/**
@name NotebookHandler
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@requires jQuery x.x.x or newer
@requires jquery.formdialog.js
@requires elpp-libs.js
@class NotebookHandler
@description A class and jQuery plugin for handling saving, network connections, etc. for Elpp Notebook.
*/


(function(window, $, ebooklocalizer){

    /**
     * Some data structures from Elpp libs
     */
    var ContentStorage = window.Elpp.ContentStorage;
    var Userinfo = window.Elpp.Userinfo;
    var Userlist = window.Elpp.Userlist;


    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var handler = new NotebookHandler(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[handlerdata]]');
            return data;
        }
    }
    
    $.fn.notebookhandler = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in notebookhandler.');
            return false;
        }
    }
    
    /*****************************
     * NotebookHandler class
     * @class NotebookHandler
     * @constructor
     * @param {jQuery} place - Place for handler
     * @param {Object} options - options for the handler
     ****************************/
    
    NotebookHandler = function(place, options){
        window.nbhandler = this;
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.place.trigger('updaterequest', {contexttype: this.context.type, contextid: this.context.id, apptype: 'notebook', appid: this.nbname});
        this.show();
    };
    
    /**
     * Init nbhandler
     * @param {Object} options - data and settings for openging notebook
     *                 options.storage - storage data in format of ContentStorage.
     */
    NotebookHandler.prototype.init = function(options){
        options = $.extend(true, {}, this.defaults, options);
        var storage = $.extend(true, {defaultlang: 'fi'}, options.storage);
        var storageparts = $.extend(true, [], options.storageparts);
        var updateparts = $.extend(true, {}, options.updateparts);
        updateparts = this.validateUpdates(updateparts);
        storageparts = storageparts.concat(updateparts);
        this.settings = options.settings;
        this.context = options.context;
        this.users = new Userlist({
            username: this.settings.username,
            contextid: this.context.id,
            contexttype: this.context.type,
            contextdescription: this.context.description,
            users: this.context.users,
            groups: this.context.groups,
            rights: this.context.rights
        });
        this.apps = this.context.apps;
        this.appdata = options.appdata;
        this.material = this.context.material;
        this.materialdata = options.materialdata;
        // Make sure, there is this.material
        if (!this.material[0]) {
            this.material[0] = {};
        };
        var material = this.material[0];
        material.type = material.type || 'notebook';
        material.jq = material.jq || 'notebookview';
        this.nbname = this.material[0] && this.material[0].id || '';
        this.configs = $.extend(true, {}, options.configs, this.loadConfigs());
        this.nbconfigs = $.extend(true, {}, this.defaults.configs, options.configs, this.configs[this.nbname]);
        this.configforms = {};
        this.selfconfigable = true;
        this.storage = new ContentStorage(storage);
        this.updatestorage = new ContentStorage();
        // Append storageparts and updateparts to the storage.
        for (var i = 0, len = storageparts.length; i < len; i++) {
            if (storageparts[i].contentinfo) {
                this.setStorageData(storageparts[i]);
            };
        };
        window.Elpp.nbhandlerstorage = this.storage;
        window.Elpp.nbhandlerupdatestorage = this.updatestorage;
    };
    
    /**
     * Get context data
     * @returns {Object} - the same kind of data that this NotebookHandler is inited with.
     */
    NotebookHandler.prototype.getContextData = function() {
        this.nbplace.trigger('getdata');
        var nbdata = this.nbplace.data('[[notebookdata]]');
        var context = {
            appdata: {},
            materialdata: {},
            context: JSON.parse(JSON.stringify(this.context)),
            storage: this.storage.getAllData(['notebookcontent']),
            storageparts: [],
            updateparts: {}
        };
        context.materialdata[this.material[0].id] = nbdata;
        return context;
    };
    
    /**
     * Get context savedata (applications' data and all storages/storageparts)
     * @returns {Object} - materialdata, appdata, storage, storageparts and updateparts
     */
    NotebookHandler.prototype.getContextSave = function() {
        this.nbplace.trigger('getdata');
        var nbdata = this.nbplace.data('[[notebookdata]]');
        var context = {
            contextinfo: {
                context: JSON.parse(JSON.stringify(this.context))
            },
            contextdata: {
                materialdata: {},
                appdata: {},
                storage: this.storage.getAllData(['notebookcontent']),
                storageparts: [],
                updateparts: {}
            },
            saveneeded: true
        };
        context.contextdata.materialdata[this.material[0].id] = nbdata;
        return context;
    };
    
    /**
     * Show everything inside this handler
     */
    NotebookHandler.prototype.show = function(){
        this.place.html(this.templates.view);
        this.nbplace = this.place.find('.context');
        this.dialog = this.place.find('.context-popupdialog');
        this.confdialog = this.place.find('.context-configdialog');
        this.notificationarea = this.place.find('.context-control-notifications');
        this.notificationarea.vnotifier($.extend({}, NotebookHandler.notification, {level: (this.adminable ? 10 : 2)}));
        //this.place.attr('data-ffwtheme', this.configs.ffwtheme);
        this.showNotebook();
    }
    
    /**
     * Show the notebook
     */
    NotebookHandler.prototype.showNotebook = function(index){
        index = index || 0;
        this.nbname = this.material[index].id;
        var nbdata = JSON.parse(JSON.stringify(this.materialdata[this.nbname] || {type: 'notebook', name: this.nbname}));
        nbdata.settings = this.settings;
        nbdata.settings.users = this.users;
        var nbname = nbdata.name || (nbdata.data && nbdata.data.bookid) || 'undefined';
        nbdata.settings.gotolink.appname = nbname;
        nbdata.configs = this.nbconfigs = this.configs[nbname] || {};
        this.nbplace.notebookview(nbdata);
        this.appsOpen = 1;
        this.nbname = this.nbplace.attr('data-appname');
    }
    
    /**
     * Set the title of the handler
     */
    NotebookHandler.prototype.setTitle = function() {
        this.place.find('.context-title').text(this.nbtitle);
    };
    
    /**
     * Close notebook
     */
    NotebookHandler.prototype.close = function() {
        this.appsToClose = this.appsOpen || 0;
        var $apps = this.place.find('[data-appname]');
        $apps.trigger('closechildrenapp');
    };
    
    /**
     * One children app is closed
     */
    NotebookHandler.prototype.appClosed = function() {
        this.appsToClose--;
        if (this.appsToClose <= 0) {
            // If all apps are closed, data can be saved and this context can be closed.
            this.saveContent();
            this.sendFromOutbox();
            this.place.trigger('getcontextsave');
            this.place.trigger('closeok', {contexttype: this.context.type, contextid: this.context.id});
        };
    };
    
    /**
     * Show import dialog
     */
    NotebookHandler.prototype.importFile = function(){
        var uilang = this.settings.uilang;
        this.dialog.html(this.templates.fileimport);
        this.dialog.find('.nbhandler-closebutton').text(ebooklocalizer.localize('nbhandler:cancel', uilang));
        //this.showDialog();
        this.dialog.find('.context-handler-importbutton').click();
    }
    
    /**
     * Show export dialog
     * @param {Object} data - Filedata
     */
    NotebookHandler.prototype.exportFile = function(data){
        var uilang = this.settings.uilang;
        var content = $(this.templates.fileexport);
        this.dialog.html(content);
        this.dialog.find('a')
            .attr('download', data.filename)
            .attr('href', data.objecturl)
            .text(ebooklocalizer.localize('nbhandler:save', uilang));
        this.dialog.find('.nbhandler-closebutton').text(ebooklocalizer.localize('nbhandler:cancel', uilang));
        var mevent = new MouseEvent('click', {view: window, publes: true, cancelable: false});
        this.dialog.find('a').get(0).dispatchEvent(mevent);
        //this.showDialog();
    };
    
    /**
     * Parse and validate data in updateparts. Check that contentType matches with the
     * type given inside the data.
     * @param {Object} updates  - updated data ordered by contentType.
     *                            Each contentType has an array of JSON.stringified
     *                            elementobjects.
     * @return {Array} An array of JSON.parsed and typechecked element data.
     */
    NotebookHandler.prototype.validateUpdates = function(updates) {
        var contentType, carr, i, len, jsonelem, element;
        var result = [];
        for (contentType in updates) {
            carr = updates[contentType];
            for (i = 0, len = carr.length; i < len; i++) {
                try {
                    element = JSON.parse(carr[i]);
                } catch(err) {
                    console.log('Error: Not valid updates', err);
                    element = null;
                };
                if (element && element.contentType === contentType && element.contentinfo && element.contentinfo.contentType === contentType) {
                    result.push(element);
                };
            };
        };
        return result;
    }

    /**
     * Get the savedata (all dirty elements)
     * @param {Boolean} readonly - if the notebook is in readonly mode
     * @returns {Array} an array of all drity elements
     */
    NotebookHandler.prototype.getSavedata = function(readonly){
        var ctypes = this.storage.getContentTypes();
        if (readonly) {
            var nbindex = ctypes.indexOf('notebookcontent');
            ctypes.splice(nbindex, 1);
        };
        var ctype;
        var savearray = [];
        var dirtydata;
        for (var i = 0, len = ctypes.length; i < len; i++) {
            // Give the outer system a list of objects to save.
            ctype = ctypes[i];
            dirtydata = this.storage.getDirtyContent({contentType: ctype});
            for (var j = 0, dirlen = dirtydata.length; j < dirlen; j++) {
                savearray.push(dirtydata[j]);
            };
        };
        return savearray;
    };
    
    /**
     * Get the senddata from outbox
     * @returns {Array} an array of element datas to be sent
     */
    NotebookHandler.prototype.getSenddata = function() {
        var ctypes = this.storage.getContentTypes();
        var ctype;
        var sendarray = [], senddata;
        var obdata;
        for (var i = 0, len = ctypes.length; i < len; i++) {
            // Send as an array with objects with needed attributes and the content data.
            // Outer systen can then just process the list.
            ctype = ctypes[i];
            obdata = this.storage.getOutboxContent({contentType: ctype});
            for (var j = 0, oblen = obdata.length; j < oblen; j++) {
                // Mark all elements as sent and dirty.
                // (If sending is successfull, that is true and the data in server is correct.)
                // (If element comes from server as update, it is dirty, i.e., must be saved locally)
                obdata[j].contentinfo.isoutbox = false;
                senddata = {
                    contexttype: 'own_notes',
                    contextid: this.nbname,
                    subcontext: obdata[j].contentinfo.subcontext,
                    dataid: obdata[j].name,
                    data: JSON.stringify(obdata[j]),
                    send_to: obdata[j].contentinfo.sendto || [],
                    datatype: obdata[j].contentType,
                    datamodified: obdata[j].contentinfo.timestamp,
                    lang: obdata[j].contentinfo.lang
                    //updatetype: 'notebook',
                    //updateid: this.nbname,
                    //dataid: obdata[j].name,
                    //pubtype: obdata[j].contentinfo.pubtype,
                    //public_to: obdata[j].contentinfo.sendto,
                    //dataclass: 'refdata',
                    //datatype: obdata[j].contentType,
                    //datamodified: obdata[j].contentinfo.timestamp,
                    //data: obdata[j]
                };
                sendarray.push(senddata);
            };
        };
        return sendarray;
    };
    
    /**
     * Save all dirty content and send all unsent content from storages.
     * @param {Boolean} readonly - if the notebook is in readonly mode
     */
    NotebookHandler.prototype.saveContent = function(readonly){
        var savearray = this.getSavedata(readonly) || [];
        var sendarray = this.getSenddata() || [];
        if (savearray.length + sendarray.length > 0) {
            this.place.trigger('contentdatasave', [savearray, sendarray, {contexttype: this.context.type, contextid: this.context.id, apptype: 'notebook', appid: this.nbname}]);
        };
    };
    
    /**
     * Save the whole storage as it is.
     */
    NotebookHandler.prototype.saveStorage = function(){
        // Get an array of all dirty elements
        var dirties = this.storage.getDirtyContent();
        var dirtylist = [];
        var dirty;
        for (var i = 0, len = dirties.length; i < len; i++) {
            dirtylist.push({
                success: true,
                data: dirties[i]
            });
        };
        // Get the data of the storage
        var storage = this.getStorageAllData();
        // Give the data of whole storage to the outer system for saving.
        this.place.trigger('storagesave', {contexttype: this.context.type, contextid: this.context.id, apptype: 'notebook', appid: this.nbname, storage: storage, dirtylist: dirtylist});
    }
    
    /**
     * Send all content from outbox
     */
    NotebookHandler.prototype.sendFromOutbox = function(){
        var sendarray = this.getSenddata();
        if (sendarray.length > 0) {
            this.place.trigger('contentdatasend', [sendarray]);
        };
    };

    /**
     * Set data to storage
     * @param {Object} contentdata - contentdata to be stored.
     *     contentdata = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     */
    NotebookHandler.prototype.setStorageData = function(contentdata) {
        this.storage.setData(contentdata);
    };
    
    /**
     * Get data from storage
     * @param {Object} query - object that tells what to get from storage.
     *     query = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *     }
     * @returns {Object} - object with all data of element in same format as given to .setStorageData(data).
     *      {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *      }
     */
    NotebookHandler.prototype.getStorageData = function(query){
        result = this.storage.getData(query);
        return result;
    };
    
    /**
     * Get all the data from the storage
     */
    NotebookHandler.prototype.getStorageAllData = function() {
        return this.storage.getAllData();
    };
    
    /**
     * Data is saved. Mark dirty elements clean
     * @param {Object|Array} item           The data for identifying the element to be cleared
     * @param {String} item[].name         The id of the content
     * @param {String} item[].contentType  The type of the content
     * @param {Object} item[].contentdata  The data of the content
     * @param {Object} item[].contentinfo  The info of the content
     */
    NotebookHandler.prototype.dataSaved = function(item) {
        if (typeof(item.length) === 'undefined') {
            item = [item];
        };
        for (var i = 0, len = item.length; i < len; i++) {
            this.storage.cleanDirty(item[i].contentinfo);
            switch (item[i].contentType) {
                case 'notebookcontent':
                    var data = {
                        name: item[i].name,
                        lang: item[i].contentinfo.lang,
                        timestamp: item[i].contentinfo.timestamp,
                        type: item[i].contentdata && item[i].contentdata.type || ''
                    };
                    this.nbplace.trigger('datasaved', [data]);
                    break;
                default:
                    break;
            };
        };
        this.notify({
            "nclass": "localsave",
            "message": "Save done.",
            "state": "black",
            "ttl": 300
        });
    };
    
    /**
     * Data save failed
     * @param {Object} item           The data for identifying the element
     * @param {String} item.name         The id of the content
     * @param {String} item.contentType  The type of the content
     * @param {Object} item.contentdata  The data of the content
     * @param {Object} item.contentinfo  The info of the content
     */
    NotebookHandler.prototype.dataSaveFailed = function(item) {
        this.notify({
            "nclass": "localsave",
            "message": "Could not save data. Trying later.",
            "state": "red",
            "ttl": 1000
        });
    };

    /**
     * Data is sent
     * @param {Object|Array} item           The data for identifying the element
     * @param {String} item[].name         The id of the content
     * @param {String} item[].contentType  The type of the content
     * @param {Object} item[].contentdata  The data of the content
     * @param {Object} item[].contentinfo  The info of the content
     */
    NotebookHandler.prototype.dataSent = function(item) {
        if (typeof(item) === 'string') {
            try {
                item = JSON.parse(item);
            } catch (err) {
                item = [];
            };
        };
        if (typeof(item.length) === 'undefined') {
            item = [item];
        };
        for (var i = 0, len = item.length; i < len; i++) {
            this.storage.cleanOutbox(item[i].contentinfo);
        };
        this.notify({
            "nclass": "network",
            "message": "Sending done.",
            "state": "black",
            "ttl": 300
        });
    };
    
    /**
     * Data send failed
     * @param {Object} item           The data for identifying the element
     * @param {String} item.name         The id of the content
     * @param {String} item.contentType  The type of the content
     * @param {Object} item.contentdata  The data of the content
     * @param {Object} item.contentinfo  The info of the content
     */
    NotebookHandler.prototype.dataSendFailed = function(item) {
        this.notify({
            "nclass": "network",
            "message": "Could not send data. Trying later.",
            "state": "red",
            "ttl": 1000
        });
    };
    
    /**
     * Trigger notifications
     * @param {Object} ntification    Notification object: {id: "...", nclass: "...", message: "...", ttl: ...}
     */
    NotebookHandler.prototype.notify = function(ntification){
        this.notificationarea.trigger('newnotification', ntification);
    }
    /**
     * Trigger removing of notifications
     * @param {Object} ntification    Notification object: {id: "...", nclass: "...", message: "...", ttl: ...}
     */
    NotebookHandler.prototype.denotify = function(ntification){
        this.notificationarea.trigger('removenotification', ntification);
    }
    
    /**
     * Show popupdialog
     */
    NotebookHandler.prototype.showDialog = function(){
        this.dialog.css('display', 'block');
    }
    
    /**
     * Hide popupdialog
     */
    NotebookHandler.prototype.hideDialog = function(){
        this.dialog.empty().css('display', 'none');
    }
    
    /**
     * Show config dialog
     */
    NotebookHandler.prototype.showConfigs = function(){
        var uilang = this.settings.uilang;
        var confforms = JSON.parse(JSON.stringify(this.configforms));
        var confvalues = JSON.parse(JSON.stringify(this.configs));
        var formsets = [];
        var apps = this.place.find('[data-appname]');
        var appname;
        for (var i = 0, len = apps.length; i < len; i++){
            appname = apps.eq(i).attr('data-appname');
            if (confforms[appname]) {
                formsets.push(appname);
            };
        };
        if (this.confdialog) {
            this.confdialog.formdialog({
                type: 'dialog',
                name: 'ElppNotebookConfig',
                data: {
                    title: ebooklocalizer.localize('nbhandler:configs', uilang),
                    icon: this.icons.config,
                    formsets: formsets,
                    forms: confforms,
                    values: confvalues
                },
                config: {
                    position: {
                        x: 300,
                        y: 300
                    }
                },
                settings: {
                    uilang: uilang,
                    lang: this.settings.lang,
                    modal: false
                }
            });
        };
    };
    
    /**
     * Save config data
     * @param {Object} data - Optional config data to be saved.
     *                        If not given, just save all configs.
     *                        format: {appname: '<application id>', configs: {<configs to be saved>}}
     */
    NotebookHandler.prototype.saveConfigs = function(data){
        if (this.selfconfigable) {
            // Save given data dataor current data in this module
            if (typeof(data) !== 'undefined') {
                // Add given config data
                var appname = data.appname;
                var configs = data.configs;
                this.configs[appname] = configs;
            };
            // Save all config data
            if (false && this.settings.ischromeapp) {
                // Use chrome.storage
                chrome.storage.local.set({'elppnotebook_configs': this.configs}, function(){
                    console.log('saved in chromeapp');
                });
            } else if (false && this.settings.ischromesandbox) {
                
            } else {
                // Use localStorage
                var confstring = JSON.stringify(this.configs);
                try {
                    localStorage.setItem('elppnotebook_configs', confstring);
                } catch (err) {
                    console.log('error saving elpp notebook configs', err);
                };
            };
        } else {
            // Save config data in parent module
            if (typeof(data) === 'undefined') {
                // Not given data, but save all. (Else save only given data)
                data = {
                    type: 'configsave',
                    appname: 'elppnotebook',
                    configs: JSON.parse(JSON.stringify(this.configs))
                };
            };
            this.place.trigger('saveconfigs', data);
        };
    };
    
    /**
     * Load configs from localStorage
     * @returns {Object} config-object
     */
    NotebookHandler.prototype.loadConfigs = function(){
        var confs = {};
        if (false && this.settings.ischromeapp) {
            // Use chrome.storage
            chrome.storage.local.get('elppnotebook_configs', function(obj){
                confs = obj['elppnotebook_configs'];
            })
        } else if (false && this.settings.ischromesandbox) {
                
        } else {
            try {
                confs = JSON.parse(localStorage.getItem('elppnotebook_configs')) || {};
            } catch (err) {
                confs = {};
            };
        };
        return confs;
    };
    
    /**
     * Put data to the storage
     * @param {Object} contentdata - contentdata to be stored.
     *     contentdata = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     */
    NotebookHandler.prototype.storageSetData = function(contentdata){
        this.storage.setData(contentdata);
        // TODO sending etc.
    };
    
    /**
     * Get data from storage
     * @param {Object} query - object that tells what to get from storage.
     *     query = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *     }
     * 
     */
    NotebookHandler.prototype.storageGetData = function(query){
        result = this.storage.getData(query);
        return result;
    }
    
    
    /**
     * Init event handlers
     */
    NotebookHandler.prototype.initHandlers = function(){
        var handler = this;
        /**
         * Importing and exporting files.
         */
        this.place.off('importfile').on('importfile', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            handler.importFile();
        }).off('exportfileas').on('exportfileas', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            handler.exportFile(data);
        }).off('change', '.context-popupdialog input[type="file"]').on('change', '.context-popupdialog input[type="file"]', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var thefile = event.target.files[0];
            handler.loadNotebook(thefile);
            handler.hideDialog();
        }).off('click', '.context-popupdialog button.popupclose').on('click', '.context-popupdialog button.popupclose', function(event, data){
            handler.hideDialog();
        });
        /**
         * Context event api
         */
        this.place.off('mergeappdata').on('mergeappdata', function(event, data) {
            event.stopPropagation();
            var appdata;
            for (var appname in data) {
                appdata = data[appname];
                handler.place.find('[data-appname="' + appname + '"]').trigger('mergecontent', appdata);
            };
        }).on('closeapp', function(event){
            event.stopPropagation();
            handler.close();
        }).off('appready').on('appready', function(event, data){
            data.contexttype = handler.context.type;
            data.contextid = handler.context.id;
            handler.nbname = data.appid;
            handler.nbtitle = data.title || 'Notebook';
            handler.setTitle();
        }).off('apptitlechanged').on('apptitlechanged', function(event, data){
            handler.nbname = data.appid;
            handler.nbtitle = data.title || 'Notebook';
            handler.setTitle();
        }).off('getuserlist').on('getuserlist', function(event, data){
            event.stopPropagation();
            var users = handler.getUserlist();
            var target = $(event.target);
            target.trigger('userlist', users);
        }).off('edit_started').on('edit_started', function(event, data){
            data.contexttype = handler.context.type;
            data.contextid = handler.context.id;
        }).off('edit_stopped').on('edit_stopped', function(event, data){
            data.contexttype = handler.context.type;
            data.contextid = handler.context.id;
        }).off('getcontextdata').on('getcontextdata', function(event, data){
            event.stopPropagation();
            var cdata = handler.getContextData();
            handler.place.trigger('contextdata', [cdata]);
        }).off('getcontextsave').on('getcontextsave', function(event, data){
            event.stopPropagation();
            var cdata = handler.getContextSave();
            var appinfo = {
                contexttype: handler.context.type,
                contextid: handler.context.id,
            };
            handler.place.trigger('contextsave', [cdata, appinfo]);
        });
        /**
         * Get contentinfo from storage. (With queries)
         */
        this.place.off('getcontentinfo').on('getcontentinfo', function(event, data) {
            // get contentinfo from storage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var target = $(event.target);
            handler.getContentinfo(data, target);
        });
        /**
         * Get content from storage. (With queries)
         */
        this.place.off('getcontent').on('getcontent', function(event, data) {
            // get content from storage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var target = $(event.target);
            handler.getContent(data, target);
        });
        /**
         * Get updated content from updatestorage. (With queries)
         */
        this.place.off('getupdatecontent').on('getupdatecontent', function(event, data) {
            // get content from updatestorage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var query, ctype;
            var target = $(event.target);
            var result;
            for (var i = 0, len = data.contentType.length; i < len; i++){
                ctype = data.contentType[i];
                if (data.anchors) {
                    result = handler.updatestorage.getContentByAnchor({
                        contentType: ctype,
                        lang: data.lang,
                        anchors: data.anchors
                    });
                } else if (data.names) {
                    result = handler.updatestorage.getContentByName({
                        contentType: ctype,
                        lang: data.lang,
                        names: data.names
                    });
                } else {
                    result = {contentType: ctype, contentdata: {}};
                };
                var contentlist = [];
                for (var cname in result.contentdata) {
                    contentlist.push(cname);
                };
                handler.updatestorage.removeContentByName(ctype, contentlist, data.lang);
                target.trigger('reply_getcontent', result);
            };
        });
        /**
         * Save content to the storage.
         */
        this.place.off('setcontent').on('setcontent', function(event, data, isdirty){
            event.stopPropagation();
            event.preventDefault();
            if (typeof(data) === 'object' && typeof(data.length) === 'undefined'){
                data = [data];
            }
            for (var i = 0, len = data.length; i < len; i++) {
                handler.storage.setData(data[i], isdirty);
                if (data[i].contentinfo && data[i].contentinfo.isoutbox) {
                    handler.storage.sendOutbox(data[i].contentinfo);
                };
                //handler.newdata(data[i]);
            };
        });
        /**
         * Save content to the updatestorage.
         */
        this.place.off('setupdatecontent').on('setupdatecontent', function(event, data, isdirty){
            event.stopPropagation();
            event.preventDefault();
            if (typeof(data) === 'object' && typeof(data.length) === 'undefined'){
                data = [data];
            }
            for (var i = 0, len = data.length; i < len; i++) {
                handler.updatestorage.setData(data[i], isdirty);
                if (data[i].contentinfo && data[i].contentinfo.isoutbox) {
                    handler.updatestorage.sendOutbox(data[i].contentinfo);
                };
            };
            if (handler.nbplace) {
                handler.nbplace.trigger('updates_available');
            };
        });
        /**
         * Save contentinfo
         */
        this.place.on('setcontentinfo', function(event, data){
            event.stopPropagation();
            handler.storage.updateContentInfo(data);
        });
        /**
         * Set and save content
         */
        this.place.off('setsavecontent').on('setsavecontent', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            handler.place.trigger('setcontent', [data, false]); // false = no isdirty
            handler.place.trigger('setupdatecontent', [data, false]);
            handler.saveStorage();
        });
        /**
         * Mark element for sending, if given, and send all content from outbox.
         */
        this.place.on('sendcontent', function(event, contentinfo){
            event.stopPropagation();
            if (contentinfo) {
                handler.storage.sendOutbox(contentinfo);
            };
            handler.sendFromOutbox();
        });
        /**
         * Save all dirty content from storage
         */
        this.place.on('savecontent', function(event, isreadonly){
            event.stopPropagation();
            handler.saveContent(isreadonly);
        });
        /**
         * Reply from contentdatasave
         */
        this.place.on('reply_contentdatasave reply_storagesave', function(event, message) {
            event.stopPropagation();
            message = message || [];
            if (typeof(message) === 'object' && typeof(message.length) === 'undefined') {
                message = [message];
            };
            var messitem;
            for (var i = 0, len = message.length; i < len; i++) {
                messitem = message[i];
                if (messitem.success) {
                    handler.dataSaved(messitem.data);
                } else {
                    handler.dataSaveFailed(messitem.data);
                };
            };
        });
        /**
         * Reply from contentdatasend
         */
        this.place.on('reply_contentdatasend', function(event, message) {
            event.stopPropagation();
            message = message || [];
            if (typeof(message) === 'object' && typeof(message.length) === 'undefined') {
                message = [message];
            };
            var messitem;
            for (var i = 0, len = message.length; i < len; i++) {
                messitem = message[i];
                if (messitem.success) {
                    handler.dataSent(messitem.data);
                } else {
                    handler.dataSendFailed(messitem.data);
                };
            };
        });
        /**
         * Handle 'closeappok'events from children applications
         */
        this.place.on('closeappok', function(event, data){
            // 'closeok' came from a children app. Don't let it pass to outer system.
            // TODO count closings and trigger 'closeok', when all children apps are closed.
            event.stopPropagation();
            handler.appClosed();
        });
        /**
         * Handle 'closeok'events from children applications
         */
        this.place.on('closeok', function(event, data){
            if (event.target !== this) {
                // 'closeok' came from a children app. Don't let it pass to outer system.
                // TODO count closings and trigger 'closeok', when all children apps are closed.
                event.stopPropagation();
            };
        });

        /**
         * Handle notifications to the notification toolbar.
         */
        this.place.on('notification', function(event, data){
            event.stopPropagation();
            handler.notify(data);
        });
        this.place.on('denotification', function(event, data){
            event.stopPropagation();
            handler.denotify(data);
        });
        // Config dialog
        this.place.off('registerconfigdialog').on('registerconfigdialog', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            handler.configforms[data.appid] = data.confform;
        }).off('formdialog-valuechange').on('formdialog-valuechange', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var dialogdata = data.dialogdata;
            var apps = dialogdata.formsets;
            var values = dialogdata.values;
            var appname, confs;
            for (var i = 0, len = apps.length; i < len; i++){
                appname = apps[i];
                confs = values[appname];
                handler.place.find('[data-appname="'+appname+'"]').trigger('setconfigs', confs);
            };
        }).off('config_changed').on('config_changed', function(event, data){
            var key = data.key;
            var config = data.config;
            handler.configs[key] = config;
            handler.saveConfigs();
        }).off('saveconfigs').on('saveconfigs', function(event, data){
            if (event.target !== this) {
                event.stopPropagation();
                handler.saveConfigs(data);
            };
        }).off('configdialogopen').on('configdialogopen', function(event, data){
            event.stopPropagation();
            handler.showConfigs();
        });
    };
    
    /**
     * Get contentinfo from storage by anchors or by names and return to asking application
     * @param {Object} query           Data for the query in the storage
     * @param {String[]} query.contentType   A list of content types to search for
     * @param {String[]} [query.anchors]     A list of anchors to search for or
     * @param {String[]} [query.names]       A list of element id's to search for
     * @param {String[]} query.lang          The language of searched elements
     * @param {jQuery} returntarget          The jQuery-element where to trigger the reply
     */
    NotebookHandler.prototype.getContentinfo = function(query, returntarget) {
        var ctype, result;
        for (var i = 0, len = query.contentType.length; i < len; i++) {
            ctype = query.contentType[i];
            if (query.anchors) {
                // Query is by anchors
                result = this.storage.getContentInfoByAnchor({
                    contentType: ctype,
                    lang: query.lang,
                    anchors: query.anchors
                });
            } else if (query.names) {
                result = this.storage.getContentInfoByName({
                    contentType: ctype,
                    lang: query.lang,
                    names: query.names
                });
            } else {
                result = {contentType: ctype, contentinfo: {}};
            };
            returntarget.trigger('reply_getcontentinfo', result);
        };
    };
    
    /**
     * Get content from storage by anchors or by names and return to asking application
     * @param {Object} query           Data for the query in the storage
     * @param {String[]} query.contentType   A list of content types to search for
     * @param {String[]} [query.anchors]     A list of anchors to search for or
     * @param {String[]} [query.names]       A list of element id's to search for
     * @param {String[]} query.lang          The language of searched elements
     * @param {jQuery} returntarget          The jQuery-element where to trigger the reply
     */
    NotebookHandler.prototype.getContent = function(query, returntarget) {
        var ctype, result;
        for (var i = 0, len = query.contentType.length; i < len; i++) {
            ctype = query.contentType[i];
            if (query.anchors) {
                // Query is by anchors
                result = this.storage.getContentByAnchor({
                    contentType: ctype,
                    lang: query.lang,
                    anchors: query.anchors
                });
            } else if (query.names) {
                result = this.storage.getContentByName({
                    contentType: ctype,
                    lang: query.lang,
                    names: query.names
                });
            } else {
                result = {contentType: ctype, contentdata: {}};
            };
            returntarget.trigger('reply_getcontent', result);
        };
    };
    
    /**
     * Get lists of users in this context (teachers, students,...?)
     * @returns {Object} object of users
     */
    NotebookHandler.prototype.getUserlist = function() {
        return this.users;
    };
    
    /**
     * Load notebook from a file
     * @param {File} thefile - file-object to load
     */
    NotebookHandler.prototype.loadNotebook = function(thefile){
        var handler = this;
        if (thefile.type === 'application/json' || thefile.type === '') {
            var reader = new FileReader();
            reader.onload = function(ev){
                var jsondata;
                try {
                    jsondata = JSON.parse(ev.target.result);
                } catch (err) {
                    jsondata = {};
                    console.log('parse error: ', err);
                };
                if (jsondata.type === 'book') {
                    handler.nbname = jsondata.name;
                    handler.apps[0].id = jsondata.name;
                    handler.appdata[jsondata.name] = jsondata;
                    handler.place.trigger('updaterequest', {contexttype: this.context.type, contextid: this.context.id, apptype: 'notebook', appid: handler.nbname});
                    handler.showNotebook();
                } else {
                    alert('The file was invalid format.');
                }
            };
            reader.readAsText(thefile);
        };
    };
    
    /**
     * Some icons
     */
    NotebookHandler.prototype.icons = {
        config: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-gear"><path style="stroke: none;" d="M17.26 2.2 a13 13 0 0 1 7.7 4.44 a3 8 60 0 0 2.256 3.91 a13 13 0 0 1 0 8.89 a3 8 120 0 0 -2.256 3.91 a13 13 0 0 1 -7.7 4.44 a3 8 0 0 0 -4.52 0 a13 13 0 0 1 -7.7 -4.44 a3 8 60 0 0 -2.256 -3.91 a13 13 0 0 1 0 -8.89 a3 8 120 0 0 2.256 -3.91 a13 13 0 0 1 7.7 -4.44 a3 8 0 0 0 4.52 0 M15 11 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8 z"></path></svg>'
    }
    
    /**
     * CSS-styles
     */
    NotebookHandler.prototype.styles = [
        '.context-handler {position: absolute; top: 0; bottom: 0; left: 0; right: 0; display: flex; flex-direction: column; align-items: stretch;}',
        '.context {flex-grow: 1;}',
        '.context-controlbar {display: flex; flex-direction: row; justify-content: space-between; flex-shrink: 0; flex-grow: 0;}',
        '.context-title {flex-grow: 1;}',
        '@media print {',
        '    .context-handler {position: static; display: block;}',
        '    .context-controlbar {display: none;}',
        '}'
    ].join('\n');
    
    /**
     * Html-templates
     */
    NotebookHandler.prototype.templates = {
        view: [
            '<style type="text/css" scoped>' + NotebookHandler.prototype.styles + '</style>',
            '<div class="context-handler">',
            '    <div class="context-controlbar ffwidget-background-colored">',
            '        <div class="context-control-buttons"></div>',
            '        <div class="context-title ffwidget-title">Notebook</div>',
            '        <div class="context-control-notifications"></div>',
            '    </div>',
            '    <div class="context"></div>',
            // File dialog
            '    <div class="context-popupdialog" style="display: none; z-index: 30; position: fixed; top: 4em; right: 3em; padding: 1em 2em; background-color: #eee; border: 1px solid black; border-radius: 1em; box-shadow: 8px 8px 12px rgba(0,0,0,0.3);"></div>',
            // Config dialog
            '    <div class="context-configdialog"></div>',
            '</div>'
        ].join('\n'),
        fileimport: [
            '<label class="ffwidget-button context-handler-importbutton"><input type="file" name="importfile" style="display: none;"><span><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-folder"><path stroke="none" fill="black" d="M2 6 l5 0 l2 2 l14 0 l0 2 l-16 0 l-4 14 l5 -12 l22 0 l-5 13 l-23 0z"></path></svg></span></input></label>',
            '<button class="popupclose nbhandler-closebutton ffwidget-button"></button>',
        ].join('\n'),
        fileexport: [
            '<a class="ffwidget-button popupclose" download="" href=""></a>',
            '<button class="popupclose nbhandler-closebutton ffwidget-button"></button>',
        ].join('\n')
    };
    
    /**
     * Default values for options
     */
    NotebookHandler.prototype.defaults = {
        context: {
            type: 'own_notes',
            id: 'nbhandler',
            material: [],
        },
        materialdata: {},
        nbsettings: {
            lang: 'en',
            uilang: 'en',
            username: 'Anonymous',
            role: 'student',
            readonly: false,
            hasparent: true,
            configable: true,
            onepagemode: false,
            gotolink: {
                appname: ''
            }
        },
        settings: {
            lang: 'en',
            uilang: 'en',
            username: 'Anonymous',
            role: 'student',
            readonly: false,
            hasparent: true,
            configable: true,
            onepagemode: false,
            gotolink: {
                appname: ''
            },
            ischromeapp: typeof(window.chrome) !== 'undefined' && typeof(window.chrome.storage) !== 'undefined',
            ischromesandbox: typeof(window.chrome) !== 'undefined' && typeof(window.chrome.storage) === 'undefined'
        },
        configs: {
            nbtheme: 'default',
            ffwtheme: 'default-blue'
        }
    };
    
    NotebookHandler.notification = {
        classes: [
            {
                name: 'network',
                label: 'Network',
                type: 'trafficlight',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="-5 -5 40 40" class="mini-icon mini-icon-cloudup"><path style="stroke: none;" d="M5 25 a4 4 0 1 1 2 -10 a6 6 0 0 1 12 0 a4 4 0 0 1 7 3 a3 3 0 0 1 0 7z m7 -2 l4 0 l0 -3 l2 0 l-4 -4 l-4 4 l2 0z" /></svg>',
                //icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z"></path><path style="stroke: none; fill: rgba(255,255,255,0.2);" d="M5 15 a10 10 0 0 1 20 0 a12 12 0 0 0 -20 0z"></path><path style="stroke: none; fill: rgba(255,255,255,0.5);" d="M5 15 a10 10 0 0 1 20 0 a10.2 10.2 0 0 0 -20 0z"></path><path style="stroke: none; fill: rgba(0,0,0,0.1);" d="M5 15 a10 10 0 0 0 20 0 a12 12 0 0 1 -20 0z"></path><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M5 15 a10 10 0 0 0 20 0 a10.2 10.2 0 0 1 -20 0z"></path></svg>',
                event: 'networknotifclick',
                level: 0
            },
            {
                name: 'localsave',
                label: 'Saving',
                type: 'trafficlight',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="-7 -7 44 44" class="mini-icon mini-icon-save mini-icon-disk"><path style="stroke: none;" d="M1 1 l23 0 l5 5 l0 23 l-28 0z m5 2 l0 8 l17 0 l0 -8z m12 1 l3 0 l0 6 l-3 0z m-13 10 l0 14 l20 0 l0 -14z m3 3 l14 0 l0 2 l-14 0z m0 3 l14 0 l0 2 l-14 0z m0 3 l14 0 l0 2 l-14 0z"></path></svg>',
                event: 'localsavenotifclick',
                level: 0
            },
            {
                name: 'messages',
                label: 'Messages',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /></svg>',
                event: 'showmessage',
                level: 0
            },
            {
                name: 'quick',
                label: 'Quick messages',
                type: 'quickmsg',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /></svg>',
                event: 'quickmsgack',
                level: 0
            },
            {
                name: 'chat',
                label: 'Chat',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-chat"><path style="stroke: none;" d="M11 6 a4 4 0 0 1 4 -4 l10 0 a4 4 0 0 1 4 4 l0 2 a4 4 0 0 1 -4 4 a8 8 0 0 0 2 4 a12 12 0 0 1 -8 -4 l-4 0 a4 4 0 0 1 -4 -4z m-1 2 l0 2 a4 4 0 0 0 4 4 l6 0 a4 4 0 0 1 -4 4 l-3 0 a12 12 0 0 1 -8 4 a8 8 0 0 0 2 -4 l-2 0 a4 4 0 0 1 -4 -4 l0 -2 a4 4 0 0 1 4 -4z" /></svg>',
                event: 'showchat',
                level: 0
            },
            {
                name: 'debug',
                label: 'Debug',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 50 50" class="pssicon pssicon-wrench"><path style="stroke: none;" fill="black" d="M35 5 a12 12 0 1 0 9 9 l-9 9 a10 10 0 0 1 -9 -9 l9 -9z m-15 17.5 l-17 17 a2 2 0 0 0 7 7 l17 -17z m-16 18 a2 2 0 0 1 5 5 a2 2 0 0 1 -5 -5z"></path></svg>',
                event: 'showdebug',
                level: 9
            }
        ],
        level: 3,
        align: 'right'
        
    }
    NotebookHandler.localization = {
        en: {
            'nbhandler:configs': 'Settings',
            'nbhandler:save': 'Save',
            'nbhandler:cancel': 'Cancel'
        },
        fi: {
            'nbhandler:configs': 'Asetukset',
            'nbhandler:save': 'Tallenna',
            'nbhandler:cancel': 'Peruuta'
        },
        sv: {
            'nbhandler:configs': 'Inställningar',
            'nbhandler:save': 'Spara',
            'nbhandler:cancel': 'Avbryt'
        }
    };
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(NotebookHandler.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(NotebookHandler.localization);
    };

    
    
})(window, jQuery, window.ebooklocalizer)