/**
 * Copyleft 2010-2011 Jay and Han (laughinghan@gmail.com)
 *   under the GNU Lesser General Public License
 *     http://www.gnu.org/licenses/lgpl.html
 * Project Website: http://mathquill.com
 */

(function() {

var $ = jQuery,
  undefined,
  _, //temp variable of prototypes
  jQueryDataKey = '[[mathquill internal data]]',
  min = Math.min,
  max = Math.max;

/*************************************************
 * Abstract base classes of blocks and commands.
 ************************************************/

/**
 * MathElement is the core Math DOM tree node prototype.
 * Both MathBlock's and MathCommand's descend from it.
 */
function MathElement(){}
_ = MathElement.prototype;
_.prev = 0;
_.next = 0;
_.parent = 0;
_.firstChild = 0;
_.lastChild = 0;
_.eachChild = function(fn) {
  for (var child = this.firstChild; child; child = child.next)
    if (fn.call(this, child) === false) break;

  return this;
};
_.foldChildren = function(fold, fn) {
  this.eachChild(function(child) {
    fold = fn.call(this, fold, child);
  });
  return fold;
};
_.keydown = function(e) {
  return this.parent.keydown(e);
};
_.textInput = function(ch) {
  return this.parent.textInput(ch);
};

/**
 * Commands and operators, like subscripts, exponents, or fractions.
 * Descendant commands are organized into blocks.
 * May be passed a MathFragment that's being replaced.
 */
function MathCommand(){}
_ = MathCommand.prototype = new MathElement;
_.init = function(cmd, html_template, text_template, replacedFragment) {
  var self = this; // minifier optimization

  if (cmd) self.cmd = cmd;
  if (html_template) self.html_template = html_template;
  if (text_template) self.text_template = text_template;

  self.jQ = $(self.html_template[0]).data(jQueryDataKey, {cmd: self});
  self.initBlocks(replacedFragment);
};
_.initBlocks = function(replacedFragment) {
  var self = this;
  //single-block commands
  if (self.html_template.length === 1) {
    self.firstChild =
    self.lastChild =
    self.jQ.data(jQueryDataKey).block =
      (replacedFragment && replacedFragment.blockify()) || new MathBlock;

    self.firstChild.parent = self;
    self.firstChild.jQ = self.jQ.append(self.firstChild.jQ);

    return;
  }
  //otherwise, the succeeding elements of html_template should be child blocks
  var newBlock, prev, num_blocks = self.html_template.length;
  this.firstChild = newBlock = prev =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;

  newBlock.parent = self;
  newBlock.jQ = $(self.html_template[1])
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(self.jQ);

  newBlock.blur();

  for (var i = 2; i < num_blocks; i += 1) {
    newBlock = new MathBlock;
    newBlock.parent = self;
    newBlock.prev = prev;
    prev.next = newBlock;
    prev = newBlock;

    newBlock.jQ = $(self.html_template[i])
      .data(jQueryDataKey, {block: newBlock})
      .appendTo(self.jQ);

    newBlock.blur();
  }
  self.lastChild = newBlock;
};
_.latex = function() {
  return this.foldChildren(this.cmd, function(latex, child) {
    return latex + '{' + (child.latex() || ' ') + '}';
  });
};
_.text_template = [''];
_.text = function() {
  var i = 0;
  return this.foldChildren(this.text_template[i], function(text, child) {
    i += 1;
    var child_text = child.text();
    if (text && this.text_template[i] === '('
        && child_text[0] === '(' && child_text.slice(-1) === ')')
      return text + child_text.slice(1, -1) + this.text_template[i];
    return text + child.text() + (this.text_template[i] || '');
  });
};
_.insertAt = function(cursor) {
  var cmd = this;

  cmd.parent = cursor.parent;
  cmd.next = cursor.next;
  cmd.prev = cursor.prev;

  if (cursor.prev)
    cursor.prev.next = cmd;
  else
    cursor.parent.firstChild = cmd;

  if (cursor.next)
    cursor.next.prev = cmd;
  else
    cursor.parent.lastChild = cmd;

  cursor.prev = cmd;

  cmd.jQ.insertBefore(cursor.jQ);

  //adjust context-sensitive spacing
  cmd.respace();
  if (cmd.next)
    cmd.next.respace();
  if (cmd.prev)
    cmd.prev.respace();

  cmd.placeCursor(cursor);

  cursor.redraw(); //this will soon be cmd.trigger('redraw')
};
_.respace = $.noop; //placeholder for context-sensitive spacing
_.placeCursor = function(cursor) {
  //append the cursor to the first empty child, or if none empty, the last one
  cursor.appendTo(this.foldChildren(this.firstChild, function(prev, child) {
    return prev.isEmpty() ? prev : child;
  }));
};
_.isEmpty = function() {
  return this.foldChildren(true, function(isEmpty, child) {
    return isEmpty && child.isEmpty();
  });
};
_.remove = function() {
  var self = this,
      prev = self.prev,
      next = self.next,
      parent = self.parent;

  if (prev)
    prev.next = next;
  else
    parent.firstChild = next;

  if (next)
    next.prev = prev;
  else
    parent.lastChild = prev;

  self.jQ.remove();

  return self;
};

/**
 * Lightweight command without blocks or children.
 */
function Symbol(cmd, html, text) {
  this.init(cmd, [ html ],
    [ text || (cmd && cmd.length > 1 ? cmd.slice(1) : cmd) ]);
}
_ = Symbol.prototype = new MathCommand;
_.initBlocks = $.noop;
_.latex = function(){ return this.cmd; };
_.text = function(){ return this.text_template; };
_.placeCursor = $.noop;
_.isEmpty = function(){ return true; };

/**
 * Children and parent of MathCommand's. Basically partitions all the
 * symbols and operators that descend (in the Math DOM tree) from
 * ancestor operators.
 */
function MathBlock(){}
_ = MathBlock.prototype = new MathElement;
_.latex = function() {
  return this.foldChildren('', function(latex, child) {
    return latex + child.latex();
  });
};
_.text = function() {
  return this.firstChild === this.lastChild ?
    this.firstChild.text() :
    this.foldChildren('(', function(text, child) {
      return text + child.text();
    }) + ')';
};
_.isEmpty = function() {
  return this.firstChild === 0 && this.lastChild === 0;
};
_.focus = function() {
  this.jQ.addClass('hasCursor');
  if (this.isEmpty())
    this.jQ.removeClass('empty');

  return this;
};
_.blur = function() {
  this.jQ.removeClass('hasCursor');
  if (this.isEmpty())
    this.jQ.addClass('empty');

  return this;
};

/**
 * An entity outside the Math DOM tree with one-way pointers (so it's only
 * a "view" of part of the tree, not an actual node/entity in the tree)
 * that delimit a list of symbols and operators.
 */
function MathFragment(parent, prev, next) {
  if (!arguments.length) return;

  var self = this;

  self.parent = parent;
  self.prev = prev || 0; //so you can do 'new MathFragment(block)' without
  self.next = next || 0; //ending up with this.prev or this.next === undefined

  self.jQinit(self.fold($(), function(jQ, child){ return child.jQ.add(jQ); }));
}
_ = MathFragment.prototype;
_.remove = MathCommand.prototype.remove;
_.jQinit = function(children) {
  this.jQ = children;
};
_.each = function(fn) {
  for (var el = this.prev.next || this.parent.firstChild; el !== this.next; el = el.next)
    if (fn.call(this, el) === false) break;

  return this;
};
_.fold = function(fold, fn) {
  this.each(function(el) {
    fold = fn.call(this, fold, el);
  });
  return fold;
};
_.latex = function() {
  return this.fold('', function(latex, el){ return latex + el.latex(); });
};
_.blockify = function() {
  var self = this,
      prev = self.prev,
      next = self.next,
      parent = self.parent,
      newBlock = new MathBlock,
      newFirstChild = newBlock.firstChild = prev.next || parent.firstChild,
      newLastChild = newBlock.lastChild = next.prev || parent.lastChild;

  if (prev)
    prev.next = next;
  else
    parent.firstChild = next;

  if (next)
    next.prev = prev;
  else
    parent.lastChild = prev;

  newFirstChild.prev = self.prev = 0;
  newLastChild.next = self.next = 0;

  self.parent = newBlock;
  self.each(function(el){ el.parent = newBlock; });

  newBlock.jQ = self.jQ;

  return newBlock;
};

/*********************************************
 * Root math elements with event delegation.
 ********************************************/

function createRoot(jQ, root, textbox, editable) {
  var contents = jQ.contents().detach();

  if (!textbox) {
    jQ.addClass('mathquill-rendered-math');
  }

  root.jQ = jQ.data(jQueryDataKey, {
    block: root,
    revert: function() {
      jQ.empty().unbind('.mathquill')
        .removeClass('mathquill-rendered-math mathquill-editable mathquill-textbox')
        .append(contents);
    }
  });

  var cursor = root.cursor = new Cursor(root);

  root.renderLatex(contents.text());

  //textarea stuff
  var textareaSpan = root.textarea = $('<span class="textarea"><textarea></textarea></span>'),
    textarea = textareaSpan.children();

  /******
   * TODO [Han]: Document this
   */
  var textareaSelectionTimeout;
  root.selectionChanged = function() {
    if (textareaSelectionTimeout === undefined) {
      textareaSelectionTimeout = setTimeout(setTextareaSelection);
    }
    forceIERedraw(jQ[0]);
  };
  function setTextareaSelection() {
    textareaSelectionTimeout = undefined;
    var latex = cursor.selection ? '$'+cursor.selection.latex()+'$' : '';
    textarea.val(latex);
    if (latex) {
      textarea[0].select();
    }
  }

  //prevent native selection except textarea
  jQ.bind('selectstart.mathquill', function(e) {
    if (e.target !== textarea[0]) e.preventDefault();
    e.stopPropagation();
  });

  //drag-to-select event handling
  var anticursor, blink = cursor.blink;
  jQ.bind('mousedown.mathquill', function(e) {
    function mousemove(e) {
      cursor.seek($(e.target), e.pageX, e.pageY);

      if (cursor.prev !== anticursor.prev
          || cursor.parent !== anticursor.parent) {
        cursor.selectFrom(anticursor);
      }

      return false;
    }

    // docmousemove is attached to the document, so that
    // selection still works when the mouse leaves the window.
    function docmousemove(e) {
      // [Han]: i delete the target because of the way seek works.
      // it will not move the mouse to the target, but will instead
      // just seek those X and Y coordinates.  If there is a target,
      // it will try to move the cursor to document, which will not work.
      // cursor.seek needs to be refactored.
      delete e.target;

      return mousemove(e);
    }

    function mouseup(e) {
      anticursor = undefined;
      cursor.blink = blink;
      if (!cursor.selection) {
        if (editable) {
          cursor.show();
        }
        else {
          textareaSpan.detach();
        }
      }

      // delete the mouse handlers now that we're not dragging anymore
      jQ.unbind('mousemove', mousemove);
      $(document).unbind('mousemove', docmousemove).unbind('mouseup', mouseup);
    }

    setTimeout(function() { textarea.focus(); });
      // preventDefault won't prevent focus on mousedown in IE<9
      // that means immediately after this mousedown, whatever was
      // mousedown-ed will receive focus
      // http://bugs.jquery.com/ticket/10345

    cursor.blink = $.noop;
    cursor.seek($(e.target), e.pageX, e.pageY);

    anticursor = new MathFragment(cursor.parent, cursor.prev, cursor.next);

    if (!editable) jQ.prepend(textareaSpan);

    jQ.mousemove(mousemove);
    $(document).mousemove(docmousemove).mouseup(mouseup);

    // E-math: added by Pesasa to make sure we get focusout.
    jQuery(document.activeElement).parents('.mathquill-editable').focusout().blur();
    return false;
  });

  // Insert to DOM before binding events to avoid memory leaks.
  jQ.prepend(textareaSpan);

  if (!editable) {
    jQ.bind('cut paste', false).bind('copy', setTextareaSelection)
      .prepend('<span class="selectable">$'+root.latex()+'$</span>');
    textarea.blur(function() {
      cursor.clearSelection();
      setTimeout(detach); //detaching during blur explodes in WebKit
    });
    function detach() {
      textareaSpan.detach();
    }
    return;
  }

  //root CSS classes
  jQ.addClass('mathquill-editable');
  if (textbox)
    jQ.addClass('mathquill-textbox');

  //focus and blur handling
  textarea.focus(function(e) {
    if (!cursor.parent)
      cursor.appendTo(root);
    cursor.parent.jQ.addClass('hasCursor');
    if (cursor.selection) {
      cursor.selection.jQ.removeClass('blur');
      setTimeout(root.selectionChanged); //select textarea after focus
    }
    else
      cursor.show();
    e.stopPropagation();
  }).blur(function(e) {
    cursor.hide().parent.blur();
    if (cursor.selection)
      cursor.selection.jQ.addClass('blur');
    e.stopPropagation();
  });

  jQ.bind('focus.mathquill blur.mathquill', function(e) {
    textarea.trigger(e);
  }).blur();

  //clipboard event handling
  jQ.bind('cut', function(e) {
    setTextareaSelection();

    if (cursor.selection) {
      setTimeout(function() {
        cursor.deleteSelection();
        cursor.redraw();
      });
    }

    e.stopPropagation();
  })
  .bind('copy', function(e) {
    setTextareaSelection();
    e.stopPropagation();
  })
  .bind('paste', function(e) {
    pasting = true;
    setTimeout(paste);
    e.stopPropagation();
  });

  function paste() {
    //FIXME HACK the parser in RootTextBlock needs to be moved to
    //Cursor::writeLatex or something so this'll work with MathQuill textboxes
    var latex = textarea.val();
    if (latex.slice(0,1) === '$' && latex.slice(-1) === '$') {
      latex = latex.slice(1, -1);
    }
    else {
      latex = '\\text{' + latex + '}';
    }
    cursor.writeLatex(latex).show();
    textarea.val('');
    pasting = false;
  }

  //keyboard events and text input, see Wiki page "Keyboard Events"
  var lastKeydn, lastKeydnHappened, lastKeypressWhich, pasting = false;
  jQ.bind('keydown.mathquill', function(e) {
    lastKeydn = e;
    lastKeydnHappened = true;
    if (cursor.parent.keydown(e) === false)
      e.preventDefault();
  }).bind('keypress.mathquill', function(e) {
    if (lastKeydnHappened)
      lastKeydnHappened = false;
    else {
      //there's two ways keypress might be triggered without a keydown happening first:
      if (lastKeypressWhich !== e.which)
        //all browsers do that if this textarea is given focus during the keydown of
        //a different focusable element, i.e. by that element's keydown event handler.
        //No way of knowing original keydown, so ignore this keypress
        return;
      else
        //some browsers do that when auto-repeating key events, replay the keydown
        cursor.parent.keydown(lastKeydn);
    }
    lastKeypressWhich = e.which;

    //make sure setTextareaSelection() doesn't happen before textInput(), where we
    //check if any text was typed
    if (textareaSelectionTimeout !== undefined)
      clearTimeout(textareaSelectionTimeout);

    //after keypress event, trigger virtual textInput event if text was
    //input to textarea
    setTimeout(textInput);
  });

  function textInput() {
    if (pasting || (
      'selectionStart' in textarea[0]
      && textarea[0].selectionStart !== textarea[0].selectionEnd
    )) return;
    var text = textarea.val();
    if (text) {
      textarea.val('');
      for (var i = 0; i < text.length; i += 1) {
        cursor.parent.textInput(text.charAt(i));
      }
      textareaSelectionTimeout = undefined;
    }
    else {
      if (textareaSelectionTimeout !== undefined)
        setTextareaSelection();
    }
  }
}

function RootMathBlock(){}
_ = RootMathBlock.prototype = new MathBlock;
_.latex = function() {
  return MathBlock.prototype.latex.call(this).replace(/(\\[a-z]+) (?![a-z])/ig,'$1');
};
_.text = function() {
  return this.foldChildren('', function(text, child) {
    return text + child.text();
  });
};
_.renderLatex = function(latex) {
  var jQ = this.jQ;

  jQ.children().slice(1).remove();
  this.firstChild = this.lastChild = 0;

  // temporarily take the element out of the displayed DOM
  // while we add stuff to it.  Grab the next or parent node
  // so we know where to put it back.
  // NOTE: careful that it be the next or parent DOM **node** and not
  // HTML element, lest we skip a text node.
  var next = jQ[0].nextSibling,
    parent = jQ[0].parentNode;

  jQ.detach();
  this.cursor.appendTo(this).writeLatex(latex);

  // Put. the element. back.
  // if there's no next element, it's at the end of its parent
  next ? jQ.insertBefore(next) : jQ.appendTo(parent);

  // XXX HACK ALERT
  this.jQ.mathquill('redraw');
  this.blur();
};
_.keydown = function(e)
{
  e.ctrlKey = e.ctrlKey || e.metaKey;
  switch ((e.originalEvent && e.originalEvent.keyIdentifier) || e.which) {
  case 8: //backspace
  case 'Backspace':
  case 'U+0008':
    if (e.ctrlKey)
      while (this.cursor.prev || this.cursor.selection)
        this.cursor.backspace();
    else
      this.cursor.backspace();
    break;
  case 27: //may as well be the same as tab until we figure out what to do with it
  case 'Esc':
  case 'U+001B':
  case 9: //tab
  case 'Tab':
  case 'U+0009':
    if (e.ctrlKey) break;

    var parent = this.cursor.parent;
    if (e.shiftKey) { //shift+Tab = go one block left if it exists, else escape left.
      if (parent === this.cursor.root) //cursor is in root editable, continue default
        return this.skipTextInput = true;
      else if (parent.prev) //go one block left
        this.cursor.appendTo(parent.prev);
      else //get out of the block
        this.cursor.insertBefore(parent.parent);
    }
    else { //plain Tab = go one block right if it exists, else escape right.
      if (parent === this.cursor.root) //cursor is in root editable, continue default
        return this.skipTextInput = true;
      else if (parent.next) //go one block right
        this.cursor.prependTo(parent.next);
      else //get out of the block
        this.cursor.insertAfter(parent.parent);
    }

    this.cursor.clearSelection();
    break;
  case 13: //enter
  case 'Enter':
    break;
  case 35: //end
  case 'End':
    if (e.shiftKey)
      while (this.cursor.next || (e.ctrlKey && this.cursor.parent !== this))
        this.cursor.selectRight();
    else //move to the end of the root block or the current block.
      this.cursor.clearSelection().appendTo(e.ctrlKey ? this : this.cursor.parent);
    break;
  case 36: //home
  case 'Home':
    if (e.shiftKey)
      while (this.cursor.prev || (e.ctrlKey && this.cursor.parent !== this))
        this.cursor.selectLeft();
    else //move to the start of the root block or the current block.
      this.cursor.clearSelection().prependTo(e.ctrlKey ? this : this.cursor.parent);
    break;
  case 37: //left
  case 'Left':
    if (e.ctrlKey) break;

    if (e.shiftKey)
      this.cursor.selectLeft();
    else
      this.cursor.moveLeft();
    break;
  case 38: //up
  case 'Up':
    if (e.ctrlKey) break;

    if (e.shiftKey) {
      if (this.cursor.prev)
        while (this.cursor.prev)
          this.cursor.selectLeft();
      else
        this.cursor.selectLeft();
    }
    else if (this.cursor.parent.prev)
      this.cursor.clearSelection().appendTo(this.cursor.parent.prev);
    else if (this.cursor.prev)
      this.cursor.clearSelection().prependTo(this.cursor.parent);
    else if (this.cursor.parent !== this)
      this.cursor.clearSelection().insertBefore(this.cursor.parent.parent);
    break;
  case 39: //right
  case 'Right':
    if (e.ctrlKey) break;

    if (e.shiftKey)
      this.cursor.selectRight();
    else
      this.cursor.moveRight();
    break;
  case 40: //down
  case 'Down':
    if (e.ctrlKey) break;

    if (e.shiftKey) {
      if (this.cursor.next)
        while (this.cursor.next)
          this.cursor.selectRight();
      else
        this.cursor.selectRight();
    }
    else if (this.cursor.parent.next)
      this.cursor.clearSelection().prependTo(this.cursor.parent.next);
    else if (this.cursor.next)
      this.cursor.clearSelection().appendTo(this.cursor.parent);
    else if (this.cursor.parent !== this)
      this.cursor.clearSelection().insertAfter(this.cursor.parent.parent);
    break;
  case 46: //delete
  case 'Del':
  case 'U+007F':
    if (e.ctrlKey)
      while (this.cursor.next || this.cursor.selection)
        this.cursor.deleteForward();
    else
      this.cursor.deleteForward();
    break;
  case 65: //the 'A' key, as in Ctrl+A Select All
  case 'A':
  case 'U+0041':
    if (e.ctrlKey && !e.shiftKey && !e.altKey) {
      if (this !== this.cursor.root) //so not stopPropagation'd at RootMathCommand
        return this.parent.keydown(e);

      this.cursor.clearSelection().appendTo(this);
      while (this.cursor.prev)
        this.cursor.selectLeft();
      break;
    }
  default:
    this.skipTextInput = false;
    return true;
  }
  this.skipTextInput = true;
  return false;
};
_.textInput = function(ch) {
  if (!this.skipTextInput)
    this.cursor.write(ch);
};

function RootMathCommand(cursor) {
  this.init('$');
  this.firstChild.cursor = cursor;
  this.firstChild.textInput = function(ch) {
    if (this.skipTextInput) return;

    if (ch !== '$' || cursor.parent !== this)
      cursor.write(ch);
    else if (this.isEmpty()) {
      cursor.insertAfter(this.parent).backspace()
        .insertNew(new VanillaSymbol('\\$','$')).show();
    }
    else if (!cursor.next)
      cursor.insertAfter(this.parent);
    else if (!cursor.prev)
      cursor.insertBefore(this.parent);
    else
      cursor.write(ch);
  };
}
_ = RootMathCommand.prototype = new MathCommand;
_.html_template = ['<span class="mathquill-rendered-math"></span>'];
_.initBlocks = function() {
  this.firstChild =
  this.lastChild =
  this.jQ.data(jQueryDataKey).block =
    new RootMathBlock;

  this.firstChild.parent = this;
  this.firstChild.jQ = this.jQ;
};
_.latex = function() {
  return '$' + this.firstChild.latex() + '$';
};

function RootTextBlock(){}
_ = RootTextBlock.prototype = new MathBlock;
_.renderLatex = function(latex) {
  var self = this, cursor = self.cursor;
  self.jQ.children().slice(1).remove();
  self.firstChild = self.lastChild = 0;
  cursor.show().appendTo(self);

  latex = latex.match(/(?:\\\$|[^$])+|\$(?:\\\$|[^$])*\$|\$(?:\\\$|[^$])*$/g) || '';
  for (var i = 0; i < latex.length; i += 1) {
    var chunk = latex[i];
    if (chunk[0] === '$') {
      if (chunk[-1+chunk.length] === '$' && chunk[-2+chunk.length] !== '\\')
        chunk = chunk.slice(1, -1);
      else
        chunk = chunk.slice(1);

      var root = new RootMathCommand(cursor);
      cursor.insertNew(root);
      root.firstChild.renderLatex(chunk);
      cursor.show().insertAfter(root);
    }
    else {
      for (var j = 0; j < chunk.length; j += 1)
        this.cursor.insertNew(new VanillaSymbol(chunk[j]));
    }
  }
};
_.keydown = RootMathBlock.prototype.keydown;
_.textInput = function(ch) {
  if (this.skipTextInput) return;

  this.cursor.deleteSelection();
  if (ch === '$') {
    this.cursor.insertNew(new RootMathCommand(this.cursor));
  } else {
    this.cursor.insertNew(new VanillaSymbol(ch));
  }
};

/***************************
 * Commands and Operators.
 **************************/

var CharCmds = {}, LatexCmds = {}; //single character commands, LaTeX commands

var scale, // = function(jQ, x, y) { ... }
//will use a CSS 2D transform to scale the jQuery-wrapped HTML elements,
//or the filter matrix transform fallback for IE 5.5-8, or gracefully degrade to
//increasing the fontSize to match the vertical Y scaling factor.

//ideas from http://github.com/louisremi/jquery.transform.js
//see also http://msdn.microsoft.com/en-us/library/ms533014(v=vs.85).aspx

  forceIERedraw = $.noop,
  div = document.createElement('div'),
  div_style = div.style,
  transformPropNames = {
    transform:1,
    WebkitTransform:1,
    MozTransform:1,
    OTransform:1,
    msTransform:1
  },
  transformPropName;

for (var prop in transformPropNames) {
  if (prop in div_style) {
    transformPropName = prop;
    break;
  }
}

if (transformPropName) {
  scale = function(jQ, x, y) {
    jQ.css(transformPropName, 'scale('+x+','+y+')');
  };
}
else if ('filter' in div_style) { //IE 6, 7, & 8 fallback, see https://github.com/laughinghan/mathquill/wiki/Transforms
  forceIERedraw = function(el){ el.className = el.className; };
  scale = function(jQ, x, y) { //NOTE: assumes y > x
    x /= (1+(y-1)/2);
    jQ.addClass('matrixed').css({
      fontSize: y + 'em',
      marginTop: '-.1em',
      filter: 'progid:DXImageTransform.Microsoft'
        + '.Matrix(M11=' + x + ",SizingMethod='auto expand')"
    });
    function calculateMarginRight() {
      jQ.css('marginRight', (1+jQ.width())*(x-1)/x + 'px');
    }
    calculateMarginRight();
    var intervalId = setInterval(calculateMarginRight);
    $(window).load(function() {
      clearTimeout(intervalId);
      calculateMarginRight();
    });
  };
}
else {
  scale = function(jQ, x, y) {
    jQ.css('fontSize', y + 'em');
  };
}

function proto(parent, child) { //shorthand for prototyping
  child.prototype = parent.prototype;
  return child;
}

function bind(cons) { //shorthand for binding arguments to constructor
  var args = Array.prototype.slice.call(arguments, 1);

  return proto(cons, function() {
    cons.apply(this, Array.prototype.concat.apply(args, arguments));
  });
}

//because I miss the <font> tag
//(that's a joke, I hate this, it's like actively *fighting*
// separation of presentation and content and everything HTML and CSS
// are about, but it's an intrinsic problem with WYSIWYG)
//TODO: WYSIWYM?
function Style(cmd, html_template, replacedFragment) {
  this.init(cmd, [ html_template ], undefined, replacedFragment);
}
proto(MathCommand, Style);
//fonts
LatexCmds.mathrm = bind(Style, '\\mathrm', '<span class="roman font"></span>');
LatexCmds.mathit = bind(Style, '\\mathit', '<i class="font"></i>');
LatexCmds.mathbf = bind(Style, '\\mathbf', '<b class="font"></b>');
LatexCmds.mathsf = bind(Style, '\\mathsf', '<span class="sans-serif font"></span>');
LatexCmds.mathtt = bind(Style, '\\mathtt', '<span class="monospace font"></span>');
// Added by pesasa/pekasa for e-math
LatexCmds.unit = bind(Style, '\\unit', '<span style="margin-left: 0.2em;" class="roman font unit"></span>');
LatexCmds.solution = bind(Style, '\\solution', '<span class="solution"></span>');
LatexCmds.extramot = bind(Style, '\\extramot', '<span class="extramotivation"></span>');
//text-decoration
LatexCmds.underline = bind(Style, '\\underline', '<span class="underline"></span>');
LatexCmds.overline = LatexCmds.bar = bind(Style, '\\overline', '<span class="overline"></span>');

// Colors for E-math by pesasa
LatexCmds.red = bind(Style, '\\red', '<span class="color_red"></span>');
LatexCmds.blue = bind(Style, '\\blue', '<span class="color_blue"></span>');
LatexCmds.green = bind(Style, '\\green', '<span class="color_green"></span>');
LatexCmds.violet = bind(Style, '\\violet', '<span class="color_violet"></span>');
LatexCmds.orange = bind(Style, '\\orange', '<span class="color_orange"></span>');


function SupSub(cmd, html, text, replacedFragment) {
  this.init(cmd, [ html ], [ text ], replacedFragment);
}
_ = SupSub.prototype = new MathCommand;
_.latex = function() {
  var latex = this.firstChild.latex();
  if (latex.length === 1)
    return this.cmd + latex;
  else
    return this.cmd + '{' + (latex || ' ') + '}';
};
_.redraw = function() {
  if (this.prev)
    this.prev.respace();
  //SupSub::respace recursively calls respace on all the following SupSubs
  //so if prev is a SupSub, no need to call respace on this or following nodes
  if (!(this.prev instanceof SupSub)) {
    this.respace();
    //and if next is a SupSub, then this.respace() will have already called
    //this.next.respace()
    if (this.next && !(this.next instanceof SupSub))
      this.next.respace();
  }
};
_.respace = function() {
  if (
    this.prev.cmd === '\\int ' || (
      this.prev instanceof SupSub && this.prev.cmd != this.cmd
      && this.prev.prev && this.prev.prev.cmd === '\\int '
    )
  ) {
    if (!this.limit) {
      this.limit = true;
      this.jQ.addClass('limit');
    }
  }
  else {
    if (this.limit) {
      this.limit = false;
      this.jQ.removeClass('limit');
    }
  }

  this.respaced = this.prev instanceof SupSub && this.prev.cmd != this.cmd && !this.prev.respaced;
  if (this.respaced) {
    var fontSize = +this.jQ.css('fontSize').slice(0,-2),
      prevWidth = this.prev.jQ.outerWidth()
      thisWidth = this.jQ.outerWidth();
    this.jQ.css({
      left: (this.limit && this.cmd === '_' ? -.25 : 0) - prevWidth/fontSize + 'em',
      marginRight: .1 - min(thisWidth, prevWidth)/fontSize + 'em'
        //1px extra so it doesn't wrap in retarded browsers (Firefox 2, I think)
    });
  }
  else if (this.limit && this.cmd === '_') {
    this.jQ.css({
      left: '-.25em',
      marginRight: ''
    });
  }
  else {
    this.jQ.css({
      left: '',
      marginRight: ''
    });
  }

  if (this.next instanceof SupSub)
    this.next.respace();

  return this;
};

LatexCmds.subscript = LatexCmds._ = proto(SupSub, function(replacedFragment) {
  SupSub.call(this, '_', '<sub></sub>', '_', replacedFragment);
});

LatexCmds.superscript =
LatexCmds.supscript =
LatexCmds['^'] = proto(SupSub, function(replacedFragment) {
  SupSub.call(this, '^', '<sup></sup>', '**', replacedFragment);
});

function Fraction(replacedFragment) {
  this.init('\\frac', undefined, undefined, replacedFragment);
  this.jQ.append('<span style="display:inline-block;width:0">&nbsp;</span>');
}
_ = Fraction.prototype = new MathCommand;
_.html_template = [
  '<span class="fraction"></span>',
  '<span class="numerator"></span>',
  '<span class="denominator"></span>'
];
_.text_template = ['(', '/', ')'];

LatexCmds.frac = LatexCmds.dfrac = LatexCmds.cfrac = LatexCmds.fraction = Fraction;

function LiveFraction() {
  Fraction.apply(this, arguments);
}
_ = LiveFraction.prototype = new Fraction;
_.placeCursor = function(cursor) { //TODO: better architecture so this can be done more cleanly, highjacking MathCommand::placeCursor doesn't seem like the right place to do this
  if (this.firstChild.isEmpty()) {
    var prev = this.prev;
    while (prev &&
      !(
        prev instanceof BinaryOperator ||
        prev instanceof TextBlock ||
        prev instanceof BigSymbol
      ) //lookbehind for operator
    )
      prev = prev.prev;

    if (prev instanceof BigSymbol && prev.next instanceof SupSub) {
      prev = prev.next;
      if (prev.next instanceof SupSub && prev.next.cmd != prev.cmd)
        prev = prev.next;
    }

    if (prev !== this.prev) { //FIXME: major Law of Demeter violation, shouldn't know here that MathCommand::initBlocks does some initialization that MathFragment::blockify doesn't
      var newBlock = new MathFragment(this.parent, prev, this).blockify();
      newBlock.jQ = this.firstChild.jQ.empty().removeClass('empty').append(newBlock.jQ).data(jQueryDataKey, { block: newBlock });
      newBlock.next = this.lastChild;
      newBlock.parent = this;
      this.firstChild = this.lastChild.prev = newBlock;
    }
  }
  cursor.appendTo(this.lastChild);
};

LatexCmds.over = CharCmds['/'] = LiveFraction;

function SquareRoot(replacedFragment) {
  this.init('\\sqrt', undefined, undefined, replacedFragment);
}
_ = SquareRoot.prototype = new MathCommand;
_.html_template = [
  '<span class="block"><span class="sqrt-prefix">&radic;</span></span>',
  '<span class="sqrt-stem"></span>'
];
_.text_template = ['sqrt(', ')'];
_.redraw = function() {
  var block = this.lastChild.jQ;
  scale(block.prev(), 1, block.innerHeight()/+block.css('fontSize').slice(0,-2) - .1);
};
_.optional_arg_command = 'nthroot';

LatexCmds.sqrt = LatexCmds['√'] = SquareRoot;

function NthRoot(replacedFragment) {
  SquareRoot.call(this, replacedFragment);
  this.jQ = this.firstChild.jQ.detach().add(this.jQ);
}
_ = NthRoot.prototype = new SquareRoot;
_.html_template = [
  '<span class="block"><span class="sqrt-prefix">&radic;</span></span>',
  '<sup class="nthroot"></sup>',
  '<span class="sqrt-stem"></span>'
];
_.text_template = ['sqrt[', '](', ')'];
_.latex = function() {
  return '\\sqrt['+this.firstChild.latex()+']{'+this.lastChild.latex()+'}';
};

LatexCmds.nthroot = NthRoot;

// Round/Square/Curly/Angle Brackets (aka Parens/Brackets/Braces)
function Bracket(open, close, cmd, end, replacedFragment) {
  this.init('\\left'+cmd,
    ['<span class="block"><span class="paren">'+open+'</span><span class="block"></span><span class="paren">'+close+'</span></span>'],
    [open, close],
    replacedFragment);
  this.end = '\\right'+end;
}
_ = Bracket.prototype = new MathCommand;
_.initBlocks = function(replacedFragment) { //FIXME: possible Law of Demeter violation, hardcore MathCommand::initBlocks knowledge needed here
  this.firstChild = this.lastChild =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  this.firstChild.parent = this;
  this.firstChild.jQ = this.jQ.children(':eq(1)')
    .data(jQueryDataKey, {block: this.firstChild})
    .append(this.firstChild.jQ);

  var block = this.blockjQ = this.firstChild.jQ;
  this.bracketjQs = block.prev().add(block.next());
};
_.latex = function() {
  return this.cmd + this.firstChild.latex() + this.end;
};
_.redraw = function() {
  var height = this.blockjQ.outerHeight()/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bracketjQs, min(1 + .2*(height - 1), 1.2), 1.05*height);
};

LatexCmds.lbrace = CharCmds['{'] = proto(Bracket, function(replacedFragment) {
  Bracket.call(this, '{', '}', '\\{', '\\}', replacedFragment);
});
LatexCmds.langle = LatexCmds.lang = proto(Bracket, function(replacedFragment) {
  Bracket.call(this,'&lang;','&rang;','\\langle ','\\rangle ',replacedFragment);
});

// Closing bracket matching opening bracket above
function CloseBracket(open, close, cmd, end, replacedFragment) {
  Bracket.apply(this, arguments);
}
_ = CloseBracket.prototype = new Bracket;
_.placeCursor = function(cursor) {
  //if I'm at the end of my parent who is a matching open-paren, and I was not passed
  //  a selection fragment, get rid of me and put cursor after my parent
  if (!this.next && this.parent.parent && this.parent.parent.end === this.end && this.firstChild.isEmpty())
    cursor.backspace().insertAfter(this.parent.parent);
  else {
    this.firstChild.blur();
    this.redraw();
  }
};

LatexCmds.rbrace = CharCmds['}'] = proto(CloseBracket, function(replacedFragment) {
  CloseBracket.call(this, '{','}','\\{','\\}',replacedFragment);
});
LatexCmds.rangle = LatexCmds.rang = proto(CloseBracket, function(replacedFragment) {
  CloseBracket.call(this,'&lang;','&rang;','\\langle ','\\rangle ',replacedFragment);
});

function Paren(open, close, replacedFragment) {
  Bracket.call(this, open, close, open, close, replacedFragment);
}
Paren.prototype = Bracket.prototype;

LatexCmds.lparen = CharCmds['('] = proto(Paren, function(replacedFragment) {
  Paren.call(this, '(', ')', replacedFragment);
});
LatexCmds.lbrack = LatexCmds.lbracket = CharCmds['['] = proto(Paren, function(replacedFragment) {
  Paren.call(this, '[', ']', replacedFragment);
});

function CloseParen(open, close, replacedFragment) {
  CloseBracket.call(this, open, close, open, close, replacedFragment);
}
CloseParen.prototype = CloseBracket.prototype;

LatexCmds.rparen = CharCmds[')'] = proto(CloseParen, function(replacedFragment) {
  CloseParen.call(this, '(', ')', replacedFragment);
});
LatexCmds.rbrack = LatexCmds.rbracket = CharCmds[']'] = proto(CloseParen, function(replacedFragment) {
  CloseParen.call(this, '[', ']', replacedFragment);
});

function Pipes(replacedFragment) {
  Paren.call(this, '|', '|', replacedFragment);
}
_ = Pipes.prototype = new Paren;
_.placeCursor = function(cursor) {
  if (!this.next && this.parent.parent && this.parent.parent.end === this.end && this.firstChild.isEmpty())
    cursor.backspace().insertAfter(this.parent.parent);
  else
    cursor.appendTo(this.firstChild);
};

_.latex = function() {
  return '\\abs{' + this.firstChild.latex() + '}';
};


LatexCmds.lpipe = LatexCmds.rpipe = CharCmds['|'] = Pipes;
//LatexCmds.lpipe = LatexCmds.rpipe = Pipes;

/***
 * Abs and Norm functions for E-Math
 ***/

function Abs(replacedFragment){
  this.init('\\abs', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">|</span>').prependTo(this.jQ)
      .add($('<span class="paren">|</span>').appendTo(this.jQ));
}
_ = Abs.prototype = new MathCommand;
_.html_template = [
  '<span class="block"></span>',
  '<span class="abs"></span>'
];
_.text_template = ['abs(',')'];
_.redraw = Bracket.prototype.redraw;
//LatexCmds.abs = CharCmds['|'] = Abs;
LatexCmds.abs = CharCmds['\u00a6'] = Abs; // Use ¦ for shortcut

function Norm(replacedFragment){
  this.init('\\norm', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">||</span>').prependTo(this.jQ)
      .add($('<span class="paren">||</span>').appendTo(this.jQ));
}
_ = Norm.prototype = new MathCommand;
_.html_template = [
  '<span class="block"></span>',
  '<span class="norm"></span>'
];
_.text_template = ['norm(',')'];
_.redraw = Bracket.prototype.redraw;
LatexCmds.norm = Norm;

/***
 * Open and half open intervals for E-Math
 ***/

function OpenBoth(replacedFragment){
  this.init('\\openBoth', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">]</span>').prependTo(this.jQ)
      .add($('<span class="paren">[</span>').appendTo(this.jQ));
}
_ = OpenBoth.prototype = new MathCommand;
_.html_template = [
  '<span class="block intervalblock"></span>'
];
_.text_template = ['openBoth(',',',')'];
_.initBlocks = function(replacedFragment){
  var newBlock, first, second;
  this.firstChild = newBlock = first =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var interval = this.jQ.append('<span class="' + this.cmd.substr(1) + ' interval block"></span>').find('.interval');
  newBlock.jQ = $('<span class="intervalelement"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(interval);
  interval.append('<span class="intervalseparator">,</span>');
  first.next = newBlock = second = new MathBlock;
  newBlock.jQ = $('<span class="intervalelement"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(interval);
  second.prev = first;
  this.lastChild = second;
  first.blur();
  second.blur();
  first.parent = second.parent = this;
}
_.redraw = Bracket.prototype.redraw;
LatexCmds.openBoth = OpenBoth;

function Closedinterval(replacedFragment){
  this.init('\\closed', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">[</span>').prependTo(this.jQ)
      .add($('<span class="paren">]</span>').appendTo(this.jQ));
}
_ = Closedinterval.prototype = new OpenBoth;
_.text_template = ['closed(',',',')'];
LatexCmds.closed = Closedinterval;

function OpenLeft(replacedFragment){
  this.init('\\openLeft', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">]</span>').prependTo(this.jQ)
      .add($('<span class="paren">]</span>').appendTo(this.jQ));
}
_ = OpenLeft.prototype = new OpenBoth;
_.text_template = ['openLeft(',',',')'];
LatexCmds.openLeft = OpenLeft;

function OpenRight(replacedFragment){
  this.init('\\openRight', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">[</span>').prependTo(this.jQ)
      .add($('<span class="paren">[</span>').appendTo(this.jQ));
}
_ = OpenRight.prototype = new OpenBoth;
_.text_template = ['openRight(',',',')'];
LatexCmds.openRight = OpenRight;

function Openleft(replacedFragment){
  this.init('\\openleft', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">(</span>').prependTo(this.jQ)
      .add($('<span class="paren">]</span>').appendTo(this.jQ));
}
_ = Openleft.prototype = new OpenBoth;
_.text_template = ['openleft(',',',')'];
LatexCmds.openleft = Openleft;

function Openright(replacedFragment){
  this.init('\\openright', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">[</span>').prependTo(this.jQ)
      .add($('<span class="paren">)</span>').appendTo(this.jQ));
}
_ = Openright.prototype = new OpenBoth;
_.text_template = ['openright(',',',')'];
LatexCmds.openright = Openright;

function Openboth(replacedFragment){
  this.init('\\openboth', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">(</span>').prependTo(this.jQ)
      .add($('<span class="paren">)</span>').appendTo(this.jQ));
}
_ = Openboth.prototype = new OpenBoth;
_.text_template = ['openboth(',',',')'];
LatexCmds.openboth = Openboth;

/***
 * Integral for E-Math
 ***/

function Integral(replacedFragment){
  this.init('\\Int', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children().eq(1);
}
_ = Integral.prototype = new MathCommand;
_.html_template = [
  '<span class="block integralblock"></span>'
];
_.text_template = ['Integral(',',',',',')'];
_.initBlocks = function(replacedFragment){
  this.blockjQ = this.jQ.children();
  console.log(this.blockjQ);
  this.bracketjQs =
    $('<span class="bigoperatorstack"><span><span class="bigoperator">&int;</span></span></span>').prependTo(this.jQ);
  this.bigoperatorjQ = this.bracketjQs.find('.bigoperator');
  var newBlock, intfrom, intto, integrand, intdx;
  intfrom = new MathBlock;
  intto = new MathBlock;
  intdx = new MathBlock;
  newBlock = integrand =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var intblock = this.jQ.append('<span class="' + this.cmd.substr(1) + ' block"></span>').find('span:last');
  intfrom.jQ = $('<span class="integralfrom limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(intfrom.jQ)
    .appendTo(this.bracketjQs);
  intto.jQ = $('<span class="integralto limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(intto.jQ)
    .prependTo(this.bracketjQs);
  newBlock.jQ = $('<span class="integral"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(intblock);
  intdx.jQ = $('<span class="integraldifferential"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(intdx.jQ)
    .appendTo(intblock);
  this.firstChild = intfrom;
  intfrom.next = intto;
  intto.prev = intfrom;
  intto.next = integrand;
  integrand.prev = intto;
  integrand.next = intdx;
  intdx.prev = integrand;
  this.lastChild = intdx;
  intfrom.blur();
  intto.blur();
  integrand.blur();
  intdx.blur();
  intfrom.parent = intto.parent = integrand.parent = intdx.parent = this;
}
_.redraw = Bracket.prototype.redraw;
_.redraw = function() {
  var outerheight = this.blockjQ.outerHeight();
  var height = outerheight/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bigoperatorjQ, min(1 + .2*(height - 1), 1.2), 0.95*height);
  this.bigoperatorjQ.css('line-height', outerheight+'px');
  this.bigoperatorjQ.parent().css('height', outerheight);
};
LatexCmds.Int = LatexCmds.Integral = Integral;

/***
 * Integral substitution: Finnish syntax for E-Math
 ***/

function IntegralSubst(replacedFragment){
  this.init('\\Intsubst', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children().eq(1);
}
_ = IntegralSubst.prototype = new MathCommand;
_.html_template = [
  '<span class="block integralblock"></span>'
];
_.text_template = ['Integralsubst(',',',')'];
_.initBlocks = function(replacedFragment){
  this.blockjQ = this.jQ.children();
  console.log(this.blockjQ);
  this.bracketjQs =
    $('<span class="bigoperatorstack"><span><span class="bigoperator">/</span></span></span>').prependTo(this.jQ);
  this.bigoperatorjQ = this.bracketjQs.find('.bigoperator');
  var newBlock, intfrom, intto, integrand, intdx;
  intfrom = new MathBlock;
  intto = new MathBlock;
  newBlock = integrand =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var intblock = this.jQ.append('<span class="' + this.cmd.substr(1) + ' block"></span>').find('span:last');
  intfrom.jQ = $('<span class="integralfrom limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(intfrom.jQ)
    .appendTo(this.bracketjQs);
  intto.jQ = $('<span class="integralto limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(intto.jQ)
    .prependTo(this.bracketjQs);
  newBlock.jQ = $('<span class="integral"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(intblock);
  this.firstChild = intfrom;
  intfrom.next = intto;
  intto.prev = intfrom;
  intto.next = integrand;
  integrand.prev = intto;
  this.lastChild = integrand;
  intfrom.blur();
  intto.blur();
  integrand.blur();
  intfrom.parent = intto.parent = integrand.parent = this;
}
_.redraw = Bracket.prototype.redraw;
_.redraw = function() {
  var outerheight = this.blockjQ.outerHeight();
  var height = outerheight/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bigoperatorjQ, min(1 + .2*(height - 1), 1.2), 0.95*height);
  this.bigoperatorjQ.css('line-height', outerheight+'px');
  this.bigoperatorjQ.parent().css('height', outerheight);
};
LatexCmds.Intsubst = LatexCmds.IntegralSubst = IntegralSubst;

/***
 * Display style sum for E-Math
 ***/

function BigSum(replacedFragment){
  this.init('\\Sum', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children().eq(1);
}
_ = BigSum.prototype = new MathCommand;
_.html_template = [
  '<span class="block opblock"></span>'
];
_.text_template = ['Sum(',',',')'];
_.initBlocks = function(replacedFragment){
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="bigoperatorstack"><span><span class="bigoperator">&Sigma;</span></span></span>').prependTo(this.jQ);
  this.bigoperatorjQ = this.bracketjQs.find('.bigoperator');
  var newBlock, opfrom, opto, expression;
  opfrom = new MathBlock;
  opto = new MathBlock;
  newBlock = expression =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var opblock = this.jQ.append('<span class="bigopblock block"></span>').find('span:last');
  opfrom.jQ = $('<span class="opfrom limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(opfrom.jQ)
    .appendTo(this.bracketjQs);
  opto.jQ = $('<span class="opto limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(opto.jQ)
    .prependTo(this.bracketjQs);
  newBlock.jQ = $('<span class="opexpression"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(opblock);
  this.firstChild = opfrom;
  opfrom.next = opto;
  opto.prev = opfrom;
  opto.next = expression;
  expression.prev = opto;
  this.lastChild = expression;
  opfrom.blur();
  opto.blur();
  expression.blur();
  opfrom.parent = opto.parent = expression.parent = this;
}
_.redraw = Bracket.prototype.redraw;
_.redraw = function() {
  var outerheight = this.blockjQ.outerHeight();
  var height = outerheight/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bigoperatorjQ, 0.95*height, 0.95*height);
  this.bigoperatorjQ.css('line-height', outerheight+'px');
  this.bigoperatorjQ.parent().css('height', outerheight);
};
LatexCmds.Sum = LatexCmds.BigSum = BigSum;

/***
 * Display style product for E-Math
 ***/

function BigProd(replacedFragment){
  this.init('\\Prod', undefined, undefined, replacedFragment);
  this.blockjQ = this.jQ.children().eq(1);
}
_ = BigProd.prototype = new MathCommand;
_.html_template = [
  '<span class="block opblock"></span>'
];
_.text_template = ['Product(',',',')'];
_.initBlocks = function(replacedFragment){
  this.blockjQ = this.jQ.children();
  console.log(this.blockjQ);
  this.bracketjQs =
    $('<span class="bigoperatorstack"><span><span class="bigoperator">&Pi;</span></span></span>').prependTo(this.jQ);
  this.bigoperatorjQ = this.bracketjQs.find('.bigoperator');
  var newBlock, opfrom, opto, expression;
  opfrom = new MathBlock;
  opto = new MathBlock;
  newBlock = expression =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var opblock = this.jQ.append('<span class="bigopblock block"></span>').find('span:last');
  opfrom.jQ = $('<span class="opfrom limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(opfrom.jQ)
    .appendTo(this.bracketjQs);
  opto.jQ = $('<span class="opto limit"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(opto.jQ)
    .prependTo(this.bracketjQs);
  newBlock.jQ = $('<span class="opexpression"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .append(newBlock.jQ)
    .appendTo(opblock);
  this.firstChild = opfrom;
  opfrom.next = opto;
  opto.prev = opfrom;
  opto.next = expression;
  expression.prev = opto;
  this.lastChild = expression;
  opfrom.blur();
  opto.blur();
  expression.blur();
  opfrom.parent = opto.parent = expression.parent = this;
}
_.redraw = Bracket.prototype.redraw;
_.redraw = function() {
  var outerheight = this.blockjQ.outerHeight();
  var height = outerheight/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bigoperatorjQ, 0.95*height, 0.95*height);
  this.bigoperatorjQ.css('line-height', outerheight+'px');
  this.bigoperatorjQ.parent().css('height', outerheight);
};
LatexCmds.Prod = LatexCmds.BigProd = BigProd;


function TextBlock(replacedText) {
  if (replacedText instanceof MathFragment)
    this.replacedText = replacedText.remove().jQ.text();
  else if (typeof replacedText === 'string')
    this.replacedText = replacedText;

  this.init();
}
_ = TextBlock.prototype = new MathCommand;
_.cmd = '\\text';
_.html_template = ['<span class="text"></span>'];
_.text_template = ['"', '"'];
_.initBlocks = function() { //FIXME: another possible Law of Demeter violation, but this seems much cleaner, like it was supposed to be done this way
  this.firstChild =
  this.lastChild =
  this.jQ.data(jQueryDataKey).block = new InnerTextBlock;

  this.firstChild.parent = this;
  this.firstChild.jQ = this.jQ.append(this.firstChild.jQ);
};
_.placeCursor = function(cursor) { //TODO: this should be done in the constructor that's passed replacedFragment, but you need the cursor to create new characters and insert them
  (this.cursor = cursor).appendTo(this.firstChild);

  if (this.replacedText)
    for (var i = 0; i < this.replacedText.length; i += 1)
      this.write(this.replacedText.charAt(i));
};
_.write = function(ch) {
  this.cursor.insertNew(new VanillaSymbol(ch));
};
_.keydown = function(e) {
  //backspace and delete and ends of block don't unwrap
  if (!this.cursor.selection &&
    (
      (e.which === 8 && !this.cursor.prev) ||
      (e.which === 46 && !this.cursor.next)
    )
  ) {
    if (this.isEmpty())
      this.cursor.insertAfter(this);
    return false;
  }
  return this.parent.keydown(e);
};
_.textInput = function(ch) {
  this.cursor.deleteSelection();
  if (ch !== '$')
    this.write(ch);
  else if (this.isEmpty())
    this.cursor.insertAfter(this).backspace().insertNew(new VanillaSymbol('\\$','$'));
  else if (!this.cursor.next)
    this.cursor.insertAfter(this);
  else if (!this.cursor.prev)
    this.cursor.insertBefore(this);
  else { //split apart
    var next = new TextBlock(new MathFragment(this.firstChild, this.cursor.prev));
    next.placeCursor = function(cursor) { //FIXME HACK: pretend no prev so they don't get merged
      this.prev = 0;
      delete this.placeCursor;
      this.placeCursor(cursor);
    };
    next.firstChild.focus = function(){ return this; };
    this.cursor.insertAfter(this).insertNew(next);
    next.prev = this;
    this.cursor.insertBefore(next);
    delete next.firstChild.focus;
  }
};
function InnerTextBlock(){}
_ = InnerTextBlock.prototype = new MathBlock;
_.blur = function() {
  this.jQ.removeClass('hasCursor');
  if (this.isEmpty()) {
    var textblock = this.parent, cursor = textblock.cursor;
    if (cursor.parent === this)
      this.jQ.addClass('empty');
    else {
      cursor.hide();
      textblock.remove();
      if (cursor.next === textblock)
        cursor.next = textblock.next;
      else if (cursor.prev === textblock)
        cursor.prev = textblock.prev;

      cursor.show().redraw();
    }
  }
  return this;
};
_.focus = function() {
  MathBlock.prototype.focus.call(this);

  var textblock = this.parent;
  if (textblock.next.cmd === textblock.cmd) { //TODO: seems like there should be a better way to move MathElements around
    var innerblock = this,
      cursor = textblock.cursor,
      next = textblock.next.firstChild;

    next.eachChild(function(child){
      child.parent = innerblock;
      child.jQ.appendTo(innerblock.jQ);
    });

    if (this.lastChild)
      this.lastChild.next = next.firstChild;
    else
      this.firstChild = next.firstChild;

    next.firstChild.prev = this.lastChild;
    this.lastChild = next.lastChild;

    next.parent.remove();

    if (cursor.prev)
      cursor.insertAfter(cursor.prev);
    else
      cursor.prependTo(this);

    cursor.redraw();
  }
  else if (textblock.prev.cmd === textblock.cmd) {
    var cursor = textblock.cursor;
    if (cursor.prev)
      textblock.prev.firstChild.focus();
    else
      cursor.appendTo(textblock.prev.firstChild);
  }
  return this;
};

CharCmds.$ =
LatexCmds.text =
LatexCmds.textnormal =
LatexCmds.textrm =
LatexCmds.textup =
LatexCmds.textmd =
  TextBlock;

function makeTextBlock(latex, html) {
  function SomeTextBlock() {
    TextBlock.apply(this, arguments);
  }
  _ = SomeTextBlock.prototype = new TextBlock;
  _.cmd = latex;
  _.html_template = [ html ];

  return SomeTextBlock;
}

LatexCmds.em = LatexCmds.italic = LatexCmds.italics =
LatexCmds.emph = LatexCmds.textit = LatexCmds.textsl =
  makeTextBlock('\\textit', '<i class="text"></i>');
LatexCmds.strong = LatexCmds.bold = LatexCmds.textbf =
  makeTextBlock('\\textbf', '<b class="text"></b>');
LatexCmds.sf = LatexCmds.textsf =
  makeTextBlock('\\textsf', '<span class="sans-serif text"></span>');
LatexCmds.tt = LatexCmds.texttt =
  makeTextBlock('\\texttt', '<span class="monospace text"></span>');
LatexCmds.textsc =
  makeTextBlock('\\textsc', '<span style="font-variant:small-caps" class="text"></span>');
LatexCmds.uppercase =
  makeTextBlock('\\uppercase', '<span style="text-transform:uppercase" class="text"></span>');
LatexCmds.lowercase =
  makeTextBlock('\\lowercase', '<span style="text-transform:lowercase" class="text"></span>');

// input box to type a variety of LaTeX commands beginning with a backslash
function LatexCommandInput(replacedFragment) {
  this.init('\\');
  if (replacedFragment) {
    this.replacedFragment = replacedFragment.detach();
    this.isEmpty = function(){ return false; };
  }
}
_ = LatexCommandInput.prototype = new MathCommand;
_.html_template = ['<span class="latex-command-input">\\</span>'];
_.text_template = ['\\'];
_.placeCursor = function(cursor) { //TODO: better architecture, better place for this to be done, and more cleanly
  this.cursor = cursor.appendTo(this.firstChild);
  if (this.replacedFragment)
    this.jQ =
      this.jQ.add(this.replacedFragment.jQ.addClass('blur').bind(
        'mousedown mousemove', //FIXME: is monkey-patching the mousedown and mousemove handlers the right way to do this?
        function(e) {
          $(e.target = this.nextSibling).trigger(e);
          return false;
        }
      ).insertBefore(this.jQ));
};
_.latex = function() {
  return '\\' + this.firstChild.latex() + ' ';
};
_.keydown = function(e) {
  if (e.which === 9 || e.which === 13) { //tab or enter
    this.renderCommand();
    return false;
  }
  return this.parent.keydown(e);
};
_.textInput = function(ch) {
  if (ch.match(/[a-z:;,%&]/i)) {
    this.cursor.deleteSelection();
    this.cursor.insertNew(new VanillaSymbol(ch));
    return;
  }
  this.renderCommand();
  if (ch === ' ' || (ch === '\\' && this.firstChild.isEmpty()))
    return;

  this.cursor.parent.textInput(ch);
};
_.renderCommand = function() {
  this.jQ = this.jQ.last();
  this.remove();
  if (this.next)
    this.cursor.insertBefore(this.next);
  else
    this.cursor.appendTo(this.parent);

  var latex = this.firstChild.latex();
  if (latex)
    this.cursor.insertCmd(latex, this.replacedFragment);
  else {
    var cmd = new VanillaSymbol('\\backslash ','\\');
    this.cursor.insertNew(cmd);
    if (this.replacedFragment)
      this.replacedFragment.remove();
  }
};

CharCmds['\\'] = LatexCommandInput;
  
function Binomial(replacedFragment) {
  this.init('\\binom', undefined, undefined, replacedFragment);
  this.jQ.wrapInner('<span class="array"></span>');
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">(</span>').prependTo(this.jQ)
    .add( $('<span class="paren">)</span>').appendTo(this.jQ) );
}
_ = Binomial.prototype = new MathCommand;
_.html_template =
  ['<span class="block"></span>', '<span></span>', '<span></span>'];
_.text_template = ['choose(',',',')'];
_.redraw = Bracket.prototype.redraw;
LatexCmds.binom = LatexCmds.binomial = Binomial;

function Choose() {
  Binomial.apply(this, arguments);
}
_ = Choose.prototype = new Binomial;
_.placeCursor = LiveFraction.prototype.placeCursor;

LatexCmds.choose = Choose;
//// Pekasa added
function Functions(replacedFragment) {
  this.init('\\func', ['<span class="block functionsblock"></span>'], ['Functions(',',',')'], replacedFragment);
  this.blockjQ = this.jQ.children();
}
_ = Functions.prototype = new MathCommand;
_.initBlocks = function(replacedFragment){
  this.blockjQ = this.jQ.children();
  var newBlock, fname, fparams;
  fname = new MathBlock;
  newBlock = fparams =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  fname.jQ = $('<span class="funcname block"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .prependTo(this.jQ);
  fparams.jQ = $('<span class="funcparams block"></span>')
    .data(jQueryDataKey, {block: newBlock})
    .appendTo(this.jQ);
  this.firstChild = fname;
  fname.next = fparams;
  fparams.prev = fname;
  this.lastChild = fparams;
  this.bracketjQs =
    $('<span class="paren">(</span>').insertBefore(fparams.jQ)
    .add( $('<span class="paren">)</span>').insertAfter(fparams.jQ) );
  fname.blur();
  fparams.blur();
  fname.parent = fparams.parent = this;
}
_.redraw = Bracket.prototype.redraw;
_.redraw = function() {
  var outerheight = this.jQ.outerHeight();
  var height = outerheight/+this.blockjQ.css('fontSize').slice(0,-2);
  scale(this.bracketjQs, min(1 + .2*(height - 1), 1.2), 1.05*height);
};
LatexCmds.func = Functions;

//// Pesasa added \cases for E-math
function Cases(replacedFragment) {
  this.init('\\cases', undefined, undefined, replacedFragment);
  this.jQ.wrapInner('<span class="array"></span>');
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">{</span>').prependTo(this.jQ)
      .add( $('<span class="paren"></span>').appendTo(this.jQ) );
}
_ = Cases.prototype = new MathCommand;
_.html_template =
  ['<span class="block cases"></span>', '<span></span>', '<span></span>'];
_.text_template = ['case(',',',')'];
_.redraw = Bracket.prototype.redraw;
LatexCmds.cases = LatexCmds.cases = Cases;

function Case(replacedFragment, token, latex) {
  if (latex && latex[0] === '[') {
    latex.shift();
    var rows = '';
    while (/^[0-9]$/.test(latex[0])){
      rows += latex.shift();
    }
    rows = parseInt(rows);
    if (latex[0] === ']') {
      latex.shift();
    }
  }
  this.rowNum = rows || 1;
  this.editlock = false;
  this.init('\\case', undefined, undefined, replacedFragment);
  this.jQ.wrapInner('<span class="casebody"></span>');
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">{</span>').prependTo(this.jQ)
      .add( $('<span class="paren"></span>').appendTo(this.jQ));
}
_ = Case.prototype = new MathCommand;
_.html_template = ['<span class="block case"></span>'];
_.text_template = ['case(',',',')'];
_.redraw = Bracket.prototype.redraw;
_.initBlocks = function(replacedFragment) {
  var self = this;
  var newTermBlock, newExplBlock, prev;
  
  this.firstChild = newTermBlock =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  newExplBlock = new MathBlock;
  
  this.jQ.append('<span class="caserow"></span>');
  var crow = this.jQ.find('.caserow:last');
  newTermBlock.jQ = $('<span class="casecell"></span>')
    .data(jQueryDataKey, {block: newTermBlock})
    .append(newTermBlock.jQ)
    .appendTo(crow);
  newExplBlock.jQ = $('<span class="casecell caseexpl"></span>')
    .data(jQueryDataKey, {block: newExplBlock})
    .append(newExplBlock.jQ)
    .appendTo(crow);
  newTermBlock.next = newExplBlock;
  newExplBlock.prev = newTermBlock;
  this.lastChild = newExplBlock;
  newTermBlock.blur();
  newExplBlock.blur();
  newTermBlock.parent = newExplBlock.parent = self;
  
  for (var i = 1; i < this.rowNum; i++) {
    this.jQ.append('<span class="caserow"></span>');
    var crow = this.jQ.find('.caserow:last');
    newTermBlock = new MathBlock;
    newExplBlock = new MathBlock;
    newTermBlock.jQ = $('<span class="casecell"></span>')
      .data(jQueryDataKey, {block: newTermBlock})
      .append(newTermBlock.jQ)
      .appendTo(crow);
    newExplBlock.jQ = $('<span class="casecell caseexpl"></span>')
      .data(jQueryDataKey, {block: newExplBlock})
      .append(newExplBlock.jQ)
      .appendTo(crow);
    newTermBlock.next = newExplBlock;
    newExplBlock.prev = newTermBlock;
    this.lastChild.next = newTermBlock;
    newTermBlock.prev = this.lastChild;
    this.lastChild = newExplBlock;
    newTermBlock.blur();
    newExplBlock.blur();
    newTermBlock.parent = newExplBlock.parent = self;
  }
}
_.placeCursor = function(cursor) {
  this.cursor = cursor.appendTo(this.firstChild);
};
_.addRow = function() {
  if (this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var newrow = $('<span class="caserow"></span>');
  var row = $('.caserow:last', this.jQ);
  
  var newTermBlock = new MathBlock;
  var newExplBlock = new MathBlock;

  newTermBlock.parent = this;
  newExplBlock.parent = this;
  newTermBlock.jQ = $('<span class="casecell"></span>').data(jQueryDataKey, {block: newTermBlock}).appendTo(newrow);
  newExplBlock.jQ = $('<span class="casecell caseexpl"></span>').data(jQueryDataKey, {block: newExplBlock}).appendTo(newrow);
  this.lastChild.next = newTermBlock;
  newTermBlock.prev = this.lastChild;
  newTermBlock.next = newExplBlock;
  newExplBlock.prev = newTermBlock;
  this.lastChild = newExplBlock;
  this.cursor.appendTo(newTermBlock);
  this.cursor.appendTo(newExplBlock);

  row.after(newrow);
  this.cursor.appendTo(newTermBlock).redraw();
  this.rowNum++;
  this.editlock = false;
  return false;
}
_.removeRow = function() {
  if (this.rowNum === 1 || this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var row = $('.caserow:last', this.jQ);
  var curr = this.lastChild;
  var prev = curr.prev;
  for (var i = 0; i < 2; i++) {
    curr.next = null;
    curr.prev = null;
    curr.parent = null;
    curr.jQ.remove();
    curr = prev;
    curr.next = null;
    prev = curr.prev;
  }
  this.lastChild = curr;
  row.remove();
  this.rowNum--;
  this.cursor.appendTo(curr).redraw();
  this.editlock = false;
  return false;
}
_.keydown = function(e) {
  var currentBlock = this.cursor.parent;
  var self = this;
  if (currentBlock.parent === this) {
    if (e.which === 8) { //backspace
      if (currentBlock.isEmpty()) {
        if (currentBlock.prev) {
          this.cursor.appendTo(currentBlock.prev);
        } else {
          this.cursor.insertBefore(this);
        }
        return false;
      }
      else if (!this.cursor.prev)
        return false;
    } else if (e.which === 38 && e.ctrlKey) {
      return self.removeRow();
    } else if (e.which === 40 && e.ctrlKey) {
      return self.addRow();
    }
  }
  return this.parent.keydown(e);
};
_.latex = function() {
  var latex = this.cmd + '[' + this.rowNum + ']';
  var child = this.firstChild;
  latex += '{' + child.latex() + '}';
  while (child.next) {
    child = child.next;
    latex += '{' + child.latex() + '}';
  }
  return latex;
}
_.text = function() {
  var text = this.cmd.substr(1) + '(';
  var child = this.firstChild;
  var matrix = [];
  for (var i = 0; i < this.rowNum; i++) {
    var row = [];
    for (var j = 0; j < 2 && !!child; j++){
      row.push(child.text().toString());
      child = child.next;
    }
    matrix.push('[' + row.join(',') + ']');
  }
  text += '[' + matrix.join(',') +']' + ')';
  return text;
}
_.optional_arg_command = 'case';
LatexCmds.case = Case;

function Determ(replacedFragment, token, latex) {
  if (latex && latex[0] === '[') {
    latex.shift();
    var cols = '';
    var rows = '';
    while (/^[0-9]$/.test(latex[0])){
      cols += latex.shift();
    }
    cols = parseInt(cols);
    if (latex[0] === ',') {
      latex.shift();
      while (/^[0-9]$/.test(latex[0])){
        rows += latex.shift();
      }
      rows = parseInt(rows);
    } else {
      rows = cols;
    }
    if (latex[0] === ']') {
      latex.shift();
    }
  }
  this.colNum = cols || 1;
  this.rowNum = rows || 1;
  this.editlock = false;
  this.init('\\determ', undefined, undefined, replacedFragment);
  this.jQ.wrapInner('<span class="matrixbody"></span>');
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">|</span>').prependTo(this.jQ)
      .add( $('<span class="paren">|</span>').appendTo(this.jQ));
}
_ = Determ.prototype = new MathCommand;
_.html_template =
  ['<span class="block determ"></span>'];
_.text_template = ['determ(',',',',',',',')'];
_.redraw = Bracket.prototype.redraw;
_.initBlocks = function(replacedFragment){
  var self = this;
  
  var newBlock, prev;
  this.firstChild = newBlock = prev =
    (replacedFragment && replacedFragment.blockify()) || new MathBlock;
  var firstDone = false;
  
  for (var i = 0; i < this.rowNum; i++) {
    this.jQ.append('<span class="matrixrow"></span>');
    var mrow = this.jQ.find('.matrixrow:last');
    for (var j = 0; j < this.colNum; j++) {
      if (!firstDone) {
        firstDone = true;
        newBlock.jQ = $('<span class="matrixcell"></span>')
          .data(jQueryDataKey, {block: newBlock})
          .append(newBlock.jQ)
          .appendTo(mrow);
        this.lastChild = newBlock;
        newBlock.blur();
        newBlock.parent = self;
      } else {
        newBlock = new MathBlock;
        newBlock.jQ = $('<span class="matrixcell"></span>')
          .data(jQueryDataKey, {block: newBlock})
          .append(newBlock.jQ)
          .appendTo(mrow);
        newBlock.prev = this.lastChild;
        newBlock.prev.next = newBlock;
        this.lastChild = newBlock;
        newBlock.blur();
        newBlock.parent = self;
      }
    }
  }
}
_.placeCursor = function(cursor) {
  this.cursor = cursor.appendTo(this.firstChild);
};
_.addRow = function() {
  if (this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var newrow = $('<span class="matrixrow"></span>');
  var row = $('.matrixrow:last', this.jQ);
  var cb;
  
  for (var i = 0; i < this.colNum; i++) {
    var newBlock = new MathBlock;
    cb = cb || newBlock;
    newBlock.parent = this;
    newBlock.jQ = $('<span class="matrixcell"></span>').data(jQueryDataKey, {block: newBlock}).appendTo(newrow);
    newBlock.prev = this.lastChild;
    this.lastChild.next = newBlock;
    newBlock.next = null;
    this.lastChild = newBlock;
    this.cursor.appendTo(newBlock);
  }
  row.after(newrow);
  this.cursor.appendTo(cb).redraw();
  this.rowNum++;
  this.editlock = false;
  return false;
}
_.addCol = function(){
  if (this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var curr = this.firstChild;
  var rows = $('.matrixrow', this.jQ);
  var cb;
  for (var i = 0; i < this.rowNum; i++) {
    for (var j = 1; j < this.colNum; j++) {
      curr = curr.next;
    }
    var newBlock = new MathBlock;
    cb = cb || newBlock;
    newBlock.parent = this;
    newBlock.jQ = $('<span class="matrixcell"></span>').data(jQueryDataKey, {block: newBlock}).appendTo(rows.eq(i));
    newBlock.prev = curr;
    newBlock.next = curr.next;
    curr.next = newBlock;
    if (newBlock.next) {
      newBlock.next.prev = newBlock;
      curr = newBlock.next;
    } else {
      this.lastChild = newBlock;
    }
    this.cursor.appendTo(newBlock).redraw();
    this.cursor.appendTo(cb).redraw();
  }
  this.colNum++;
  this.editlock = false;
  return false;
}
_.removeRow = function() {
  if (this.rowNum === 1 || this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var row = $('.matrixrow:last', this.jQ);
  var curr = this.lastChild;
  var prev = curr.prev;
  for (var i = 0; i < this.colNum; i++) {
    curr.next = null;
    curr.prev = null;
    curr.parent = null;
    curr.jQ.remove();
    curr = prev;
    curr.next = null;
    prev = curr.prev;
  }
  this.lastChild = curr;
  row.remove();
  this.rowNum--;
  this.cursor.appendTo(curr).redraw();
  this.editlock = false;
  return false;
}
_.removeCol = function() {
  if (this.colNum === 1 || this.editlock) {
    return false;
  } else {
    this.editlock = true;
  }
  var curr = this.firstChild;
  var prev;
  for (var i = 0; i < this.rowNum; i++) {
    for (var j = 1; j < this.colNum; j++){
      prev = curr;
      curr = curr.next;
    }
    prev.next = curr.next;
    if (curr.next) {
      curr.next.prev = prev;
    }
    curr.prev = null;
    curr.next = null;
    curr.parent = null;
    curr.jQ.remove();
    curr = prev.next;
  }
  curr = prev;
  this.lastChild = curr;
  this.cursor.appendTo(curr).redraw();
  this.colNum--;
  this.editlock = false;
  return false;
}
_.keydown = function(e) {
  var currentBlock = this.cursor.parent;
  var self = this;
  if (currentBlock.parent === this) {
    if (e.which === 8) { //backspace
      if (currentBlock.isEmpty()) {
        if (currentBlock.prev) {
          this.cursor.appendTo(currentBlock.prev);
        } else {
          this.cursor.insertBefore(this);
        }
        return false;
      }
      else if (!this.cursor.prev)
        return false;
    } else if (e.which === 37 && e.ctrlKey) {
      return self.removeCol();
    } else if (e.which === 38 && e.ctrlKey) {
      return self.removeRow();
    } else if (e.which === 39 && e.ctrlKey) {
      return self.addCol();
    } else if (e.which === 40 && e.ctrlKey) {
      return self.addRow();
    }
  }
  return this.parent.keydown(e);
};
_.latex = function() {
  var latex = this.cmd + '[' + this.colNum + ',' + this.rowNum + ']';
  var child = this.firstChild;
  latex += '{' + child.latex() + '}';
  while (child.next) {
    child = child.next;
    latex += '{' + child.latex() + '}';
  }
  return latex;
}
_.text = function() {
  var text = this.cmd.substr(1) + '(';
  var child = this.firstChild;
  var matrix = [];
  for (var i = 0; i < this.rowNum; i++) {
    var row = [];
    for (var j = 0; j < this.colNum && !!child; j++){
      row.push(child.text().toString());
      child = child.next;
    }
    matrix.push('[' + row.join(',') + ']');
  }
  text += '[' + matrix.join(',') +']' + ')';
  return text;
}
_.optional_arg_command = 'determ';
LatexCmds.determ = Determ;

function Matrix(replacedFragment, token, latex) {
  if (latex && latex[0] === '[') {
    latex.shift();
    var cols = '';
    var rows = '';
    while (/^[0-9]$/.test(latex[0])){
      cols += latex.shift();
    }
    cols = parseInt(cols);
    if (latex[0] === ',') {
      latex.shift();
      while (/^[0-9]$/.test(latex[0])){
        rows += latex.shift();
      }
      rows = parseInt(rows);
    } else {
      rows = cols;
    }
    if (latex[0] === ']') {
      latex.shift();
    }
  }
  this.colNum = cols || 1;
  this.rowNum = rows || 1;
  this.init('\\matrix', undefined, undefined, replacedFragment);
  this.jQ.wrapInner('<span class="matrixbody"></span>');
  this.blockjQ = this.jQ.children();
  this.bracketjQs =
    $('<span class="paren">(</span>').prependTo(this.jQ)
      .add( $('<span class="paren">)</span>').appendTo(this.jQ));
}
_ = Matrix.prototype = new Determ;
_.html_template =
  ['<span class="block matrix"></span>'];
_.text_template = ['matrix(',',',',',',',')'];
_.optional_arg_command = 'matrix';
LatexCmds.matrix = Matrix;

function Vector(replacedFragment) {
  this.init('\\vector', undefined, undefined, replacedFragment);
}
_ = Vector.prototype = new MathCommand;
_.html_template = ['<span class="array"></span>', '<span></span>'];
_.latex = function() {
  return '\\begin{matrix}' + this.foldChildren([], function(latex, child) {
    latex.push(child.latex());
    return latex;
  }).join('\\\\') + '\\end{matrix}';
};
_.text = function() {
  return '[' + this.foldChildren([], function(text, child) {
    text.push(child.text());
    return text;
  }).join() + ']';
}
_.placeCursor = function(cursor) {
  this.cursor = cursor.appendTo(this.firstChild);
};
_.keydown = function(e) {
  var currentBlock = this.cursor.parent;

  if (currentBlock.parent === this) {
    if (e.which === 13) { //enter
      var newBlock = new MathBlock;
      newBlock.parent = this;
      newBlock.jQ = $('<span></span>')
        .data(jQueryDataKey, {block: newBlock})
        .insertAfter(currentBlock.jQ);
      if (currentBlock.next)
        currentBlock.next.prev = newBlock;
      else
        this.lastChild = newBlock;

      newBlock.next = currentBlock.next;
      currentBlock.next = newBlock;
      newBlock.prev = currentBlock;
      this.cursor.appendTo(newBlock).redraw();
      return false;
    }
    else if (e.which === 9 && !e.shiftKey && !currentBlock.next) { //tab
      if (currentBlock.isEmpty()) {
        if (currentBlock.prev) {
          this.cursor.insertAfter(this);
          delete currentBlock.prev.next;
          this.lastChild = currentBlock.prev;
          currentBlock.jQ.remove();
          this.cursor.redraw();
          return false;
        }
        else
          return this.parent.keydown(e);
      }

      var newBlock = new MathBlock;
      newBlock.parent = this;
      newBlock.jQ = $('<span></span>').data(jQueryDataKey, {block: newBlock}).appendTo(this.jQ);
      this.lastChild = newBlock;
      currentBlock.next = newBlock;
      newBlock.prev = currentBlock;
      this.cursor.appendTo(newBlock).redraw();
      return false;
    }
    else if (e.which === 8) { //backspace
      if (currentBlock.isEmpty()) {
        if (currentBlock.prev) {
          this.cursor.appendTo(currentBlock.prev)
          currentBlock.prev.next = currentBlock.next;
        }
        else {
          this.cursor.insertBefore(this);
          this.firstChild = currentBlock.next;
        }

        if (currentBlock.next)
          currentBlock.next.prev = currentBlock.prev;
        else
          this.lastChild = currentBlock.prev;

        currentBlock.jQ.remove();
        if (this.isEmpty())
          this.cursor.deleteForward();
        else
          this.cursor.redraw();

        return false;
      }
      else if (!this.cursor.prev)
        return false;
    }
  }
  return this.parent.keydown(e);
};

LatexCmds.vector = Vector;

LatexCmds.editable = proto(RootMathCommand, function() {
  this.init('\\editable');
  createRoot(this.jQ, this.firstChild, false, true);
  var cursor;
  this.placeCursor = function(c) { cursor = c.appendTo(this.firstChild); };
  this.firstChild.blur = function() {
    if (cursor.prev !== this.parent) return; //when cursor is inserted after editable, append own cursor FIXME HACK
    delete this.blur;
    this.cursor.appendTo(this);
    MathBlock.prototype.blur.call(this);
  };
  this.latex = function(){ return this.firstChild.latex(); };
  this.text = function(){ return this.firstChild.text(); };
});

/**********************************
 * Symbols and Special Characters
 *********************************/

LatexCmds.f = bind(Symbol, 'f', '<var class="florin">&fnof;</var><span style="display:inline-block;width:0">&nbsp;</span>');

function Variable(ch, html) {
  Symbol.call(this, ch, '<var>'+(html || ch)+'</var>');
}
_ = Variable.prototype = new Symbol;
_.text = function() {
  var text = this.cmd;
  if (this.prev && !(this.prev instanceof Variable)
      && !(this.prev instanceof BinaryOperator))
    text = '*' + text;
  if (this.next && !(this.next instanceof BinaryOperator)
      && !(this.next.cmd === '^'))
    text += '*';
  return text;
};

function VanillaSymbol(ch, html) {
  Symbol.call(this, ch, '<span>'+(html || ch)+'</span>');
}
VanillaSymbol.prototype = Symbol.prototype;

CharCmds[' '] = bind(VanillaSymbol, '\\:', ' ');

LatexCmds.prime = CharCmds["'"] = bind(VanillaSymbol, "'", '&prime;');

function NonSymbolaSymbol(ch, html) { //does not use Symbola font
  Symbol.call(this, ch, '<span class="nonSymbola">'+(html || ch)+'</span>');
}
NonSymbolaSymbol.prototype = Symbol.prototype;

LatexCmds['@'] = NonSymbolaSymbol;
LatexCmds['&'] = bind(NonSymbolaSymbol, '\\&', '&');
LatexCmds['%'] = bind(NonSymbolaSymbol, '\\%', '%');

//the following are all Greek to me, but this helped a lot: http://www.ams.org/STIX/ion/stixsig03.html

//lowercase Greek letter variables
LatexCmds.alpha =
LatexCmds.beta =
LatexCmds.gamma =
LatexCmds.delta =
LatexCmds.zeta =
LatexCmds.eta =
LatexCmds.theta =
LatexCmds.iota =
LatexCmds.kappa =
LatexCmds.mu =
LatexCmds.nu =
LatexCmds.xi =
LatexCmds.rho =
LatexCmds.sigma =
LatexCmds.tau =
LatexCmds.chi =
LatexCmds.psi =
LatexCmds.omega = proto(Symbol, function(replacedFragment, latex) {
  Variable.call(this,'\\'+latex+' ','&'+latex+';');
});

//why can't anybody FUCKING agree on these
LatexCmds.phi = //W3C or Unicode?
  bind(Variable,'\\phi ','&#981;');

LatexCmds.phiv = //Elsevier and 9573-13
LatexCmds.varphi = //AMS and LaTeX
  bind(Variable,'\\varphi ','&phi;');

LatexCmds.epsilon = //W3C or Unicode?
  bind(Variable,'\\epsilon ','&#1013;');

LatexCmds.epsiv = //Elsevier and 9573-13
LatexCmds.varepsilon = //AMS and LaTeX
  bind(Variable,'\\varepsilon ','&epsilon;');

LatexCmds.piv = //W3C/Unicode and Elsevier and 9573-13
LatexCmds.varpi = //AMS and LaTeX
  bind(Variable,'\\varpi ','&piv;');

LatexCmds.sigmaf = //W3C/Unicode
LatexCmds.sigmav = //Elsevier
LatexCmds.varsigma = //LaTeX
  bind(Variable,'\\varsigma ','&sigmaf;');

LatexCmds.thetav = //Elsevier and 9573-13
LatexCmds.vartheta = //AMS and LaTeX
LatexCmds.thetasym = //W3C/Unicode
  bind(Variable,'\\vartheta ','&thetasym;');

LatexCmds.upsilon = //AMS and LaTeX and W3C/Unicode
LatexCmds.upsi = //Elsevier and 9573-13
  bind(Variable,'\\upsilon ','&upsilon;');

//these aren't even mentioned in the HTML character entity references
LatexCmds.gammad = //Elsevier
LatexCmds.Gammad = //9573-13 -- WTF, right? I dunno if this was a typo in the reference (see above)
LatexCmds.digamma = //LaTeX
  bind(Variable,'\\digamma ','&#989;');

LatexCmds.kappav = //Elsevier
LatexCmds.varkappa = //AMS and LaTeX
  bind(Variable,'\\varkappa ','&#1008;');

LatexCmds.rhov = //Elsevier and 9573-13
LatexCmds.varrho = //AMS and LaTeX
  bind(Variable,'\\varrho ','&#1009;');

//Greek constants, look best in un-italicised Times New Roman
LatexCmds.pi = LatexCmds['π'] = bind(NonSymbolaSymbol,'\\pi ','&pi;');
LatexCmds.lambda = bind(NonSymbolaSymbol,'\\lambda ','&lambda;');

//uppercase greek letters

LatexCmds.Upsilon = //LaTeX
LatexCmds.Upsi = //Elsevier and 9573-13
LatexCmds.upsih = //W3C/Unicode "upsilon with hook"
LatexCmds.Upsih = //'cos it makes sense to me
  bind(Symbol,'\\Upsilon ','<var style="font-family: serif">&upsih;</var>'); //Symbola's 'upsilon with a hook' is a capital Y without hooks :(

LatexCmds.Gamma =
LatexCmds.Delta =
LatexCmds.Theta =
LatexCmds.Lambda =
LatexCmds.Xi =
LatexCmds.Pi =
LatexCmds.Sigma =
LatexCmds.Phi =
LatexCmds.Psi =
LatexCmds.Omega =

//other symbols with the same LaTeX command and HTML character entity reference
LatexCmds.forall = proto(Symbol, function(replacedFragment, latex) {
  VanillaSymbol.call(this,'\\'+latex+' ','&'+latex+';');
});

function BinaryOperator(cmd, html, text) {
  Symbol.call(this, cmd, '<span class="binary-operator">'+html+'</span>', text);
}
BinaryOperator.prototype = new Symbol; //so instanceof will work

function PlusMinus(cmd, html) {
  VanillaSymbol.apply(this, arguments);
}
_ = PlusMinus.prototype = new BinaryOperator; //so instanceof will work
_.respace = function() {
  if (!this.prev) {
    this.jQ[0].className = '';
  }
  else if (
    this.prev instanceof BinaryOperator &&
    this.next && !(this.next instanceof BinaryOperator)
  ) {
    this.jQ[0].className = 'unary-operator';
  }
  else {
    this.jQ[0].className = 'binary-operator';
  }
  return this;
};

LatexCmds['+'] = bind(PlusMinus, '+', '+');
//yes, these are different dashes, I think one is an en dash and the other is a hyphen
LatexCmds['–'] = LatexCmds['-'] = bind(PlusMinus, '-', '&minus;');
LatexCmds['±'] = LatexCmds.pm = LatexCmds.plusmn = LatexCmds.plusminus =
  bind(PlusMinus,'\\pm ','&plusmn;');
LatexCmds.mp = LatexCmds.mnplus = LatexCmds.minusplus =
  bind(PlusMinus,'\\mp ','&#8723;');

CharCmds['*'] = LatexCmds.sdot = LatexCmds.cdot =
  bind(BinaryOperator, '\\cdot ', '&middot;');
//semantically should be &sdot;, but &middot; looks better

LatexCmds['='] = bind(BinaryOperator, '=', '=');
LatexCmds['lt'] = bind(BinaryOperator, '\\lt ', '&lt;');
LatexCmds['gt'] = bind(BinaryOperator, '\\gt ', '&gt;');

LatexCmds.notin =
LatexCmds.sim =
LatexCmds.cong =
LatexCmds.equiv =
LatexCmds.oplus =
LatexCmds.otimes = proto(BinaryOperator, function(replacedFragment, latex) {
  BinaryOperator.call(this, '\\'+latex+' ', '&'+latex+';');
});

LatexCmds.times = proto(BinaryOperator, function(replacedFragment, latex) {
  BinaryOperator.call(this, '\\times ', '&times;', '[x]')
});

LatexCmds['÷'] = LatexCmds.div = LatexCmds.divide = LatexCmds.divides =
  bind(BinaryOperator,'\\div ','&divide;', '[/]');

LatexCmds['≠'] = LatexCmds.ne = LatexCmds.neq = bind(BinaryOperator,'\\ne ','&ne;');

LatexCmds.ast = LatexCmds.star = LatexCmds.loast = LatexCmds.lowast =
  bind(BinaryOperator,'\\ast ','&lowast;');
  //case 'there4 = // a special exception for this one, perhaps?
LatexCmds.therefor = LatexCmds.therefore =
  bind(BinaryOperator,'\\therefore ','&there4;');

LatexCmds.cuz = // l33t
LatexCmds.because = bind(BinaryOperator,'\\because ','&#8757;');

LatexCmds.prop = LatexCmds.propto = bind(BinaryOperator,'\\propto ','&prop;');

LatexCmds['≈'] = LatexCmds.asymp = LatexCmds.approx = bind(BinaryOperator,'\\approx ','&asymp;');

LatexCmds['<'] = bind(BinaryOperator,'\\lt ','&lt;');

LatexCmds['>'] = bind(BinaryOperator,'\\gt ','&gt;');

LatexCmds['≤'] = LatexCmds.le = LatexCmds.leq = bind(BinaryOperator,'\\le ','&le;');

LatexCmds['≥'] = LatexCmds.ge = LatexCmds.geq = bind(BinaryOperator,'\\ge ','&ge;');

LatexCmds.isin = LatexCmds['in'] = bind(BinaryOperator,'\\in ','&isin;');

LatexCmds.ni = LatexCmds.contains = bind(BinaryOperator,'\\ni ','&ni;');

LatexCmds.notni = LatexCmds.niton = LatexCmds.notcontains = LatexCmds.doesnotcontain =
  bind(BinaryOperator,'\\not\\ni ','&#8716;');

LatexCmds.sub = LatexCmds.subset = bind(BinaryOperator,'\\subset ','&sub;');

LatexCmds.sup = LatexCmds.supset = LatexCmds.superset =
  bind(BinaryOperator,'\\supset ','&sup;');

LatexCmds.nsub = LatexCmds.notsub =
LatexCmds.nsubset = LatexCmds.notsubset =
  bind(BinaryOperator,'\\not\\subset ','&#8836;');

LatexCmds.nsup = LatexCmds.notsup =
LatexCmds.nsupset = LatexCmds.notsupset =
LatexCmds.nsuperset = LatexCmds.notsuperset =
  bind(BinaryOperator,'\\not\\supset ','&#8837;');

LatexCmds.sube = LatexCmds.subeq = LatexCmds.subsete = LatexCmds.subseteq =
  bind(BinaryOperator,'\\subseteq ','&sube;');

LatexCmds.supe = LatexCmds.supeq =
LatexCmds.supsete = LatexCmds.supseteq =
LatexCmds.supersete = LatexCmds.superseteq =
  bind(BinaryOperator,'\\supseteq ','&supe;');

LatexCmds.nsube = LatexCmds.nsubeq =
LatexCmds.notsube = LatexCmds.notsubeq =
LatexCmds.nsubsete = LatexCmds.nsubseteq =
LatexCmds.notsubsete = LatexCmds.notsubseteq =
  bind(BinaryOperator,'\\not\\subseteq ','&#8840;');

LatexCmds.nsupe = LatexCmds.nsupeq =
LatexCmds.notsupe = LatexCmds.notsupeq =
LatexCmds.nsupsete = LatexCmds.nsupseteq =
LatexCmds.notsupsete = LatexCmds.notsupseteq =
LatexCmds.nsupersete = LatexCmds.nsuperseteq =
LatexCmds.notsupersete = LatexCmds.notsuperseteq =
  bind(BinaryOperator,'\\not\\supseteq ','&#8841;');


//sum, product, coproduct, integral
function BigSymbol(ch, html) {
  Symbol.call(this, ch, '<big>'+html+'</big>');
}
BigSymbol.prototype = new Symbol; //so instanceof will work

LatexCmds['∑'] = LatexCmds.sum = LatexCmds.summation = bind(BigSymbol,'\\sum ','&sum;');
LatexCmds['∏'] = LatexCmds.prod = LatexCmds.product = bind(BigSymbol,'\\prod ','&prod;');
LatexCmds.coprod = LatexCmds.coproduct = bind(BigSymbol,'\\coprod ','&#8720;');
LatexCmds['∫'] = LatexCmds['int'] = LatexCmds.integral = bind(BigSymbol,'\\int ','&int;');



//the canonical sets of numbers
LatexCmds.N = LatexCmds.naturals = LatexCmds.Naturals =
  bind(VanillaSymbol,'\\mathbb{N}','&#8469;');

LatexCmds.P =
LatexCmds.primes = LatexCmds.Primes =
LatexCmds.projective = LatexCmds.Projective =
LatexCmds.probability = LatexCmds.Probability =
  bind(VanillaSymbol,'\\mathbb{P}','&#8473;');

LatexCmds.Z = LatexCmds.integers = LatexCmds.Integers =
  bind(VanillaSymbol,'\\mathbb{Z}','&#8484;');

LatexCmds.Q = LatexCmds.rationals = LatexCmds.Rationals =
  bind(VanillaSymbol,'\\mathbb{Q}','&#8474;');

LatexCmds.R = LatexCmds.reals = LatexCmds.Reals =
  bind(VanillaSymbol,'\\mathbb{R}','&#8477;');

LatexCmds.C =
LatexCmds.complex = LatexCmds.Complex =
LatexCmds.complexes = LatexCmds.Complexes =
LatexCmds.complexplane = LatexCmds.Complexplane = LatexCmds.ComplexPlane =
  bind(VanillaSymbol,'\\mathbb{C}','&#8450;');

LatexCmds.H = LatexCmds.Hamiltonian = LatexCmds.quaternions = LatexCmds.Quaternions =
  bind(VanillaSymbol,'\\mathbb{H}','&#8461;');

// pekasa added true and false
LatexCmds.dollar = bind(VanillaSymbol, '\\dollar','$');
// pekasa added true and false
LatexCmds.T = LatexCmds.true = bind(VanillaSymbol, '\\T','T');
LatexCmds.F = LatexCmds.false = bind(VanillaSymbol, '\\F','F');

// pesasa added some working alternatives for sets of numbers as long as \\mathbb{} is broken.
LatexCmds.NN = bind(VanillaSymbol, '\\NN','&#8469;');
LatexCmds.PP = bind(VanillaSymbol, '\\PP','&#8473;');
LatexCmds.ZZ = bind(VanillaSymbol, '\\ZZ','&#8484;');
LatexCmds.QQ = bind(VanillaSymbol, '\\QQ','&#8474;');
LatexCmds.RR = bind(VanillaSymbol, '\\RR','&#8477;');
LatexCmds.CC = bind(VanillaSymbol, '\\CC','&#8450;');
LatexCmds.HH = bind(VanillaSymbol, '\\HH','&#8461;');
LatexCmds.AA = bind(VanillaSymbol, '\\AA','&#120120;');
LatexCmds.BB = bind(VanillaSymbol, '\\BB','&#120121;');
LatexCmds.DD = bind(VanillaSymbol, '\\DD','&#120123;');
LatexCmds.EE = bind(VanillaSymbol, '\\EE','&#120124;');
LatexCmds.FF = bind(VanillaSymbol, '\\FF','&#120125;');
LatexCmds.GG = bind(VanillaSymbol, '\\GG','&#120126;');
LatexCmds.II = bind(VanillaSymbol, '\\II','&#120128;');
LatexCmds.JJ = bind(VanillaSymbol, '\\JJ','&#120129;');
LatexCmds.KK = bind(VanillaSymbol, '\\KK','&#120130;');
LatexCmds.LL = bind(VanillaSymbol, '\\LL','&#120131;');
LatexCmds.MM = bind(VanillaSymbol, '\\MM','&#120132;');
LatexCmds.OO = bind(VanillaSymbol, '\\OO','&#120134;');
LatexCmds.SS = bind(VanillaSymbol, '\\SS','&#120138;');
LatexCmds.TT = bind(VanillaSymbol, '\\TT','&#120139;');
LatexCmds.UU = bind(VanillaSymbol, '\\UU','&#120140;');
LatexCmds.VV = bind(VanillaSymbol, '\\VV','&#120141;');
LatexCmds.WW = bind(VanillaSymbol, '\\WW','&#120142;');
LatexCmds.XX = bind(VanillaSymbol, '\\XX','&#120143;');
LatexCmds.YY = bind(VanillaSymbol, '\\YY','&#120144;');

//spacing
LatexCmds.quad = LatexCmds.emsp = bind(VanillaSymbol,'\\quad ','    ');
LatexCmds.qquad = bind(VanillaSymbol,'\\qquad ','        ');
LatexCmds[','] = bind(VanillaSymbol,'\\, ',' ');
LatexCmds[':'] = bind(VanillaSymbol,'\\: ','  ');
LatexCmds[';'] = bind(VanillaSymbol,'\\; ','   ');
LatexCmds['%'] = bind(VanillaSymbol,'\\% ','<span style="margin-left:0.2em;margin-right:0.2em;">%</span>');
LatexCmds['&'] = bind(VanillaSymbol,'\\& ','<span style="margin-left:0.2em;margin-right:0.2em;">&</span>');

/* spacing special characters, gonna have to implement this in LatexCommandInput::textInput somehow
case ',':
  return new VanillaSymbol('\\, ',' ');
case ':':
  return new VanillaSymbol('\\: ','  ');
case ';':
  return new VanillaSymbol('\\; ','   ');
case '!':
  return new Symbol('\\! ','<span style="margin-right:-.2em"></span>');
*/

//binary operators
LatexCmds.diamond = bind(VanillaSymbol, '\\diamond ', '&#9671;');
LatexCmds.bigtriangleup = bind(VanillaSymbol, '\\bigtriangleup ', '&#9651;');
LatexCmds.ominus = bind(VanillaSymbol, '\\ominus ', '&#8854;');
LatexCmds.uplus = bind(VanillaSymbol, '\\uplus ', '&#8846;');
LatexCmds.bigtriangledown = bind(VanillaSymbol, '\\bigtriangledown ', '&#9661;');
LatexCmds.sqcap = bind(VanillaSymbol, '\\sqcap ', '&#8851;');
LatexCmds.triangleleft = bind(VanillaSymbol, '\\triangleleft ', '&#8882;');
LatexCmds.sqcup = bind(VanillaSymbol, '\\sqcup ', '&#8852;');
LatexCmds.triangleright = bind(VanillaSymbol, '\\triangleright ', '&#8883;');
LatexCmds.odot = bind(VanillaSymbol, '\\odot ', '&#8857;');
LatexCmds.bigcirc = bind(VanillaSymbol, '\\bigcirc ', '&#9711;');
LatexCmds.dagger = bind(VanillaSymbol, '\\dagger ', '&#0134;');
LatexCmds.ddagger = bind(VanillaSymbol, '\\ddagger ', '&#135;');
LatexCmds.wr = bind(VanillaSymbol, '\\wr ', '&#8768;');
LatexCmds.amalg = bind(VanillaSymbol, '\\amalg ', '&#8720;');

//relationship symbols
LatexCmds.models = bind(VanillaSymbol, '\\models ', '&#8872;');
LatexCmds.prec = bind(VanillaSymbol, '\\prec ', '&#8826;');
LatexCmds.succ = bind(VanillaSymbol, '\\succ ', '&#8827;');
LatexCmds.preceq = bind(VanillaSymbol, '\\preceq ', '&#8828;');
LatexCmds.succeq = bind(VanillaSymbol, '\\succeq ', '&#8829;');
LatexCmds.simeq = bind(VanillaSymbol, '\\simeq ', '&#8771;');
LatexCmds.mid = bind(VanillaSymbol, '\\mid ', '&#8739;');
LatexCmds.ll = bind(VanillaSymbol, '\\ll ', '&#8810;');
LatexCmds.gg = bind(VanillaSymbol, '\\gg ', '&#8811;');
LatexCmds.parallel = bind(VanillaSymbol, '\\parallel ', '&#8741;');
/*** pesasa added for e-math ****/
LatexCmds.nparallel = bind(VanillaSymbol, '\\nparallel ', '&#8742;');
LatexCmds.bowtie = bind(VanillaSymbol, '\\bowtie ', '&#8904;');
LatexCmds.sqsubset = bind(VanillaSymbol, '\\sqsubset ', '&#8847;');
LatexCmds.sqsupset = bind(VanillaSymbol, '\\sqsupset ', '&#8848;');
LatexCmds.smile = bind(VanillaSymbol, '\\smile ', '&#8995;');
LatexCmds.sqsubseteq = bind(VanillaSymbol, '\\sqsubseteq ', '&#8849;');
LatexCmds.sqsupseteq = bind(VanillaSymbol, '\\sqsupseteq ', '&#8850;');
LatexCmds.doteq = bind(VanillaSymbol, '\\doteq ', '&#8784;');
LatexCmds.frown = bind(VanillaSymbol, '\\frown ', '&#8994;');
LatexCmds.vdash = bind(VanillaSymbol, '\\vdash ', '&#8870;');
LatexCmds.dashv = bind(VanillaSymbol, '\\dashv ', '&#8867;');
LatexCmds.Vdash = bind(VanillaSymbol, '\\Vdash ', '&#8873;');
LatexCmds.nmid = bind(VanillaSymbol, '\\nmid ', '&#8740;');
LatexCmds.square = bind(VanillaSymbol, '\\square ', '&#9633;');

//arrows
LatexCmds.longleftarrow = bind(VanillaSymbol, '\\longleftarrow ', '&#8592;');
LatexCmds.longrightarrow = bind(VanillaSymbol, '\\longrightarrow ', '&#8594;');
LatexCmds.Longleftarrow = bind(VanillaSymbol, '\\Longleftarrow ', '&#8656;');
LatexCmds.Longrightarrow = bind(VanillaSymbol, '\\Longrightarrow ', '&#8658;');
LatexCmds.longleftrightarrow = bind(VanillaSymbol, '\\longleftrightarrow ', '&#8596;');
LatexCmds.updownarrow = bind(VanillaSymbol, '\\updownarrow ', '&#8597;');
LatexCmds.Longleftrightarrow = bind(VanillaSymbol, '\\Longleftrightarrow ', '&#8660;');
LatexCmds.Updownarrow = bind(VanillaSymbol, '\\Updownarrow ', '&#8661;');
LatexCmds.mapsto = bind(VanillaSymbol, '\\mapsto ', '&#8614;');
LatexCmds.nearrow = bind(VanillaSymbol, '\\nearrow ', '&#8599;');
LatexCmds.hookleftarrow = bind(VanillaSymbol, '\\hookleftarrow ', '&#8617;');
LatexCmds.hookrightarrow = bind(VanillaSymbol, '\\hookrightarrow ', '&#8618;');
LatexCmds.searrow = bind(VanillaSymbol, '\\searrow ', '&#8600;');
LatexCmds.leftharpoonup = bind(VanillaSymbol, '\\leftharpoonup ', '&#8636;');
LatexCmds.rightharpoonup = bind(VanillaSymbol, '\\rightharpoonup ', '&#8640;');
LatexCmds.swarrow = bind(VanillaSymbol, '\\swarrow ', '&#8601;');
LatexCmds.leftharpoondown = bind(VanillaSymbol, '\\leftharpoondown ', '&#8637;');
LatexCmds.rightharpoondown = bind(VanillaSymbol, '\\rightharpoondown ', '&#8641;');
LatexCmds.nwarrow = bind(VanillaSymbol, '\\nwarrow ', '&#8598;');

//Misc
LatexCmds.ldots = bind(VanillaSymbol, '\\ldots ', '&#8230;');
LatexCmds.cdots = bind(VanillaSymbol, '\\cdots ', '&#8943;');
LatexCmds.vdots = bind(VanillaSymbol, '\\vdots ', '&#8942;');
LatexCmds.ddots = bind(VanillaSymbol, '\\ddots ', '&#8944;');
LatexCmds.surd = bind(VanillaSymbol, '\\surd ', '&#8730;');
LatexCmds.triangle = bind(VanillaSymbol, '\\triangle ', '&#9653;');
LatexCmds.ell = bind(VanillaSymbol, '\\ell ', '&#8467;');
LatexCmds.top = bind(VanillaSymbol, '\\top ', '&#8868;');
LatexCmds.flat = bind(VanillaSymbol, '\\flat ', '&#9837;');
LatexCmds.natural = bind(VanillaSymbol, '\\natural ', '&#9838;');
LatexCmds.sharp = bind(VanillaSymbol, '\\sharp ', '&#9839;');
LatexCmds.wp = bind(VanillaSymbol, '\\wp ', '&#8472;');
LatexCmds.bot = bind(VanillaSymbol, '\\bot ', '&#8869;');
LatexCmds.clubsuit = bind(VanillaSymbol, '\\clubsuit ', '&#9827;');
LatexCmds.diamondsuit = bind(VanillaSymbol, '\\diamondsuit ', '&#9826;');
LatexCmds.heartsuit = bind(VanillaSymbol, '\\heartsuit ', '&#9825;');
LatexCmds.spadesuit = bind(VanillaSymbol, '\\spadesuit ', '&#9824;');

//variable-sized
LatexCmds.oint = bind(VanillaSymbol, '\\oint ', '&#8750;');
LatexCmds.bigcap = bind(VanillaSymbol, '\\bigcap ', '&#8745;');
LatexCmds.bigcup = bind(VanillaSymbol, '\\bigcup ', '&#8746;');
LatexCmds.bigsqcup = bind(VanillaSymbol, '\\bigsqcup ', '&#8852;');
LatexCmds.bigvee = bind(VanillaSymbol, '\\bigvee ', '&#8744;');
LatexCmds.bigwedge = bind(VanillaSymbol, '\\bigwedge ', '&#8743;');
LatexCmds.bigodot = bind(VanillaSymbol, '\\bigodot ', '&#8857;');
LatexCmds.bigotimes = bind(VanillaSymbol, '\\bigotimes ', '&#8855;');
LatexCmds.bigoplus = bind(VanillaSymbol, '\\bigoplus ', '&#8853;');
LatexCmds.biguplus = bind(VanillaSymbol, '\\biguplus ', '&#8846;');

//delimiters
LatexCmds.lfloor = bind(VanillaSymbol, '\\lfloor ', '&#8970;');
LatexCmds.rfloor = bind(VanillaSymbol, '\\rfloor ', '&#8971;');
LatexCmds.lceil = bind(VanillaSymbol, '\\lceil ', '&#8968;');
LatexCmds.rceil = bind(VanillaSymbol, '\\rceil ', '&#8969;');
LatexCmds.slash = bind(VanillaSymbol, '\\slash ', '&#47;');
LatexCmds.opencurlybrace = bind(VanillaSymbol, '\\opencurlybrace ', '&#123;');
LatexCmds.closecurlybrace = bind(VanillaSymbol, '\\closecurlybrace ', '&#125;');

//various symbols

LatexCmds.caret = bind(VanillaSymbol,'\\caret ','^');
LatexCmds.underscore = bind(VanillaSymbol,'\\underscore ','_');
LatexCmds.backslash = bind(VanillaSymbol,'\\backslash ','\\');
LatexCmds.vert = bind(VanillaSymbol,'|');
LatexCmds.perp = LatexCmds.perpendicular = bind(VanillaSymbol,'\\perp ','&perp;');
LatexCmds.nabla = LatexCmds.del = bind(VanillaSymbol,'\\nabla ','&nabla;');
LatexCmds.hbar = bind(VanillaSymbol,'\\hbar ','&#8463;');

LatexCmds.AA = LatexCmds.Angstrom = LatexCmds.angstrom =
  bind(VanillaSymbol,'\\text\\AA ','&#8491;');

LatexCmds.ring = LatexCmds.circ = LatexCmds.circle =
  bind(VanillaSymbol,'\\circ ','&#8728;');

LatexCmds.bull = LatexCmds.bullet = bind(VanillaSymbol,'\\bullet ','&bull;');

LatexCmds.setminus = LatexCmds.smallsetminus =
  bind(VanillaSymbol,'\\setminus ','&#8726;');

LatexCmds.not = //bind(Symbol,'\\not ','<span class="not">/</span>');
LatexCmds['¬'] = LatexCmds.neg = bind(VanillaSymbol,'\\neg ','&not;');

LatexCmds['…'] = LatexCmds.dots = LatexCmds.ellip = LatexCmds.hellip =
LatexCmds.ellipsis = LatexCmds.hellipsis =
  bind(VanillaSymbol,'\\dots ','&hellip;');

LatexCmds.converges =
LatexCmds.darr = LatexCmds.dnarr = LatexCmds.dnarrow = LatexCmds.downarrow =
  bind(VanillaSymbol,'\\downarrow ','&darr;');

LatexCmds.dArr = LatexCmds.dnArr = LatexCmds.dnArrow = LatexCmds.Downarrow =
  bind(VanillaSymbol,'\\Downarrow ','&dArr;');

LatexCmds.diverges = LatexCmds.uarr = LatexCmds.uparrow =
  bind(VanillaSymbol,'\\uparrow ','&uarr;');

LatexCmds.uArr = LatexCmds.Uparrow = bind(VanillaSymbol,'\\Uparrow ','&uArr;');

LatexCmds.to = bind(BinaryOperator,'\\to ','&rarr;');

//LatexCmds.rarr = LatexCmds.rightarrow = bind(VanillaSymbol,'\\rightarrow ','&rarr;');
// By pesasa: change rightarrow to binary operator
LatexCmds.rarr = LatexCmds.rightarrow = bind(BinaryOperator,'\\rightarrow ','&rarr;');

LatexCmds.implies = bind(BinaryOperator,'\\Rightarrow ','&rArr;');

//LatexCmds.rArr = LatexCmds.Rightarrow = bind(VanillaSymbol,'\\Rightarrow ','&rArr;');
// By pesasa: change Rightarrow to binary operator
LatexCmds.rArr = LatexCmds.Rightarrow = bind(BinaryOperator,'\\Rightarrow ','&rArr;');

LatexCmds.gets = bind(BinaryOperator,'\\gets ','&larr;');

//LatexCmds.larr = LatexCmds.leftarrow = bind(VanillaSymbol,'\\leftarrow ','&larr;');
// By pesasa: change leftarrow to binary operator
LatexCmds.larr = LatexCmds.leftarrow = bind(BinaryOperator,'\\leftarrow ','&larr;');

LatexCmds.impliedby = bind(BinaryOperator,'\\Leftarrow ','&lArr;');

//LatexCmds.lArr = LatexCmds.Leftarrow = bind(VanillaSymbol,'\\Leftarrow ','&lArr;');
// By pesasa: change Leftarrow to binary operator
LatexCmds.lArr = LatexCmds.Leftarrow = bind(BinaryOperator,'\\Leftarrow ','&lArr;');

//LatexCmds.harr = LatexCmds.lrarr = LatexCmds.leftrightarrow =
//  bind(VanillaSymbol,'\\leftrightarrow ','&harr;');
// By pesasa: change leftrightarrow to binary operator
LatexCmds.harr = LatexCmds.lrarr = LatexCmds.leftrightarrow =
  bind(BinaryOperator,'\\leftrightarrow ','&harr;');

LatexCmds.iff = bind(BinaryOperator,'\\Leftrightarrow ','&hArr;');

//LatexCmds.hArr = LatexCmds.lrArr = LatexCmds.Leftrightarrow =
//  bind(VanillaSymbol,'\\Leftrightarrow ','&hArr;');
// By pesasa: change Leftrightarrow to binary operator
LatexCmds.hArr = LatexCmds.lrArr = LatexCmds.Leftrightarrow =
  bind(BinaryOperator,'\\Leftrightarrow ','&hArr;');

LatexCmds.Re = LatexCmds.Real = LatexCmds.real = bind(VanillaSymbol,'\\Re ','&real;');

LatexCmds.Im = LatexCmds.imag =
LatexCmds.image = LatexCmds.imagin = LatexCmds.imaginary = LatexCmds.Imaginary =
  bind(VanillaSymbol,'\\Im ','&image;');

LatexCmds.part = LatexCmds.partial = bind(VanillaSymbol,'\\partial ','&part;');

LatexCmds.inf = LatexCmds.infin = LatexCmds.infty = LatexCmds.infinity =
  bind(VanillaSymbol,'\\infty ','&infin;');

LatexCmds.alef = LatexCmds.alefsym = LatexCmds.aleph = LatexCmds.alephsym =
  bind(VanillaSymbol,'\\aleph ','&alefsym;');

LatexCmds.xist = //LOL
LatexCmds.xists = LatexCmds.exist = LatexCmds.exists =
  bind(VanillaSymbol,'\\exists ','&exist;');

//LatexCmds.and = LatexCmds.land = LatexCmds.wedge =
//  bind(VanillaSymbol,'\\wedge ','&and;');
// By pesasa: change land to binary operator
LatexCmds.and = LatexCmds.land = LatexCmds.wedge =
  bind(BinaryOperator,'\\wedge ','&and;');

//LatexCmds.or = LatexCmds.lor = LatexCmds.vee = bind(VanillaSymbol,'\\vee ','&or;');
// By pesasa: change lor to binary operator
LatexCmds.or = LatexCmds.lor = LatexCmds.vee = bind(BinaryOperator,'\\vee ','&or;');

LatexCmds.o = LatexCmds.O =
LatexCmds.empty = LatexCmds.emptyset =
LatexCmds.oslash = LatexCmds.Oslash =
LatexCmds.nothing = LatexCmds.varnothing =
  bind(BinaryOperator,'\\varnothing ','&empty;');

LatexCmds.cup = LatexCmds.union = bind(BinaryOperator,'\\cup ','&cup;');

LatexCmds.cap = LatexCmds.intersect = LatexCmds.intersection =
  bind(BinaryOperator,'\\cap ','&cap;');

LatexCmds.deg = LatexCmds.degree = bind(VanillaSymbol,'\\deg ','&deg;');

LatexCmds.ang = LatexCmds.angle = bind(VanillaSymbol,'\\angle ','&ang;');


function NonItalicizedFunction(replacedFragment, fn) {
  Symbol.call(this, '\\'+fn+' ', '<span>'+fn+'</span>');
}
_ = NonItalicizedFunction.prototype = new Symbol;
_.respace = function()
{
  this.jQ[0].className =
    (this.next instanceof SupSub || this.next instanceof Bracket) ?
    '' : 'non-italicized-function';
};

LatexCmds.ln =
LatexCmds.lg =
LatexCmds.log =
LatexCmds.span =
LatexCmds.proj =
LatexCmds.det =
LatexCmds.dim =
LatexCmds.min =
LatexCmds.max =
LatexCmds.mod =
LatexCmds.lcm =
LatexCmds.gcd =
LatexCmds.gcf =
LatexCmds.hcf =
LatexCmds.lim = NonItalicizedFunction;

(function() {
  var trig = ['sin', 'cos', 'tan', 'sec', 'cosec', 'csc', 'cotan', 'cot'];
  for (var i in trig) {
    LatexCmds[trig[i]] =
    LatexCmds[trig[i]+'h'] =
    LatexCmds['a'+trig[i]] = LatexCmds['arc'+trig[i]] =
    LatexCmds['a'+trig[i]+'h'] = LatexCmds['arc'+trig[i]+'h'] =
      NonItalicizedFunction;
  }
}());

/********************************************
 * Cursor and Selection "singleton" classes
 *******************************************/

/* The main thing that manipulates the Math DOM. Makes sure to manipulate the
HTML DOM to match. */

/* Sort of singletons, since there should only be one per editable math
textbox, but any one HTML document can contain many such textboxes, so any one
JS environment could actually contain many instances. */

//A fake cursor in the fake textbox that the math is rendered in.
function Cursor(root) {
  this.parent = this.root = root;
  var jQ = this.jQ = this._jQ = $('<span class="cursor">&zwj;</span>');

  //closured for setInterval
  this.blink = function(){ jQ.toggleClass('blink'); }
}
_ = Cursor.prototype;
_.prev = 0;
_.next = 0;
_.parent = 0;
_.show = function() {
  this.jQ = this._jQ.removeClass('blink');
  if ('intervalId' in this) //already was shown, just restart interval
    clearInterval(this.intervalId);
  else { //was hidden and detached, insert this.jQ back into HTML DOM
    if (this.next) {
      if (this.selection && this.selection.prev === this.prev)
        this.jQ.insertBefore(this.selection.jQ);
      else
        this.jQ.insertBefore(this.next.jQ.first());
    }
    else
      this.jQ.appendTo(this.parent.jQ);
    this.parent.focus();
  }
  this.intervalId = setInterval(this.blink, 500);
  return this;
};
_.hide = function() {
  if ('intervalId' in this)
    clearInterval(this.intervalId);
  delete this.intervalId;
  this.jQ.detach();
  this.jQ = $();
  return this;
};
_.redraw = function() {
  for (var ancestor = this.parent; ancestor; ancestor = ancestor.parent)
    if (ancestor.redraw)
      ancestor.redraw();
};
_.insertAt = function(parent, prev, next) {
  var old_parent = this.parent;

  this.parent = parent;
  this.prev = prev;
  this.next = next;

  old_parent.blur(); //blur may need to know cursor's destination
};
_.insertBefore = function(el) {
  this.insertAt(el.parent, el.prev, el)
  this.parent.jQ.addClass('hasCursor');
  this.jQ.insertBefore(el.jQ.first());
  return this;
};
_.insertAfter = function(el) {
  this.insertAt(el.parent, el, el.next);
  this.parent.jQ.addClass('hasCursor');
  this.jQ.insertAfter(el.jQ.last());
  return this;
};
_.prependTo = function(el) {
  this.insertAt(el, 0, el.firstChild);
  if (el.textarea) //never insert before textarea
    this.jQ.insertAfter(el.textarea);
  else
    this.jQ.prependTo(el.jQ);
  el.focus();
  return this;
};
_.appendTo = function(el) {
  this.insertAt(el, el.lastChild, 0);
  this.jQ.appendTo(el.jQ);
  el.focus();
  return this;
};
_.hopLeft = function() {
  this.jQ.insertBefore(this.prev.jQ.first());
  this.next = this.prev;
  this.prev = this.prev.prev;
  return this;
};
_.hopRight = function() {
  this.jQ.insertAfter(this.next.jQ.last());
  this.prev = this.next;
  this.next = this.next.next;
  return this;
};
_.moveLeft = function() {
  if (this.selection)
    this.insertBefore(this.selection.prev.next || this.parent.firstChild).clearSelection();
  else {
    if (this.prev) {
      if (this.prev.lastChild)
        this.appendTo(this.prev.lastChild)
      else
        this.hopLeft();
    }
    else { //we're at the beginning of a block
      if (this.parent.prev)
        this.appendTo(this.parent.prev);
      else if (this.parent !== this.root)
        this.insertBefore(this.parent.parent);
      //else we're at the beginning of the root, so do nothing.
    }
  }
  return this.show();
};
_.moveRight = function() {
  if (this.selection)
    this.insertAfter(this.selection.next.prev || this.parent.lastChild).clearSelection();
  else {
    if (this.next) {
      if (this.next.firstChild)
        this.prependTo(this.next.firstChild)
      else
        this.hopRight();
    }
    else { //we're at the end of a block
      if (this.parent.next)
        this.prependTo(this.parent.next);
      else if (this.parent !== this.root)
        this.insertAfter(this.parent.parent);
      //else we're at the end of the root, so do nothing.
    }
  }
  return this.show();
};
_.seek = function(target, pageX, pageY) {
  var cursor = this.clearSelection();
  if (target.hasClass('empty')) {
    cursor.prependTo(target.data(jQueryDataKey).block);
    return cursor;
  }

  var data = target.data(jQueryDataKey);
  if (data) {
    //if clicked a symbol, insert at whichever side is closer
    if (data.cmd && !data.block) {
      if (target.outerWidth() > 2*(pageX - target.offset().left))
        cursor.insertBefore(data.cmd);
      else
        cursor.insertAfter(data.cmd);

      return cursor;
    }
  }
  //if no MathQuill data, try parent, if still no, forget it
  else {
    target = target.parent();
    data = target.data(jQueryDataKey);
    if (!data)
      data = {block: cursor.root};
  }

  if (data.cmd)
    cursor.insertAfter(data.cmd);
  else
    cursor.appendTo(data.block);

  //move cursor to position closest to click
  var dist = cursor.offset().left - pageX, prevDist;
  do {
    cursor.moveLeft();
    prevDist = dist;
    dist = cursor.offset().left - pageX;
  }
  while (dist > 0 && (cursor.prev || cursor.parent !== cursor.root));

  if (-dist > prevDist)
    cursor.moveRight();

  return cursor;
};
_.offset = function() {
  //in Opera 11.62, .getBoundingClientRect() and hence jQuery::offset()
  //returns all 0's on inline elements with negative margin-right (like
  //the cursor) at the end of their parent, so temporarily remove the
  //negative margin-right when calling jQuery::offset()
  //Opera bug DSK-360043
  //http://bugs.jquery.com/ticket/11523
  //https://github.com/jquery/jquery/pull/717
  var jQ = this.jQ.removeClass('cursor'),
    offset = jQ.offset();
  jQ.addClass('cursor');
  return offset;
};
_.writeLatex = function(latex) {
  this.deleteSelection();
  latex = ( latex && latex.match(/\\text\{([^}]|\\\})*\}|\\:|\\;|\\,|\\%|\\&|\\[a-z]*|[^\s]/ig) ) || 0;
  (function writeLatexBlock(cursor) {
    while (latex.length) {
      var token = latex.shift(); //pop first item
      if (!token || token === '}' || token === ']') return;

      var cmd;
      if (token.slice(0, 6) === '\\text{') {
        cmd = new TextBlock(token.slice(6, -1));
        cursor.insertNew(cmd).insertAfter(cmd);
        continue; //skip recursing through children
      }
      else if (token === '\\left' || token === '\\right') { //FIXME HACK: implement real \left and \right LaTeX commands, rather than special casing them here
        token = latex.shift();
        if (token === '\\')
          token = latex.shift();

        cursor.insertCh(token);
        cmd = cursor.prev || cursor.parent.parent;

        if (cursor.prev) //was a close-paren, so break recursion
          return;
        else //was an open-paren, hack to put the following latex
          latex.unshift('{'); //in the ParenBlock in the math DOM
      }
      else if (/^\\[a-z:;,%&]+$/i.test(token)) {
        token = token.slice(1);
        var cmd = LatexCmds[token];
        if (cmd) {
          cmd = new cmd(undefined, token);
          if (latex[0] === '[' && cmd.optional_arg_command) {
            //e.g. \sqrt{m} -> SquareRoot, \sqrt[n]{m} -> NthRoot
            token = cmd.optional_arg_command;
            cmd = new LatexCmds[token](undefined, token, latex);
          }
          cursor.insertNew(cmd);
        }
        else {
          cmd = new TextBlock(token);
          cursor.insertNew(cmd).insertAfter(cmd);
          continue; //skip recursing through children
        }
      }
      else {
        if (token.match(/[a-eg-zA-Z]/)) //exclude f because want florin
          cmd = new Variable(token);
        else if (token.match(/[:;,]/)) // these are backslash-sequences, not symbols (this is a hack)
          cmd = new VanillaSymbol(token);
        else if (cmd = LatexCmds[token])
          cmd = new cmd;
        else
          cmd = new VanillaSymbol(token);

        cursor.insertNew(cmd);
      }
      cmd.eachChild(function(child) {
        cursor.appendTo(child);
        var token = latex.shift();
        if (!token) return false;

        if (token === '{' || token === '[')
          writeLatexBlock(cursor);
        else
          cursor.insertCh(token);
      });
      cursor.insertAfter(cmd);
    }
  }(this));
  return this.hide();
};
_.write = function(ch) {
  return this.show().insertCh(ch);
};
_.insertCh = function(ch) {
  if (this.selection) {
    //gotta do this before this.selection is mutated by 'new cmd(this.selection)'
    this.prev = this.selection.prev;
    this.next = this.selection.next;
  }

  var cmd;
  if (ch.match(/^[a-eg-zA-Z]$/)) //exclude f because want florin
    cmd = new Variable(ch);
  else if ((cmd = CharCmds[ch] || LatexCmds[ch]) &&
      !(ch.match(/[:;,]/))) // exclude spaces symbols here, they are backslash-sequences (this is also a hack)
    cmd = new cmd(this.selection, ch);
  else
    cmd = new VanillaSymbol(ch);

  if (this.selection) {
    if (cmd instanceof Symbol)
      this.selection.remove();
    delete this.selection;
  }

  return this.insertNew(cmd);
};
_.insertNew = function(cmd) {
  cmd.insertAt(this);
  return this;
};
_.insertCmd = function(latexCmd, replacedFragment) {
  var cmd = LatexCmds[latexCmd];
  if (cmd) {
    cmd = new cmd(replacedFragment, latexCmd);
    this.insertNew(cmd);
    if (cmd instanceof Symbol && replacedFragment)
      replacedFragment.remove();
  }
  else {
    cmd = new TextBlock(latexCmd);
    cmd.firstChild.focus = function(){ delete this.focus; return this; };
    this.insertNew(cmd).insertAfter(cmd);
    if (replacedFragment)
      replacedFragment.remove();
  }
  return this;
};
_.unwrapGramp = function() {
  var gramp = this.parent.parent,
    greatgramp = gramp.parent,
    prev = gramp.prev,
    cursor = this;

  gramp.eachChild(function(uncle) {
    if (uncle.isEmpty()) return;

    uncle.eachChild(function(cousin) {
      cousin.parent = greatgramp;
      cousin.jQ.insertBefore(gramp.jQ.first());
    });
    uncle.firstChild.prev = prev;
    if (prev)
      prev.next = uncle.firstChild;
    else
      greatgramp.firstChild = uncle.firstChild;

    prev = uncle.lastChild;
  });
  prev.next = gramp.next;
  if (gramp.next)
    gramp.next.prev = prev;
  else
    greatgramp.lastChild = prev;

  if (!this.next) { //then find something to be next to insertBefore
    if (this.prev)
      this.next = this.prev.next;
    else {
      while (!this.next) {
        this.parent = this.parent.next;
        if (this.parent)
          this.next = this.parent.firstChild;
        else {
          this.next = gramp.next;
          this.parent = greatgramp;
          break;
        }
      }
    }
  }
  if (this.next)
    this.insertBefore(this.next);
  else
    this.appendTo(greatgramp);

  gramp.jQ.remove();

  if (gramp.prev)
    gramp.prev.respace();
  if (gramp.next)
    gramp.next.respace();
};
_.backspace = function() {
  if (this.deleteSelection());
  else if (this.prev) {
    if (this.prev.isEmpty())
      this.prev = this.prev.remove().prev;
    else
      this.selectLeft();
  }
  else if (this.parent !== this.root) {
    if (this.parent.parent.isEmpty())
      return this.insertAfter(this.parent.parent).backspace();
    else
      this.unwrapGramp();
  }

  if (this.prev)
    this.prev.respace();
  if (this.next)
    this.next.respace();
  this.redraw();

  return this;
};
_.deleteForward = function() {
  if (this.deleteSelection());
  else if (this.next) {
    if (this.next.isEmpty())
      this.next = this.next.remove().next;
    else
      this.selectRight();
  }
  else if (this.parent !== this.root) {
    if (this.parent.parent.isEmpty())
      return this.insertBefore(this.parent.parent).deleteForward();
    else
      this.unwrapGramp();
  }

  if (this.prev)
    this.prev.respace();
  if (this.next)
    this.next.respace();
  this.redraw();

  return this;
};
_.selectFrom = function(anticursor) {
  //find ancestors of each with common parent
  var oneA = this, otherA = anticursor; //one ancestor, the other ancestor
  loopThroughAncestors: while (true) {
    for (var oneI = this; oneI !== oneA.parent.parent; oneI = oneI.parent.parent) //one intermediate, the other intermediate
      if (oneI.parent === otherA.parent) {
        left = oneI;
        right = otherA;
        break loopThroughAncestors;
      }

    for (var otherI = anticursor; otherI !== otherA.parent.parent; otherI = otherI.parent.parent)
      if (oneA.parent === otherI.parent) {
        left = oneA;
        right = otherI;
        break loopThroughAncestors;
      }

    if (oneA.parent.parent)
      oneA = oneA.parent.parent;
    if (otherA.parent.parent)
      otherA = otherA.parent.parent;
  }
  //figure out which is left/prev and which is right/next
  var left, right, leftRight;
  if (left.next !== right) {
    for (var next = left; next; next = next.next) {
      if (next === right.prev) {
        leftRight = true;
        break;
      }
    }
    if (!leftRight) {
      leftRight = right;
      right = left;
      left = leftRight;
    }
  }
  this.hide().selection = new Selection(
    left.parent,
    left.prev,
    right.next
  );
  this.insertAfter(right.next.prev || right.parent.lastChild);
  this.root.selectionChanged();
};
_.selectLeft = function() {
  if (this.selection) {
    if (this.selection.prev === this.prev) { //if cursor is at left edge of selection;
      if (this.prev) { //then extend left if possible
        this.hopLeft().next.jQ.prependTo(this.selection.jQ);
        this.selection.prev = this.prev;
      }
      else if (this.parent !== this.root) //else level up if possible
        this.insertBefore(this.parent.parent).selection.levelUp();
    }
    else { //else cursor is at right edge of selection, retract left
      this.prev.jQ.insertAfter(this.selection.jQ);
      this.hopLeft().selection.next = this.next;
      if (this.selection.prev === this.prev) {
        this.deleteSelection();
        return;
      }
    }
  }
  else {
    if (this.prev)
      this.hopLeft();
    else //end of a block
      if (this.parent !== this.root)
        this.insertBefore(this.parent.parent);
      else
        return;

    this.hide().selection = new Selection(this.parent, this.prev, this.next.next);
  }
  this.root.selectionChanged();
};
_.selectRight = function() {
  if (this.selection) {
    if (this.selection.next === this.next) { //if cursor is at right edge of selection;
      if (this.next) { //then extend right if possible
        this.hopRight().prev.jQ.appendTo(this.selection.jQ);
        this.selection.next = this.next;
      }
      else if (this.parent !== this.root) //else level up if possible
        this.insertAfter(this.parent.parent).selection.levelUp();
    }
    else { //else cursor is at left edge of selection, retract right
      this.next.jQ.insertBefore(this.selection.jQ);
      this.hopRight().selection.prev = this.prev;
      if (this.selection.next === this.next) {
        this.deleteSelection();
        return;
      }
    }
  }
  else {
    if (this.next)
      this.hopRight();
    else //end of a block
      if (this.parent !== this.root)
        this.insertAfter(this.parent.parent);
      else
        return;

    this.hide().selection = new Selection(this.parent, this.prev.prev, this.next);
  }
  this.root.selectionChanged();
};
_.clearSelection = function() {
  if (this.show().selection) {
    this.selection.clear();
    delete this.selection;
    this.root.selectionChanged();
  }
  return this;
};
_.deleteSelection = function() {
  if (!this.show().selection) return false;

  this.prev = this.selection.prev;
  this.next = this.selection.next;
  this.selection.remove();
  delete this.selection;
  this.root.selectionChanged();
  return true;
};

function Selection(parent, prev, next) {
  MathFragment.apply(this, arguments);
}
_ = Selection.prototype = new MathFragment;
_.jQinit = function(children) {
  this.jQ = children.wrapAll('<span class="selection"></span>').parent();
    //can't do wrapAll(this.jQ = $(...)) because wrapAll will clone it
};
_.levelUp = function() {
  this.clear().jQinit(this.parent.parent.jQ);

  this.prev = this.parent.parent.prev;
  this.next = this.parent.parent.next;
  this.parent = this.parent.parent.parent;

  return this;
};
_.clear = function() {
  this.jQ.replaceWith(this.jQ.children());
  return this;
};
_.blockify = function() {
  this.jQ.replaceWith(this.jQ = this.jQ.children());
  return MathFragment.prototype.blockify.call(this);
};
_.detach = function() {
  var block = MathFragment.prototype.blockify.call(this);
  this.blockify = function() {
    this.jQ.replaceWith(block.jQ = this.jQ = this.jQ.children());
    return block;
  };
  return this;
};

/*********************************************************
 * The actual jQuery plugin and document ready handlers.
 ********************************************************/

//The publicy exposed method of jQuery.prototype, available (and meant to be
//called) on jQuery-wrapped HTML DOM elements.
$.fn.mathquill = function(cmd, latex) {
  switch (cmd) {
  case 'redraw':
    this.find(':not(:has(:first))').each(function() {
      var data = $(this).data(jQueryDataKey);
      if (data && (data.cmd || data.block)) Cursor.prototype.redraw.call(data.cmd || data.block);
    });
    return this;
  case 'revert':
    return this.each(function() {
      var data = $(this).data(jQueryDataKey);
      if (data && data.revert)
        data.revert();
    });
  case 'latex':
    if (arguments.length > 1) {
      return this.each(function() {
        var data = $(this).data(jQueryDataKey);
        if (data && data.block && data.block.renderLatex)
          data.block.renderLatex(latex);
      });
    }

    var data = this.data(jQueryDataKey);
    return data && data.block && data.block.latex();
  case 'text':
    var data = this.data(jQueryDataKey);
    return data && data.block && data.block.text();
  case 'html':
    return this.html().replace(/ ?hasCursor|hasCursor /, '')
      .replace(/ class=(""|(?= |>))/g, '')
      .replace(/<span class="?cursor( blink)?"?><\/span>/i, '')
      .replace(/<span class="?textarea"?><textarea><\/textarea><\/span>/i, '');
  case 'write':
    if (arguments.length > 1)
      return this.each(function() {
        var data = $(this).data(jQueryDataKey),
          block = data && data.block,
          cursor = block && block.cursor;

        if (cursor)
          cursor.writeLatex(latex).parent.blur();
      });
  case 'cmd':
    if (arguments.length > 1)
      return this.each(function() {
        var data = $(this).data(jQueryDataKey),
          block = data && data.block,
          cursor = block && block.cursor;

        if (cursor) {
          cursor.show();
          if (/^\\[a-z]+$/i.test(latex)) {
            if (cursor.selection) {
              //gotta do cursor before cursor.selection is mutated by 'new cmd(cursor.selection)'
              cursor.prev = cursor.selection.prev;
              cursor.next = cursor.selection.next;
            }
            cursor.insertCmd(latex.slice(1), cursor.selection);
            delete cursor.selection;
          }
          else
            cursor.insertCh(latex);
          cursor.hide().parent.blur();
        }
      });
  default:
    var textbox = cmd === 'textbox',
      editable = textbox || cmd === 'editable',
      RootBlock = textbox ? RootTextBlock : RootMathBlock;
    return this.each(function() {
      $(this).attr('tabindex', '-1');
      createRoot($(this), new RootBlock, textbox, editable);
    });
  }
};

//on document ready, mathquill-ify all `<tag class="mathquill-*">latex</tag>`
//elements according to their CSS class.
//$(function() {
//  $('.mathquill-editable:not(.mathquill-rendered-math)').mathquill('editable');
//  $('.mathquill-textbox:not(.mathquill-rendered-math)').mathquill('textbox');
//  $('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
//});


}());
