/**
 * Requirements:
 * - jQuery
 * - wikiEngine.js
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
    typeof(wikify) === 'undefined' && wikify.apply;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - jquery.commentor.js
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(jQuery.fn.commentor) === 'undefined' && jQuery.fn.commentor.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - Mathquill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    
    var WikiElement = function(place, options){
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlersCommon();
        this.show();
    };
    
    WikiElement.prototype.init = function(options) {
        this.type = 'wikielement';
        this.metadata = options.metadata;
        this.data = options.data;
        this.settings = options.settings;
        this.setMode(options.settings.mode);
        if ($.fn.commentor && typeof($.fn.commentor) === 'function') {
            this.selfreviewable = true;
            this.place.addClass('selfreviewable');
        };
        this.name = options.name || this.place.attr('data-element-name');
        this.reviewdata = false;
        this.getMargin = false;
    };
    
    WikiElement.prototype.setMode = function(mode) {
        this.settings.mode = mode || 'view';
        var modesettings = this.modes[mode] || this.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
        this.noteviewable = modesettings.noteviewable;
    };
    
    WikiElement.prototype.changeMode = function(mode) {
        this.setMode(mode);
        this.show();
    };
    
    WikiElement.prototype.show = function() {
        this.place.empty().html(this.templates.html);
        var place = this.place.children('.wikielement-content');
        var text = this.data.text.replace(/\n\n+/g, '\n\n');
        wikify(text, place[0]);
        if (this.reviewable || this.noteviewable) {
            this.editReview();
        } else if (this.reviewdata) {
            this.showReview();
        };
    };
    
    WikiElement.prototype.initHandlersCommon = function() {
        var element = this;
        this.place.on('setmode', function(event, data) {
            event.stopPropagation();
            if (data in element.modes) {
                element.changeMode(data);
            };
        }).on('setnoteview', function(event, data) {
            event.stopPropagation();
            if (data && element.selfnotereviewable) {
                element.setReview(data);
            };
            if (element.noteviewable) {
                element.editReview();
            } else if (!element.editable) {
                element.showReview();
            };
        }).on('showselfreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (data && element.selfreviewable) {
                element.setReview(data);
                if (element.reviewable) {
                    element.editReview();
                } else if (!element.editable) {
                    element.showReview();
                };
            } else {
                element.unhideReview();
            };
            //element.place.trigger('showcomments');
        }).on('hideselfreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (element.selfreviewable) {
                element.hideReview();
            };
        }).on('commentorchanged', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            element.setReview(data);
            if (element.reviewable) {
                element.reviewChanged();
            } else if (element.noteviewable) {
                element.noteviewChanged();
            };
        }).on('reply_getmarginplace', function(event, data) {
            event.stopPropagation();
            if (typeof(data.getMargin) === 'function') {
                element.getMargin = data.getMargin;
            };
        });
    };

    /******
     * Trigger 'review_changed' -event
     ******/
    WikiElement.prototype.reviewChanged = function(){
        this.place.trigger('review_changed');
        var data = {
            id: this.name,
            reviewdata: this.reviewdata
        };
        this.place.trigger('elementreview_changed', data);
    };
    
    /******
     * Trigger 'noteview_changed' -event
     ******/
    WikiElement.prototype.noteviewChanged = function(){
        this.place.trigger('noteview_changed');
        var data = {
            id: this.reviewdata.data.id,
            anchorid: this.name,
            contentType: 'marginnote',
            lang: 'common',
            sendto: [],
            sendfrom: this.settings.username,
            data: this.reviewdata.data
        };
        this.place.trigger('marginnote_changed', data);
    };

    /******
     * Set review data
     * @param {Object} revdata - review data
     ******/
    WikiElement.prototype.setReview = function(revdata) {
        this.reviewdata = $.extend(true, {data: {}}, revdata);
    };
    
    /******
     * Show review
     ******/
    WikiElement.prototype.showReview = function(){
        if (this.selfreviewable) {
            var appid = this.settings.gotolink && this.settings.gotolink.appname || '';
            var settings = {
                margin: (this.getMargin && this.settings.usemargin ? this.getMargin(this.name, appid) : this.place.children('.wikielement-margin')),
                usemargin: this.settings.usemargin && (typeof(this.getMargin) === 'function'),
                mode: 'view'
            };
            var rdata = $.extend(true, {settings: settings}, this.reviewdata);
            this.place.children('.wikielement-content').commentor(rdata);
        };
    };
    
    /******
     * Unhide review
     ******/
    WikiElement.prototype.unhideReview = function(){
        if (this.selfreviewable) {
            this.place.children('.wikielement-content').trigger('showcomments');
        };
    };
    
    /******
     * Hide review
     ******/
    WikiElement.prototype.hideReview = function(){
        if (this.selfreviewable) {
            this.place.children('.wikielement-content').trigger('hidecomments');
        };
    };
    
    /******
     * Edit review
     ******/
    WikiElement.prototype.editReview = function(){
        if (this.selfreviewable) {
            var appid = this.settings.gotolink && this.settings.gotolink.appname || '';
            var settings = {
                margin: (this.getMargin && this.settings.usemargin ? this.getMargin(this.name, appid) : this.place.children('.wikielement-margin')),
                usemargin: (this.settings.usemargin && typeof(this.getMargin) === 'function'),
                mode: 'edit'
            };
            var rdata = $.extend(true, {settings: settings}, this.reviewdata);
            this.place.children('.wikielement-content').trigger('commentordestroy').commentor(rdata);
        };
    };
    
    
    WikiElement.prototype.defaults = {
        type: 'wikielement',
        metadata: {},
        data: {
            text: ''
        },
        settings: {
            mode: 'view'
        }
    };
    
    WikiElement.prototype.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            noteviewable: false
        },
        noteview: {
            editable: false,
            authorable: false,
            reviewable: false,
            noteviewable: true
        }
    }
    
    WikiElement.prototype.templates = {
        html: [
            '<div class="wikielement-content"></div>',
            '<div class="wikielement-margin"></div>'
        ].join('\n')
    };
    
    
    WikiElement.elementinfo = {
        type : 'wikielement',
        elementtype : 'elements',
        jquery : 'wikielement',
        name : 'TWikielement',
        icon : '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-functiongraph"><path style="stroke: none;" d="M1 17 l26 0 l0 -2 l3 2.5 l-3 2.5 l0 -2 l-26 0z M13 29 l0 -26 l-2 0 l2.5 -3 l2.5 3 l-2 0 l0 26z M1 29 q5 -28 14 -17 q7 7 12 -10 l1 0 q-5 19 -14 10 q-8 -8 -12 17z" /></svg>',
        description : {
            en : 'Wiki text element',
            fi : 'Wiki-tekstielementti'
        },
        classes : [ 'viewonly' ]
    }

    WikiElement.plaintextinfo = {
        type : 'plaintext',
        elementtype : 'elements',
        jquery : 'wikielement',
        name : 'Plaintextassignment',
        icon : '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-functiongraph"><path style="stroke: none;" d="M1 17 l26 0 l0 -2 l3 2.5 l-3 2.5 l0 -2 l-26 0z M13 29 l0 -26 l-2 0 l2.5 -3 l2.5 3 l-2 0 l0 26z M1 29 q5 -28 14 -17 q7 7 12 -10 l1 0 q-5 19 -14 10 q-8 -8 -12 17z" /></svg>',
        description : {
            en : 'Plain text assignment',
            fi : 'Pelkkä teksti -tehtävä'
        },
        classes : [ 'viewonly' ]
    }

    // For possible listing/registration to elementset
    if (typeof ($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', WikiElement.elementinfo);
        $.fn.elementset('addelementtype', WikiElement.plaintextinfo);
    }
    
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', WikiElement.elementinfo);
        $.fn.elementpanel('addelementtype', WikiElement.plaintextinfo);
    }
    
    var convert2new = function(params){
        try {
            var content = params.data.contents[0].id;
            var data = params.data.contentdata[content];
            data.metadata = params.metadata;
            params = data;
        } catch (err) {
            console.log('"plaintext"-element not in expected format');
        }
        return params;
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            if (params.type === 'plaintext') {
                params = convert2new(params);
            }
            return this.each(function(){
                var wikielem = new WikiElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[wikielementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.wikielement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in wikielement.');
            return false;
        }
    }
    
})(jQuery);