/***
 * VNotifier
 * A tool for notifications
 * Petri Salmela <pesasa@iki.fi>
 ***/
//{{{

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function ($) {
    { /*** jQuery plugin ***/
        $.fn.vnotifier = function(options){
            if (methods[options]){
                return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
            } else if (typeof(options) === 'object' || !options) {
                return methods.init.apply(this, arguments);
            } else {
                $.error( 'Method ' +  method + ' does not exist on VNotifier' );
                return this;
            }
        }
        
        var methods = {
            init: function(params){
                params = $.extend(true, {
                }, params);
                var notifier = new Notifier(this, params);
            },
            getdata: function(params){
                var $place = $(this);
                $place.trigger('getdata');
                var data = $place.data('[[notifierdata]]');
                return data;
            }
        }
    }
    
    /**
     * Helper functions
     */
    
    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Class for Notifier
     * @constructor
     * @param {jQuery} place - place for navbar
     * @param {Object} options - settings for navbar
     ******/
    var Notifier = function(place, options){
        this.place = $(place);
        options = $.extend(true, {}, Notifier.defaults, options);
        this.setStyles();
        this.initData(options);
        this.setAttrs();
        this.initHandlers();
        this.show();
    };
    
    /******
     * Set attributes
     ******/
    Notifier.prototype.setAttrs = function(){
        this.place.addClass('vnotifier-wrapper');
        if (this.background) {
            this.place.addClass('ffwidget-background');
        }
        switch (this.align){
            case 'right':
                this.place.addClass('vnotifier-alignright');
                break;
            case 'left':
                this.place.addClass('vnotifier-alignleft');
                break;
            default:
                break;
        };
    }
    
    /******
     * Set stylesheets
     ******/
    Notifier.prototype.setStyles = function(){
        if ($('head style#vnotifier-style').length === 0) {
            $('head').append('<style id="vnotifier-style" type="text/css">'+Notifier.styles+'</style>')
        };
    };
    
    /******
     * Init data
     ******/
    Notifier.prototype.initData = function(options){
        this.classes = {};
        this.classlist = [];
        this.size = options.size;
        this.level = options.level;
        this.align = options.align;
        this.background = options.background;
        var notifclass, classname, classtype;
        for (var i = 0, len = options.classes.length; i < len; i++){
            notifclass = options.classes[i];
            classname = notifclass.name;
            classtype = notifclass.type || 'list';
            if (typeof(this.classTypes[classtype]) === 'function') {
                this.classes[classname] = new this.classTypes[classtype](notifclass);
                this.classlist.push(classname);
            };
        };
    };
    
    Notifier.prototype.classTypes = {};
    
    /******
     * Init handlers
     ******/
    Notifier.prototype.initHandlers = function(){
        var notier = this;
        this.place.off('getdata').on('getdata', function(event, data){
            event.stopPropagation();
            var result = notier.getData();
            notier.place.data('[[notifierdata]]', result);
        });
        this.place.off('click', '.vnotifier-classbutton').on('click', '.vnotifier-classbutton', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var itemlist = button.closest('li').children('.vnotifier-notiflist');
            var alllists = button.closest('.vnotifier-classlist').find('.vnotifier-notiflist');
            var ison = itemlist.hasClass('vnotifier-show');
            alllists.removeClass('vnotifier-show');
            if (!ison) {
                itemlist.addClass('vnotifier-show');
            };
        });
        this.place.off('newnotification').on('newnotification', function(event, data){
            notier.addNotification(data);
        });
        this.place.off('removenotification').on('removenotification', function(event, data) {
            notier.removeNotification(data);
        });
        this.place.off('click', '.vnotifier-notification').on('click', '.vnotifier-notification', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var item = $(this);
            var classitem = item.closest('.vnotifier-classitem');
            var notifclass = classitem.attr('data-notifclass');
            var eventname = notier.getClassEvent(notifclass);
            var notifid = item.attr('data-notification-id');
            var notifdata = notier.getNotifData(notifclass, notifid);
            var resultdata = {id: notifid};
            if (typeof(notifdata) !== 'undefined') {
                resultdata.data = notifdata;
            }
            item.trigger(eventname, resultdata);
            notier.acknowledgeNotification(notifclass, notifid);
        });
        this.place.off('mouseleave', '.vnotifier-notiflist.vnotifier-show').on('mouseleave', '.vnotifier-notiflist.vnotifier-show', function(event, data){
            // If mouse leaves the notification (class)list, close the list with dealy of 2 secons.
            event.stopPropagation();
            event.preventDefault();
            var place = $(this);
            var classname = place.parent().attr('data-notifclass');
            notier.mouseleave = {classname: classname};
            if (event.target === place[0]) {
                notier.mouseleave.timeout = setTimeout(function(){
                    var place = notier.place.find('[data-notifclass="'+notier.mouseleave.classname+'"] .vnotifier-notiflist');
                    place.removeClass('vnotifier-show');
                }, 2000);
            };
        });
        this.place.off('mouseenter', '.vnotifier-notiflist.vnotifier-show').on('mouseenter', '.vnotifier-notiflist.vnotifier-show', function(event, data){
            // If the mouse returns the list, cancel the closing of the list.
            event.stopPropagation();
            event.preventDefault();
            if (notier.mouseleave) {
                clearTimeout(notier.mouseleave.timeout);
            };
        });
    }
    
    /******
     * Show the notifier
     ******/
    Notifier.prototype.show = function(){
        this.place.html(Notifier.templates.html);
        var classlist = this.place.children('.vnotifier-classlist');
        classlist.empty();
        var classelem, classname, classitem;
        for (var i = 0, len = this.classlist.length; i < len; i++) {
            classname = this.classlist[i];
            classitem = this.classes[classname];
            if (classitem.getLevel() < this.level) {
                classelem = $('<li class="vnotifier-classitem" data-notifclass="'+escapeHTML(classname)+'"></li>');
                classlist.append(classelem);
                classitem.show(classelem);
            };
        };
    };
    
    /******
     * Add notification
     ******/
    Notifier.prototype.addNotification = function(data){
        var notifclass;
        if (data.nclass && this.classes[data.nclass]) {
            notifclass = this.classes[data.nclass];
            notifclass.addNotification(data);
        };
    };
    
    /**
     * Remove notification
     * @param {Object} data   Identifications for removing notification
     * @param {String} data.nclass   Notification class
     * @param {String} data.id       Notification id
     */
    Notifier.prototype.removeNotification = function(data) {
        this.acknowledgeNotification(data.nclass, data.id);
    };
    
    /******
     * Acknowledge the notification (by click)
     ******/
    Notifier.prototype.acknowledgeNotification = function(notifclass, notifid){
        var notclass;
        if (this.classes[notifclass]) {
            notclass = this.classes[notifclass];
            notclass.acknowledgeNotification(notifid);
        };
    };
    
    /******
     * Get the name of event for given class
     ******/
    Notifier.prototype.getClassEvent = function(classname){
        var notifclass = this.classes[classname];
        return notifclass.getEvent();
    }
    
    /******
     * Get data of given notification in given class
     ******/
    Notifier.prototype.getNotifData = function(nclass, nid){
        var notifclass = this.classes[nclass];
        return notifclass.getNotifData(nid);
    }
    
    /******
     * Get data of the whole notifier
     ******/
    Notifier.prototype.getData = function(){
        var result = {
            classes: [],
            level: this.level,
            align: this.align,
            background: this.background
        };
        var classname;
        for (var i = 0, len = this.classlist.length; i < len; i++) {
            classname = this.classlist[i];
            result.classes.push(this.classes[classname].getData());
        };
        return result;
    }
    
    /******
     * Default settings for menubar
     ******/
    Notifier.defaults = {
        classes: [],
        size: 30,
        level: 1,
        align: 'left',
        background: false
    };
    
    Notifier.templates = {
        html: [
            '<ul class="vnotifier-classlist">',
            '</ul>'
        ].join('\n')
    };

    Notifier.styles = [
        '.vnotifier-wrapper {display: inline-block; padding: 2px; min-width: 1em; min-height: 34px; position: relative;}',
        'ul.vnotifier-classlist {list-style: none; margin: 0; padding: 0; line-height: 0;}',
        'li.vnotifier-classitem {display: inline-block; position: relative; width: 34px; height: 34px; line-height: normal; margin: 0 2px;}',
        'li.vnotifier-classitem.vnotifier-empty {display: none;}',
        '.vnotifier-classbutton {width: 30px; height: 30px; padding: 2px; overflow: hidden; cursor: pointer; border-radius: 4px; border-left: 1px solid transparent; border-right: 1px solid transparent;}',
        '.vnotifier-classbutton:hover {border-left: 1px solid #999; border-right: 1px solid #999; border-radius: 0; background-color: rgba(255,255,255,0.5);}',
//         '.vnotifier-classbutton:hover {box-shadow: inset 0 4px 4px rgba(255,255,255,0.5), inset 0 -4px 4px rgba(0,0,0,0.2);}',
        '.vnotifier-classbutton svg {width: 30px; height: 30px;}',
        '.vnotifier-notifcount {display: block; font-size: 10px; font-weight: bold; min-width: 14px; height: 14px; text-align: center; line-height: 14px; position: absolute; top: 0; right: 0; color: white; background-color: #a00; border-radius: 4px; box-shadow: inset 0 3px 3px rgba(255,255,255,0.5), inset 0 -3px 3px rgba(0,0,0,0.2);}',
        'ul.vnotifier-notiflist {display: none; min-height: 1em; background-color: #ccc; border-top: 3px solid #ccc; border-bottom: 3px solid #ccc; min-width: 13em; position: absolute; list-style: none; padding: 0.2em 1.5em 0.2em 0.2em; font-family: sans-serif; font-size: 80%; max-height: 20em; overflow-y: auto; overflow-x: hidden; z-index: 20; border: 1px solid #999; box-shadow: 3px 3px 5px rgba(0,0,0,0.2);}',
        '.vnotifier-alignright ul.vnotifier-notiflist {right: 0;}',
        'ul.vnotifier-notiflist.vnotifier-show {display: block;}',
        'li.vnotifier-notification {background-color: #eee; color: black; margin: 0.3em; padding: 0.3em 0.5em; cursor: pointer; min-width: 20em; min-height: 2.5em; border-radius: 0.5em; border: 1px solid #999; box-shadow: 1px 1px 2px rgba(0,0,0,0.2); white-space: nowrap; overflow: hidden; text-align: left;}',
        'li.vnotifier-notification:hover {background-color: #fefefe;}',
        // Notificatio html
        '.vnotifier-notification-icon {display: inline-block; width: 20px; text-align: center; white-space: normal; vertical-align: top;}',
        '.vnotifier-notification-icon svg {width: 20px; height: 20px; vertical-align: middle;}',
        '.vnotifier-notification-message {display: inline-block; white-space: normal; text-align: left;}',
        // Trafficlights
        '.vnotifier-classitem[data-notiftype="trafficlight"][data-notifstate="red"] svg path {fill: #a00;}',
        '.vnotifier-classitem[data-notiftype="trafficlight"][data-notifstate="yellow"] svg path {fill: #F7C001;}',
        '.vnotifier-classitem[data-notiftype="trafficlight"][data-notifstate="green"] svg path {fill: #06CB00;}',
        '.vnotifier-classitem[data-notiftype="trafficlight"][data-notifstate="none"] svg path {fill: none;}',
        // Systray
        '.vnotifier-classitem[data-notiftype="systray"] {width: auto; vertical-align: top; height: 30px; line-height: 30px; box-shadow: inset 1px 1px 3px rgba(0,0,0,0.3); border-radius: 0.2em;}',
        '.vnotifier-classitem[data-notiftype="systray"] > ul {list-style: none; margin: 0; padding: 0; vertical-align: middle; display: inline-block;}',
        '.vnotifier-classitem[data-notiftype="systray"] > ul > .vnotifier-systray-notification {display: inline-block; height: 30px; width: 30px; margin: 0; padding: 0; text-align: center;}',
        // Quickmsg
        'li.vnotifier-classitem[data-notiftype="quickmsg"] {width: 0; height: 0; position: static; border: none; margin: 0;}',
        '.vnotifier-classitem[data-notiftype="quickmsg"] > .vnotifier-classbutton {display: none;}',
        '.vnotifier-classitem[data-notiftype="quickmsg"] > ul {background-color: transparent; border: none; position: absolute; top: 35px; margin: 0; padding: 0; z-index: 12;}',
        '.vnotifier-wrapper.vnotifier-alignleft .vnotifier-classitem[data-notiftype="quickmsg"] > ul {left: 0;}',
        '.vnotifier-wrapper.vnotifier-alignright .vnotifier-classitem[data-notiftype="quickmsg"] > ul {right: 0;}',
        '.vnotifier-classitem[data-notiftype="quickmsg"] > ul > .vnotifier-notification {border: 5px solid rgba(0,0,0,0.4); border-radius: 0.6em; background-color: rgba(0,0,0,0.7); color: white; font-family: sans-serif; font-size: 80%;}',
        '.vnotifier-classitem[data-notiftype="quickmsg"] > ul > .vnotifier-notification .vnotifier-notification-message {min-width: 20em; white-space: pre-line;}'
    ].join('\n');
    
    
    /********************************************************************************************************/
    /********************************************************************************************************/
    /******
     * Class for notification class
     * @constructor
     ******/
    var NotifClass = function(options){
        options = $.extend(true, {}, NotifClass.defaults, options);
        this.init(options);
    };
    
    NotifClass.prototype.init = function(options){
        this.name = options.name;
        this.type = options.type;
        this.label = options.label;
        this.icon = sanitize(options.icon);
        this.event = options.event;
        this.level = options.level;
        this.notifications = [];
        this.notifById = {};
        this.archive = [];
        if (options.notifications) {
            for (var i = 0, len = options.notifications.length; i < len; i++){
                this.addNotification(options.notifications[i]);
            };
        };
    };
    
    NotifClass.prototype.addNotification = function(data){
        if (typeof(data.id) === 'undefined') {
            data.id = this.generateId();
        };
        if (typeof(data.icon) === 'undefined') {
            data.icon = this.icon;
        };
        if (typeof(data.nclass) === 'undefined') {
            data.nclass = this.name;
        };
        if (typeof(data.ttl) === 'undefined') {
            data.ttl = this.ttl;
        };
        var notification = new Notification(data);
        var nid = notification.getId();
        this.removeNotification(nid);
        this.notifications.push(notification);
        this.notifById[data.id] = notification;
        this.showNew(data.id);
        if (data.ttl) {
            var notifclass = this;
            setTimeout(function(){notifclass.removeNotification(data.id);}, data.ttl);
        };
    };
    
    NotifClass.prototype.removeNotification = function(nid){
        var notif = this.notifById[nid];
        var index = this.notifications.indexOf(notif);
        if (index !== -1) {
            this.archive.push(this.notifications.splice(index, 1)[0]);
        };
        delete this.notifById[nid];
        this.update();
    }
    
    NotifClass.prototype.acknowledgeNotification = function(notifid){
        this.removeNotification(notifid);
    }
    
    NotifClass.prototype.size = function(){
        return this.notifications.length;
    }
    
    NotifClass.prototype.getLevel = function(){
        return this.level;
    }
    
    NotifClass.prototype.getEvent = function(){
        return this.event;
    }
    
    NotifClass.prototype.getNotifData = function(nid){
        var notification = this.notifById[nid];
        return notification && notification.data;
    }
    
    NotifClass.prototype.show = function(place){
        this.place = place;
        this.update();
    }
    
    NotifClass.prototype.getHtml = function(){}
    
    NotifClass.prototype.showNew = function(){
        this.update();
    }
    
    NotifClass.prototype.showCount = function(){}
    
    NotifClass.prototype.update = function(){
        if (this.place) {
            // Update only, if the place exists.
            this.setAttr();
            var html = '';
            if (this.size() > 0) {
                this.place.removeClass('vnotifier-empty');
            } else {
                this.place.addClass('vnotifier-empty');
            };
            html = this.getHtml();
            this.place.html(html);
            this.showCount();
        };
    };
    
    NotifClass.prototype.setAttr = function(){};
    
    NotifClass.prototype.generateId = function(){
        return 'notification-' + this.name + '-' + (new Date()).getTime() + '-' + Math.floor(1000*Math.random());
    };
    
    NotifClass.prototype.getData = function(){
        var result = {
            name: this.name,
            type: this.type,
            label: this.label,
            icon: sanitize(this.icon),
            event: this.event,
            level: this.level,
            notifications: [],
            archive: []
        };
        var notif;
        for (var i = 0, len = this.notifications.length; i < len; i++) {
            notif = this.notifications[i];
            result.notifications.push(notif.getData());
        };
        for (var i = 0, len = this.archive.length; i < len; i++) {
            notif = this.archive[i];
            result.archive.push(notif.getData());
        };
        return result;
    }
    
    /******
     * Default values for notification class.
     * type: "list","trafficlight","onoff"
     ******/
    NotifClass.defaults = {
        name: 'default',
        label: 'Notifications',
        type: 'list',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none;" d="M2 10 a8 8 0 0 1 8 -8 l10 0 a8 8 0 0 1 8 8 l0 2 a8 8 0 0 1 -8 8 a8 8 0 0 0 2 4 a12 12 0 0 1 -8 -4 l-4 0 a8 8 0 0 1 -8 -8z m11 -6 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l2 0 l1 -3 l3 0 l-1 3 l2 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l-2 0 l-1 3 l-3 0 l1 -3z m0.5 5 l3 0 l-1 3 l-3 0z" /></svg>',
        event: 'notificationclick',
        level: 0
    }
    
    NotifClass.templates = {
        html: [
            '<ul class="vnotifier-notiflist">',
            '</ul>'
        ].join('\n')
    };
    
    /******
     * NotifClassList
     ******/
    var NotifClassList = function(options){
        options = $.extend(true, {}, NotifClassList.defaults, options);
        this.init(options);
    };
    
    NotifClassList.prototype = Object.create(NotifClass.prototype);
    NotifClassList.prototype.constructor = NotifClassList;
    NotifClassList.prototype.parentClass = NotifClass.prototype;
    
    Notifier.prototype.classTypes['list'] = NotifClassList;

    NotifClassList.prototype.acknowledgeNotification = function(notifid){
        this.removeNotification(notifid);
    }
    
    NotifClassList.prototype.showNew = function(notifid){
        console.log('notifid', notifid);
        var notification = this.notifById[notifid];
        var html = notification.getHtml();
        if (this.place) {
            // Do only, when this.place exists.
            this.place.removeClass('vnotifier-empty');
            this.place.find('.vnotifier-notiflist').prepend(html);
            this.showCount();
        } else {
            this.update();
        };
    };
    
    NotifClassList.prototype.showCount = function(){
        var count = this.size();
        var countplace = this.place.find('.vnotifier-notifcount');
        countplace.text(count);
        if (count > 0) {
            countplace.removeClass('vnotifier-empty');
        } else {
            countplace.addClass('vnotifier-empty');
        }
    }
    
    NotifClassList.prototype.getHtml = function(){
        var notiflist = [];
        var notif;
        for (var i = 0, len = this.notifications.length; i < len; i++) {
            notif = this.notifications[i];
            notiflist.push(notif.getHtml());
        };
        notiflist.reverse();
        notiflist.unshift('<ul class="vnotifier-notiflist">');
        notiflist.push('</ul>');
        var size = this.size();
        var html = [
            '<div class="vnotifier-classbutton" title="'+escapeHTML(this.label)+'">'+sanitize(this.icon),
            '    <div class="vnotifier-notifcount vnotifier-empty"></div>',
            '</div>',
            notiflist.join('\n')
        ];
        return html.join('\n');
    };

    /******
     * Default values for notification class.
     * type: "list","trafficlight","onoff"
     ******/
    NotifClassList.defaults = {
        name: 'default',
        label: 'Notifications',
        type: 'list',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none;" d="M2 10 a8 8 0 0 1 8 -8 l10 0 a8 8 0 0 1 8 8 l0 2 a8 8 0 0 1 -8 8 a8 8 0 0 0 2 4 a12 12 0 0 1 -8 -4 l-4 0 a8 8 0 0 1 -8 -8z m11 -6 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l2 0 l1 -3 l3 0 l-1 3 l2 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l-2 0 l-1 3 l-3 0 l1 -3z m0.5 5 l3 0 l-1 3 l-3 0z" /></svg>',
        event: 'notificationclick',
        level: 0
    }
    
    /******
     * NotifClassTraffic
     ******/
    var NotifClassTraffic = function(options){
        options = $.extend(true, {}, NotifClassTraffic.defaults, options);
        this.init(options);
    };

    NotifClassTraffic.prototype = Object.create(NotifClass.prototype);
    NotifClassTraffic.prototype.constructor = NotifClassTraffic;
    NotifClassTraffic.prototype.parentClass = NotifClass.prototype;
    
    Notifier.prototype.classTypes['trafficlight'] = NotifClassTraffic;
    
    NotifClassTraffic.prototype.setAttr = function(){
        var lastnotif = this.notifications[this.notifications.length -1];
        var value = lastnotif && lastnotif.getState() || '';
        value = value.replace(/[^a-zA-Z]/g, '');
        this.place.attr('data-notiftype', 'trafficlight')
            .attr('data-notifstate', escapeHTML(value))
            .addClass('vnotifier-classitem');
    }
    
    NotifClassTraffic.prototype.getHtml = function(){
        var lastnotif = this.notifications[this.notifications.length -1];
        var nid = lastnotif && lastnotif.getId() || '';
        var message = lastnotif && lastnotif.getMessage() || '';
        var html = [
            '<div class="vnotifier-classbutton vnotifier-notification" title="'+escapeHTML(message)+'" data-notification-id="'+escapeHTML(nid)+'">'+sanitize(this.icon)+'</div>'
        ];
        return html.join('\n');
    };
    

    /******
     * Default values for notification class.
     * type: "list","trafficlight","onoff"
     ******/
    NotifClassTraffic.defaults = {
        name: 'default',
        label: 'Notifications',
        type: 'trafficlight',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z" /><path style="stroke: none; fill: rgba(255,255,255,0.2);" d="M5 15 a10 10 0 0 1 20 0 a12 12 0 0 0 -20 0z" /><path style="stroke: none; fill: rgba(255,255,255,0.5);" d="M5 15 a10 10 0 0 1 20 0 a10.2 10.2 0 0 0 -20 0z" /><path style="stroke: none; fill: rgba(0,0,0,0.1);" d="M5 15 a10 10 0 0 0 20 0 a12 12 0 0 1 -20 0z" /><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M5 15 a10 10 0 0 0 20 0 a10.2 10.2 0 0 1 -20 0z" /></svg>',
        event: 'notificationclick',
        level: 0
    }
    
    /******
     * NotifClassSystray
     ******/
    var NotifClassSystray = function(options){
        options = $.extend(true, {}, NotifClassSystray.defaults, options);
        this.ttl = options.ttl;
        this.init(options);
    }
    
    NotifClassSystray.prototype = Object.create(NotifClass.prototype);
    NotifClassSystray.prototype.constructor = NotifClassSystray;
    NotifClassSystray.prototype.parentClass = NotifClass.prototype;
    
    Notifier.prototype.classTypes['systray'] = NotifClassSystray;
    
    NotifClassSystray.prototype.setAttr = function(){
        this.place.attr('data-notiftype', 'systray');
    }

    NotifClassSystray.prototype.getHtml = function(){
        var notiflist = ['<ul class="vnotifier-notifsystray">'];
        var notif;
        for (var i = 0, len = this.notifications.length; i < len; i++) {
            notif = this.notifications[i];
            notiflist.push(notif.getHtmlIcon());
        };
        notiflist.push('</ul>');
        return notiflist.join('\n');
    }

    /******
     * Default values for notification class.
     * type: "list","trafficlight","onoff"
     ******/
    NotifClassSystray.defaults = {
        name: 'default',
        label: 'Notifications',
        type: 'systray',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z" /></svg>',
        event: 'notificationclick',
        level: 0,
        ttl: 2000 // Time to live
    }
    
    
    /******
     * NotifClassQuickmsg
     ******/
    var NotifClassQuickmsg = function(options){
        options = $.extend(true, {}, NotifClassQuickmsg.defaults, options);
        this.ttl = options.ttl;
        this.init(options);
    }
    
    NotifClassQuickmsg.prototype = Object.create(NotifClass.prototype);
    NotifClassQuickmsg.prototype.constructor = NotifClassQuickmsg;
    NotifClassQuickmsg.prototype.parentClass = NotifClass.prototype;
    
    Notifier.prototype.classTypes['quickmsg'] = NotifClassQuickmsg;
    
    NotifClassQuickmsg.prototype.setAttr = function(){
        this.place.attr('data-notiftype', 'quickmsg');
    }

    NotifClassQuickmsg.prototype.acknowledgeNotification = function(notifid){
        this.removeNotification(notifid);
    }
    
    NotifClassQuickmsg.prototype.getHtml = function(){
        var notiflist = ['<ul class="vnotifier-quickmsg">'];
        var notif;
        for (var i = 0, len = this.notifications.length; i < len; i++) {
            notif = this.notifications[i];
            notiflist.push(notif.getHtml());
        };
        notiflist.push('</ul>');
        return notiflist.join('\n');
    }

    /******
     * Default values for notification class.
     * type: "list","trafficlight","onoff", "quickmsg"
     ******/
    NotifClassQuickmsg.defaults = {
        name: 'default',
        label: 'Notifications',
        type: 'quickmsg',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z" /></svg>',
        event: 'notificationclick',
        level: 0,
        ttl: 4000 // Time to live
    }
    
    
    /********************************************************************************************************/
    /********************************************************************************************************/
    /******
     * Class for notification item
     * @constructor
     ******/
    var Notification = function(options){
        options = $.extend(true, {}, Notification.defaults, options);
        this.nclass = options.nclass,
        // Sanitize the html from the message.
        this.message = $('<span>' + options.message + '</span>').text();
        this.state = $('<span>' + options.state + '</span>').text();
        this.id = options.id;
        this.data = options.data;
        this.icon = sanitize(options.icon);
        this.timestamp = options.timestamp || (new Date()).getTime();
    };
    
    Notification.prototype.getHtml = function(){
        var html = [
            '<li class="vnotifier-notification" data-notification-id="'+escapeHTML(this.id)+'">',
            '    ' + '<div class="vnotifier-notification-icon">' + (this.icon ? sanitize(this.icon) : '') + '</div>',
            '    <div class="vnotifier-notification-message">'+escapeHTML(this.message)+'</div>',
            '</li>'
        ];
        return html.join('\n');
    }
    
    Notification.prototype.getHtmlIcon = function(){
        var html = [
            '<li class="vnotifier-systray-notification" data-notification-id="'+escapeHTML(this.id)+'">',
            '   <div>'+sanitize(this.icon)+'</div>',
            '</li>'
        ];
        return html.join('\n');
    }
    
    Notification.prototype.getMessage = function(){
        return this.message;
    };
    
    Notification.prototype.getState = function(){
        return this.state;
    };
    
    Notification.prototype.getId = function(){
        return this.id || '';
    }
    
    Notification.prototype.getData = function(){
        var result = {
            id: this.id,
            nclass: this.nclass,
            message: this.message,
            icon: sanitize(this.icon),
            timestamp: this.timestamp,
            data: this.data
        };
        return result;
    }
    
    Notification.defaults = {
        "nclass": "default",
        "message": "",
        "state": "",
        "icon": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none;" d="M2 10 a8 8 0 0 1 8 -8 l10 0 a8 8 0 0 1 8 8 l0 2 a8 8 0 0 1 -8 8 a8 8 0 0 0 2 4 a12 12 0 0 1 -8 -4 l-4 0 a8 8 0 0 1 -8 -8z m11 -6 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l-3 0 l-0.5 2 l3 0 l-1 3 l2 0 l1 -3 l3 0 l-1 3 l2 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l3 0 l0.5 -2 l-3 0 l1 -3 l-2 0 l-1 3 l-3 0 l1 -3z m0.5 5 l3 0 l-1 3 l-3 0z" /></svg>',
//        "icon": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z" /></svg>'
    }
    
})(jQuery);
//}}}