/******
@name VideoConf
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class VideoConf
@description A class and jQuery-plugin for a course video conferencing.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Course video conferencing
     * @constructor
     * @param {jQuery} place    DOM-place for course videoconf
     * @param {Object} options  data for course videoconf
     */
    var VideoConf = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.initData();
        this.setAttrs();
        this.setStyles();
        this.show();
    }
    
    VideoConf.prototype.init = function(options) {
        this.settings = options.settings;
        this.users = this.settings.users;
        this.appid = this.settings.appid;
        this.apptype = this.settings.apptype;
    };
    
    VideoConf.prototype.initData = function() {
    }
    
    VideoConf.prototype.initHandlers = function() {
        var videoconf = this;
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            videoconf.close()
        });
        this.place.off('click', '.coursevideoconf-actions button').on('click', '.coursevideoconf-actions button', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            videoconf.place.trigger(action);
        });
    };
    
    VideoConf.prototype.setAttrs = function() {
        this.place.addClass('coursevideoconf');
    };
    
    VideoConf.prototype.show = function() {
        this.view();
    };
    
    VideoConf.prototype.view = function() {
        var uilang = this.settings.uilang;
        var html = this.template.html;
        this.place.html(html);
        var headplace = this.place.find('.coursevideoconf-header');
        var descplace = this.place.find('.coursevideoconf-description');
        var startbutton = this.place.find('.coursevideoconf-button-start');
        headplace.text(ebooklocalizer.localize('videoconf:header', uilang));
        descplace.html(ebooklocalizer.localize('videoconf:description', uilang));
        startbutton.text(ebooklocalizer.localize('videoconf:start', uilang));
    };
    
    VideoConf.prototype.close = function() {
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    }
    
    VideoConf.prototype.setStyles = function() {
        if ($('head style#coursevideoconfstyles').length === 0) {
            $('head').append('<style type="text/css" id="coursevideoconfstyles">'+this.styles+'</style>')
        }
    }
    
    VideoConf.prototype.defaults = {
        settings: {
            uilang: 'en'
        }
    };
    
    VideoConf.prototype.styles = [
        '.coursevideoconf {background-color: #eee; font-family: helvetica, Arial, sans-serif; padding: 1em;}',
        '.coursevideoconf-actions.coursevideoconf-actions.coursevideoconf-actions {margin: 1em 0;}',
        '.coursevideoconf-iconarea {text-align: center;}',
        '.coursevideoconf-icon {display: inline-block;}',
        '.coursevideoconf-icon svg {width: 200px; height: auto;}'
    ].join('\n');
    
    VideoConf.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-videoconf"><path style="stroke: none;" d="M15 5 a5 5 0 0 1 0 10 a5 5 0 0 1 0 -10z m-10 16 a6 6 0 0 1 6 -6 l4 4 l4 -4 a6 6 0 0 1 6 6z m-4 -17 a2 2 0 0 1 2 -2 h24 a2 2 0 0 1 2 2 v18 a2 2 0 0 1 -2 2 h-11 v2 h6 v2 h-14 v-2 h6 v-2 h-11 a2 2 0 0 1 -2 -2z m2 0 v18 h24 v-18z" /></svg>',
        refresh: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-refresh"><path style="stroke: none;" d="M16 14 l8 0 l0 -8 l-3 3 a8.5 8.5 0 1 0 0 12 l-2 -2 a5.6 5.6 0 1 1 0 -8 z"></path></svg>'
    }
    
    VideoConf.prototype.template = {
        html: [
            '<div class="coursevideoconf-controlbuttons ffwidget-buttonset ffwidget-horizontal"></div>',
            '<h1 class="coursevideoconf-header"></h1>',
            '<div class="coursevideoconf-iconarea"><div class="coursevideoconf-icon">'+VideoConf.prototype.icons.appicon+'</div></div>',
            '<div class="coursevideoconf-description"></div>',
            '<div class="coursevideoconf-actions ffwidget-buttonset ffwidget-horizontal"><button class="coursevideoconf-button-start ffwidget-setbutton" data-action="videoconf_start"></button></div>'
        ].join('\n')
    };
    
    VideoConf.localization = {
        "en": {
            "videoconf:header": "Videochat",
            "videoconf:description": "Here you can start videoconferencing for this course.\nCurrently this feature uses <a href=\"https://meet.jit.si/\">Jitsi</a> as conference service.",
            "videoconf:start": "Start"
        },
        "fi": {
            "videoconf:header": "Videochat",
            "videoconf:description": "Tästä voit käynnistää kurssin videotapaamisen.\nTällä hetkellä kokouspalveluna käytetään <a href=\"https://meet.jit.si/\">Jitsiä</a>.",
            "videoconf:start": "Aloita"
        },
        "sv": {
            "videoconf:header": "Videochat",
            "videoconf:description": "Här kan du starta videokonferens för denna kurs.\nFör närvarande denna funktion använder <a href=\"https://meet.jit.si/\">Jitsi</a> som konferensservice.",
            "videoconf:start": "Start"
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(VideoConf.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(VideoConf.localization);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var db = new VideoConf(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[videoconfdata]]');
            return data;
        },
        'geticon': function() {
            return VideoConf.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.coursevideoconf = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in coursevideoconf.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
