/***
 * Togglebutton
 * Petri Salmela <pesasa@iki.fi>
 ***/
//{{{

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

(function ($) {
    { /*** jQuery plugin ***/
        $.fn.togglebutton = function(options){
            if (methods[options]){
                return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
            } else if (typeof(options) === 'object' || !options) {
                return methods.init.apply(this, arguments);
            } else {
                $.error( 'Method ' +  method + ' does not exist on Togglebutton' );
                return this;
            }
        }
        
        var methods = {
            init: function(params){
                params = $.extend(true, {
                }, params);
                var menu = new Togglebutton(this, params);
            },
            getdata: function(params){
                var $place = $(this);
                $place.trigger('getdata');
                var data = $place.data('[[togglebuttondata]]');
                return data;
            }
        }
    }
    
    /******
     * Class for Togglebutton
     * @constructor
     * @param {jQuery} place - place for navbar
     * @param {Object} options - settings for navbar
     ******/
    var Togglebutton = function(place, options){
        this.place = $(place);
        options = $.extend(true, {}, Togglebutton.defaults, options);
        this.setStyles();
        this.initData(options);
        this.setAttrs();
        this.initHandlers();
        this.show();
    };
    
    /******
     * Set attributes
     ******/
    Togglebutton.prototype.setAttrs = function(){
        this.place.addClass('togglebutton-wrapper');
        if (this.ffbackground) {
            this.place.addClass('ffwidget-background');
        };
        if (this.showlabels) {
            this.place.addClass('togglebutton-showlabels');
        };
        if (typeof(this.size) === 'number') {
            this.place.attr('style', 'width: ' + this.size + 'px; height: ' + this.size + 'px;');
        }
    }
    
    /******
     * Set stylesheets
     ******/
    Togglebutton.prototype.setStyles = function(){
        if ($('head style#togglebutton-style').length === 0) {
            $('head').append('<style id="togglebutton-style" type="text/css">'+Togglebutton.styles+'</style>')
        };
    };
    
    /******
     * Init menudata
     ******/
    Togglebutton.prototype.initData = function(options){
        this.ffbackground = options.ffbackground;
        this.showlabels = options.showlabels;
        this.size = options.size;
        this.states = [];
        this.statedata = {};
        var item, stateitem;
        for (var i = 0, len = options.states.length; i < len; i++){
            item = options.states[i];
            stateitem = new Stateitem(item);
            this.states.push(item.name);
            this.statedata[item.name] = stateitem;
        };
        this.current = 0;
    };
    
    /**
     * Get the name of current state
     * @returns {String} the name of the current status.
     */
    Togglebutton.prototype.getCurrentState = function() {
        return this.states[this.current];
    };
    
    /**
     * Move to the next state
     * @param {Boolean} trigger - whether to trigger the event or not
     */
    Togglebutton.prototype.next = function(trigger) {
        if (trigger) {
            this.triggerOff();
        };
        this.current = (this.current + 1) % this.states.length;
        this.setState(trigger);
    };
    
    /**
     * Move to the previous state
     * @param {Boolean} trigger - whether to trigger the event or not
     */
    Togglebutton.prototype.prev = function(trigger) {
        if (trigger) {
            this.triggerOff();
        };
        this.current = (this.current + this.states.length - 1) % this.states.length;
        this.setState(trigger);
    };
    
    /**
     * Move to the given status
     * @param {Boolean} trigger - whether to trigger the event or not
     * @param {String} state - the name of the state
     */
    Togglebutton.prototype.set = function(trigger, state) {
        var index = this.states.indexOf(state);
        if (index >= 0) {
            if (trigger) {
                this.triggerOff();
            };
            this.current = index;
            this.setState(trigger);
        };
    };
    
    /**
     * Set the state
     * @param {Boolean} trigger - whether to trigger the event or not
     * @param {String} state - the name of the state
     */
    Togglebutton.prototype.setState = function(trigger, state) {
        if (typeof(state) === 'undefined') {
            state = this.getCurrentState();
        };
        this.place.children().removeClass('togglebutton-currentstate');
        this.place.children('[data-name="'+state+'"]').addClass('togglebutton-currentstate');
        if (trigger) {
            this.trigger();
        };
    };
    
    /**
     * Trigger the current event
     */
    Togglebutton.prototype.trigger = function() {
        var event = this.getEvent();
        var data = this.getEventData() || [];
        if (event) {
            this.place.trigger(event, data);
        };
    };

    /**
     * Trigger the current event_off
     */
    Togglebutton.prototype.triggerOff = function() {
        var event = this.getEventOff();
        var data = this.getEventOffData() || [];
        if (event) {
            this.place.trigger(event, data);
        };
    };

    /**
     * Get the event of current state
     * @returns {String} the event name of current state
     */
    Togglebutton.prototype.getEvent = function() {
        return this.statedata[this.getCurrentState()].getEvent();
    };
    
    /**
     * Get the event_off of current state
     * @returns {String} the event_off name of current state
     */
    Togglebutton.prototype.getEventOff = function() {
        return this.statedata[this.getCurrentState()].getEventOff();
    };
    
    /**
     * Get the event data of the current state
     * @return {Object} the data for event of current state
     */
    Togglebutton.prototype.getEventData = function() {
        return this.statedata[this.getCurrentState()].getEventData();
    };
    
    /**
     * Get the event_off data of the current state
     * @return {Object} the data for event_off of current state
     */
    Togglebutton.prototype.getEventOffData = function() {
        return this.statedata[this.getCurrentState()].getEventOffData();
    };
    
    /******
     * Init handlers
     ******/
    Togglebutton.prototype.initHandlers = function(){
        var button = this;
        this.place.off();
        this.place.on('click', '.togglebutton-buttonitem', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            button.next(true);
        }).on('nextstate', function(event, data){
            event.stopPropagation();
            button.next(true);
        }).on('prevstate', function(event, data){
            event.stopPropagation();
            button.prev(true);
        }).on('setstate', function(event, data){
            event.stopPropagation();
            button.set(true, data);
        }).on('shownext', function(event, data){
            event.stopPropagation();
            button.next(false);
        }).on('showprev', function(event, data){
            event.stopPropagation();
            button.prev(false);
        }).on('showstate', function(event, data){
            event.stopPropagation();
            button.set(false, data);
        });
    };
    
    /******
     * Show the button
     ******/
    Togglebutton.prototype.show = function(){
        this.place.html(this.getHtml());
        this.place.find('svg').attr('width', this.size).attr('height', this.size);
        this.setState();
    };
    
    /******
     * Get the html for the menu.
     ******/
    Togglebutton.prototype.getHtml = function(){
        var html = [];
        var state;
        for (var i = 0, len = this.states.length; i < len; i++) {
            state = this.states[i];
            html.push(this.statedata[state].getHtml());
        };
        this.setState();
        return html;
    }
    
    /******
     * Default settings for menubar
     ******/
    Togglebutton.defaults = {
        showlabels: true,
        size: 30,
        states: [
        ],
        ffbackground: true
    };
    
    Togglebutton.templates = {
        html: [
        ].join('\n')
    };

    Togglebutton.styles = [
        '.togglebutton-wrapper {display: inline-block; text-align: center; padding: 2px; border-radius: 3px;}',
        '.togglebutton-buttonitem {display: none;}',
        '.togglebutton-buttonitem.togglebutton-currentstate {display: inline-block; cursor: pointer;}',
        // Labels
        '.togglebutton-buttonlabel {display: none; font-size: 80%;}',
        '.togglebutton-showlabels .togglebutton-buttonlabel {display: block;}'
   ].join('\n');
    
    
    /**********************************************************************************************/
    /**********************************************************************************************/
    /**
     * Class for menu item
     * @constructor
     */
    var Stateitem = function(options){
        options = $.extend(true, {}, Stateitem.defaults, options);
        this.name = options.name;
        this.icon = options.icon;
        this.title = options.title;
        this.event = options.event;
        this.event_off = options.event_off;
        this.data = options.data || null;
        this.data_off = options.data || null;
    };
    
    /**
     * Get the name of event
     * @returns {String} the event name
     */
    Stateitem.prototype.getEvent = function() {
        return this.event;
    };
    
    
    /**
     * Get the name of event_off
     * @returns {String} the event_off name
     */
    Stateitem.prototype.getEventOff = function() {
        return this.event_off;
    };
    
    /**
     * Get the event data
     * @returns {Object} data to be triggered with event
     */
    Stateitem.prototype.getEventData = function() {
        return JSON.parse(JSON.stringify(this.data));
    };
    
    /**
     * Get the event_off data
     * @returns {Object} data to be triggered with event_off
     */
    Stateitem.prototype.getEventOffData = function() {
        return JSON.parse(JSON.stringify(this.data_off));
    };
    
    /******
     * Get html for this menuitem.
     ******/
    Stateitem.prototype.getHtml = function(){
        var html = [
            '<div class="togglebutton-buttonitem" title="' + this.title + '" data-name="' + this.name + '">',
            '<div class="togglebutton-buttonicon">' + this.icon + '</div>',
            '<div class="togglebutton-buttonlabel">' + this.title + '</div>',
            '</div>'
        ];
        return html.join('\n');
    };
    
    Stateitem.defaults = {
        name: 'default',
        icon: '',
        title: '',
        event: ''
    }
    
})(jQuery);
//}}}