/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}


;(function($){
    
    /**
     * Helper functions
     */
    
    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    /******
     * TOC element to show and edit the table of contents of the notebook.
     * @requires 
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var TocElement = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init tocelement
     * @param {Object} options - initing data of the element
     ******/
    TocElement.prototype.init = function(options){
        options = $.extend(true, {}, TocElement.defaults, options);
        this.settings = options.settings;
        this.type = 'tocelement';
        this.setMode(options.settings.mode);
        this.metadata = options.toc.metadata;
        this.data = options.toc.data;
        this.booktitle = options.title;
        this.pagetitles = options.pagetitles;
        this.defaultlang = options.defaultlang;
        this.currentPage = options.currentPage || this.data.firstpage.chapid;
        this.pages = new TocPage(this.data.firstpage, 0);
        this.makeLinear();
        this.makeIndex();
        this.setAttrs();
    };
    
    /******
     * Create the index of pages (maps chapid to the TocPage-object)
     ******/
    TocElement.prototype.makeIndex = function(){
        this.pageIndex = {};
        var page;
        for (var i = 0, len = this.pagesLinear.length; i < len; i++) {
            page = this.pagesLinear[i];
            this.pageIndex[page.chapid] = page;
        };
    };
    
    /******
     * Create the linear pagelist
     ******/
    TocElement.prototype.makeLinear = function(){
        this.pagesLinear = this.pages.getLinear();
    }
    
    /******
     * Set some attributes
     ******/
    TocElement.prototype.setAttrs = function() {
        this.place.addClass('tocelement');
    }
    
    /******
     * Set style-tag if needed.
     ******/
    TocElement.prototype.setStyles = function(){
        if ($('head style#tocelement-style').length === 0) {
            $('head').append('<style id="tocelement-style" type="text/css">'+TocElement.styles+'</style>')
        };
    };
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    TocElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = TocElement.modes[mode] || TocElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
    };
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    TocElement.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    };
    
    /******
     * Show the TocElement element in different modes
     ******/
    TocElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', escapeHTML(this.metadata.lang));
            this.place.attr('uilang', escapeHTML(this.settings.uilang));
        };
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        };
    };
    
    /******
     * Show the element in non-editable view mode.
     ******/
    TocElement.prototype.view = function(){
        var uilang = this.settings.uilang;
        var lang = this.settings.lang;
        var booktitle = this.booktitle[lang] || this.booktitle.common;
        this.place.empty().html(TocElement.templates.view);
        this.place.children('.tocelement-toctitle').html(ebooklocalizer.localize('tocelement:toc', uilang));
        this.place.find('.tocelement-booktitle').text(booktitle);
        var toclist = this.place.children('.tocelement-toclist');
        toclist.html(this.getHtml(this.pages, []));
        this.openToc();
        this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
    };
    
    /******
     * Show the element in editable mode.
     ******/
    TocElement.prototype.edit = function(){
        var uilang = this.settings.uilang;
        var lang = this.settings.lang;
        var booktitle = this.booktitle[lang] || this.booktitle.common;
        this.place.empty().html(TocElement.templates.edit);
        this.place.children('.tocelement-toctitle').html(ebooklocalizer.localize('tocelement:toc', uilang));
        this.place.find('input.tocelement-booktitle').attr('lang', escapeHTML(lang)).val(booktitle);
        var toclist = this.place.children('.tocelement-toclist');
        toclist.html(this.getHtmlEditable(this.pages, []));
        if (this.settings.lang === this.defaultlang) {
            // Ordering is allowed only in defaultlang of the notebook.
            this.showOrderbar();
        };
        this.openToc();
        this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
    };
    
    /******
     * Open the toc
     ******/
    TocElement.prototype.openToc = function(){
        this.place.addClass('tocelement-open');
        var width = this.place.width();
        var ul = this.place.children('ul.tocelement-toclist');
        ul.width(ul.width());
        this.place.removeClass('tocelement-open');
        this.place.width(0);
        this.place.addClass('tocelement-open');
        this.place.width(width);
        setTimeout(function(){
            ul.removeAttr('style');
        }, 600);
        // Scroll the current page into the view.
        var currpage = this.place.find('.tocelement-tocitem-currentpage');
        if (currpage[0]) {
            try {
                currpage[0].scrollIntoView({behavior: 'smooth', block: 'center'});
            } catch (err) {
            };
        };
        if (Math.abs(ul.offset().top - currpage.offset().top) < 5 ) {
            ul[0].scrollTop -= ul.height() / 2;
        };
    };
    
    /******
     * Close the toc
     ******/
    TocElement.prototype.closeToc = function(){
        var toc = this;
        var place = this.place;
        var ul = this.place.children('ul.tocelement-toclist');
        ul.width(ul.width());
        place.addClass('tocelement-closing').removeClass('tocelement-open').removeAttr('style');
        setTimeout(function(){
            toc.removeHandlers();
            place.removeClass('tocelement-closing').empty();
        }, 500);
    };
    
    /******
     * Get the html of the toclist
     * @param {Object} startpage - Page object to start with. Get children recoursively.
     * @param {Array} pagenums - The current pagenumber as an array
     * @returns {String} the table of contents as html list.
     ******/
    TocElement.prototype.getHtml = function(startpage, pagenums) {
        if (typeof(startpage) === 'undefined') {
            startpage = this.pages;
        };
        var unnumbered = false;
        if (typeof(pagenums) === 'undefined') {
            pagenums = [];
            unnumbered = true;
        };
        var html = [];
        var chapid = startpage.chapid;
        var pnums;
        var title = this.getTitle(chapid);
        var iscurrent = (chapid === this.currentPage ? ' tocelement-tocitem-currentpage' : '');
        var currenthl = (chapid === this.currentPage ? ' ffwidget-highlight' : '');
        html.push('<li class="tocelement-tocitem'+iscurrent+'" data-chapid="'+escapeHTML(chapid)+'"><div class="tocelement-tocitem-title ffwidget-hoverhighlight'+currenthl+'"><span class="tocelement-tocitem-pagenum">'+ pagenums.join('.') + '</span> <span class="tocelement-tocitem-pagetitle">' + (escapeHTML(title).replace(/\$([^$]*)\$/g,'<span class="mathquill-embedded-latex">$1</span>') || '<span class="tocelement-listplaceholder">[&nbsp;&nbsp;----&nbsp;&nbsp;]</span>') + '</span></div>');
        if (startpage.before && startpage.before.chapid) {
            html.push('<ul class="tocelement-sublist tocelement-beforelist">');
            html.push(this.getHtml(startpage.before));
            html.push('</ul>');
        }
        html.push('<ul class="tocelement-sublist">');
        for (var i = 0, len = startpage.chapters.length; i < len; i++) {
            if (!unnumbered) {
                pnums = pagenums.slice();
                pnums.push(i+1);
            };
            html.push(this.getHtml(startpage.chapters[i], pnums));
        };
        html.push('</ul>');
        if (startpage.after && startpage.after.chapid) {
            html.push('<ul class="tocelement-sublist tocelement-afterlist">');
            html.push(this.getHtml(startpage.after));
            html.push('</ul>');
        }
        html.push('</li>');
        //html = sanitize(html.join('\n'));
        return html.join('\n');
    };
    
    /******
     * Get the html of the toclist in editable mode
     * @param {Object} startpage - Page object to start with. Get children recoursively.
     * @param {Array} pagenums - The current pagenumber as an array
     * @returns {String} the table of contents as html list.
     ******/
    TocElement.prototype.getHtmlEditable = function(startpage, pagenums) {
        if (typeof(startpage) === 'undefined') {
            startpage = this.pages;
        };
        var unnumbered = false;
        if (typeof(pagenums) === 'undefined') {
            pagenums = [];
            unnumbered = true;
        };
        var isdeflang = (this.settings.lang === this.defaultlang);
        var html = [];
        var pnums;
        var chapid = startpage.chapid;
        var title = this.getTitle(chapid);
        var iscurrent = (chapid === this.currentPage ? ' tocelement-tocitem-currentpage' : '');
        var currenthl = (chapid === this.currentPage ? ' ffwidget-highlight' : '');
        html.push('<li class="tocelement-tocitem'+iscurrent+'" data-chapid="'+escapeHTML(chapid)+'"><div class="tocelement-tocitem-title ffwidget-hoverhighlight'+currenthl+'"><span class="tocelement-tocitem-pagenum">'+ pagenums.join('.') + '</span> <span class="tocelement-tocitem-pagetitle">'+ escapeHTML(title) + '</span></div>');
        if (startpage.before && startpage.before.chapid) {
            html.push('<ul class="tocelement-sublist tocelement-beforelist">');
            html.push(this.getHtml(startpage.before));
            html.push('</ul>');
        } else if (false && isdeflang) {
            html.push('<ul class="tocelement-sublist tocelement-beforelist">');
            html.push('<li class="tocelement-tocitem tocelement-tocitem-addbeforepage"><div class="tocelement-tocitem-title">Add beforepages</div></li>');
            html.push('</ul>');
        }
        html.push('<ul class="tocelement-sublist">');
        for (var i = 0, len = startpage.chapters.length; i < len; i++) {
            if (!unnumbered) {
                pnums = pagenums.slice();
                pnums.push(i+1);
            };
            html.push(this.getHtml(startpage.chapters[i], pnums));
        };
        html.push('</ul>');
        if (startpage.after && startpage.after.chapid) {
            html.push('<ul class="tocelement-sublist tocelement-afterlist">');
            html.push(this.getHtml(startpage.after));
            html.push('</ul>');
        } else if (false && isdeflang) {
            html.push('<ul class="tocelement-sublist tocelement-beforelist">');
            html.push('<li class="tocelement-tocitem tocelement-tocitem-addafterpage"><div class="tocelement-tocitem-title">Add afterpages</div></li>');
            html.push('</ul>');
        }
        html.push('</li>');
        //html = sanitize(html.join('\n'));
        return html.join('\n');
    };
    
    /******
     * Get the title by chapid and language.
     * @param {String} chapid - the id of the page
     * @returns {String} the title of the page
     ******/
    TocElement.prototype.getTitle = function(chapid) {
        var lang = this.settings.lang;
        var deflang = this.defaultlang;
        var title = this.pagetitles[lang] && this.pagetitles[lang][chapid]
                     || (this.pagetitles[deflang] && this.pagetitles[deflang][chapid])
                     || (this.pagetitles.common && this.pagetitles.common[chapid])
                     || '';
        return title;
    }
    
    /******
     * Change the current page to given chapid
     * @param {String} chapid - chapid of the page to move to
     ******/
    TocElement.prototype.changeToPage = function(chapid) {
        this.currentPage = chapid;
        this.place.find('li.tocelement-tocitem-currentpage').removeClass('tocelement-tocitem-currentpage')
            .children('.tocelement-tocitem-title').removeClass('ffwidget-highlight');
        this.place.find('li[data-chapid="'+escapeHTML(chapid)+'"]').addClass('tocelement-tocitem-currentpage')
            .children('.tocelement-tocitem-title').addClass('ffwidget-highlight');
    };
    
    /******
     * Show tools for ordering pages
     ******/
    TocElement.prototype.showOrderbar = function(){
        var uilang = this.settings.uilang;
        var buttons = TocElement.orderbuttons, button;
        var html = ['<ul class="tocelement-orderbuttons ffwidget-buttonset ffwidget-horizontal">'];
        for (var i = 0, len = buttons.length; i < len; i++) {
            button = buttons[i];
            html.push('<li class="tocelement-orderbutton ffwidget-setbutton" data-action="'+button.action+'" title="'+ebooklocalizer.localize(button.label, uilang)+'"><div class="tocelement-buttonicon">'+button.icon+'</div></li>')
        }
        html.push('</ul>');
        this.place.find('.tocelement-orderbar').html(html.join(''));
    }
    
    /******
     * Move the current page to lower level of hierarchy (chapter to section, section to page)
     ******/
    TocElement.prototype.indentPage = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        var level = page.level;
        if (!this.isFirst(chapid) && page.parent.before !== page && page.parent.after !== page) {
            // Find the previous sibling and move the page as its last child.
            var index = this.pagesLinear.indexOf(page);
            var parent = page.parent;
            var pindex = parent.chapters.indexOf(page);
            var prev, found = false;
            for (var i = index - 1; i > -1; i--){
                prev = this.pagesLinear[i];
                if (prev.level === level) {
                    // We found the new parent.
                    found = true;
                    break;
                } else if(prev.level < level) {
                    // New parent does not exist.
                    break;
                };
            };
            if (found) {
                // Remove page from current parent and add as the last child of new parent.
                parent.chapters.splice(pindex, 1);
                prev.chapters.push(page);
                page.parent = prev;
                page.level = prev.level + 1;
                page.fixLevels();
                this.makeLinear();
            };
            this.place.trigger('tocincreasenesting');
        };
    };
    
    /******
     * Move the current page to higher level of hierarchy (section to chapter, page to section)
     ******/
    TocElement.prototype.unindentPage = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        var level = page.level;
        if (!this.isFirst(chapid) && page.parent && page.parent.parent && page.parent.before !== page && page.parent.after !== page && page.parent.parent.before !== page.parent && page.parent.parent.after !== page.parent) {
            // Don't do anything, if page is the frontpage, does not have grandparent or is before or after or this is child of before or after.
            // Take from parent and put as child of grandparent, after parent.
            var parent = page.parent;
            var gparent = parent.parent;
            var pindex = parent.chapters.indexOf(page);
            var gpindex = gparent.chapters.indexOf(parent);
            parent.chapters.splice(pindex, 1);
            gparent.chapters.splice(gpindex + 1, 0, page);
            page.parent = gparent;
            page.level = gparent.level + 1;
            page.fixLevels();
            this.makeLinear();
            this.place.trigger('tocdecreasenesting');
        };
    };
    
    /******
     * Move the current page to lower level of hierarchy, but keep the linear place. (chapter to section, section to page)
     ******/
    TocElement.prototype.demotePage = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        if (!this.isFirst(chapid) && page.parent.before !== page && page.parent.after !== page) {
            // Find the previous sibling and move the page and children as its children.
            var index = this.pagesLinear.indexOf(page);
            var parent = page.parent;
            var pindex = parent.chapters.indexOf(page);
            var prev;
            if (pindex !== 0) {
                prev = parent.chapters[pindex - 1];
                page.parent = prev;
                parent.chapters.splice(pindex, 1);
                prev.chapters.push(page);
                page.level = prev.level + 1;
                prev.chapters = prev.chapters.concat(page.chapters.splice(0));
                prev.fixParents();
            };
            this.place.trigger('tocdemotehere');
        };
    };
    
    /******
     * Move the current page to higher level of hierarchy, but keep the linear place. (section to chapter, page to section)
     ******/
    TocElement.prototype.promotePage = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        var level = page.level;
        if (!this.isFirst(chapid) && page.parent && page.parent.parent && page.parent.before !== page && page.parent.after !== page && page.parent.parent.before !== page.parent && page.parent.parent.after !== page.parent) {
            // Take from parent and put as child of grandparent, after parent.
            var parent = page.parent;
            var gparent = parent.parent;
            var pindex = parent.chapters.indexOf(page);
            var gpindex = gparent.chapters.indexOf(parent);
            parent.chapters.splice(pindex, 1);
            var children = parent.chapters.splice(pindex);
            gparent.chapters.splice(gpindex + 1, 0, page);
            page.parent = gparent;
            page.level = gparent.level + 1;
            page.chapters = page.chapters.concat(children);
            page.fixParents();
            page.fixLevels();
            this.makeLinear();
            this.place.trigger('tocpromotehere');
        };
    };
    
    /******
     * Move the current page forward in the toc (with same level of hierarchy)
     ******/
    TocElement.prototype.movePageForward = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        if (!this.isLast(chapid) && !this.isFirst(chapid) && page.parent.before !== page && page.parent.after !== page) {
            var index = this.pagesLinear.indexOf(page);
            var parent = page.parent;
            var pindex = parent.chapters.indexOf(page);
            if (pindex !== parent.chapters.length - 1) {
                // Page is not its parent's last child.
                // Switch Page with its next sibling.
                parent.chapters.splice(pindex, 1);
                parent.chapters.splice(pindex + 1, 0, page);
                this.makeLinear();
            } else {
                // Page is its parent's last child.
                // Move page before the next page with same level, as that pages first child.
                var nextParent, found = false;
                for (var i = index + 1, len = this.pagesLinear.length; i < len; i++){
                    nextParent = this.pagesLinear[i];
                    if (nextParent.level === parent.level) {
                        found = true;
                        break;
                    };
                };
                if (found) {
                    // Move only, if the page wasn't the last of its level.
                    //var nindex = this.pagesLinear.indexOf(next);
                    parent.chapters.splice(pindex, 1);
                    nextParent.chapters.unshift(page);
                    this.makeLinear();
                    page.parent = nextParent;
                };
            };
            this.place.trigger('tocmovepageforward');
        };
    };
    
    /******
     * Move the current page backward in the toc (with same level of hierarchy)
     ******/
    TocElement.prototype.movePageBackward = function(){
        var chapid = this.currentPage;
        var page = this.pageIndex[chapid];
        if (!this.isFirst(chapid) && page.parent.before !== page && page.parent.after !== page) {
            var index = this.pagesLinear.indexOf(page);
            var parent = page.parent;
            var pindex = parent.chapters.indexOf(page);
            if (pindex !== 0) {
                // Page is not its parent's first child.
                // Switch Page with its previous sibling.
                parent.chapters.splice(pindex, 1);
                parent.chapters.splice(pindex - 1, 0, page);
                this.makeLinear();
            } else {
                // Page is its parent's first child.
                // Move page as the last child of the previous page with same level as the parent.
                var parentIndex = this.pagesLinear.indexOf(parent);
                var prevParent, found = false;
                for (var i = parentIndex - 1; i > -1; i--){
                    prevParent = this.pagesLinear[i];
                    if (prevParent.level === parent.level) {
                        found = true;
                        break;
                    };
                };
                if (found) {
                    // Move only, if the parent wasn't the first of its level.
                    parent.chapters.splice(pindex, 1);
                    prevParent.chapters.push(page);
                    this.makeLinear();
                    page.parent = prevParent;
                };
            };
            this.place.trigger('tocmovepagebackward');
        };
    }
    
    /******
     * Change the booktitle
     * @param {String} lang - language
     * @param {String} title - titletext
     ******/
    TocElement.prototype.booktitleChange = function(lang, title) {
        this.booktitle[lang] = title;
        if (lang === this.defaultlang) {
            this.booktitle.common = title;
        };
        this.place.trigger('tocchangebooktitle', [JSON.parse(JSON.stringify(this.booktitle))]);
    };
    
    /******
     * Check, if the given chapid is the id of the last page in pagesLinear.
     * @param {String} chapid - the id of the page
     ******/
    TocElement.prototype.isLast = function(chapid){
        return this.pagesLinear[this.pagesLinear.length - 1].chapid === chapid;
    }
    
    /******
     * Check, if the given chapid is the id of the first page in pagesLinear.
     * @param {String} chapid - the id of the page
     ******/
    TocElement.prototype.isFirst = function(chapid){
        return this.pagesLinear[0].chapid === chapid;
    }
    
    /******
     * Remove all event handlers
     ******/
    TocElement.prototype.removeHandlers = function(){
        this.place.off();
    };
    
    /******
     * Init handlers for all modes.
     ******/
    TocElement.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in CodeElement.modes) {
                element.changeMode(data);
            }
        }).on('goto', function(e, chapid){
            e.stopPropagation();
            element.changeToPage(chapid);
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        }).on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        }).on('destroy', function(e){
            e.stopPropagation();
            element.closeToc();
        }).on('starteditingtoc', function(e, data){
            e.stopPropagation();
            element.setMode('edit');
            element.show();
        }).on('stopeditingtoc', function(e, data){
            e.stopPropagation();
            element.setMode('view');
            element.show();
        });

    };
    
    /******
     * Init handlers for non-editable mode
     ******/
    TocElement.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        }).on('click', '.tocelement-tocitem', function(e){
            e.stopPropagation();
            var item = $(this);
            var chapid = item.attr('data-chapid');
            item.trigger('gotopage', {chapid: chapid}).trigger('destroy');
        });
    };
    
    /******
     * Init handlers for editable mode
     ******/
    TocElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('click', '.tocelement-orderbar .tocelement-orderbutton', function(event, data){
            event.stopPropagation();
            var button = $(this);
            var action = button.attr('data-action');
            button.trigger(action);
        }).on('click', '.tocelement-booktitle', function(e){
            e.stopPropagation();
        }).on('change', '.tocelement-booktitle', function(e){
            e.stopPropagation();
            var input = $(this);
            var lang = input.attr('lang');
            var value = input.val();
            element.booktitleChange(lang, value);
        }).on('click', '.tocelement-tocitem', function(e){
            e.stopPropagation();
            var item = $(this);
            var chapid = item.attr('data-chapid');
            var addbefore = item.is('.tocelement-tocitem-addbeforepage');
            var addafter = item.is('.tocelement-tocitem-addafterpage');
            if (addbefore) {
                item.trigger('menubarevent', {eventname: 'addbeforepages'});
                item.trigger('menubarevent', {eventname: 'refreshtoc'});
            } else if (addafter) {
                item.trigger('menubarevent', {eventname: 'addafterpages'});
                item.trigger('menubarevent', {eventname: 'refreshtoc'});
            } else {
                item.trigger('gotopage', {chapid: chapid});
            };
        }).on('increasenesting', function(event, data){
            event.stopPropagation();
            element.indentPage();
            element.edit();
        }).on('decreasenesting', function(event, data){
            event.stopPropagation();
            element.unindentPage();
            element.edit();
        }).on('demotehere', function(event, data){
            event.stopPropagation();
            element.demotePage();
            element.edit();
        }).on('promotehere', function(event, data){
            event.stopPropagation();
            element.promotePage();
            element.edit();
        }).on('movepageforward', function(event, data){
            event.stopPropagation();
            element.movePageForward();
            element.edit();
        }).on('movepagebackward', function(event, data){
            event.stopPropagation();
            element.movePageBackward();
            element.edit();
        });
    }
    
    /******
     * Update the metadata of the element.
     ******/
    TocElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Trigger element_changed event.
     ******/
    TocElement.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'tocelement'});
    }

    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "markdownelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {
     *          "firstpage": {
     *              "chapid": "front",
     *              "content": "frontpage",
     *              "chapters": [...]
     *          }
     *      }
     *   }
     ******/
    TocElement.prototype.getData = function(){
        var result = {
            title: JSON.parse(JSON.stringify(this.booktitle)),
            toc: {
                type: this.type,
                metadata: JSON.parse(JSON.stringify(this.metadata)),
                data: this.getTocData()
            },
            pagetitles: this.getPagetitles()
        }
        return result;
    }
    
    /******
     * Returns the data of the toc
     * @returns {Object} the data of tocelement
     ******/
    TocElement.prototype.getTocData = function(){
        return $.extend(true, {}, this.data);
    };
    
    /******
     * Returns the pagetitles data
     * @returns {Object} pagetitles
     ******/
    TocElement.prototype.getPagetitles = function(){
        return $.extend(true, {}, this.pagetitles);
    };
    
    /******
     * Default settings
     ******/
    TocElement.defaults = {
        title: {
            common: 'Book name'
        },
        toc: {
            type: 'tocelement',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                firstpage: {}
            }
        },
        pagetitles: {},
        currentPage: '',
        defaultlang: 'en',
        settings: {
            mode: 'view',
            uilang: 'en',
            lang: 'en',
            role: 'student',
            readonly: true
        }
    }
    
    /******
     * Modes
     ******/
    TocElement.modes = {
        view: {
            editable: false,
            authorable: false
        },
        edit: {
            editable: true,
            authorable: false
        },
        author: {
            editable: true,
            authorable: true
        }
    }
    
    /******
     * Templates
     ******/
    TocElement.templates = {
        view: [
            '<div class="tocelement-control ffwidget-background"><span class="tocelement-booktitle"></span></div>',
            '<h1 class="tocelement-toctitle"></h1>',
            '<ul class="tocelement-toclist"></ul>'
        ].join('\n'),
        edit: [
            '<div class="tocelement-control ffwidget-background"><input type="text" class="tocelement-booktitle" /></div>',
            '<h1 class="tocelement-toctitle"></h1>',
            '<ul class="tocelement-toclist"></ul>',
            '<div class="tocelement-orderbar ffwidget-background"></div>'
        ].join('\n')
    }
    
    /******
     * Icons
     ******/
    TocElement.icons = {
        edit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>',
        ready: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-ready"><path style="stroke: none; fill: #0a0" d="M3 12 l8 8 l14 -14 l2 2 l-16 16 l-10 -10z"></path></svg>',
        cancel: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-cancelq"><path style="stroke: none; fill: #a00" d="M5 3 l10 10 l10 -10 l2 2 l-10 10 l10 10 l-2 2 l-10 -10 l-10 10 l-2 -2 l10 -10 l-10 -10z"></path></svg>',
        arrowup: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-arrowup"><path style="stroke: none;" d="M15 2 l10 10 l-6 -1 l0 10 l-8 0 l0 -10 l-6 1z" /></svg>',
        arrowdown: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-arrodown"><path style="stroke: none;" d="M15 28 l-10 -10 l6 1 l0 -10 l8 0 l0 10 l6 -1z" /></svg>',
        indent: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-indent"><path style="stroke: none;" d="M2 2 l26 0 l0 6 l-26 0z m0 10 m8 0 l12 0 l0 -3 l7 6 l-7 6 l0 -3 l-12 0z m-8 0 m0 10 l26 0 l0 6 l-26 0z" /></svg>',
        unindent: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-unindent"><path style="stroke: none;" d="M2 2 l26 0 l0 6 l-26 0z m0 10 m4 3 l7 -6 l0 3 l16 0 l0 6 l-16  0 l0 3z m-4 -3 m0 10 l26 0 l0 6 l-26 0z" /></svg>',
        promotehere: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-promote"><path style="stroke: none;" d="M2 2 l26 0 l0 4 l-26 0z m8 6 l18 0 l0 3 l-18 0z m-8 -2 m0 6 m0 3 l7 -4 l0 2 l19 0 l0 4 l-19 0 l0 2z m0 -3 m8 7 l18 0 l0 3 l-18 0z m-8 0 m0 5 l26 0 l0 4 l-26 0z" /></svg>',
        demotehere: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-demote"><path style="stroke: none;" d="M2 2 l26 0 l0 4 l-26 0z m8 6 l18 0 l0 3 l-18 0z m-2 3 l0 2 l12 0 l0 -2 l7 4 l-7 4 l0 -2 l-12 0 l0 2z m2 8 l18 0 l0 3 l-18 0z m-8 0 m0 5 l26 0 l0 4 l-26 0z" /></svg>'
    };
    
    /******
     * Orderbuttons
     ******/
    TocElement.orderbuttons = [
        {
            name: 'nesting_decrease',
            label: 'tocelement:decreasenesting',
            icon: TocElement.icons.unindent,
            action: 'decreasenesting'
        },
        {
            name: 'nesting_increase',
            label: 'tocelement:increasenesting',
            icon: TocElement.icons.indent,
            action: 'increasenesting'
        },
        {
            name: 'promotehere',
            label: 'tocelement:promotehere',
            icon: TocElement.icons.promotehere,
            action: 'promotehere'
        },
        {
            name: 'demotehere',
            label: 'tocelement:demotehere',
            icon: TocElement.icons.demotehere,
            action: 'demotehere'
        },
        {
            name: 'move_forward',
            label: 'tocelement:moveforwardpage',
            icon: TocElement.icons.arrowdown,
            action: 'movepageforward'
        },
        {
            name: 'move_backward',
            label: 'tocelement:movebackwardpage',
            icon: TocElement.icons.arrowup,
            action: 'movepagebackward'
        }
    ]
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    TocElement.elementinfo = {
        type: 'tocelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'tocelement',
        name: 'TocElement',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-codeelement"><path style="stroke: none;" d="M1 10 a5 5 0 0 1 10 0 l0 10 a5 5 0 0 1 -10 0z m2 0 l0 8 l6 -8 a3 3 0 0 0 -6 0z m6 2 l-6 8 a3 3 0 0 0 6 0z m5 -7 l2 0 l0 18 l2 0 l0 2 l-6 0 l0 -2 l2 0 l0 -14 l-2 0z m5 5 a5 5 0 0 1 10 0 l0 10 a5 5 0 0 1 -10 0z m2 0 l0 8 l6 -8 a3 3 0 0 0 -6 0z m6 2 l-6 8 a3 3 0 0 0 6 0z" /></svg>',
        description: {
            en: 'Table of Contents',
            fi: 'Sisällysluettelo',
            sv: 'Innehållsförteckning'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['general']
    }
    
    /******
     * Styles
     ******/
    TocElement.styles = [
        // Tocelement
        '.tocelement {position: absolute; top: 100px; bottom: 100px; left: 9px; overflow: hidden; width: 0; padding: 0; z-index: 20; font-family: helvetica, sans-serif; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; -webkit-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch;}',
        '.tocelement.tocelement-open {border: 1px solid #aaa; border-left: none; background-color: rgba(255,255,255,0.95); box-shadow: 5px 5px 15px rgba(0,0,0,0.3); transition: width 0.5s, padding 0.5s; width: auto;}',
        '.tocelement.tocelement-closing {border: 1px solid #aaa; border-left: none; background-color: rgba(255,255,255,0.95); box-shadow: 5px 5px 15px rgba(0,0,0,0.3); transition: width 0.5s, padding 0.5s; width: 0;}',
        // Control
        '.tocelement-control {min-height: 20px; padding: 2px 0.5em; font-weight: bold; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        '.tocelement-control svg {width: 20px; height: 20px;}',
        // Title
        '.tocelement-toctitle {font-size: 150%; font-family: helvetica, sans-serif; font-weight: bold; padding: 0.2em 0.5em; margin: 0; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        '.tocelement-booktitle {white-space: nowrap; width: 100%; box-sizing: border-box;}',
        // Toc-list
        '.tocelement ul.tocelement-toclist, .tocelement ul.tocelement-toclist ul {list-style: none; padding: 0; margin: 0; min-width: 15em;}',
        '.tocelement > ul.tocelement-toclist {margin: 0.5em 0 0.5em 0; padding-right: 2em; box-shadow: inset 0 2px 8px rgba(0,0,0,0.3); overflow-y: auto; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1;}',
        '.tocelement li.tocelement-tocitem {margin: 0; padding: 0 0 0 1em; cursor: pointer; background: url(\'data:image/svg+xml;charset=utf8,%3Csvg xmlns="http://www.w3.org/2000/svg" version="1.1" width="1.5em" height="1.5em" viewBox="0 0 30 30" %3E%3Cpath stroke="rgb(153,154,153)" stroke-width="1" style="fill: none;" d="M15 0 v30" /%3E%3C/svg%3E\') left top repeat-y, url(\'data:image/svg+xml;charset=utf8,%3Csvg xmlns="http://www.w3.org/2000/svg" version="1.1" width="1.5em" height="1.5em" viewBox="0 0 30 30" %3E%3Cpath stroke="rgb(153,154,153)" stroke-width="1" style="fill: none;" d="M15 15 h13" /%3E%3C/svg%3E\') left top no-repeat;}',
        '.tocelement li.tocelement-tocitem:last-child {background: url(\'data:image/svg+xml;charset=utf8,%3Csvg xmlns="http://www.w3.org/2000/svg" version="1.1" width="1.5em" height="1.5em" viewBox="0 0 30 30" %3E%3Cpath stroke="rgb(153,154,153)" stroke-width="1" style="fill: none;" d="M15 0 v15 h13" /%3E%3C/svg%3E\') left top no-repeat;}',
        '.tocelement ul.tocelement-toclist > li.tocelement-tocitem {background: transparent;}',
        '.tocelement > ul.tocelement-toclist > li.tocelement-tocitem > ul.tocelement-sublist > li {background: transparent; padding-left: 0;}',
        '.tocelement .tocelement-tocitem-title {padding: 0 0.4em; min-height: 1.2em;}',
        '.tocelement .tocelement-tocitem-title:hover {border-radius: 0.5em;}',
        '.tocelement .tocelement-tocitem-currentpage > .tocelement-tocitem-title {font-weight: bold; background-color: rgba(256, 221, 0, 0.5); border-radius: 0.5em;}',
        '.tocelement-listplaceholder {color: #888; font-size: 90%;}',
        '.tocelement-beforelist, .tocelement-afterlist {font-style: italic; color: #555;}',
        '.tocelement-tocitem-pagenum {margin-right: 0.4em;}',
        // Orderbar
        '.tocelement-orderbar {min-height: 2em; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        '.tocelement-orderbuttons {margin: 0.2em;}',
        '.tocelement-orderbuttons li {padding: 4px;}',
        '.tocelement-orderbuttons .tocelement-buttonicon, .tocelement-orderbuttons svg {width: 20px; height: 20px;}',
        ''
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    TocElement.localization = {
        "en": {
            "tocelement:toc": "Table of Contents",
            "tocelement:increasenesting": "Increase the nesting level.",
            "tocelement:decreasenesting": "Decrease the nesting level.",
            "tocelement:moveforwardpage": "Move the page forward.",
            "tocelement:movebackwardpage": "Move the page backward",
            "tocelement:promotehere": "Promote the page level here.",
            "tocelement:demotehere": "Demote the page level here"
        },
        "fi": {
            "tocelement:toc": "Sisällysluettelo",
            "tocelement:increasenesting": "Siirrä alemmalle tasolle.",
            "tocelement:decreasenesting": "Siirrä ylemmälle tasolle.",
            "tocelement:moveforwardpage": "Siirrä sivu loppuun päin.",
            "tocelement:movebackwardpage": "Siirrä sivu alkuun päin",
            "tocelement:promotehere": "Siirrä ylemmälle tasolle tässä.",
            "tocelement:demotehere": "Siirrä alemmalle tasolle tässä."
        },
        "sv": {
            "tocelement:toc": "Innehållsförteckning",
            "tocelement:increasenesting": "Increase the nesting level.",
            "tocelement:decreasenesting": "Decrease the nesting level.",
            "tocelement:moveforwardpage": "Move the page forward.",
            "tocelement:movebackwardpage": "Move the page backward",
            "tocelement:promotehere": "Promote the page level here.",
            "tocelement:demotehere": "Demote the page level here."
        }
    }
    
    
    /***************************************************************
     * Class for tocpage
     ***************************************************************/
    var TocPage = function(data, level, parent){
        this.level = level;
        this.parent = parent;
        this.chapid = data.chapid;
        this.content = data.content;
        this.types = data.types && data.types.slice() || [];
        if (data.before && data.before.chapid) {
            this.before = new TocPage(data.before, level + 1, this);
        };
        if (data.after && data.after.chapid) {
            this.after = new TocPage(data.after, level + 1, this);
        };
        this.chapters = [];
        for (var i = 0, len = data.chapters.length; i < len; i++) {
            this.chapters.push(new TocPage(data.chapters[i], level + 1, this));
        };
    };
    
    TocPage.prototype.getLinear = function(){
        var list = [this];
        if (this.before && this.before.chapid) {
            list = list.concat(this.before.getLinear());
        };
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            list = list.concat(this.chapters[i].getLinear());
        };
        if (this.after && this.after.chapid) {
            list = list.concat(this.after.getLinear());
        };
        return list;
    };
    
    TocPage.prototype.fixLevels = function(){
        var page;
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            page = this.chapters[i];
            page.level = this.level + 1;
            page.fixLevels();
        };
    };
    
    TocPage.prototype.fixParents = function(){
        var page;
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            page = this.chapters[i];
            page.parent = this;
            page.fixParents();
        };
    };
    
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(TocElement.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(TocElement.localization);
    }

    
    //if (typeof($.fn.elementset) === 'function') {
    //    $.fn.elementset('addelementtype', TocElement.elementinfo);
    //}
    //if (typeof($.fn.elementpanel) === 'function') {
    //    $.fn.elementpanel('addelementtype', TocElement.elementinfo);
    //}
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var tocelem = new TocElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.tocelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in tocelement.');
            return false;
        }
    }

})(jQuery);