/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - Emathtable
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(jQuery.fn.emathtable) === 'undefined' && jQuery.fn.emathtable.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

;(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true, {}, Filltable.defaults.userdata, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        for (var i = 0, len = this.data.total || 0; i < len; i++) {
            this.data.results[i] = 0;
        };
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime;
        };
    };
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };
    
    /**
     * Add a new answer (and mark correct or not, if given)
     * @param {String}   str       The answer
     * @param {Number}   index     The index of the answer
     * @param {Boolean}  correct   The correctness of the answer
     */
    Userdata.prototype.addAnswer = function(str, index, correct){
        var length = this.data.answers.length;
        if (length === 0) {
            this.data.answers.push([]);
            length++;
        };
        this.data.answers[length - 1][index] = str;
        this.data.results[index] = correct | 0;
        var nowtime = (new Date()).getTime();
        if (!this.metadata.created) {
            this.metadata.created = nowtime;
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime;
        this.metadata.modifier = this.settings.username;
    };
    
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    }
    
    //Userdata.prototype.isAnswer = function(str){
    //    return (this.data.answers.indexOf(str) > -1);
    //};
    
    /**
     * Get the answer.
     * @param {Number} [index]   optional index of the answer
     * @returns {Array|String}   either the whole answer array or index'th item, if index is given
     */
    Userdata.prototype.getAnswer = function(index){
        var answer = this.data.answers[this.data.answers.length - 1] || [];
        if (typeof(index) === 'number') {
            answer = answer[index];
        };
        return answer;
    }
    
    Userdata.prototype.getNewId = function(){
        var idparts = ['filltableanswer'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };

    
    /******
     * Filltable question element
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for element
     ******/
    var Filltable = function(place, options){
        this.place = $(place);
        this.place.addClass('filltableelement');
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    Filltable.prototype.init = function(options){
        options = $.extend(true, {}, Filltable.defaults, options);
        this.settings = options.settings;
        if (this.settings.mode === 'view') {
            // Use 'appendview' instead of 'view' by default. If the multichoice is already
            // solved, change to 'view' later.
            this.settings.mode = 'appendview'
        };
        this.solvable = this.settings.users && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.data = options.data;
        this.metadata = options.metadata;
        this.assignmentid = this.place.attr('data-element-name');
        var userdata = options.userdata
        userdata.data.total = this.data.correct.length;
        userdata.settings = {username: this.settings.username, lang: this.settings.lang || ''};
        this.userdata = new Userdata(userdata);
        this.setMode(this.settings.mode);
        this.setAttrs();
    }
    
    Filltable.prototype.getData = function(){
        var data = {
            type: 'filltableelement',
            metadata: $.extend(true, {}, this.metadata),
            data: $.extend(true, {}, this.data)
        };
        return data;
    }
    
    Filltable.prototype.setAttrs = function(){
        this.place.attr('lang', escapeHTML(this.metadata.lang))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .attr('data-tags', escapeHTML(this.metadata.tags.join(',')))
            .attr('data-quiztotal', escapeHTML(this.data.correct.length))
            .addClass('quizquestion');
    };
    
    Filltable.prototype.setMode = function(mode, dontredraw){
        // If mode is incorrect, use 'view' as a default mode.
        if (!Filltable.modes[mode]) {
            mode = 'view';
        }
        this.place.attr('data-elementmode', mode);
        var modedata = Filltable.modes[mode];
        this.editable = modedata.editable;
        this.authorable = modedata.authorable;
        this.reviewable = modedata.reviewable;
        this.appendable = modedata.appendable;
        this.settings.mode = mode;
        if (!dontredraw) {
            this.show();
        }
        this.addHandlers();
    };

    Filltable.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    }
    
    Filltable.prototype.view = function(){
        var name = this.assignmentid;
        var html = Filltable.templates.view;
        this.place.html(html);
        var qarea = this.place.children('.filltableelement-questionarea');
        var qdata = {
            type: 'elementset',
            metadata: {},
            data: {
                contents: this.data.contents,
                contentdata: this.data.contentdata
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'view',
                role: this.settings.role
            }
        }
        qarea.elementset(qdata);
        if (!this.userdata.isClosed()) {
            this.place.trigger('get_userdata', {refid: name});
            this.place.trigger('getcontentbyref', {refs: [name]});
        } else {
            this.appendUserData();
        }
        this.place.find('.filltableelement-questionarea .emathtable').off('focusout');
        this.place.find('.mathquill-rendered-math:not(.mathquill-editable) > .textarea > textarea').remove();
    }
    
    //Filltable.prototype.edit = function(){
    //    var html = Filltable.templates.edit;
    //    this.place.html(html);
    //    var text = this.data.text;
    //    var qarea = this.place.children('.shortanswer-questionarea');
    //    var aarea = this.place.find('> .shortanswer-answerarea ul');
    //    var labelarea = this.place.find('> .shortanswer-answerarea .shortanswer-correctlabel');
    //    labelarea.html(ebooklocalizer.localize('shortanswer:correct_answers', this.settings.uilang));
    //    qarea.html('<div class="shortanswer-markdown"></div>');
    //    var mddata = {
    //        type: 'markdownelement',
    //        metadata: {},
    //        data: {
    //            text: text
    //        },
    //        settings: {
    //            uilang: this.settings.uilang,
    //            mode: 'edit',
    //            role: this.settings.role
    //        }
    //    };
    //    qarea.children('.shortanswer-markdown').markdownelement(mddata);
    //    var corrects = [], correct;
    //    var trash = (this.data.corrects.length > 1 ? '<span class="shortanswer-trash">'+Filltable.icons.trash+'</span>' : '');
    //    for (var i = 0, len = this.data.corrects.length; i < len; i++) {
    //        var correct = this.data.corrects[i];
    //        corrects.push('<li data-correct-number="'+i+'"><input type="text" class="shortanswer-correcttext" value="'+correct+'" data-correct-number="'+i+'" />'+trash+'</li>');
    //    };
    //    corrects.push('<li data-correct-number="'+i+'" class="shortanswer-addcorrect"><div>+</div></li>');
    //    aarea.append(corrects.join('\n'));
    //}
    
    // Dummy edit for now.
    Filltable.prototype.edit = Filltable.prototype.view;
    
    Filltable.prototype.appendUserData = function(){
        var inputs = this.place.find('> .filltableelement-questionarea .mathquill-editable');
        var useranswer, mqtext, mqbox;
        var solved = this.userdata.isSolved();
        var closed = this.userdata.isClosed();
        var currentanswers = this.userdata.getAnswer();
        if ((this.settings.feedback && solved && this.appendable) || (closed && this.appendable)) {
            this.setMode('view');
        } else {
            if (solved && this.settings.feedback || closed) {
                for (var i = 0, len = inputs.length; i < len; i++) {
                    mqbox = inputs.eq(i);
                    mqbox.mathquill('revert');
                    mqbox.addClass('filltableelement-useranswer');
                    mqtext = escapeHTML(currentanswers[i]);
                    mqbox.html(mqtext).addClass('filltableelement-math').mathquill();
                    if (this.isCorrect(currentanswers[i], i)) {
                        mqbox.addClass('filltableelement-iscorrect').removeClass('filltableelement-iswrong');
                    } else {
                        mqbox.addClass('filltableelement-iswrong').removeClass('filltableelement-iscorrect');
                    };
                };
            } else {
                for (var i = 0, len = inputs.length; i < len; i++) {
                    mqbox = inputs.eq(i);
                    mqbox.mathquill('revert');
                    mqbox.html(escapeHTML(currentanswers[i] || ''));
                    mqbox.mathquill('editable');
                }
            };
        };
    };
    
    Filltable.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    Filltable.prototype.addHandlers = function(){
        this.removeHandlers();
        this.addHandlersCommon();
        if (this.editable) {
            this.addHandlersEdit();
        } else {
            this.addHandlersView();
        }
        if (this.appendable) {
            this.addHandlersAppendable();
        };
    };
    
    Filltable.prototype.addHandlersCommon = function(){
        var tfill = this;
        this.place.on('setmode', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            tfill.setMode(data);
        });
        this.place.on('reply_get_userdata', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            data.settings = {
                username: tfill.settings.username,
                lang: tfill.settings.lang || '',
                uilang: tfill.settings.uilang,
                role: tfill.settings.role
            };
            tfill.userdata = new Userdata(data);
            tfill.appendUserData();
        });
        this.place.on('reply_getcontentbyref', function(e, data){
            var refid = tfill.assignmentid;
            var reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: tfill.settings.username,
                    lang: tfill.settings.lang || '',
                    uilang: tfill.settings.uilang,
                    role: tfill.settings.role
                };
                tfill.userdata = new Userdata(userdata);
                tfill.appendUserData();
            };
        });
        this.place.on('showcorrectanswer', function(event, data){
            event.stopPropagation();
            tfill.showCorrect();
        });
        this.place.on('clearanswer', function(event, data){
            event.stopPropagation();
            tfill.clearAnswer();
        });
        this.place.on('getdata', function(e){
            var data = tfill.getData();
            tfill.place.data('[[elementdata]]', data);
        });
    }

    Filltable.prototype.addHandlersEdit = function(){
        var tfill = this;
        //this.place.on('change', 'input.shortanswer-correcttext', function(e, data){
        //    e.stopPropagation();
        //    e.preventDefault();
        //    var input = $(this);
        //    var value = input.val();
        //    var index = input.attr('data-correct-number') | 0;
        //    tfill.data.corrects[index] = value;
        //    tfill.changed();
        //});
        //this.place.on('keyup', 'input.shortanswer-correcttext', function(e, data){
        //    e.stopPropagation();
        //    e.preventDefault();
        //    var input = $(this);
        //    var nextli = input.closest('li').next('li');
        //    var next = nextli.children('input');
        //    switch (e.which) {
        //        case 13:
        //            if (next.length > 0) {
        //                next.focus().select();
        //            } else {
        //                nextli.click();
        //            };
        //            break;
        //        default:
        //            break;
        //    };
        //});
        //this.place.on('element_changed', '.shortanswer-markdown', function(e, data){
        //    e.stopPropagation();
        //    e.preventDefault();
        //    var mdelem = $(this);
        //    mdelem.trigger('getdata');
        //    var mddata = mdelem.data('[[elementdata]]');
        //    tfill.data.text = mddata.data.text;
        //    tfill.changed();
        //});
        //this.place.on('click', '.shortanswer-addcorrect', function(e, data){
        //    e.stopPropagation();
        //    e.preventDefault();
        //    var button = $(this);
        //    var index = button.attr('data-correct-number') | 0;
        //    button.before('<li data-correct-number="'+index+'"><input type="text" class="shortanswer-correcttext" data-correct-number="'+index+'" /><span class="shortanswer-trash">'+Textfill.icons.trash+'</span></li>');
        //    button.attr('data-correct-number', index + 1);
        //    button.prev().children('input').focus();
        //});
        //this.place.on('click', '.shortanswer-corrects li .shortanswer-trash', function(e, data){
        //    e.stopPropagation();
        //    e.preventDefault();
        //    var button = $(this);
        //    var liitem = button.closest('li');
        //    var list = liitem.closest('ul.shortanswer-corrects');
        //    var input = liitem.children('input');
        //    var index = (input.attr('data-correct-number') | 0);
        //    tfill.data.corrects.splice(index, 1);
        //    liitem.remove();
        //    var items = list.children('li');
        //    for (var i = 0, len = items.length; i < len; i++) {
        //        items.eq(i).attr('data-correct-number', i).children('input').attr('data-correct-number', i);
        //    };
        //    tfill.changed();
        //});
    };
    
    Filltable.prototype.addHandlersView = function(){}
    
    Filltable.prototype.addHandlersAppendable = function(){
        var tfill = this;
        this.place.on('focusout', '.filltableelement-questionarea .mathquill-editable', function(e){
            e.preventDefault();
            var input = $(this);
            var value = input.mathquill('latex');
            var allblanks = tfill.place.find('.filltableelement-questionarea .mathquill-editable');
            var index = allblanks.index(input);
            var userdata = tfill.userdata;
            var oldanswer = userdata.getAnswer(index);
            if (!(typeof(oldanswer) === 'undefined' && value === '') && value !== oldanswer) {
                userdata.addAnswer(value, index, tfill.isCorrect(value, index));
                var answers = userdata.getAnswer();
                userdata.solved(tfill.isCorrect(answers));
                if (tfill.settings.feedback) {
                    // TODO!
                    tfill.saveUserData({sendforward: false});
                } else {
                    tfill.saveUserData();
                };
            };
        });
        this.place.on('close_quizquestion', function(event, data){
            event.stopPropagation();
            tfill.close();
            var localsave = !!data.localsave;
            tfill.saveUserData({localsave: localsave, sendforward: false});
        });
    }
    
    /******
     * Check, if the given answer is correct. Case insensitive.
     * @param {Array|String} answers    An array of all given answers or the answer of index
     * @param {Number} [index]          An optional index of the answer. If index is not given, check all
     *                                  answers in the array.
     ******/
    Filltable.prototype.isCorrect = function(answers, index){
        var correct = this.data.correct;
        var found = true;
        if (typeof(index) === 'number' && index >= 0 && index < correct.length) {
            found = (correct[index].toLowerCase() === (answers || '').toLowerCase());
        } else {
            for (var i = 0, len = correct.length; i < len && found; i++) {
                found = found && (correct[i].toLowerCase() === (answers[i] || '').toLowerCase());
            };
        };
        return found;
    }
    
    Filltable.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'filltableelement'});
    }
    
    /******
     * Close the quiz question.
     ******/
    Filltable.prototype.close = function(){
        this.userdata.closed(true);
    };
    
    Filltable.prototype.saveUserData = function(options){
        if (this.solvable) {
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            var localsave = options && !!options.localsave;
            if (options && options.sendforward) {
                var senddata = {};
                senddata[name] = userdata;
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    /******
     * Show correct answer
     ******/
    Filltable.prototype.showCorrect = function(){
        this.setMode('view');
        var inputs = this.place.find('> .filltableelement-questionarea .mathquill-editable');
        var mqbox, mqtext;
        if (!this.feedback) {
            for (var i = 0, len = inputs.length; i < len; i++) {
                mqbox = inputs.eq(i);
                mqbox.mathquill('revert');
                mqbox.addClass('filltableelement-correctanswer');
                mqtext = escapeHTML(this.data.correct[i]);
                mqbox.html(mqtext).addClass('filltableelement-math').mathquill();
            };
        };
    };
    
    /******
     * Clear answer
     ******/
    Filltable.prototype.clearAnswer = function(){
        var userdata = {settings: {username: this.settings.username, lang: this.settings.lang || ''}};
        this.userdata = new Userdata(userdata);
        this.setMode('appendview');
        this.appendUserData();
    };
    
    
    
    /******
     * Set style-tag if needed
     ******/
    Filltable.prototype.setStyles = function(){
        if ($('head style#filltableelement-style').length === 0) {
            $('head').append('<style id="filltableelement-style" type="text/css">'+Filltable.styles+'</style>');
        }
    }

    /******
     * Mathquillify the string, i.e., replace \( \)-pairs and $ $-pairs with a span.
     ******/
    Filltable.prototype.mathfy = function(str){
        str = str.replace(/\$([^$]*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>');
        return str;
    }
    
    
    Filltable.styles = [
        '.ebook-elementset.filltableelement-questionarea {clear: none;}',
        '.filltableelement-questionarea > .ebook-elementset-body > .elementset-elementwrapper {margin: 0;}',
        '.filltableelement-questionarea > .ebook-elementset-body > .elementset-elementwrapper > [data-ebookelement-type="wikielement"], .filltableelement-questionarea > .ebook-elementset-body > .elementset-elementwrapper > [data-ebookelement-type="markdownelement"] {margin: 0; padding: 0;}',
        '.filltableelement-questionarea > .ebook-elementset-body > .elementset-elementwrapper > [data-ebookelement-type="tableelement"] .mathquill-embedded-latex .solution {white-space: nowrap;}',
        '.filltableelement-questionarea > .ebook-elementset-body > .elementset-elementwrapper > [data-ebookelement-type="tableelement"] .mathquill-embedded-latex .solution .mathquill-editable {white-space: nowrap; min-height: 1.2em; min-width: 3em; background-color: white;}',
        '.filltableelement-iswrong, .filltableelement-iscorrect {font-weight: bold; border-bottom: 1px solid black; background-color: rgba(255,255,255,0.8);}',
        '.questionhighlights .filltableelement-iswrong {color: #a00; font-weight: bold; text-decoration: line-through; border-bottom: none; background-color: transparent;}',
        '.questionhighlights .filltableelement-iscorrect {color: black; font-weight: bold; border-bottom: 2px solid #0a0; background-color: transparent;}',
        '.filltableelement-correctanswer {background-color: #fafafa; border-radius: 0.2em; box-shadow: inset 1px 1px 2px rgba(0,0,0,0.3); padding: 0.2em 0.5em; color: #070;}',
        '.filltableelement-questionarea .emathtable[tabletype] table.emtable tbody td {padding: 0.1em 0.1em;}'
    ].join('\n');
    
    Filltable.defaults = {
        type: 'filltableelement',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            lang: '',
            tags: []
        },
        data: {
            title: '',
            contents: [],
            contentdata: {},
            correct: []
        },
        userdata: {
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                answers: [],
                results: [],
                total: 0,
                solved: false
            },
            settings: {
                username: 'Anonymous123',
                uilang: 'en',
                lang: 'en'
            }
        },
        settings: {
            mode: 'view',
            role: 'student',
            uilang: 'en',
            username: 'Anonymous',
            feedback: true
        }
    };
    
    Filltable.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        appendview: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: true
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            appendable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            appendable: false
        }
    };

    Filltable.templates = {
        view: [
            '<div class="filltableelement-questionarea ebookassignment-questionarea"></div>'
        ].join('\n'),
        edit: [
        ].join('\n')
    };

    Filltable.icons = {
        trash: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>'
    };
    
    /******
     * Localization strings
     ******/
    Filltable.localization = {
        "en": {
            "filltableelement:correct_answers": "Correct answers"
        },
        "fi": {
            "filltableelement:correct_answers": "Oikeat vastaukset"
        }
    };
    

    Filltable.elementinfo = {
        type: 'filltableelement',
        elementtype: ['assignments','quizelement'],
        jquery: 'filltableelement',
        name: 'Filltable',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice"><text x="15" y="18" text-anchor="middle" style="font-weight: bold; font-size: 15px; font-family: sans;">??</text><path style="stroke: none;" d="M5 22 l22 0 l0 8 l-22 0z m1 1 l0 6 l20 0 l0 -6z" /></svg>',
        description: {
            en: 'Table fill',
            fi: 'Taulukkoaukkotehtävä',
            sv: 'Tabelluppgift'
        },
        roles: ['teacher', 'author'],
        classes: ['viewonly']
    };

    Filltable.ftelementinfo = {
        type: 'filltable',
        elementtype: ['assignments','quizelement'],
        jquery: 'filltableelement',
        name: 'Filltable',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice"><text x="15" y="18" text-anchor="middle" style="font-weight: bold; font-size: 15px; font-family: sans;">??</text><path style="stroke: none;" d="M5 22 l22 0 l0 8 l-22 0z m1 1 l0 6 l20 0 l0 -6z" /></svg>',
        description: {
            en: 'Table fill',
            fi: 'Taulukkoaukkotehtävä',
            sv: 'Tabelluppgift'
        },
        roles: ['teacher', 'author'],
        classes: ['viewonly']
    };

    // Register multichoice as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Filltable.elementinfo);
        $.fn.elementset('addelementtype', Filltable.ftelementinfo);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Filltable.elementinfo);
        $.fn.elementpanel('addelementtype', Filltable.ftelementinfo);
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Filltable.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Filltable.localization);
    }


    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var tfill = new Filltable(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.filltableelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in filltableelement.');
            return false;
        }
    };
    
})(jQuery)