/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Solution tool to show and edit SolutionElements.
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var SolutionTool = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.initHandlers();
        this.show();
        this.initReady();
    };

    /******
     * Init the solutiontool
     * @param {Object} options - settings for this solutiontool
     ******/
    SolutionTool.prototype.init = function(options){
        options = $.extend(true, {}, SolutionTool.defaults, options);
        this.place.addClass('solutiontool-initing');
        this.assignmentid = options.data.assignmentid;
        this.settings = options.settings;
        this.allSolutions = {};
        this.allSolutions[this.settings.username] = options.data.contents;
        this.solutiondata = {};
        this.solutiondata[this.settings.username] = options.data.contentdata;
        this.setMode(this.settings.mode);
        this.setRole(this.settings.role);
        this.allReviews = {
            refs: {},
            contentdata: {},
            contentinfo: {},
            anchors: {}
        };
        this.solutioncount = 0;
        this.unrevcount = 0;
        this.setAttrs();
        this.currentSolution = {};
        this.setCurrent();
    };
    
    /**
     * Initing is ready
     */
    SolutionTool.prototype.initReady = function() {
        this.place.removeClass('solutiontool-initing');
    };
    
    /******
     * Init all event handlers
     ******/
    SolutionTool.prototype.initHandlers = function(){
        this.removeHandlers();
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersNonEdit();
        };
        if (this.reviewable) {
            this.initHandlersReview();
        };
    };
    
    /******
     * Remove all event handlers
     ******/
    SolutionTool.prototype.removeHandlers = function(){
        this.place.off();
    };
    
    /******
     * Init event handlers common for all modes
     ******/
    SolutionTool.prototype.initHandlersCommon = function(){
        var tool = this;
        var username = this.settings.username;
        // Set the mode of solutiontool with event
        this.place.off('setmode').on('setmode', function(event, mode){
            event.stopPropagation();
            event.preventDefault();
            tool.setMode(mode);
            tool.show();
        });
        // Select area to view
        this.place.off('click', '.solutiontool-selector button').on('click', '.solutiontool-selector button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var seldata = button.attr('data-selectarea');
            var select = !button.is('.buttonselected');
            button.parent().children('button').removeClass('buttonselected');
            if (select) {
                button.addClass('buttonselected');
                tool.place.attr('data-selectedarea', seldata);
                if (seldata === 'view' && !tool.place.find('.solutiontool-viewarea').is('.solutiontool-view-inited')) {
                    tool.edit();
                    tool.place.find('.solutiontool-viewarea').addClass('solutiontool-view-inited');
                }
            } else {
                tool.place.removeAttr('data-selectedarea')
            };
        });
        // Click for tabs in tablist of solutiontool
        this.place.off('click', '.solutiontool-solutiontablist > li').on('click', '.solutiontool-solutiontablist > li', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var tab = $(this);
            var index = tab.attr('data-solution-tabnumber') | 0;
            var place = tab.closest('.solutiontool-area');
            var wasopen = tab.parent().children('.solutiontool-currenttab').attr('data-solution-isopen') === 'true';
            var isopen = tab.attr('data-solution-isopen') === 'true';
            var username = tab.closest('[data-username]').attr('data-username');
            tool.showTab(index, username, place);
            if (wasopen && !isopen) {
                tool.place.trigger('stopediting');
            };
            if (!wasopen && isopen) {
                tool.place.trigger('startediting');
            };
        });
        // Click for buttons in solutiontool. (Trigger corresponding actions.)
        this.place.off('click', 'button.solutiontool-button').on('click', 'button.solutiontool-button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            if (action) {
                button.trigger(action);
            };
        });
        // Send solution button action. (Get data and trigger both saving and sending events.)
        this.place.off('sendsolution').on('sendsolution', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var nonsent = tool.getOpenSolution();
            var senddata = {};
            var setdata;
            var solid;
            var soldata;
            for (var i = 0, len = nonsent.length; i < len; i++) {
                // For all unsent solutions. (Should be only one.)
                solid = nonsent[i];
                soldata = tool.solutiondata[solid];
                soldata.name = solid;
                soldata.data.issent = true;
                //// TODO old format
                ////senddata[solid] = {
                ////    name: solid,
                ////    reftype: 'solutiondata',
                ////    reftos: [tool.assignmentid],
                ////    sendto: 'teacher',
                ////    sendfrom: tool.settings.username,
                ////    senddate: (new Date()).toString(),
                ////    read: true,
                ////    recipientopened: false
                ////};
                ////setdata = {
                ////    reftos: [soldata.metadata.ref],
                ////    reftype: 'solutiondata',
                ////    name: solid,
                ////    refcontentdata: soldata
                ////};
                ////// Trigger setcontentbyref-event (saving and sending).
                ////tool.place.trigger('setcontentbyref', [setdata, senddata[solid]]);
                var recipients = tool.settings.users.getRecipients('solution');
                var reclist = [];
                for (var i = 0, len = recipients.length; i < len; i++) {
                    reclist.push(recipients[i].send_to);
                };
                // New with storage
                var solutiondata = {
                    name: solid,
                    contentType: 'solution',
                    lang: 'common',
                    anchors: [tool.assignmentid],
                    contentdata: soldata,
                    contentinfo: {
                        name: solid,
                        contentType: 'solution',
                        lang: 'common',
                        pubtype: 'teacher',
                        sendto: reclist,
                        sendfrom: tool.settings.username,
                        timestamp: (new Date()).getTime(),
                        read: true,
                        recipientopened: false,
                        isoutbox: true
                    }
                };
                tool.place.trigger('setcontent', [solutiondata, true]);
                tool.place.trigger('savecontent');
                //tool.place.trigger('sendcontent', solutiondata.contentinfo);
            };
            //tool.show();
            tool.edit();
            tool.place.trigger('stopediting');
        });
        // New with storage
        this.place.off('reply_getcontent').on('reply_getcontent', function(event, data){
            // Don't handle, if the target was some children in DOM-tree.
            if (event.target === this && data) {
                event.stopPropagation();
                switch (data.contentType) {
                    case 'solution':
                        // Sort solutions with time and username
                        tool.sortSolutions(data);
                        // Get info about reviews
                        tool.getReviewInfo();
                        // Come back to .show() again, now that the solutions have been fetched.
                        tool.show();
                        break;
                    case 'review':
                        // Sort all reviews
                        tool.sortReviews(data);
                        // Show the list of available reviews.
                        tool.showReviewlist();
                        // Update the attribute indicating unpublished reviews
                        tool.setUnpubCount();
                        break;
                    default:
                        break;
                }
            };
        });
        this.place.off('gotolink').on('gotolink', function(event, data){
            var link = data && data.data && data.data.link;
            if (link && link.solution) {
                event.stopPropagation();
                tool.goToLink(data);
            };
        });
        this.place.on('click', '.solutiontool-reviewlist-item', function(event, forceshow){
            event.stopPropagation();
            var item = $(this);
            var reviewid = item.attr('data-reviewid');
            var solutionid = item.attr('data-solutionid');
            var isselected = item.is('.selected');
            var isreviewtool = item.closest('.solutiontool-reviewarea').length > 0;
            if (!isselected || forceshow) {
                tool.showReviewData(solutionid, reviewid, isreviewtool);
            } else {
                tool.showReviewData(solutionid, '', isreviewtool)
            };
        });
        this.place.on('click', '.solutiontool-reviewstart-button', function(event, data){
            event.stopPropagation();
            var item = $(this);
            var solarea = item.closest('.solutiontool-reviewtools').next('.solutiontool-content-solution');
            var solutionid = solarea.attr('data-solutionid') || '';
            item.parent().children().removeClass('selected').children('button').removeClass('buttonselected');
            tool.showReviewData(solutionid, '', true, true);
        });
        this.place.off('reply_getcontentinfo').on('reply_getcontentinfo', function(event, data){
            event.stopPropagation();
            if (data.contentType === 'review') {
                tool.setReviewInfo(data);
            };
        });
        this.place.off('review_read').on('review_read', function(event, revid){
            event.stopPropagation();
            tool.markReviewRead(revid);
        });
        this.place.off('click', '.assignment-teacherbuttons button').on('click', '.assignment-teacherbuttons button', function(event, data) {
            event.stopPropagation();
            var button = $(this);
            var action = button.attr('data-action');
            var assignment = tool.place.prev('.ebook-assignment');
            assignment.trigger(action);
        });
        this.place.off('dragstart', '.assignment-teacherbuttons button').on('dragstart', '.assignment-teacherbuttons button', function(event, data) {
            event.stopPropagation();
            var button = $(this);
            var action = button.attr('data-action');
            var assignment = tool.place.prev('.ebook-assignment');
            assignment.trigger('customdragstart', [{action: action, event: event.originalEvent}]);
        });
        this.place.off('element_modified').on('element_modified', function(event, revid){
            event.stopPropagation();
            // Stop element_modified from solution. We don't want changes in index.
        });
    };
    
    /******
     * Init event handlers for noneditable modes
     ******/
    SolutionTool.prototype.initHandlersNonEdit = function(){
        var tool = this;
    };
    
    /******
     * Init event handlers for reviewable modes
     ******/
    SolutionTool.prototype.initHandlersReview = function(){
        var tool = this;
        this.place.off('click', '.solutiontool-review-username').on('click', '.solutiontool-review-username', function(event){
            // Open / close the accordion
            var header = $(this);
            var content = header.next('.solutiontool-reviewitem');
            var user = header.attr('data-username');
            var isopen = header.attr('data-isopen') || false;
            if (isopen) {
                header.removeAttr('data-isopen');
                content.empty();
            } else {
                header.attr('data-isopen', 1);
                tool.showReview(content, user);
            };
        });
        this.place.off('elementreview_changed').on('elementreview_changed', function(event, data){
            event.stopPropagation();
            if (data) {
                tool.setReview(data);
            };
        });
        // Teacherbuttons
        this.place.off('click', '.solutiontool-teacherbuttons button').on('click', '.solutiontool-teacherbuttons button', function(event, data){
            event.stopPropagation();
            var action = $(this).attr('data-action');
            if (action) {
                $(this).trigger(action);
            };
        });
        // Click for button in reviewtool.
        this.place.off('click', 'button.solutiontool-reviewbutton-send').on('click', 'button.solutiontool-reviewbutton-send', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            var actiondata = button.closest('.solutiontool-content[data-unsentrevs="true"]').children('.solutiontool-content-solution').attr('data-unsentreviews');
            if (actiondata) {
                actiondata = actiondata.split(' ');
            };
            if (action) {
                button.trigger(action, [actiondata]);
            };
        });
        this.place.off('sendreviews').on('sendreviews', function(event, data) {
            event.stopPropagation();
            tool.sendReviews(data);
        });
    };
    
    /******
     * Init event handlers for editable modes
     ******/
    SolutionTool.prototype.initHandlersEdit = function(){
        var tool = this;
        // Element (solutionelement) has changed. Get data and save it.
        this.place.off('element_changed').on('element_changed', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var eltype = data.type;
            var element = $(event.target);
            var elname = element.attr('data-element-name');
            element.trigger('getdata');
            var eldata = element.data('[[elementdata]]');
            if (eldata && eldata.metadata) {
                eldata.metadata.lang = 'common';
            };
            tool.setData(elname, eldata);
            var solutiondata = {
                name: elname,
                contentType: 'solution',
                lang: 'common', // tool.settings.lang,
                anchors: [tool.assignmentid],
                contentdata: eldata,
                contentinfo: {
                    name: elname,
                    contentType: 'solution',
                    lang: 'common', // tool.settings.lang,
                    pubtype: 'own',
                    sendto: [],
                    sendfrom: tool.settings.username,
                    timestamp: (new Date()).getTime(),
                    isoutbox: true
                }
            };
            tool.place.trigger('setcontent', [solutiondata, true]);
        });
        // New solution button action. (Create a new empty solution.)
        this.place.off('newsolution').on('newsolution', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            tool.showEditButtons(['send']);
            tool.addNew();
        });
        // Copy solution button action. (Create a new solution as a copy of old one.)
        this.place.off('copysolution').on('copysolution', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            tool.showEditButtons(['send']);
            var user = tool.settings.username;
            tool.addNew(tool.currentSolution[user]);
        });
    };
    
    /******
     * Show the element (common part for all modes)
     ******/
    SolutionTool.prototype.show = function(){
        var username = this.settings.username;
        if (!this.solutionsFetched) {
            //If we haven't tried to fetch solutions from outside yet.
            //Trigger 'getcontent' and listen for reply.
            this.place.trigger('getcontent', {anchors: [this.assignmentid], contentType: ['solution']});
        } else {
            // If the solutions were already fetched, use them.
            if (this.reviewable) {
                this.place.html(SolutionTool.templates.view);
                this.review();
            } else {
                this.place.html(SolutionTool.templates.studentview);
                if (this.editable) {
                    this.edit();
                };
            };
            // Update solution counts
            this.updateSolutionCount();
        };
    };
    
    /******
     * Show the element (editable modes);
     ******/
    SolutionTool.prototype.edit = function(){
        this.getReviewInfo();
        var username = this.settings.username;
        var buttonlist = [];
        var solcount = this.showTabList();
        // If the tablist is not empty, show the last solution.
        if (solcount > 0) {
            this.showTab(solcount - 1, this.settings.username, this.place.find('.solutiontool-viewarea'));
        };
        if (!this.hasOpenSolution()) {
            buttonlist.push('new');
            if (solcount > 0) {
                buttonlist.push('copy');
            };
            this.place.attr('data-hasopensolution', false);
        };
        if (this.hasOpenSolution()) {
            buttonlist.push('send');
            this.place.trigger('startediting');
            this.place.attr('data-hasopensolution', true);
        };
        this.showEditButtons(buttonlist);
    };
    
    /**
     * Show editing view buttons
     * @param {String[]} buttonlist    An array of names of buttons to show.
     */
    SolutionTool.prototype.showEditButtons = function(buttonlist) {
        var buttonbar = this.place.find('.solutiontool-buttonbar');
        buttonbar.empty();
        var button;
        for (var i = 0, len = buttonlist.length; i < len; i++) {
            button = SolutionTool.buttons[buttonlist[i]];
            buttonbar.append('<button class="ffwidget-setbutton solutiontool-button solutiontool-button-' + button.name + '" title="'+(button.label[this.settings.uilang] || button.label['en'])+'" data-action="'+button.event+'"><span class="solutiontool-buttonicon">'+button.icon+'</span> <span class="solutiontool-buttontext">'+(button.label[this.settings.uilang] || button.label['en'])+'</span></button>')
        };
    }

    /******
     * Show the solutions in accordion by sover and in review mode.
     * Shows only the accordions. The solutions are shown with click on accordion with .showReview()
     ******/
    SolutionTool.prototype.review = function(){
        var uilang = this.settings.uilang;
        var reviewarea = this.place.children('.solutiontool-reviewarea');
        var teacherbarea = reviewarea.find('.solutiontool-teacherutils .solutiontool-teacherbuttons');
        var button;
        for (var bname in SolutionTool.teacherbuttons) {
            button = SolutionTool.teacherbuttons[bname];
            teacherbarea.append('<button class="ffwidget-setbutton" data-buttonname="'+bname+'" data-action="' + button.event + '" title="'+(button.label[uilang] || button.label['en'])+'" data-info="">'+button.icon+'</button>');
        };
        var accordion = this.place.find('> .solutiontool-reviewarea > ul.solutiontool-reviewaccordion');
        var users = [];
        for (var uname in this.allSolutions){
            users.push(uname);
        };
        var tool = this;
        users.sort(function(a, b){
            var aname = a, bname = b;
            if (tool.settings.users && tool.settings.users.getRealnameLnameFirst) {
                aname = tool.settings.users.getRealnameLnameFirst(a) || a;
                bname = tool.settings.users.getRealnameLnameFirst(b) || b;
            };
            return (aname < bname ? -1 : 1);
        });
        var lilist = [];
        var realname;
        var solnum, unrevnum;
        for (var i = 0, len = users.length; i < len; i++){
            realname = this.settings.users && this.settings.users.getRealnameLnameFirst(users[i]) || users[i];
            solnum = this.getSolutionNumber(users[i]);
            unrevnum = this.getUnreviewedNumber(users[i]);
            lilist.push('<li class="solutiontool-review-useritem"><div class="solutiontool-review-username ffwidget-background" data-username="'+escapeHTML(users[i])+'"><div class="solutiontool-review-name">'+escapeHTML(realname)+'</div><div class="solutiontool-review-userstats"><span class="solutiontool-review-userdone">'+escapeHTML(solnum)+'</span><span class="solutiontool-review-userunreviewed">'+escapeHTML(unrevnum)+'</span></div></div><div class="solutiontool-reviewitem" data-username="'+escapeHTML(users[i])+'"></div></li>')
        };
        accordion.html(lilist.join('\n'));
        this.setUnpubCount();
        this.updateSolutionCount();
    };
    
    /******
     * Show the set of solutions of one student in review mode for teacher inside an accordion.
     ******/
    SolutionTool.prototype.showReview = function(place, username){
        place.html(SolutionTool.templates.reviewitem);
        var solutiontablist = place.find('ul.solutiontool-solutiontablist');
        //var buttonlist = [];
        var solution;
        var sollist = this.allSolutions[username] || [];
        var reviews = this.allReviews.refs || {};
        var hasrev;
        // Add tabs in the tablist.
        for (var i = 0, len = sollist.length; i < len; i++){
            solution = sollist[i];
            hasrev = !!reviews[solution.id] && reviews[solution.id].length > 0;
            soldata = this.solutiondata[solution.id].data;
            solutiontablist.append('<li data-solution-name="'+escapeHTML(solution.id)+'" data-solution-isopen="'+(!soldata.issent)+'" data-solution-tabnumber="'+i+'" data-hasreview="'+(hasrev + '')+'"><span>'+(i+1)+'</span></li>');
        };
        // If the tablist is not empty, show the last solution.
        if (sollist.length > 0) {
            this.showTab(sollist.length - 1, username, place);
        };
    }
    
    /**
     * Readraw the tablist
     */
    SolutionTool.prototype.showTabList = function() {
        var username = this.settings.username;
        var viewarea = this.place.children('.solutiontool-viewarea');
        var solutiontablist = viewarea.find('ul.solutiontool-solutiontablist');
        var sollist = this.allSolutions[username] = this.allSolutions[username] || [];
        var reviews = this.allReviews.refs || {};
        var solution, hasrev, unreadrevs, revname, cinfo;
        // Add tabs in the tablist.
        solutiontablist.empty();
        for (var i = 0, len = sollist.length; i < len; i++){
            solution = sollist[i];
            unreadrevs = 0;
            hasrev = !!reviews[solution.id] && reviews[solution.id].length > 0;
            if (hasrev) {
                for (var j = 0, rlen = reviews[solution.id].length; j < rlen; j++) {
                    revname = reviews[solution.id][j];
                    cinfo = this.allReviews.contentinfo[revname];
                    unreadrevs += (cinfo && cinfo.recipientopened ? 0 : 1);
                };
            };
            soldata = this.solutiondata[solution.id].data;
            solutiontablist.append('<li data-solution-name="'+escapeHTML(solution.id)+'" data-solution-isopen="'+(!soldata.issent)+'" data-solution-tabnumber="'+i+'" data-hasreview="'+(hasrev+'')+'" data-unreadrevs="'+unreadrevs+'"><span>'+(i+1)+'</span></li>');
        };
        return sollist.length;
    }
    
    /******
     * Create the selected tab
     * @param {Number} num - the index number of the selected solution.
     * @param {String} username - optional parameter, show solution of this user
     ******/
    SolutionTool.prototype.showTab = function(num, username, place){
        username = username || this.settings.username;
        place = place || this.place;
        this.place.find('.solutiontool-content.solutiontool-empty').removeClass('solutiontool-empty');
        var sollist = this.allSolutions[username] || [];
        this.setCurrent(num, username);
        var tablist = place.find('.solutiontool-solutiontablist');
        var isreview = tablist.closest('.solutiontool-reviewarea').length > 0;
        tablist.children('li').removeClass('solutiontool-currenttab').eq(num).addClass('solutiontool-currenttab');
        var solelem = $('<div></div>');
        var solution = sollist[num];
        var solid = solution.id;
        this.place.trigger('getcontent', {anchors: [solid], contentType: ['review']});
        solelem.attr('data-element-name', solid);
        var soldata = JSON.parse(JSON.stringify(this.solutiondata[solid]));
        var solcontainer = place.find('.solutiontool-content-solution');
        solcontainer.attr('data-solutionid', escapeHTML(solid));
        solcontainer.empty();
        solcontainer.prepend(solelem);
        soldata.settings = {
            uilang: this.settings.uilang,
            lang: this.settings.lang,
            mode: (this.editable ? 'edit' : 'view'),
            username: this.settings.username,
            users: this.settings.users,
            gotolink: JSON.parse(JSON.stringify(this.settings.gotolink || {})),
            usemargin: false
        };
        solelem.solutionelement(soldata);
        this.showReviewlist();
        this.showLastReview();
        this.updateReviewstatus();
    }
    
    /******
     * Add a new solution
     ******/
    SolutionTool.prototype.addNew = function(copyfrom){
        var newdata;
        var username = this.settings.username;
        if (!this.allSolutions[username]) {
            this.allSolutions[username] = [];
        }
        var sollist = this.allSolutions[this.settings.username];
        if (typeof(copyfrom) === 'number') {
            var copyid = sollist[copyfrom].id;
            newdata = $.extend(true, {}, this.solutiondata[copyid]);
        } else {
            newdata = SolutionTool.emptysolution;
        };
        newdata.data.issent = false;
        newdata.metadata.created = new Date()+'';
        newdata.metadata.modified = new Date()+'';
        newdata.metadata.ref = this.assignmentid;
        var newid = this.getNewId();
        sollist.push({id: newid});
        this.solutiondata[newid] = newdata;
        var solcount = this.showTabList();
        this.showTab(solcount - 1), this.settings.username, this.place.find('.solutiontool-viewarea');
        if (copyfrom) {
            this.place.find('.solutiontool-viewarea .solutiontool-content-solution > div').trigger('element_changed', {type: 'solutionelement'});
        }
        this.place.trigger('startediting');
    }
    
    /******
     * Set/update the data of solution with given id and given data
     ******/
    SolutionTool.prototype.setData = function(solid, soldata){
        var username = this.settings.username;
        this.solutiondata[solid] = soldata;
    };
    
    /******
     * Sort solutions in time order and by username
     * @param {Object} data - data from 'reply_getcontent' event.
     ******/
    SolutionTool.prototype.sortSolutions = function(data) {
        // Make a copy so that changes don't affect elsewhere!
        data = $.extend(true, {}, data);
        var username = this.settings.username;
        var solutions = [];
        var solutiondata = {};
        var user;
        var solid;
        var contentdata = data.contentdata;
        var refs = data.refs[this.assignmentid] || [];
        var i, len, key;
        for (i = 0, len = refs.length; i < len; i++) {
            key = refs[i];
            if (contentdata[key].type === 'solutionelement') {
                solutions.push({id: key});
                solutiondata[key] = contentdata[key];
            };
        };
        // Order by modification time.
        solutions.sort(function(a,b){
            try {
                var dateA = new Date(solutiondata[a.id].metadata.modified);
                var dateB = new Date(solutiondata[b.id].metadata.modified);
            } catch (err) {
                var dateA = 1;
                var dateB = 0;
            };
            return (dateA > dateB ? 1 : -1);
        });
        this.allSolutions = {};
        // Create an array of solutions for each username
        for (i = 0, len = solutions.length; i < len; i++) {
            solid = solutions[i].id;
            if (solid && solutiondata[solid]) {
                user = solutiondata[solid].metadata.creator;
                if (!this.allSolutions[user]) {
                    this.allSolutions[user] = [];
                }
                this.allSolutions[user].push(solutions[i]);
            };
        };
        // Current user's own solutions.
        this.solutions = this.allSolutions[username] || [];
        // All solutions data (no need to be separated by name)
        this.solutiondata = solutiondata;
        // Now the solutions are fetched.
        this.solutionsFetched = true;
    };
    
    /******
     * Sort reviews from 'reply_getcontent' event and show them.
     * @param {Object} data - data from event
     ******/
    SolutionTool.prototype.sortReviews = function(data) {
        if (!this.allReviews) {
            this.allReviews = {
                refs: {},
                contentdata: {},
                contentinfo: {},
                anchors: {}
            };
        };
        var anchor, i, len, reflist, cname;
        for (anchor in data.refs) {
            reflist = data.refs[anchor].slice();
            this.allReviews.refs[anchor] = reflist;
            for (i = 0, len = reflist.length; i < len; i++) {
                cname = reflist[i];
                this.allReviews.contentdata[cname] = JSON.parse(JSON.stringify(data.contentdata[cname]));
                this.allReviews.contentinfo[cname] = JSON.parse(JSON.stringify(data.contentinfo[cname]));
                if (!this.allReviews.anchors[cname]) {
                    this.allReviews.anchors[cname] = [];
                };
                if (this.allReviews.anchors[cname].indexOf(anchor) === -1) {
                    this.allReviews.anchors[cname].push(anchor);
                };
            };
        };
    };
    
    /******
     * Show reviewlist of current solution
     ******/
    SolutionTool.prototype.showReviewlist = function(){
        var uilang = this.settings.uilang;
        var solname, $rvlist, reviewnames, reviews, rname;
        var reviewbuttons, rvbuttondata, realname;
        var $visibleSols = this.place.find('.solutiontool-content:visible .ebook-solutionelement');
        if ($visibleSols.length > 0) {
            this.place.find('.solutiontool-content.solutiontool-empty').removeClass('solutiontool-empty');
        }
        // Go through all students
        for (var sol = 0; sol < $visibleSols.length; sol++){
            $rvlist = $visibleSols.eq(sol).closest('.solutiontool-content').find('.solutiontool-reviewlist');
            solname = $visibleSols.eq(sol).attr('data-element-name');
            reviewnames = this.allReviews.refs[solname] || [];
            reviews = [];
            for (var i = 0, len = reviewnames.length; i < len; i++) {
                rname = reviewnames[i];
                reviews.push(this.allReviews.contentinfo[rname]);
            };
            reviews.sort(function(a, b) {
                return ((new Date(a.timestamp)).getTime() < (new Date(b.timestamp)).getTime() ? -1 : 1);
            });
            reviewbuttons = [];
            for (var i = 0, len = reviews.length; i < len; i++) {
                realname = this.settings.users && this.settings.users.getRealname(reviews[i].sendfrom) || reviews[i].sendfrom;
                rvbuttondata = {
                    reviewer: realname,
                    date: (new Date(reviews[i].timestamp)).toLocaleString(uilang),
                    dateshort: (new Date(reviews[i].timestamp)).toLocaleDateString(uilang),
                    reviewid: reviews[i].name,
                    solutionid: solname
                };
                reviewbuttons.push(this.fillVars(SolutionTool.templates.reviewbutton, rvbuttondata));
            };
            $rvlist.html(reviewbuttons.join('\n'));
        };
    };
    
    /******
     * Show the last review in review list
     * @param {Boolean} forceshow      Force showing instead of toggling.
     ******/
    SolutionTool.prototype.showLastReview = function(forceshow){
        this.place.find('.solutiontool-content .solutiontool-reviewlist li.solutiontool-reviewlist-item:last-child:visible button').trigger('click', [forceshow]);
    }
    
    /******
     * Show (or hide) reviewdata
     * @param {String} solutionid - id of the solution to which we want to show reviewdata
     * @param {String} reviewid - id of the wanted review or empty string (or false) to hide
     * @param {Boolean} isreviewtool   - True, if the reviewdata should be shown in reviewtool.
     * @param {Boolean} createnew      - True, if we want to create a new review.
     ******/
    SolutionTool.prototype.showReviewData = function(solutionid, reviewid, isreviewtool, createnew) {
        var $solutionwrapper;
        var areatype = (isreviewtool ? 'reviewarea' : 'viewarea');
        var $solutionelem = this.place.find('.solutiontool-'+areatype+' .solutiontool-content-solution .ebook-solutionelement[data-element-name="'+solutionid+'"]');
        this.currentReview = reviewid;
        var $reviewbuttons = $solutionelem.closest('.solutiontool-content').find('> .solutiontool-reviewtools .solutiontool-reviewlist .solutiontool-reviewlist-item');
        $reviewbuttons.removeClass('selected').children('button').removeClass('buttonselected');
        if (reviewid) {
            $reviewbuttons.filter('[data-reviewid="'+reviewid+'"]').addClass('selected').children('button').addClass('buttonselected');
            try {
                var review = this.allReviews.contentdata[reviewid];
                var revinfo = this.allReviews.contentinfo[reviewid];
            } catch (err) {
                var review = {};
                var revinfo = {sendto: []};
            };
            if (review.type === 'reviewelement') {
                $solutionwrapper = $solutionelem.find('> .ebook-elementset-body > .elementset-elementwrapper');
                if (review.metadata.creator === this.settings.username && revinfo.sendto.length === 0) {
                    $solutionelem.trigger('setmode', 'review');
                } else {
                    $solutionelem.trigger('setmode', 'view');
                }
                $solutionelem.trigger('hidereview').trigger('showreview', [reviewid, review])
            };
        } else {
            $solutionwrapper = $solutionelem.find('> .ebook-elementset-body > .elementset-elementwrapper');
            if (createnew) {
                $solutionelem.trigger('setmode', 'review');
                $solutionelem.trigger('hidereview').trigger('showreview');
            } else {
                $solutionelem.trigger('setmode', 'view');
                $solutionelem.trigger('hidereview');
            }
        };
    };
    
    /******
     * Get info about reviews
     ******/
    SolutionTool.prototype.getReviewInfo = function(){
        var sollist = [];
        for (var solname in this.solutiondata) {
            sollist.push(solname);
        };
        this.place.trigger('getcontentinfo', {contentType: ['review'], anchors: sollist, lang: 'common'});
    };
    
    /******
     * Handle info about reviews
     * @param {Object} data - DataByAnchors (without contentdata)
     *     {
     *         contentType: 'review',
     *         lang: 'common',
     *         refs: {
     *             <anchorid>: ["<elementid>", ...],
     *             ....
     *         },
     *         contentinfo: {
     *             <elementid>: {<element's contentinfo>},
     *             ....
     *         }
     *     }
     ******/
    SolutionTool.prototype.setReviewInfo = function(data) {
        if (data.contentType === 'review') {
            var revid;
            for (revid in data.contentinfo) {
                this.allReviews.contentinfo[revid] = data.contentinfo[revid];
            };
            var anchor;
            for (anchor in data.refs) {
                for (var i = 0, len = data.refs[anchor].length; i < len; i++) {
                    revid = data.refs[anchor][i];
                    if (!this.allReviews.refs[anchor]) {
                        this.allReviews.refs[anchor] = [];
                    };
                    if (this.allReviews.refs[anchor].indexOf(revid) === -1) {
                        this.allReviews.refs[anchor].push(revid);
                    };
                };
            };
        };
        this.setUnpubCount();
    };

    /******
     * Mark review as read
     * @param {String} revid - id of the review
     ******/
    SolutionTool.prototype.markReviewRead = function(revid) {
        var cinfo = this.allReviews.contentinfo[revid];
        if (cinfo) {
            cinfo.recipientopened = true;
            this.place.trigger('setcontentinfo', cinfo);
        }
    }
    
    /******
     * Set the review data
     * @param {Object} data - the data of the review: {name: "...", contentType: "review", contentdata: "...", contentinfo: "...", anchors: "..."}
     ******/
    SolutionTool.prototype.setReview = function(data) {
        var uilang = this.settings.uilang;
        if (data.contentType === 'review') {
            var revid = data.name;
            var wasnew = !this.allReviews.contentdata[revid];
            this.allReviews.contentdata[revid] = data.contentdata;
            if (this.allReviews.contentinfo[revid]) {
                // If review is already published, keep it as such.
                data.contentinfo.pubtype = (this.allReviews.contentinfo[revid].pubtype === 'user' ? 'user' : data.contentinfo.pubtype);
                data.contentinfo.sendto = (this.allReviews.contentinfo[revid].sendto && this.allReviews.contentinfo[revid].sendto.length > 0 ? this.allReviews.contentinfo[revid].sendto : data.contentinfo.sendto);
                data.contentinfo.isoutbox = this.allReviews.contentinfo[revid].isoutbox || data.contentinfo.isoutbox;
            }
            this.allReviews.contentinfo[revid] = data.contentinfo;
            var anchor;
            for (var i = 0, len = data.anchors.length; i < len; i++) {
                anchor = data.anchors[i];
                if (!this.allReviews.refs[anchor]) {
                    this.allReviews.refs[anchor] = [];
                };
                if (this.allReviews.refs[anchor].indexOf(data.name) === -1) {
                    this.allReviews.refs[anchor].push(data.name);
                };
            };
            if (wasnew) {
                var reviewer = data.contentinfo.sendfrom;
                reviewer = this.settings.users && this.settings.users.getRealnameLnameFirst(reviewer) || reviewer;
                this.place.find('[data-element-name="'+escapeHTML(anchor)+'"]').closest('.solutiontool-content').find('.solutiontool-reviewlist').append('<li class="solutiontool-reviewlist-item selected" data-reviewid="'+escapeHTML(revid)+'" data-solutionid="'+escapeHTML(data.anchors.join(' '))+'"><button class="ffwidget-button buttonselected" title="'+(new Date(data.contentinfo.timestamp)).toLocaleString(uilang)+'"><span class="solutiontool-reviewlist-reviewer">'+escapeHTML(data.contentinfo.sendfrom)+'</span> <span class="solutiontool-reviewlist-date">'+(new Date(data.contentinfo.timestamp)).toLocaleDateString(uilang)+'</span></button></li>');
            };
            this.setUnpubCount();
            this.updateReviewstatus();
            this.updateReviewAccordion();
            this.place.trigger('setcontent', [data, true]);
            this.place.trigger('savecontent');
        };
    };
    
    /******
     * Send (publish to student) the reviews
     * @param {Array} revs - a list of reviews to send. If not given, send all.
     ******/
    SolutionTool.prototype.sendReviews = function(revs) {
        var revname, recip;
        if (typeof revs === 'undefined') {
            revs = [];
            for (revname in this.allReviews.contentinfo) {
                revs.push(revname);
            };
        };
        var contentinfo;
        var recipients = this.settings.users.getRecipients('review');
        var reclist = [];
        for (var index = 0, rlen = recipients.length; index < rlen; index++) {
            if (recipients[index].idtype === 'group' && recipients[index].togroup) {
                reclist.push(JSON.parse(JSON.stringify(recipients[index].send_to)));
            };
        };
        for (var i = 0, len = revs.length; i < len; i++) {
            revname = revs[i];
            contentinfo = this.allReviews.contentinfo[revname];
            if (contentinfo.pubtype !== 'user' || contentinfo.isoutbox) {
                contentinfo.pubtype = 'user';
                contentinfo.sendto = reclist;
                for (var j = 0, reslen = contentinfo.recipients.length; j < reslen; j++) {
                    recip = contentinfo.recipients[j];
                    if (this.settings.users.canPublish({ctype: 'review', toid: recip.to, totype: recip.pubtype})) {
                        contentinfo.sendto.push(recip);
                    };
                };
                this.place.trigger('sendcontent', JSON.parse(JSON.stringify(contentinfo)));
                contentinfo.isoutbox = false;
            };
        };
        this.place.trigger('savecontent');
        this.setUnpubCount();
        this.updateReviewstatus();
        this.showLastReview(true);
        this.updateSolutionCount();
    };
    
    /******
     * Set the index of currently selected solution (the index of the tab).
     * @param {Number} index - The index of the solution
     * @param {String} username - the name of the user whose solutiontab this is.
     ******/
    SolutionTool.prototype.setCurrent = function(index, username) {
        username = username || this.settings.username;
        var solutions = this.allSolutions[username] || [];
        index = (typeof(index) === 'number' ? index : solutions.length - 1);
        this.currentSolution[username] = index;
    };
    
    /******
     * Set an attribute indicating unpublished reviews
     ******/
    SolutionTool.prototype.setUnpubCount = function(){
        var count = 0;
        for (var rname in this.allReviews.contentinfo) {
            var cinfo = this.allReviews.contentinfo[rname];
            if (cinfo.pubtype !== 'user' || cinfo.isoutbox) {
                count++;
            };
        };
        this.unpublishedReviews = count;
        this.place.attr('data-unpublishedreviews', count);
        this.place.find('.solutiontool-teacherbuttons button[data-buttonname="sendall"]').attr('data-info', count);
    };
    
    /******
     * Update reviewstatus and unread count in solutiontabs.
     ******/
    SolutionTool.prototype.updateReviewstatus = function(){
        var solutiontablist = this.place.find('ul.solutiontool-solutiontablist > li');
        var solutioncontent, solcontsol, revnames;
        var username = this.settings.username;
        var reviews = this.allReviews.refs || {};
        var hasrev, unreadrevs, unsentrevs, tab, solname, revname, cinfo, iscurrent;
        for (var i = 0, len = solutiontablist.length; i < len; i++) {
            tab = solutiontablist.eq(i);
            iscurrent = tab.is('.solutiontool-currenttab');
            solutioncontent = tab.parent().next('.solutiontool-content');
            solcontsol = solutioncontent.children('.solutiontool-content-solution');
            revnames = [];
            solname = tab.attr('data-solution-name');
            unreadrevs = 0;
            unsentrevs = 0;
            hasrev = !!reviews[solname] && reviews[solname].length > 0;
            if (hasrev) {
                for (var j = 0, rlen = reviews[solname].length; j < rlen; j++) {
                    revname = reviews[solname][j];
                    revnames.push(revname);
                    cinfo = this.allReviews.contentinfo[revname];
                    unreadrevs += (cinfo && cinfo.recipientopened ? 0 : 1);
                    unsentrevs += (cinfo && (cinfo.pubtype !== 'user' || cinfo.isoutbox) ? 1 : 0);
                };
            };
            tab.attr('data-hasreview', hasrev+'');
            tab.attr('data-unreadrevs', unreadrevs);
            tab.attr('data-hasunsentrevs', (unsentrevs > 0) + '');
            if (iscurrent) {
                solutioncontent.attr('data-hasreview', hasrev+'');
                solutioncontent.attr('data-unreadrevs', unreadrevs);
                solutioncontent.attr('data-unsentrevs', (unsentrevs > 0) + '');
                solcontsol.attr('data-unsentreviews', revnames.join(' '));
            };
        };
    };
    
    /**
     * Update the number of solutions and open solutions.
     */
    SolutionTool.prototype.updateSolutionCount = function() {
        this.solutioncount = 0;
        this.unrevcount = 0;
        for (var user in this.allSolutions) {
            this.solutioncount += this.getSolutionNumber(user);
            this.unrevcount += this.getUnreviewedNumber(user);
        };
        this.place.attr('data-totalsolution', this.solutioncount);
        this.place.attr('data-totalunrevsol', this.unrevcount);
    };
    
    /******
     * Update reviewstatus in teacher mode / accrodions
     ******/
    SolutionTool.prototype.updateReviewAccordion = function(){
        var $userdone, $userunreviewed;
        var users = [];
        for (var uname in this.allSolutions) {
            users.push(uname);
        };
        var tool = this;
        users.sort(function(a, b){
            var aname = a, bname = b;
            if (tool.settings.users && tool.settings.users.getRealnameLnameFirst) {
                aname = tool.settings.users.getRealnameLnameFirst(a) || a;
                bname = tool.settings.users.getRealnameLnameFirst(b) || b;
            };
            return (aname < bname ? -1 : 1);
        });
        for (var i = 0, len = users.length; i < len; i++) {
            $userdone = this.place.find('.solutiontool-review-username[data-username="'+escapeHTML(users[i])+'"] .solutiontool-review-userdone');
            $userunreviewed = this.place.find('.solutiontool-review-username[data-username="'+escapeHTML(users[i])+'"] .solutiontool-review-userunreviewed');
            $userdone.html(this.getSolutionNumber(users[i]));
            $userunreviewed.html(this.getUnreviewedNumber(users[i]));
        }
    }
    
    /******
     * Set the mode (view, edit, review, ...)
     * @param {String} mode - the name of the mode
     ******/
    SolutionTool.prototype.setMode = function(mode){
        this.settings.mode = (SolutionTool.modes[mode] && mode || 'view');
        this.editable = SolutionTool.modes[this.settings.mode].editable;
        this.reviewable = SolutionTool.modes[this.settings.mode].reviewable;
        if (this.editable) {
            this.place.attr('data-dragtype', 'element container studentelement studentcontainer');
        } else {
            this.place.removeAttr('data-dragtype');
        };
    };
    
    /******
     * Set user role
     * @param {String} role - the role to set
     ******/
    SolutionTool.prototype.setRole = function(role){
        if (this.settings.users) {
            this.editable = this.settings.users.canEdit({ctype: 'solution', contextid: this.assignmentid});
            this.reviewable = this.settings.users.canEdit({ctype: 'review', contextid: this.assignmentid});
            if (this.editable && !this.reviewable) {
                this.place.attr('data-selectedarea', 'view');
            } else if (this.reviewable && !this.editable) {
                this.place.attr('data-selectedarea', 'review');
            } else if (this.reviewable && this.editable) {
                this.place.addClass('solutiontool-areaselectable');
            };
        } else {
            if (typeof(SolutionTool.roles[role]) === 'undefined') {
                role = this.settings.role;
            };
            var roledata = SolutionTool.roles[role];
            if (this.rolechangeable || typeof(this.rolechangeable) === 'undefined') {
                this.settings.role = role;
                this.rolechangeable = roledata.rolechangeable;
                this.teacherable = roledata.teacherable;
                this.studentable = roledata.studentable;
            }
        }
    }
    
    /******
     * Set all attributes and classes for the html-content
     ******/
    SolutionTool.prototype.setAttrs = function(){
        this.place.addClass('solutiontool');
        this.place.attr('data-referenceto', escapeHTML(this.assignmentid));
        this.setUnpubCount();
    };

    /******
     * Set the style sheets to the head, if needed.
     ******/
    SolutionTool.prototype.setStyles = function(){
        if ($('#solutiontool-styles').length === 0) {
            $('head').append('<style id="solutiontool-styles" type="text/css">'+SolutionTool.styles+'</style>');
        }
    };
    
    /******
     * Check if this solution set has any non-sent solutions.
     * @returns {Boolean} true, if there is a non-sent solution, else false
     ******/
    SolutionTool.prototype.hasOpenSolution = function(username){
        username = username || this.settings.username;
        var found = false;
        var solid;
        var sollist = this.allSolutions[username];
        for (var i = 0, len = sollist.length; i < len; i++) {
            solid = sollist[i].id;
            found = found || !this.solutiondata[solid].data.issent;
        };
        return found;
    };
    
    /******
     * Find the solutios that have not been sent.
     * @returns {Array} an array of non-sent solutions
     ******/
    SolutionTool.prototype.getOpenSolution = function(){
        var username = this.settings.username;
        var sollist = this.allSolutions[username];
        var found = [];
        var solid;
        for (var i = 0, len = sollist.length; i < len; i++) {
            solid = sollist[i].id;
            if (!this.solutiondata[solid].data.issent){
                found.push(solid);
            };
        };
        return found;
    };
    
    /******
     * Get the number of solutions by username
     * @param {String} username - username
     ******/
    SolutionTool.prototype.getSolutionNumber = function(username) {
        var sollist = this.allSolutions[username] || [];
        return sollist.length;
    };
    
    /******
     * Get the number of unreviewed by username
     ******/
    SolutionTool.prototype.getUnreviewedNumber = function(username) {
        var sollist = this.allSolutions[username] || [];
        var solname, reflist;
        var count = 0;
        for (var i = 0, len = sollist.length; i < len; i++) {
            solname = sollist[i].id;
            reflist = this.allReviews.refs[solname] || [];
            if (reflist.length === 0) {
                count++;
            };
        };
        return count;
    };
    
    /******
     * Go to the link target
     * @param {Object} linkdata - link data to the target
     ******/
    SolutionTool.prototype.goToLink = function(linkdata){
        var link = linkdata && linkdata.data && linkdata.data.link;
        delete link.solution;
        var element = this.place;
        if (this.settings.users.canEdit({ctype: 'review'})) {
            element.find('.assignment-teacherarea .solutiontool-selector button[data-selectarea="review"]:not(.buttonselected)').click();
        };
        if (link.solver) {
            var listelement = element.parent().find('.solutiontool[data-referenceto="'+escapeHTML(this.assignmentid)+'"] .solutiontool-review-username[data-username="'+escapeHTML(link.solver)+'"]');
            if (!listelement.is('[data-isopen="1"]')) {
                listelement.click();
            };
            if (listelement.length > 0) {
                element = listelement.parent();
            };
            if (link.solname) {
                element.find('.solutiontool-solutiontablist li[data-solution-name="'+escapeHTML(link.solname)+'"]').click();
            };
        };
    };
    
    /******
     * Get an id for a new solution element.
     * @returns {String} An id for the new element.
     ******/
    SolutionTool.prototype.getNewId = function(){
        var idparts = ['solutionelement'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    SolutionTool.prototype.fillVars = function(text, map){
        var rex, result = text;
        for (var key in map) {
            rex = RegExp('{{ *'+key+' *}}', 'g')
            result = result.replace(rex, map[key]);
        };
        return result;
    };
    
    /******
     * Some icons
     ******/
    SolutionTool.icons = {
        preview: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-preview"><path style="stroke: none;" d="M3 15 a13 13 0 0 0 24 0 a13 13 0 0 0 -24 0z m1 0 a13 13 0 0 1 22 0 a13 13 0 0 1 -22 0z m6 -1 a5 5 0 0 0 10 0 a5 5 0 0 0 -10 0z m1.5 -1.6 a3 3 0 0 1 3 -2.1 a2 0.8 0 0 1 0 1 a3 3 0 0 0 -1.8 1.2 a0.5 0.5 0 0 1 -1.16 0z"></path><path class="nopreview" stroke="none" style="fill: none; stroke-width: 4;" d="M3 27 l24 -24"></path></svg>',
        edit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>',
        review: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="3 3 24 24" class="mini-icon mini-icon-thumbs"><path style="stroke: none;" d="M6 11 h2 v10 h-2z m4 0 a4 6 0 0 0 4 -6 a3 6 25 0 1 2 6 a25 25 0 0 1 9 1 a25 25 0 0 1 -3 10 a24 24 0 0 1 -12 -1z"></path></svg>',
        reviewadd: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="3 3 24 24" class="mini-icon mini-icon-thumbs"><path style="stroke: none;" d="M6 11 h2 v10 h-2z m4 0 a4 6 0 0 0 4 -6 a3 6 25 0 1 2 6 a25 25 0 0 1 9 1 a25 25 0 0 1 -3 10 a24 24 0 0 1 -12 -1z"></path><path style="fill: white; stroke: none;" d="M16 14 h2 v2 h2 v2 h-2 v2 h-2 v-2 h-2 v-2 h2z" /></svg>',
        send: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-send"><path style="stroke: none;" d="M2 18 l24 -12 l-4 18 l-7 -3 l-4 4 l-1 0 l-1 -6z m8 0 l0.7 6 l2 -4.5 l12 -12z"></path></svg>'
    }
    
    /******
     * Templates for the html-content for different modes
     ******/
    SolutionTool.templates = {
        studentview: [
            '<div class="solutiontool-viewarea solutiontool-view-inited" data-dragtype="element">',
            '    <div class="solutiontool-buttonbar ffwidget-buttonset ffwidget-horizontal"></div>',
            '    <div class="solutiontool-area">',
            '        <ul class="solutiontool-solutiontablist"></ul>',
            '        <div class="solutiontool-content solutiontool-empty">',
            '            <div class="solutiontool-reviewtools">',
            '                <ul class="solutiontool-reviewlist"></ul>',
            '            </div>',
            '            <div class="solutiontool-content-solution"></div>',
            '        </div>',
            '    </div>',
            '</div>'
        ].join('\n'),
        view: [
            '<div class="assignment-teacherarea">',
            '  <div class="assignment-teachercontrol ffwidget-background">',
            '    <div class="solutiontool-selector ffwidget-buttonset ffwidget-horizontal"><button class="ffwidget-setbutton" data-selectarea="view">'+SolutionTool.icons.edit+'</button><button class="ffwidget-setbutton" data-selectarea="review">'+SolutionTool.icons.review+'</button></div>',
            '  </div>',
            '</div>',
            '<div class="solutiontool-viewarea" data-dragtype="element">',
            '    <div class="solutiontool-buttonbar ffwidget-buttonset ffwidget-horizontal"></div>',
            '    <div class="solutiontool-area">',
            '        <ul class="solutiontool-solutiontablist"></ul>',
            '        <div class="solutiontool-content solutiontool-empty">',
            '            <div class="solutiontool-reviewtools">',
            '                <ul class="solutiontool-reviewlist"></ul>',
            '            </div>',
            '            <div class="solutiontool-content-solution"></div>',
            '        </div>',
            '    </div>',
            '</div>',
            '<div class="solutiontool-reviewarea">',
            '    <div class="solutiontool-teacherutils ffwidget-background">',
            '        <div class="solutiontool-teacherinfo"></div>',
            '        <div class="solutiontool-teacherbuttons ffwidget-buttonset"></div>',
            '    </div>',
            '    <ul class="solutiontool-reviewaccordion">',
            '    </ul>',
            '</div>'
        ].join('\n'),
        review: [
            '<div class="solutiontool-teacherutils ffwidget-background">',
            '    <div class="solutiontool-teacherinfo"></div>',
            '    <div class="solutiontool-teacherbuttons ffwidget-buttonset"></div>',
            '</div>',
            '<ul class="solutiontool-reviewaccordion">',
            '</ul>'
        ].join('\n'),
        reviewitem: [
            '<div class="solutiontool-buttonbar ffwidget-buttonset ffwidget-horizontal">',
            '</div>',
            '<div class="solutiontool-area">',
            '    <ul class="solutiontool-solutiontablist"></ul>',
            '    <div class="solutiontool-content" data-unsentrevs="false">',
            '        <div class="solutiontool-reviewtools">',
            '            <ul class="solutiontool-reviewlist"></ul>',
            '            <div class="solutiontool-reviewtools-buttons"><button class="solutiontool-reviewbutton-newreview solutiontool-reviewstart-button ffwidget-button" data-action="startreview">'+SolutionTool.icons.reviewadd+'</button><button class="solutiontool-reviewbutton-send ffwidget-button" data-action="sendreviews">'+SolutionTool.icons.send+'</button></div>',
            '        </div>',
            '        <div class="solutiontool-content-solution"></div>',
            '    </div>',
            '</div>'
        ].join('\n'),
        edit: [
            '<div class="solutiontool-buttonbar ffwidget-buttonset ffwidget-horizontal">',
            '</div>',
            '<div class="solutiontool-area">',
            '    <ul class="solutiontool-solutiontablist"></ul>',
            '    <div class="solutiontool-content solutiontool-empty">',
            '        <ul class="solutiontool-reviewlist"></ul>',
            '        <div class="solutiontool-content-solution"></div>',
            '    </div>',
            '</div>'
        ].join('\n'),
        reviewbutton: [
            '<li class="solutiontool-reviewlist-item" data-reviewid="{{ reviewid }}" data-solutionid="{{ solutionid }}">',
            '    <button class="ffwidget-button" title="{{ date }}">',
            '        <span class="solutiontool-reviewlist-reviewer">{{ reviewer }}</span>',
            '        <span class="solutiontool-reviewlist-date">{{ dateshort }}</span>',
            '    </button>',
            '</li>'
        ].join('\n')
    };
    
    /******
     * Template for empty solution
     ******/
    SolutionTool.emptysolution = {
        type: 'solutionelement',
        metadata: {},
        data: {
            issent: false,
            contents: [],
            contentdata: {}
        }
    };
    
    /******
     * Available buttons for the edit mode
     ******/
    SolutionTool.buttons = {
        "new": {
            name: 'new',
            label: {
                en: 'New solution',
                fi: 'Uusi ratkaisu'
            },
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-new"><path style="stroke: none;" d="M15 2 a13 13 0 0 1 0 26 a13 13 0 0 1 0 -26z m-3 4 l0 6 l-6 0 l0 6 l6 0 l0 6 l6 0 l0 -6 l6 0 l0 -6 l-6 0 l0 -6z"></path></svg>',
            event: 'newsolution'
        },
        "copy": {
            name: 'copy',
            label: {
                en: 'Copy this solution',
                fi: 'Kopioi tämä ratkaisu'
            },
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-copy"><path style="stroke: none;" d="M3 3 l12 0 l4 4 l0 1 l3 0 l4 4 l0 15 l-16 0 l0 -5 l-7 0z m1 1 l0 17 l6 0 l0 -5 l-4 0 l0 -1 l4 0 l0 -2 l-4 0 l0 -1 l4 0 l0 -2 l-4 0 l0 -1 l4 0 l0 -1 l8 0 l0 -1 l-3 0 l0 -3 z m7 5 l0 17 l14 0 l0 -14 l-3 0 l0 -3z m1 5 l12 0 l0 1 l-12 0z m0 3 l12 0 l0 1 l-12 0z m0 3 l12 0 l0 1 l-12 0z"></path></svg>',
            event: 'copysolution'
        },
        "send": {
            name: 'send',
            label: {
                en: 'Send solution',
                fi: 'Lähetä ratkaisu'
            },
            icon: SolutionTool.icons.send,
            event: 'sendsolution'
        }
    }
    
    /******
     * Available buttons for teacher in review mode
     ******/
    SolutionTool.teacherbuttons = {
        "sendall": {
            name: 'sendall',
            label: {
                en: 'Send all reviews',
                fi: 'Lähetä kaikki arviot'
            },
            event: 'sendreviews',
            icon: SolutionTool.icons.send
        }
    }
    
    /******
     * The css-styles for solution tool
     ******/
    SolutionTool.styles = [
        '[data-elementmode="view"][data-elementrole="teacher"] + .solutiontool {margin: 0; margin-bottom: 2em; border: 1px dashed black; border-top: none; background-color: aliceblue;}',
        '[data-elementrole="student"] + .solutiontool {margin: 0; margin-bottom: 2em; border: 1px dashed black; border-top: none; background-color: aliceblue;}',
        '.solutiontool-button {padding: 2px;}',
        '.solutiontool-buttontext {display: none;}',
        '.solutiontool-buttonicon {display: inline-block; vertical-align: text-bottom; height: 25px;}',
        '.solutiontool-buttonicon > svg {width: 25px; height: 25px; margin: 0 0.5em;}',
        '.solutiontool-buttonbar {margin: 0; padding: 0.3em 0 0 1em;}',
        '.solutiontool-area {margin: 0; padding: 1em;}',
        '[data-elementrole="student"] + .solutiontool .solutiontool-area {margin: 0.5em 0 0; padding: 0 1em 1em;}',
        '.solutiontool-solutiontablist {list-style: none; margin: 0; padding: 0 0.5em;}',
        '.solutiontool-solutiontablist > li {display: inline-block; margin: 0 -5px 0 0; border-radius: 0.5em 0.5em 0 0; border: 1px solid #666; border-bottom: none; background-color: #f0f0f0; min-width: 2em; text-align: center; cursor: pointer; position: relative;}',
        '.solutiontool-solutiontablist > li.solutiontool-currenttab {font-weight: bold; background-color: #444; color: white; z-index: 2; padding-top: 0.3em; border-top-width: 3px; border-top-color: black;}',
        '.solutiontool-solutiontablist > li.solutiontool-currenttab:only-child {visibility: hidden;}',
        '.solutiontool-solutiontablist > li[data-solution-isopen="true"] {color: black; background-color: #f66; font-weight: bold;}',
        '.solutiontool-content {border: 1px solid #666;}',
        '[data-elementrole="student"] + .solutiontool .solutiontool-content {background-color: white;}',
        '.solutiontool-area .solutiontool-content {background-color: white;}',
        'ul.solutiontool-reviewaccordion {border: 1px solid #666; border-radius: 6px; list-style: none; padding: 0; margin: 1em;}',
        'ul.solutiontool-reviewaccordion:empty {display: none;}',
        'ul.solutiontool-reviewaccordion.withmaxheight {max-height: 30em; overflow-y: auto;}',
        '.solutiontool-review-useritem {margin: 0; padding: 0; background-color: white;}',
        '.solutiontool-review-useritem .solutiontool-review-username {margin: 0; padding: 0.2em 0.5em; font-size: 110%; font-weight: bold; font-family: sans-serif; border-bottom: 1px solid #666; cursor: pointer; display: flex; flex-direction: row; justify-content: space-between;}',
        '.solutiontool-review-useritem .solutiontool-reviewitem {border-bottom: 1px solid #666; padding: 0.5em 1em; display: none;}',
        '.solutiontool-review-useritem:first-child {border-radius: 6px 6px 0 0;}',
        '.solutiontool-review-useritem:last-child {border-radius: 0 0 6px 6px;}',
        '.solutiontool-review-useritem:first-child:last-child {border-radius: 6px;}',
        '.solutiontool-review-useritem:first-child .solutiontool-review-username {border-radius: 6px 6px 0 0;}',
        '.solutiontool-review-useritem:last-child .solutiontool-review-username:not([data-isopen="1"]) {border-bottom: none; border-radius: 0 0 6px 6px;}',
        '.solutiontool-review-useritem:first-child:last-child .solutiontool-review-username:not([data-isopen="1"]) {border-radius: 6px;}',
        '.solutiontool-review-useritem:last-child .solutiontool-reviewitem {border-bottom: none;}',
        '.solutiontool-review-useritem .solutiontool-review-username[data-isopen="1"] + .solutiontool-reviewitem {display: block;}',
        
        // Selector (solution edit, review)
        '.solutiontool-selector {display: none; margin: 0; padding: 0.5em 0 0.5em 1em;}',
        '.solutiontool-selector button {padding: 2px 2em;}',
        '.solutiontool-areaselectable .solutiontool-selector {display: block;}',
        '.solutiontool-viewarea, .solutiontool-reviewarea {display: none;}',
        '.solutiontool-initing .solutiontool-viewarea {display: block;}',
        '[data-selectedarea="view"] .solutiontool-viewarea {display: block;}',
        '[data-selectedarea="review"] .solutiontool-reviewarea {display: block;}',
        
        // Reviewlist
        '.solutiontool-empty {display: none;}',
        '.solutiontool-reviewtools {background-color: #f0f0f0; box-shadow: 0 0 3px rgba(0,0,0,0.5); border-bottom: 1px solid #777; display: flex; flex-direction: row; justify-content: space-between;}',
        '.solutiontool-reviewlist {list-style: none; margin: 0; padding: 0.2em 0.3em; background-color: #eee;}',
        '.solutiontool-reviewlist .solutiontool-reviewstart-button {display: inline-block; margin: 0.2em 0.5em;}',
        '.solutiontool-reviewlist .solutiontool-reviewstart-button button {padding: 0em 0.3em 0.2em; border-radius: 1em; font-size: 100%; vertical-align: middle; line-height: 100%;}',
        '.solutiontool-reviewlist .solutiontool-reviewlist-item {font-size: 70%; display: inline-block; margin: 0.2em 0.3em;}',
        '.solutiontool-reviewlist .solutiontool-reviewlist-item button {padding: 0em 0.3em 0.2em; border-radius: 1em; font-size: 100%;}',
        '.solutiontool-review-userstats {font-weight: normal; font-family: monospace; font-size: 80%;}',
        '.solutiontool-review-userstats span {display: inline-block; min-width: 3em; text-align: center;}',
        '.solutiontool-review-userunreviewed {color: red;}',
        '.solutiontool-reviewitem .solutiontool-solutiontablist li[data-hasreview="false"] {box-shadow: inset 0 0 20px rgba(255,0,0,0.5);}',
        '[data-elementrole="student"] + .solutiontool .solutiontool-solutiontablist li[data-hasreview="true"] {box-shadow: inset 0 0 8px rgba(0,170,0,0.5);}',
        '[data-elementrole="student"] + .solutiontool .solutiontool-solutiontablist li[data-hasreview="true"]:not([data-unreadrevs="0"]) {font-weight: bold; color: red;}',
        '[data-elementrole="teacher"] + .solutiontool .solutiontool-solutiontablist li[data-hasunsentrevs="true"] {font-weight: bold; color: red;}',
        
        // Teacherbuttons
        '.solutiontool-teacherutils {border-bottom: 1px solid #999; box-shadow: 0 3px 6px rgba(0,0,0,0.3); padding: 3px 6px; display: flex; flex-direction: row; justify-content: space-between;}',
        '.solutiontool-teacherbuttons {margin: 2px;}',
        '.solutiontool-teacherbuttons button {padding: 0;}',
        '.solutiontool-teacherbuttons button svg {width: 20px; height: 20px;}',
        '.solutiontool-teacherbuttons button[data-buttonname="sendall"]::before {content: attr(data-info) " "; vertical-align: top; font-weight: bold;}',
        '.solutiontool-teacherbuttons button[data-buttonname="sendall"]:not([data-info="0"])::before {color: #a00;}',
        '.solutiontool-reviewtools-buttons {margin: 0; padding: 3px;}',
        '.solutiontool-reviewtools-buttons button {padding: 0; cursor: pointer;}',
        '.solutiontool-reviewtools-buttons button svg {width: 20px; height: 20px;}',
        '.solutiontool-content[data-unsentrevs="false"] .solutiontool-reviewtools button.solutiontool-reviewbutton-send {display: none;}',
        '.solutiontool-content[data-unsentrevs="true"] .solutiontool-reviewtools button.solutiontool-reviewstart-button {display: none;}',
        
        // Droptarget
        '.solutiontool-content .elementset-droptarget[data-droptarget-cancontain="elements"] {background-color: #fafafa; box-shadow: 0 0 1px 1px rgba(0,0,0,0.3);}',
        
        // Indicators for solution and review counts
        '.solutiontool:not([data-totalsolution="0"]) .assignment-teachercontrol button[data-selectarea="review"] svg {fill: green;}',
        '.solutiontool:not([data-totalunrevsol="0"]) .assignment-teachercontrol button[data-selectarea="review"] svg {fill: red;}',
        '.solutiontool:not([data-unpublishedreviews="0"]) .assignment-teachercontrol button[data-selectarea="review"] svg {fill: red;}',
        '.solutiontool[data-hasopensolution="true"] .assignment-teachercontrol button[data-selectarea="view"] svg {fill: red;}'
        
    ].join('\n');
    
    /******
     * Properties in different modes
     ******/
    SolutionTool.modes = {
        view: {
            editable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            reviewable: false
        },
        review: {
            editable: false,
            reviewable: true
        }
    };
    
    /******
     * Properties of different roles
     ******/
    
    SolutionTool.roles = {
        student: {
            studentable: true,
            teacherable: false,
            rolechangeable: false
        },
        teacher: {
            studentable: false,
            teacherable: true,
            rolechangeable: true
        },
        teacherstudent: {
            studentable: true,
            teacherable: false,
            rolechangeable: true
        }
    }
    
    /******
     * Default settings for a solution tool.
     ******/
    SolutionTool.defaults = {
        type: 'solutiontool',
        metadata: {},
        data: {
            assignmentid: '',
            contents: [],
            contentdata: {}
        },
        settings: {
            mode: 'view',
            role: 'student',
            isindependent: true,
            lang: 'en',
            uilang: 'en'
        }
    }
    
    /******
     * Localization strings
     ******/
    SolutionTool.localization = {
        
    }
    
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var solutiontool = new SolutionTool(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.solutiontool = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in solutiontool.');
            return false;
        }
    }
    
})(jQuery);