/*********************************************************
 * jquery.signchart.js
 * jQuery-plugin for creating a sign chart and derivative sign chart (curve sketcing)
 * Petri Salmela
 * petri.salmela@fourferries.fi
 * 08.11.2017
 *
 * Copyright: Four Ferries Oy
 * https://fourferries.com
 ********************************************************/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    // jQuery plugin
    var methods = {
        'init': function(params){
            return this.each(function(){
                var signchart = new Signchart(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        }
    }
    

    $.fn.signchart = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else if (typeof(method) === 'string') {
            var cmd = method;
            var options = arguments[1] || {};
            if (typeof(options) === 'string') {
                options = {name: options};
            }
            // Placeholder variable for returning value.
            options.result = this;
            $(this).children('div.signchart').trigger(cmd, options);
            return options.result;
        } else {
            $.error('Method ' + method + ' does not exist in Signchart.');
            return false;
        };
    };
    
    /**
     * Sign chart element as a class
     * @class Signchart
     * @constructor
     */
    var Signchart = function(place, options){
        // Constructor for Signchart object.
        options = $.extend(true, {}, this.defaults, options);
        this.type = 'signchartelement';
        this.settings = options.settings;
        this.metadata = options.metadata;
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.data = options.data;
        //this.wrapper = $(place);
        //this.wrapper.html('<div class="signchart"></div>').addClass('signchartwrapper');
        //this.place = this.wrapper.find('.signchart');
        this.place = $(place);
        this.place.addClass('signchart');
        this.rows = [];
        this.roots = [];
        
        if ($('head style#signchartstyle').length == 0){
            $('head').append('<style id="signchartstyle" type="text/css">'+this.styles+'</style>');
        }
        this.init();
    };
    
    /**
     * Init the signchart
     */
    Signchart.prototype.init = function(){
        // Init and draw the signchart
        var signchart = this;
        if (this.place.hasClass('sc_rendered')){
            return false;
        };
        this.place.addClass('sc_rendered');
        var schart = `
            <div class="signchartwrapper">
            </div>
            <div class="sc_caption">${escapeHTML(this.data.caption)}</div>
        `;
        this.place.html(schart);
        this.wrapper = this.place.children('.signchartwrapper');
        this.captionelem = this.place.children('.sc_caption');
        this.setMode(this.settings.mode);
        this.setData(this.data, true);
        this.show();
        this.initHandlers();
    }
    
    /**
     * Set the mode
     * @param {String} mode   The mode of the element ('view'|'edit'|'review'|'author')
     */
    Signchart.prototype.setMode = function(mode){
        if (!this.modes[mode]) {
            mode = 'view';
        };
        this.settings.mode = mode;
        var modedata = this.modes[mode];
        this.editable = modedata.editable;
        this.reviewable = modedata.reviewable;
    };
    
    /**
     * Show the signchart
     */
    Signchart.prototype.show = function(){
        if (this.editable) {
            this.place.addClass('editmode').attr('data-elementmode', this.settings.mode);
            this.edit();
        } else {
            this.place.removeClass('editmode').attr('data-elementmode', this.settings.mode);
            this.view();
        }
    }
    
    /**
     * Show the element in view mode
     */
    Signchart.prototype.view = function() {
        this.wrapper.empty();
        this.wrapper.append(this.getRootsHtmlView());
        var isroots;
        var roots = this.roots.count();
        for (let i = 0, len = this.rows.length; i < len; i++) {
            isroots = this.roots.getRootsOfRow(i);
            this.wrapper.append(this.rows[i].getHtml(isroots, i));
        };
        if (this.rows.length > 1) {
            this.wrapper.append(this.total.getHtml(roots));
            this.wrapper.append(this.summary.getHtml(roots, this.total.getFunc()));
        } else if (this.rows.length === 1) {
            this.wrapper.append(this.summary.getHtml(roots));
        };
        this.makeMath();
    };
    
    /**
     * Show the element in edit mode
     */
    Signchart.prototype.edit = function() {
        this.wrapper.empty();
        this.wrapper.append(this.getRootsHtmlView());
        var isroots;
        var roots = this.roots.count();
        for (let i = 0, len = this.rows.length; i < len; i++) {
            isroots = this.roots.getRootsOfRow(i);
            this.wrapper.append(this.rows[i].getHtml(isroots, i));
        };
        this.wrapper.append(`<div class="signchart-addrow"><div class="signchart-addicon">${Signchart.icons.add}</div></div>`);
        if (this.rows.length > 1) {
            this.wrapper.append(this.total.getHtml(roots));
            this.wrapper.append(this.summary.getHtml(roots, this.total.getFunc(), 'edit'));
        } else if (this.rows.length === 1) {
            this.wrapper.append(this.summary.getHtml(roots, this.rows[0].getFunc(), 'edit'));
        };
        this.makeMath();
    };
    
    /**
     * Mathquillify the math
     */
    Signchart.prototype.makeMath = function() {
        if (this.editable) {
            this.wrapper.find('.signchart-math:not(.mathquill-rendered-math)').mathquill('editable');
            this.wrapper.find('.signchart-math-noneditable:not(.mathquill-rendered-math)').mathquill();
        } else {
            this.wrapper.find('.signchart-math:not(.mathquill-rendered-math)').mathquill();
        }
    }
    
    /**
     * Get html for roots in view mode
     * @returns {String} the html for roots
     */
    Signchart.prototype.getRootsHtmlView = function() {
        var html = [
            '<div class="signchart-rootsempty signchart-emptyleft"></div>'
        ];
        html = html.concat(this.roots.getHtml());
        //html.push('<div class="signchart-rootsempty"></div>');
        return html.join('\n');
    };

    /**
     * Add a new row with function.
     * @param {Object} options   Data for the row
     */
    Signchart.prototype.addFunc = function(options) {
        this.rows.push(new ScRow(options));
    };
    
    /**
     * Get the data of the signchart
     * @returns {Object}    The all data of the signchartelement
     */
    Signchart.prototype.getData = function(){
        var result = {
            type: this.type,
            metadata: $.extend(true, {}, this.defaults.metadata, this.metadata),
            data: {
                roots: this.roots.getData(),
                rows: [],
                total: this.total.getData(),
                summary: this.summary.getData()
            }
        };
        for (let i = 0, len = this.rows.length; i < len; i++) {
            result.data.rows.push(this.rows[i].getData());
        };
        return result;
    };
    
    /**
     * Set data of the signchart
     * @param {Object} options    The data
     * @param {Boolean} isinit    Is the first init.
     */
    Signchart.prototype.setData = function(options, isinit){
        this.caption = options.caption || '';
        this.roots = new ScRoots(this.data.roots);
        this.rows = [];
        for (let i = 0, len = options.rows.length; i < len; i++) {
            this.addFunc(options.rows[i])
        };
        this.total = new ScTotal(options.total);
        this.summary = new ScSummary(options.summary);
    };
        
    /**
     * Show the selector for the motivation on given row
     * @param {Number} row     The index of the row
     */
    Signchart.prototype.showMotivationSelector = function(row) {
        var place = this.place.find(`.signchart-motivationslot[data-rowid="${row}"]`);
        var html = this.rows[row].getMotivationPanel();
        var panel = $(html);
        panel.attr('data-motivation-for-row', row);
        place.append(panel);
    };
    
    /**
     * Change the value of root label
     * @param {Number} index   The index of the root
     * @param {String} label   Changed label
     */
    Signchart.prototype.changeRootLabel = function(index, label) {
        var changed = false;
        if (label) {
            changed = this.roots.changeLabel(index, label);
        };
        if (changed) {
            this.changed();
        };
    };
    
    /**
     * Change the value of row function
     * @param {Number} index   The index of the row
     * @param {String} func    The changed function
     */
    Signchart.prototype.changeRowFunc = function(index, func) {
        var changed = false;
        if (func && typeof(index) === 'number' && 0 <= index && index < this.rows.length) {
            changed = this.rows[index].setFunc(func);
        };
        if (changed) {
            this.changed();
        };
    };
    
    /**
     * Change the value of total function
     * @param {String} func    The changed function
     */
    Signchart.prototype.changeTotalFunc = function(func) {
        var changed = false;
        if (func) {
            changed = this.total.setFunc(func);
        };
        if (changed) {
            this.changed();
            var roots = this.roots.count();
            var total = this.place.find('.signchart-totalrow');
            total.last().after(this.total.getHtml(roots));
            total.remove();
            var summary = this.place.find('.signchart-summaryrow');
            summary.last().after(this.summary.getHtml(roots, this.total.getFunc(), 'edit'));
            summary.remove();
            this.makeMath();
        };
    };
    
    /**
     * Change the value of summary function
     * @param {String} func    The changed function
     */
    Signchart.prototype.changeSummaryFunc = function(func) {
        var changed = false;
        if (typeof(func) === 'string') {
            changed = this.summary.setFunc(func);
        };
        if (changed) {
            this.changed();
            var roots = this.roots.count();
            var summary = this.place.find('.signchart-summaryrow');
            summary.last().after(this.summary.getHtml(roots, '', 'edit'));
            summary.remove();
            this.makeMath();
        };
    };
    
    /**
     * Change the relation to the next one
     */
    Signchart.prototype.changeNextRelation = function() {
        this.summary.setNextRelation();
        this.changed();
        var summary = this.place.find('.signchart-summaryrow');
        summary.last().after(this.summary.getHtml(this.roots.count(), this.total.getFunc(), 'edit'));
        summary.remove();
        this.makeMath();
    };
    
    /**
     * Change the motivation of given row
     * @param {Number} row     The row index
     * @param {String} mot     The motivation
     */
    Signchart.prototype.changeMotivation = function(row, mot) {
        if (typeof(row) === 'number' && 0 <= row && row < this.rows.length) {
            this.rows[row].setMotivation(mot);
            var symbol = this.rows[row].getMotivationSymbol();
            this.place.find(`.signchart-motivationslot[data-rowid="${row}"] .signchart-motivation`).html(symbol);
        };
        this.changed();
    }
    
    /**
     * Add a new root
     * @param {Number} index    The index where to add the new root.
     */
    Signchart.prototype.addRoot = function(index) {
        this.roots.addRoot(index, '');
        this.changed();
        this.show();
        this.place.find(`.signchart-rootitem[data-rootindex="${index}"] .signchart-rootwrapper .signchart-math`).focus();
    };
    
    /**
     * Remove a root
     * @param {Number} index    The index of the root to remove.
     */
    Signchart.prototype.removeRoot = function(index) {
        this.roots.removeRoot(index);
        this.changed();
        this.show();
    };
    
    /**
     * Toggle root in given row
     * @param {Number} row   The index of the row
     * @param {Number} root  The index of the root
     */
    Signchart.prototype.toggleRoot = function(row, root) {
        this.roots.toggleRootInRow(row, root);
        this.changed();
        this.show();
    };
    
    /**
     * Toggle sign of the slot in given row
     * @param {Number} row     The index of the row
     * @param {Number} signid  The index of the sign
     */
    Signchart.prototype.toggleRowSign = function(row, signid) {
        if (0 <= row && row < this.rows.length) {
            var newsign = this.rows[row].setNextSign(signid);
            this.changed();
            this.place.find(`.signchart-signslot[data-rowid=${row}][data-signid="${signid}"]`).attr('data-sign', newsign);
        };
    };
    
    /**
     * Toggle sign of the slot in total row
     * @param {Number} signid  The index of the sign
     */
    Signchart.prototype.toggleTotalSign = function(signid) {
        var newsign = this.total.setNextSign(signid);
        this.changed();
        this.place.find(`.signchart-signslot.signchart-totalrow[data-signid="${signid}"]`).attr('data-sign', newsign);
    };
    
    /**
     * Toggle undefinedness of the root in total row
     * @param {Number} rootid  The index of the root
     */
    Signchart.prototype.toggleTotalUndefined = function(rootid) {
        var newState = this.total.toggleUndefinedPoint(rootid);
        this.changed();
        var place = this.place.find(`.signchart-signslot.signchart-totalrow[data-rootid="${rootid}"] .signchart-totalundefined`);
        if (newState) {
            place.addClass('isundefined');
        } else {
            place.removeClass('isundefined');
        };
    };
    
    /**
     * Toggle the direction on summary row
     * @param {Number} rootid  The index of the root
     */
    Signchart.prototype.toggleDirection = function(rootid) {
        var newdir = this.summary.setNextDir(rootid);
        this.changed();
        var slot = this.place.find(`.signchart-summary-directionslot[data-rootid="${rootid}"]`);
        slot.attr('data-direction', newdir.name);
        slot.children('.signchart-directionsymbol').html(newdir.symbol);
    };
    
    /**
     * Toggle the interval
     * @param {Number} index  The index of the interval
     */
    Signchart.prototype.toggleSummaryInterval = function(index) {
        var newstate = this.summary.toggleInterval(index);
        this.changed();
        var slot = this.place.find(`.signchart-interval[data-intervalindex="${index}"]`);
        slot.attr('data-intervaltype', newstate);
    };
    
    /**
     * Toggle the interval stop
     * @param {Number} index  The index of the intervalstop
     */
    Signchart.prototype.toggleSummaryRootpoint = function(index) {
        var newstate = this.summary.nextRootpoint(index);
        this.changed();
        var slot = this.place.find(`.signchart-intervalstop[data-intervalroot="${index}"]`);
        slot.attr('data-intervalroot-type', newstate);
    };
    
    /**
     * Toggle type of summary row between "intervals" and "directions"
     */
    Signchart.prototype.toggleSummaryType = function() {
        this.summary.toggleType();
        this.changed();
        var summary = this.place.find('.signchart-summaryrow');
        summary.last().after(this.summary.getHtml(this.roots.count(), this.total.getFunc(), 'edit'));
        summary.remove();
        this.makeMath();
    };
    
    /**
     * Add a new row
     */
    Signchart.prototype.addNewRow = function() {
        var index = this.rows.length;
        this.rows.push(new ScRow());
        if (index > 1) {
            var place = this.place.find('.signchart-addrow');
            var isroots = this.roots.getRootsOfRow(index);
            place.before(this.rows[index].getHtml(isroots, index));
            this.makeMath();
        } else {
            // Moving from one row to more: must add the total row.
            this.show();
        }
        this.place.find(`.signchart-formula[data-rowid="${index}"] .signchart-math`).focus();
        this.changed();
    };
    
    /**
     * Remove row
     * @param {Number} row   The index of the row
     */
    Signchart.prototype.removeRow = function(row) {
        this.rows.splice(row, 1);
        this.roots.removeRow(row);
        this.changed();
        this.show();
    };
    
    
    /**
     * Element has changed. Inform any entities outside.
     */
    Signchart.prototype.changed = function(){
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
        this.place.trigger('element_changed', {type: this.type});
    };
    
    /**
     * Init event handlers
     */
    Signchart.prototype.initHandlers = function() {
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersView();
        };
    };
    
    /**
     * Init handlers for all modes
     */
    Signchart.prototype.initHandlersCommon = function() {
        var element = this;
        this.place.on('getdata', function(event) {
            event.stopPropagation();
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    };
    
    /**
     * Init handlers for view mode
     */
    Signchart.prototype.initHandlersView = function() {
        
    }
    
    /**
     * Init handlers for edit mode
     */
    Signchart.prototype.initHandlersEdit = function() {
        var element = this;
        // Key events in Mathquill
        this.place.off('keyup', '.signchart-math.mathquill-editable').on('keyup', '.signchart-math.mathquill-editable', function(event) {
            switch (event.which) {
                case 13: // Enter
                    $(this).trigger('focusout').trigger('blur')
                    var mq = $(this);
                    setTimeout(function(){mq.focusout();}, 0);
                    break;
                case 40: // Down
                    var all = element.place.find('.signchart-math.mathquill-editable');
                    var current = $(this);
                    var index = all.index(current);
                    if (current.is('.hasCursor') && index < all.length - 1) {
                        all.eq(index + 1).focus();
                    };
                    break;
                case 38: // Up
                    var all = element.place.find('.signchart-math.mathquill-editable');
                    var current = $(this);
                    var index = all.index(current);
                    if (current.is('.hasCursor') && index > 0) {
                        all.eq(index - 1).focus();
                    };
                    break;
                default:
                    break;
            };
        }).off('focusout', '.signchart-rootitem .mathquill-editable').on('focusout', '.signchart-rootitem .mathquill-editable', function(event) {
           var mq = $(this);
           var latex = mq.mathquill('latex');
           var rootitem = mq.closest('.signchart-rootitem');
           var rootindex = rootitem.attr('data-rootindex') | 0;
           element.changeRootLabel(rootindex, latex);
        }).off('focusout', '.signchart-formula .mathquill-editable').on('focusout', '.signchart-formula .mathquill-editable', function(event) {
           var mq = $(this);
           var latex = mq.mathquill('latex');
           var formulaitem = mq.closest('.signchart-formula');
           var rowindex = formulaitem.attr('data-rowid') | 0;
           element.changeRowFunc(rowindex, latex);
        }).off('focusout', '.signchart-totalformula .mathquill-editable').on('focusout', '.signchart-totalformula .mathquill-editable', function(event) {
           var mq = $(this);
           var latex = mq.mathquill('latex');
           element.changeTotalFunc(latex);
        }).off('focusout', '.signchart-summary-directionformula .mathquill-editable').on('focusout', '.signchart-summary-directionformula .mathquill-editable', function(event) {
           var mq = $(this);
           var latex = mq.mathquill('latex');
           element.changeSummaryFunc(latex);
        }).off('click', '.signchart-summary-formula .signchart-relation').on('click', '.signchart-summary-formula .signchart-relation', function(event) {
            event.stopPropagation();
            var relation = $(this).closest('.signchart-relation').attr('data-relation');
            element.changeNextRelation();
        }).off('click', '.signchart-addroot').on('click', '.signchart-addroot', function(event) {
            event.stopPropagation();
            var adder = $(this).parent();
            var index = adder.attr('data-rootindex') | 0;
            element.addRoot(index);
        }).off('click', '.signchart-removeroot').on('click', '.signchart-removeroot', function(event) {
            event.stopPropagation();
            var remover = $(this).closest('.signchart-rootitem');
            var index = remover.attr('data-rootindex') | 0;
            element.removeRoot(index);
        }).off('click', '.signchart-rowroot-addremove').on('click', '.signchart-rowroot-addremove', function(event) {
            event.stopPropagation();
            var changer = $(this).closest('.signchart-signslot');
            var row = changer.attr('data-rowid') | 0;
            var root = changer.attr('data-rootid') | 0;
            element.toggleRoot(row, root);
        }).off('click', '.signchart-signslot[data-rowid]').on('click', '.signchart-signslot[data-rowid]', function(event) {
            event.stopPropagation();
            var slot = $(this);
            var row = slot.attr('data-rowid') | 0;
            var signid = slot.attr('data-signid') | 0;
            element.toggleRowSign(row, signid);
        }).off('click', '.signchart-signslot.signchart-totalrow').on('click', '.signchart-signslot.signchart-totalrow', function(event) {
            event.stopPropagation();
            var slot = $(this);
            var signid = slot.attr('data-signid') | 0;
            element.toggleTotalSign(signid);
        }).off('click', '.signchart-summary-directionslot').on('click', '.signchart-summary-directionslot', function(event) {
            event.stopPropagation();
            var slot = $(this);
            var rootid = slot.attr('data-rootid') | 0;
            element.toggleDirection(rootid);
        }).off('click', '.signchart-totalundefined').on('click', '.signchart-totalundefined', function(event) {
            event.stopPropagation();
            var changer = $(this).closest('.signchart-signslot');
            var root = changer.attr('data-rootid') | 0;
            element.toggleTotalUndefined(root);
        }).off('click', '.signchart-interval').on('click', '.signchart-interval', function(event) {
            event.stopPropagation();
            var index = $(this).attr('data-intervalindex') | 0;
            element.toggleSummaryInterval(index);
        }).off('click', '.signchart-intervalstop').on('click', '.signchart-intervalstop', function(event) {
            event.stopPropagation();
            var root = $(this).attr('data-intervalroot') | 0;
            element.toggleSummaryRootpoint(root);
        }).off('click', '.signchart-motivation').on('click', '.signchart-motivation', function(event) {
            event.stopPropagation();
            var row = $(this).parent().attr('data-rowid') | 0;
            element.showMotivationSelector(row);
        }).off('click', '.signchart-motivationitem').on('click', '.signchart-motivationitem', function(event) {
            event.stopPropagation();
            var item = $(this);
            var panel = item.parent();
            var mot = item.attr('data-motivationid');
            var row = panel.attr('data-motivation-for-row') | 0;
            panel.remove();
            element.changeMotivation(row, mot);
        }).off('click', '.signchart-summary-type').on('click', '.signchart-summary-type', function(event) {
            event.stopPropagation();
            element.toggleSummaryType();
        }).off('click', '.signchart-addrow').on('click', '.signchart-addrow', function(event) {
            event.stopPropagation();
            element.addNewRow();
        }).off('click', '.signchart-removerow').on('click', '.signchart-removerow', function(event) {
            event.stopPropagation();
            var row = $(this).parent().attr('data-rowid') | 0;
            element.removeRow(row);
        });
    }
    
    /**
     * Default values for empty signchart
     */
    Signchart.prototype.defaults = {
        "type": "signchart",
        "metadata": {
            "creator": "",
            "created": 0,
            "modifier": "",
            "modified": 0,
            "tags": []
        },
        "data": {
            caption: '',
            rows: [{func: ''}],
            total: {func: '', signs: [''], relation: '\\lt'},
            summary: {
                type: 'intervals',
                intervals: [],
                rootpoints: [],
                undefinedpoint: [],
                directions: []
            }
        },
        "settings": {
            mode: 'view'
        }
    };
    
    /**
     * Editing ect. rights for different modes
     */
    Signchart.prototype.modes = {
        view: {
            editable: false
        },
        edit: {
            editable: true
        }
    };
    
    /**
     * CSS stylesheets
     */
    Signchart.prototype.styles = `
        .signchart {
            padding: 1em;
            margin: 0;
            background-color: transparent;
        }
        .signchartwrapper {
            display: grid;
            grid-template-columns: [formula] auto [motivation] 40px;
            grid-gap: 0;
            padding-top: 2em;
        }
        .signchart-rootsempty {
            grid-row: 1 / 2;
            min-width: 2em;
            border-bottom: 1px solid #555;
        }
        .signchart-rootsempty.signchart-emptyleft {
            grid-column: 1 / 3;
        }
        .signchart-rootitem {
            grid-row: 1 / 2;
            border-bottom: 1px solid #555;
            border-right: 1px solid #555;
            position: relative;
            min-height: 10px;
        }
        .signchart-addroot {
            height: 6px;
            display: none;
            background-transparent;
            border-top: 2px solid transparent;
            border-bottom: 2px solid transparent;
        }
        .signchart-rootwrapper {
            position: absolute;
            right: 0;
            bottom: 0;
            margin-bottom: 10px;
            overflow: visible;
            text-align: center;
            font-size: 90%;
        }
        .signchart-rootwrapper .signchart-math {
            margin-right: -100%;
        }
        .signchart-rootwrapper .signchart-math.mathquill-editable {
            border-radius: 4px;
        }
        .signchart-removeroot {
            display: none;
            position: absolute;
            top: -25px;
            left: 100%;
            margin-left: -10px;
            cursor: pointer;
        }
        .signchart-removeroot svg {
            width: 20px;
            height: 20px;
        }
        .signchart-formula {
            border-left: 1px solid black;
            padding: 0.2em 0.5em;
            background-color: white;
        }
        .signchart-totalformula {
            border: 1px solid black;
            border-top: 2px solid black;
            grid-column: 1 / 3;
            padding: 0.2em 0.5em;
            background-color: white;
        }
        .signchart-formula .signchart-math.mathquill-editable,
        .signchart-totalformula .signchart-math.mathquill-editable,
        .signchart-summary-directionformula .signchart-math.mathquill-editable {
            display: block;
        }
        .signchart-motivationslot {
            border-right: 1px solid black;
            background-color: white;
            position: relative;
        }
        .signchart-motivation {
            display: inline-block;
            padding: 3px;
            margin: 2px;
        }
        .signchart-motivation svg {
            width: 25px;
            height: 25px;
        }
        .signchart[data-elementmode="edit"] .signchart-motivationslot .signchart-motivation {
            border-radius: 4px;
            border: 1px solid #999;
            background-color: #eee;
            cursor: pointer;
        }
        .signchart-signslot {
            text-align: center;
            font-weight: bold;
            border-right: 1px dotted black;
            background-color: white;
            font-family: sans-serif;
            position: relative;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }
        .signchart-lastsign {
            border-right: 1px solid black;
        }
        .signchart-totalsign {
            border-top: 2px solid black;
            border-bottom: 1px solid black;
        }
        .signchart-signslot.signchart-nextisroot {
            border-right: 2px solid black;
        }
        .signchart-signslot.signchart-nextisroot + .signchart-signslot {
            border-left: 1px solid black;
        }
        .signchart-signslot[data-sign="plus"] {
            color: white;
            background-color: #c00;
            text-shadow: 0 0 2px black;
            background: linear-gradient(to bottom, rgba(248,80,50,1) 0%,rgba(241,111,92,1) 50%,rgba(246,41,12,1) 51%,rgba(240,47,23,1) 71%,rgba(231,56,39,1) 100%);
        }
        .signchart-signslot[data-sign="minus"] {
            color: white;
            background-color: #00c;
            text-shadow: 0 0 2px black;
            background: linear-gradient(to bottom, rgba(183,222,237,1) 0%,rgba(113,206,239,1) 50%,rgba(33,180,226,1) 51%,rgba(183,222,237,1) 100%);
        }
        .signchart-signslot[data-sign="plus"]::before {
            content: "+";
            display: block;
        }
        .signchart-signslot[data-sign="minus"]::before {
            content: "–";
            display: block;
        }
        .signchart-rowroot-addremove {
            position: absolute;
            top: 0;
            bottom: 0;
            width: 10px;
            right: -5px;
            z-index: 10;
            display: none;
        }
        .signchart-totalundefined {
            position: absolute;
            top: 0;
            bottom: 0;
            right: 0;
            width: 10px;
            margin-right: -5px;
            z-index: 10;
        }
        .signchart-totalundefined.isundefined {
            background: transparent url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAkAAAAGCAYAAAARx7TFAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAEJwAABCcB2U8dgAAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAABESURBVAiZjc6xDYBADAPAC12moqb9ghlZg2kYIzSP9AVFLFkubNmOqrIiIhIHTuy4zVBi4MIzdSC/kl9j5aaDzlx0jr/6Nimp6cXzJwAAAABJRU5ErkJggg==) center top repeat-y;
        }
        .signchart-summary-formula {
            grid-column: 1 / 3;
            padding: 0.2em 0.5em;
            text-align: center;
        }
        .signchart-summary-type {
            grid-column: 2 / 3;
            padding: 0.2em;
            text-align: center;
            display: none;
        }
        .signchart-summary-directiontype {
            background-color: white;
            border-bottom: 1px solid black;
            border-top: 1px solid black;
            border-right: 1px solid black;
        }
        .signchart-summary-intervalslot {
            position: relative;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }
        .signchart-interval {
            width: 100%;
            height: 0;
            border-bottom: 1px dotted black;
        }
        .signchart-interval[data-intervaltype="true"] {
            border-bottom: 3px solid red;
        }
        .signchart-intervalstop {
            position: absolute;
            right: 0;
            top: 50%;
            height: 8px;
            width: 8px;
            border-radius: 6px;
            margin-top: -6px;
            margin-right: -5px;
            border: 2px solid transparent;
            z-index: 10;
        }
        .signchart-intervalstop[data-intervalroot-type="open"] {
            border: 2px solid red;
            background-color: white;
        }
        .signchart-intervalstop[data-intervalroot-type="closed"] {
            border: 2px solid red;
            background-color: red;
        }
        .signchart-summary-directionformula {
            grid-column: 1 / 3;
            padding: 0.2em 0.5em;
            text-align-left;
            background-color: white;
            border: 1px solid black;
        }
        .signchart-summary-directionslot {
            background-color: white;
            text-align: center;
            border: 1px solid black;
            border-left: none;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }
        .signchart-motivationpanel {
            position: absolute;
            left: 0;
            top: -50%;
            background-color: #eee;
            border: 1px solid #bbb;
            border-radius: 5px;
            box-shadow: 2px 2px 10px rgba(0,0,0,0.2);
            display: grid;
            grid-template-columns: 32px 32px 32px;
            grid-gap: 4px;
            padding: 4px;
            z-index: 11;
        }
        .signchart-motivationpanel .signchart-motivationitem {
            border: 1px solid #ddd;
            border-radius: 2px;
            cursor: pointer;
        }
        .signchart-motivationpanel .signchart-motivationitem.signchart-motivationitem-current {
            box-shadow: 0 0 1px 2px rgba(255, 77, 77, 0.5);
            background-color: rgba(255,255,255,0.5);
        }
        .signchart-motivationpanel .signchart-motivationitem:hover {
            background-color: rgba(255,255,0,0.5);
            box-shadow: 0 0 0 2px rgba(200,200,200,0.5);
        }
        .signchart-totalrow {
            border-bottom: none;
        }
        .signchart-summaryrow {
            border-top: 1px solid black;
        }
        .signchart-removerow {
            display: none;
        }
        /********** Editing *********/
        .signchart[data-elementmode="edit"] .signchart-summary-formula .signchart-relation .binary-operator {
            color: red;
            font-weight: bold;
            cursor: pointer;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-formula .signchart-relation:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-addroot {
            display: block;
        }
        .signchart[data-elementmode="edit"] .signchart-rootitem .signchart-addroot:hover {
            background-color: rgba(255, 255, 0, 0.5);
            border-top: 2px solid rgba(200, 200, 200, 0.5);
            border-bottom: 2px solid rgba(200, 200, 200, 0.5);
            cursor: copy;
        }
        .signchart[data-elementmode="edit"] .signchart-rootwrapper:hover .signchart-removeroot {
            display: block;
            position: absolute;
        }
        .signchart[data-elementmode="edit"] .signchart-rowroot-addremove {
            display: block;
            cursor: pointer;
        }
        .signchart[data-elementmode="edit"] .signchart-rowroot-addremove:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-lastsign .signchart-rowroot-addremove {
            display: none;
        }
        .signchart[data-elementmode="edit"] .signchart-signslot {
            cursor: pointer;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-directionslot {
            cursor: pointer;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-directionslot:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-totalundefined:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-interval {
            cursor: pointer;
            padding-top: 3px;
            border-top: 2px solid transparent;
        }
        .signchart[data-elementmode="edit"] .signchart-interval:hover {
            box-shadow: 0 3px 0 rgba(255,255,0,0.5), 0 5px 0 rgba(200, 200, 200, 0.5);
            background-color: rgba(255,255,0,0.5);
            border-top: 2px solid rgba(200, 200, 200, 0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-intervalstop {
            cursor: pointer;
            margin-top: -4px;
            
        }
        .signchart[data-elementmode="edit"] .signchart-intervalstop:hover {
            box-shadow: 0 0 0 2px rgba(255,255,0,0.5), 0 0 0 5px rgba(200, 200, 200, 0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-summary-formula,
        .signchart[data-elementmode="edit"] .signchart-summary-directionformula {
            grid-column: 1 / 2;
            border-right: none;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-type {
            display: flex;
            flex-direction: column;
            justify-content: center;
            text-align: center;
            cursor: pointer;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-type svg {
            display: inline-block;
            margin: 5px;
        }
        .signchart[data-elementmode="edit"] .signchart-summary-type:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-addrow {
            height: 10px;
            background-color: transparent;
            margin: -5px 0;
            z-index: 10;
            cursor: copy;
            position: relative;
        }
        .signchart[data-elementmode="edit"] .signchart-addicon {
            position: absolute;
            top: -5px;
            left: -20px;
            height: 20px;
            width: 20px;
            opacity: 0.5;
        }
        .signchart[data-elementmode="edit"] .signchart-addrow:hover {
            background-color: rgba(255,255,0,0.5);
        }
        .signchart[data-elementmode="edit"] .signchart-addrow:hover .signchart-addicon {
            opacity: 1;
        }
        .signchart[data-elementmode="edit"] .signchart-removerow {
            display: block;
            position: absolute;
            right: -22px;
            width: 20px;
            height: 20px;
            cursor: pointer;
            opacity: 0.5;
        }
        .signchart[data-elementmode="edit"] .signchart-removerow:hover {
            opacity: 1;
        }
        /********** Mathquill **************/
        .signchart-math.mathquill-editable {
            background-color: white;
            border-color: #ddd;
        }
        .signchart-math.mathquill-editable .cursor {
            margin-top: -5px;
            margin-bottom: -2px;
        }
        `

    /**
     * Some icons in svg format
     */
    Signchart.icons = {
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-removeminus"><path style="fill: #e43; stroke: none;" d="M15 1 a14 14 0 0 1 0 28 a14 14 0 0 1 0 -28z"></path><path style="fill: white; stroke: none;" d="M8 13 h14 v4 h-14z"></path></svg>',
        add: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-addplus"><path style="fill: #8c6; stroke: none;" d="M15 1 a14 14 0 0 1 0 28 a14 14 0 0 1 0 -28z"></path><path style="fill: white; stroke: none;" d="M8 13 h5 v-5 h4 v5 h5 v4 h-5 v5 h-4 v-5 h-5z"></path></svg>'
    }
    

    
    
    
/*********************************************************************************/    
    
    
    
    
    
    
    /***********************************************************
     * Roots as a class
     * @class ScRoots
     * @constructor
     * @param {Array} options    Array of roots.
     */
    var ScRoots = function(options) {
        this.roots = options || [];
    };
    
    /**
     * Get count of roots
     * @returns {Number}   The count of the roots
     */
    ScRoots.prototype.count = function() {
        return this.roots.length;
    }
    
    /**
     * Add a new root in given place
     * @param {Number} index          Place of the new root
     * @param {Object|String} root    New root
     */
    ScRoots.prototype.addRoot = function(index, root) {
        if (typeof(root) === 'string') {
            root = {
                label: root,
                row: []
            };
        };
        if (0 <= index && index <= this.roots.length) {
            this.roots.splice(index, 0, root);
        };
    };
    
    /**
     * Remove a root
     * @param {Number} index    Index of the root to remove
     * @returns {Object}    The removed root
     */
    ScRoots.prototype.removeRoot = function(index) {
        var result = false;
        if (0 <= index && index < this.roots.length) {
            result = this.roots.splice(index, 1);
        };
        return result;
    };
    
    /**
     * Replace a root
     * @param {Number} index          Place for the new root to replace
     * @param {Object|String} root    New root
     */
    ScRoots.prototype.replaceRoot = function(index, root) {
        if (typeof(root) === 'string') {
            root = {
                label: root,
                row: []
            };
        };
        if (0 <= index && index < this.root.length) {
            this.roots.splice(index, 1, root);
        };
    };
    
    /**
     * Change label
     * @param {Number} index     Place of the root to change
     * @param {String} label     The new label for the root
     * @returns {Boolean}        True, if the root label was changed
     */
    ScRoots.prototype.changeLabel = function(index, label) {
        var root = this.roots[index];
        var changed = false;
        if (root && root.label !== label) {
            root.label = label;
            changed = true;
        };
        return changed;
    };
    
    /**
     * Toggle the root as root of the givenrow
     * @param {Number} row   The row index
     * @param {Number} index The index of the root
     */
    ScRoots.prototype.toggleRootInRow = function(row, index) {
        var root = this.roots[index];
        if (root) {
            var place = root.row.indexOf(row);
            if (place > -1) {
                root.row.splice(place, 1);
            } else {
                root.row.push(row);
            };
        };
    };
    
    /**
     * Remove a given row and move others if needed
     * @param {Number} row    The index of the row
     */
    ScRoots.prototype.removeRow = function(row) {
        if (typeof(row) === 'number') {
            var root, rowlist, newrows;
            for (let i = 0, len = this.roots.length; i < len; i++) {
                root = this.roots[i];
                rowlist = root.row;
                newrows = [];
                for (let j = 0, jlen = rowlist.length; j < jlen; j++) {
                    if (rowlist[j] < row) {
                        newrows.push(rowlist[j]);
                    } else if (rowlist[j] > row) {
                        newrows.push(rowlist[j] -1 );
                    };
                };
            };
        };
    };
    
    /**
     * Get the html for the roots
     * @returns {String[]}   Array of html-strings (one string for each root)
     */
    ScRoots.prototype.getHtml = function() {
        var result = [];
        for (let i = 0, len = this.roots.length; i < len; i++) {
            result.push(`
                <div class="signchart-rootitem" data-rootindex="${i}" data-rows="${this.roots[i].row.join(' ')}">
                    <div class="signchart-addroot"></div>
                    <div class="signchart-rootwrapper">
                        <span class="signchart-math">${this.roots[i].label}</span>
                        <div class="signchart-removeroot">${Signchart.icons.remove}</div>
                    </div>
                </div>
            `);
        };
        result.push(`<div class="signchart-rootsempty signchart-rootitem" data-rootindex="${this.roots.length}"><div class="signchart-addroot"></div></div>`)
        return result;
    };
    
    /**
     * Get an array with "isroots" of given row
     * @param {Number} row   The id of the row
     * @returns {Boolean[]}  An array of boolean values. True, if n'th root is root of given row.
     */
    ScRoots.prototype.getRootsOfRow = function(row) {
        var isroots = [];
        for (let i = 0, len = this.roots.length; i < len; i++) {
            isroots.push(this.roots[i].row.includes(row));
        };
        return isroots;
    };
    
    /**
     * Get the data of the roots
     * @returns {Object} the data of the roots
     */
    ScRoots.prototype.getData = function() {
        var result = JSON.parse(JSON.stringify(this.roots));
        return result;
    };

    
    
    
/*********************************************************************************/    
    
    
    
    
    /******************************************
     * Row of signchart
     * @class ScRow
     * @constructor
     ******************************************/
    var ScRow = function(options){
        options = $.extend({
            func: '',
            motivation: 'none',
            signs: []
        }, options);
        this.func = options.func;
        this.motivation = options.motivation;
        this.signs = options.signs;
    }
    
    /**
     * Get the function LaTeX-string
     * @returns {String} LaTeX-string
     */
    ScRow.prototype.getFunc = function(){
        return this.func || '';
    };
    
    /**
     * Set the function LaTeX-string
     * @param {String} func   The LaTeX-string
     * @returns {Boolean}     True, if the new string was different tha current
     */
    ScRow.prototype.setFunc = function(func) {
        var changed = this.func !== func;
        this.func = func;
        return changed;
    };
    
    /**
     * Get the sign in given index.
     * @param {Number} n    Index of the sign.
     */
    ScRow.prototype.getSign = function(n){
        return this.signs[n] || 'none';
    };
    
    /**
     * Set the sign in given index
     * @param {Number} n       The index of the sign
     * @param {String} sign    The sign to set
     */
    ScRow.prototype.setSign = function(n, sign){
        return this.signs[n] = sign || 'none';
    };
    
    /**
     * Set the sign the next in given index
     * @param {Number} n       The index of the sign
     * @returns {String}       The sign that was set
     */
    ScRow.prototype.setNextSign = function(n){
        var current = this.signs[n] || 'none';
        var index = (this.availableSigns.indexOf(current) + 1) % this.availableSigns.length;
        this.signs[n] = this.availableSigns[index];
        return this.signs[n];
    };
    
    /**
     * Get the motivation of the form of formula
     * @returns {String}   The motivation as a string.
     */
    ScRow.prototype.getMotivation = function(){
        return this.motivation || 'none';
    };
    
    /**
     * Set the motivation of the form of formula
     * @param {String} mot    The motivation
     */
    ScRow.prototype.setMotivation = function(mot){
        return this.motivation = mot || 'none';
    };
    
    /**
     * Get the motivation symbol of the form of formula
     * @returns {String}   The motivation as a string.
     */
    ScRow.prototype.getMotivationSymbol = function(){
        return this.mot[this.motivation || 'none'].symbol;
    };
    
    /**
     * Get the data of this row
     * @returns {Object} The whole row as an object
     */
    ScRow.prototype.getData = function(){
        return jQuery.extend({}, {
            func: this.func,
            motivation: this.motivation,
            signs: this.signs
        });
    };
    
    /**
     * Get the next value for motivation
     * @returns {String}   Next motivation in this.motList;
     */
    ScRow.prototype.nextMot = function(){
        var newindex = ((this.motList.indexOf(this.motivation) + 1) % this.motList.length);
        this.motivation = this.motList[newindex];
        return this.motivation;
    };
    
    /**
     * Get motivation panel
     * @returns {String}   The html for selecting motivation
     */
    ScRow.prototype.getMotivationPanel = function() {
        var html = ['<div class="signchart-motivationpanel">'];
        for (let i = 0, len = this.motList.length; i < len; i++) {
            html.push(`<div class="signchart-motivationitem ${(this.motList[i] === this.motivation ? 'signchart-motivationitem-current' : '')}" data-motivationid="${this.motList[i]}">
                ${this.mot[this.motList[i]].symbol}
                </div>`)
        }
        html.push('</div>');
        return html.join('\n');
    };
    
    /**
     * Get row as html
     * @param {Boolean[]} isroots   An array of Boolean values for "is this root for this row".
     * @param {Number} index    The index of this row.
     * @returns {String}    Html for this row.
     **/
    ScRow.prototype.getHtml = function(isroots, index) {
        index = index || 0;
        var html = [], sign, rootnum = 0, islast;
        html.push(`
            <div class="signchart-formula" data-rowid="${index}"><span class="signchart-math">${escapeHTML(this.func)}</span></div>
            <div class="signchart-motivationslot" data-rowid="${index}"><span class="signchart-motivation">${this.getMotivationSymbol()}</span></div>`);
        for (let i = 0, len = isroots.length + 1; i < len; i++) {
            sign = this.getSign(rootnum);
            islast = i + 1 === len;
            html.push(`<div class="signchart-signslot ${isroots[i] ? 'signchart-nextisroot' : ''} ${islast ? 'signchart-lastsign' : ''}" data-rowid="${index}" data-signid="${rootnum}" data-rootid="${i}" data-sign="${escapeHTML(sign)}">${islast ? `<div class="signchart-removerow">${Signchart.icons.remove}</div>` : `<div class="signchart-rowroot-addremove" data-rootid="${i}"></div>`}</div>`)
            if (isroots[i]) {
                rootnum++;
            };
        };
        return html.join('\n');
    };
    
    /**
     * List of available motivation types
     */
    ScRow.prototype.motList = [
        'none',
        'linear-asc',
        'linear-desc',
        'parab-up-0',
        'parab-up-1',
        'parab-up-2',
        'parab-down-0',
        'parab-down-1',
        'parab-down-2',
        'third-deg-asc-monot',
        'third-deg-desc-monot',
        'third-deg-asc',
        'third-deg-desc',
        'log-asc',
        'log-desc'
    ];
    
    /**
     * Properties of motivations (symbol/icon)
     */
    ScRow.prototype.mot = {
        "none": {
            "id": "none",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-empty"></svg>'
        },
        "linear-asc": {
            "id": "linear-asc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-linear-up"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 15 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 30 l24 -30"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M23 9 h6 m-3 -3 v6"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M1 21 h6"></path></svg>'
        },
        "linear-desc": {
            "id": "linear-desc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-linear-down"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 15 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 0 l24 30"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M1 9 h6 m-3 -3 v6"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M23 21 h6"></path></svg>'
        },
        "parab-up-0": {
            "id": "parab-up-0",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 25 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 5 c5 20 19 20 24 0"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M12 11 h6 m-3 -3 v6"></path></svg>'
        },
        "parab-up-1": {
            "id": "parab-up-1",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 25 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 10 c6 20 18 20 24 0"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M0 20 h6 m-3 -3 v6 m21 -3 h6 m-3 -3 v6"></path></svg>'
        },
        "parab-up-2": {
            "id": "parab-up-2",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 15 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 0 c10 35 14 35 24 0"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M0 10 h6 m-3 -3 v6 m21 -3 h6 m-3 -3 v6"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M12 20 h6"></path></svg>'
        },
        "parab-down-0": {
            "id": "parab-down-0",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 10 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 30 c5 -20 19 -20 24 0"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M12 24 h6 "></path></svg>'
        },
        "parab-down-1": {
            "id": "parab-down-1",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 10 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 25 c6 -20 18 -20 24 0"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M0 15 h6 m18 0 h6"></path></svg>'
        },
        "parab-down-2": {
            "id": "parab-down-2",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-parab-up-0"><path style="stroke: #777; fill: none; stroke-width: 1.5" d="M0 15 h30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 30 c10 -35 14 -35 24 0"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M0 10 h6 m18 0 h6"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M12 20 h6 m-3 -3 v6"></path></svg>'
        },
        "third-deg-asc-monot": {
            "id": "third-deg-asc-monot",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-third-deg-asc-monot"><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 30 c5 -22 19 -8 24 -30"></path></svg>'
        },
        "third-deg-desc-monot": {
            "id": "third-deg-desc-monot",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-third-deg-desc-monot"><path style="stroke: black; fill: none; stroke-width: 1.5" d="M3 0 c5 22 19 8 24 30"></path></svg>'
        },
        "third-deg-asc": {
            "id": "third-deg-asc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-third-deg-asc"><path style="stroke: black; fill: none; stroke-width: 1.5" d="M1 30 c5 -25 10 -25 14 -15 c4 10 9 10 14 -15"></path></svg>'
        },
        "third-deg-desc": {
            "id": "third-deg-desc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-third-deg-desc"><path style="stroke: black; fill: none; stroke-width: 1.5" d="M1 0 c5 25 10 25 14 15 c4 -10 9 -10 14 15"></path></svg>'
        },
        "log-asc": {
            "id": "log-asc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-log-asc"><path style="stroke: #777; fill: none; stroke-width: 1" d="M0 15 h30 M5 0 v30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M7 30 c0 -15 10 -23 23 -25"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M22 11 h6 m-3 -3 v6"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M1 18 h6"></path></svg>'
        },
        "log-desc": {
            "id": "log-desc",
            "symbol": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-graph-log-desc"><path style="stroke: #777; fill: none; stroke-width: 1" d="M0 15 h30 M5 0 v30"></path><path style="stroke: black; fill: none; stroke-width: 1.5" d="M7 0 c0 15 10 23 23 25"></path><path style="stroke: red; fill none; stroke-width: 1.5" d="M1 11 h6 m-3 -3 v6"></path><path style="stroke: blue; fill none; stroke-width: 1.5" d="M21 19 h6"></path></svg>'
        }
    };
    
    /**
     * Available signs for the sign chart
     */
    ScRow.prototype.availableSigns = ["none", "plus", "minus"];
    

    

/*********************************************************************************/    




    /**********************************************
     * Total row
     * @class ScTotal
     * @constructor
     * @param {Object} options   Data for total row
     */
    var ScTotal = function(options) {
        options = $.extend(true, {
            func: '',
            signs: [],
            undefinedpoints: []
        }, options);
        this.func = options.func;
        this.signs = options.signs;
        this.undefinedpoints = options.undefinedpoints;
    };

    /**
     * Total row is like a row, but special. So inherit.
     */
    ScTotal.prototype = Object.create(ScRow.prototype);
    ScTotal.prototype.constructor = ScTotal;
    ScTotal.prototype.parentClass = ScRow.prototype;
    
    /**
     * Get the list of undefined points
     */
    ScTotal.prototype.getUndefinedpoints = function() {
        for (let i = 0, len = this.undefinedpoints.length; i < len; i++) {
            this.undefinedpoints[i] = this.undefinedpoints[i] || false;
        };
        return this.undefinedpoints.slice();
    };
    
    /**
     * Set the state of root point to undefined or not.
     * @param {Number} index    The indes of the root.
     * @param {Boolean} state   The state of given root (true == undefined, false == defined)
     */
    ScTotal.prototype.setUndefinedpoint = function(index, state) {
        if (index >= 0) {
            this.undefinedpoints[index] = !!state;
        };
    };
    
    /**
     * Toggle undefined state of a root point
     * @param {Number} index    The index of the root
     * @returns {Boolean}       The current state of the root
     */
    ScTotal.prototype.toggleUndefinedPoint = function(index) {
        if (index >= 0) {
            this.undefinedpoints[index] = !this.undefinedpoints[index];
        };
        return !!this.undefinedpoints[index];
    };
    
    /**
     * Check, if root with index is undefined point
     * @param {Number} index    The index of the root
     * @returns {Boolena}       True, if the point is undefined
     */
    ScTotal.prototype.isUndefined = function(index) {
        return !!this.undefinedpoints[index];
    };
    
    /**
     * Get the data
     * @returns {Object}  The data of the total row
     */
    ScTotal.prototype.getData = function() {
        return jQuery.extend({}, {
            func: this.func,
            signs: this.signs,
            undefinedpoints: this.getUndefinedpoints()
        });
    };

    /**
     * Get total row as html
     * @param {Number} roots    The number of roots in total.
     * @returns {String}    Html for this row.
     **/
    ScTotal.prototype.getHtml = function(roots) {
        var html = [], sign;
        html.push(`
            <div class="signchart-totalformula signchart-totalrow"><span class="signchart-math">${escapeHTML(this.func)}</span></div>
            `);
        for (let i = 0, len = roots + 1; i < len; i++) {
            sign = this.getSign(i);
            html.push(`<div class="signchart-signslot signchart-totalrow signchart-totalsign ${i + 1 === len ? 'signchart-lastsign' : ''}" data-signid="${i}" data-rootid="${i}" data-sign="${escapeHTML(sign)}"><div class="signchart-totalundefined ${this.isUndefined(i) ? 'isundefined' : ''}"></div></div>`)
        };
        return html.join('\n');
    };





/*********************************************************************************/    



    /**
     * Summary row for the sign chart
     * @class ScSummary
     * @constructor
     * @param {Object} options   Data for summary
     */
    var ScSummary = function(options) {
        options = $.extend(true, {
            type: 'intervals',
            relation: 'less',
            func: '',
            intervals: [],
            rootpoints: [],
            directions: []
        }, options);
        this.type = options.type;
        this.relation = options.relation;
        this.func = options.func;
        this.intervals = options.intervals;
        this.rootpoints = options.rootpoints;
        this.directions = options.directions;
    };

    /**
     * Get the data of summary row
     * @returns {Object}   The data of summary row
     */
    ScSummary.prototype.getData = function() {
        var result = {
            type: this.type
        };
        if (this.type === 'intervals') {
            result.relation = this.getRelation();
            result.intervals = this.getIntervals();
            result.rootpoints = this.getRootpoints();
        } else if (this.type === 'directions') {
            result.func = this.getFunc();
            result.directions = this.getDirections();
        };
        return result;
    };
    
    /**
     * Set the type of the summary
     * @param {String}   stype   The type of the summary
     */
    ScSummary.prototype.setType = function(stype) {
        if (this.availTypes.includes(stype)) {
            this.type = stype;
        };
    };
    
    /**
     * Get the type of the summary
     * @returns {String}   The type of the summary
     */
    ScSummary.prototype.getType = function() {
        return this.type;
    };
    
    /**
     * Toggle the type of the summary between "intervals" and "directions"
     */
    ScSummary.prototype.toggleType = function() {
        switch (this.type) {
            case 'intervals':
                this.type = 'directions';
                break;
            case 'directions':
                this.type = 'intervals';
                break;
            default:
                break;
        };
    };
    
    /**
     * Set the given interval selected or not
     * @param {Number} index   Index of the interval
     * @param {Boolean} selected   True, if selected, else false.
     */
    ScSummary.prototype.setInterval = function(index, selected) {
        this.intervals[index] = selected;
    };
    
    /**
     * Toggle interval state
     * @param {Number} index   Index of the interval
     * @return {String}        The state of the interval
     */
    ScSummary.prototype.toggleInterval = function(index) {
        if (typeof(index) === 'number' && index >= 0) {
            this.intervals[index] = !this.intervals[index];
        };
        return this.intervals[index];
    }
    
    /**
     * Get the state of the given interval
     * @param {Number} index   The index of the interval
     * @returns {Boolean}      The state of the interal in given index
     */
    ScSummary.prototype.getIntervalAt = function(index) {
        return !!this.intervals[index];
    };
    
    /**
     * Get the state of all intervals (with given optional length)
     * @param {Number} length   Optional number of intervals
     * @returns {Boolean[]}     An array of states of intervals
     */
    ScSummary.prototype.getIntervals = function(length) {
        if (typeof(length) !== 'number') {
            length = this.intervals.length;
        };
        for (let i = 0, len = length; i < len; i++) {
            // Make sure, each index has true/false (not undefined)
            this.intervals[i] = !!this.intervals[i];
        };
        return this.intervals.slice();
    };
    
    /**
     * Set the state of rootpoint ("open", "closed", false)
     * @param {Number} index   The index of rootpoint
     * @param {String | Boolean}   state  The state of the rootpoint
     */
    ScSummary.prototype.setRootpoint = function(index, state) {
        if (typeof(index) === 'number' && index >= 0
            && this.availRootStates.includes(state)) {
            this.rootpoints[index] = state;
        };
    };
    
    /**
     * Set the state of rootpoint to the next one in rotation
     * @param {Number} index    The index of the rootpoint
     * @return {String|Boolean} The new state of the rootpoint
     */
    ScSummary.prototype.nextRootpoint = function(index) {
        if (typeof(index) === 'number' && index >= 0) {
            var current = this.rootpoints[index] || false;
            var newindex = (this.availRootStates.indexOf(current) + 1) % this.availRootStates.length;
            this.rootpoints[index] = this.availRootStates[newindex];
        };
        return this.rootpoints[index] || false;
    };
    
    /**
     * Get the state of rootpoint in index.
     * @param {Number} index   The index of rootpoint
     * @returns {String|Boolean}  The state of the rootpoint
     */
    ScSummary.prototype.getRootpointAt = function(index) {
        return this.rootpoints[index] || false;
    };
    
    /**
     * Get states of all rootpoints (with given optional length)
     * @param {Number} length   Optional number of rootpoints
     * @returns {Array}   An array of strings/booleans
     */
    ScSummary.prototype.getRootpoints = function(length) {
        if (typeof(length) !== 'number') {
            length = this.rootpoints.length;
        };
        for (let i = 0, len = length; i < len; i++) {
            // Make sure, each index has value of 'open', 'closed' or false
            this.rootpoints[i] = this.rootpoints[i] || false;
        };
        return this.rootpoints.slice();
    };
    
    /**
     * Set direction in given index
     * @param {Number} index   The index of place
     * @param {String} direction   The direction ('none', 'asc', 'desc', 'horiz')
     */
    ScSummary.prototype.setDirection = function(index, direction) {
        if (typeof(index) === 'number' && index >= 0
            && this.dirSymbol.hasOwnProperty(direction)) {
            this.directions[index] = direction;
        };
    };
    
    /**
     * Get the direction in given index
     * @param {Number} index   The index of place
     * @return {String}   The direction
     */
    ScSummary.prototype.getDirectionAt = function(index) {
        return this.directions[index] || 'none';
    };
    
    /**
     * Get all directions
     * @param {Number} length    Optional number of directions
     * @returns {String[]}   An array of directions
     */
    ScSummary.prototype.getDirections = function(length) {
        if (typeof(length) !== 'number') {
            length = this.directions.length;
        };
        for (let i = 0, len = length; i < len; i++) {
            // Make sure, each direction has value of 'none', 'asc', 'desc', 'horiz'
            this.directions[i] = this.directions[i] || 'none';
        };
        return this.directions.slice();
    };
    
    /**
     * Get the next direction
     * @param {Number} index   The index of place
     * @returns {String}       The next direction in ratating order
     */
    ScSummary.prototype.getNextDir = function(index) {
        var pos = (this.availDirections.indexOf(this.getDirectionAt(index)) + 1) % this.availDirections.length;
        return this.availDirections[pos];
    };
    
    /**
     * Set the next direction in rotating order
     * @param {Number} index   The index of place
     * @return {Object}        An object with the next direction and the next direction symbol
     */
    ScSummary.prototype.setNextDir = function(index) {
        var newDir = this.getNextDir(index);
        this.setDirection(index, newDir);
        return {name: newDir, symbol: this.dirSymbol[newDir]};
    };
    
    /**
     * Set the relation
     * @param {String} rel   The relation
     */
    ScSummary.prototype.setRelation = function(rel) {
        if (this.availRelations.includes(rel)) {
            this.relation = rel;
        };
    };
    
    /**
     * Set the next relation with rotation
     */
    ScSummary.prototype.setNextRelation = function() {
        this.setRelation(this.getNextRelation());
    };
    
    /**
     * Get the relation
     * @returns {String}   The name of the relation
     */
    ScSummary.prototype.getRelation = function() {
        return this.relation || 'less';
    };
    
    /**
     * Get the next relation
     * @returns {String}   The name of the next relation
     */
    ScSummary.prototype.getNextRelation = function() {
        var index = (this.availRelations.indexOf(this.getRelation()) + 1) % this.availRelations.length;
        console.log(index);
        return this.availRelations[index];
    };
    
    /**
     * Get the relation symbol
     * @returns {String}   The LaTeX code for the symbol of relation.
     */
    ScSummary.prototype.getRelationSymbol = function() {
        return this.relationSymbol[this.getRelation()];
    };
    
    /**
     * Set the function LaTeX-formula
     * @param {String} func    LaTeX-formula
     * @returns {Boolean}     True, if the new string was different tha current
     */
    ScSummary.prototype.setFunc = function(func) {
        var changed = this.func !== func;
        this.func = func;
        return changed;
    }
    
    /**
     * Get the function LaTeX-string
     * @returns {String} LaTeX-string
     */
    ScSummary.prototype.getFunc = function(){
        return this.func || '';
    };
    
    /**
     * Get the content for inequation formula
     * @param {String} formula   LaTeX code
     * @param {Boolean|String} editable   Whether the fields are editable
     * @returns {String}   Html content for ".signchart-summary-formula"
     */
    ScSummary.prototype.getHtmlInequation = function(formula, editable) {
        var result = '';
        if (editable) {
            result = `<span class="signchart-math-noneditable">${escapeHTML(formula) + '</span><span class="signchart-math-noneditable signchart-relation" data-relation="${escapeHTML(this.getRelation())}">' + this.getRelationSymbol() + ' 0'}</span>`;
        } else {
            result = `<span class="signchart-math">${escapeHTML(formula) + ' ' + this.getRelationSymbol() + ' 0'}</span>`;
        };
        return result;
    };
    
    /**
     * Get the content for formula
     * @param {String} formula   LaTeX code
     * @returns {String}   Html content for ".signchart-summary-formula"
     */
    ScSummary.prototype.getHtmlFormula = function(formula) {
        return `<span class="signchart-math">${escapeHTML(formula)}</span>`;
    };
    
    /**
     * Get the html for viewing summary row
     * @param {Number} roots    The number of roots in the summary.
     * @param {String} formula  The LaTeX code for the formula.
     * @param {Boolean|String} editable   Whether the fields are editable
     * @returns {String}   Html-code
     */
    ScSummary.prototype.getHtml = function(roots, formula, editable) {
        var html = '';
        switch (this.type) {
            case 'intervals':
                html = this.getHtmlIntervals(roots, formula, editable);
                break;
            case 'directions':
                html = this.getHtmlDirections(roots, formula);
                break;
            default:
                break;
        };
        return html;
    };
    
    /**
     * Get html for intervals type
     * @param {Number} roots   The number of roots in the summary.
     * @param {String} formula  The LaTeX code for the formula.
     * @param {Boolean|String} editable   Whether the fields are editable
     * @returns {String}    Html-code
     */
    ScSummary.prototype.getHtmlIntervals = function(roots, formula, editable) {
        var html = [], inttype, roottype;
        html.push(`
            <div class="signchart-summary-formula signchart-summaryrow">
                ${this.getHtmlInequation(formula, editable)}
            </div>
            <div class="signchart-summary-type signchart-summaryrow">${this.icons.switch}</div>
        `);
        for (let i = 0, len = roots + 1; i < len; i++) {
            inttype = this.getIntervalAt(i);
            roottype = this.getRootpointAt(i) || '';
            html.push(`
                <div class="signchart-summary-intervalslot signchart-summaryrow" data-intervalindex="${i}">
                    <div class="signchart-interval" data-intervalindex="${i}" data-intervaltype="${inttype}"></div>
                    <div class="signchart-intervalstop" data-intervalroot="${i}" data-intervalroot-type="${roottype}"></div>
                </div>
            `);
        };
        return html.join('\n');
    }
    
    /**
     * Get html for directions type
     * @param {Number} roots   The number of roots in the summary.
     * @param {String} formula  The LaTeX code for the formula.
     * @returns {String}    Html-code
     */
    ScSummary.prototype.getHtmlDirections = function(roots, formula) {
        formula = this.func || formula;
        var html = [], direction, dirSymbol;
        html.push(`
            <div class="signchart-summary-directionformula signchart-summaryrow">
                ${this.getHtmlFormula(formula)}
            </div>
            <div class="signchart-summary-type signchart-summary-directiontype signchart-summaryrow">${this.icons.switch}</div>
        `);
        for (let i = 0, len = roots + 1; i < len; i++) {
            direction = this.getDirectionAt(i);
            dirSymbol = this.dirSymbol[direction];
            html.push(`
                <div class="signchart-summary-directionslot signchart-summaryrow" data-direction="${direction}" data-rootid="${i}">
                    <div class="signchart-directionsymbol">${dirSymbol}</div>
                </div>
            `);
        };
        return html.join('\n');
    }
    
    /**
     * Available types
     */
    ScSummary.prototype.availTypes = ['intervals', 'directions'];
    
    /**
     * Available relations
     */
    ScSummary.prototype.availRelations = ['less', 'greater', 'lessequal', 'greaterequal'];
    
    /**
     * Relation symbols
     */
    ScSummary.prototype.relationSymbol = {
        less: '\\lt',
        greater: '\\gt',
        lessequal: '\\leq',
        greaterequal: '\\geq'
    };
    
    /**
     * Available rootpoint states
     */
    ScSummary.prototype.availRootStates = ['open', 'closed', false];
    
    /**
     * Available directions
     */
    ScSummary.prototype.availDirections = ['none', 'asc', 'desc', 'horiz'];
    
    /**
     * Symbols for directions
     */
    ScSummary.prototype.dirSymbol = {
        "none": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="40" height="30" viewBox="0 0 40 30" class="mini-icon mini-icon-arrow-none"></svg>',
        "asc": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="40" height="30" viewBox="0 0 40 30" class="mini-icon mini-icon-arrow-ne"><path style="stroke: none; fill: black;" d="M39 1 l-6 10 l0 -4 l-28 21 l-1.5 -2 l28 -21 l-3 -1z"></path></svg>',
        "desc": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="40" height="30" viewBox="0 0 40 30" class="mini-icon mini-icon-arrow-se"><path style="stroke: none; fill: black;" d="M39 29 l-6 -10 l0 4 l-28 -21 l-1.5 2 l28 21 l-3 1z"></path></svg>',
        "horiz": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="40" height="30" viewBox="0 0 40 30" class="mini-icon mini-icon-arrow-e"><path style="stroke: none; fill: black;" d="M39 15 l-8 3 l1 -2 h-31 v-2 h31 l-1 -2z"></path></svg>'
    }
    
    /**
     * Icons
     */
    ScSummary.prototype.icons = {
        'switch': '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-switch"><circle class="mini-icon-background" stroke="none" fill="none" cx="15" cy="15" r="15" /><path class="mini-icon-foreground" style="stroke: none;" d="M20 7 l1.4 -4 l4 8 l-8 4 l1.4 -4 c-5 -1 -10 -2 -15 3 c2 -6 6 -10 16.2 -7z m-10 16 l-1.4 4 l-4 -8 l8 -4 l-1.4 4 c5 1 10 2 15 -3 c-2 6 -6 10 -16.2 7z"></path></svg>'
    }

    
    
    
    
    
/*********************************************************************************/    


    /******
     * Info about element for the elementset/elementpanel (icon, description, etc.)
     ******/
    Signchart.elementinfo = {
        type: 'signchartelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'signchart',
        name: 'Signchart',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-signchart"><path style="stroke: none;" d="M2 14 h26 v2 h-26z m12 13 v-24 h2 v24z M5 8 h6 v2 h-6 z m16 0 h2 v-2 h2 v2 h2 v2 h-2 v2 h-2 v-2 h-2z M3 18 l8 8 l1.5 -1.5 v4 h-4 l1.5 -1.5 l-8 -8z m15 9 l8 -8 l-1.5 -1.5 h4 v4 l-1.5 -1.5 l-8 8z" /></svg>',
        description: {
            en: 'Sign chart element',
            fi: 'Merkkikaavio ja kulkukaavio',
            sv: 'Teckenschema'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['math']
    }

    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Signchart.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', Signchart.elementinfo);
    }

    
})(jQuery)

