/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 * - MarkdownElement
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
        typeof(jQuery.fn.markdownelement) === 'undefined' && jQuery.fn.markdownelement.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

;(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true, {}, Shuffleassignment.defaults.userdata, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime;
        };
    };
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };

    Userdata.prototype.setSelectedOrder = function(list, corrects){
        this.data.order = list;
        this.data.results = corrects.map(function(a){return a|0;});
        var nowtime = (new Date()).getTime();
        if (!this.metadata.created) {
            this.metadata.created = nowtime;
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime;
        this.metadata.modifier = this.settings.username;
    }
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    }
    
    
    Userdata.prototype.getNewId = function(){
        var idparts = ['shuffleassignmentanswer'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    
    /******
     * Shuffleassignment question element
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for element
     ******/
    var Shuffleassignment = function(place, options){
        this.place = $(place);
        this.place.addClass('shuffleassignment');
        this.setStyles();
        this.init(options);
        // this.addSounds();
        this.show();
    };
     /******
     * Init shuffleassignment element
     * @param {Object} options - initing data of the element
     ******/
    Shuffleassignment.prototype.init = function(options){
        options = $.extend(true, {}, Shuffleassignment.defaults, options);
        this.settings = options.settings;
        if (this.settings.mode === 'view') {
            // Use 'appendview' instead of 'view' by default. If the Shuffleassignment is already
            // solved, change to 'view' later.
            this.settings.mode = 'appendview';
        }
        this.solvable = this.settings.users && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.text = options.data.text;
        this.shuffleassignmentData = {
            solid: options.data.solid,
            shuffle: options.data.shuffle
        }
        this.metadata = options.metadata;
        this.assignmentid = this.place.attr('data-element-name');
        var userdata = options.userdata;
        userdata.data.total = options.data.solid.length;
        userdata.settings = {username: options.settings.username, lang: options.settings.lang || ''}
        this.userdata = new Userdata(userdata);
        this.setMode(this.settings.mode, true);
        this.setAttrs();
    };
    
    /******
     * Get the data of shuffleassignment element.
     * @returns {Object} the data of this shuffleassignment element
     ******/
    Shuffleassignment.prototype.getData = function(){
        var data = {
            type: 'shuffleassignment',
            metadata: $.extend(true, {}, this.metadata),
            data: {
                title: '',
                text: this.text,
                assignmentType: '12',//not used anyway?
                solid: this.shuffleassignmentData.solid,
                shuffle: this.shuffleassignmentData.shuffle
            }
        };
        return data;
    }
    
    /******
     * Set attributes for the container
     ******/
    Shuffleassignment.prototype.setAttrs = function(){
        this.place.attr('lang', escapeHTML(this.metadata.lang))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .attr('data-tags', escapeHTML(this.metadata.tags.join(',')))
            .attr('data-quiztotal', escapeHTML(this.shuffleassignmentData.solid.length))
            .addClass('quizquestion');
    };
    
    /******
     * Set modes
     ******/
    Shuffleassignment.prototype.setMode = function(mode, dontredraw){
        // If mode is incorrect, use 'view' as a default mode.
        if (!Shuffleassignment.modes[mode]) {
            mode = 'view';
        }
        this.place.attr('data-elementmode', mode);
        var modedata = Shuffleassignment.modes[mode];
        this.editable = modedata.editable;
        this.authorable = modedata.authorable;
        this.reviewable = modedata.reviewable;
        this.appendable = modedata.appendable;
        this.settings.mode = mode;
        if (!dontredraw) {
            this.show();
        }
        this.addHandlers();
    }
    
    /******
     * Show shuffleassignment
     ******/
    Shuffleassignment.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    }
    
    /*****
     * Show the shuffleassignment in view mode
     ******/
    Shuffleassignment.prototype.view = function(){
        var name = this.place.attr('data-element-name');
        var html = Shuffleassignment.templates.html;
        this.place.html(html);
        var text = this.text;
        var qarea = this.place.children('.shuffleassignment-questionarea');
        var solidarea = this.place.find('> .shuffleassignment-answerarea ul.shuffleassignment-solid');
        var shufflearea = this.place.find('> .shuffleassignment-answerarea ul.shuffleassignment-shuffle');
        qarea.html('<div class="shuffleassignment-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'view',
                role: this.settings.role
            }
        }
        if(typeof($.fn.markdownelement) ==="function"){
            qarea.children('.shuffleassignment-markdown').markdownelement(mddata);
        }else{
            qarea.children('.shuffleassignment-markdown').text(this.mathfy(mddata.data.text));
        }
        var solidelements = [];
        var shuffleelements = [];
        
        for (var i = 0; i < this.shuffleassignmentData.solid.length; i++) {
            solidelements.push('<li class="shuffleassignment-solidelement"><span class="shuffleassignment-solidtext">'+this.mathfy(escapeHTML(this.shuffleassignmentData.solid[i]))+'</span></li>');
            shuffleelements.push('<li class="shuffleassignement-shuffleelement" data-shuffleorder="order_'+(i)+'"><span class="shuffleassignment-shuffletext">'+this.mathfy(escapeHTML(this.shuffleassignmentData.shuffle[i]))+'</span></li>')
        };
        for (var i = 0; i < 10; i++){
            shuffleelements.sort(function(a,b){return (Math.random() - 0.5);});
        };
        
        solidarea.append(solidelements.join('\n'));
        shufflearea.append(shuffleelements.join('\n'));
        if(this.isCorrect(shufflearea)){
            //if still in correct order swap first element to last
            shufflearea.append(shufflearea.find('li').eq(0));
        }
        if(typeof($.fn.mathquill) ==="function"){
            this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        }
        shufflearea.sortable({placeholder:"shuffle_placeholder"}).disableSelection();
        var max_height = 0;
        var shuffle_elements = shufflearea.find('li');
        for(var i=0;i<shuffle_elements.length;i++){
                max_height = Math.max(shuffle_elements.eq(i).height(),max_height);
        }
        var solid_elements = solidarea.find('li');
        for(var i=0;i<solid_elements.length;i++){
                max_height = Math.max(solid_elements.eq(i).height(),max_height);
        }
        this.place.find('li.shuffleassignment-solidelement,li.shuffleassignement-shuffleelement').height(max_height);
        if (!this.userdata.isClosed()) {
            this.place.trigger('get_userdata', {refid: name});
            this.place.trigger('getcontentbyref', {refs: [name]})
        } else {
            this.appendUserData();
        }
    }

    Shuffleassignment.prototype.appendUserData = function(){
        var shufflelist = this.place.find('.shuffleassignment-answerarea ul.shuffleassignment-shuffle');
        //Now just set data
        //TODO: make 
        var solved = this.userdata.isSolved();
        var closed = this.userdata.isClosed();
        
        if(this.userdata && this.userdata.data && this.userdata.data.order && this.userdata.data.order.length){
            this.setOrder(this.userdata.data.order);
        }
        if (closed){
            shufflelist.sortable("disable");
            var correctness = this.getCorrectness(shufflelist);
            var lielements = shufflelist.children('li');
            lielements.removeClass('shuffleassignment-iscorrect shuffleassignment-iswrong');
            for (var i = 0, len = lielements.length; i < len; i++) {
                lielements.eq(i).addClass(correctness[i] ? 'shuffleassignment-iscorrect' : 'shuffleassignment-iswrong');
            };
        };
        /*
        if ((this.settings.feedback && solved && this.appendable) || (closed && this.appendable)) {
            this.setMode('view');
        } else {
            
        };
        */
    };
    
    Shuffleassignment.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    Shuffleassignment.prototype.addHandlers = function(){
        this.removeHandlers();
        this.addHandlersCommon();
        if (this.editable) {
            this.addHandlersEdit();
        } else {
            this.addHandlersView();
        }
        if (this.appendable) {
            this.addHandlersAppendable();
        };
        
    };
    
    Shuffleassignment.prototype.addHandlersCommon = function(){
        var shuffle = this;
        this.place.on('setmode', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            shuffle.setMode(data);
        });
        this.place.on('reply_get_userdata', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            data.settings = {
                username: shuffle.settings.username,
                lang: shuffle.settings.lang || '',
                uilang: shuffle.settings.uilang,
                role: shuffle.settings.role
            }
            shuffle.userdata = new Userdata(data);
            shuffle.appendUserData();
        });
        this.place.on('reply_getcontentbyref', function(e, data){
            var refid = shuffle.assignmentid;
            var reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: shuffle.settings.username,
                    lang: shuffle.settings.lang || '',
                    uilang: shuffle.settings.uilang,
                    role: shuffle.settings.role
                };
                shuffle.userdata = new Userdata(userdata);
                shuffle.appendUserData();
            };
        });
        this.place.on('showcorrectanswer', function(event, data){
            event.stopPropagation();
            shuffle.showCorrect();
        });
        this.place.on('clearanswer', function(event, data){
            event.stopPropagation();
            shuffle.clearAnswer();
        });
        this.place.on('getdata', function(e){
            var data = shuffle.getData();
            shuffle.place.data('[[elementdata]]', data);
        });
    };
    
    Shuffleassignment.prototype.addHandlersEdit = function(){
        var shuffle = this;
        this.place.on('change', '.shuffleassignment-markdown textarea', function(e, data){
            shuffle.text = jQuery(this).val();
            shuffle.changed();
        });
        this.place.on('change', 'input', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var value = input.val();
            var li = input.parents('li').eq(0);
            var ul = input.parents('ul').eq(0);
            var index = ul.find('li').index(li);
            var type = (ul.hasClass('shuffleassignment-solid')?'solid':'shuffle');
            shuffle.shuffleassignmentData[type][index]=value;
            shuffle.changed();
        });
        this.place.on('click','span.shuffleassignment-trash', function(e, data){
            var li = jQuery(this).parent();
            var ul = li.parent();
            var lilist = ul.find('li');
            var index = lilist.index(li);
            if(lilist.length <=2){
                return alert("2 is min");
            }
            shuffle.shuffleassignmentData['solid'].splice(index,1);
            shuffle.shuffleassignmentData['shuffle'].splice(index,1);
            ul.parent().find('.shuffleassignment-solid li').eq(index).remove();
            lilist.eq(index).remove();
        });
        this.place.on('click','button.shuffleassignment-addrow', function(e, data){
            shuffle.place.find('ul.shuffleassignment-solid').append('<li class="shuffleassignment-solidelement"><span class="shuffleassignment-solidtext"><input type="text" value="" /></span></li>');
            shuffle.place.find('ul.shuffleassignment-shuffle').append('<li class="shuffleassignement-shuffleelement" ><span class="shuffleassignment-shuffletext"><input type="text" class="shuffleassignment-choicetext"  value="" /></span>'+'<span class="shuffleassignment-trash">'+Shuffleassignment.icons.trash+'</span>'+'</li>');
            shuffle.shuffleassignmentData['solid'].push("");
            shuffle.shuffleassignmentData['shuffle'].push("");
        });
        /*TODO: 'enter' moves to next input (what is next input?)
        this.place.on('keyup', 'input.multichoice-choicetext', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var nextli = input.closest('li').next('li');
            var next = nextli.children('input');
            switch (e.which){
                case 13:
                    if (next.length > 0) {
                        next.focus().select();
                    } else {
                        nextli.click();
                    };
                    break;
                default:
                    break;
            };
        });
        */
        this.place.on('element_changed', '.shuffleassignment-markdown', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var mdelem = $(this);
            mdelem.trigger('getdata');
            var mddata = mdelem.data('[[elementdata]]');
            shuffle.text = mddata.data.text;
            shuffle.changed();
        });
        
    };
    
    Shuffleassignment.prototype.addHandlersView = function(){
        
        
    };
    
    Shuffleassignment.prototype.addHandlersAppendable = function(){
        var shuffle = this;
        this.place.on('sortupdate',function(e,ui){
            var sortablelist = shuffle.place.find('ul.shuffleassignment-shuffle');
            var userdata = shuffle.userdata;
            var iscorrect = shuffle.isCorrect(sortablelist);
            userdata.setSelectedOrder(shuffle.getOrder(sortablelist), shuffle.getCorrectness(sortablelist));
            userdata.solved(iscorrect);
            if (shuffle.settings.feedback) {
                if(iscorrect){
                    userdata.solved(iscorrect);
                    shuffle.saveUserData({sendforward: true});
                    shuffle.setMode('view', true); // Change mode to 'view', don't redraw.
                }else{
                    shuffle.saveUserData();
                }
            }else{
                shuffle.saveUserData();
            }
        });
        
        this.place.on('close_quizquestion', function(event, data){
            event.stopPropagation();
            shuffle.close();
            var localsave = !!data.localsave;
            shuffle.saveUserData({localsave: localsave, sendforward: false});
        });
        
    };
    
    /******
     * Show the shuffleassignment in edit mode
     ******/
    Shuffleassignment.prototype.edit = function(){
        // var name = this.place.attr('data-element-name');
        var html = Shuffleassignment.templates.html;
        this.place.html(html);
        var text = this.text;
        var qarea = this.place.children('.shuffleassignment-questionarea');
        var solidarea = this.place.find('> .shuffleassignment-answerarea ul.shuffleassignment-solid');
        var shufflearea = this.place.find('> .shuffleassignment-answerarea ul.shuffleassignment-shuffle');
        qarea.html('<div class="shuffleassignment-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'edit',
                role: this.settings.role
            }
        }
        if(typeof($.fn.markdownelement) ==="function"){
            qarea.children('.shuffleassignment-markdown').markdownelement(mddata);
        }else{
            qarea.children('.shuffleassignment-markdown').html('<textarea>'+mddata.data.text+'</textarea>');
        }
        var solidelements = [];
        var shuffleelements = [];
        
        var trash = '<span class="shuffleassignment-trash">'+Shuffleassignment.icons.trash+'</span>' ;
        for (var i = 0; i < this.shuffleassignmentData.solid.length; i++) {
            solidelements.push('<li class="shuffleassignment-solidelement"><span class="shuffleassignment-solidtext"><input type="text" value="'+escapeHTML(this.shuffleassignmentData.solid[i])+'" /></span></li>');
            shuffleelements.push('<li class="shuffleassignement-shuffleelement" ><span class="shuffleassignment-shuffletext"><input type="text" class="shuffleassignment-choicetext"  value="'+escapeHTML(this.shuffleassignmentData.shuffle[i])+'" /></span>'+trash+'</li>')
        };
        solidarea.append(solidelements.join('\n'));
        shufflearea.append(shuffleelements.join('\n'));
        this.place.find('> .shuffleassignment-answerarea').append('<div><button class="shuffleassignment-addrow">+</button></div>');
    }
    
    /******
     * Trigger changed event
     ******/
    Shuffleassignment.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'shuffleassignment'});
    }
    
    /******
     * Close the quiz question.
     ******/
    Shuffleassignment.prototype.close = function(){
        this.userdata.closed(true);
    };
    
    
    
    Shuffleassignment.prototype.saveUserData = function(options){
        if (this.solvable) {
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            var localsave = options && !!options.localsave;
            if (options && options.sendforward) {
                var senddata = {};
                senddata[name] = userdata;
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    /******
     * Show correct answer
     ******/
    Shuffleassignment.prototype.showCorrect = function(){
        this.setMode('view');
        var shufflelist = this.place.find('ul.shuffleassignment-shuffle');
        var shuffleelements = shufflelist.find('li');
        for (var i=0; i < shuffleelements.length; i++){
            shufflelist.append(shufflelist.find('li[data-shuffleorder="order_'+i+'"]'));
        }
        if (!this.feedback) {
            
        };
    };
    
    /******
     * Clear answer
     ******/
    Shuffleassignment.prototype.clearAnswer = function(){
        var userdata = {settings: {username: this.settings.username, lang: this.settings.lang || ''}};
        this.userdata = new Userdata(userdata);
        this.setMode('appendview');
        this.appendUserData();
    };
    
    /******
     * Set style-tag if needed
     ******/
    Shuffleassignment.prototype.setStyles = function(){
        if ($('head style#shuffleassignmentelement-style').length === 0) {
            $('head').append('<style id="shuffleassignmentelement-style" type="text/css">'+Shuffleassignment.styles+'</style>');
        }
    }
    
    /******
     * Correctness check
     *****/
    Shuffleassignment.prototype.isCorrect = function(listelement){
        var lielements = listelement.find('li');
        for(var i=0;i<lielements.length;i++){
            if(lielements.eq(i).attr('data-shuffleorder')!=="order_"+i) return false;
        }
        return true;
    };
    
    /**
     * Check the correctness
     * @param {jQuery} listelement    A jQuery-object with <ul>-element.
     * @returns {Boolean[]}  An array with correctness of each line
     */
    Shuffleassignment.prototype.getCorrectness = function(listelement){
        var lielements = listelement.find('li');
        var result = [];
        for (var i = 0, len = lielements.length; i < len; i++) {
            result.push(lielements.eq(i).attr('data-shuffleorder') === 'order_' + i);
        };
        return result;
    };
    
    /******
     * Get order
     *****/
    Shuffleassignment.prototype.getOrder = function(listelement){
        var lielements = listelement.find('li');
        var order = [];
        for(var i=0;i<lielements.length;i++){
            order.push(lielements.eq(i).attr('data-shuffleorder'));
        }
        return order;
    };
    /******
     * Set order
     *****/
    Shuffleassignment.prototype.setOrder = function(list){
        try{
            var listelement = this.place.find('ul.shuffleassignment-shuffle');
            for(var i=0;i<list.length;i++){
                listelement.append(listelement.find('li[data-shuffleorder="'+list[i]+'"]'));
            }
        }catch(e){
            console.log('Shuffleassignment.setOrder error:',e);
            return false;
        }

    };
    /******
     * Sounds
     * TODO: add later if needed
     *****/
    Shuffleassignment.prototype.playSounds = function(id){
    };
    Shuffleassignment.prototype.addSounds = function(){
    };
    
    /******
     * Mathquillify the string, i.e., replace \( \)-pairs and $ $-pairs with a span.
     ******/
    Shuffleassignment.prototype.mathfy = function(str){
        str = str.replace(/\$([^$]*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>');
        return str;
    }
    
    Shuffleassignment.defaults = {
        type: 'shuffleassignment',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            lang: '',
            tags: []
        },
        data: {
            title: '',
            text: '',
            solid: ['',''],
            shuffle: ['','']
        },
        userdata: {
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                order: [],
                results: [],
                correct: 0,
                total: 0,
                solved: false,
                closed: false
            },
            settings: {
                username: 'Anonymous',
                uilang: 'en',
                lang: 'en'
            }
        },
        settings: {
            mode: 'view',
            role: 'student',
            uilang: 'en',
            username: 'Anonymous',
            feedback: true
        }
    };
    
    Shuffleassignment.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        appendview: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: true
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            appendable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            appendable: false
        }
    };
    
    Shuffleassignment.templates = {
        html: [
            '<div class="shuffleassignment-questionarea ebookassignment-questionarea"></div>',
            '<div class="shuffleassignment-answerarea ebookassignment-answerarea">',
            '    <ul class="shuffleassignment-solid">',
            '    </ul>',
            '    <ul class="shuffleassignment-shuffle">',
            '    </ul>',
            '</div>'
        ].join('\n')
    };
    
    Shuffleassignment.icons = {
        trash: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>',
        correct: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment-correct"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /><path style="stroke: none; fill: #0a0;" d="M29 1 l-16 24 l-1 0 l-6 -12 a2 2 0 0 1 4 -2 l3 6 l13 -18z" /></svg>',
        wrong: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment-wrong"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /><path style="stroke: none; fill: #a00;" d="M4 2 l25 25 l-4 2 l-23 -24z M2 24 l26 -22 l2 2 l-25 23z" /></svg>',
        empty: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment-empty"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /></svg>',
        selected: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment-selected"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z m0 3 a8 8 0 0 0 0 16 a8 8 0 0 0 0 -16z" /></svg>',
        shuffleassignmentmark: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment-mark"><circle style="stroke: none;" fill="black" cx="15" cy="15" r="13" /><circle class="shuffleassignment-selectfill" style="stroke: none;" fill="white" cx="15" cy="15" r="11" /><circle class="shuffleassignment-selectedmark" style="stroke: none;" fill="black" cx="15" cy="15" r="7" /><path class="shuffleassignment-correctmark" style="stroke: none; fill: #0a0;" d="M29 1 l-16 24 l-1 0 l-6 -12 a2 2 0 0 1 4 -2 l3 6 l13 -18z" /><path class="shuffleassignment-wrongmark" style="stroke: none; fill: #a00;" d="M4 2 l25 25 l-4 2 l-23 -24z M2 24 l26 -22 l2 2 l-25 23z" /></svg>'
    }
    
    Shuffleassignment.styles = [
        //'.shuffleassignmentelement {background-color: white;}',
        '.shuffleassignment-answerarea ul.shuffleassignment-shuffle,',
        '.shuffleassignment-answerarea ul.shuffleassignment-solid {display:inline-block;list-style: none;vertical-align:top;}',
        '[data-elementmode="view"] .shuffleassignment-answerarea li.shuffleassignement-shuffleelement{background-color:#F9F9F9;padding-left:1em;padding-right:1em;border:1px solid #EEE;}',
        '[data-elementmode="view"] .shuffleassignment-answerarea li.shuffleassignment-solidelement{padding-left:1em;padding-right:1em;border:1px solid #EEE;}',
        '[data-elementmode="view"] .shuffleassignment-answerarea ul.shuffleassignment-shuffle{padding:0;}',
        '[data-elementmode="appendview"] .shuffleassignment-answerarea li.shuffleassignement-shuffleelement{background-color:#F9F9F9;padding-left:1em;padding-right:1em;border:1px solid #EEE;}',
        '[data-elementmode="appendview"] .shuffleassignment-answerarea li.shuffleassignment-solidelement{padding-left:1em;padding-right:1em;border:1px solid #EEE;}',
        '[data-elementmode="appendview"] .shuffleassignment-answerarea ul.shuffleassignment-shuffle{padding:0;}',
        '.shuffleassignment-answerarea .shuffle_placeholder{background-color:#EEFFEE;min-height:1em;}',
        '.shuffleassignment-answerarea ul.shuffleassignment-solid li,.shuffleassignment-answerarea ul.shuffleassignment-shuffle li {margin-bottom:4px;}',
        '.questionhighlights .shuffleassignment-iscorrect .shuffleassignment-shuffletext {color: black; font-weight: bold; border-bottom: 2px solid #0a0;}',
        '.questionhighlights .shuffleassignment-iswrong .shuffleassignment-shuffletext {color: #a00; font-weight: bold; text-decoration: line-through;}'
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    Shuffleassignment.localization = {
        "en": {
        },
        "fi": {
        }
    };
    
    Shuffleassignment.sounds ={
        "shuffleassignment_correct" :"",
        "shuffleassignment_wrong" : ""
        };

    Shuffleassignment.elementinfo = {
        type: 'shuffleassignment',
        elementtype: ['assignments','quizelement'],
        jquery: 'shuffleassignment',
        name: 'Shuffleassignment',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment mini-icon-shuffle"><path style="stroke: none;" d="M4 8 h6 l8 14 h5 l-2 -2 l1 -1 l4 4 l-4 4 l-1 -1 l2 -2 h-6 l-8 -14 h-5z m0 14 h5 l8 -14 h6 l-2 -2 l1 -1 l4 4 l-4 4 l-1 -1 l2 -2 h-5 l-8 14 h-6z" /></svg>',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-shuffleassignment"><path style="stroke: none;" d="M 1 4 l8 0 l0 2 l-8 0z m0 10 l8 0 l0 2 l-8 0z m0 10 l8 0 l0 2 l-8 0z m12 -20 l8 0 l0 2 l-8 0z m0 10 l8 0 l0 2 l-8 0z m0 10 l8 0 l0 2 l-8 0z m14 0 a10 16 0 0 0 0 -16 l4 0 l0 -1 l-8 -2 l-1 9 l1 0 l2 -5 a10 20 0 0 1 0 16z " /></svg>',
        description: {
            en: 'Shuffle assignment',
            fi: 'Sekoitustehtävä',
            sv: 'Hopparningsuppgift'
        },
        weight: 25,
        roles: ['teacher', 'author'],
        classes: ['assignments']
    };


    // Register shuffleassignment as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Shuffleassignment.elementinfo);
        $.fn.elementset('addelementtype', Shuffleassignment.elementinfoold);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Shuffleassignment.elementinfo);
    }
    
    if (typeof(ebooklocalizer) !=="undefined") {
        ebooklocalizer.addTranslations(Shuffleassignment.localization);
    };
    
    
        /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var shuffleassignment = new Shuffleassignment(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.shuffleassignment = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in shuffleassignment.');
            return false;
        }
    };
    
})(jQuery);