/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

;(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true, {}, Shortanswer.defaults.userdata, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime;
        };
    };
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };
    
    Userdata.prototype.addAnswer = function(str, correct){
        this.data.answers = [str];
        this.data.results[0] = correct | 0;
        var nowtime = (new Date()).getTime();
        if (!this.metadata.created) {
            this.metadata.created = nowtime;
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime;
        this.metadata.modifier = this.settings.username;
    };
    
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    }
    
    Userdata.prototype.isAnswer = function(str){
        return (this.data.answers.indexOf(str) > -1);
    };
    
    Userdata.prototype.getAnswer = function(){
        var answer = this.data.answers[this.data.answers.length - 1] || '';
        return answer;
    }
    
    Userdata.prototype.getNewId = function(){
        var idparts = ['shortansweranswer'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };

    
    /******
     * Shortanswer question element
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for element
     ******/
    var Shortanswer = function(place, options){
        this.place = $(place);
        this.place.addClass('shortanswerelement');
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    Shortanswer.prototype.init = function(options){
        options = $.extend(true, {}, Shortanswer.defaults, options);
        this.settings = options.settings;
        if (this.settings.mode === 'view') {
            // Use 'appendview' instead of 'view' by default. If the multichoice is already
            // solved, change to 'view' later.
            this.settings.mode = 'appendview'
        };
        this.solvable = this.settings.users && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.data = options.data;
        this.metadata = options.metadata;
        this.assignmentid = this.place.attr('data-element-name');
        var userdata = options.userdata
        userdata.settings = {username: this.settings.username, lang: this.settings.lang || ''};
        this.userdata = new Userdata(userdata);
        this.setMode(this.settings.mode, true);
        this.setAttrs();
    }
    
    Shortanswer.prototype.getData = function(){
        var data = {
            type: 'shortanswer',
            metadata: $.extend(true, {}, this.metadata),
            data: $.extend(true, {}, this.data)
        };
        return data;
    }
    
    Shortanswer.prototype.setAttrs = function(){
        this.place.attr('lang', escapeHTML(this.metadata.lang))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .attr('data-tags', escapeHTML(this.metadata.tags.join(',')))
            .attr('data-quiztotal', '1')
            .addClass('quizquestion');
    };
    
    Shortanswer.prototype.setMode = function(mode, dontredraw){
        // If mode is incorrect, use 'view' as a default mode.
        if (!Shortanswer.modes[mode]) {
            mode = 'view';
        }
        this.place.attr('data-elementmode', mode);
        var modedata = Shortanswer.modes[mode];
        this.editable = modedata.editable;
        this.authorable = modedata.authorable;
        this.reviewable = modedata.reviewable;
        this.appendable = modedata.appendable;
        this.settings.mode = mode;
        if (!dontredraw) {
            this.show();
        }
        this.addHandlers();
    };

    Shortanswer.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    }
    
    Shortanswer.prototype.view = function(){
        var name = this.assignmentid;
        var html = Shortanswer.templates.view;
        this.place.html(html);
        var text = this.data.text;
        var qarea = this.place.children('.shortanswer-questionarea');
        qarea.html('<div class="shortanswer-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'view',
                role: this.settings.role
            }
        }
        qarea.children('.shortanswer-markdown').markdownelement(mddata);
        this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        if (!this.userdata.isClosed()) {
            this.place.trigger('get_userdata', {refid: name});
            this.place.trigger('getcontentbyref', {refs: [name]});
        } else {
            this.appendUserData();
        }
    }
    
    Shortanswer.prototype.edit = function(){
        var html = Shortanswer.templates.edit;
        this.place.html(html);
        var text = this.data.text;
        var qarea = this.place.children('.shortanswer-questionarea');
        var aarea = this.place.find('> .shortanswer-answerarea ul');
        var labelarea = this.place.find('> .shortanswer-answerarea .shortanswer-correctlabel');
        labelarea.html(ebooklocalizer.localize('shortanswer:correct_answers', this.settings.uilang));
        qarea.html('<div class="shortanswer-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'edit',
                role: this.settings.role
            }
        };
        qarea.children('.shortanswer-markdown').markdownelement(mddata);
        var corrects = [], correct;
        var trash = (this.data.corrects.length > 1 ? '<span class="shortanswer-trash">'+Shortanswer.icons.trash+'</span>' : '');
        for (var i = 0, len = this.data.corrects.length; i < len; i++) {
            var correct = this.data.corrects[i];
            corrects.push('<li data-correct-number="'+i+'"><input type="text" class="shortanswer-correcttext" value="'+escapeHTML(correct)+'" data-correct-number="'+i+'" />'+trash+'</li>');
        };
        corrects.push('<li data-correct-number="'+i+'" class="shortanswer-addcorrect"><div>+</div></li>');
        aarea.append(corrects.join('\n'));
    }
    
    Shortanswer.prototype.appendUserData = function(){
        var input = this.place.find('> .shortanswer-answerarea input.shortanswer-answerbox');
        var useranswer = this.place.find('> .shortanswer-answerarea .shortanswer-useranswer');
        var solved = this.userdata.isSolved();
        var closed = this.userdata.isClosed();
        var currentanswer = this.userdata.getAnswer();
        if ((this.settings.feedback && solved && this.appendable) || (closed && this.appendable)) {
            this.setMode('view');
        } else {
            if (solved && this.settings.feedback || closed) {
                if (input.length > 0) {
                    useranswer = $('<span class="shortanswer-useranswer"></span>');
                    input.after(useranswer);
                    input.remove();
                }
                if (solved) {
                    useranswer.addClass('shortanswer-iscorrect').removeClass('shortanswer-iswrong');
                } else {
                    useranswer.addClass('shortanswer-iswrong').removeClass('shortanswer-iscorrect');
                }
                useranswer.html(this.mathfy(currentanswer));
                useranswer.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
            } else {
                input.val(currentanswer);
            };
        };
    };
    
    Shortanswer.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    Shortanswer.prototype.addHandlers = function(){
        this.removeHandlers();
        this.addHandlersCommon();
        if (this.editable) {
            this.addHandlersEdit();
        } else {
            this.addHandlersView();
        }
        if (this.appendable) {
            this.addHandlersAppendable();
        };
    };
    
    Shortanswer.prototype.addHandlersCommon = function(){
        var shortans = this;
        this.place.on('setmode', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            shortans.setMode(data);
        });
        this.place.on('reply_get_userdata', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            data.settings = {
                username: shortans.settings.username,
                lang: shortans.settings.lang || '',
                uilang: shortans.settings.uilang,
                role: shortans.settings.role
            };
            shortans.userdata = new Userdata(data);
            shortans.appendUserData();
        });
        this.place.on('reply_getcontentbyref', function(e, data){
            var refid = shortans.assignmentid;
            var reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: shortans.settings.username,
                    lang: shortans.settings.lang || '',
                    uilang: shortans.settings.uilang,
                    role: shortans.settings.role
                };
                shortans.userdata = new Userdata(userdata);
                shortans.appendUserData();
            };
        });
        this.place.on('showcorrectanswer', function(event, data){
            event.stopPropagation();
            shortans.showCorrect();
        });
        this.place.on('clearanswer', function(event, data){
            event.stopPropagation();
            shortans.clearAnswer();
        });
        this.place.on('getdata', function(e){
            var data = shortans.getData();
            shortans.place.data('[[elementdata]]', data);
        });
    }

    Shortanswer.prototype.addHandlersEdit = function(){
        var shortans = this;
        this.place.on('change', 'input.shortanswer-correcttext', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var value = input.val();
            var index = input.attr('data-correct-number') | 0;
            shortans.data.corrects[index] = value;
            shortans.changed();
        });
        this.place.on('keyup', 'input.shortanswer-correcttext', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var nextli = input.closest('li').next('li');
            var next = nextli.children('input');
            switch (e.which) {
                case 13:
                    if (next.length > 0) {
                        next.focus().select();
                    } else {
                        nextli.click();
                    };
                    break;
                default:
                    break;
            };
        });
        this.place.on('element_changed', '.shortanswer-markdown', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var mdelem = $(this);
            mdelem.trigger('getdata');
            var mddata = mdelem.data('[[elementdata]]');
            shortans.data.text = mddata.data.text;
            shortans.changed();
        });
        this.place.on('click', '.shortanswer-addcorrect', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var button = $(this);
            var index = button.attr('data-correct-number') | 0;
            button.before('<li data-correct-number="'+index+'"><input type="text" class="shortanswer-correcttext" data-correct-number="'+index+'" /><span class="shortanswer-trash">'+Shortanswer.icons.trash+'</span></li>');
            button.attr('data-correct-number', index + 1);
            button.prev().children('input').focus();
        });
        this.place.on('click', '.shortanswer-corrects li .shortanswer-trash', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var button = $(this);
            var liitem = button.closest('li');
            var list = liitem.closest('ul.shortanswer-corrects');
            var input = liitem.children('input');
            var index = (input.attr('data-correct-number') | 0);
            shortans.data.corrects.splice(index, 1);
            liitem.remove();
            var items = list.children('li');
            for (var i = 0, len = items.length; i < len; i++) {
                items.eq(i).attr('data-correct-number', i).children('input').attr('data-correct-number', i);
            };
            shortans.changed();
        });
    };
    
    Shortanswer.prototype.addHandlersView = function(){}
    
    Shortanswer.prototype.addHandlersAppendable = function(){
        var shortans = this;
        this.place.on('change', '.shortanswer-answerarea input', function(e){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var value = input.val();
            var userdata = shortans.userdata;
            var iscorrect = shortans.isCorrect(value);
            if (shortans.settings.feedback) {
                // TODO!
                userdata.addAnswer(value, iscorrect);
                userdata.solved(iscorrect);
                shortans.saveUserData({sendforward: false});
            } else {
                userdata.addAnswer(value, iscorrect);
                userdata.solved(shortans.isCorrect(value));
                shortans.saveUserData();
            };
        });
        this.place.on('close_quizquestion', function(event, data){
            event.stopPropagation();
            shortans.close();
            var localsave = !!data.localsave;
            shortans.saveUserData({localsave: localsave, sendforward: false});
        });
    }
    
    /******
     * Check, if the given answer is correct. Case insensitive.
     ******/
    Shortanswer.prototype.isCorrect = function(answer){
        var lowans = answer.toLowerCase();
        var found = false;
        var corrects = this.data.corrects;
        for (var i = 0, len = corrects.length; i < len && !found; i++) {
            found = (lowans === corrects[i].toLowerCase());
        };
        return found;
    }
    
    Shortanswer.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'shortanswer'});
    }
    
    /******
     * Close the quiz question.
     ******/
    Shortanswer.prototype.close = function(){
        this.userdata.closed(true);
    };
    
    Shortanswer.prototype.saveUserData = function(options){
        if (this.solvable) {
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            var localsave = options && !!options.localsave;
            if (options && options.sendforward) {
                var senddata = {};
                senddata[name] = userdata;
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    /******
     * Show correct answer
     ******/
    Shortanswer.prototype.showCorrect = function(){
        this.setMode('view');
        var input = this.place.find('.shortanswer-answerarea input.shortanswer-answerbox');
        var useranswer = this.place.find('.shortanswer-useranswer');
        if (!this.feedback) {
            if (input.length > 0) {
                useranswer = $('<span class="shortanswer-useranswer"></span>');
                input.after(useranswer);
                input.remove();
            };
            useranswer.html(this.mathfy(escapeHTML(this.data.corrects[0])));
            useranswer.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        };
    };
    
    /******
     * Clear answer
     ******/
    Shortanswer.prototype.clearAnswer = function(){
        var userdata = {settings: {username: this.settings.username, lang: this.settings.lang || ''}};
        this.userdata = new Userdata(userdata);
        this.setMode('appendview');
        this.appendUserData();
    };
    
    
    
    /******
     * Set style-tag if needed
     ******/
    Shortanswer.prototype.setStyles = function(){
        if ($('head style#shortanswerelement-style').length === 0) {
            $('head').append('<style id="shortanswerelement-style" type="text/css">'+Shortanswer.styles+'</style>');
        }
    }

    /******
     * Mathquillify the string, i.e., replace \( \)-pairs and $ $-pairs with a span.
     ******/
    Shortanswer.prototype.mathfy = function(str){
        str = str.replace(/\$([^$]*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>');
        return str;
    }
    
    
    Shortanswer.styles = [
        '.shortanswer-answerarea {margin: 0.5em 2em;}',
        '[data-ebookelement-type="shortanswer"] > .shortanswer-answerarea {margin-left: 0.6em; border-left: 5px solid #ddd; padding-left: 2em;}',
        '.shortanswer-answerarea ul.shortanswer-corrects {list-style: none; margin: 0; padding: 0;}',
        '.shortanswer-answerarea li.shortanswer-addcorrect {font-weight: bold; text-align: center; max-width: 10em; cursor: pointer;}',
        '.shortanswer-answerarea li.shortanswer-addcorrect:hover {background-color: #fafafa;}',
        '.shortanswer-trash:hover svg path {fill: #a00;}',
        '.shortanswer-trash {cursor: pointer;}',
        '.questionhighlights span.shortanswer-useranswer.shortanswer-iscorrect {color: black; font-weight: bold; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; border-bottom: 2px solid #0a0; text-shadow: 1px 1px 1px white, -1px -1px 1px white; min-width: 3em;}',
        '.questionhighlights span.shortanswer-useranswer.shortanswer-iswrong {color: #a00; font-weight: bold; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; text-decoration: line-through;}'
        //'.questionhighlights span.shortanswer-useranswer.shortanswer-iscorrect {color: black; background-color: #afa; border-radius: 0.3em; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; border: 2px solid #0a0; text-shadow: 1px 1px 1px white, -1px -1px 1px white; min-width: 3em;}',
        //'.questionhighlights span.shortanswer-useranswer.shortanswer-iswrong {color: black; background-color: #faa; border-radius: 0.3em; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; border: 2px solid #a00; text-shadow: 1px 1px 2px white, -1px -1px 2px white; min-width: 3em;}'
    ].join('\n');
    
    Shortanswer.defaults = {
        type: 'shortanswer',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            lang: '',
            tags: []
        },
        data: {
            title: '',
            text: '',
            corrects: ['']
        },
        userdata: {
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                answers: [],
                total: 1,
                results: [0],
                solved: false
            },
            settings: {
                username: 'Anonymous123',
                uilang: 'en',
                lang: 'en'
            }
        },
        settings: {
            mode: 'view',
            role: 'student',
            uilang: 'en',
            username: 'Anonymous',
            feedback: true
        }
    };
    
    Shortanswer.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        appendview: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: true
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            appendable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            appendable: false
        }
    };

    Shortanswer.templates = {
        view: [
            '<div class="shortanswer-questionarea ebookassignment-questionarea"></div>',
            '<div class="shortanswer-answerarea ebookassignment-answerarea">',
            '    <input type="text" class="shortanswer-answerbox" />',
            '</div>'
        ].join('\n'),
        edit: [
            '<div class="shortanswer-questionarea ebookassignment-questionarea"></div>',
            '<div class="shortanswer-answerarea ebookassignment-answerarea">',
            '    <div class="shortanswer-correctlabel"></div>',
            '    <ul class="shortanswer-corrects">',
            '    </ul>',
            '</div>'
        ].join('\n')
    };

    Shortanswer.icons = {
        trash: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>'
    };
    
    /******
     * Localization strings
     ******/
    Shortanswer.localization = {
        "en": {
            "shortanswer:correct_answers": "Correct answers"
        },
        "fi": {
            "shortanswer:correct_answers": "Oikeat vastaukset"
        }
    };
    

    Shortanswer.elementinfo = {
        type: 'shortanswer',
        elementtype: ['assignments','quizelement'],
        jquery: 'shortanswer',
        name: 'Short answer',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice"><path style="stroke: none;" d="M4 23 h22 v2 h-22z m3 -1 l2 -6 l12 -12 l4 4 l-12 12z m1 -1 l4 -2 l-2 -2z m3 -5 l2 2 l8 -8 l-2 -2z m9 -9 l2 2 l1 -1 l-2 -2z"></path></svg>',
        description: {
            en: 'Short answer assignment',
            fi: 'Lyhytvastaustehtävä',
            sv: 'Kortsvarsuppgift'
        },
        weight: 30,
        roles: ['teacher', 'author'],
        classes: ['assignments']
    };

    // Register multichoice as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Shortanswer.elementinfo);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Shortanswer.elementinfo);
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Shortanswer.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Shortanswer.localization);
    }


    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var shortans = new Shortanswer(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.shortanswer = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in shortanswer.');
            return false;
        }
    };
    
})(jQuery)