/******
@name Sharer
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Sharer
@description A class and jQuery-plugin for sharing things from the context. (For example a page from a notebook).

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /**
     * Tool for sharing things from the context.
     * @constructor
     * @param {jQuery} place    DOM-place for sharer
     * @param {Object} options  data for sharer
     */
    var Sharer = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initData();
        if (this.settings.users.canEdit({ctype: 'sharecontent'})) {
            this.initHandlers();
            this.setAttrs();
            this.setStyles();
            this.show();
            this.getShareables();
            this.getShareContent();
            //this.registerConfForm();
            this.appReady();
        } else {
            this.close();
        };
    };
    
    Sharer.prototype.init = function(options) {
        this.settings = options.settings;
        this.configs = options.configs;
        this.users = this.settings.users;
    };
    
    Sharer.prototype.initData = function() {
        this.appid = this.settings.appid;
        this.apptype = 'sharer';
        this.shareables = {};
        this.place.attr('data-appname', escapeHTML(this.appid));
    };
    
    Sharer.prototype.initHandlers = function() {
        var sharer = this;
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            sharer.close();
        }).off('closeapp').on('closeapp', function(event, data){
            event.stopPropagation();
            sharer.close();
        }).off('setconfigs').on('setconfigs', function(event, data){
            event.stopPropagation();
            sharer.setConfigs(data);
        }).off('reply_getshareables').on('reply_getshareables', function(event, data){
            event.stopPropagation();
            sharer.updateShareables(data);
        }).off('sharedone').on('sharedone', function(event, data){
            event.stopPropagation();
            sharer.addShared(data);
        }).off('reply_getcontent').on('reply_getcontent', function(event, data){
            event.stopPropagation();
            data = data || {};
            switch (data.contentType) {
                case 'sharecontent':
                    var elemlist = data.refs[sharer.appid] || [];
                    var eldata;
                    for (var i = 0, len = elemlist.length; i < len; i++) {
                        eldata = data.contentdata[elemlist[i]];
                        sharer.addShared(eldata);
                    };
                    break;
                default:
                    break;
            }
        }).off('click', '.sharer-button').on('click', '.sharer-button', function(event, data){
            event.stopPropagation();
            var button = $(this);
            var shevent = button.attr('data-event');
            var appid = button.attr('data-toappid');
            button.trigger('shareevent', {event: shevent, appid: appid});
        });
    };
    
    Sharer.prototype.setAttrs = function() {
        this.place.addClass('sharer');
    };
    
    //Sharer.prototype.setConfigs = function(data) {
    //    var noteview = data.noteview;
    //    this.configs = $.extend(this.configs, data);
    //    if (typeof(noteview) === 'boolean') {
    //        this.setNoteview();
    //    };
    //    this.saveConfigs();
    //};
    
    //Sharer.prototype.saveConfigs = function(data) {
    //    // Save config data in parent module
    //    if (typeof(data) === 'undefined') {
    //        // Not given data, but save all. (Else save only given data)
    //        data = {
    //            type: 'configsave',
    //            appname: this.appid,
    //            appid: this.appid,
    //            apptype: this.apptype,
    //            configs: JSON.parse(JSON.stringify(this.configs))
    //        };
    //    };
    //    this.place.trigger('saveconfigs', data);
    //}
    
    Sharer.prototype.show = function() {
        this.view();
    };
    
    Sharer.prototype.view = function() {
        var uilang = this.settings.uilang;
        var html = this.template.html;
        this.place.html(html);
        //this.drawButtons();
        this.place.find('.sharer-title').text(ebooklocalizer.localize('sharer:sharertool', uilang));
    };
    
    /**
     * Request for elements of contentType: 'sharecontent'
     */
    Sharer.prototype.getShareContent = function() {
        this.place.trigger('getcontent', {anchors: [this.appid], contentType: ['sharecontent']});
    };
    
    /**
     * As the context to give information about things to share
     */
    Sharer.prototype.getShareables = function() {
        this.place.trigger('getshareables');
    };
    
    /**
     * Update the list of shareables
     * @param {Array} shareables  A list of shareable-objects
     */
    Sharer.prototype.updateShareables = function(shareables) {
        var item;
        for (var i = 0, len = shareables.length; i < len; i++) {
            item = shareables[i];
            if (!this.shareables[item.appid]) {
                this.shareables[item.appid] = [];
            };
            this.shareables[item.appid].push(item);
        };
        this.drawButtons();
    }
    
    Sharer.prototype.drawButtons = function() {
        var uilang = this.settings.uilang;
        var buttonset = this.place.find('.sharer-controlbuttons');
        var appshares, buttelem, butt, button, label;
        for (var appid in this.shareables) {
            appshares = this.shareables[appid];
            for (var i = 0, len = appshares.length; i < len; i++) {
                buttelem = $(this.template.setbutton);
                button = appshares[i];
                label = button.label;
                buttelem.attr('data-event', escapeHTML(button.event)).attr('data-button-name', escapeHTML(button.name)).attr('data-toappid', escapeHTML(button.appid)).attr('title', escapeHTML(button.title));
                buttelem.html('<span class="sharer-button-icon">' + (button.icon || '') + '</span><span class="sharer-button-label">' + escapeHTML(label || '') + '</span>');
            };
            buttonset.append(buttelem);
        };
    };
    
    Sharer.prototype.addShared = function(data) {
        var uilang = this.settings.uilang;
        var sharelist = this.place.find('.sharer-sharelist');
        var vars = {
            title: escapeHTML(data.data.title),
            url: escapeHTML(data.data.url),
            date: (new Intl.DateTimeFormat(uilang, {
                weekday: 'short',
                year: 'numeric',
                month: 'numeric',
                day: 'numeric',
                hour: 'numeric',
                minute: 'numeric'
            }).format(new Date(data.metadata.modified)))
        };
        var item = $(this.fillTemplate(this.template.shareitem, vars));
        sharelist.prepend(item);
        this.saveSendContent({
            contentType: 'sharecontent',
            id: data.id || data.name,
            anchorid: [this.appid],
            lang: 'common',
            data: data,
            sendto: [],
            sendfrom: this.settings.username
        });
    };
    
    /**
     * Save / Send content (trigger event).
     * @param {Object} content       data of the content
     * @param {String} content.contentType    Type of the content
     * @param {String} content.id    id of the content
     * @param {String} content.anchorid   id where to anchor the content
     * @param {String} content.lang  The content language
     * @param {Object} content.data  Data of the content
     * @param {Array}  content.sendto    A list of recipients
     * @param {Array}  content.sendfrom  The userid of the sender
     */
    Sharer.prototype.saveSendContent = function(content) {
        var uilang = this.settings.uilang;
        var anchors = [content.anchorid || this.name];
        var sdata = {
            "name": content.id,
            "contentType": content.contentType,
            "lang": content.lang || "common",
            "anchors": anchors,
            "contentdata": content.data,
            "contentinfo": {
                "name": content.name,
                "contentType": content.contentType,
                "subcontext": this.name,
                "lang": content.lang || "common",
                "sendto": content.sendto || [],
                "sendfrom": content.sendfrom || this.settings.username,
                "timestamp": (new Date()).getTime(),
                "read": false,
                "recipientopened": false,
                "isoutbox": true
            }
        };
        this.place.trigger('setcontent', [sdata, true]);
        this.place.trigger('savecontent');
    }    
    
    /**
     * Fill data in template text
     * @param {String} template   Text with placeholders as '{{% variablename }}'
     * @param {Object} mapping    Object that maps variablenames to values
     * @returns {String}          Template replaced with given values for variables.
     */
    Sharer.prototype.fillTemplate = function(template, mapping) {
        var rex;
        for (var key in mapping) {
            rex = RegExp('{{%\\s*'+key+'\\s*%}}', 'g')
            template = template.replace(rex, mapping[key] + '');
        };
        return template;
    };
    
    //Sharer.prototype.registerConfForm = function() {
    //    var formdata = {apptype: this.apptype, appid: this.appid, confform: this.getConfigForm()};
    //    this.place.trigger('registerconfigdialog', formdata);
    //}
    
    /**
     * Inform outer system that the application is ready
     */
    Sharer.prototype.appReady = function(){
        var uilang = this.settings.uilang;
        this.place.trigger('appready', {apptype: this.apptype, appid: this.appid, title: ebooklocalizer.localize('sharer:sharertool', uilang)});
    };

    Sharer.prototype.setStyles = function() {
        if ($('head style#sharertoolstyles').length === 0) {
            $('head').append('<style type="text/css" id="sharertoolstyles">'+this.styles+'</style>')
        }
    };
    
    Sharer.prototype.close = function() {
        this.place.off();
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    };
    
    //Sharer.prototype.getConfigForm = function() {
    //    var uilang = this.settings.uilang;
    //    var form = {
    //        name: this.appid,
    //        title: ebooklocalizer.localize('marginnotes:margin', uilang),
    //        icon: this.icons.appicon,
    //        form: [],
    //        weight: 8
    //    };
    //    
    //    return form;
    //};
    
    Sharer.prototype.defaults = {
        settings: {
            uilang: 'en'
        }
    };
    
    Sharer.prototype.styles = [
        '.sharer {background-color: #eee; font-family: helvetica, Arial, sans-serif; padding: 1em; display: flex; flex-direction: column; justify-content: flex-start; overflow: hidden;}',
        '.sharer-header {flex-grow: 0; flex-shrink: 0;}',
        '.sharer-iconarea {display: inline-block;}',
        '.sharer-controlbuttons {flex-grow: 0; flex-shrink: 0;}',
        '.sharer-sharelist {margin: 0; margin-right: 1.5em; padding: 0; flex-grow: 1; flex-shrink: 1; overflow: auto; list-style: none;}',
        '.sharer-button-icon {display: block;}',
        '.sharer-button-label {display: block; font-size: 80%; max-width: 50px; white-space: normal; line-height: 0.9em;}',
        '.sharer-shareitem {margin: 0.1em 0; padding: 0; border: 1px solid #888; border-radius: 2px; background-color: rgba(255,255,200, 0.5); color: black;}',
        '.sharer-shareitem-date {font-size: 75%; text-align: right; background-color: rgba(200,200,200,0.3); padding: 0 0.7em;}',
        '.sharer-shareitem-title {font-size: 110%; font-weight: bold; padding: 0 0.4em;}',
        '.sharer-shareitem-link {padding: 0 0.5em;}',
        '.sharer-shareitem-link a {color: #006;}',
        '.sharer-shareitem-link a:visited {color: #a00;}'
    ].join('\n');
    
    Sharer.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-share"><path style="stroke: none;" d="M7 11 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m15 -8 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 16 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m-13 -4 l15 8 v2 l-15 -8z m0 -2 l15 -8 v2 l-15 8z"></path></svg>',
        sharerotate: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-share-rotating"><path style="stroke: none;" d="M7 11 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m15 -8 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 16 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m-13 -4 l15 8 v2 l-15 -8z m0 -2 l15 -8 v2 l-15 8z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 15 15" to="360 15 15" dur="4s" repeatCount="indefinite"></animateTransform></path></svg>'
    }
    
    Sharer.prototype.template = {
        html: [
            '<h1 class="sharer-header">',
            '    <div class="sharer-iconarea"><div class="sharer-icon">'+Sharer.prototype.icons.appicon+'</div></div>',
            '    <span class="sharer-title"></span>',
            '</h1>',
            '<div class="sharer-controlbuttons ffwidget-buttonset ffwidget-horizontal"></div>',
            '<ul class="sharer-sharelist"></ul>'
            //'<div class="marginnotes-actions ffwidget-buttonset ffwidget-horizontal"><button class="marginnotes-button-start ffwidget-setbutton" data-action="videoconf_start"></button></div>'
        ].join('\n'),
        setbutton: [
            '<button class="sharer-button ffwidget-setbutton"></button>'
        ].join(''),
        shareitem: [
            '<li class="sharer-shareitem"><div class="sharer-shareitem-date">{{% date %}}</div><div class="sharer-shareitem-title">{{% title %}}</div><div class="sharer-shareitem-link"><a target="_blank" href="{{% url %}}">{{% url %}}</a></div></li>'
        ].join('')
    };
    
    //Sharer.prototype.radiobuttons = [
    //    {
    //        name: 'notehide',
    //        label: 'marginnotes:hidenotesaction',
    //        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-hidenotes"><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path></svg>',
    //        action: 'notehide'
    //        
    //    },
    //    {
    //        name: 'noteview',
    //        label: 'marginnotes:shownotesaction',
    //        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-shownotes"><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path><path style="fill: red; stroke: none;" d="M5 22 h20 v2 h-20z"></path></svg>',
    //        action: 'noteview'
    //    },
    //    {
    //        name: 'noteedit',
    //        label: 'marginnotes:editnotesaction',
    //        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-editnotes"><path style="fill-opacity: 1; stroke: none;" d="M6 8 m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z"></path><path style="fill: red; stroke: none;" d="M5 22 h12 v2 h-12z"></path><path style="fill: #d00; stroke: none;" d="M18 23 l0 -4 l7.5 -14 l4 2 l-7.5 14z"></path></svg>',
    //        action: 'noteedit'
    //    }
    //];
    
    Sharer.localization = {
        "en": {
            "sharer:sharertool": "Share center"
        },
        "fi": {
            "sharer:sharertool": "Jakokeskus"
        },
        "sv": {
            "sharer:sharertool": "Delningscenter"
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Sharer.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Sharer.localization);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var sharer = new Sharer(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[sharerdata]]');
            return data;
        },
        'geticon': function() {
            return Sharer.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.sharertool = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in sharertool.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
