/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - Sdeditor
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.sdeditor) === 'undefined' && jQuery.fn.sdeditor.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

;(function($){
    
    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true, {}, Sdshuffle.defaults.userdata, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime.getTime();
        };
    }
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };
    
    Userdata.prototype.setSelectedOrder = function(list, corrects) {
        this.data.order = list;
        this.data.results = corrects.map(function(a){return a|0;});
        var nowtime = new Date();
        if (!this.metadata.created) {
            this.metadata.created = nowtime.getTime();
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime.getTime();
        this.metadata.modifier = this.settings.username;
    }
    
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    }
    
    //Userdata.prototype.isAnswer = function(str){
    //    return (this.data.answers.indexOf(str) > -1);
    //};
    //
    //Userdata.prototype.getAnswer = function(){
    //    var answer = this.data.answers[this.data.answers.length - 1] || '';
    //    return answer;
    //}
    //
    Userdata.prototype.getNewId = function(){
        var idparts = ['sdshuffleelement'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };

    /******
     * Shuffle as a jQuery-function
     ******/
    (function(d){d.fn.shuffle=function(c){c=[];return this.each(function(){c.push(d(this).clone(true))}).each(function(a,b){d(b).replaceWith(c[a=Math.floor(Math.random()*c.length)]);c.splice(a,1)})};d.shuffle=function(a){return d(a).shuffle()}})(jQuery);
    
    /******
     * SdShuffle question element
     * @constructor
     * @param {jQuery} place - the place for the element
     * @param {Object} options - data for element
     ******/
    var Sdshuffle = function(place, options){
        this.place = $(place);
        this.place.addClass('sdshuffleelement');
        this.setStyles();
        this.init(options);
        //this.addSounds();
        this.show();
    };
    
    /******
     * Init the sdshuffle
     ******/
    Sdshuffle.prototype.init = function(options) {
        options = $.extend(true, {}, Sdshuffle.defaults, options);
        this.assignmentid = this.place.attr('data-element-name');
        this.settings = options.settings;
        if (this.settings.mode === 'view') {
            // Use 'appendview' instead of 'view' by default. If the multichoice is already
            // solved, change to 'view' later.
            this.settings.mode = 'appendview'
        };
        this.solvable = this.settings.users && this.settings.users.canEdit && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.metadata = options.metadata;
        this.data = options.data;
        var userdata = options.userdata;
        userdata.settings = {username: this.settings.username, lang: this.settings.lang || ''}
        this.userdata = new Userdata(userdata);
        this.setMode(this.settings.mode, true);
        this.setAttrs();
        //this.addSounds();
    };
    
    /******
     * Get the data of the sdshuffle
     ******/
    Sdshuffle.prototype.getData = function(){
        var data = {
            "type": "sdshuffle",
            "metadata": $.extend(true, {}, this.metadata),
            "data": {
                "text": this.data.text || "",
                "derivation": $.extend(true, {}, this.data.derivation)
            }
        };
        return data;
    };

    Sdshuffle.prototype.setAttrs = function(){
        this.place.attr('lang', escapeHTML(this.metadata.lang))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .attr('data-tags', escapeHTML(this.metadata.tags.join(',')))
            .attr('data-quiztotal', '1')
            .addClass('quizquestion');
    };
    
    /******
     * Set modes
     ******/
    Sdshuffle.prototype.setMode = function(mode, dontredraw){
        if (!Sdshuffle.modes[mode]) {
            mode = 'view';
        };
        this.place.attr('data-elementmode', escapeHTML(mode));
        var modedata = Sdshuffle.modes[mode];
        this.editable = modedata.editable;
        this.authorable = modedata.authorable;
        this.reviewable = modedata.reviewable;
        this.appendable = modedata.appendable;
        this.settings.mode = mode;
        if (!dontredraw) {
            this.show();
        }
        this.initHandlers();
    }

    /**
     * Remove all handlers
     */
    Sdshuffle.prototype.removeHandlers = function() {
        this.place.off();
    };
    
    /******
     * Init event handlers
     ******/
    Sdshuffle.prototype.initHandlers = function() {
        this.removeHandlers();
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersView();
        };
        if (this.appendable) {
            this.initHandlersAppendable();
        }
    }
    
    /******
     * Init event handlers
     ******/
    Sdshuffle.prototype.initHandlersCommon = function() {
        var shuffle = this;
        this.place.on('getdata', function(event){
            var data = shuffle.getData();
            shuffle.place.data('[[elementdata]]', data);
        });
        this.place.on('reply_get_userdata', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            data.settings = {
                username: shuffle.settings.username,
                lang: shuffle.settings.lang || '',
                uilang: shuffle.settings.uilang,
                role: shuffle.settings.role
            }
            shuffle.userdata = new Userdata(data);
            if (shuffle.userdata.isSolved()) {
                shuffle.show();
            } else {
                shuffle.appendUserData();
            };
        });
        this.place.on('reply_getcontentbyref', function(e, data) {
            var refid = shuffle.assignmentid;
            reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: shuffle.settings.username,
                    lang: shuffle.settings.lang || '',
                    uilang: shuffle.settings.uilang,
                    role: shuffle.settings.role
                };
                shuffle.userdata = new Userdata(userdata);
                shuffle.appendUserData();
            };
        });
        this.place.on('showcorrectanswer', function(event, data) {
            event.stopPropagation();
            shuffle.showCorrect();
        });
        this.place.on('clearanswer', function(event, data) {
            event.stopPropagation();
            shuffle.clearAnswer();
        });
        this.place.on('getdata', function(e) {
            var data = shuffle.getData();
            shuffle.place.data('[[elementdata]]', data);
        });
    };
    
    /******
     * Init event handlers
     ******/
    Sdshuffle.prototype.initHandlersView = function() {
        var shuffle = this;
    }
    
    /******
     * Init event handlers
     ******/
    Sdshuffle.prototype.initHandlersEdit = function() {
        var shuffle = this;
        var place = this.place;
        this.place.on('element_changed', '.sdshuffle-derivation', function(event, data){
            event.stopPropagation();
            shuffle.updateSdData();
            shuffle.place.trigger('element_changed', {type: 'sdshuffle'});
        });
    };
    
    /******
     * Init event handlers
     ******/
    Sdshuffle.prototype.initHandlersAppendable = function() {
        var shuffle = this;
        var place = this.place;
        var shuffleElem = this.assignmentid;
        this.place.on('sortstart', '.'+shuffleElem+'shuffle', function(event, ui){
            shuffle.playSounds('#sdshuffle_moving');
        });
        this.place.on('sortupdate', '.'+shuffleElem+'shuffle', function(event, ui){
            shuffle.playSounds('#sdshuffle_moved');
            var userdata = shuffle.userdata;
            var tableRows = place.find('.sdeditor-derivationstep > .sdlayouttable > tbody > tr');
            var iscorrect = shuffle.isCorrect(tableRows);
            var neworder = shuffle.getOrder(tableRows);
            var correctness = shuffle.getCorrectness(neworder)
            userdata.setSelectedOrder(neworder, correctness);
            userdata.solved(iscorrect);
            if (shuffle.settings.feedback) {
                if (iscorrect) {
                    userdata.solved(iscorrect);
                    shuffle.saveUserData({sendforward: true});
                    shuffle.setMode('view', true); // Change mode to 'view', don't redraw.
                    shuffle.orderReady(shuffleElem);
                } else {
                    shuffle.saveUserData();
                }
            } else {
                shuffle.saveUserData();
            }
            //for (var i = 0; i < tableRows.length; i++) {
            //    if (tableRows.eq(i).attr('data-order') != i) {
            //        break;
            //    };
            //    if (i === tableRows.length - 1) {
            //        shuffle.playSounds('#sdshuffle_done');
            //        shuffle.userdata.solved(true);
            //        shuffle.saveUserData(true);
            //        shuffle.orderReady(shuffleElem);
            //    };
            //};
        });
    }
    
    /******
     * Show the sdshuffle
     ******/
    Sdshuffle.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    }
    
    /******
     * View the sdshuffle in view mode
     ******/
    Sdshuffle.prototype.view = function() {
        this.place.html('<div class="sdshuffle-derivation"></div>');
        this.place.children('.sdshuffle-derivation').sdeditor(this.data.derivation);
        var tableRows = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody > tr');
        tableRows.find('.sdmath').off(); // Remove Mathquill bindings to enable dragging from math.
        for (var i = 0; i < tableRows.length; i++) {
            tableRows.eq(i).attr('data-order', i);
        }
        var allDerivations = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody');
        allDerivations.addClass(this.assignmentid + 'shuffle');
        if (this.userdata.isSolved()) {
            this.orderReady();
        } else {
            this.place.trigger('get_userdata', {refid: this.assignmentid});
            this.issortable = true;
            for (var i = 0; i < allDerivations.length; i++) {
                allDerivations.eq(i).children('tr').shuffle();
                allDerivations.eq(i).sortable().disableSelection();
            };
        }
        var total = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody > tr').length;
        this.place.attr('data-quiztotal', escapeHTML(total));
        if (!this.userdata.isClosed()) {
            this.place.trigger('get_userdata', {refid: this.assignmentid});
            this.place.trigger('getcontentbyref', {refs: [this.assignmentid]});
        } else {
            this.appendUserData();
        }
    };
    
    Sdshuffle.prototype.appendUserData = function() {
        var solved = this.userdata.isSolved();
        var closed = this.userdata.isClosed();
        
        
        if (this.userdata && this.userdata.data && this.userdata.data.order && this.userdata.data.order.length) {
            this.setOrder(this.userdata.data.order);
        };
        
        var shuffleElements = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody');
        var shuffleItems = shuffleElements.children('tr');
        if (closed) {
            shuffleElements.sortable('disable');
            var order = this.getOrder(shuffleItems);
            var correctness = this.getCorrectness(order);
            shuffleItems.removeClass('sdshuffle-iscorrect sdshuffle-iswrong');
            for (var i = 0, len = shuffleItems.length; i < len; i++) {
                shuffleItems.eq(i).addClass(correctness[i] ? 'sdshuffle-iscorrect' : 'sdshuffle-iswrong');
            }
        }
    }
    
    /******
     * View the sdshuffle in edit mode
     ******/
    Sdshuffle.prototype.edit = function(){
        this.place.html('<div class="sdshuffle-derivation"></div>');
        var data = this.data.derivation;
        data.settings = {
            mode: 'edit',
            username: this.settings.username,
            role: this.settings.role,
            lang: this.settings.lang,
            uilang: this.settings.uilang
        };
        var sdplace = this.place.children('.sdshuffle-derivation');
        sdplace.sdeditor(data);
        if (this.data.derivation) {
            // Old data needs to be put in edit mode.
            sdplace.sdeditor('edit');
        };
    };

    /******
     * Update the data from sdeditor
     ******/
    Sdshuffle.prototype.updateSdData = function(){
        this.data.derivation = this.place.children('.sdshuffle-derivation').sdeditor('get');
        this.metadata.modified = (new Date()).getTime();
    }
    
    /******
     * Stop being sortable
     ******/
    Sdshuffle.prototype.orderReady = function(){
        if (this.issortable) {
            this.place.find('.'+this.assignmentid+'shuffle').sortable('destroy');
            this.issortable = false;
        }
        this.place.addClass('sdInOrder');
    };
    
    /******
     * Save userdata
     ******/
    Sdshuffle.prototype.saveUserData = function(options){
        if (this.solvable) {
            var sendforward = options && options.sendforward;
            var localsave = options && !!options.localsave;
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            if (sendforward) {
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    /**
     * Show correct answer
     */
    Sdshuffle.prototype.showCorrect = function() {
        this.setMode('view');
        var rows = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody > tr');
        var list = [];
        for (var i = 0; i < rows.length; i++) {
            list.push(i);
        };
        this.setOrder(list);
    };
    
    /**
     * Clear answer
     */
    Sdshuffle.prototype.clearAnswer = function() {
        var userdata = {settings: {username: this.settings.username, lang: this.settings.lang || ''}};
        this.userdata = new Userdata(userdata);
        this.setMode('appendview');
        this.appendUserData();
    }
    
    /**
     * Get the current order
     */
    Sdshuffle.prototype.getOrder = function(elements) {
        var order = [];
        for (var i = 0, len = elements.length; i < len; i++) {
            order.push(elements.eq(i).attr('data-order') | 0);
        }
        return order;
    };
    
    /**
     * Set order
     */
    Sdshuffle.prototype.setOrder = function(list) {
        try {
            var listitems = this.place.find('.sdeditor-derivationstep > .sdlayouttable > tbody > tr');
            var item, parent;
            for (var i = 0, len = list.length; i < len; i++) {
                item = listitems.filter('[data-order="'+list[i]+'"]');
                parent = item.closest('tbody');
                parent.append(item);
            }
        } catch (err) {
            console.log('Sdshuffle.setOrder error:', err);
        };
    };
    
    /**
     * Check the correctness
     */
    Sdshuffle.prototype.isCorrect = function(rowitems) {
        var iscorrect = true;
        for (var i = 0, len = rowitems.length; i < len && iscorrect; i++) {
            iscorrect = ((rowitems.attr('data-order') | 0) === i);
        };
        return iscorrect;
    }
    
    /**
     * Check the correctness
     * @param {Array} order  An array with a permutation of numbers 0..n.
     * @returns {Boolean[]}  An array with correctness of each row.
     */
    Sdshuffle.prototype.getCorrectness = function(order) {
        var result = [];
        for (var i = 0, len = order.length; i < len; i++) {
            result.push(order[i] === i);
        };
        return result;
    };
    
    /******
     * Add sounds to the html body
     ******/
    Sdshuffle.prototype.addSounds = function() {
        if (jQuery("#soundEffects").length === 0){
            var effectsPlace = jQuery('body').append('<div id="soundEffects"/>').find('#soundEffects');
        } else {
            var effectsPlace = jQuery("#soundEffects");
        };
        for (var sound in Sdshuffle.sounds) {
            if (jQuery("#" + sound).length === 0) {
                jQuery('#soundEffects').append('<audio id="'+escapeHTML(sound)+'" autobuffer="autobuffer" ><source src="data:audio/mpeg;base64,'+Sdshuffle.sounds[sound]+'"></source></audio>');
            };
        };
    };

    /******
     * Play the needed sound
     ******/
    Sdshuffle.prototype.playSounds = function(id){
        if (jQuery('html').hasClass('ipad')) {
            if (typeof(this.audios[id])==="undefined") {
                this.audios[id]={};
                var audioelement = this.audios[id];
                if (typeof(window.myAudioContext) === "undefined"){
                    window.myAudioContext = new webkitAudioContext();
                };
                this.audios[id]['arrayBuff'] = Base64Binary.decodeArrayBuffer(this.sounds[id.replace("#", "")]);
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                context.decodeAudioData(this.audios[id]['arrayBuff'], function(audioData) {
                    element.buffer = audioData;
                    audioelement['buffer'] = audioData;
                    element.connect(context.destination);
                    element.noteOn(0);
                });
            } else {
                if (typeof(window.myAudioContext) === "undefined"){
                    window.myAudioContext = new webkitAudioContext();
                };
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                element.buffer = this.audios[id]['buffer'];
                element.connect(context.destination);
                element.noteOn(0);
            };
        } else {
            jQuery(id)[0].play();
        };
    };
    
    Sdshuffle.prototype.playSounds = function(id){};
    
    /******
     * Set style-tag if needed
     ******/
    Sdshuffle.prototype.setStyles = function(){
        if ($('head style#sdshuffleelement-style').length === 0) {
            $('head').append('<style id="sdshuffleelement-style" type="text/css">'+Sdshuffle.styles+'</style>');
        }
    }
    
    /******
     * Default values for input data
     ******/
    Sdshuffle.defaults = {
        type: 'sdshuffle',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            lang: '',
            tags: []
        },
        data: {
            title: '',
            text: '',
            derivation: {}
        },
        userdata: {
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                solved: false
            }
        },
        settings: {
            mode: 'view',
            uilang: 'en',
            role: 'student'
        }
    };
    
    /******
     * Modes
     ******/
    Sdshuffle.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        appendview: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: true
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            appendable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            appendable: false
        }
    };

    /******
     * Stylesheets
     ******/
    Sdshuffle.styles = [
        '.sdshuffle-derivation tr.ui-sortable-helper {background-color: #f9f9f9; box-shadow: 4px 4px 15px rgba(0,0,0,0.5);}',
        '.sdshuffle-derivation .ui-sortable tr, .sdshuffle-derivation .ui-sortable tr .sdmath {cursor: pointer;}',
        '.sdshuffleelement.sdInOrder {box-shadow: 0 0 5px #0a0;}',
        '.sdshuffle-iswrong {color: #a00; font-weight: bold; text-decoration: line-through;}',
        '.sdshuffle-iscorrect {font-weight: normal; color: black;}'
    ].join('\n');
    
    /******
     * Sound files
     ******/
    Sdshuffle.sounds ={
        "sdshuffle_done":   "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",
        "sdshuffle_moved":"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",
        "sdshuffle_moving":"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"
    };
    
    /******
     * Elementinfo for elementsets
     ******/
    Sdshuffle.elementinfo = {
        type: 'sdshuffle',
        elementtype: ['assignments', 'quizelement'],
        jquery: 'sdshuffle',
        name: 'Sdshuffle',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-sdshuffle"><path style="stroke: none;" d="M4 4 h22 v2 h-22 m0 18 h22 v2 h-22z m5 -3 v-10 h-3 l5 -4 l5 4 h-3 v10z m10 2 l-5 -4 h3 v-10 h4 v10 h3z" /></svg>',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 10 l10 0 l0 -10z m0 12 l0 10 l10 0 l0 -10z m12 0 l0 10 l10 0 l0 -10z m0 -12 l0 10 l10 0 l0 -10z m-7 2 l4 6 l-8 0z m-4 12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z m16 -4 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m-4 -12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z" /></svg>',
        description: {
            en: 'Order the structured derivation assignment.',
            fi: 'Päättelyketjun järjestelytehtävä'
        },
        weight: 0,
        roles: ['teacher', 'author'],
        classes: ['assignments']
    };

    // Register multichoice as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Sdshuffle.elementinfo);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Sdshuffle.elementinfo);
    }


    /******
     * Convert data from old format to the new one
     ******/
    var convert2new = function(options) {
        options = JSON.parse(JSON.stringify(options));
        var contentid = options.data.contents[0].id;
        var derivation = {};
        if (options.data && options.data.contentdata) {
            derivation = options.data.contentdata[contentid];
        };
        options.data.derivation = derivation;
        options.data.text = '';
        delete options.data.assignmentType;
        delete options.data.contentdata;
        delete options.data.content;
        return options;
    };
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            if (params.data && params.data.contentdata) {
                params = convert2new(params);
            }
            return this.each(function(){
                var Sdshuffleassignment = new Sdshuffle(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.sdshuffle = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in sdshuffle.');
            return false;
        }
    }
    
})(jQuery);