//{{{
/*********************************************************
 * jquery.sdeditor.js
 * jQuery-plugin for sdeditor
 * Petri Salmela
 * Petri Sallasmaa
 * 25.03.2013
 ********************************************************/

 /**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * Helper functions
     */

    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /**
     * sdeditor
     * @param options
     */
    
    /**********
     * Compatibility with older (<1.7) jQuery
     **********/
    if (typeof($.fn.on) !== 'function') {
        $.fn.on = function(a, b, c, d){
            if (arguments.length === 2 && typeof(b) === 'function') {
                return this.bind(a, b);
            } else {
                return this.delegate(b, a, c, d);
            }
        }
    }
    if (typeof($.fn.off) !== 'function') {
        $.fn.off = function(a, b, c, d){return 1===arguments.length ? this.undelegate(a, '**'): this.undelegate(b, a, c, d);}
    }

    $.fn.sdeditor = function(options){
        if (methods[options]){
            return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
        } else if (typeof(options) === 'object' || !options) {
            return methods.init.apply(this, arguments);
        } else {
            $.error( 'Method ' +  method + ' does not exist on jQuery.fn.sdeditor' );
            return this;
        }
    }

    var methods = {
        init: function( options ){
            if (options.data && options.data.main) {
                options = options.data;
            }
            if (options.main) {
                options = convertOldToNew(options);
            }
            var settings = $.extend({
            }, options);

            return this.each(function(){
                var sdeditor = new SdEditor(this, settings);
            });
        },
        get: function(){
            // Return the derivation data
            this.trigger('get');
            return this.eq(0).data('sdeditor');
        },
        getjson: function(){
            // Return the derivation data as json string
            this.trigger('get');
            return JSON.stringify(this.eq(0).data('sdeditor'));
        },
        setmode: function( options ){
            if (typeof(options) === 'undefined' || this.hasClass('sdeditorwrapper')) {
                // Trigger view mode by default
                this.trigger('view');
                return true;
            } else {
                this.trigger('setmode', [options]);
                return true;
            }
        },
        edit: function( options ){
            if (typeof(options) === 'undefined' || this.hasClass('sdeditorwrapper')) {
                // Trigger edit mode
                this.trigger('edit');
                return true;
            } else {
                options.mode = 'edit';
                return methods.init.call(this, options);
            }
        },
        view: function( options ){
            if (typeof(options) === 'undefined') {
                // Trigger view mode
                this.trigger('view');
                return true;
            } else {
                options.mode = 'view';
                return methods.init.call(this, options);
            }
        },
        show: function( options ){
            if (typeof(options) === 'undefined') {
                // Trigger view mode
                this.trigger('view');
                return true;
            } else {
                options.mode = 'view';
                return methods.init.call(this, options);
            }
        },
        review: function( options ){
            if (typeof(options) === 'undefined' || this.hasClass('sdeditorwrapper')) {
                // Trigger review mode
                this.trigger('review');
                return true;
            } else {
                options.mode = 'review';
                return methods.init.call(this, options);
            }
        },
        settings: function( options ){
            if (typeof(options) !== 'undefined') {
                SdEditor.user.settings = $.extend(true, SdEditor.user.settings, options);
                return true;
            } else {
                return false;
            }
        },
        getsettings: function( options ){
            return JSON.parse(JSON.stringify(SdEditor.user.settings));
        },
        addplugin: function( options ){
            for (name in options) {
                var pluginset = options[name];
                var label = pluginset.label || {'en': name};
                var order = pluginset.order || 0;
                var modes = pluginset.modes || [];
                var items = [];
                for (var i = 0; i < pluginset.items.length; i++) {
                    var item = pluginset.items[i];
                    var valid = (typeof(item.name) === 'string' &&
                                 typeof(item.action) === 'function' &&
                                 (item.description && item.description.en && true || false));
                    if (valid) {
                        items.push({
                            name: item.name,
                            label: item.label || {'en': item.name},
                            icon: item.icon || '',
                            description: item.description || {'en': ''},
                            action: item.action
                        });
                    }
                }
                SdEditor.plugins[name] = {order: order, modes: modes, label: label, items: items};
            }
        },
        getreview: function(){
            // Return the review data
            var options = {result: ''};
            this.trigger('getreview', options);
            //return options.result;
            return this.eq(0).data('sdeditorreview');
        }
    };

    /******
     * Convert data from old sdeditor format to new sdeditor2 format
     ******/
    var convertOldToNew = function(data){
        data = JSON.parse(JSON.stringify(data));
        var settings = {
            mode: data.mode || (data.editable ? 'edit' : 'view'),
            stepmode: data.stepmode || false,
            theme: data.sdtheme || 'default',
            lang: data.lang || 'en',
            uilang: data.lang || 'en'
        }
        var result = {
            derivation: data.derivations[data.main]
        };
        for (var subname in data.derivations) {
            var subder = data.derivations[subname];
            for (var i = 0; i < subder.observation.length; i++) {
                // Search for all subderivations in observations.
                var motiv = subder.observation[i].motivation;
                for (var j = 0; j < motiv.subderiv.length; j++){
                    var sdname = motiv.subderiv[j];
                    motiv.subderiv[j] = data.derivations[sdname];
                }
                // Convert observation texts from mathquill-text to mathquill-math.
                subder.observation[i].text = (subder.observation[i].text.replace(/^([^$]*)/, '\\text{$1').replace(/\$([^$]*)\$/g, '}$1\\text{') + '}').replace(/\\text{}/g, '');

            }
            for (var k = 0; k < subder.motivation.length; k++) {
                // Search for all subderivations in (step) motivations.
                var motiv = subder.motivation[k];
                for (var l = 0; l < motiv.subderiv.length; l++){
                    var sdname = motiv.subderiv[l];
                    motiv.subderiv[l] = data.derivations[sdname];
                }
            }
            var tasklist = [];
            for (var j = 0; j < subder.task.length; j++) {
                tasklist.push({
                    text: subder.task[j],
                    readonly: false,
                    answer: {
                        text: '',
                        readonly: false
                    }
                });
            };
            subder.task = tasklist;
        }
        //var options = {metadata: {}, mode: data.mode || 'view', data: result};
        var options = {metadata: {}, data: result, settings: settings};
        return options;
    }

    /******************************************
     * Dictinary
     * Translations for the user interface.
     ******************************************/
    var Dictionary = {
        localize: function(term, lang){
            lang = lang || 'en';
            if (!Dictionary.terms[lang]) {
                lang = 'en';
            };
            term = Dictionary.terms[lang] && Dictionary.terms[lang][term] && term || 'undefined localization';
            return Dictionary.terms[lang][term];
        },
        terms: {
            en: {
                'undefined localization': '[Localization not found]',
                'add task': 'Add a question [ctrl+j]',
                'remove task': 'Remove the question',
                'add assumption': 'Add an assumption [Enter]',
                'remove assumption': 'Remove the assumption [ctrl+Backspace]',
                'add observation': 'Add a fact',
                'add definition': 'Add a definition',
                'add declaration': 'Add a declaration',
                'remove observation': 'Remove the observation',
                'add step': 'Add a step [Enter]',
                'remove step': 'Remove the step [ctrl+Backspace]',
                'add derivationmotivation': 'Add taskjustification',
                'remove derivationmotivation': 'Remove taskjustification',
                'add subderivation': 'Add a nested task [ctrl+b]',
                'remove subderivation': 'Remove the nested task [ctrl+shift+b]',
                'undo': 'Undo [ctrl+z]',
                'redo': 'Redo [ctrl+shift+z]',
                'steps:shownextstep': 'Show next step',
                'steps:hidelaststep': 'Hide the last step',
                'steps:showallsteps': 'Show all steps',
                'add task step': 'Add task step',
                'add assumption step': 'Add assumption step',
                'add fact step': 'Add fact step',
                'add definition step': 'Add definition step',
                'add declaration step': 'Add declaration step',
                'remove derivation step': 'Remove derivation step',
                'areyousure': 'Are you sure?',
                'question:simplify': 'Simplify',
                'question:calculate': 'Calculate',
                'question:prove': 'Prove',
                'question:solve': 'Solve'
            },
            fi: {
                'undefined localization': '[Käännöstä ei löytynyt]',
                'add task': 'Lisää tehtävä [ctrl+j]',
                'remove task': 'Poista tehtävä',
                'add assumption': 'Lisää oletus [Enter]',
                'remove assumption': 'Poista oletus [ctrl+Backspace]',
                'add observation': 'Lisää havainto',
                'add definition': 'Lisää määritelmä',
                'add declaration': 'Lisää esittely',
                'remove observation': 'Poista havainto',
                'add step': 'Lisää askel [Enter]',
                'remove step': 'Poista askel [ctrl+Backspace]',
                'add derivationmotivation': 'Lisää päättelyketjun perustelu',
                'remove derivationmotivation': 'Poista päättelyketjun perustelu',
                'add subderivation': 'Lisää alipäättely [ctrl+b]',
                'remove subderivation': 'Poista alipäättely [ctrl+shift+b]',
                'undo': 'Kumoa [ctrl+z]',
                'redo': 'Tee uudelleen [ctrl+shift+z]',
                'steps:shownextstep': 'Näytä seuraava askel',
                'steps:hidelaststep': 'Piilota viimeisin askel',
                'steps:showallsteps': 'Näytä kaikki askeleet',
                'add task step': 'Lisää tehtäväaskel',
                'add assumption step': 'Lisää oletusaskel',
                'add fact step': 'Lisää fakta-askel',
                'add definition step': 'Lisää määritelmäaskel',
                'add declaration step': 'Lisää esittelyaskel',
                'remove derivation step': 'Poista päättelyaskel',
                'areyousure': 'Oletko varma?',
                'question:simplify': 'Sievennä',
                'question:calculate': 'Laske',
                'question:prove': 'Todista',
                'question:solve': 'Ratkaise'
            },
            sv: {
                'undefined localization': '[Översättningen saknas]',
                'add task': 'Lägg till en uppgift [ctrl+j]',
                'remove task': 'Ta bort en uppgift',
                'add assumption': 'Lägg till ett antagande [Enter]',
                'remove assumption': 'Ta bort ett antagande [ctrl+Backspace]',
                'add observation': 'Lägg till en observation',
                'add definition': 'Lägg till en definition',
                'add declaration': 'Lägg till en deklaration',
                'remove observation': 'Ta bort en observation',
                'add step': 'Lägg till ett steg [Enter]',
                'remove step': 'Ta bort ett steg [ctrl+Backspace]',
                'add derivationmotivation': 'Lägg till en motivering för härledningen',
                'remove derivationmotivation': 'Ta bort motiveringen för härledningen',
                'add subderivation': 'Lägg till en delhärledning [ctrl+b]',
                'remove subderivation': 'Ta bort en delhärledning [ctrl+shift+b]',
                'undo': 'Återställ [ctrl+z]',
                'redo': 'Gör igen [ctrl+shift+z]',
                'steps:shownextstep': 'Visa nästa steg',
                'steps:hidelaststep': 'Göm det sista steget',
                'steps:showallsteps': 'Visa alla stegen',
                'add task step': 'Add task step',
                'add assumption step': 'Add assumption step',
                'add fact step': 'Add fact step',
                'add definition step': 'Add definition step',
                'add declaration step': 'Add declaration step',
                'remove derivation step': 'Remove derivation step',
                'areyousure': 'Är du säker?',
                'question:simplify': 'Förenkla',
                'question:calculate': 'Beräkna',
                'question:prove': 'Bevisa',
                'question:solve': 'Lösa'
            }
        }
    }
    
    /******
     * Constructor of SdEditor-object.
     * @constructor
     * @param {jQuery} place - jQuery-element (or DOM-element) for the place where to render derivation/editor.
     * @param {Object} options - The data of the derivation.
     *   @param {String} options.type - "sderivation"
     *   @param {Object} options.metadata - {creator, created, modifier, modified, tags,...}
     *   @param {Object} options.data - The actual data for the derivation.
     *   @param {Object} options.settings - the settings for SdEditor (mode, lang, uilang, sdtheme, username, ...)
     ******/
    var SdEditor = function(place, options){
        window.sdeditor = this;
        this.place = $(place);
        this.setSettings(options);
        this.setMetadata(options);
        this.setData(options);
        this.init();
        this.show();
    }
    
    /******
     * Init the SdEditor
     ******/
    SdEditor.prototype.init = function(){
        this.place.addClass('sdeditorwrapper selfreviewable').attr('style', 'position: relative;');
        this.initHandlers();
    }
    
    /******
     * Set mode of the editor
     ******/
    SdEditor.prototype.setMode = function(mode){
        if (!SdEditor.modes[mode]) {
            mode = 'view';
        };
        if (mode) {
            this.settings.mode = mode;
        } else {
            mode = this.settings.mode;
        };
        this.editable = SdEditor.modes[mode] && SdEditor.modes[mode].editable || false;
        this.authorable = SdEditor.modes[mode] && SdEditor.modes[mode].authorable || false;
        this.reviewable = SdEditor.modes[mode] && SdEditor.modes[mode].reviewable || false;
        this.initHandlers();
    };
    
    /******
     * Init event handlers of this editor.
     ******/
    SdEditor.prototype.initEvents = function(){
        var sdeditor = this;
    }
    
    /******
     * Set the theme of the editor
     ******/
    SdEditor.prototype.setTheme = function(theme){
        this.settings.theme = theme;
        this.place.trigger('sdeditor_themechanged', [{theme: theme}]);
        this.show();
    }

    /******
     * Set the settings of this editor.
     ******/
    SdEditor.prototype.setSettings = function(options){
        options = options || this.settings || {settings: {}};
        var currents = {
            // Existing language or the one closest in DOM.
            lang: (this.settings && this.settings.lang) || this.place.closest('[lang]').attr('lang'),
            uilang: (this.settings && this.settings.uilang) || this.place.closest('[uilang]').attr('uilang') || this.place.closest('[lang]').attr('lang'),
            theme: this.settings && this.settings.theme,
            mode: this.settings && this.settings.mode,
            stepmode: this.settings && this.settings.stepmode
        }
        this.settings = $.extend(true, {}, SdEditor.defaults.settings, currents, SdEditor.user.settings, options.settings);
        this.place.attr('lang', this.settings.uilang);
        this.setMode(this.settings.mode);
    }

    /******
     * Set the metadata of this editor.
     ******/
    SdEditor.prototype.setMetadata = function(options){
        options = options || {metadata: {}};
        this.metadata = $.extend({}, SdEditor.defaults.metadata, options.metadata);
    }

    /******
     * Set the data of this editor.
     ******/
    SdEditor.prototype.setData = function(options){
        var data
        this.name = options.name || this.place.attr('data-element-name') || this.generateId('structuredderivation');
        this.locname = this.name.replace(/[-_]/g, '');
        this.dsteps = [];
        this.dstepdata = {};
        var dstep, dstepdata;
        //if (options.type === 'sderivation' && options.data && options.data.dsteps) {
        if (options.type === 'sderivation' || options.type === 'sdeditor' && options.data && options.data.dsteps) {
            // If the data is multistepdata ('sderivation', in new format) or one step data ('sdeditor', in new format)
            this.type = options.type;
            this.place.addClass('sdeditor-dstepmode');
            data = $.extend({}, SdEditor.stepdefaults.data, options.data);
            for (var i = 0, len = data.dsteps.length; i < len; i++) {
                dstep = data.dsteps[i];
                this.dsteps.push(dstep);
                data.dstepdata[dstep.id].name = dstep.id;
                switch (dstep.type) {
                    case 'task':
                        dstepdata = new Sderivation(data.dstepdata[dstep.id]);
                        break;
                    case 'assumption':
                        dstepdata = new Sdassumption(data.dstepdata[dstep.id]);
                        break;
                    case 'observation':
                        dstepdata = new Sdobservation(data.dstepdata[dstep.id]);
                        break;
                    default:
                        break;
                }
                this.dstepdata[dstep.id] = dstepdata;
            };
        } else {
            // If the data is one step data in old format => convert it.
            this.type = 'sdeditor';
            this.place.addClass('sdeditor-dstepmode');
            data = $.extend({}, SdEditor.defaults.data, options.data);
            dstep = {id: this.generateId('stask'), type: 'task'};
            this.dsteps.push(dstep);
            data.derivation.name = dstep.id;
            this.dstepdata[dstep.id] = new Sderivation(data.derivation);
            this.derivation = this.dstepdata[dstep.id];
        }
        this.undoStack = new SdEventStack();
        this.redoStack = new SdEventStack();
        //this.derivation = new Sderivation(data.derivation);

        this.markings = new SdMarkings(data.markings);
        this.markings.setLang(this.settings.uilang);
        this.marksvisible = true;
    }
    
    /******
     * Return the settings of this editor.
     ******/
    SdEditor.prototype.getSettings = function(){
        return JSON.parse(JSON.stringify(this.settings));
    }
    
    /******
     * Return the data of this editor.
     ******/
    SdEditor.prototype.getData = function(){
        var data = {
            //sdtheme: this.sdtheme
            //editable: this.editable
            dstepdata: {}
        };
        data.dsteps = JSON.parse(JSON.stringify(this.dsteps));
        var dstep;
        for (var i = 0, len = this.dsteps.length; i < len; i++){
            dstep = this.dsteps[i];
            data.dstepdata[dstep.id] = this.dstepdata[dstep.id].getData();
        };
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date() + '');
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
        this.metadata.lang = this.settings.lang;
        var result = {
            type: this.type,
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: data
        }
        return result;
    }
    
    /******
     * Return the data of the derivation as an old format JSON object.
     ******/
    SdEditor.prototype.getJsonObject = function(){
        var result = {
            derivations: []
        }
        var dstep;
        //result.derivations.push(this.derivation.jsonobject());
        for (var i = 0, len = this.dsteps.length; i < len; i++) {
            dstep = this.dsteps[i];
            result.derivations.push(this.dstepdata[dstep.id].jsonobject());
        }
        return result;
    }
    
    /******
     * Get the data object with location strings.
     ******/
    SdEditor.prototype.getObject = function(){
        var result = {
            derivations: []
            //derivation: this.derivation.getData(true)
        }
        var dstep;
        for (var i = 0, len = this.dsteps.length; i < len; i++) {
            dstep = this.dsteps[i];
            result.derivations.push(this.dstepdata[dstep.id].getData(true));
        }
        return result;
    }
    
    /******
     * Get data for checker
     ******/
    SdEditor.prototype.getCheckData = function(){
        var name = this.locname || 'derivation';
        var result = {
            name: name,
            steps: []
        };
        var dstep;
        for (var i = 0, len = this.dsteps.length; i < len; i++){
            dstep = this.dsteps[i];
            result.steps.push(this.dstepdata[dstep.id].getCheckData([name, 'step', i], true));
        };
        return result;
    }
    
    /******
     * Get derivationstep by location
     * @param {Array} location - location path
     ******/
    SdEditor.prototype.getDstep = function(location){
        location.shift();
        location.shift();
        var stepnum = location.shift() | 0;
        var dstepid = this.dsteps[stepnum].id;
        var dstep = this.dstepdata[dstepid];
        return dstep;
    }

    /******
     * Edit given data to location [deriv, element, pos, element, pos,...]
     ******/
    SdEditor.prototype.editLocation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.editLocation(location, data);
        } else {
            return {};
        }
    }

    /******
     * Set given attributes to location [deriv, element, pos, element, pos,...]
     ******/
    SdEditor.prototype.setLocationAttrs = function(location, attrs){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.setLocationAttrs(location, attrs);
        } else {
            return {};
        }
    }

    /******
     * Get data from location
     ******/
    SdEditor.prototype.getLocationData = function(location){
        var dstep = this.getDstep(location);
        if (dstep && location.length > 0) {
            return dstep.getLocationData(location);
        } else {
            return {};
        }
    }
    
    /******
     * Show/Hide subderivation in location.
     ******/
    SdEditor.prototype.showHideLocation = function(location, status){
        var result = false;
        var dstep = this.getDstep(location);
        if (dstep) {
            if (status) {
                result = dstep.hideLocation(location);
            } else {
                result = dstep.showLocation(location);
            }
        };
        return result;
    }
    
    /******
     * Check that the location exists.
     ******/
    SdEditor.prototype.locationExists = function(location){
        var dstep = this.getDstep(location);
        if (dstep && location.length > 0) {
            return dstep.locationExists(location);
        } else {
            return false;
        }
    }
    
    /******
     * Get elemtypes of given location
     ******/
    SdEditor.prototype.getLocationTypes = function(location){
        var dstep = this.getDstep(location);
        return dstep.getLocationTypes(location);
    }

    /******
     * Get previous location. Optionally only editable locations searched.
     ******/
    SdEditor.prototype.getPrevLoc = function(location, editable){
        var loc = (typeof(location) === 'string' ? location : '');
        if (typeof(location) === 'object' && location.hasOwnProperty('length')) {
            loc = location.join('_');
        }
        var search = (editable ? '[data-loc] .mathquill-editable' : '[data-loc] .sdmath, [data-loc] .sdmathtext');
        var current = this.editorelem.find('[data-loc="'+location+'"] .sdmath, [data-loc="'+location+'"] .sdmathtext');
        var allloc = this.editorelem.find(search);
        var index = allloc.index(current);
        index--;
        if (index > -1) {
            var newloc = allloc.eq(index).parents('td').eq(0).attr('data-loc') || '';
        } else {
            var newloc = '';
        }
        return newloc;
    }

    /******
     * Get next location. Optionally only editable locations searched.
     ******/
    SdEditor.prototype.getNextLoc = function(location, editable){
        var loc = (typeof(location) === 'string' ? location : '');
        if (typeof(location) === 'object' && location.hasOwnProperty('length')) {
            loc = location.join('_');
        }
        var search = (editable ? '[data-loc] .mathquill-editable' : '[data-loc] .sdmath, [data-loc] .sdmathtext');
        var current = this.editorelem.find('[data-loc="'+location+'"] .sdmath, [data-loc="'+location+'"] .sdmathtext');
        var allloc = this.editorelem.find(search);
        var index = allloc.index(current);
        index++;
        if (index < allloc.length) {
            var newloc = allloc.eq(index).parents('td').eq(0).attr('data-loc') || '';
        } else {
            var newloc = '';
        }
        return newloc;
    }
    
    /**
     * The derivation has changed
     */
    SdEditor.prototype.changed = function() {
        this.place.trigger('sderiv_changed');
        this.place.trigger('element_changed', [{type: 'sdeditor'}]);
    }
    
    /******
     * Set focus to given location.
     ******/
    SdEditor.prototype.setFocus = function(location){
        var loc = (typeof(location) === 'string' ? location : '');
        if (typeof(location) === 'object' && location.hasOwnProperty('length')) {
            loc = location.join('_');
        }
        this.editorelem.find('td[data-loc="'+loc+'"] .mathquill-editable').focus();
    }
    /******
     * Add a new step with given data to a given location.
     ******/
    SdEditor.prototype.addStep = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addStep(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Remove the step in given location.
     * Return the old data to be put in undo/redo-stack.
     ******/
    SdEditor.prototype.removeStep = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeStep(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Add the task in given location with given data.
     ******/
    SdEditor.prototype.addTask = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addTask(location, data);
        } else {
            return false;
        }
    }
    
    /******
     * Remove the task from given location.
     ******/
    SdEditor.prototype.removeTask = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeTask(location, data);
        } else {
            return false;
        }
    }
    
    /******
     * Add a new assumption in given location with given data.
     ******/
    SdEditor.prototype.addAssumption = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addAssumption(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Remove an assumption from given location.
     ******/
    SdEditor.prototype.removeAssumption = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeAssumption(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Add a new observation in given location with given data.
     ******/
    SdEditor.prototype.addObservation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addObservation(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Remove an observation from given location.
     ******/
    SdEditor.prototype.removeObservation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeObservation(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Add a new subderivation in given location with given data.
     ******/
    SdEditor.prototype.addSubderivation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addSubderivation(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Remove a subderivation from given location.
     ******/
    SdEditor.prototype.removeSubderivation = function(location){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeSubderivation(location);
        } else {
            return {};
        }
    }
    
    /******
     * Add a new derivation motivation in given location with given data.
     ******/
    SdEditor.prototype.addDerivMotivation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.addDerivMotivation(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Remove derivation motivation from given location.
     ******/
    SdEditor.prototype.removeDerivMotivation = function(location, data){
        var dstep = this.getDstep(location);
        if (dstep) {
            return dstep.removeDerivMotivation(location, data);
        } else {
            return {};
        }
    }
    
    /******
     * Add derivation step after some step
     * @param {String} location - The name of derivation to add this step at (before)
     * @param {String} dstype - The type of new derivation step (task, assumption, fact, definition)
     * @param {Object} dsdata - optional data from initing the new derivation step
     * @returns {Array} location path for the first fillable field.
     ******/
    SdEditor.prototype.addDSAt = function(location, dstype, dsdata){
        var index = location[2] | 0;
        var newname = dsdata.name || this.generateId(dstype);
        var newloc = location.slice(0,3);
        var type = dstype;
        if (type === 'fact' || type === 'definition' || type === 'declaration') {
            type = 'observation';
        }
        this.dsteps.splice(index, 0, {id: newname, type: type});
        if (!(dsdata && dsdata.name) || dsdata.dstepcopy) {
            // If the dsdata is not a valid instance of one of [Sderivation, Sdassumption, Sdobservation], then make it.
            var data;
            if (dsdata.dstepcopy) {
                data = dsdata.data;
            } else {
                data = {name: newname, type: dstype}
            };
            //var data = (!(dsdata && dsdata.name) ? dsdata : {name: newname, type: dstype});
            switch (dstype) {
                case 'task':
                    dsdata = new Sderivation(data);
                    break;
                case 'assumption':
                    dsdata = new Sdassumption(data);
                    break;
                case 'fact':
                    dsdata = new Sdobservation(data);
                    break;
                case 'definition':
                    data.definition = {};
                    dsdata = new Sdobservation(data);
                    break;
                case 'declaration':
                    dsdata = new Sdobservation(data);
                    break;
                default:
                    break;
            };
        }
        this.dstepdata[newname] = dsdata;
        switch (dstype) {
            case 'task':
                if (dsdata.elements.task.length > 0) {
                    newloc.push('task');
                } else {
                    newloc.push('term');
                }
                newloc.push('0');
                break;
            case 'fact':
                newloc.push('motivation');
                break;
            case 'definition':
                newloc.push('declaration');
                break;
            case 'declaration':
                newloc.push('declaration');
                break;
            default:
                break;
        }
        return newloc;
    };
    
    /******
     * Add derivation step after some step
     * @param {String} dsafter - The name of derivation to add this step after
     * @param {String} dstype - The type of new derivation step (task, assumption, fact, definition)
     * @param {Object} dsdata - optional data from initing the new derivation step
     * @returns {Array} location path for the first fillable field.
     ******/
    SdEditor.prototype.addDSAfter = function(location, dstype, dsdata){
        var dsnames = this.getDstepNames();
        var index = location[2] | 0;
        var newname = dsdata.name || this.generateId(dstype);
        var newloc = location.slice(0, 2);
        newloc.push(index + 1);
        var type = dstype;
        if (type === 'fact' || type === 'definition' || type === 'declaration') {
            type = 'observation';
        }
        this.dsteps.splice(index + 1, 0, {id: newname, type: type});
        if (!(dsdata && dsdata.name)) {
            var data = {name: newname, type: dstype};
            switch (dstype) {
                case 'task':
                    dsdata = new Sderivation(data);
                    break;
                case 'assumption':
                    dsdata = new Sdassumption(data);
                    break;
                case 'fact':
                    dsdata = new Sdobservation(data);
                    break;
                case 'definition':
                    data.definition = {};
                    dsdata = new Sdobservation(data);
                    break;
                case 'declaration':
                    dsdata = new Sdobservation(data);
                    break;
                default:
                    break;
            };
        }
        this.dstepdata[newname] = dsdata;
        switch (dstype) {
            case 'task':
                if (dsdata.elements.task.length > 0) {
                    newloc.push('task');
                } else {
                    newloc.push('term');
                }
                newloc.push('0');
                break;
            case 'fact':
                newloc.push('motivation');
                break;
            case 'definition':
                newloc.push('definition');
                break;
            case 'declaration':
                newloc.push('declaration');
                break;
            default:
                break;
        }
        return newloc;
    };
    
    /******
     * Remove derivation step with given id
     * @param {String} dsname - The name of the derivation step to be removed
     ******/
    SdEditor.prototype.removeDStep = function(location){
        var dsnames = this.getDstepNames();
        var index = location[2] | 0;
        var dsname = dsnames[index];
        var data = this.dstepdata[dsname];
        this.dsteps.splice(index, 1);
        delete this.dstepdata[dsname];
        return data;
    }
    
    /******
     * Exec the event and record it in the undo-redo-stack.
     ******/
    SdEditor.prototype.doEvent = function(event, forceshow){
        event = this.execEvent(event, forceshow);
        if (event) {
            event.invert();
            this.undoStack.push(event);
            this.redoStack.clear();
            if (this.undoStack.size() > 0) {
                this.place.attr('data-undoredo', 'undo');
            }
        }
    }
    
    /******
     * Execution of the event. Depens on the event type.
     ******/
    SdEditor.prototype.execEvent = function(event, forceshow){
        var success = true;
        switch (event.event) {
            case 'edit':
                var location = event.loc.slice(0);
                // "Send" the new data, event.data[0], to the location,
                // get back the old data and put it to event.data[1].
                event.data[1] = this.editLocation(location, event.data[0]);
                // If caller wants the derivation to be redrawn. For example undo/redo and remote events.
                if (forceshow) {
                    this.show(event.loc);
                }
                break;
            case 'addtask':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., the first term.
                // 'derivation_step_0_term_0'
                success = (location.length > 4 &&
                           location[location.length - 2] === 'term' &&
                           location[location.length - 1] === '0');
                if (success) {
                    this.addTask(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length - 2] = 'task';
                    newloc[newloc.length - 1] = '0';
                    this.show(newloc);
                }
                break;
            case 'removetask':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., the task.
                // 'derivation_step_0_task_0'
                success = (location.length > 4 &&
                           location[location.length - 2] === 'task');
                if (success) {
                    event.data[1] = this.removeTask(location, event.data[0]);
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length-2] = 'term';
                    this.show(newloc);
                }
                break;
            case 'addstep':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., a term in
                // derivation body.
                // 'derivation_step_0_..._term_3'
                success = (location.length > 4 &&
                           location[location.length - 2] === 'term');
                if (success) {
                    this.addStep(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length - 2] = 'relation';
                    var copyloc = newloc.slice(0);
                    if (this.getLocationData(copyloc).text !== '') {
                        // If new step has nonempty relation (copied from previous),
                        // then move to the motivation.
                        newloc[newloc.length - 2] = 'motivation';
                    }
                    this.show(newloc);
                }
                break;
            case 'removestep':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., a motivation
                // or a relation, but not a motivation of observation.
                // 'derivation_step_0_..._motivation_2'
                // 'derivation_step_0_..._relation_2'
                // ! 'derivation_step_0_..._observation_1_motivation'
                success = (location.length > 4 &&
                           location[location.length -3] !== 'observation' &&
                           (location[location.length - 2] === 'motivation' || location[location.length - 2] === 'relation'));
                if (success) {
                    event.loc[location.length - 2] = 'motivation';
                    location[location.length - 2] = 'motivation';
                    event.data[1] = this.removeStep(location, event.data[0]);
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length - 2] = 'term';
                    this.show(newloc);
                }
                break;
            case 'addassumption':
                var loclength = event.loc.length;
                if (event.loc[loclength - 2] === 'task') {
                    // If the event was initiated from task, location of the new observation is as
                    // the first one (before current index 0, i.e., the index right after -1).
                    event.loc[loclength - 2] = 'assumption';
                    event.loc[loclength - 1] = '-1';
                    event.loc.push('proposition');
                }
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., an assumption.
                // 'derivation_step_0_..._assumption_1_proposition'
                success = (location.length > 5 && location[location.length - 3] === 'assumption');
                if (success) {
                    this.addAssumption(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    // The location of the new assumption is the next one.
                    // '34' | 0 === 34, but faster than parseInt();
                    newloc[newloc.length-2] = (newloc[newloc.length-2] | 0)+1;
                    this.show(newloc);
                }
                break;
            case 'removeassumption':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., an assumption.
                // 'derivation_step_0_..._assumption_1_proposition'
                success = (location.length > 5 &&
                           location[location.length - 3] === 'assumption');
                if (success) {
                    var newloc = this.getPrevLoc(location.join('_'), true);
                    event.data[1] = this.removeAssumption(location, event.data[0]);
                    this.show(newloc);
                }
                break;
            case 'addobservation':
            case 'adddefinition':
            case 'adddeclaration':
                var loclength = event.loc.length;
                if (event.loc[loclength - 2] === 'task') {
                    // If the event was initiated from task, location of the new observation is as
                    // the first one (before current index 0, i.e., the index right after -1).
                    event.loc[loclength - 2] = 'observation';
                    event.loc[loclength - 1] = '-1';
                } else if (event.loc[loclength - 3] === 'assumption') {
                    // If the event was initiated from assumption, location of the new observation is as
                    // the first one (before current index 0, i.e., the index right after -1).
                    event.loc.pop(); // Remove 'proposition' from the end.
                    loclength--;
                    event.loc[loclength - 2] = 'observation';
                    event.loc[loclength - 1] = '-1';
                };
                if (event.loc[loclength - 1] === 'proposition' && event.loc[loclength -3] === 'observation') {
                    event.loc.pop();
                    loclength = loclength - 1;
                };
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., an observation.
                // 'derivation_step_0_observation_1'
                success = (location.length > 4 &&
                           (location[location.length - 2] === 'observation' ||
                            location[location.length - 1] === 'declaration'));
                if (success) {
                    if (event.event === 'adddefinition') {
                        event.data[0].definition = {text: ''};
                        event.data[0].type = 'definition';
                    };
                    if (event.event === 'adddeclaration') {
                        event.data[0].definition = {text: ''};
                        event.data[0].type = 'declaration';
                    };
                    this.addObservation(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    // '34' | 0 === 34, but faster than parseInt();
                    newloc[newloc.length-1] = (newloc[newloc.length-1] | 0)+1;
                    if (event.data[0].type === 'definition') {
                        newloc.push('definition');
                    } else if (event.data[0].type === 'declaration') {
                        newloc.push('declaration');
                    } else {
                        newloc.push('motivation');
                    }
                    this.show(newloc);
                };
                break;
            case 'removeobservation':
                var loclength = event.loc.length;
                if (event.loc[loclength - 2] === 'observation') {
                    event.loc.push('motivation');
                }
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., a motivation of an observation.
                // 'derivation_step_0_..._observation_0_motivation'
                success = (location.length > 5 &&
                           (location[location.length - 3] === 'observation' &&
                           (location[location.length - 1] === 'motivation') ||
                            location[location.length - 1] === 'declaration'));
                if (success) {
                    var newloc = this.getPrevLoc(location.join('_'), true);
                    event.data[1] = this.removeObservation(location, event.data[0]);
                    this.show(newloc);
                }
                break;
            case 'addsubderiv':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., a motivation of an observation or a motivation of a step.
                var loclength = location.length;
                // 'derivation_step_0_motivation_1'
                // 'derivation_step_0_derivmotivation_1'
                var successMot = (loclength > 4 && (location[loclength - 2] === 'motivation' || location[loclength - 2] === 'derivmotivation'));
                // 'derivation_step_0_observation_2_motivation'
                // 'derivation_step_0_motivation'
                var successObs = ((loclength > 5 && location[loclength - 3] === 'observation' && location[loclength -1] === 'motivation') || (loclength === 4 && location[3] === 'motivation'));
                // 'derivation_step_0_motivation_1_subderivation_2_term_1'
                // 'derivation_step_0_observation_1_motivation_subderivation_2_term_1'
                // 'derivation_step_0_derivmotivation_0_subderivation_2_term_1'
                var successTerm = ((loclength > 5 && location[loclength - 4] === 'subderivation' && location[loclength -2] === 'term'));
                if (successMot || successObs) {
                    event.loc.push('subderivation');
                    event.loc.push('-1');
                    location = event.loc.slice(0);
                    loclength = loclength + 2;
                }
                if (successTerm) {
                    event.loc.pop();
                    event.loc.pop();
                    location.pop();
                    location.pop();
                    loclength = loclength - 2;
                };
                var success = (loclength > 5 &&
                                (location[loclength - 4] === 'motivation' || location[loclength - 3] === 'motivation' || location[loclength - 4] === 'derivmotivation') &&
                                location[loclength - 2] === 'subderivation') ||
                                (loclength > 6 && location[loclength - 5] === 'observation' &&
                                 location[loclength - 3] === 'motivation' &&
                                 location[loclength - 2] === 'subderivation');
                if (success) {
                    this.addSubderivation(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length - 1] = ((newloc[newloc.length - 1] | 0) + 1) + '';
                    newloc.push('term');
                    newloc.push('0');
                    this.show(newloc);
                }
                break;
            case 'removesubderiv':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., task or the first term inside a subderivation.
                // 'derivation_step_0_..._subderivation_0_task_0'
                // 'derivation_step_0_..._subderivation_0_term_0'
                var successTaskTerm = (location.length > 5 &&
                           location[location.length - 4] === 'subderivation' &&
                           (location[location.length - 2] === 'task' ||
                            (location[location.length - 2] === 'term' &&
                             location[location.length - 1] === '0')));
                if (successTaskTerm) {
                    event.loc.pop();
                    event.loc.pop();
                    location.pop();
                    location.pop();
                }
                success = (location.length > 3 && location[location.length - 2] === 'subderivation');
                if (success) {
                    var oldloc = this.editorelem.find('[data-loc^="'+location.join('_')+'"]').eq(1).attr('data-loc');
                    var newloc = this.getPrevLoc(oldloc, true);
                    event.data[1] = this.removeSubderivation(location, event.data[0]);
                    this.show(newloc);
                }
                break;
            case 'addderivationmotivation':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., the task.
                // 'derivation_step_0_..._task_0'
                success = (location.length > 4 &&
                           location[location.length - 2] === 'task' &&
                           location[location.length - 1] === '0');
                if (success) {
                    this.addDerivMotivation(location, event.data[0]);
                    event.data[1] = {};
                    var newloc = event.loc.slice(0);
                    newloc[newloc.length - 2] = 'derivmotivation';
                    this.show(newloc);
                }
                break;
            case 'removederivationmotivation':
                var location = event.loc.slice(0);
                // Check that the format of location is correct, i.e., derivationmotivation.
                // 'derivation_step_0_..._derivmotivation_0'
                success = (location.length > 4 &&
                           location[location.length - 2] === 'derivmotivation' &&
                           location[location.length - 1] === '0');
                if (success) {
                    var newloc = this.getPrevLoc(location.join('_'), true);
                    event.data[1] = this.removeDerivMotivation(location, event.data[0]);
                    this.show(newloc);
                }
                break;
            case 'showhidesubderivation':
                var location = event.loc.slice(0);
                var newloc = event.loc.slice(0);
                var loclen = location.length;
                // Check that the format of location is correct, i.e., motivation.
                // 'derivation_step_0_..._motivation'
                // 'derivation_step_0_..._motivation_0'
                success = (loclen > 3 && (location[loclen-1] === 'motivation' || location[loclen-2] === 'motivation'));
                if (success) {
                    var locstring = event.loc.join('_');
                    event.data[1] = this.showHideLocation(location, event.data[0]);
                    var $tbody = this.place.find('[data-loc="'+locstring+'"]').closest('tbody');
                    var $elemdiv = $tbody.children('tr').children('[data-loc^="'+locstring+'_subderivation"]').children('.subderivation');
                    if (event.data[0]) {
                        $elemdiv.addClass('sdsubhidden');
                    } else {
                        $elemdiv.removeClass('sdsubhidden');
                    }
                    this.setFocus(newloc);
                }
                break;
            case 'addsteptask':
                var location = event.loc.slice(0);
                var dstype = 'task';
                event.data[0].type = dstype;
                var newloc = this.addDSAt(location, dstype, event.data[0]);
                event.loc = newloc.slice(0,3);
                // event.loc[0] === after derivation step with this id
                // event.loc[1] === id of the new derivation step
                event.data[1] = {type: dstype};
                this.show(newloc);
                break;
            case 'addstepassumption':
                var location = event.loc.slice(0);
                var dstype = 'assumption';
                event.data[0].type = dstype;
                var newloc = this.addDSAt(location, dstype, event.data[0]);
                event.loc = newloc.slice(0,3);
                // event.loc[0] === after derivation step with this id
                // event.loc[1] === id of the new derivation step
                event.data[1] = {type: dstype};
                this.show(newloc);
                break;
            case 'addstepfact':
                var location = event.loc.slice(0);
                var dstype = 'fact';
                event.data[0].type = dstype;
                var newloc = this.addDSAt(location, dstype, event.data[0]);
                event.loc = newloc.slice(0,3);
                // event.loc[0] === after derivation step with this id
                // event.loc[1] === id of the new derivation step
                event.data[1] = {type: dstype};
                this.show(newloc);
                break;
            case 'addstepdefinition':
                var location = event.loc.slice(0);
                var dstype = 'definition';
                event.data[0].type = dstype;
                var newloc = this.addDSAt(location, dstype, event.data[0]);
                event.loc = newloc.slice(0,3);
                // event.loc[0] === after derivation step with this id
                // event.loc[1] === id of the new derivation step
                event.data[1] = {type: dstype};
                this.show(newloc);
                break;
            case 'addstepdeclaration':
                var location = event.loc.slice(0);
                var dstype = 'declaration';
                event.data[0].type = dstype;
                var newloc = this.addDSAt(location, dstype, event.data[0]);
                event.loc = newloc.slice(0,3);
                // event.loc[0] === after derivation step with this id
                // event.loc[1] === id of the new derivation step
                event.data[1] = {type: dstype};
                this.show(newloc);
                break;
            case 'removedstep':
            case 'removesteptask':
            case 'removestepassumption':
            case 'removestepfact':
            case 'removestepdefinition':
            case 'removestepdeclaration':
                var location = event.loc.slice(0, 3);
                var dstepname = location[0];
                var dsnames = this.getDstepNames();
                var index = dsnames.indexOf(dstepname);
                var newindex = Math.max(index - 1, 0);
                var prevname = dsnames[index - 1] || '';
                var data = this.removeDStep(location);
                event.data[1] = data;
                event.data[0].type = event.data[0].type || event.data[1].type;
                event.loc = location;
                this.show();
                break;
            case 'movedstep':
                var indexfrom = event.data[0].indexfrom;
                var indexto = event.data[0].indexto;
                var element;
                if (indexfrom !== indexto) {
                    element = this.dsteps.splice(indexfrom, 1);
                    this.dsteps.splice(indexto, 0, element[0]);
                };
                this.show();
                break;
            case 'setqtype':
                var qtype = event.data[0].qtype;
                var location = event.loc.slice(0);
                event.data[1] = this.setQtype(location, qtype);
                break;
            case 'undo':
                success = true;
                this.undo();
                break;
            case 'redo':
                success = true;
                this.redo();
                break;
            default:
                break;
        }
        if (this.editable) {
            this.changed();
        };
        return success && event;
    }

    /******
     * Undo the last event.
     ******/
    SdEditor.prototype.undo = function(){
        if (this.undoStack.size() > 0) {
            var event = this.undoStack.pop();
            event = this.execEvent(event, true);
            event.invert();
            this.redoStack.push(event);
        }
        var undoredo = [];
        if (this.undoStack.size() > 0) {
            undoredo.push('undo');
        }
        if (this.redoStack.size() > 0) {
            undoredo.push('redo');
        }
        this.place.attr('data-undoredo', undoredo.join(' '));
    }
    
    /******
     * Redo the last undone event.
     ******/
    SdEditor.prototype.redo = function(){
        if (this.redoStack.size() > 0) {
            var event = this.redoStack.pop();
            event = this.execEvent(event, true);
            event.invert();
            this.undoStack.push(event);
        }
        var undoredo = [];
        if (this.undoStack.size() > 0) {
            undoredo.push('undo');
        }
        if (this.redoStack.size() > 0) {
            undoredo.push('redo');
        }
        this.place.attr('data-undoredo', undoredo.join(' '));
    }
    
    /******
     * Create a new event.
     * Mainly for testing.
     ******/
    SdEditor.prototype.createEvent = function(event, loc, data){
        var aaa = new SdEditevent(event, loc, data);
        return aaa;
    }
    
    /******
     * Slide the button bar near the focus.
     ******/
    SdEditor.prototype.slideButtons = function(){
        var current = this.place.data('currentmq') || this.place.data('currentplaceholder');
        var dsplace = current.closest('.sdeditor-derivationstep, .sdeditor-dstepplaceholder');
        var curroffset;
        if (current.is('.sdeditor-dstepplaceholder')) {
            curroffset = dsplace.position().top;
        } else {
            curroffset = current.position().top + dsplace.position().top;
        };
        var buttonbarheight = this.editorpanel.height();
        var buttonlist = this.editorpanel.find('ul.sdeditor-buttonlist');
        var buttonlistheight = buttonlist.height() + 10;
        var newmargin = Math.min(buttonbarheight - buttonlistheight, curroffset);
        this.buttonmargin = newmargin;
        // Animate buttonslide with css3 transition.
        buttonlist.css('margin-top', newmargin);
    }
    
    /******
     * Show plugin menubar
     ******/
    SdEditor.prototype.showPlugins = function(){
        var uilang = this.settings.uilang || 'en';
        var pluginhtml = [];
        var pluginlist = [];
        for (var plug in SdEditor.plugins) {
            SdEditor.plugins[plug].name = plug;
            var pluginmodes = SdEditor.plugins[plug].modes;
            if (pluginmodes.length === 0 || pluginmodes.indexOf(this.settings.mode) !== -1) {
                pluginlist.push(SdEditor.plugins[plug]);
            }
        }
        pluginlist.sort(function(a, b){
            a.order = a.order || 0;
            b.order = b.order || 0;
            return a.order - b.order;
        });
        for (var i = 0; i < pluginlist.length; i++) {
            var plugin = pluginlist[i];
            pluginhtml.push('<li class="sdeditor-menuitem sdeditor-plugins"><div tabindex="0">'+escapeHTML(plugin.label[uilang] ||plugin.label['en'])+'</div><ul class="sdeditor-menulist">');
            for (j = 0; j < plugin.items.length; j++) {
                var item = plugin.items[j];
                var titletext = escapeHTML(item.description[uilang] || item.description['en'] || '');
                pluginhtml.push('<li title="'+titletext+'" data-pluginname="'+escapeHTML(plugin.name)+'" data-pluginitem="'+j+'">'+sanitize(item.icon) + ' '+escapeHTML(item.label[uilang] || item.label['en'])+'</li>');
            }
            pluginhtml.push('</ul></li>');
        }
        this.place.children('.sdeditor-menubar').html(pluginhtml.join(''));
        if (pluginlist.length === 0) {
            this.place.children('.sdeditor-menubar').addClass('sdeditor-plugins-empty');
        }
    }

    /******
     * Render the derivation (edit or view mode). Removed: and bind required handlers.
     ******/
    SdEditor.prototype.show = function(focusloc){
        if (this.editable) {
            this.place.css('position','relative');
            if (this.dsteps.length === 0 && this.type === 'sderivation') {
                this.place.addClass('sdeditor-nodsteps');
            } else {
                this.place.removeClass('sdeditor-nodsteps');
            }
        }
        var uilang = this.settings.uilang || 'en';
        this.place.attr('data-elementmode', this.settings.mode);
        var templ = (this.editable ? 'edit' : 'view');
        this.place.empty();
        this.place.html(SdEditor.template[templ]);
        this.showPlugins();
        this.editorelem = this.place.find('.sdeditor');
        this.editorelem.attr('data-loc', this.locname);
        var html = [];
        var dstep, htmltext;
        var loc;
        var haschecker = !!(SdEditor.plugins['Checker'] && SdEditor.plugins['Checker'].modes.indexOf(this.settings.mode) !== -1);
        for (var i = 0, len = this.dsteps.length; i < len; i++) {
            dstep = this.dsteps[i];
            loc = [this.locname, 'step', i];
            html.push('<div class="sdeditor-dstepplaceholder" tabindex="0" data-stepid="'+escapeHTML(dstep.id)+'" data-loc="'+escapeHTML(loc.join('_'))+'" data-dstepindex="'+i+'"></div>');
            html.push('<div class="sdeditor-derivationstep" data-stepid="'+escapeHTML(dstep.id)+'" data-loc="'+escapeHTML(loc.join('_'))+'">');
            html.push('<div class="sdeditor-derivationstep-draghandle" draggable="true" data-dstepindex="'+i+'"></div>');
            if (this.editable) {
                html.push('<div class="sdeditor-removestep-button" data-dstepindex="'+i+'"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg></div>');
            };
            htmltext = this.dstepdata[dstep.id].getHtml(!this.editable, loc, true /*isdstep*/, uilang, haschecker);
            html.push(htmltext);
            html.push('</div>');
        }
        html.push('<div class="sdeditor-dstepplaceholder" data-loc="'+escapeHTML(this.name.replace(/[-_]/g, ''))+'_step_'+i+'" tabindex="0"></div>')
        this.editorelem.html(html.join('\n'));
        this.place.attr('data-sdtheme', this.settings.theme);
        this.editorpanel = this.place.find('.sdeditor-buttonbar');
        // Render math with MathQuill
        if (this.reviewable) {
            // Reviewable is always non-editable!
            this.place.find('.sdmath').not('.mathquill-rendered-math').mathquill('embedded-latex');
            this.place.find('.sdmathtext .sdmath').not('.mathquill-rendered-math').mathquill('embedded-latex');
        } else {
            this.place.find('.sdmath[data-sdreadonly="true"]').not('.mathquill-rendered-math').mathquill('embedded-latex');
            this.place.find('.sdmath[data-sdreadonly="false"]').not('.mathquill-rendered-math').mathquill('editable');
            this.place.find('.sdmathtext[data-sdreadonly="true"] .sdmath').not('.mathquill-rendered-math').mathquill('embedded-latex');
            this.place.find('.sdmathtext[data-sdreadonly="false"]').not('.mathquill-rendered-math, .mathquill-textbox').mathquill('textbox');
        }
        //this.initHandlers();
        this.initButtons();
        if (focusloc) {
            this.setFocus(focusloc);
        }
        if (this.reviewable) {
            if (this.marksnote && this.marksnote.length) {
                this.marksnote.remove();
            }
            this.marksnote = false;
            this.editMarkings();
        } else if (this.markings && typeof(this.markings.size) === 'function' && this.markings.size() > 0 || this.markings.general) {
            if (this.marksnote && this.marksnote.length) {
                this.marksnote.remove();
            }
            this.marksnote = false;
            this.place.find('[data-noteloc]').empty();
            this.showMarkings();
        }
        if (!this.editable && !this.reviewable && this.settings.stepmode) {
            this.startStepping();
        }
    }
    
    /******
     * Start showing the derivation in step-by-step-mode.
     ******/
    SdEditor.prototype.startStepping = function(){
        var editor = this;
        var uilang = this.settings.uilang;
        this.editorelem.append('<div class="sdeditor-afterbar ffwidget-buttonset ffwidget-horizontal"><span class="sdeditor-showallstep sdeditor-stepbutton ffwidget-setbutton" title="'+Dictionary.localize('steps:showallsteps', uilang)+'">'+SdEditor.icons.stepall+'</span><span class="sdeditor-shownextstep sdeditor-stepbutton ffwidget-setbutton" title="'+Dictionary.localize('steps:shownextstep', uilang)+'">'+SdEditor.icons.stepnext+'</span></div><div class="sdeditor-steppingstore" style="display:none;"><span class="sdeditor-hideprevstep sdeditor-stepbutton ffwidget-setbutton" title="'+Dictionary.localize('steps:hidelaststep', uilang)+'">'+SdEditor.icons.stepprev+'</span></div>');
        this.afterbar = this.editorelem.children('.sdeditor-afterbar');
        this.stepperstore = this.editorelem.children('.sdeditor-steppingstore');
        this.editorelem.css('min-height', this.editorelem.height());
        this.steplist = this.place.find('.sdlayouttable tr:not(.sdsubhidden tr)');
        this.steplist.addClass('sdstepbystephidden');
        this.stepByStepCurrent = 0;
        this.steplist.eq(0).removeClass('sdstepbystephidden');
        this.place.off('contextmenu.stepbystep', '.sdeditor-container').on('contextmenu.stepbystep', '.sdeditor-container', function(e){return false;});
        this.place.off('mousedown.stepbystep', '.sdeditor-container').on('mousedown.stepbystep', '.sdeditor-container', function(e){
            if (!$(e.target).hasClass('sdeditor-stepbutton')) {
                switch (e.which){
                    case 1:
                        editor.showNextStep();
                        editor.place.focus();
                        break;
                    case 3:
                        editor.hidePrevStep();
                        editor.place.focus();
                        break;
                    default:
                        break;
                }
            }
        });
        this.place.find('.mathquill-rendered-math').off('mousedown.stepbystep').on('mousedown.stepbystep', function(e){
            switch (e.which){
                case 1:
                    editor.showNextStep();
                    editor.place.focus();
                    break;
                case 3:
                    editor.hidePrevStep();
                    editor.place.focus();
                    break;
                default:
                    break;
            }
        });
        this.place.off('keydown.stepbystep').on('keydown.stepbystep', function(e){
            switch (e.which){
                case 32:
                case 34:
                case 13:
                    e.stopPropagation();
                    e.preventDefault();
                    editor.showNextStep();
                    break;
                case 8:
                case 33:
                    e.stopPropagation();
                    e.preventDefault();
                    editor.hidePrevStep();
                    break;
                default:
                    break;
            }
        });
        this.place.off('mousedown.stepbystepbutton', '.sdeditor-stepbutton').on('mousedown.stepbystepbutton', '.sdeditor-stepbutton', function(e){
            e.stopPropagation();
        }).off('mouseup.stepbystepbutton', '.sdeditor-shownextstep').on('mouseup.stepbystepbutton', '.sdeditor-shownextstep', function(e){
            e.stopPropagation();
            e.preventDefault();
            editor.showNextStep();
        }).off('mouseup.stepbystepbutton', '.sdeditor-hideprevstep').on('mouseup.stepbystepbutton', '.sdeditor-hideprevstep', function(e){
            e.stopPropagation();
            e.preventDefault();
            editor.hidePrevStep();
        }).off('mousedown.stepbystepbutton', '.sdeditor-hideprevstep').on('mousedown.stepbystepbutton', '.sdeditor-hideprevstep', function(e){
            e.stopPropagation();
            e.preventDefault();
        }).off('mouseup.stepbystepbutton', '.sdeditor-showallstep').on('mouseup.stepbystepbutton', '.sdeditor-showallstep', function(e){
            e.stopPropagation();
            e.preventDefault();
            editor.showAllSteps();
        });
    }
    
    /******
     * Show next step in step-by-step-mode
     ******/
    SdEditor.prototype.showNextStep = function(){
        if (this.stepByStepCurrent === 0) {
            this.afterbar.prepend(this.stepperstore.children('.sdeditor-hideprevstep'));
        }
        if (this.stepByStepCurrent !== this.steplist.length - 1) {
            this.steplist.eq(++this.stepByStepCurrent).removeClass('sdstepbystephidden');
            while (this.steplist.eq(this.stepByStepCurrent + 1).is('.sdsubderivationrow, .sdsubhidden tr')){
                this.steplist.eq(++this.stepByStepCurrent).removeClass('sdstepbystephidden');
            }
            if (this.stepByStepCurrent === this.steplist.length - 2) {
                this.steplist.eq(++this.stepByStepCurrent).removeClass('sdstepbystephidden');
            }
        }
        if (this.stepByStepCurrent === this.steplist.length - 1) {
            this.stepperstore.append(this.afterbar.children('.sdeditor-showallstep, .sdeditor-shownextstep'));
            this.afterbar.addClass('laststep');
        }
    }
    
    /******
     * Show all steps in step-by-step-mode
     ******/
    SdEditor.prototype.showAllSteps = function(){
        this.steplist.removeClass('sdstepbystephidden');
        this.stepByStepCurrent = this.steplist.length - 1;
        this.afterbar.prepend(this.stepperstore.children('.sdeditor-hideprevstep'));
        this.stepperstore.append(this.afterbar.children('.sdeditor-showallstep, .sdeditor-shownextstep'));
        this.afterbar.addClass('laststep');
    }
    
    /******
     * Hide the last visible step in step-by-step-mode.
     ******/
    SdEditor.prototype.hidePrevStep = function(){
        if (this.stepByStepCurrent === this.steplist.length - 1) {
            this.afterbar.append(this.stepperstore.children('.sdeditor-showallstep, .sdeditor-shownextstep'));
            this.afterbar.removeClass('laststep');
        }
        if (this.stepByStepCurrent === this.steplist.length -1) {
            this.steplist.eq(this.stepByStepCurrent--).addClass('sdstepbystephidden');
        }
        if (this.stepByStepCurrent !== 0) {
            while (this.steplist.eq(this.stepByStepCurrent).is('.sdsubderivationrow, .sdsubhidden tr')){
                this.steplist.eq(this.stepByStepCurrent--).addClass('sdstepbystephidden');
            }
            this.steplist.eq(this.stepByStepCurrent--).addClass('sdstepbystephidden');
        }
        if (this.stepByStepCurrent === 0) {
            this.stepperstore.prepend(this.afterbar.children('.sdeditor-hideprevstep'));
        }
    }
    
    /******
     * Update markings with show or edit.
     ******/
    SdEditor.prototype.updateMarkings = function(){
        if (this.reviewable) {
            this.editMarkings();
        } else {
            this.showMarkings();
        }
    }
    
    /******
     * Show markings in view mode.
     ******/
    SdEditor.prototype.showMarkings = function(locs){
        if (typeof(locs) === 'undefined') {
            locs = this.markings.getLocations();
        }
        var sdeditor = this;
        var clipper = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="24" viewBox="0 0 50 24" class="sdimg-clipper"><path style="stroke: none; fill: rgba(0,0,0,0.5);" transform="rotate(90, 50, 50), translate(0,50)" d="M3 10 l0 -2 a5 5 0 0 1 5 -5 l8 0 a5 5 0 0 1 5 5 l0 34 a4 4 0 0 1 -4 4 l-6 0 a4 4 0 0 1 -4 -4 l0 -27 l3 0 l0 26 a2 2 0 0 0 2 2 l4 0 a2 2 0 0 0 2 -2 l0 -32 a3 3 0 0 0 -3 -3 l-6 0 a3 3 0 0 0 -3 3 l0 1z" /><path style="stroke: none; fill: #a00;" transform="rotate(90, 50, 50), translate(0,50)" d="M2 10 l0 -3 a5 5 0 0 1 5 -5 l8 0 a5 5 0 0 1 5 5 l0 34 a4 4 0 0 1 -4 4 l-6 0 a4 4 0 0 1 -4 -4 l0 -27 l3 0 l0 26 a2 2 0 0 0 2 2 l4 0 a2 2 0 0 0 2 -2 l0 -32 a3 3 0 0 0 -3 -3 l-6 0 a3 3 0 0 0 -3 3 l0 2z" /></svg>';
        var dsteps = this.place.find('.sdeditor-derivationstep');
        var dsloc, dsname, marktype, marksymbol;
        for (var i = 0, len = this.dsteps.length; i < len; i++) {
            dsloc = this.locname + '_step_' + i;
            dsname = this.dsteps[i].id;
            marktype = this.markings.getMark(dsloc);
            marksymbol = marktype ? this.markings.getSymbol(marktype) : '';
            dsteps.eq(i).children('.sdmarkings-available').remove();
            dsteps.eq(i).append([
                '<div class="sdmarkings-available sdmarkings-dstepnote'+(this.marksvisible ? '' : ' sdmarkings-hidden')+'">',
                '    <div class="sdmarkings-note" data-marktype="'+marktype+'">',
                '        <div class="sdmarkings-generalnote" data-noteloc="'+escapeHTML(dsloc)+'" data-marktype="'+marktype+'">',
                '            <span class="sdcheckmark">'+marksymbol+'</span> '+this.markings.getComment(dsloc),
                '        </div>',
                '    </div>',
                //'    <div class="sdmarkings-clipper">'+clipper+'</div>',
                '</div>'
            ].join(''));
        };
        // General feedback for whole derivation (only show if non-empty)
        if (!!this.markings.getComment(this.locname)) {
            var sdeddiv = this.place.find('.sdeditor')
            sdeddiv.children('.sdeditor-generalfeedback').remove();
            marktype = this.markings.getMark(this.locname);
            marksymbol = marktype ? this.markings.getSymbol(marktype) : '';
            sdeddiv.append([
                '<div class="sdeditor-generalfeedback sdmarkings-available sdmarkings-generalfeedback' + (this.marksvisible ? '' : ' sdmarkings-hidden') + '">',
                '    <div class="sdmarkings-note">',
                '        <div class="sdmarkings-generalnote" data-noteloc="' + escapeHTML(this.locname) + '" data-marktype="' + marktype + '">',
                '            <span class="sdcheckmark">' + marksymbol + '</span> <span class="sdcheckcomment">' + this.markings.getComment(this.locname) + '</span>',
                '        </div>',
                '    </div>',
                '    <div class="sdmarkings-clipper">' + clipper + '</div>',
                '</div>'
            ].join(''));
        }
        this.marksnote = this.place.find('.sdmarkings-available');
        if (this.marksvisible) {
            for (var i = 0; i < locs.length; i++) {
                var loc = locs[i];
                var place = this.place.find('[data-noteloc="'+loc+'"]:not(.sdmarkings-generalnote)');
                if (this.markings.messageExists(loc) && this.markings.getMark(loc) !== 'empty') {
                    this.markings.showMark(loc, place);
                } else {
                    place.empty();
                }
            }
            this.marksnote.removeClass('sdmarkings-hidden');
        } else {
            //this.marksnote.remove();
            //this.marksnote = false;
            this.place.find('[data-noteloc]:not(.sdmarkings-generalnote)').empty();
            this.marksnote.addClass('sdmarkings-hidden');
        }
    }
    
    /******
     * Clear all markings from screen
     ******/
    SdEditor.prototype.clearShownMarkings = function(){
        this.place.find('[data-noteloc]').empty();
        if (this.marksnote) {
            this.marksnote.remove();
            this.marksnote = false;
        };
    };
    
    /******
     * Edit markings in review mode.
     ******/
    SdEditor.prototype.editMarkings = function(){
        var sdeditor = this;
        //var generalLoc = this.derivation.name;
        var generalComment;
        var generalMark;
        var sdgeneral;
        if (!this.marksnote) {
            var clipper = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="24" viewBox="0 0 50 24" class="sdimg-clipper"><path style="stroke: none; fill: rgba(0,0,0,0.5);" transform="rotate(90, 50, 50), translate(0,50)" d="M3 10 l0 -2 a5 5 0 0 1 5 -5 l8 0 a5 5 0 0 1 5 5 l0 34 a4 4 0 0 1 -4 4 l-6 0 a4 4 0 0 1 -4 -4 l0 -27 l3 0 l0 26 a2 2 0 0 0 2 2 l4 0 a2 2 0 0 0 2 -2 l0 -32 a3 3 0 0 0 -3 -3 l-6 0 a3 3 0 0 0 -3 3 l0 1z" /><path style="stroke: none; fill: #a00;" transform="rotate(90, 50, 50), translate(0,50)" d="M2 10 l0 -3 a5 5 0 0 1 5 -5 l8 0 a5 5 0 0 1 5 5 l0 34 a4 4 0 0 1 -4 4 l-6 0 a4 4 0 0 1 -4 -4 l0 -27 l3 0 l0 26 a2 2 0 0 0 2 2 l4 0 a2 2 0 0 0 2 -2 l0 -32 a3 3 0 0 0 -3 -3 l-6 0 a3 3 0 0 0 -3 3 l0 2z" /></svg>';
            var dsteps = this.place.find('.sdeditor-derivationstep');
            var dsloc, dsname, generalhtml;
            for (var i = 0, len = this.dsteps.length; i < len; i++){
                dsloc = this.locname + '_step_' + i;
                dsname = this.dsteps[i].id;
                generalComment = escapeHTML(this.markings.getComment(dsloc));
                generalMark = escapeHTML(this.markings.getMark(dsloc) || 'empty');
                generalhtml = [
                    '<div class="sdmarkings-available"><div class="sdmarkings-note" data-marktype="'+generalMark+'">',
                    '<span class="sdcorrectionmarks">',
                    '<span class="sdmark sdcorrect" data-sdmark="correct" title="Correct"><span>'+this.markings.getSymbol('correct')+'</span></span>',
                    '<span class="sdmark sdwrong" data-sdmark="wrong" title="Wrong"><span>'+this.markings.getSymbol('wrong')+'</span></span>',
                    '<span class="sdmark sdnote" data-sdmark="note" title="Note"><span>'+this.markings.getSymbol('note')+'</span></span>',
                    '<span class="sdmark sdempty" data-sdmark="empty" title="Remove comment"><span>'+this.markings.getSymbol('empty')+'</span></span>',
                    '</span>',
                    '<div class="sdmarkings-generalnote" data-noteloc="'+escapeHTML(dsloc)+'">'+generalComment+'</div></div>',
                    //'<div class="sdmarkings-clipper">'+clipper+'</div></div>'
                ].join('\n');
                dsteps.eq(i).append(generalhtml);
                //sdgeneral = this.marksnote.find('[data-noteloc]');
                dsteps.eq(i).find('.sdmarkings-generalnote').mathquill('revert').html(generalComment).mathquill('textbox');
            };
            generalcomment = escapeHTML(this.markings.getComment(this.locname));
            generalMark = escapeHTML(this.markings.getMark(this.locname) || 'empty');
            generalhtml = [
                '<div class="sdeditor-generalfeedback sdmarkings-available sdmarkings-generalfeedback"><div class="sdmarkings-note" data-marktype="'+generalMark+'">',
                '<span class="sdcorrectionmarks">',
                '<span class="sdmark sdcorrect" data-sdmark="correct" title="Correct"><span>'+this.markings.getSymbol('correct')+'</span></span>',
                '<span class="sdmark sdwrong" data-sdmark="wrong" title="Wrong"><span>'+this.markings.getSymbol('wrong')+'</span></span>',
                '<span class="sdmark sdnote" data-sdmark="note" title="Note"><span>'+this.markings.getSymbol('note')+'</span></span>',
                '<span class="sdmark sdempty" data-sdmark="empty" title="Remove comment"><span>'+this.markings.getSymbol('empty')+'</span></span>',
                '</span>',
                '<div class="sdmarkings-generalnote" data-noteloc="'+escapeHTML(this.locname)+'">'+generalcomment+'</div></div>',
                '<div class="sdmarkings-clipper">'+clipper+'</div></div>'
            ].join('\n');
            this.place.find('.sdeditor').append(generalhtml)
            this.place.find('.sdeditor').find('.sdmarkings-generalfeedback .sdmarkings-generalnote').mathquill('revert').html(generalcomment).mathquill('textbox');
            this.marksnote = this.place.find('.sdmarkings-available');
        }
        var $markPlaces = this.place.find('td[data-noteloc]');
        for (var i = 0, length = $markPlaces.length; i < length; i++) {
            var $mplace = $markPlaces.eq(i);
            var loc = $mplace.attr('data-noteloc');
            var mark = this.markings.getMark(loc) || 'empty';
            var html = [
                '<div class="sdmanualreviewwrapper"><div class="sdmanualreviewbox" data-marktype="'+mark+'"><span class="sdcorrectionmarks">',
                '<span class="sdmark sdcorrect" data-sdmark="correct" title="Correct"><span>'+this.markings.getSymbol('correct')+'</span></span>',
                '<span class="sdmark sdwrong" data-sdmark="wrong" title="Wrong"><span>'+this.markings.getSymbol('wrong')+'</span></span>',
                '<span class="sdmark sdnote" data-sdmark="note" title="Note"><span>'+this.markings.getSymbol('note')+'</span></span>',
                '<span class="sdmark sdempty" data-sdmark="empty" title="Remove comment"><span>'+this.markings.getSymbol('empty')+'</span></span>',
                '</span><input type="text" class="sdmanualreviewcomment" value="'+(this.markings.getComment(loc) || '')+'" /><button class="sdmanualreviewready">OK</button>',
                '</div></div>'
            ].join('\n');
            $mplace.empty().html(html);
        }
        //var generalLoc = sdgeneral.attr('data-noteloc');
    }
    
    /******
     * Set markings visibility.
     ******/
    SdEditor.prototype.setMarksVisible = function(visible){
        this.marksvisible = visible;
    }
    
    /******
     * Get markings.
     ******/
    SdEditor.prototype.getMarkings = function(){
        return this.markings.getData();
    }

    /******
     * Set markings.
     ******/
    SdEditor.prototype.setMarkings = function(markings){
        //markings.dstepnames = this.getDstepNames();
        this.markings = new SdMarkings(markings);
        this.markings.setLang(this.settings.uilang);
    }
    
    /******
     * Add several markings
     * @param {Array} messages - Array of messages to add as markings
     * @returns {Array} an array of location id's of all messages
     ******/
    SdEditor.prototype.addMarkings = function(markings){
        var locs = [];
        if (markings.messages && typeof(markings.messages.length) === 'number') {
            var messages = markings.messages;
            for (var i = 0, len = messages.length; i < len; i++) {
                this.addMarking(messages[i]);
                if (messages[i].loc) {
                    locs.push(messages[i].loc)
                }
            }
        }
        return locs;
    }
    
    /******
     * Add marking message
     * @param {Object} options - A message object
     *   @param {String} options.loc - A location identifier
     *   @param {String} options.mark - A mark type (correct, wrong, note, empty)
     *   @param {String} options.comment - Marking message text
     *****/
    SdEditor.prototype.addMarking = function(options){
        options = $.extend({loc: '', mark: 'empty', comment: ''}, options);
        this.markings.addMessage(options);
    }
    
    /******
     * Clear a list of markings
     * @param {Array} locs - list of locations to clear from markings
     ******/
    SdEditor.prototype.clearMarkings = function(locs){
        if (typeof(locs.length) === 'number') {
            for (var i = 0, len = locs.length; i < len; i++) {
                this.clearMarking(locs[i]);
            };
        };
    };
    
    /******
     * Clear marking - remove a message from given location
     * @param {String} loc - A location identifier
     ******/
    SdEditor.prototype.clearMarking = function(loc){
        this.markings.clearMarking(loc);
    }
    
    /******
     * Clear all markings
     ******/
    SdEditor.prototype.clearReview = function(){
        this.markings.clearAll();
    }
    
    /******
     * Trigger event about review changing
     ******/
    SdEditor.prototype.reviewChanged = function(){
        this.place.trigger('review_changed');
        var data = {
            id: this.name,
            reviewdata: this.getMarkings()
        };
        this.place.trigger('elementreview_changed', data);
    }
    
    /******
     * Get the comment for location loc
     * @param {String} loc - Location identifier
     * @returns {String} Comment text for given location
     ******/
    SdEditor.prototype.getMarksComment = function(loc){
        return this.markings.getComment('loc');
    }

    /******
     * Get the mark for location loc
     * @param {String} loc - Location identifier
     * @returns {String} Mark name for given location
     ******/
    SdEditor.prototype.getMarksMark = function(loc){
        return this.markings.getMark('loc');
    }
    
    /******
     * Get the list of id's of all derivation steps.
     * @returns {Array} list of id's
     ******/
    SdEditor.prototype.getDstepNames = function(){
        var result = [];
        for (var i = 0, len = this.dsteps.length; i < len; i++) {
            result.push(this.dsteps[i].id);
        };
        return result;
    }
    
    /**
     * Init event handlers
     */
    SdEditor.prototype.initHandlers = function() {
        this.removeHandlers();
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersNonedit();
        };
        if (this.reviewable) {
            this.initHandlersReview();
        };
    };

    /**
     * Remove all event handlers
     * removeHandlers
     */
    SdEditor.prototype.removeHandlers = function() {
        this.place.off();
    };
    /******
     * Bind event handlers for keypresses etc.
     ******/
    SdEditor.prototype.initHandlersCommon = function(){
        var sdeditor = this;
        //this.place.off('keydown.sdeditorundo').on('keydown.sdeditorundo', '.sdeditor', function(e){
        //    switch (e.which) {
        //        // Undo (ctrl+z) and Redo (shift+ctrl+z)
        //        case 90:
        //            if (e.ctrlKey) {
        //                if (e.shiftKey) {
        //                    //var $current = sdeditor.place.find('td.currentFocus');
        //                    //var loc = $current.attr('data-loc');
        //                    //var loc = [''];
        //                    var event = new SdEditevent('redo', [], '');
        //                    sdeditor.execEvent(event);
        //                } else {
        //                    //var $current = sdeditor.place.find('td.currentFocus');
        //                    //$(this).focusout().focus();
        //                    //var loc = $current.attr('data-loc');
        //                    var event = new SdEditevent('undo', [], '');
        //                    sdeditor.execEvent(event);
        //                }
        //            }
        //            break;
        //        default:
        //            break;
        //    };
        //});
        this.place.off('get getdata').on('get getdata', function(e){
            var data = sdeditor.getData();
            $(this).data('sdeditor', data);
            $(this).data('[[elementdata]]', data);
        }).off('set').on('set', function(e, options){
            var settings = options.settings;
            if (options.data && options.data.main) {
                options = options.data;
            };
            if (options.main) {
                options = convertOldToNew(options);
                options.settings = settings;
            };
            sdeditor.setSettings(options);
            sdeditor.setMetadata(options);
            sdeditor.setData(options);
            sdeditor.init();
            sdeditor.show();
        }).off('getreview').on('getreview', function(e){
            var markings = sdeditor.getMarkings();
            $(this).trigger('reviewdata', [markings]).data('sdeditorreview', markings);
        }).off('setmode').on('setmode', function(e, mode){
            sdeditor.setMode(mode);
            sdeditor.show();
        }).off('edit').on('edit', function(){
            sdeditor.setMode('edit');
            sdeditor.show();
        }).off('refresh').on('refresh', function(){
            sdeditor.show();
        }).off('view').on('view', function(){
            sdeditor.setMode('view');
            sdeditor.show();
            sdeditor.place.attr('data-focusonelem','');
        }).off('show').on('show', function(){
            sdeditor.setMode('view');
            sdeditor.show();
            sdeditor.place.attr('data-focusonelem','');
        }).off('review').on('review', function(e, data){
            sdeditor.setMode('review');
            if (data) {
                sdeditor.setMarkings(data);
            }
            sdeditor.show();
            sdeditor.place.attr('data-focusonelem','');
        }).off('stepbystep').on('stepbystep', function(e, mode){
            sdeditor.settings.stepmode = mode;
            sdeditor.show();
        }).off('setreview').on('setreview', function(e, data){
            sdeditor.setMarkings(data);
            sdeditor.clearShownMarkings();
            if (sdeditor.reviewable) {
                sdeditor.editMarkings();
            } else {
                sdeditor.showMarkings();
            };
        }).off('setmarkings').on('setmarkings', function(e, data){
            sdeditor.setMarkings(data);
            sdeditor.clearShownMarkings();
            if (sdeditor.reviewable) {
                sdeditor.editMarkings();
            } else {
                sdeditor.showMarkings();
            };
        }).off('addmarkings').on('addmarkings', function(e, data){
            var locs = sdeditor.addMarkings(data);
            if (sdeditor.reviewable) {
                sdeditor.editMarkings();
            } else {
                sdeditor.showMarkings();
            };
        }).off('clearreview').on('clearreview', function(e, data){
            sdeditor.clearReview(data);
            sdeditor.clearShownMarkings();
        }).off('clearmarkings').on('clearmarkings', function(e, locs){
            sdeditor.clearMarkings(locs);
            sdeditor.showMarkings(locs);
        }).off('settheme').on('settheme', function(e, data){
            sdeditor.setTheme(data);
        }).off('getcheckdata').on('getcheckdata', function(e, data){
            var cdata = sdeditor.getCheckData();
            sdeditor.place.data('[[checkdata]]', cdata);
        });
        
        
        this.place.off('click.ldots').on('click.ldots', 'td.sdsubldots', function(e){
            e.stopPropagation();
            e.preventDefault();
            var $elemtr = $(this).parent('tr').prev();
            var $elemtd = $elemtr.children('td').eq(1);
            var sdtype = $elemtd.attr('data-sdtype');
            while (sdtype === 'sdsubderivation'){
                $elemtr = $elemtr.prev();
                $elemtd = $elemtr.children('td').eq(1);
                sdtype = $elemtd.attr('data-sdtype');
            }
            if (sdtype === 'sdmotivation') {
                var locstring = $elemtd.attr('data-loc');
                var loc = locstring.split('_');
                var $subder = $elemtr.next().children('td').eq(1).children('.subderivation');
                var status = ($subder.length > 0 ? $subder.hasClass('sdsubhidden') : false);
                if (sdeditor.editable) {
                    var event = new SdEditevent('showhidesubderivation', loc, [!status, status]);
                    sdeditor.doEvent(event);
                } else {
                    var $tbody = sdeditor.place.find('[data-loc="'+locstring+'"]').closest('tbody');
                    $tbody.children('tr').children('[data-loc^="'+locstring+'_subderivation"]').children('.subderivation').toggleClass('sdsubhidden');
                }
            }
        });
        
        
        this.place.off('mousedown.plugins', '.sdeditor-plugins ul.sdeditor-menulist li').on('mousedown.plugins', '.sdeditor-plugins ul.sdeditor-menulist li', function(e){
            e.stopPropagation();
            e.preventDefault();
            $(this).closest('.sdeditor-plugins').children('[tabindex]').blur();
            var pluginname = $(this).attr('data-pluginname');
            var item = $(this).attr('data-pluginitem') | 0;
            if (SdEditor.plugins[pluginname] && SdEditor.plugins[pluginname].items[item]) {
                SdEditor.plugins[pluginname].items[item].action(sdeditor);
            }
        });
        this.place.off('click', '.sdmarkings-clipper').on('click', '.sdmarkings-clipper', function(event){
            event.stopPropagation();
            sdeditor.marksvisible = !sdeditor.marksvisible;
            sdeditor.updateMarkings();
        });
        
        /***************
         * Drag handler
         ***************/
        this.place.off('dragstart', '.sdeditor-dragicon').on('dragstart', '.sdeditor-dragicon', function(e){
            e.stopPropagation();
            var event = e.originalEvent;
            var data = sdeditor.getData();
            data.id = sdeditor.place.closest('[data-element-name]').attr('data-element-name');
            event.dataTransfer.setData('text', JSON.stringify(data, null, 4));
            event.dataTransfer.effectAllowed = 'copy';
        });
        
        
        /******
         * Adding and showing review
         ******/
        this.place.off('showselfreview').on('showselfreview', function(event, data){
            event.stopPropagation();
            sdeditor.setMarkings(data);
            sdeditor.show();
        });
        /******
         * Hiding review
         ******/
        this.place.off('hideselfreview').on('hideselfreview', function(event, data){
            event.stopPropagation();
            sdeditor.clearShownMarkings();
        });
        
        /******
         * Changing the type of question
         ******/
        this.place.off('click', '.sdeditor-questiontype').on('click', '.sdeditor-questiontype', function(event, data) {
            event.stopPropagation();
            var button = $(this);
            var location = button.closest('tr').next().children('[data-loc]').attr('data-loc');
            var loc = location.split('_');
            var qtype = button.attr('data-qtype');
            var newqtype = Sdtask.nextType(qtype);
            if (newqtype) {
                button.attr('data-qtype', newqtype);
                button.html(Dictionary.localize('question:'+newqtype, sdeditor.settings.uilang));;
            };
            var edevent = new SdEditevent('edit', loc, [{type: newqtype}, {type: qtype}]);
            sdeditor.doEvent(edevent);
        });
    };
    
    /**
     * initHandlersEdit
     * Init event handlers in editing modes
     */
    SdEditor.prototype.initHandlersEdit = function() {
        var sdeditor = this;
        this.place.off('keydown.sdeditor').on('keydown.sdeditor', '.mathquill-editable', function(e){
            var mqelem = $(this);
            var allmq = sdeditor.place.find('.mathquill-editable');
            var index = allmq.index(mqelem);
            switch (e.which) {
                // Enter
                case 13:
                    var loc = mqelem.closest('td[data-loc]').attr('data-loc').split('_');
                    mqelem.focusout();
                    if (loc[loc.length - 2] === 'task') {
                        var event = new SdEditevent('addassumption', loc, [{},{}]);
                        sdeditor.doEvent(event);
                    } else if (loc[loc.length - 2] === 'term') {
                        var event = new SdEditevent('addstep', loc, [{},{}]);
                        sdeditor.doEvent(event);
                    } else if (loc[loc.length - 3] === 'assumption'){
                        var event = new SdEditevent('addassumption', loc, [{},{}]);
                        sdeditor.doEvent(event);
                    } else if (loc[loc.length - 2] === 'observation'){
                        var event = new SdEditevent('addobservation', loc, [{},{}]);
                        sdeditor.doEvent(event);
                    } else {
                        allmq.eq(index + 1).focus();
                    }
                    break;
                // Backspace
                case 8:
                    if (e.ctrlKey) {
                        var locstring = mqelem.parents('td[data-loc]').attr('data-loc');
                        var loc = locstring.split('_');
                        if (loc[loc.length - 2] === 'task') {
                            var event = new SdEditevent('removetask', loc, [{},{}]);
                            sdeditor.doEvent(event);
                        } else if (loc[loc.length - 2] === 'motivation' || loc[loc.length - 2] === 'relation') {
                            var event = new SdEditevent('removestep', loc, [{},{}]);
                            sdeditor.doEvent(event);
                        } else if (loc[loc.length - 2] === 'derivmotivation') {
                            var event = new SdEditevent('removederivationmotivation', loc, [{},{}]);
                            sdeditor.doEvent(event);
                        } else if (loc[loc.length - 3] === 'assumption') {
                            var event = new SdEditevent('removeassumption', loc, [{},{}]);
                            sdeditor.doEvent(event);
                        } else if (loc[loc.length - 1] === 'motivation' && loc[loc.length - 3] === 'observation' || loc[loc.length -2] === 'observation') {
                            var event = new SdEditevent('removeobservation', loc, [{},{}]);
                            sdeditor.doEvent(event);
                        }
                    }
                    break;
                // Up arrow
                case 38:
                    if (index > 0 && (mqelem.is('.hasCursor') || (mqelem.is('.mathquill-textbox') && mqelem.children('.mathquill-rendered-math.hasCursor').length > 0))){
                        mqelem.focusout();
                        allmq.eq(index - 1).focus();
                    }
                    break;
                // Down arrow
                case 40:
                    if (index < allmq.length -1 && (mqelem.is('.hasCursor') || (mqelem.is('.mathquill-textbox') && mqelem.children('.mathquill-rendered-math.hasCursor').length > 0))){
                        mqelem.focusout();
                        allmq.eq(index + 1).focus();
                    }
                    break;
                // Undo (ctrl+z) and Redo (shift+ctrl+z)
                case 90:
                    if (e.ctrlKey) {
                        e.stopPropagation();
                        e.preventDefault();
                        if (e.shiftKey) {
                            var $current = sdeditor.place.find('td.currentFocus');
                            var loc = $current.attr('data-loc');
                            var event = new SdEditevent('redo', [], '');
                            sdeditor.execEvent(event);
                        } else {
                            var $current = sdeditor.place.find('td.currentFocus');
                            $(this).focusout().focus();
                            var loc = $current.attr('data-loc');
                            var event = new SdEditevent('undo', [], '');
                            sdeditor.execEvent(event);
                        }
                    }
                    break;
                // Arrow right, Arrow left (break for undo-/redostack)
                case 37:
                case 39:
                    //$(this).focusout().focus();
                    $(this).trigger('makeeditevent');
                    break;
                // Ctrl+x, Ctrl+c, Ctrl+v (break for undo-/redostack)
                case 67:
                case 86:
                case 88:
                    if (e.ctrlKey) {
                        $(this).focusout().focus();
                    }
                    break;
                // Esc (focusout)
                case 27:
                    sdeditor.place.find('td.currentFocus span.mathquill-editable').blur().focusout();
                    break;
                // ctrl+j (add question)
                case 74:
                    if (e.ctrlKey) {
                        var loc = mqelem.closest('td[data-loc]').attr('data-loc').split('_');
                        var loclen = loc.length;
                        if (parseInt(loc[loclen - 1]) + '' === loc[loclen - 1] + '') {
                            // The last part in loc is a number
                            loc[loclen - 2] = 'term';
                            loc[loclen - 1] = '0';
                            var event = new SdEditevent('addtask', loc, [{},{}]);
                            sdeditor.doEvent(event);
                            e.stopPropagation();
                            e.preventDefault();
                        };
                    };
                    break;
                // ctrl+k (add assumption)
                case 75:
                    if (e.ctrlKey) {
                        var loc = mqelem.closest('td[data-loc]').attr('data-loc').split('_');
                        var loclen = loc.length;
                        if (parseInt(loc[loclen - 1]) + '' !== loc[loclen - 1] + '') {
                            // The last part in loc is not a number
                            loc.pop();
                        };
                        loc[loclen - 2] = 'assumption';
                        loc[loclen - 1] = 0;
                        while (sdeditor.locationExists(loc.slice())) {
                            loc[loclen - 1]++;
                        };
                        loc[loclen - 1] = (loc[loclen - 1] - 1) + '';
                        loc.push('proposition');
                        var event = new SdEditevent('addassumption', loc, [{},{}]);
                        sdeditor.doEvent(event);
                        e.stopPropagation();
                        e.preventDefault();
                    };
                    break;
                // ctrl+b (add-remove subderivation)
                case 66:
                    if (e.ctrlKey) {
                        var loc = mqelem.closest('td[data-loc]').attr('data-loc').split('_');
                        var loclen = loc.length;
                        if (loc[loclen - 1] === 'motivation' || loc[loclen -2] === 'motivation' || loc[loclen - 2] === 'term') {
                            if (e.shiftKey) {
                                var event = new SdEditevent('removesubderiv', loc, [{},{}]);
                            } else {
                                var event = new SdEditevent('addsubderiv', loc, [{},{}]);
                            }
                            sdeditor.doEvent(event);
                            e.stopPropagation();
                            e.preventDefault();
                        }
                    }
                    break;
                default:
                    break;
            }
        });
        this.place.off('makeeditevent.sdeditor').on('makeeditevent.sdeditor', 'span.mathquill-editable',function(e){
            var $mqelem = $(this);
            var currentsd = $mqelem.closest('td[data-loc]').eq(0);
            var loc = currentsd.attr('data-loc');
            var latex = $mqelem.mathquill('latex');
            if (typeof(latex) === 'string') {
                // If this element was removed, don't come here, but just ignore this focusout.
                latex = latex.replace(/\$\$/g, '').replace(/\\text\{\s*\}/g, '');
                $mqelem.trigger('editevent', [loc, {text: latex, readonly: false}]);
                if (latex === '') {
                    currentsd.addClass('sdempty');
                } else {
                    currentsd.removeClass('sdempty');
                }
            }
        });
        this.place.off('focusout.sdeditor').on('focusout.sdeditor', 'span.mathquill-editable',function(e){
            var $mqelem = $(this);
            $mqelem.trigger('makeeditevent');
            var currentsd = $mqelem.closest('td[data-loc]').eq(0);
            currentsd.removeClass('currentFocus');
        });
        this.place.off('focus.sdeditor').on('focus.sdeditor', 'span.mathquill-editable textarea', function(){
            // When a mathquill box gets focus, update some classes and attributes
            // so that correct buttons are shown in right places and focus is highlighted.
            var $mqelem = $(this).parents('.mathquill-editable').eq(0);
            var $currenttd = $mqelem.parents('td').eq(0);
            sdeditor.place.data('currentmq', $mqelem);
            var focusonelem = $mqelem.parents('td[data-loc]').eq(0).attr('data-sdtype');
            sdeditor.place.attr('data-focusonelem', focusonelem);
            var focuselemtype = '';
            var loclist = $currenttd.attr('data-loc').split('_');
            var loctypes = sdeditor.getLocationTypes(loclist);
            focuselemtype = loctypes.join(' ');
            sdeditor.place.attr('data-focuselemtype', focuselemtype);
            //var currentsd = sdeditor.place.find('td.currentFocus');
            $currenttd.addClass('currentFocus');
            sdeditor.slideButtons();
        });
        this.place.off('focus.dsplaceholder').on('focus.dsplaceholder', '.sdeditor-dstepplaceholder', function(event){
            event.stopPropagation();
            sdeditor.place.attr('data-focusonelem', '');
            sdeditor.place.attr('data-focuselemtype', 'dstep');
            $(this).addClass('currentFocus');
            sdeditor.place.data('currentmq', null);
            sdeditor.place.data('currentplaceholder', $(this));
            sdeditor.slideButtons();
        });
        this.place.off('focusout.dsplaceholder').on('focusout.dsplaceholder', '.sdeditor-dstepplaceholder', function(event){
            event.stopPropagation();
            $(this).removeClass('currentFocus');
            //setTimeout(function(){sdeditor.place.attr('data-focuselemtype', '');}, 500);
        });
        
        this.place.off('editevent.sdeditor').on('editevent.sdeditor', function(e, loc, data){
            var location = loc.split('_');
            var locarr = loc.split('_');
            var olddata = sdeditor.getLocationData(locarr);
            var isdifferent = false;
            for (var attr in data) {
                isdifferent = isdifferent || data[attr] !== olddata[attr];
            }
            if (isdifferent) {
                var event = new SdEditevent('edit', location, [data, {}]);
                sdeditor.doEvent(event);
            }
        });

        /******
         * Handle editorevent coming as javascript event.
         * Event format:
         * {
         *     "type": <eventtype>,
         *     "loc": "path_to_the_location" | ["path", "to", "the", "location"],
         *     "setfocus": true | false,
         *     "data": [<new_data>] | <new_data>
         * }
         ******/
        this.place.off('editorevent.sdeditor').on('editorevent.sdeditor', function(e, eventdata){
            var etype = eventdata.type;
            if (typeof(eventdata.loc) === 'string') {
                var location = eventdata.loc && eventdata.loc.split('_') || [];
            } else if (typeof(eventdata.loc) === 'object' && eventdata.loc.hasOwnProperty('length')) {
                var location = eventdata.loc.slice();
            }
            var setfocus = !!eventdata.setfocus;
            var data = eventdata.data[0] || eventdata.data;
            var event = new SdEditevent(etype, location, [data, {}]);
            sdeditor.doEvent(event);
            var newlocstr = sdeditor.place.find('td.currentFocus').attr('data-loc') || '';
            var newloc = newlocstr.split('_');
            sdeditor.show(setfocus ? newloc : null);
        });

        this.place.off('elementfocus').on('elementfocus', function(e){
            e.stopPropagation();
            sdeditor.place.find('td.sdterm .mathquill-editable').trigger('focus');
        });

        /***************
         * Drag handler
         ***************/
        this.place.off('dragstart', '.sdeditor-derivationstep-draghandle').on('dragstart', '.sdeditor-derivationstep-draghandle', function(e){
            e.stopPropagation();
            var target = $(e.target);
            var event = e.originalEvent;
            var derivid = sdeditor.name.replace(/[-_]/g, '');
            var wrapper = $(this).closest('.sdeditor-derivationstep');
            wrapper.addClass('sdeditor-draggingelement');
            var index = target.attr('data-dstepindex') | 0;
            var data = {
                derivid: derivid,
                dstype: sdeditor.dsteps[index].type,
                movedata: {
                    index: index
                },
                copydata: {}
            };
            if (sdeditor.authorable) {
                var dsnames = sdeditor.getDstepNames();
                var dsname = dsnames[index];
                data.copydata = sdeditor.dstepdata[dsname].getData();
            };
            event.dataTransfer.setData('text', JSON.stringify(data, null, 4));
            event.dataTransfer.effectAllowed = 'copy';
        });
        
        this.place.off('dragend', '.sdeditor-derivationstep-draghandle').on('dragend', '.sdeditor-derivationstep-draghandle', function(e){
            sdeditor.place.find('.sdeditor-derivationstep.sdeditor-draggingelement').removeClass('sdeditor-draggingelement');
            sdeditor.place.find('.sdeditor-draghover').removeClass('sdeditor-draghover');
        });
        
        this.place.off('dragover', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep').on('dragover', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep', function(event, data){
            // Prevent default on dragover to allow drop.
            event.preventDefault();
            $(this).addClass('sdeditor-draghover');
        });
        
        this.place.off('dragleave', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep').on('dragleave', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep', function(event, data){
            if (event.target === this) {
                $(this).removeClass('sdeditor-draghover');
            };
        });
        
        this.place.off('drop', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep').on('drop', '.sdeditor-dstepplaceholder, .sdeditor-derivationstep', function(event){
            event.stopPropagation();
            event.preventDefault();
            var ev = event.originalEvent;
            var placeto = $(this);
            var data;
            var datastr = ev.dataTransfer.getData('text');
            try {
                data = JSON.parse(datastr);
            } catch (err) {
                
            };
            if (data) {
                var loc = placeto.attr('data-loc');
                var location = loc.split('_');
                var indexto = location[2] | 0;
                if (sdeditor.name.replace(/[-_]/g, '') === data.derivid && data.movedata && typeof(data.movedata.index) === 'number') {
                    // Moving inside the derivation.
                    var editevent = new SdEditevent('movedstep', [data.derivid], [{indexfrom: data.movedata.index, indexto: indexto}, {}])
                    sdeditor.doEvent(editevent);
                } else {
                    // Copying to another derivation.
                    var derivid = sdeditor.name.replace(/[-_]/g, '');
                    var editevent = new SdEditevent('addstep' + data.dstype, location, [{dstepcopy: true, data: data.copydata}, {}]);
                    sdeditor.doEvent(editevent);
                };
            };
        });
        
        /******
         * Trashcan for removing dstep
         ******/
        this.place.off('click', '.sdeditor-removestep-button').on('click', '.sdeditor-removestep-button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var uilang = sdeditor.settings.uilang;
            var location = $(this).closest('[data-loc]').attr('data-loc') || '';
            location = location.split('_');
            var editevent = new SdEditevent('removedstep', location, [{},{}]);
            if (confirm(Dictionary.localize('areyousure', uilang))) {
                sdeditor.doEvent(editevent);
            };
        });
    };

    /**
     * initHandlersNonedit
     * Init event handlers in nonediting modes
     */
    SdEditor.prototype.initHandlersNonedit = function() {
        
    };

    /**
     * initHandlersReview
     * Init event handlers in reviewable mode
     */
    SdEditor.prototype.initHandlersReview = function() {
        var sdeditor = this;
        /******
         * Review marking handlers
         ******/
        this.place.off('click.markclick', '.sdmanualreviewbox .sdmark').on('click.markclick', '.sdmanualreviewbox .sdmark', function(e){
            // Change the mark for a step.
            var markbutton = $(this);
            var mark = markbutton.attr('data-sdmark');
            var checkbox = markbutton.closest('.sdmanualreviewbox');
            var loc = checkbox.closest('[data-noteloc]').attr('data-noteloc');
            var comment = checkbox.find('input[type="text"]').val() || '';
            comment = comment.replace(/\s*<\s*/g, ' < ').replace(/\s*>\s*/g, ' > ');
            checkbox.attr('data-marktype', mark);
            sdeditor.addMarking({
                loc: loc,
                mark: mark,
                comment: comment
            });
            sdeditor.reviewChanged();
            //markbutton.trigger('review_changed');
        });

        this.place.off('click.markopen', '.sdmanualreviewbox').on('click.markopen', '.sdmanualreviewbox', function(e){
            // Open marktool for a step.
            $(this).addClass('sdmarkboxopen');
        });
        this.place.off('click.markclose', '.sdmanualreviewbox button').on('click.markclose', '.sdmanualreviewbox button', function(e){
            // Close marktool for a step. (OK-button)
            e.stopPropagation();
            e.preventDefault();
            $(this).closest('.sdmanualreviewbox').removeClass('sdmarkboxopen');
        });
        this.place.off('focusout.markready', '.sdmanualreviewbox input[type="text"]')
        .on('focusout.markready', '.sdmanualreviewbox input[type="text"]', function(e){
            // Focusout from input in marktool for a step.
            var $input = $(this);
            var comment = $input.val();
            comment = comment.replace(/\s*<\s*/g, ' < ').replace(/\s*>\s*/g, ' > ');
            var mark = $input.closest('.sdmanualreviewbox').attr('data-marktype');
            var loc = $input.closest('[data-noteloc]').attr('data-noteloc');
            var oldcomment = sdeditor.getMarksComment(loc);
            if (comment !== oldcomment) {
                sdeditor.addMarking({
                    loc: loc,
                    mark: mark,
                    comment: comment
                });
                sdeditor.reviewChanged();
                //$input.trigger('review_changed');
            }
        });
        this.place.off('focusout.markready', '.sdmarkings-note .mathquill-editable')
        .on('focusout.markready', '.sdmarkings-note .mathquill-editable', function(e){
            // Focusout from general feedback MathQuill.
            var $elem = $(this);
            var comment = $elem.mathquill('latex');
            comment = comment.replace(/\s*<\s*/g, ' < ').replace(/\s*>\s*/g, ' > ');
            var mark = $elem.closest('.sdmarkings-note').attr('data-marktype') || 'empty';
            var loc = $elem.attr('data-noteloc');
            var oldcomment = sdeditor.getMarksComment(loc);
            if (comment !== oldcomment) {
                sdeditor.addMarking({
                    loc: loc,
                    mark: mark,
                    comment: comment
                });
                sdeditor.reviewChanged();
                //$elem.trigger('review_changed');
            }
        });
        this.place.off('click.generalmark', '.sdmarkings-available .sdmark').on('click.generalmark', '.sdmarkings-available .sdmark', function(e){
            // Change the marktype of general feedback.
            var $elem = $(this);
            var $marknote = $elem.closest('.sdmarkings-note');
            var $note = $marknote.find('[data-noteloc]');
            var loc = $note.attr('data-noteloc');
            var mark = $elem.closest('[data-sdmark]').attr('data-sdmark');
            var comment = $note.mathquill('latex') || '';
            comment = comment.replace(/\s*<\s*/g, ' < ').replace(/\s*>\s*/g, ' > ');
            if (mark === 'empty' || mark === '') {
                comment = '';
                $note.mathquill('latex', comment);
            }
            $marknote.attr('data-marktype', mark);
            sdeditor.addMarking({
                loc: loc,
                mark: mark,
                comment: comment
            });
            $note.focus();
            sdeditor.reviewChanged();
            //$elem.trigger('review_changed');
        });
    };

    /******
     * Create and init buttons in buttonbar.
     ******/
    SdEditor.prototype.initButtons = function(){
        var sdeditor = this;
        var uilang = this.settings.uilang;
        var html = ['<ul class="sdeditor-buttonlist" '+(this.buttonmargin ? 'style="margin-top: '+this.buttonmargin+'px;"' : '')+'>'];
        for (var i = 0; i < SdEditor.buttons.length; i++) {
            var button = SdEditor.buttons[i];
            html.push('<li class="sdeditor-button sdeditor-'+button.name+'" data-sdeditor-event="'+button.event+'"><a href="javascript:;" title="'+Dictionary.localize(button.title, uilang)+'"><span>'+(button.icon || button.text)+'</span></a></li>');
        }
        html.push('</ul>');
        this.editorpanel.html(html.join('\n'));
        this.editorpanel.off('click.sdbutton').on('click.sdbutton', 'li.sdeditor-button a', function(e){
            var location;
            var current = sdeditor.place.data('currentmq') || sdeditor.place.data('currentplaceholder');
            if (current) {
                location = current.closest('[data-loc]').attr('data-loc').split('_');
            } else {
                location = [''];
            };
            var $buttonli = $(this).parent('li.sdeditor-button');
            var eventname = $buttonli.attr('data-sdeditor-event');
            var event = new SdEditevent(eventname, location, [{},{}]);
            if (eventname === 'undo' || eventname === 'redo'){
                sdeditor.execEvent(event);
            } else {
                sdeditor.doEvent(event);
            }
        });
    };
    
    /******
     * Generate a new id for derivationstep (type, username, datetime, randomnumber)
     * @param {String} type - Type of derivationstep
     ******/
    SdEditor.prototype.generateId = function(type){
        type = type || 'stask';
        var id = [type];
        id.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear() + '';
        var month = ('0' + (now.getUTCMonth() + 1)).slice(-2);
        var day = ('0' + now.getUTCDate()).slice(-2);
        var hour = ('0' + now.getUTCHours()).slice(-2);
        var minute = ('0' + now.getUTCMinutes()).slice(-2);
        var second = ('0' + now.getUTCSeconds()).slice(-2);
        var msecs = ('00' + now.getUTCMilliseconds()).slice(-3);
        id.push(year + month + day + hour + minute + second + msecs);
        id.push(Math.floor(1000 * Math.random()));
        return id.join('-');
    };
    
    /*************************
     * Constants for editor
     *************************/
    SdEditor.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        }
    }
    
    SdEditor.stepdefaults = {
        type: 'sderivation',
        // Default metadata
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        // Default empty data
        data: {
            dsteps: [],
            dstepdata: {},
            markings: {}
        },
        // Default settings for editor.
        settings: {
            lang: 'en',
            uilang: 'en',
            mode: 'view',
            stepmode: false,
            theme: 'default',
            username: 'Anonymous'        }
    }
    
    SdEditor.defaults = {
        // Default metadata
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        // Default empty data.
        data: {
            derivation:  {
                "task": [],
                "assumption": [],
                "observation": [],
                "derivmotivation": [],
                "term": [
                    {"text": "" }
                ],
                "relation": [],
                "motivation":[]
            },
            markings: {}
        },
        // Default settings for editor.
        settings: {
            lang: 'en',
            uilang: 'en',
            mode: 'view',
            stepmode: false,
            theme: 'default',
            username: 'Anonymous'
        },
    }
    
    SdEditor.user = {
        // Current user settings for the editor. Empty by default, but can be changed with jQuery-plugin with:
        // $.fn.sdeditor('settings', usersettings);
        settings: {
        }
    }
    
    // Plugins will be added here.
    SdEditor.plugins = {};
    
    SdEditor.buttons = [
        {
            // Undo
            name: 'undo',
            title: 'undo',
            event: 'undo',
            text: '&lt;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-undo sdicon-control"><path style="stroke: none; fill: #aaa;" d="M-2 8 l0 10 l10 0 l-3 -3 a10 10 0 0 1 18 4 a12 12 0 0 0 -22 -8z" /><path style="stroke: none; fill: white;" d="M2 12 l0 10 l10 0 l-3 -3 a10 10 0 0 1 18 4 a12 12 0 0 0 -22 -8z" /><path style="stroke: none;" d="M0 10 l0 10 l10 0 l-3 -3 a10 10 0 0 1 18 4 a12 12 0 0 0 -22 -8z" /></svg>'
        },
        {
            // Redo
            name: 'redo',
            title: 'redo',
            event: 'redo',
            text: '&gt;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-redo sdicon-control"><path style="stroke: none; fill: #aaa;" d="M28 9 l0 10 l-10 0 l3 -3 a10 10 0 0 0 -18 4 a12 12 0 0 1 22 -8z" /><path style="stroke: none; fill: white;" d="M30 11 l0 10 l-10 0 l3 -3 a10 10 0 0 0 -18 4 a12 12 0 0 1 22 -8z" /><path style="stroke: none;" d="M29 10 l0 10 l-10 0 l3 -3 a10 10 0 0 0 -18 4 a12 12 0 0 1 22 -8z" /></svg>'
        },
        {
            // Add task 
            name: 'addtask',
            title: 'add task',
            event: 'addtask',
            text: '&bull;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addtask sdicon-add"><path style="stroke: none; fill: #aaa;" d="M13 13 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none; fill: white;" d="M17 17 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none;" d="M15 15 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none; fill: #aaa;" d="M1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/><path style="stroke: none; fill: white;" d="M3 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/><path style="stroke: none; fill: #070;" d="M2 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/></svg>'
        },
        {
            // Remove task
            name: 'removetask',
            title: 'remove task',
            event: 'removetask',
            text: '&bull;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-removetask sdicon-remove"><path style="stroke: none; fill: #aaa;" d="M13 13 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none; fill: white;" d="M17 17 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none;" d="M15 15 a6 6 0 0 0 12 0 a6 6 0 0 0 -12 0z" /><path style="stroke: none; fill: #aaa;" d="M-6 12 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/><path style="stroke: none; fill: white;" d="M-4 14 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/><path style="stroke: none; fill: #a00;" d="M-5 13 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/></svg>'
        },
        {
            // Add assumption
            name: 'addass',
            title: 'add assumption',
            event: 'addassumption',
            text: '(+)',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-plusassumption sdicon-add"><path style="stroke: none; fill: #aaa;" d="M12 5 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: white;" d="M14 7 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none;" d="M13 6 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: #aaa;" d="M1 4 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none; fill: white;" d="M3 6 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none;" d="M2 5 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 4 a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /><path style="stroke: none; fill: white;" d="M28 6  a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /><path style="stroke: none;" d="M27 5  a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /></svg>'
        },
        {
            // Remove assumption
            name: 'removeass',
            title: 'remove assumption',
            event: 'removeassumption',
            text: '(&mdash;)',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-minusassumption sdicon-remove"><path style="stroke: none; fill: white;" d="M7 15 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 12 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M6 13 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M1 4 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none; fill: white;" d="M3 6 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none;" d="M2 5 a4 10 0 0 0 0 20 l-1 1 a5 10 0 0 1 0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 4 a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /><path style="stroke: none; fill: white;" d="M28 6  a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /><path style="stroke: none;" d="M27 5  a4 10 0 0 1 0 20 l1 1 a5 10 0 0 0 0 -22z" /></svg>'
        },
        {
            // Remove observation
            name: 'removeobs',
            title: 'remove observation',
            event: 'removeobservation',
            text: '[&mdash;]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-minusobservation sdicon-remove"><path style="stroke: none; fill: white;" d="M7 15 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 12 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M6 13 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M1 3 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M3 6 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none;" d="M2 4 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 3 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M28 6 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none;" d="M27 4 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /></svg>'
        },
        {
            // Add observation
            name: 'addobs',
            title: 'add observation',
            event: 'addobservation',
            text: '[+]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-plusobservation sdicon-add"><path style="stroke: none; fill: #aaa;" d="M12 5 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: white;" d="M14 7 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none;" d="M13 6 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: #aaa;" d="M1 3 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M3 6 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none;" d="M2 4 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 3 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M28 6 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none;" d="M27 4 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /></svg>'
        },
        {
            // Add definition
            name: 'adddef',
            title: 'add definition',
            event: 'adddefinition',
            text: '[=]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-plusdefinition sdicon-add"><path style="stroke: none; fill: white;" d="M7 10 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 7 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M6 8 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: white;" d="M7 20 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 17 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M6 18 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M1 3 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M3 6 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none;" d="M2 4 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 3 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M28 6 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none;" d="M27 4 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /></svg>'
        },
        {
            // Add declaration
            name: 'adddec',
            title: 'add declaration',
            event: 'adddeclaration',
            text: '[-]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-plusdeclaration sdicon-add"><path style="stroke: none; fill: white;" d="M13 7 l4 0 l0 12 l-4 0z m0 14 h4 v4 h-4z" /><path style="stroke: none; fill: #aaa;" d="M11 5 l4 0 l0 12 l-4 0z m0 14 h4 v4 h-4z" /><path style="stroke: none;" d="M12 6 l4 0 l0 12 l-4 0z m0 14 h4 v4 h-4z" /><path style="stroke: none; fill: white;" d="M3 6 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: #aaa;" d="M1 3 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none;" d="M2 4 l0 2 l-4 0 l0 18 l4 0 l0 2 l-6 0 l0 -22z" /><path style="stroke: none; fill: white;" d="M28 6 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none; fill: #aaa;" d="M26 3 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /><path style="stroke: none;" d="M27 4 l0 2 l4 0 l0 18 l-4 0 l0 2 l6 0 l0 -22z" /></svg>'
        },
        {
            // Add step
            name: 'addstep',
            title: 'add step',
            event: 'addstep',
            text: '+',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-plusstep sdicon-add"><path style="stroke: none; fill: #aaa;" d="M11 5 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: white;" d="M13 7 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none;" d="M12 6 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /></svg>'
        },
        {
            // Remove step
            name: 'removestep',
            title: 'remove step',
            event: 'removestep',
            text: '&mdash;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-minusstep sdicon-remove"><path style="stroke: none; fill: white;" d="M9 13 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 11 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M7 12 l18 0 l0 4 l-18 0 l0 -6z" /></svg>'
        },
        {
            // Add derivationmotivation
            name: 'addderivationmotivation',
            title: 'add derivationmotivation',
            event: 'addderivationmotivation',
            text: '{+}',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-plusderivmot sdicon-add"><path style="stroke: none; fill: #aaa;" d="M12 5 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: white;" d="M14 7 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none;" d="M13 6 l4 0 l0 7 l7 0 l0 4 l-7 0 l0 7 l-4 0 l0 -7 l-7 0 l0 -4 l7 0z" /><path style="stroke: none; fill: #aaa;" d="M3 3 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none; fill: white;" d="M5 6 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none;" d="M4 4 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none; fill: #aaa;" d="M24 3 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /><path style="stroke: none; fill: white;" d="M26 6 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /><path style="stroke: none;" d="M25 4 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /></svg>'
        },
        {
            // Remove derivationmotivation
            name: 'removederivationmotivation',
            title: 'remove derivationmotivation',
            event: 'removederivationmotivation',
            text: '{&ndash;}',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-minusderivmot sdicon-remove"><path style="stroke: none; fill: white;" d="M7 15 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M5 12 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none;" d="M6 13 l18 0 l0 4 l-18 0 l0 -6z" /><path style="stroke: none; fill: #aaa;" d="M3 3 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none; fill: white;" d="M5 6 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none;" d="M4 4 l0 2 l-2 0 a2 2 0 0 0 -2 2 l0 5 a3 2 0 0 1 -3 2 a3 2 0 0 1 3 2 l0 5 a2 2 0 0 0 2 2 l2 0 l0 2 l-3 0 a3 3 0 0 1 -3 -3 l0 -5 a5 3 0 0 0 -5 -3 a5 3 0 0 0 5 -3 l0 -5 a3 3 0 0 1 3 -3z" /><path style="stroke: none; fill: #aaa;" d="M24 3 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /><path style="stroke: none; fill: white;" d="M26 6 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /><path style="stroke: none;" d="M25 4 l0 2 l2 0 a2 2 0 0 1 2 2 l0 5 a3 2 0 0 0 3 2 a3 2 0 0 0 -3 2 l0 5 a2 2 0 0 1 -2 2 l-2 0 l0 2 l3 0 a3 3 0 0 0 3 -3 l0 -5 a5 3 0 0 1 5 -3 a5 3 0 0 1 -5 -3 l0 -5 a3 3 0 0 0 -3 -3z" /></svg>'
        },
        {
            // Add subderivation
            name: 'addsubderiv',
            title: 'add subderivation',
            event: 'addsubderiv',
            text: '+...',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addsubderiv sdicon-add"><path style="stroke: none;" d="M4 4 l25 0 l0 3 l-25 0z m6 7 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m-12 5 l25 0 l0 3 l-25 0z" /><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"/></svg>'
        },
        {
            // Remove subderivation
            name: 'removesubderiv',
            title: 'remove subderivation',
            event: 'removesubderiv',
            text: '-...',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-removesubderiv sdicon-remove"><path style="stroke: none;" d="M4 4 l25 0 l0 3 l-25 0z m6 7 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m-12 5 l25 0 l0 3 l-25 0z" /><path style="stroke: none; fill: #aaa;" d="M-6 12 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/><path style="stroke: none; fill: white;" d="M-4 14 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/><path style="stroke: none; fill: #a00;" d="M-5 13 l13 0 l0 3 l-13 0 l0 -3 l5 0z"/></svg>'
        },
        {
            // Add derivation step task 
            name: 'addsteptask',
            title: 'add task step',
            event: 'addsteptask',
            text: '+&bull;',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addsteptask sdicon-add"><path style="stroke: none;" d="M10 6 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z m-6 0 l4 0 l0 3 l-4 0z"></path><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path></svg>'
        },
        {
            // Add derivation step assumption
            name: 'addstepassumption',
            title: 'add assumption step',
            event: 'addstepassumption',
            text: '+()',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addstepassumption sdicon-add"><path style="stroke: none;" d="M10 12 l6 0 l0 2 l-6 0z m9 -1 l12 0 l0 4 l-12 0z"></path><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path></svg>'
        },
        {
            // Add derivation step declaration
            name: 'addstepdeclaration',
            title: 'add declaration step',
            event: 'addstepdeclaration',
            text: '+[-]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addstepdeclaration sdicon-add"><path style="stroke: none;" d="M10 12 l2 0 l0 -2 l2 0 l0 2 l2 0 l0 2 l-2 0 l0 2 l-2 0 l0 -2 l-2 0z m9 -1 l13 0 l0 4 l-13 0z"></path><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path></svg>'
        },
        {
            // Add derivation step fact
            name: 'addstepfact',
            title: 'add fact step',
            event: 'addstepfact',
            text: '+[]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addstepfact sdicon-add"><path style="stroke: none;" d="M10 12 l2 0 l0 -2 l2 0 l0 2 l2 0 l0 2 l-2 0 l0 2 l-2 0 l0 -2 l-2 0z m9 -1 l13 0 l0 4 l-13 0z m0 5 l13 0 l0 3 l-13 0z"></path><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path></svg>'
        },
        {
            // Add derivation step definition
            name: 'addstepdefinition',
            title: 'add definition step',
            event: 'addstepdefinition',
            text: '+[=]',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="20" viewBox="-7.5 0 45 30" class="mini-icon mini-icon-addstepdefinition sdicon-add"><path style="stroke: none;" d="M10 12 l2 0 l0 -2 l2 0 l0 2 l2 0 l0 2 l-2 0 l0 2 l-2 0 l0 -2 l-2 0z m9 -1 l13 0 l0 4 l-13 0z m0 5 l13 0 l0 3 l-13 0z m0 5 l13 0 l0 3 l-13 0z"></path><path style="stroke: none; fill: #aaa;" d="M-1 7 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: white;" d="M1 9 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path><path style="stroke: none; fill: #070;" d="M0 8 l3 0 l0 5 l5 0 l0 3 l-5 0 l0 5 l-3 0 l0 -5 l-5 0 l0 -3 l5 0z"></path></svg>'
        }
    ];
    
    SdEditor.template = {
        edit: [
            '<ul class="sdeditor-menubar"></ul>',
            '<div class="sdeditor-container" style="position: relative;">',
            '    <div class="sdeditor" tabindex="0"></div>',
            '    <div class="sdeditor-buttonbar"></div>',
            '    <div class="sdeditor-dragicon" draggable="true"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="pssicon pssicon-drag"><path style="stroke: none;" d="M15 3 l5 5 l-4 0 l0 6 l6 0 l0 -4 l5 5 l-5 5 l0 -4 l-6 0 l0 6 l4 0 l-5 5 l-5 -5 l4 0 l0 -6 l-6 0 l0 4 l-5 -5 l5 -5 l0 4 l6 0 l0 -6 l-4 0 z"></path></svg></div>',
            '</div>'
        ].join('\n'),
        view: [
            '<ul class="sdeditor-menubar"></ul>',
            '<div class="sdeditor-container" style="position: relative;">',
            '    <div class="sdeditor"></div>',
            '    <div class="sdeditor-buttonbar"></div>',
            '    <div class="sdeditor-dragicon" draggable="true"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="pssicon pssicon-drag"><path style="stroke: none;" d="M15 3 l5 5 l-4 0 l0 6 l6 0 l0 -4 l5 5 l-5 5 l0 -4 l-6 0 l0 6 l4 0 l-5 5 l-5 -5 l4 0 l0 -6 l-6 0 l0 4 l-5 -5 l5 -5 l0 4 l6 0 l0 -6 l-4 0 z"></path></svg></div>',
            '</div>'
        ].join('\n')
    };
    
    SdEditor.icons = {
        stepprev: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="sdicon-stepnext"><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M2 23 l22 0 l-11 -20z" /><path style="stroke: none; fill: white;" d="M6 27 l22 0 l-11 -20z" /><path style="stroke: none; fill: black;" d="M4 25 l22 0 l-11 -20z" /></svg>',
        stepnext: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="sdicon-stepnext"><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M2 3 l22 0 l-11 20z" /><path style="stroke: none; fill: white;" d="M6 7 l22 0 l-11 20z" /><path style="stroke: none; fill: black;" d="M4 5 l22 0 l-11 20z" /></svg>',
        stepall: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="sdicon-stepnext"><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M2 3 l22 0 l-11 20z" /><path style="stroke: none; fill: white;" d="M6 7 l22 0 l-11 20z m0 18 l22 0 l0 3 l-22 0z" /><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M2 3 m0 18 l22 0 l0 3 l-22 0z" /><path style="stroke: none; fill: black;" d="M4 5 l22 0 l-11 20z m0 18 l22 0 l0 3 l-22 0z" /></svg>'
    }
    
    SdEditor.elementinfo = {
        type: 'sdeditor',
        elementtype: ['elements','studentelements'],
        jquery: 'sdeditor',
        name: 'Structured task',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-sderivation"><path style="stroke: none;" d="M10 5 a3 4 0 0 0 -3 4 v4 a2 2 0 0 1 -2 2 a2 2 0 0 1 2 2 v4 a3 4 0 0 0 3 4 v1 a5 4 0 0 1 -5 -4 v-4 a3 3 0 0 0 -3 -3 a3 3 0 0 0 3 -3 v-4 a5 4 0 0 1 5 -4z m10 0 v-1 a5 4 0 0 1 5 4 v4 a3 3 0 0 0 3 3 a3 3 0 0 0 -3 3 v4 a5 4 0 0 1 -5 4 v-1 a3 4 0 0 0 3 -4 v-4 a2 2 0 0 1 2 -2 a2 2 0 0 1 -2 -2 v-4 a5 4 0 0 0 -3 -4z m-5 6 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8z" /></svg>',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30"><circle cx="6" cy="4" r="2" /><line stroke="black" x1="15" y1="4" x2="30" y2="4" /><path stroke="black" stroke-width="1" fill="none" d="M1 15 l3 -3 l0 1 l4 0 l0 -1 l3 3 l-3 3 l0 -1 l-4 0 l0 1z" /><path stroke="black" stroke-width="1" fill="none" d="M17 9 l-1 1 l0 3 l-2 2 l2 2 l0 3 l1 1" /><line stroke="black" x1="18" y1="15" x2="25" y2="15" /><path stroke="black" stroke-width="1" fill="none" d="M26 9 l1 1 l0 3 l2 2 l-2 2 l0 3 l-1 1" /><line stroke="black" x1="15" y1="26" x2="30" y2="26" /></svg>',
        description: {
            en: 'Structured task',
            fi: 'Rakenteinen päättelyketju',
            sv: 'Strukturerad härledning'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['math'],
        weight: 20
    }
    
    SdEditor.elementinfoderiv = {
        type: 'sderivation',
        elementtype: ['elements','studentelements'],
        jquery: 'sdeditor',
        name: 'Structured derivation',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-sderivation"><path stroke="none" fill="black" d="M6 3 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m5 1 l17 0 l0 4 l-17 0z m-6 7 l-2 0 l0 6 l2 0 l0 -1 l-1 0 l0 -4 l1 0 l0 -1z m2 0 l0 1 l1 0 l0 4 l-1 0 l0 1 l2 0 l0 -6z m4 1 l17 0 l0 4 l-17 0z m-8 7 l0 6 l6 0 l0 -6z m1 1 l4 0 l0 4 l-4 0z m7 0 l17 0 l0 4 l-17 0z" /></svg>',
        description: {
            en: 'Structured derivation.',
            fi: 'Rakenteinen päättelyketju.'
        },
        classes: ['math', 'content']
    }
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', SdEditor.elementinfo);
//         $.fn.elementset('addelementtype', SdEditor.elementinfoderiv);
    }
    
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', SdEditor.elementinfo);
//         $.fn.elementpanel('addelementtype', SdEditor.elementinfoderiv);
    }
    
    
    
    /*****************************************
     * Sderivation object
     * constructor
     *****************************************/
    var Sderivation = function(derivation){
        derivation = $.extend(true, {
            name: 'derivation1',
            readonly: false,
            task: [],
            assumption: [],
            observation: [],
            definition: [],
            derivmotivation: [],
            term: [{}],
            relation: [],
            motivation: []
        }, derivation);
        this.issubderiv = !!derivation.issubderiv;
        this.name = (this.issubderiv ? 'subderiv' : derivation.name);
        this.type = 'task';
        this.readonly = derivation.readonly;
        this.elements = {
            task: [],
            assumption: [],
            observation: [],
            definition: [],
            derivmotivation: [],
            term: [],
            relation: [],
            motivation: []
        }
        for (var i = 0, length = Sderivation.components.length; i < length; i++) {
            var component = Sderivation.components[i];
            var constr = Sderivation.constr[component];
            for (var j = 0, clength = derivation[component].length; j < clength; j++) {
                var newelement = new constr(derivation[component][j]);
                this.elements[component].push(newelement);
            }
        }
        return this;
    }

    /******
     * Return the name of this derivation.
     ******/
    Sderivation.prototype.getName = function(){
        return this.name;
    }
    
    /******
     * Get html-code.
     ******/
    Sderivation.prototype.getHtml = function(readonly, location, issdstep, uilang, haschecker){
        var classes = (this.issubderiv ? 'sdlayouttable subdertable' : 'sdlayouttable');
        var html = [];
        var qtyperow = '';
        if (haschecker && this.elements.task.length > 0) {
            var qtype = this.elements.task[0].type;
            var qtypetext = Dictionary.localize('question:'+qtype, uilang);
            qtyperow = '<tr><td></td><td><span class="sdeditor-questiontype" data-qtype="'+escapeHTML(qtype)+'">'+qtypetext+'</span></td></tr>\n';
        };
        var attrs = [];
        if (this.elements.task.length > 0) {
            for (var attr in this.elements.task[0].attrs) {
                if (this.elements.task[0].attrs[attr]) {
                    attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.elements.task[0].attrs[attr]) + '"');
                }
            }
        }
        html.push('<table class="'+classes+'">\n<tbody>\n'+qtyperow+'<tr '+attrs.join(' ')+'><td><span class="sdmath" data-sdreadonly="true">\\bullet</span></td>');
        for (var i = 0; i < this.elements.task.length; i++) {
            html.push(this.elements.task[i].getHtml(readonly || this.readonly, location.concat(['task','0']), false, uilang, haschecker));
        }
        for (var i = 0; i < this.elements.assumption.length; i++) {
            html.push(this.elements.assumption[i].getHtml(readonly || this.readonly, location.concat(['assumption',''+i]), false, uilang, haschecker));
        }
        for (var i = 0; i < this.elements.observation.length; i++) {
            html.push(this.elements.observation[i].getHtml(readonly || this.readonly, location.concat(['observation',''+i]), false, uilang, haschecker));
        }
        if (this.elements.task.length >0) {
            var termattrs = [];
            if (this.elements.term.length > 0) {
                for (var attr in this.elements.term[0].attrs) {
                    if (this.elements.term[0].attrs[attr]) {
                        termattrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.elements.term[0].attrs[attr]) + '"');
                    }
                }
            }
            html.push('<tr '+termattrs.join(' ')+'><td><span class="sdmath" data-sdreadonly="true">\\Vdash</span></td>');
        }
        for (var i = 0; i < this.elements.derivmotivation.length; i++) {
            html.push(this.elements.derivmotivation[i].getHtml(readonly || this.readonly, location.concat(['derivmotivation',''+i]), false, uilang, haschecker));
        }
        var termcase;
        if (this.elements.derivmotivation.length === 0) {
            termcase = (this.elements.task.length > 0 ? 'vdash' : 'bullet');
        }
        html.push(this.elements.term[0].getHtml(readonly || this.readonly, location.concat(['term','0']), termcase, uilang, haschecker));
        for (var i=0; i < this.elements.motivation.length; i++) {
            termcase = '';
            html.push(this.elements.relation[i].getHtml(readonly || this.readonly, location.concat(['relation',''+i]), false, uilang, haschecker));
            html.push(this.elements.motivation[i].getHtml(readonly || this.readonly, location.concat(['motivation',''+i]), false, uilang, haschecker));
            if (this.elements.motivation[i].subderivation.length > 0) {
                termcase = 'dots';
                html.push('<tr><td class="sdsubldots"><span class="sdmath" data-sdreadonly="true">\\ldots</span></td>');
            }
            html.push(this.elements.term[i+1].getHtml(readonly || this.readonly, location.concat(['term',''+(i+1)]), termcase, uilang, haschecker));
        }
        //if (!this.issubderiv) {
        //    if (this.elements.task.length > 0) {
        //        html.push('<tr><td><span class="sdmath" data-sdreadonly="true">\\square</span></td>'+this.elements.answer[0].getHtml(readonly || this.readonly, location.concat(['answer','0']))+'</tr>');
        //    } else {
        //        html.push('<tr><td><span class="sdmath" data-sdreadonly="true">\\square</span></td><td></td><td class="sdnoteplace"></td></tr>');
        //    }
        //}
        var ansattrs = [];
        if (this.elements.task.length > 0 && this.elements.task[0].answer) {
            for (var attr in this.elements.task[0].answer.attrs) {
                if (this.elements.task[0].answer.attrs[attr]) {
                    ansattrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.elements.task[0].answer.attrs[attr]) + '"');
                }
            }
        }
        html.push('<tr '+ansattrs.join(' ')+'><td><span class="sdmath" data-sdreadonly="true">\\square</span></td>');
        if (this.elements.task.length > 0) {
            html.push(this.elements.task[0].getHtmlAnswer(readonly || this.readonly, location.concat(['task','0','answer']), uilang, haschecker));
        } else {
            html.push('<td></td><td class="sdnoteplace"></td>');
        }
        html.push('</tr>');
        html.push('</tbody>\n</table>');
        return html.join('\n');
    };

    /******
     * Return the data of all derivations as an object.
     ******/
    Sderivation.prototype.getData = function(prefix){
        var loc;
        var result = {
            readonly: this.readonly,
            task: [],
            assumption: [],
            observation: [],
            derivmotivation: [],
            term: [],
            relation: [],
            motivation: []
        };
        if (prefix) {
            if (prefix.length) {
                result.loc = prefix.join('_');
            } else {
                result.loc = this.name;
                prefix = [this.name];
            }
        };
        // Get data for this derivation.
        for (var i = 0; i < this.elements.task.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('task');
                loc.push(i);
            };
            result.task.push(this.elements.task[i].getData(loc));
        };
        for (var i = 0; i < this.elements.assumption.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('assumption');
                loc.push(i);
            };
            result.assumption.push(this.elements.assumption[i].getData(loc));
        };
        for (var i = 0; i < this.elements.observation.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('observation');
                loc.push(i);
            };
            result.observation.push(this.elements.observation[i].getData(loc));
        };
        for (var i = 0; i < this.elements.derivmotivation.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('derivmotivation');
                loc.push(i);
            };
            result.derivmotivation.push(this.elements.derivmotivation[i].getData(loc));
        };
        for (var i = 0; i < this.elements.term.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('term');
                loc.push(i);
            };
            result.term.push(this.elements.term[i].getData(loc));
        };
        for (var i = 0; i < this.elements.relation.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('relation');
                loc.push(i);
            };
            result.relation.push(this.elements.relation[i].getData(loc));
        }
        for (var i = 0; i < this.elements.motivation.length; i++) {
            if (prefix) {
                loc = prefix.slice();
                loc.push('motivation');
                loc.push(i);
            };
            result.motivation.push(this.elements.motivation[i].getData(loc));
        }
        return result;
    }
    
    /******
     * Return the data of all derivations as an object. Old version.
     ******/
    Sderivation.prototype.getDataOld = function(){
        var result = {};
        result[this.name] = {
            task: [],
            assumption: [],
            observation: [],
            derivmotivation: [],
            term: [],
            relation: [],
            motivation: []
        }
        // Get data for this derivation.
        for (var i = 0; i < this.elements.task.length; i++) {
            result[this.name].task.push(this.elements.task[i].getData());
        }
        for (var i = 0; i < this.elements.assumption.length; i++) {
            result[this.name].assumption.push(this.elements.assumption[i].getData());
        }
        for (var i = 0; i < this.elements.observation.length; i++) {
            result[this.name].observation.push(this.elements.observation[i].getData());
        }
        for (var i = 0; i < this.elements.derivmotivation.length; i++) {
            result[this.name].derivmotivation.push(this.elements.derivmotivation[i].getData());
        }
        for (var i = 0; i < this.elements.term.length; i++) {
            result[this.name].term.push(this.elements.term[i].getData());
        }
        for (var i = 0; i < this.elements.relation.length; i++) {
            result[this.name].relation.push(this.elements.relation[i].getData());
        }
        for (var i = 0; i < this.elements.motivation.length; i++) {
            result[this.name].motivation.push(this.elements.motivation[i].getData());
        }
        // Get data of subderivations in observations.
        for (var i = 0; i < this.elements.observation.length; i++) {
            var motiv = this.elements.observation[i].motivation;
            for (var j = 0; j < motiv.subderivation.length; j++) {
                $.extend(result, motiv.subderivation[j].getData());
            }
        }
        // Get data of subderivations in derivmotivation.
        for (var i = 0; i < this.elements.derivmotivation.length; i++) {
            var motiv = this.elements.derivmotivation[i];
            for (var j = 0; j < motiv.subderivation.length; j++) {
                $.extend(result, motiv.subderivation[j].getData());
            }
        }
        // Get data of subderivations in motivations.
        for (var i = 0; i < this.elements.motivation.length; i++) {
            var motiv = this.elements.motivation[i];
            for (var j = 0; j < motiv.subderivation.length; j++) {
                $.extend(result, motiv.subderivation[j].getData());
            }
        }
        return result;
    }

    /******
     * Return the data of the derivation in old JSON format. (For checker)
     ******/
    Sderivation.prototype.jsonobject = function(counter){
        var derivobj = {};
        if (typeof(counter) === 'undefined') {
            derivobj.name = this.name || 'derivation0';
            counter = {index: 0, name: derivobj.name};
        } else {
            derivobj.name = counter.name + 'sub' + counter.index++;
        }

        // Get data for this derivation.
        derivobj.task = [];
        for (var i = 0; i < this.elements.task.length; i++) {
            derivobj.task.push(this.elements.task[i].jsonobject());
        }
        derivobj.assumptions = [];
        for (var i = 0; i < this.elements.assumption.length; i++) {
            derivobj.assumptions.push(this.elements.assumption[i].jsonobject());
        }
        derivobj.observations = [];
        for (var i = 0; i < this.elements.observation.length; i++) {
            derivobj.observations.push(this.elements.observation[i].jsonobject(counter));
        }
        
        derivobj.derivmotivation = [];
        for (var i = 0; i < this.elements.derivmotivation.length; i++) {
            derivobj.derivmotivation.push(this.elements.derivmotivation[i].jsonobject(counter));
        }

        derivobj.chain = {
            terms: [],
            relations: [],
            motivations: []
        };
        for (var i = 0; i < this.elements.term.length; i++) {
            derivobj.chain.terms.push(this.elements.term[i].jsonobject());
        }
        for (var i = 0; i < this.elements.relation.length; i++) {
            derivobj.chain.relations.push(this.elements.relation[i].jsonobject());
        }
        for (var i = 0; i < this.elements.motivation.length; i++) {
            derivobj.chain.motivations.push(this.elements.motivation[i].jsonobject(counter));
        }
        
        return derivobj;
    }
    
    /******
     * Get data for checker.
     * @param {Array} loc - location path
     * @param {Boolean} isstep - true, if this is a derivation step
     ******/
    Sderivation.prototype.getCheckData = function(loc, isstep){
        loc = loc.slice();
        var location = loc.join('_');
        var result = {
            question: [],
            assumptions: [],
            observations: [],
            taskJustification: [],
            terms: [],
            relations: [],
            justifications: [],
            answer: [],
            loc: location
        };
        if (isstep) {
            result.steptype = 'task';
        };
        if (this.elements.task.length > 0) {
            var task = this.elements.task[0].jsonobject();
            var questionobject = {
                text: task.question,
                type: task.type,
                loc: location + '_task_0'
            };
            result.question.push(questionobject);
            result.answer = [
                {
                    text: task.answer,
                    loc: location + '_task_0_answer'
                }
            ];
        };
        var i, len;
        for (i = 0, len = this.elements.assumption.length; i < len; i++){
            result.assumptions.push(this.elements.assumption[i].getCheckData(loc.concat(['assumption', i])));
        };
        for (i = 0, len = this.elements.observation.length; i < len; i++){
            result.observations.push(this.elements.observation[i].getCheckData(loc.concat(['observation', i])));
        };
        for (i = 0, len = this.elements.derivmotivation.length; i < len; i++){
            result.taskJustification.push(this.elements.derivmotivation[i].getCheckData(loc.concat(['derivmotivation', i])));
        };
        for (i = 0, len = this.elements.term.length; i < len; i++){
            result.terms.push(this.elements.term[i].getCheckData(loc.concat(['term', i])));
        };
        for (i = 0, len = this.elements.relation.length; i < len; i++){
            result.relations.push(this.elements.relation[i].getCheckData(loc.concat(['relation', i])));
        };
        for (i = 0, len = this.elements.motivation.length; i < len; i++){
            result.justifications.push(this.elements.motivation[i].getCheckData(loc.concat(['motivation', i])));
        };
        return result;
    }
    
    /******
     * Edit data to location.
     ******/
    Sderivation.prototype.editLocation = function(location, data){
        var element = location.shift();
        var position = location.shift();
        return this.elements[element] && this.elements[element][position] && this.elements[element][position].editLocation(location, data);
    }
    
    /******
     * Set attributes to location.
     ******/
    Sderivation.prototype.setLocationAttrs = function(location, attrs){
        var element = location.shift();
        var position = location.shift();
        return this.elements[element] && this.elements[element][position] && this.elements[element][position].setLocationAttrs(location, attrs);
    }
    
    /******
     * Get data from location.
     ******/
    Sderivation.prototype.getLocationData = function(location){
        var element = location.shift();
        var position = location.shift();
        return this.elements[element] && this.elements[element][position] && this.elements[element][position].getLocationData(location);
    }
    
    /******
     * Hide subderivation in location.
     ******/
    Sderivation.prototype.hideLocation = function(location){
        var result;
        if (location.length === 2 && location[0] === 'motivation'){
            var mot = location.shift();
            var pos = location.shift();
            result = this.elements[mot][pos].hideSubderivs();
        } else if (location.length === 3 && location[0] === 'observation') {
            var obs = location.shift();
            var pos = location.shift();
            var mot = location.shift();
            result = this.elements[obs][pos][mot].hideSubderivs();
        } else {
            var element = location.shift();
            var pos = location.shift();
            switch (element){
                case 'observation':
                    var mot = location.shift();
                    var subder = location.shift();
                    var subderpos = location.shift();
                    result = this.elements.observation[pos][mot][subder][subderpos].hideLocation(location);
                    break;
                case 'motivation':
                case 'derivationmotivation':
                    var subder = location.shift();
                    var subderpos = location.shift();
                    result = this.elements[element][pos][subder][subderpos].hideLocation(location);
                    break;
                default:
                    result = false;
                    break;
            }
        }
        return result;
    }
    
    /******
     * Show subderivation in location.
     ******/
    Sderivation.prototype.showLocation = function(location){
        var result;
        if (location.length === 2 && location[0] === 'motivation'){
            var mot = location.shift();
            var pos = location.shift();
            result = this.elements[mot][pos].showSubderivs();
        } else if (location.length === 3 && location[0] === 'observation') {
            var obs = location.shift();
            var pos = location.shift();
            var mot = location.shift();
            result = this.elements[obs][pos][mot].showSubderivs();
        } else {
            var element = location.shift();
            var pos = location.shift();
            switch (element){
                case 'observation':
                    var mot = location.shift();
                    var subder = location.shift();
                    var subderpos = location.shift();
                    result = this.elements.observation[pos][mot][subder][subderpos].showLocation(location);
                    break;
                case 'motivation':
                case 'derivationmotivation':
                    var subder = location.shift();
                    var subderpos = location.shift();
                    result = this.elements[element][pos][subder][subderpos].showLocation(location);
                    break;
                default:
                    result = false;
                    break;
            }
        }
        return result;
    }
    
    /******
     * Check if the location exists in the derivation.
     ******/
    Sderivation.prototype.locationExists = function(location){
        var element = location.shift();
        var position = location.shift();
        return (this.elements[element] && this.elements[element][position] && this.elements[element][position].locationExists(location)) || false;
    }
    
    /******
     * Get elemtypes of the location.
     ******/
    Sderivation.prototype.getLocationTypes = function(location, issubder){
        if (location.length > 3) {
            // If location is in subderivations.
            var element = location.shift();
            var position = location.shift();
            if ((element === 'motivation' || element === 'derivmotivation') && location.length > 2 ) {
                var subder = location.shift();
                var subdpos = location.shift();
                return this.elements[element][position][subder][subdpos].getLocationTypes(location, true);
            } else if (element === 'observation' && location.length > 3) {
                var mot = location.shift();
                var subder = location.shift();
                var subdpos = location.shift();
                return this.elements[element][position][mot][subder][subdpos].getLocationTypes(location, true);
            }
        } else if (location.length > 1) {
            // Locations in this derivation (not in subderivs).
            var element = location.shift();
            var position = location.shift();
            var result = [];
            switch (element){
                case 'task':
                    if (this.elements.assumption.length === 0 &&
                        this.elements.observation.length === 0 &&
                        this.elements.derivmotivation.length === 0) {
                        result.push('removabletask');
                    }
                    if (!issubder) {
                        result.push('dsteptask');
                        result.push('dstepremove');
                    }
                    if (this.elements.derivmotivation.length === 0) {
                        result.push('hasnoderivmot');
                    }
                    if (this.elements.derivmotivation.length === 1 && this.elements.derivmotivation[0].nOfSubs() === 0) {
                        result.push('removablederivmot');
                    }
                    break;
                case 'answer':
                    result.push('answer');
                    break;
                case 'term':
                    if (position === '0' && this.elements.task.length === 0) {
                        result.push('firstterm');
                        if (issubder) {
                            result.push('insubder');
                        } else {
                            result.push('dsteptask');
                            result.push('dstepremove');
                        }
                    }
                    if (position == this.elements.term.length - 1 && issubder) {
                        result.push('lastterm');
                    }
                    break;
                case 'observation':
                    if (element === 'observation' && location.length > 0 && location[0] === 'motivation') {
                        result.push('obsmotivation');
                    }
                    if (element === 'observation' && location.length > 0 && location[0] === 'declaration') {
                        result.push('obsdeclaration');
                    }
                    if (element === 'observation' && location.length > 0 && location[0] === 'proposition') {
                        result.push('obsproposition');
                    }
                    break;
                case 'derivmotivation':
                    if (element === 'derivmotivation') {
                        result.push('derivmotivation');
                    }
                    break;
                default:
                    result = [];
            }
            return result;
        }
        return [];
    }
    
    /******
     * Add a new step with given data into the given location and position.
     ******/
    Sderivation.prototype.addStep = function(location, data){
        if (location.length === 2 && location[0] === 'term') {
            // Step in this derivation (not in subderivation).
            var pos = (location[1] | 0);
            if (pos === 0) {
                var reltext = '';
                if (this.elements.relation.length > 0) {
                    reltext = this.elements.relation[0].text;
                }
                var rel = new Sdrelation($.extend({text: reltext, readonly: false}, data.relation));
                var mot = new Sdmotivation($.extend({text: '', subderivation: [], readonly: false}, data.motivation));
                var ter = new Sdterm($.extend({text: this.elements.term[0].text, readonly: false}, data.term));
            } else {
                var rel = new Sdrelation($.extend({text: this.elements.relation[pos-1].text, readonly: false}, data.relation));
                var mot = new Sdmotivation($.extend({text: '', subderivation: [], readonly: false}, data.motivation));
                var ter = new Sdterm($.extend({text: this.elements.term[pos].text, readonly: false}, data.term));
            }
            this.elements.relation.splice(pos, 0, rel);
            this.elements.motivation.splice(pos, 0, mot);
            this.elements.term.splice(pos + 1, 0, ter);
        } else {
            // Step in subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements.observation[location[1]].motivation.subderivation[location[4]];
                location = location.slice(5);
            } else if (location[0] === 'derivmotivation'){
                var deriv = this.elements.derivmotivation[0].subderivation[location[3]];
                location = location.slice(4);
            } else {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addStep(location, data);
        }
    }
    
    /******
     * Remove the step in given location and position. Return the old data to be put in undo/redo-stack.
     ******/
    Sderivation.prototype.removeStep = function(location, data){
        var olddata = {};
        if (location.length === 2) {
            // Step in this derivation (not in subderivation).
            if (location[0] === 'relation' || location[0] === 'motivation') {
                var pos = (location[1] | 0);
            }
            olddata.relation = this.elements.relation.splice(pos,1)[0];
            var mot = this.elements.motivation.splice(pos,1)[0];
            olddata.motivation = mot.getData();
            olddata.term = this.elements.term.splice(pos + 1, 1)[0];
        } else {
            // Step in some subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements.observation[location[1]].motivation.subderivation[location[4]];
                location = location.slice(5);
            } else if (location[0] === 'derivmotivation'){
                var deriv = this.elements.derivmotivation[0].subderivation[location[3]];
                location = location.slice(4);
            } else {
                var deriv = this.elements.motivation[location[1]].subderivation[location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeStep(location, data);
        }
        return olddata;
    }
    
    /******
     * Add an assumption in given location with given data.
     ******/
    Sderivation.prototype.addAssumption = function(location, data){
        if (location.length === 3 && location[0] === 'assumption') {
            // Assumption in this derivation (not in subderivation).
            var pos = (location[1] | 0);
            if (pos < this.elements.assumption.length) {
                pos++;
                var ass = new Sdassumption($.extend({text: '', label: Sderivation.numtoalpha(pos),  readonly: false}, data));
                this.elements.assumption.splice(pos, 0, ass);
                for (var i = pos+1; i < this.elements.assumption.length; i++) {
                    this.elements.assumption[i].shiftLabel(1);
                }
            }
        } else {
            // Assumption in subderivation of observation motivation, derivmotivation or motivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addAssumption(location, data);
        }
    }
    
    /******
     * Remove the assumption in given location. Return the old data.
     ******/
    Sderivation.prototype.removeAssumption = function(location, data){
        var olddata = {};
        if (location.length === 3 && location[0] === 'assumption') {
            // Assumption in this derivation (not in subderivation).
            var pos = (location[1] | 0);
            olddata = this.elements.assumption.splice(pos, 1)[0];
            for (var i = pos; i < this.elements.assumption.length; i++) {
                this.elements.assumption[i].shiftLabel(-1);
            }
        } else {
            // Assumption in subderivation of observation motivation, derivmotivation or motivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeAssumption(location, data);
        }
        return olddata;
    }

    /******
     * Add an observation in given location with given data.
     ******/
    Sderivation.prototype.addObservation = function(location, data){
        if ((location.length === 2 || location.length === 3) && location[0] === 'observation') {
            // Observation of this derivation
            var pos = (location[1] | 0);
            if (pos < this.elements.observation.length) {
                pos++;
                var obs = new Sdobservation($.extend({text: '', label: ''+(pos+1), motivation: {text: '', readonly: false, subderivation: []}, readonly: false}, data));
                this.elements.observation.splice(pos, 0, obs);
                for (var i = pos+1; i < this.elements.observation.length; i++) {
                    this.elements.observation[i].shiftLabel(1);
                }
            }
        } else {
            // Observation is in some subderivation
            if (location[0] === 'observation') {
                // In subderivation of observation
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                // in subderivation of derivation motivation or some step.
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addObservation(location, data);
        }
    }

    /******
     * Remove the observation in given location. Return the old data.
     ******/
    Sderivation.prototype.removeObservation = function(location, data){
        var olddata = {};
        if (location.length === 3 && location[0] === 'observation') {
            // Observation in this derivation (not in subderivation).
            // Remove initiated in obsmotivation.
            location.pop();
            var pos = (location[1] | 0);
            olddata = this.elements.observation.splice(pos, 1)[0];
            for (var i = pos; i < this.elements.observation.length; i++) {
                this.elements.observation[i].shiftLabel(-1);
            }
        } else {
            // Observation in subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeObservation(location, data);
        }
        return olddata;
    }

    /******
     * Add the task.
     ******/
    Sderivation.prototype.addTask = function(location, data){
        if (location.length === 2 && location[0] === 'term' && location[1] === '0') {
            // Task in this derivation (not in subderivation).
            var pos = 0;
            if (this.elements.task.length === 0 && this.elements.assumption.length === 0 && this.elements.observation.length === 0) {
                var task = new Sdtask($.extend({text: '', readonly: false}, data));
                this.elements.task = [task];
            }
        } else {
            // Task in some subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location [2]][location[3]][location[4]];
                location = location.slice(5);
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addTask(location, data);
        }
    }
    
    /******
     * Remove task, if there are no assumptions, observations nor derivationmotivations.
     ******/
    Sderivation.prototype.removeTask = function(location, data){
        var olddata = false;
        if (location.length === 2 && location[0] === 'task') {
            // Task in this derivation (not in subderivation).
            var pos = 0;
            if (this.elements.assumption.length === 0 &&
                this.elements.observation.length === 0 &&
                this.elements.derivmotivation.length === 0) {
                // There are no assumptions, observations nor derivmotivation.
                olddata = this.elements.task.splice(pos,1)[0];
            }
        } else {
            // Task in subderivation.
            if (location[0] === 'observation') {
                // location is of form: 'observation_n_motivation_subderivation_m_...'
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                // location is of form: 'motivation_subderivation_m_...' or 'derivationmotivation_subderivation_m_...'
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeTask(location, data);
        }
        return olddata;
    }
    
    /******
     * Add derivation motivation
     ******/
    Sderivation.prototype.addDerivMotivation = function(location, data){
        if (location.length === 2 && location[0] === 'task' && location[1] === '0') {
            if (this.elements.derivmotivation.length === 0) {
                var derivmot = new Sdmotivation($.extend({text: '', readonly: false}, data));
                this.elements.derivmotivation = [derivmot];
            }
        } else {
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location [2]][location[3]][location[4]];
                location = location.slice(5);
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addDerivMotivation(location, data);
        }
    }
    
    /******
     * Remove derivation motivation
     ******/
    Sderivation.prototype.removeDerivMotivation = function(location, data){
        var olddata = false;
        if (location.length === 2 && location[0] === 'derivmotivation') {
            var pos = 0;
            //if (this.elements.derivmotivation[0].nOfSubs() === 0) {
            olddata = this.elements.derivmotivation.splice(0,1)[0];
            olddata = olddata.getData();
            //}
        } else {
            if (location[0] === 'observation') {
                // location is of form: 'observation_n_motivation_subderivation_m_...'
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                // location is of form: 'motivation_subderivation_m_...' or 'derivationmotivation_subderivation_m_...'
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeDerivMotivation(location, data);
        }
        return olddata;
    }
    
    /******
     * Add a subderivation in given location with given data.
     ******/
    Sderivation.prototype.addSubderivation = function(location, data){
        if (location.length === 5 && location[0] === 'observation' && location[2] === 'motivation' && location[3] === 'subderivation') {
            // Subderivation of observation (in this derivation).
            var obspos = (location[1] | 0);
            var pos = (location[4] | 0);
            pos++;
            if (obspos < this.elements.observation.length) {
                this.elements.observation[obspos].motivation.addSubderivation(data, pos);
            }
        } else if (location.length === 4 && location[0] === 'motivation' && location[2] === 'subderivation'){
            // Subderivation of motivation (in this derivation).
            var motpos = (location[1] | 0);
            var pos = (location[3] | 0);
            pos++;
            if (motpos < this.elements.motivation.length && pos <= this.elements.motivation[motpos].nOfSubs()) {
                this.elements.motivation[motpos].addSubderivation(data, pos);
            }
        } else if (location.length === 4 && location[0] === 'derivmotivation' && location[2] === 'subderivation'){
            // Subderivation of derivmotivation (in this derivation).
            var motpos = 0;
            var pos = (location[3] | 0);
            if (pos < this.elements.derivmotivation[0].nOfSubs()) {
                pos++;
                this.elements.derivmotivation[motpos].addSubderivation(data, pos);
            }
        } else {
            // Subderivation of some subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            deriv.addSubderivation(location, data);
        }
    }
    
    /******
     * Remove a subderivation in given location with given data.
     ******/
    Sderivation.prototype.removeSubderivation = function(location){
        var olddata = false;
        if (location.length === 5 && location[0] === 'observation' && location[2] === 'motivation' && location[3] === 'subderivation') {
            // Subderivation of observation (in this derivation).
            var obspos = (location[1] | 0);
            var pos = (location[4] | 0);
            if (obspos < this.elements.observation.length && pos < this.elements.observation[obspos].motivation.nOfSubs()) {
                olddata = this.elements.observation[obspos].motivation.removeSubderivation(pos);
            }
        } else if (location.length === 4 && location[0] === 'motivation' && location[2] === 'subderivation'){
            // Subderivation of motivation (in this derivation).
            var motpos = (location[1] | 0);
            var pos = (location[3] | 0);
            if (motpos < this.elements.motivation.length && pos < this.elements.motivation[motpos].nOfSubs()) {
                olddata = this.elements.motivation[motpos].removeSubderivation(pos);
            }
        } else if (location.length === 4 && location[0] === 'derivmotivation' && location[2] === 'subderivation'){
            // Subderivation of derivmotivation (in this derivation).
            var motpos = 0;
            var pos = (location[3] | 0);
            if (pos < this.elements.derivmotivation[0].nOfSubs()) {
                olddata = this.elements.derivmotivation[motpos].removeSubderivation(pos);
            }
        } else {
            // Subderivation of some subderivation.
            if (location[0] === 'observation') {
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]][location[4]];
                location = location.slice(5)
            } else if (location[0] === 'derivmotivation' || location[0] === 'motivation'){
                var deriv = this.elements[location[0]][location[1]][location[2]][location[3]];
                location = location.slice(4);
            }
            olddata = deriv.removeSubderivation(location);
        }
        return olddata;
    }
    
    /****************************************
     * Virtual Sdelement class
     * Constructor
     ****************************************/
    var Sdelement = function(options){
        if (options) {
            this.text = options.text;
            this.attrs = options.attrs || {};
            this.options = options;
        }
    }
    
    /******
     * Get Html-code
     ******/
    Sdelement.prototype.getHtml = function(){
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        }
        return '<tr '+attrs.join(' ')+'><td>'+this.loc+'</td><td>'+this.text+'</td><td class="sdnoteplace"></td></tr>';
    }
    
    /******
     * Return the data of this element.
     ******/
    Sdelement.prototype.getData = function(loc){
        var result =  {
            text: this.text,
            readonly: this.readonly,
            fillable: !!this.fillable,
            attrs: this.attrs
        };
        if (loc && typeof(loc.length) === 'number') {
            result.loc = loc.join('_');
        };
        return result;
    };
    
    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdelement.prototype.jsonobject = function(){
        // Implemented in each element.
        return '';
    }
    
    /**
     * Set attributes
     * @param {Array} location   Location array
     * @param {Object} attrs   Attributes that should be changed.
     */
    Sdelement.prototype.setLocationAttrs = function(location, attrs) {
        attrs = attrs || {};
        if (location.length === 0) {
            for (var key in attrs) {
                this.attrs[key] = attrs[key];
            };
        } else {
            var elem = location.shift();
            if (this instanceof Sdobservation && elem === 'motivation') {
                this.motivation.setLocationAttrs(location, attrs);
            } else if (this instanceof Sdobservation && elem === 'declaration') {
                this.definition.setLocationAttrs(location, attrs);
                if (this.type === 'declaration') {
                    this.setLocationAttrs([], attrs);
                }
            } else if (this instanceof Sdobservation && elem === 'proposition') {
                for (var key in attrs) {
                    this.attrs[key] = attrs[key];
                };
            } else if (this instanceof Sdassumption && elem === 'proposition') {
                for (var key in attrs) {
                    this.attrs[key] = attrs[key];
                };
            } else if (this instanceof Sdtask && elem === 'answer') {
                this.answer.setLocationAttrs(location, attrs);
            } else {
                var pos = location.shift();
                this[elem][pos].setLocationAttrs(location, attrs);
            };
        }
    };
    
    /******
     * Edit data to this location.
     ******/
    Sdelement.prototype.editLocation = function(location, data){
        var olddata = {};
        if (location.length === 0) {
            for (var attr in data) {
                olddata[attr] = this[attr];
                this[attr] = data[attr]
            }
        } else {
            var elem = location.shift();
            if (this instanceof Sdobservation && elem === 'motivation') {
                olddata = this.motivation.editLocation(location, data);
            } else if (this instanceof Sdobservation && elem === 'declaration') {
                olddata = this.definition.editLocation(location, data);
            } else if (this instanceof Sdobservation && elem === 'proposition') {
                for (var attr in data) {
                    olddata[attr] = this[attr];
                    this[attr] = data[attr];
                };
            } else if (this instanceof Sdassumption && elem === 'proposition') {
                for (var attr in data) {
                    olddata[attr] = this[attr];
                    this[attr] = data[attr];
                };
            } else if (this instanceof Sdtask && elem === 'answer') {
                olddata = this.answer.editLocation(location, data);
            } else {
                var pos = location.shift();
                olddata = this[elem][pos].editLocation(location, data);
            };
        };
        return olddata;
    }
    
    /******
     * Get data from location.
     ******/
    Sdelement.prototype.getLocationData = function(location){
        var data;
        if (location.length === 0) {
            data = {text: this.text, readonly: this.readonly};
        } else {
            var elem = location.shift();
            if (this instanceof Sdobservation && elem === 'motivation') {
                data = this.motivation.getLocationData(location);
            } else if (this instanceof Sdobservation && elem === 'declaration') {
                data = this.definition.getLocationData(location);
            } else if (this instanceof Sdobservation && elem === 'definition') {
                data = this.definition.getLocationData(location);
            } else if (this instanceof Sdobservation && elem === 'proposition') {
                data = {text: this.text, readonly: this.readonly};
            } else if (this instanceof Sdassumption && elem === 'proposition') {
                data = {text: this.text, readonly: this.readonly};
            } else if (this instanceof Sdtask && elem === 'answer') {
                data = this.answer.getLocationData(location);
            } else {
                var pos = location.shift();
                data = this[elem] && this[elem][pos] && this[elem][pos].getLocationData(location);
            }
        }
        return data;
    }
    
    /******
     * Check if given location does exist.
     ******/
    Sdelement.prototype.locationExists = function(location){
        var result = true;
        if (location.length === 0) {
            result = true;
        } else {
            var elem = location.shift();
            if (this instanceof Sdobservation && elem === 'motivation') {
                result = this.motivation.locationExists(location);
            } else {
                var pos = location.shift();
                result = (this[elem] && this[elem][pos] && this[elem][pos].locationExists(location)) || false;
            }
        }
        return result;
    }
    
    /******
     * Return text as text and math, if text should be readonly,
     * replace $math$ with <span class="sdmath">math</span>
     ******/
    Sdelement.prototype.mathtext = function(readonly, text){
        text = escapeHTML(typeof(text) === 'string' ? text : this.text);
        return (readonly ? text.replace(/\$([^\$]+)\$/g, '<span class="sdmath">$1</span>') : text);
    }
    
    /******
     * Check if this element is empty.
     ******/
    Sdelement.prototype.isempty = function(){
        return this.text === '';
    }
    
    /******
     * Get elemtypes of the location (used when assumption or observation is a derivation step)
     * @param {Array} location - location path as an array
     ******/
    Sdelement.prototype.getLocationTypes = function(location){
        location = location.slice();
        if (location.length > 3) {
            // Observation and location is in subderivation
            var mot = location.shift();
            var subder = location.shift();
            var subdpos = location.shift();
            return this[mot][subder][subdpos].getLocationTypes(location, true);
        } else if (location.length === 1 && location[0] === 'motivation') {
            // The location is motivation under derivation step observation.
            return ['obsmotivation', 'dstepobservation', 'dstepremove']
        } else if (location.length === 1 && location[0] === 'declaration') {
            // The location is declaration under derivation step observation.
            return ['obsdeclaration', 'dstepobservation', 'dstepremove']
        } else {
            // The location is assumption or observation
            if (this instanceof Sdassumption) {
                return ['dsassumption', 'dstepremove'];
            } else {
                return ['dsobservation', 'dstepremove'];
            }
        }
    }
    
    /*****************************************
     * Sdtask class
     * Constructor
     *****************************************/
    var Sdtask = function(options){
        this.type = options && options.type || Sdtask.availtypes[0];
        this.text = options && options.text || '';
        this.answer = new Sdanswer(options && options.answer || {text: '', readonly: options.readonly});
        this.readonly = options && options.readonly || false;
        this.fillable = options && options.fillable || false;
        this.attrs = options.attrs || {};
    }
    
    // Inherit from Sdelement
    Sdtask.prototype = new Sdelement();
    Sdtask.prototype.constructor = Sdtask;

    
    /******
     * Return the data of this element.
     ******/
    Sdtask.prototype.getData = function(loc){
        var result =  {
            text: this.text,
            type: this.type,
            readonly: this.readonly,
            fillable: !!this.fillable,
            attrs: this.attrs
        };
        if (loc && typeof(loc.length) === 'number') {
            result.loc = loc.join('_');
            loc = loc.slice();
            loc.push('answer');
        };
        result.answer = this.answer.getData(loc);
        return result;
    };
    
    /******
     * Get html-code
     ******/
    Sdtask.prototype.getHtml = function(readonly, location, uilang, haschecker){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        readonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        var loc = escapeHTML(location.join('_'));
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        };
        //var qtypetext = Dictionary.localize('question:'+this.type, uilang);
        //var qtype = (haschecker ? '<span class="sdeditor-questiontype" data-qtype="'+this.type+'">'+qtypetext+'</span>' : '');
        return '<td '+attrs.join(' ')+' data-loc="'+loc+'" data-sdtype="sdtask" class="sdtask' + (this.isempty() ? ' sdempty' : '') + '"><span class="sdmathtext" data-sdreadonly="'+readonly+'">'+this.mathtext(readonly)+'</td><td class="sdnoteplace" data-noteloc="'+loc+'"></td></tr>\n';
    }
    
    /******
     * Get html-code of answer
     ******/
    Sdtask.prototype.getHtmlAnswer = function(readonly, location){
        return this.answer.getHtml(readonly, location);
    }
    
    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdtask.prototype.jsonobject = function(){
        // Don't!
        //// Convert mathmode mathquill to textbox mathquill.
        //var text = '$\\text{'+this.text.replace(/\$(.*?)\$/g, '}$1\\text{') + '}$';
        var text = this.text;
        var answer = this.answer;
        // Remove empty '$$' and '\text{}' -strings.
        text = text.replace(/\$\$/g, '').replace(/\\text\{\}/g, '')
        return {question: text, answer: this.answer.jsonobject(), type: this.type};
    }

    /******
     * Get the next question type
     * @param {String} qtype - current qtype
     * @returns {String} the name of the next type.
     ******/
    Sdtask.nextType = function(qtype) {
        var index = Sdtask.availtypes.indexOf(qtype);
        var newqtype = Sdtask.availtypes[(index + 1) % Sdtask.availtypes.length];
        return newqtype;
    };
    
    Sdtask.availtypes = ['calculate', 'simplify', 'solve', 'prove'];

    /*****************************************
     * Sdanswer class
     * Constructor
     *****************************************/
    var Sdanswer = function(options){
        this.text = options && options.text || '';
        this.readonly = options && options.readonly || false;
        this.fillable = options && options.fillable || false;
        this.attrs = options && options.attrs || {};
    }
    
    // Inherit from Sdelement
    Sdanswer.prototype = new Sdelement();
    Sdanswer.prototype.constructor = Sdanswer;
    
    /******
     * Get html-code
     ******/
    Sdanswer.prototype.getHtml = function(readonly, location){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        readonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        var loc = escapeHTML(location.join('_'));
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        };
        return '<td '+attrs.join(' ')+' data-loc="' + loc + '" data-sdtype="sdanswer" class="sdanswer' + (this.isempty() ? ' sdempty':'') + '"><span class="sdmathtext" data-sdreadonly="' + readonly + '">' + this.mathtext(readonly) + '</td><td class="sdnoteplace" data-noteloc="' + loc + '"></td>';
    }
    
    Sdanswer.prototype.jsonobject = function(){
        var text = this.text;
        // Remove empty '$$' and '\text{}' -strings.
        text = text.replace(/\$\$/g, '').replace(/\\text\{\}/g, '')
        return text;
    }
    
    /*****************************************
     * Sdassumption class
     * Constructor
     *****************************************/
    var Sdassumption = function(options){
        if (options) {
            this.name = options.name;
            this.type = 'assumption';
            this.text = options.text || '';
            this.label = options.label || '-';
            this.readonly = options.readonly || false;
            this.fillable = options.fillable || false;
            this.attrs = options.attrs || {};
            this.options = options;
        }
    }
    
    // Inherit from Sdelement
    Sdassumption.prototype = new Sdelement();
    Sdassumption.prototype.constructor = Sdassumption;
    
    /******
     * Get html-code
     * @param {Boolean} readonly - If true, give html in non-editable mode
     * @param {Array} location - an array of location path
     * @param {Boolean} isdstep - If the assumption is a derivation step, add the table around the rows.
     ******/
    Sdassumption.prototype.getHtml = function(readonly, location, isdstep, uilang){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var loc = escapeHTML(location.join('_'));
        var proploc = escapeHTML(loc + '_proposition');
        var html = [];
        if (isdstep) {
            html.push('<table class="sdlayouttable"><tbody>');
        }
        if (this.attrs[attr]) {
            attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
        }
        var attrs = [];
        for (var attr in this.attrs) {
            attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
        }
        html.push('<tr '+attrs.join(' ')+'><td>('+escapeHTML(this.label)+')</td><td data-loc="'+proploc+'" data-sdtype="sdassumption" class="sdassumption'+(this.isempty()?' sdempty':'')+'"><span class="sdmathtext" data-sdreadonly="'+sdreadonly+'">'+this.mathtext(sdreadonly)+'</td><td class="sdnoteplace" data-noteloc="'+proploc+'"></td></tr>\n');
        if (isdstep) {
            html.push('</tbody></table>');
        }
        return html.join('\n');
    }

    /******
     * Return the data of this element.
     ******/
    Sdassumption.prototype.getData = function(loc){
        var result = {
            label: this.label,
            text: this.text,
            readonly: this.readonly,
            fillable: !!this.fillable,
            attrs: this.attrs
        };
        if (loc && typeof(loc.length) === 'number') {
            result.loc = loc.join('_');
        };
        return result;
    }
    
    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdassumption.prototype.jsonobject = function(){
        return [this.label, this.text];
    }
    
    /******
     * Get data for checker.
     * @param {Array} loc - location path
     * @param {Boolean} isstep - true, if this is a derivation step
     ******/
    Sdassumption.prototype.getCheckData = function(loc, isstep){
        var result = {
            label: this.label,
            proposition: {
                expr: this.text,
                loc: loc.join('_') + '_proposition'
            }
        };
        if (isstep) {
            result.steptype = 'assumption';
            result.loc = loc.join('_');
        };
        return result;
    };

    /******
     * Move the label of the assumption with delta.
     ******/
    Sdassumption.prototype.shiftLabel = function(delta){
        var num = Sderivation.alphatonum(this.label) + delta;
        this.label = Sderivation.numtoalpha(num);
    }
    
    
    
    /*****************************************
     * Sdobservation class
     * Constructor
     *****************************************/
    var Sdobservation = function(options){
        if (options) {
            this.name = options.name;
            this.type = (options.type ? options.type : (options.definition ? 'definition' : 'fact'));
            this.text = options.text || '';
            this.label = options.label || '+';
            if (this.type === 'definition' || this.type === 'fact') {
                this.motivation = new Sdmotivation(options.motivation);
            };
            this.readonly = options.readonly || false;
            this.fillable = options.fillable || false;
            this.attrs = options.attrs || {};
            if (options.definition || options.type === 'definition' || options.type === 'declaration') {
                this.definition = new Sddefstring(options.definition || {text: ''});
            };
        };
    };
    
    // Inherit from Sdelement
    Sdobservation.prototype = new Sdelement();
    Sdobservation.prototype.constructor = Sdobservation;
    
    /******
     * Get html-code
     * @param {Boolean} readonly - If true, give html in non-editable mode
     * @param {Array} location - an array of location path
     * @param {Boolean} isdstep - If the observation is a derivation step, add the table around the rows.
     ******/
    Sdobservation.prototype.getHtml = function(readonly, location, isdstep, uilang, haschecker){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var loc = escapeHTML(location.join('_'));
        var decloc = escapeHTML(loc + '_declaration')
        var proploc = escapeHTML(loc + '_proposition');
        var html = [];
        if (isdstep) {
            html.push('<table class="sdlayouttable"><tbody>');
        };
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        }
        html.push('<tr '+attrs.join(' ')+'><td>['+this.label+']</td>');
        switch (this.type) {
            case 'declaration':
                html.push(this.definition.getHtml(readonly, location.concat(['declaration'])) + '</tr>');
                break;
            case 'definition':
                html.push(this.definition.getHtml(readonly, location.concat(['declaration'])) + '</tr><tr><td></td>');
                html.push(this.motivation.getHtml(readonly, location.concat(['motivation']), false, uilang, haschecker)+(this.motivation.subderivation.length > 0 ? '<tr '+attrs.join(' ')+'><td class="sdsubldots"><span class="sdmath" data-sdreadonly="true">\\ldots</span></td>' : '<tr '+attrs.join(' ')+'><td></td>'));
                html.push('<td data-loc="'+proploc+'" data-sdtype="sdobservation" class="sdobservation'+(this.isempty()?' sdempty':'')+'"><span class="sdmath" data-sdreadonly="'+sdreadonly+'">'+escapeHTML(this.text)+'</td><td class="sdnoteplace" data-noteloc="'+proploc+'"></td></tr>\n')
                break;
            case 'fact':
                html.push(this.motivation.getHtml(readonly, location.concat(['motivation']))+(this.motivation.subderivation.length > 0 ? '<tr '+attrs.join(' ')+'><td class="sdsubldots"><span class="sdmath" data-sdreadonly="true">\\ldots</span></td>' : '<tr '+attrs.join(' ')+'><td></td>'));
                html.push('<td data-loc="'+proploc+'" data-sdtype="sdobservation" class="sdobservation'+(this.isempty()?' sdempty':'')+'"><span class="sdmath" data-sdreadonly="'+sdreadonly+'">'+escapeHTML(this.text)+'</td><td class="sdnoteplace" data-noteloc="'+proploc+'"></td></tr>\n')
                break;
            default:
                break;
        }
        if (isdstep) {
            html.push('</tbody></table>');
        }
        return html.join('\n');
    }

    /******
     * Return the data of this element.
     ******/
    Sdobservation.prototype.getData = function(loc){
        var result = {
            label: this.label,
            text: this.text,
            readonly: this.readonly,
            fillable: this.fillable,
            attrs: this.attrs,
            type: this.type
        };
        var defloc;
        if (loc && typeof(loc.length) === 'number') {
            result.loc = loc.join('_');
            defloc = loc.slice();
            defloc.push('declaration');
            loc = loc.slice();
            loc.push('motivation');
        };
        if (this.motivation) {
            result.motivation = this.motivation.getData(loc);
        };
        if (this.definition) {
            result.definition = this.definition.getData(defloc);
        };
        return result;
    }

    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdobservation.prototype.jsonobject = function(counter){
        // Convert mathmode mathquill to textbox mathquill.
        var text = '$\\text{'+ ('$' + this.text + '$').replace(/\$(.*?)\$/g, '}$1\\text{') + '}$';
        // Remove empty '$$' and '\text{}' -strings.
        text = text.replace(/\$\$/g, '').replace(/\\text\{\}/g, '')
        var result =  {label: this.label, text: text};
        if (this.motivation) {
            result.motivation = this.motivation.jsonobject(counter);
        }
        if (this.definition) {
            result.definition = this.definition.jsonobject();
        }
        return result;
    }

    /******
     * Get data for checker.
     * @param {Array} loc - location path
     * @param {Boolean} isstep - true, if this is a derivation step
     ******/
    Sdobservation.prototype.getCheckData = function(loc, isstep){
        var result = {
            label: this.label,
            type: this.type,
            declaration: {
                expr: '',
                loc: loc.join('_') + '_declaration'
            },
            proposition: {
                expr: '',
                loc: loc.join('_') + '_proposition'
            },
            observJustification: {
                text: '',
                hidden: false,
                nestedTask: [],
                loc: loc.join('_') + '_motivation'
            }
        };
        if (isstep) {
            result.steptype = 'observation';
            result.loc = loc.join('_');
        }
        if (this.type === 'definition') {
            result.declaration = this.definition.getCheckData(loc);
        };
        if (this.motivation) {
            result.observJustification = this.motivation.getCheckData(loc.concat(['motivation']));
            result.proposition = {
                expr: this.text,
                loc: loc.join('_') + '_proposition'
            };
        };
        if (this.type === 'declaration') {
            result.declaration = this.definition.getCheckData(loc);
        };
        return result;
    };
    
    /******
     * Add a new step with given data into given location and position
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the next step
     ******/
    Sdobservation.prototype.addStep = function(location, data){
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        deriv.addStep(location, data);
    }
    
    /******
     * Remove the step in given location. Return the old data to be put in undo/redo-stack.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the next step
     ******/
    Sdobservation.prototype.removeStep = function(location, data){
        var olddata = {};
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        olddata = deriv.removeStep(location, data);
        return olddata;
    }
    
    /******
     * Add an assumption in given location with given data.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for assumption
     ******/
    Sdobservation.prototype.addAssumption = function(location, data){
        var deriv = this.motivation[location[1]][location[2]];
        location = location.slice(3);
        deriv.addAssumption(location, data);
    }

    /******
     * Remove the assumption in given location. Return the old data to be put in undo/redo-stack.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the assumption
     ******/
    Sdobservation.prototype.removeAssumption = function(location, data){
        var olddata = {};
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        olddata = deriv.removeAssumption(location, data);
        return olddata;
    }
    
    /******
     * Add an observation in given location with given data.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for observation
     ******/
    Sdobservation.prototype.addObservation = function(location, data){
        var deriv = this.motivation[location[1]][location[2]];
        location = location.slice(3);
        deriv.addObservation(location, data);
    }

    /******
     * Remove the observation in given location. Return the old data to be put in undo/redo-stack.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the observation
     ******/
    Sdobservation.prototype.removeObservation = function(location, data){
        var olddata = {};
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        olddata = deriv.removeObservation(location, data);
        return olddata;
    }
    
    /******
     * Add a task question in given location with given data.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for task question
     ******/
    Sdobservation.prototype.addTask = function(location, data){
        var deriv = this.motivation[location[1]][location[2]];
        location = location.slice(3);
        deriv.addTask(location, data);
    }

    /******
     * Remove the task question in given location. Return the old data to be put in undo/redo-stack.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the task question
     ******/
    Sdobservation.prototype.removeTask = function(location, data){
        var olddata = {};
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        olddata = deriv.removeTask(location, data);
        return olddata;
    }
    
    /******
     * Add a derivation motivation in given location with given data.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for derivation motivation
     ******/
    Sdobservation.prototype.addDerivMotivation = function(location, data){
        var deriv = this.motivation[location[1]][location[2]];
        location = location.slice(3);
        deriv.addDerivMotivation(location, data);
    }

    /******
     * Remove the derivation motivation in given location. Return the old data to be put in undo/redo-stack.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the derivation motivation
     ******/
    Sdobservation.prototype.removeDerivMotivation = function(location, data){
        var olddata = {};
        var deriv = this.motivation.subderivation[location[2]];
        location = location.slice(3);
        olddata = deriv.removeDerivMotivation(location, data);
        return olddata;
    }
    
    /******
     * Add a subderivation with given data into given location and position
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for the next step
     ******/
    Sdobservation.prototype.addSubderivation = function(location, data){
        if (location.length === 3 && location[1] === 'subderivation') {
            // Location is a subderivation
            var pos = (location[2] | 0);
            pos++;
            this.motivation.addSubderivation(data, pos);
        } else {
            // Subderivation of some subderivation.
            var deriv = this.motivation[location[1]][location[2]];
            location = location.slice(3);
            deriv.addSubderivation(location, data);
        };
    };
    
    /******
     * Remove the subderivation from given location. Return old data.
     * (used when an observation is as a derivation step)
     * @param {Array} location - location path as an array
     * @param {Object} data - data for subderivation
     * @returns {Object} the old data
     ******/
    Sdobservation.prototype.removeSubderivation = function(location, data){
        var olddata = {};
        if (location.length === 3 && location[1] === 'subderivation') {
            // Location is a subderivation
            var pos = (location[2] | 0);
            olddata = this.motivation.removeSubderivation(pos);
        } else {
            // Subderivation of some subderivation.
            var deriv = this.motivation[location[1]][location[2]];
            location = location.slice(3);
            olddata = deriv.removeSubderivation(location, data);
        };
        return olddata;
    };
    
    /******
     * Hide subderivation in location
     * (used when on observation is as a derivation step)
     * @param {Array} location - location path as an Array
     ******/
    Sdobservation.prototype.hideLocation = function(location){
        // In this case the location always starts with 'motivation'.
        var result;
        if (location.length > 3) {
            // Location in subderivation's subderivation
            var mot = location.shift(); // 'motivation'
            var subder = location.shift(); //'subderivation'
            var pos = location.shift();
            result = this[mot][subder][pos].hideLocation(location);
        } else {
            result = this.motivation.hideSubderivs();
        };
        return result;
    }
    
    /******
     * Show subderivation in location
     * (used when on observation is as a derivation step)
     * @param {Array} location - location path as an Array
     ******/
    Sdobservation.prototype.showLocation = function(location){
        // In this case the location always starts with 'motivation'.
        var result;
        if (location.length > 3) {
            // Location in subderivation's subderivation
            var mot = location.shift(); // 'motivation'
            var subder = location.shift(); //'subderivation'
            var pos = location.shift();
            result = this[mog][subder][pos].showLocation(location);
        } else {
            result = this.motivation.showSubderivs();
        };
        return result;
    };


    /******
     * Move the label of the observation with delta.
     ******/
    Sdobservation.prototype.shiftLabel = function(delta){
        this.label = ''+((this.label | 0) + delta);
    }



    /*****************************************
     * Sddefstring class
     * Constructor
     *****************************************/
    var Sddefstring = function(options){
        if (options) {
            this.text = options.text || '';
            this.readonly = options.readonly || false;
            this.fillable = options.fillable || false;
            this.attrs = options.attrs || {};
        }
    }
    
    // Inherit from Sdelement
    Sddefstring.prototype = new Sdelement();
    Sddefstring.prototype.constructor = Sddefstring;
    
    /******
     * Get html-code
     ******/
    Sddefstring.prototype.getHtml = function(readonly, location, isdstep, uilang){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var loc = escapeHTML(location.join('_'));
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        };
        var html = '<td '+attrs.join(' ')+' class="sddefinition'+(this.isempty()?' sdempty':'')+'" data-sdtype="sddefinition" data-loc="' + loc + '"><span class="sdmath" data-sdreadonly="'+sdreadonly+'">'+escapeHTML(this.text)+'</span></td><td class="sdnoteplace" data-noteloc="'+loc+'"></td>';
        return html;
    }

    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sddefstring.prototype.jsonobject = function(){
        return '$' + this.text + '$';
    }
    
    /******
     * Return the object for checkdata
     * @param {Array} loc - location array
     ******/
    Sddefstring.prototype.getCheckData = function(loc) {
        var result = {
            expr: this.text,
            loc: loc.join('_') + '_declaration'
        };
        return result;
    };

    /*****************************************
     * Sdterm class
     * Constructor
     *****************************************/
    var Sdterm = function(options){
        if (options) {
            this.text = options.text || '';
            this.readonly = options.readonly || false;
            this.fillable = options.fillable || false;
            this.attrs = options.attrs || {};
            this.options = options;
        }
    }
    
    // Inherit from Sdelement
    Sdterm.prototype = new Sdelement();
    Sdterm.prototype.constructor = Sdterm;
    
    /******
     * Get html-code
     ******/
    Sdterm.prototype.getHtml = function(readonly, location, termcase, uilang){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var loc = escapeHTML(location.join('_'));
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        }
        var html = '';
        if (termcase === 'vdash' || termcase === 'bullet' || termcase === 'dots') {
            html += '';
        } else {
            html += '<tr '+attrs.join(' ')+'><td></td>';
        }
        html += '<td '+attrs.join(' ')+' data-loc="'+loc+'" data-sdtype="sdterm" class="sdterm'+(this.isempty()?' sdempty':'')+'"><span class="sdmath" data-sdreadonly="'+sdreadonly+'">'+escapeHTML(this.text)+'</td><td class="sdnoteplace" data-noteloc="'+loc+'"></td></tr>\n';
        return html;
    }

    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdterm.prototype.jsonobject = function(){
        return '$' + this.text + '$';
    }
    
    /******
     * Get data for checker.
     * @param {Array} loc - location path
     ******/
    Sdterm.prototype.getCheckData = function(loc){
        var result = {
            expr: this.text,
            loc: loc.join('_'),
        };
        return result;
    };
    
    /*****************************************
     * Sdrelation class
     * Constructor
     *****************************************/
    var Sdrelation = function(options){
        if (options) {
            this.text = options.text;
            this.readonly = options.readonly || false;
            this.fillable = options.fillable || false;
            this.attrs = options.attrs || {};
            this.options = options;
        }
    }
    
    // Inherit from Sdelement
    Sdrelation.prototype = new Sdelement();
    Sdrelation.prototype.constructor = Sdrelation;
    
    /******
     * Get html-code
     ******/
    Sdrelation.prototype.getHtml = function(readonly, location, issdstep, uilang){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        }
        return '<tr '+attrs.join(' ')+'><td data-loc="'+escapeHTML(location.join('_'))+'" data-sdtype="sdrelation" class="sdrelation'+(this.isempty()?' sdempty':'')+'"><span class="sdmath" data-sdreadonly="'+sdreadonly+'">'+escapeHTML(this.text)+'</td>';
    }

    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdrelation.prototype.jsonobject = function(){
        return '$' + this.text + '$';
    }

    /******
     * Get data for checker.
     * @param {Array} loc - location path
     ******/
    Sdrelation.prototype.getCheckData = function(loc){
        var result = {
            expr: this.text,
            loc: loc.join('_'),
        };
        return result;
    };
    
    /*****************************************
     * Sdmotivation class
     * Constructor
     *****************************************/
    var Sdmotivation = function(options){
        options = $.extend({
            text: '',
            readonly: false,
            fillable: false,
            subderiv: [],
            subshidden: false
        }, options);
        this.name = 'motivation';
        if (options) {
            this.text = options.text;
            this.readonly = options.readonly;
            this.fillable = options.fillable;
            this.subshidden = options.subshidden;
            this.subderivation = [];
            for (var i = 0; i < options.subderiv.length; i++) {
                var subderiv = options.subderiv[i];
                this.addSubderivation(subderiv, i);
            }
            this.attrs = options.attrs || {};
            this.options = options;
        }
    }
    
    // Inherit from Sdelement
    Sdmotivation.prototype = new Sdelement();
    Sdmotivation.prototype.constructor = Sdmotivation;
    
    /******
     * Add a new subderivation with data in position pos.
     ******/
    Sdmotivation.prototype.addSubderivation = function(data, pos){
        if (typeof(pos) === 'undefined') {
            pos = this.subderivation.length;
        }
        data.issubderiv = true;
        var subder = new Sderivation(data);
        this.subderivation.splice(pos, 0, subder);
    }
    
    /******
     * Remove subderivation in position pos.
     ******/
    Sdmotivation.prototype.removeSubderivation = function(pos){
        var data = this.subderivation.splice(pos, 1)[0];
        return data.getData();
    }
    
    /******
     * Hide subderivations
     ******/
    Sdmotivation.prototype.hideSubderivs = function(){
        var oldval = this.subshidden;
        this.subshidden = true;
        return oldval;
    }
    
    /******
     * Show subderivations
     ******/
    Sdmotivation.prototype.showSubderivs = function(){
        var oldval = this.subshidden;
        this.subshidden = false;
        return oldval;
    }
    
    /******
     * Get html-code
     ******/
    Sdmotivation.prototype.getHtml = function(readonly, location, issdstep, uilang, haschecker){
        // Edit-mode: readonly = this.readonly,  View-mode: readonly = !this.fillable
        var sdreadonly = (!readonly && this.readonly) || (readonly && !this.fillable);
        readonly = readonly || this.readonly;
        var loc = escapeHTML(location.join('_'));
        var attrs = [];
        for (var attr in this.attrs) {
            if (this.attrs[attr]) {
                attrs.push('data-' + escapeHTML(attr) + '="' + escapeHTML(this.attrs[attr]) + '"');
            }
        }
        var html = '<td '+attrs.join(' ')+' data-loc="'+loc+'" data-sdtype="sdmotivation" class="sdmotivation'+(this.isempty()?' sdempty':'')+'"><span class="sdmathtext" data-sdreadonly="'+sdreadonly+'">'+this.mathtext(sdreadonly)+'</td><td class="sdnoteplace" data-noteloc="'+loc+'"></td></tr>\n';
        for (var i = 0; i < this.subderivation.length; i++) {
            html += '<tr class="sdsubderivationrow"><td></td><td data-loc="'+escapeHTML(location.join('_'))+'_subderivation_'+i+'" data-sdtype="sdsubderivation" class="sdsubderivation" colspan="2"><div class="subderivation'+(this.subshidden ? ' sdsubhidden' : '')+'">';
            html += this.subderivation[i].getHtml(readonly, location.concat(['subderivation', ''+i]), false, uilang, haschecker);
            html += '</td></tr>';
        }
        return html;
    }
    
    /******
     * Return the data of this element.
     ******/
    Sdmotivation.prototype.getData = function(loc){
        var result = {
            text: this.text,
            readonly: this.readonly,
            fillable: this.fillable,
            subshidden: this.subshidden,
            subderiv: [],
            attrs: this.attrs
        };
        if (loc && typeof(loc.length) === 'number') {
            result.loc = loc.join('_');
            loc = loc.slice();
            loc.push('subderivation');
            loc.push('');
        };
        for (var i = 0; i < this.subderivation.length; i++) {
            if (loc) {
                loc[loc.length - 1] = i;
            };
            result.subderiv.push(this.subderivation[i].getData(loc));
        };
        return result;
    };
    
    /******
     * Return the data of this element. (Old format)
     * TODO!
     ******/
    Sdmotivation.prototype.getDataOld = function(){
        return {text: this.text, readonly: this.readonly, subderiv: this.subderivnames.slice(0)};
    }

    /******
     * Return the data as JSON object (old format for checker).
     ******/
    Sdmotivation.prototype.jsonobject = function(counter){
        var mot = [this.text];
        for (var i = 0, length = this.subderivation.length; i < length; i++) {
            mot.push({
                derivations: [this.subderivation[i].jsonobject(counter)]
            });
        }
        return mot;
    }
    
    /******
     * Get data for checker.
     * @param {Array} loc - location path
     ******/
    Sdmotivation.prototype.getCheckData = function(loc){
        var result = {
            text: this.text,
            hidden: this.subshidden,
            loc: loc.join('_'),
            nestedTask: []
        };
        for (var i = 0, len = this.subderivation.length; i < len; i++) {
            result.nestedTask.push(this.subderivation[i].getCheckData(loc.concat(['subderivation', i])));
        };
        return result;
    };
    
    /******
     * Return the number of subderivations in this motivation.
     ******/
    Sdmotivation.prototype.nOfSubs = function(){
        return this.subderivation.length;
    }
    
    /*************************************
    * Some constants for Sderivation
    **************************************/
    Sderivation.components = [
        'task',
        'assumption',
        'observation',
        'derivmotivation',
        'term',
        'relation',
        'motivation'
    ];
    
    Sderivation.constr = {
        task: Sdtask,
        assumption: Sdassumption,
        observation: Sdobservation,
        derivmotivation: Sdmotivation,
        term: Sdterm,
        relation: Sdrelation,
        motivation: Sdmotivation
    }
    
    Sderivation.numtoalpha = function(num){
        /***************************************
          * Converts number to alphabetical label
          * 0 -> a, 1 -> b, ..., 25 -> z, 26 -> aa,
          * 27 -> ab, ...
          ***************************************/
         var remain=0;
         var result = '';
         while (result === '' || num > 25){
             remain = num % 26;
             result = String.fromCharCode(remain+97)+result;
             num = Math.floor(num / 26);
         };
         if (num > 0){
             remain = num % 26;
             result = String.fromCharCode(remain+96)+result;
         }
         return result
     }

    Sderivation.alphatonum = function(alpha){
        /***************************************
         * Converts alphabetical label to a number
         * a -> 0, b -> 1, ..., z -> 25, aa -> 26,
         * ab -> 27, ..., az -> 51,...
         ***************************************/
        var ll = 0;
        for (var i = 0; i<alpha.length; i++){
            if (alpha.length > 1 && i == 0){
                ll += alpha.charCodeAt(i)-96;
            } else {
                ll = 26*ll;
                ll += alpha.charCodeAt(i)-97;
            };
        };
        return ll;
    }
    
    /**************************************
     * SdEditevent class
     **************************************/
    var SdEditevent = function(event, loc, data){
        // Class for editing events.
        //options = {
        //    event: 'nop',   // Operation, what we are doing
        //    loc: '',        // Element's location array [deriv,element,pos,element,pos,...]
        //    data: ['','']   // [new_data, old_data]
        //};
        this.event = event || 'nop';
        this.loc = loc || '';
        this.data = $.extend(true, [], data);
    }
    
    /******
     * Change this event to it's inverse. (add vs. remove, edit vs. edit,...)
     ******/
    SdEditevent.prototype.invert = function(){
        switch (this.event) {
            case 'edit':
                this.data.reverse();
                break;
            case 'addstep':
                this.event = 'removestep';
                this.data.reverse();
                //this.loc[this.loc.length - 1] = ''+(parseInt(this.loc[this.loc.length-1]) + 1);
                this.loc[this.loc.length - 2] = 'motivation';
                break;
            case 'removestep':
                this.event = 'addstep';
                this.data.reverse();
                //this.loc[this.loc.length - 1] = ''+(parseInt(this.loc[this.loc.length-1]) - 1);
                this.loc[this.loc.length - 2] = 'term';
                break;
            case 'addtask':
                this.event = 'removetask';
                this.data.reverse();
                this.loc[this.loc.length -2] = 'task';
                this.loc[this.loc.length -1] = '0';
                break;
            case 'removetask':
                this.event = 'addtask';
                this.data.reverse();
                this.loc[this.loc.length -2] = 'term';
                this.loc[this.loc.length -1] = '0';
                break;
            case 'addassumption':
                this.event = 'removeassumption';
                this.data.reverse();
                if (this.loc[this.loc.length -2] === 'task') {
                    this.loc[this.loc.length -2] = 'assumption';
                    this.loc[this.loc.length -1] = '0';
                    this.loc.push('proposition');
                } else {
                    this.loc[this.loc.length -2] = ((this.loc[this.loc.length -2] | 0) + 1) + '';
                }
                break;
            case 'removeassumption':
                this.event = 'addassumption';
                this.data.reverse();
                this.loc[this.loc.length -2] = ((this.loc[this.loc.length -2] | 0) - 1) + '';
                break;
            case 'addobservation':
            case 'adddefinition':
            case 'adddeclaration':
                this.event = 'removeobservation';
                this.data.reverse();
                if (this.loc[this.loc.length -2] === 'task') {
                    this.loc[this.loc.length -2] = 'observation';
                    this.loc[this.loc.length -1] = '0';
                } else if (this.loc[this.loc.length -3] === 'assumption') {
                    this.loc.pop(); // Remove 'proposition' from the end.
                    this.loc[this.loc.length -2] = 'observation';
                    this.loc[this.loc.length -1] = '0';
                } else {
                    this.loc[this.loc.length -1] = ((this.loc[this.loc.length -1] | 0) + 1) + '';
                }
                this.loc.push('motivation');
                break;
            case 'removeobservation':
                this.event = 'addobservation';
                this.data.reverse();
                this.loc.pop();
                if (this.loc[this.loc.length -1] === '0') {
                    this.loc[this.loc.length -2] = 'assumption';
                    this.loc.push('proposition');
                    // There is no need to change the index to the last assumption.
                    // If 'addobservation' event comes from 'assumption', it will
                    // always be done to the beginning of the 'observation' array.
                } else {
                    this.loc[this.loc.length -1] = ((this.loc[this.loc.length -1] | 0) - 1) + '';
                }
                break;
            case 'addderivationmotivation':
                this.event = 'removederivationmotivation';
                this.data.reverse();
                this.loc[this.loc.length - 2] = 'derivmotivation';
                break;
            case 'removederivationmotivation':
                this.event = 'addderivationmotivation';
                this.data.reverse();
                this.loc[this.loc.length - 2] = 'task';
                break;
            case 'addsubderiv':
                this.event = 'removesubderiv';
                this.data.reverse();
                this.loc[this.loc.length - 1] = ((this.loc[this.loc.length - 1] | 0) + 1) + '';
                break;
            case 'removesubderiv':
                this.event = 'addsubderiv';
                this.data.reverse();
                this.loc[this.loc.length - 1] = ((this.loc[this.loc.length - 1] | 0) - 1) + '';
                break;
            case 'showhidesubderivation':
                this.data.reverse();
                break;
            case 'addsteptask':
                this.event = 'removesteptask';
                this.data.reverse();
                break;
            case 'addstepassumption':
                this.event = 'removestepassumption';
                this.data.reverse();
                break;
            case 'addstepfact':
                this.event = 'removestepfact';
                this.data.reverse();
                break;
            case 'addstepdefinition':
                this.event = 'removestepdefinition';
                this.data.reverse();
                break;
            case 'addstepdeclaration':
                this.event = 'removestepdeclaration';
                this.data.reverse();
                break;
            case 'removesteptask':
                this.event = 'addsteptask';
                this.data.reverse();
                break;
            case 'removestepassumption':
                this.event = 'addstepassumption';
                this.data.reverse();
                break;
            case 'removestepfact':
                this.event = 'addstepfact';
                this.data.reverse();
                break;
            case 'removestepdefinition':
                this.event = 'addstepdefinition';
                this.data.reverse();
                break;
            case 'removestepdeclaration':
                this.event = 'addstepdeclaration';
                this.data.reverse();
                break;
            case 'removedstep':
                this.event = 'addstep'+this.data[0].type;
                this.data.reverse();
                break;
            case 'movedstep':
                this.data.reverse();
                var data1 = this.data[1];
                this.data[0] = {indexfrom: data1.indexto, indexto: data1.indexfrom};
                break;
            case 'undo':
                this.event = 'redo';
                break;
            case 'redo':
                this.event = 'undo';
                break;
            default:
                break;
        }
        return this;
    }
    
    /******
     * Return the event type.
     ******/
    SdEditevent.prototype.getEvent = function(){
        return this.event;
    }
    
    /******
     * Return the location of the parent.
     ******/
    SdEditevent.prototype.getLoc = function(){
        return this.loc;
    }
    
    
    /**********************************
     * SdEventStack class
     * Constructor
     **********************************/
    var SdEventStack = function(size){
        this.maxsize = size || 100;
        this.events = [];
    }
    
    /******
     * Push a new event in to the stack. If the size exceeds the maxsize,
     *  then drop the oldest event.
     ******/
    SdEventStack.prototype.push = function(event){
        this.events.push(event);
        if (this.events.length > this.maxsize) {
            this.events.shift();
        }
    }
    
    /******
     * Pop the last event from the stack.
     ******/
    SdEventStack.prototype.pop = function(){
        return this.events.pop();
    }
    
    /******
     * Return the curent size of the stack.
     ******/
    SdEventStack.prototype.size = function(){
        return this.events.length;
    }
    
    /******
     * Clear the stack.
     ******/
    SdEventStack.prototype.clear = function(){
        this.events = [];
    }




    /***************************************************************************
     * Teacher's marking tool
     ***************************************************************************/
    
    /******
     * Constructor of one marking message object.
     * @constructor
     * @param {Object} options - Message data
     *   @param {String} options.loc - Location identifier
     *   @param {String} options.mark - The name of the mark ('correct', 'wrong', 'note')
     *   @param {String} options.comment - A free text comment
     ******/
    var SdMarkingMessage = function(options){
        this.loc = (options && options.loc) || '';
        this.mark = (options && options.mark) || '';
        this.comment = (options && options.comment) || '';
        return this;
    }
    
    /******
     * Set location
     * @param {String} location - Location identifier
     ******/
    SdMarkingMessage.prototype.setLoc = function(location){
        this.loc = location;
    }
    
    /******
     * Set mark
     * @param {String} mark - The name of the mark ('correct', 'wrong', 'note', 'empty')
     ******/
    SdMarkingMessage.prototype.setMark = function(mark){
        this.mark = mark;
    }
    
    /******
     * Set comment
     * @param {String} comment - A free text comment
     ******/
    SdMarkingMessage.prototype.setComment = function(comment){
        this.comment = comment;
    }
    
    /******
     * Get location
     * @returns {String} The location identifier
     ******/
    SdMarkingMessage.prototype.getLoc = function(){
        return this.loc;
    }
    
    /******
     * Get mark
     * @returns {String} The name of the mark ('correct', 'wrong', 'note', 'empty')
     ******/
    SdMarkingMessage.prototype.getMark = function(){
        return this.mark;
    }
    
    /******
     * Get comment
     * @returns {String} The free text comment
     ******/
    SdMarkingMessage.prototype.getComment = function(){
        return this.comment;
    }
    
    /******
     * Get the data
     * @returns {Object} A message object with loc, mark and comment
     ******/
    SdMarkingMessage.prototype.getData = function(){
        var obj = {
            "loc": this.loc,
            "mark": this.mark,
            "comment": this.comment
        }
        return obj;
    }
    
    /******
     * Get the data as JSON-string.
     * @returns {String} Amessage object with loc, mark and comment as a JSON string
     ******/
    SdMarkingMessage.prototype.getJson = function(){
        var obj = this.getObject();
        return JSON.stringify(obj);
    }
    
    
    /*****************************************************************
     * Class for set of markings and comments (SdMarkingMessage elements) plus general comment
     * @constructor
     * @param {Object} options - The content of a review with all messages
     *   @param {String} options.derivname - the name of current derivation
     *   @param {Array} options.messages - An array of message objects
     *   @param {String} options.general - A free text general feedback
     ******/
    var SdMarkings = function(options){
        this.lang = 'en';
        this.messages = {};
        this.count = 0;
        var loc;
        if (options && options.messages && options.messages.length > 0){
            for (var i = 0; i < options.messages.length; i++){
                loc = options.messages[i].loc;
                var message = new SdMarkingMessage(options.messages[i]);
                this.messages[message.getLoc()] = message;
                this.count++;
            }
        }
        this.general = (options && options.general) || '';
        this.symbols = $.extend(SdMarkings.checkmarks);
    }
    
    /******
     * Constants for symbols.
     ******/
    SdMarkings.checkmarks = {"correct": "&#x2714;", "wrong": "&#x2718;", "note": "!", "empty": "-", "finnishcorrect":"&#x2052;", "unsure": "?", "warning":"&#9888;"};
    SdMarkings.localCheckMarks = {
        fi: {
            correct: "&#x2052;"
        }
    }
    
    /******
     * Get the number of messages.
     * @returns {Number} The number of messages in this set
     ******/
    SdMarkings.prototype.size = function(){
        return this.count;
    }
    
    /******
     * Set tha language/locale of the marks
     * @param {String} lang - the language/name of the locale
     *****/
    SdMarkings.prototype.setLang = function(lang){
        this.lang = lang;
        this.symbols = $.extend({}, SdMarkings.checkmarks, SdMarkings.localCheckMarks[lang] || {});
    }
    
    /******
     * Get a marking symbol (localized)
     * @param {String} mark - name of the mark
     * @returns {String} the symbol accosiated with given mark.
     ******/
    SdMarkings.prototype.getSymbol = function(mark){
        return this.symbols[mark] || this.symbols['note'];
    }
    
    /******
     * Add new message.
     * @param {Object} options - A new message to be added
     *   @param {String} options.loc - Location identifier
     *   @param {String} options.mark - The name of the mark ('correct', 'wrong', 'note')
     *   @param {String} options.comment - A free text comment
     ******/
    SdMarkings.prototype.addMessage = function(options){
        options = $.extend({"loc":"","mark":"","comment":""}, options);
        var message = new SdMarkingMessage(options);
        var loc = message.getLoc();
        if (loc && !this.messages[loc]) {
            this.count++;
        }
        this.messages[message.getLoc()] = message;
        return this;    
    }
    
    /******
     * Clear message - remove a message from given location
     * @param {String} loc - A location identifier
     ******/
    SdMarkings.prototype.clearMarking = function(loc){
        delete this.messages[loc];
        this.count--;
    }
    
    /******
     * Clear all markings
     ******/
    SdMarkings.prototype.clearAll = function(){
        this.messages = {};
        this.general = '';
        this.count = 0;
    }
    
    /******
     * Check if marking exists for given location.
     * @param {String} loc - A location identifier
     * @returns {Boolean} Returns true, if a message for given location exists.
     ******/
    SdMarkings.prototype.messageExists = function(loc){
        return !!this.messages[loc];
    }
    
    /******
     * Get message for given location.
     * @param {String} loc - A location identifier
     * @returns {Object} Message object for given location.
     ******/
    SdMarkings.prototype.getMessage = function(loc){
        return (this.messages[loc] || false);
    }
    
    /******
     * Get location for given location.
     * @param {String} loc - The location identifier
     * @returns {String} The location identifier
     ******/
    SdMarkings.prototype.getLoc = function(loc){
        return (this.messages[loc] && this.messages[loc].loc) || '';
    }
    
    /******
     * Get the array of locations of all markings.
     * @returns {Array} An array of all location identifiers that have messages
     ******/
    SdMarkings.prototype.getLocations = function(){
        var locs = [];
        for (var loc in this.messages) {
            locs.push(loc);
        }
        return locs;
    }
    
    /******
     * Get mark for given location.
     * @param {String} loc - Location identifier
     * @returns {String} The mark for the given location
     ******/
    SdMarkings.prototype.getMark = function(loc){
        return (this.messages[loc] && this.messages[loc].mark) || '';
    }
    
    /******
     * Get comment for given location.
     * @param {String} loc - Location identifier
     * @returns {String} The comment for given location
     ******/
    SdMarkings.prototype.getComment = function(loc){
        return (this.messages[loc] && this.messages[loc].comment) || '';
    }
    
    /******
     * Set the general comment.
     * @param {String} general - The general comment for this review
     * @returns {Object} This Markings set
     ******/
    SdMarkings.prototype.setGeneral = function(general){
        this.general = general;
        return this;
    }
    
    /******
     * Get the general comment.
     * @returns {String} The general comment for this review
     ******/
    SdMarkings.prototype.getGeneral = function(){
        return this.general;
    }
    
    /******
     * Get data of all markings.
     * @returns {Object} The data of this review as a general Javascript object (general and messages)
     ******/
    SdMarkings.prototype.getData = function(){
        var obj = {
            "general": this.general,
            "messages": []
        };
        for (var key in this.messages){
            obj.messages.push(this.messages[key].getData());
        }
        return obj;
    }
    
    /******
     * Prepare LaTeX-math for MathQuill
     * @param {String} str - String that contains math in LaTeX format
     ******/
    SdMarkings.prototype.makeMq = function(str){
        var result = str.replace(/\\\(/g, '<span class="mathquill-embedded-latex">').replace(/\\\)/g, '</span>')
            .replace(/\$([^$]*)\$/g, '<span class="mathquill-embedded-latex">$1</span>');
        return result;
    }

    /******
     * Show marking of given location in given place
     * @param {String} loc - Location identifier
     * @param {Object} place - jQuery-object or DOM-element where the marking will be shown
     ******/
    SdMarkings.prototype.showMark = function(loc, place){
        var message = this.getMessage(loc);
        var html = [
            '<div class="manualreview">',
            '    <span class="checkmark sd' + message.getMark() + '">',
            //'        <span>' + SdMarkings.checkmarks[message.getMark()] + '</span>',
            '        <span>' + this.getSymbol(message.getMark()) + '</span>',
            '    </span>',
            '    <span class="checkcomment">' + this.makeMq(message.getComment()) + '</span>',
            '</div>'
        ].join('\n');
        var newelem = $(html);
        newelem.find('.mathquill-embedded-latex').mathquill();
        newelem.on('mouseenter', function(e){
            $(this).find('.mathquill-embedded-latex').mathquill('redraw');
        }).on('click', function(e){
            $(this).closest('.manualreview').toggleClass('checkvisible');
        });
        $(place).html(newelem);
    }

    if (typeof(config) !== 'undefined' && config.macros){
        config.macros.sdeditor2 = {
            /******************************
            * Show sdeditor2
            ******************************/
            handler: function (place, macroName, params, wikifier, paramString, tiddler)
            {
                if (params.length < 1){
                    wikify('Missing structured derivation.', place);
                    return false;
                }
                var thispage = jQuery(place).parents('.bookpage');
                if (thispage.length > 0){
                    var pageno = Math.max(jQuery('#pageOne, #pageTwo').index(thispage), 0);
                } else {
                    var pageno = 0;
                }
                var stepbystepmode = false;
                var ui_lang = config.options.txtLang || 'en';
                if(typeof(Emathbook.options.user.showstepbystep) !== "undefined" && Emathbook.options.user.showstepbystep){
                    stepbystepmode = true;
                }
                var sdtheme = 'default';
                if(typeof(Emathbook.options.user.sdtheme) !== "undefined"){
                    sdtheme = Emathbook.options.user.sdtheme;
                }
                var elementid = params[0];
                var mode = (params[1] || 'view');
                if (mode === 'authordialog') {
                    mode = 'author';
                }else if (mode === 'savehere') {
                    mode = 'view';
                }
                var containertext = '{{emathsdeditor sderivation_'+elementid+'{\n}}}';
                var initData = 
                {
                    "settings":{
                        mode: mode,
                        theme: sdtheme, 
                        username: config.options.txtUserName,
                        stepmode : stepbystepmode,
                        uilang : ui_lang
                    },
                    "data":{}
                }
                wikify(containertext, place);
                if (tiddler) {
                    var settings = jQuery.extend(true, {},DataTiddler.getData(tiddler, 'sderivation2',{}));
                    if(typeof(settings[elementid]) !=="undefined"){
                        //settings[elementid] = settings[elementid] || {data: {}};
                        settings[elementid].data = settings[elementid].data || {};
                        settings[elementid].settings = {
                            mode: mode,
                            theme: sdtheme, 
                            username: config.options.txtUserName,
                            stepmode : stepbystepmode,
                            uilang : ui_lang
                        };
                        initData = settings[elementid];
                    }else{
                        var derivations = jQuery.extend(true, {},DataTiddler.getData(tiddler, 'derivations',{}));
                        if(typeof(derivations[elementid]) !=="undefined"){
                            initData = 
                            {
                                "data":
                                    {
                                        main: elementid,
                                        mode: mode,
                                        sdtheme: sdtheme,
                                        username: config.options.txtUserName,
                                        derivations: derivations,
                                        stepmode : stepbystepmode,
                                        lang : ui_lang
                                    }
                            }
                        }
                    }
                }         
                var sdeditor2 = jQuery(place).find('.emathsdeditor.sderivation_'+elementid).last();
                    sdeditor2.sdeditor(initData);
                if ((mode === 'edit' || mode === 'author') &&  params[1] !== 'authordialog') {
                    sdeditor2.unbind('sderiv_changed').bind('sderiv_changed', function(e){
                        var $derivplace = jQuery(this);
                        var data = $derivplace.sdeditor('get');
                        var settings = DataTiddler.getData(tiddler, 'sderivation2',{});
                        settings[elementid] = data;
                        var autosavestatus = config.options.chkAutoSave;
                        config.options.chkAutoSave = false;
                        DataTiddler.setData(tiddler,'sderivation2', settings);
                        config.options.chkAutoSave = autosavestatus;
                    });
                }
                return true;
            }
        };
        
        /********
         * To replace the old editor in TiddlyWiki
         ********/
         /*
        config.macros.qedderiv = {
            handler: function(place, macroName, params, wikifier, paramString, tiddler)
            {
                if (params.length < 1){
                    wikify('Missing structured derivation.', place);
                    return false;
                }
                var thispage = jQuery(place).parents('.bookpage');
                if (thispage.length > 0){
                    var pageno = Math.max(jQuery('#pageOne, #pageTwo').index(thispage), 0);
                } else {
                    var pageno = 0;
                }
                var elementid = params[0];
                var mode = (params[1] || 'view');
                if (mode === 'authordialog') {
                    mode = 'author';
                }
                if (mode === 'savehere') {
                    mode = 'view';
                }
                var containertext = '{{emathsdeditor sderivation_'+elementid+'{\n}}}';
                wikify(containertext, place);
                if (tiddler) {
                    var derivations = jQuery.extend(true, {},DataTiddler.getData(tiddler, 'derivations',{}));
                } else {
                    var derivations = {};
                }
                var data = {
                    main: elementid,
                    mode: mode,
                    sdtheme: 'paper',
                    username: config.options.txtUserName,
                    derivations: derivations
                }
                var settings = {data: data};
                var sdeditor2 = jQuery(place).find('.emathsdeditor.sderivation_'+elementid).last().sdeditor(settings);
                //if ((mode === 'edit' || mode === 'author') &&  params[1] !== 'authordialog') {
                //    sdeditor2.unbind('sderiv_changed').bind('sderiv_changed', function(e){
                //        var $derivplace = jQuery(this);
                //        var data = $derivplace.sdeditor('get');
                //        var settings = DataTiddler.getData(tiddler, 'sderivation2',{});
                //        settings[elementid] = data;
                //        var autosavestatus = config.options.chkAutoSave;
                //        config.options.chkAutoSave = false;
                //        tiddler.setData('sderivation2', settings);
                //        config.options.chkAutoSave = autosavestatus;
                //    });
                //}
                return true;
            }
        }
        */
        config.macros.qedderiv = {
            handler: config.macros.sdeditor2.handler
        };
    }


})(jQuery);
//}}}