/******
@name ReaderView
@version 0.1
@author Petri Salmela <petri.salmela@fourferries.fi>
@type plugin
@requires jQuery x.x.x or newer
@description Class and jQuery-plugin for showing a book.
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(ebooklocalizer) === 'undefined' && ebooklocalizer.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/******
 * Create a localizer
 ******/
if (typeof(ebooklocalizer) === 'undefined'){
    if (typeof(EbookLocalizer) !== 'undefined') {
        var ebooklocalizer = new EbookLocalizer();
    } else {
        var ebooklocalizer = {};
        ebooklocalizer.localize = function(term){return term;}
    }
}

if (typeof(ebooklocalizer) !== 'undefined') {
    ebooklocalizer.addTranslations({
        'en': {
            'reader:TOC': 'Table of contents',
            'reader:language': 'Language',
            'reader:widget theme': 'Widget themes',
            'reader:theme': 'Content themes',
            'reader:settingsmenu': 'Settings',
            'reader:aboutmenu': 'About',
            'reader:helpmenu': 'Help',
            'reader:fullscreen': 'Fullscreen',
            'reader:imageinfo': 'Information about the image',
            'reader:author': 'Author',
            'reader:license': 'License',
            'reader:source': 'Source',
            'reader:notes': 'Notes',
            'readerthemes:plain text': 'Plain text',
            'readerthemes:theory text': 'Theory text',
            'readerthemes:example text': 'Example text',
            'readerthemes:theory box': 'Theory box',
            'readerthemes:example box': 'Example box',
            'reader:structured derivations': 'Structured derivations',
            'reader:showstepbystep': 'Show derivations step by step',
            'reader:font': "Font",
            'reader:usedyslexiafont': "Use font increasing readability for readers with dyslexia.",
            'reader:theorybox': 'Theory',
            'reader:examplebox': 'Example',
            'reader:definitionbox': 'Definition',
            'reader:assignment': 'Assignment',
            'reader:fontattribution': "Fonts used in this e-book",
            'reader:softwareattribution': "Software libraries used in this e-book",
            'reader:graphicsattribution': "Graphics used in this e-book",
            'reader:instructionstitle': 'How to use this book?',
            'reader:instructionsdescription': "There's multiple ways to navigate the book. Page can be changed with navigation bar's arrow buttons, slider and by swiping the screen. Settings of the book can be changed from settings button located in the top left corner of the book. Further intructions of this book can be found online from the 4ferries website.",
            'reader:instructionsurl': 'http://www.4ferries.fi/?p=358',
            'reader:instructionsurldescription': 'Click here to open the instructions page'

        },
        'fi': {
            'reader:TOC': 'Sisällysluettelo',
            'reader:language': 'Kieli',
            'reader:widget theme': 'Käyttöliittymän tyyli',
            'reader:theme': 'Sisällön teemat',
            'reader:settingsmenu': 'Asetukset',
            'reader:aboutmenu': 'Tietoa sovelluksesta',
            'reader:helpmenu' : 'Apua',
            'reader:fullscreen': 'Kokoruututila',
            'reader:imageinfo': 'Tietoja kuvasta',
            'reader:author': 'Tekijä',
            'reader:license': 'Lisenssi',
            'reader:source': 'Lähde',
            'reader:notes': 'Huomioita',
            'readerthemes:plain text': 'Pelkkää tekstiä',
            'readerthemes:theory text': 'Teorian tekstiä',
            'readerthemes:example text': 'Esimerkin tekstiä',
            'readerthemes:theory box': 'Teorialaatikko',
            'readerthemes:example box': 'Esimerkkilaatikko',
            'reader:structured derivations': 'Rakenteiset päättelyketjut',
            'reader:showstepbystep': 'Näytä päättelyketjut askeleittain',
            'reader:font': "Fontti",
            'reader:usedyslexiafont': "Käytä dyslexia -fonttia",
            'reader:theorybox': 'Teoria',
            'reader:examplebox': 'Esimerkki',
            'reader:definitionbox': 'Määritelmä',
            'reader:assignment': 'Tehtävä',
            'reader:fontattribution': "Tässä e-kirjassa kätetyt fontit",
            'reader:softwareattribution': "Tässä e-kirjassa käytetyt ohjelmistot",
            'reader:graphicsattribution': "Tässä e-kirjassa käytetyt grafiikat",
            'reader:instructionstitle': 'Kirjan käyttöohjeet',
            'reader:instructionsdescription': "Voit selata kirjaa usealla eri tavalla. Saat vaihdettua sivua navigointipalkin nuolipainikkeista, liu'uttimesta tai pyyhkäisemällä näyttöä. Asetuksia saa vaihdettua vasemman ylälaidan 'asetukset' -painikkeesta avautuvasta valikkorivistä. Tarkemmat käyttöohjeet löydät internetistä 4ferriesin verkkosivuilta.",
            'reader:instructionsurl': 'http://www.4ferries.fi/?p=197',
            'reader:instructionsurldescription': 'Klikkaa tästä siirtyäksesi käyttöohjeiden sivulle'

        },
        'sv': {
            "reader:TOC": "Innehållsförteckning",
            "reader:language": "Språk",
            "reader:widget theme": "Tema för användargränssnitt",
            "reader:theme": "Tema för innehållet",
            "reader:settingsmenu": "Inställningar",
            "reader:aboutmenu": "Information om appen",
            "reader:helpmenu": "Hjälp",
            "reader:fullscreen": "Helskärmsläge",
            "reader:imageinfo": "Bildinformation",
            "reader:author": "Författare",
            "reader:license": "Licens",
            "reader:source": "Källa",
            "reader:notes": "Anmärkningar",
            "readerthemes:plain text": "Standardtext",
            "readerthemes:theory text": "Teoritext",
            "readerthemes:example text": "Exempeltext",
            "readerthemes:theory box": "Teorilåda",
            "readerthemes:example box": "Exempellåda",
            "reader:structured derivations": "Strukturerade härledningar",
            "reader:showstepbystep": "Visa steg för steg",
            "reader:font": "Typsnitt",
            "reader:usedyslexiafont": "Använd dyslexi-typsnitt",
            "reader:theorybox": "Teori",
            "reader:examplebox": "Exempel",
            "reader:definitionbox": "Definition",
            "reader:assignment": "Uppgift",
            "reader:fontattribution": "Typsnitt som används i den här boken",
            "reader:softwareattribution": "Programvara som används i den här boken",
            "reader:graphicsattribution": "Grafik som används i den här boken",
            "reader:instructionstitle": "Bruksanvisningar för boken",
            "reader:instructionsdescription": "Du kan navigera i boken genom att bläddra framåt och bakåt sida för sida, genom att svepa med fingret över sidan eller genom att använda pilknapparna i de övre hörnen. Bokens huvudmeny är placerad i skärmens övre vänstra hörn. Där kan du välja bokens språk (svenska eller finska), layout och temafärger enligt dina önskemål. För dyslektiker finns det även möjlighet att välja en lättläslig font i inställningarna. Närmare instruktioner hittas på 4ferries websida",
            "reader:instructionsurl": "http://www.4ferries.fi/?p=358",
            "reader:instructionsurldescription": "Klicka här för att öppna webbsidan med användarinformation"
       },
        'et': {
            'reader:language': 'Keel',
            'reader:theme': 'Teema'
        }
    });
}

(function($){
    /******
     * ReaderView -class
     * @constructor
     * @param {jQuery} place - a jQuery-object to place the reader.
     * @param {Object} options - initial options for the reader.
     ******/
    
    Hammer.defaults.stop_browser_behavior.userSelect = false;
    
    var ReaderView = function(place, options){
        if (options.book && !options.data) {
            // Backwards compatibility
            options.data = options.book;
        };
        options = $.extend({}, ReaderView.defaults, options);
        var bookid = options.bookid || options.name || options.config && options.config.bookid || '';
        this.config = $.extend({}, options.config || {}, this.loadConf(bookid));
        var defaultconf = {
            mode: 'view',
            ticklevel: 1,
            currentpage: 'front',
            lang: options.settings.lang || 'fi',
            uilang: options.settings.uilang || 'en',
            theme: 'default',
            settings: {
                fontsettings: {
                    useDyslexiaFont: false
                },
                sdsettings: {
                    stepmode: false
                }
            },
            pageUpdateTimeThreshold: 350 //if set to non-zero value, the book will update the page body only after this time(ms) has passed from the last 'navigationmoved' event
        };
        //defaultconf = $.extend(defaultconf, this.loadConf());
        for (var conf in defaultconf) {
            this.config[conf] = (typeof(this.config[conf]) !== 'undefined' ? this.config[conf] : defaultconf[conf]);
            //this.saveConf();
        }
        this.settings = options.settings;
        this.place = place;
        this.name = options.bookid || options.name;
        this.book = (options.data.type === 'Book' ? options.data : new Book(options.data));
        // make sure, the selected language is available in the book, if not, use the default language.
        this.setConf('lang', this.book.getUsableLang(this.getConf('lang')));
        this.pagehistory = [];
    }
    
    ReaderView.defaults = {
        name: 'undefined',
        data: {
            contentPrefix: '',
            content: {}
        }
    }

    /******
     * Initialization of ReaderView
     ******/
    ReaderView.prototype.init = function(){
        this.place.addClass('ebookreader-wrapper');
        if ($('head style#ebookreaderstyle').length == 0){
            $('head').prepend('<style id="ebookreaderstyle" type="text/css">'+ReaderView.strings.css+'</style>');
        }
        this.place.html(ReaderView.strings.template);
        this.controlarea = this.place.children('.ebookreader-controlarea');
        this.menuarea = this.place.children('.ebookreader-menuarea');
        this.menulist = this.menuarea.find('.ebookreader-menulist');
        this.breadcrumbarea = this.menuarea.children('.ebookreader-breadcrumb');
        this.viewarea = this.place.children('.ebookreader-viewarea');
        //this.pageheader = this.viewarea.children('.ebookreader-pageheader');
        //this.pageneck = this.viewarea.children('.ebookreader-pageneck');
        //this.pagebody = this.viewarea.children('.ebookreader-pagebody');
        this.notearea = this.place.children('.ebookreader-notearea');
        this.notetitle = this.notearea.find('.ebookreader-notetitle .titlearea');
        this.notecontent = this.notearea.children('.ebookreader-notecontent');
        this.footerarea = this.place.children('.ebookreader-footerarea');
        this.naviarea = this.place.children('.ebookreader-naviarea');
        this.navlabeldown = $('html').hasClass('touch');
        this.navlabeldown = true;
        this.booklinear = this.book.getLinearToc(this.getConf('lang'));
        this.name = this.name || 'reader-' + this.getBookid();
        this.place.attr('data-appname', this.name);
        this.setNavbar();
        this.setTheme();
        this.setWidgetTheme();
        this.initHandlers();
        this.setFooter();
        this.initSettings();
        this.updatePage();
        this.announceTitle(this.config['lang']);
    }
    
    /**
     * Announce the titledata to the outside
     */
    ReaderView.prototype.announceTitle = function(lang) {
        this.place.trigger('titledata', [this.book.getTitleData(lang)]);
    };

    /******
     * Init event handlers.
     ******/
    ReaderView.prototype.initHandlers = function(){
        var viewer = this;
        // Change page, when navigation moves.
        this.place.off('navigationmoved').on('navigationmoved', function(event, data){
            event.preventDefault();
            event.stopPropagation();
            viewer.closeMenu();
            viewer.setPage(data);
            //lastNavigationMoveTime is used in updatePage() to increase performance
            viewer.lastNavigationMoveTime = new Date().getTime();
        });
//        this.viewarea.off('touchstart').on('touchstart', function(event){});
        // Change page and move navigation, when 'setpage'-event is triggered.
        this.place.on('setpage', function(event, data){
            event.preventDefault();
            event.stopPropagation();
            viewer.setPage(data);
            viewer.naviarea.trigger('setnaviposition', [data]);
        });
        // Get the data of elements from book and send it to the asking element as a reply. Async.
        this.place.off('get_bookelement').on('get_bookelement', function(event, data){
            event.preventDefault();
            event.stopPropagation();
            var target = $(event.target);
            var defaultlang = viewer.getDefaultLang();
            // Use original language, if it is asked for or if container element is already using original (default) language.
            data.origlang = data.origlang || data.lang === defaultlang;
            // As lang use one of following:
            // null, if the element is not language specific, i.e., for example the page element
            // books default language, if data.origlang is true, i.e., if the translation has not been found and origlang has been asked
            // viewers selected language, otherwise
            viewer.getContentData({
                contentlist: data.contentlist,
                //lang: (data.uselangpath ? (data.origlang ? defaultlang : viewer.getConf('lang')) : null),
                lang: data.lang,
                callback: function(replydata){
                    target.trigger('reply_get_bookelement', [replydata]);
                }
            });
        });
        // Tap on breadcrumb changes to that page
        this.place.hammer().on('tap', '.ebookreader-breadcrumb .breadcrumbchapter', function(event){
            viewer.closeMenu();
            var $bcchap = $(this);
            var chapid = $bcchap.attr('chapid');
            $bcchap.trigger('setpage', [chapid]);
        });
        // Hold on breadcrumb is same as taping '>' between items.
        this.place.hammer().on('hold', '.ebookreader-breadcrumb .breadcrumbchapter', function(event){
            var $bcchap = $(this);
            var $bcdelim = $bcchap.prev('.breadcrumbdelimiter');
            $bcdelim.trigger('tap');
        });
        // Tap on '>' between breadcrumb items opens submenu.
        this.place.hammer().on('tap', '.ebookreader-breadcrumb .breadcrumbdelimiter', function(event){
            event.preventDefault();
            event.stopPropagation();
            var $bcdelim = $(this);
            var $oldlist = $bcdelim.find('ul.ebookreader-breadcrumb-childlist');
            var hadold = $oldlist.length > 0;
            $('body').trigger('tap');
            if (hadold) {
                return false;
            }
            var $bcchap = $bcdelim.prev('.breadcrumbchapter');
            var currchap = $bcdelim.next().attr('chapid')/* || $bcdelim.next().next().attr('chapid')*/;
            var chapid = $bcchap.attr('chapid');
            var children = viewer.getChildren(chapid);
            if (children.length > 0) {
                var html = ['<ul class="ebookreader-breadcrumb-childlist ffwidget-background">'];
                for (var i = 0; i < children.length; i++) {
                    var child = children[i];
                    html.push('<li class="breadcrumbchapter breadcrumbchildchapter'+ (child.chapid === currchap ? ' bccurrentpage' : '') +'" chapid="' + child.chapid + '">' + (child.chapnumber !== '' ? child.chapnumber + '. ' : '') + child.title + '</li>');
                }
                html.push('</ul>');
                var htmltext = html.join('\n');
                htmltext = htmltext.replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>')
                    .replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>');
                var $listelem = $(htmltext);
                $bcdelim.append($listelem);
                $bcdelim.find('.mathquill-embedded-latex').mathquill();
                $('body').hammer().on('tap.childlisttap', function(event){
                    $listelem.remove();
                    $('body').hammer().off('tap.childlisttap');
                });
            }
            return false;
        })

        this.updateMenu(); // WAT?

        // Tap the menu icon to open/close the menu.
        this.place.hammer().on('tap', '.ebookreader-menu', function(event){
            viewer.menulist.toggleClass('closed');
            if (viewer.place.is('[data-opencontent]')) {
                viewer.updatePage();
            }
        });
        // Swipe left to go to the next page.
        this.viewarea.hammer().off('swipeleft').on('swipeleft', function(event){
            if ($('html').hasClass('touch')) {
                viewer.naviarea.trigger('gonext');
            }
        })
        // Swipe right to go to the previous page.
        this.viewarea.hammer().off('swiperight').on('swiperight', function(event){
            if ($('html').hasClass('touch')) {
                viewer.naviarea.trigger('goprevious');
            }
        })
        // Click on link of headimage
        this.place.off('click', 'a.ebook-imagelink').on('click', 'a.ebook-imagelink', function(e){
            e.stopPropagation();
            e.preventDefault();
            var $place = $(this).parents('.ebookreader-pageheader, .ebook-rasterimage').eq(0);
            var $element = $place.find('.ebook-imagelink-popup');
            $element.addClass('ebookreader-showimginfopopup');
            $element.off('click').on('click', function(event){
                return (event.target.tagName === 'A' || $(event.target).closest('.ebook-popupclose').length > 0);
            });
            viewer.place.on('click.ebook-image-popup', function(event){
                $element.removeClass('ebookreader-showimginfopopup');
                $(this).off('click.ebook-image-popup');
            })
        });
        // Click on extraboxes
        this.place.off('click', '.ebookreader-extrabox').on('click', '.ebookreader-extrabox', function(e){
            var $element = $(this);
            var elementName = $element.attr('data-element-name');
            var extratype = $element.attr('data-extratype');
            var title = ReaderView.strings.icons[extratype + 'icon'];
            //var lang = viewer.getConf('lang');
            //elementName = lang + '/' + elementName;
            viewer.showNote({
                title: title,
                note: elementName,
                iselement: true
            });
        });
        // Close notearea
        this.notearea.off('click.closenote', '.ebookreader-notetitle-button.closenotes')
            .on('click.closenote', '.ebookreader-notetitle-button.closenotes', function(e){
                viewer.closeNote();
        });
        // Solutionhandlers
        this.place.off('getusersolutiondata').on('getusersolutiondata', function(e,assignment){
            var data = viewer.getSolutionData(assignment);
            jQuery(e.target).trigger('usersolutiondata', [data]);
        });
        this.place.off('setusersolutiondata').on('setusersolutiondata', function(e,assignment,solutiondata){
            var solutionid = viewer.setSolutionData(assignment, solutiondata);
            jQuery(e.target).trigger('usersolutiondataset', [solutionid]);
        });
        
        // Handlers for userdata saving and returning.
        this.place.off('save_userdata').on('save_userdata', function(e, data){
            // Add some information to data
            data.bookid = viewer.getBookid();
            data.username = viewer.config.settings.username || 'Anonymous';
        });
        
        this.place.off('get_userdata').on('get_userdata', function(e, data){
            // Add some information to data.
            data.bookid = viewer.getBookid();
            data.username = viewer.config.settings.username || 'Anonymous';
        });
        
        this.place.off('resizerequest').on('resizerequest', function(event, data) {
            event.stopPropagation();
            viewer.place.addClass('resizing');
            viewer.onWindowResize(data.change)
            setTimeout(function(){viewer.onWindowResize(0), viewer.place.removeClass('resizing')}, 600);
        });

        this.place.off('gotolink').on('gotolink', function(event, data){
            var link = data.data && data.data.link;
            if (link && link.appname && link.appname === viewer.name) {
                event.stopPropagation();
                viewer.goToLink(data);
            };
        });
        
        this.place.off('setcontent').on('setcontent', function(event, data) {
            // Pass the event through and add the id of this notebook as anchor.
            // Don't stop propagation!!!
            var appid = viewer.getBookid();
            var dataarr;
            if (typeof(data) === 'object' && typeof(data.length) === 'number') {
                dataarr = data;
            } else {
                dataarr = [data];
            };
            for (var i = 0, len = dataarr.length; i < len; i++) {
                if (dataarr[i].anchors.indexOf(appid) === -1) {
                    dataarr[i].anchors.push(appid);
                };
            };
        });
        
        this.place.off('gettitle').on('gettitle', function(event, data) {
            event.stopPropagation();
            data.title = viewer.book.getBookTitle(viewer.getConf('lang'));
        });
        
        this.place.off('closechildrenapp').on('closechildrenapp', function(event, data){
            event.stopPropagation();
            viewer.close();
        });

        this.place.off('closeapp').on('closeapp', function(event, data){
            event.stopPropagation();
            viewer.close();
        });

        //init window resize event listener
        $(window).on('resize', function(){viewer.onWindowResize();});
        
        // Marginnotes
        this.place.off('marginnotes_on').on('marginnotes_on', function(event, margin) {
            event.stopPropagation();
            viewer.setNoteview(true, margin);
        }).off('marginnotes_off').on('marginnotes_off', function(event, data) {
            event.stopPropagation();
            viewer.setNoteview(false);
        }).off('marginnotes_edit_on').on('marginnotes_edit_on', function(event, data) {
            event.stopPropagation();
            viewer.setNoteedit(true);
        }).off('marginnotes_edit_off').on('marginnotes_edit_off', function(event, data) {
            event.stopPropagation();
            viewer.setNoteedit(false);
        });
    };
    
    /**
     * Set noteview (marginnotes) on or off
     * @param {Boolean} [noteview] Optional parameter to switch the status. If not given, use the current status.
     * @param {Object} [margin]    Optional parameter with functions form margin handling.
     */
    ReaderView.prototype.setNoteview = function(noteview, margin) {
        if (typeof(noteview) === 'boolean') {
            this.noteview = noteview;
        };
        if (typeof(margin) === 'object') {
            this.margin = margin;
        };
        var elementsets = this.place.find('.ebookreader-pageframe > .ebookreader-pagebody > .ebook-elementset');
        if (this.noteview) {
            elementsets.trigger('showmarginnotes', [this.margin]);
        } else {
            elementsets.trigger('hidemarginnotes');
        }
    };
    
    /**
     * Set noteedit (marginnotes editmode) on or off
     * @param {Boolean} [noteedit] Optional parameter to switch the status. If not given, use the current status.
     */
    ReaderView.prototype.setNoteedit = function(noteedit) {
        if (typeof(noteedit) === 'boolean') {
            this.noteedit = noteedit;
        };
        var elementsets = this.place.find('.ebookreader-pageframe > .ebookreader-pagebody > .ebook-elementset');
        if (this.noteedit) {
            elementsets.trigger('editmarginnotes');
        } else {
            elementsets.trigger('viewmarginnotes');
        };
    };


    /******
     * Initializes the book's content to match settings specified in config.settings
     ******/
    ReaderView.prototype.initSettings = function(){
        var settings = this.config.settings;

        //Font Settings
        var wrapper = this.place;
        settings.fontsettings.dyslexiamode ? wrapper.addClass('useDyslexiaFont') : wrapper.removeClass('useDyslexiaFont');

        //Structured derivation settings
        if (typeof($.fn.sdeditor) === 'function') {
            var sdsettings = $.fn.sdeditor('getsettings');
            sdsettings.stepmode = settings.sdsettings.stepmode;
            $.fn.sdeditor('settings', sdsettings);
        }
        
        //Update page wasn't enough to immediately show step-by-step mode when the page is loaded
        //this.setPage(this.currentpage);

    }

    /******
     * Get bookid
     * @returns {string} The id of the current book.
     ******/
    ReaderView.prototype.getBookid = function(){
        return this.book && this.book.getBookid() || this.getConf('bookid') || '';
    }
    
    /******
     * Set config data and save it to localStorage
     * @param {String} key - the key of the setting
     * @param {?} value - the value of the setting
     ******/
    ReaderView.prototype.setConf = function(key, value){
        this.config[key] = value;
        this.saveConf();
    }
    
    /******
     * Get config data
     * @param {String} key - the key of the setting
     * @returns {?} the value of asked setting
     ******/
    ReaderView.prototype.getConf = function(key){
        return this.config && this.config[key] || '';
    }
    
    /******
     * Load config data from localStorage
     * @param {String} bookid - The id of the book
     * @returns {Object} the settings object
     ******/
    ReaderView.prototype.loadConf = function(bookid){
        bookid = bookid || this.getBookid();
        var config;
        try {
            config = JSON.parse(localStorage.getItem('readerview_' + bookid));
        } catch (e){
            console.log(e);
            config = {};
        }
        return config;
    }
    
    /******
     * Save config data to localStorage
     ******/
    ReaderView.prototype.saveConf = function(){
        var conf = JSON.stringify(this.config);
        var confdata = {
            conftype: 'readerview',
            bookid: this.getBookid(),
            config: conf
        }
        this.place.trigger('ebooksaveconfig', [confdata]);
        localStorage.setItem('readerview_' + this.getBookid(), conf);
    }
    
    var updateTimeout;  // WAT?

    /**
     * Go to the link inside this reader
     * @param {Object} linkdata - the link to the target
     */
    ReaderView.prototype.goToLink = function(linkdata){
        var link = linkdata.data && linkdata.data.link;
        delete link.appname;
        if (link.chapid && link.pageelementid) {
            var chapid = link.chapid;
            this.setPage(chapid);
            this.naviarea.trigger('goto', [chapid, false]);
            this.place.find('.ebookreader-viewarea [data-pageframe="current"] .ebook-pageelement[data-element-name="'+link.pageelementid+'"]').trigger('gotolink', linkdata);
        };
    };
    

    
    /******
     * Redraw page with data of corresponding to chapid in this.currentpage.
     * If update is done too quickly after another update, a white page is rendered and the real content
     * is rendered after a time specified by var this.pageUpdateTimeThreshold from last update has passed.
     * This technique is used to increase page change performance.
     * @param {Boolean} refreshAll - whether to refresh all page frames or just the current (visible) one.
     ******/
    ReaderView.prototype.updatePage = function(refreshAll){
        var viewer = this;
        this.hideSettingsFrame();
        this.updateBreadcrumb();
        this.clearMenuSelections();
        this.closeNote();
        this.setPageTypes();
        this.viewarea.scrollTop(0);
        
        //Check if update time threshold is in use and if it is, check if the page will be updated
        var curTime = new Date().getTime();
        var timeFromLastCall = (this.lastNavigationMoveTime) ? curTime - this.lastNavigationMoveTime : curTime;
        var willUpdate = timeFromLastCall > this.config.pageUpdateTimeThreshold;
        
        var pageids = {
            prev: this.getConf('prevpage'),
            current: this.getConf('currentpage'),
            next: this.getConf('nextpage')
        };
        var frames = {
            prev: this.viewarea.children('[data-pageframe="prev"]'),
            current: this.viewarea.children('[data-pageframe="current"]'),
            next: this.viewarea.children('[data-pageframe="next"]')
        };
        if (willUpdate) {
            var frameset = {'prev': true, 'current': true, 'next': true};
            for (var name in frameset) {
                var frame = frames[name];
                var pageid = pageids[name];
                var contentlevel = this.book.getContentlevel(pageid);
                frame.css('width', this.viewarea.width());
                if (refreshAll || (frame.attr('data-chapid') !== pageid && pageid !== '')) {
                    frame.attr('data-chapid', pageid).attr('data-contentlevel', contentlevel);
                    // Update page header
                    var headelem = frame.children('.ebookreader-pageheader');
                    this.updatePageHead(headelem, pageid);
                    
                    // Update page "neck"
                    var neckelem = frame.children('.ebookreader-pageneck');
                    this.updatePageNeck(neckelem, pageid);
                    
                    // Update page body
                    var bodyelem = frame.children('.ebookreader-pagebody');
                    this.updatePageBody(bodyelem, {pageid : pageid});
                }
            }
        } else {
            //Render just white page body this time and make real update with timeout
            var frameset = {'current': true};
            for (var name in frameset) {
                var frame = frames[name];
                var pageid = pageids[name];
                var contentlevel = this.book.getContentlevel(pageid);
                frame.css('width', this.viewarea.width());
                if (refreshAll || (frame.attr('data-chapid') !== pageid && pageid !== '')) {
                    frame.attr('data-chapid', pageid).attr('data-contentlevel', contentlevel);
                    // Update page header
                    var headelem = frame.children('.ebookreader-pageheader');
                    this.updatePageHead(headelem, pageid);
                    
                    // Update page "neck"
                    var neckelem = frame.children('.ebookreader-pageneck');
                    this.updatePageNeck(neckelem, pageid);
                    
                    //Render empty page body to improve performane
                    var bodyelem = frame.children('.ebookreader-pagebody');
                    this.updatePageBody(bodyelem, {renderEmpty:true, pageid : pageid});
                }
            }
            var nextUpdate = this.pageUpdateTimeThreshold + 10;
            clearTimeout(updateTimeout);
            updateTimeout = setTimeout(function(){viewer.updatePage(true);}, nextUpdate);
        };
        this.place.trigger('pageredraw', {apptype: 'reader', appid: this.getBookid()})
        // Set marginnote viewing and editing as it was on previous page.
        this.setNoteview();
        this.setNoteedit();
    };
    
    /******
     * Update pagehead in given element with headers of page with pageid.
     * @param {jQuery} element - jQuery-object with head to update.
     * @param {String} pageid - the id of the page
     ******/
    ReaderView.prototype.updatePageHead = function(element, pageid){
        var viewer = this,
            title,
            titlenumber,
            headimage,
            lang = this.getConf('lang');
        title = this.book.getTitle(pageid, lang);
        title = title.replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>');
        titlenumber = this.book.getChapNumberLong(pageid);
        element.html('<h1 class="ebookreader-headplain"><span>' + titlenumber + ' ' + title + '</span></h1><h1 class="ebookreader-headimageh1"><span class="ebookreader-headtext">' + titlenumber + ' ' + title + '</span></h1>');
        element.find('.mathquill-embedded-latex').mathquill();
        headimage = this.book.getHeadimage(pageid);
        if (headimage.text !== '') {
            // Update the background of the page header.
            this.getContentData({
                contentlist: [headimage.text],
                lang: lang,
                callback: function(data){
                    var hidata = data[headimage.text];
                    if (hidata && hidata.type !== 'response') {
                        var tablet = $('html').hasClass('touch');
                        var imgurl = hidata.data.source.match(/(?:https?:\/\/|file:\/\/)\S*/);
                        var imgtitle = [
                            ebooklocalizer.localize('reader:author', lang) + ': ' + hidata.data.author,
                            ebooklocalizer.localize('reader:license', lang) + ': ' + hidata.data.license,
                            ebooklocalizer.localize('reader:source', lang) + ': ' + hidata.data.source,
                            ebooklocalizer.localize('reader:notes', lang) + ': ' + hidata.data.notes
                        ].join('&#013;');
                        imgtitle = imgtitle.replace(/\"/g, '&quot;');
                        var imglink = (imgurl ? '<a href="'+imgurl[0]+'" class="ebook-imagelink" target="_blank" title="'
                                        +imgtitle+'">'+ ReaderView.strings.icons.infoicon+'</a>' : '');
                        var imgpopup = '<div class="ebook-imagelink-popup"><strong>'+ebooklocalizer.localize('reader:imageinfo', lang)+'</strong><table><tr><td>'
                            + ([ebooklocalizer.localize('reader:author', lang)+':</td><td>' + hidata.data.author,
                                ebooklocalizer.localize('reader:license', lang)+':</td><td>' + hidata.data.license,
                                ebooklocalizer.localize('reader:source', lang)+':</td><td>' + (imgurl && !tablet ? '<a href="' + imgurl[0] + '" class="ebook-info-imagelink" target="_blank">' + hidata.data.source +'</a>' : hidata.data.source),
                                ebooklocalizer.localize('reader:notes', lang)+':</td><td>' + hidata.data.notes
                            ].join('</td></tr><tr><td>')) + '</td></tr></table><span class="ebook-popupclose">'+ReaderView.strings.icons.closeicon+'</span></div>';
                            
                        element.find('h1.ebookreader-headimageh1')
                            .append(imglink)
                            .addClass('ebookreader-headimage')
                            .attr('style', 'background-image: url('+hidata.data.content+');')
                            .attr('data-headimagetype', headimage.type);
                        element.append(imgpopup);
                    };
                }
            });
        };
    };

    /******
     * Update pageneck in given element with neck things of page with pageid.
     ******/
    ReaderView.prototype.updatePageNeck = function(element, pageid){
        element.empty().attr('data-element-name', '');
    }

    /******
     * Update pagebody in given element with body content of page with pageid.
     ******/
    /*ReaderView.prototype.updatePageBody = function(element, pageid){
        var content = this.book.getContentById(pageid);
        if (content !== '' && element.hasClass('ebook-pageelement')) {
            // Just update the data.
            this.getContentData({
                contentlist: [content],
                callback: function(data){
                    element.attr('data-element-name', content).pageelement('setdata', data[content]);
                }
            });
        } else if (content !== '') {
            // First rendering (init the pageelement with jQuery-plugin).
            this.getContentData({
                contentlist: [content],
                callback: function (data) {
                    element.attr('data-element-name', content).pageelement(data[content]);
                }
            });
        }
    }*/
    
    /******
     * Update pagebody in given element with body content of page with pageid.
     ******/
    ReaderView.prototype.updatePageBody = function(element, options){
        var reader = this;
        var content = this.book.getContentById(options.pageid);
        var lang = this.getConf('lang');
        var uilang = this.getConf('uilang');
        var showsettings = {
            mode: 'view',
            lang: lang,
            uilang: lang,
            username: this.settings.username,
            users: this.settings.users,
            role: this.settings.role,
            gotolink: JSON.parse(JSON.stringify(this.settings.gotolink || {})),
            usemargin: true
        };
        showsettings.gotolink.chapid = options.pageid;
        showsettings.gotolink.pageelementid = content;
        if (content !== '') {
            if (!options.renderEmpty) {
                if (element.hasClass('ebook-pageelement')) {
                    // Just update the data.
                    if (options.pageid === this.book.frontId()) {
                        // If the page is frontpage, show some special data.
                        var pelement = {
                            type: 'pageelement',
                            metadata: {},
                            data: {
                                contents: [
                                    {id: 'titledata0'},
                                    {id: 'authordata0'}
                                ],
                                contentdata: {
                                    'titledata0': {
                                        type: 'coverpageelement',
                                        metadata: {},
                                        data: this.book.getTitleData(lang)
                                    },
                                    'authordata0': {
                                        type: 'authorselement',
                                        metadata: {},
                                        data: this.book.getAuthors()
                                    }
                                }
                            }
                        };
                        var frontimage = this.book.getFrontImage(options.pageid);
                        if (frontimage) {
                            pelement.data.contents.unshift({id: frontimage});
                        };
                        var bookserielogo = this.book.getBookSerieLogo();
                        if (bookserielogo) {
                            pelement.data.contents.unshift({id: bookserielogo});
                        };
                        element.pageelement('setdata', pelement);
                    } else {
                        this.getContentData({
                            contentlist: [content],
                            callback: function(data){
                                data[content].metadata.lang = data[content].metadata.lang || lang;
                                data[content].settings = showsettings;
                                element.attr('data-element-name', content).pageelement('setdata', data[content]);
                            }
                        });
                    }
                } else {
                    // First rendering (init the pageelement with jQuery-plugin).
                    if (options.pageid === this.book.frontId()) {
                        // If the page is frontpage, show some special data.
                        // If the page is frontpage, show some special data.
                        var pelement = {
                            type: 'pageelement',
                            metadata: {},
                            data: {
                                contents: [
                                    {id: 'titledata0'},
                                    {id: 'authordata0'}
                                ],
                                contentdata: {
                                    'titledata0': {
                                        type: 'coverpageelement',
                                        metadata: {},
                                        data: this.book.getTitleData(lang)
                                    },
                                    'authordata0': {
                                        type: 'authorselement',
                                        metadata: {},
                                        data: this.book.getAuthors()
                                    }
                                }
                            }
                        };
                        var frontimage = this.book.getFrontImage(options.pageid);
                        if (frontimage) {
                            pelement.data.contents.unshift({id: frontimage});
                        };
                        var bookserielogo = this.book.getBookSerieLogo();
                        if (bookserielogo) {
                            pelement.data.contents.unshift({id: bookserielogo});
                        };
                        element.pageelement(pelement);
                    } else {
                        this.getContentData({
                            contentlist: [content],
                            callback: function (data) {
                                data[content].metadata.lang = data[content].metadata.lang || lang;
                                data[content].settings = showsettings;
                                element.attr('data-element-name', content).pageelement(data[content]);
                            }
                        });
                    };
                }
            } else {
                if (element.hasClass('ebook-pageelement')) {
                    // Just update the data.
                    this.getContentData({
                        contentlist: [content],
                        callback: function(data){
                            var empty = {metadata:{lang: lang},data:{},type:"pageelement"}
                            element.attr('data-element-name', content).pageelement('setdata', empty);
                        }
                    });
                }
            }
        } else{
            console.log("content was null");
        }
        
    }
    
    /******
     * Update menu
     ******/
    ReaderView.prototype.updateMenu = function(){
        this.menulist.empty();
        for (var item in ReaderView.menu) {
            this.addMenuItem(item);
        }
    }
    
    /****
     *Closes the menu if it's open. Does nothing otherwise.
     *
     *@returns true if menu was closed
     ****/
    ReaderView.prototype.closeMenu = function(){
        if (!this.menulist.hasClass('closed')) {
            this.menulist.addClass('closed');
            return true;
        }
        else{
            return false;
        }
    }
    
    /******
     * Update breadcrumb
     ******/
    ReaderView.prototype.updateBreadcrumb = function(){
        var html = this.book.getBreadcrumb(this.getConf('currentpage'), this.getConf('lang'));
        html = html.replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>');
        this.breadcrumbarea.html(html);
        this.breadcrumbarea.find('.mathquill-embedded-latex').mathquill();
    }
    
    /******
     * Set pagetypes
     ******/
    ReaderView.prototype.setPageTypes = function(types){
        if (typeof(types) === 'undefined' || typeof(types.join) === 'undefined') {
            types = this.book.getTypes(this.getConf('currentpage'));
        }
        this.viewarea.children('[data-pageframe="current"]').attr('data-pagetypes', types.join(' '));
        if (types.indexOf('front') !== -1) {
            this.place.addClass('showfront');
        } else {
            this.place.removeClass('showfront');
        }
    }
    
    /******
     * Set footer content for the front page
     ******/
    ReaderView.prototype.setFooter = function(){
        var viewer = this;
        var logoarea = this.footerarea.children('.vendorlogo');
        //var authorarea = this.footerarea.children('.authorlist');
        //var sauthorarea = this.footerarea.children('.secondaryauthorlist');
        var vendorLogo = this.book.getVendorLogo();
        if (vendorLogo) {
            //vendorLogo = this.getConf('lang') + '/' + vendorLogo;
            this.getContentData({
                contentlist: [vendorLogo],
                lang: this.getConf('lang'),
                callback: function(data){
                    var cdata = data[vendorLogo];
                    var ctype = cdata.type;
                    if ($.fn[ctype]) {
                        logoarea.attr('data-element-name', vendorLogo)[ctype](cdata);
                    }
                }
            })
        }
        //var authors = this.book.getAuthors();
        //var authorhtml = ['<ul>'];
        //for (var i = 0; i < authors.author.length; i++) {
        //    authorhtml.push('<li>'+authors.author[i]+'</li>')
        //}
        //authorhtml.push('</ul>');
        //var sauthorhtml = [];
        //for (var i = 0; i < authors.secondaryauthor.length; i++) {
        //    var authorset = authors.secondaryauthor[i];
        //    sauthorhtml.push('<div class="secondaryauthorset">');
        //    sauthorhtml.push('<h2>'+authorset.name+'</h2>');
        //    sauthorhtml.push('<ul>');
        //    for (var j = 0; j < authorset.author.length; j++) {
        //        sauthorhtml.push('<li>'+authorset.author[j]+'</li>')
        //    }
        //    sauthorhtml.push('</ul>');
        //    sauthorhtml.push('</div>');
        //}
        //authorarea.html(authorhtml.join('\n'));
        //sauthorarea.html(sauthorhtml.join('\n'));
        //this.footerarea.off('click', '.secondaryauthorset h2').on('click', '.secondaryauthorset h2', function(e){
        //    e.stopPropagation();
        //    var header = $(this);
        //    var isopen = header.parent().hasClass('open');
        //    header.closest('.secondaryauthorlist').children('.secondaryauthorset').removeClass('open');
        //    if (isopen) {
        //        header.closest('.secondaryauthorset').removeClass('open');
        //    } else {
        //        header.closest('.secondaryauthorset').addClass('open');
        //        viewer.place.on('click.secondaryauthorsetpopup', function(e){
        //            viewer.place.off('click.secondaryauthorsetpopup');
        //            viewer.footerarea.find('.secondaryauthorset').removeClass('open');
        //        })
        //    }
        //});
    }
    
    /******
     * Set navbar
     ******/
    ReaderView.prototype.setNavbar = function(){
        var navioptions = {
            data: this.booklinear,
            ticklevel: this.getConf('ticklevel'),
            labeldown: this.navlabeldown,
            current: this.getConf('currentpage') || this.booklinear[0].id
        }
        this.naviarea.navbar(navioptions);
    }
    
    /******
     * Set the current language
     ******/
    ReaderView.prototype.setLanguage = function(lang){
        this.setConf('lang', lang);
        //Setting current page again refreshes prev and next pages so they update to new lang setting too
        this.setPage(this.currentpage);
        this.booklinear = this.book.getLinearToc(this.getConf('lang'));
        this.setNavbar();
        this.announceTitle(lang);
    }
    
    ReaderView.prototype.hasLanguage = function(lang){
        return ebooklocalizer.hasLanguage(lang);
    }
    
    /******
     * Set the current widget theme
     ******/
    ReaderView.prototype.setWidgetTheme = function(theme){
        if (typeof(theme) === 'string') {
            this.setConf('widgettheme', theme);
        }
        this.place.attr('data-ffwtheme', this.getConf('widgettheme'));
    }
    
    /******
     * Set the current theme
     ******/
    ReaderView.prototype.setTheme = function(theme){
        if (typeof(theme) === 'string') {
            this.setConf('theme', theme);
        }
        this.place.attr('data-readertheme', this.getConf('theme'));
    }
    
    /******
     * Show settings frame
     ******/
    ReaderView.prototype.showSettingsFrame = function(){
        this.viewarea.children('.ebookreader-settingsframe').addClass('settingson');
    }
    
    /******
     * Hide settings frame
     ******/
    ReaderView.prototype.hideSettingsFrame = function(){
        this.viewarea.children('.ebookreader-settingsframe').removeClass('settingson');
        if (this.needsupdate) {
            this.needsupdate = false;
            this.setPage();
        }
    }
    
    /******
     * Show toc 
     ******/
    ReaderView.prototype.showToc = function(){
        var reader = this;
        var html = this.book.getTocHtml(null, this.getConf('lang'));
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        //frame.attr('data-chapid','');
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        this.clearMenuSelections();
        this.setPageTypes([]);
        //this.pageheader.empty();
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.toc+'</div> <span class="ebookreader-menupage-title">'+ebooklocalizer.localize('reader:TOC', this.getConf('lang'))+'</span></h1>');
        pagebody.html(html);
        pagebody.find('.mathquill-embedded-latex').mathquill();
        pagebody.find('li[data-ebtocchapid="'+this.getConf('currentpage')+'"]').addClass('ebookreader-toccurrent');
        pagebody.on('click.toclink', 'li.ebooktocitem', function(event){
            event.stopPropagation();
            event.preventDefault();
            pagebody.off('click.toclink');
            var chapterli = $(this);
            var chapid = chapterli.attr('data-ebtocchapid');
            chapterli.trigger('setpage', [chapid]);
            reader.hideSettingsFrame();
        });
    }
    
    /******
     * Show language chooser
     ******/
    ReaderView.prototype.showLangchooser = function(){
        var reader = this;
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        //frame.attr('data-chapid','');
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        this.clearMenuSelections();
        this.setPageTypes([]);
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.langchooser+'</div> <span class="ebookreader-menupage-title">'+ebooklocalizer.localize('reader:language', this.getConf('lang'))+'</span></h1>');
        var languages = this.book.getLangs();
        languages.sort(function(a,b){
            return (a.name > b.name ? 1 : -1);
        });
        var langrows = [];
        for (var i = 0; i < languages.length; i++){
            langrows.push('<div data-lang-value="' + languages[i].id + '" class="ebookreader-langselect'+(languages[i].id === this.getConf('lang') ? ' buttonselected' : '')+' ffwidget-setbutton"> ' + languages[i].name + '</div>');
        }
        var html = [
            '<div class="ffwidget-buttonset">',
            langrows.join('\n'),
            '</div>'
        ].join('\n');
        pagebody.html(html);
        pagebody.off('click.selectlang').on('click.selectlang', '.ebookreader-langselect', function(event){
            event.preventDefault();
            event.stopPropagation();
            pagebody.off('click.selectlang');
            var lang = $(this).attr('data-lang-value') || reader.getConf('lang');
            reader.setLanguage(lang);
            reader.updateMenu();
            reader.hideSettingsFrame();
        })
    }
    
    /******
     * Show widget theme chooser
     ******/
    ReaderView.prototype.showWidgetThemechooser = function(){
        var reader = this;
        var lang = this.getConf('lang');
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        //frame.attr('data-chapid','');
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        this.clearMenuSelections();
        this.setPageTypes([]);
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.widgetthemechooser+'</div> <span class="ebookreader-menupage-title">'+ebooklocalizer.localize('reader:widget theme', this.getConf('lang'))+'</span></h1>');
        var themes = ReaderView.widgetthemes;
        //themes.sort(function(a,b){
        //    return a.name > b.name;
        //});
        var current = this.getConf('widgettheme') || this.place.attr('data-ffwtheme') || '';
        var optionrows = [];
        for (var i = 0; i < themes.length; i++){
            optionrows.push('<div data-widgettheme-value="' + themes[i].id + '" class="ebookreader-widgetthemeselect'+(themes[i].id === current ? ' buttonselected' : '')+' ffwidget-setbutton"> ' + (themes[i].name[lang] || themes[i].name['en']) + '</div>');
        }
        var html = [
            '<div class="ffwidget-buttonset">',
            optionrows.join('\n'),
            '</div>'
        ].join('\n');
        pagebody.html(html);
        pagebody.off('click.selecttheme').on('click.selecttheme', '.ebookreader-widgetthemeselect', function(event){
            event.preventDefault();
            event.stopPropagation();
            var $button = $(this);
            var theme = $button.attr('data-widgettheme-value') || '';
            reader.setWidgetTheme(theme);
            $button.parent().find('.buttonselected').removeClass('buttonselected');
            $button.addClass('buttonselected');
        })
    }
    
    /******
     * Show theme chooser
     ******/
    ReaderView.prototype.showThemechooser = function(){
        var reader = this;
        var debug = 0;
        var lang = this.getConf('lang');
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        //frame.attr('data-chapid','');
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        this.clearMenuSelections();
        this.setPageTypes([]);
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.themechooser+'</div> <span class="ebookreader-menupage-title">'+ebooklocalizer.localize('reader:theme', this.getConf('lang'))+'</span></h1>');
        var themes = ReaderView.themes;
        //themes.sort(function(a,b){
        //    return a.name > b.name;
        //});
        var current = this.getConf('theme') || this.place.attr('data-readertheme') || '';
        var optionrows = [];
        for (var i in themes){
            optionrows.push('<div data-theme-value="' + themes[i].id + '" class="ebookreader-themeselect'+(themes[i].id === current ? ' buttonselected' : '')+' ffwidget-setbutton"> ' + (themes[i].name[lang] || themes[i].name['en']) + '</div>');
        }
        var html = [
            '<table class="ebookreader-themeselect-table">',
            '<tr><td class="ebookreader-themeselect-box">',
            '<div>',
            '<div class="ffwidget-buttonset">',
            optionrows.join('\n'),
            '</div>',
            '</div>',
            '</td><td class="ebookreader-theme-description"><div>' + (ReaderView.themes[current] ? ReaderView.themes[current].description[lang] || ReaderView.themes[current].description['en'] : '') + '</div></td></tr>',
            '<tr><td colspan="2">' + ebooklocalizer.localizeText(ReaderView.strings.samplepage, lang) + '</td></tr></table>'
        ].join('\n');
        pagebody.html(html);
        var descriptiontd = pagebody.find('td.ebookreader-theme-description');
        pagebody.off('click.selecttheme').on('click.selecttheme', '.ebookreader-themeselect', function(event){
            event.preventDefault();
            event.stopPropagation();
            var $button = $(this);
            var theme = $button.attr('data-theme-value') || '';
            reader.setTheme(theme);
            $button.parent().find('.buttonselected').removeClass('buttonselected');
            $button.addClass('buttonselected');
            var description = ReaderView.themes[theme].description[lang] || ReaderView.themes[theme].description['en'] || '';
            descriptiontd.html(description);
        })
    }
    
    /******
     * Show settings dialog
     ******/
    ReaderView.prototype.showSettings = function(){
        var reader = this;
        var debug = 0;
        var lang = this.getConf('lang');
        this.clearMenuSelections();
        this.setPageTypes([]);
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        //frame.attr('data-chapid','');
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.wrench+'</div> <span class="ebookreader-menupage-title">'+ebooklocalizer.localize('reader:settingsmenu', this.getConf('lang'))+'</span></h1>');

        var settings = reader.config.settings;
        //settings.fontsettings = {};
        //settings.sdsettings = {};
        var html = [
            //Structured derivation settings
            '<fieldset data-setting="sdsettings">',
            '<legend>'+ebooklocalizer.localize('reader:structured derivations', lang)+'</legend>',
            '<label><input type="checkbox" '+(reader.config.settings.sdsettings.stepmode ? 'checked="checked"' : '')+' data-settingskey="stepmode"> '+ ebooklocalizer.localize('reader:showstepbystep', lang) +'</label>',
            '</fieldset>',

            //Font settings
            '<fieldset data-setting="fontsettings">',
            '<legend>'+ebooklocalizer.localize('reader:font', lang)+'</legend>',
            '<label><input type="checkbox" '+(reader.config.settings.fontsettings.dyslexiamode ? 'checked="checked"' : '')+' data-settingskey="dyslexiamode"> '+ ebooklocalizer.localize('reader:usedyslexiafont', lang) +'</label>',
            '</fieldset>'
        ].join('\n');
        pagebody.html(html);
        pagebody.off('change.settings').on('change.settings', 'input', function(event){
            event.preventDefault();
            event.stopPropagation();
            var $input = $(this);
            var key = $input.attr('data-settingskey');
            var settingset = $input.closest('[data-setting]').attr('data-setting');
            var value = ($input.attr('type') === 'checkbox' ? $input.is(':checked') : '');
            settings[settingset][key] = value;

            //Take action
            switch (settingset){
                case 'sdsettings':
                    var sdsettings = $.fn.sdeditor('getsettings');
                    sdsettings.stepmode = value;
                    $.fn.sdeditor('settings', settings.sdsettings);
                    reader.needsupdate = true;
                    break;
                case 'fontsettings':
                    var wrapper = $input.closest('.ebookreader-wrapper');
                    if(value){
                        wrapper.addClass('useDyslexiaFont');
                    }
                    else{
                        wrapper.removeClass('useDyslexiaFont');
                    }
                default:
                    break;
            }
            reader.config.settings = settings;
            reader.saveConf();
        });
    };


    /******
     * Show about dialog
     ******/
    ReaderView.prototype.showAbout = function(){
        var reader = this;
        var debug = 0;
        var lang = this.getConf('lang');
        this.clearMenuSelections();
        this.setPageTypes([]);
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.infoicon+'</div> <span class="ebookreader-menupage-title">'+ ebooklocalizer.localize('reader:aboutmenu', this.     getConf('lang'))+'</span></h1>');

        var aboutData = ReaderView.strings.aboutArray.slice();
        aboutData.sort(function(a,b){return(a.type < b.type ? -1 : 1);})
        //HTML of the body, each cell represents one line of html
        var bodyHtml = [];
        bodyHtml.push('<div class="aboutcontainer">');
        var fieldSets = [];
        var lastType;
        //Form <fieldset> for each type of data and insert the object data into corresponding fieldset. 
        for (i=0; i< aboutData.length; i++) {
            var data = aboutData[i];
            var type = aboutData[i].type;
            //If first of it's type, add new fieldset
            if (!lastType || lastType !== type) {
                bodyHtml.push('   <fieldset datatype=' + type + '>');
                bodyHtml.push('<legend>' + ebooklocalizer.localize('reader:'+type, lang) + '</legend><br>');
            }
           
            //Multiple var's used for readability
            var title = '<a target="_blank" class="title" href=\"' + data.titleLink + '\">' + data.title + '</a><label> ';
            var author = '<li class="author">' + ebooklocalizer.localize('reader:author', this.getConf('lang')) + ':  <a target="_blank"  href=\"' + data.authorLink + '\">' + data.author +  '</a></li>';
            var license = '<li class="license">' + ebooklocalizer.localize('reader:license', this.getConf('lang')) + ':  <a target="_blank"  href=\"' + data.licenseLink + '\">' + data.license +  '</a></li>';
            bodyHtml.push(title +'<ul>' + author + license + '</ul>');
            
            lastType = type;
            //If last element or last of it's type, close fieldset
            if ((i+1) === aboutData.length || aboutData[i+1].type !== type) {
                bodyHtml.push( '   </fieldset>');
            }
            
        }
        bodyHtml.push('</div>');
        pagebody.html(bodyHtml.join('\n'));
    };
    
    /******
     * Show help dialog
     ******/
    // TODO: Fix this!
    ReaderView.prototype.showHelp = function(){
        var reader = this;
        var debug = 0;
        var lang = this.getConf('lang');
        this.clearMenuSelections();
        this.setPageTypes([]);
        var frame = this.viewarea.children('.ebookreader-settingsframe');
        this.showSettingsFrame();
        var pageheader = frame.children('.ebookreader-pageheader');
        var pagebody = frame.children('.ebookreader-pagebody');
        pageheader.html('<h1><div class="ebookreader-menupage-icon">'+ReaderView.strings.icons.questionmark+'</div> <span class="ebookreader-menupage-title">'+ ebooklocalizer.localize('reader:helpmenu', lang)+'</span></h1>');
        var bodyHtml = [];
        bodyHtml.push('<div class="helpcontainer">');
        bodyHtml.push('   <h1>'+ebooklocalizer.localize('reader:instructionstitle',lang)+'</h1>');
        bodyHtml.push('      <p>'+ebooklocalizer.localize('reader:instructionsdescription',lang)+'</p><br>');
        bodyHtml.push('      <a href="'+ebooklocalizer.localize('reader:instructionsurl',lang)+'" target="_blank">'+ ebooklocalizer.localize('reader:instructionsurldescription',lang) + '</a>');
        bodyHtml.push('</div>');
        pagebody.html(bodyHtml.join('\n'));
    }


    
    /******
     * Show content in notearea.
     ******/
    ReaderView.prototype.showNote = function(options){
        options = $.extend({
            title: '',
            note: '',
            iselement: false,
            fullscreen: false
        }, options);
        this.place.addClass('notevisible');
        if (options.fullscreen) {
            this.place.addClass('notefullpage');
        }
        this.notetitle.html(options.title);
        if (options.iselement) {
            var lang = this.getConf('lang');
            var container = $('<div>');
            this.notecontent.html(container);
            this.getContentData({
                lang: lang,
                contentlist: [options.note],
                callback: function(data){
                    container.attr('data-element-name', options.note).elementbox(data[options.note]);
                }
            });
        } else {
            this.notecontent.html(options.note);
        }
    };
    
    /******
     * Close notearea
     ******/
    ReaderView.prototype.closeNote = function(){
        this.place.removeClass('notevisible notefullpage');
    }
    
    /******
     * Clear menuselections
     ******/
    ReaderView.prototype.clearMenuSelections = function(){
        this.place.removeAttr('data-opencontent');
        this.menulist.find('li.menuactive').removeClass('menuactive');
    }

    /******
     * Get data of the page from the book.
     ******/
    ReaderView.prototype.getContentData = function(options){
        options = $.extend({
            lang: null
        }, options);
        this.book.getContentData(options.contentlist, options.lang, options.callback);
    }


    
    /******
     * Set the current page to one with given chapid.
     ******/
    ReaderView.prototype.setPage = function(pagedata){
        var viewer = this;
        var timeout = 0;
        var refreshAll = false;
        if (pagedata) {
            if (typeof(pagedata) === 'string') {
                pagedata = {current: pagedata}
            }
            pagedata = $.extend({
                prev: '',
                current: '',
                next: ''
            }, pagedata)
            this.pagehistory.push(pagedata.current);
            this.setConf('prevpage', pagedata.prev);
            this.setConf('currentpage', pagedata.current);
            this.setConf('nextpage', pagedata.next);
        } else {
            pagedata = {
                prev: this.getConf('prevpage'),
                current: this.getConf('currentpage'),
                next: this.getConf('nextpage')
            }
            refreshAll = true;
        }
        var prevframe = this.viewarea.children('[data-pageframe="prev"]'),
            currframe = this.viewarea.children('[data-pageframe="current"]'),
            nextframe = this.viewarea.children('[data-pageframe="next"]');
        if (nextframe.attr('data-chapid') === pagedata.current) {
            nextframe.removeClass('lastprev').addClass('lastnext');
            currframe.removeClass('lastprev lastnext');
            prevframe.removeClass('lastnext').addClass('lastprev');
            nextframe.attr('data-pageframe', 'current');
            currframe.attr('data-pageframe', 'prev');
            prevframe.attr('data-pageframe', 'next');
            timeout = 600;
        } else if (prevframe.attr('data-chapid') === pagedata.current) {
            nextframe.removeClass('lastprev').addClass('lastnext');
            currframe.removeClass('lastprev lastnext');
            prevframe.removeClass('lastnext').addClass('lastprev');
            nextframe.attr('data-pageframe', 'prev');
            currframe.attr('data-pageframe', 'next');
            prevframe.attr('data-pageframe', 'current');
            timeout = 600;
        }
        this.updateBreadcrumb();
        this.clearMenuSelections();
        this.closeNote();
        this.setPageTypes();

        if (timeout) {
            setTimeout(function(){viewer.updatePage(refreshAll);}, timeout);
        } else {
            // Old mobile safari would miss timeouts if scrolling.
            viewer.updatePage(refreshAll);
        }
        
    }
    
    /*****
     * Get the default language of the book.
     ******/
    ReaderView.prototype.getDefaultLang = function(){
        return this.book.getDefaultLang();
    }
    
    /*****
     * Get the list of children of given chapter
     ******/
    ReaderView.prototype.getChildren = function(chapid){
        return this.book.getChildren(chapid, this.getConf('lang'));
    }
    
    /******
     * Add an item to the menu
     ******/
    ReaderView.prototype.addMenuItem = function(itemName){
        var menuitem = ReaderView.menu[itemName];
        this.menulist.append('<li class="ebookreader-menuitem ebookreader-'+menuitem.name+' ffwidget-menubutton" title="'+ebooklocalizer.localize(menuitem.title, this.getConf('lang'))+'">'+menuitem.icon+'</li>');
        var data = {viewer: this};
        this.place.hammer().off('tap', '.ebookreader-'+menuitem.name);
        this.place.hammer().on('tap', '.ebookreader-'+menuitem.name, data, menuitem.action);
    }
    
    /*****************************************
     * Solutionhandling
     ******/
    
    /******
     * Check, if solutions exist.
     ******/
    ReaderView.prototype.checkLocalstorageSolutions = function(){
        if(typeof(this.solutions) === "undefined"){
            if(typeof(localStorage.solutions) !== "undefined"){
                this.solutions = JSON.parse(localStorage.solutions);
            }else{
                this.solutions = {};
            }
        }
    }
    
    /******
     * Find free id for solution
     ******/
    ReaderView.prototype.createSolution = function(assignment){
        var i = 0;
        while (typeof(this.solutions["solution"+i]) !== "undefined"){
            i = i +1;
        }
        return "solution"+i;
    }
    
    /******
     * Get solution data
     ******/
    ReaderView.prototype.getSolutionData = function(assignment){
        var data = [];
        this.checkLocalstorageSolutions();
        for (var solutionid in this.solutions){
            if (this.solutions[solutionid].solutionto && this.solutions[solutionid].solutionto === assignment){
                data.push(this.solutions[solutionid]);
            }
        }
        return data;
    }
    
    /******
     * Set solution data
     ******/
    ReaderView.prototype.setSolutionData = function(assignment,solutiondata){
        try{
            this.checkLocalstorageSolutions();
            if (typeof(solutiondata.solutionid) ==="undefined"){
                var solutionid = this.createSolution(assignment);
            }else{
                var solutionid = solutiondata.solutionid;
            }
            this.solutions[solutionid] = solutiondata;
            localStorage.solutions = JSON.stringify(this.solutions);
            return solutionid;
        }catch (e){
            return e+"fail";
        }
    }
    
    ReaderView.prototype.onWindowResize = function(change){
        var width = this.viewarea.width() + (-change || 0);
        this.viewarea.children('[data-pageframe="current"]').css('width', width);
        this.viewarea.children('[data-pageframe="prev"]').css('width', width);
        this.viewarea.children('[data-pageframe="next"]').css('width', width);
    };
    
    ReaderView.prototype.close = function() {
        this.place.trigger('closeappok', {appid: this.getBookid(), apptype: 'reader'});
    };
    
    
    
    
    /******
     * Some constant strings for ReaderView: templates, svg-icons, css, etc.
     ******/
    ReaderView.strings = {};
    
    // Some svg-icons for ReaderView
    ReaderView.strings.icons = {
        menu1: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 20 20" class="mini-icon-menu mini-icon"><path style="stroke: none;" d="M2 3 l16 0 l0 3 l-16 0z m0 6 l16 0 l0 3 l-16 0z m0 6 l16 0 l0 3 l-16 0z" /></svg>',
        menu2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="3 3 44 44" class="mini-icon ebook-icon ebook-icon-book"><path stroke="none" fill="white" d="M21 6 a20 20 0 0 1 10 0 a3 6 22.5 0 0 5.3 2.2 a20 20 0 0 1 7 7 a3 6 67.5 0 0 2.2 5.3 a20 20 0 0 1 0 10 a3 6 112.5 0 0 -2.2 5.3 a20 20 0 0 1 -7 7 a3 6 157.5 0 0 -5.3 2.2 a20 20 0 0 1 -10 0 a3 6 -157.5 0 0 -5.3 -2.2 a20 20 0 0 1 -7 -7 a3 6 -112.5 0 0 -2.2 -5.3 a20 20 0 0 1 0 -10 a3 6 -67.5 0 0 2.2 -5.3 a20 20 0 0 1 7 -7 a3 6 -22.5 0 0 5.3 -2.2z m5 13 a7 7 0 0 0 0 14 a7 7 0 0 0 0 -14z" /><path stroke="none" fill="black" d="M20 5 a20 20 0 0 1 10 0 a3 6 22.5 0 0 5.3 2.2 a20 20 0 0 1 7 7 a3 6 67.5 0 0 2.2 5.3 a20 20 0 0 1 0 10 a3 6 112.5 0 0 -2.2 5.3 a20 20 0 0 1 -7 7 a3 6 157.5 0 0 -5.3 2.2 a20 20 0 0 1 -10 0 a3 6 -157.5 0 0 -5.3 -2.2 a20 20 0 0 1 -7 -7 a3 6 -112.5 0 0 -2.2 -5.3 a20 20 0 0 1 0 -10 a3 6 -67.5 0 0 2.2 -5.3 a20 20 0 0 1 7 -7 a3 6 -22.5 0 0 5.3 -2.2z m5 13 a7 7 0 0 0 0 14 a7 7 0 0 0 0 -14z" /></svg>',
        menu: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 32 32" class="mini-icon mini-icon-gear"><path style="stroke: none;" d="M12 0 h8 l0.9 3.5 l3.5 2 l3.55 -1.1 l4.1 7 l-2.7 2.5 v4.1 l2.7 2.5 l-4.1 7 l-3.55 -1.1 l-3.5 2 l-0.9 3.5 h-8 l-0.9 -3.5 l-3.5 -2 l-3.55 1.1 l-4.1 -7 l2.7 -2.5 v-4.1 l-2.7 -2.5 l4.1 -7 l3.55 1.1 l3.5 -2z m1.55 2 l-0.7 2.9 l-5 2.8 l-2.8 -0.8 l-2.5 4.3 l2.15 2 v5.65 l-2.15 2 l2.5 4.3 l2.8 -0.8 l5 2.8 l0.7 2.9 h4.9 l0.7 -2.9 l5 -2.8 l2.8 0.8 l2.5 -4.3 l-2.15 -2 v-5.65 l2.15 -2 l-2.5 -4.3 l-2.8 0.8 l-5 -2.8 l-0.7 -2.9z m2.45 7.55 a6.45 6.45 0 0 1 0 12.9 a6.45 6.45 0 0 1 0 -12.9z m0 2 a4.45 4.45 0 0 0 0 8.9 a4.45 4.45 0 0 0 0 -8.9z"></path></svg>',
        wrench: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 50 50" class="ebook-icon mini-icon mini-icon-wrench"><path stroke="none" fill="white" d="M34 4 a12 12 0 1 0 9 9 l-9 9 a10 10 0 0 1 -9 -9 l9 -9z m-15 17.5 l-17 17 a2 2 0 0 0 7 7 l17 -17z m-16 18 a2 2 0 0 1 5 5 a2 2 0 0 1 -5 -5z" /><path stroke="none" fill="black" d="M35 5 a12 12 0 1 0 9 9 l-9 9 a10 10 0 0 1 -9 -9 l9 -9z m-15 17.5 l-17 17 a2 2 0 0 0 7 7 l17 -17z m-16 18 a2 2 0 0 1 5 5 a2 2 0 0 1 -5 -5z" /></svg>',
        toc: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-toc"><path style="stroke: none;" d="M5 4 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m7 -18 l14 0 a2.5 2.5 0 0 1 0 5 l-14 0 a2.5 2.5 0 0 1 0 -5z m0 9 l14 0 a2.5 2.5 0 0 1 0 5 l-14 0 a2.5 2.5 0 0 1 0 -5z m0 9 l14 0 a2.5 2.5 0 0 1 0 5 l-14 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        toc2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-toc"><path style="stroke: none;" d="M4 4 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l15 0 l0 4 l-15 0z M4 10 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l15 0 l0 4 l-15 0z M10 16 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l15 0 l0 4 l-15 0z M10 22 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l15 0 l0 4 l-15 0z M4 28 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -2 l15 0 l0 4 l-15 0z" /></svg>',
        langchooser: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 20 20" class="mini-icon-langchoose mini-icon"><path style="fill: white;" d="M3 2 l14 0 a2 3 0 0 1 2 2 l0 10 a2 2 0 0 1 -2 2 l-4 0 l-6 3 l1 -3 l-5 0 a2 2 0 0 1 -2 -2 l0 -10 a2 2 0 0 1 2 -2z" /><path style="stroke: none;" d="M3 2 l14 0 a2 3 0 0 1 2 2 l0 10 a2 2 0 0 1 -2 2 l-4 0 l-6 3 l1 -3 l-5 0 a2 2 0 0 1 -2 -2 l0 -10 a2 2 0 0 1 2 -2z m0 1 a2 2 0 0 0 -1 1 l0 10 a2 2 0 0 0 1 1 l6.5 0 l-1 2.5 l4.5 -2.5 l4 0 a2 2 0 0 0 1 -1 l0 -10 a2 2 0 0 0 -1 -1 l-14 0z M12 6 l 2 0 l3 8 l-1.5 0 l-1.2 -3 l-2.6 0 l-1.2 3 l-1.5 0 l3 -8 m1 1 l-1 3 l2 0 l-1 -3z M3.5 5.5 l6 -0.3 l0.5 1 l-6 0.2z m3.5 1 l1 0 a7 7 0 0 1 -5 7 l-0.2 -0.5 a8 8 0 0 0 4.2 -6.5z m-3.5 0.8 l0.5 -0.3 a10 10 0 0 0 5.5 5 l-0.5 1 a11 11 0 0 1 -5.5 -5.7z m2.6 -4 l2 0.6 l0 1 l-2 -1.2z" /></svg>',
        widgetthemechooser: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 20 20" class="mini-icon-widgetthemechooser mini-icon"><path style="stroke: black; fill: red;" d="M0 0 l8 0 l0 12 l-8 0z" transform="translate(1 8) rotate(-30)" /><path style="stroke: black; fill: blue;" d="M0 0 l8 0 l0 12 l-8 0z" transform="translate(12 4) rotate(30)" /><path style="stroke: black; fill: yellow;" d="M0 0 l8 0 l0 12 l-8 0z" transform="translate(6 4)" /></svg>',
        themechooser: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 20 20" class="mini-icon-themechooser mini-icon"><path style="stroke: none; fill: #a00;" d="M1 1 l18 0 l0 8 l-18 0z" /><path style="fill: white; stroke: none;" d="M2 4 l16 0 l0 4 l-16 0z" /><path style="stroke: none; fill: #55a;" d="M1 10 l18 0 l0 8 l-18 0z" /><path style="fill: white; stroke: none;" d="M2 13 l16 0 l0 4 l-16 0z" /></svg>',
        fullscreen: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-presentation"><path style="fill: white;" d=" M4 4 l0 13 l22 0 z" /><path style="stroke: none;" d="M3 3 l24 0 l0 15 l-24 0 z m1 1 l0 13 l22 0 z m9 -1 l0 -2 l4 0 l0 2 z m0 15 l4 0 l0 8 l5 3 l-2 0 l-5 -3 l-5 3 l-2 0 l5 -3" /></svg>',
        linkicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z" /></svg>',
        infoicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-info"><path style="stroke: none;" d="M15 4 a2.5 2.5 0 0 0 0 5 a2.5 2.5 0 0 0 0 -5z m3 7 l-7 1 l0 2 a3 2 0 0 1 3 2 l0 6 a3 2 0 0 1 -3 2 l0 2 10 0 l0 -2 a3 2 0 0 1 -3 -2z" /></svg>',
        closeicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon ebook-icon ebook-icon-close"><path fill="white" style="stroke: none;" d="M2 16 m3 -6 l4 -4 l6 6 l6 -6 l4 4 l-6 6 l6 6 l-4 4 l-6 -6 l-6 6 l-4 -4 l6 -6 z" /><path class="icon-maincolor" style="stroke: none;" d="M1 15 m3 -6 l4 -4 l6 6 l6 -6 l4 4 l-6 6 l6 6 l-4 4 l-6 -6 l-6 6 l-4 -4 l6 -6 z" /></svg>',
        closeicon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-cancel"><path style="stroke: none; fill: #a00;" d="M1 15 a14 14 0 0 0 28 0 a14 14 0 0 0 -28 0z" /><path style="fill: red; stroke: none;" d="M2 15 a13 13 0 0 1 26 0 a13 13 0 0 1 -26 0z"/><path style="fill: white; stroke: #a00;" d="M6 10 l4 -4 l5 5 l5 -5 l4 4 l-5 5 l5 5 l-4 4 l-5 -5 l-5 5 l-4 -4 l5 -5 z" /><path style=" stroke: none; fill: white; opacity: 0.4;" d="M2 15 a13 13 0 0 1 26 0 a13 13 0 0 0 -13 0 a13 13 0 0 1 -13 0z" /></svg>',
        extraicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="50" viewBox="0 0 50 50" class="mini-icon ebook-icon ebook-icon-extra"><path stroke="none" fill="white" d="M19 11 a2 2 0 0 1 2 -2 l6 0 a2 2 0 0 1 2 2 l0 8 l8 0 a2 2 0 0 1 2 2 l0 6 a2 2 0 0 1 -2 2 l-8 0 l0 8 a2 2 0 0 1 -2 2 l-6 0 a2 2 0 0 1 -2 -2 l0 -8 l-8 0 a2 2 0 0 1 -2 -2 l0 -6 a2 2 0 0 1 2 -2 l8 0 l0 -8z" /><path class="icon-maincolor" stroke="none" fill="black" d="M20 12 a2 2 0 0 1 2 -2 l6 0 a2 2 0 0 1 2 2 l0 8 l8 0 a2 2 0 0 1 2 2 l0 6 a2 2 0 0 1 -2 2 l-8 0 l0 8 a2 2 0 0 1 -2 2 l-6 0 a2 2 0 0 1 -2 -2 l0 -8 l-8 0 a2 2 0 0 1 -2 -2 l0 -6 a2 2 0 0 1 2 -2 l8 0 l0 -8z" /></svg>',
        hinticon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="50" viewBox="0 0 50 50" class="mini-icon ebook-icon ebook-icon-hint"><text fill="white" x="12" y="39" style="font-size: 45px; font-weight: bold; font-family: Lucida, \'Times New Roman\', serif;">?</text><text class="icon-maincolor" fill="black" x="13" y="40" style="font-size: 45px; font-weight: bold; font-family: Lucida, \'Times New Roman\', serif;">?</text></svg>',
        questionmark: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon ebook-icon-gear ebook-icon-settings"><path stroke="none" fill="white" d="M9.5 11 a7 7 0 1 1 10 5 a4.5 4.5 0 0 0 -3 4 l0 1 a0.5 0.5 0 0 1 -1 0 a12 12 0 0 1 2 -8 a6 6 0 0 0 -2 -8 a2.6 2.6 0 0 0 -3.6 3.5 a2 2 0 1 1 -2.4 1.5z m6.5 13 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z" /><path stroke="none" fill="black" d="M8.5 10 a7 7 0 1 1 10 5 a4.5 4.5 0 0 0 -3 4 l0 1 a0.5 0.5 0 0 1 -1 0 a12 12 0 0 1 2 -8 a6 6 0 0 0 -2 -8 a2.6 2.6 0 0 0 -3.6 3.5 a2 2 0 1 1 -2.4 1.5z m6.5 13 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z" /></svg>',
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 50 50" class="ebook-icon ebook-icon-book mini-icon mini-icon-book"><path stroke="none" d="M19 10 q2 -4 6 -4 l20 0 q5 0 3 5 l-14 20 q-3.5 5 -8 5 l-21 0 a4 6 0 0 0 0 10 l21 0 q4.5 0 8 -5 l11.2 -16 q3 -1 2 3 l-11.2 16 q-3.5 5 -8 5 l-24 0 a8 10 0 0 1 0 -16z" /></svg>'
    }
    
    //TODO: redudant code in action function. Refactor.
    ReaderView.menu = {
        toc: {
            name: 'showtoc',
            title: 'reader:TOC',
            icon: ReaderView.strings.icons.toc,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'tocisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showToc();
                    viewer.place.attr('data-opencontent', 'tocisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        langchooser: {
            name: 'langchooser',
            title: 'reader:language',
            icon: ReaderView.strings.icons.langchooser,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'langchooserisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showLangchooser();
                    viewer.place.attr('data-opencontent', 'langchooserisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        widgetthemechooser: {
            name: 'widgetthemechooser',
            title: 'reader:widget theme',
            icon: ReaderView.strings.icons.widgetthemechooser,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'widgetthemechooserisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showWidgetThemechooser();
                    viewer.place.attr('data-opencontent', 'widgetthemechooserisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        themechooser: {
            name: 'themechooser',
            title: 'reader:theme',
            icon: ReaderView.strings.icons.themechooser,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'themechooserisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showThemechooser();
                    viewer.place.attr('data-opencontent', 'themechooserisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        settings: {
            name: 'settings',
            title: 'reader:settingsmenu',
            icon: ReaderView.strings.icons.wrench,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'settingsisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showSettings();
                    viewer.place.attr('data-opencontent', 'settingsisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        about: {
            name: 'about',
            title: 'reader:aboutmenu',
            icon: ReaderView.strings.icons.infoicon,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'aboutisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showAbout();
                    viewer.place.attr('data-opencontent', 'aboutisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        help: {
            name: 'help',
            title:'reader:helpmenu',
            icon: ReaderView.strings.icons.questionmark,
            action: function(event){
                var viewer = event.data.viewer;
                if (viewer.place.attr('data-opencontent') === 'helpisopen') {
                    viewer.updatePage();
                } else {
                    viewer.showHelp();
                    viewer.place.attr('data-opencontent', 'helpisopen');
                    $(this).addClass('menuactive');
                }
            }
        },
        fullscreen: {
            name: 'fullscreen',
            title: 'reader:fullscreen',
            icon: ReaderView.strings.icons.fullscreen,
            action: function(event){
                var viewer = event.data.viewer;
                var viewarea = viewer.viewarea[0];
                if (viewarea.mozRequestFullScreen) {
                    viewarea.mozRequestFullScreen();
                } else if (viewarea.webkitRequestFullScreen) {
                    viewarea.webkitRequestFullScreen();
                } else if (viewarea.msRequestFullScreen) {
                    viewarea.msRequestFullScreen();
                }
            }
        }
    }
    
    

    /****************************************
     * Data for about menu (attributions, thanks to, etc...)
     *
     * All objects having the same type are represented in the same <fieldset> element.
     * Objects type is used as the localization key to fetch the correct localization for the <legend>,   (there's "readerview" prefix in the key i.e  'readerview:fontattribution')
     * if one is not found, the type property's value itself is used as the <legend>
     *
     */
    
    
    ReaderView.strings.aboutArray = [
        //FONTS
        {
            type: "fontattribution",
            title: "OpenDyslexic",
            titleLink: "http://opendyslexic.org/",
            author: "Abelardo Gonzalez",
            authorLink: "http://abbiecod.es/",
            license: "CC BY 3.0",
            licenseLink: "http://creativecommons.org/licenses/by/3.0/",
            source: "http://opendyslexic.org"
        },
        {
            type: "fontattribution",
            title: "Intuitive",
            titleLink: "http://openfontlibrary.org/en/font/intuitive",
            author: "Bruno de Souza Leão",
            authorLink: "https://www.flickr.com/photos/kylesatori",
            license: "OFL",
            licenseLink: "http://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL",
            source: "http://openfontlibrary.org/en/font/intuitive"
        },
        {
            type: "fontattribution",
            title: "Open Baskerville",
            titleLink: "http://klepas.org/openbaskerville/",
            author: "Isaac Moore, James Puckett, Rob Mientjes",
            authorLink: "http://klepas.org/openbaskerville",
            license: "OFL",
            licenseLink: "http://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL",
            source: "http://klepas.org/openbaskerville/"
        },
        
        //SOFTWARE
        {
            type: "softwareattribution",
            title: "jQuery JavaScript Library v1.10.2",
            titleLink: "http://jquery.com/",
            author: "The jQuery Foundation",
            authorLink: "https://jquery.org/",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "http://jquery.com/"
        },
        {
            type: "softwareattribution",
            title: "jQuery UI",
            titleLink: "http://jqueryui.com/",
            author: "The jQuery Foundation",
            authorLink: "https://jquery.org/",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "http://jqueryui.com/"
        },
        {
            type: "softwareattribution",
            title: "jQuery UI Touch Punch",
            titleLink: "http://touchpunch.furf.com/",
            author: "David Furfero",
            authorLink: "http://www.furf.com/",
            license: "MIT or GPL Version 2",
            licenseLink: "http://opensource.org/licenses/MIT",
            dualLicenseLink: "http://www.gnu.org/licenses/gpl-2.0.html",
            source: "http://touchpunch.furf.com/"
        },
        //{
        //    type: "softwareattribution",
        //    title: "Base64Binary converter",
        //    titleLink: "http://touchpunch.furf.com/",
        //    author: "Copyright (c) 2011, Daniel Guerrero. All rights reserved.",
        //    authorLink: "http://blog.danguer.com/",
        //    license: "BSD 2",
        //    licenseLink: "http://opensource.org/licenses/BSD-2-Clause",
        //    source: "https://github.com/danguer/blog-examples/blob/master/js/base64-binary.js"
        //},
        {
            type: "softwareattribution",
            title: "MathQuill",
            titleLink: "http://mathquill.com/",
            author: "Han and Jay",
            authorLink: "http://mathquill.com/",
            license: "Mozilla Public License, v. 2.0",
            licenseLink: "http://mozilla.org/MPL/2.0/",
            source: "https://github.com/mathquill/mathquill"
        },
        {
            type: "softwareattribution",
            title: "TiddlyWiki",
            titleLink: "http://tiddlywiki.com/",
            author: "JeremyRuston, UnaMesa Association",
            authorLink: "",
            license: "BSD 2",
            licenseLink: "http://opensource.org/licenses/BSD-2-Clause",
            source: "https://github.com/TiddlyWiki/tiddlywiki"
        },
        {
            type: "softwareattribution",
            title: "Modernizr",
            titleLink: "http://modernizr.com/",
            author: "Modernizr Team",
            authorLink: "http://modernizr.com/",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "https://github.com/Modernizr/Modernizr"
        },
        {
            type: "softwareattribution",
            title: "Hammer.Js",
            titleLink: "http://hammerjs.github.io/",
            author: "Jorik Tangelder, Eight Media",
            authorLink: "http://modernizr.com/",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "https://github.com/hammerjs/hammer.js"
        },
        {
            type: "softwareattribution",
            title: "JSXGraph",
            titleLink: "http://jsxgraph.uni-bayreuth.de/wp/",
            author: "Matthias Ehmann, Michael Gerhäuser, Carsten Miller, Bianca Valentin, Alfred Wassermann, Peter Wilfahrt",
            authorLink: "http://modernizr.com/",
            license: "MIT & LGPL",
            licenseLink: "http://opensource.org/licenses/MIT",
            dualLicenseLink: "http://www.gnu.org/licenses/licenses.html#LGPL",
            source: "https://github.com/hammerjs/hammer.js"
        },
        {
            type: "softwareattribution",
            title: "Marked - Markdown parser",
            titleLink: "https://github.com/chjj/marked",
            author: "Christopher Jeffrey",
            authorLink: "",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "https://github.com/chjj/marked"
        },
        /*{
            type: "graphicsattribution",
            title: "Delta - jQuery UI Theme",
            titleLink: "https://github.com/kiandra/Delta-jQuery-UI-Theme",
            author: "Kiandra",
            authorLink: "https://github.com/kiandra",
            license: "MIT/GPL",
            licenseLink: "http://opensource.org/licenses/MIT",
            dualLicenseLink: "http://www.gnu.org/copyleft/gpl.html",
            source: "https://github.com/chjj/marked"
        },
        {
            type: "graphicsattribution",
            title: "Sunny - jQuery UI Theme",
            titleLink: "http://jqueryui.com/themeroller/",
            author: "The jQuery Foundation",
            authorLink: "https://jqueryui.com",
            license: "MIT",
            licenseLink: "http://opensource.org/licenses/MIT",
            source: "http://jqueryui.com/themeroller/"
        }*/
        
    ]
    

    // Html-template for reader.
    ReaderView.strings.template = [
        '<div class="ebookreader-menuarea ffwidget-background"><div class="ebookreader-menu">'+ReaderView.strings.icons.menu+'</div><ul class="ebookreader-menulist closed ffwidget-background-inverse"></ul><div class="ebookreader-breadcrumb"></div></div>',
        '<div class="ebookreader-viewarea ffwidget-content">',
        '    <div class="ebookreader-pageframe" data-pageframe="prev">',
        '        <div class="ebookreader-pageheader"></div>',
        '        <div class="ebookreader-pageneck"></div>',
        '        <div class="ebookreader-pagebody"></div>',
        '    </div>',
        '    <div class="ebookreader-pageframe" data-pageframe="current">',
        '        <div class="ebookreader-pageheader"></div>',
        '        <div class="ebookreader-pageneck"></div>',
        '        <div class="ebookreader-pagebody"></div>',
        '    </div>',
        '    <div class="ebookreader-pageframe" data-pageframe="next">',
        '        <div class="ebookreader-pageheader"></div>',
        '        <div class="ebookreader-pageneck"></div>',
        '        <div class="ebookreader-pagebody"></div>',
        '    </div>',
        '    <div class="ebookreader-settingsframe">',
        '        <div class="ebookreader-pageheader"></div>',
        '        <div class="ebookreader-pageneck"></div>',
        '        <div class="ebookreader-pagebody"></div>',
        '    </div>',
        '</div>',
        '<div class="ebookreader-notearea ffwidget-background">',
        '    <div class="ebookreader-notetitle"><span class="titlearea"></span><span class="buttonarea"><span class="ebookreader-notetitle-button closenotes">'+ReaderView.strings.icons.closeicon+'</span></span></div>',
        '    <div class="ebookreader-notecontent"></div>',
        '</div>',
        '<div class="ebookreader-footerarea">',
        '    <div class="vendorlogo"></div>',
        '</div>',
        '<div class="ebookreader-naviarea"></div>'
    ].join('\n')
    
    // Css-styles for reader.
    ReaderView.strings.css = [
        'div.ebookreader-wrapper {display: -webkit-box; display: -webkit-flex; display: flex; -webkit-flex-wrap: nowrap; flex-wrap: nowrap; -webkit-flex-direction: column; flex-direction: column; -webkit-align-items: stretch; align-items: stretch; justify-content: space-between; -webkit-justify-content: space-between; overflow: hidden;}',
        '.ebookreader-wrapper .ebookreader-menuarea {-webkit-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; flex-shrink: 0; padding: 0; background-color: white; border: none; padding: 0.5em; -webkit-order: 1; order: 1;}',
        '.ebookreader-wrapper .ebookreader-viewarea {-webkit-flex-grow: 1; flex-grow: 1; background-color: white; border: none; -webkit-order: 3; order: 3;}',
        '.ebookreader-wrapper .ebookreader-naviarea {-webkit-flex-grow: 0; -webkit-flex-shring: 0; flex-grow: 0; flex-shrink: 0; -webkit-order: 2; order: 2;}',
        '.ebookreader-wrapper .ebookreader-notearea {-webkit-flex-grow: 0; -webkit-flex-shring: 0; flex-grow: 0; flex-shrink: 0; -webkit-order: 3; order: 3; max-height: 40%; padding: 0; height: 0;}',
        '.ebookreader-wrapper .ebookreader-footerarea {-webkit-flex-grow: 0; -webkit-flex-shring: 0; flex-grow: 0; flex-shrink: 0; -webkit-order: 4; order: 4; max-height: 200px; overflow: auto;}',
        'html.touch .ebookreader-wrapper .ebookreader-naviarea {-webkit-box-ordinal-group: 1; -moz-box-ordinal-group: 1; -ms-flex-order: 1; -webkit-order: 1; order: 1;  border-bottom: 1px solid black;}',
        'html.touch .ebookreader-wrapper .ebookreader-menuarea {-webkit-box-ordinal-group: 2; -moz-box-ordinal-group: 2; -ms-flex-order: 2; -webkit-order: 2; order: 2; border-bottom: none; border-top: 1px solid black; height: 2em;}',
        'html.touch .ebookreader-wrapper .ebookreader-viewarea {-webkit-box-ordinal-group: 3; -moz-box-ordinal-group: 3; -ms-flex-order: 3; -webkit-order: 3; order: 3; -webkit-overflow-scrolling-y: touch; -webkit-box-pack: start; justify-content: flex-start; -webkit-justify-content: flex-start;}',
        'html.touch .ebookreader-wrapper .ebookreader-naviarea .navbar-wrapper { border-radius: 0; border: none;}',
        '.ebookreader-menuarea .ebookreader-menulist li.ebookreader-menuitem svg {width: 20px; height: 20px;}',
        'body[data-ffwtheme] .ebookreader-menuarea .ebookreader-menulist li.ebookreader-widgetthemechooser {display: none;}',
        '.ebookreader-wrapper .ebookreader-menuarea .ebookreader-menulist li.ebookreader-fullscreen {display: none;}'
    ].join('\n')
    
    // Sample page for themeselector
    ReaderView.strings.samplepage = [
        '<div class="ebookreader-samplepage" data-readertheme="" data-pagetypes="section">',
        //////////'    <div class="ebookreader-pageheader"><h1>2.4 Sample title</h1></div>',
        '    <div class="ebookreader-pagebody">',
        '        <div class="ebook-elementbox" data-boxclass="textbox">{%readerthemes:plain text%}</div>',
        '        <div class="ebook-elementbox" data-boxclass="theorybox"><h1 class="ebook-elementbox-title">{%readerthemes:theory box%}</h1><p>{%readerthemes:theory text%}.</p></div>',
        '        <div class="ebook-elementbox" data-boxclass="textbox"><div data-ebookelement-type="markdownelement" data-element-name="elementid12" lang="fi"><p>{%readerthemes:plain text%}.</p></div><div data-ebookelement-type="imageelement" data-element-name="elementid13" lang="fi"><img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAGQAAABpCAYAAADMfIaKAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMegAADHoBW7Y6bwAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAW2SURBVHic7d35qxVlHMfx9+d6MfyhHyIDQwyvpmmZuZSWVhpBv5jtm2W2mm1KC/0RUVD9JJntG+25by22Z1ZatmkYQURFEIgZFMi3H57neI7j9Z5t5swzZ543iN7xmZmvvJg59xw85wrYB/xJLMuGAP+YWV+9hb3AXjMbmflIJU3SOcBy4L9G1vdkO0658xiPAvOBfxvZpzfTiUqcpNk4jDnA3kb3i1dIBnmMZcAcM/uxmX0jSMpJmkWLGBBBUs1jPAac3woGRJDUknQ2VYxdrR4ngqSQx1gOzG0HA+J3WW0n6SzgcdyVsbPd48UrpI08xhOkhAERpOVqMOamhQERpKUknUkV44c0jx1BmsxjPAlckDYGRJCmkjSTKsb3WZwjgjSYx3iKDDEggjSUpBk4jAuzxIAIUjeP8TQO47uszxefGA6Qx3iGDmFAvEIOm6QzqGJ826nzRpB+8hjP0mEMiCCHJOl0HMZFncaACHJQHuM5HMY3ecwQQXw1GBfnhQERBABJ06li7MhzltKDeIznCQADSg4iaRoO45IQMKDEIB7jBRzG13nPU6mUIJJOw2FcGhIGlBDEY7yIw/gq73mSlQqkBuOyEDGgRCCSTqWKsT3veQ5XKV7t9RgvETgGlOAKkTSVKsa2vOepV9Agco2o+bpX0rAm9p8KvExBMCDgW5Z/njAL2CnpEeBB4BhgjKRBZnZ/nf0rGJcXBQMCvUIkDQbmm9kDZrYSOBpYaGYrgOHA4Dr7T6GK8WXmA6fc72ZGSL+AicAw/2cBfwHzGtx3CrAbmJr3v6NmpmHA9kbWBnnLSjx7HgccBbxfbz9/ZbwCXGFmX2Q0XqYFectKNAPYbWa/woEH+kPmljQZh3FlkTAk9fjZgUBBJE2QdJL/cjawpeavFwNHJtZPBl7FYXzekSHbTNIkSQuArcDSyvYgQXAv/E2T1AeMAPYASDoeGGRmeyoLJU3CYVxVFAzfEcAGYFPtxlBB7sM9mE8GzgV2SFoETDOzhyqLPMZrOIytuUzaYma2xcz+SG4P9UF9Q2LT0uSaImMMVKhXyIBJOgWHMa+bMKCAIB7jdRzGZ3nPk3aFApE0EYdxdTdiQIFAPMYbOIwt9dYXtUKA1GBc080YUAAQSSdTxfg073myLshveyt5jDdxr/x2FYak4cAoYCYwVtJ5wC/BgiQwPsl7ngwaAxwLPAwY7gXUoUGCSJqAw7i2SzEws839bQ/uMcRjrMBhfJz3PJ0uKBD/Cu8KYEEZMSAgEI+xEofxUd7z5FUQIDUY15UZAwIAkXQiVYwP854n73IF8RirgOsjhis3kATGB3nNEVq5gEgaj7tN3RAxDq7jIB5jFXCjmdX9rz1lq6MgNRg3RYz+6xiIpHFUMd7r1HmLVkdAPMZq4OaIMXCZgyQwNmd9vqKXKYikE3AYCyNGY2UG4jHW4DDezeo83VYmIDUYt0SM5kodRNJYqhjvpH38bi9VEI+xFlgUMVorNRBJY6hivJ3WcctWKiAeYx1wa8Ror7ZBEhhvtT9SuWsLxL+BZi1wW8RIp5ZBPMY64HYz21RvfayxWgKpwbgjYqRb0yCSRlPF2Jj+SOWuKRCPsR64M2JkU8MgCYzkewBjKdUQiKRROIzFESPb6oJ4jA3AEjNbn/1I5W5AkJorY4mZrevMSOXusCA1GHdFjM7VL4j/SIsKxtrOjlTu+vtUnT7cY8bdEaPzHQSSwFiTz0jl7gCIpJE4jHsiRn71wAGMjTiM1XkOVPZ6gEE4jHsjRv4J2A/8Bvyd8yzd3CBgn5lNqrewF/cRrLHs29/IIvmPMY0FUu7vMWw2SSMlLZc0NO9ZsqgwV4ik2cB0YDSwEOgzs5/znCmLCgNSyX/W4ja6FKRwt6xuL4IEVgQJrAgSWBEksCJIYEWQwCoiyJDE711VYZ4Y+p/SuQAYDxwH/ATsApZZjj+QPu3+B74eL+J27JwZAAAAAElFTkSuQmCC" alt="Author: Pesasa, License: PD, Source: E-Math" title="Author: Pesasa, License: PD, Source: E-Math"></div></div>',
        '        <div data-ebookelement-type="elementbox" data-modified="Thu Feb 06 2014 10:08:00 GMT+0200 (EET)" data-modifier="pesasa" data-created="Thu Feb 06 2014 10:06:00 GMT+0200 (EET)" data-creator="pesasa" data-tags="tagi1 tagi2" class="ebook-elementbox" data-boxclass="examplebox" data-element-name="elementbox14" lang="fi"><h1 class="ebook-elementbox-title">{%readerthemes:example box%}</h1><div data-ebookelement-type="markdownelement" data-element-name="elementid15" lang="fi"><p>{%readerthemes:example text%}.</p></div></div>',
        '        <div class="ebook-elementbox" data-boxclass="textbox"></div>',
        '    </div>',
        '</div>'
    ].join('\n');
    
    // Available themes
    ReaderView.themes = {
        default: {
            name: {
                en: 'Default',
                fi: 'Oletus'
            },
            id: 'default',
            description: {
                en: 'Default theme',
                fi: 'Oletusteema'
            }
        },
        blanco: {
            name: {
                en: 'Blanco',
            },
            id: 'blanco',
            description: {
                en: 'Very simple theme',
                fi: 'Hyvin yksinkertainen teema'
            }
        },
        notes: {
            name: {
                en: 'Notes',
                fi: 'Muistilaput'
            },
            id: 'notes',
            description: {
                en: 'Theme looking like note taking',
                fi: 'Muistilappuja muistuttava teema'
            }
        },
        rosy: {
            name: {
                en: 'Rosy',
                fi: 'Ruusu'
            },
            id: 'rosy',
            description: {
                en: 'Pink rosy theme',
                fi: 'Vaaleanpunainen ruusuteema'
            }
        },
        night: {
            name: {
                en: 'Night',
                fi: 'Yö'
            },
            id: 'night',
            description: {
                en: 'Dark theme for reading in dark.',
                fi: 'Tumma teema pimeässä lukemiseen.'
            }
        },
        /*sporty: {
            name: {
                en: 'Sporty',
                fi: 'Sportti'
            },
            id: 'sporty',
            description: {
                en: 'Theme with sports',
                fi: 'Urheilullinen teema'
            }
        }*/
    }
    
    // Available widget themes
    ReaderView.widgetthemes = [
        {
            name: {
                en: 'Default',
                fi: 'Oletus'
            },
            id: '',
            description: {
                en: 'Default grayish theme',
                fi: 'Harmaa oletusteema'
            }
        },
        {
            name: {
                en: 'Default (red)',
                fi: 'Oletus (punainen)'
            },
            id: 'default-red',
            description: {
                en: 'Default theme with red highlights',
                fi: 'Oletusteema punaisilla korostuksilla'
            }
        },
        {
            name: {
                en: 'Default (green)',
                fi: 'Oletus (vihreä)'
            },
            id: 'default-green',
            description: {
                en: 'Default theme with green highlights',
                fi: 'Oletusteema vihreillä korostuksilla'
            }
        },
        {
            name: {
                en: 'Default (blue)',
                fi: 'Oletus (sininen)'
            },
            id: 'default-blue',
            description: {
                en: 'Default theme with blue highlights',
                fi: 'Oletusteema sinisillä korostuksilla'
            }
        },
        {
            name: {
                en: 'Flat (blue)',
                fi: 'Litteä (sininen)'
            },
            id: 'flat-blue',
            description: {
                en: 'Flat blue theme',
                fi: 'Litteä teema sinisellä värillä'
            }
        }
    ]
    
    { /*** jQuery plugin ***/
        $.fn.readerview = function(options){
            if (methods[options]){
                return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
            } else if (typeof(options) === 'object' || !options) {
                return methods.init.apply(this, arguments);
            } else {
                $.error( 'Method ' +  options + ' does not exist on readerview' );
                return this;
            }
        }
        
        var methods = {
            init: function(params){
                
                //params = $.extend(true, {
                //}, params);
                var readerview = new ReaderView(this, params);
                readerview.init();
            },
            getdata: function(params){
                var $place = $(this);
                $place.trigger('getdata');
                var data = $place.data('[[readerviewdata]]');
                return data;
            },
            geticon: function() {
                return ReaderView.strings.icons.appicon;
            },
            gettitle: function() {
                var result = {title: ''};
                $(this).trigger('gettitle', [result]);
                return result.title;
            }
        }
    }
})(jQuery);

//
///******
// * System connector.
// ******/
//function Reader(place, config){
//    //this.preRequisites = ['ContentView'];
//    this.preRequisites = ['EbookApp'];
//    this.type = 'Readerview';
//    this.version = 0.1;
//    this.currentPage = null;
//    this.pageHistory = [];
//    this.place = place;
//    
//    // Run initialize-method from superclass.
//    this.initialize(place, config);
//}
//    
//if (typeof(EbookApp) !== 'undefined' && typeof(Reader.inheritsFrom) !== 'undefined' ) {
//    //Reader.inheritsFrom(ContentView);
//    Reader.inheritsFrom(EbookApp);
//}
//
//// initialize-method from superclass runs init(), which is specific for this view.
////Reader.prototype.init = function(){
////    var reader = this;
////    var system = this.place.parents('.system').data('system');
////    var store = new Store({fs: system.fs, contentPrefix: this.config.bookpath + this.config.bookid + '/'});
////    this.book = new Book(store);
////    var params = {
////        lang: this.config.lang,
////        initCallback: function(){
////            reader.show();
////        }
////    };
////    this.book.initialize(params);
////}
//
//Reader.prototype.init = function(){
//    var reader = this;
//    //var system = this.place.parents('.system').data('system');
//    var fs = new OfflineFS({asyncLoad: false});
//    var store = new Store({fs: fs, contentPrefix: 'data/books/' + this.config.bookid + '/'});
//    this.book = new Book(store);
//    var params = {
//        lang: this.config.lang,
//    };
//    this.book.initialize(params);
//    this.show();
//}
//
///******
// * Render the reader
// ******/
//Reader.prototype.show = function(){
//    if (!this.config.lang || !this.book.isAvailableLang(this.config.lang)) {
//        this.config.lang = this.book.getDefaultLang();
//    }
//    this.place.readerview({book: this.book, config: this.config});
//}
//
///******
// * Redraw the content
// ******/
//Reader.prototype.redraw = function(){
//    this.place.trigger('navigationmoved');
//}
//
///******
// * Get data of the reader
// ******/
//Reader.prototype.getData = function(){
//    return this.place.readerview('getdata');
//}
//
////Reader.prototype.register = function(){
////    return null;
////}
////
////Reader.prototype.draw = function(){
////    return true;
////}
//
///******
// * Register Reader as an EbookApp
// ******/
//if (typeof(EbookApp) === 'function') {
//    EbookApp.register('Reader', Reader);
//}
