/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
        typeof(jQuery.fn.emathquiz) === 'undefined' && jQuery.fn.emathquiz.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true,{
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                
            },
            settings: {
                username: 'Anonymous',
                uilang: 'en',
                lang: 'en'
            }
        }, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime;
        };
    };
    
    /**
     * Add score to userdata
     * @param {Object} score   the score of the quiz
     * @param {Number} score.correct     Number of correct answers
     * @param {Number} score.total       Total number of questions
     */
    Userdata.prototype.addScore = function(score) {
        this.data.results = score.results.map(function(a){return a|0});
        this.data.correct = score.correct;
        this.data.answered = score.answered;
        this.data.total = score.total;
    };
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };
    
    /**
     * Get the current answer (Clicked object)
     * @return {Object}  the clicked-object
     */
    Userdata.prototype.getAnswer = function() {
        return JSON.parse(JSON.stringify(this.data.clicked));
    };
    
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    };

    Userdata.prototype.getNewId = function(){
        var idparts = ['emathquizanswer'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    var Quiz = function(place, options){
        this.place = $(place);
        var openQuiz = options['data']['openQuiz'].split('##');
        this.openMode = openQuiz[0];
        this.openTitle = openQuiz[1];
        this.callFunctions = options['data']['callFunction'];
        this.neededContent = [];
        for( var oneFunction in this.callFunctions){
            this.neededContent.push(this.callFunctions[oneFunction].split("##")[0]);
        }
        this.overlaymode = (this.openMode === 'modal');
        this.total = options.total;
        this.quizFuncs = [];
        this.init(options);
        this.setStyles();
    }
    Quiz.prototype.init = function(options){
        this.initData(options);
        this.solvable = this.settings.users && this.settings.users.canEdit && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.initHandlers();
        this.showContent();
    }

    Quiz.prototype.initData = function(options){
        options = jQuery.extend(true, {
            type: "elementset",
            metadata: {
                creator: "",
                created: new Date(),
                modifier: "",
                modified: new Date(),
                tags: []
            },
            userdata: {
                type: 'userdata',
                metadata: {
                    creator: '',
                    created: '',
                    modifier: '',
                    modified: '',
                    tags: []
                },
                data: {
                },
                settings: {
                    username: 'Anonymous',
                    uilang: 'en',
                    lang: 'common'
                }
            },
            data: {
                contents: [],
                
            }
        }, options);
        this.type = options.type;
        this.metadata = options.metadata;
        this.data = options.data;
        this.settings = options.settings;
        this.contents = options.data && options.data.contents || [];
        this.contentdata = options.data && options.data.contentdata || {};
        this.contentPlaces = this.contentPlaces || {};
        this.place.addClass('ebook-' + escapeHTML(this.type));
        this.place.attr('data-tags', escapeHTML(this.metadata.tags.join(' ')))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .addClass('quizquestion');
        // By default elementsets use language in the path of element ids.
        this.uselangpath = true;
        var userdata = options.userdata;
        userdata.settings = {username: options.settings.username, lang: options.settings.lang || ''}
        this.userdata = new Userdata(userdata);
        this.assignmentid = this.place.attr('data-element-name');
    }
    
    /******
     * Set style-tag if needed
     ******/
    Quiz.prototype.setStyles = function(){
        if ($('head style#quizelement-style').length === 0) {
            $('head').append('<style id="quizelement-style" type="text/css">'+Quiz.styles+'</style>');
        }
    }
    
    Quiz.prototype.initHandlers = function(){
        var assignment = this; var element = this;

        this.place.off('emathquiz_changed').on('emathquiz_changed', function(event, score){
            element.userdata.addScore(score);
            element.place.attr('data-quiztotal', escapeHTML(score.total));
            var refid = element.assignmentid;
            var userdata = element.userdata.getData();
            var name = userdata.name;
            var senddata = {};
            senddata[name] = userdata;
            element.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
            element.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, true]);
            element.place.trigger('sendcontentdata', senddata);
        });
        this.place.off('reply_get_bookelement').on('reply_get_bookelement', function(event, data){
            var parseData = {};
            for( var oneData in data){
                var splitted = oneData.split("/");
                if(splitted.length>1){
                    parseData[splitted[1]] = data[oneData]['data']['quizzJavascript'];
                }else{
                    parseData[splitted[0]] = data[oneData]['data']['quizzJavascript'];
                }
            }
            for( var oneFunction in assignment.callFunctions){
                var quizF = assignment.callFunctions[oneFunction].split("##");
                assignment.quizFuncs.push(parseData[quizF[0]][quizF[1]]);
            }
            assignment.drawQuiz();
        });
        this.place.off('emathquiz_changed').on('emathquiz_changed', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            assignment.userdata.addScore(data);
            assignment.userdata.solved(data.correct === data.total);
            assignment.userdata.closed(data.answered === data.total);
            assignment.saveUserData();
        });
        this.place.off('close_quizquestion').on('close_quizquestion', function(event, data) {
            event.stopPropagation();
            assignment.userdata.closed(true);
            var localsave = !!data.localsave;
            assignment.saveUserData({localsave: localsave, sendforward: false});
        });
        this.place.off('clearanswer').on('clearanswer', function(event, data) {
            event.stopPropagation();
            assignment.clear();
        })
        //this.place.off('reply_get_userdata').on('reply_get_userdata', function(event, data) {
        //    event.stopPropagation();
        //    event.preventDefault();
        //    data.settings = {
        //        username: assignment.settings.username,
        //        lang: assignment.settings.lang || '',
        //        uilang: assignment.settings.uilang,
        //        role: assignment.settings.role
        //    };
        //    assignment.userdata = new Userdata(data);
        //    assignment.drawQuiz();
        //});
        this.place.off('reply_getcontentbyref').on('reply_getcontentbyref', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var refid = assignment.assignmentid;
            var reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: assignment.settings.username,
                    lang: assignment.settings.lang || '',
                    uilang: assignment.settings.uilang,
                    role: assignment.settings.role
                };
                assignment.userdata = new Userdata(userdata);
                assignment.drawQuiz();
            };
        });
    }
    Quiz.prototype.showContent = function(){
        this.contentPlaces = {};
        this.place.empty();
        var contentlist = [];
        var elemblock;
        var currcontainer = this.place;
        for (var i = 0; i < this.contents.length; i++){
            var content = this.contents[i].id;
            var element = jQuery('<div>');
            element.attr('data-element-name', escapeHTML(content));
            this.contentPlaces[content] = element;
            currcontainer.append(element);
            if (this.contentdata[content]) {
                // If the data of element is embedded in the elementset.
                var cdata = this.contentdata[content];
                var ctype = cdata.type;
                element.attr('data-ebookelement-type', escapeHTML(ctype));
                // Use jQuery-plugin named as variable ctype.
                try {
                    element[ctype](cdata);
                } catch (e){
                    console.log(e, ctype);
                }
            } else {
                // If the data of element is separate from elementset.
                contentlist.push(content);
            }
        }
        this.emquiz = $('<div class="quiz-emquiz-wrapper"></div>');
        this.place.append(this.emquiz);
        this.place.trigger('get_bookelement', {contentlist: this.neededContent, uselangpath: this.uselangpath, lang: this.metadata.lang});
        this.place.trigger('get_userdata', {refid: this.assignmentid});
        this.place.trigger('getcontentbyref', {refs: [this.assignmentid]});
    }

    Quiz.prototype.drawQuiz = function(){
        if (!this.userdata.isClosed()) {
            var quizdata = {
                title: this.openTitle,
                func: this.quizFuncs,
                overlay: this.overlaymode,
                score: this.userdata.data.correct || 0,
                answered: this.userdata.data.answered || 0,
                total: this.total
            };
            this.emquiz.empty().off();
            this.emquiz.emathquiz(quizdata);
        } else {
            this.emquiz.trigger('disable');
        }
    }
    
    Quiz.prototype.clear = function() {
        var quizdata = {
            title: this.openTitle,
            func: this.quizFuncs,
            overlay: this.overlaymode,
            score: 0,
            answered: 0,
            total: this.total
        };
        this.emquiz.empty().off();
        this.emquiz.emathquiz(quizdata);
    }
    
    Quiz.prototype.saveUserData = function(options) {
        if (this.solvable) {
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            var localsave = options && !!options.localsave;
            if (options && options.sendforward) {
                var senddata = {};
                senddata[name] = userdata;
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    Quiz.prototype.playSounds = function(id){
        if(jQuery('html').hasClass('ipad')){
            if(typeof(this.audios[id])==="undefined"){
                this.audios[id]={};
                var audioelement = this.audios[id];
                if(typeof(window.myAudioContext)==="undefined"){
                    window.myAudioContext = new webkitAudioContext();
                }
                this.audios[id]['arrayBuff'] = Base64Binary.decodeArrayBuffer(this.sounds[id.replace("#","")]);
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                context.decodeAudioData(this.audios[id]['arrayBuff'], function(audioData) {
                    element.buffer = audioData;
                    audioelement['buffer'] = audioData;
                    element.connect(context.destination);
                    
                    element.noteOn(0);
                });
            }else{
                if(typeof(window.myAudioContext)==="undefined"){
                    window.myAudioContext = new webkitAudioContext();
                }
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                element.buffer = this.audios[id]['buffer'];
                element.connect(context.destination);
                element.noteOn(0);
            }
        }else{
            jQuery(id)[0].play();
        }
    }
    Quiz.prototype.addSounds = function(){
        if(jQuery("#soundEffects").length ===0){
            var effectsPlace = jQuery('body').append('<div id="soundEffects"/>').find('#soundEffects');
        }else{
            var effectsPlace = jQuery("#soundEffects");
        }
        for(var sound in this.sounds){
            if(jQuery("#"+sound).length ===0){
                jQuery('#soundEffects').append('<audio id="'+escapeHTML(sound)+'" autobuffer="autobuffer" ><source src="data:audio/mpeg;base64,'+this.sounds[sound]+'"></source></audio>');
            }
        }
        
    }
    Quiz.prototype.sounds ={};
    
    Quiz.styles = [
        '.quizquestion[data-quizset-question][data-ebookelement-type="quiz"]::before {display: none;}'
    ];
    
    var quizinfo = {
        type: 'quiz',
        elementtype: 'elements',
        jquery: 'quiz',
        name: 'emquiz',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-geoimage"><circle style="stroke: black; fill: none; stroke-width: 1;" cx="15" cy="15" r="10" /></svg>',
        description: {
            en: 'emquiz',
            fi: 'emquiz'
        },
        classes: ['viewonly']
    };
    
    // Register geoimage as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', quizinfo);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', quizinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var Quizassignment = new Quiz(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[Quizdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.quiz = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in quiz.');
            return false;
        }
    }
    
})(jQuery, ebooklocalizer);