/******
@name PageElement
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class PageElement
@description A class and jQuery-plugin for pageelement on notebook. A few possible pagetemplates
    with different kinds of content and layout: coverpage, contentpage, table of contents, table of pictures,
    authorlist, etc.

TODO:
*******/


/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - ElementSet
 * - ElementPanel
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(ebooklocalizer) === 'undefined' && ebooklocalizer.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var element = new PageElement(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.pageelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in %cpageelement%c.', 'color: red; background: #fdd;');
            return false;
        }
    }

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /******
     * PageElement class
     * @class GenContent
     * @constructor
     * @param {jQuery} place - Place for the element
     * @param {Object} options - options for the element
     ******/

    var PageElement = function(place, options) {
        this.place = $(place);
        this.init(options);
        this.show();
        this.setAttrs();
        this.setStyles();
    };
    
    /******
     * Init the element
     * @param {Object} options - initing data
     ******/
    PageElement.prototype.init = function(options) {
        var setlang = this.place.closest('[lang]').attr('lang') || 'common';
        var lang = 'common';
        options = $.extend(true, {}, this.defaults, {metadata: {lang: lang}, settings: {lang: setlang}}, options);
        this.settings = options.settings;
        this.type = 'pageelement';
        this.settings.gotolink.pageelementid = this.settings.gotolink.pageelementid || this.place.attr('data-element-name') || options.name;
        this.name = options.name || this.place.attr('data-element-name');
        this.data = options.data;
        this.pagetype = this.data.pagetype || 'contentpage';
        this.templatedata = {};
        this.templatedata[this.pagetype] = this.data;
        this.metadata = $.extend(true, {
            creator: this.settings.username,
            created: (new Date()).getTime(),
            modifier: this.settings.username,
            modified: (new Date()).getTime(),
            tags: []
        }, options.metadata);
        this.setRole(this.settings.role);
        this.setMode(this.settings.mode);
        this.initHandlers();
    };
    
    /******
     * Remove all event handlers
     ******/
    PageElement.prototype.removeHandlers = function() {
        this.place.off();
    };
    
    /******
     * Init event handlers
     ******/
    PageElement.prototype.initHandlers = function() {
        this.removeHandlers();
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersView();
        };
    };
    
    /******
     * Init handlers common for all modes
     ******/
    PageElement.prototype.initHandlersCommon = function() {
        var element = this;
        this.place.off('getdata').on('getdata', function(event, options){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        }).off('setdata').on('setdata', function(event, data) {
            element.init(data);
            element.show();
        }).off('setmode').on('setmode', function(event, mode) {
            if (event.target === this) {
                event.stopPropagation();
                element.changeMode(mode);
            };
        }).off('setrole').on('setrole', function(event, role) {
            if (event.target === this) {
                event.stopPropagation();
                element.setRole(role);
                element.show();
            };
        }).off('refreshfull').on('refreshfull', function(event) {
            event.stopPropagation();
            if (event.target === this) {
                element.contentplace && element.contentplace.trigger('refreshfull');
            };
        }).off('refresh').on('refresh', function(event) {
            event.stopPropagation();
            if (event.target === this) {
                element.setAttrs();
                element.contentplace && element.contentplace.trigger('refresh');
            };
        }).off('gotolink').on('gotolink', function(event, data){
            var link = data.data && data.data.link;
            if (link && link.pageelementid && link.pageelementid === element.place.attr('data-element-name')) {
                event.stopPropagation();
                element.goToLink(data);
            };
        }).off('getelements').on('getelements', function(event, data) {
            event.stopPropagation();
            if (typeof(data) === 'object') {
                data.elements = element.getElements(!!data.rename);
            };
        });
    };
    
    /******
     * Init handlers for editable modes
     ******/
    PageElement.prototype.initHandlersEdit = function() {
        var element = this;
        this.place.off('element_changed').on('element_changed', function(event, data) {
            if (event.target !== this && event.target === element.contentplace.get(0)) {
                // From children and we are not supposed to let the data pass through.
                event.stopPropagation();
                event.preventDefault();
                var elname = $(event.target).attr('data-element-name');
                element.contentplace.trigger('getdata');
                var eldata = element.contentplace.data('[[elementdata]]');
                element.name = elname;
                element.data = eldata.data;
                element.data.pagetype = element.pagetype;
                element.metadata = eldata.metadata;
                element.templatedata[element.pagetype] = element.data;
                element.changed();
            };
        });
    };
    
    /******
     * Init handlers for non-editable modes
     ******/
    PageElement.prototype.initHandlersView = function() {
        
    }
    
    /******
     * Change the pagetype that is in use.
     * @param {String} pagetype - the name of template to use
     ******/
    PageElement.prototype.setPagetype = function(pagetype) {
        this.pagetype = pagetype;
        if (this.templatedata[pagetype]) {
            this.data = this.templatedata[pagetype];
        } else {
            this.data = $.extend(true, {}, this.templates[pagetype]);
            this.templatedata[pagetype] = this.data;
        };
        this.show();
    };
    
    /******
     * Show the element
     ******/
    PageElement.prototype.show = function() {
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    };
    
    /******
     * Show the element in view mode
     ******/
    PageElement.prototype.view = function() {
        var jq = this.templates[this.pagetype] && this.templates[this.pagetype].jq;
        if (!this.contentplace) {
            var canmodify = (this.metadata.lang !== 'common' || (this.settings.lang === this.settings.defaultlang) ?  true : false);
            this.contentplace = $('<div class="pageelement-content" data-canmodify="'+canmodify+'"></div>');
            this.place.html(this.contentplace);
        };
        if (jq && typeof($.fn[jq]) === 'function') {
            this.contentplace.empty();
            var options = {
                type: 'elementset',
                name: this.name,
                metadata: JSON.parse(JSON.stringify(this.metadata)),
                data: JSON.parse(JSON.stringify(this.data)),
                settings: $.extend(true, {}, this.settings)
            };
            //if (this.settings.savemode === 'semilocal') {
            //    options.settings.savemode = 'local';
            //}
            this.contentplace[jq](options);
        };
    };
    
    /******
     * Show the element in edit mode
     ******/
    PageElement.prototype.edit = function() {
        this.view();
    };
    
    /******
     * Element has been changed
     ******/
    PageElement.prototype.changed = function() {
        this.metadata.modified = (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        if (this.settings.savemode !== 'local') {
            this.metadata.lang = 'common';
        };
        this.place.trigger('element_changed', [{type: this.type, lang: this.metadata.lang}]);
    };
    
    /******
     * Get the data of the page
     ******/
    PageElement.prototype.getData = function() {
        var result = {
            type: 'pageelement',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        };
        return result;
    };
    
    /**
     * Get an array of all elements on this page, including the page itself.
     * @param {Boolean} rename     Rename all elements.
     * @returns {Array} an array of elements
     */
    PageElement.prototype.getElements = function(rename) {
        var getelems = {elements: [], rename: rename};
        this.contentplace.trigger('getelements', getelems);
        var last = getelems.elements[getelems.elements.length - 1];
        if (last.name === this.name || !last.name) {
            last.type = 'pageelement';
        };
        return getelems.elements;
    };
    
    
    /******
     * Set some attributes in html
     ******/
    PageElement.prototype.setAttrs = function() {
        this.place.addClass('ebook-elementset ebook-pageelement');
        this.place.attr('data-tags', escapeHTML(this.metadata.tags.join(', ')))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified));
    };
    
    /******
     * Set role of the user
     * @param {Object} role - the role of the user
     ******/
    PageElement.prototype.setRole = function(role) {
        var roledata = this.roles[role]
        if (typeof(roledata) === 'undefined') {
            role = this.settings.role || 'student';
            roledata = this.roles[role];
        };
        if (this.rolechangeable || typeof(this.rolechangeable) === 'undefined') {
            this.rolechangeable = roledata.rolechangeable;
            this.settings.role = role;
            this.teacherable = roledata.teacherable;
            this.studentable = roledata.studentable;
        };
    };
    
    /******
     * Set mode
     * @param {String} mode - the new mode
     ******/
    PageElement.prototype.setMode = function(mode) {
        var rights = this.modes[mode];
        if (typeof(rights) === 'undefined') {
            mode = this.settings.mode || 'view';
            rights = this.modes[mode];
        };
        this.settings.mode = mode;
        this.editable = rights.editable;
        this.authorable = rights.authorable;
        this.reviewable = rights.reviewable;
    };
    
    /******
     * Change mode (set mode and re-show the page)
     * @param {String} mode - the new mode
     ******/
    PageElement.prototype.changeMode = function(mode) {
        this.setMode(mode);
        this.initHandlers();
        this.show();
    };
    
    /******
     * Go to the link target
     * @param {Object} linkdata - link data to the target
     ******/
    PageElement.prototype.goToLink = function(linkdata){
        var link = linkdata && linkdata.data && linkdata.data.link;
        delete link.chapid;
        delete link.pageelementid;
        var nextstop = link.elementsetpath.shift();
        if (nextstop === this.name) {
            nextstop = link.elementsetpath.shift();
        };
        if (nextstop) {
            var element = this.place.find('.elementset-element[data-element-name="'+nextstop+'"]');
            if (element.length > 0) {
                element.get(0).scrollIntoView({behavior: 'smooth', block: 'start'});
                if (link.elementsetpath.length === 0 || nextstop === link.elementid) {
                    this.highlightElement(element, 2000);
                };
                element.trigger('gotolink', linkdata);
            };
        } else if (link.elementid) {
            var element = this.place.find('.elementset-element[data-element-name="'+link.elementid+'"]');
            if (element.length > 0) {
                element.get(0).scrollIntoView({behavior: 'smooth', block: 'start'});
                this.highlightElement(element, 2000);
                element.trigger('gotolink', linkdata);
            };
        };
    };

    /******
     * Highlight element for a small time
     * @param {jQuery} element - element to highlight
     * @param {Number} time - the time in ms for highlighting
     ******/
    PageElement.prototype.highlightElement = function(element, time){
        $(element).addClass('elementset-elementhighlight');
        if (time) {
            setTimeout(function(){$(element).removeClass('elementset-elementhighlight')}, time);
        };
    };

    /******
     * Set some stylesheets
     ******/
    PageElement.prototype.setStyles = function() {
        if ($('head style#pageelement-style').length === 0) {
            $('head').append('<style id="pageelement-style" type="text/css">'+this.styles+'</style>')
        };
    };
    
    PageElement.prototype.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    };
    
    PageElement.prototype.roles = {
        student: {
            studentable: true,
            teacherable: false,
            rolechangeable: false
        },
        teacher: {
            studentable: false,
            teacherable: true,
            rolechangeable: true
        }
    };
    
    PageElement.prototype.defaults = {
        type: 'pageelement',
        metadata: {},
        data: {
            template: 'contentpage',
            contents: [],
            contentdata: {}
        },
        settings: {
            lang: 'common',
            uilang: 'en',
            mode: 'view',
            role: 'student',
            username: 'Anonymous',
            savemode: 'local',    // 'local' | 'semilocal' | 'global'
            gotolink: {
                chapid: '',
                pageelementid: ''
            }
        }
    };
    
    PageElement.prototype.styles = [
        '.ebook-pageelement {min-width: 20em; padding-bottom: 1.5em;}',
        '.ebook-pageelement {position: relative;}',
        '.ebook-pageelement[data-elementmode="edit"] > .pageelement-content, .ebook-pageelement[data-elementmode="author"] > .pageelement-content {border: none;}',
        '.ebook-pageelement .pageelement-content {margin-left: 1em; margin-right: 1em;}',
        ''
    ].join('\n');
    
    PageElement.prototype.templates = {
        contentpage: {
            pagetype: 'contentpage',
            jq: 'elementset',
            label: {
                en: 'Content page',
                fi: 'Sisältösivu',
                sv: 'Innehållssida'
            },
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="100" height="100" viewBox="0 0 30 30" class="template-icon template-icon-contentpage"><path style="stroke: none; fill: rgb(200, 200, 200);" d="M4 2 h12 v1 h-12z m0 2 h12 v1 h-12z m3 2 h6 v6 h-6z m-3 7 h12 v1 h-12z m0 2 h12 v1 h-12z m0 2 h12 v1 h-12z"></path></svg>',
            data: {
                pagetype: 'contentpage',
                contents: [],
                contentdata: {}
            }
        },
        coverpage: {
            pagetype: 'coverpage',
            jq: 'coverpage',
            label: {
                en: 'Coverpage',
                fi: 'Kansisivu',
                sv: 'Omslag'
            },
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="100" height="100" viewBox="0 0 30 30" class="template-icon template-icon-coverpage"><path style="stroke: none; fill: rgb(200, 200, 200);" d="M6 4 a4 2 0 0 0 8 0 a4 2 0 0 0 -8 0z m-1 4 h10 v8 h-10z m1 1 v5 l2 -2 l2 1 l2 -3 l2 2 v-3z m-1 10 h3 v1 h-1 v4 h-1 v-4 h-1z m3 0 h1 l1 2 l1 -2 h1 l-1 2.5 l1 2.5 h-1 l-1 -2 l-1 2 h-1 l1 -2.5z m4 0 h3 v1 h-1 v4 h-1 v-4 h-1z"></path></svg>',
            data: {
                pagetype: 'coverpage',
                contents: [
                    {
                        type: 'vendorLogo'
                    },
                    {
                        type: 'imageelement',
                        title: '',
                        contentid: ''
                    },
                    {
                        type: 'titletext',
                        title: ''
                    }
                ]
            }
        },
        toc: {
            pagetype: 'toc',
            jq: 'tocelement',
            label: {
                en: 'Table of Contents',
                fi: 'Sisällysluettelo',
                sv: 'Innehållsförteckning'
            },
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="100" height="100" viewBox="0 0 30 30" class="template-icon template-icon-toc"><path style="stroke: none; fill: rgb(170, 170, 170);" d="M3 6 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h14 v2 h-14z m0 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h10 v2 h-10z m-4 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h10 v2 h-10z m-8 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h14 v2 h-14z m0 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h10 v2 h-10z m-4 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h10 v2 h-10z m-8 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h14 v2 h-14z m0 4 a1 1 0 0 0 2 0 a1 1 0 0 0 -2 0z m4 -1 h10 v2 h-10z"></path></svg>',
            contents: [
                {
                    type: 'tocelement',
                    startpage: ''
                }
            ]
        },
        authorlist: {
            pagetype: 'authorlist',
            label: {
                en: 'Authorlist',
                fi: 'Tekijäluettelo',
                sv: 'Författare lista'
            },
            content: [
                {
                    type: 'authorlist'
                }
            ]
        },
        picturelist: {
            pagetype: 'picturelist',
            label: {
                en: 'List of pictures',
                fi: 'Kuvaluettelo',
                sv: 'Bildlista'
            },
            content: [
                {
                    type: 'picturelist',
                    liststyle: 'list'    // 'list' | 'grid'
                }
            ]
        },
        copyrightlist: {
            pagetype: 'copyrightlist',
            label: {
                en: 'Credits / Copyrights',
                fi: 'Tekijänoikeudet',
                sv: 'Upphovsrätt'
            },
            content: [
                {
                    type: 'copyrightlist'
                }
            ]
        }
    };

    
    PageElement.elementinfo = {
        type: 'pageelement',
        elementtype: ['pagecontainers'],
        jquery: 'pageelement',
        name: 'PageElement',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 5 l0 18 l22 0 l0 -18z m3 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z" /></svg>',
        description: {
            en: 'Page for normal content.',
            fi: 'Sivu tavalliselle sisällölle.',
            sv: 'Standardsida'
        },
        roles: [],
        classes: ['pagecontainer'],
        cancontain: ['containers', 'elements', 'assignments', 'assignmentsets', 'refs']
    }
    
    $.fn.elementset('addelementtype', PageElement.elementinfo);

    
})(window, jQuery, window.ebooklocalizer);