/**
@name Notebook
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Notebook
@description A class and jQuery-plugin for an editable notebook with different types of pages.

TODO:
*/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - ElementSet
 * - EbookLocalizer
 * - Elpp
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(ebooklocalizer) === 'undefined' && ebooklocalizer.apply;
        typeof(Elpp) === 'undefined' && Elpp.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var notebook = new NotebookView(this, params);
                window.nnn = notebook;
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[notebookdata]]');
            return data;
        },
        'geticon': function() {
            return NotebookView.icons.notebook;
        },
        'gettitle': function() {
            var result = {title: ''};
            $(this).trigger('gettitle', [result]);
            return result.title;
        }
    }
    
    $.fn.notebookview = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in notebookview.');
            return false;
        }
    }
    
    /**
     * Helper functions
     */
    
    /**
     * Sanitize the input text
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /**
     * Html-escaping for sanitizing
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    /**
     * NotebookView class
     * @class Notebook
     * @constructor
     * @param {jQuery} place - Place for notebook
     * @param {Object} options - options for the notebook
     */
    var NotebookView = function(place, options, refdata){
        this.place = $(place);
        this.readytosend = false;
        this.place.empty();
        this.settings = $.extend(true, {}, NotebookView.defaults.settings, options.settings || {});
        if (this.settings.hasparent) {
            this.selfconfigable = false;
            if (this.settings.configable) {
                this.configable = true;
            } else {
                this.configable = false;
            }
        } else {
            this.selfconfigable = true;
            this.configable = true;
        };
        var updatedata = $.extend(true, [], options.updatedata);
        options.settings = JSON.parse(JSON.stringify(this.settings));
        this.notebook = new Notebook(options, refdata);
        this.settings.gotolink.appname = this.notebook.getBookId();
        this.configs = $.extend(true, {}, NotebookView.defaults.configs, options.configs, this.loadConfigs());
        this.filter = 'all';
        this.pagecontainers = $.fn.elementset('getelementtypes', 'pagecontainers');
        this.dragtypestack = [];
        this.epanelcounter = 0;
        this.changelog = new NbChangelog();
        this.setStyles();
        this.setReadonly(this.settings.readonly);
        this.setMode(this.settings.mode);
        this.setRole(this.settings.role, true);
        this.setLang(this.settings.lang);
        this.createMenu();
        this.mergeUpdates(updatedata);
        this.currentPage = this.configs.currentpage || this.notebook.getFirstPage();
        this.initHandlers();
        this.requestUpdates(true);
        this.readytosend = true;
        this.show();
        if (!this.selfconfigable) {
            this.registerConfForm();
        };
        this.setAttrs();
        this.setThemes();
        this.registerShareables();
        this.appReady();
    };
    
    /**
     * Set the readonly mode
     * @param {Boolean} readonly - true/false
     */
    NotebookView.prototype.setReadonly = function(readonly){
        this.readonly = readonly;
    };
    
    /**
     * Set the mode of NotebookView (view, edit, author,...)
     * @param {String} mode - the name of the mode
     */
    NotebookView.prototype.setMode = function(mode) {
        var uilang = this.settings.uilang;
        var waseditable = this.editable;
        if (typeof(NotebookView.modes[mode]) === 'undefined') {
            mode = 'view';
        };
        this.settings.mode = mode;
        this.editable = NotebookView.modes[mode].editable;
        if (this.pageview) {
            this.pageview.trigger('setmode', mode);
        };
        this.updateTitle(this.getPageTitle(), this.getPageNumber());
        if (this.editable) {
            this.place.trigger('edit_started', [{apptype: 'notebook', appid: this.getBookId()}]);
            let modmenu = jQuery.extend(true, {}, NotebookView.pagemodmenu);
            for (let i = 0, len = modmenu.menu.length; i < len; i++) {
                modmenu.menu[i].label = ebooklocalizer.localize(modmenu.menu[i].label, uilang);
            };
            this.place.find('.notebook-viewarea > .notebook-pagemodarea').addClass('notebook-pagemodable').menubar(modmenu);
        } else if (waseditable) {
            this.place.find('.notebook-viewarea > .notebook-pagemodarea').removeClass('notebook-pagemodable').empty();
            this.place.trigger('edit_stopped', [{apptype: 'notebook', appid: this.getBookId()}]);
            this.setNoteview();
            this.setNoteedit();
        };
    };
    
    /**
     * Set the role of the user
     * @param {String} role - the role of the user
     * @param {Boolean} firstset - True, if we are initing notebook, not  changing role.
     */
    NotebookView.prototype.setRole = function(role, firstset){
        if (firstset && typeof(NotebookView.roles[this.settings.role]) === 'undefined' ) {
            // If this is initing setRole() and the role is invalid, use 'student' role.
            role = this.settings.role = 'student';
        }
        var roledata = NotebookView.roles[role];
        var roleindex = NotebookView.roles[this.settings.role].roleindex;
        if (roledata && (firstset || (roleindex > roledata.roleindex))) {
            // If the new role is valid and (initing role or changing to lesser role),
            // set the capabilities according to the role.
            // The real role stays in this.settings.role, the effective role is in this.role
            this.role = role;
            this.teacherable = roledata.teacherable;
            this.studentable = roledata.studentable;
            this.createMenu();
        };
    };
    
    /**
     * Set configs of this notebook (themes etc.)
     * @param {Object} data - key - value pairs.
     */
    NotebookView.prototype.setConfigs = function(data){
        data = data || {};
        this.configs = $.extend(this.configs, data);
        this.setAttrs();
        if (typeof(data.nbtheme) === 'string') {
            this.setThemes();
        };
        if (data.usenumbering) {
            this.updatePage();
        };
        if (typeof(data.zoomlevel) === 'string' || typeof(data.zoomlevel) === 'number') {
            this.configs.zoomlevel = (this.configs.zoomlevel | 0) || 10; // Convert to number, but default to 10 in case weird happens.
            this.place.find('.notebook-viewarea').attr('data-zoomlevel', this.configs.zoomlevel);
        };
        this.saveConfigs();
    };
    
    /**
     * Set the language of the notebook
     * @param {String} lang - the language
     * @param {Boolean} refresh - is refreshing of the view needed
     */
    NotebookView.prototype.setLang = function(lang, refresh) {
        lang = this.notebook.setLang(lang);
        this.settings.lang = lang;
        this.place.attr('lang', lang);
        if (this.settings.mode !== 'view') {
            this.editmodeOff();
        };
        if (refresh) {
            this.show();
        };
    };
    
    /**
     * Register the config form to parent application, if one exists.
     */
    NotebookView.prototype.registerConfForm = function(){
        var formdata = {apptype: 'notebook', appid: this.getBookId(), confform: this.getConfigForm()};
        this.place.trigger('registerconfigdialog', formdata);
    };
    
    /**
     * Register the the context things that can be shared
     */
    NotebookView.prototype.registerShareables = function() {
        var uilang = this.settings.uilang;
        this.place.trigger('register_shareables', {
            appid: this.getBookId(),
            apptype: 'notebook',
            shareables: [
                {
                    appid: this.getBookId(),
                    apptype: 'notebook',
                    name: 'pageshare',
                    icon: NotebookView.icons.sharepage,
                    title: this.getBookTitle(),
                    label: ebooklocalizer.localize('notebook:sharethepage', uilang),
                    event: 'sharethepageonline'
                }
            ]
        })
    };
    
    /**
     * Return back to the original role
     */
    NotebookView.prototype.returnOwnRole = function(){
        var roledata = NotebookView.roles[this.settings.role];
        if (roledata) {
            this.role = this.settings.role;
            this.teacherable = roledata.teacherable;
            this.studentable = roledata.studentable;
            this.createMenu();
        };
    };
    
    /**
     * Show the Notebook viewer.
     */
    NotebookView.prototype.show = function(){
        var template = NotebookView.templates.html;
        if (this.settings.onepagemode) {
            template = NotebookView.templates.onepage;
        }
        this.place.html(template);
        this.updateElementPanel();
        this.updateNavbar();
        this.updateEditbutton();
        this.updateMenu();
        this.updatePage();
    };
    
    /**
     * Start editing mode
     */
    NotebookView.prototype.editmodeOn = function() {
        this.setMode('edit');
        this.place.attr('data-dragtype', 'element studentelement studentcontainer assignment quizelement');
        // Trigger for elementpanel
        this.place.trigger('startediting');
        // Trigger for toc
        this.place.find('.notebook-toc').trigger('starteditingtoc')
    };
    
    /**
     * Stop editing mode
     */
    NotebookView.prototype.editmodeOff = function() {
        this.setMode('view');
        this.saveData();
        this.place.attr('data-dragtype', '');
        // Trigger for elementpanel
        this.place.trigger('stopediting');
        // Trigger for toc
        this.place.find('.notebook-toc').trigger('stopeditingtoc');
        // Trigger saving the content
        this.place.trigger('savecontent', this.readonly);
        this.place.trigger('sendcontent');
    };
    
    /**
     * Prepare all dirty stuff for saving
     */
    NotebookView.prototype.flushDirty = function() {
        this.saveData();
        // Trigger saving the content
        this.place.trigger('savecontent', this.readonly);
        this.place.trigger('sendcontent');
    };
    
    /**
     * Create the menudata for menubar.
     */
    NotebookView.prototype.createMenu = function(){
        var uilang = this.settings.uilang;
        
        // Drop down menu
        var mainmenu = {
            name: 'mainmenu',
            type: 'toggle',
            icon: NotebookView.icons.menu,
            label: 'Menu',
            event: 'mainmenuopen',
            event_off: 'mainmenuoff',
            menu: []
        };
        
        if (this.settings.devmode) {
            mainmenu.menu.unshift(NotebookView.filemenu.menu[0]);
        };
        
        if (this.configable) {
            mainmenu.menu.unshift({
                name: 'settingsdialog',
                type: 'click',
                icon: NotebookView.icons.settings,
                label: 'Settings',
                event: 'configdialogopen'
            });
        };
        this.dmenudata = {
            menustyle: 'dropmenu',
            ffbacground: false,
            menu: []
        };
        if (mainmenu.menu.length > 0) {
            this.dmenudata.menu.push(mainmenu);
        }
        this.dmenudata.halign = 'right';
    };
    
    /**
     * Update the menubar (jQuery-plugin) with current menudata.
     */
    NotebookView.prototype.updateMenu = function(){
        var uilang = this.settings.uilang;
        //this.place.find('.notebook-menuarea').menubar(this.menudata);
        // Toc
        var tocarea = this.place.find('.notebook-tocarea');
        tocarea.menubar({
            "menustyle": "dropmenu",
            "ffbackground": false,
            "menu": [
                {
                    "name": "toc",
                    "type": "click",
                    "icon": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="36" height="36" viewBox="-3 -3 36 36" class="mini-icon mini-icon-toc"><path class="mini-icon-extrapath" style="stroke: none;" opacity="0" d="M15 -2 a17 17 0 0 1 0 34 a17 17 0 0 1 0 -34z m0 2 a 15 15 0 0 0 0 30 a15 15 0 0 0 0 -30z"></path><path style="stroke: none;" d="M8 6 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m0 7 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m0 7 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m5 -14 l10 0 a2 2 0 0 1 0 4 l-10 0 a2 2 0 0 1 0 -4z m0 7 l10 0 a2 2 0 0 1 0 4 l-10 0 a2 2 0 0 1 0 -4z m0 7 l10 0 a2 2 0 0 1 0 4 l-10 0 a2 2 0 0 1 0 -4z"></path></svg>',
                    "label": ebooklocalizer.localize('notebook:toc', uilang),
                    "event": "opentoc"
                }
            ]
        });
        // Dropmenu
        var dropmenu = this.place.find('.notebook-dropmenuarea');
        if (this.dmenudata.menu.length > 0) {
            dropmenu.menubar(this.dmenudata);
        };
    };
    
    /**
     * Update the edittoggle (jQuery-plugin) with current data.
     */
    NotebookView.prototype.updateEditbutton = function(){
        var uilang = this.settings.uilang;
        var buttonarea = this.place.find('.notebook-edittogglearea');
        if (!this.readonly && !buttonarea.is('.togglebutton-wrapper') && this.settings.users.canEdit({ctype: 'notebookcontent'})) {
            buttonarea.togglebutton({
                "name": "edittoggle",
                "ffbackground": false,
                "showlabels": false,
                "size": 30,
                "states": [
                    {
                        "name": "view",  // We are in view state. Show button for editing.
                        "event": "editmodeoff",  // On entering this state.
                        "title": ebooklocalizer.localize('notebook:edit', uilang),
                        "icon": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>'
                    },
                    {
                        "name": "edit", // We are in edit state. Show button for viewing.
                        "event": "editmodeon",  // On entering this state.
                        "title": ebooklocalizer.localize('notebook:view', uilang),
                        "icon": '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-view"><path style="stroke: none;" d="M3 15 a13 13 0 0 0 24 0 a13 13 0 0 0 -24 0z m1 0 a13 13 0 0 1 22 0 a13 13 0 0 1 -22 0z m6 0 a5 5 0 0 0 10 0 a5 5 0 0 0 -2.2 -4 l-2.8 4 l0 -5 a5 5 0 0 0 -5 5z"></path></svg>'
                    }
                ]
            });
        }
    };
    
    /**
     * Update the navbar (jQuery-plugin) with the data filtered with current filter.
     */
    NotebookView.prototype.updateNavbar = function(){
        var navbar = this.place.find('.notebook-naviarea');
        var navdata = this.getNavdata();
        navdata.current = this.currentPage || navdata.current;
        navbar.navbar(navdata);
    };
    
    /**
     * Update the elementpanel, if it is needed.
     */
    NotebookView.prototype.updateElementPanel = function(){
        if (this.settings.elementpanel && typeof($.fn.elementpanel) === 'function') {
            var settings = $.extend(true, {}, this.settings);
            var config = $.extend(true, {}, this.configs.elementpanel, {minimized: true});
            this.place.children('.notebook-elementpanelarea').elementpanel({settings: settings, config: config});
        };
    };
    
    /**
     * Update the page with the content of the current page.
     */
    NotebookView.prototype.updatePage = function(newpage){
        var lang;
        if (this.settings.savemode !== 'local') {
            lang = 'common';
        } else {
            lang = this.settings.lang;
        };
        for (var i = 0; i < this.epanelcounter; i++ ) {
            this.place.trigger('stopediting');
        };
        if (this.readytosend) {
            this.place.trigger('savecontent', this.readonly);
            this.place.trigger('sendcontent');
        };
        this.epanelcounter = 0;
        this.place.find('.notebook-viewarea :focus').blur();
        if (newpage) {
            this.currentPage = newpage;
        };
        if (!this.currentPage) {
            this.currentPage = this.configs.currentpage || this.notebook.getFirstPage();
        };
        if (!this.notebook.pageExists(this.currentPage)) {
            this.currentPage = this.notebook.getFirstPage();
        };
        var viewarea = this.place.find('.notebook-viewarea');
        var level = this.getPageLevel();
        var pagetypes = this.getPageTypes();
        viewarea.attr('data-pagelevel', level);
        viewarea.attr('data-pagetypes', pagetypes.join(' '));
        viewarea.attr('data-zoomlevel', this.configs.zoomlevel || 10);
        viewarea.scrollTop(0);
        if (this.bookSeriesSet !== this.notebook.getBookId()) {
            this.updateBookSeries();
        };
        var data = this.notebook.getPageData(this.currentPage, lang);
        var ellang = data.metadata && data.metadata.lang || '';
        var contentid = this.notebook.getPageId(this.currentPage);
        var title = this.notebook.getPageTitle(this.currentPage);
        var pagenumber = this.getPageNumber(this.currentPage);
        this.updateTitle(title, pagenumber);
        this.pageview = this.place.find('.notebook-pageview');
        this.pageview.attr('data-element-name', contentid)
            .attr('data-chapid', this.currentPage)
            .attr('lang', escapeHTML(ellang));
        if (ellang && ellang !== this.settings.lang && ellang !== 'common') {
            this.pageview.addClass('nottranslated');
        } else {
            this.pageview.removeClass('nottranslated');
        };
        data.settings = {
            uilang: this.settings.uilang,
            lang: this.settings.lang,
            defaultlang: this.notebook.getDefaultlang(),
            username: this.settings.username,
            mode: this.settings.mode,
            role: this.role,
            users: this.settings.users,
            savemode: this.notebook.getSavemode() || this.settings.savemode || 'local',
            gotolink: JSON.parse(JSON.stringify(this.settings.gotolink)),
            usemargin: true,
            userinfo: this.settings.userinfo || {}
        };
        data.settings.gotolink.chapid = this.currentPage;
        data.settings.gotolink.pageelementid = contentid;
        var jq = this.pagecontainers[data.type] && this.pagecontainers[data.type].jquery || 'pageelement';
        this.place.trigger('pageredraw', {apptype: 'notebook', appid: this.getBookId()});
        try {
            this.pageview[jq](data);
        } catch (err) {
            console.log(data,err);
        };
        // Set marginnote viewing and editing as it was on previous page.
        this.setNoteview();
        this.setNoteedit();
    };
    
    /**
     * Show and update the title of the page
     * @param {String} title - The title of the page
     */
    NotebookView.prototype.updateTitle = function(title, pagenumber) {
        var uilang = this.settings.uilang;
        pagenumber = pagenumber || '';
        var pagetitle = this.place.find('.notebook-pagetitle');
        var infoicon = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-info"><path style="stroke: none;" d="M15 4 a2.5 2.5 0 0 0 0 5 a2.5 2.5 0 0 0 0 -5z m3 7 l-7 1 l0 2 a3 2 0 0 1 3 2 l0 6 a3 2 0 0 1 -3 2 l0 2 10 0 l0 -2 a3 2 0 0 1 -3 -2z"></path></svg>';
        var copyicon = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-copy"><path style="stroke: none;" d="M4 2 l16 0 l0 3 l-11 0 l0 18 l-5 0z m7 5 l8 0 l0 8 l8 0 l0 11 l-16 0z m10 0 l6 6 l-6 0z"></path></svg>';
        var linkicon = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z"></path></svg>';
        var metadata, headsource;
        if (this.editable) {
            var pagemods = (this.isDefaultLang() ? '<div class="notebookview-pagemods"></div>' : '');
            // Show title in editable mode. With or without MathQuill.
            if (false && typeof($.fn.mathquill) === 'function') {
                title = escapeHTML(title).replace(/\\\((.*?)\\\)/g, '\$ $1\$');
                pagetitle.html('<h1><span class="notebookview-titlenumber">'+pagenumber+'</span> <span class="notebookview-titletext notebookview-mathtitle">'+title+'</span></h1>' + pagemods);
                pagetitle.find('.notebookview-titletext').mathquill('textbox');
            } else {
                pagetitle.html('<h1><span class="notebookview-titlenumber">'+pagenumber+'</span> <input type="text" class="notebookview-titletext" value="'+escapeHTML(title)+'" /></h1>' + pagemods);
            };
        } else {
            var headimage = this.getHeadimage();
            if (headimage.type && headimage.data.type === 'imageelement') {
                pagetitle.attr('data-headimagetype', headimage.type);
                pagetitle.css({
                    'background-image': 'url('+headimage.data.data.content+')',
                    'background-repeat': 'no-repeat'
                });
                metadata = [
                    ebooklocalizer.localize('notebook:author', uilang) + ': ' + escapeHTML(headimage.data.data.author || ''),
                    ebooklocalizer.localize('notebook:license', uilang) + ': ' + escapeHTML(headimage.data.data.license || ''),
                    ebooklocalizer.localize('notebook:source', uilang) + ': ' + escapeHTML(headimage.data.data.source || ''),
                    ebooklocalizer.localize('notebook:notes', uilang) + ': ' + escapeHTML(headimage.data.data.notes || '')
                ];
                headsource = escapeHTML(headimage.data.data.source || '');
            } else {
                pagetitle.removeAttr('style').removeAttr('data-headimagetype');
                metadata = false;
                headsource = '';
            };
            if (typeof($.fn.mathquill) === 'function') {
                title = escapeHTML(title).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>');
                pagetitle.html(`
                    <h1>
                        <span class="notebookview-titlenumber">${pagenumber}</span>
                        <span class="notebookview-titletext notebookview-mathtitle">${title}</span>
                        <div class="notebookview-iconbar">
                            <div class="notebookview-pagecopy" draggable="true">${copyicon}</div>
                            <div class="notebookview-pagelink" draggable="true">${linkicon}</div>
                        </div>
                    </h1>
                    ${metadata ? '<a target="_blank" href="'+ headsource +'" title="'+metadata.join('\n')+'" class="notebook-headimage-metainfo">'+infoicon+'</a>' : ''}`
                );
                pagetitle.find('.mathquill-embedded-latex').mathquill();
            } else {
                pagetitle.html(`
                    <h1>
                        <span class="notebookview-titlenumber">${pagenumber}</span>
                        <span class="notebookview-title">${escapeHTML(title)}</span>
                        <div class="notebookview-iconbar">
                            <div class="notebookview-pagecopy" draggable="true">${copyicon}</div>
                            <div class="notebookview-pagelink" draggable="true">${linkicon}</div>
                        </div>
                    </h1>
                    ${metadata ? '<a target="_blank" href="'+ headsource +'" title="'+metadata.join('\n')+'" class="notebook-headimage-metainfo">'+infoicon+'</a>' : ''}`
                );
            };
        };
    };
    
    /**
     * Update bookseries logo
     */
    NotebookView.prototype.updateBookSeries = function() {
        var logoarea = this.place.find('.notebook-viewarea .notebook-bookserieslogo');
        var serie = this.notebook.getBookSerie(this.settings.lang)
        var logo = serie.logo || '';
        if (serie.name === 'eMath' && !logo) {
            logo = NotebookView.emathLogo;
        };
        logo = logo.replace('{{booknumber}}', serie.number).replace('{{booknumbercolor}}', serie.color);
        logoarea.html(logo);
        this.bookSeriesSet = this.notebook.getBookId();
    };
    
    /**
     * Zoom viewarea closer (plus).
     */
    NotebookView.prototype.zoomPlus = function() {
        var zoomlevel = this.configs.zoomlevel;
        zoomlevel = Math.min(zoomlevel + 1, 15);
        this.setConfigs({zoomlevel: zoomlevel});
    };
    
    /**
     * Zoom viewarea further (minus).
     */
    NotebookView.prototype.zoomMinus = function() {
        var zoomlevel = this.configs.zoomlevel;
        zoomlevel = Math.max(zoomlevel - 1, 5);
        this.setConfigs({zoomlevel: zoomlevel});
    };
    
    /**
     * Zoom viewarea reset.
     */
    NotebookView.prototype.zoomReset = function() {
        this.setConfigs({zoomlevel: 10});
    };
    
    /**
     * TOC has changed: update navbar
     */
    NotebookView.prototype.tocChanged = function(){
        this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
        this.updateNavbar();
    };
    
    /**
     * Titles have changed
     */
    NotebookView.prototype.titlesChanged = function() {
        this.changelog.addDirty({
            name: this.getBookId() + '_pagetitles',
            type: 'pagetitles',
            timestamp: (new Date()).getTime(),
            lang: this.settings.lang
        });
    };
    
    /**
     * Index has changed
     */
    NotebookView.prototype.indexChanged = function(lang) {
        lang = lang || 'common';
        this.changelog.addDirty({name: this.getBookId() + '_index', type: 'index', timestamp: (new Date()).getTime(), lang: lang});
    };
    
    /**
     * Init the event handlers.
     */
    NotebookView.prototype.initHandlers = function(){
        var notebook = this;
        // Prevent default action (pageload), if some file is dropped on the notebook.
        this.place.off('drop.notebook dragover.notebook').on('drop.notebook dragover.notebook', function(event){
            event.stopPropagation();
            event.preventDefault();
        });
        // Scroll page up, when dragging over "updragscroller".
        this.place.off('dragover', '.notebook-updragscroller').on('dragover', '.notebook-updragscroller', function(event){
            event.stopPropagation();
            event.preventDefault();
            notebook.place.trigger('scrollup');
        });
        // Scroll page down, when dragging over "downdragscroller".
        this.place.off('dragover', '.notebook-downdragscroller').on('dragover', '.notebook-downdragscroller', function(event){
            event.stopPropagation();
            event.preventDefault();
            notebook.place.trigger('scrolldown');
        });
        this.place.off('scrollup').on('scrollup', function(event){
            event.stopPropagation();
            event.preventDefault();
            var viewarea = notebook.place.find('.notebook-viewarea');
            var scrollOffset = viewarea.scrollTop();
            scrollOffset = scrollOffset - 10;
            viewarea.scrollTop(scrollOffset);
        });
        this.place.off('scrolldown').on('scrolldown', function(event){
            event.stopPropagation();
            event.preventDefault();
            var viewarea = notebook.place.find('.notebook-viewarea');
            var scrollOffset = viewarea.scrollTop();
            scrollOffset = scrollOffset + 10;
            viewarea.scrollTop(scrollOffset);
        });
        // The page was changed with navbar.
        this.place.off('navigationmoved').on('navigationmoved', function(event, data){
            event.stopPropagation();
            notebook.updatePage(data.current);
            notebook.goToPage(data.current, true);
        });
        // Open config dialog
        this.place.off('configdialogopen').on('configdialogopen', function(event, data){
            if (notebook.selfconfigable) {
                event.stopPropagation();
                notebook.openConfigForm();
            };
        });
        this.place.off('editmodeon').on('editmodeon', function(event, data) {
            event.stopPropagation();
            notebook.editmodeOn();
        }).off('editmodeoff').on('editmodeoff', function(event, data) {
            event.stopPropagation();
            notebook.editmodeOff();
        }).off('flushdirty').on('flushdirty', function(event, data) {
            event.stopPropagation();
            notebook.flushDirty();
        });
        this.place.off('dragstart.pagelink', '.notebook-pagetitle .notebookview-pagelink').on('dragstart.pagelink', '.notebook-pagetitle .notebookview-pagelink', function(event, data){
            event.stopPropagation();
            var linkData = notebook.getPageLink();
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text/plain', JSON.stringify(linkData, null, 4));
            ev.dataTransfer.effectAllowed = 'copyMove';
        });
        this.place.off('dragstart.pagecopy', '.notebook-pagetitle .notebookview-pagecopy').on('dragstart.pagecopy', '.notebook-pagetitle .notebookview-pagecopy', function(event, data){
            event.stopPropagation();
            var pageData = notebook.getPageCopy();
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text/plain', JSON.stringify(pageData, null, 4));
            ev.dataTransfer.effectAllowed = 'copyMove';
        });
        this.place.off('dragover.pagecopy').on('dragover.pagecopy', function(event, data) {
            // Prevent default to allow drop!
            event.preventDefault();
        });
        this.place.off('drop.pagecopy').on('drop.pagecopy', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            if (notebook.editable) {
                var ev = event.originalEvent;
                var datastr = ev.dataTransfer.getData('text');
                var dropdata = {};
                try {
                    dropdata = JSON.parse(datastr);
                } catch (err) {
                    dropdata = {type: 'error', data: err};
                };
                if (dropdata.type && dropdata.type === 'dragelementpage') {
                    notebook.addPageCopy(dropdata.data);
                } else if (dropdata.type && dropdata.type === 'error') {
                    console.log(dropdata.err);
                };
            };
        });
        // Elementpanelevents for elementpanel
        this.place.off('elementpanelevent').on('elementpanelevent', function(event, data){
            event.stopPropagation();
            switch (data.eventname){
                case 'paletaddelem':
                    var focused = notebook.currentFocus;
                    var elset, pagewrap;
                    pagewrap = focused && focused.closest('.pageelement-content') || $();
                    if (focused && typeof(focused.length) === 'number' && pagewrap.length > 0 && focused.length > 0 && focused.is(':visible')) {
                        var droptarget, wrapper, elset, parentwrapper;
                        wrapper = focused.closest('.elementset-elementwrapper');
                        elset = focused.closest('.ebook-elementset');
                        parentwrapper = elset.closest('.elementset-elementwrapper');
                        // Find the correct droptarget to use.
                        if (focused.is('.elementset-droptarget')) {
                            // Focus was on droptarget
                            droptarget = focused;
                        } else {
                            if (wrapper.length > 0 && wrapper.get(0) !== parentwrapper.get(0)) {
                                // Focus was in some element inside this elementset
                                droptarget = wrapper.next('.elementset-droptarget');
                            } else {
                                // There was focus, but on something else than droptarget or element. Maybe the title.
                                droptarget = elset.find('> .ebook-elementset-body > .elementset-droptarget').last();
                            };
                        }
                        var index = droptarget.attr('data-dropindex') | 0;
                        elset.trigger('addelement', {index: index, data: data.data});
                    } else {
                        // There was no focus. Add to the end of the page.
                        elset = notebook.pageview.children('.pageelement-content');
                        elset.trigger('addelement', {data: data.data});
                    }
                    break;
                default:
                    break;
            }
        });
        // Menuevents from the menubar
        this.place.off('menubarevent').on('menubarevent', function(event, data){
            event.stopPropagation();
            switch (data.eventname){
                // Start editing
                case 'editmodeon':
                    notebook.editmodeOn();
                    break;
                // Stop Editing
                case 'editmodeoff':
                    notebook.editmodeOff();
                    break;
                // Containers can be dragged to the page
                case 'assignmentpaleton':
                    notebook.dragtypestack.push(notebook.place.attr('data-dragtype'));
                    notebook.place.attr('data-dragtype', 'assignment quizelement');
                    break;
                // Containers can not be dragged to the page
                case 'assignmentpaletoff':
                    var dragtype = notebook.dragtypestack.pop();
                    notebook.place.attr('data-dragtype', dragtype);
                    break;
                // Containers can be dragged to the page
                case 'containerpaleton':
                    notebook.dragtypestack.push(notebook.place.attr('data-dragtype'));
                    notebook.place.attr('data-dragtype', 'container');
                    break;
                // Containers can not be dragged to the page
                case 'containerpaletoff':
                    var dragtype = notebook.dragtypestack.pop();
                    notebook.place.attr('data-dragtype', dragtype);
                    break;
                // Tabcontainers can be dragged to the page
                case 'tabcontainerpaleton':
                    notebook.dragtypestack.push(notebook.place.attr('data-dragtype'));
                    notebook.place.attr('data-dragtype', 'tabcontainer');
                    break;
                // Tabcontainers can not be dragged to the page
                case 'tabcontainerpaletoff':
                    var dragtype = notebook.dragtypestack.pop();
                    notebook.place.attr('data-dragtype', dragtype);
                    break;
                // Add page after the current page.
                case 'addpageafter':
                    var newchapid = notebook.addPageAfter();
                    //notebook.currentPage = newchapid;
                    //notebook.show();
                    //notebook.updateNavbar();
                    //notebook.updatePage();
                    break;
                // Add chapter after the current page.
                case 'addchapterafter':
                    var newchapid = notebook.addChapterAfter();
                    //notebook.currentPage = newchapid;
                    //notebook.show();
                    //notebook.updateNavbar();
                    //notebook.updatePage();
                    break;
                // Remove current page.
                case 'removepage':
                    notebook.removePage();
                    break;
                case 'addbeforepages':
                    notebook.addBeforePages();
                    break;
                case 'addafterpages':
                    notebook.addAfterPages();
                    break;
                // Export notebook (save in a JSON-file)
                case 'exportnotebook':
                    var json = notebook.getJSON(true);
                    var blob = new Blob([json], {type: 'application/json'});
                    var url = URL.createObjectURL(blob);
                    var filename = 'notebook.json';
                    notebook.place.trigger('exportfileas', {filename: filename, objecturl: url});
                    break;
                case 'importnotebook':
                    notebook.place.trigger('importfile');
                    break;
                case 'paletaddelem':
                    var focused = notebook.currentFocus;
                    var elset;
                    if (focused.length > 0) {
                        var wrapper = focused.closest('.elementset-elementwrapper');
                        var elset = focused.closest('.ebook-elementset');
                        var droptarget;
                        if (wrapper.length > 0) {
                            droptarget = wrapper.next('.elementset-droptarget');
                        } else {
                            droptarget = elset.find('> .ebook-elementset-body > .elementset-droptarget').last();
                        };
                        var index = droptarget.attr('data-dropindex') | 0;
                        elset.trigger('addelement', {index: index, data: data.data});
                    } else {
                        elset = notebook.pageview;
                        elset.trigger('addelement', {data: data.data});
                    }
                    break;
                case 'configdialogopen':
                    notebook.place.trigger('configdialogopen');
                    break;
                case 'opentoc':
                    notebook.place.trigger('opentoc');
                    break;
                default:
                    notebook.place.trigger(data.eventname, [data.data]);
                    break;
            };
        });
        // Zooming events with wheel
        this.place.off('wheel.zoom', '.notebook-viewarea').on('wheel.zoom', '.notebook-viewarea', function(event) {
            if (event.ctrlKey && !event.altKey) {
                if (event.originalEvent.wheelDeltaY > 0) {
                    notebook.zoomPlus();
                    event.stopPropagation();
                } else if (event.originalEvent.wheelDeltaY < 0) {
                    notebook.zoomMinus();
                    event.stopPropagation();
                };
            };
        });
        // Zooming events with keyboard (ctrl+, ctrl-, ctrl0)
        $(window).off('keydown.nbzoom').on('keydown.nbzoom', function(event) {
            if (event.ctrlKey && !event.altKey) {
                if (event.keyCode === 187) {
                    notebook.zoomPlus();
                    event.stopPropagation();
                } else if (event.keyCode === 189) {
                    notebook.zoomMinus();
                    event.stopPropagation();
                } else if (event.keyCode === 48) {
                    notebook.zoomReset();
                    event.stopPropagation();
                };
            };
        });
        // Get the currently focused content element when moving mouse over elementpanel
        this.place.off('mouseenter', '.elementpanel-elementicon').on('mouseenter', '.elementpanel-elementicon', function(event){
            notebook.currentFocus = notebook.place.find(':focus');
        });
        // Get the currently focused content element when it is focused out
        this.place.off('focusout.notebookcurrentfocus').on('focusout.notebookcurrentfocus', function(event){
            notebook.currentFocus = $(event.target);
        });
        // Set content language
        this.place.off('setlang').on('setlang', function(event, lang){
            if (typeof(lang) === 'string' && lang.length === 2) {
                event.stopPropagation();
                notebook.setLang(lang, true);
            };
        });
        // Set readonly-mode ON
        this.place.off('readonly_on').on('readonly_on', function(event, lang){
            event.stopPropagation();
            if (!notebook.editable) {
                notebook.setReadonly(true);
                notebook.createMenu();
                notebook.show();
            };
        });
        // Set readonly-mode OFF
        this.place.off('readonly_off').on('readonly_off', function(event, lang){
            event.stopPropagation();
            if (!notebook.editable) {
                notebook.setReadonly(false);
                notebook.createMenu();
                notebook.show();
            };
        });
        // Get data
        this.place.off('getdata').on('getdata', function(event){
            event.stopPropagation();
            var data = notebook.getData();
            notebook.place.data('[[notebookdata]]', data);
        });
        // Get data
        this.place.off('getconfigs').on('getconfigs', function(event){
            event.stopPropagation();
            var confs = notebook.getConfigs();
            notebook.place.data('[[applicationconfigs]]', confs);
        });
        // Handle the saving of the changed elements.
        this.place.off('element_changed').on('element_changed', function(event, data){
            event.stopPropagation();
            if (notebook.editable) {
                var eltype = data.type;
                var lang = data.lang;
                var place = $(event.target);
                var elname = place.attr('data-element-name');
                place.trigger('getdata');
                var eldata = place.data('[[elementdata]]');
                console.log('elementdata', eldata);
                if (eldata) {
                    if (!eldata.metadata) {
                        eldata.metadata = {
                            modified: (new Date()).getTime()
                        };
                    };
                    eldata.metadata.lang = lang;
                    notebook.changeContent(elname, eldata);
                };
            };
        });
        this.place.off('focusout.titlefield').on('focusout.titlefield', '.notebook-pagetitle .notebookview-titletext', function(event, data) {
            var field = $(this);
            var title = '';
            if (field.prop('tagName') === 'INPUT') {
                title = sanitize(field.val());
            } else if (field.hasClass('notebookview-mathtitle')) {
                title = field.mathquill('latex');
            };
            notebook.setPageTitle(title);
        });
        this.place.off('reply_getcontent').on('reply_getcontent', function(event, data, isinit) {
            event.stopPropagation();
            if (event.target === this && data.contentType === 'notebookcontent') {
                var nonempty = false;
                var mergedata = {};
                var name, elinfo, query, hasnewdirty, element,
                    lang = data.lang,
                    refs = data.refs[notebook.getBookId()] || [];
                for (var i = 0, len = refs.length; i < len; i++) {
                    name = refs[i];
                    elinfo = JSON.parse(JSON.stringify(data.contentinfo[name] || {}));
                    element = JSON.parse(JSON.stringify(data.contentdata[name] || {}));
                    query = {
                        name: elinfo.name,
                        lang: elinfo.lang,
                        type: element.type,
                        timestamp: elinfo.timestamp
                    };
                    hasnewdirty = notebook.changelog.hasNewerDirty(query);
                    if (!hasnewdirty) {
                        // If the notebook hasn't newer element as dirty, merge this one to the content.
                        if (!element.metadata) {
                            element.metadata = {lang: lang};
                        };
                        mergedata[name] = element;
                        nonempty = true;
                    };
                };
                if (nonempty) {
                    notebook.mergeContent(mergedata, isinit);
                };
            };
        });
        this.place.off('clear_refs').on('clear_refs', function(event, data){
            notebook.notebook.clearRefs(data);
        });
        this.place.off('get_bookelement').on('get_bookelement', function(event, data){
            event.preventDefault();
            event.stopPropagation();
            var target = $(event.target);
            var replydata = {};
            var cid;
            var lang = data.lang;
            for (var i = 0, len = data.contentlist.length; i < len; i++) {
                cid = data.contentlist[i];
                replydata[cid] = notebook.notebook.getContentById(cid, lang);
            }
            target.trigger('reply_get_bookelement', [replydata]);
        });
        this.place.off('updates_available').on('updates_available', function(event, data) {
            event.stopPropagation();
            notebook.requestNewUpdates();
        });
        this.place.off('getauthors').on('getauthors', function(event, data) {
            event.stopPropagation();
            var target = $(event.target);
            var replydata = notebook.notebook.getAuthors();
            target.trigger('reply_getauthors', [replydata]);
        });
        this.place.off('gettitleinfo').on('gettitleinfo', function(event, data) {
            event.stopPropagation();
            var target = $(event.target);
            var replydata = notebook.getTitleinfo();
            target.trigger('reply_gettitleinfo', [replydata]);
        });
        this.place.off('getcopyrightinfo').on('getcopyrightinfo', function(event, data) {
            event.stopPropagation();
            var target = $(event.target);
            var replydata = notebook.getCopyrightInfo();
            target.trigger('reply_getcopyrightinfo', [replydata]);
        });
        this.place.off('getelementboxnumber').on('getelementboxnumber', function(event, data) {
            event.stopPropagation();
            if (notebook.configs.usenumbering) {
                var boxnumber = notebook.getBoxnumber(data.boxclass, data.id);
                if (boxnumber) {
                    data.callback(boxnumber);
                };
            };
        });
        this.place.off('setrole').on('setrole', function(event, data){
            event.stopPropagation();
            notebook.setRole(data);
            notebook.show();
        });
        this.place.off('returnrole').on('returnrole', function(event, data){
            event.stopPropagation();
            notebook.returnOwnRole();
            notebook.show();
        });
        this.place.off('gotolink').on('gotolink', function(event, data){
            var link = data.data && data.data.link;
            if (link && link.appname && link.appname === notebook.getBookId()) {
                event.stopPropagation();
                notebook.goToLink(data);
            };
        });
        this.place.off('gotopage').on('gotopage', function(event, data){
            event.stopPropagation();
            notebook.goToPage(data.chapid);
        });
        this.place.off('datasaved').on('datasaved', function(event, data){
            event.stopPropagation();
            notebook.dataSaved(data);
        });
        this.place.off('mergecontent').on('mergecontent', function(event, data){
            event.stopPropagation();
            notebook.mergeContent(data);
        });
        this.place.off('setcontent').on('setcontent', function(event, data) {
            // Pass the event through and add the id of this notebook as anchor.
            // Don't stop propagation!!!
            var appid = notebook.getBookId();
            var dataarr;
            if (typeof(data) === 'object' && typeof(data.length) === 'number') {
                dataarr = data;
            } else {
                dataarr = [data];
            };
            for (var i = 0, len = dataarr.length; i < len; i++) {
                if (dataarr[i].anchors.indexOf(appid) === -1) {
                    dataarr[i].anchors.push(appid);
                };
            };
        });
        this.place.off('saveconfigs').on('saveconfigs', function(event, data){
            if (event.target !== this) {
                event.stopPropagation();
                notebook.saveConfigs(data);
            };
        });
        this.place.off('formdialog-valuechange').on('formdialog-valuechange', function(event, data){
            var dialogdata = data.dialogdata;
            var apps = dialogdata.formsets;
            var values = dialogdata.values;
            var appname = notebook.getBookId();
            var appindex = apps.indexOf(appname);
            if (notebook.selfconfigable && appindex !== -1) {
                apps.splice(appindex, 1);
                var confs = values[appname];
                delete values[appname];
                notebook.place.trigger('setconfigs', confs);
                if (apps.length === 0) {
                    event.stopPropagation();
                };
            };
        });
        //this.place.off('click', '.notebook-tocicon').on('click', '.notebook-tocicon', function(event, data){
        this.place.off('opentoc').on('opentoc', function(event, data){
            event.stopPropagation();
            var lang = notebook.settings.lang;
            var place = notebook.place.find('.notebook-toc');
            if (place.is('.tocelement-open')) {
                place.trigger('destroy');
            } else {
                var booktitle = notebook.notebook.getBookTitlesAll();
                var tocdata = {
                    title: booktitle,
                    toc: $.extend(true, {}, notebook.notebook.getTocData()),
                    pagetitles: notebook.notebook.getPagetitles(),
                    currentPage: notebook.currentPage,
                    isdefaultlang: notebook.isDefaultLang(),
                    defaultlang: notebook.notebook.getDefaultlang(),
                    settings: {
                        mode: notebook.settings.mode,
                        role: notebook.settings.role,
                        lang: notebook.settings.lang,
                        uilang: notebook.settings.uilang,
                        readonly: notebook.settings.readonly
                    }
                };
                place.tocelement(tocdata);
            };
        });
        this.place.off('refreshtoc').on('refreshtoc', function(event, data) {
            event.stopPropagation();
            var place = notebook.place.find('.notebook-toc');
            var tocdata = {
                toc: $.extend(true, {}, notebook.notebook.getTocData()),
                pagetitles: $.extend(true, {}, notebook.notebook.getPagetitles()),
                currentPage: notebook.currentPage,
                settings: {
                    mode: notebook.settings.mode,
                    role: notebook.settings.role,
                    lang: notebook.settings.lang,
                    uilang: notebook.settings.uilang,
                    readonly: notebook.settings.readonly
                }
            };
            place.tocelement(tocdata);
        });
        this.place.off('click.tocclose').on('click.tocclose', function(event){
            notebook.place.find('.tocelement').trigger('destroy');
        });
        this.place.off('tocmovepageforward').on('tocmovepageforward', function(event, data){
            event.stopPropagation();
            notebook.notebook.movePageForward(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocmovepagebackward').on('tocmovepagebackward', function(event, data){
            event.stopPropagation();
            notebook.notebook.movePageBackward(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocincreasenesting').on('tocincreasenesting', function(event, data){
            event.stopPropagation();
            notebook.notebook.indentPage(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocdecreasenesting').on('tocdecreasenesting', function(event, data){
            event.stopPropagation();
            notebook.notebook.unindentPage(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocdemotehere').on('tocdemotehere', function(event, data){
            event.stopPropagation();
            notebook.notebook.demotePage(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocpromotehere').on('tocpromotehere', function(event, data){
            event.stopPropagation();
            notebook.notebook.promotePage(notebook.currentPage);
            notebook.tocChanged();
        });
        this.place.off('tocchangebooktitle').on('tocchangebooktitle', function(event, data){
            event.stopPropagation();
            notebook.setBookTitle(data);
        });
        this.place.off('setconfigs').on('setconfigs', function(event, data){
            event.stopPropagation();
            notebook.setConfigs(data);
        }).off('closechildrenapp').on('closechildrenapp', function(event, data){
            event.stopPropagation();
            notebook.close();
        }).off('closeapp').on('closeapp', function(event, data){
            event.stopPropagation();
            notebook.close();
        });
        // Counter for elementpanel
        this.place.off('startediting', '.notebook-viewarea').on('startediting', '.notebook-viewarea', function(event, data){
            notebook.epanelcounter++;
        });
        // Counter for elementpanel
        this.place.off('stopediting', '.notebook-viewarea').on('stopediting', '.notebook-viewarea', function(event, data){
            notebook.epanelcounter--;
        });
        // Keep notebooks index updated.
        this.place.off('element_modified').on('element_modified', function(event, data) {
            event.stopPropagation();
            notebook.modifyIndex(data);
        });
        this.place.off('element_removed').on('element_removed', function(event, data) {
            event.stopPropagation();
            notebook.removeIndex(data);
        });
        this.place.off('getindexdata').on('getindexdata', function(event, data) {
            event.stopPropagation();
            var eltype = data.type;
            var target = $(event.target);
            var indexdata = notebook.getIndexDataByType(eltype);
            var idata, pageid, lang = notebook.settings.lang;
            for (var i = 0, len = indexdata.length; i < len; i++) {
                idata = indexdata[i];
                pageid = idata.gotolink && idata.gotolink.chapid || '';
                if (pageid) {
                    idata.pagetitle = notebook.getPageTitle(pageid, lang);
                    idata.pagenumber = notebook.getPageNumber(pageid);
                };
            };
            target.trigger('reply_getindexdata', {type: eltype, indexdata: indexdata});
        });
        this.place.off('gettitle').on('gettitle', function(event, data) {
            event.stopPropagation();
            data.title = notebook.getBookTitle();
        });
        // Marginnotes
        this.place.off('marginnotes_on').on('marginnotes_on', function(event, margin) {
            event.stopPropagation();
            notebook.setNoteview(true, margin);
        }).off('marginnotes_off').on('marginnotes_off', function(event, data) {
            event.stopPropagation();
            notebook.setNoteview(false);
        }).off('marginnotes_edit_on').on('marginnotes_edit_on', function(event, data) {
            event.stopPropagation();
            notebook.setNoteedit(true);
        }).off('marginnotes_edit_off').on('marginnotes_edit_off', function(event, data) {
            event.stopPropagation();
            notebook.setNoteedit(false);
        });
        // Page elements
        this.place.off('getelements').on('getelements', function(event, data) {
            event.stopPropagation();
            data.exports = notebook.getPageExport();
        });
        // Share the current page to online onepage notebook
        this.place.off('sharethepageonline').on('sharethepageonline', function(event, data) {
            event.stopPropagation();
            notebook.exportToOnline();
        });
    };
    
    /**
     * Inform outer system that the application is ready
     */
    NotebookView.prototype.appReady = function(){
        this.place.trigger('appready', {apptype: 'notebook', appid: this.getBookId(), title: this.getBookTitle()});
    };
    
    /**
     * Inform outer system that the title has changed is ready
     */
    NotebookView.prototype.titleChanged = function(){
        this.changelog.addDirty({
            name: this.getBookId() + '_booktitles',
            type: 'booktitles',
            timestamp: (new Date()).getTime(),
            lang: this.settings.lang
        });
        this.place.trigger('apptitlechanged', {apptype: 'notebook', appid: this.getBookId(), title: this.getBookTitle(), lang: this.settings.lang});
        this.appdataChanged([{key: 'titles', chtype: 'edit'}]);
        //this.place.trigger('titledata', [{title: this.getBookTitle()}]);
    };
    
    /**
     * Change the content data of an element.
     * @param {String} elname - The name of the element
     * @param {Object} eldata - The data of the element
     */
    NotebookView.prototype.changeContent = function(elname, eldata){
        this.notebook.setData(elname, eldata);
        this.changelog.addDirty({
            name: elname,
            type: eldata.type,
            timestamp: eldata.metadata.modified || (new Date()).getTime(),
            lang: eldata.metadata.lang
        });
    };
    
    /**
     * Trigger saving of unsaved "dirty" data with some outer system
     * Save all "dirty" items and move them to "sending".
     */
    NotebookView.prototype.saveData = function(){
        var ellist = this.changelog.getList();
        var elitem, eldata, lang, savedata;
        var recipients = this.settings.users.getRecipientSendto('notebookcontent');
        // New with storage
        for (var i = 0, len = ellist.length; i < len; i++) {
            elitem = ellist[i];
            eldata = this.getElementData(elitem);
            lang = (elitem.type === 'toc' ? 'common' : eldata.metadata && eldata.metadata.lang) || this.settings.lang;
            savedata = {
                "name": elitem.name,
                "contentType": "notebookcontent",
                "lang": lang,
                "anchors": [this.getBookId()],
                "contentdata": eldata,
                "contentinfo": {
                    "name": elitem.name,
                    "contentType": "notebookcontent",
                    "lang": lang,
                    "pubtype": (this.settings.role === 'teacher' ? 'all' : 'own'),
                    "sendto": recipients,
                    "subcontext": this.getBookId(),
                    "timestamp": elitem.timestamp,
                    "isoutbox": true
                }
            };
            this.changelog.moveToSaving(elitem);
            this.place.trigger('setcontent', [savedata, true]);  // True = isdirty
        };
    };
    
    /**
     * Mark data as saved (remove from changelog's "saving"), when saving is successfull.
     * @param {Object} item                Data describing the saved item
     * @param {String} item.name              The id of the element
     * @param {String} item.lang              The language of the element
     * @param {String} item.type              The type of the element
     * @param {String|Number} item.timestamp  The timestamp of the element
     */
    NotebookView.prototype.dataSaved = function(data){
        this.changelog.removeSaving(data);
    };
    
    /**
     * Get navigation data (a linear presentation of pages)
     */
    NotebookView.prototype.getNavdata = function(){
        var navdata = this.notebook.getNavdata(this.filter);
        var data = navdata.data;
        var page;
        for (var i = 0, len = data.length; i < len; i++){
            page = data[i];
            page.title = this.getPageTitle(page.id);
        };
        return navdata;
    };
    
    /**
     * Get the level of the page
     * @param {String} chapid - id of the page
     * @returns {Number} the level (0 === frontpage, 1 === chapter, 2 === section, ...)
     */
    NotebookView.prototype.getPageLevel = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.notebook.getPageLevel(chapid);
    }
    
    /**
     * Get the list of pagetypes of the page
     * @param {String} chapid   The id of the page
     * @returns {Array} an array of pagetitles
     */
    NotebookView.prototype.getPageTypes = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.notebook.getPageTypes(chapid);
    };
    
    /**
     * Get the number for asked element in boxclass
     * @param {String} boxclass   The class of the box (theorybox, examplebox, definitionbox,...);
     * @param {String} elemid     The elementid of the box.
     * @return {Number|boolean}   The boxnumber in that class or false, if not found.
     */
    NotebookView.prototype.getBoxnumber = function(boxclass, elemid) {
        return this.notebook.getBoxnumber(boxclass, elemid, this.settings.lang);
    };
    
    /**
     * Recalculate boxnumbering
     */
    NotebookView.prototype.refreshBoxnumbers = function() {
        this.notebook.computeBoxnumbers();
    };
    
    /**
     * Set the title of the book
     * @param {Object} titledata - of format: {common: 'title', fi: 'title', ...}
     */
    NotebookView.prototype.setBookTitle = function(titledata) {
        this.notebook.setBookTitle(titledata);
        this.titleChanged();
    };
    
    NotebookView.prototype.getPageTitle = function(chapid, lang) {
        lang = lang || this.settings.lang;
        chapid = chapid || this.currentPage;
        return this.notebook.getPageTitle(chapid, lang);
    }
    
    NotebookView.prototype.setPageTitle = function(title, chapid, lang) {
        lang = lang || this.settings.lang;
        chapid = chapid || this.currentPage;
        this.notebook.setPageTitle(chapid, title, lang);
        this.updateNavbar();
        this.titlesChanged();
    };
    
    /**
     * Get the pagenumber as an array
     * @param {String} chapid - id of the page
     */
    NotebookView.prototype.getPageNumberList = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.notebook.getPageNumber(chapid);
    };
    
    /**
     * Get the pagenumber as a string
     * @param {String} chapid - id of the page
     */
    NotebookView.prototype.getPageNumber = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.getPageNumberList(chapid).join('.');
    };
    
    /**
     * Get the pageid (the id of the pageelement) of the given page (or current)
     * @param {String} chapid    The chapter id of the page.
     */
    NotebookView.prototype.getPageId = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.notebook.getPageId(chapid);
    };
    
    /**
     * Get the headimage of the page
     * @param {String} chapid - id of the page
     * @returns {Object} the data for headimage {type: <headimage|fullheadimage>, text: <elemid>}
     */
    NotebookView.prototype.getHeadimage = function(chapid) {
        chapid = chapid || this.currentPage;
        return this.notebook.getHeadimage(chapid);
    };
    
    /**
     * Get titleinfo with current language
     * @returns {Object}    The titleinfo (title, shorttitle, shorttitlefill,...)
     */
    NotebookView.prototype.getTitleinfo = function() {
        return this.notebook.getTitleinfo(this.settings.lang);
    }
    
    /**
     * Get copyrightinfo with current language
     * @returns {Object}    The titleinfo (title, shorttitle, shorttitlefill,...)
     */
    NotebookView.prototype.getCopyrightInfo = function() {
        return this.notebook.getCopyrightInfo();
    }
    
    /**
     * Get the data of one element from notebook
     * @param {Object} item - data describing the element to get {name: "<elementid>", type: "<toc|pagetitles|index|content>", lang: "<language of the element>"}
     */
    NotebookView.prototype.getElementData = function(item){
        return this.notebook.getElementData(item);
    }
    
    /**
     * Get the data of the notebook as an object.
     * @param {Boolean} isexport - If this is export, generate a new bookid/name
     */
    NotebookView.prototype.getData = function(isexport){
        return this.notebook.getData(isexport);
    };
    
    /**
     * Get the id of the current notebook
     */
    NotebookView.prototype.getBookId = function(){
        return this.notebook.getBookId();
    };
    
    /**
     * Get the title of the current notebook
     */
    NotebookView.prototype.getBookTitle = function(){
        return this.notebook.getBookTitle();
    };

    /**
     * Get index data by type
     * @param {String} eltype - type of elements we are looking index for
     * @returns {Array} a list of index data of elements of given type
     */
    NotebookView.prototype.getIndexDataByType = function(eltype) {
        return this.notebook.getIndexDataByType(eltype);
    }
    
    /**
     * Get the data of the notebook as a JSON-string.
     * @param {Boolean} isexport - If this is export, generate a new bookid/name
     */
    NotebookView.prototype.getJSON = function(isexport){
        return JSON.stringify(this.getData(isexport), null, 4);
    };
    
    /**
     * Get the data of the current page as updateparts
     * @param {String} lang    Language to be forced for elements
     * @return {Array} an array with elements for the current page (as "frontpage") and the pagetitles.
     */
    NotebookView.prototype.getPageExport = function(lang) {
        var data = {elements: []};
        this.pageview.trigger('getelements', data);
        var elems = data.elements;
        var parts = [], cdata, eldata, element;
        var bookid = this.getBookId().slice(0, 100);
        for (var i = 0, len = elems.length; i < len; i++) {
            cdata = elems[i];
            if (cdata.type === 'pageelement') {
                cdata.name = 'frontpage';
            };
            eldata = {
                name: cdata.name,
                contentType: 'notebookcontent',
                lang: lang || cdata.metadata.lang || 'common',
                anchors: [bookid],
                contentdata: cdata,
                contentinfo: {
                    name: cdata.name,
                    contentType: 'notebookcontent',
                    lang: lang || cdata.metadata.lang || 'common',
                    sendto: [
                        {
                            to: 'demobook',
                            pubtype: 'context'
                        }
                    ],
                    sendfrom: '',
                    read: false,
                    recipientopened: false,
                    isoutbox: false,
                    pubtype: 'own',
                    timestamp: cdata.metadata.modified || (new Date()).getTime()
                }
            };
            element = {
                contexttype: 'own_notes',
                contextid: bookid,
                dataid: cdata.name,
                data: JSON.stringify(eldata),
                send_to: [
                    {
                        to: 'demobook',
                        pubtype: 'context'
                    }
                ],
                datatype: 'notebookcontent',
                datamodified: cdata.metadata.modified || (new Date()).getTime(),
                lang: lang || cdata.metadata.lang || 'common'
            };
            parts.push(element);
        };
        var booktoc = {
            name: bookid + '_toc',
            contentType: 'notebookcontent',
            lang: 'common',
            anchors: [bookid],
            contentdata: {
                type: 'toc',
                metadata: {},
                data: {
                    firstpage: {
                        chapid: 'front',
                        content: 'frontpage'
                    }
                }
            },
            contentinfo: {
                name: bookid + '_toc',
                contentType: 'notebookcontent',
                lang: 'common',
                sendto: [
                    {
                        to: 'demobook',
                        pubtype: 'context'
                    }
                ],
                senfrom: '',
                read: false,
                recipientopened: false,
                isoutbox: false,
                pubtype: 'own',
                timestamp: (new Date()).getTime()
            }
        };
        parts.push({
            contexttype: 'own_notes',
            contextid: bookid,
            dataid: bookid + '_toc',
            data: JSON.stringify(booktoc),
            send_to: [
                {
                    to: 'demobook',
                    pubtype: 'context'
                }
            ],
            datatype: 'notebookcontent',
            datamodified: cdata.metadata.modified || (new Date()).getTime(),
            lang: cdata.metadata.lang || 'common'
        });
        var pagetitles = {
            name: bookid + '_pagetitles',
            contentType: 'notebookcontent',
            lang: 'common',
            anchors: [bookid],
            contentdata: {
                type: 'pagetitles',
                name: bookid + '_pagetitles',
                metadata: {
                    lang: 'common',
                    modified: (new Date()).getTime()
                },
                data: {
                    front: this.getPageTitle(this.currentPage, this.settings.lang)
                }
            },
            contentinfo: {
                name: bookid + '_pagetitles',
                contentType: 'notebookcontent',
                lang: 'common',
                sendto: [
                    {
                        to: 'demobook',
                        pubtype: 'context'
                    }
                ],
                sendfrom: '',
                read: false,
                recipientopened: false,
                isoutbox: false,
                pubtype: 'own',
                timestamp: (new Date()).getTime()
            }
        };
        parts.push({
            contexttype: 'own_notes',
            contextid: bookid,
            dataid: bookid + '_pagetitles',
            data: JSON.stringify(pagetitles),
            send_to: [
                {
                    to: 'demobook',
                    pubtype: 'context'
                }
            ],
            datatype: 'notebookcontent',
            datamodified: cdata.metadata.modified || (new Date()).getTime(),
            lang: cdata.metadata.lang || 'common'
        });
        return parts;
    };
    
    /**
     * Get page data for copying whole page
     * @returns {Object}  Data of the page as "dragelement" with title.
     */
    NotebookView.prototype.getPageCopy = function() {
        var getdata = {elements: [], rename: true};
        this.pageview.trigger('getelements', getdata);
        var elements = getdata.elements;
        var pageElem = elements[elements.length - 1];
        pageElem.name = this.notebook.newContentid('pageelement');
        var dragdata = {
            type: 'dragelementpage',
            data: {
                dragelement: elements[elements.length - 1].name,
                title: this.getPageTitle(this.currentPage, this.settings.lang),
                dragdata: {}
            }
        };
        var ddata = dragdata.data.dragdata;
        for (let i = 0, len = elements.length; i < len; i++) {
            let item = elements[i];
            ddata[item.name] = item;
        };
        return dragdata;
    };
    
    /**
     * Get data for page link
     * @returns {Object}  Data of the page link as "contentlink"
     */
    NotebookView.prototype.getPageLink = function() {
        let pageid = this.notebook.getPageId(this.currentPage);
        let linkdata = {
            "type": "contentlink",
            "metadata": {
                "creator": this.settings.username,
                "created": (new Date()).getTime(),
                "modifier": this.settings.username,
                "modified": (new Date()).getTime(),
                "tags": []
            },
            "data": {
                "title": this.getPageNumber() + ' ' + this.getPageTitle(this.currentPage, this.settings.lang),
                "text": "",
                "type": "pageelement",
                "link": {
                    "elementsetpath": [pageid],
                    "elementid": pageid,
                    "pageelementid": pageid,
                    chapid: this.currentPage,
                    appname: this.settings.gotolink.appname,
                    courseid: this.settings.gotolink.courseid,
                    contextid: this.settings.gotolink.contextid
                }
            }
        };
        return linkdata;
    };
    
    /**
     * Add page copy after given pageid (defaults to current page)
     * @param {Object} pagedata    The data of type 'dragelementpage'
     * @param {String} chapid      Optional chapid, after which the new page is added. If not given, use current page.
     */
    NotebookView.prototype.addPageCopy = function(pagedata, chapid) {
        var newchapid = this.addPageAfter(chapid);
        var elements = pagedata.dragdata;
        var oldname = pagedata.dragelement;
        var newname = this.getPageId(newchapid);
        elements[newname] = elements[oldname];
        elements[newname].name = newname;
        delete elements[oldname];
        for (let elname in elements) {
            let eldata = elements[elname];
            this.changeContent(elname, eldata);
        };
        this.setPageTitle(pagedata.title);
        this.updatePage();
    };
    
    /**
     * Check if current language is default language or a translation.
     * @param {String} [lang]    Language to check (optional parameter, if not given, use current)
     * @returns {Boolean}     True, if the language is default language.
     */
    NotebookView.prototype.isDefaultLang = function(lang) {
        lang = lang || this.settings.lang;
        return lang === this.notebook.getDefaultlang();
    };
    
    /**
     * Add a page after given page
     * @param {String} chapid - id of the page/chapter
     */
    NotebookView.prototype.addPageAfter = function(chapid){
        chapid = chapid || this.currentPage;
        var newchapid = this.notebook.addPageAfter(chapid);
        this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
        this.currentPage = newchapid;
        this.updateNavbar();
        this.updatePage();
        return newchapid;
    };
    
    /**
     * Add a chapter after given chapter (or parent chapter of the page)
     * @param {String} chapid - id of the page/chapter
     */
    NotebookView.prototype.addChapterAfter = function(chapid){
        chapid = chapid || this.currentPage;
        var newchapid = this.notebook.addChapterAfter(chapid);
        this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
        this.currentPage = newchapid;
        this.updateNavbar();
        this.updatePage();
        return newchapid;
    };
    
    /**
     * Remove the given page
     * @param {String} chapid - id of the page/chapter
     */
    NotebookView.prototype.removePage = function(chapid){
        chapid = chapid || this.currentPage;
        var movechapid = this.notebook.removePage(chapid);
        this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
        if (movechapid){
            this.currentPage = movechapid;
            this.updateNavbar();
            this.updatePage();
        };
        return movechapid;
    };
    
    /**
     * Add the root before-page.
     * @returns {String} the chapid of the new page created. If before-page already existed, return empty string.
     */
    NotebookView.prototype.addBeforePages = function() {
        var newchapid = this.notebook.addBeforePage();
        if (newchapid) {
            this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
            this.currentPage = newchapid;
            this.updateNavbar();
            this.updatePage();
        };
        return newchapid;
    };
    
    /**
     * Add the root after-page.
     * @returns {String} the chapid of the new page created. If after-page already existed, return empty string.
     */
    NotebookView.prototype.addAfterPages = function() {
        var newchapid = this.notebook.addAfterPage();
        if (newchapid) {
            this.changelog.addDirty({name: this.getBookId() + '_toc', type: 'toc', timestamp: (new Date()).getTime(), lang: 'common'});
            this.currentPage = newchapid;
            this.updateNavbar();
            this.updatePage();
        };
        return newchapid;
    };
    
    /**
     * Merge updatedata given in init.
     * @param {Array} updatedata - An array with updated elements.
     */
    NotebookView.prototype.mergeUpdates = function(updatedata) {
        var contentdata = {}, item;
        if (typeof(updatedata) === 'object' && typeof(updatedata.length) === 'number') {
            for (var i = 0, len = updatedata.length; i < len; i++){
                item = updatedata[i];
                if (typeof(item.data) !== 'object') {
                    try {
                        item.data = JSON.parse(item.data);
                    } catch (err) {
                        console.log('parse error', err);
                        continue;
                    };
                };
                contentdata[item.dataid] = item.data;
            };
            this.mergeContent(contentdata);
        };
    };
    
    /**
     * Merge new content (toc|pagetitles|index|contentelements) to the current notebook.
     * @param {Object} data - an object with element names as keys and element data as values.
     * @param {Boolean} isinit   Whether this is initing merge. (No need to prompt the user for confirmation.)
     */
    NotebookView.prototype.mergeContent = function(data, isinit){
        var uilang = this.settings.uilang;
        var hastoc = !!data[this.getBookId() + '_toc'];
        var haselements = [];
        var viewarea = this.place.find('.notebook-viewarea');
        for (var elid in data) {
            // Find elements that are on current page.
            if (viewarea.find('[data-element-name="'+elid+'"]').length > 0) {
                haselements.push(elid);
            };
        };
        // Merge and get list of changed elements.
        var wasChanged = this.notebook.mergeContent(data);
        if (hastoc) {
            this.updateNavbar();
        };
        for (var i = haselements.length - 1; i > -1; i--) {
            // Remove from list of elements on current page those that were not changed.
            if (wasChanged.indexOf(haselements[i]) === -1) {
                haselements.splice(i, 1);
            };
        };
        // TODO: Some clever thing to detect, if page refresh is needed and
        // Maybe to ask, if the user wants it.
        if (!isinit && !this.editable && (haselements.length > 0) && confirm(ebooklocalizer.localize('notebook:want_refresh', uilang))) {
            // If this notebook is in read-only mode (another window is in editing mode?),
            // then update this page in case the 
            this.updatePage();
        };
    };
    
    /**
     * Go to given page
     * @param {String} chapid - chapid of the page
     */
    NotebookView.prototype.goToPage = function(chapid, nonavi){
        if (this.configs.currentpage !== chapid) {
            var navbar = this.place.find('.notebook-naviarea');
            if (!nonavi) {
                // Move navbar only, if it is not prevented.
                navbar.trigger('goto', chapid);
            };
            var toclist = this.place.find('.notebook-toc');
            toclist.trigger('goto', chapid);
            this.configs.currentpage = chapid;
            this.saveConfigs();
        };
    };
    
    /**
     * Go to the link inside this notebook
     * @param {Object} linkdata - the link to the target
     */
    NotebookView.prototype.goToLink = function(linkdata){
        var link = linkdata.data && linkdata.data.link;
        delete link.appname;
        if (link.chapid && link.pageelementid) {
            var chapid = link.chapid;
            this.goToPage(chapid);
            this.place.find('.notebook-viewarea .notebook-pageview[data-element-name="'+link.pageelementid+'"]').trigger('gotolink', linkdata);
        };
    };
    
    /**
     * Modify notebook's index
     * @param {Object} data - the data of indexed element
     */
    NotebookView.prototype.modifyIndex = function(data) {
        this.notebook.addIndex(data);
        this.indexChanged(data.lang);
    };
    
    /**
     * Remove notebook's index
     * @param {Object} data - the data of indexed element
     */
    NotebookView.prototype.removeIndex = function(data) {
        var lang = data.lang;
        this.notebook.removeIndex(data);
        this.indexChanged(lang);
    };
    
    /**
     * Set html classes and other attributes for NotebookView
     */
    NotebookView.prototype.setAttrs = function(){
        this.place
            .addClass('notebookview-wrapper')
            .attr('data-appname', this.notebook.getBookId())
            .attr('textsize', this.configs.textsize);
        if (this.configs.usedyslexia) {
            this.place.addClass('usedyslexiafont');
        } else {
            this.place.removeClass('usedyslexiafont');
        };
        if (this.configs.usenumbering) {
            this.refreshBoxnumbers();
            this.place.addClass('usenumbering');
        } else {
            this.place.removeClass('usenumbering');
        };
        if (this.settings.onepagemode) {
            this.place.addClass('onepagemode');
        } else {
            this.place.removeClass('onepagemode');
        };
    };
    
    /**
     * Set themes
     */
    NotebookView.prototype.setThemes = function() {
        if (this.configs.ffwtheme && !this.settings.hasparent) {
            this.place.attr('data-ffwtheme', this.configs.ffwtheme);
        }
        if (this.configs.nbtheme) {
            this.place.attr('data-nbtheme', this.configs.nbtheme);
        } else {
            this.place.removeAttr('data-nbtheme');
        };
    };
    
    /**
     * Set noteview (marginnotes) on or off
     * @param {Boolean} [noteview] Optional parameter to switch the status. If not given, use the current status.
     * @param {Object} [margin]    Optional parameter with functions form margin handling.
     */
    NotebookView.prototype.setNoteview = function(noteview, margin) {
        if (typeof(noteview) === 'boolean') {
            this.noteview = noteview;
        };
        if (typeof(margin) === 'object') {
            this.margin = margin;
        };
        if (!this.editable) {
            var elementsets = this.place.find('.notebook-viewarea > .notebook-pageview > .ebook-elementset');
            if (this.noteview) {
                elementsets.trigger('showmarginnotes', [this.margin]);
            } else {
                elementsets.trigger('hidemarginnotes');
            };
        };
    };
    
    /**
     * Set noteedit (marginnotes editmode) on or off
     * @param {Boolean} [noteedit] Optional parameter to switch the status. If not given, use the current status.
     */
    NotebookView.prototype.setNoteedit = function(noteedit) {
        if (typeof(noteedit) === 'boolean') {
            this.noteedit = noteedit;
        };
        if (!this.editable) {
            var elementsets = this.place.find('.notebook-viewarea > .notebook-pageview > .ebook-elementset');
            if (this.noteedit) {
                elementsets.trigger('editmarginnotes');
            } else {
                elementsets.trigger('viewmarginnotes');
            };
        };
    };
    
    /**
     * Open config form dialog
     */
    NotebookView.prototype.openConfigForm = function(){
        var uilang = this.settings.uilang;
        var bookid = this.getBookId();
        var confform = this.getConfigForm();
        var values = JSON.parse(JSON.stringify(this.configs));
        var dialog = $('<div class="notebook-configformdialog"></div>');
        this.place.append(dialog);
        var formdata = {
            type: 'dialog',
            name: 'notebookconfig',
            data: {
                title: ebooklocalizer.localize('notebook:configs', uilang),
                icon: NotebookView.icons.settings,
                formsets: [bookid],
                forms: {},
                values: {}
            },
            config: {
                position: {
                    x: 300,
                    y: 500
                }
            },
            settings: {
                uilang: uilang,
                lang: this.settings.lang,
                modal: false
            }
        };
        formdata.data.forms[bookid] = confform;
        formdata.data.values[bookid] = values;
        dialog.formdialog(formdata);
    };
    
    /**
     * Get config form as JSON for sending to the config dialog.
     */
    NotebookView.prototype.getConfigForm = function(){
        var uilang = this.settings.uilang;
        var form = {
            name: this.getBookId(),
            title: this.getBookTitle(),
            icon: NotebookView.icons.notebook,
            form: []
        };
        // Add form element for theme
        var themeform = {
            name: 'nbtheme',
            label: ebooklocalizer.localize('notebook:theming', uilang),
            type: 'select',
            attribute: 'nbtheme',
            defval: 'default',
            options: []
        };
        // Get all link-tags with themes
        var themestyles = $('head link[type="text/css"][href*="notebooktheme-"]');
        // Extract the theme names from href-attributes to an array.
        var themenames = themestyles.map(function(i, style){
            var tname = $(style).attr('href').replace(/^.*notebooktheme-([^.]*)\.css$/, '$1');
            return tname;
        }).get();
        var themeoption, tname;
        for (var i = 0, len = themenames.length; i < len; i++) {
            tname = themenames[i];
            themeoption = {
                label: tname.charAt(0).toUpperCase() + tname.slice(1),
                value: tname
            };
            themeform.options.push(themeoption);
        };
        themeform.options.sort(function(a, b){
            return (a.label < b.label ? -1 : 1);
        });
        themeform.options.unshift({
            label: 'Plain',
            value: ''
        });
        form.form.push(themeform);
        form.form.push({
            name: 'usedyslexia',
            label: ebooklocalizer.localize('notebook:dyslexiafont', uilang),
            type: 'checkbox',
            attribute: 'usedyslexia',
            defval: false
        });
        form.form.push({
            name: 'textsize',
            label: 'Text size',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-textsize"><path style="stroke: none;" d="M3 26 l3 -9 h2 l3 9 h-2 l-1 -3 h-2 l-1 3z m4 -7 l-0.7 2.5 h1.4z m4 7 l6 -18 h4 l6 18 h-4 l-2 -6 h-4 l-2 6z m8 -14 l-1.4 5 h2.8z"></path></svg>',
            type: 'range',
            output: true,
            attribute: 'textsize',
            defval: 0,
            min: -3,
            max: 3
        });
        form.form.push({
            name: 'zoomlevel',
            label: 'Zoom-level',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-zoom"><circle class="mini-icon-background" stroke="none" fill="none" cx="15" cy="15" r="15" /><path class="mini-icon-foreground" style="stroke: none;" d="M18 4 a8 8 0 0 1 0 16 a8 8 0 0 1 0 -16z m0 2 a6 6 0 0 0 0 12 a6 6 0 0 0 0 -12z m-1 2 h2 v3 h3 v2 h-3 v3 h-2 v-3 h-3 v-2 h3z m-7 10 a1.4 1.4 0 0 1 2 2 l-5 5 a1.4 1.4 0 0 1 -2 -2z"></path></svg>',
            type: 'range',
            output: true,
            attribute: 'zoomlevel',
            defval: 10,
            min: 5,
            max: 15
        })
        form.form.push({
            name: 'usenumbering',
            label: ebooklocalizer.localize('notebook:boxnumbering', uilang),
            type: 'checkbox',
            attribute: 'usenumbering',
            defval: true
        });
        return form;
    }
    
    /**
     * Get config data
     */
    NotebookView.prototype.getConfigs = function(){
        var confs = JSON.parse(JSON.stringify(this.configs));
        return confs;
    }
    
    /**
     * Save configs either locally (localStorage) or in parent module.
     * @param {Object} data - Optional config data to be saved.
     *                        If not given, just save all configs.
     *                        format: {appname: '<application id>', configs: {<configs to be saved>}}
     */
    NotebookView.prototype.saveConfigs = function(data){
        if (this.selfconfigable) {
            // Save given data or current data in this module
            if (typeof(data) !== 'undefined') {
                // Add given config data
                var appname = data.appname;
                var configs = data.configs;
                this.configs[appname] = configs;
            }
            // Save all config data
            var confstring = JSON.stringify(this.configs);
            try {
                localStorage.setItem('notebookapp_'+this.getBookId()+'_configs', confstring);
            } catch (err) {
                console.log('error saving notebook configs', err);
            };
        } else {
            // Save config data in parent module
            if (typeof(data) === 'undefined') {
                // Not given data, but save all. (Else save only given data)
                data = {
                    type: 'configsave',
                    appname: this.getBookId(),
                    configs: JSON.parse(JSON.stringify(this.configs))
                };
            };
            this.place.trigger('saveconfigs', data);
        }
    };
    
    /**
     * Load configs from localStorage
     * @returns {Object} config-object
     */
    NotebookView.prototype.loadConfigs = function(){
        var confs = {};
        if (this.selfconfigable) {
            try {
                confs = JSON.parse(localStorage.getItem('notebookapp_'+this.getBookId()+'_configs')) || {};
            } catch (err){
                confs = {};
            };
        };
        return confs;
    };
    
    /**
     * Request updates from outer system
     * @param {Boolean} isinit    Whether this is initing request
     */
    NotebookView.prototype.requestUpdates = function(isinit) {
        var langs = this.notebook.getLangs();
        for (var i = 0, len = langs.length; i < len; i++) {
            this.place.trigger('getcontent', [{anchors: [this.getBookId()], contentType: ['notebookcontent'], lang: langs[i]}, isinit]);
        };
    };
    
    /**
     * Request new updates from outer system
     */
    NotebookView.prototype.requestNewUpdates = function() {
        var langs = this.notebook.getLangs();
        for (var i = 0, len = langs.length; i < len; i++) {
            this.place.trigger('getupdatecontent', {anchors: [this.getBookId()], contentType: ['notebookcontent'], lang: langs[i]});
        };
        if (this.settings.readonly) {
            this.updateNavbar();
            this.updatePage();
        };
    };
    
    /**
     * Announce titledata
     */
    NotebookView.prototype.announceTitledata = function() {
        var titledata = {
            title: this.notebook.getBookTitle(),
            shorttitle: ''
        };
        this.place.trigger('titledata', [titledata]);
    };
    
    /**
     * Appdata has changed (Inform outside about application's title etc.)
     * @param {Array} changed   A list of changed keys
     */
    NotebookView.prototype.appdataChanged = function(changed) {
        var appdata ={
            info: {
                appid: this.getBookId(),
                apptype: 'notebook',
            },
            data: []
        };
        var change, item;
        for (var i = 0, len = changed.length; i < len; i++) {
            change = changed[i];
            item = {key: change.key, chtype: change.chtype, data: {}};
            switch (change.key) {
                case 'titles':
                    item.data['titles'] = this.notebook.getBookTitlesAll();
                    break;
                case 'shorttitle':
                    item.data['shorttitle'] = '';
                    break;
                case 'username':
                    item.data['username'] = this.settings.user;
                    break;
                default:
                    break;
            };
            appdata.data.push(item);
        };
        this.place.trigger('appdatachanged', [appdata]);
    };
    
    /**
     * Close notebook and do whatever is needed before closing (savedata, etc...)
     */
    NotebookView.prototype.close = function(){
        if (this.editable) {
            this.editmodeOff();
            this.updateMenu();
        };
        this.saveData();
        this.place.trigger('closeappok', {appid: this.getBookId(), apptype: 'notebook'});
    };
    
    /**
     * Set styles of NotebookView to the head of the html document.
     */
    NotebookView.prototype.setStyles = function(){
        if ($('head style#notebook-style').length === 0) {
            $('head').append('<style id="notebook-style" type="text/css">'+NotebookView.styles+'</style>')
        };
    };
    
    /**
     * Send the current page to the online onepagenotebook
     */
    NotebookView.prototype.exportToOnline = function() {
        var notebook = this;
        var uilang = this.settings.uilang;
        var data = this.getPageExport('common');
        var shareid = (new Date()).getTime() + '-' + Math.ceil(Math.random() * 10000);
        this.place.trigger('notification', {
            id: 'pageexportworking' + shareid,
            nclass: 'quick',
            message: ebooklocalizer.localize('notebook:sharing_page', uilang),
            icon: NotebookView.icons.sharerotate
        });
        var send = jQuery.post(this.settings.onepageonlineurls.saveurl, {
            "setData": JSON.stringify(data),
            "title": this.getPageTitle(this.currentPage, this.settings.lang),
            "hash": "",
            "bookhash": this.getBookId().slice(0, 100),
            "checksum": ""
        }, function(serverData) {
            notebook.place.trigger('denotification', {
                id: 'pageexportworking' + shareid,
                nclass: 'quick'
            });
            try {
                var resp_data = JSON.parse(serverData);
            } catch (err) {
                alert(ebooklocalizer.localize('notebook:share_error', uilang));
            };
            if (!resp_data.success) {
                notebook.place.trigger('notification', {
                    id: 'pageexporterror',
                    nclass: 'quick',
                    message: ebooklocalizer.localize('notebook:share_error', uilang),
                    icon: NotebookView.icons.share
                });
                alert("Error in network. Try again.");
            } else {
                var exporturl = notebook.settings.onepageonlineurls.showurl[notebook.settings.lang || 'en'] + resp_data.hash + '/' + resp_data.version;
                notebook.place.trigger('notification', {
                    id: 'pageexportsuccess' + shareid,
                    nclass: 'quick',
                    ttl: 10000,
                    message: ebooklocalizer.localize('notebook:page_shared', uilang) + exporturl,
                    data: {appid: notebook.getBookId, events: ['openurl'], data: [{url: exporturl}]},
                    icon: NotebookView.icons.share
                });
                console.log('exporturl:', exporturl);
                var timestamp = (new Date()).getTime();
                notebook.place.trigger('pagesharedonline', {
                    id: 'pageshare-' + shareid,
                    name: 'pageshare-' + shareid,
                    type: 'pageshare',
                    metadata: {
                        creator: notebook.settings.username,
                        modifier: notebook.settings.username,
                        created: timestamp,
                        modified: timestamp,
                        lang: 'common'
                    },
                    data: {
                        title: notebook.getPageTitle(notebook.currentPage, notebook.settings.lang),
                        url: exporturl
                    }
                });
            };
        })
    };
    
    /**
     * Default values of NotebookView.
     */
    NotebookView.defaults = {
        updatedata: [],
        settings: {
            mode: 'view',
            lang: 'common',
            uilang: 'en',
            username: 'Anonymous',
            role: 'student',
            readonly: false,
            elementpanel: true,
            hasparent: false,
            onepagemode: false,
            savemode: 'semilocal',
            gotolink: {
                appname: ''
            },
            onepageonlineurls: {
                saveurl: 'https://fourferries.com/onepagenotebook/webstore.php',
                showurl: {
                    common: 'https://fourferries.com/notes/',
                    en: 'https://fourferries.com/notes/',
                    fi: 'https://fourferries.com/vihko/',
                    sv: 'https://fourferries.com/hafte/'
                }
            }
        },
        configs: {
            "nbtheme": "default",
            "ffwtheme": "",
            "usedyslexia": false,
            "usenumbering": true,
            "zoomlevel": 10
        }
    }
    
    /**
     * Icons
     */
    NotebookView.icons = {
        menu: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-menu"><path style="stroke: none;" d="M5 4 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z m0 9 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z m0 9 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        settings: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 32 32" class="mini-icon mini-icon-gear"><path style="stroke: none;" d="M12 0 h8 l0.9 3.5 l3.5 2 l3.55 -1.1 l4.1 7 l-2.7 2.5 v4.1 l2.7 2.5 l-4.1 7 l-3.55 -1.1 l-3.5 2 l-0.9 3.5 h-8 l-0.9 -3.5 l-3.5 -2 l-3.55 1.1 l-4.1 -7 l2.7 -2.5 v-4.1 l-2.7 -2.5 l4.1 -7 l3.55 1.1 l3.5 -2z m1.55 2 l-0.7 2.9 l-5 2.8 l-2.8 -0.8 l-2.5 4.3 l2.15 2 v5.65 l-2.15 2 l2.5 4.3 l2.8 -0.8 l5 2.8 l0.7 2.9 h4.9 l0.7 -2.9 l5 -2.8 l2.8 0.8 l2.5 -4.3 l-2.15 -2 v-5.65 l2.15 -2 l-2.5 -4.3 l-2.8 0.8 l-5 -2.8 l-0.7 -2.9z m2.45 7.55 a6.45 6.45 0 0 1 0 12.9 a6.45 6.45 0 0 1 0 -12.9z m0 2 a4.45 4.45 0 0 0 0 8.9 a4.45 4.45 0 0 0 0 -8.9z"></path></svg>',
        edit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>',
        view: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-view"><path style="stroke: none;" d="M3 15 a13 13 0 0 0 24 0 a13 13 0 0 0 -24 0z m1 0 a13 13 0 0 1 22 0 a13 13 0 0 1 -22 0z m6 0 a5 5 0 0 0 10 0 a5 5 0 0 0 -2.2 -4 l-2.8 4 l0 -5 a5 5 0 0 0 -5 5z"></path></svg>',
        mathpanel: '&radic;<span style="border-top: 1px solid black;">&alpha;</span>',
        notebook: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-notebook"><path style="stroke: none;" d="M6 4 a2 2 0 0 1 2 -2 h14 a2 2 0 0 1 2 2 v22 a2 2 0 0 1 -2 2 h-14 a2 2 0 0 1 -2 -2 v-2 h1 a1 1 0 0 0 0 -2 h-1 v-2 h1 a1 1 0 0 0 0 -2 h-1 v-2 h1 a1 1 0 0 0 0 -2 h-1 v-2 h1 a1 1 0 0 0 0 -2 h-1 v-2 h1 a1 1 0 0 0 0 -2 h-1z m6 1 a1 1 0 0 0 -1 1 v3 a1 1 0 0 0 1 1 h8 a1 1 0 0 0 1 -1 v-3 a1 1 0 0 0 -1 -1z m11 -3 h1 a2 2 0 0 1 2 2 v22 a2 2 0 0 1 -2 2 a1 2 0 0 0 1 -2 v-22 a1 2 0 0 0 -1 -2z" /></svg>',
        sharerotate: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-share-rotating"><path style="stroke: none;" d="M7 11 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m15 -8 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 16 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m-13 -4 l15 8 v2 l-15 -8z m0 -2 l15 -8 v2 l-15 8z"><animateTransform attributeType="xml" attributeName="transform" type="rotate" from="0 15 15" to="360 15 15" dur="4s" repeatCount="indefinite"></animateTransform></path></svg>',
        sharepage: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-sharepage"><path style="stroke: none;" d="M3 1 h19 l5 5 v23 h-24z m1 1 v26 h22 v-21 h-5 v-5z M9 14 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z m12 -6 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z m0 12 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z m-12 -4 l12 6 v2 l-12 -6z l12 -6 v2 l-12 6z"></path></svg>',
        share: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-share"><path style="stroke: none;" d="M7 11 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m15 -8 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 16 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m-13 -4 l15 8 v2 l-15 -8z m0 -2 l15 -8 v2 l-15 8z"></path></svg>'
    }
    
    NotebookView.emathLogo = [
        '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" viewBox="0 0 370 141" height="200" width="520" style="margin-right: -100px;">',
        '  <g transform="translate(-239.78233,-125.43578)">',
        '    <path class="emath-textbook-logo-emath" d="m 239.79912,261.29803 c 0,-0.97275 6.3721,-1.09986 65.39063,-1.30428 23.33203,-0.0808 42.42187,0.19804 42.42187,0.61968 0,0.9027 -0.80156,0.91933 -61.17187,1.26952 -27.68501,0.16059 -46.64063,-0.0771 -46.64063,-0.58492 z m 109.21875,-0.85548 c -0.34412,-0.55679 0.75307,-0.9375 2.70185,-0.9375 1.80468,0 3.54197,0.42187 3.86065,0.9375 0.34412,0.55679 -0.75308,0.9375 -2.70185,0.9375 -1.80468,0 -3.54198,-0.42188 -3.86065,-0.9375 z m 10.29003,-0.0562 c 1.72535,-0.51482 3.13848,-0.48407 3.70312,0.0805 0.59247,0.59247 -0.38726,0.88481 -2.79003,0.83252 -3.65969,-0.0796 -3.6704,-0.0904 -0.91309,-0.91309 z m 12.44435,0.37279 c 0.64453,-0.26007 1.69921,-0.26007 2.34375,0 0.64453,0.26007 0.11718,0.47287 -1.17188,0.47287 -1.28906,0 -1.81641,-0.2128 -1.17187,-0.47287 z m 114.84375,-0.0759 c 3.99609,-0.17573 10.53515,-0.17573 14.53125,0 3.99609,0.17574 0.36942,1.21238 -7.62277,1.21238 -7.99219,0 -10.90458,-1.03664 -6.90848,-1.21238 z M 303.73911,227.39023 c -0.65304,-0.78687 -0.69539,-3.19648 -0.12769,-7.26563 1.30644,-9.36431 2.10053,-49.91425 1.00848,-51.49759 -1.44555,-2.09589 -4.19528,-0.30872 -9.2843,6.03429 -5.14635,6.41447 -8.66148,7.89429 -8.66148,3.64635 0,-4.67688 14.52938,-16.23592 18.73161,-14.90219 3.87803,1.23084 5.10096,5.63488 5.29274,19.06036 0.0981,6.87062 0.36054,12.67412 0.58309,12.89667 0.22256,0.22256 0.9352,-1.39104 1.58365,-3.58577 2.03906,-6.90131 9.14618,-20.73699 12.29013,-23.92562 4.12324,-4.18184 6.95673,-4.76719 9.86247,-2.03739 3.03141,2.84786 4.26778,7.68082 5.02143,19.62884 l 0.62092,9.84375 1.15932,-4.21875 c 5.6692,-20.63016 13.36378,-34.43077 20.14612,-36.13303 3.3537,-0.84172 6.95853,1.55572 8.73611,5.81009 0.91295,2.18499 1.83821,11.16548 2.7695,26.88021 1.9505,32.9136 3.07197,37.65234 8.46215,35.75671 2.02774,-0.71311 2.80075,-1.75081 3.37959,-4.53678 3.13745,-15.10068 25.50808,-40.01616 32.54672,-36.2492 3.72158,1.99173 3.02638,10.347 -0.86093,10.347 -1.71917,0 -1.84625,-0.39412 -1.27635,-3.95814 l 0.63294,-3.95814 -2.73123,1.73847 c -4.50669,2.86857 -6.87369,5.30441 -13.7211,14.12021 -10.15408,13.073 -13.87141,22.0576 -9.12622,22.0576 3.8667,0 20.63934,-18.33517 22.94888,-25.08678 0.83467,-2.44006 1.83831,-3.60334 3.299,-3.82374 1.99527,-0.30107 2.06958,-0.0918 1.49848,4.21875 -1.79512,13.5491 -1.82119,17.66344 -0.12085,19.07459 1.48074,1.2289 1.79979,1.15309 3.046,-0.72382 1.58241,-2.38323 3.97583,-2.79875 3.97583,-0.69025 0,2.19274 -3.50234,5.15625 -6.09375,5.15625 -3.49689,0 -5.9363,-3.04988 -6.63157,-8.29118 l -0.6123,-4.61564 -6.64036,6.99057 c -7.67369,8.0784 -13.64637,12.01135 -16.97283,11.17646 -1.24061,-0.31137 -3.42175,0.0592 -4.90689,0.83385 -3.58563,1.87008 -5.24446,1.7658 -8.13983,-0.51171 -4.16652,-3.27738 -5.74655,-12.83814 -7.11986,-43.08235 -0.45656,-10.05469 -1.28559,-19.45222 -1.84229,-20.8834 -2.35155,-6.04544 -6.48165,-3.50743 -11.63568,7.15029 -6.30366,13.03496 -12.5405,34.2844 -13.47794,45.92049 -0.61177,7.59352 -1.99796,10.36962 -4.66652,9.34559 -1.32312,-0.50773 -1.42513,-2.3368 -0.87357,-15.66329 0.95716,-23.12599 -1.42275,-40.86968 -5.48172,-40.86968 -2.09806,0 -5.51154,5.66531 -10.28408,17.0684 -5.73286,13.69761 -8.66027,23.32132 -9.94232,32.68493 -0.5945,4.34195 -1.52014,8.4237 -2.05699,9.07056 -1.24744,1.50308 -3.05848,1.50257 -4.30651,-9.4e-4 z m 154.79502,-0.0355 c -1.35433,-2.18791 -0.4638,-23.78118 1.53518,-37.22465 3.2508,-21.86209 5.83799,-30.14966 8.57088,-27.45519 0.91389,0.90104 0.73834,2.57186 -0.84157,8.00924 -2.23636,7.69668 -4.49479,21.05106 -5.3497,31.63345 l -0.56802,7.03125 4.24711,-7.96875 c 9.82441,-18.43331 15.84191,-16.28265 18.3617,6.5625 1.3398,12.14704 1.4811,12.50003 4.74244,11.84777 3.54719,-0.70944 4.34173,0.80826 1.424,2.72004 -5.48804,3.5959 -9.0826,-1.86607 -10.8211,-16.44281 -1.35538,-11.36437 -1.81676,-13.125 -3.43946,-13.125 -3.16763,0 -11.58005,17.57548 -13.56267,28.33553 -1.11119,6.03061 -2.82644,8.45526 -4.29879,6.07665 z m -17.92269,-2.58381 c -3.38407,-1.35787 -5.8295,-8.1602 -6.56818,-18.27036 l -0.65039,-8.90188 -3.40247,2.31636 c -1.87136,1.27399 -3.66433,2.05449 -3.98437,1.73445 -1.64963,-1.64963 -0.17442,-3.9432 3.99627,-6.21318 l 4.57818,-2.49175 0.67285,-6.283 c 0.80787,-7.54383 1.85617,-10.59398 3.64103,-10.59398 1.9537,0 2.33016,1.80271 1.4708,7.04302 -1.32483,8.07872 -1.21288,8.61844 1.69959,8.19423 3.22573,-0.46983 7.60257,1.14092 8.34296,3.07036 0.84751,2.20858 -0.34362,2.77604 -2.92602,1.39398 -1.663,-0.89001 -3.18015,-0.99123 -5.3892,-0.35954 l -3.07462,0.87921 -0.28525,7.93624 c -0.34285,9.53888 0.24015,13.1877 2.56602,16.06003 1.64123,2.02683 1.97023,2.08498 5.1965,0.91833 4.4632,-1.6139 5.40686,-0.57594 2.07983,2.28764 -2.79244,2.40346 -4.48529,2.67553 -7.96353,1.27988 z m -176.54227,-1.35925 c -5.56652,-3.02535 -5.89814,-11.58283 -0.83538,-21.55726 3.46186,-6.82042 11.06294,-15.78068 15.2031,-17.92164 8.3088,-4.29665 11.0072,4.67164 4.49519,14.94003 -3.69663,5.82897 -7.71552,8.26262 -13.08799,7.92546 l -4.17378,-0.26193 -1.2963,4.5415 c -2.35486,8.25011 0.53783,11.46992 8.23325,9.16432 4.09304,-1.22631 12.07565,-8.02957 15.02958,-12.80913 1.10149,-1.78224 2.39017,-3.24044 2.86374,-3.24044 3.03684,0 -2.16457,9.82768 -7.65324,14.46026 -6.67384,5.6329 -13.8321,7.44696 -18.77817,4.75883 z m 10.44323,-21.75419 c 3.88432,-2.67717 7.15417,-7.14257 8.39282,-11.4615 1.13832,-3.96908 -1.05394,-4.89201 -4.3353,-1.82513 -2.96979,2.77567 -10.6458,12.89875 -10.6458,14.03962 0,1.41477 4.12752,0.94302 6.58828,-0.75299 z m -28.61953,-74.40872 c -3.60938,-0.29567 -6.35156,-0.61407 -6.09375,-0.70757 1.10764,-0.40167 60.99048,-0.98443 99.375,-0.96708 56.80622,0.0257 53.83827,0.0298 80.15625,-0.10992 12.89063,-0.0684 36.51563,-0.009 52.5,0.13218 l 29.0625,0.25657 -26.25,0.49794 c -14.4375,0.27387 -65.27344,0.62467 -112.96875,0.77954 -47.69531,0.15488 -91.78125,0.36582 -97.96875,0.46875 -6.1875,0.10294 -14.20313,-0.0547 -17.8125,-0.35041 z m 256.23047,-1.5091 c 0.67675,-0.27082 1.48535,-0.23751 1.79688,0.074 0.31152,0.31153 -0.2422,0.5331 -1.23047,0.4924 -1.09214,-0.045 -1.31428,-0.26713 -0.56641,-0.5664 z m 3.75,0 c 0.67675,-0.27082 1.48535,-0.23751 1.79688,0.074 0.31152,0.31153 -0.2422,0.5331 -1.23047,0.4924 -1.09214,-0.045 -1.31428,-0.26713 -0.56641,-0.5664 z" fill="#000000" />',
        '    <text class="emath-textbook-logo-number" style="fill: {{booknumbercolor}}; text-shadow: 1px 1px 1px black; font-family: ArchitectsDaughter, Comic Sans MS, Purisa, serif; font-style: italic; letter-spacing: -7; font-size: 80px;" x="500px" y="225px">{{booknumber}}</text>',
        '  </g>',
        '</svg>'
    ].join('\n');
    
    /**
     * Html-templates
     */
    NotebookView.templates = {
        html: [
            '<div class="notebook-filterarea notebook-updragscroller"></div>',
            '<div class="notebook-toc"></div>',
            '<div class="notebook-controlarea ffwidget-background">',
            '    <div class="notebook-tocarea"></div>',
            '    <div class="notebook-naviarea notebook-updragscroller"></div>',
            '    <div class="notebook-edittogglearea"></div>',
            '    <div class="notebook-dropmenuarea"></div>',
            '</div>',
            '<div class="notebook-viewarea">',
            '    <div class="notebook-pagemodarea ffwidget-background"></div>',
            '    <div class="notebook-bookserieslogo"></div>',
            '    <div class="notebook-pagetitle"></div>',
            '    <div class="notebook-pageview"></div>',
            '</div>',
            '<div class="notebook-downdragscroller"></div>',
            '<div class="notebook-elementpanelarea"></div>',
            '<div class="notebook-mqpanelarea"></div>'
        ].join('\n'),
        onepage: [
            '<div class="notebook-filterarea notebook-updragscroller"></div>',
            '<div class="notebook-controlarea ffwidget-background">',
            '    <div class="notebook-updragscroller notebook-controlfiller"></div>',
            '    <div class="notebook-naviarea notebook-updragscroller"></div>',
            '    <div class="notebook-edittogglearea"></div>',
            '    <div class="notebook-dropmenuarea"></div>',
            '</div>',
            '<div class="notebook-viewarea">',
            '    <div class="notebook-bookserieslogo"></div>',
            '    <div class="notebook-pagetitle"></div>',
            '    <div class="notebook-pageview"></div>',
            '</div>',
            '<div class="notebook-downdragscroller"></div>',
            '<div class="notebook-elementpanelarea"></div>',
            '<div class="notebook-mqpanelarea"></div>'
        ].join('\n')
    };
    
    /**
     * Properties of different modes
     */
    NotebookView.modes = {
        view: {
            editable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            reviewable: false
        },
        review: {
            editable: false,
            reviewable: true
        }
    }
    
    /**
     * Properties of roles
     */
    NotebookView.roles = {
        student: {
            roleindex: 1,
            studentable: true,
            teacherable: false,
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-student"><path style="stroke: none;" d="M9 9 a6 6 0 0 1 12 0 a6 8 0 0 1 -12 0z m1 0 a5 7 0 0 0 5 7 a5 7 0 0 0 5 -7 l-1 1 l-5 -3 l-1 1 l-2 -1 a5 5 0 0 0 -1 2z m0 6 a6 6 0 0 0 10 0 a3 6 -30 0 1 5 5 a14 14 0 0 1 -20 0 a3 6 30 0 1 5 -5z" /></svg>'
        },
        teacher: {
            roleindex: 5,
            studentable: false,
            teacherable: true,
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-teacher"><path style="stroke: none;" d="M9 9 a6 6 0 0 1 12 0 a1 5 -1 0 0 1 8 a3 6 -25 0 1 5 8 a20 5 0 0 1 -6 2 l0 -6 l-6 2 l-6 -2 l0 6 a20 5 0 0 1 -6 -2 a3 6 25 0 1 5 -8 a1 5 1 0 0 1 -8z m1 0 a5 7 0 0 0 5 7 a5 7 0 0 0 5 -7 l-1 1 l-5 -3 l-1 1 l-2 -1 a5 5 0 0 0 -1 2z m2 6.5 a3 3 0 0 1 -2 2 a5 3 0 0 0 10 0 a3 3 0 0 1 -2 -2 a5 7 0 0 1 -6 0z" /><path class="icon-highlight" style="stroke: none;" fill="#0a0" d="M9.5 22 l5 2 l0 6 l-5 -2z m6 2 l5 -2 l0 6 l-5 2z m-10 0 l-2 -22 l1 0 l3 22z" /></svg>'
        }
    };
    
    /**
     * Some nonchanging menu elements
     */
    NotebookView.pagemenu = {
        menu: [
            {
                name: 'addpagemenu',
                type: 'toggle',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpage"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M11 5 l4 0 l0 4 l4 0 l0 4 l-4 0 l0 4 l-4 0 l0 -4 l-4 0 l0 -4 l4 0 l0 -4 z"></path><path stroke="none" fill="black" d="M13 20 l10 0 l0 4 l-10 0 l0 -4 z"></svg>',
                label: 'Add pages',
                event: 'addpagemenuon',
                event_off: 'addpagemenuoff',
                menu: [
                    {
                        name: 'addchapterafter',
                        type: 'click',
                        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpage"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M15 13 l4 0 l0 4 l4 0 l0 4 l-4 0 l0 4 l-4 0 l0 -4 l-4 0 l0 -4 l4 0 l0 -4 z"></path><path stroke="none" fill="black" d="M16 5.5 a 5 5 0 1 0 0 6 l-1 -1 a3.5 4 0 1 1 0 -4 z"></path></svg>',
                        label: 'Add a chapter',
                        event: 'addchapterafter'
                    },
                    {
                        name: 'addpageafter',
                        type: 'click',
                        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpage"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M15 13 l4 0 l0 4 l4 0 l0 4 l-4 0 l0 4 l-4 0 l0 -4 l-4 0 l0 -4 l4 0 l0 -4 z"></path></svg>',
                        label: 'Add a page',
                        event: 'addpageafter'
                    },
                    {
                        name: 'removepage',
                        type: 'click',
                        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpage"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M10 17 l12 0 l0 4 l-12 0 l0 -4 z"></path></svg>',
                        label: 'Remove page',
                        event: 'removepage'
                    }
                ]
            }
        ]
    };
    
    NotebookView.pagemodmenu = {
        menustyle: 'dropmenu',
        ffbackground: false,
        menu: [
            {
                name: 'addchapterafter',
                type: 'click',
                iconi: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpagechapter"><path stroke="none" fill="black" d="M4 2 h17 l5 5 v21 h-22 0z m1 1 v24 h20 v-19 h-5 v-5z m10 10 h4 v4 h4 v4 h-4 v4 h-4 v-4 h-4 v-4 h4 v-4 z m1 -7.5 a 5 5 0 1 0 0 6 l-1 -1 a3.5 4 0 1 1 0 -4 z"></path></svg>',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-pageaddchapter"><path stroke="none" fill="black" d="M4 2 h22 v20 a8 8 0 0 1 -6 6 h-16 0z m1 1 v24 h15 v-5 h5 v-19z m13 3 h2 v-2 h2 v2 h2 v2 h-2 v2 h-2 v-2 h-2z m-11 1 h2 v3 h2 v-3 h2 v8 h-2 v-3 h-2 v3 h-2z m8 6 h6 v2 h-6z m-7 4 h13 v1 h-13z m0 3 h13 v1 h-13z m0 8 v2 h21 v-25 h-3 v1 h2 v23 h-19 v-1z"></path></svg>',
                label: 'notebook:Add a chapter',
                event: 'addchapterafter'
            },
            {
                name: 'addpageafter',
                type: 'click',
                iconi: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-addpage"><path stroke="none" fill="black" d="M4 2 h17 l5 5 v21 h-22 0z m1 1 v24 h20 v-19 h-5 v-5z m10 10 h4 v4 h4 v4 h-4 v4 h-4 v-4 h-4 v-4 h4 v-4 z"></path></svg>',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-pageadd"><path stroke="none" fill="black" d="M4 2 h22 v20 a8 8 0 0 1 -6 6 h-16 0z m1 1 v24 h15 v-5 h5 v-19z m13 3 h2 v-2 h2 v2 h2 v2 h-2 v2 h-2 v-2 h-2z m-10 5 h12 v1 h-12z m0 3 h12 v1 h-12z m0 3 h12 v1 h-12z m0 3 h12 v1 h-12z"></path></svg>',
                label: 'notebook:Add a page',
                event: 'addpageafter'
            },
            {
                name: 'removepage',
                type: 'click',
                iconi: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-removepage"><path stroke="none" fill="black" d="M4 2 h17 l5 5 v21 h-22 0z m1 1 v24 h20 v-19 h-5 v-5z m6 14 h12 v4 h-12z"></path></svg>',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-pageremove"><path stroke="none" fill="black" d="M4 2 h22 v20 a8 8 0 0 1 -6 6 h-16 0z m1 1 v24 h15 v-5 h5 v-19z m12 5 h6 v-2 h-6 v2z m-9 3 h12 v1 h-12z m0 3 h12 v1 h-12z m0 3 h12 v1 h-12z m0 3 h12 v1 h-12z"></path></svg>',
                label: 'notebook:Remove this page',
                event: 'removepage'
            }
        ]
    }
    
    NotebookView.basemenu =
    NotebookView.filemenu =
    {
        menu: [
            {
                name: 'filemenu',
                type: 'toggle',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-folder"><path stroke="none" fill="black" d="M2 6 l5 0 l2 2 l14 0 l0 2 l-16 0 l-4 14 l5 -12 l22 0 l-5 13 l-23 0z"></path></svg>',
                label: 'File menu',
                event: 'filemenuon',
                event_off: 'filemenuoff',
                menu: [
                    {
                        name: 'import',
                        type: 'click',
                        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-import"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M1 15 l8 0 l0 -3 l5 5 l-5 5 l0 -3 l-8 0z"></path></svg>',
                        label: 'Import a notebook',
                        event: 'importnotebook'
                    },
                    {
                        name: 'export',
                        type: 'click',
                        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-export"><path stroke="none" fill="black" d="M4 2 l17 0 l 5 5 l0 21 l-22 0z"></path><path stroke="none" style="fill: white;" d="M5 3 l15 0 l0 5 l5 0 l0 19 l-20 0z"></path><path stroke="none" fill="black" d="M17 15 l8 0 l0 -3 l5 5 l-5 5 l0 -3 l-8 0z"></path></svg>',
                        label: 'Export the notebook',
                        event: 'exportnotebook'
                    }
                ]
            }
        ]
    };

    /**
     * Drop down menus
     */
    NotebookView.dropmenu = {
        menustyle: 'dropmenu',
        ffbacground: false,
        menu: [
            {
                // Editmode - elements
                name: 'edit',
                type: 'toggleclick',
                icon: NotebookView.icons.edit,
                icon_off: NotebookView.icons.view,
                label: 'Edit',
                label_off: 'Stop editing',
                event: 'editmodeon',
                event_off: 'editmodeoff',
            },
            {
                name: 'mainmenu',
                type: 'toggle',
                icon: NotebookView.icons.menu,
                label: 'Menu',
                event: 'mainmenuopen',
                event_off: 'mainmenuoff',
                menu: [
                    {
                        name: 'mqpanel',
                        type: 'click',
                        icon: NotebookView.icons.mathpanel,
                        label: 'Mathpanel',
                        event: 'togglemqpanel'
                    },
                    NotebookView.filemenu.menu[0]
               ]
            }
        ]
    };
    
    NotebookView.dropmenuwithconfig = {
        menustyle: 'dropmenu',
        ffbacground: false,
        menu: [
            {
                // Editmode - elements
                name: 'edit',
                type: 'toggleclick',
                icon: NotebookView.icons.edit,
                icon_off: NotebookView.icons.view,
                label: 'Edit',
                label_off: 'Stop editing',
                event: 'editmodeon',
                event_off: 'editmodeoff',
            },
            {
                name: 'mainmenu',
                type: 'toggle',
                icon: NotebookView.icons.menu,
                label: 'Menu',
                event: 'mainmenuopen',
                event_off: 'mainmenuoff',
                menu: [
                    {
                        name: 'settingsdialog',
                        type: 'click',
                        icon: NotebookView.icons.settings,
                        label: 'Settings',
                        event: 'configdialogopen'
                    },
                    {
                        name: 'mqpanel',
                        type: 'click',
                        icon: NotebookView.icons.mathpanel,
                        label: 'Mathpanel',
                        event: 'togglemqpanel'
                    },
                    NotebookView.filemenu.menu[0]
                ]
            }
        ]
    };
    
    /**
     * CSS-styles for NotebookView
     */
    NotebookView.styles = [
        '.notebookview-wrapper {background-color: #ccc; overflow: hidden; position: relative; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; -webkit-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        //'.notebookview-wrapper .notebook-filterarea {min-height: 30px; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0;  flex-shrink: 0; background-color: #ffa;}',
        '.notebookview-wrapper .notebook-menuarea {min-height: 40px; padding: 5px; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        // Controlarea
        '.notebookview-wrapper .notebook-controlarea {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; -webkit-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        // Dropmenu
        '.notebook-dropmenuarea.menubar-wrapper {position: relative; z-index: 10; padding: 10px 5px;}',
        '.notebook-dropmenuarea svg.mini-icon {width: 30px; height: 30px;}',
        // Editbutton
        '.notebook-edittogglearea {position: relative; z-index: 10; margin: 7px 5px;}',
        '.notebook-edittogglearea svg.mini-icon {width: 30px; height: 30px;}',
        // TOC button
        '.notebook-tocarea.menubar-wrapper {position: relative; z-index: 10; padding: 10px 5px;}',
        '.notebook-tocarea svg.mini-icon {width: 30px; height: 30px;}',
        // TOC
        '.notebook-tocarea {-webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0;}',
        '.notebook-tocicon {display: inline-block; padding: 8px; cursor: pointer;}',
        // Naviarea
        '.notebookview-wrapper .notebook-naviarea {min-height: 20px; -webkit-flex-grow: 01; -ms-flex-grow: 1; flex-grow: 1; -webkit-flex-shrink: 1; -ms-flex-shrink: 1; flex-shrink: 1; z-index: 11;}',
        '.notebookview-wrapper.onepagemode .notebook-naviarea {display: none;}',
        '.notebookview-wrapper .notebook-naviarea .navbar-wrapper {border: none;}',
        '.notebookview-wrapper .notebook-viewarea {-webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; position: relative; margin: 0 8px 8px; padding: 20px 0 20px 0; border: 1px solid #777; box-shadow: 5px 5px 8px rgba(0,0,0,0.5); overflow: auto; background-color: white; overflow-y: scroll; padding: 0.5em;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pageview {max-width: 60em; margin: 0 auto 5em auto;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle {max-width: 60em; margin: 0 auto; border-bottom: 3px double black; position: relative;}',
        '.notebook-pagetitle h1 {margin: 0.5em 0 0.1em; padding-left: 0.7em;}',
        '.notebook-pagetitle h1 span.notebookview-titletext {margin-left: 0.5em; margin-right: 0.5em;}',
        '.notebook-viewarea[data-zoomlevel="5"] {zoom: 0.5;}',
        '.notebook-viewarea[data-zoomlevel="6"] {zoom: 0.6;}',
        '.notebook-viewarea[data-zoomlevel="7"] {zoom: 0.7;}',
        '.notebook-viewarea[data-zoomlevel="8"] {zoom: 0.8;}',
        '.notebook-viewarea[data-zoomlevel="9"] {zoom: 0.9;}',
        '.notebook-viewarea[data-zoomlevel="10"] {zoom: 1;}',
        '.notebook-viewarea[data-zoomlevel="11"] {zoom: 1.1;}',
        '.notebook-viewarea[data-zoomlevel="12"] {zoom: 1.2;}',
        '.notebook-viewarea[data-zoomlevel="13"] {zoom: 1.3;}',
        '.notebook-viewarea[data-zoomlevel="14"] {zoom: 1.4;}',
        '.notebook-viewarea[data-zoomlevel="15"] {zoom: 1.5;}',
        '.notebook-viewarea[data-pagelevel="0"] .notebook-pagetitle h1 {font-size: 300%}',
        '.notebook-viewarea[data-pagelevel="1"] .notebook-pagetitle h1 {font-size: 220%}',
        '.notebook-viewarea[data-pagelevel="2"] .notebook-pagetitle h1 {font-size: 180%}',
        '.notebook-viewarea[data-pagelevel="3"] .notebook-pagetitle h1 {font-size: 150%}',
        //'.notebook-viewarea[data-pagetypes~="noheader"] .notebook-pagetitle h1 .notebookview-titletext {display: none;}',
        '.notebook-pagetitle h1 input.notebookview-titletext {font-size: 130%; width: 94%; -webkit-box-sizing: border-box; -moz-box-sizing: border-box; box-sizing: border-box;}',
        '.notebook-headimage-metainfo {position: absolute; right: 2px; bottom: 2px; background-color: rgba(0,0,0,0.2); border-radius: 50%; width: 24px; height: 24px; padding: 2px; display: inline-block; line-height: 24px; text-align: center; z-index: 50; opacity: 0; transition: opacity 0.2s;}',
        '.notebook-pagetitle:hover .notebook-headimage-metainfo {opacity: 1;}',
        // Pagemodarea
        '.notebook-pagemodarea {display: none; border-top: none; border-radius: 0 0 8px 0; position: absolute; z-index: 5; left: 0; top: 0; border-right: 1px solid #777; border-bottom: 1px solid #777; box-shadow: 1px 1px 1px rgba(0,0,0,0.3);}',
        '.notebook-pagemodarea.notebook-pagemodable {display: block;}',
        '.notebook-pagemodarea.notebook-pagemodable + div + .notebook-pagetitle {margin-top: 20px;}',
        // Scroller areas
        '.notebookview-wrapper .notebook-updragscroller {}',
        '.notebookview-wrapper .notebook-controlfiller {-webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1;}',
        '.notebookview-wrapper .notebook-downdragscroller {height: 25px; margin-top: -8px; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0; position: relative;}',
        // View area
        '',
        '.notebookview-wrapper .notebook-viewarea {font-family: Helvetica, sans-serif;}',
        '.nottranslated.ebook-pageelement {box-shadow: 0 0 6px 1px #faa; border-radius: 3px; background-color: #fffafa;}',
        '@font-face {font-family: OpenDyslexic; src: url(./css/fonts/OpenDyslexic-Regular.otf) format("opentype");}',
        '.notebookview-wrapper.usedyslexiafont .notebook-viewarea, .notebookview-wrapper.usedyslexiafont .notebook-viewarea .notebook-pagetitle, .notebookview-wrapper.usedyslexiafont .notebook-viewarea .mathquill-rendered-math, .notebookview-wrapper.usedyslexiafont .notebook-viewarea .sdeditorwrapper div.sdeditor {font-family: OpenDyslexic;}',
        '.notebookview-wrapper[textsize="-3"] .notebook-viewarea {font-size: 85%;}',
        '.notebookview-wrapper[textsize="-2"] .notebook-viewarea {font-size: 90%;}',
        '.notebookview-wrapper[textsize="-1"] .notebook-viewarea {font-size: 95%;}',
        '.notebookview-wrapper[textsize="1"] .notebook-viewarea {font-size: 110%;}',
        '.notebookview-wrapper[textsize="2"] .notebook-viewarea {font-size: 115%;}',
        '.notebookview-wrapper[textsize="3"] .notebook-viewarea {font-size: 120%;}',
        
        // Elementbox and titles
        '.notebook-viewarea .ebook-elementbox h1.ebook-elementbox-title .ebook-elementbox-boxnumber {display: block; font-size: 80%; font-weight: normal; text-decoration: none;}',
        '.notebook-viewarea .ebook-elementbox h1.ebook-elementbox-title .ebook-elementbox-boxnumber:empty {display: none;}',
        '.ebook-elementbox-boxnumberplace {display: none;}',
        '.notebookview-wrapper.usenumbering .ebook-elementbox-boxnumberplace {display: inline;}',
        
        // Assignmentlists
        '.ebook-assignmentindent {list-style: none;}',
        
        // Title and headimage
        '.notebookview-wrapper .notebook-bookserieslogo {display: none; text-align: center;}',
        '.notebookview-wrapper .notebook-viewarea[data-pagetypes="front"] .notebook-bookserieslogo {display: block; margin-top: 5px;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype="frontimage"] {margin: 0 auto; width: 640px; height: 640px; position: relative; border-bottom: none; margin-top: 0; padding: 0; background-position: center center;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype="fullheadimage"] {width: 640px; height: 640px; position: relative; border-bottom: none; margin-top: 1em; padding: 0; background-position: center center;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype="headimage"] {margin-top: 0.5em; margin-bottom: 1.5em; border-bottom: 1px solid black; margin-top: 1em; padding: 0; background-position: center center;}',
        '.notebookview-wrapper .notebook-viewarea[data-pagelevel="1"] .notebook-pagetitle[data-headimagetype="headimage"] {height: 150px; position: relative;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype] h1 {color: white; background-color: rgba(0,0,0,0.6)!important; border-radius: 0.3em; padding: 0.3em 0.5em; display: inline-block; margin: 0.1em 1em; text-shadow: 3px 3px 5px black;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype="fullheadimage"] h1 {margin: 50px 20px;}',
        '.notebookview-wrapper .notebook-viewarea .notebook-pagetitle[data-headimagetype="frontimage"] h1 {display: none;}',
        '.notebookview-wrapper .notebook-viewarea[data-pagelevel="1"] .notebook-pagetitle[data-headimagetype="headimage"] h1 {position: absolute; left: 0.5em; bottom: 0.5em;}',
        
        '.notebook-viewarea .notebook-pagetitle .notebookview-iconbar {position: absolute; right: 5px; top: 5px; background-color: rgba(230,230,230,0.8); border: 1px solid #aaa; border-radius: 3px; display: none;}',
        '.notebook-viewarea .notebook-pagetitle:hover .notebookview-iconbar {display: block;}',
        '.notebook-viewarea .notebook-pagetitle .notebookview-pagecopy {cursor: move; display: inline-block;}',
        '.notebook-viewarea .notebook-pagetitle .notebookview-pagelink {cursor: move; display: inline-block;}',
        
        // Elementpanel (hidden)
        '.notebook-elementpanelarea {overflow: hidden; width: 0; height: 0; border: none; margin: 0; padding: 0;}',
        
        // Mathquill-fix
        // Show non-math text in editable textboxes with Times New Roman instead of Symbola. (Scandinavic letters in Mac!)
        '.mathquill-editable.mathquill-textbox > span:not(.mathquill-rendered-math) {font-family: "Times New Roman";}',
        
        // Print css
        '@media print {',
            '.notebookview-wrapper {font-size: 9pt;}',
            '.notebookview-wrapper .notebook-filterarea, .notebookview-wrapper .notebook-naviarea, .notebookview-wrapper .notebook-menuarea, .notebookview-wrapper .notebook-downdragscroller {display: none;}',
            '.notebookview-wrapper {position: static!important; display: block!important; background-color: transparent!important;}',
            '.notebookview-wrapper .notebook-controlarea {display: none;}',
            '.notebookview-wrapper .notebook-viewarea {border: none!important; box-shadow: none!important; background-color: white; overflow: visible!important;}',
            '.notebookview-wrapper .notebook-viewarea h1, .notebookview-wrapper .notebook-viewarea h2, .notebookview-wrapper .notebook-viewarea h3, .notebookview-wrapper .notebook-viewarea h4, .notebookview-wrapper .notebook-viewarea h5, .notebookview-wrapper .notebook-viewarea h6 {page-break-after: avoid;}',
            '.notebook-viewarea .ebook-elementbox {page-break-inside: avoid;}',
            '.notebook-pagetitle {page-break-after: avoid;}',
            '.notebook-viewarea button {display: none;}',
        '}'
    ].join('\n');
    
    /**
     * Localization strings
     */
    NotebookView.localization = {
        "en": {
            "notebook:change role": "Change your role",
            "notebook:Return to own role": "Return back to your own role",
            "notebook:role_teacher": "Teacher",
            "notebook:role_student": "Student",
            "notebook:configs": "Settings",
            "notebook:toc": "Table of Contents",
            "notebook:view": "View",
            "notebook:edit": "Edit",
            "notebook:sharing_page": "Sharing the page to an online service.",
            "notebook:share_error": "Error in sharing. Please try again.",
            "notebook:page_shared": "Page shared in address: ",
            "notebook:sharethepage": "Share the current page.",
            "notebook:theming": "Theme",
            "notebook:dyslexiafont": "Dyslexiafont",
            "notebook:boxnumbering": "Box numbering",
            "notebook:author": "Author",
            "notebook:license": "License",
            "notebook:source": "Source",
            "notebook:notes": "Notes",
            "notebook:want_refresh": "The content of this page has changed. Do you want to refresh the page now?",
            "notebook:Add a chapter": "Add a chapter",
            "notebook:Add a page": "Add a page",
            "notebook:Remove this page": "Remove this page"
        },
        "fi": {
            "notebook:change role": "Vaihda roolia",
            "notebook:Return to own role": "Palaa omaan rooliisi",
            "notebook:role_teacher": "Opettaja",
            "notebook:role_student": "Opiskelija",
            "notebook:configs": "Asetukset",
            "notebook:toc": "Sisällysluettelo",
            "notebook:view": "Näytä",
            "notebook:edit": "Muokkaa",
            "notebook:sharing_page": "Jaetaan sivu online-palveluun.",
            "notebook:share_error": "Virhe jakamisessa. Ole hyvä ja yritä uudelleen.",
            "notebook:page_shared": "Sivu jaettu osoitteeseen: ",
            "notebook:sharethepage": "Jaa nykyinen sivu.",
            "notebook:theming": "Teema",
            "notebook:dyslexiafont": "Dysleksiafontti",
            "notebook:boxnumbering": "Laatikoiden numerointi",
            "notebook:author": "Tekijä",
            "notebook:license": "Lisenssi",
            "notebook:source": "Lähde",
            "notebook:notes": "Huomioita",
            "notebook:want_refresh": "Tämän sivun sisältö on muuttunut. Haluatko päivittää sen nyt?",
            "notebook:Add a chapter": "Lisää kappale",
            "notebook:Add a page": "Lisää sivu",
            "notebook:Remove this page": "Poista tämä sivu"
        },
        "sv": {
            "notebook:change role": "Ändra din roll",
            "notebook:Return to own role": "Återgå till din egen roll",
            "notebook:role_teacher": "Lärare",
            "notebook:role_student": "Student",
            "notebook:configs": "Inställningar",
            "notebook:toc": "Innehållsförteckning",
            "notebook:view": "Visa",
            "notebook:edit": "Redigera",
            "notebook:share_error": "Problem med delningen. Försök igen senare.",
            "notebook:sharing_page": "Dela sidan via internet.",
            "notebook:page_shared": "Sidan finns på adressen: ",
            "notebook:sharethepage": "Dela den nuvarande sidan",
            "notebook:theming": "Tema",
            "notebook:dyslexiafont": "Dyslexifont",
            "notebook:boxnumbering": "Numrering av faktarutor",
            "notebook:author": "Författare",
            "notebook:license": "Licens",
            "notebook:source": "Källa",
            "notebook:notes": "Anmärkningar",
            "notebook:want_refresh": "Inehållet på denna sida har ändrats. Vill du ladda om sidan nu?",
            "notebook:Add a chapter": "Lägg till kapitel",
            "notebook:Add a page": "Lägg till sida",
            "notebook:Remove this page": "Ta bort den här sidan"
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(NotebookView.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(NotebookView.localization);
    }
    
    /*****************************************************************
     ****************************************************************/
    
    /**
     * Notebook class
     * @constructor
     * @param {Object} options - initing options of a Notebook
     */
    var Notebook = function(options){
        if (options.data && options.data.content && options.data.content.frontpage) {
            var common = options.data.content;
            options.data.content = {
                common: common
            };
        };
        options = $.extend(true, {}, Notebook.defaults, options);
        this.data = options.data;
        this.metadata = options.metadata;
        this.settings = options.settings;
        var lang = this.settings.lang;
        this.settings.lang = this.data.defaultlang;
        this.setLang(lang);
        this.refdata = $.extend(true, {}, Notebook.refdefaults, this.data.refdata);
        this.bookid = this.data.bookid || options.name || this.newBookId();
        this.booktitle = this.data.title || this.data.titles;
        if (!this.data.toc.data.firstpage || !this.data.toc.data.firstpage.chapid) {
            this.initFront();
        };
        this.initToc();
        this.initIndex();
        this.setAllTitles();
    };
    
    /**
     * Create an empty frontpage
     */
    Notebook.prototype.initFront = function() {
        this.data.toc.data.firstpage = {
            chapid: 'front',
            content: this.bookid + '_frontpage'
        };
        if (typeof(this.data.toc.metadata) === 'undefined') {
            this.data.toc.metadata = {};
            this.data.toc.metadata.creator = this.settings.username || 'Anonymous';
            this.data.toc.metadata.created = (new Date()).getTime();
        };
        this.metadata.modified  = this.data.toc.metadata.modified = (new Date()).getTime();
        this.metadata.modifier = this.data.toc.metadata.modifier = this.settings.username || 'Anonymous';
    };
    
    /**
     * Init the toc datastructures from this.data.toc
     */
    Notebook.prototype.initToc = function(){
        var toc = this.data.toc;
        this.pages = new NotebookPage(toc.data.firstpage, null, 0);
        this.makeLinear();
        this.makeById();
    }
    
    /**
     * Init index
     */
    Notebook.prototype.initIndex = function() {
        this.index = new IndexTool({indexdata: this.data.index, pageorder: this.getPagelist()});
    }
    
    /**
     * Get the data of the notebook as an object.
     * @param {Boolean} isexport - If this is export, generate a new bookid/name
     */
    Notebook.prototype.getData = function(isexport){
        var bookid = (isexport ? this.newBookId() : this.bookid);
        var data = {
            type: 'book',
            name: bookid,
            metadata: $.extend(true, {
                creator: this.settings.username,
                created: (new Date() + ''),
                modifier: this.settings.username,
                modified: (new Date() + ''),
                tags: []
            }, this.metadata),
            data: {
                bookid: bookid,
                titles: $.extend(true, {}, this.booktitle),
                booktype: ['notebook'],
                toc: this.getTocData(),
                pagetitles: this.getPagetitles(),
                index: this.getIndexData(bookid),
                bookinfo: {
                    author: $.extend([], this.data.bookinfo && this.data.bookinfo.author, this.data.author),
                    secondaryauthor: $.extend([], this.data.bookinfo && this.data.bookinfo.secondaryauthor, this.data.secondaryauthor),
                    vendorURL: this.data.bookinfo.vendorURL || this.data.vendorURL || '',
                    vendorLogo: this.data.bookinfo.vendorLogo || this.data.vendorLogo || ''
                }
            }
        };
        var btype;
        if (typeof(this.data.booktype) !== 'undefined') {
            for (var i = 0, len = this.data.booktype.length; i < len; i++){
                btype = this.data.booktype[i];
                if (btype !== 'notebook') {
                    data.data.booktype.push(btype);
                };
            };
        }
        if (typeof(this.data.curriculum) !== 'undefined') {
            data.data.curriculum = this.data.curriculum;
        };
        data.data.defaultlang = this.data.defaultlang || 'common';
        if (typeof(this.data.langs) !== 'undefined') {
            data.data.langs = $.extend(true, [], this.data.langs);
        };
        data.data.content = $.extend(true, {}, this.data.content);
        data.data.refdata = this.getRefdata();
        return data;
    };
    
    /**
     * Get authordata
     * @returns {Object} an object with data about authors
     */
    Notebook.prototype.getAuthors = function() {
        var result = {
            author: JSON.parse(JSON.stringify(this.data.bookinfo.author || this.data.author)),
            secondaryauthor: JSON.parse(JSON.stringify(this.data.bookinfo.secondaryauthor || this.data.secondaryauthor)),
            vendorURL: this.data.bookinfo.vendorURL || this.data.vendorURL,
            vendorLogo: this.data.bookinfo.vendorLogo || this.data.vendorLogo
        };
        return result;
    };
    
    /**
     * Get copyrightinfo
     * @returns {Object} an object with data about authors
     */
    Notebook.prototype.getCopyrightInfo = function() {
        var lang = this.settings.lang;
        var result = JSON.parse(JSON.stringify(this.data.bookinfo && this.data.bookinfo.copyright || {}));
        if (result.ownerlogo) {
            var logodata = this.getContentById(result.ownerlogo, 'common');
            if (logodata.type === 'imageelement') {
                result.ownerlogodata = logodata.data.content;
            };
        };
        return result;
    };
    
    /**
     * Get info about title, shorttitle, etc.
     * @param {String} lang    The language
     * @returns {Object}  The titleinfo: {}
     */
    Notebook.prototype.getTitleinfo = function(lang) {
        var result = {
            title: this.data.title[lang] || '',
            shorttitle: this.data.shorttitle[lang] || '',
            shorttitlefill: this.data.shorttitlefill[lang] || '',
            fronttitle: this.getPageTitle(this.pagesLinear[0].chapid, lang)
        };
        return result;
    }
    
    /**
     * Get toc-data
     */
    Notebook.prototype.getTocData = function(){
        var toc = {
            name: this.getBookId() + '_toc',
            type: 'toc',
            metadata: $.extend(true, {
                creator: this.settings.username,
                created: (new Date() + ''),
                modifier: this.settings.username,
                modified: (new Date() + ''),
                tags: []
            }, this.data.toc.metadata),
            data: {}
        };
        toc.data.firstpage = this.pages.getData();
        return toc;
    };
    
    /**
     * Get pagetitles
     */
    Notebook.prototype.getPagetitles = function(){
        return $.extend(true, {}, this.data.pagetitles);
    };
    
    /**
     * Get index data
     * @param {String} bookid - optional bookid
     * @returns {Object} data of the index
     */
    Notebook.prototype.getIndexData = function(bookid) {
        return this.index.getData(bookid);
    };
    
    /**
     * Get index data by type
     * @param {String} eltype - type of elements we are looking index for
     * @returns {Array} a list of index data of elements of given type
     */
    Notebook.prototype.getIndexDataByType = function(eltype) {
        return this.index.getElementsByType(eltype);
    }
    
    /**
     * Make linear list of pages.
     */
    Notebook.prototype.makeLinear = function(){
        this.pagesLinear = this.pages.getLinear();
    };
    
    /**
     * Make a pagemap by id
     */
    Notebook.prototype.makeById = function(){
        var pagesById = {};
        var page;
        for (var i = 0, len = this.pagesLinear.length; i < len; i++){
            page = this.pagesLinear[i];
            pagesById[page.chapid] = page;
        }
        this.pagesById = pagesById;
    }
    
    /**
     * Test if the page exists
     * @param {String} chapid - id of the page
     */
    Notebook.prototype.pageExists = function(chapid){
        return !!this.pagesById[chapid];
    }
    
    /**
     * Get all pages of the notebook in linear order for navbar.
     * @returns {Object} data suitable for navbar.
     */
    Notebook.prototype.getAllLinear = function(){
        var toc = this.data.toc;
        var linData = {
            data: [],
            ticklevel: 1,
            labeldown: true
        };
        linData.data = this.pages.getNavdata(0);
        linData.current = linData.data[0].id;
        return linData;
    }
    
    /**
     * Get the linear list of all (or filtered) pages in format suitable for navbar.
     * @param {String} filter - how to filter the pages (TODO: list of multiple filters?)
     */
    Notebook.prototype.getNavdata = function(filter){
        filter = filter || 'all';
        var data;
        switch (filter) {
            case 'all':
                data = this.getAllLinear();
                break;
            default:
                data = this.getAllLinear();
        }
        return data;
    };
    
    /**
     * Get the list of available langs
     * @returns {Array} - array of langs, including 'common'
     */
    Notebook.prototype.getLangs = function() {
        result = this.data.langs && this.data.langs.slice() || [];
        result.push('common');
        return result;
    }
    
    /**
     * Get the default language of the notebook
     * @returns {String} the default language
     */
    Notebook.prototype.getDefaultlang = function() {
        return this.data.defaultlang;
    };
    
    /**
     * Get the id of the book
     */
    Notebook.prototype.getBookId = function(){
        return this.bookid || '';
    }
    
    /**
     * Get chapid of the first page
     * @returns {String} chapid of the first page
     */
    Notebook.prototype.getFirstPage = function() {
        return this.pages.getId();
    }
    
    /**
     * Get the title of the book
     */
    Notebook.prototype.getBookTitle = function(){
        return this.settings.lang && this.booktitle[this.settings.lang] || this.booktitle.common;
    }
    
    /**
     * Get all language versions of the title of the book
     */
    Notebook.prototype.getBookTitlesAll = function(){
        return $.extend(true, {}, this.booktitle);
    }

    /**
     * Set the booktitle
     * @param {Object} titledata - of format: {common: 'title', fi: 'title', ...}
     */
    Notebook.prototype.setBookTitle = function(titledata) {
        this.booktitle = $.extend(true, this.booktitle, titledata);
    };
    
    /**
     * Get the data of bookserie
     * @returns {Object} the data of the bookserie 
     */
    Notebook.prototype.getBookSerie = function(lang) {
        var bookserie = this.data.bookinfo && this.data.bookinfo.bookserie || {};
        var seriename = bookserie.name || '';
        if (!seriename) {
            // If there is no seriename, but the bookid is between 'readerbook-1' and 'readerbook-42',
            // then use 'eMath' as the seriename.
            var bookidnum = this.getBookId().replace(/^readerbook-/, '') | 0;
            if (0 < bookidnum && bookidnum < 43) {
                seriename = 'eMath';
            };
        };
        var shorttitle = this.data.shorttitle && this.data.shorttitle[lang] || '';
        var serienumber = bookserie.number || shorttitle.replace(/[^0-9]/g, '') | 0 || '';
        var color = bookserie.color || this.data.shorttitlefill && this.data.shorttitlefill[lang] || '';
        var serielogo = bookserie.logo || '';
        var result = {
            name: seriename,
            number: serienumber,
            color: color,
            logo: serielogo
        };
        return result;
    };
    
    
    /**
     * Get the savemode of current book.
     */
    Notebook.prototype.getSavemode = function(){
        return this.data.savemode;
    }
    
    /**
     * Get the pagelist in linear order
     * @returns {Array} a list of all pageid's in linear order
     */
    Notebook.prototype.getPagelist = function() {
        return this.pages.getPagelist();
    };
    
    /**
     * Get the level of the page
     * @param {String} chapid - the id if the page
     * @returns {Number} the number of the level (0 === frontpage, 1 === chapter, 2 === section, ...)
     */
    Notebook.prototype.getPageLevel = function(chapid) {
        var page = this.pagesById[chapid];
        return page.level;
    };
    
    /**
     * Get the title of a page wit chapid.
     * @param {String} chapid - id of the page.
     * @param {String} lang - language of the title
     */
    Notebook.prototype.getPageTitle = function(chapid, lang){
        lang = lang || this.settings.lang;
        var deflang = this.data.defaultlang;
        var title = lang && this.data.pagetitles[lang] &&
                    this.data.pagetitles[lang][chapid] ||
                    deflang && this.data.pagetitles[deflang] && this.data.pagetitles[deflang][chapid] ||
                    this.data.pagetitles.common[chapid] || '';
        return title;
    };
    
    /**
     * Get the pagenumber as an array
     * @param {String} chapid - id of the page
     */
    Notebook.prototype.getPageNumber = function(chapid) {
        var cnumlist = [], parent, parentcid, page;
        parent = this.pagesById[chapid];
        while (parent && parent.level > 0 && !parent.unnumbered) {
            page = parent;
            parentcid = page.parent;
            parent = this.pagesById[parentcid];
            cnumlist.unshift(parent.indexOf(page.chapid) + 1);
        };
        return cnumlist;
    };
    
    /**
     * Get list of pagetypes
     * @param {String} chapid   The chapter id of the page
     * @return {Array} An array of pagetypes (strings)
     */
    Notebook.prototype.getPageTypes = function(chapid) {
        var result = [], page;
        if (chapid) {
            page = this.pagesById[chapid];
            result = page.getTypes();
        };
        return result;
    };
    
    /**
     * Get the headimage of the page
     * @param {String} chapid  the chapter id of the page
     * @returns {Object}  The headimage as {type: <headimage|fullheadimage>, text: <elemid>, data: <hidata>}
     */
    Notebook.prototype.getHeadimage = function(chapid) {
        var result = {}, tmpdata = {level: 1}, page, parent, i = 20;
        if (chapid) {
            page = this.pagesById[chapid];
            result = page.getHeadimage();
            result.text = (result.level === 0 ? result.frontimage : (result.level < 2 && result.subheadimage ? result.subheadimage : result.headimage));
            while (tmpdata.level > 0 && !result.text && result.parent && i > 0) {
                i--;
                parent = this.pagesById[result.parent];
                tmpdata = parent.getHeadimage();
                result.text = (result.level < 2 && tmpdata.subheadimage ? tmpdata.subheadimage : tmpdata.headimage);
                result.parent = tmpdata.parent;
            };
            result.data = (result.text ? this.getElementData({name: result.text, type: 'content'}) : {});
        };
        return result;
    };
    
    /**
     * Get the content data of the page.
     * @param {String} chapid - the id of the page
     * @param {String} lang - the asked lang (optional)
     */
    Notebook.prototype.getPageData = function(chapid, lang){
        var page = this.pagesById[chapid];
        var contentid = page.content;
        var data = this.getContentById(contentid, lang);
        return data;
    };
    
    /**
     * Get content data by content id
     * @param {String} cid - the content id of the content
     * @param {String} lang - the asked lang (optional)
     */
    Notebook.prototype.getContentById = function(cid, lang){
        lang = lang || this.settings.lang;
        var deflang = this.data.defaultlang;
        var pagedata = (lang && this.data.content[lang] && this.data.content[lang][cid]) ||
            (deflang && this.data.content[deflang] && this.data.content[deflang][cid]) ||
            (this.data.content['common'] && this.data.content['common'][cid]);
        return $.extend(true, {}, pagedata);
    }
    
    /**
     * Get data of a single element
     * @param {Object} item - description of the element to get {name: "<elementid>", type: "<toc|pagetitles|booktitles|index|content>"}
     */
    Notebook.prototype.getElementData = function(item){
        var result;
        var lang = item.lang || this.settings.lang;
        switch (item.type){
            case 'toc':
                result = this.getTocData();
                break;
            case 'index':
                var allindex = this.getIndexData();
                result = {
                    type: 'index',
                    name: this.bookid + '_index',
                    metadata: {lang: lang},
                    data: allindex[lang] || {}
                };
                break;
            case 'pagetitles':
                var alltitles = this.getPagetitles();
                result = {
                    type: 'pagetitles',
                    name: this.bookid + '_pagetitles',
                    metadata: {lang: lang},
                    data: alltitles[lang] || {}
                };
                break;
            case 'booktitles':
                var alltitles = this.getBookTitlesAll();
                result = {
                    type: 'booktitles',
                    name: this.bookid + '_booktitles',
                    metadata: {lang: lang},
                    data: {}
                };
                result.data[lang] = alltitles[lang];
                break;
            default:
                result = this.getContentById(item.name, item.lang);
                break;
        }
        return result;
    }
    
    /**
     * Get the content id of the page
     * @param {String} chapid - the id of the page
     */
    Notebook.prototype.getPageId = function(chapid){
        var page = this.pagesById[chapid];
        var contentid = page.content;
        return contentid;
    };

    /**
     * Set the title of the page
     * @param {String} chapid - The id of the page
     * @param {String} title - The title of the page
     * @param {String} lang - The language of the title
     */
    Notebook.prototype.setPageTitle = function(chapid, title, lang) {
        lang = lang || this.settings.lang;
        var deflang = this.data.defaultlang;
        if (typeof(this.data.pagetitles[lang]) === 'undefined') {
            this.data.pagetitles[lang] = {};
        };
        var titlemap = this.data.pagetitles[lang] || this.data.pagetitles[deflang] || this.data.pagetitles.common;
        // Update pagetitles map
        titlemap[chapid] = title;
        // Update pages datastructure
        this.pagesById[chapid].setTitle(title);
    };
    
    /**
     * Set the titles of all pages from the titlemap.
     */
    Notebook.prototype.setAllTitles = function() {
        var titlemap = this.data.pagetitles[this.settings.lang] || this.data.pagetitles.common;
        var page, title;
        for (var i = 0, len = this.pagesLinear.length; i < len; i++) {
            page = this.pagesLinear[i];
            title = titlemap[page.chapid];
            page.setTitle(title);
        };
    };
    
    /**
     * Change the content data of the page
     * @param {String} elname - The name of the content element.
     * @param {Object} eldata - The data object of the element
     */
    Notebook.prototype.setData = function(elname, eldata) {
        // Use the language in element's metadata, if it exists. Otherwise use lang from settings or the default language.
        var lang = eldata.metadata && eldata.metadata.lang || this.settings.lang;
        var deflang = this.data.defaultlang;
        if (typeof(lang) === 'undefined') {
            lang = deflang;
        };
        if (!this.data.content[lang]) {
            this.data.content[lang] = {};
        };
        this.data.content[lang][elname] = eldata;
    };
    
    /**
     * Set the current language
     * @param {String} lang - current language to use
     */
    Notebook.prototype.setLang = function(lang){
        if (this.data.langs.indexOf(lang) > -1) {
            this.settings.lang = lang;
        } else {
            this.settings.lang = this.data.defaultlang;
        };
        return this.settings.lang;
    };
    
    /**
     * Add an empty chapter after chapter of the given page.
     * If the page is frontpage, add a chapter page at begining of the book.
     * When adding pages, use always defaultlanguage.
     * @param {String} chapid - The id of the page
     */
    Notebook.prototype.addChapterAfter = function(chapid) {
        var lang = 'common'; // this.data.defaultlang;
        var chapter = this.pagesById[chapid];
        var newchapid = this.newChapid();
        if(chapter.level === 0){
        // frontpage
            var frontpage = chapter;                        
            var index = 0;// add to first           
        }else{
            // not frontpage
            while (chapter.level !== 1){
                // seach page parent chapter
                chapter = this.pagesById[chapter.parent];
            }
            var frontpage = this.pagesById[chapter.parent]; 
            // chapters parent is frontpage
            var index = frontpage.indexOf(chapter.chapid)+1;          
            // add after current chapter
        }
        frontpage.addPageAt(index, newchapid);
        // Update linear list and key-value-mapping.
        this.makeLinear();
        this.makeById();
        // Make connection between chapid and the content (key).
        var content = this.newContentid('pageelement');
        this.pagesById[newchapid].setContent(content);
        this.setPageTitle(newchapid, 'New chapter');
        this.setData(content, {type: 'pageelement', metadata: {lang: lang}});
        this.tocChanged();
        return newchapid;
    };
    /**
     * Removes given page. If the page is frontpage, it can't be removed. If "page" have subocontent it can't be removed
     * @param {String} chapid - The id of the page
     */
    Notebook.prototype.removePage = function(chapid) {
        var page = this.pagesById[chapid];
        var respond = false;
        if(page.level === 0){
            alert("You can't remove frontpage");
        }else if(page.chapters.length >0){
            alert("Only empty chapters can removed");
        }else{
            if(confirm("Are you sure! All page data is destroyed.")){
                var parent = this.pagesById[page.parent];
                var index = parent.indexOf(chapid);
                parent.removePageAt(index);
                var content, ptitles;
                for (var langname in this.data.content) {
                    content = this.data.content[langname];
                    delete content[this.getPageId(chapid)];
                };
                for (var langname in this.data.content) {
                    ptitles = this.data.pagetitles[langname];
                    delete ptitles[this.getPageId(chapid)];
                };
                var prevChapId = this.pagesLinear[this.pagesLinear.indexOf(page)-1].chapid;
                // Update linear list and key-value-mapping.
                this.makeLinear();
                this.makeById();
                respond = prevChapId;
                this.tocChanged();
            };
        };
        return respond;
    };
    /**
     * Add an empty page after given page. If the page is frontpage, add a chapter page otherwise add a normal page.
     * @param {String} chapid - The id of the page
     */
    Notebook.prototype.addPageAfter = function(chapid) {
        var lang = 'common'; // this.data.defaultlang;
        var chapter = this.pagesById[chapid];
        var newchapid = this.newChapid();
        if (chapter.level > 1) {
            var parent = this.pagesById[chapter.parent];
            var index = parent.indexOf(chapid);
            parent.addPageAt(index + 1, newchapid);
        } else {
            chapter.addPageAt(0, newchapid);
        };
        // Update linear list and key-value-mapping.
        this.makeLinear();
        this.makeById();
        // Make connection between chapid and the content (key).
        var content = this.newContentid('pageelement');
        this.pagesById[newchapid].setContent(content);
        if(chapter.level === 0){
            this.setPageTitle(newchapid, 'New chapter');
        }else{
            this.setPageTitle(newchapid, 'New page');
        }
        this.setData(content, {type: 'pageelement', metadata: {lang: lang}});
        this.tocChanged();
        return newchapid;
    };
    
    /**
     * Add the root before-page.
     * @returns {String} the chapid of the new page created. If before-page already existed, return empty string.
     */
    Notebook.prototype.addBeforePage = function(){
        var lang = 'common'; // this.data.defaultlang;
        var newchapid;
        if (this.pages.before && this.pages.before.chapid) {
            newchapid = '';
        } else {
            newchapid = this.newChapid();
            this.pages.before = new NotebookPage({chapid: newchapid, title: "-"}, this.pages.chapid, 1, true);
            // Update linear list and key-value-mapping.
            this.makeLinear();
            this.makeById();
            // Make connection between chapid and the content (key).
            var content = this.newContentid('pageelement');
            this.pagesById[newchapid].setContent(content);
            this.setPageTitle(newchapid, 'Before');
            this.setData(content, {type: 'pageelement', metadata: {lang: lang}});
        };
        this.tocChanged();
        return newchapid;
    };

    /**
     * Add the root after-page.
     * @returns {String} the chapid of the new page created. If after-page already existed, return empty string.
     */
    Notebook.prototype.addAfterPage = function(){
        var lang = 'common'; // this.data.defaultlang;
        var newchapid;
        if (this.pages.after && this.pages.after.chapid) {
            newchapid = '';
        } else {
            newchapid = this.newChapid();
            this.pages.after = new NotebookPage({chapid: newchapid, title: "-"}, this.pages.chapid, 1, true);
            // Update linear list and key-value-mapping.
            this.makeLinear();
            this.makeById();
            // Make connection between chapid and the content (key).
            var content = this.newContentid('pageelement');
            this.pagesById[newchapid].setContent(content);
            this.setPageTitle(newchapid, 'After');
            this.setData(content, {type: 'pageelement', metadata: {lang: lang}});
        };
        this.tocChanged();
        return newchapid;
    };

    /**
     * Move the current page to lower level of hierarchy (chapter to section, section to page)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.indentPage = function(chapid){
        var page = this.pagesById[chapid];
        var level = page.level;
        if (!this.isFirst(chapid)) {
            // Find the previous sibling and move the page as its last child.
            var index = this.pagesLinear.indexOf(page);
            var parent = this.pagesById[page.parent];
            var pindex = parent.chapters.indexOf(page);
            var prev, found = false;
            for (var i = index - 1; i > -1; i--) {
                prev = this.pagesLinear[i];
                if (prev.level === level) {
                    // We found the new parent.
                    found = true;
                    break;
                } else if (prev.level < level) {
                    // New parent does not exist.
                    break;
                };
            };
            if (found) {
                // Remove page from current parent and add as the last child of new parent.
                parent.chapters.splice(pindex, 1);
                prev.chapters.push(page);
                page.parent = prev.chapid;
                page.level = prev.level + 1;
                page.fixLevels();
                this.makeLinear();
            };
            this.tocChanged();
        };
    };
    
    /**
     * Move the current page to higher level of hierarchy (section to chapter, page to section)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.unindentPage = function(chapid){
        var page = this.pagesById[chapid];
        var level = page.level;
        if (!this.isFirst(chapid) && page.parent) {
            var parent = this.pagesById[page.parent];
            if (parent.parent) {
                var gparent = this.pagesById[parent.parent];
                var pindex = parent.chapters.indexOf(page);
                var gpindex = gparent.chapters.indexOf(parent);
                parent.chapters.splice(pindex, 1);
                gparent.chapters.splice(gpindex + 1, 0, page);
                page.parent = gparent.chapid;
                page.level = gparent.level + 1;
                page.fixLevels();
                this.makeLinear();
            };
            this.tocChanged();
        };
    };
    
    /**
     * Move the current page to lower level of hierarchy, but keep the linear place (chapter to section, section to page)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.demotePage = function(chapid){
        var page = this.pagesById[chapid];
        var level = page.level;
        if (!this.isFirst(chapid)) {
            // Find the previous sibling and move the page and children as its children.
            var index = this.pagesLinear.indexOf(page);
            var parent = this.pagesById[page.parent];
            var pindex = parent.chapters.indexOf(page);
            var prev;
            if (pindex !== 0) {
                prev = parent.chapters[pindex - 1];
                page.parent = prev.chapid;
                parent.chapters.splice(pindex, 1);
                prev.chapters.push(page);
                page.level = prev.level + 1;
                prev.chapters = prev.chapters.concat(page.chapters.splice(0));
                prev.fixParents();
            };
            this.tocChanged();
        };
    };
    
    /**
     * Move the current page to higher level of hierarchy, but keep the linear place (section to chapter, page to section)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.promotePage = function(chapid){
        var page = this.pagesById[chapid];
        var level = page.level;
        if (!this.isFirst(chapid) && page.parent) {
            var parent = this.pagesById[page.parent];
            if (parent.parent) {
                var gparent = this.pagesById[parent.parent];
                var pindex = parent.chapters.indexOf(page);
                var gpindex = gparent.chapters.indexOf(parent);
                parent.chapters.splice(pindex, 1);
                var children = parent.chapters.splice(pindex);
                gparent.chapters.splice(gpindex + 1, 0, page);
                page.parent = gparent.chapid;
                page.level = gparent.level + 1;
                page.chapters = page.chapters.concat(children);
                page.fixParents();
                page.fixLevels();
                this.makeLinear();
            };
            this.tocChanged();
        };
    };
    
    /**
     * Move the current page forward in the toc (with same level of hierarchy)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.movePageForward = function(chapid){
        var page = this.pagesById[chapid];
        if (!this.isLast(chapid) && !this.isFirst(chapid)) {
            var index = this.pagesLinear.indexOf(page);
            var parent = this.pagesById[page.parent];
            var pindex = parent.chapters.indexOf(page);
            if (pindex !== parent.chapters.length - 1) {
                // Page is not its parent's last child.
                // Switch Page with its next sibling.
                parent.chapters.splice(pindex, 1);
                parent.chapters.splice(pindex + 1, 0, page);
                this.makeLinear();
            } else {
                // Page is its parent's last child.
                // Move page as the first child of the next page with same level as the parent.
                var nextParent, found = false;
                for (var i = index + 1, len = this.pagesLinear.length; i < len; i++){
                    // Find the next page with the same level as the parent.
                    nextParent = this.pagesLinear[i];
                    if (nextParent.level === parent.level) {
                        found = true;
                        break;
                    };
                };
                if (found) {
                    // Move only, if the parent wasn't the last of its level.
                    parent.chapters.splice(pindex, 1);
                    nextParent.chapters.unshift(page);
                    this.makeLinear();
                    page.parent = nextParent.chapid;
                };
            };
            this.tocChanged();
        };
    };
    
    /**
     * Move the current page backward in the toc (with same level of hierarchy)
     * @param {String} chapid - Id of the page
     */
    Notebook.prototype.movePageBackward = function(chapid){
        var page = this.pagesById[chapid];
        if (!this.isFirst(chapid)) {
            var index = this.pagesLinear.indexOf(page);
            var parent = this.pagesById[page.parent];
            var pindex = parent.chapters.indexOf(page);
            if (pindex !== 0) {
                // The Page is not its parent's first child.
                // Switch Page with its previous sibling.
                parent.chapters.splice(pindex, 1);
                parent.chapters.splice(pindex - 1, 0, page);
                this.makeLinear();
            } else {
                // Page is the first child of its parent.
                // Move page as the last child of the previous page with same level as the parent.
                var parentIndex = this.pagesLinear.indexOf(parent);
                var prevParent, found = false;
                for (var i = parentIndex -1; i > -1; i--) {
                    // Find the previous page with the same level as the parent.
                    prevParent = this.pagesLinear[i];
                    if (prevParent.level === parent.level) {
                        found = true;
                        break;
                    };
                };
                if (found) {
                    // Move only, if the parent wasn't the first of its level.
                    parent.chapters.splice(pindex, 1);
                    prevParent.chapters.push(page);
                    this.makeLinear();
                    page.parent = prevParent.chapid;
                };
            };
            this.tocChanged();
        };
    };

    /**
     * Toc changed: Do some things that must be done, when toc is changed.
     */
    Notebook.prototype.tocChanged = function() {
        if (typeof(this.data.toc.metadata) === 'undefined') {
            this.data.toc.metadata = {};
        };
        this.metadata.modified  = this.data.toc.metadata.modified = (new Date()).getTime();
        this.metadata.modifier = this.data.toc.metadata.modifier = this.settings.username || 'Anonymous';
        this.index.setPageorder(this.pages.getPagelist());
    };
    
    /**
     * Add data to index
     * @param {Object} data - index data
     */
    Notebook.prototype.addIndex = function(data) {
        this.index.addElement(data);
    };
    
    /**
     * Remove data from index
     * @param {Object} data - index data
     */
    Notebook.prototype.removeIndex = function(data) {
        this.index.removeElement(data);
    };
    
    /**
     * Check, if the given chapid is the id of the last page in pagesLinear.
     * @param {String} chapid - the id of the page
     */
    Notebook.prototype.isLast = function(chapid){
        return this.pagesLinear[this.pagesLinear.length - 1].chapid === chapid;
    }
    
    /**
     * Check, if the given chapid is the id of the first page in pagesLinear.
     * @param {String} chapid - the id of the page
     */
    Notebook.prototype.isFirst = function(chapid){
        return this.pagesLinear[0].chapid === chapid;
    }
    
    /**
     * Merge content to this notebook.
     * @param {Object} data - an object with element names as keys and element data as values.
     * @returns {String[]}     Array of id's of changed elements if the data was changed (modified timestamp was different)
     */
    Notebook.prototype.mergeContent = function(data){
        // TODO! Fix the language choosing.
        // For now, assuming read from metadata or use 'common' as default.
        var deflang = 'common';
        var lang, eldata, eltype, changed = [];
        for (var name in data){
            eldata = data[name];
            eltype = eldata.type;
            switch (eltype){
                case 'toc':
                    this.data.toc = eldata;
                    this.initToc();
                    changed.push(name);
                    this.tocChanged();
                    break;
                case 'index':
                    // Metadata should always have the language!
                    lang = eldata.metadata && eldata.metadata.lang || deflang;
                    changed.push(name);
                    this.data.index[lang] = eldata.data;
                    break;
                case 'pagetitles':
                    // Metadata should always have the language!
                    lang = eldata.metadata && eldata.metadata.lang || deflang;
                    changed.push(name);
                    this.data.pagetitles[lang] = eldata.data;
                    break;
                case 'booktitles':
                    // Metadata should always have the language!
                    lang = eldata.metadata && eldata.metadata.lang || deflang;
                    changed.push(name);
                    this.setBookTitle(eldata.data);
                    break;
                default:
                    lang = data[name].metadata && data[name].metadata.lang || deflang;
                    if (!this.data.content[lang]) {
                        this.data.content[lang] = {};
                    };
                    try {
                        if (this.data.content[lang][name].metadata.modified !== data[name].metadata.modified){
                            changed.push(name)
                        };
                    } catch (err) {};
                    this.data.content[lang][name] = data[name];
                    break;
            };
        };
        this.initIndex();
        return changed;
    };
    
    /**
     * Generate new bookid
     */
    Notebook.prototype.newBookId = function(){
        return this.settings.username + (new Date()).getTime();
    };
    
    /**
     * Generate a new chapter id. Uses username, uilang and date.
     */
    Notebook.prototype.newChapid = function() {
        var chapid = 'front';
        while (this.pagesById[chapid]) {
            chapid = (this.settings.username + this.settings.uilang + (new Date()).toJSON()).replace(/[-:.]/g, '');
        };
        return chapid;
    };

    /**
     * Generate a new id for content. Uses element type, username, and datetime.
     * @param {String} eltype - Type of the element.
     */
    Notebook.prototype.newContentid = function(eltype) {
        var idparts = [eltype];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    /**
     * Add referenced data
     * @param {Object} refdata - new data by reference
     *   @param {Object} refdata.refs - {key: value}, where key is reference and value is an array of contents.
     *   @param {Object} refdata.refcontentdata - {key: value}, where key is content id and value is content data.
     */
    Notebook.prototype.addRefdata = function(refdata){
        refdata = $.extend(true, {}, Notebook.refdefaults, refdata);
        var refnames, name;
        // Go for all reference elements
        for (var ref in refdata.refs){
            refnames = refdata.refs[ref];
            // If there are elements refering to this ref.
            if (refnames.length > 0) {
                if (!this.refdata.refs[ref]) {
                    this.refdata.refs[ref] = [];
                };
                // Go through all elements thet refer to this ref and add their names to ref's list and
                // their data to the refcontentdata with element's name as a key.
                for (var i = 0, len = refnames.length; i < len; i++) {
                    name = refnames[i];
                    if (this.refdata.refs[ref].indexOf(name) === -1) {
                        this.refdata.refs[ref].push(name);
                    };
                    this.refdata.refcontentdata[name] = refdata.refcontentdata[name];
                };
            };
        };
    };
    
    /**
     * Clear reference data. (Reference data is outdated and therefore cleared.)
     */
    Notebook.prototype.clearRefs = function(){
        this.refdata.refs = {};
        this.refdata.refcontentdata = {};
    }
    
    /**
     * Get referenced data by reference
     */
    Notebook.prototype.getRefdata = function(refs){
        if (typeof(refs) === 'undefined') {
            refs = [];
            for (var ref in this.refdata.refs) {
                refs.push(ref);
            };
        };
        var result = {
            refs: {},
            refcontentdata: {}
        };
        var refname, reflist, elemname;
        for (var i = 0, len = refs.length; i < len; i++) {
            refname = refs[i];
            reflist = this.refdata.refs[refname];
            if (reflist) {
                result.refs[refname] = reflist.slice();
                for (var j = 0, rlen = reflist.length; j < rlen; j++) {
                    elemname = reflist[j];
                    result.refcontentdata[elemname] = $.extend(true, {}, this.refdata.refcontentdata[elemname]);
                };
            };
        };
        return result;
    };
    
    /**
     * Generate numbering for elementboxes
     */
    Notebook.prototype.computeBoxnumbers = function() {
        this.boxnumbers = {};
        var boxnumbers = this.boxnumbers;
        var lang, boxnums, pageid, pagedata, clist, elemid, elem, boxtype;
        var langs = this.getLangs();
        langs.splice(langs.indexOf('common'), 1);
        for (var k = 0; k < langs.length; k++) {
            lang = langs[k];
            boxnumbers[lang] = boxnums = {};
            for (var i = 0, len = this.pagesLinear.length; i < len; i++) {
                pageid = this.pagesLinear[i].content;
                pagedata = this.getElementData({name: pageid, type: 'content', lang: 'common'});
                clist = pagedata.data && pagedata.data.contents || [];
                for (var j = 0, clen = clist.length; j < clen; j++) {
                    elemid = clist[j].id;
                    elem = this.getElementData({name: elemid, type: 'content', lang: lang}) || this.getElementData({name: elemid, type: 'content', lang: 'common'});
                    if (elem.data && elem.data.boxtype) {
                        boxtype = elem.data.boxclass || elem.data.boxtype;
                        boxnums[boxtype] = boxnums[boxtype] || {lastnum: 0, numbers: {}};
                        boxnums[boxtype].numbers[elemid] = ++(boxnums[boxtype].lastnum);
                    };
                };
            };
        };
    };
    
    /**
     * Get the boxnumber
     * @param {String} boxtype    The type of the box (theorybox, examplebox, definitionbox,...)
     * @param {String} elemid     The elementid of the box
     * @param {String} lang       The language of the content
     * @returns {Number}          The number of the box or false, if not found.
     */
    Notebook.prototype.getBoxnumber = function(boxtype, elemid, lang) {
        lang = lang || 'common';
        if (!this.boxnumbers) {
            this.computeBoxnumbers();
        };
        var result = this.boxnumbers[lang] && this.boxnumbers[lang][boxtype] && this.boxnumbers[lang][boxtype].numbers[elemid];
        result = result || this.boxnumbers['common'] && this.boxnumbers['common'][boxtype] && this.boxnumbers['common'][boxtype].numbers[elemid] || false;
        return result;
    }

    /**
     * Default data for Notebook
     */
    Notebook.defaults = {
        type: 'book',
        metadata: {
            
        },
        data: {
            bookid: '',
            titles: {
                common: 'Notebook'
            },
            booktype: [],
            langs: [],
            bookinfo: {
                author: [],
                secondaryauthor: [],
                vendorURL: '',
                vendorLogo: ''
            },
            toc: {
                data: {
                    //firstpage: {
                    //    "chapid": "front",
                    //    "content": "frontpage"
                    //}
                }
            },
            defaultlang: "common",
            pagetitles: {
                common: {}
            },
            content: {
                //common: {
                //    "frontpage": {
                //        "type": "pageelement",
                //    }
                //}
            },
            index: {}
        },
        settings: {
            username: 'Anonymous',
            uilang: 'en',
            lang: 'common',
            teachermode: false
        }
    };
    
    Notebook.refdefaults = {
        refs: {},
        refcontentdata: {}
    }
    
    /**
     * Default data for an empty pageelement.
     */
    Notebook.emptypage = {
        "type": "pageelement",
        "metadata": {},
        "data": {
            "contents": []
        }
    };
    
    /*****************************************************************
     ****************************************************************/
    
    /**
     * NotebookPage class
     * @constructor
     * @param {Object} options - initing options for this page (and subpages)
     * @param {String} parent - The chapid of the parent node
     * @param {Number} level - the level of this page in page hierarchy. (0,1,2,...)
     * @param {Boolean} unnumbered - true, if the page and it's children should be unnumbered
     */
    var NotebookPage = function(options, parent, level, unnumbered){
        options = $.extend(true, {}, NotebookPage.defaults, options);
        this.chapid = options.chapid;
        this.content = options.content;
        this.title = options.title;
        this.label = options.label;
        this.parent = parent;
        this.unnumbered = unnumbered || false;
        this.level = level || 0;
        this.headimage = (options.headimage || '');
        this.subheadimage = (options.subheadimage || '');
        this.types = options.types;
        this.frontimage = (options.frontImage || '');
        if (options.before && options.before.chapid) {
            this.before = new NotebookPage(options.before, this.chapid, this.level + 1, true);
        };
        if (options.after && options.after.chapid) {
            this.after = new NotebookPage(options.after, this.chapid, this.level + 1, true);
        };
        this.chapters = [];
        for (var i = 0, len = options.chapters.length; i < len; i++){
            this.chapters.push(new NotebookPage(options.chapters[i], this.chapid, this.level + 1, this.unnumbered));
        };
    };
    
    /**
     * Get the chapid of this page
     * @returns {String} chapid of this page
     */
    NotebookPage.prototype.getId = function(){
        return this.chapid;
    };
    
    /**
     * Get the headimage
     * @returns {Object} headimage {type: <headimage|fullheadimage>, text: <elemid>}
     */
    NotebookPage.prototype.getHeadimage = function() {
        var result = {
            type: (this.level === 0 ? 'frontimage' : (this.level < 2 && this.subheadimage ? 'fullheadimage' : 'headimage')),
            headimage: this.headimage,
            subheadimage: this.subheadimage,
            frontimage: this.frontimage || '',
            parent: this.parent,
            level: this.level
        };
        return result;
    };
    
    /**
     * Get the list of pagetypes
     * @returns {Array} An array of strings of types
     */
    NotebookPage.prototype.getTypes = function() {
        return this.types.slice();
    };
    
    /**
     * Get the data of this page (and subpages).
     * @returns {Object}
     */
    NotebookPage.prototype.getData = function(){
        var data = {
            chapid: this.chapid,
            content: this.content,
            types: [],
            chapters: []
        }
        if (this.label) {
            data.label = this.label;
        }
        for (var i = 0, len = this.chapters.length; i < len; i++){
            data.chapters.push(this.chapters[i].getData());
        }
        if (this.level === 0 && this.before && this.before.chapid) {
            data.before = this.before.getData();
        }
        if (this.level === 0 && this.after && this.after.chapid) {
            data.after = this.after.getData();
        }
        return data;
    }
    
    /**
     * Find the index of chapid in this.chapters.
     * @param {String} chapid - chapter id
     * @returns {Number} index of chapid in this.chapter (-1 === not found)
     */
    NotebookPage.prototype.indexOf = function(chapid) {
        var index = -1;
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            if (chapid === this.chapters[i].chapid) {
                index = i;
                break;
            };
        };
        return index;
    };
    
    /**
     * Check if this page has subpages
     * @returns {Boolean} true, if this.pages.length > 0 
     */
    NotebookPage.prototype.hasSubpages = function(){
        return this.chapters.length > 0;
    };
    
    /**
     * Set content id
     * @param {String} contentid - reference to the content.
     */
    NotebookPage.prototype.setContent = function(content) {
        this.content = content;
    };
    
    /**
     * Set the title of the page
     * @param {String} title - the title of the page
     */
    NotebookPage.prototype.setTitle = function(title) {
        this.title = title || '';
    };
    
    /**
     *Remove page at given position.
     */
    NotebookPage.prototype.removePageAt = function(index) {
        this.chapters.splice(index, 1);
    };
    /**
     * Add a new page at given position.
     */
    NotebookPage.prototype.addPageAt = function(index, chapid) {
        this.chapters.splice(index, 0, new NotebookPage({chapid: chapid,title:"New Chapter"}, this.chapid, this.level + 1, this.unnumbered));
    };
    
    /**
     * Get a linear array of subpages recursively.
     * @returns {Array} an array of subpages.
     */
    NotebookPage.prototype.getLinear = function(){
        var list = [this];
        if (this.before && this.before.chapid) {
            list = list.concat(this.before.getLinear());
        };
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            list = list.concat(this.chapters[i].getLinear());
        };
        if (this.after && this.after.chapid) {
            list = list.concat(this.after.getLinear());
        };
        return list;
    };
    
    /**
     * Get the data for navbar.
     * @returns {Array} an array of objects with info about this page and its subpages.
     */
    NotebookPage.prototype.getNavdata = function(level, label){
        label = label || '';
        var nextlabel;
        var data = [{
            id: this.chapid,
            level: level,
            title: this.title,
            label: this.label || label
        }];
        if (this.before && this.before.chapid) {
            data = data.concat(this.before.getNavdata(level + 1, ''));
        };
        for (var i = 0, len = this.chapters.length; i < len; i++){
            nextlabel = label ? label + '.' + (i+1) : (i+1)+'';
            data = data.concat(this.chapters[i].getNavdata(level + 1, nextlabel));
        };
        if (this.after && this.after.chapid) {
            data = data.concat(this.after.getNavdata(level + 1, ''));
        };
        return data;
    };
    
    /**
     * Get a list of all pageid's in linear order
     * @returns {Array} An array with pageid's
     */
    NotebookPage.prototype.getPagelist = function(){
        var list = [this.chapid];
        if (this.before && this.before.chapid) {
            list = list.concat(this.before.getPagelist());
        };
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            list = list.concat(this.chapters[i].getPagelist());
        };
        if (this.after && this.after.chapid) {
            list = list.concat(this.after.getPagelist());
        };
        return list;
    }
    
    /**
     * Recoursively fix the levels of chapters. (For example after moving pages in the toc hierarchy.)
     */
    NotebookPage.prototype.fixLevels = function(){
        var level = this.level;
        var page;
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            page = this.chapters[i];
            page.level = level + 1;
            page.fixLevels();
        };
    };
    
    /**
     * Recoursively fix the parent attribute of chapters to reference this page.
     */
    NotebookPage.prototype.fixParents = function(){
        var chapid = this.chapid;
        var page;
        for (var i = 0, len = this.chapters.length; i < len; i++) {
            page = this.chapters[i];
            page.parent = chapid;
            page.fixParents();
        };
    };
    
    /**
     * Default options for constructor of NotebookPage.
     */
    NotebookPage.defaults = {
        chapid: '',
        content: '',
        title: 'New page',
        label: '',
        chapters: [],
        level: 0,
        types: []
    }
    
    
    
    /*****************************************************************
     ****************************************************************/
    
    /**
     * NbChangelog class - keeps log, which parts of the notebook have been changed, but not saved. (are "dirty")
     * @name NbChangelog
     * @constructor
     */
    var NbChangelog = function(){
        this.dirty = {};
        this.saving = {};
        this.dirtycount = 0;
        this.savingcount = 0;
    };
    
    /**
     * Get the count of dirty items.
     * @returns {Number} count of dirty items
     */
    NbChangelog.prototype.dirtysize = function(){
        return this.dirtycount;
    };
    
    /**
     * Get the count of saving items.
     * @returns {Number} count of saving items
     */
    NbChangelog.prototype.savingsize = function(){
        return this.savingcount;
    };
    
    /**
     * Add a new dirty item
     * @param {Object} data            The object to add to dirties
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     */
    NbChangelog.prototype.addDirty = function(data){
        var item = {
            name: data.name || '',
            lang: data.lang || 'common',
            type: data.type || '',
            timestamp: data.timestamp || (new Date()).getTime()
        };
        if (!this.dirty[item.lang]) {
            this.dirty[item.lang] = {};
        }
        if (!this.dirty[item.lang][item.name]) {
            // If new item, add the counter.
            this.dirtycount++;
        };
        this.dirty[item.lang][item.name] = item;
    };
    
    /**
     * Remove item from dirties
     * @param {Object} data         The object to remove from dirties
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     * @returns {Object|Boolean}       The removed object or false
     */
    NbChangelog.prototype.removeDirty = function(data){
        var langdata = this.dirty[data.lang] || {};
        var item = langdata[data.name];
        var timestamp;
        var result = false;
        var rdate = (new Date(data.timestamp || 0)).getTime();
        if (item) {
            timestamp = (new Date(item.timestamp)).getTime();
            if (rdate >= timestamp) {
                result = langdata[data.name];
                delete langdata[data.name];
                this.dirtycount--;
            };
        };
        return result;
    };
    
    /**
     * Add a new saving item
     * @param {Object} data         The object of data that is under saving.
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     */
    NbChangelog.prototype.addSaving = function(data){
        var item = {
            name: data.name || '',
            lang: data.lang || 'common',
            type: data.type || '',
            timestamp: data.timestamp || (new Date()).getTime()
        };
        if (!this.saving[item.lang]) {
            this.saving[item.lang] = {};
        };
        if (!this.saving[item.lang][item.name]) {
            // If new item, add the counter.
            this.savingcount++;
        };
        this.saving[item.lang][item.name] = item;
    };
    
    /**
     * Remove item from savings
     * @param {Object} data         The object of data that is under saving.
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     */
    NbChangelog.prototype.removeSaving = function(data){
        var langdata = this.saving[data.lang] || {};
        var item = langdata[data.name];
        var timestamp;
        var result = false;
        var rdate = (new Date(data.timestamp || 0)).getTime();
        if (item) {
            timestamp = (new Date(item.timestamp)).getTime();
            if (rdate >= timestamp) {
                result = langdata[data.name];
                delete langdata[data.name];
                this.savingcount--;
            };
        };
        return result;
    };
    
    /**
     * Move from dirty to saving
     * @param {Object} data            The object of data to be moved.
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     */
    NbChangelog.prototype.moveToSaving = function(data){
        var ddata = this.removeDirty(data); // || data;
        if (ddata) {
            this.addSaving(ddata);
        };
    };
    
    /**
     * Check if there is newer version of element as dirty
     * @param {Object} data            The object of data to be moved.
     * @param {String} data.name       The id of the element
     * @param {String} data.lang       The language of the changed element
     * @param {String} data.type       The type of the content element
     * @param {String} data.timestamp  The timestamp of the change
     */
    NbChangelog.prototype.hasNewerDirty = function(data) {
        var result = false;
        var item = this.dirty[data.lang] && this.dirty[data.lang][data.name] || false;
        if (item) {
            result = (data.type === item.type && (new Date(data.timestamp)).getTime() <= (new Date(item.timestamp)).getTime());
        };
        return result;
    }
    
    
    /**
     * Get list of dirty items
     * @returns {Array} an array of all dirty items.
     */
    NbChangelog.prototype.getList = function(){
        var result = [];
        var lang, item;
        for (lang in this.dirty) {
            for (item in this.dirty[lang]) {
                result.push(JSON.parse(JSON.stringify(this.dirty[lang][item])));
            };
        };
        return result;
    };
    
    NbChangelog.defaultitem = {
        "name": "",
        "type": "",
        "timestamp": ""
    }
    
    
    
    /*********************************************
     * Indextool for indexing notebooks elements by type.
     * @class IndexTool
     * @constructor
     * @param {Object} options - initing data
     ********************************************/
    var IndexTool = function(options) {
        options = $.extend(true, {}, this.defaults, options);
        this.init(options);
    };
    
    /**
     * Init the index
     * @param {Object} options - initing data
     */
    IndexTool.prototype.init = function(options) {
        this.allElements = {};
        this.bytypes = {};
        this.pageorder = options.pageorder;
        var ellang, elname, eldata, ctype;
        for (ellang in options.indexdata) {
            for (elname in options.indexdata[ellang]) {
                eldata = options.indexdata[ellang][elname];
                this.addElement(eldata);
            };
        };
    };
    
    /**
     * Change the pageorder
     * @param {Array} pageorder - the array of pageid's in linear order
     */
    IndexTool.prototype.setPageorder = function(pageorder) {
        this.pageorder = pageorder;
        for (var ctype in this.bytypes) {
            this.bytypes[ctype].setPageorder(pageorder);
        };
    };
    
    /**
     * Add (or update) the data of an element in the index.
     * @param {Object} eldata - The element index data
     *      {
     *          type: <type of element>,
     *          id: <id of the element>,
     *          lang: <the language of the element>,
     *          gotolink: {<data of the gotolink>}
     *      }
     */
    IndexTool.prototype.addElement = function(eldata) {
        var eltype = eldata.type;
        var lang = eldata.lang;
        if (typeof(eltype) !== 'undefined' && typeof(eldata.id) !== 'undefined') {
            if (typeof(this.bytypes[eltype]) === 'undefined') {
                this.bytypes[eltype] = new IndexList({type: eltype, pageorder: this.pageorder});
            };
            if (!this.allElements[lang]) {
                this.allElements[lang] = {};
            };
            if (this.allElements[lang][eldata.id]) {
                this.bytypes[eltype].removeItem(eldata);
            };
            this.bytypes[eltype].addItem(eldata);
            this.allElements[lang][eldata.id] = eldata;
        };
    };
    
    /**
     * Remove the data of an element in the index.
     * @param {Object} eldata - The element index data
     *      {
     *          type: <type of element>,
     *          id: <id of the element>,
     *          lang: <the language of the element>,
     *      }
     */
    IndexTool.prototype.removeElement = function(eldata) {
        if (this.bytypes[eldata.type]) {
            this.bytypes[eldata.type].removeItem(eldata);
        };
        if (this.allElements[eldata.lang][eldata.id]) {
            delete this.allElements[eldata.lang][eldata.id];
        };
    };
    
    /**
     * Get the index data of element by id
     * @param {String} elemid - id of the element
     * @returns {Object} the index data of the element
     */
    IndexTool.prototype.getElement = function(elemid) {
        var result = JSON.parse(JSON.stringify(this.allElements[elemid]));
        return result;
    };
    
    /**
     * Get a list of index data of elements by the type
     * @param {String} eltype - the type of the element
     * @returns {Array} a list of index data of all elements of asked type in pageorder
     */
    IndexTool.prototype.getElementsByType = function(eltype) {
        result = this.bytypes[eltype] && this.bytypes[eltype].getData() || [];
        return result;
    };
    
    /**
     * Get data for saving
     * @param {String} bookid - optional replacement for bookid
     * @returns {Object} a copy of this.allElements
     */
    IndexTool.prototype.getData = function(bookid) {
        var result = JSON.parse(JSON.stringify(this.allElements));
        if (bookid) {
            var lang, elem;
            for (lang in result) {
                for (elem in result[lang]) {
                    result[lang][elem].gotolink.appname = bookid;
                };
            };
        };
        return result;
    };
    
    IndexTool.prototype.defaults = {
        indexdata: {},
        pageorder: []
    };
    
    
    
    /***********************************************
     * IndexList - list of elements of the same type ordered by the pageorder
     * @class IndexList
     * @constructor
     * @param {Object} options - initing data
     *     {
     *         type: <type of elements in this list>,
     *         list: [<index objects of elements>,...],
     *         pageorder: [<pageid-strings in linear order>]
     *     }
     **********************************************/
    var IndexList = function(options) {
        options = $.extend({}, this.defaults, options);
        this.init(options);
    };
    
    /**
     * Init the index list
     * @param {Object} options - initing data
     */
    IndexList.prototype.init = function(options) {
        this.type = options.type;
        this.pageorder = options.pageorder;
        this.list = [];
        for (var i = 0, len = options.list.length; i < len; i++) {
            this.addItem(options.list[i]);
        };
    };
    
    /**
     * Change the pageorder
     * @param {Array} pageorder - an array of pageid's
     */
    IndexList.prototype.setPageorder = function(pageorder) {
        this.pageorder = pageorder;
        this.sort();
    };
    
    /**
     * Add new index list item to the list in the correct place
     * @param {Object} data - data of the index list item
     *      {
     *          type: <type of element>,
     *          id: <id of the element>,
     *          lang: <the language of the element>,
     *          gotolink: {<data of the gotolink>}
     *      }
     */
    IndexList.prototype.addItem = function(data) {
        var pageid = data.gotolink && data.gotolink.chapid || '';
        var index = this.pageorder.indexOf(pageid);
        if (pageid && data.type === this.type && index > -1) {
            // Dont add, if there is no chapid
            // Positions on the list
            var bottom = 0;
            var top = this.list.length;
            var bottomid = this.list[bottom] && this.list[bottom].gotolink.chapid || '';
            var topid = '';
            //var bottomindex = this.pageorder.indexOf(this.list[bottom].gotolink.chapid);
            //var topindex = this.pageorder.indexOf(this.list[top].gotolink.chapid);
            var cursor, curid, curindex, diff = top-bottom, prevdiff = diff+1, pos = 0;
            while (diff > 0 && prevdiff > diff && bottomid !== topid) {
                // Binary search for the right place.
                // While top and bottom are more than 1 apart and they don't link to the page with same id.
                // Make sure, the iteration advances (diff gets smaller).
                prevdiff = diff;
                cursor = Math.floor((top + bottom) / 2);
                curid = this.list[cursor].gotolink.chapid;
                curindex = this.pageorder.indexOf(curid);
                if (index < curindex) {
                    top = cursor;
                    topid = curid;
                    // The right place is before the cursor.
                    pos = bottom;
                } else if (index > curindex) {
                    bottom = cursor;
                    bottomid = curid;
                    // The right place is after the cursor.
                    pos = bottom + 1;
                } else {
                    bottom = cursor;
                    bottomid = curid;
                    top = cursor;
                    topid = curid;
                    // The correct place is where the cursor was.
                    pos = bottom;
                };
                diff = top - bottom;
            };
            // Add the new element between top and bottom at the place of cursor.
            this.list.splice(pos, 0, data);
        };
    };
    
    /**
     * Remove index list item from the index list
     * @param {Object} data - data identifying the item to be removed
     *     {
     *         type: <type of element>,
     *         id: <id of the element>,
     *         lang: <the language of the element>
     *     }
     */
    IndexList.prototype.removeItem = function(data) {
        var index = this.indexOf(data.id, data.lang);
        this.list.splice(index, 1);
    };
    
    /**
     * Find the index of searched item in the list
     * @param {String} elemid - the id of the element
     * @param {String} lang - the language of the element
     */
    IndexList.prototype.indexOf = function(elemid, lang) {
        var index, len;
        for (index = 0, len = this.list.length; index < len; index++) {
            if (this.list[index].id === elemid && this.list[index].lang === lang) {
                break;
            };
        };
        return index;
    };
    
    /**
     * Get the data
     * @returns {Array} a copy of the ordered list of index data
     */
    IndexList.prototype.getData = function() {
        return JSON.parse(JSON.stringify(this.list));
    };
    
    /**
     * Sort the list by the pageid
     */
    IndexList.prototype.sort = function() {
        var plist = this.pageorder;
        this.list.sort(function(a, b) {
            aid = a.gotolink.chapid;
            bid = b.gotolink.chapid;
            aindex = plist.indexOf(aid);
            bindex = plist.indexOf(bid);
            return aindex - bindex;
        });
    };
    
    IndexList.prototype.defaults = {
        type: '',
        pageorder: [],
        list: []
    };


    if (window.Elpp) {
        window.Elpp.Notebook = Notebook;
    };

    
})(window, jQuery, window.ebooklocalizer);