/***
 * NavBar - Navigation bar
 * Petri Salmela <pesasa@iki.fi>
 ***/
//{{{

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

(function ($) {
    { /*** jQuery plugin ***/
        $.fn.navbar = function(options){
            if (methods[options]){
                return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
            } else if (typeof(options) === 'object' || !options) {
                return methods.init.apply(this, arguments);
            } else {
                $.error( 'Method ' +  method + ' does not exist on Navbar' );
                return this;
            }
        }
        
        var methods = {
            init: function(params){
                
                params = $.extend(true, {
                }, params);
                var navbar = new Navbar(this, params);
            },
            getdata: function(params){
                var $place = $(this);
                $place.trigger('getdata');
                var data = $place.data('[[navbardata]]');
                return data;
            }
        }
    }
    
    { /*** Navbar class ***/
        /******
         * Class for Navigation bar
         * @constructor
         * @param {jQuery} place - place for navbar
         * @param {Object} params - settings for navbar
         ******/
        var Navbar = function(place, params){
            this.place = $(place);
            this.options = $.extend(Navbar.defaults, params);
            this.labeldown = this.options.labeldown;
            this.data = this.options.data;
            this.current = this.options.current;
            this.ticklevel = this.options.ticklevel;
            this.initData();
            this.initView();
        }
        
        /******
         * Default settings for navbar
         * @property {Boolean} labeldown - Show the label of an item under the bar
         * @property {Number} ticklevel - Show tick marks upto this level
         * @property {Array} data - An array of items shown in the bar
         *     Items:
         *     @property {String} current - the id of the current item
         *     @property {String} id - id for the item
         *     @property {Number} level - the level of hierarchy 0,1,2,...
         *     @property {String} title - the title for the item
         *     @property {String} label - the label for the item (for example 1.3.2)
         ******/
        Navbar.defaults = {
            labeldown: false,
            ticklevel: 1,
            data : []
        }
        
        /******
         * Init navigation data.
         ******/
        Navbar.prototype.initData = function(){
            this.byid = {};
            this.ticks = [];
            for (var i = 0; i < this.data.length; i++){
                this.byid[this.data[i].id] = this.data[i];
                if (this.data[i].level <= this.ticklevel) {
                    this.ticks.push(this.data[i]);
                }
            };
            if (!this.byid[this.current]) {
                this.current = this.data[0].id;
            };
        };
        
        /******
         * Init the navbar
         ******/
        Navbar.prototype.initView = function(){
            if ($('head style#navbarstyle').length == 0){
                $('head').prepend('<style id="navbarstyle" type="text/css">'+Navbar.strings.css+'</style>');
            }
            this.draw();
            this.addHandlers();
        }
        
        /******
         * Draw the navbar
         ******/
        Navbar.prototype.draw = function(){
            var width = this.place.width();
            var tickshtml = [];
            for (var i = 0; i < this.ticks.length; i++){
                var tick = this.ticks[i];
                var index = this.data.indexOf(tick);
                var position = (index/(this.data.length - 1)) * 100;
                tickshtml.push('<div class="navbar-tick ffwidget-tick" data-chapid="'+this.escapeHTML(tick.id)+'" data-ctitle="'+this.escapeHTML(tick.title)+'" style="left: '+position+'%;"><div class="navbar-tick-button ffwidget-tick-button">'+Navbar.strings.svg.pointerLabelTemplate.replace('{%label%}', this.escapeHTML(tick.label || '')).replace('{%textlength%}', (tick.label.length > 1 ? '20' : '14'))+Navbar.strings.svg.pointerLabelTemplateDown.replace('{%label%}', this.escapeHTML(tick.label || '')).replace('{%textlength%}', (tick.label.length > 1 ? '20' : '14'))+'</div><span class="navbar-tick-title">'+this.escapeHTML(tick.title)+'</span></div>');
            }
            var html = Navbar.strings.template.replace(/{%navbartickset%}/g, tickshtml.join('\n')).replace(/{%slidermax%}/g, this.data.length-1);
            this.place.html(html);
            this.wrapper = this.place.children('.navbar-wrapper');
            if (this.labeldown) {
                this.wrapper.addClass('navlabel-down')
            }
            this.slider = this.place.find('.navbar-slider');
            this.sliderbefore = this.place.find('.navbar-sliderbefore');
            this.handletitle = this.place.find('.navbar-sliderhandle-title');
            this.range = this.place.find('.navbar-slider-range');
            this.setSlider(false);
        }
        
        /******
         * Update slider
         * @param {String} chapid - id of the item (chapter) to which we need to move
         * @param {Boolean} sendevent - trigger 'navigationmoved' event. 
         ******/
        Navbar.prototype.setSlider = function(chapid, sendevent){
            if (typeof(chapid) === 'boolean') {
                sendevent = chapid;
                chapid = this.current;
            } else if (chapid) {
                this.current = chapid;
            }
            this.currentindex = this.getIndex();
            var position = this.getPos();
            this.sliderbefore.css('width', position + '%');
            this.range.val(this.currentindex);
            if (sendevent) {
                var prev = this.data[this.currentindex - 1];
                var next = this.data[this.currentindex + 1];
                var edata = {
                    prev: prev && prev.id || '',
                    current: this.current,
                    next: next && next.id || ''
                }
                this.sendEvent('navigationmoved', edata);
            }
            this.handletitle.html('<span class="navbar-tooltip-chapnumber">' + this.escapeHTML(this.byid[this.current].label) + '</span>  <span class="navbar-tooltip-chaptitle">' + this.escapeHTML(this.byid[this.current].title) + '</span>');
        }
        
        /******
         * Add event handlers.
         ******/
        Navbar.prototype.addHandlers = function(){
            var navbar = this;
            // Prevent scrolling when swiping
            //$(document).off('touchmove.naviprevent').on('touchmove.naviprevent', function(event){
            //    event.preventDefault();
            //});
            // Move to asked position without triggering event.
            this.place.off('setnaviposition').on('setnaviposition', function(event, data){
                navbar.current = data;
                navbar.setSlider(false);
            })
            // Click on tick-link
            this.place.off('click', '.navbar-tick .navbar-tick-button').on('click', '.navbar-tick .navbar-tick-button', function(event){
                event.preventDefault();
                event.stopPropagation();
                var chapid = $(this).parent().attr('data-chapid');
                navbar.setSlider(chapid, true);
                return false;
            });
            // Range slider changed
            this.place.off('change.navbar', '.navbar-slider-range').on('change.navbar', '.navbar-slider-range', function(event){
                event.preventDefault();
                event.stopPropagation();
                var slider = $(this);
                var value = slider.val();
                var max = navbar.data.length -1;
                var position = value / max;
                var chapid = navbar.getByPosition(position);
                navbar.setSlider(chapid, true);
            });
            // Update the slider-before and title while dragging the thumb.
            this.place.off('input.navbarthumb', '.navbar-slider-range').on('input.navbarthumb', '.navbar-slider-range', function(event){
                var value = $(this).val();
                var max = navbar.data.length -1;
                var position = (max ? value/max : 0);
                var chapid = navbar.getByPosition(position);
                navbar.setSlider(chapid, false);
            });
            // Force change event also on Firefox, when moving with keyboard stops.
            this.place.off('keyup.navbarthumb', '.navbar-slider-range').on('keyup.navbarthumb', '.navbar-slider-range', function(event){
                event.stopPropagation();
                event.preventDefault();
                var place = $(this);
                switch (event.which){
                    case 37:
                    case 38:
                    case 39:
                    case 40:
                        place.blur().focus();
                        break;
                    default:
                        break;
                };
            });
            // Set navbar-dragging -class (show the title), when dragging of range starts.
            this.place.off('touchstart.navbarthumb, mousedown.navbarthumb', '.navbar-slider-range').on('touchstart.navbarthumb, mousedown.navbarthumb', '.navbar-slider-range', function(event){
                navbar.place.addClass('navbar-dragging');
            });
            // Remove navbar-dragging -class (hide the title), when dragging of range ends.
            this.place.off('touchend.navbarthumb, mouseup.navbarthumb', '.navbar-slider-range').on('touchend.navbarthumb, mouseup.navbarthumb', '.navbar-slider-range', function(event){
                navbar.place.removeClass('navbar-dragging');
            });
            // click previous
            this.place.off('click', '.navbar-button-previous').on('click', '.navbar-button-previous', function(event){
                event.preventDefault();
                navbar.goPrev(true);
            });
            // click next
            this.place.off('click', '.navbar-button-next').on('click', '.navbar-button-next', function(event){
                event.preventDefault();
                navbar.goNext(true);
            });
            // Go next event
            this.place.off('gonext').on('gonext', function(event){
                navbar.goNext(true);
            });
            // Go previous event
            this.place.off('goprevious').on('goprevious', function(event){
                navbar.goPrev(true);
            });
            this.place.off('goto').on('goto', function(event, data, sendevent){
                if (typeof(sendevent) === 'undefined') {
                    sendevent = true;
                };
                navbar.setSlider(data, sendevent);
            });
            // Scroll up/down
            this.place.off('DOMMouseScroll mousewheel').on('DOMMouseScroll mousewheel', function(event){
                event.preventDefault();
                event.stopPropagation();
                if (navbar.delayedEvent) {
                    clearTimeout(navbar.delayedEvent);
                }
                if (event.originalEvent.wheelDelta > 0 || event.originalEvent.detail < 0) {
                    navbar.goPrev(false);
                } else{
                    navbar.goNext(false);
                }
                navbar.delayedEvent = setTimeout(function(){navbar.setSlider(true);}, 200);
            })
        }
        
        /******
         * Go next
         * @param {Boolean} sendevent - trigger 'navigationmoved' event.
         ******/
        Navbar.prototype.goNext = function(sendevent){
            var index = this.currentindex + 1;
            if (index < this.data.length) {
                var chapid = this.data[index].id;
                this.setSlider(chapid, sendevent); // || (index === this.data.length - 1));
            }
        }
        
        /******
         * Go prev
         * @param {Boolean} sendevent - trigger 'navigationmoved' event.
         ******/
        Navbar.prototype.goPrev = function(sendevent){
            var index = this.currentindex - 1;
            if (index > -1) {
                var chapid = this.data[index].id;
                this.setSlider(chapid, sendevent); // || (index === 0));
            }
        }
        
        /******
         * Get index of chapid in data array.
         * @param {String} chapid - id of the item (chapter)
         * @returns {Number} The index of given item in the data array
         ******/
        Navbar.prototype.getIndex = function(chapid){
            if (typeof(chapid) === 'undefined') {
                chapid = this.current;
            }
            var index = this.data.indexOf(this.byid[chapid]);
            return (index > -1 ? index : 0);
        }
        
        /******
         * Get position of current as percentages.
         * @param {String} chapid - id of the item (chapter)
         * @returns {Number} position of slider as percentages from left.
         ******/
        Navbar.prototype.getPos = function(chapid){
            chapid = chapid || this.current;
            var index = this.data.indexOf(this.byid[chapid]);
            var position = Math.round((index / (this.data.length -1)) * 1000) / 10;
            return position;
        }
        
        /******
         * Get chapid by position (as fraction of total width of the slider).
         * @param {Number} position
         * @returns {String} chapid the id of the item in given position
         ******/
        Navbar.prototype.getByPosition = function(position){
            position = Math.max(0, Math.min(1, position));
            var index = Math.round(position * ( this.data.length - 1 ));
            return this.data[index] && this.data[index].id || '';
        }
        
        /******
         * Send event to containing element.
         * @param {String} eventName - the name of an event to be sent
         * @param {Object} data - the data to be sent with the event
         ******/
        Navbar.prototype.sendEvent = function(eventName, data){
            this.place.trigger(eventName, [data]);
        }
        
        /**
         * Escape html to entities (for sanitation)
         */
        Navbar.prototype.escapeHTML = function(html) {
            return document.createElement('div')
                .appendChild(document.createTextNode(html))
                .parentNode
                .innerHTML
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;')
        };

        /******
         * Named colors
         ******/
        Navbar.prototype.colornames = {
        }
        
        /******
         * Some strings (css, svg)
         ******/
        
        Navbar.strings = {};

        Navbar.strings.svg = {
            goPrev: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewbox="-3 -3 36 36" class="mini-icon mini-icon-left"><path class="mini-icon-extrapath" style="stroke: none;" opacity="0" d="M15 -2 a17 17 0 0 1 0 34 a17 17 0 0 1 0 -34z m0 2 a 15 15 0 0 0 0 30 a15 15 0 0 0 0 -30z"></path><path style="stroke: none;" d="M19 25 l-15 -10 l15 -10z" /></svg>',
            goNext: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewbox="-3 -3 36 36" class="mini-icon mini-icon-right"><path class="mini-icon-extrapath" style="stroke: none;" opacity="0" d="M15 -2 a17 17 0 0 1 0 34 a17 17 0 0 1 0 -34z m0 2 a 15 15 0 0 0 0 30 a15 15 0 0 0 0 -30z"></path><path style="stroke: none;" d="M10 5 l15 10 l-15 10z" /></svg>',
            pointerLabelTemplate: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="40" viewbox="0 0 30 40" class="pointer-label pointer-label-up"><circle class="labelbackground ffwidget-labelbackground" fill="white" stroke="none" cx="15" cy="15" r="13" /><path class="stylable ffwidget-labelborder" stroke="none" d="M4 23.5 l11 14 l11 -14 a 14 14 0 1 0 -22 0z M3 15 a12 12 0 0 1 24 0 a12 12 0 0 1 -24 0 z" /><text class="ffwidget-labeltext" x="15.5" y="21" fill="black" stroke="black" style="font-family: sans-serif; font-size: 16px;" text-anchor="middle" textLength="{%textlength%}" lengthAdjust="spacingAndGlyphs">{%label%}</text></svg>',
            pointerLabelTemplateDown: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="40" viewbox="0 0 30 40" class="pointer-label pointer-label-down"><circle class="labelbackground ffwidget-labelbackground" fill="white" stroke="none" cx="15" cy="25" r="13" /><path class="stylable ffwidget-labelborder" stroke="none" d="M4 16.5 l11 -14 l11 14 a 14 14 0 1 1 -22 0z M3 25 a12 12 0 0 0 24 0 a12 12 0 0 0 -24 0 z" /><text class="ffwidget-labeltext" x="15.5" y="31" fill="black" stroke="black" style="font-family: sans-serif; font-size: 16px;" text-anchor="middle" textLength="{%textlength%}" lengthAdjust="spacingAndGlyphs">{%label%}</text></svg>'
        }

        Navbar.strings.template = [
            '<div class="navbar-wrapper ffwidget-background navbar-show-buttons">',
            //'<div class="navbar-buttons">',
            '<div class="navbar-button navbar-button-previous ffwidget-button">'+Navbar.strings.svg.goPrev+'</div>',
            //'</div>',
            '<div class="navbar-button navbar-button-next ffwidget-button">'+Navbar.strings.svg.goNext+'</div>',
            '<div class="navbar-slider ffwidget-groove">',
            '<div class="navbar-tickset">{%navbartickset%}</div>',
            '<div class="navbar-sliderbefore ffwidget-ridge"></div>',
            '<div class="navbar-sliderhandle"><div class="navbar-sliderhandle-title ffwidget-tooltip"></div></div>',
            '<div class="navbar-slider-rangewrapper"><input class="navbar-slider-range ffwidget-rangethumb" type="range" min="0" max="{%slidermax%}" /></div>',
            '</div>',
            '</div>'
        ].join('\n');

        Navbar.strings.css = [
            '.navbar-wrapper {-moz-user-select: none; -webkit-user-select: none; -ms-user-select: none; user-select: none; width: 100%; padding: 0.4em 0.2em; position: relative; box-sizing: border-box; -moz-box-sizing: border-box; display: -webkit-box; display: -webkit-flex; display: flex; -webkit-flex-wrap: nowrap; flex-wrap: nowrap; -webkit-flex-direction: row; flex-direction: row; -webkit-align-items: center; align-items: center;}',
            'html.no-flexbox .navbar-wrapper {display: block;}',
            'html.touch .navbar-wrapper {height: 50px;}',
            '.navbar-slider {height: 6px; border: 1px solid #444; border-radius: 3px; background-color: #ddd; position: relative; margin: 10px 0.4em; padding: 0; line-height: 0; cursor: pointer; white-space: nowrap; -webkit-box-ordinal-group: 3; -moz-box-ordinal-group: 3; -ms-flex-order: 3; -webkit-order: 3; order: 3; align-content: center; -webkit-box-flex: 1; -webkit-flex: 1 1 100%; flex: 1 1 100%;}',
            'html.touch .navbar-slider {-webkit-box-ordinal-group: 2; -moz-box-ordinal-group: 2; -ms-flex-order: 2; -webkit-order: 2; order: 2;}',
            '.navbar-sliderbefore {height: 6px; border-radius: 3px 0 0 3px; width: 35%; background-color: #666; display: inline-block; margin: 0; padding: 0; vertical-align: middle; -webkit-transition: width 0.3s; transition: width 0.3s;}',
            '.navbar-dragging .navbar-sliderbefore {-webkit-transition: width 0s; transition: width 0s;}',
            '.navbar-sliderhandle {position: relative; padding: 0; width: 16px; height: 16px; border-radius: 3px; border: 1px solid black; margin: -6px -14px; background-color: #888; display: inline-block; vertical-align: middle; cursor: pointer;}',
            '.navbar-sliderhandle {width: 0; height: 0; border: 2px solid red; background: transparent; margin: -6px -4px; height: 8px;}',
            
            '.navbar-sliderhandle-title {display: none; position: absolute; top: 20px; left: -10px; background-color: #ffa; border: 1px solid #666; border-radius: 5px; padding: 0.3em; line-height: 1.2em;}',
            '.navbar-wrapper.navlabel-down .navbar-sliderhandle-title {top: auto; bottom: -50px;}',
            '.navbar-sliderhandle:hover .navbar-sliderhandle-title, .navbar-dragging .navbar-sliderhandle-title {display: block;}',
            '.navbar-tooltip-chapnumber {padding-right: 0.8em;}',
            '.navbar-tickset {position: absolute; top: -0.8em; left: 0; right: 0; line-height: 0.8em; height: 0.8em;}',
            '.navbar-tick {position: absolute; display: inline-block; border-left: 1px solid #666; border-right: 1px solid #666; width: 0; text-align: center; top: 0; height:100%;}',
            '.navbar-tick .navbar-tick-button {position: relative; z-index: 1; display: inline-block; visibility: hidden; font-family: sans-serif; font-size: 100%; width: auto; height: auto; border: none; border-radius: 0; background-color: transparent; padding: 0; margin-top: -30px; margin-left: -500px; margin-right: -500px; opacity: 0; overflow: hidden;}',
            '.navbar-wrapper:hover .navbar-tick .navbar-tick-button {display: inline-block; visibility: visible; opacity: 1; -webkit-transition: opacity 0.6s linear 0s; transition: opacity 0.6s linear 0s;}',
            '.navbar-tick:hover .navbar-tick-button {width: auto; height: auto; font-size: 100%; border: none; border-radius: 0; padding: 0; background-color: transparent; margin-top: -30px; z-index: 2;}',
            ':hover > .labelbackground {fill: #f77;}',
            '.navbar-tick .navbar-tick-title {display: none; line-height: 1.2em;}',
            '.navbar-tick .navbar-tick-button:hover + .navbar-tick-title {display: block; position: absolute; z-index: 2; top: -1.5em; left: -2.5em; padding: 0.3em 0.6em; border-radius: 0.3em; background-color: #ffa; color: black; border: 1px solid black; box-shadow: 5px 5px 4px rgba(0,0,0,0.5);}',
            '.navbar-show-buttons .navbar-buttons {padding-top: 0.5em; border: 1px solid red;}',
            '.navbar-button {font-weight: bold; width: 30px; height: 30px; padding: 0; margin: 0 2px; border: 1px solid #666; border-radius: 0.2em; text-align: center; vertical-align: middle; line-height: 30px; background-color: white;}',
            '.navbar-button-previous {-webkit-box-ordinal-group: 1; -webkit-order: 1; order: 1; -webkit-box-flex: 0; -webkit-flex: 0 0 30px; flex: 0 0 30px;}',
            '.navbar-button-next {-webkit-order: 2; order: 2;  -webkit-box-flex: 0; -webkit-flex: 0 0 30px; flex: 0 0 30px;}',
            'html.touch .navbar-button-previous {-webkit-box-ordinal-group: 1; -webkit-order: 1; order: 1; -webkit-box-flex: 0; -webkit-flex: 0 0 50px; flex: 0 0 50px;}',
            'html.touch .navbar-button-next {-webkit-box-ordinal-group: 3; -moz-box-ordinal-group: 3; -ms-flex-order: 3; -webkit-order: 3; order: 3; -webkit-box-flex: 0; -webkit-flex: 0 0 50px; flex: 0 0 50px;}',
            '.navbar-slider svg.pointer-label-down {display: none;}',
            '.navbar-wrapper.navlabel-down .navbar-slider svg.pointer-label-down {display: inline-block;}',
            '.navbar-wrapper.navlabel-down .navbar-slider svg.pointer-label-up {display: none;}',
            '.navbar-wrapper.navlabel-down .navbar-tickset {top: auto; bottom: -0.8em;}',
            '.navbar-wrapper.navlabel-down .navbar-tick .navbar-tick-button {margin-top: 0; margin-bottom: -30px;}',
            '.navbar-slider-rangewrapper {margin: -15px -10px 0 -10px;}',
            '.navbar-slider-range {display: block; position: relative; width: 100%; box-sizing: border-box; -moz-box-sizing: border-box; border: 0; margin: 0; -webkit-appearance:none; background-color: transparent; height: 22px;}',
            '.navbar-slider-range::-webkit-slider-runnable-track {background: none;}',
            '.navbar-slider-range::-moz-range-track {background: none;}'
        ].join('\n');
    }
    
})(jQuery);
//}}}