/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    
    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Userdata class
     * @constructor
     ******/
    var Userdata = function(options){
        options = $.extend(true, {}, Multichoice.defaults.userdata, options);
        this.settings = options.settings;
        this.name = options.name || this.getNewId();
        this.metadata = options.metadata;
        this.data = options.data;
        if (this.metadata.creator === '') {
            this.metadata.creator = this.settings.username;
        };
        if (this.metadata.created === '') {
            var nowtime = new Date();
            this.metadata.created = nowtime;
        };
    };
    
    Userdata.prototype.getData = function(){
        var userdata = {
            type: 'userdata',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return userdata;
    };
    
    Userdata.prototype.addClick = function(str, correct){
        this.data.clicked[str] = true;
        this.data.results[0] = correct | 0;
        var nowtime = (new Date()).getTime();
        if (!this.metadata.created) {
            this.metadata.created = nowtime;
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime;
        this.metadata.modifier = this.settings.username;
    };
    
    Userdata.prototype.setSelection = function(str, correct){
        this.data.clicked = {};
        this.data.clicked[str] = true;
        this.data.results[0] = correct | 0;
        var nowtime = (new Date()).getTime();
        if (!this.metadata.created) {
            this.metadata.created = nowtime;
            this.metadata.creator = this.settings.username;
        };
        this.metadata.modified = nowtime;
        this.metadata.modifier = this.settings.username;
    }
    
    /**
     * Get the current answer (Clicked object)
     * @return {Object}  the clicked-object
     */
    Userdata.prototype.getAnswer = function() {
        return JSON.parse(JSON.stringify(this.data.clicked));
    };
    
    Userdata.prototype.solved = function(bool){
        this.data.solved = !!bool;
    };
    
    Userdata.prototype.isSolved = function(){
        return this.data.solved;
    };
    
    Userdata.prototype.closed = function(bool){
        this.data.closed = !!bool;
    };
    
    Userdata.prototype.isClosed = function(){
        return this.data.closed;
    }
    
    Userdata.prototype.isClicked = function(choiceid){
        return !!this.data.clicked[choiceid];
    }
    
    Userdata.prototype.getNewId = function(){
        var idparts = ['multichoiceanswer'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    
    /******
     * Multichoice question element
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for element
     ******/
    var Multichoice = function(place, options){
        this.place = $(place);
        this.place.addClass('multichoiceelement');
        this.setStyles();
        this.init(options);
        this.addSounds();
        this.show();
    };
    
    /******
     * Init multichoice element
     * @param {Object} options - initing data of the element
     ******/
    Multichoice.prototype.init = function(options){
        options = $.extend(true, {}, Multichoice.defaults, options);
        this.settings = options.settings;
        if (this.settings.mode === 'view') {
            // Use 'appendview' instead of 'view' by default. If the multichoice is already
            // solved, change to 'view' later.
            this.settings.mode = 'appendview';
        }
        this.solvable = this.settings.users && this.settings.users.canEdit({ctype: 'solution'}) || false;
        this.text = options.data.text;
        this.multichoiceData = {
            correct: options.data.correct,
            wrong: options.data.wrong
        }
        this.metadata = options.metadata;
        this.assignmentid = this.place.attr('data-element-name');
        var userdata = options.userdata;
        userdata.settings = {username: options.settings.username, lang: options.settings.lang || ''}
        this.userdata = new Userdata(userdata);
        this.setMode(this.settings.mode, true);
        this.setAttrs();
    };
    
    /******
     * Get the data of multichoice element.
     * @returns {Object} the data of this multichoice element
     ******/
    Multichoice.prototype.getData = function(){
        var data = {
            type: 'multichoice',
            metadata: $.extend(true, {}, this.metadata),
            data: {
                title: '',
                text: this.text,
                assignmentType: '2',
                correct: this.multichoiceData.correct,
                wrong: this.multichoiceData.wrong.slice()
            }
        };
        return data;
    }
    
    /******
     * Set attributes for the container
     ******/
    Multichoice.prototype.setAttrs = function(){
        this.place.attr('lang', escapeHTML(this.metadata.lang))
            .attr('data-creator', escapeHTML(this.metadata.creator))
            .attr('data-created', escapeHTML(this.metadata.created))
            .attr('data-modifier', escapeHTML(this.metadata.modifier))
            .attr('data-modified', escapeHTML(this.metadata.modified))
            .attr('data-tags', escapeHTML(this.metadata.tags.join(',')))
            .attr('data-quiztotal', '1')
            .addClass('quizquestion');
    };
    
    /******
     * Set modes
     ******/
    Multichoice.prototype.setMode = function(mode, dontredraw){
        // If mode is incorrect, use 'view' as a default mode.
        if (!Multichoice.modes[mode]) {
            mode = 'view';
        }
        this.place.attr('data-elementmode', mode);
        var modedata = Multichoice.modes[mode];
        this.editable = modedata.editable;
        this.authorable = modedata.authorable;
        this.reviewable = modedata.reviewable;
        this.appendable = modedata.appendable;
        this.settings.mode = mode;
        if (!dontredraw) {
            this.show();
        }
        this.addHandlers();
    }
    
    /******
     * Show multichoice
     ******/
    Multichoice.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        }
    }
    
    /******
     * Show the multichoice in view mode
     ******/
    Multichoice.prototype.view = function(){
        var name = this.place.attr('data-element-name');
        var html = Multichoice.templates.html;
        this.place.html(html);
        var text = this.text;
        var qarea = this.place.children('.multichoice-questionarea');
        var aarea = this.place.find('> .multichoice-answerarea ul');
        qarea.html('<div class="multichoice-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'view',
                role: this.settings.role
            }
        }
        qarea.children('.multichoice-markdown').markdownelement(mddata);
        var choices = [];
        choices.push('<li data-choiceid="choice_1" data-status="empty"><span class="multichoice-select">'+Multichoice.icons.multichoicemark+'</span><span class="multichoice-choicetext">'+this.mathfy(escapeHTML(this.multichoiceData.correct))+'</span></li>');
        for (var i = 0; i < this.multichoiceData.wrong.length; i++) {
            choices.push('<li data-choiceid="choice_'+(i+2)+'" data-status="empty"><span class="multichoice-select">'+Multichoice.icons.multichoicemark+'</span><span class="multichoice-choicetext">'+this.mathfy(escapeHTML(this.multichoiceData.wrong[i]))+'</span></li>')
        };
        for (var i = 0; i < 10; i++){
            choices.sort(function(a,b){return (Math.random() - 0.5);});
        };
        for (var i = 0; i < choices.length; i++) {
            aarea.append(choices[i]);
        };
        this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        if (!this.userdata.isClosed()) {
            this.place.trigger('get_userdata', {refid: name});
            this.place.trigger('getcontentbyref', {refs: [name]})
        } else {
            this.appendUserData();
        }
    }

    Multichoice.prototype.appendUserData = function(){
        var choices = this.place.find('.multichoice-answerarea ul.multichoice-choices > li');
        var solved = this.userdata.isSolved();
        var closed = this.userdata.isClosed();
        if ((this.settings.feedback && solved && this.appendable) || (closed && this.appendable)) {
            this.setMode('view');
        } else {
            var choiceid, $choice;
            if (solved && this.settings.feedback || closed) {
                choices.attr('data-status', 'disabled');
            };
            for (var i = 0; i < choices.length; i++) {
                $choice = choices.eq(i);
                choiceid = $choice.attr('data-choiceid');
                if (this.userdata.isClicked(choiceid)) {
                    if (this.settings.feedback) {
                        if (choiceid === 'choice_1') {
                            $choice.attr('data-status', 'correct');
                        } else {
                            $choice.attr('data-status', 'wrong');
                        };
                    } else {
                        $choice.attr('data-status', 'selected');
                    }
                };
            };
        };
    };
    
    Multichoice.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    Multichoice.prototype.addHandlers = function(){
        this.removeHandlers();
        this.addHandlersCommon();
        if (this.editable) {
            this.addHandlersEdit();
        } else {
            this.addHandlersView();
        }
        if (this.appendable) {
            this.addHandlersAppendable();
        };
    };
    
    Multichoice.prototype.addHandlersCommon = function(){
        var multi = this;
        this.place.on('setmode', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            multi.setMode(data);
        });
        this.place.on('reply_get_userdata', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            data.settings = {
                username: multi.settings.username,
                lang: multi.settings.lang || '',
                uilang: multi.settings.uilang,
                role: multi.settings.role
            }
            multi.userdata = new Userdata(data);
            multi.appendUserData();
        });
        this.place.on('reply_getcontentbyref', function(e, data){
            var refid = multi.assignmentid;
            var reflist = data.refs[refid];
            var userdata = data.refcontentdata[reflist[0]];
            if (userdata) {
                userdata.settings = {
                    username: multi.settings.username,
                    lang: multi.settings.lang || '',
                    uilang: multi.settings.uilang,
                    role: multi.settings.role
                };
                multi.userdata = new Userdata(userdata);
                multi.appendUserData();
            };
        });
        this.place.on('showcorrectanswer', function(event, data){
            event.stopPropagation();
            multi.showCorrect();
        });
        this.place.on('clearanswer', function(event, data){
            event.stopPropagation();
            multi.clearAnswer();
        });
        this.place.on('getdata', function(e){
            var data = multi.getData();
            multi.place.data('[[elementdata]]', data);
        });
    };
    
    Multichoice.prototype.addHandlersEdit = function(){
        var multi = this;
        this.place.on('change', 'input.multichoice-choicetext', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var value = input.val();
            var index = input.attr('data-choice-number') | 0;
            var status = input.closest('li').attr('data-status');
            if (status === 'wrong') {
                multi.multichoiceData.wrong[index - 2] = value;
            } else {
                multi.multichoiceData.correct = value;
            };
            multi.changed();
        });
        this.place.on('keyup', 'input.multichoice-choicetext', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var input = $(this);
            var nextli = input.closest('li').next('li');
            var next = nextli.children('input');
            switch (e.which){
                case 13:
                    if (next.length > 0) {
                        next.focus().select();
                    } else {
                        nextli.click();
                    };
                    break;
                default:
                    break;
            };
        });
        this.place.on('element_changed', '.multichoice-markdown', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var mdelem = $(this);
            mdelem.trigger('getdata');
            var mddata = mdelem.data('[[elementdata]]');
            multi.text = mddata.data.text;
            multi.changed();
        });
        this.place.on('click', '.multichoice-addwrong', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var button = $(this);
            var index = button.attr('data-choice-number') | 0;
            button.before('<li data-choiceid="choice_' + index + '" data-status="wrong"><span class="multichoice-select">'+Multichoice.icons.multichoicemark+'</span><input type="text" class="multichoice-choicetext" data-choice-number="'+index+'" /><span class="multichoice-trash">'+Multichoice.icons.trash+'</span></li>');
            button.attr('data-choice-number', index + 1);
            button.prev().children('input').focus();
        });
        this.place.on('click', '.multichoice-choices li .multichoice-trash', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            var button = $(this);
            var liitem = button.closest('li');
            var list = liitem.closest('ul.multichoice-choices');
            var input = liitem.children('input');
            var index = (input.attr('data-choice-number') | 0) - 2;
            if (index > -1) {
                multi.multichoiceData.wrong.splice(index, 1);
            }
            liitem.remove();
            var items = list.children('li');
            for (var i = 0, len = items.length; i < len; i++) {
                items.eq(i).attr('data-choiceid', 'choice_' + (i+1));
                items.eq(i).children('input').attr('data-choice-number', (i+1));
            };
            multi.changed();
        });
    }
    
    Multichoice.prototype.addHandlersView = function(){
        
    }
    
    Multichoice.prototype.addHandlersAppendable = function(){
        var multi = this;
        this.place.on('click', '.multichoice-answerarea ul.multichoice-choices > li', function(e){
            e.stopPropagation();
            e.preventDefault();
            var selection = $(this);
            var choice = selection.attr('data-choiceid');
            var userdata = multi.userdata;
            var clickeds = userdata.getAnswer();
            if (!clickeds[choice]) {
                if (multi.settings.feedback) {
                    if (choice === 'choice_1') {
                        selection.attr('data-status', 'correct');
                        selection.parent().children('li[data-status="empty"]').attr('data-status', 'disabled');
                        userdata.addClick(choice, true);
                        userdata.solved(true)
                        multi.saveUserData({sendforward: true});
                        multi.playSounds('multichoice_correct');
                        multi.setMode('view', true); // Change mode to 'view', don't redraw.
                    } else {
                        selection.attr('data-status', 'wrong');
                        userdata.addClick(choice, false);
                        multi.saveUserData();
                        multi.playSounds('multichoice_wrong');
                    }
                } else {
                    selection.closest('ul').children('li').attr('data-status', 'empty');
                    selection.attr('data-status', 'selected');
                    userdata.setSelection(choice, choice === 'choice_1');
                    userdata.solved(choice === 'choice_1');
                    multi.saveUserData();
                }
            }
        });
        this.place.on('close_quizquestion', function(event, data){
            event.stopPropagation();
            multi.close();
            var localsave = !!data.localsave;
            multi.saveUserData({localsave: localsave, sendforward: false});
        });
    };
    
    /******
     * Show the multichoice in edit mode
     ******/
    Multichoice.prototype.edit = function(){
        var name = this.place.attr('data-element-name');
        var html = Multichoice.templates.html;
        this.place.html(html);
        var text = this.text;
        var qarea = this.place.children('.multichoice-questionarea');
        var aarea = this.place.find('> .multichoice-answerarea ul');
        qarea.html('<div class="multichoice-markdown"></div>');
        var mddata = {
            type: 'markdownelement',
            metadata: {},
            data: {
                text: text
            },
            settings: {
                uilang: this.settings.uilang,
                mode: 'edit',
                role: this.settings.role
            }
        }
        qarea.children('.multichoice-markdown').markdownelement(mddata);
        var choices = [];
        choices.push('<li data-choiceid="choice_1" data-status="correct"><span class="multichoice-select">'+Multichoice.icons.multichoicemark+'</span><input type="text" class="multichoice-choicetext" data-choice-number="1" value="'+escapeHTML(this.multichoiceData.correct)+'" /></li>');
        var trash = (this.multichoiceData.wrong.length > 1 ? '<span class="multichoice-trash">'+Multichoice.icons.trash+'</span>' : '');
        for (var i = 0; i < this.multichoiceData.wrong.length; i++) {
            choices.push('<li data-choiceid="choice_'+(i+2)+'" data-status="wrong"><span class="multichoice-select">'+Multichoice.icons.multichoicemark+'</span><input type="text" class="multichoice-choicetext" data-choice-number="'+(i+2)+'" value="'+escapeHTML(this.multichoiceData.wrong[i])+'" />'+trash+'</li>')
        };
        choices.push('<li data-choice-number="'+(i+2)+'" class="multichoice-addwrong"><div>+</div></li>');
        aarea.append(choices.join('\n'));
    }
    
    /******
     * Trigger changed event
     ******/
    Multichoice.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'multichoice'});
    }
    
    /******
     * Close the quiz question.
     ******/
    Multichoice.prototype.close = function(){
        this.userdata.closed(true);
    };
    
    
    
    Multichoice.prototype.saveUserData = function(options){
        //if (this.settings.role === 'student') {
        if (this.solvable) {
            var refid = this.assignmentid;
            var userdata = this.userdata.getData();
            var name = userdata.name;
            var localsave = options && !!options.localsave;
            if (options && options.sendforward) {
                var senddata = {};
                senddata[name] = userdata;
                this.place.trigger('save_userdata', {refid: refid, 'public': true, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
                this.place.trigger('sendcontentdata', senddata);
            } else {
                this.place.trigger('save_userdata', {refid: refid, 'public': false, data: userdata});
                this.place.trigger('setcontentbyref', [{reftos: [refid], name: name, refcontentdata: userdata}, localsave]);
            };
        };
    };
    
    /******
     * Show correct answer
     ******/
    Multichoice.prototype.showCorrect = function(){
        this.setMode('view');
        var choices = this.place.find('.multichoice-answerarea .multichoice-choices li');
        var correct = this.place.find('.multichoice-answerarea .multichoice-choices li[data-choiceid="choice_1"]');
        if (!this.feedback) {
            choices.attr('data-status', 'empty');
            correct.attr('data-status', 'selected');
        };
    };
    
    /******
     * Clear answer
     ******/
    Multichoice.prototype.clearAnswer = function(){
        var userdata = {settings: {username: this.settings.username, lang: this.settings.lang || ''}};
        this.userdata = new Userdata(userdata);
        this.setMode('appendview');
        this.appendUserData();
    };
    
    /******
     * Set style-tag if needed
     ******/
    Multichoice.prototype.setStyles = function(){
        if ($('head style#multichoiceelement-style').length === 0) {
            $('head').append('<style id="multichoiceelement-style" type="text/css">'+Multichoice.styles+'</style>');
        }
    }

    Multichoice.prototype.playSounds = function(id){
        if (jQuery('html').hasClass('ipad')) {
            if (typeof(this.audios[id]) === "undefined" ){
                this.audios[id]= {};
                var audioelement = this.audios[id];
                if (typeof(window.myAudioContext) === "undefined"){
                    window.myAudioContext = new webkitAudioContext();
                }
                this.audios[id]['arrayBuff'] = Base64Binary.decodeArrayBuffer(this.sounds[id.replace("#", "")]);
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                context.decodeAudioData(this.audios[id]['arrayBuff'], function(audioData) {
                    element.buffer = audioData;
                    audioelement['buffer'] = audioData;
                    element.connect(context.destination);
                    
                    element.noteOn(0);
                });
            }else{
                if(typeof(window.myAudioContext)==="undefined"){
                    window.myAudioContext = new webkitAudioContext();
                }
                var context = window.myAudioContext;
                var element = context.createBufferSource();
                element.buffer = this.audios[id]['buffer'];
                element.connect(context.destination);
                element.noteOn(0);
            }
        }else{
            jQuery('#'+id)[0].play();
        };
    }

    Multichoice.prototype.playSounds = function(id){
    };
    
    Multichoice.prototype.addSounds = function(){
        if (jQuery("#soundEffects").length === 0 ){
            var effectsPlace = jQuery('body').append('<div id="soundEffects"/>').find('#soundEffects');
        } else {
            var effectsPlace = jQuery("#soundEffects");
        };
        for (var sound in Multichoice.sounds){
            if (jQuery("#"+sound).length ===0){
                jQuery('#soundEffects').append('<audio id="'+escapeHTML(sound)+'" autobuffer="autobuffer" ><source src="data:audio/mpeg;base64,'+Multichoice.sounds[sound]+'"></source></audio>');
            };
        };
    };
    
    Multichoice.prototype.addSounds = function(){
    };
    
    /******
     * Mathquillify the string, i.e., replace \( \)-pairs and $ $-pairs with a span.
     ******/
    Multichoice.prototype.mathfy = function(str){
        str = str.replace(/\$([^$]*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>')
            .replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>');
        return str;
    }
    
    Multichoice.defaults = {
        type: 'multichoice',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            lang: '',
            tags: []
        },
        data: {
            title: '',
            text: '',
            correct: '',
            wrong: ['']
        },
        userdata: {
            type: 'userdata',
            metadata: {
                creator: '',
                created: '',
                modifier: '',
                modified: '',
                tags: []
            },
            data: {
                clicked: {},
                results: [0],
                total: 1,
                solved: false
            },
            settings: {
                username: 'Anonymous',
                uilang: 'en',
                lang: 'en'
            }
        },
        settings: {
            mode: 'view',
            role: 'student',
            uilang: 'en',
            username: 'Anonymous',
            feedback: true
        }
    };
    
    Multichoice.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        appendview: {
            editable: false,
            authorable: false,
            reviewable: false,
            appendable: true
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            appendable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            appendable: false
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            appendable: false
        }
    };
    
    Multichoice.templates = {
        html: [
            '<div class="multichoice-questionarea ebookassignment-questionarea"></div>',
            '<div class="multichoice-answerarea ebookassignment-answerarea">',
            '    <ul class="multichoice-choices">',
            '    </ul>',
            '</div>'
        ].join('\n')
    };
    
    Multichoice.icons = {
        trash: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>',
        correct: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice-correct"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /><path style="stroke: none; fill: #0a0;" d="M29 1 l-16 24 l-1 0 l-6 -12 a2 2 0 0 1 4 -2 l3 6 l13 -18z" /></svg>',
        wrong: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice-wrong"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /><path style="stroke: none; fill: #a00;" d="M4 2 l25 25 l-4 2 l-23 -24z M2 24 l26 -22 l2 2 l-25 23z" /></svg>',
        empty: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice-empty"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z" /></svg>',
        selected: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice-selected"><circle style="stroke: none; fill: white;" cx="15" cy="15" r="14" /><path style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z m0 2 a11 11 0 0 1 0 22 a11 11 0 0 1 0 -22z m0 3 a8 8 0 0 0 0 16 a8 8 0 0 0 0 -16z" /></svg>',
        multichoicemark: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice-mark"><circle style="stroke: none;" fill="black" cx="15" cy="15" r="13" /><circle class="multichoice-selectfill" style="stroke: none;" fill="white" cx="15" cy="15" r="11" /><circle class="multichoice-selectedmark" style="stroke: none;" fill="black" cx="15" cy="15" r="7" /><path class="multichoice-correctmark" style="stroke: none; fill: #0a0;" d="M29 1 l-16 24 l-1 0 l-6 -12 a2 2 0 0 1 4 -2 l3 6 l13 -18z" /><path class="multichoice-wrongmark" style="stroke: none; fill: #a00;" d="M4 2 l25 25 l-4 2 l-23 -24z M2 24 l26 -22 l2 2 l-25 23z" /></svg>'
    }
    
    Multichoice.styles = [
        //'.multichoiceelement {background-color: white;}',
        '.multichoice-answerarea ul.multichoice-choices {list-style: none;}',
        '.multichoice-answerarea ul.multichoice-choices li {cursor: pointer; padding-left: 30px;}',
        '.multichoice-answerarea .multichoice-select {margin-right: -20px; position: relative; left: -35px; vertical-align: top;}',
        '.multichoice-answerarea .multichoice-select path.multichoice-correctmark {display: none;}',
        '.multichoice-answerarea .multichoice-select circle.multichoice-selectedmark {display: none;}',
        '.multichoice-answerarea li[data-status="selected"] .multichoice-select circle.multichoice-selectedmark {display: inline;}',
        '.multichoice-answerarea li[data-status="selected"] .multichoice-select path {display: none;}',
        //'.multichoice-answerarea li[data-status="empty"] .multichoice-select path.multichoice-correctmark {display: none;}',
        '.multichoice-answerarea li[data-status="empty"] .multichoice-select path.multichoice-wrongmark {display: none;}',
        //'.multichoice-answerarea li[data-status="disabled"] .multichoice-select path.multichoice-correctmark {display: none;}',
        '.multichoice-answerarea li[data-status="disabled"] .multichoice-select path.multichoice-wrongmark {display: none;}',
        '.multichoice-answerarea li[data-status="disabled"] .multichoice-select circle {fill: #eee;}',
        '.multichoice-answerarea li[data-status="correct"] .multichoice-select path.multichoice-wrongmark {display: none;}',
        '.multichoice-answerarea li[data-status="correct"] .multichoice-select circle {fill: #050;}',
        '.multichoice-answerarea li[data-status="correct"] .multichoice-select circle.multichoice-selectfill {fill: #5f5;}',
        '.multichoice-answerarea li[data-status="wrong"] .multichoice-select path.multichoice-correctmark {display: none;}',
        '.multichoice-answerarea li[data-status="wrong"] .multichoice-select circle {fill: #eee;}',
        '.multichoice-answerarea ul.multichoice-choices li.multichoice-addwrong {text-align: center; font-weight: bold; max-width: 15em; padding: 0;}',
        'li.multichoice-addwrong:hover {background-color: #fafafa;}',
        '.multichoice-choices li span.multichoice-trash:hover svg path {fill: #a00;}',
        '.multichoice-choices li input {min-height: 20px; vertical-align: top;}',
        '[data-ebookelement-type="multichoice"] > .multichoice-answerarea {margin-left: 0.6em; border-left: 5px solid #ddd;}',
        // Highlight questions correct and wrong solutions.
        '.questionhighlights .multichoice-choices li[data-status="selected"][data-choiceid="choice_1"] .multichoice-select svg circle.multichoice-selectedmark {fill: #0a0;}',
        '.questionhighlights .multichoice-choices li[data-status="selected"]:not([data-choiceid="choice_1"]) .multichoice-select svg circle.multichoice-selectedmark {fill: #a00;}',
        //'.questionhighlights .multichoice-choices li[data-choiceid="choice_1"] .multichoice-choicetext {color: black; background-color: transparent;margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; font-weight: bold; border-bottom: 2px solid #0a0;}', //  border: 2px solid #0a0; border-radius: 0.3em;
        '.questionhighlights .multichoice-choices li[data-status="selected"][data-choiceid="choice_1"] .multichoice-choicetext {color: black; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; text-shadow: 1px 1px 1px white, -1px -1px 1px white; font-weight: bold; border-bottom: 2px solid #0a0;}', //  border-radius: 0.3em; background-color: #afa;
        '.questionhighlights .multichoice-choices li[data-status="selected"]:not([data-choiceid="choice_1"]) .multichoice-choicetext {color: black; margin-left: -0.3em; margin-right: -0.3em; padding-left: 0.3em; padding-right: 0.3em; color: #a00; text-decoration: line-through;}' // background-color: #faa; border-radius: 0.3em; border: 2px solid #a00; text-shadow: 1px 1px 2px white, -1px -1px 2px white;
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    Multichoice.localization = {
        "en": {
        },
        "fi": {
        }
    };
    
    Multichoice.sounds ={
        "multichoice_correct" :"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",
        "multichoice_wrong" : "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"
        };

    Multichoice.elementinfo = {
        type: 'multichoice',
        elementtype: ['assignments','quizelement'],
        jquery: 'multichoice',
        name: 'Multichoice',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-multichoice"><path style="stroke: none;" d="M5 1 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8z m0 1 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z m0 9 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8z m0 1 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z m0 1 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z m0 8 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8z m0 1 a3 3 0 0 1 0 6 a3 3 0 0 1 0 -6z M 12 4 l15 0 l0 2 l-15 0z m0 10 l15 0 l0 2 l-15 0z m0 10 l15 0 l0 2 l-15 0z" /></svg>',
        description: {
            en: 'Multiple choice assignment',
            fi: 'Monivalintatehtävä',
            sv: 'Flervalsuppgift'
        },
        weight: 30,
        roles: ['teacher', 'author'],
        classes: ['assignments']
    };
    
    Multichoice.elementinfoold = {
        type: 'multichoise',
        elementtype: 'assignments',
        jquery: 'multichoice',
        name: 'Multichoice',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 10 l10 0 l0 -10z m0 12 l0 10 l10 0 l0 -10z m12 0 l0 10 l10 0 l0 -10z m0 -12 l0 10 l10 0 l0 -10z m-7 2 l4 6 l-8 0z m-4 12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z m16 -4 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m-4 -12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z" /></svg>',
        description: {
            en: 'Multiple choice assignment',
            fi: 'Monivalintatehtävä',
            sv: 'Flervalsuppgift'
        },
        weight: 30,
        classes: ['general', 'content']
    };

    // Register multichoice as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Multichoice.elementinfo);
        $.fn.elementset('addelementtype', Multichoice.elementinfoold);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Multichoice.elementinfo);
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Multichoice.localization);
    };
    
    var convert2new = function(options){
        options.type = 'multichoice';
        if (options.data.contents.length > 0) {
            var elid = options.data.contents[0].id;
            options.data.text = options.data.contentdata[elid].data.text;
        } else {
            options.data.text = '';
        };
        delete options.data.contents;
        delete options.data.contentdata;
        delete options.data.assignmentType;
        return options;
    }

    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            if (params.type === 'multichoise') {
                params = convert2new(params);
            }
            return this.each(function(){
                var multichoice = new Multichoice(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.multichoice = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in multichoice.');
            return false;
        }
    };
    
    $.fn.multichoise = $.fn.multichoice;
    
})(jQuery);