/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - OggVorbisEncoder
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(OggVorbisEncoder) === 'undefined' && OggVorbisEncoder.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    
    var SIZELIMIT = 3000000; // Around 2 megabytes of raw data encoded to base64.

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Media element to play and record media.
     * @requires OggVorbisEncoder.js library <https://github.com/higuma/ogg-vorbis-encoder-js>
     *     (libvorbis: BSD-like license)
     *     (js-wrapper: MIT-license)
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var MediaElement = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init mediaelement
     * @param {Object} options - initing data of the element
     ******/
    MediaElement.prototype.init = function(options){
        options = $.extend(true, {}, MediaElement.defaults, options);
        this.settings = options.settings;
        this.type = options.type;
        this.setMode(options.settings.mode);
        if (options.data.media.length <= SIZELIMIT) {
            this.data = JSON.parse(JSON.stringify(options.data));
        } else {
            this.data = JSON.parse(JSON.stringify(MediaElement.defaults.data));
            alert(ebooklocalizer.localize('mediaelement:file too large', this.settings.uilang) + '\nIn init');
        };
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        switch (this.type){
            case 'videoelement':
                this.data.mediatype = 'video';
                break;
            case 'audioelement':
                this.data.mediatype = 'audio';
                break;
            default:
                this.data.mediatype = 'media';
                break;
        };
        this.setAttrs();
    };
    
    /******
     * Set style-tag if needed.
     ******/
    MediaElement.prototype.setStyles = function(){
        if ($('head style#mediaelement-style').length === 0) {
            $('head').append('<style id="mediaelement-style" type="text/css">'+MediaElement.styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    MediaElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = MediaElement.modes[mode] || MediaElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Set attributes
     ******/
    MediaElement.prototype.setAttrs = function(){
        this.place.addClass('mediaelement');
        this.place.removeClass('mediaelement-recordmode mediaelement-isplaying');
        this.isplaying = false;
        if (!this.editable && !this.data.controls) {
            this.place.addClass('mediaelement-nocontrols');
        } else {
            this.place.removeClass('mediaelement-nocontrols');
        }
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    MediaElement.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the MediaElement element in different modes
     ******/
    MediaElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
            this.place.attr('uilang', this.settings.uilang);
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable();
        };
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    MediaElement.prototype.view = function(){
        this.place.empty().html(MediaElement.templates.view);
        this.setAttrs();
        this.place.removeClass('ffwidget-background');
        var element = this;
        var marea = this.place.find('.mediaelement-mediaarea');
        var controls = (this.data.controls ? ' controls' : '');
        var autoplay = (this.data.autoplay ? ' autoplay' : '');
        var dataurl = this.data.media.replace(/;base64/, ';rnd=' + this.getRnd(1, 100000) + ';base64');
        var media;
        if ((this.data.mediatype === 'audio' || this.data.mediatype === 'video') && this.data.media) {
            media = $('<'+this.data.mediatype+' '+autoplay+' src="' + dataurl + '">');
            marea.append(media);
            media.trigger('canplay');
            media[0].oncanplaythrough = function(){media.trigger('mediaready');};
            media[0].ontimeupdate = function(){
                media.trigger('mediatime', [this.currentTime]);
            };
            media[0].onseeked = function(){
                //element.currentBreakpoint = null;
            };
            media[0].onplay = function(){
                media.trigger('playing');
            };
            media[0].onpause = function(){
                media.trigger('paused');
            };
            media[0].onended = function(){
                media.trigger('mediaended');
            };
        //} else {
            //marea.append('<div class="mediaelement-empty">'+MediaElement.icons[this.data.mediatype]+'</div>')
        };
        this.mediatag = media;
        if (this.data.controls) {
            this.showControls();
        };
    };
    
    /******
     * Show the element in editable mode.
     ******/
    MediaElement.prototype.edit = function(){
        var uilang = this.settings.uilang;
        this.place.empty().html(MediaElement.templates.edit);
        this.setAttrs();
        this.place.addClass('ffwidget-background');
        this.showControlbar();
        var element = this;
        var marea = this.place.find('.mediaelement-mediaarea');
        //var controls = (this.data.controls ? ' controls' : '');
        //var autoplay = (this.data.autoplay ? ' autoplay' : '');
        var dataurl = this.data.media.replace(/;base64/, ';rnd=' + this.getRnd(1, 100000) + ';base64');
        var media;
        if ((this.data.mediatype === 'audio' || this.data.mediatype === 'video') && this.data.media) {
            media = $('<'+this.data.mediatype+' src="' + dataurl + '">');
            marea.append(media);
            media.trigger('canplay');
            media[0].oncanplaythrough = function(){media.trigger('mediaready');};
            media[0].ontimeupdate = function(){
                media.trigger('mediatime', [this.currentTime]);
            };
            media[0].onseeked = function(){
                //element.currentBreakpoint = null;
            };
            media[0].onplay = function(){
                media.trigger('playing');
            };
            media[0].onpause = function(){
                media.trigger('paused');
            };
            media[0].onended = function(){
                media.trigger('mediaended');
            };
        //} else {
        //    marea.append('<div class="mediaelement-empty">'+MediaElement.icons[this.data.mediatype]+'</div>')
        };
        this.mediatag = media;
        this.showControls();
    };
   
    /******
    * Update the data in recording monitor.
    * @param {Object} data - data of format {buffer: <Array of numbers>, time: <time in ms>}
    ******/
    MediaElement.prototype.updateMonitor = function(data){
        var buffer = data.buffer || [];
        var length = buffer.length | 0;
        var svg = [
            '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" align="none" width="'+length+'" height="100" viewBox="0 -10 '+length+' 20" class="mediaelement-wavemonitor">'
        ];
        var path = ['<path style="stroke: black; fill: none; stroke-width: 4;" d="M0 10'];
        for (var i = 0; i < length; i++){
            path.push('L'+ i +' ' + (100 * buffer[i]))
        };
        path.push('" />');
        svg.push(path.join(' '));
        svg.push('</svg>');
        this.place.find('.mediaelement-visualizer').html(svg.join(''));
        var time = data.time || 0;
        var seconds = Math.floor(time / 1000);
        var msecs = time % 1000;
        var dsecs = Math.floor(msecs / 100);
        var minutes = Math.floor(seconds / 60);
        var seconds = seconds % 60;
        this.place.find('.mediaelement-timer').html(minutes +':' + ('00' + seconds).slice(-2) + '.' + dsecs);
    }
    
    /******
     * Start recordingmode
     ******/
    MediaElement.prototype.startRecordingMode = function(mediatype){
        var element = this;
        var eventarea = this.place.children('.mediaelement-eventarea');
        if (mediatype === 'audio') {
            if (!this.recorder || !this.recorder.mediatype() === 'audio') {
                this.recorder = new AudioRecorder({}, eventarea);
            }
        } else {
        };
    };
    
    /******
     * Stop recordingmode
     ******/
    MediaElement.prototype.stopRecordingMode = function(){
        var dataurl = '';
        if (this.recorder) {
            dataurl = this.recorder.getDataurl();
            if (dataurl) {
                this.data.media = dataurl;
                var mediatype = this.recorder.mediatype();
                this.data.audio = true;
                this.data.video = (mediatype === 'video');
                this.clearBreakpoints();
                this.show();
            };
            this.recorder.close();
            delete this.recorder;
            this.changed();
        };
    };
    
    /******
     * Show recording user interface
     ******/
    MediaElement.prototype.showRecui = function(mediatype){
        this.place.addClass('mediaelement-recordmode');
        this.stop();
    };
    
    /******
     * Hide recording user interface
     ******/
    MediaElement.prototype.hideRecui = function(){
        this.place.removeClass('mediaelement-recordmode');
    };
    
    /******
     * Show controlbar for editing
     ******/
    MediaElement.prototype.showControlbar = function(){
        var uilang = this.settings.uilang;
        var controlbar = this.place.find('.mediaelement-controlbar');
        controlbar.empty();
        var autoplay = (this.data.autoplay ? 'checked' : '');
        var controls = (this.data.controls ? 'checked' : '');
        var recbutton;
        switch (this.data.mediatype){
            case 'audio':
                recbutton = '<button class="mediaelement-recording-button mediaelement-button" data-action="recordaudio">'+MediaElement.icons.microphone+'</button>'
                break;
            case 'video':
                // Disabled for now
                //recbutton = '<button class="mediaelement-recording-button mediaelement-button" data-action="recordvideo">'+MediaElement.icons.webcam+'</button>'
                break;
            default:
                recbutton = '';
                break;
        }
        var html = ['<form>',
            '<label class="mediaelement-fileinput"><input type="file" data-inputtype="file" value=""></input><span class="mediaelement-filelabel">'+MediaElement.icons.file+'</span></label>',
            recbutton,
            '<label class="mediaelement-autoplayinput"><input type="checkbox" data-key="autoplay" '+autoplay+'></input> <span class="mediaelement-checkboxlabel">'+ebooklocalizer.localize('mediaelement:autoplay', uilang) + '</span></label>',
            '<label class="mediaelement-controlsinput"><input type="checkbox" data-key="controls" '+controls+'></input> <span class="mediaelement-checkboxlabel">'+ebooklocalizer.localize('mediaelement:showcontrols', uilang) + '</span></label>'
        ];

        html.push('</form>');
        controlbar.append(html.join('\n'));
    };
    
    /******
     * Show custom controls for playing
     ******/
    MediaElement.prototype.showControls = function(){
        var controls = this.place.find('.mediaelement-controlarea');
        var buttons = [
            '    <button class="mediaelement-button ffwidget-button" data-action="play">'+MediaElement.icons.play+'</button>',
            '    <button class="mediaelement-button ffwidget-button" data-action="pause">'+MediaElement.icons.pause+'</button>',
            '    <button class="mediaelement-button ffwidget-button" data-action="seekprev">'+MediaElement.icons.seekprev+'</button>',
            '    <button class="mediaelement-button ffwidget-button" data-action="seeknext">'+MediaElement.icons.seeknext+'</button>',
        ];
        if (this.editable) {
            buttons.push('    <button class="mediaelement-button ffwidget-button" data-action="addbreakpoint">'+MediaElement.icons.breakpoint+'</button>');
        };
        var html = [
            '<div class="mediaelement-mediacontrolbuttons">',
            buttons.join(''),
            '</div>',
            '<div class="mediaelement-track">',
            '    <div class="mediaelement-progressbar ffwidget-primarycolor">',
            '        <div class="mediaelement-progresshandle ffwidget-primarycolor"></div>',
            '        <div class="mediaelement-progressinfo">0:00</div>',
            '    </div>',
            '    <div class="mediaelement-breakpointarea"></div>',
            '</div>',
            '<div class="mediaelement-mediacontrolinfo">',
            '    <div class="mediaelement-mediacontrol-duration">0:00</div>',
            '</div>'
        ];
        controls.html(html.join('\n'));
        this.progressbar = controls.find('.mediaelement-progressbar');
        this.progressinfo = this.progressbar.children('.mediaelement-progressinfo');
        this.durationinfo = controls.find('.mediaelement-mediacontrol-duration');
        this.breakpointarea = controls.find('.mediaelement-breakpointarea');
        this.updateProgress();
    };
    
    /******
     * Show breakpoints
     ******/
    MediaElement.prototype.showBreakpoints = function(){
        var uilang = this.settings.uilang;
        var breakpoints = this.data.breakpoints;
        if (breakpoints.length > 0 && (this.editable || this.data.controls)) {
            var duration, bppos, relpos;
            if (this.mediatag && this.mediatag[0]) {
                duration = this.mediatag[0].duration;
            };
            var html = [];
            for (var i = 0, len = breakpoints.length; i < len; i++) {
                bppos = breakpoints[i];
                relpos = 100 * bppos / duration;
                if (0 <= relpos && relpos <= 100) {
                    html.push('<div class="mediaelement-breakpoint" data-bptime="' + escapeHTML(breakpoints[i]) + '" title="' + escapeHTML(breakpoints[i]) + ' s" style="left: '+relpos+'%"><div class="mediaelement-breakpoint-info ffwidget-background"><div class="mediaelement-breakpoint-infotext">'+escapeHTML(breakpoints[i])+' s</div><button class="mediaelement-button" data-action="removebreakpoint">'+MediaElement.icons.remove+'</button></div></div>');
                };
            };
            this.breakpointarea.html(html.join(''));
            this.updateProgress();
        } else if (this.breakpointarea) {
            this.breakpointarea.html('');
        };
    };
    
    /******
     * Play
     ******/
    MediaElement.prototype.play = function(){
        if (this.mediatag) {
            var element = this;
            this.mediatag[0].play();
            setTimeout(function(){element.currentBreakpoint = null;}, 601);
        };
    };
    
    /******
     * Pause
     ******/
    MediaElement.prototype.pause = function(){
        if (this.mediatag) {
            this.mediatag[0].pause();
        };
    };
    
    /******
     * Stop
     ******/
    MediaElement.prototype.stop = function(){
        if (this.mediatag) {
            var tag = this.mediatag[0];
            tag.pause();
            tag.currentTime = 0;
        };
    };
    
    /******
     * Seek
     ******/
    MediaElement.prototype.seek = function(time){
        if (this.mediatag) {
            var tag = this.mediatag[0];
            if (typeof(time) === 'number' && 0 <= time && time <= tag.duration) {
                tag.currentTime = time;
            };
        };
    };
    
    /******
     * Seek by the ratio
     * @param {Number} ratio - ratio between position and total duration
     ******/
    MediaElement.prototype.seekRatio = function(ratio){
        if (this.mediatag) {
            var tag = this.mediatag[0];
            var duration = tag.duration || 0;
            var time = ratio * duration;
            this.seek(time);
        };
    };
    
    /******
     * Seek to the next break
     ******/
    MediaElement.prototype.seekNext = function(){
        if (this.mediatag && this.mediatag[0]) {
            var tag = this.mediatag[0];
            var time = tag.currentTime;
            var position;
            for (var i = 0, len = this.data.breakpoints.length; i < len; i++){
                if (this.data.breakpoints[i] > time) {
                    position = this.data.breakpoints[i];
                    break;
                };
            };
            position = position || tag.duration;
            this.seek(position);
            this.currentBreakpoint = position;
            this.pause();
        };
    };
    
    /******
     * Seek to the previous break
     ******/
    MediaElement.prototype.seekPrev = function(){
        if (this.mediatag && this.mediatag[0]) {
            var tag = this.mediatag[0];
            var time = tag.currentTime;
            var position;
            for (var i = this.data.breakpoints.length -1; i > -1; i--){
                if (this.data.breakpoints[i] < time) {
                    position = this.data.breakpoints[i];
                    break;
                };
            };
            position = position || 0;
            this.seek(position);
            this.currentBreakpoint = position;
            this.pause();
        };
    };
    
    /******
     * Start recording
     ******/
    MediaElement.prototype.startRecording = function(){
        this.place.find('.mediaelement-recuiarea').addClass('mediaelement-recording');
        this.recorder.start();
    }

    /******
     * Stop recording
     ******/
    MediaElement.prototype.stopRecording = function(){
        this.place.find('.mediaelement-recuiarea').removeClass('mediaelement-recording');
        this.recorder.stop();
    }
    
    /******
     * Stop recording
     ******/
    MediaElement.prototype.cancelRecording = function(){
        this.place.find('.mediaelement-recuiarea').removeClass('mediaelement-recording');
        this.recorder.cancel();
    }
    
    /******
     * Pause recording
     ******/
    MediaElement.prototype.pauseRecording = function(){
        this.recorder.pause();
    }
    
    /******
     * Play recording
     ******/
    MediaElement.prototype.playRecording = function(){
        this.recorder.play();
    }
    
    /******
     * Clear recording
     ******/
    MediaElement.prototype.clearRecording = function(){
        this.recorder.clear();
    }

    /******
     * Get the length of recording
     ******/
    MediaElement.prototype.clearRecording = function(){
        return this.recorder.getLength();
    }
    
    /******
     * Add breakpoint
     * @param {Number} position - time as seconds (optional)
     ******/
    MediaElement.prototype.addBreakpoint = function(position){
        if (typeof(position) !== 'number') {
            if (this.mediatag && this.mediatag[0]) {
                position = this.mediatag[0].currentTime;
            } else {
                position = 0;
            };
        };
        position = Math.floor(position * 2) / 2; // Round to halfs.
        var duration = 0;
        if (this.mediatag && this.mediatag[0]) {
            duration = this.mediatag[0].duration;
        }
        if (position > 0 && position < duration) {
            if (this.data.breakpoints.indexOf(position) === -1) {
                this.data.breakpoints.push(position);
                this.data.breakpoints.sort(function(a, b){return a > b ? 1 : a < b ? -1 : 0;});
                this.changed();
                this.currentBreakpoint = position;
                this.showBreakpoints();
            };
        };
    };
    
    /******
     * Remove breakpoint
     * @param {Number} position - time as seconds
     ******/
    MediaElement.prototype.removeBreakpoint = function(position){
        if (typeof(position) === 'number') {
            var index = this.data.breakpoints.indexOf(position);
            if (index !== -1) {
                this.data.breakpoints.splice(index, 1);
                this.changed();
                this.showBreakpoints();
            };
        };
    };
    
    /******
     * Clear all breakpoints
     ******/
    MediaElement.prototype.clearBreakpoints = function(){
        this.data.breakpoints = [];
    }
    
    /******
     * Stop playing, if we are in a breakpoint.
     * @param {Number} time - the current time of playing
     ******/
    MediaElement.prototype.stopIfBreakpoint = function(time){
        if (time !== this.currentBreakpoint && this.data.breakpoints.indexOf(time) !== -1) {
            this.pause();
            this.seek(time);
            this.currentBreakpoint = time;
        };
    };
    
    /******
     * Update progressbar
     * @param {Number} time - the current time of playing
     ******/
    MediaElement.prototype.updateProgress = function(){
        if ((this.editable || this.data.controls) && this.mediatag && this.mediatag[0]) {
            var media = this.mediatag[0];
            var time = media.currentTime;
            var duration = media.duration;
            var width = 100 * time / duration;
            this.progressbar.css('width', width + '%');
            var timesecs = Math.round(time);
            var minutes = Math.floor(timesecs / 60);
            var seconds = timesecs % 60;
            this.progressinfo.html(minutes + ':' + ('00' + seconds).slice(-2));
            var dursecs = Math.round(duration);
            var dminutes = Math.floor(dursecs / 60);
            var dseconds = dursecs % 60;
            if (!isNaN(dminutes)) {
                this.durationinfo.html(dminutes + ':' + ('00' + dseconds).slice(-2));
            };
        };
    };
    
    /******
     * Start moving with mouse
     * @param {Object} data - {startpos: position, total: total, startx: startx}
     ******/
    MediaElement.prototype.startMoving = function(data){
        var element = this;
        this.moveOffset = {
            trackleft: data.startx - data.startpos,
            tracktotal: data.total
        };
        $('body').addClass('mediaelementhandlemoving')
            .off('mousemove.mediaelement')
            .on('mousemove.mediaelement', function(event, data){
                var leftx = event.clientX;
                var offset = element.moveOffset;
                var ratio = (leftx - offset.trackleft) / offset.tracktotal;
                element.seekRatio(ratio);
            })
            .off('mouseup.mediaelement')
            .on('mouseup.mediaelement', function(event, data){
                element.stopMoving();
            });
    };
    
    /******
     * Stop moving with mouse
     ******/
    MediaElement.prototype.stopMoving = function(){
        $('body').removeClass('mediaelementhandlemoving')
            .off('mousemove.mediaelement')
            .off('mouseup.mediaelement');
    };
    
    /******
     * Remove all event handlers
     ******/
    MediaElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    MediaElement.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in MediaElement.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        }).on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        }).on('getindexmetadata', function(event){
            event.stopPropagation();
            var data = element.getIndexData();
            element.place.data('[[elementindexmetadata]]', data);
        });
        
        this.place.on('canplay', '.mediaelement-mediaarea audio, .mediaelement-mediaarea video', function(event, data){
            element.place.find('.mediaelement-infoarea').empty();
        }).on('play', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.play();
        }).on('pause', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.pause();
        }).on('playtoggle', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            if (element.isplaying) {
                element.stop();
            } else {
                element.play();
            };
        }).on('stop', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.stop();
        }).on('seek', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.seek(data);
        }).on('seekprev', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.seekPrev();
        }).on('seeknext', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.seekNext();
        }).on('mediatime', function(event, time){
            event.stopPropagation();
            event.preventDefault();
            var mediatime = Math.floor(2 * time) / 2;
            element.updateProgress();
            element.stopIfBreakpoint(mediatime);
        }).on('playing', function(event, time){
            //event.stopPropagation();
            event.preventDefault();
            element.isplaying = true;
            element.place.addClass('mediaelement-isplaying');
        }).on('paused', function(event, time){
            //event.stopPropagation();
            event.preventDefault();
            element.isplaying = false;
            element.place.removeClass('mediaelement-isplaying');
        }).on('click', '.mediaelement-track', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var track = $(this);
            var total = track.width();
            var trackleft = track.offset().left;
            var position = event.clientX - trackleft;
            var ratio = position / total;
            element.seekRatio(ratio);
        }).on('click', '.mediaelement-button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            button.trigger(action);
        }).on('mousedown', '.mediaelement-progresshandle', function(event, data){
            event.stopPropagation();
            var handle = $(this);
            var proggressbar = handle.parent();
            var track = handle.closest('.mediaelement-track');
            var total = track.width();
            var position = event.offsetX + proggressbar.width();
            var startx = event.clientX;
            element.startMoving({startpos: position, total: total, startx: startx});
        }).on('mediaready', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.showBreakpoints();
        });
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    MediaElement.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    MediaElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('change', '.mediaelement-controlbar .mediaelement-input', function(event, data){
            var input = $(this);
            var value = input.val();
            var key = input.attr('data-key');
            element[key] = value;
            element.changed();
            element.show();
        }).on('click', '.mediaelement-controlbar input[type="file"]', function(e){
            var uilang = element.settings.uilang;
            if (!confirm(ebooklocalizer.localize('mediaelement:replaces_old', uilang) + '\n' + ebooklocalizer.localize('mediaelement:areyousure', uilang))) {
                e.stopPropagation();
                e.preventDefault();
            };
        }).on('dragover', function(event, data){
            event.preventDefault();
        }).on('drop', function(event, data){
            event.preventDefault();
            event.stopPropagation();
            var dataurl = '';
            var oevent = event.originalEvent;
            var files = oevent.dataTransfer.files;
            if (files.length > 0) {
                element.loadFile(files[0]);
            };
        //}).on('change', '.mediaelement-controlbar input[data-inputtype="recordaudio"]', function(event, data){
        //    event.stopPropagation();
        //    event.preventDefault();
        //    var input = $(this);
        //    var selected = input.is(':checked');
        //    if (selected) {
        //        element.showRecui('audio');
        //    } else {
        //        element.hideRecui();
        //    }
        }).on('click', '.mediaelement-recbuttons button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            button.trigger(action);
        }).on('start_recording', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.startRecording();
        }).on('stop_recording', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.stopRecording();
        }).on('cancel_recording', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.cancelRecording();
        }).on('enterrecmode', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.showRecui();
        }).on('dataurlready', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.stopRecordingMode();
            element.hideRecui();
        }).on('recordaudio', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.startRecordingMode('audio');
        }).on('recordvideo', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.startRecordingMode('video');
        }).on('recordingmonitordata', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.updateMonitor(data);
        }).on('addbreakpoint', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            element.addBreakpoint();
        }).on('removebreakpoint', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(event.target);
            var bpoint = button.closest('[data-bptime]');
            var bptime = bpoint.attr('data-bptime');
            bptime = parseFloat(bptime) || '';
            element.removeBreakpoint(bptime);
        }).on('change', '.mediaelement-controlbar input[type="file"]', function(e){
            var fileinput = $(this);
            //fileinput.next('input[type="hidden"]').val(fileinput.val());
            var files = fileinput.get(0).files;
            if (files.length > 0) {
                element.loadFile(files[0]);
            }
        }).on('change', '.mediaelement-controlbar input[type="checkbox"]', function(event, data){
            var input = $(this);
            var key = input.attr('data-key');
            var value = input.is(':checked');
            element.data[key] = value;
            element.changed();
        }).on('openfile', function(event, data){
            event.stopPropagation();
            if (data.file && (data.file instanceof File)) {
                element.loadFile(data.file);
            };
        });
        this.place.find('.mediaelement-breakpointarea').on('click', '.mediaelement-button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-action');
            button.trigger(action);
        });
    };
    
    /******
     * Update the metadata of the element.
     ******/
    MediaElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Trigger element_changed event.
     ******/
    MediaElement.prototype.changed = function(){
        this.updateMetadata();
        this.place.trigger('element_changed', {type: 'mediaelement'});
    }

    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "markdownelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {"text": this.text}
     *   }
     ******/
    MediaElement.prototype.getData = function(){
        var result = {
            type: this.type,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return result;
    }
    
    /******
     * Get some metadata for indexing
     ******/
    MediaElement.prototype.getIndexData = function(){
        var result = {
            notes: this.data.notes || '',
            file: this.data.file || 'recording.ogg',
            url: this.data.url || '',
            name: this.data.name || '',
            author: this.data.author || '',
            source: this.data.source || '',
            license: this.data.license || '',
            thumbnail: MediaElement.icons[this.data.mediatype]
        };
        return result;
    };
    
    /******
     * Load media from file
     * @param {File} file - the file to be opened
     ******/
    MediaElement.prototype.loadFile = function(file){
        var uilang = this.settings.uilang;
        var element = this;
        var dataurl = '';
        var filename = file.name || '';
        var infoarea = this.place.find('.mediaelement-infoarea');
        if (file && ((file.type.indexOf('audio') === 0 && this.data.mediatype === 'audio') ||
                     (file.type.indexOf('video') === 0 && this.data.mediatype === 'video') ||
                     ((/\/ogg$/).test(file.type) && this.data.mediatype === 'audio'))) {
            var filetype = file.type.split('/')[1];
            var mediatype = file.type.split('/')[0];
            var reader = new FileReader();
            reader.onloadend = function(ev){
                dataurl = reader.result;
                if (dataurl.length <= SIZELIMIT) {
                    element.data.media = dataurl;
                    element.data.type = filetype;
                    element.data.file = filename;
                    switch (mediatype){
                        case 'audio':
                            element.data.audio = true;
                            element.data.video = false;
                            break;
                        case 'video':
                            var mdata = atob(dataurl.split(',')[1]);
                            element.data.audio = true;
                            element.data.video = (filetype === 'ogg' ? /Content-Type: video/.test(mdata) : true);
                            // Firefox gives mime-type 'video/ogg' for all files with extension .ogg. Even for audio.
                            break;
                        default:
                            element.data.audio = true;
                            element.data.video = false;
                            break;
                    };
                    element.clearBreakpoints();
                    element.changed();
                    element.setAttrs();
                    element.show();
                } else {
                    //alert(ebooklocalizer.localize('mediaelement:file too large', uilang));
                    //infoarea.empty();
                    infoarea.html('<div class="mediaelement-note"><strong>'+ebooklocalizer.localize('mediaelement:file too large', uilang)+ '</strong>: ' +filename+ '</div>');
                }
            };
            infoarea.html('<div class="mediaelement-note">'+ebooklocalizer.localize('mediaelement:loading', uilang) +'<span class="mediaelement-note-blink">...</span></div>');
            reader.readAsDataURL(file);
        } else {
            alert(ebooklocalizer.localize('mediaelement:wrong filetype', uilang));
        };
    };
    
    /******
     * Get random number
     * @param {Number} min - minimum
     * @param {Number} max - maximum
     ******/
    MediaElement.prototype.getRnd = function(min, max){
        return Math.random() * (max - min) + min;
    }

    
    /******
     * Default settings
     ******/
    MediaElement.defaults = {
        type: 'mediaelement',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            audio: false,
            video: false,
            media: '',         // url (http/data/file/...)
            controls: true,
            autoplay: false,
            breakpoints: []
        },
        settings: {
            mode: 'view',
            uilang: 'en',
            lang: 'en',
            role: 'student'
        }
    }
    
    
    /******
     * Modes
     ******/
    MediaElement.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    /******
     * Icons
     ******/
    MediaElement.icons = {
        height: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-height"><path style="stroke: none;" d="M8 1 l14 0 l0 2 l-7 0 l4 4 l-2.5 0 l0 16 l2.5 0 l-4 4 l7 0 l0 2 l-14 0 l0 -2 l7 0 l-4 -4 l2.5 0 l0 -16 l-2.5 0 l4 -4 l-7 0z" /></svg>',
        file: "<svg xmlns='http://www.w3.org/2000/svg' version='1.1' width='30' height='30' viewBox='0 0 30 30' class='mini-icon mini-icon-folder'><path stroke='none' fill='black' d='M2 6 l5 0 l2 2 l14 0 l0 2 l-16 0 l-4 14 l5 -12 l22 0 l-5 13 l-23 0z'></path></svg>",
        microphone: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-microphone"><path class="reccolor highlightcolor" style="stroke: none;" d="M10 6 a5 5 0 0 1 10 0 l0 7 a5 5 0 0 1 -10 0z m-2 5 l0 2 a7 7 0 0 0 14 0 l0 -2 l2 0 l0 2 a9 9 0 0 1 -8 9 l0 4 l4 0 l0 2 l-10 0 l0 -2 l4 0 l0 -4 a9 9 0 0 1 -8 -9 l0 -2z" /></svg>',
        webcam: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-webcam"><path class="mini-icon-maincontent" style="stroke: none;" d="M15 2 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z m0 5 a5 5 0 0 1 0 10 a5 5 0 0 1 0 -10z m0 2 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m1 1 a1 1 0 0 1 0 2 a1 1 0 0 1 0 -2z m-1 13 a13 13 0 0 0 8 -3 a15 4 20 0 1 0 6 a16 3 0 0 0 -16 0 a15 4 -20 0 1 0 -6 a13 13 0 0 0 8 3z" /></svg>',
        record: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-record"><path fill="red" style="stroke: none;" d="M15 2 a13 13 0 0 0 0 26 a13 13 0 0 0 0 -26z" /></svg>',
        stop: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-stop"><path style="stroke: none;" fill="#333" d="M2 2 h26 v26 h-26z" /></svg>',
        play: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-play"><path style="stroke: none;" fill="#333" d="M4 2 l22 13 l-22 13 z" /></svg>',
        pause: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-pause"><path style="stroke: none;" fill="#333" d="M2 2 h10 v26 h-10z m16 0 h10 v26 h-10z" /></svg>',
        cancelback: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-cancelback"><path class="mini-icon-maincontent" style="stroke: none;" d="M1 15 l7 -10 a2 2 0 0 1 2 -1 l17 0 a3 3 0 0 1 3 3 l0 16 a3 3 0 0 1 -3 3 l-17 0 a2 2 0 0 1 -2 -1z m13 -6 a1 1 0 0 0 -2 2 l4 4 l-4 4 a1 1 0 0 0 2 2 l4 -4 l4 4 a1 1 0 0 0 2 -2 l-4 -4 l4 -4 a1 1 0 0 0 -2 -2 l-4 4z" /></svg>',
        audio: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-audio"><path class="mini-icon-maincontent" style="stroke: none;" d="M5 12 h3 l8 -8 v22 l-8 -8 h-3z m13 -1 a8 8 0 0 1 0 8 l1 1 a9 9 0 0 0 0 -10z m3 -3 a11 11 0 0 1 0 14 l1 1 a12 12 0 0 0 0 -16z" /></svg>',
        video: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-video"><path class="mini-icon-maincontent" style="stroke: none;" d="M1 5 l28 0 l0 20 l-28 0z m4 2 l0 16 l20 0 l0 -16z m-3 1 v2 h2 v-2z m0 4 v2 h2 v-2z m0 4 v2 h2 v-2z m0 4 v2 h2 v-2z m24 0 v2 h2 v-2z m0 -4 v2 h2 v-2z m0 -4 v2 h2 v-2z m0 -4 v2 h2 v-2z m-14 1 l8 6 l-8 6 z" /></svg>',
        media: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-video"><path class="mini-icon-maincontent" style="stroke: none;" d="M1 5 l28 0 l0 20 l-28 0z m4 2 l0 16 l20 0 l0 -16z m-3 1 v2 h2 v-2z m0 4 v2 h2 v-2z m0 4 v2 h2 v-2z m0 4 v2 h2 v-2z m24 0 v2 h2 v-2z m0 -4 v2 h2 v-2z m0 -4 v2 h2 v-2z m0 -4 v2 h2 v-2z m-14 1 l8 6 l-8 6 z" /></svg>',
        add: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-add"><path class="mini-icon-maincontent" style="stroke: none;" d="M13 5 h4 v8 h8 v4 h-8 v8 h-4 v-8 h-8 v-4 h8z" /></svg>',
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-remove"><path class="mini-icon-maincontent" style="stroke: none;" d="M5 13 h20 v4 h-20 z" /></svg>',
        breakpoint: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-breakpoint"><path style="stroke: none;" d="M13 13 a6 6 0 1 1 4 0 l-2 16z m2 -2.5 a3 3 0 0 0 0 -6 a3 3 0 0 0 0 6z" /></svg>',
        seeknext: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-seekprev"><path class="mini-icon-maincontent" style="stroke: none;" d="M4 2 l22 13 v-13 h4 v26 h-4 v-13 l-22 13 z" /></svg>',
        seekprev: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-seekprev"><path class="mini-icon-maincontent" style="stroke: none;" d="M26 2 l-22 13 v-13 h-4 v26 h4 v-13 l22 13z" /></svg>'
    }

    /******
     * Templates
     ******/
    MediaElement.templates = {
        view: [
            '<div class="mediaelement-mediabox">',
            '    <div class="mediaelement-mediaarea"></div>',
            '    <div class="mediaelement-controlarea ffwidget-background"></div>',
            '</div>'
        ].join('\n'),
        edit: [
            //'<div class="mediaelement-controlbar ffwidget-background"></div>',
            '<div class="mediaelement-controlbar"></div>',
            '<div class="mediaelement-recuiarea">',
            '    <div class="mediaelement-recbuttons">',
            '        <button class="mediaelement-cancel-recording ffwidget-button" data-action="cancel_recording">'+MediaElement.icons.cancelback+'</button>',
            '        <button class="mediaelement-start-recording ffwidget-button" data-action="start_recording">'+MediaElement.icons.record+'</button>',
            '        <button class="mediaelement-stop-recording ffwidget-button" data-action="stop_recording">'+MediaElement.icons.stop+'</button>',
            '    </div>',
            '    <div class="mediaelement-recmonitor">',
            '        <div  class="mediaelement-visualizer"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" align="none" width="2048" height="100" viewBox="0 -10 2048 20" class="mediaelement-wavemonitor"><path style="stroke: black; fill: none; stroke-width: 4;" d="M0 10 l2048 0" /></svg></div>',
            '        <div  class="mediaelement-timemonitor">',
            '            <div  class="mediaelement-timer">0:00.0</div>',
            '        </div>',
            '    </div>',
            '</div>',
            '<div class="mediaelement-mediabox">',
            '    <div class="mediaelement-mediaarea"></div>',
            '    <div class="mediaelement-controlarea"></div>',
            '</div>',
            '<div class="mediaelement-infoarea"></div>',
            '<div class="mediaelement-eventarea"></div>'
            //'<div class="mediaelement-infoarea ffwidget-background"></div>'
        ].join('\n')
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    MediaElement.elementinfo = {
        type: 'mediaelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'mediaelement',
        name: 'Media element',
        icon: MediaElement.icons.audio,
        description: {
            en: 'Media element',
            fi: 'Mediaelementti',
            sv: 'Mediaelement'
        },
        roles: [],
        classes: ['viewonly'],
        weight: 500
    }
    
    MediaElement.audioelementinfo = {
        type: 'audioelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'mediaelement',
        name: 'Media element',
        icon: MediaElement.icons.audio,
        description: {
            en: 'Audio element',
            fi: 'Audioelementti',
            sv: 'Ljudelement'
        },
        roles: ['teacher', 'student'],
        classes: ['media'],
        weight: 500
    }
    
    MediaElement.videoelementinfo = {
        type: 'videoelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'mediaelement',
        name: 'Media element',
        icon: MediaElement.icons.video,
        description: {
            en: 'Video element',
            fi: 'Videoelementti',
            sv: 'Videoelement'
        },
        roles: ['teacher', 'student'],
        classes: ['media'],
        weight: 500
    }
    
    /******
     * Styles
     ******/
    MediaElement.styles = [
        '.mediaelement {margin: 0; position: relative; clear: both;}',
        '.mediaelement[data-elementmode="view"] {margin: 0 auto; padding: 5px; border-radius: 5px; max-width: 800px; min-width: 50px;}',
        '.mediaelement[data-elementmode="view"].mediaelement-nocontrols {padding: 0;}',
        '.mediaelement-empty {text-align: center;}',
        '.mediaelement-empty svg path.mini-icon-maincontent {fill: #888;}',
        // Buttons
        '.mediaelement-button {background: transparent; border: none; margin: 0; padding: 2px; width: 26px; height: 26px; cursor: pointer; display: inline-block; line-height: 100%; box-sizing: content-box; vertical-align: middle;}',
        '.mediaelement-button:hover {background-color: rgba(255,255,255,0.4);}',
        '.mediaelement-button svg {width: 20px; height: 20px;}',
        
        // Controlbar
        '.mediaelement-controlbar {text-align: left; padding: 0 3px;}',
        '.mediaelement-controlbar svg {height: 20px; width: auto; display: inline-block;}',
        '.mediaelement-controlbar form {text-align: left; display: inline-block;}',
        '.mediaelement-controlbar form > * {display: inline-block; vertical-align: middle;}',
        '.mediaelement-controlbar input[type="file"] {display: none;}',
        '.mediaelement-controlbar .mediaelement-filelabel {display: inline-block; cursor: pointer;}',
        '.mediaelement-controlbar .mediaelement-recording-button {display: inline-block; cursor: pointer;}',
        '.mediaelement-controlbar .mediaelement-recording-button input[type="checkbox"] {display: none;}',
        '.mediaelement-controlbar .mediaelement-recording-button input[type="checkbox"]:checked + svg .reccolor {fill: red;}',
        '.mediaelement-recordmode .mediaelement-controlbar {display: none;}',
        // Mediaarea
        '.mediaelement-mediaarea audio {display: block; width: 100%; margin-top: 10px;}',
        '.mediaelement-mediaarea video {display: block; margin: 0 auto;}',
        '.mediaelement-recordmode .mediaelement-mediabox {display: none;}',
        // Infoarea
        '.mediaelement-infoarea {text-align: center; padding: 2px 10px; min-height: 1em; font-family: monospace; font-size: 80%;}',
        '.mediaelement-infoarea .mediaelement-note .mediaelement-note-blink {animation: blink 1s step-start 0s infinite; -webkit-animation: blink 1s step-start 0s infinite;}',
        '@keyframes blink {50% {opacity: 0.0;}} @-webkit-keyframes blink {50% {opacity: 0.0;}}',
        // Recording ui area
        '.mediaelement-recuiarea {display: none;}',
        '.mediaelement-recordmode .mediaelement-recuiarea {margin: 0 1em; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; align-items: center;}',
        '.mediaelement-recuiarea .mediaelement-stop-recording {display: none}',
        '.mediaelement-recuiarea.mediaelement-recording .mediaelement-stop-recording {display: inline-block}',
        '.mediaelement-recuiarea.mediaelement-recording .mediaelement-start-recording {display: none}',
        
        '.mediaelement-recbuttons {white-space: nowrap;}',
        
        '.mediaelement-recmonitor {min-width: 10em; flex-grow: 1; font-weight: bold; font-family: monospace; font-size: 120%; min-height: 30px; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; ms-flex-flow: row nowrap; flex-flow: row nowrap;}',
        '.mediaelement-recmonitor svg.mediaelement-wavemonitor {width: 100%; height: 100%;}',
        '.mediaelement-visualizer {background: #eee; border: 1px solid #aaa; margin: 10px; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1;}',
        '.mediaelement-timer {min-width: 6em; text-align: right; padding: 0.2em; margin: 10px; font-size: 150%; background-color: #ddd; box-shadow: inset 1px 1px 2px rgba(0,0,0,0.5), inset -1px -1px 2px rgba(255,255,255,0.5);}',
        
        // Controls
        '.mediaelement-controlarea {margin: 0.2em 0.5em; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; align-items: center;  -webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none;}',
        '.mediaelement[data-elementmode="view"] .mediaelement-controlarea {padding: 4px 8px; border-radius: 4px;}',
        '.mediaelement[data-elementmode="view"].mediaelement-nocontrols .mediaelement-controlarea {padding: 0;}',
        '.mediaelement-mediacontrolbuttons {flex-grow: 0; flex-shrink: 0;}',
        '.mediaelement-mediacontrolbuttons .mediaelement-button svg {width: 24px; height: 24px;}',
        '.mediaelement-track {margin: 0 12px; background-color: #777; height: 6px; border-radius: 3px; position: relative; flex-grow: 1; flex-shrink: 0; transition: height 0.2s;}',
        '.mediaelement-progressbar {margin: 0; position: absolute; left: 0; top: 0; height: 6px; background-color: #a00; border-radius: 3px; transition: height 0.2s;}',
        '.mediaelement:hover .mediaelement-track {height: 8px;}',
        '.mediaelement:hover .mediaelement-progressbar {height: 8px;}',
        
        '.mediaelement-progresshandle {width: 0; height: 0; border-radius: 50%; background-color: #a00; position: absolute; right: 0; top: 3px; transition: width 0.1s, height 0.1s, margin 0.1s, top 0.1s; cursor: pointer; z-index: 2;}',
        '.mediaelement-track:hover .mediaelement-progresshandle {width: 16px; height: 16px; margin-right: -8px; top: -4px;}',
        
        '.mediaelement-progressinfo {display: none; font-family: monospace; font-size: 80%; color: black; background-color: #eee; position: absolute; top: 14px; padding: 3px; border: 1px solid #aaa; border-radius: 3px; right: -16px;}',
        '.mediaelement-mediacontrol-duration {font-family: monospace; font-size: 90%; padding: 0.2em;}',
        '.mediaelement:hover .mediaelement-progressinfo {display: block;}',
        '.mediaelement-button[data-action="pause"] {display: none;}',
        '.mediaelement-isplaying .mediaelement-button[data-action="pause"] {display: inline-block;}',
        '.mediaelement-isplaying .mediaelement-button[data-action="play"] {display: none;}',
        
        // Breakpoints
        '.mediaelement-breakpointarea {padding: 0.1em 0.5em 0.5em 0.5em;}',
        '.mediaelement-breakpoint {position: absolute; height: 10px; width: 4px; top: -3px; left: -3px; background-color: #fd0; border: 1px solid black; transition: height 0.2s; z-index: 1;}',
        '.mediaelement:hover .mediaelement-breakpoint {height: 16px;}',
        '.mediaelement-breakpoint-info {display: none; position: absolute; top: 10px; left: -10px; background-color: #ddd; color: black; border-radius: 4px; padding: 3px; box-shadow: 1px 1px 2px rgba(0,0,0,0.4); text-align: center;}',
        '.mediaelement-breakpoint:hover .mediaelement-breakpoint-info {display: block;}',
        '.mediaelement-breakpoint-infotext {white-space: nowrap;}',
        //'.mediaelement-bptrack {margin: 0.2em 0; background-color: #777; height: 6px; border-radius: 3px; position: relative;}',
        //'.mediaelement-progressbar {margin: 0; position: absolute; left: 0; top: 0; height: 6px; background-color: #a00; border-radius: 3px;}',
        //'.mediaelement-breakpoint {position: absolute; height: 10px; width: 4px; top: -3px; left: -3px; background-color: #fd0; border: 1px solid black;}',
        ''
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    MediaElement.localization = {
        "en": {
            "mediaelement:audio": "Audio",
            "mediaelement:replaces_old": "The new file replaces the old one.",
            "mediaelement:areyousure": "Are you sure?",
            "mediaelement:loading": "Loading",
            "mediaelement:autoplay": "Autoplay",
            "mediaelement:showcontrols": "Show controls",
            "mediaelement:breakpoints": "Breakpoints",
            "mediaelement:file too large": "File too large",
            "mediaelement:wrong filetype": "Wrong filetype.\nSorry!"
        },
        "fi": {
            "mediaelement:audio": "Ääni",
            "mediaelement:replaces_old": "Uusi tiedosto korvaa aiemman.",
            "mediaelement:areyousure": "Oletko varma?",
            "mediaelement:loading": "Ladataan",
            "mediaelement:autoplay": "Soita suoraan",
            "mediaelement:showcontrols": "Näytä ohjaimet",
            "mediaelement:breakpoints": "Pysäytyskohdat",
            "mediaelement:file too large": "Tiedosto on liian suuri",
            "mediaelement:wrong filetype": "Väärä tiedostotyyppi"
        },
        "sv": {
            "mediaelement:audio": "Ljud",
            "mediaelement:replaces_old": "Ny fil ersätter den gammal.",
            "mediaelement:areyousure": "Är du sekärt?",
            "mediaelement:loading": "Laddar",
            "mediaelement:autoplay": "Autoplay",
            "mediaelement:showcontrols": "Visa kontrollpanel",
            "mediaelement:breakpoints": "Brytpunkter",
            "mediaelement:file too large": "Filen är för stor",
            "mediaelement:wrong filetype": "Fel filtyp"
        }
    }
    
    
    
    var AudioRecorder = function(config, eventplace){
        config = $.extend(true, {}, AudioRecorder.defaults, config);
        this.eventplace = eventplace;
        this.maxLength = config.maxLength;
        this.quality = config.quality;
        this.channels = config.channels;
        this.recording = false;
        this.dataurl = '';
        this.blob = false;
        this.buffers = [];
        this.recLength = 0;
        this.bufferSize = 2048;
        if (!navigator.getUserMedia) {
            navigator.getUserMedia = (navigator.getUserMedia || navigator.webkitGetUserMedia ||
                                      navigator.mozGetUserMedia || navigator.msGetUserMedia);
        };
        var recorder = this;
        var AudioContext = window.AudioContext || window.webkitAudioContext;
        this.context = new AudioContext();
        this.sampleRate = this.context.sampleRate;
        navigator.getUserMedia({audio: true}, function(stream){
            recorder.eventplace.trigger('enterrecmode');
            recorder.mediastream = stream;
            recorder.audioInput = recorder.context.createMediaStreamSource(stream);
            recorder.node = recorder.context.createScriptProcessor(recorder.bufferSize, 1, 1);
            recorder.node.onaudioprocess = function(e){
                var buff = new Float32Array(e.inputBuffer.getChannelData(0));
                if (recorder.recording) {
                    recorder.buffers.push(buff);
                    recorder.recLength += recorder.bufferSize;
                    var rectime = recorder.recLength / recorder.sampleRate * 1000;
                    if (rectime > recorder.maxLength) {
                        recorder.eventplace.trigger('stop_recording');
                    }
                };
                if (recorder.eventplace) {
                    recorder.eventplace.trigger('recordingmonitordata', [{buffer: new Float32Array(buff), time: Math.floor(recorder.recLength / recorder.sampleRate * 1000)}]);
                };
            };
            recorder.audioInput.connect(recorder.node);
            recorder.node.connect(recorder.context.destination);
        }, function(err){
            console.log('Error: ', err);
        });
    };
    
    AudioRecorder.prototype.start = function(){
        this.recording = true;
    }
    
    AudioRecorder.prototype.stop = function(){
        var recorder = this;
        this.recording = false;
        this.blob = this.getOgg();
        if (this.blob) {
            var reader = new window.FileReader();
            reader.onloadend = function(){
                recorder.dataurl = reader.result;
                recorder.eventplace.trigger('dataurlready');
            };
            reader.readAsDataURL(this.blob);
        };
    };
    
    AudioRecorder.prototype.cancel = function(){
        this.dataurl = '';
        this.eventplace.trigger('dataurlready');
    };
    
    AudioRecorder.prototype.pause = function(){
        this.recording = !this.recording;
    }
    
    AudioRecorder.prototype.play = function(){
        
    }
    
    AudioRecorder.prototype.clear = function(){
        this.recording = false;
        this.buffer = [];
        this.recLength = 0;
    }
    
    AudioRecorder.prototype.close = function(){
        this.mediastream.getAudioTracks()[0].stop();
        if (this.mediastream.stop) {
            this.mediastream.stop();
        };
        this.node.disconnect();
        this.node.onaudioprocess = null;
        delete this.node;
    }
    
    AudioRecorder.prototype.getLength = function(){
        return this.recLength;
    }
    
    AudioRecorder.prototype.getOgg = function(){
        var result;
        if (typeof OggVorbisEncoder === 'function') {
            var encoder = new OggVorbisEncoder(this.sampleRate, this.channels, this.quality);
            for (var i = 0, len = this.buffers.length; i < len; i++){
                encoder.encode([this.buffers[i]]);
            };
            result = encoder.finish('audio/ogg');
        } else {
            result = false;
        };
        return result;
    };
    
    AudioRecorder.prototype.getDataurl = function(){
        return this.dataurl;
    }
    
    AudioRecorder.prototype.hasAudio = function(){
        return true;
    }
    
    AudioRecorder.prototype.hasVideo = function(){
        return false;
    }
    
    AudioRecorder.prototype.mediatype = function(){
        return 'audio';
    }
    
    AudioRecorder.defaults = {
        maxLength: 5*60*1000, // ms (5 minutes)
        quality: -0.1, // [-0.1 , 1] (-0.1 ~ 45kbps, 0.0 ~ 64kbps, 0.1 ~ 80kbps, 0.2 ~ 96kbps, 0.3 ~ 112kbps, 0.4 ~ 128kbps, 0.5 ~ 160kbps, 0.6 ~ 192kbps, 0.7 ~ 224kbps, 0.8 ~ 256kbps, 0.9 ~ 320kbps, 1.0 ~ 500kbps)
        channels: 1 // mono
    }
    
    
    
    
    
    
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(MediaElement.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(MediaElement.localization);
    }

    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', MediaElement.elementinfo);
        $.fn.elementset('addelementtype', MediaElement.audioelementinfo);
        $.fn.elementset('addelementtype', MediaElement.videoelementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', MediaElement.elementinfo);
        $.fn.elementpanel('addelementtype', MediaElement.audioelementinfo);
        $.fn.elementpanel('addelementtype', MediaElement.videoelementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var mediaelem = new MediaElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.mediaelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in mediaelement.');
            return false;
        }
    }

})(jQuery);