/**
 * Requirements:
 * - jQuery
 * - Marked.js
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
    typeof(marked) === 'undefined' && marked.apply;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - Highlight.js
 * - jquery.commentor.js
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(hljs) === 'undefined' && hljs.apply;
        typeof(jQuery.fn.commentor) === 'undefined' && jQuery.fn.commentor.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - Mathquill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Optional runtime requirements
 * - MediumEditor
 * - Upndown
 * - htmlparser.js
 */

if (typeof(checkOptionalRuntimeRequirements) !== 'undefined' && checkOptionalRuntimeRequirements) {
    try {
        typeof(MediumEditor) === 'undefined' && MediumEditor.apply;
        typeof(upndown) === 'undefined' && upndown.apply;
        typeof(HTMLtoDOM) === 'undefined' && HTMLtoDOM.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function(window, $, ebooklocalizer){
    
    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Markdown element to show and edit Markdown.
     * @requires marked.js Markdown library
     * @requires mathquill.js Mathquill-library.
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var MarkdownElement = function(place, options){
        this.place = $(place);
        this.place.addClass('markdownelement');
        this.setStyles();
        this.init(options);
        this.initHandlersCommon();
        //this.place.trigger('getmarginplace');
        this.show();
    }
    
    /******
     * Init markdown element
     * @param {Object} options - initing data of the element
     ******/
    MarkdownElement.prototype.init = function(options){
        options = $.extend({}, MarkdownElement.defaults, options);
        this.settings = $.extend(true, {}, MarkdownElement.defaults.settings, options.settings); 
        this.settings.wysiwyg = (typeof(MediumEditor) ==="function" && typeof(upndown) ==="function" && typeof(HTMLtoDOM) ==="function"?this.settings.wysiwyg:false);
        this.setMode(options.settings.mode);
        this.text = options.data.text;
        this.metadata = {
            creator: options.metadata.creator || this.settings.username,
            created: options.metadata.created || (new Date).getTime(),
            modifier: options.metadata.modifier || this.settings.username,
            modified: options.metadata.modified || (new Date).getTime(),
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        if ($.fn.commentor && typeof($.fn.commentor) === 'function') {
            this.selfreviewable = true;
            this.place.addClass('selfreviewable');
        };
        this.name = options.name || this.place.attr('data-element-name');
        this.reviewdata = false;
        this.getMargin = false;
    }
    
    /******
     * Set style-tag if needed.
     ******/
    MarkdownElement.prototype.setStyles = function(){
        if ($('head style#markdownelement-style').length === 0) {
            $('head').append('<style id="markdownelement-style" type="text/css">'+MarkdownElement.styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    MarkdownElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = MarkdownElement.modes[mode] || MarkdownElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
        this.noteviewable = modesettings.noteviewable;
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    MarkdownElement.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the Markdown element in different modes
     ******/
    MarkdownElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
            this.place.attr('uilang', this.settings.uilang);
        }
        this.removeHandlers();
        if (!this.editable) {
            this.view();
            this.initHandlersCommon();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersCommon();
            this.initHandlersEditable()
        }
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    MarkdownElement.prototype.view = function(){
        if(this.place.find('.markdownelement-content').length){
            this.place.find('.markdownelement-content').empty().html(marked(this.text, {wysiwyg: this.settings.wysiwyg}).replace(/\u200B/g, ''));
        }else{
            this.place.html('<div class="markdownelement-content">' + marked(this.text, {wysiwyg: this.settings.wysiwyg}).replace(/\u200B/g, '') + '</div>');
        }
        if(!this.place.find('.markdownelement-margin').length){this.place.append('<div class="markdownelement-margin"></div>')};
        this.place.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        if (this.reviewable || this.noteviewable) {
            this.editReview();
        } else if (this.reviewdata) {
            this.showReview();
        };
    };
    
    /******
     * Set information where cursor is in textblock
     ******/
    MarkdownElement.prototype.setAtBlock = function(place){
        this.atBlock = {};
        if(typeof(place)!=="undefined")this.atBlock[place]=true;
    }
    /******
     * move cursor in mathblock
     ******/
    MarkdownElement.prototype.moveInMathblock = function(place,posit){
        var $math_box = place;
        var data = $math_box.data('[[mathquill internal data]]');
        var block = data && data.block;
        var cursor = block && block.cursor;
        var i=0;
        if(!cursor)return false;
        if(posit){
            while(!place.children('span').last().hasClass('cursor') && i<100){
                cursor.moveRight();
                i++;//just for be sure
            }
            this.setAtBlock('End');
        
        }else{
            while(jQuery(this).children('span').index(jQuery(this).find('.cursor')) !== 1 && i<100){
                cursor.moveLeft();
                i++;//just for be sure
            }
            this.setAtBlock('Start');
            
        }
    }

    // Place cursor/carret before some element
    MarkdownElement.prototype.placeCursorBefore = function(placeElement){
        jQuery('#markdownelement-wysiwygedit-temp').remove();
        placeElement.parents('.markdownelement-wysiwygedit').focus();
        var tmp = placeElement.before( jQuery('<span id="markdownelement-wysiwygedit-temp"/>')),
            node = jQuery('#markdownelement-wysiwygedit-temp').get(0),
            range = null,
            sel = null;

        if (document.selection) {
            range = document.body.createTextRange();
            range.moveToElementText(node);
            range.select();
        } else if (window.getSelection) {
            range = document.createRange();
            range.selectNode(node);
            range.collapse(true);
            sel = window.getSelection();
            sel.removeAllRanges();
            sel.addRange(range);
        }
        this.setAtBlock('End');
    }
    // Place cursor/carret after some element
    MarkdownElement.prototype.placeCursorAfter = function(placeElement){
        jQuery('#markdownelement-wysiwygedit-temp').remove();
        placeElement.parents('.markdownelement-wysiwygedit').focus();
        var tmp = placeElement.after( jQuery('<span id="markdownelement-wysiwygedit-temp"/>')),
            node = jQuery('#markdownelement-wysiwygedit-temp').get(0),
            range = null,
            sel = null;

        if (document.selection) {
            range = document.body.createTextRange();
            range.moveToElementText(node);
            range.select();
        } else if (window.getSelection) {
            range = document.createRange();
            range.selectNode(node);
            sel = window.getSelection();
            sel.removeAllRanges();
            sel.addRange(range);
        }
        jQuery('#markdownelement-wysiwygedit-temp').remove();
        this.setAtBlock('Start');
    };
    
    /******
     * Show the element in editable mode.
     ******/
    MarkdownElement.prototype.edit = function(){
        var element = this;
        if(this.settings.wysiwyg){
            // Place cursor/carret before some element
            var placeCursorBefore = function(placeElement){
                jQuery('#markdownelement-wysiwygedit-temp').remove();
                placeElement.parents('.markdownelement-wysiwygedit').focus();
                var tmp = placeElement.before( jQuery('<span id="markdownelement-wysiwygedit-temp"/>')),
                    node = jQuery('#markdownelement-wysiwygedit-temp').get(0),
                    range = null,
                    sel = null;

                if (document.selection) {
                    range = document.body.createTextRange();
                    range.moveToElementText(node);
                    range.select();
                } else if (window.getSelection) {
                    range = document.createRange();
                    range.selectNode(node);
                    range.collapse(true);
                    sel = window.getSelection();
                    sel.removeAllRanges();
                    sel.addRange(range);
                }
                element.setAtBlock('End');
            }
            // Place cursor/carret after some element
            var placeCursorAfter = function(placeElement){
                jQuery('#markdownelement-wysiwygedit-temp').remove();
                placeElement.parents('.markdownelement-wysiwygedit').focus();
                var tmp = placeElement.after( jQuery('<span id="markdownelement-wysiwygedit-temp"/>')),
                    node = jQuery('#markdownelement-wysiwygedit-temp').get(0),
                    range = null,
                    sel = null;

                if (document.selection) {
                    range = document.body.createTextRange();
                    range.moveToElementText(node);
                    range.select();
                } else if (window.getSelection) {
                    range = document.createRange();
                    range.selectNode(node);
                    sel = window.getSelection();
                    sel.removeAllRanges();
                    sel.addRange(range);
                }
                jQuery('#markdownelement-wysiwygedit-temp').remove();
                element.setAtBlock('Start');
            }
            // Fix structure of html-lists 
            var fixlist = function(htmlListElement){
                var childrenlists = htmlListElement.children('ul,ol');
                if(childrenlists.length>0){
                    for(var i=0;i<childrenlists.length;i++){
                        if(childrenlists.eq(i).prev() && childrenlists.prev().prop("tagName").toLowerCase() ==="li"){
                            childrenlists.eq(i).appendTo(childrenlists.eq(i).prev());
                        }
                    }
                }
            }
            // Save html as markdowntext
            var saveText = function(editorElement){
                var removeColoredSpans = function(spanHTML){
                    var child = spanHTML.children('.colored');
                    for(var i=0;i<child.length;i++){
                        removeColoredSpans(child.eq(i));
                    }
                    if(typeof(spanHTML.attr('data-dataid')) !== "undefined" && (oneColor = spanHTML.attr('data-dataid').match(/colored_([a-z]*)/))){
                        spanHTML.replaceWith("@£c"+oneColor[1]+"@£"+spanHTML.html()+"£@£@");
                    }
                }
                if(typeof(editorElement)==="undefined"){
                    editorElement = element.place.find('.markdownelement-wysiwygedit');
                }
                //replace mathquill elements with latex
                var editorHtml = jQuery('<div>').append(jQuery(editorElement).html());
                var origMathElements = jQuery(editorElement).find('.mathquill-rendered-math');
                var mathElements = editorHtml.find('.mathquill-rendered-math');
                var latex, delimiter;
                for(var i=0;i<mathElements.length;i++){
                    latex = origMathElements.eq(i).mathquill('latex');
                    delimiter = (origMathElements.eq(i).is('.mathdisplaystyle') ? '$$' : '$');
                    if (latex !== "") {
                        mathElements.eq(i).replaceWith(delimiter+latex+delimiter);
                    } else {
                        mathElements.eq(i).replaceWith('');
                    };
                }
                
                //replace codeblocks with markdown
                var codeElements = editorHtml.find('pre');
                for(var i=0;i<codeElements.length;i++){
                    codeElements.eq(i).find('br').after('\n').remove();
                }
                var lists = editorHtml.find('ul,ol');
                for(var i=0;i< lists.length;i++){
                    fixlist(lists.eq(i));
                }
                // remove link-spans from header
                editorHtml.find('a.anchor,span.header-link').remove();
                // replace colorer elements
                var oneColor,coloredElems = editorHtml.find('span.colored');
                for(var i=0;i<coloredElems.length;i++)
                {
                    removeColoredSpans(coloredElems.eq(i));
                    // if(coloredElems.eq(i).attr('data-dataid')&&(oneColor = coloredElems.eq(i).attr('data-dataid').match(/colored_([a-z]*)/))){
                        // coloredElems.eq(i).after("@£c"+oneColor[1]+"@£"+coloredElems.eq(i).html()+"£@£@");
                    // }
                    
                }
                // coloredElems.remove();
                //remove empty spans (span is also empty then when it only contains zerolength spaces)
                // This is done because old chrome making spans for content-editeble content
                var allSpans = editorHtml.find('span');
                for (var i=0;i<allSpans.length ;i++){
                    if(allSpans.eq(i).html().replace(/[\u200B-\u200D\uFEFF]/g, '') === "" ){
                        allSpans.eq(i).remove();
                    }else{
                        allSpans.eq(i).replaceWith(allSpans.eq(i).html());
                    }
                }
                var upndn = new upndown();
                // Override '*' ems with '_' ems.
                upndn.methods['close']['em'] = (function(_this) {
                    return function(node) {
                        var depth, html, prefix;
                        depth = _this.currentdepth();
                        _this.depth--;
                        prefix = _this.prefixstack_pop();
                        html = prefix + '_' + _this.buffer[depth].join('').split('\n').join('\n' + prefix) + '_';
                        return _this.buffer[depth - 1].push(html);
                    };
                })(upndn);
                
                element.text = upndn.convert(editorHtml.html());
                element.text = element.text.replace(/[\u200B-\u200D\uFEFF]+/g, '\u200B')
                    .replace(/\*{4}/g, '').replace(/_{2}/g, '');
                element.changed();
                
            }
            // Remove math-elements with "browser-cut" so that ctrl-z(undo) works
            var cutMath = function(mathElement){
                    var currentPlace = document.getSelection().getRangeAt(0);
                    var selection = window.getSelection();            
                    var range = document.createRange();
                    range.selectNodeContents(mathElement);
                    selection.removeAllRanges();
                    selection.addRange(range);
                    document.execCommand('cut',false,null);
                    
                }
                
            // Adde target blank to links. Now user editor option for that but can br user if want to be sure about targets
            var addTargetBlank = function(linklist){
                // used option on medium-editor
                return true;
                /*open previewlinks in new tab
                for(var i=0;i<linklist.length;i++){
                    if( !linklist.eq(i).attr('target') ||(linklist.eq(i).attr('target') && linklist.eq(i).attr('target') !== "_blank")){
                        linklist.eq(i).attr('target',"_blank");
                    } 
                }*/
            }
            // Check if some editor shortcuts are pressed
            var checkPressed = function(keyevent){
                keyevent = keyevent || window.event;
                var charCode = keyevent.which || keyevent.keyCode;
                var cursors = [37,38,39,40];
                if(cursors.indexOf(parseInt(charCode)) !== -1)return true;
                var charStr = String.fromCharCode(charCode);
                for(var i=0;i<keyCommands.length;i++){
                    if(keyCommands[i].key ===charStr 
                        && keyCommands[i].meta ===keyevent.ctrlKey 
                        && keyCommands[i].shift ===keyevent.shiftKey )return true;
                }
                return false;
                
            }
            // sets activebuttons in buttonsets to be shown
            var setActive = function(){
                element.place.find('.medium-editor-toolbar ul.activeButtons').removeClass('activeButtons');
                var activeButtons = element.place.find('.medium-editor-toolbar ul.medium-editor-toolbar-actions button.medium-editor-button-active');
                for(var i=0;i<activeButtons.length;i++){
                    activeButtons.parents('.submenulist').addClass('activeButtons');
                }
                
            }
            // Makes possible sublist's of toolbar
            var makeToolbarSublists = function(){
                // return true;
                var buttonList = element.place.find('.medium-editor-toolbar ul.medium-editor-toolbar-actions');
                var sublistButtons = buttonList.find('li button.sublist');
                
                var neededSublists = {};
                for(var i=0;i<sublistButtons.length;i++){
                    var sublistname= sublistButtons.eq(i).attr('sublist');
                    if(typeof(neededSublists[sublistname]) === "undefined"){
                        neededSublists[sublistname] = {name:sublistname,label:sublistButtons.eq(i).attr('sublistlabel'),textlen:sublistButtons.eq(i).text().length};
                    }
                    if(neededSublists[sublistname].textlen < sublistButtons.eq(i).text().length )neededSublists[sublistname].textlen = sublistButtons.eq(i).text().length;
                }
                for(var key in neededSublists){
                    buttonList.append('<li><button style="width:'+Math.max(parseInt(neededSublists[key].textlen),5)+'em;" title="submenu '+neededSublists[key].label+'" class="medium-editor-action submenubutton"><b>'+neededSublists[key].label+'</b></button> <ul '+/*style="width:'+Math.max(parseInt(neededSublists[key].textlen),5)+'em;"*/ ' class="submenulist medium-editor-toolbar-actions ffwidget-background '+neededSublists[key].name+'"></ul></li>');
                }
                for(var i=0;i<sublistButtons.length;i++){
                    var sublistname= sublistButtons.eq(i).attr('sublist');
                    sublistButtons.eq(i).css('width',Math.max(parseInt(neededSublists[sublistname].textlen),5)+'em').parent().appendTo(buttonList.find('ul.'+sublistname));
                }
                buttonList.find('.medium-editor-button-last').removeClass('medium-editor-button-last');
                buttonList.children('li').last().children('button').addClass('medium-editor-button-last');
            }
            
            // informative classes to editor's html-element
            this.place.addClass('markdownelement-wysiwygmode');
            if(this.settings.wysiwygstatictoolbar){
                this.place.addClass('markdownelement-wysiwygstatictoolbar');
            }
            var html = this.__format(MarkdownElement.templates.wysiwyg, {'markdowntext': marked(this.text, {wysiwyg: this.settings.wysiwyg}).replace(/\u200b+/g, '\u200b')});
            this.place.html(html);
            //keyboard short-cut for editor
            var keyCommands = [
                        {
                            command: 'bold',key: 'B', meta: true,shift: false
                        },
                        {
                            command: 'italic',key: 'I',meta: true,shift: false
                        },
                        {
                            command: 'underline',key: 'U',meta: true,shift: false
                        },
                        {
                            command: 'append-h1',key: '1',meta: true,shift: false
                        },
                        {
                            command: 'append-h2',key: '2',meta: true,shift: false
                        },
                        {
                            command: 'append-h3',key: '3',meta: true,shift: false
                        },
                        {
                            command: 'insertunorderedlist',key: 'L',meta: true,shift: false
                        },
                        {
                            command: 'append-pre',key: 'C',meta: true,shift: true
                        },
                        {
                            command: 'insertorderedlist',key: 'L',meta: true,shift: true
                        },
                        {
                            command: 'append-blockquote',key: 'Q',meta: true,shift: false
                        }
                    ];
            this.wysiwygeditor = new MediumEditor([this.place.find('.markdownelement-wysiwygedit')[0]], {
                targetBlank: true,
                elementsContainer:this.place[0],
                toolbar: {
                    static: this.settings.wysiwygstatictoolbar,
                    sticky: this.settings.wysiwygstatictoolbar,
                    updateOnEmptySelection: this.settings.wysiwygstatictoolbar,
                    buttons: [  
                        'bold',
                        'italic',
                        /* Disabled for now
                        'underline',
                        'strikethrough',
                        'subscript',
                        'superscript',
                        */
                        {
                            name: 'anchor',
                            action: 'createLink',
                            aria: 'link',
                            tagNames: ['a'],
                            contentDefault: '<b>&#x1F517;</b>',
                            contentFA: '<i class="fa fa-link"></i>'
                        },
                        'quote',
                        'pre',
                        'orderedlist',
                        'unorderedlist',
                        'indent',
                        'outdent',
                        {
                            name: 'h1',
                            action: 'append-h1',
                            aria: 'header type 1',
                            tagNames: ['h1'],
                            contentDefault: '<b>H1</b>',
                            classList: ['sublist'],
                            attrs: {
                                'sublist': 'h',
                                'sublistlabel': 'H'
                            }
                        },
                        {
                            name: 'h2',
                            action: 'append-h2',
                            aria: 'header type 2',
                            tagNames: ['h2'],
                            contentDefault: '<b>H2</b>',
                            classList: ['sublist'],
                            attrs: {
                                'sublist': 'h',
                                'sublistlabel': 'H'
                            }
                        },
                        {
                            name: 'h3',
                            action: 'append-h3',
                            aria: 'header type 3',
                            tagNames: ['h3'],
                            contentDefault: '<b>H3</b>',
                            classList: ['sublist'],
                            attrs: {
                                'sublist': 'h',
                                'sublistlabel': 'H'
                            }
                        },
                        {
                            name: 'h4',
                            action: 'append-h4',
                            aria: 'header type 4',
                            tagNames: ['h4'],
                            contentDefault: '<b>H4</b>',
                            classList: ['sublist'],
                            attrs: {
                                'sublist': 'h',
                                'sublistlabel': 'H'
                            }
                        }
                        ,
                        // "table",
                        // "gold",
                        "red",
                        "green",
                        "blue",
                        "yellow",
                        "white"
                        ]
                },
                keyboardCommands: {
                    commands: keyCommands
                },
                anchorPreview:{
                    showWhenToolbarIsVisible:true,
                    positionPreview: function (activeAnchor) {
                        // Added to fix medium-editor problem to calculate anchor-preview elements position
                        // added 'relative' property to elementsContainerAbsolute check
                        // original function can be found in medium-editor.js:4103
                        activeAnchor = activeAnchor || this.activeAnchor;
                        var containerWidth = this.window.innerWidth,
                            buttonHeight = this.anchorPreview.offsetHeight,
                            boundary = activeAnchor.getBoundingClientRect(),
                            diffLeft = this.diffLeft,
                            diffTop = this.diffTop,
                            elementsContainer = this.getEditorOption('elementsContainer'),
                            elementsContainerAbsolute = ['absolute', 'fixed','relative'].indexOf(window.getComputedStyle(elementsContainer).getPropertyValue('position')) > -1,//This is fixed
                            relativeBoundary = {},
                            halfOffsetWidth, defaultLeft, middleBoundary, elementsContainerBoundary, top;

                        halfOffsetWidth = this.anchorPreview.offsetWidth / 2;
                        var toolbarExtension = this.base.getExtensionByName('toolbar');
                        if (toolbarExtension) {
                            diffLeft = toolbarExtension.diffLeft;
                            diffTop = toolbarExtension.diffTop;
                        }
                        defaultLeft = diffLeft - halfOffsetWidth;

                        // If container element is absolute / fixed, recalculate boundaries to be relative to the container
                        if (elementsContainerAbsolute) {
                            elementsContainerBoundary = elementsContainer.getBoundingClientRect();
                            ['top', 'left'].forEach(function (key) {
                                relativeBoundary[key] = boundary[key] - elementsContainerBoundary[key];
                            });

                            relativeBoundary.width = boundary.width;
                            relativeBoundary.height = boundary.height;
                            boundary = relativeBoundary;

                            containerWidth = elementsContainerBoundary.width;

                            // Adjust top position according to container scroll position
                            top = elementsContainer.scrollTop;
                        } else {
                            // Adjust top position according to window scroll position
                            top = this.window.pageYOffset;
                        }

                        middleBoundary = boundary.left + boundary.width / 2;
                        top += buttonHeight + boundary.top + boundary.height - diffTop - this.anchorPreview.offsetHeight;

                        this.anchorPreview.style.top = Math.round(top) + 'px';
                        this.anchorPreview.style.right = 'initial';
                        if (middleBoundary < halfOffsetWidth) {
                            this.anchorPreview.style.left = defaultLeft + halfOffsetWidth + 'px';
                            this.anchorPreview.style.right = 'initial';
                        } else if ((containerWidth - middleBoundary) < halfOffsetWidth) {
                            this.anchorPreview.style.left = 'auto';
                            this.anchorPreview.style.right = 0;
                        } else {
                            this.anchorPreview.style.left = defaultLeft + middleBoundary + 'px';
                            this.anchorPreview.style.right = 'initial';
                        }
                    },
                },
                anchor :{
                    linkValidation: true,
                    placeholderText: ebooklocalizer.localize('markdownelement:typelink', this.settings.uilang)
                },
                extensions: {
                    green : new MediumExtraButton({
                        label:'<b>green<b>',
                        start:'span',
                        end:'span',
                        addClass:'colored',
                        dataId: 'colored_green',
                        buttonAttrs: {
                            'sublist': 'color',
                            'sublistlabel': 'Color'
                        },
                        buttonClassList: ['sublist']
                        }),
                    red : new MediumExtraButton({
                        label:'<b>red<b>',
                        start:'span',
                        end:'span',
                        addClass:'colored',
                        dataId: 'colored_red',
                        buttonAttrs: {
                            'sublist': 'color',
                            'sublistlabel': 'Color'
                        },
                        buttonClassList: ['sublist']
                        }),
                    yellow : new MediumExtraButton({
                        label:'<b>yellow<b>',
                        start:'span',
                        end:'span',
                        addClass:'colored',
                        dataId: 'colored_yellow',
                        buttonAttrs: {
                            'sublist': 'color',
                            'sublistlabel': 'Color'
                        },
                        buttonClassList: ['sublist']
                        }),
                    blue : new MediumExtraButton({
                        label:'<b>blue<b>',
                        start:'span',
                        end:'span',
                        addClass:'colored',
                        dataId: 'colored_blue',
                        buttonAttrs: {
                            'sublist': 'color',
                            'sublistlabel': 'Color'
                        },
                        buttonClassList: ['sublist']
                        }),
                    white : new MediumExtraButton({
                        label:'<b>white<b>',
                        start:'span',
                        end:'span',
                        addClass:'colored',
                        dataId: 'colored_white',
                        buttonAttrs: {
                            'sublist': 'color',
                            'sublistlabel': 'Color'
                        },
                        buttonClassList: ['sublist']
                        })
                }
            });
            this.place.find('.medium-editor-toolbar').addClass('ffwidget-background');
            makeToolbarSublists();
            //Set needed eventhandlers for wysiwyg-editing
            this.place.on('mouseup','.medium-editor-toolbar ul.medium-editor-toolbar-actions button', function(e) {
                setTimeout(setActive,1);
            });
            this.place.on('keydown', function(e) {
                var checkRange = document.getSelection().getRangeAt(0);
                
                if(jQuery(checkRange.startContainer).hasClass('mathquill-editable')||jQuery(checkRange.startContainer).parents('.mathquill-editable').length>0)// is mathblock
                {
                    element.startBlock = "math";
                }else if(checkRange.startContainer.nodeType ===3) // is the block textblock
                {
                    element.startBlock = "text";
                }else{
                    element.startBlock = "other";
                }
                if(e.keyCode ===39){//arrow right
                    jQuery('#markdownelement-wysiwygedit-temp').remove();
                        
                    if(element.atBlock && element.atBlock['End']&&document.getSelection().getRangeAt(0).startContainer.tagName ==="P"){
                        //if cursor has moved right and into mathquill element, give focus to mathquill
                        var nnode = document.createElement("span");
                        checkRange.surroundContents(nnode);
                        nnode.setAttribute("id", "wysiwygedit-rightCursorTemp");
                        if(jQuery("#wysiwygedit-rightCursorTemp").next().hasClass('mathquill-editable')){
                            jQuery("#wysiwygedit-rightCursorTemp").next().focus();
                        }
                        jQuery("#wysiwygedit-rightCursorTemp").remove();
                    }
                        
                        
                        
                    if(checkRange.endOffset === checkRange.startContainer.length && element.atBlock && element.atBlock['End']){
                        if(jQuery(checkRange.startContainer.nextElementSibling).hasClass('mathquill-editable'))
                        {
                            var mathElement = jQuery(checkRange.startContainer.nextElementSibling);
                            mathElement.focus();
                            element.moveInMathblock(mathElement,false);
                            return false;
                        }
                        
                    }
                }else if(e.keyCode === 37){//arrow left
                    if(element.atBlock && element.atBlock['Start']&&document.getSelection().getRangeAt(0).startContainer.tagName ==="P"){
                        //if cursor has moved right and into mathquill element, give focus to mathquill
                        var nnode = document.createElement("span");
                        checkRange.surroundContents(nnode);
                        nnode.setAttribute("id", "wysiwygedit-leftCursorTemp");
                        if(jQuery("#wysiwygedit-leftCursorTemp").prev().hasClass('mathquill-editable')){
                            jQuery("#wysiwygedit-leftCursorTemp").prev().focus();
                        }
                        jQuery("#wysiwygedit-leftCursorTemp").remove();
                    }
                    if(checkRange.endOffset === 0 && element.atBlock && element.atBlock['Start']){
                        if(jQuery(checkRange.startContainer.previousElementSibling).hasClass('mathquill-editable'))
                        {
                            var mathElement = jQuery(checkRange.startContainer.previousElementSibling);
                            mathElement.focus();
                            element.moveInMathblock(mathElement,true);
                            return false;
                        }
                        
                    }
                }
                   
            });
            this.place.on('keyup', function(e) {
                if(checkPressed(e)){
                    setTimeout(setActive,1);
                }
                var checkRange = document.getSelection().getRangeAt(0);
                
                if(e.keyCode == 77 && e.ctrlKey){//crtl+m for new mathmode
                    if(jQuery(e.target).parents('.mathquill-editable').length === 0){
                        var range = checkRange;
                        var nnode = document.createElement("span");
                        range.surroundContents(nnode);
                        nnode.setAttribute("id", "wysiwygedit-newMath");
                        nnode.setAttribute("class", "mathquill-embedded-latex");
                        jQuery('#wysiwygedit-newMath').mathquill('editable').attr('id','').attr('contenteditable','false')
                        .focus();
                    }
                }else if(e.keyCode === 38 || e.keyCode === 40){//arrow up or arrow down
                    if(checkRange.startOffset === 0){
                        element.setAtBlock('Start');
                    }else if(checkRange.startOffset ===checkRange.startContainer.length){
                        element.setAtBlock('End');
                    }
                    if(checkRange.startContainer.tagName ==="P"){
                        //if cursor has moved up or down and into mathquill element, give focus to mathquill
                        var nnode = document.createElement("span");
                        checkRange.surroundContents(nnode);
                        nnode.setAttribute("id", "wysiwygedit-downCursorTemp");
                        if(jQuery("#wysiwygedit-downCursorTemp").prev().hasClass('mathquill-editable')){
                            jQuery("#wysiwygedit-downCursorTemp").prev().focus();
                        }
                        jQuery("#wysiwygedit-downCursorTemp").remove();
                    }
                }else if(e.keyCode === 8){//backspace
                    /*
                    if(jQuery(e.target).parents('.mathquill-editable.hasCursor').length >0){
                        saveText(jQuery(e.target).closest('.markdownelement-wysiwygedit'));                        
                    }else{
                    */
                        if(checkRange.endOffset === 0 && element.atBlock && element.atBlock['Start']){
                            if(jQuery(checkRange.startContainer.previousElementSibling).hasClass('mathquill-editable'))
                            {
                                var mathElement= checkRange.startContainer.previousElementSibling;
                                try{
                                    cutMath(mathElement);
                                }catch(error){
                                    // No suport for execCommand cut, remove with jQuery
                                    jQuery(mathElement).remove();
                                }
                                element.setAtBlock();
                                saveText(jQuery(e.target).closest('.markdownelement-wysiwygedit'));
                                return false;
                            }
                            
                        }
                        if(element.atBlock && element.atBlock['Start']&&document.getSelection().getRangeAt(0).startContainer.tagName ==="P"){
                            //if backspace presses and previous element is mathquill element, remove it
                            //TODO:FIX with cut when browsers suports it
                            var nnode = document.createElement("span");
                            checkRange.surroundContents(nnode);
                            nnode.setAttribute("id", "wysiwygedit-backspaceTemp");
                            if(jQuery("#wysiwygedit-backspaceTemp").prev().hasClass('mathquill-editable')){
                                var mathElement = jQuery("#wysiwygedit-backspaceTemp").prev()[0];
                                try{
                                    cutMath(mathElement)
                                }catch(error){
                                    // No suport for execCommand cut, remove with jQuery
                                    jQuery(mathElement).remove();
                                }
                            }
                            jQuery("#wysiwygedit-backspaceTemp").remove();
                            element.setAtBlock();
                            saveText(jQuery(e.target).closest('.markdownelement-wysiwygedit'));
                            return false;
                        }
                    // }
                }else if(e.keyCode === 46){//delete
                    jQuery('#markdownelement-wysiwygedit-temp').remove();
                    if(element.atBlock && element.atBlock['End']&&document.getSelection().getRangeAt(0).startContainer.tagName ==="P"){
                        //if delete pressed and next element is mathquill element, remove it
                        var nnode = document.createElement("span");
                        checkRange.surroundContents(nnode);
                        nnode.setAttribute("id", "wysiwygedit-deleteTemp");
                        if(jQuery("#wysiwygedit-deleteTemp").next().hasClass('mathquill-editable')){
                            var mathElement = jQuery("#wysiwygedit-backspaceTemp").next()[0];
                            try{
                                cutMath(mathElement)
                            }catch(error){
                                // No suport for execCommand cut, remove with jQuery
                                jQuery(mathElement).remove();
                            }
                        }
                        jQuery("#wysiwygedit-deleteTemp").remove();
                        element.setAtBlock();
                        saveText(jQuery(e.target).closest('.markdownelement-wysiwygedit'));
                        return false;
                    }
                        
                        
                        
                    if(checkRange.endOffset === checkRange.startContainer.length && element.atBlock && element.atBlock['End']){
                        if(jQuery(checkRange.startContainer.nextElementSibling).hasClass('mathquill-editable'))
                        {
                            
                            var mathElement = checkRange.startContainer.nextElementSibling;
                            try{
                                cutMath(mathElement)
                            }catch(error){
                                // No suport for execCommand cut, remove with jQuery
                                jQuery(mathElement).remove();
                            }
                            element.setAtBlock();
                            saveText(jQuery(e.target).closest('.markdownelement-wysiwygedit'));
                            return false;
                        }
                        
                    }
                }else{
                    var checkRange = document.getSelection().getRangeAt(0);
                    if(element.startBlock && element.startBlock !=="math" && (jQuery(checkRange.startContainer).hasClass('mathquill-editable')||jQuery(checkRange.startContainer).parents('.mathquill-editable').length>0))//block is mathquill-block
                    {
                        var mathElement =  jQuery(checkRange.startContainer);//.closest('.mathquill-editable');
                        
                        mathElement.focus();
                        element.moveInMathblock(mathElement,false);
                    }
                    else if(checkRange.startContainer.nodeType ===3) // is the block textblock
                    {
                        if(checkRange.endOffset === 0){// carret at begining of the block
                            if(e.keyCode === 37){//arrow left
                                element.setAtBlock('Start');
                            }
                            
                        }else if(checkRange.endOffset === checkRange.startContainer.length){//  carret at end of the block
                            if(e.keyCode === 39){//arrow right
                                element.setAtBlock('End');
                            }
                            
                        }else{
                             element.setAtBlock();
                        }
                    }
                }
            });
            /* not needed for now
            this.wysiwygeditor.subscribe('editableKeyup', function (event, editable) {
            });
            */
            /**
             * Revert all mathquill-elements to span+latex
             */
            var doRevert = function(event) {
                var node = MediumEditor.selection.getSelectionStart(this.options.ownerDocument);
                var ismq = $(event.target).closest('.mathquill-editable').length > 0;
                if (!ismq && (MediumEditor.util.isKey(event, [MediumEditor.util.keyCode.BACKSPACE, MediumEditor.util.keyCode.DELETE, MediumEditor.util.keyCode.ENTER, MediumEditor.util.keyCode.TAB]) || MediumEditor.util.isMetaCtrlKey(event))) {
                    element.place.trigger('revertmath');
                };
                var prev, next;
                if (MediumEditor.selection.getCaretOffsets(node).left === 0) {
                    next = $(node);
                    prev = next.prev();
                } else if (MediumEditor.selection.getCaretOffsets(node).right === 0) {
                    prev = $(node);
                    next = prev.next();
                };
                if (MediumEditor.util.isKey(event, [MediumEditor.util.keyCode.DELETE, MediumEditor.util.keyCode.BACKSPACE]) &&
                    prev && next) {
                    console.log(prev, next);
                    if (next.length > 0 && $(next[0].childNodes[0]).is('.mathquill-reverted')) {
                        $(next[0].childNodes[0]).before('\u200b');
                    };
                    if (prev.length > 0 && $(prev[0].childNodes[prev[0].childNodes.length - 1]).is('.mathquill-reverted')) {
                        $(prev[0].childNodes[prev[0].childNodes.length - 1]).after('\u200b');
                    };
                };
            };
            /**
             * Undo reverted math from span+latex back to mathquill
             */
            var doUnrevert = function(event) {
                var node = MediumEditor.selection.getSelectionStart(this.options.ownerDocument);
                element.place.trigger('redrawrevertedmath');
                element.place[0].dispatchEvent(new Event('change'));
            };
            this.wysiwygeditor.subscribe('editableKeydown', doRevert.bind(this.wysiwygeditor));
            this.wysiwygeditor.subscribe('editableKeyup', doUnrevert.bind(this.wysiwygeditor));
            
            this.wysiwygeditor.subscribe('editableInput', function (event, editable) {
                jQuery('#markdownelement-wysiwygedit-temp').remove();
                var previewLinks = jQuery('a.medium-editor-toolbar-anchor-preview-inner');
                if(previewLinks)addTargetBlank(previewLinks);
                if(jQuery(event.target).parents('.mathquill-editable').length > 0){
                    setTimeout(saveText,1);
                }else{
                    saveText(editable);
                }
            });
            this.place.on("focusout",".markdownelement-wysiwygedit",function(){
                var setFocusBack = function(){
                    var focusedElement = jQuery(":focus");
                    if(!(focusedElement.hasClass('medium-editor-toolbar-input')||focusedElement.hasClass('medium-editor-action-anchor')||focusedElement.hasClass('medium-editor-anchor-preview')) && focusedElement.parents('.medium-editor-toolbar.static-toolbar.medium-editor-toolbar-active').length >0){
                        element.place.find('.markdownelement-wysiwygedit').focus();
                    }
                }
                //delay call for setting focus back to editor if static toolbar is clicked
                setTimeout(setFocusBack,1);
                
                });
            //Handle mathquill elements in contenteditable.
            //Disable contenteditable of mathquill and fix movement with arrowkeys
            this.place.find('.mathquill-embedded-latex').mathquill('editable').attr('contenteditable','false');
            //Handle code editing
            var codeBlocks = jQuery('pre.hljs');
            for(var i=0;i<codeBlocks.length;i++){
                codeBlocks.eq(i).after('<pre>'+codeBlocks.eq(i).text().replace(/</g,"&lt;").replace(/>/g,"&gt;")+'</pre>');
            };
            codeBlocks.remove();
            //Handle pure javascript change events
            this.place[0].addEventListener('change',function(e){
                saveText();
            });
        }else{
            var html = this.__format(MarkdownElement.templates.edithtml, {'markdowntext': escapeHTML(this.text)});
            this.place.html(html);
            this.initButtons();
            this.resizeTextarea();
            this.refreshPreview();
        }
    }
    
    /******
     * Refresh preview
     ******/
    MarkdownElement.prototype.refreshPreview = function(){
        if (this.settings.preview) {
            if (this.place.width() > 300) {
                this.place.addClass('markdownelement-iswide');
            } else {
                this.place.removeClass('markdownelement-iswide');
            }
            this.place.children('.markdownelement-preview').html(marked(this.text, {wysiwyg: this.settings.wysiwyg}))
                .find('.mathquill-embedded-latex').mathquill();
        }
    }
    
    /**
     * Element has changed
     */
    MarkdownElement.prototype.changed = function() {
        this.updateMetadata();
        this.place.trigger('element_changed', [{type: 'markdownelement', lang: this.metadata.lang}]);
    };
    
    /******
     * Init buttons on their places
     ******/
    MarkdownElement.prototype.initButtons = function(){
        var bset = this.place.children('.markdownelement-buttonset');
        bset.append('<span class="markdownelement-button" data-buttontype="preview" title="' + ebooklocalizer.localize('markdownelement:preview', this.settings.uilang) + '">' + MarkdownElement.icons.preview + '</span>');
        //var ebar = this.place.children('.markdownelement-editbar');
        //ebar.append('<span class="markdownelement-button" data-buttontype="remove" title="' + ebooklocalizer.localize('markdownelement:remove', this.settings.uilang) + '">' + MarkdownElement.icons.remove + '</span>');
    }
    
    /******
     * Remove all event handlers
     ******/
    MarkdownElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    MarkdownElement.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in MarkdownElement.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e, data){
            e.stopPropagation();
            if (data && element.selfreviewable) {
                element.setReview(data);
            };
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        }).on('getdata', function(e){
            e.stopPropagation();
            var data = element.getData();
            element.place.data('[[markdownelementdata]]', data);
            element.place.data('[[elementdata]]', data);
        }).on('gethtml', function(e){
            e.stopPropagation();
            var data = element.getHtmltext();
            element.place.data('[[elementhtml]]', data);
        }).on('setreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (data && element.selfreviewable) {
                element.setReview(data);
            };
            if (element.reviewable) {
                element.editReview();
            } else if (!element.editable) {
                element.showReview();
            };
        }).on('setnoteview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (data && element.selfnoteviewable) {
                element.setReview(data);
            };
            if (element.noteviewable) {
                element.editReview();
            } else if (!element.editable) {
                element.showReview();
            };
        }).on('showselfreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (data && element.selfreviewable) {
                element.setReview(data);
                if (element.reviewable) {
                    element.editReview();
                } else if (!element.editable) {
                    element.showReview();
                };
            } else {
                element.unhideReview();
            };
            //element.place.trigger('showcomments');
        }).on('hideselfreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            if (element.selfreviewable) {
                element.hideReview();
            };
        }).on('getreview', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            element.place.trigger('reviewdata', [JSON.parse(JSON.stringify(element.reviewdata || {}))]);
        }).on('commentorchanged', function(e, data){
            e.stopPropagation();
            e.preventDefault();
            element.setReview(data);
            if (element.reviewable) {
                element.reviewChanged();
            } else if (element.noteviewable) {
                element.noteviewChanged();
            };
        }).on('reply_getmarginplace', function(event, data) {
            event.stopPropagation();
            if (typeof(data.getMargin) === 'function') {
                element.getMargin = data.getMargin;
            };
        });
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    MarkdownElement.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    MarkdownElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('drop', 'textarea.markdownelement-editarea', function(e){
            // Trigger change with timeout, since drop event is fired before changing the content.
            var ta = $(this);
            setTimeout(function(){ta.trigger('change')}, 1);
        });
        this.place.on('change', 'textarea.markdownelement-editarea', function(e){
            element.text = $(this).val();
            element.changed();
        });
        this.place.on('keyup', 'textarea.markdownelement-editarea', function(e){
            switch (e.which){
                case 13:
                    element.text = $(this).val();
                    element.changed();
                    break;
                default:
                    element.refreshPreview();
                    break;
            }
            element.resizeTextarea();
        });
        this.place.on('keydown', '.mathquill-editable', function(e){
            if(e.keyCode == '39'){//arrow right
                if(jQuery(this).children('span').last().hasClass('cursor')){
                    if(element.atBlock && element.atBlock['End']){
                        element.setAtBlock();
                        element.placeCursorAfter(jQuery(this));
                    }else{
                        element.setAtBlock('End');
                    }
                }else{
                    element.setAtBlock();
                }
            }

            if(e.keyCode == '37' ){//arrow left
                if(jQuery(this).children('span').index(jQuery(this).find('.cursor')) === 1){
                    if(element.atBlock && element.atBlock['Start']){
                        element.setAtBlock();
                        element.placeCursorBefore(jQuery(this));
                    }else{
                        element.setAtBlock('Start');
                    }
                }else{
                    element.setAtBlock();
                }
            }
        });
        this.place.on('revertmath', function(event) {
            event.stopPropagation();
            var mq = element.place.find('.mathquill-editable');
            var latex;
            for (var i = 0, len = mq.length; i < len; i++) {
                let width = mq.eq(i).width();
                let height = mq.eq(i).height();
                latex = mq.eq(i).css({width: width, height: height, display: 'inline-block'}).mathquill('latex');
                mq.eq(i).mathquill('revert')
                .addClass('mathquill-reverted')
                .text(latex);
            };
        });
        this.place.on('redrawrevertedmath', function(event) {
            event.stopPropagation();
            element.place.find('.mathquill-reverted')
                .removeAttr('style')
                .removeClass('mathquill-reverted')
                .removeAttr('style')
                .mathquill('editable')
                .attr('contenteditable', 'false');
            // Trigger that the content has changed.
            element.place[0].dispatchEvent(new Event('change'));
        });
        this.place.on('cleanhtml', function(event) {
            event.stopPropagation();
            var temp = jQuery('<b>temp</b>');
            var i, len;
            // Remember the place, where the cursor was.
            var temp2 = jQuery('<b class="markdownelement-cursorplaceholder">temp</b>');
            var sel = window.getSelection();
            var range = sel.getRangeAt(0);
            range.insertNode(temp2[0]);
            var nestedp = element.place.find('.markdownelement-wysiwygedit p > p');
            for (i = 0, len = nestedp.length; i < len; i++) {
                nestedp.eq(i).before(temp);
                temp.after(nestedp.eq(i).contents());
                nestedp.eq(i).remove();
            };
            var styled = element.place.find('.markdownelement-wysiwygedit [style]:not(.mathquill-editable, .mathquill-reverted)');
            for (i = 0, len = styled.length; i < len; i++) {
                styled.eq(i).before(temp);
                temp.after(styled.eq(i).contents());
                styled.eq(i).remove();
            };
            temp.remove();
            // Place the cursor back.
            range.setStart(temp2[0], 0);
            range.collapse(true);
            sel.removeAllRanges();
            sel.addRange(range);
            temp2.remove();
        });
        this.place.on('elementfocus', function(e){
            e.stopPropagation();
            var focusplace = 'textarea.markdownelement-editarea';
            if(element.settings.wysiwyg){
                focusplace = '.markdownelement-wysiwygedit';
            }
            element.place.find(focusplace).trigger('focus');
        });
        this.place.on('element_changed', function(e){
            element.refreshPreview();
        });
        this.place.on('preview_toggle', function(e){
            element.refreshPreview();
        });
        this.place.on('click', '.markdownelement-button[data-buttontype="preview"]', function(e){
            element.settings.preview = !element.settings.preview;
            element.place.toggleClass('markdownelement-previewmode');
            element.place.trigger('preview_toggle');
        });
        this.place.on('click', '.markdownelement-button[data-buttontype="remove"]', function(e){
            element.place.trigger('removeelement', {type: 'markdownelement'});
        });
        this.place.on('click', function(e){
            e.stopPropagation();
            element.place.children('textarea').focus();
        });
        this.place.on('renameelements', function(e) {
            e.stopPropagation();
            element.name = element.place.attr('data-element-name');
        });
        this.place.on('mousedown', '.medium-editor-toolbar button', function(event) {
            console.log('button mousedown');
            element.place.trigger('revertmath');
        });
        this.place.on('mouseup', '.medium-editor-toolbar button', function(event) {
            console.log('button mouseup');
            setTimeout(function(){element.place.trigger('redrawrevertedmath');}, 10);
        });
    }
    
    /******
     * Update the metadata of the element.
     ******/
    MarkdownElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    }
    
    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "markdownelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {"text": this.text}
     *   }
     ******/
    MarkdownElement.prototype.getData = function(){
        var result = {
            type: 'markdownelement',
            name: this.name,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: {
                text: this.text
            }
        }
        return result;
    }
    
    /******
     * Get data of the element as html
     * @returns {String} data of this markdown element as html-text
     ******/
    MarkdownElement.prototype.getHtmltext = function(){
        var text = this.text;
        var html = '';
        var rex = /\$([^\$]*)\$/g;
        var match, last = 0, capture;
        while (match = rex.exec(text)) {
            html += text.substring(last, match.index)
                .replace(/@£c[a-z]*@£(.*?)£@£@/g, '$1')
                .replace(/\n\* /g, '\n- ')
                .replace(/(^|\n)# ([^\n]*)\n/g, '$1$2\n=========================================\n')
                .replace(/(^|\n)## ([^\n]*)\n/g, '$1$2\n------------------------------------------\n')
                .replace(/\n/g, '<br>\n');
            capture = match[1]
                .replace(/\\\\/g, '\\')
                ;
            html += '<img src="/math.svg?latex='
                +encodeURIComponent(capture.replace(/\\(?:(N)N|(Z)Z|(Q)Q|(R)R|(C)C|(P)P|(H)H)/g, '\\mathbb{$1$2$3$4$5$6$7}'))
                +'" alt="'
                +escapeHTML(capture.replace(/\\(?:(N)N|(Z)Z|(Q)Q|(R)R|(C)C|(P)P|(H)H)/g, '\\$1$2$3$4$5$6$7'))
                +'" >';
            last = rex.lastIndex;
        };
        html += text.substring(last)
            .replace(/@£c[a-z]*@£(.*?)£@£@/g, '$1')
            .replace(/\n\* /g, '\n- ')
            .replace(/(^|\n)# ([^\n]*)\n/g, '$1$2\n=========================================\n')
            .replace(/(^|\n)## ([^\n]*)\n/g, '$1$2\n------------------------------------------\n')
            .replace(/\n/g, '<br>\n');
        return html;
    }
    
    /******
     * Trigger 'review_changed' -event
     ******/
    MarkdownElement.prototype.reviewChanged = function(){
        this.place.trigger('review_changed');
        var data = {
            id: this.name,
            reviewdata: this.reviewdata
        };
        this.place.trigger('elementreview_changed', data);
    };
    
    /******
     * Trigger 'noteview_changed' -event
     ******/
    MarkdownElement.prototype.noteviewChanged = function(){
        this.place.trigger('noteview_changed');
        var data = {
            id: this.reviewdata.data.id,
            anchorid: this.name,
            contentType: 'marginnote',
            lang: 'common',
            sendto: [],
            sendfrom: this.settings.username,
            data: this.reviewdata.data
        };
        this.place.trigger('marginnote_changed', data);
    };
    
    /******
     * Set review data
     * @param {Object} revdata - review data
     ******/
    MarkdownElement.prototype.setReview = function(revdata) {
        this.reviewdata = $.extend(true, {data: {}}, revdata);
    };
    
    /******
     * Show review
     ******/
    MarkdownElement.prototype.showReview = function(){
        if (this.selfreviewable && !this.editable) {
            var appid = this.settings.gotolink && this.settings.gotolink.appname || '';
            var settings = {
                margin: (this.getMargin && this.settings.usemargin ? this.getMargin(this.name, appid) : this.place.children('.markdownelement-margin')),
                usemargin: this.settings.usemargin && (typeof(this.getMargin) === 'function'),
                mode: 'view',
                username: this.settings.username,
                uilang: this.settings.uilang
            };
            var rdata = $.extend(true, {settings: settings}, this.reviewdata);
            this.place.children('.markdownelement-content').commentor(rdata);
        };
    };
    
    /******
     * Unhide review
     ******/
    MarkdownElement.prototype.unhideReview = function(){
        if (this.selfreviewable) {
            this.place.children('.markdownelement-content').trigger('showcomments');
        };
    };
    
    /******
     * Hide review
     ******/
    MarkdownElement.prototype.hideReview = function(){
        if (this.selfreviewable) {
            this.place.children('.markdownelement-content').trigger('hidecomments');
        };
    };
    
    /******
     * Edit review
     ******/
    MarkdownElement.prototype.editReview = function(){
        if (this.selfreviewable) {
            var appid = this.settings.gotolink && this.settings.gotolink.appname || '';
            var settings = {
                margin: (this.getMargin && this.settings.usemargin ? this.getMargin(this.name, appid) : this.place.children('.markdownelement-margin')),
                usemargin: (this.settings.usemargin && typeof(this.getMargin) === 'function'),
                mode: 'edit',
                username: this.settings.username,
                uilang: this.settings.uilang
            };
            var rdata = $.extend(true, {settings: settings}, this.reviewdata);
            this.place.children('.markdownelement-content').trigger('commentordestroy').commentor(rdata);
        };
    };
    
    MarkdownElement.prototype.resizeTextarea = function(){
        this.place.css('height', this.place.height());
        var telem = this.place.find('textarea');
        telem.css({height: 'auto'});
        var newh = (telem[0].scrollHeight > 32 ? telem[0].scrollHeight + 30 : 32);
        telem.css('height', newh + 'px');
        this.place.css('height', '');
    }
    
    /******
     * Template variable replacer.
     * @param {String} text - template with replaceable placeholders: '{%key%}'
     * @param {Object} data - Object with key-value-pairs
     * @returns {String} Original text with placeholders replaced with corresponding values.
     ******/
    MarkdownElement.prototype.__format = function(text, data){
        values = $.extend({}, data);
        for (var key in data) {
            var rex = RegExp('{%'+key+'%}', 'g');
            text = text.replace(rex, data[key]);
        }
        return text;
    }
    
    /******
     * Default settings
     ******/
    MarkdownElement.defaults = {
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            text: ""
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en',
            wysiwyg: true,
            wysiwygstatictoolbar:true
        }
    }
    
    // By default, marked adds id's for each heading. We don't want that.
    // Override heading rendering with GitHub-like anchors.
    var renderer = new marked.Renderer();
    renderer.heading = function(text, level) {
        var escapedText = text.toLowerCase().replace(/[^\w]+/g, '-');

        return '<h' + level + '><a name="' +
                      escapedText +
                       '" class="anchor" href="#' +
                       escapedText +
                       '"><span class="header-link"></span></a>' +
                        text + '</h' + level + '>';
    };

    renderer.link = function(href, title, text) {
        if (href.trim() === '') {
            return text;
        }
        if (this.options.sanitize) {
            try {
                var prot = decodeURIComponent(unescape(href))
                .replace(/[^\w:]/g, '')
                .toLowerCase();
            } catch (e) {
                return text;
            }
            if (prot.indexOf('javascript:') >= 0 || prot.indexOf('vbscript:') === 0) {
                return text;
            }
        }
        var out = '<a href="' + href + '" target="_blank"';
        if (title) {
            out += ' title="' + title + '"';
        }
        out += '>' + text + '</a>';
        return out;
    };
    
    MarkdownElement.markedOptions = (typeof(hljs) === 'object' ? {
        highlight: function(code){
            return hljs.highlightAuto(code).value;
        },                 // Highlight code blocks with hightlight.js
        highlightClass: 'hljs',
        sanitize: true,    // Sanititze html!
        gfm: true,         // Github Flavoured Markdown
        renderer: renderer // Use the renderer defined above.
    } : {});
    // Set the rendering options for marked.
    marked.setOptions(MarkdownElement.markedOptions);
    
    /******
     * Modes
     ******/
    MarkdownElement.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false,
            noteviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false,
            noteviewable: false
        },
        edit_html: {
            editable: true,
            authorable: false,
            reviewable: false,
            noteviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true,
            noteviewable: false
        },
        noteview: {
            editable: false,
            authorable: false,
            reviewable: false,
            noteviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false,
            noteviewable: false
        }
    }
    
    /******
     * Templates
     ******/
    MarkdownElement.templates = {
        edithtml: [
            '<textarea class="markdownelement-editarea">{%markdowntext%}</textarea>',
            '<div class="markdownelement-preview"></div>',
            '<div class="markdownelement-buttonset"></div>'
        ].join('\n'),
        wysiwyg: 
            '<div class="markdownelement-wysiwygedit">{%markdowntext%}</div>'
    }
    
    /******
     * Icons
     ******/
    MarkdownElement.icons = {
        preview: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-preview"><path style="stroke: none;" d="M3 15 a13 13 0 0 0 24 0 a13 13 0 0 0 -24 0z m1 0 a13 13 0 0 1 22 0 a13 13 0 0 1 -22 0z m6 -1 a5 5 0 0 0 10 0 a5 5 0 0 0 -10 0z m1.5 -1.6 a3 3 0 0 1 3 -2.1 a2 0.8 0 0 1 0 1 a3 3 0 0 0 -1.8 1.2 a0.5 0.5 0 0 1 -1.16 0z"></path><path class="nopreview" stroke="none" style="fill: none; stroke-width: 4;" d="M3 27 l24 -24" /></svg>',
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z" /><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    MarkdownElement.elementinfo = {
        type: 'markdownelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'markdownelement',
        name: 'Markdown',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-text"><path style="stroke: none;" d="M5 6 h20 v4 h-1 a2 2 0 0 0 -2 -2 h-5 v16 a2 2 0 0 0 2 2 v1 h-8 v-1 a2 2 0 0 0 2 -2 v-16 h-5 a2 2 0 0 0 -2 2 h-1z"></path></svg>',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-textelement"><path transform="scale(0.72) translate(1 12)" style="stroke: none;" d="M0 15 l5.625 -15 l4 0 l5.625 15 l-4 0 l-1.5 -4 l-4.25 0 l-1.5 4z m4 0 m1.5 -4 m1 -2.6666 l2.25 0 l-1.125 -3z" /><path transform="scale(0.72) translate(17 12)" style="stroke: none;" d="M0 0 l4 0 a4.2 4 0 0 1 2 7 a2.3 2.5 0 0 1 0 8 l-6 0z m3 2 l0 4 a2 1.5 0 0 0 0 -4z  m0 6 l0 5 a4 2.5 0 0 0 0 -5z" /><path transform="scale(0.8) translate(25 11)" style="stroke: none;" d="M9 0.5 l0 3 a4 4 0 1 0 0 6 l0 3 a6.7 6.7 0 1 1 0 -12z" /></svg>',
        description: {
            en: 'Text content with support for math',
            fi: 'Tekstisisältöä matematiikkatuella',
            sv: 'Text med stöd för matematik'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['general'],
        weight: 1000
    }
    
    /******
     * Styles
     ******/
    MarkdownElement.styles = [
        //'.markdownelement {font-family: Helvetica, sans-serif;}',
        '.markdownelement[data-elementmode$="view"] {margin: 0; width: auto; text-align: justify; cursor: text;}',
        '.markdownelement[data-elementmode="noteview"] {cursor: url("data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' version=\'1.1\' width=\'30\' height=\'30\' viewBox=\'0 0 30 30\' class=\'mini-icon mini-icon-marginnotes\'%3E%3Cpath style=\'stroke: none;\' d=\'M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6z m-4 2 h25 v2 h-25z\'%3E%3C/path%3E%3C/svg%3E") 3 27, cell;}',
        // Textarea
        '.markdownelement textarea.markdownelement-editarea {min-width: 10em; min-height: 3em; width: 100%; margin: 0; padding: 0.5em; border: none; box-sizing: border-box; -webkit-box-sizing: border-box; -moz-box-sizing: border-box; transition: height 0.1s; overflow: hidden; background-color: white; color: black;}',
        // Edit / Author mode
        '.markdownelement[data-elementmode="edit"], .markdownelement[data-elementmode="author"] {border: 1px solid #aaa; margin: 0; position: relative; min-width: 15em;}',
        // Review mode
        //'.markdownelement[data-elementmode="review"] {border: 1px solid red; border-radius: 0.5em; padding: 0.5em;}',
        // Clickable buttons
        '.markdownelement .markdownelement-button {cursor: pointer;}',
        '.markdownelement .markdownelement-buttonset {position: absolute; top: 3px; right: 5px;}',
        '.markdownelement .markdownelement-buttonset span {margin: 0 5px;}',
        '.markdownelement .markdownelement-buttonset span svg {width: 20px; height: 20px;}',
        '.markdownelement .markdownelement-editbar {height: 20px; border: 1px solid #888; border-radius: 0 0 0.5em 0.5em; text-align: right; padding: 2px;}',
        '.markdownelement .markdownelement-editbar span {text-align: left; margin: 0 5px;}',
        // Preview
        '.markdownelement .markdownelement-preview {font-size: 80%; padding: 0 0.5em; display: none; background-color: #f0f0f0; overflow: auto; box-sizing: border-box;}',
        '.markdownelement.markdownelement-previewmode .markdownelement-preview {display: block; margin: 0;}',
        '.markdownelement.markdownelement-previewmode svg.mini-icon-preview path.nopreview {stroke: #a00;}',
        '.markdownelement.markdownelement-iswide.markdownelement-previewmode textarea.markdownelement-editarea {width: 50%; display: inline-block; vertical-align: top;}',
        '.markdownelement.markdownelement-iswide.markdownelement-previewmode .markdownelement-preview {width: 50%; display: inline-block; vertical-align: top; margin-left: -0.5em;}',
        // Paragraphs
        '.markdownelement p {line-height: 1.7em;}',
        // Headings
        '.markdownelement h1 {text-align: left; margin: 1em 0;}',
        '.markdownelement h2 {text-align: left; margin: 2em 0 1em;}',
        '.markdownelement h3 {text-align: left; margin: 2em 0 1em;}',
        // Links
        '.markdownelement a {color: #4965BF; font-weight: bold;}',
        // Highlights
        // '.markdownelement p em {color: #B51C1C;}',//removed for now
        // Lists (fix left positioning, when we have left floated elements)
        '.markdownelement ul, .markdownelement ol {width: auto; overflow: hidden;}',
        // Math displaystyle
        '.markdownelement .mathdisplaystyle {display: block; text-align: left; margin: 1.5em 0 1.5em 5em; font-size: 110%;}',
        // Tables
        '.markdownelement table {border-collapse: collapse; margin: 0.5em 1em;}',
        '.markdownelement thead {border-top: 1px solid black;}',
        '.markdownelement tbody {border-top: 1px solid black; border-bottom: 1px solid black;}',
        '.markdownelement tr:nth-child(odd) {background-color: #fefefe;}',
        '.markdownelement tr:nth-child(even) {background-color: #f4f4f4;}',
        '.markdownelement tr:nth-child(odd) {background-color: rgba(230,230,230,0.2);}',
        '.markdownelement tr:nth-child(even) {background-color: rgba(100,100,100,0.15);}',
        '.markdownelement td, .markdownelement th {padding: 0.2em 0.5em; vertical-align: top;}',
        '.markdownelement th:empty {padding: 0;}',
        // Inline code
        '.markdownelement p code {background-color: #eee; color: black; padding: 0.1em 0.3em;}',
        // Code blocks
        '.markdownelement pre code {display: block; overflow: auto; padding: 0.8em;}',
        // Block quote
        '.markdownelement blockquote {border-left: 6px solid #aaa; margin-left: 0; padding-left: 0.6em;}',
        // Wysiwyg editing
        '.markdownelement.markdownelement-wysiwygmode {min-height:3em;}',
        '.markdownelement.markdownelement-wysiwygmode .medium-editor-toolbar.medium-editor-toolbar-active{position:fixed;}',//TODO:remove if positionproblem fixed
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-wysiwygedit {min-height: 3em; padding: 0.2em;background-color:white; color: black; border-radius: 0 0 0.5em 0.5em;}',
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-wysiwygedit.medium-editor-placeholder::after {padding: 0.5em 2em;}',
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-preview {display: block; margin: 0;}',
        '.markdownelement.markdownelement-wysiwygmode textarea.markdownelement-editarea,.markdownelement.markdownelement-wysiwygmode .markdownelement-buttonset {display: none;}',
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-wysiwygedit:focus { outline: 0px solid transparent; }',
        '#medium-editor-toolbar-anchor-preview a.medium-editor-toolbar-anchor-preview-inner { color:black; }',
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-wysiwygedit pre{border:1px dashed blue;padding:1em 0.2em;margin:10px 3px 3px 3px;border-radius:5px;}',
        '.markdownelement.markdownelement-wysiwygmode .markdownelement-wysiwygedit pre:before{content:"code";border:1px solid blue;border-radius:4px;position:absolute;margin-top:-1.5em;background-color:white; color: black; padding:0 2px;}',
        //'.markdownelement.markdownelement-wysiwygmode.markdownelement-wysiwygstatictoolbar .medium-editor-toolbar.static-toolbar.medium-editor-toolbar-active{left: 10px!important;top: 6.5em!important;width: 50px;}',
        '.markdownelement.markdownelement-wysiwygmode.markdownelement-wysiwygstatictoolbar .medium-editor-toolbar.static-toolbar.medium-editor-toolbar-active{left: 10px!important;bottom: 50px!important;top:auto!important;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions ul.medium-editor-toolbar-actions{display:none;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li:hover ul.medium-editor-toolbar-actions{display:inline-block;position:absolute;bottom:0;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li:hover ul.medium-editor-toolbar-actions li button{display:inline-block;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li ul.medium-editor-toolbar-actions.activeButtons{display:inline-block;position:absolute;bottom:0;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions .medium-editor-toolbar-actions.activeButtons  button{display:none;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions  ul.medium-editor-toolbar-actions.activeButtons li .medium-editor-button-active{display:inline-block;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li button.submenubutton::before {content: "+";border: 1px solid #000; border-radius: 10px;padding: 0; font-size: 14px;position: absolute; top: 2px; right: 2px; align-content: top; background-color: rgba(255,255,255,0.5); color: black; width: 20px; height: 20px; line-height: 20px; display: block;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li:hover button.submenubutton::before {content: ""}',
        //'.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li ul{width:55px;}',
        '.markdownelement.markdownelement-wysiwygmode ul.medium-editor-toolbar-actions li button.submenubutton{position:relative;}',
        '.markdownelement.markdownelement-wysiwygmode .medium-editor-toolbar:not(.medium-editor-toolbar-active) {display: none;}',
        //coloring of elements
        '.markdownelement span.colored[data-dataid="colored_blue"]{color:blue;}',
        '.markdownelement span.colored[data-dataid="colored_red"]{color:red;}',
        '.markdownelement span.colored[data-dataid="colored_green"]{color:green;}',
        '.markdownelement span.colored[data-dataid="colored_yellow"]{color:yellow;}',
        '.markdownelement span.colored[data-dataid="colored_white"]{color:white;}',
        // overwrite medium-editor styles
        '.medium-editor-toolbar li button {color: inherit; box-shadow: none; background: transparent; text-shadow: none;}',
        '.medium-editor-toolbar li li {float: none;}',
        '.medium-editor-toolbar li li button {display: block;}',
        // override mathquill styles
        '.markdownelement.markdownelement-wysiwygmode .mathquill-editable, .markdownelement.markdownelement-wysiwygmode .mathquill-embedded-latex .mathquill-editable {border: 1px solid #ccc;}',
        '.markdownelement.markdownelement-wysiwygmode .mathquill-reverted {display: inline-block; vertical-align: middle; overflow: hidden; color: #f3f3f3; background-color: #f3f3f3; border: 1px solid transparent; padding: 2px;}'
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    MarkdownElement.localization = {
        "en": {
            "markdownelement:preview": "Preview",
            "markdownelement:remove": "Remove",
            "markdownelement:typelink": "Paste or type a link"
        },
        "fi": {
            "markdownelement:preview": "Esikatselu",
            "markdownelement:remove": "Poista",
            "markdownelement:typelink": "Liitä tai kirjoita linkin osoite"
        },
        "sv": {
            "markdownelement:preview": "Förhandsgranskning",
            "markdownelement:remove": "Ta bort",
            "markdownelement:typelink": "Klistra eller skriv in en länk"
        }
    }

    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(MarkdownElement.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(MarkdownElement.localization);
    }

    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', MarkdownElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', MarkdownElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var markdownelem = new MarkdownElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.markdownelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in markdownelement.');
            return false;
        }
    }
    
})(window, jQuery, window.ebooklocalizer);