/******
@name MarginNotes
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class MarginNotes
@description A class and jQuery-plugin for a adding and showing user's own notes and highlights on notebook (or reader?).

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /**
     * Course note tool for markings, highlightings and notes on notebooks.
     * @constructor
     * @param {jQuery} place    DOM-place for course videoconf
     * @param {Object} options  data for course videoconf
     */
    var MarginNotes = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.initData();
        this.setAttrs();
        this.setStyles();
        this.show();
        //this.registerConfForm();
        this.setNoteview();
        this.appReady();
    }
    
    MarginNotes.prototype.init = function(options) {
        this.settings = options.settings;
        this.configs = options.configs;
        this.users = this.settings.users;
        this.noteedit = false;
    };
    
    MarginNotes.prototype.initData = function() {
        this.appid = this.settings.appid;
        this.apptype = 'marginnotes';
        this.place.attr('data-appname', this.appid);
    };
    
    MarginNotes.prototype.initHandlers = function() {
        var mnotes = this;
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            mnotes.close();
        }).off('closeapp').on('closeapp', function(event, data){
            event.stopPropagation();
            mnotes.close();
        }).off('setconfigs').on('setconfigs', function(event, data){
            event.stopPropagation();
            mnotes.setConfigs(data);
        }).off('change', '.marginnotes-marginstatus').on('change', '.marginnotes-marginstatus', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var action = input.val();
            switch (action) {
                case 'notehide':
                    mnotes.setNoteedit(false);
                    mnotes.setConfigs({noteview: false});
                    break;
                case 'noteview':
                    mnotes.setNoteedit(false);
                    mnotes.setConfigs({noteview: true});
                    break;
                case 'noteedit':
                    mnotes.setConfigs({noteview: true});
                    mnotes.setNoteedit(true);
                    break;
                default:
                    break;
            };
        });
    };
    
    MarginNotes.prototype.setAttrs = function() {
        this.place.addClass('marginnotes');
    };
    
    MarginNotes.prototype.setConfigs = function(data) {
        var noteview = data.noteview;
        this.configs = $.extend(this.configs, data);
        if (typeof(noteview) === 'boolean') {
            this.setNoteview();
        };
        this.saveConfigs();
    };
    
    MarginNotes.prototype.saveConfigs = function(data) {
        // Save config data in parent module
        if (typeof(data) === 'undefined') {
            // Not given data, but save all. (Else save only given data)
            data = {
                type: 'configsave',
                appname: this.appid,
                appid: this.appid,
                apptype: this.apptype,
                configs: JSON.parse(JSON.stringify(this.configs))
            };
        };
        this.place.trigger('saveconfigs', data);
    }
    
    MarginNotes.prototype.setNoteview = function() {
        if (this.configs.noteview) {
            this.place.trigger('setmarginnotes_on');
        } else {
            this.place.trigger('setmarginnotes_off');
        };
    };
    
    MarginNotes.prototype.setNoteedit = function(mode) {
        var changed = (this.noteedit === !mode);
        this.noteedit = !!mode;
        if (changed) {
            this.place.trigger(this.noteedit ? 'setmarginnotes_edit_on' : 'setmarginnotes_edit_off');
        };
    };
    
    MarginNotes.prototype.show = function() {
        this.view();
        this.announceMargin();
    };
    
    MarginNotes.prototype.view = function() {
        var uilang = this.settings.uilang;
        var html = this.template.html;
        this.place.html(html);
        this.drawButtons();
        this.place.find('.marginnotes-title').text(ebooklocalizer.localize('marginnotes:margin_notes', uilang));
    };
    
    MarginNotes.prototype.drawButtons = function() {
        var uilang = this.settings.uilang;
        var buttonset = this.place.find('.marginnotes-controlbuttons');
        var noteview = this.configs.noteview;
        var buttons = this.radiobuttons;
        var button, buttelem, butt, input, label;
        for (var i = 0, len = buttons.length; i < len; i++) {
            buttelem = $(this.template.setbutton);
            butt = buttelem.children('.marginnotes-button');
            button = buttons[i];
            label = ebooklocalizer.localize(button.label, uilang);
            butt.attr('data-action', button.action).attr('data-button-name', button.name).attr('title', label);
            butt.html('<span>' + (buttons[i].icon || label) + '</span>');
            input = buttelem.children('input');
            input.attr('value', button.action);
            if ((button.name === 'noteview' && noteview) || (button.name === 'notehide' && !noteview)) {
                input.attr('checked', 'checked');
            };
            buttonset.append(buttelem);
        };
    };
    
    MarginNotes.prototype.announceMargin = function() {
        var margin = this.place.find('.marginnotes-marginarea');
        var getMargin = function(id, appid) {
            appid = appid || 'noname';
            var appmargin = margin.children('[data-marginappid="' + escapeHTML(appid) + '"]');
            if (appmargin.length === 0) {
                appmargin = $('<div class="marginnotes-marginapparea" data-marginappid="'+escapeHTML(appid)+'"></div>');
                margin.append(appmargin);
            }
            var newarea = appmargin.children('[data-marginid="' + escapeHTML(id) + '"]');
            if (newarea.length === 0) {
                newarea = $('<div class="marginnotes-marginitem" data-marginid="'+escapeHTML(id)+'"></div>');
                appmargin.append(newarea);
            };
            return newarea;
        };
        var clearMargin = function(appid) {
            var appmargin = margin.children('[data-marginappid="' + escapeHTML(appid) + '"]');
            appmargin.empty();
        };
        this.place.trigger('marginannouncement', [{
            info: {
                appid: this.appid,
                apptype: this.apptype
            },
            data: {
                getMargin: getMargin,
                clearMargin: clearMargin
            }
        }]);
    };
    
    MarginNotes.prototype.registerConfForm = function() {
        var formdata = {apptype: this.apptype, appid: this.appid, confform: this.getConfigForm()};
        this.place.trigger('registerconfigdialog', formdata);
    }
    
    /**
     * Inform outer system that the application is ready
     */
    MarginNotes.prototype.appReady = function(){
        var uilang = this.settings.uilang;
        this.place.trigger('appready', {apptype: this.apptype, appid: this.appid, title: ebooklocalizer.localize('marginnotes:margin_notes', uilang)});
    };

    MarginNotes.prototype.setStyles = function() {
        if ($('head style#marginnotesstyles').length === 0) {
            $('head').append('<style type="text/css" id="marginnotesstyles">'+this.styles+'</style>')
        }
    };
    
    MarginNotes.prototype.close = function() {
        this.place.off();
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    };
    
    MarginNotes.prototype.getConfigForm = function() {
        var uilang = this.settings.uilang;
        var form = {
            name: this.appid,
            title: ebooklocalizer.localize('marginnotes:margin', uilang),
            icon: this.icons.appicon,
            form: [],
            weight: 8
        };
        
        //form.form.push({
        //    name: 'noteview',
        //    label: ebooklocalizer.localize('marginnotes:show_notes', uilang),
        //    icon: this.icons.appicon,
        //    type: 'checkbox',
        //    attribute: 'noteview',
        //    defval: false
        //});
        return form;
    };
    
    MarginNotes.prototype.defaults = {
        settings: {
            uilang: 'en'
        }
    };
    
    MarginNotes.prototype.styles = [
        '.marginnotes {background-color: #eee; font-family: helvetica, Arial, sans-serif; padding: 1em;}',
        '.marginnotes-iconarea {display: inline-block;}',
        '.marginnotes-marginarea {margin: 0; margin-right: 1.5em;}'
    ].join('\n');
    
    MarginNotes.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-marginnotes"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6z m-4 2 h25 v2 h-25z"></path></svg>',
        refresh: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-refresh"><path style="stroke: none;" d="M16 14 l8 0 l0 -8 l-3 3 a8.5 8.5 0 1 0 0 12 l-2 -2 a5.6 5.6 0 1 1 0 -8 z"></path></svg>'
    }
    
    MarginNotes.prototype.template = {
        html: [
            '<h1 class="marginnotes-header">',
            '    <div class="marginnotes-iconarea"><div class="marginnotes-icon">'+MarginNotes.prototype.icons.appicon+'</div></div>',
            '    <span class="marginnotes-title"></span>',
            '</h1>',
            '<div class="marginnotes-controlbuttons ffwidget-buttonset ffwidget-horizontal"></div>',
            '<div class="marginnotes-marginarea"></div>'
            //'<div class="marginnotes-actions ffwidget-buttonset ffwidget-horizontal"><button class="marginnotes-button-start ffwidget-setbutton" data-action="videoconf_start"></button></div>'
        ].join('\n'),
        setbutton: [
            '<label class="ffwidget-setbutton-wrapper"><input type="radio" name="marginnotes-marginstatus" class="marginnotes-marginstatus" /><span class="marginnotes-button ffwidget-setbutton"></span></label>'
        ].join('')
    };
    
    MarginNotes.prototype.radiobuttons = [
        {
            name: 'notehide',
            label: 'marginnotes:hidenotesaction',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-hidenotes"><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path></svg>',
            action: 'notehide'
            
        },
        {
            name: 'noteview',
            label: 'marginnotes:shownotesaction',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-shownotes"><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path><path style="fill: red; stroke: none;" d="M5 22 h20 v2 h-20z"></path></svg>',
            action: 'noteview'
        },
        {
            name: 'noteedit',
            label: 'marginnotes:editnotesaction',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-editnotes"><path style="fill-opacity: 1; stroke: none;" d="M6 8 m14 0 a4 6 0 0 1 0 12 a4 6 0 0 1 0 -12z m0 2 a2 4 0 0 0 0 8 a2 4 0 0 0 0 -8z"></path><path style="stroke: none;" d="M6 8 h2 v8 a2 2 0 0 0 4 0 v-8 h2 v8 a4 4 0 0 1 -8 0z"></path><path style="fill: red; stroke: none;" d="M5 22 h12 v2 h-12z"></path><path style="fill: #d00; stroke: none;" d="M18 23 l0 -4 l7.5 -14 l4 2 l-7.5 14z"></path></svg>',
            action: 'noteedit'
        }
    ];
    
    MarginNotes.localization = {
        "en": {
            "marginnotes:margin_notes": "Margin notes",
            "marginnotes:margin": "Margin",
            "marginnotes:show_notes": "Show notes",
            "marginnotes:hidenotesaction": "Hide notes",
            "marginnotes:shownotesaction": "Show notes",
            "marginnotes:editnotesaction": "Edit notes",
        },
        "fi": {
            "marginnotes:margin_notes": "Marginaalimerkinnät",
            "marginnotes:margin": "Marginaali",
            "marginnotes:show_notes": "Näytä merkinnät",
            "marginnotes:hidenotesaction": "Piilota merkinnät",
            "marginnotes:shownotesaction": "Näytä merkinnät",
            "marginnotes:editnotesaction": "Muokkaa merkintöjä",
        },
        "sv": {
            "marginnotes:margin_notes": "Marginalanteckningar",
            "marginnotes:margin": "Marginal",
            "marginnotes:show_notes": "Visa anteckningar",
            "marginnotes:hidenotesaction": "Göm anteckningar",
            "marginnotes:shownotesaction": "Visa anteckningar",
            "marginnotes:editnotesaction": "Redigera anteckningar",
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(MarginNotes.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(MarginNotes.localization);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var mnotes = new MarginNotes(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[marginnotesdata]]');
            return data;
        },
        'geticon': function() {
            return MarginNotes.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.marginnotes = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in marginnotes.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
