/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * Helper functions
     */
    
    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Linklist element to show and edit inner links in course.
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var LinkList = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init LinkList element
     * @param {Object} options - initing data of the element
     ******/
    LinkList.prototype.init = function(options){
        options = $.extend(true, {}, LinkList.defaults, options);
        this.settings = options.settings;
        this.type = 'linklistelement';
        this.setMode(options.settings.mode);
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator || this.settings.username,
            created: options.metadata.created || (new Date()).getTime(),
            modifier: options.metadata.modifier || this.settings.username,
            modified: options.metadata.modified || (new Date()).getTime(),
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.elementtypes = $.fn.elementset('getelementtypes', 'all');
        this.setAttrs();
    };
    
    /******
     * Set style-tag if needed.
     ******/
    LinkList.prototype.setStyles = function(){
        if ($('head style#linklistelement-style').length === 0) {
            $('head').append('<style id="linklistelement-style" type="text/css">'+LinkList.styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    LinkList.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = LinkList.modes[mode] || LinkList.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Set some attributes
     ******/
    LinkList.prototype.setAttrs = function(){
        this.place.addClass('linklistelement');
        this.place.attr('data-linkviewmode', this.data.viewmode);
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    LinkList.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the CodeElement element in different modes
     ******/
    LinkList.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
            this.place.attr('uilang', this.settings.uilang);
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        }
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    LinkList.prototype.view = function(){
        this.place.empty().html(LinkList.templates.view);
        var linklist = this.place.find('.linklistelement-linklist');
        var list = [];
        var cname, cdata, ctype, icon, title, text;
        for (var i = 0, len = this.data.contents.length; i < len; i++){
            cname = this.data.contents[i].id;
            cdata = this.data.contentdata[cname];
            ctype = cdata.data.type || '';
            icon = sanitize(cdata.data.icon || (this.elementtypes[ctype] && this.elementtypes[ctype].icon) || LinkList.icons.link);
            title = escapeHTML(cdata.data.title);
            text = escapeHTML(cdata.data.text);
            list.push('<li class="linklistelement-listitem"><div class="linklistelement-link" data-linkname="'+escapeHTML(cname)+'"><span class="linklistelement-icon">'+icon+'</span><span class="linklistelement-title">'+title+'</span></div><div class="linklistelement-description">'+text+'</div></li>')
        }
        linklist.append(list.join('\n'));
    }
    
    /******
     * Show the element in editable mode.
     ******/
    LinkList.prototype.edit = function(){
        var uilang = this.settings.uilang;
        this.place.empty();
        this.place.html(LinkList.templates.edit);
        var linkcontrol = [
            '<div class="linklistelement-linkcontrol">',
            '    <div class="linklistelement-linkcontrol-icon" draggable="true" data-action="linkmove">'+LinkList.icons.move+'</div>',
            '    <div class="linklistelement-linkcontrol-icon" data-action="linkremove">'+LinkList.icons.trashcan+'</div>',
            '</div>'
        ].join('');
        var linklist = this.place.find('.linklistelement-linklist');
        var list = [];
        var cname, cdata, ctype, icon, title, text;
        for (var i = 0, len = this.data.contents.length; i < len; i++){
            cname = this.data.contents[i].id;
            cdata = this.data.contentdata[cname];
            ctype = cdata.data.type || '';
            icon = sanitize(cdata.data.icon || (this.elementtypes[ctype] && this.elementtypes[ctype].icon) || LinkList.icons.link);
            title = escapeHTML(cdata.data.title);
            text = escapeHTML(cdata.data.text);
            list.push('<li class="linklistelement-listitem" data-linkname="'+cname+'" data-linkindex="'+i+'">'+linkcontrol+'<div class="linklistelement-linkdata"><div class="linklistelement-link"><span class="linklistelement-icon">'+icon+'</span><span class="linklistelement-title"><input type="text" placeholder="'+ebooklocalizer.localize('linklistelement:title', uilang)+'" value="'+title+'" /></span></div><div class="linklistelement-description"><textarea placeholder="'+ebooklocalizer.localize('linklistelement:description', uilang)+'">'+text+'</textarea></div></div></li>')
        }
        linklist.append(list.join('\n'));
        this.showControls();
    };
    
    /******
     * Show controlbar for editing
     ******/
    LinkList.prototype.showControls = function(){
        var uilang = this.settings.uilang;
        var controlbar = this.place.find('.linklistelement-controlbar');
        controlbar.empty();
        var viewmodes = ['list', 'grid'], classes;
        var html = ['<ul class="ffwidget-buttonset ffwidget-horizontal">'];
        for (var i = 0, len = viewmodes.length; i < len; i++) {
            classes = [];
            classes.push(this.data.viewmode === viewmodes[i] ? 'buttonselected' : '');
            html.push('<li class="linklistelement-viewmodeitem ffwidget-setbutton '+classes.join(' ')+'" data-key="viewmode" data-value="'+viewmodes[i]+'"><span class="linklistelement-viewmode-icon">'+LinkList.icons[viewmodes[i]]+'</span></li>');
        };
        html.push('</ul>');
        controlbar.append(html.join(''));
    };
    
    /******
     * Remove all event handlers
     ******/
    LinkList.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    LinkList.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in LinkList.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        });
        this.place.on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    LinkList.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        }).on('click', '.linklistelement-link', function(event, data){
            event.stopPropagation();
            var linkname = $(this).closest('[data-linkname]').attr('data-linkname');
            element.triggerLink(linkname);
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    LinkList.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('click', '.linklistelement-controlbar .linklistelement-viewmodeitem', function(event, data){
            var button = $(this);
            var value = button.attr('data-value');
            var key = button.attr('data-key');
            button.parent().children('.linklistelement-viewmodeitem').removeClass('buttonselected');
            button.addClass('buttonselected');
            element.data[key] = value;
            element.changed();
            element.show();
        }).on('change', '.linklistelement-title input', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            var linkname = input.closest('[data-linkname]').attr('data-linkname') || '';
            element.setLinkTitle(linkname, value);
            element.changed();
        }).on('change', '.linklistelement-description textarea', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            var linkname = input.closest('[data-linkname]').attr('data-linkname') || '';
            element.setLinkText(linkname, value);
            element.changed();
        }).on('click', '.linklistelement-icon', function(event, data){
            event.stopPropagation();
            var linkname = $(this).closest('.linklistelement-link').attr('data-linkname');
            element.triggerLink(linkname);
        }).on('click', '.linklistelement-linkcontrol-icon', function(event){
            event.stopPropagation();
            var icon = $(this);
            var action = icon.attr('data-action');
            var index = icon.closest('[data-linkindex]').attr('data-linkindex') | 0;
            icon.trigger(action, {index: index});
        }).on('dragenter dragover', '.linklistelement-droptarget, .linklistelement-listitem', function(event){
            // Highlight the droptarget
            var current = $(this);
            current.addClass('linklistelement-draghover');
            var dragitem = current.closest('.linklistelement-linklist').children('.linklistelement-currentlydragged');
            var dragindex = (dragitem.length > 0 ? dragitem.attr('data-linkindex') | 0 : 5000),
                hoverindex = current.attr('data-linkindex') | 0;
            if (dragindex < hoverindex) {
                dragitem.addClass('dropunder').removeClass('dropover');
                current.addClass('isunder').removeClass('isover');
            } else if (dragindex > hoverindex) {
                dragitem.addClass('dropover').removeClass('dropunder');
                current.addClass('isover').removeClass('isunder');
            };
        }).on('dragleave', '.linklistelement-droptarget, .linklistelement-listitem', function(event){
            // Remove highlight from the droptarget
            $(this).removeClass('linklistelement-draghover isover isunder');
            $(this).closest('.linklistelement-linklist').children().removeClass('dropunder dropover');
        }).on('dragover', '.linklistelement-droptarget', function(event){
            // Prevent default to allow drop!
            event.preventDefault();
        }).on('drop', '.linklistelement-droptarget, .linklistelement-listitem', function(event){
            // Do the drop magic
            event.stopPropagation();
            event.preventDefault();
            var place = $(this);
            var toid;
            if (place.is('.linklistelement-listitem')) {
                toid = $(this).attr('data-linkindex') | 0;
            };
            var ev = event.originalEvent;
            var datastr = ev.dataTransfer.getData('text');
            var data = {};
            try {
                data = JSON.parse(datastr);
            } catch (err) {
                data = {type: 'error'};
            };
            switch (data.type) {
                case 'contentlink':
                    element.addLink(data, toid);
                    break;
                case 'linkmovedata':
                    var listid = element.place.attr('data-element-name');
                    if (data.listid === listid && element.data.contents[data.index].id === data.id) {
                        element.moveLink(data.index, toid);
                    } else {
                        element.addLink(data.data, toid);
                    }
                    break;
                default:
                    break;
            };
        }).on('drop', '.linklistelement-listitem', function(event){
            // Drop on the place of existing link
            event.stopPropagation();
            event.preventDefault();
            var ev = event.originalEvent;
            var datastr = ev.dataTransfer.getData('text');
            var data = {};
            try {
                data = JSON.parse(datastr);
            } catch (err) {
                data = {type: 'error'};
            };
        }).on('dragstart', '.linklistelement-linkcontrol-icon[data-action="linkmove"]', function(event){
            event.stopPropagation();
            var wrapper = $(this).closest('li.linklistelement-listitem');
            wrapper.addClass('linklistelement-currentlydragged');
            var listid = element.place.attr('data-element-name');
            var index = wrapper.attr('data-linkindex') | 0;
            var cid = element.data.contents[index].id;
            var movedata = {
                type: 'linkmovedata',
                listid: listid,
                index: index,
                id: cid,
                data: $.extend(true, {}, element.data.contentdata[cid])
            };
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text', JSON.stringify(movedata, null, 4));
            ev.dataTransfer.effectAllowed = 'move';
        }).on('dragend', '.linklistelement-linkcontrol-icon[data-action="linkmove"]', function(event){
            var wrapper = $(this).closest('li.linklistelement-linkitem');
            wrapper.removeClass('linklistelement-currentlydragged');
        }).on('linkadd', '.linklistelement-droptarget, .linklistelement-listitem', function(event, data){
            event.stopPropagation();
            var place = $(this);
            var toid;
            if (place.is('.linklistelement-listitem')) {
                toid = $(this).attr('data-linkindex') | 0;
            };
            if (data.type === 'contentlink') {
                element.addLink(data, toid);
            };
        }).on('linkremove', function(event, data){
            event.stopPropagation();
            element.removeLink(data.index);
        });
    };
    
    /******
     * Update the metadata of the element.
     ******/
    LinkList.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Add a new link
     ******/
    LinkList.prototype.addLink = function(data, index){
        if (typeof index === 'undefined') {
            index = this.data.contents.length;
        };
        var cid = this.getNewId();
        this.data.contents.splice(index, 0, {id: cid});
        this.data.contentdata[cid] = data;
        this.changed();
        this.show();
    }
    
    /******
     * Remove a link
     ******/
    LinkList.prototype.removeLink = function(index){
        if (index >= 0 && index < this.data.contents.length) {
            var cid = this.data.contents[index].id;
            this.data.contents.splice(index, 1);
            delete this.data.contentdata[cid];
            this.changed();
            this.show();
        };
    };
    
    /******
     * Move a link
     ******/
    LinkList.prototype.moveLink = function(index, toindex){
        var length = this.data.contents.length;
        if (typeof toindex === 'undefined') {
            toindex = length;
        };
        if (index >= 0 && index < length && toindex >= 0 && toindex < length + 1) {
            var cid = this.data.contents[index].id;
            //var toid = data.id;
            this.data.contents.splice(index, 1);
            this.data.contents.splice(toindex, 0, {id: cid});
            this.changed();
            this.show();
        };
    };
    
    /******
     * Set the title of given link
     ******/
    LinkList.prototype.setLinkTitle = function(linkname, title){
        this.data.contentdata[linkname].data.title = title;
    }
    
    /******
     * Set the text of given link
     ******/
    LinkList.prototype.setLinkText = function(linkname, text){
        this.data.contentdata[linkname].data.text = text;
    }
    
    /******
     * Trigger a gotolink-event with data of the link.
     ******/
    LinkList.prototype.triggerLink = function(linkname){
        var linkdata = JSON.parse(JSON.stringify(this.data.contentdata[linkname]));
        if (linkdata.type === 'contentlink') {
            this.place.trigger('gotolink', linkdata);
        };
    };
    
    /******
     * Trigger element_changed event.
     ******/
    LinkList.prototype.changed = function(){
        this.updateMetadata();
        this.place.trigger('element_changed', {type: 'linklistelement'});
    }

    
    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "linklistelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {}
     *   }
     ******/
    LinkList.prototype.getData = function(){
        var result = {
            type: this.type,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return result;
    }
    
    /******
     * Get an id for a new element.
     * @param {String} eltype - Type of an element.
     * @returns {String} An id for the new element.
     ******/
    LinkList.prototype.getNewId = function(){
        var idparts = ['contentlink'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };

    /******
     * Default settings
     ******/
    LinkList.defaults = {
        type: 'linklistelement',
        metadata: {
            creator: '',
            created: 0,
            modifier: '',
            modified: 0,
            tags: []
        },
        data: {
            viewmode: 'list',
            contents: [],
            contentdata: {}
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en',
            lang: 'en'
        }
    }
    
    /******
     * Modes
     ******/
    LinkList.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    
    
    /******
     * Templates
     ******/
    LinkList.templates = {
        view: [
            '<ul class="linklistelement-linklist">',
            '</ul>'
        ].join('\n'),
        edit: [
            '<div class="linklistelement-controlbar ffwidget-background"></div>',
            '<ul class="linklistelement-linklist">',
            '</ul>',
            '<div class="linklistelement-droptarget"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-add"><path style="stroke: none;" d="M12 2 l8 0 l0 10 l10 0 l0 8 l-10 0 l0 10 l-8 0 l0 -10 l-10 0 l0 -8 l10 0z"></path></svg></div>'
        ].join('\n')
    }
    
    /******
     * Icons
     ******/
    LinkList.icons = {
        link: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z"></path></svg>',
        list: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-list"><path style="stroke: none;" d="M5 4 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m7.5 -18 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        grid: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-grid"><path style="stroke: none;" d="M3 3 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z"></path></svg>',
        trashcan: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>',
        move: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-move mini-icon-drag"><path style="stroke: none;" d="M15 2 l5 5 l-4 0 l0 7 l7 0 l0 -4 l5 5 l-5 5 l0 -4 l-7 0 l0 7 l4 0 l-5 5 l-5 -5 l4 0 l0 -7 l-7 0 l0 4 l-5 -5 l5 -5 l0 4 l7 0 l0 -7 l-4 0z"></path></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    LinkList.elementinfo = {
        type: 'linklistelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'linklistelement',
        name: 'LinkList',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z"></path></svg>',
        description: {
            en: 'List of links',
            fi: 'Linkkilista',
            sv: 'Interna länkar'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['bookmarks']
    }
    
    /******
     * Styles
     ******/
    LinkList.styles = [
        '.linklistelement[data-elementmode="edit"] {border: 1px solid #aaa;}',
        
        // List in view / grid modes
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-linklist {list-style: none; margin: 0; padding: 0;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem {display: inline-block; vertical-align: top; max-height: 8em; max-width: 6em; min-width: 3em; padding: 3px; text-align: center; position: relative;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem .linklistelement-link {cursor: pointer;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem .linklistelement-icon {display: inline-block; width: 30px; height: 30px; border: 1px solid #aaa; border-radius: 6px; background-color: #fafafa; box-shadow: 2px 2px 5px rgba(0,0,0,0.3);}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-icon svg {width: 30px; height: 30px;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem .linklistelement-title {display: block; font-size: 70%; font-family: mono; white-space: no-wrap; overflow: hidden; text-overflow: ellipsis;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem .linklistelement-description {display: none;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="grid"] .linklistelement-listitem:hover .linklistelement-description:not(:empty) {display: block; position: absolute; top: 50px; width: 15em; border: 1px solid #aaa; background-color: #ffa; padding: 0.5em; z-index: 3; text-align: left; font-size: 80%;}',
        
        // List in view / list mode
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-linklist {list-style: none; margin: 0 0 0 1em; padding: 0;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-link {cursor: pointer;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-icon svg {width: 15px; height: 15px;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-icon {margin-right: 5px;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-title {text-decoration: underline; color: blue;}',
        '.linklistelement[data-elementmode="view"][data-linkviewmode="list"] .linklistelement-description {margin-left: 25px; font-size: 80%;}',
        
        // List in edit (always list mode)
        '.linklistelement[data-elementmode="edit"] .linklistelement-linklist {list-style: none; margin: 0.5em; padding: 0;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem {border: 1px solid #aaa; margin: 0.3em 0; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; align-content: flex-start; background-color: white; transition: margin 0.5s; position: relative; z-index: 3;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-currentlydragged {opacity: 0.3; z-index: 1;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-currentlydragged.dropunder {margin-bottom: -1.2em;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-currentlydragged.dropover {margin-top: -1.2em;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-draghover {box-shadow: 2px 2px 4px rgba(0,0,0,0.5); z-index: 2; transition: margin 0.5s;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-draghover.isover {margin-top: 1.5em; box-shadow: 0 -1.5em 0 rgba(215,215,215,0.5);}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-listitem.linklistelement-draghover.isunder {margin-bottom: 1.5em; box-shadow: 0 1.5em 0 rgba(215,215,215,0.5);}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkcontrol {width: 20px; text-align: center; background-color: #eee; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; align-content: flex-start;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkcontrol-icon {padding: 3px 0;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkcontrol svg {width: 15px; height: 15px;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkcontrol [data-action="linkmove"] {cursor: move;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkcontrol [data-action="linkremove"] {cursor: pointer;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-linkdata {-webkitflex-grow: 1; -ms-flex-grow: 1; flex-grow: 1;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-icon {margin: 0 3px; display: inline-block; float: left;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-icon svg {width: 30px; height: 30px;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-icon {margin: 0 5px 5px 0; cursor: pointer;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-title {margin-left: 36px; display: block; border-left: 1px solid #aaa;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-title input {border: 1px solid #ddd; border-bottom: 1px solid #777; background-color: #eee; width: 100%; box-sizing: border-box;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-title input:focus {background-color: white;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-description {margin: 0 0 0 36px; font-size: 80%; border-left: 1px solid #aaa;}',
        '.linklistelement[data-elementmode="edit"] .linklistelement-description textarea {box-sizing: border-box; width: 100%; border: none; background-color: #ffa;}',
        
        // Controls
        '.linklistelement-controlbar ul {list-style: none; margin: 0; padding: 3px 1em;}',
        '.linklistelement-controlbar .linklistelement-viewmodeitem {display: inline-block; margin: 0; padding: 2px 8px;}',
        '.linklistelement-viewmodeitem svg {width: 15px; height: 15px;}',
        
        // Droptarget
        '.linklistelement-droptarget {margin: 0.5em; padding: 3px; background-color: #fafafa; border: 1px dashed #ccc; text-align: center;}',
        '.linklistelement-droptarget svg {width: 17px; height: 17px;}',
        '.linklistelement-droptarget svg path {fill: #888;}',
        '.linklistelement-droptarget.linklistelement-draghover {background-color: #999;}',
        '.linklistelement-droptarget.linklistelement-draghover svg path {fill: white;}'

    ].join('\n');
    
    /******
     * Localization strings
     ******/
    LinkList.localization = {
        "en": {
            "linklistelement:title": "Link title",
            "linklistelement:description": "Longer description of the link"
        },
        "fi": {
            "linklistelement:title": "Linkin otsikko",
            "linklistelement:description": "Linkin pidempi kuvaus"
        },
        "sv": {
            "linklistelement:title": "Länktitel",
            "linklistelement:description": "Längre beskrivning"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(LinkList.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(LinkList.localization);
    }

    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', LinkList.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', LinkList.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var element = new LinkList(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.linklistelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in linklistelement.');
            return false;
        }
    }

})(jQuery);