/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    
    /**
     * Helper functions
     */
    
    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * ImageElement for showing images.
     * @requires jQuery
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for element
     ******/
    var ImageElement = function(place, options){
        this.place = $(place);
        this.init(options);
        this.show();
        this.setAttrs();
    }
    
    ImageElement.prototype.init = function(options){
        options = $.extend(true, {}, ImageElement.defaults, options);
        this.settings = $.extend(true, {}, ImageElement.defaults.settings, options.settings);
        this.setMode(options.settings.mode);
        this.type = options.type;
        this.metadata = options.metadata;
        this.data = options.data;
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    ImageElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = ImageElement.modes[mode] || ImageElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    };
    
    /******
     * Set some attributes
     ******/
    ImageElement.prototype.setAttrs = function(){
        this.place.addClass('imageelement');
        if (this.data.alignment) {
            this.place.closest('.elementset-elementwrapper')
                .addClass('imageelement-wrapper')
                .attr('data-alignment', this.data.alignment);
        };
        if (typeof(this.data['size-x']) !== 'undefined') {
            var size = this.data['size-x'];
            if (this.editable) {
                this.place
                    .css('width', size)
                    .closest('.elementset-elementwrapper').css({'max-width': size, width: size});
                if (size === 'auto') {
                    this.place.closest('.elementset-elementwrapper').css({'max-width': '', width: ''});
                    // remove scaling
                    this.place.find('.imageelement-image img').removeClass('imageelement-scaled');
                    var svg = this.place.find('.imageelement-image svg');
                    svg.attr('class', (svg.attr('class') || '').replace(/(\s|^)imageelement-scaled(\s|$)/g, ' '));
                } else {
                    // add scaling
                    this.place.find('.imageelement-image img').addClass('imageelement-scaled');
                    var svg = this.place.find('.imageelement-image svg');
                    var svgclasses = svg.attr('class');
                    svgclasses = (svgclasses && /(\s|^)imageelement-scaled(\s|$)/.test(svgclasses) ? svgclasses : (svgclasses || '') + ' imageelement-scaled');
                    svg.attr('class', svgclasses);
                }
            } else {
                if (this.data.alignment !== 'center') {
                    this.place.closest('.elementset-elementwrapper').css({'max-width': size, width: size});
                } else {
                    this.place.css('width', size);
                };
                if (size === 'auto') {
                    //this.place.closest('.elementset-elementwrapper').css({'max-width': '', width: ''});
                    // remove scaling
                    this.place.find('.imageelement-image img').removeClass('imageelement-scaled');
                    var svg = this.place.find('.imageelement-image > svg').eq(0);
                    svg.attr('class', (svg.attr('class') || '').replace(/(\s|^)imageelement-scaled(\s|$)/g, ' '));
                } else {
                    // add scaling
                    this.place.find('.imageelement-container img').addClass('imageelement-scaled');
                    var svg = this.place.find('.imageelement-container > svg').eq(0);
                    var svgclasses = svg.attr('class');
                    svgclasses = (svgclasses && /(\s|^)imageelement-scaled(\s|$)/.test(svgclasses) ? svgclasses : (svgclasses || '') + ' imageelement-scaled');
                    svg.attr('class', svgclasses);
                }
            };
        };
    };
    
    ImageElement.prototype.changed = function(){
        var inputs = this.place.find('.imageelement-dataform input');
        var imdatatype, field;
        for (var i = 0, len = inputs.length; i < len; i++) {
            field = inputs.eq(i);
            imdatatype = field.attr('data-imdatatype');
            this.data[imdatatype] = field.val();
        };
        this.updateMetadata();
        this.place.trigger('element_changed', {type: 'imageelement', lang: this.settings.lang});
    };
    
    ImageElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.modifier = this.metadata.modifier || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modified = (new Date()).getTime();
    };

    ImageElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        }
    }
    
    ImageElement.prototype.view = function(){
        //var tablet = $('html').hasClass('touch');
        var tablet = false;
        var lang = this.place.closest('[lang]').attr('lang') || 'en';
        var uilang = this.settings.uilang;
        var html;
        var title = [
            ebooklocalizer.localize('imageelement:author', uilang)+': ' + escapeHTML(this.data.author),
            ebooklocalizer.localize('imageelement:license', uilang)+': ' + escapeHTML(this.data.license),
            ebooklocalizer.localize('imageelement:source', uilang)+': ' + escapeHTML(this.data.source),
            ebooklocalizer.localize('imageelement:notes', uilang)+': ' + escapeHTML(this.data.notes)
        ].join('&#013;');
        var imgurl = this.data.source.match(/(?:https?:\/\/|file:\/\/)\S*/);
        var imglink = '<a href="javascript:;" class="ebook-imagelink" target="_blank">'+ImageElement.icons.info+'</a>';
        var $imgbox = this.place.closest('[data-boxtype="imagebox"]');
        var width = $imgbox.attr('data-imgwidth') || '';
        width = (width ? width+'px' : 'auto');
        var height = $imgbox.attr('data-imgheight') || '';
        height = (height ? height+'px' : 'auto');
        var databits = ['author', 'license', 'source', 'notes'];
        var imagedata = [];
        for (var i = 0, length = databits.length; i < length; i++) {
            var dataname = databits[i];
            if (this.data[dataname]) {
                if (dataname === 'source') {
                    imagedata.push('<td>'+ebooklocalizer.localize('imageelement:' + dataname, uilang) + '</td><td>' + (imgurl && !tablet ? '<a href="' + escapeHTML(imgurl[0]) + '" class="ebook-info-imagelink" target="_blank">' + escapeHTML(this.data.source) +'</a>' : escapeHTML(this.data.source)) + '</td>');
                } else {
                    imagedata.push('<td>'+ebooklocalizer.localize('imageelement:' + dataname, uilang) + '</td><td>' + escapeHTML(this.data[dataname]) + '</td>');
                }
            }
        }
        var imgpopup = '<div class="ebook-imagelink-popup"><strong>'+ebooklocalizer.localize('imageelement:imageinfo', uilang)+'</strong><table><tr>' + imagedata.join('</tr>\n<tr>') + '</tr></table><span class="ebook-popupclose">'+ImageElement.icons.close+'</span></div>';
        switch (this.data.type) {
            case 'svg':
                html = '<figure class="imageelement-container" title="'+title+'">' + sanitize(this.data.content) +'<figcaption>'+escapeHTML(this.data.caption || '')+'</figcaption>' + imglink + imgpopup + '</figure>';
                break;
            default:
                //html = '<div class="imageelement-container"><img src="' + this.data.content + '" alt="'+(this.data.content ? title : '')+'" title="'+title+'" style="max-width: 100%; width: '+(width === 'auto' ? '100%' : width)+'; max-height: '+height+';" />'+imglink+imgpopup+'</div>';
                html = '<figure class="imageelement-container"><img src="' + escapeHTML(this.data.content) + '" alt="'+(this.data.content ? title : '')+'" title="'+title+'" style="max-width: 100%; max-height: '+height+';" /><figcaption>'+escapeHTML(this.data.caption || '')+'</figcaption>'+imglink+imgpopup+'</figure>';
                break;
        }
        
        this.place.html(html);
        if (this.data.type === 'svg') {
            var svgimg = this.place.find('.imageelement-container > svg');
            if (width !== 'auto') {
                svgimg.attr('width', width);
            }
            if (height !== 'auto') {
                svgimg.attr('height', height);
            }
        }
    }
    
    ImageElement.prototype.edit = function(){
        var lang = this.place.closest('[lang]').attr('lang') || 'en';
        var uilang = this.settings.uilang;
        var $imgbox = this.place.closest('[data-boxtype="imagebox"]');
        var width = $imgbox.attr('data-imgwidth') || '';
        width = (width ? width+'px' : 'auto');
        var height = $imgbox.attr('data-imgheight') || '';
        height = (height ? height+'px' : 'auto');
        var form = [
            '<div class="imageelement-dataform-title"><span class="imageelement-dataform-icon">'+ImageElement.icons.info+'</span></div>',
            '<table class="imageelement-datatable"><tbody>',
            '<tr><td><label>' + ebooklocalizer.localize('imageelement:author', uilang) + '</label></td><td><input type="text" data-imdatatype="author" value="'+escapeHTML(this.data.author)+'" placeholder="John Smith" required></input></td></tr>',
            '<tr><td><label>' + ebooklocalizer.localize('imageelement:license', uilang) + '</label></td><td><input type="text" data-imdatatype="license" value="'+escapeHTML(this.data.license)+'" placeholder="CC-by" required></input></td></tr>',
            '<tr><td><label>' + ebooklocalizer.localize('imageelement:source', uilang) + '</label></td><td><input type="text" data-imdatatype="source" value="'+escapeHTML(this.data.source)+'" placeholder="http://opencliparts.org" required></input></td></tr>',
            '<tr><td><label>' + ebooklocalizer.localize('imageelement:notes', uilang) + '</label></td><td><input type="text" data-imdatatype="notes" value="'+escapeHTML(this.data.notes)+'"></input></td></tr>',
            '</tbody></table>'
        ].join('\n');
        var imagedata;
        if (this.data.content !== '') {
            switch (this.data.type){
                case 'svg':
                    imagedata = '<div class="imageelement-container">'+sanitize(this.data.content)+'</div>';
                    break;
                default:
                    //imagedata = '<div class="imageelement-container"><img src="'+this.data.content+'" style="width: ' + (width === 'auto' ? '100%' : width) + '; height: ' + height + '; max-width: 100%;" /></div>';
                    imagedata = '<div class="imageelement-container"><img src="'+escapeHTML(this.data.content)+'" style="max-width: 100%;" /></div>';
                    break;
            }
        } else {
            imagedata = '<div class="imageelement-droparea">'+ebooklocalizer.localize('imageelement:drophere', uilang)+'</div>';
        };
        var dialog = this.getDialog();
        var html = [
            '<div class="imageelement-dialog ffwidget-background"><label class="imageelement-fileinput"><input type="file" data-imdatatype="file" value="'+escapeHTML(this.data.file)+'"></input><span class="imageelement-filelabel">'+ImageElement.icons.file+'</span></label><input type="hidden" data-imdatatype="file" value="'+escapeHTML(this.data.file)+'"></input>'+dialog+'</div>',
            '<div class="imageelement-dataform">' + form + '</div>',
            '<figure class="imageelement-image">'+imagedata+'<figcaption><input data-attribute="caption" type="text" placeholder="'+ebooklocalizer.localize('imageelement:caption', uilang)+'" value="'+escapeHTML(this.data.caption || '')+'" /></figcaption></figure>'
            //'<div class="imageelement-editbar ffwidget-background"><span class="imageelement-button" data-buttontype="remove" title="'+ebooklocalizer.localize('imageelement:remove', uilang)+'">' + ImageElement.removeicon + '</span></div>'
        ].join('\n');
        this.place.html(html);
        if (this.data.type === 'svg') {
            var svgimg = this.place.find('.imageelement-container > svg');
            if (width !== 'auto') {
                svgimg.attr('width', width);
            }
            if (height !== 'auto') {
                svgimg.attr('height', height);
            }
        };
        if (!this.data.author || !this.data.license || !this.data.source) {
            this.place.addClass('imageelement-showinfo');
        }
        if (!this.data.author) {
            this.place.find('[data-imdatatype="author"]').focus();
        }
    }
    
    ImageElement.prototype.loadFile = function(file){
        var element = this;
        var $imgbox = this.place.closest('[data-boxtype="imagebox"]');
        var dataurl = '';
        var filename = file.name || '';
        var width = $imgbox.attr('data-imgwidth') || '';
        width = (width ? width+'px' : 'auto');
        var height = $imgbox.attr('data-imgheight') || '';
        height = (height ? height+'px' : 'auto');
        var imageplace = this.place.find('.imageelement-container, .imageelement-droparea');
        if (file && (file.type === 'image/png' || file.type === 'image/gif' || file.type === 'image/jpg' || file.type === 'image/jpeg')) {
            var imgtype = file.type.split('/')[1];
            var reader = new FileReader();
            reader.onloadend = function(ev){
                dataurl = reader.result;
                imageplace.children().remove();
                var tmpimage = new Image();
                tmpimage.onload = function(){
                    var imwidth = tmpimage.width;
                    var imheight = tmpimage.height;
                    if (imwidth > 1024 || imheight > 1024) {
                        dataurl = element.scaleImage(tmpimage, 1024, file.type);
                    };
                    var thumbnail = element.scaleImage(tmpimage, 50, file.type);
                    imageplace.html('<img src="'+escapeHTML(dataurl)+'" style="max-width: ' + width + '; max-height: ' + height + ';" />').removeClass('imageelement-droparea').addClass('imageelement-container');
                    element.data.content = dataurl;
                    element.data.type = imgtype;
                    element.data.file = filename;
                    element.data.thumbnail = thumbnail || '';
                    element.changed();
                    element.setAttrs();
                }
                tmpimage.src = dataurl;
            }
            reader.readAsDataURL(file);
        } else if (file && file.type === 'image/svg+xml') {
            var reader = new FileReader();
            reader.onloadend = function(ev){
                data = reader.result;
                data = $('<div>'+data+'</div>').find('svg').wrap('<div>').parent().html();
                imageplace.children().remove();
                imageplace.html(sanitize(data)).removeClass('imageelement-droparea').addClass('imageelement-container');
                element.data.content = sanitize(data);
                element.data.type = 'svg';
                element.data.file = filename;
                element.data.thumbnail = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-image"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 22 l22 0 l0 -22z m7 2 a5 5 0 0 0 0 10 a5 5 0 0 0 0 -10 m8 7 l5 10 l-16 0 l4 -6 l3 3z"></path></svg>';
                element.changed();
                element.setAttrs();
            };
            reader.readAsText(file);
        } else {
            console.log(file.type);
        };
    };
    
    ImageElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    ImageElement.prototype.initHandlersCommon = function(){
        var image = this;
        this.place.on('refresh', function(event){
            event.stopPropagation();
            image.show();
        });
        this.place.on('getdata', function(event){
            event.stopPropagation();
            var data = image.getData();
            image.place.data('[[imageelementdata]]', data);
            image.place.data('[[elementdata]]', data);
        });
        this.place.on('gethtml', function(event){
            event.stopPropagation();
            var data = image.getHtmltext();
            image.place.data('[[elementhtml]]', data);
        });
        this.place.on('getindexmetadata', function(event){
            event.stopPropagation();
            var data = image.getIndexData();
            image.place.data('[[elementindexmetadata]]', data);
        });
    };
    
    ImageElement.prototype.initHandlersNoneditable = function(){
        var image = this;
        this.place.on('click', 'a.ebook-imagelink', function(event){
            event.stopPropagation();
            event.preventDefault();
            $(this).closest('.imageelement-container').addClass('ebook-popup-visible');
        });
        this.place.on('click', '.ebook-imagelink-popup', function(event){
            //event.stopPropagation();
            //event.preventDefault();
            $(this).closest('.imageelement-container').removeClass('ebook-popup-visible');
        });
    }
    
    ImageElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('click', '.imageelement-dataform-title .imageelement-dataform-icon', function(event, data){
            event.stopPropagation();
            element.place.toggleClass('imageelement-showinfo');
        });
        this.place.on('change', '.imageelement-dataform input[type="text"]', function(e){
            element.changed();
        });
        this.place.on('dragover', '.imageelement-image', function(e){
            e.preventDefault();
            $(this).addClass('imageelement-draghover');
        });
        this.place.on('dragleave', '.imageelement-image', function(e){
            e.preventDefault();
            $(this).removeClass('imageelement-draghover');
        });
        this.place.on('change', '.imageelement-dialog input[type="file"]', function(e){
            var fileinput = $(this);
            fileinput.next('input[type="hidden"]').val(fileinput.val());
            var files = fileinput.get(0).files;
            if (files.length > 0) {
                element.loadFile(files[0]);
            }
        });
        this.place.on('change', '.imageelement-dialog input, .imageelement-dialog select, figcaption input', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value;
            var itype = input.attr('type');
            switch (itype) {
                case 'number':
                case 'range':
                    value = input.val() | 0;
                    break;
                case 'checkbox':
                    value = input.is(':checked');
                    break;
                default:
                    value = input.val();
                    break;
            }
            var key = input.attr('data-attribute');
            element.setData(key, value);
        });
        this.place.on('drop', function(e){
            e.preventDefault();
            e.stopPropagation();
            var dataurl = '';
            var $imgbox = element.place.closest('[data-boxtype="imagebox"]');
            var width = $imgbox.attr('data-imgwidth') || '';
            width = (width ? width+'px' : 'auto');
            var height = $imgbox.attr('data-imgheight') || '';
            height = (height ? height+'px' : 'auto');
            var imageplace = element.place.find('.imageelement-image');
            var event = e.originalEvent;
            var files = e.originalEvent.dataTransfer.files;
            if (files.length > 0) {
                element.loadFile(files[0]);
            }
        });
        this.place.on('openfile', function(event, data){
            event.stopPropagation();
            var dataurl = '';
            var $imgbox = element.place.closest('[data-boxtype="imagebox"]');
            var width = $imgbox.attr('data-imgwidth') || '';
            width = (width ? width+'px' : 'auto');
            var height = $imgbox.attr('data-imgheight') || '';
            height = (height ? height+'px' : 'auto');
            var imageplace = element.place.find('.imageelement-image');
            if (data.file && (data.file instanceof File)) {
                element.loadFile(data.file);
            };
        });
        //this.place.on('click', '.imageelement-editbar .imageelement-button[data-buttontype="remove"]', function(event){
        //    event.preventDefault();
        //    event.stopPropagation();
        //    element.place.trigger('removeelement', {type: 'imageelement'});
        //});
    }
    
    ImageElement.prototype.setData = function(key, value){
        this.data[key] = value;
        this.setAttrs();
        this.changed();
    }

    ImageElement.prototype.updateSize = function(){
        var $imgbox = this.place.closest('[data-boxtype="imagebox"]');
        var width = $imgbox.attr('data-imgwidth') || '';
        width = (width ? width+'px' : 'auto');
        var height = $imgbox.attr('data-imgheight') || '';
        height = (height ? height+'px' : 'auto');
        var imageplace = this.place.find('.imageelement-image img');
        imageplace.css({'max-width': width, 'max-height': height});
    };
    
    ImageElement.prototype.getData = function(){
        var result = {
            type: 'imageelement',
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return result;
    };
    
    ImageElement.prototype.getHtmltext = function(){
        var result = '<img src="'+this.data.content+'">';
        return result;
    };
    
    ImageElement.prototype.getIndexData = function(){
        if (!this.data.thumbnail || this.data.thumbnail.length < 20 || true) {
            var tmpimage = new Image();
            tmpimage.src = this.data.content;
            this.data.thumbnail = this.scaleImage(tmpimage, 50, this.data.type);
        };
        var result = {
            notes: this.data.notes,
            file: this.data.file,
            url: this.data.url,
            name: this.data.name,
            author: this.data.author,
            source: this.data.source,
            license: this.data.license,
            thumbnail: this.data.thumbnail
        };
        return result;
    };
    
    ImageElement.prototype.scaleImage = function(image, maxsize, mimetype) {
        var pngrex = /(png)/i;
        var gifrex = /(gif)/i;
        mimetype = (pngrex.test(mimetype) ? 'image/png' : (gifrex.test(mimetype) ? 'image/gif' : 'image/jpeg'));
        var imwidth = image.width;
        var imheight = image.height;
        var scale = imwidth / imheight;
        if (scale >= 1) {
            var newwidth = maxsize;
            var newheight = maxsize / scale;
        } else {
            var newheight = maxsize;
            var newwidth = scale * maxsize;
        };
        var canvas = document.createElement('canvas');
        canvas.width = newwidth;
        canvas.height = newheight;
        var ctx = canvas.getContext('2d');
        ctx.drawImage(image, 0, 0, newwidth, newheight);
        return canvas.toDataURL(mimetype);
    }
    
    ImageElement.prototype.getDialog = function(){
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        var eltype = this.data.boxtype || this.type;
        var dialogdata = ImageElement.dialogs;
        var html = [];
        if (dialogdata) {
            for (var i = 0, len = dialogdata.length; i < len; i++) {
                var data = dialogdata[i];
                var value = this.data[data.attribute] || data.defval;
                var text = data.label[uilang] || data.label['en'];
                switch (data.type){
                    case 'range':
                        html.push([
                            '<form>',
                            '<label title="'+text+'">'+(data.icon || text)+'</label>',
                            '<input type="range" min="' + data.min + '" max="' + data.max + '" value="'+(typeof(value) !== 'undefined' ? value : data.defval || 1)+'" step="'+(typeof(data.step) !== 'undefined' ? data.step : 1)+'" data-attribute="'+data.attribute+'" />',
                            '</form>'
                        ].join(''));
                        break;
                    case 'number':
                        html.push('<form><label title="'+text+'">'+(data.icon || text)+'</label><input type="number" min="' + data.min + '" max="' + data.max + '" value="'+(typeof(value) !== 'undefined' ? value : data.defval || 1)+'" step="1" data-attribute="'+data.attribute+'" placeholder="'+(data.placeholder || '')+'" /></form>');
                        break;
                    case 'select':
                        html.push('<form><label title="'+text+'">'+(data.icon || text)+'</label><select data-attribute="' + data.attribute + '">');
                        for (var ops = 0, opslen = data.options.length; ops < opslen; ops++) {
                            var option = data.options[ops];
                            html.push('<option value="'+option.value+'" ' + (value === option.value ? 'selected="selected"' : '') + '>'+(option.label[uilang] || option.label['en'])+'</option>');
                        }
                        html.push('</select></form>');
                        break;
                    case 'radio':
                        html.push('<form><span title="'+text+'">'+(data.icon || text)+'</span>');
                        var radioname = 'dialogradio-'+(Math.random() + '').replace(/^0./,'');
                        html.push('<div class="elementset-dialog-radio ffwidget-buttonset ffwidget-horizontal">');
                        for (var ops = 0, opslen = data.options.length; ops < opslen; ops++) {
                            var option = data.options[ops];
                            html.push('<label class="elementset-radiolabel"><input type="radio" name="' + radioname + '" value="'+option.value+'" ' + (value === option.value ? 'checked="checked"' : '') + ' data-attribute="'+data.attribute+'" /><span class="elementset-radiobutton ffwidget-label ffwidget-setbutton" title="'+(option.label[uilang] || option.label['en'])+'">'+(option.icon || option.label[uilang] || option.label['en'])+'</span></label>');
                        }
                        html.push('</div>', '</form>');
                        break;
                    case 'checkbox':
                        html.push('<form><label><span title="'+text+'">'+(data.icon || text)+'</span>');
                        var checkname = 'dialogcheckbox-'+(Math.random() + '').replace(/^0./,'');
                        html.push('<input type="checkbox" name="' + checkname + '" ' + (value === true ? 'checked="checked"' : '') + ' data-attribute="' + data.attribute + '" /></label></form>');
                    default:
                        break;
                }
            }
        }
        return html.join('');
    }

    ImageElement.dialogs = [
        {
            name: 'alignment',
            label: {
                en: 'Alignment',
                fi: 'Sijoittelu'
            },
            attribute: 'alignment',
            defval: 'center',
            type: 'radio',
            icon: ' ',
            options: [
                {
                    label: {
                        en: 'left',
                        fi: 'vasen',
                        sv: 'vänster'
                    },
                    value: 'left',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatleft"><path style="stroke: none;" d="M2 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="m2 3 l28 0 l0 2 l-28 0z m19 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m-19 6 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'center',
                        fi: 'keski',
                        sv: 'centrum'
                    },
                    value: 'center',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatcenter"><path style="stroke: none;" d="M9 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="M2 3 l28 0 l0 2 l-28 0z m0 24 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'right',
                        fi: 'oikea',
                        sv: 'höger'
                    },
                    value: 'right',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatright"><path style="stroke: none;" d="M17 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="M2 3 l28 0 l0 2 l-28 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l28 0 l0 2 l-28 0z" /></svg>'
                }
            ]
        },
        {
            name: 'size-x',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-widthpercent"><path style="stroke: none;" d="M1 16 l0 14 l2 0 l0 -7 l4 4 l0 -2.5 l16 0 l0 2.5 l4 -4 l0 7 l2 0 l0 -14 l-2 0 l0 7 l-4 -4 l0 2.5 l-16 0 l0 -2.5 l-4 4 l0 -7z M9 3 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 2 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z m8 -3 l2 0 l-6 14 l-2 0z m4 5 a4 4 0 0 1 0 8 a4 4 0 0 1 0 -8z m0 2 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z"></path></svg>',
            label: {
                en: 'Width',
                fi: 'Leveys',
                sv: 'Bredd'
            },
            attribute: 'size-x',
            type: 'select',
            defval: 'auto',
            options: [
                {
                    label: {
                        en: 'Orig.',
                        fi: 'Alkup.',
                        sv: 'Orig.'
                    },
                    value: 'auto'
                },
                {
                    label: {
                        en: '30%'
                    },
                    value: '30%'
                },
                {
                    label: {
                        en: '40%'
                    },
                    value: '40%'
                },
                {
                    label: {
                        en: '50%'
                    },
                    value: '50%'
                },
                {
                    label: {
                        en: '60%'
                    },
                    value: '60%'
                },
                {
                    label: {
                        en: '70%'
                    },
                    value: '70%'
                },
                {
                    label: {
                        en: '80%'
                    },
                    value: '80%'
                },
                {
                    label: {
                        en: '90%'
                    },
                    value: '90%'
                },
                {
                    label: {
                        en: '100%'
                    },
                    value: '100%'
                }
            ]
        }
    ]
    
    
    ImageElement.icons = {
        link: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z" /></svg>',
        info: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-info"><path style="stroke: none;" d="M15 4 a2.5 2.5 0 0 0 0 5 a2.5 2.5 0 0 0 0 -5z m3 7 l-7 1 l0 2 a3 2 0 0 1 3 2 l0 6 a3 2 0 0 1 -3 2 l0 2 10 0 l0 -2 a3 2 0 0 1 -3 -2z" /></svg>',
        close: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-cancel"><path style="stroke: none; fill: #a00;" d="M1 15 a14 14 0 0 0 28 0 a14 14 0 0 0 -28 0z" /><path style="fill: red; stroke: none;" d="M2 15 a13 13 0 0 1 26 0 a13 13 0 0 1 -26 0z"/><path style="fill: white; stroke: #a00;" d="M6 10 l4 -4 l5 5 l5 -5 l4 4 l-5 5 l5 5 l-4 4 l-5 -5 l-5 5 l-4 -4 l5 -5 z" /><path style=" stroke: none; fill: white; opacity: 0.4;" d="M2 15 a13 13 0 0 1 26 0 a13 13 0 0 0 -13 0 a13 13 0 0 1 -13 0z" /></svg>',
        file: "<svg xmlns='http://www.w3.org/2000/svg' version='1.1' width='30' height='30' viewBox='0 0 30 30' class='mini-icon mini-icon-folder'><path stroke='none' fill='black' d='M2 6 l5 0 l2 2 l14 0 l0 2 l-16 0 l-4 14 l5 -12 l22 0 l-5 13 l-23 0z'></path></svg>"
    }
    //ImageElement.removeicon = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z" /><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>';
    
    /******
     * Default data
     ******/
    ImageElement.defaults = {
        type: "imageelement",
        metadata: {
            "creator": "",
            "created": "",
            "modifier": "",
            "modified": "",
            "tags": []
        },
        data: {
            "notes": "",
            "type": "",
            "file": "",
            "url": "",
            "name": "",
            "author": "",
            "source": "",
            "license": "",
            "content": "",
            "alignment": "center" // "center" | "left" | "right"
        },
        settings: {
            mode: 'view',
            uilang: 'en'
        }
    }
    
    /******
     * Modes
     ******/
    ImageElement.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    /******
     * Stylesheet for images.
     ******/
    ImageElement.styles = [
        '[data-ebookelement-type="imageelement"] {text-align: left;}',
        '[data-ebookelement-type="imageelement"] img {max-width: 100%;}',
        'table.ebook-contenttable td [data-ebookelement-type="imageelement"] img {max-width: none!important;}',
        '[data-ebookelement-type="imageelement"][data-elementmode="edit"] {display: block; border: 1px solid #bbb; border-radius: 0;}',
        '.imageelement-container {position: relative; overflow: hidden; padding: 10px; text-align: center;}',
        
        // Popup
        '.imageelement-container .ebook-imagelink-popup {position: absolute; top: 0; bottom: 0; right: 100%; width: 100%; transition: right 0.5s; font-family: helvetica, Arial, sans-serif; font-size: 70%; padding: 0 0.5em 0 0.5em; box-sizing: border-box; -webkit-box-sizing: border-box; border: 2px solid black; overflow-y: auto; background-color: rgba(0, 0, 0, 0.7); color: white; border-radius: 4px; box-shadow: 0 0 2px 1px rgba(255,255,255,0.5); box-shadow: inset 0 0 2px 1px rgba(255,255,255,0.5);}',
        '.imageelement-container.ebook-popup-visible {}',
        '.imageelement-container.ebook-popup-visible .ebook-imagelink-popup {right: 0; text-align: left;}',
        '.imageelement-container .ebook-imagelink-popup .ebook-popupclose {position: absolute; top: 2px; right: 2px; cursor: pointer; display: none;}',
        '.imageelement-container .ebook-imagelink-popup td {padding: 0.2em 0.3em; line-height: 0.9em;}',
        '.imageelement-container .ebook-imagelink {display: block; position: absolute; top: 1em; right: 1em; background-color: rgba(255,255,255,0.8); border-radius: 3px; opacity: 0.2;}',
        '.imageelement-container:hover .ebook-imagelink {opacity: 1;}',
        'a.ebook-info-imagelink {color: #ddf;}',
        // Image
        '.imageelement[data-elementmode="edit"] .imageelement-image {background-color: #f0f0f0; min-height: 40px; padding: 5px; text-align: center; display: block; vertical-align: top; border: 1px solid transparent;}',
        'img.imageelement-scaled, svg.imageelement-scaled {width: 100%; height: auto;}',
        // Caption
        '.imageelement figcaption input {display: block; width: 100%; padding: 0.5em; box-sizing: border-box;}',
        '.imageelement figcaption {text-align: left; font-size: 90%; font-family: helvetica,sans-serif; line-height: 1.3em;}',
        // Metadata form
        '.imageelement-dataform .imageelement-datatable {display: none;}',
        '.imageelement-showinfo .imageelement-dataform .imageelement-datatable {display: table;}',
        '.imageelement-dataform {display: inline-block; vertical-align: top; position: absolute; top: 0.5em; right: 0.5em; background-color: transparent; color: black;}',
        '.imageelement-showinfo .imageelement-dataform {display: inline-block; vertical-align: top; position: absolute; top: 0.5em; right: 0.5em; background-color: rgba(0,0,0,0.6); color: white; border: 2px solid black; border-radius: 0.5em; box-shadow: inset 0 0 1px 1px rgba(255,255,255,0.4), 3px 3px 5px rgba(0,0,0,0.3); font-size: 80%; font-family: helvetica, sans-serif; padding: 0.5em; z-index: 3;}',
        '.imageelement-dataform input {background-color: white; color: black;}',
        '.imageelement-dataform input[type="text"] {border: 1px solid black; border-radius: 4px; padding: 0 0.5em;}',
        '.imageelement-dataform input[required]:invalid {background-color: #ff5;}',
        '.imageelement-dataform-title {text-align: right;}',
        '.imageelement-dataform-title .imageelement-dataform-icon {display: inline-block; width: 20px; height: 20px; padding: 2px; border-radius: 12px; margin-top: -0.3em; cursor: pointer;}',
        '.imageelement-dataform-title .imageelement-dataform-icon svg {fill: black; width: 20px; height: 20px;}',
        '.imageelement-showinfo .imageelement-dataform-title .imageelement-dataform-icon svg {fill: white;}',
        // Droparea
        '.imageelement-image .imageelement-droparea {min-width: 100px; min-height: 100px; padding: 0.5em; border: 1px dashed #666;}',
        '.imageelement[data-elementmode="edit"] .imageelement-image.imageelement-draghover {border: 1px dashed #666; background-color: #ddd;}',
        // File input
        '.imageelement-dialog {padding: 0 1em;}',
        '.imageelement-dialog svg {height: 20px; width: 20px;}',
        '.imageelement-fileinput {cursor: pointer;}',
        '.imageelement-fileinput input[type="file"] {display: none;}',
        '.imageelement-fileinput .imageelement-filelabel {display: inline-block; vertical-align: middle;}',
        '.imageelement-fileinput .imageelement-filelabel svg {height: 30px; width: 30px;}',
        '.imageelement-dialog form {display: inline-block; vertical-align: middle; margin: 0 1em;}',
        // Floating
        '.imageelement-wrapper[data-alignment="center"] .imageelement {margin: 0 auto; clear: both;}',
        '.imageelement-wrapper[data-alignment="left"] {float: left; clear: left;}',
        '.imageelement-wrapper[data-alignment="right"] {float: right; clear: right;}',
        '.imageelement-wrapper figure {margin: 0.5em; padding: 0.2em;}',
        '.imageelement-wrapper[data-alignment="left"] figure {margin-top: 0; margin-left: 0; padding-top: 0; padding-left: 0;}',
        '.imageelement-wrapper[data-alignment="right"] figure {margin-top: 0; margin-right: 0; padding-top: 0; padding-right: 0;}',
        ''
    ].join('\n');
    
    if ($('head style#imageelement-style').length === 0) {
        $('head').append('<style id="imageelement-style" type="text/css">'+ImageElement.styles+'</style>')
    }
    
    /******
     * Localization strings
     ******/
    ImageElement.localization = {
        "en": {
            "imageelement:author": "Author",
            "imageelement:license": "License",
            "imageelement:source": "Source",
            "imageelement:notes": "Notes",
            "imageelement:imageinfo": "Information",
            "imageelement:file": "File",
            "imageelement:drophere": "Drag'n'drop here an image file",
            "imageelement:remove": "Remove",
            "imageelement:caption": "Caption"
        },
        "fi": {
            "imageelement:author": "Tekijä",
            "imageelement:license": "Lisenssi",
            "imageelement:source": "Lähde",
            "imageelement:notes": "Huomioita",
            "imageelement:imageinfo": "Tietoa kuvasta",
            "imageelement:file": "Tiedosto",
            "imageelement:drophere": "Raahaa tähän kuvatiedosto",
            "imageelement:remove": "Poista",
            "imageelement:caption": "Kuvateksti"
        },
        "sv": {
            "imageelement:author": "Upphovsman",
            "imageelement:license": "Licens",
            "imageelement:source": "Källa",
            "imageelement:notes": "Anmärkningar",
            "imageelement:imageinfo": "Information",
            "imageelement:file": "Fil",
            "imageelement:drophere": "Dra och släpp hit en bildfil",
            "imageelement:remove": "Ta bort",
            "imageelement:caption": "Bildtext"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(ImageElement.localization);
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    ImageElement.elementinfo = {
        type: 'imageelement',
        elementtype: ['elements','studentelements'],
        jquery: 'imageelement',
        name: 'Image',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-image"><path style="stroke: none;" d="M3 6 h24 v18 h-24z m2 2 v14 h20 v-14z m1 12 l5 -5 l3 3 l6 -6 l4 4 v2 l-4 -4 l-6 6 l-3 -3 l-5 5z"></path></svg>',
        description: {
            en: 'Image',
            fi: 'Kuva',
            sv: 'Bild'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['general'],
        weight: 950
    }
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', ImageElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', ImageElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var imageelem = new ImageElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[imageelementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.imageelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in imageelement.');
            return false;
        }
    }
    
})(jQuery);