/******
@name Homeassignment
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Homeassignment
@description A class and jQuery-plugin for a homeassignment.
            Homeassignment has a list of goto-links to assignments,
            given and due-to dates, some instructions and
            color coding.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Dependency problem in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Homeassignment element to show and edit goto-links and information for homeassignment.
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var Homeassignment = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init Homeassignment element
     * @param {Object} options - initing data of the element
     ******/
    Homeassignment.prototype.init = function(options){
        options = $.extend(true, {}, Homeassignment.defaults, options);
        this.settings = options.settings;
        this.type = 'homeassignmentelement';
        this.setMode(options.settings.mode);
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator || this.settings.username,
            created: options.metadata.created || (new Date()).toString(),
            modifier: options.metadata.modifier || this.settings.username,
            modified: options.metadata.modified || (new Date()).toString(),
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.setAttrs();
    };
    
    /******
     * Set style-tag if needed.
     ******/
    Homeassignment.prototype.setStyles = function(){
        if ($('head style#homeassignmentelement-style').length === 0) {
            var styles = Homeassignment.styles;
            var colorstyles = [];
            for (var color in Homeassignment.colors) {
                colorstyles.push('.homeassignment-color-' + color + ' svg path {fill: ' + Homeassignment.colors[color] + ';}');
                //colorstyles.push('.homeassignment-color-input.homeassignment-color-' + color + ' {border-bottom: 3px solid ' + Homeassignment.colors[color] + ';}')
                //colorstyles.push('.homeassignment-color-input.homeassignment-color-' + color + ':not(.selected) svg path {fill: #aaa; stroke: ' + Homeassignment.colors[color] + ';}')
            };
            styles = styles + '\n' + colorstyles.join('\n');
            $('head').append('<style id="homeassignmentelement-style" type="text/css">'+styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    Homeassignment.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = Homeassignment.modes[mode] || Homeassignment.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Set some attributes
     ******/
    Homeassignment.prototype.setAttrs = function(){
        this.place.addClass('homeassignmentelement');
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    Homeassignment.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the Homeassignment element in different modes
     ******/
    Homeassignment.prototype.show = function(){
        this.place.attr('data-elementmode', escapeHTML(this.settings.mode));
        if (this.metadata.lang) {
            this.place.attr('lang', escapeHTML(this.metadata.lang));
            this.place.attr('uilang', escapeHTML(this.settings.uilang));
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        }
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    Homeassignment.prototype.view = function(){
        var uilang = this.settings.uilang;
        this.place.html(Homeassignment.templates.view);
        var iconplace = this.place.find('.homeassignment-icons');
        for (var color in Homeassignment.colors) {
            if (this.data.colors[color]) {
                iconplace.append([
                    '<div class="homeassignment-icon homeassignment-color-' + escapeHTML(color) + '">',
                    Homeassignment.icons.home,
                    '</div>'
                ].join(''));
            };
        };
        var levelplace = this.place.find('.homeassignment-level');
        var levelarr = [];
        levelarr[this.data.level + 1] = '';
        levelplace.append(levelarr.join(Homeassignment.icons.star));
        var titleplace = this.place.find('.homeassignment-title');
        titleplace.html(escapeHTML(this.data.title));
        var textplace = this.place.find('.homeassignment-text');
        var text = (typeof(marked) === 'function' ? marked(this.data.text) : escapeHTML(this.data.text));
        textplace.html(text);
        if ($.fn.mathquill) {
            textplace.find('.mathquill-embedded-latex:not(.mathquill-rendered-math)').mathquill();
        };
    };
    
    /******
     * Show the element in editable mode.
     ******/
    Homeassignment.prototype.edit = function(){
        var uilang = this.settings.uilang;
        this.place.html(Homeassignment.templates.edit);
        this.showControls();
        var iconplace = this.place.find('.homeassignment-icons');
        var selected;
        for (var color in Homeassignment.colors) {
            selected = (this.data.colors[color] ? ' selected' : '')
            iconplace.append([
                '<div class="homeassignment-icon homeassignment-color-input homeassignment-color-' + escapeHTML(color) + selected + '" data-color="'+escapeHTML(color)+'">',
                Homeassignment.icons.home,
                '</div>'
            ].join(''));
        };
        var levelplace = this.place.find('.homeassignment-level');
        var levelarr = [];
        levelarr[this.data.level + 1] = '';
        levelplace.append(levelarr.join(Homeassignment.icons.star));
        var titleplace = this.place.find('.homeassignment-title');
        titleplace.html('<input class="homeassignment-title-input" type="text" value="' + escapeHTML(this.data.title) + '" />');
        var textplace = this.place.find('.homeassignment-text');
        var text = escapeHTML(this.data.text);
        textplace.html('<textarea class="homeassignment-text-input">' + text + '</textarea>');
        var linkelemid = this.data.gotolink.data && this.data.gotolink.data.link.elementid;
        this.place.trigger('addmessageanchor', {anchor: linkelemid});
    };
    
    /******
     * Show controlbar for editing
     ******/
    Homeassignment.prototype.showControls = function(){
        console.log('haelement', this);
        var uilang = this.settings.uilang;
        var controlbar = this.place.find('.homeassignment-controlbar');
        controlbar.empty();
        var classes;
        var html = ['<ul class="ffwidget-buttonset ffwidget-horizontal homeassignment-colorselector">'];
        for (var color in Homeassignment.colors) {
            classes = [];
            classes.push(this.data.colors[color] ? 'buttonselected' : '');
            html.push('<li class="homeassignment-coloritem ffwidget-setbutton '+classes.join(' ')+'" data-color="'+escapeHTML(color)+'"><span class="homeassignment-colordot" style="background-color: '+Homeassignment.colors[color]+';">&nbsp;</span></li>');
        };
        html.push('</ul>');
        html.push('<div class="homeassignment-multiselector ffwidget-buttonset ffwidget-horizontal">');
        html.push('<button class="homeassignment-multiitem ffwidget-setbutton '+(this.data.multi ? '' : 'buttonselected')+'" data-multi="false">' + Homeassignment.icons.home + '</button>');
        html.push('<button class="homeassignment-multiitem ffwidget-setbutton '+(this.data.multi ?  'buttonselected' : '')+'" data-multi="true">' + Homeassignment.icons.multihome + '</button>');
        controlbar.append(html.join(''));
        html.push('</div>');
    };
    
    /******
     * Remove all event handlers
     ******/
    Homeassignment.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    Homeassignment.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in Homeassignment.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        }).on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        }).on('destroy', function(event, data) {
            event.stopPropagation();
            var linkelemid = element.data.gotolink.data && element.data.gotolink.data.link.elementid;
            element.place.trigger('removemessageanchor', {anchor: linkelemid});
        });
        
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    Homeassignment.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        }).on('click', function(event, data){
            event.stopPropagation();
            element.triggerLink();
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    Homeassignment.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('click', '.homeassignment-color-input', function(event, data) {
            event.stopPropagation();
            //var input = $(this);
            //input.toggleClass('selected');
            //var color = input.attr('data-color');
            //var status = input.is('.selected');
            //element.setColor(color, status);
        }).on('click', '.homeassignment-coloritem', function(event, data) {
            event.stopPropagation();
            var button = $(this);
            var selected = button.parent().find('.buttonselected');
            var color = button.attr('data-color');
            var oldcolor;
            var status = button.is('.buttonselected');
            if (element.data.multi) {
                if (!(selected.length === 1 && button.is('.buttonselected'))) {
                    // If this wasn't the last selected button.
                    button.toggleClass('buttonselected');
                    element.setColor(color, !status);
                };
            } else {
                selected.removeClass('buttonselected');
                oldcolor = selected.attr('data-color');
                element.setColor(oldcolor, false);
                button.addClass('buttonselected');
                element.setColor(color, true);
            };
        }).on('click', '.homeassignment-multiitem', function(event, data) {
            event.stopPropagation();
            var button = $(this);
            var status = (button.attr('data-multi') === 'true');
            var allbuttons = button.parent().find('button');
            allbuttons.removeClass('buttonselected');
            button.addClass('buttonselected');
            element.setMulti(status);
        }).on('change', '.homeassignment-title-input', function(event, data) {
            event.stopPropagation();
            var text = $(this).val();
            element.setTitle(text);
        }).on('change', '.homeassignment-text-input', function(event, data) {
            event.stopPropagation();
            var text = $(this).val();
            element.setText(text);
        });
    };
    
    /******
     * Update the metadata of the element.
     ******/
    Homeassignment.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Trigger a gotolink-event with data of the gotolink.
     ******/
    Homeassignment.prototype.triggerLink = function(){
        var linkdata = JSON.parse(JSON.stringify(this.data.gotolink));
        if (linkdata.type === 'contentlink') {
            this.place.trigger('gotolink', linkdata);
        };
    };
    
    /**
     * Set the color on or off
     * @param {String} color - the color to set
     * @param {Boolean} status - set the color on (true) or off (false)
     */
    Homeassignment.prototype.setColor = function(color, status) {
        if (Homeassignment.colors[color]) {
            this.data.colors[color] = status;
            if (status) {
                this.place.find('.homeassignment-color-' + color).addClass('selected');
                this.latestColor = color;
            } else {
                this.place.find('.homeassignment-color-' + color).removeClass('selected');
            };
        };
        this.changed();
    };
    
    /**
     * Change the multi color status
     * @param {Boolean} status - The status of selecting many colors. true === "multi", false === "single"
     */
    Homeassignment.prototype.setMulti = function(status) {
        this.data.multi = status;
        var cselector, color, oldcolor, allselected;
        if (!status) {
            color = this.latestColor || 'black';
            cselector = this.place.find('.homeassignment-colorselector');
            allselected = cselector.children('.buttonselected');
            for (var i = 0, len = allselected.length; i < len; i++) {
                oldcolor = allselected.eq(i).attr('data-color');
                allselected.removeClass('buttonselected');
                this.setColor(oldcolor, false);
            }
            cselector.children('[data-color="'+escapeHTML(color)+'"]').addClass('buttonselected');
            this.setColor(color, true);
        };
        this.changed();
    };
    
    /**
     * Set the title
     * @param {String} title - the title text
     */
    Homeassignment.prototype.setTitle = function(title) {
        this.data.title = title;
        this.changed();
    }
    
    /**
     * Set the text
     * @param {String} text - the title text
     */
    Homeassignment.prototype.setText = function(text) {
        this.data.text = text;
        this.changed();
    }
    
    /******
     * Trigger element_changed event.
     ******/
    Homeassignment.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'homeassignmentelement'});
    }

    
    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "linklistelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {}
     *   }
     ******/
    Homeassignment.prototype.getData = function(){
        var result = {
            type: this.type,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return result;
    }
    
    /******
     * Get an id for a new element.
     * @param {String} eltype - Type of an element.
     * @returns {String} An id for the new element.
     ******/
    Homeassignment.prototype.getNewId = function(){
        var idparts = ['contentlink'];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };

    /******
     * Default settings
     ******/
    Homeassignment.defaults = {
        type: 'homeassignmentelement',
        metadata: {
            creator: '',
            created: 0,
            modifier: '',
            modified: 0,
            tags: []
        },
        data: {
            title: '',
            text: '',
            colors: {
                black: true
            },
            level: 0,
            gotolink: {type: 'contentlink'}
        },
        settings: {
            mode: 'view',
            uilang: 'en',
            lang: 'en'
        }
    }
    
    Homeassignment.colors = {
        black: '#2d2d2d',
        red: '#ff4141',
        orange: '#ff9900',
        //yellow: '#ffff3e',
        yellow: 'gold',
        green: '#009100',
        blue: '#0084c8',
        violet: '#ba00ff',
        brown: '#804d00'
    }
    
    /******
     * Modes
     ******/
    Homeassignment.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    
    
    /******
     * Templates
     ******/
    Homeassignment.templates = {
        view: [
            '<div class="homeassignment-metadata">',
            '    <div class="homeassignment-icons"></div>',
            '    <div class="homeassignment-level"></div>',
            '</div>',
            '<div class="homeassignment-title"></div>',
            '<div class="homeassignment-text"></div>',
        ].join('\n'),
        edit: [
            '<div class="homeassignment-controlbar ffwidget-background"></div>',
            '<div class="homeassignment-content">',
            '    <div class="homeassignment-metadata">',
            '        <div class="homeassignment-icons"></div>',
            '        <div class="homeassignment-level"></div>',
            '    </div>',
            '    <div class="homeassignment-title"></div>',
            '    <div class="homeassignment-text"></div>',
            '</div>'
        ].join('\n')
    }
    
    /******
     * Icons
     ******/
    Homeassignment.icons = {
        home: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path stroke="none" d="M4 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        multihome: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="60" height="30" viewBox="0 0 90 30" class="mini-icon mini-icon-multihome"><path stroke="none" d="M4 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path><path stroke="none" d="M34 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path><path stroke="none" d="M64 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        star: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-star"><path d="M15 2 l3.7 8 l9.3 1 l-7 6 l2 9 l-8 -4.8 l-8 4.8 l2 -9 l-7 -6 l9.3 -1z"></path></svg>',
        link: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z"></path></svg>',
        list: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-list"><path style="stroke: none;" d="M5 4 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m7.5 -18 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        grid: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-grid"><path style="stroke: none;" d="M3 3 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z"></path></svg>',
        trashcan: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z"></path><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>',
        move: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-move pssicon-drag"><path style="stroke: none;" d="M15 2 l5 5 l-4 0 l0 7 l7 0 l0 -4 l5 5 l-5 5 l0 -4 l-7 0 l0 7 l4 0 l-5 5 l-5 -5 l4 0 l0 -7 l-7 0 l0 4 l-5 -5 l5 -5 l0 4 l7 0 l0 -7 l-4 0z"></path></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    Homeassignment.elementinfo = {
        type: 'homeassignmentelement',
        elementtype: ['elements'],
        jquery: 'homeassignmentelement',
        name: 'Homeassignment',
        icon: Homeassignment.icons.home,
        description: {
            en: 'Homeassignments',
            fi: 'Kotitehtävät',
            sv: 'Hemuppgifter'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['message']
    }
    
    /******
     * Styles
     ******/
    Homeassignment.styles = [
        '.homeassignmentelement[data-elementmode="view"], .homeassignmentelement[data-elementmode="review"] {padding: 0.5em; border-radius: 0.5em; border: 1px solid #999; cursor: pointer; background-color: #f8f8f8;}',
        '.homeassignment-metadata {display: flex; flex-direction: row; flex-wrap: nowrap; justify-content: space-between; align-items: center;}',
        '.homeassignment-icon {display: inline-block;}',
        '.homeassignment-metadata svg {width: 25px; height: 25px;}',
        '.homeassignment-title {font-weight: bold; font-size: 100%; border-bottom: 1px solid black;}',
        '.homeassignment-title:hover {color: blue; text-decoration: underline;}',
        '.homeassignment-text {font-size: 90%;}',
        '.homeassignment-level svg path {fill: gold; stroke: black; stroke-width: 1;}',
        // Editing
        // controlbar
        '.homeassignment-controlbar {min-height: 20px; padding: 3px; display: flex; flex-flow: row nowrap;}',
        '.homeassignment-colorselector {margin: 0 1em; padding: 0;}',
        '.homeassignment-coloritem {padding: 0; line-height: 100%;}',
        '.homeassignment-colordot {display: inline-block; width: 10px; height: 10px; border-radius: 50%; vertical-align: middle; margin: 3px;}',
        '.homeassignment-multiselector svg {height: 15px; width: auto; margin: 1px;}',
        '.homeassignment-controlbar .ffwidget-buttonset {margin: 0 1em 0 1em;}',
        '.homeassignment-controlbar .ffwidget-setbutton {padding: 0; line-height: 100%;}',
        // content
        '.homeassignment-content {padding: 0.5em;}',
        //'.homeassignment-color-input {cursor: pointer; border-bottom: 2px solid transparent;}',
        //'.homeassignment-color-input:not(.selected) svg path {fill: #aaa; stroke-width: 1;}',
        '.homeassignment-color-input:not(.selected) {display: none;}',
        '.homeassignment-title-input {box-sizing: border-box; width: 100%; padding: 0.4em 0.5em; border: 1px solid #ccc; font-weight: bold;}',
        '.homeassignment-text-input {box-sizing: border-box; width: 100%; min-height: 3em;}',
        
        // Markings on assignments
        '.homeassignmentelement-marking {display: inline-block;}',
        '.homeassignmentelement-marking-info-text {font-weight: bold;}',
        ''
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    Homeassignment.localization = {
        "en": {
            //"linklistelement:title": "Link title",
            //"linklistelement:description": "Longer description of the link"
        },
        "fi": {
            //"linklistelement:title": "Linkin otsikko",
            //"linklistelement:description": "Linkin pidempi kuvaus"
        },
        "sv": {
            //"linklistelement:title": "Länktitel",
            //"linklistelement:description": "Längre beskrivning"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Homeassignment.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Homeassignment.localization);
    }
    
    /******
     * Set some needed styles
     ******/
    (function(){
        if ($('head style#homeassignmentelement-style').length === 0) {
            var styles = Homeassignment.styles;
            var colorstyles = [];
            for (var color in Homeassignment.colors) {
                colorstyles.push('.homeassignment-color-' + color + ' svg.mini-icon path {fill: ' + Homeassignment.colors[color] + '!important;}');
                //colorstyles.push('.homeassignment-color-input.homeassignment-color-' + color + ' {border-bottom: 3px solid ' + Homeassignment.colors[color] + ';}')
                //colorstyles.push('.homeassignment-color-input.homeassignment-color-' + color + ':not(.selected) svg path {fill: #aaa; stroke: ' + Homeassignment.colors[color] + ';}')
            };
            styles = styles + '\n' + colorstyles.join('\n');
            $('head').append('<style id="homeassignmentelement-style" type="text/css">'+styles+'</style>')
        }
    })()

    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Homeassignment.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', Homeassignment.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var element = new Homeassignment(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.homeassignmentelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in homeassignmentelement.');
            return false;
        }
    }

})(jQuery);