/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}


(function($){
    
    /******
     * CoverPage-element to show coverimage and title
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var CoverPage = function(place, options){
        this.place = $(place);
        this.place.addClass('coverpageelement');
        this.setStyles();
        this.init(options);
        this.initHandlers();
        this.show();
    }
    
    /******
     * Init coverpage element
     * @param {Object} options - initing data of the element
     ******/
    CoverPage.prototype.init = function(options){
        options = $.extend({}, CoverPage.defaults, options);
        this.settings = $.extend(true, {}, CoverPage.defaults.settings, options.settings);
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.datafetched = false;
    }
    
    /******
     * Set style-tag if needed.
     ******/
    CoverPage.prototype.setStyles = function(){
        if ($('head style#coverpageelement-style').length === 0) {
            $('head').append('<style id="coverpageelement-style" type="text/css">'+CoverPage.styles+'</style>')
        }
    }
    
    /******
     * Show the CoverPage element 
     ******/
    CoverPage.prototype.show = function(){
        if (this.datafetched) {
            this.place.html(CoverPage.templates.html);
            var shorttitle = this.data.shorttitle;
            var booknumber = shorttitle.replace(/[^0-9]/g, '') | 0;
            var shorttitlefill = this.data.shorttitlefill;
            var title = this.data.title;
            var fronttitle = this.data.fronttitle || title || '';
            //var isnumber = (parseInt(shorttitle, 10).toString() === shorttitle.toString());
            //var shortfont = (isnumber ? 'Angelina' : 'Waukegan-ldo');
            //var shortsize = (isnumber ? 100 : 50);
            //var svg = [
            //    '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="150" height="135" viewBox="0 0 200 180" class="bookshortname" style="display: block;">',
            //    '<ellipse fill="'+shorttitlefill+'" cx="100" cy="90" rx="100" ry="90"></ellipse>',
            //    '<text fill="white" font-family="'+shortfont+'" font-size="'+shortsize+'" x="100" y="90" style="text-anchor: middle; dominant-baseline: middle;">'+shorttitle+'</text>',
            //    '</svg>'
            //].join('');
            //this.place.children('.coverpageelement-shorttitle').html(svg);
            this.place.children('.coverpageelement-fronttitle').text(fronttitle);
        } else {
            this.place.trigger('gettitleinfo');
        }
    }
    
    /******
     * Remove all event handlers
     ******/
    CoverPage.prototype.removeHandlers = function(){
    }
    
    /******
     * Init handlers.
     ******/
    CoverPage.prototype.initHandlers = function(){
        var element = this;
        this.place.off('reply_gettitleinfo').on('reply_gettitleinfo', function(event, data) {
            event.stopPropagation();
            element.data = data;
            element.datafetched = true;
            element.show();
        });
        this.place.on('getdata', function(event) {
            event.stopPropagation();
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    };
    
    /**
     * Get data
     */
    CoverPage.prototype.getData = function() {
        return {type: 'coverpageelement', metadata: {}};
    };
    
    /******
     * Default settings
     ******/
    CoverPage.defaults = {
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            shorttitle: '',
            shorttitlefill: '#66cc33',
            title: ''
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en'
        }
    }
    
    /******
     * Templates
     ******/
    CoverPage.templates = {
        html: [
            '<div class="coverpageelement-shorttitle"></div>',
            '<div class="coverpageelement-fronttitle"></div>'
        ].join('\n')
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    CoverPage.elementinfo = {
        type: 'coverpageelement',
        elementtype: 'elements',
        jquery: 'coverpageelement',
        name: 'CoverPage',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-markdown mini-icon-commonmark"><path style="stroke: none;" d="M3 6 l24 0 a3 3 0 0 1 3 3 l0 12 a3 3 0 0 1 -3 3 l-24 0 a3 3 0 0 1 -3 -3 l0 -12 a3 3 0 0 1 3 -3z m0 2 a1 1 0 0 0 -1 1 l0 12 a1 1 0 0 0 1 1 l24 0 a1 1 0 0 0 1 -1 l0 -12 a1 1 0 0 0 -1 -1z m2 3 l2.5 0 l2.5 3 l2.5 -3 l2.5 0 l0 8 l-2.5 0 l0 -5 l-2.5 3 l-2.5 -3 l0 5 l-2.5 0z m15 0 l2.5 0 l0 4 l2.5 0 l-3.75 4 l-3.75 -4 l2.5 0z" /></svg>',
        description: {
            en: 'Show the cover image and title',
            fi: 'Näytetään kirjan etusivun kuva ja otsikko.'
        },
        classes: ['viewonly']
    }
    
    /******
     * Styles
     ******/
    CoverPage.styles = [
        '@font-face{ font-family: Angelina;  src: url(data:font/ttf;base64,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) format("truetype");}',
        '@font-face{ font-family: Waukegan-ldo; src: url(data:font/ttf;base64,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) format("truetype");}',
        '[data-ebookelement-type="coverpageelement"].coverpageelement {margin-left: auto; margin-right: auto; margin-bottom: 2.5em; width: 480px; width: auto;}',
        '.coverpageelement-shorttitle {margin: -100px 0 -40px; margin: 0; display: none;}',
        '.coverpageelement-fronttitle {margin-top: 0.3em; font-size: 250%; text-transform: uppercase; font-family: Waukegan-ldo; text-align: center; text-shadow: 0 0 1px white, 0 0 2px white;}',
        '[data-pagetypes~="front"] .imageelement img {max-width: 460px!important;}'
    ].join('\n');    
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', CoverPage.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', CoverPage.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var coverpage = new CoverPage(this, params);
            });
        }
    }
    
    $.fn.coverpageelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in coverpageelement.');
            return false;
        }
    }
    
})(jQuery);

(function($){
    
    /******
     * Authors-element to show names of authors.
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var AuthorsElement = function(place, options){
        this.place = $(place);
        this.place.addClass('authorselement');
        this.setStyles();
        this.init(options);
        this.initHandlers();
        this.show();
    }
    
    /******
     * Init authors element
     * @param {Object} options - initing data of the element
     ******/
    AuthorsElement.prototype.init = function(options){
        options = $.extend({}, AuthorsElement.defaults, options);
        this.settings = $.extend(true, {}, AuthorsElement.defaults.settings, options.settings);
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.datafetched = false;
    }
    
    /******
     * Set style-tag if needed.
     ******/
    AuthorsElement.prototype.setStyles = function(){
        if ($('head style#authorselement-style').length === 0) {
            $('head').append('<style id="authorselement-style" type="text/css">'+AuthorsElement.styles+'</style>')
        }
    }
    
    /******
     * Show the Authors element 
     ******/
    AuthorsElement.prototype.show = function(){
        if (this.datafetched) {
            this.place.html(AuthorsElement.templates.html);
            var secondaryset, secondaryauthor;
            var authorselem = this.place.children('.authorselement-authors');
            var secondaryelem = this.place.children('.authorselement-secondaryauthors');
            var authors = ['<ul class="authorselement-authorlist">'];
            for (var i = 0, len = this.data.author.length; i < len; i++) {
                authors.push('<li><h3>'+this.data.author[i]+'</h3></li>');
            }
            authors.push('</ul>');
            var secondary = ['<ul class="authorselement-secondarysetlist">'];
            for (var i = 0, len = this.data.secondaryauthor.length; i < len; i++){
                secondaryset = this.data.secondaryauthor[i];
                secondary.push('<li class="authorselement-secondaryset">');
                secondary.push('<span class="authorselement-secondarygroupname">'+secondaryset.name+'</span>');
                secondary.push('<div class="authorselement-secondarygrouplist ffwidget-background"><ul>');
                for (var j = 0, seclen = secondaryset.author.length; j < seclen; j++) {
                    secondaryauthor = secondaryset.author[j];
                    secondary.push('<li authorselement-secondaryauthorname">'+secondaryauthor+'</li>')
                }
                secondary.push('</ul>');
                secondary.push('</li>');
            }
            secondary.push('</ul></div>');
            authorselem.html(authors.join('\n'));
            secondaryelem.html(secondary.join('\n'));
        } else {
            this.place.trigger('getauthors')
        }
    }
    
    /******
     * Remove all event handlers
     ******/
    AuthorsElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers.
     ******/
    AuthorsElement.prototype.initHandlers = function(){
        var element = this;
        this.removeHandlers();
        this.place.on('click.authors', 'li.authorselement-secondaryset', function(event){
            event.stopPropagation();
            event.preventDefault();
            var item = $(this);
            if (item.hasClass('isopen')) {
                item.removeClass('isopen');
            } else {
                item.trigger('closeauthors');
                item.addClass('isopen');
            };
        });
        this.place.on('click.authors, closeauthors', function(event){
            event.stopPropagation();
            event.preventDefault();
            $(this).find('li.authorselement-secondaryset.isopen').removeClass('isopen');
        });
        this.place.off('reply_getauthors').on('reply_getauthors', function(event, data) {
            event.stopPropagation();
            element.data = data;
            element.datafetched = true;
            element.show();
        });
        this.place.on('getdata', function(event) {
            event.stopPropagation();
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    };
    
    /**
     * Get data
     */
    AuthorsElement.prototype.getData = function() {
        return {type: 'authorselement', metadata: {}};
    };
    
    /******
     * Default settings
     ******/
    AuthorsElement.defaults = {
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        //data: {
        //    author: [],
        //    secondaryauthor: [],
        //    vendorURL: '',
        //    vendorLogo: ''
        //},
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en'
        }
    }
    
    /******
     * Templates
     ******/
    AuthorsElement.templates = {
        html: [
            '<div class="authorselement-authors"></div>',
            '<div class="authorselement-secondaryauthors"></div>'
        ].join('\n')
    }
    
    /******
     * Icons
     ******/
    AuthorsElement.icons = {
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z" /><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    AuthorsElement.elementinfo = {
        type: 'authorselement',
        elementtype: 'elements',
        jquery: 'authorselement',
        name: 'Authors',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-markdown mini-icon-commonmark"><path style="stroke: none;" d="M3 6 l24 0 a3 3 0 0 1 3 3 l0 12 a3 3 0 0 1 -3 3 l-24 0 a3 3 0 0 1 -3 -3 l0 -12 a3 3 0 0 1 3 -3z m0 2 a1 1 0 0 0 -1 1 l0 12 a1 1 0 0 0 1 1 l24 0 a1 1 0 0 0 1 -1 l0 -12 a1 1 0 0 0 -1 -1z m2 3 l2.5 0 l2.5 3 l2.5 -3 l2.5 0 l0 8 l-2.5 0 l0 -5 l-2.5 3 l-2.5 -3 l0 5 l-2.5 0z m15 0 l2.5 0 l0 4 l2.5 0 l-3.75 4 l-3.75 -4 l2.5 0z" /></svg>',
        description: {
            en: 'Show the lists of names of authors',
            fi: 'Näytetään tekijöiden nimien listat.'
        },
        classes: ['viewonly']
    }
    
    /******
     * Styles
     ******/
    AuthorsElement.styles = [
        '.authorselement-authors {text-align: center;}',
        '.authorselement-secondaryauthors {text-align: center;}',
        '.authorselement-authorlist {list-style: none; margin: 0 auto; padding: 0; font-size: 120%;}',
        '.authorselement-authorlist li {display: inline-block; margin-right: 1.5em;}',
        '.authorselement-authorlist li h3 {margin: 0.5em 0;}',
        '.authorselement-secondarysetlist {list-style: none;}',
        '.authorselement-secondarysetlist > li:before {display: inline-block; content: "\\25B6";}',
        'li.authorselement-secondaryset {position: relative; display: inline-block;}',
        'li.authorselement-secondaryset span.authorselement-secondarygroupname {font-weight: bold; font-size: 105%; padding-right: 1em; cursor: pointer;}',
        'li.authorselement-secondaryset > .authorselement-secondarygrouplist {display: none; position: absolute; bottom: 1.2em; padding: 1em 0; border-radius: 1em; border: 2px solid black; box-shadow: 8px 8px 8px rgba(0,0,0,0.5); font-size: 85%; line-height: 1.3em; z-index: 2; text-align: left; font-family: Helvetica, Arial, sans-serif;}',
        //'li.authorselement-secondaryset > .authorselement-secondarygrouplist {display: none; position: absolute; bottom: 1.2em; background-color: rgba(0,0,0,0.8); padding: 1em 0; border-radius: 1em; border: 2px solid #0195de; box-shadow: 8px 8px 8px rgba(0,0,0,0.5); color: white; font-size: 85%; line-height: 1.3em; z-index: 2; text-align: left; font-family: Helvetica, Arial, sans-serif;}',
        'li.authorselement-secondaryset > .authorselement-secondarygrouplist > ul {padding: 0em 2em;  margin: 0; list-style: outside none square; overflow-y: auto; max-height: 30em; min-width: 12em;}',
        'li.authorselement-secondaryset > .authorselement-secondarygrouplist > ul::-webkit-scrollbar {width: 4px; background: rgba(255,255,255,0.5); border-radius: 2px;}',
        'li.authorselement-secondaryset > .authorselement-secondarygrouplist > ul::-webkit-scrollbar-thumb {width: 4px; background: rgba(0,0,0,0.5); border-radius: 2px;}',
        'li.authorselement-secondaryset.isopen > .authorselement-secondarygrouplist {display: block;}'
    ].join('\n');
    
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', AuthorsElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', AuthorsElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var authorselem = new AuthorsElement(this, params);
            });
        }
    }
    
    $.fn.authorselement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in authorselement.');
            return false;
        }
    }
    
})(jQuery);

(function($){

    /**
     * Helper functions
     */
    
    /**
     * Sanitize the input text
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /**
     * Html-escaping for sanitizing
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /**
     * Copyright-element to show the copyright information of the notebook.
     * @constructor
     * @param {jQuery} place    place of the element
     * @param {Object} options  data for the element
     */
    var CopyrightElement = function(place, options) {
        this.place = $(place);
        this.place.addClass('copyrightelement');
        this.setStyles();
        this.init(options);
        this.initHandlers();
        this.show();
    };
    
    /**
     * Init copyright element
     * @param {Object} options    data for the element
     */
    CopyrightElement.prototype.init = function(options) {
        options = $.extend(true, {}, this.defaults, options);
        this.settings = options.settings;
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.datafetched = false;
    };

    /**
     * Init event handlers
     */
    CopyrightElement.prototype.initHandlers = function(options) {
        var element = this;
        this.removeHandlers();
        this.place.off('reply_getcopyrightinfo').on('reply_getcopyrightinfo', function(event, data) {
            event.stopPropagation();
            element.data = $.extend({}, this.defaults, data);
            element.datafetched = true;
            element.show();
        });
        this.place.on('getdata', function(event) {
            event.stopPropagation();
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    };
    
    /**
     * Remove all event handlers
     */
    CopyrightElement.prototype.removeHandlers = function(){
        this.place.off();
    };

    /**
     * Show the element
     */
    CopyrightElement.prototype.show = function(options) {
        var lang = this.settings.lang;
        if (this.datafetched) {
            this.place.html(this.templates.html);
            if (this.data.ownerlogo && this.data.ownerlogodata) {
                this.place.find('.copyrightelement-ownerlogo').html(this.data.ownerurl ? '<a href="'+escapeHTML(this.data.ownerurl)+'" target="_blank">' + this.getOwnerLogo() + '</a>' : this.getOwnerLogo());
                this.place.find('.copyrightelement-ownerwrapper').addClass('haslogo');
            };
            this.place.find('.copyrightelement-owner').html(this.data.ownerurl ? '<a href="'+escapeHTML(this.data.ownerurl)+'" target="_blank">' + escapeHTML(this.data.owner) + '</a>' : escapeHTML(this.data.owner));
            this.place.find('.copyrightelement-license').html(this.getLicensetext());
            this.place.find('.copyrightelement-licensing').attr('data-license', escapeHTML(this.data.license));
            this.place.find('.copyrightelement-licenseicons').html(this.getLicenseicons());
        } else {
            this.place.html('');
            this.place.trigger('getcopyrightinfo');
        };
    };

    /**
     * Get data
     */
    CopyrightElement.prototype.getData = function() {
        return {type: 'copyrightelement', metadata: {}};
    };
    
    /**
     * Get localized license text
     * @returns {String} license text in localized language
     */
    CopyrightElement.prototype.getLicensetext = function() {
        var lang = this.settings.lang;
        var code = this.data.license;
        var text = (this.licenses[lang] && this.licenses[lang][code]) ||
            (this.licenses['en'] && this.licenses['en'][code]) || code;
        var url = (this.licenseurls[lang] && this.licenseurls[lang][code]) ||
            (this.licenseurls['en'] && this.licenseurls['en'][code]) || this.data.licenseurl || '';
        var result = (url ? '<a href="'+escapeHTML(url)+'" target="_blank">' + text + '</a>' : text);
        return result;
    };
    
    /**
     * Get the copyright icons
     * @return {String} a string of html with icons for copyright
     */
    CopyrightElement.prototype.getLicenseicons = function() {
        var html = [];
        var code = this.data.license;
        var symbols = this.licensesymbols[code] || [];
        for (var i = 0, len = symbols.length; i < len; i++) {
            html.push(this.licenseicons[symbols[i]]);
        };
        return html.join('\n');
    };
    
    /**
     * Get the owner logo as html
     * @return {String} a html img-tag with the owner logo
     */
    CopyrightElement.prototype.getOwnerLogo = function() {
        var result = '';
        if (this.data.ownerlogo && this.data.ownerlogodata) {
            result = '<img src="'+escapeHTML(this.data.ownerlogodata)+'" title="'+escapeHTML(this.data.owner)+'" alt="'+escapeHTML(this.data.owner)+'">';
        };
        return result;
    };
    
    /**
     * Set the css-styles
     */
    CopyrightElement.prototype.setStyles = function(options) {
        if ($('head style#copyrightelement-style').length === 0) {
            $('head').append('<style id="copyrightelement-style" type="text/css">' + this.styles + '</style>');
        };
    };
    
    /**
     * Default values for copyright element
     */
    CopyrightElement.prototype.defaults = {
        type: 'copyrightelement',
        data: {
            owner: '',
            ownerlogo: '',
            ownerurl: '',
            license: '',
            licenseurl: ''
        },
        metadata: {},
        settings: {}
    };
    
    /**
     * License mapping
     */
    CopyrightElement.prototype.licenses = {
        "en": {
            "copyright": "All rights reserved",
            "pd": "Public Domain (PD)",
            "cc0": "Creative Commons Zero (CC0)",
            "cc-by": "Creative Commons Attribution (CC BY)",
            "cc-by-sa": "Creative Commons Nimeä-ShareAlike (CC BY-SA)",
            "cc-by-nc": "Creative Commons Attribution-NonCommercial (CC BY-NC)",
            "cc-by-nd": "Creative Commons Attribution-NoDerivatives (CC BY-ND)",
            "cc-by-nc-sa": "Creative Commons Attribution-NonCommercial-ShareAlike (CC BY-NC-SA)",
            "cc-by-nc-nd": "Creative Commons Attribution-NonCommercial-NoDerivatives (CC BY-NC-ND)"
        },
        "fi": {
            "copyright": "Kaikki oikeudet pidätetään",
            "pd": "Public Domain (PD)",
            "cc0": "Creative Commons Zero (CC0)",
            "cc-by": "Creative Commons Nimeä (CC BY)",
            "cc-by-sa": "Creative Commons Attribution-JaaSamoin (CC BY-SA)",
            "cc-by-nc": "Creative Commons Nimeä-EiKaupallinen (CC BY-NC)",
            "cc-by-nd": "Creative Commons Nimeä-EiMuutoksia (CC BY-ND)",
            "cc-by-nc-sa": "Creative Commons Nimeä-EiKaupallinen-JaaSamoin (CC BY-NC-SA)",
            "cc-by-nc-nd": "Creative Commons Nimeä-EiKaupallinen-EiMuutoksia (CC BY-NC-ND)"
        },
        "sv": {
            "copyright": "Alla rättigheter förbehållna",
            "pd": "Public Domain (PD)",
            "cc0": "Creative Commons Zero (CC0)",
            "cc-by": "Creative Commons Erkännande (CC BY)",
            "cc-by-sa": "Creative Commons Erkännande-DelaLika (CC BY-SA)",
            "cc-by-nc": "Creative Commons Erkännande-IckeKommersiell (CC BY-NC)",
            "cc-by-nd": "Creative Commons Erkännande-IngaBearbetningar (CC BY-ND)",
            "cc-by-nc-sa": "Creative Commons Erkännande-IckeKommersiell-DelaLika (CC BY-NC-SA)",
            "cc-by-nc-nd": "Creative Commons Erkännande-IckeKommersiell-IngaBearbetningar (CC BY-NC-ND)"
        }
    };
    
    CopyrightElement.prototype.licenseurls = {
        "en": {
            "cc0": "https://creativecommons.org/publicdomain/zero/1.0/",
            "cc-by": "https://creativecommons.org/licenses/by/4.0/",
            "cc-by-sa": "https://creativecommons.org/licenses/by-sa/4.0/",
            "cc-by-nc": "https://creativecommons.org/licenses/by-nc/4.0/",
            "cc-by-nd": "https://creativecommons.org/licenses/by-nd/4.0/",
            "cc-by-nc-sa": "https://creativecommons.org/licenses/by-nc-sa/4.0/",
            "cc-by-nc-nd": "https://creativecommons.org/licenses/by-nc-nd/4.0/"
        },
        "fi": {
            "cc0": "https://creativecommons.org/publicdomain/zero/1.0/deed.fi",
            "cc-by": "https://creativecommons.org/licenses/by/4.0/deed.fi",
            "cc-by-sa": "https://creativecommons.org/licenses/by-sa/4.0/deed.fi",
            "cc-by-nc": "https://creativecommons.org/licenses/by-nc/4.0/deed.fi",
            "cc-by-nd": "https://creativecommons.org/licenses/by-nd/4.0/deed.fi",
            "cc-by-nc-sa": "https://creativecommons.org/licenses/by-nc-sa/4.0/deed.fi",
            "cc-by-nc-nd": "https://creativecommons.org/licenses/by-nc-nd/4.0/deed.fi"
        },
        "sv": {
            "cc0": "https://creativecommons.org/publicdomain/zero/1.0/",
            "cc-by": "https://creativecommons.org/licenses/by/2.5/se/",
            "cc-by-sa": "https://creativecommons.org/licenses/by-sa/2.5/se/",
            "cc-by-nc": "https://creativecommons.org/licenses/by-nc/2.5/se/",
            "cc-by-nd": "https://creativecommons.org/licenses/by-nd/2.5/se/",
            "cc-by-nc-sa": "https://creativecommons.org/licenses/by-nc-sa/2.5/se/",
            "cc-by-nc-nd": "https://creativecommons.org/licenses/by-nc-nd/2.5/se/"
        }
    };
    
    CopyrightElement.prototype.licensesymbols = {
        'cc0': ['cczero'],
        'cc-by': ['ccby'],
        'cc-by-sa': ['ccby', 'ccsa'],
        'cc-by-nc': ['ccby', 'ccnc'],
        'cc-by-nd': ['ccby', 'ccnd'],
        'cc-by-nc-sa': ['ccby', 'ccnc', 'ccsa'],
        'cc-by-nc-nd': ['ccby', 'ccnc', 'ccnd'],
        'copyright': ['copyright']
    }
    
    CopyrightElement.prototype.licenseicons = {
        'pd': '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="0 0 384 384.017" enable-background="new 0 0 384 384.017" xml:space="preserve"><path d="M192,0.017c-106.036,0-192,85.964-192,192c0,106.024,85.964,192,192,192c106.036,0,192-85.976,192-192 	C384,85.98,298.036,0.017,192,0.017z M192,348.015c-86.023,0-156.004-69.987-156.004-155.998c0-18.007,3.103-35.299,8.746-51.394 l61.013,27.169c-1.643,8.165-2.534,16.682-2.534,25.479c0,76.65,57.212,98.247,93.208,98.247c21.227,0,39.598-6.852,53.979-16.904 c2.733-1.959,5.209-3.93,7.484-5.936l-27.99-34.009c-0.821,0.938-1.678,1.83-2.557,2.628 c-10.887,10.323-22.853,10.323-25.527,10.323c-29.286,0-41.545-30.899-41.709-55.651l168.157,74.867 c0.199,0.035,0.363,0.035,0.527,0.07C302.259,315.215,250.89,348.015,192,348.015z M342.332,231.878l-173.436-77.412 c6.258-11.332,16.218-19.849,31.122-19.849c8.64,0,15.491,2.862,20.887,6.476c2.24,1.619,4.188,3.237,5.83,4.868l31.263-32.178 c-20.998-18.054-44.871-21.28-60.484-21.28c-36.688,0-62.649,15.86-77.952,39.944L61.564,106.58 C89.478,64.114,137.51,36.007,192,36.007c86.022,0,156.01,69.976,156.01,156.01c0,13.925-1.865,27.403-5.303,40.261 C342.566,232.137,342.473,232.019,342.332,231.878z"/></svg>',
        'cczero': '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="-0.5 0.5 64 64" enable-background="new -0.5 0.5 64 64" xml:space="preserve"><g><circle fill="#FFFFFF" cx="31.325" cy="32.873" r="30.096"/><path id="text2809_1_" d="M31.5,14.08c-10.565,0-13.222,9.969-13.222,18.42c0,8.452,2.656,18.42,13.222,18.42 c10.564,0,13.221-9.968,13.221-18.42C44.721,24.049,42.064,14.08,31.5,14.08z M31.5,21.026c0.429,0,0.82,0.066,1.188,0.157 c0.761,0.656,1.133,1.561,0.403,2.823l-7.036,12.93c-0.216-1.636-0.247-3.24-0.247-4.437C25.808,28.777,26.066,21.026,31.5,21.026z M36.766,26.987c0.373,1.984,0.426,4.056,0.426,5.513c0,3.723-0.258,11.475-5.69,11.475c-0.428,0-0.822-0.045-1.188-0.136 c-0.07-0.021-0.134-0.043-0.202-0.067c-0.112-0.032-0.23-0.068-0.336-0.11c-1.21-0.515-1.972-1.446-0.874-3.093L36.766,26.987z"/><path id="path2815_1_" d="M31.433,0.5c-8.877,0-16.359,3.09-22.454,9.3c-3.087,3.087-5.443,6.607-7.082,10.532 C0.297,24.219-0.5,28.271-0.5,32.5c0,4.268,0.797,8.32,2.397,12.168c1.6,3.85,3.921,7.312,6.969,10.396 c3.085,3.049,6.549,5.399,10.398,7.037c3.886,1.602,7.939,2.398,12.169,2.398c4.229,0,8.34-0.826,12.303-2.465 c3.962-1.639,7.496-3.994,10.621-7.081c3.011-2.933,5.289-6.297,6.812-10.106C62.73,41,63.5,36.883,63.5,32.5 c0-4.343-0.77-8.454-2.33-12.303c-1.562-3.885-3.848-7.32-6.857-10.33C48.025,3.619,40.385,0.5,31.433,0.5z M31.567,6.259 c7.238,0,13.412,2.566,18.554,7.709c2.477,2.477,4.375,5.31,5.67,8.471c1.296,3.162,1.949,6.518,1.949,10.061 c0,7.354-2.516,13.454-7.506,18.33c-2.592,2.516-5.502,4.447-8.74,5.781c-3.2,1.334-6.498,1.994-9.927,1.994 c-3.468,0-6.788-0.653-9.949-1.948c-3.163-1.334-6.001-3.238-8.516-5.716c-2.515-2.514-4.455-5.353-5.826-8.516 c-1.333-3.199-2.017-6.498-2.017-9.927c0-3.467,0.684-6.787,2.017-9.949c1.371-3.2,3.312-6.074,5.826-8.628 C18.092,8.818,24.252,6.259,31.567,6.259z"/></g></svg>',
        'ccby': '<svg version="1.0" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="5.5 -3.5 64 64" enable-background="new 5.5 -3.5 64 64" xml:space="preserve"><g><circle fill="#FFFFFF" cx="37.637" cy="28.806" r="28.276"/><g><path d="M37.443-3.5c8.988,0,16.57,3.085,22.742,9.257C66.393,11.967,69.5,19.548,69.5,28.5c0,8.991-3.049,16.476-9.145,22.456 C53.879,57.319,46.242,60.5,37.443,60.5c-8.649,0-16.153-3.144-22.514-9.43C8.644,44.784,5.5,37.262,5.5,28.5 c0-8.761,3.144-16.342,9.429-22.742C21.101-0.415,28.604-3.5,37.443-3.5z M37.557,2.272c-7.276,0-13.428,2.553-18.457,7.657 c-5.22,5.334-7.829,11.525-7.829,18.572c0,7.086,2.59,13.22,7.77,18.398c5.181,5.182,11.352,7.771,18.514,7.771 c7.123,0,13.334-2.607,18.629-7.828c5.029-4.838,7.543-10.952,7.543-18.343c0-7.276-2.553-13.465-7.656-18.571 C50.967,4.824,44.795,2.272,37.557,2.272z M46.129,20.557v13.085h-3.656v15.542h-9.944V33.643h-3.656V20.557 c0-0.572,0.2-1.057,0.599-1.457c0.401-0.399,0.887-0.6,1.457-0.6h13.144c0.533,0,1.01,0.2,1.428,0.6 C45.918,19.5,46.129,19.986,46.129,20.557z M33.042,12.329c0-3.008,1.485-4.514,4.458-4.514s4.457,1.504,4.457,4.514 c0,2.971-1.486,4.457-4.457,4.457S33.042,15.3,33.042,12.329z"/></g></g></svg>',
        'ccsa': '<svg version="1.0" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="5.5 -3.5 64 64" enable-background="new 5.5 -3.5 64 64" xml:space="preserve"><g><circle fill="#FFFFFF" cx="36.944" cy="28.631" r="29.105"/><g><path d="M37.443-3.5c8.951,0,16.531,3.105,22.742,9.315C66.393,11.987,69.5,19.548,69.5,28.5c0,8.954-3.049,16.457-9.145,22.514 C53.918,57.338,46.279,60.5,37.443,60.5c-8.649,0-16.153-3.143-22.514-9.429C8.644,44.786,5.5,37.264,5.5,28.501 c0-8.723,3.144-16.285,9.429-22.685C21.138-0.395,28.643-3.5,37.443-3.5z M37.557,2.272c-7.276,0-13.428,2.572-18.457,7.715 c-5.22,5.296-7.829,11.467-7.829,18.513c0,7.125,2.59,13.257,7.77,18.4c5.181,5.182,11.352,7.771,18.514,7.771 c7.123,0,13.334-2.609,18.629-7.828c5.029-4.876,7.543-10.99,7.543-18.343c0-7.313-2.553-13.485-7.656-18.513 C51.004,4.842,44.832,2.272,37.557,2.272z M23.271,23.985c0.609-3.924,2.189-6.962,4.742-9.114 c2.552-2.152,5.656-3.228,9.314-3.228c5.027,0,9.029,1.62,12,4.856c2.971,3.238,4.457,7.391,4.457,12.457 c0,4.915-1.543,9-4.627,12.256c-3.088,3.256-7.086,4.886-12.002,4.886c-3.619,0-6.743-1.085-9.371-3.257 c-2.629-2.172-4.209-5.257-4.743-9.257H31.1c0.19,3.886,2.533,5.829,7.029,5.829c2.246,0,4.057-0.972,5.428-2.914 c1.373-1.942,2.059-4.534,2.059-7.771c0-3.391-0.629-5.971-1.885-7.743c-1.258-1.771-3.066-2.657-5.43-2.657 c-4.268,0-6.667,1.885-7.2,5.656h2.343l-6.342,6.343l-6.343-6.343L23.271,23.985L23.271,23.985z"/></g></g></svg>',
        'ccnd': '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64.000977px" height="64px" viewBox="0 0 64.000977 64" enable-background="new 0 0 64.000977 64" xml:space="preserve"><g><circle fill="#FFFFFF" cx="32.064453" cy="31.788086" r="29.012695"/><g><path d="M31.943848,0C40.896484,0,48.476562,3.105469,54.6875,9.314453C60.894531,15.486328,64.000977,23.045898,64.000977,32 s-3.048828,16.457031-9.145508,22.513672C48.417969,60.837891,40.779297,64,31.942871,64 	c-8.648926,0-16.152832-3.142578-22.513672-9.429688C3.144043,48.286133,0,40.761719,0,32.000977 	c0-8.723633,3.144043-16.285156,9.429199-22.68457C15.640137,3.105469,23.14502,0,31.943848,0z M32.060547,5.771484 c-7.275391,0-13.429688,2.570312-18.458496,7.714844C8.381836,18.783203,5.772949,24.954102,5.772949,32 c0,7.125,2.589844,13.256836,7.77002,18.400391c5.181152,5.181641,11.352051,7.770508,18.515625,7.770508 c7.123047,0,13.332031-2.608398,18.626953-7.828125C55.713867,45.466797,58.228516,39.353516,58.228516,32 c0-7.3125-2.553711-13.484375-7.65625-18.513672C45.504883,8.341797,39.333984,5.771484,32.060547,5.771484z M44.117188,24.456055 v5.485352H20.859863v-5.485352H44.117188z M44.117188,34.743164v5.481445H20.859863v-5.481445H44.117188z"/></g></g></svg>',
        'ccnc': '<svg version="1.0" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="64px" height="64px" viewBox="5.5 -3.5 64 64" enable-background="new 5.5 -3.5 64 64" xml:space="preserve"><g><circle fill="#FFFFFF" cx="37.47" cy="28.736" r="29.471"/><g><path d="M37.442-3.5c8.99,0,16.571,3.085,22.743,9.256C66.393,11.928,69.5,19.509,69.5,28.5c0,8.992-3.048,16.476-9.145,22.458 C53.88,57.32,46.241,60.5,37.442,60.5c-8.686,0-16.19-3.162-22.513-9.485C8.644,44.728,5.5,37.225,5.5,28.5 c0-8.762,3.144-16.343,9.429-22.743C21.1-0.414,28.604-3.5,37.442-3.5z M12.7,19.872c-0.952,2.628-1.429,5.505-1.429,8.629 c0,7.086,2.59,13.22,7.77,18.4c5.219,5.144,11.391,7.715,18.514,7.715c7.201,0,13.409-2.608,18.63-7.829 c1.867-1.79,3.332-3.657,4.398-5.602l-12.056-5.371c-0.421,2.02-1.439,3.667-3.057,4.942c-1.622,1.276-3.535,2.011-5.744,2.2 v4.915h-3.714v-4.915c-3.543-0.036-6.782-1.312-9.714-3.827l4.4-4.457c2.094,1.942,4.476,2.913,7.143,2.913 c1.104,0,2.048-0.246,2.83-0.743c0.78-0.494,1.172-1.312,1.172-2.457c0-0.801-0.287-1.448-0.858-1.943l-3.085-1.315l-3.771-1.715 l-5.086-2.229L12.7,19.872z M37.557,2.214c-7.276,0-13.428,2.571-18.457,7.714c-1.258,1.258-2.439,2.686-3.543,4.287L27.786,19.7 c0.533-1.676,1.542-3.019,3.029-4.028c1.484-1.009,3.218-1.571,5.2-1.686V9.071h3.715v4.915c2.934,0.153,5.6,1.143,8,2.971 l-4.172,4.286c-1.793-1.257-3.619-1.885-5.486-1.885c-0.991,0-1.876,0.191-2.656,0.571c-0.781,0.381-1.172,1.029-1.172,1.943 c0,0.267,0.095,0.533,0.285,0.8l4.057,1.83l2.8,1.257l5.144,2.285l16.397,7.314c0.535-2.248,0.801-4.533,0.801-6.857 c0-7.353-2.552-13.543-7.656-18.573C51.005,4.785,44.831,2.214,37.557,2.214z"/></g></g></svg>',
        'copyright': '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="32px" height="32px" viewBox="0 0 384 384.017" enable-background="new 0 0 384 384.017" xml:space="preserve"><path d="M192 0 a192 192 0 0 1 0 384 a192 192 0 0 1 0 -384z m0 36 a156 156 0 0 0 0 312 a156 156 0 0 0 0 -312z m66 78 l-32 32 a42 57 -2 1 0 4 88 l28 34 a94 101 7 1 1 0 -154z"/></svg>'
    }
    
    /**
     * Templates
     */
    CopyrightElement.prototype.templates = {
        html: [
            '<div class="copyrightelement-wrapper">',
            '    <div class="copyrightelement-ownerwrapper">',
            '        <div class="copyrightelement-ownerlogo"></div>',
            '        <div class="copyrightelement-owner"></div>',
            '    </div>',
            '    <div class="copyrightelement-licensing">',
            '        <div class="copyrightelement-licenseicons"></div>',
            '        <div class="copyrightelement-license"></div>',
            '    </div>',
            '</div>'
        ].join('\n')
    };
    
    /**
     * Styles
     */
    CopyrightElement.prototype.styles = [
        '.copyrightelement-wrapper {text-align: center;}',
        '.copyrightelement-wrapper a {text-decoration: none; color: black;}',
        '.copyrightelement-wrapper a:hover {text-decoration: underline; color: #300;}',
        '.copyrightelement-owner {font-size: 220%; font-weight: bold; margin: 1em 0 0.3em;}',
        '.copyrightelement-ownerwrapper.haslogo .copyrightelement-owner {display: none;}',
        '.copyrightelement-ownerwrapper.haslogo .copyrightelement-ownerlogo {display: block;}',
        '.copyrightelement-ownerlogo {display: none;}',
        '.copyrightelement-ownerlogo a {display: block; margin:0; padding: 0; text-decoration: none;}',
        '.copyrightelement-licensing[data-license^="cc-"] svg {width: 3em; height: 3em;}',
        '.copyrightelement-licensing[data-license="copyright"] div {display: inline-block;}',
        '.copyrightelement-licensing[data-license="copyright"] .copyrightelement-licenseicons {vertical-align: middle;}',
        '.copyrightelement-licensing[data-license="copyright"] svg {width: 1.5em; height: 1.5em;}'
    ].join('\n');


    /******
     * Info about element (icon, description, etc.)
     ******/
    CopyrightElement.elementinfo = {
        type: 'copyrightelement',
        elementtype: 'elements',
        jquery: 'copyrightelement',
        name: 'Copyright',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-markdown mini-icon-commonmark"><path style="stroke: none;" d="M3 6 l24 0 a3 3 0 0 1 3 3 l0 12 a3 3 0 0 1 -3 3 l-24 0 a3 3 0 0 1 -3 -3 l0 -12 a3 3 0 0 1 3 -3z m0 2 a1 1 0 0 0 -1 1 l0 12 a1 1 0 0 0 1 1 l24 0 a1 1 0 0 0 1 -1 l0 -12 a1 1 0 0 0 -1 -1z m2 3 l2.5 0 l2.5 3 l2.5 -3 l2.5 0 l0 8 l-2.5 0 l0 -5 l-2.5 3 l-2.5 -3 l0 5 l-2.5 0z m15 0 l2.5 0 l0 4 l2.5 0 l-3.75 4 l-3.75 -4 l2.5 0z" /></svg>',
        description: {
            en: 'Show the information about copyrights',
            fi: 'Näytetään tieto tekijänoikeuksista'
        },
        classes: ['viewonly']
    }
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', CopyrightElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', CopyrightElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var elem = new CopyrightElement(this, params);
            });
        }
    }
    
    $.fn.copyrightelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in copyrightelement.');
            return false;
        }
    }


})(jQuery);