/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}


;(function($){

    /**
     * Helper functions
     */
    
    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };

    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Form dialog for showing config dialogs etc.
     * @requires 
     * @constructor
     * @param {jQuery} place - place for dialog
     * @param {Object} options - data for the dialog
     ******/
    var FormDialog = function(place, options){
        this.place = $(place);
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init form dialog
     * @param {Object} options - initing data of the dialog
     ******/
    FormDialog.prototype.init = function(options){
        options = $.extend(true, {}, FormDialog.defaults, options);
        this.type = options.type;
        this.name = options.name;
        this.data = options.data;
        // Keep a copy of original values in case the user wants to cancel.
        this.originals = $.extend(true, {}, this.data.values);
        this.settings = options.settings;
        this.configs = options.configs;
        this.setAttrs();
        this.removeHandlers();
        this.initHandlers();
        this.currentTab = this.data.formsets[0] || '';
    };
    
    /******
     * Destroy this dialog
     ******/
    FormDialog.prototype.destroy = function(){
        this.removeHandlers();
        this.place.empty();
    };
    
    /******
     * Set some attributes
     ******/
    FormDialog.prototype.setAttrs = function() {
        this.place.addClass('formdialog');
        if (this.settings.modal === 'window') {
            this.place.attr('data-modal','windowmodal');
        } else if (this.settings.modal) {
            this.place.attr('data-modal', 'modal');
        } else {
            this.place.removeAttr('data-modal');
        };
    };
    
    /******
     * Set style-tag if needed.
     ******/
    FormDialog.prototype.setStyles = function(){
        if ($('head style#formdialog-style').length === 0) {
            $('head').append('<style id="formdialog-style" type="text/css">'+FormDialog.styles+'</style>')
        };
    };
    
    
    /******
     * Show the form dialog
     ******/
    FormDialog.prototype.show = function(){
        this.place.empty().html(FormDialog.templates.view);
        this.window = this.place.children('.formdialog-dialogbox');
        this.setPosition();
        this.window.find('.formdialog-title .formdialog-title-icon').html(this.data.icon);
        this.window.find('.formdialog-title h1').html(this.data.title);
        if (this.data.formsets.length > 1) {
            this.showTabs();
        } else if (this.data.formsets.length === 1) {
            this.showForm();
        };
        this.showButtons();
    };
    
    /******
     * Show form tabs
     ******/
    FormDialog.prototype.showTabs = function(){
        var dialog = this;
        var tablist = this.window.find('.formdialog-tabbox');
        var tabname, tabdata, html = [];
        this.data.formsets.sort(function(a, b) {
            var aweight = dialog.data.forms[a].weight | 0;
            var bweight = dialog.data.forms[b].weight | 0;
            return aweight - bweight;
        });
        for (var i = 0, len = this.data.formsets.length; i < len; i++) {
            tabname = this.data.formsets[i];
            tabdata = this.data.forms[tabname];
            html.push('<li class="formdialog-tabitem" data-tabname="' + escapeHTML(tabname) + '"><span class="formdialog-tabitem-icon">' + (sanitize(tabdata.icon) || '') + '</span> <span class="formdialog-tabitem-title">' + escapeHTML(tabdata.title) + '</span></li>');
        };
        tablist.html(html.join('\n'));
        tablist.children().eq(0).click();
    };
    
    /******
     * Show the form of selected tab
     * @param {String} tabname - name (id) of the tab
     ******/
    FormDialog.prototype.showForm = function(tabname) {
        tabname = tabname || this.currentTab;
        var formbox = this.window.find('.formdialog-formbox');
        formbox.attr('data-formset', tabname);
        var formdata = this.data.forms[tabname] || [];
        var html = [], finput, value;
        html.push('<fieldset>');
        html.push('<legend class="formdialog-formbox-title">' + escapeHTML(formdata.title) + '</legend>');
        html.push('<table><tbody>');
        for (var i = 0, len = formdata.form.length; i < len; i++) {
            finput = formdata.form[i];
            value = this.data.values[tabname] &&  this.data.values[tabname][finput.name];
            html.push('<div>' + this.getInputElement(finput, value) + '</div>');
        };
        html.push('</table></tbody>');
        html.push('</fieldset>');
        formbox.html(html.join('\n'));
    };
    
    /******
     * Show OK and Cancel buttons
     ******/
    FormDialog.prototype.showButtons = function(){
        var uilang = this.settings.uilang;
        var buttons = ['ok', 'cancel'], action;
        var html = [];
        for (var i = 0, len = buttons.length; i < len; i++) {
            action = buttons[i];
            html.push('<button class="ffwidget-button" data-action="'+escapeHTML(action)+'">'+ebooklocalizer.localize(action, uilang)+'</button></div>');
        };
        this.window.find('.formdialog-buttonarea').html(html.join(''));
    };
    
    /******
     * Convert object-formatted input element to HTML
     * @param {Object} data - the data of form input as an object.
     ******/
    FormDialog.prototype.getInputElement = function(data, value){
        if (typeof(value) === 'undefined') {
            value = data.defval;
        };
        var html = [];
        var label = sanitize(data.icon) || escapeHTML(data.label) || '';
        switch (data.type) {
            case 'label':
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td colspan="2"><span class="formdialog-formlabel">' + label + '</span></td></tr>');
                break;
            case 'select':
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td><label class="formdialog-label-text">'+label+'</label></td><td><select data-attribute="' + escapeHTML(data.attribute) + '">');
                var option, selected;
                for (var i = 0, len = data.options.length; i < len; i++) {
                    option = data.options[i];
                    selected = (option.value === value);
                    html.push('<option value="' + escapeHTML(option.value) + '"'+(selected ? ' selected' : '')+'>' + escapeHTML(option.label) + '</option>');
                };
                html.push('</select></td></tr>');
                break;
            case 'radio':
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td><label><span class="formdialog-label-text">' + label + '</span></label></td><td>');
                var option, selected;
                var rnd = Math.floor(Math.random() * 100000);
                for (var i = 0, len = data.options.length; i < len; i++) {
                    option = data.options[i];
                    selected = (option.value === value);
                    html.push('<div><label><input type="radio" name="' + escapeHTML(data.name) + '-' + rnd + '" data-attribute="' + escapeHTML(data.attribute) + '" value="'+escapeHTML(option.value)+'"'+(selected ? ' checked' : '')+'> ' + escapeHTML(option.label) + '</label></div>');
                };
                html.push('</td></tr>');
                break;
            case 'checkbox':
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td><label><span class="formdialog-label-text">' + label + '</span></label></td><td><input type="checkbox" data-attribute="' + escapeHTML(data.attribute) + '" ' + (value ? 'checked="checked"' : '') + ' /></td></tr>');
                break;
            case 'range':
                var output = (data.output ? '<output class="formdialog-rangeoutput">' + escapeHTML(value) + '</output>' : '');
                var oninput = (data.output ? ' oninput="this.nextSibling.value = this.value"' : '');
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td><label><span class="formdialog-label-text">' + label + '</span></label></td><td><input type="range" class="ffwidget-range ffwidget-rangethumb" data-attribute="' + escapeHTML(data.attribute) + '" value="' + escapeHTML(value) + '" min="'+(data.min | 0)+'" max="'+escapeHTML(data.max)+'"'+oninput+' />'+output+'</td></tr>');
                break;
            case 'checkboxset':
                html.push('<tr data-widgettype="'+escapeHTML(data.type)+'"><td><label><span class="formdialog-label-text">' + label + '</span></label></td><td>');
                var option, selected;
                for (var i = 0, len = data.options.length; i < len; i++) {
                    option = data.options[i];
                    selected = (value.indexOf(option.name) !== -1);
                    html.push('<div><label><input type="checkbox" data-name="' + escapeHTML(data.name) + '" data-attribute="' + escapeHTML(data.attribute) + '" value="'+escapeHTML(option.value)+'"'+(selected ? ' checked' : '')+'> ' + escapeHTML(option.label) + '</label></div>');
                }
                html.push('</td></tr>')
                break;
            default:
                break;
        };
        return html.join('');
    }
    
    /******
     * Remove all event handlers
     ******/
    FormDialog.prototype.removeHandlers = function(){
        this.place.off();
        $(window).off('resize.formdialog');
    };
    
    /******
     * Init handlers
     ******/
    FormDialog.prototype.initHandlers = function(){
        var dialog = this;
        this.removeHandlers();
        this.place.on('mousedown touchstart', '.formdialog-headbar', function(event, data){
            event.stopPropagation();
            var parent = $(this).parent();
            var offset = parent.offset();
            dialog.startMoving({x: event.clientX - offset.left, y: event.clientY - offset.top});
        }).on('click', 'li.formdialog-tabitem', function(event, data){
            event.stopPropagation();
            var tab = $(this);
            var tabname = tab.attr('data-tabname');
            dialog.currentTab = tabname;
            tab.parent().children('li').removeClass('selected');
            tab.addClass('selected');
            dialog.showForm(tabname);
        }).on('mousedown touchstart', '.formdialog-closebutton', function(event, data){
            event.stopPropagation();
        }).on('click', '.formdialog-closebutton', function(event, data){
            event.stopPropagation();
            dialog.destroy();
        }).on('click', '.formdialog-buttonarea button', function(event, data){
            event.stopPropagation();
            var button = $(this);
            var action = button.attr('data-action');
            button.trigger('formdialog-'+action);
        }).on('formdialog-ok', function(event, data){
            event.stopPropagation();
            dialog.destroy();
        }).on('formdialog-cancel', function(event, data){
            event.stopPropagation();
            dialog.cancel();
            dialog.destroy();
        }).on('change', '.formdialog-formbox [data-widgettype="checkbox"] input[type="checkbox"]', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.is(':checked');
            var attribute = input.attr('data-attribute');
            var formset = input.closest('.formdialog-formbox').attr('data-formset');
            dialog.setValue(formset, attribute, value);
        }).on('input', '.formdialog-formbox input[type="range"]', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            var attribute = input.attr('data-attribute');
            var formset = input.closest('.formdialog-formbox').attr('data-formset');
            dialog.setValue(formset, attribute, value);
        }).on('change', '.formdialog-formbox input[type="radio"]', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            var attribute = input.attr('data-attribute');
            var formset = input.closest('.formdialog-formbox').attr('data-formset');
            dialog.setValue(formset, attribute, value);
        }).on('change', '.formdialog-formbox select', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            var attribute = input.attr('data-attribute');
            var formset = input.closest('.formdialog-formbox').attr('data-formset');
            dialog.setValue(formset, attribute, value);
        }).on('change', '.formdialog-formbox [data-widgettype="checkboxset"] input[type="checkbox"]', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var onoff = input.is(':checked');
            var value = input.attr('value');
            var attribute = input.attr('data-attribute');
            var formset = input.closest('.formdialog-formbox').attr('data-formset');
            var selecteds = dialog.getValue(formset, attribute) || [];
            var index = selecteds.indexOf(value);
            if (onoff) {
                if (index === -1) {
                    selecteds.push(value);
                };
            } else {
                if (index !== -1) {
                    selecteds.splice(index, 1);
                };
            };
            dialog.setValue(formset, attribute, selecteds);
        });
        $(window).on('resize.formdialog', function(event, data){
            dialog.setPosition();
        });
    };
    
    /******
     * Start moving the dialog
     ******/
    FormDialog.prototype.startMoving = function(coords) {
        var dialog = this;
        this.moveOffset = coords;
        $('body').addClass('formdialogmoving')
            .off('mousemove.formdialog touchmove.formdialog')
            .on('mousemove.formdialog touchmove.formdialog', function(event, data){
                var coords = {x: event.clientX, y: event.clientY};
                dialog.move(coords);
            })
            .off('mouseup.formdialog touchend.formdialog')
            .on('mouseup.formdialog touchend.formdialog', function(event, data){
                dialog.stopMoving();
            });
    };
    
    /******
     * Stop moving the dialog
     ******/
    FormDialog.prototype.stopMoving = function(){
        $('body').removeClass('formdialogmoving')
            .off('mousemove.formdialog touchmove.formdialog')
            .off('mousemove.formdialog touchend.formdialog');
        this.configChanged();
    };
    
    /******
     * Move the form dialog
     ******/
    FormDialog.prototype.move = function(coords){
        var newcoords = {
            x: coords.x - this.moveOffset.x,
            y: coords.y - this.moveOffset.y
        };
        this.setPosition(newcoords);
    };
    
    /******
     * Set the position of dialog window
     ******/
    FormDialog.prototype.setPosition = function(coords){
        if (typeof(coords) === 'object') {
            this.configs.position.x = coords.x;
            this.configs.position.y = coords.y;
        };
        var width = this.window.width();
        var height = this.window.height();
        var wwidth = $(window).width();
        var wheight = $(window).height();
        var maxx = wwidth - width;
        var maxy = wheight - height;
        var positions = {
            top: Math.max(0, Math.min(this.configs.position.y, maxy)),
            left: Math.max(0, Math.min(this.configs.position.x, maxx)),
            right: 'auto',
            bottom: 'auto'
        };
        this.window.css(positions);
    };
    
    /**
     * Get the value of some attribute
     */
    FormDialog.prototype.getValue = function(formset, attribute) {
        return this.data.values[formset][attribute];
    };
    
    /******
     * Change the value of some attribute
     ******/
    FormDialog.prototype.setValue = function(formset, attribute, value){
        if (typeof(this.data.values[formset]) === 'undefined') {
            this.data.values[formset] = {};
        };
        this.data.values[formset][attribute] = value;
        this.valueChanged(formset, attribute);
    };
    
    /******
     * Actions for changed values. Trigger an event.
     ******/
    FormDialog.prototype.valueChanged = function(formset, attribute){
        var senddata = {
            formsets: [],
            values: {}
        };
        if (typeof(formset) === 'undefined') {
            // Send all values
            senddata.formsets = this.data.formsets.slice();
            senddata.values = JSON.parse(JSON.stringify(this.data.values));
        } else if (typeof(attribute) === 'undefined') {
            // Send all values in given formset
            senddata.formsets.push(formset);
            senddata.values[formset] = JSON.parse(JSON.stringify(this.data.values[formset]));
        } else {
            // Send only value of given attribute in given formset
            senddata.formsets.push(formset);
            senddata.values[formset] = {};
            senddata.values[formset][attribute] = this.data.values[formset][attribute];
        };
        this.place.trigger('formdialog-valuechange', {dialogname: this.name, dialogdata: senddata});
    }
    
    /******
     * Cancel and reset the values back to the original ones.
     ******/
    FormDialog.prototype.cancel = function(){
        this.data.values = JSON.parse(JSON.stringify(this.originals));
        this.valueChanged();
    };
    
    /******
     * Get configs of this dialog
     ******/
    FormDialog.prototype.getConfig = function(){
        var configs = $.extend(true, {}, this.configs);
        return configs;
    };
    
    /******
     * Trigger config changed
     ******/
    FormDialog.prototype.configChanged = function(){
        this.place.trigger('config_changed', {type: this.type, key: this.name, toolname: this.name, config: this.getConfig()});
    }
    
    /******
     * Default settings
     ******/
    FormDialog.defaults = {
        type: 'dialog',
        name: '',
        data: {
            title: '',
            formsets: [],
            forms: {},
            values: {}
        },
        configs: {
            position: {
                x: 100,
                y: 100
            }
        },
        settings: {
            uilang: 'en',
            lang: 'en',
            modal: false // false | true | 'window'
        }
    }
    
    /******
     * Icons
     ******/
    FormDialog.icons = {
        tool: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 50 50" class="pssicon pssicon-wrench"><path style="stroke: none;" fill="black" d="M35 5 a12 12 0 1 0 9 9 l-9 9 a10 10 0 0 1 -9 -9 l9 -9z m-15 17.5 l-17 17 a2 2 0 0 0 7 7 l17 -17z m-16 18 a2 2 0 0 1 5 5 a2 2 0 0 1 -5 -5z"></path></svg>',
        settings: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 32 32" class="mini-icon mini-icon-gear"><path style="stroke: none;" d="M12 0 h8 l0.9 3.5 l3.5 2 l3.55 -1.1 l4.1 7 l-2.7 2.5 v4.1 l2.7 2.5 l-4.1 7 l-3.55 -1.1 l-3.5 2 l-0.9 3.5 h-8 l-0.9 -3.5 l-3.5 -2 l-3.55 1.1 l-4.1 -7 l2.7 -2.5 v-4.1 l-2.7 -2.5 l4.1 -7 l3.55 1.1 l3.5 -2z m1.55 2 l-0.7 2.9 l-5 2.8 l-2.8 -0.8 l-2.5 4.3 l2.15 2 v5.65 l-2.15 2 l2.5 4.3 l2.8 -0.8 l5 2.8 l0.7 2.9 h4.9 l0.7 -2.9 l5 -2.8 l2.8 0.8 l2.5 -4.3 l-2.15 -2 v-5.65 l2.15 -2 l-2.5 -4.3 l-2.8 0.8 l-5 -2.8 l-0.7 -2.9z m2.45 7.55 a6.45 6.45 0 0 1 0 12.9 a6.45 6.45 0 0 1 0 -12.9z m0 2 a4.45 4.45 0 0 0 0 8.9 a4.45 4.45 0 0 0 0 -8.9z"></path></svg>',
        exit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-exit pssicon-close pssicon-cross"><path style="stroke: none;" d="M4 7 a1.5 1.5 0 0 1 3 -3 l8 8 l8 -8 a1.5 1.5 0 0 1 3 3 l-8 8 l8 8 a1.5 1.5 0 0 1 -3 3 l-8 -8 l-8 8 a1.5 1.5 0 0 1 -3 -3 l8 -8 l-8 -8 z"></path></svg>'
    }
    
    /******
     * Templates
     ******/
    FormDialog.templates = {
        view: [
            '<div class="formdialog-modalshadow"></div>',
            '<div class="formdialog-dialogbox ffwidget-background">',
            '    <div class="formdialog-headbar ffwidget-background-colored"><div class="formdialog-closebutton">'+FormDialog.icons.exit+'</div></div>',
            '    <div class="formdialog-title"><div class="formdialog-title-icon"></div><h1></h1></div>',
            '    <div class="formdialog-formarea">',
            '        <ul class="formdialog-tabbox"></ul>',
            '        <div class="formdialog-formbox"></div>',
            '    </div>',
            '    <div class="formdialog-buttonarea"></div>',
            '</div>'
        ].join('\n')
    }
    
    /******
     * Styles
     ******/
    FormDialog.styles = [
        // Modalshadow
        '.formdialog-modalshadow {display: none; top: 0; bottom: 0; left: 0; right: 0; background: rgba(0,0,0,0.4); z-index: 999;}',
        '.formdialog[data-modal="modal"] .formdialog-modalshadow {display: block; position: absolute;}',
        '.formdialog[data-modal="windowmodal"] .formdialog-modalshadow {display: block; position: fixed;}',
        // Dialog
        '.formdialog-dialogbox {position: fixed; top: 80px; top: 100px; left: 100px; min-width: 100px; min-height: 100px; border: 1px solid #666; border-radius: 0.5em; box-shadow: 10px 10px 20px rgba(0,0,0,0.4); -webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none; z-index: 1000;}',
        // Headbar
        '.formdialog-headbar {margin: 4px 5px; background-color: rgba(0,0,0,0.4); height: 15px; border: 1px solid #999; border-radius: 5px; cursor: move; text-align: right; position: relative;}',
        '.formdialog-closebutton {position: absolute; top: 2px; right: 2px; margin: 0; padding: 0; width: 12px; height: 12px; border-radius: 3px; background-color: black; cursor: pointer;}',
        '.formdialog-closebutton svg {display: block; width: 12px; height: 12px;}',
        '.formdialog-closebutton svg path {fill: white;}',
        // Title
        '.formdialog-title {margin: 0 1em; padding: 2px 0;}',
        '.formdialog-title-icon {float: right; width: 30px; height: 30px;}',
        '.formdialog-title-icon svg {width: 30px; height: 30px;}',
        '.formdialog-title h1 {font-family: Helvetica, sans-serif; font-size: 120%; margin: 0; padding: 0;}',
        // Formarea
        '.formdialog-formarea {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; clear: both;}',
        // Tablist
        '.formdialog-tabbox {list-style: none; margin: 0.5em 0 0.5em 1em; padding: 0.5em 0; position: relative; z-index: 1;}',
        '.formdialog-tabbox li {border: 1px solid #aaa; border-right: none; background-color: rgba(0,0,0,0.1); padding: 0.5em 1em; cursor: pointer; border-radius: 0.5em 0 0 0.5em; font-size: 80%; white-space: nowrap; max-width: 14em; overflow: hidden; text-overflow: ellipsis;}',
        '.formdialog-tabbox li.selected {border: 1px solid #aaa; border-right: 1px solid #eee; margin-right: -1px; background-color: rgba(255,255,255,0.1);}',
        '.formdialog-tabitem-icon {display: inline-block; vertical-align: middle; width: 20px; height: 20px; padding: 2px;}',
        '.formdialog-tabitem-icon svg {width: 20px; height: 20px;}',
        // Formbox
        '.formdialog-formbox {position: relative; background-color: rgba(255,255,255,0.05); padding: 0.5em; border: 1px solid #aaa;-webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; margin: 0.5em 1em 0.5em 0; border-radius: 0.5em;}',
        '.formdialog-formbox-title {font-size: 120%; font-family: Helvetica, sans-serif; margin: 0 1em;}',
        '.formdialog-formbox table td {vertical-align: top;}',
        '.formdialog-label-text {text-transform: capitalize;}',
        '.formdialog-rangeoutput {vertical-align: top; font-size: 90%;}',
        // Buttonarea
        '.formdialog-buttonarea {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row-reverse nowrap; -ms-flex-flow: row-reverse nowrap; flex-flow: row-reverse nowrap; border-top: 1px solid black; padding: 8px;}',
        '.formdialog-buttonarea button {margin: 0 0 0 0.5em; min-width: 4em;}',
        // Form label
        '.formdialog-formlabel {font-weight: bold;}',
        ''
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    FormDialog.localization = {
        "en": {
            "formdialog:settings": "Settings",
            "ok": "OK",
            "cancel": "Cancel"
        },
        "fi": {
            "formdialog:settings": "Asetukset",
            "ok": "OK",
            "cancel": "Peruuta"
        },
        "sv": {
            "formdialog:settings": "Inställningar",
            "ok": "OK",
            "cancel": "Avbryt"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(FormDialog.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(FormDialog.localization);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var dialog = new FormDialog(this, params);
            });
        }
    }
    
    $.fn.formdialog = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in formdialog.');
            return false;
        }
    }

})(jQuery);