//{{{
/*
 * Examplepicture
 * A framework for geometric examples with jsxgraph
 *
 * Petri Salmela (pesalmel@abo.fi)
 * Petri Sallasmaa (pekasa@utu.fi)
 * 
 */

/**
 * Requirements:
 * - jQuery
 * - jQuery-ui
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
    typeof(jQuery.fn.button) === 'undefined' && jQuery.fn.button.apply;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - JSXgraph
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(JXG) === 'undefined' && JXG.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    // jQuery plugin
    $.fn.emathexamplepic = function(options){
        // Test for examplepic commands and trigger command with options.
        if (typeof(options) === 'string'){
            options = {data:{example: options}};
            // Placeholder variable for returning value.
        }
        // Extend default settings with user given options.
        var settings = $.extend({
            xscale: [-6, 6],
            yscale: [-6, 6]
        }, options.data);

        // Return this so that methods of jQuery element can be chained.
        return this.each(function(){
            // Create new Examplepicture object.
            var examplepic = new Examplepicture(this, settings);
            // Init the examplepicture
            if(!examplepic.settings.modal){
                examplepic.init();
            }else{
                var buttontext = (typeof(examplepic.settings.buttontitle) ==="string" ? examplepic.settings.buttontitle:"open");//"nappula";//settings.buttontext
                
                var emexpicbutton = examplepic.place.empty().addClass('modalexample').append('<a href="javascript:;" class="emathex-button">'+escapeHTML(buttontext)+'</a>').find('a.emathex-button').last();
                emexpicbutton.button({
                    icons: {
                        primary: 'em-icon-expic'
                    }
                }).click(function(){
                    var backdiv = jQuery('<div class="emathexpic-back"></div>');
                    backdiv.appendTo('body');
                    backdiv.html('<div class="emexpic-wrapper"></div><a href="javascript:;">x</a>');
                    backdiv.css({'position': 'fixed', 'top': '0', 'right': '0', 'bottom': '0', 'left': '0', 'background-color': 'rgba(240,240,240,0.9)', 'z-index': '201'});
                    var closebutton = backdiv.find('a');
                    closebutton.click(function(){
                        backdiv.remove();
                    }).css({'display': 'inline-block', 'position': 'absolute', 'top': '20px', 'right': '20px', 'height': '1.5em', 'width': '1.5em', 'border-radius': '0.4em', 'border': '2px solid black', 'background': '#f00', 'color': 'white', 'text-align': 'center', 'vertical-align': 'middle', 'font-weight': 'bold', 'box-shadow': 'inset 2px 2px 3px rgba(255,255,255,0.5), inset -2px -2px 3px rgba(0,0,0,0.5)'});
                    var content = backdiv.find('.emexpic-wrapper');
                    content.css({'max-width': '1000px', 'margin': '5em auto 0 auto','text-align': 'center'});
                    examplepic.place = content;
                    examplepic.init();
                });
            }
        });
    }



    var Examplepicture = function(place, settings){
        this.place = $(place);
        this.settings = settings;
        this.place.addClass('examplepicture');
        this.xscale = this.settings.xscale || [-6, 6];
        this.yscale = this.settings.yscale || [-6, 6];
        this.lang = this.settings.lang || this.place.attr('lang') || this.place.parents('[lang]').attr('lang');
    }
    
    Examplepicture.prototype.init = function(){
        this.drawingNum = -1;
        while ($('#examplepicture_'+(++this.drawingNum)).length > 0){};
        this.drawingId = 'examplepicture_'+this.drawingNum;
        this.place.empty()
            .append('<div class="examplepicture_header"></div><div class="examplepicture_drawingarea" id="'+escapeHTML(this.drawingId)+'"></div><div class="examplepicture_controlarea"></div>');
        this.header = this.place.find('.examplepicture_header');
        this.drawing = this.place.find('.examplepicture_drawingarea');
        this.control = this.place.find('.examplepicture_controlarea');
        this.drawing.css({width: '300px', height: '300px','display':'inline-block'});
        if ($('head style#examplepicturecss').length < 1){
            $('head').append('<style  id="examplepicturecss" type="text/css">'+this.strings.css+'</style>');
        }
        this.board = JXG.JSXGraph.initBoard(this.drawingId, {
            grid: false,
            boundingbox: [this.xscale[0], this.yscale[1], this.xscale[1], this.yscale[0]],
            keepaspectratio: true,
            axis: false,
            showNavigation: false,
            showCopyright: false
        });
        if (typeof(this.settings.example) === 'function'){
            this.action = this.settings.example;
            this.title = exthisample.settings.title || '';
        } else if (typeof(this.settings.example) === 'string' && (this.settings.example in this.examples)){
            this.action = this.examples[this.settings.example].func;
            this.dict = this.examples[this.settings.example].dict;
            this.title = this.settings.title || this.localize(this.examples[this.settings.example].title);
        } else if (typeof(this.settings.example) === 'object'
                          && typeof(this.settings.example.func) == 'function'){
            this.action = this.settings.example.func;
            this.dict = this.settings.example.dict;
            this.title = this.settings.title || this.localize(this.settings.example.title);
        }
        if (typeof(this.action) === 'function'){
            this.setTitle();
            this.action();
            this.update();
        }
    }
    
    Examplepicture.prototype.setTitle = function(){
        if (typeof(this.title) === 'string' && this.title.length > 0){
            this.header.html('<h1>'+escapeHTML(this.title)+'</h1>');
        }
    }
    
    Examplepicture.prototype.update = function(){
        this.board.update();
    }
    
    Examplepicture.prototype.localize = function(text){
        var result;
        if (this.dict[text]){
            if (this.dict[text][this.lang]){
                result = this.dict[text][this.lang];
            }
            result = result || this.dict[text]['en'];
        }
        result = result || text;
        return result;
    }
    
    Examplepicture.prototype.examples = {
        circleapprox: {
            title: 'approximation of pi',
            func: function(){
                var example = this;
                var brd = this.board;

                var n = brd.createElement('slider',[[-5,5],[3,5],[3,6,96]],{name:'n',snapWidth:1, precision: 0});
                var center = brd.create('point', [0,0], {visible: false});
                var rot = brd.create('transform', [function() {return Math.PI*2.0/n.Value();},center], {type:'rotate'}); 
                var ptmp = brd.create('point',[0,0],{visible:false,withLabel:false});
                var radius = brd.create('point', [4,0], {visible: false});
                var outer = brd.create('curve', [[0],[0]],{fillColor:'#ffff00',highlightFillColor:'#ffff00',fillOpacity:1,highlightFillOpacity:1,strokeColor:'black',highlightStrokeColor:'black'});
                var circle = brd.create('circle', [center, radius], {fixed: true, fillColor: '#00ffff',highlightFillcolor: '#00ffff',fillOpacity:0.5,highlightFillOpacity:0.5, strokeColor: 'black', highlightColor:'black'})
                var inner = brd.create('curve',[[0,1,2],[0,1,2]],{fillColor:'#ff00ff',highlightFillColor:'#ff00ff',fillOpacity:1,highlightFillOpacity:1,strokeColor:'black',highlightStrokeColor:'black'});
                example.control.html('<table><tbody><tr><td>'+ example.localize('value of pi')+':</td><td style="text-align: right;">&approx;'+(Math.PI).toFixed(5)+'</td></tr><tr class="examplepicture_innerarea"></tr><tr class="examplepicture_outerarea"></tr></tbody></table>');
                
                inner.updateDataArray = function() {
                    var i, len = n.Value();
                    this.dataX = [center.X()+circle.getRadius()];
                    this.dataY = [center.Y()];
                    ptmp.setPositionDirectly(JXG.COORDS_BY_USER, [center.X()+circle.getRadius(), center.Y()]);
                    for (i=0;i<len;i++) {
                        rot.applyOnce(ptmp);
                        this.dataX.push(ptmp.X());
                        this.dataY.push(ptmp.Y());
                    }
                    example.control.find('.examplepicture_innerarea').html('<td>'+example.localize('approx of pi from inscribed').replace(/%1/,''+n.Value())+':</td><td style="text-align: right;">'+(n.Value()*Math.sin(Math.PI/n.Value())).toFixed(5)+'</td>');
                }
                   
                outer.updateDataArray = function() {
                    var i, len = n.Value();
                    var s = circle.getRadius()/Math.cos(Math.PI/n.Value());
                    this.dataX = [center.X()+s];
                    this.dataY = [center.Y()];
                    ptmp.setPositionDirectly(JXG.COORDS_BY_USER, [center.X()+s,center.Y()]);
                    for (i=0;i<len;i++) {
                        rot.applyOnce(ptmp);
                        this.dataX.push(ptmp.X());
                        this.dataY.push(ptmp.Y());
                    }
                    example.control.find('.examplepicture_outerarea').html('<td>'+example.localize('approx of pi from circumscribed').replace(/%1/,''+n.Value())+':</td><td style="text-align: right;">'+(n.Value()*Math.tan(Math.PI/n.Value())).toFixed(5)+'</td>');
                }
            },
            dict: {
                'approximation of pi': {
                    'en': 'Archimeds\' Approximation of Pi',
                    'fi': 'Piin likiarvo Arkhimedeen tapaan',
                    'sv': 'Arkimedes approximation av Pi'
                },
                'value of pi': {
                    'en': 'Value of Pi',
                    'fi': 'Piin arvo',
                    'sv': 'Pi:s värde'
                },
                'approx of pi from inscribed': {
                    'en': 'Inscribed %1-gon',
                    'fi': 'Sisään piirretystä %1-kulmiosta',
                    'sv': 'Den inskrivna %1-hörningen'
                },
                'approx of pi from circumscribed': {
                    'en': 'Circumscribed %1-gon',
                    'fi': 'Ympäri piirretystä %1-kulmiosta',
                    'sv': 'Den omskrivna %1-hörningen'
                }
            }
        },
        
        anglenames: {
            title: 'classification of angles',
            func: function(){
                var example = this;
                var brd = this.board;

                var radius = 4;
                var n = brd.createElement('slider', [[-5.5,5.5],[-5.5,-5],[0,30,360]],{name: '&alpha;', snapWidth: 5, precision: 0});
                var center = brd.create('point', [0,0], {visible: true, fixed: true, name: ''});
                var point1 = brd.create('point', [radius,0], {visible: false});
                var point2 = brd.create('point', [function(){return 4*Math.cos(n.Value()*Math.PI/180);}, function(){return 4*Math.sin(n.Value()*Math.PI/180);}], {visible: false});
                var line1 = brd.create('line', [center, point1], {straightFirst: false, straightLast: false})
                var line2 = brd.create('line', [center, point2], {straightFirst: false, straightLast: false})
                var angle = brd.create('angle', [point1, center, point2], {type: 'sector', orthoType: 'square', orthoSensitivity: 0.5, radius: 0.8, name: '&alpha;'});
                
                point1.update = function(){
                    var angletext;
                    if (n.Value() === 0){
                        angletext = example.localize('zero angle')
                    } else if (n.Value() < 90) {
                        angletext = example.localize('acute angle') + ', ' + example.localize('concave angle');
                    } else if (n.Value() === 90) {
                        angletext = example.localize('right angle') + ', ' + example.localize('concave angle');
                    } else if (n.Value() < 180) {
                        angletext = example.localize('obtuse angle') + ', ' + example.localize('concave angle');
                    } else if (n.Value() === 180) {
                        angletext = example.localize('straight angle');
                    } else if (n.Value() < 360) {
                        angletext = example.localize('reflex angle');
                    } else if (n.Value() === 360) {
                        angletext = example.localize('full angle');
                    } else {
                        angletext = example.localize('empty');
                    }
                    example.control.html('<p>'+angletext+'</p>');
                }
                
            },
            dict: {
                'and': {
                    'en': 'and',
                    'fi': 'ja',
                    'sv': 'och'
                },
                'classification of angles': {
                    'en': 'Classification of angles',
                    'fi': 'Kulmien luokittelu',
                    'sv': 'Olika vinklar'
                },
                'zero angle': {
                    'en': 'Zero angle',
                    'fi': 'Nollakulma',
                    'sv': 'Nollvinkel'
                },
                'right angle': {
                    'en': 'Right angle',
                    'fi': 'Suora kulma',
                    'sv': 'Rät vinkel'
                },
                'straight angle': {
                    'en': 'Straight angle',
                    'fi': 'Oikokulma',
                    'sv': 'Rak vinkel'
                },
                'full angle': {
                    'en': 'Full angle',
                    'fi': 'Täysikulma',
                    'sv': 'Full vinkel'
                },
                'acute angle': {
                    'en': 'Acute angle',
                    'fi': 'Terävä kulma',
                    'sv': 'Spetsig vinkel'
                },
                'obtuse angle': {
                    'en': 'Obtuse angle',
                    'fi': 'Tylppä kulma',
                    'sv': 'Trubbig vinkel'
                },
                'reflex angle': {
                    'en': 'Reflex angle',
                    'fi': 'Kupera kulma',
                    'sv': 'Konvex vinkel'
                },
                'concave angle': {
                    'en': 'concave angle',
                    'fi': 'kovera kulma',
                    'sv': 'konkav vinkel'
                },
                'empty': {
                    'en': '&nbsp;',
                    'fi': '&nbsp;',
                    'sv': '&nbsp;'
                }
            }
        }
    }
    
    Examplepicture.prototype.strings = {
        css:
            '.examplepicture:not(.modalexample) {margin: 0.5em auto 1em auto; width: 310px; border: 1px solid #999; border-radius: 10px; box-shadow: 0 0 5px rgba(0,0,0,0.5);}'
            +'.examplepicture_drawingarea {width: 300px; height: 400px; display: block; border-radius: 5px; border: 1px solid #777; margin: 5px; box-shadow: inset 2px 2px 5px rgba(0,0,0,0.2); background: white;}'
            +'.examplepicture_controlarea {margin: 10px 5px;}'
            +'.examplepicture_header h1 {margin: 0.2em 0.5em 0.5em 0.5em; padding: 0; font-size: 120%; color: black; border-bottom: 1px solid black;}'
            +'.examplepicture_controlarea table {width: 100%; border: none; -moz-box-sizing: border-box; -webkit-box-sizing: border-box; box-sizing: border-box;}'
            +'.examplepicture_controlarea table tr {border: none;}'
            +'.examplepicture_controlarea table td {border: none;}'
            +'.examplepicture_controlarea p {margin: 0 0.5em;text-shadow:1px 1px 1px white;}'
            +'.examplepicture:not(.modalexample) {background: rgb(255,255,255); background: -moz-linear-gradient(top,  rgba(255,255,255,1) 0%, rgba(229,229,229,1) 100%);'
            +'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(255,255,255,1)), color-stop(100%,rgba(229,229,229,1)));'
            +'background: -webkit-linear-gradient(top,  rgba(255,255,255,1) 0%,rgba(229,229,229,1) 100%);'
            +'background: -o-linear-gradient(top,  rgba(255,255,255,1) 0%,rgba(229,229,229,1) 100%);'
            +'background: -ms-linear-gradient(top,  rgba(255,255,255,1) 0%,rgba(229,229,229,1) 100%);'
            +'background: linear-gradient(to bottom,  rgba(255,255,255,1) 0%,rgba(229,229,229,1) 100%);'
            +'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#ffffff\', endColorstr=\'#e5e5e5\',GradientType=0 );}'
    }
     /******
     * Elementinfo for elementsets
     ******/
    Examplepicture.elementinfo = {
        type: 'emathexamplepic',
        elementtype: ['jsexample'],
        jquery: 'emathexamplepic',
        name: 'Emathexamplepic',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-emathexamplepic mini-icon-emathexamplepic"></svg>',
        description: {
            en: 'emathexamplepic',
            fi: 'emathexamplepic'
        },
        classes: ['viewonly']
    }
    // Register multichoice as an element to the elementset and to the elementpanel.
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', Examplepicture.elementinfo);
    }
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Examplepicture.elementinfo);
    }
})(jQuery);

if (typeof(config) !== 'undefined' && typeof(config.macros) !== 'undefined'){
    // Create macro for TiddlyWiki
    config.macros.examplepic = {
        /******************************
        * Show examplepic
        ******************************/
        handler: function (place, macroName, params, wikifier, paramString, tiddler)
        {
            if (params.length < 1){
                wikify('Missing examplepic.', place, null, tiddler);
                return false;
            }
            var bookpage = {'pageOne': 0, 'pageTwo': 1}[jQuery(place).parents('.bookpage').attr('id')];
            var examplepic = params[0];
            var modal = (params[1] === 'modal');
            var textparams = (params[2] || '').split('##');
            var buttontext = textparams[0];
            var booklang = textparams[1] || jQuery(place).attr('lang') || EbookPages[bookpage].currentlang || jQuery(place).parents('[lang]').attr('lang');
            var data = DataTiddler.getData(tiddler, 'examplepic', {});
            var exampleobj = {example: {}, lang: booklang};
            if (data[examplepic]){
                exampleobj.example.title = data[examplepic].title;
                exampleobj.example.dict = data[examplepic].dict;
                exampleobj.example.func = new Function(data[examplepic].func);
            } else {
                exampleobj = {example: examplepic, lang: booklang};
            }
            examplepic = exampleobj;
            if (modal){
                var expictext = '<html><div class="emathexpic-place" lang="'+booklang+'"><a href="javascript:;" class="emathex-button">'+buttontext+'</a></div></html>';
                wikify(expictext, place, null, tiddler);
                var emexpicbutton = jQuery(place).find('a.emathex-button').last();
                emexpicbutton.button({
                    icons: {
                        primary: 'em-icon-expic'
                    }
                }).click(function(){
                    var backdiv = jQuery('<div class="emathexpic-back"></div>');
                    backdiv.appendTo('body');
                    backdiv.html('<div class="emexpic-wrapper"></div><a href="javascript:;">x</a>');
                    backdiv.css({'position': 'fixed', 'top': '0', 'right': '0', 'bottom': '0', 'left': '0', 'background-color': 'rgba(255,255,255,0.9)', 'z-index': '201'});
                    var closebutton = backdiv.find('a');
                    closebutton.click(function(){
                        backdiv.remove();
                    }).css({'display': 'inline-block', 'position': 'absolute', 'top': '20px', 'right': '20px', 'height': '1.5em', 'width': '1.5em', 'border-radius': '0.4em', 'border': '2px solid black', 'background': '#f00', 'color': 'white', 'text-align': 'center', 'vertical-align': 'middle', 'font-weight': 'bold', 'box-shadow': 'inset 2px 2px 3px rgba(255,255,255,0.5), inset -2px -2px 3px rgba(0,0,0,0.5)'});
                    var content = backdiv.find('.emexpic-wrapper');
                    content.css({'max-width': '1000px', 'margin': '5em auto 0 auto'})
                        .emathexamplepic(examplepic);
                });
            } else {
                var expictext = '<html><div class="emathexpic" lang="'+booklang+'">\n</div></html>';
                wikify(expictext, place, null, tiddler);
                jQuery(place).find('.emathexpic').last().emathexamplepic(examplepic);
            }
            return true;
        }
    }
}

//}}}