/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * emathquiz
     * @param options
     */

    $.fn.emathquiz = function(options){
        if (methods[options]){
            return methods[options].apply( this, Array.prototype.slice.call( arguments, 1));
        } else if (typeof(options) === 'object' || !options) {
            return methods.init.apply(this, arguments);
        } else {
            $.error( 'Method ' +  method + ' does not exist on emathquiz' );
            return this;
        }
    }

    var methods = {
        init: function( options ){
            var settings = $.extend(true, {}, options);

            return this.each(function(){
                var emquiz = new EmathQuiz(this, settings);
                emquiz.init();
            });
        },
        score: function(){
            // Return the scores as {correct: <number>, total: <number>}
            return this.eq(0).data('emathquiz');
        }
    };


    /**
     * Helper functions
     */
    
    /**
     * Sanitize the input text
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    var EmathQuiz = function(place, settings){
        // Create EmathQuiz-object. Put empty html-elements in the place and
        // find some elements that are needed later.
        window.testilogit = {};
        settings = $.extend({}, this.defaults, settings);
        testilogit.emq = this;
        this.place = $(place);
        this.settings = settings;
        this.lang = this.settings.lang || this.place.closest('[lang]').eq(0).attr('lang') || 'en';
        this.dict = {};
        // Keep the score
        this.totalquestions = this.settings.total;
        this.answered = this.settings.answered;
        this.correctanswers = this.settings.score;
        this.results = this.settings.results;
        this.place.html(this.template);
        this.wrapper = this.place.find('.emathquizwrapper');
        //this.wrapper.addClass('emquiz-gradblue');
        if (this.settings.overlay){
            // If we use quiz as overlay instead of inline.
            this.wrapper.addClass('emquizoverlay')
        }
        // Place for title
        this.titlearea = this.place.find('.emquiz-title');
        // Place for picture
        this.picarea = this.place.find('.emquiz-picarea');
        this.quizid = -1;
        while ($('#emquiz-picarea-'+(++this.quizid)).length > 0){};
        this.picarea.attr('id','emquiz-picarea-'+this.quizid);
        // Place for question text
        this.qtextarea = this.place.find('.emquiz-question-textarea');
        // Place for answer
        this.qanswersarea = this.place.find('.emquiz-question-answersarea');
        // Place for feedback
        this.feedbackarea = this.place.find('.emquiz-feedbackarea');
        // Place for scorebar
        this.scorebar = this.place.find('.emquiz-scorebar');
        // Place for controls like "answer" and "next" buttons and "scores"
        this.controlarea = this.place.find('.emquiz-controlarea');
        this.controlarea.html('<div class="emquiz-resultarea">'+escapeHTML(this.correctanswers)+'/'+escapeHTML(this.totalquestions)+'</div>'
            +'<a href="javascript:;" class="emquiz-answerbutton emquiz-button">OK</a>'
            +'<a href="javascript:;" class="emquiz-nextbutton emquiz-button">&raquo;</a>');
        this.resultarea = this.controlarea.find('.emquiz-resultarea');
        
        // Put css in use.
        $('head style#emathquizcss').remove();
        $('head').append('<style id="emathquizcss" type="text/css">'+this.strings.css+'</style>')
    }
    
    EmathQuiz.prototype.init = function(){
        // Init the quiz from options given.
        var emquiz = this;
        // Put the title on its place
        this.title = this.settings.title || '';
        this.titlearea.html('<h1>'+escapeHTML(this.title)+'</h1>');
        
        // Show scoreblocks
        if (this.answered === this.results.length) {
            for (var i = 0, len = this.results.length; i < len; i++) {
                this.addScoreblock(this.results[i]);
            };
        } else {
            for (var i = 0, len = this.answered; i < len; i++) {
                this.addScoreblock(i < this.correctanswers);
            };
        };
        // Check the type of given action function: 'function', 'string' or 'array' of functions.
        if (typeof(this.settings.func) === 'function'){
            // Function is given as function
            this.nextquestion = this.settings.func;
        } else if (typeof(this.settings.func) === 'string'){
            // Function is given as a string of javascript code
            try {
                this.nextquestion = new Function(this.settings.func);
            } catch (err){
                $.error('Emathquiz: error(Code:W8827) in nextquestion function.\n'+err);
            }
        } else if (typeof(this.settings.func) === 'object' && typeof(this.settings.func.length) === 'number'){
            // Functions are given as an array of functions. Either strings or functions.
            var nfuncs = [];
            for (var i = 0; i < this.settings.func.length; i++){
                if (typeof(this.settings.func[i]) === 'function'){
                    nfuncs.push(this.settings.func[i]);
                } else if (typeof(this.settings.func[i]) === 'string'){
                    try {
                        nfuncs.push(new Function(this.settings.func[i]));
                    } catch (err){
                        $.error('Emathquiz: error(Code:D8832) in nextquestion function.\n'+err);
                    }
                }
            }
            this.settings.func = nfuncs;
            this.nextquestion = function(){
                return emquiz.settings.func[Math.floor(Math.random()*emquiz.settings.func.length)].call(this);
            }
        } else {
            // No function was given. Use default example question.
            this.nextquestion = this.defaultQuestion;
        }
        
        // Init some events. ('Answer' and 'Next' buttons)
        this.initEvents();
        
        // Start the next (first) question.
        this.update();
    }
    
    EmathQuiz.prototype.initEvents = function(){
        // Init some events. ('Answer' and 'Next' buttons)
        var emquiz = this;
        this.place.on('click', 'a.emquiz-answerbutton', function(event) {
            event.stopPropagation();
            event.preventDefault();
            emquiz.checkAnswer();
        });
        
        this.place.on('click', 'a.emquiz-nextbutton', function(event) {
            event.stopPropagation();
            event.preventDefault();
            emquiz.update();
        });
        
        this.place.on('keydown', '.emquiz-asking .emquiz-question-answersarea .mathquill-editable', function(event) {
            if (event.which === 13) {
                event.stopPropagation();
                event.preventDefault();
                emquiz.checkAnswer();
            };
        });
        
        this.place.on('change', '.emquiz-asking .emquiz-question-answersarea input[type="radio"]', function(event) {
            event.stopPropagation();
            event.preventDefault();
            if (emquiz.settings.autonext) {
                if (emquiz.answertimeout) {
                    clearTimeout(emquiz.answertimeout);
                };
                emquiz.answertimeout = setTimeout(function() {
                    emquiz.checkAnswer();
                }, 1000);
            };
        });
        
        
    }
    
    EmathQuiz.prototype.update = function() {
        if (this.nexttimeout) {
            clearTimeout(this.nexttimeout);
        };
        if (this.answered < this.totalquestions) {
            this.updateView();
        } else {
            this.wrapper.addClass('emquiz-finished').removeClass('emquiz-asking');
        };
    };
    
    EmathQuiz.prototype.updateView = function(){
        // Start next round by generating next question and using it.
        var emquiz = this;
        this.wrapper.removeClass('emquiz-correctanswer').addClass('emquiz-asking');
        this.controlarea.find('a.emquiz-answerbutton').show()
            .end().find('a.emquiz-nextbutton').hide();
        // Get data for next question.
        this.question = this.nextquestion();
        // Set the style
        this.question.qstyle = this.question.qstyle || 'default';
        this.wrapper.attr('data-qstyle', this.question.qstyle).attr('data-qtype', this.question.qtype);
        // Clear the feedback
        this.feedbackarea.empty().removeClass('emquiz-feedback-correct emquiz-feedback-wrong');
        // Show the question. Show math with mathquill.
        this.qtextarea.html(sanitize(this.question.qtext).replace(/\\\(/g, '<span class="mathquill-embedded-latex">').replace(/\\\)/g, '</span>'))
            .find('.mathquill-embedded-latex').mathquill('embedded-latex');
            
        // Init the answering section according to the type of question (multichoice, shortanswer)
        var answers = '';
        switch (this.question.qtype){
            case 'multichoice':
                // Show multichoice answers
                answers = ['<ul class="emquiz-multichoice">'];
                for (var i = 0; i < this.question.qanswers.length; i++){
                    answers.push(
                        '<li><label><input type="radio" name="emquiz-'+escapeHTML(this.quizid) +'-multichoice" value="'+i+'" />',
                        '<span class="emquiz-multichoice-text">' + escapeHTML(this.question.qanswers[i])
                            .replace(/\\\(/, '<span class="mathquill-embedded-latex">')
                            .replace(/\\\)/, '</span>')+'</span></label></li>'
                    );
                }
                answers.push('</ul>');
                answers = answers.join('');
                break;
            case 'shortanswer':
                // Show the answering field for short answer.
                answers = '<div class="emquiz-shortanswer">';
                answers += '<span class="emquiz-shortanswer-useranswer mathquill-editable"></span>';
                answers += '</div>';
                break;
            case 'image':
                // Show (don't show) the answering widgets for image quiz.
                break;
            default:
                break;
        }
        
        // Show math with mathquill
        this.qanswersarea.html(answers).find('.mathquill-editable').mathquill('editable').end().find('.mathquill-embedded-latex').mathquill();

        // Show or hide picarea.
        if (this.question.haspicture){
            this.wrapper.addClass('emquiz-haspicture');
        } else {
            this.wrapper.removeClass('emquiz-haspicture');
        }
        // Draw the picture with jsxgraph+jessiescript, if this.question.jessie exists.
        if (this.question.jessie){
            this.picarea.empty();
            this.construction = [];
            var jxgoptions = JXG.Options;
            // Some default settings for picture.
            JXG.Options = JXG.deepCopy(JXG.Options, {
                showNavigation: false,
                text: {
                    fontSize: 20
                },
                angle: {
                    radius: 1.1,
                    label: {
                        strokeColor: '#ff0000'
                    },
                    orthoSensitivity: 0.5
                },
                point: {
                    fixed: true,
                    size: 1.3
                },
                glider: {
                    fixed: false,
                    color: '#f00',
                    size: 5,
                    strokeColor: 'black',
                    strokeWidth: 3,
                    face: 'x'
                },
                line: {
                    strokeWidth: 3,
                    shadow: true
                }
             });
            // Init the board
            if (this.board){
                JXG.JSXGraph.freeBoard(this.board);
            }
            this.board = JXG.JSXGraph.initBoard("emquiz-picarea-"+this.quizid, {
                boundingbox: this.question.jessiebb || [0,10,10,0],
                keepaspectratio: true,
                grid: true,
                showCopyright: false,
                showNavigation: false,
                pan: false,
                zoom: false
            });
            this.construction = [];
            this.board.suspendUpdate();
            this.construction['question'] = this.board.construct(this.question.jessie);
            if (this.question.lblabels){
                for (var i = 0; i < this.question.lblabels.length; i++){
                    this.board.elementsByName[this.question.lblabels[i]].label.content.setProperty({offset: [-20,-20]});
                }
            }
            if (this.question.ltlabels){
                for (var i = 0; i < this.question.ltlabels.length; i++){
                    this.board.elementsByName[this.question.ltlabels[i]].label.content.setProperty({offset: [-20,20]});
                }
            }
            if (this.question.rblabels){
                for (var i = 0; i < this.question.rblabels.length; i++){
                    this.board.elementsByName[this.question.rblabels[i]].label.content.setProperty({offset: [10,-20]});
                }
            }
            if (this.question.rtlabels){
                for (var i = 0; i < this.question.rtlabels.length; i++){
                    this.board.elementsByName[this.question.rtlabels[i]].label.content.setProperty({offset: [10,20]});
                }
            }
            if (this.question.jessiestyles && this.question.jessiestyles['question']){
                this.setJsxProperty('question');
            }
            if (this.question.jessielabels && this.question.jessielabels['question']){
                this.setJsxLabels('question');
            }
            this.board.unsuspendUpdate();
            testilogit.board = this.board;
            testilogit.construction = this.construction;
            JXG.Options = jxgoptions;
        }
    }
    
    EmathQuiz.prototype.setJsxProperty = function(elementset, ruleset){
    /**
     * @param String elementset - the name of construction set of jsxelements
     * @param String ruleset - the name of set of style rules for jsxelements. If null, then use elementset.
     */
        ruleset = ruleset || elementset;
        for (var elems in this.question.jessiestyles[elementset]){
            if (typeof(this.construction[elementset][elems].length) === 'undefined'){
                this.construction[elementset][elems].setProperty(this.question.jessiestyles[ruleset][elems]);
            } else {
                for (var i = 0, length = this.construction[elementset][elems].length; i < length; i++){
                    this.construction[elementset][elems][i].setProperty(this.question.jessiestyles[ruleset][elems]);
                }
            }
        }
    }
    
    EmathQuiz.prototype.setJsxLabels = function(elementset, labelset){
    /**
     * @param String elementset - the name of construction set of jsxelements
     * @param String labelset - the name of set of labels for jsxelements. If null, then use elementset.
     */
        labelset = labelset || elementset;
        var labels = this.question.jessielabels[elementset];
        var construction = this.construction[elementset];
        for (var elem in labels){
            if (construction[elem] && typeof(construction[elem].length) === 'undefined'){
                this.construction[elementset][elem].label.content.setText(labels[elem](construction));
            }
        }
    }
    
    EmathQuiz.prototype.localize = function(text){
        var result;
        if (this.dict[text]){
            if (this.dict[text][this.lang]){
                result = this.dict[text][this.lang];
            }
            result = result || this.dict[text]['en'];
        }
        result = result || text;
        return result;
    }
    
    EmathQuiz.prototype.defaultQuestion = function(){
        // "Dummy" default function for questions, if real question function is not given.
        // Returns randomly either multichoice question or short answer question.
        var questions = [
            {
                qtype: 'multichoice',
                qtext: '<p>Mikä on vastaus?</p>',
                qanswers: [
                    'Eka \\(\\beta\\)',
                    'Toka \\(\\frac{1}{2}\\)',
                    'Kolmas',
                    'Neljäs: Tähän oikein pitkä vastaus. Vaikka jotain lorem ipsumia.'
                ],
                correct: '2',
                jessie: 'A(1,2); B(9,3); C(3,8); [A B] nolabel; [BC] nolabel; [C A] nolabel; 45\u00b0=<(B,A,C)',
                lblabels: ['A']
            },
            {
                qtype: 'shortanswer',
                qtext: '<p>Kuinka paljon?</p>',
                correct: [2.5, 3.5],
                check: function(answer, corrects){
                    var result = false;
                    var replacers = [
                        [/\\frac{([^{}]+)}{([^{}]+)}/ig, '(($1)/($2))'],
                        [/\\sqrt{([^{}]+)}/ig, 'Math.sqrt($1)'],
                        [/\\cdot/ig, '*'],
                        [/\\left\(/ig, '('],
                        [/\\right\)/ig, ')'],
                        [/((?:[0-9]+)|(?:\([^\(\)]\)))\^((?:[0-9])|(?:{[0-9]+}))/ig, 'pow($1, $2)'],
                        [/,/ig, '.']
                    ];
                    var oldanswer = '';
                    while (answer !== oldanswer){
                        oldanswer = answer;
                        for (var i = 0; i < replacers.length; i++){
                            answer = answer.replace(replacers[i][0], replacers[i][1]);
                        }
                    }
                    try {
                        var corrfunc = new Function('return ('+ answer +');');
                        for (var i = 0; i < corrects.length; i++){
                            result = result || (corrects[i] == corrfunc());
                        }
                    } catch (err){
                    }
                    return result;
                },
                jessie: 'A(1,2); B(9,3); C(3,8); [A B] nolabel; [BC] nolabel; [C A] nolabel; 45\u00b0=<(B,A,C)',
                lblabels: ['A']
            }
        ]
        return questions[Math.floor(Math.random()*questions.length)];
    }
    
    EmathQuiz.prototype.checkAnswer = function(){
        // Check the answer and show the correct one before continuing
        // to the next one.
        if (this.answertimeout) {
            clearTimeout(this.answertimeout);
        };
        this.wrapper.removeClass('emquiz-asking').addClass('emquiz-correctanswer');
        this.controlarea.find('a.emquiz-answerbutton').hide()
            .end().find('a.emquiz-nextbutton').show();
        var correct = false;
        switch (this.question.qtype){
            case 'multichoice':
                this.answer = parseInt(this.qanswersarea.find(':checked').val());
                this.qanswersarea.find('li').eq(parseInt(this.question.correct)).addClass('emquiz-multichoice-correct');
                this.qanswersarea.find('li').eq(parseInt(this.answer)).addClass('emquiz-multichoice-useranswer');
                correct = (this.question.correct === this.answer);
                break;
            case 'shortanswer':
                this.answer = this.qanswersarea.find('.emquiz-shortanswer-useranswer').mathquill('latex') || '';
                var useranswer = this.qanswersarea.find('.emquiz-shortanswer-useranswer');
                useranswer.mathquill('revert').html(this.answer).mathquill('embedded-latex');
                if (typeof(this.question.check) === 'function'){
                    correct = this.question.check(this.answer, this.question.correct);
                } else {
                    for (var i = 0; i < this.question.correct.length; i++){
                        correct = correct || (this.answer == this.question.correct[i]);
                    }
                }
                if (correct){
                    this.qanswersarea.find('.emquiz-shortanswer-useranswer').addClass('emquiz-shortanswer-correct');
                } else {
                    useranswer.after('<div class="emquiz-shortanswer-correct">'+escapeHTML(this.question.correct[0])+'</div>');
                }
                break;
            case 'image':
                this.answer = this.question.getAnswer(this.board);
                correct = this.question.check(this.answer, this.question.correct);
                if (!(correct && this.question.jessiehidecorrect)){
                    this.board.suspendUpdate();
                    this.construction['correct'] = this.board.construct(this.question.jessiecorrect);
                    if (this.question.jessiestyles){
                        this.setJsxProperty('question', 'questionafter');
                        this.setJsxProperty('correct');
                    }
                    this.board.unsuspendUpdate();
                }
                break
            default:
                this.answer = '';
                break;
        }
        //this.totalquestions++;
        this.answered++;
        if (correct){
            this.correctanswers++;
        };
        this.results.push(correct);
        this.addScoreblock(correct);
        if (this.question.feedback){
            this.feedbackarea.addClass(correct ? 'emquiz-feedback-correct' : 'emquiz-feedback-wrong')
                .html(sanitize(this.question.feedback)
                      .replace(/\\\(/g, '<span class="mathquill-embedded-latex">')
                      .replace(/\\\)/g, '</span>')
                ).find('.mathquill-embedded-latex').mathquill('embedded-latex');
        }
        this.resultarea.html(escapeHTML(this.correctanswers) +'/'+ escapeHTML(this.totalquestions));
        this.controlarea.find('a.emquiz-nextbutton').focus();
        this.place.data('emathquiz', {correct: this.correctanswers, answered: this.answered, total: this.totalquestions});
        this.place.trigger('startsolving');
        this.place.trigger('emathquiz_changed', [{correct: this.correctanswers, answered: this.answered, total: this.totalquestions, results: this.results}]);
        if (this.answered === this.totalquestions) {
            this.update();
        } else if (this.settings.autonext) {
            var emquiz = this;
            //clearTimeout(this.nexttimeout);
            this.nexttimeout = setTimeout(function(){
                emquiz.update();
            }, (correct ? 1000 : 5000));
        };
    };
    
    EmathQuiz.prototype.addScoreblock = function(correct) {
        this.scorebar.append('<span class="emquiz-scoreblock emquiz-scoreblock-' + (correct ? 'correct' : 'wrong') + '" style="width: '+(Math.round(1000 / this.totalquestions) / 10).toFixed(1)+'%"></span>');
    };
    
    EmathQuiz.prototype.strings = {
        // CSS style rules.
        css: [
            '.emathquizwrapper {border: 1px solid #777; border-radius: 1em; box-shadow: 5px 5px 5px rgba(0,0,0,0.5), inset 2px 2px 4px rgba(255,255,255,0.8), inset -2px -2px 4px black; min-height: 5em; margin: 1em; padding: 0.5em;}',
            '.emathquizwrapper.emquizoverlay {position: fixed; top: 5em; width: 90%; max-width: 1000px; margin: 0 auto; left: 5%; right: 5%; z-index: 200;}',
            '.emathquizwrapper:after {content: " "; clear: both; display: block;}',
            '.emquiz-title h1 {text-align: center; border: none; margin: 0.2em 0.5em;}',
            '.emquiz-picarea {border: 1px solid #777; border-radius: 5px; background: white; box-shadow: inset 5px 5px 5px rgba(0,0,0,0.2); margin: 5px; display: none;}',
            '.emquiz-questionarea {display: flex; flex-flow: row nowrap;}',
            '[data-qstyle="default"] .emquiz-picarea {width: 300px; height: 300px;}',
            '[data-qstyle="bigpic"] .emquiz-picarea {height: 300px;}',
            '[data-qstyle="widepic"] .emquiz-picarea {height: 150px;}',
            '[data-qstyle="bigpic"] .emquiz-questionarea, [data-qstyle="widepic"] .emquiz-questionarea {display: block;}',
            '.emquiz-haspicture .emquiz-picarea {display: block;}',
            '.emquiz-question {margin: 0px 5px 5px 5px; padding-top: 0px; flex-grow: 1;}',
            //'.emquiz-haspicture[data-qstyle="widepic"] .emquiz-question, .emquiz-haspicture[data-qstyle="bigpic"] .emquiz-question {margin: 6px 5px 5px 5px; padding-top: 5px;}',
            //'.emquiz-haspicture .emquiz-question {margin: 6px 5px 5px 315px;}',
            '.emquiz-question-textarea, .emquiz-question-answersarea, .emquiz-feedbackarea {border: 1px solid #777; border-radius: 5px; color: black; background-color: rgba(255,255,255,0.8); padding: 0 1em; margin: 0 0 1em 0; min-height: 2em; box-shadow: inset 4px 4px 6px rgba(0,0,0,0.3);}',
            '[data-qtype="image"] .emquiz-question-answersarea {display: none;}',
            '.emquiz-feedbackarea {visibility: hidden;}',
            '.emquiz-feedbackarea.emquiz-feedback-correct {background-color: rgb(248,255,232); visibility: visible;}',
            '.emquiz-feedbackarea.emquiz-feedback-wrong {background-color: rgb(254,187,187); visibility: visible;}',
            '.emquiz-feedbackarea p.corrfeedback, .emquiz-feedbackarea p.wrongfeedback {display: none;}',
            '.emquiz-feedbackarea.emquiz-feedback-correct p.corrfeedback, .emquiz-feedbackarea.emquiz-feedback-wrong p.wrongfeedback {display: block;}',
            
            'ul.emquiz-multichoice {list-style: none; padding-left: 1em;}',
            'ul.emquiz-multichoice li {min-height: 2em; padding: 0; margin: 0.5em 0;}',
            'ul.emquiz-multichoice li label {cursor: pointer; display: block; padding: 0.5em 0}',
            'ul.emquiz-multichoice li label:hover {background-color: rgba(255,255,255,0.3);}',
            'ul.emquiz-multichoice li label .emquiz-multichoice-text {padding-left: 1em; vertical-align: baseline;}',
            '.emquiz-correctanswer ul.emquiz-multichoice input[type="radio"] {display: none;}',
            '.emquiz-correctanswer ul.emquiz-multichoice li.emquiz-multichoice-correct, .emquiz-correctanswer .emquiz-shortanswer-correct, .emquiz-correctanswer ul.emquiz-multichoice li.emquiz-multichoice-correct.emquiz-multichoice-useranswer',
                '{border-radius: 10px; border: 2px solid #a9c314; box-shadow: 0 0 2px white;',
                'background: rgb(248,255,232);',
                'background: -moz-linear-gradient(top,  rgba(248,255,232,1) 0%, rgba(227,245,171,1) 33%, rgba(183,223,45,1) 100%);',
                'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(248,255,232,1)), color-stop(33%,rgba(227,245,171,1)), color-stop(100%,rgba(183,223,45,1)));',
                'background: -webkit-linear-gradient(top,  rgba(248,255,232,1) 0%,rgba(227,245,171,1) 33%,rgba(183,223,45,1) 100%);',
                'background: -o-linear-gradient(top,  rgba(248,255,232,1) 0%,rgba(227,245,171,1) 33%,rgba(183,223,45,1) 100%);',
                'background: -ms-linear-gradient(top,  rgba(248,255,232,1) 0%,rgba(227,245,171,1) 33%,rgba(183,223,45,1) 100%);',
                'background: linear-gradient(to bottom,  rgba(248,255,232,1) 0%,rgba(227,245,171,1) 33%,rgba(183,223,45,1) 100%);',
                'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#f8ffe8\', endColorstr=\'#b7df2d\',GradientType=0 );}',
            '.emquiz-correctanswer ul.emquiz-multichoice li.emquiz-multichoice-useranswer {border-radius: 10px; border: 2px solid #e81818; box-shadow: 0 0 2px white;',
                'background: rgb(254,187,187);',
                'background: -moz-linear-gradient(top,  rgba(254,187,187,1) 0%, rgba(254,144,144,1) 45%, rgba(255,92,92,1) 100%);',
                'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(254,187,187,1)), color-stop(45%,rgba(254,144,144,1)), color-stop(100%,rgba(255,92,92,1)));',
                'background: -webkit-linear-gradient(top,  rgba(254,187,187,1) 0%,rgba(254,144,144,1) 45%,rgba(255,92,92,1) 100%);',
                'background: -o-linear-gradient(top,  rgba(254,187,187,1) 0%,rgba(254,144,144,1) 45%,rgba(255,92,92,1) 100%);',
                'background: -ms-linear-gradient(top,  rgba(254,187,187,1) 0%,rgba(254,144,144,1) 45%,rgba(255,92,92,1) 100%);',
                'background: linear-gradient(to bottom,  rgba(254,187,187,1) 0%,rgba(254,144,144,1) 45%,rgba(255,92,92,1) 100%);',
                'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#febbbb\', endColorstr=\'#ff5c5c\',GradientType=0 );}',

            '.emquiz-shortanswer {padding: 1em; min-height: 2em;}',            
            '.emquiz-shortanswer .emquiz-shortanswer-correct {display: block; padding: 0.3em 1em; margin: 0.3em 0;}',
            '.emquiz-shortanswer .emquiz-shortanswer-useranswer {display: block; margin: 0.3em; background-color: white; padding: 0.2em;}',
            
            '.emquiz-controlarea {display: flex; flex-flow: row nowrap; justify-content: space-between;}',
            '.emquiz-resultarea {color: black; background-color: rgba(255,255,255,0.8); border: 1px solid #777; border-radius: 5px; min-width: 8em; padding: 0.5em; text-align: center; box-shadow: inset 4px 4px 6px rgba(0,0,0,0.3);}',
            '.emquiz-button, .emathquizwrapper a.emquiz-button:hover {padding: 0.5em 1em; border-radius: 0.3em; text-decoration: none; margin: 0 0.5em; min-width: 6em;',
                'border: 1px solid black; text-align: center; color: black; font-weight: bold; text-shadow: 1px 1px 1px white;',
                'background: rgb(254,252,234);',
                'background: -moz-linear-gradient(top,  rgba(254,252,234,1) 0%, rgba(241,218,54,1) 100%);',
                'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(254,252,234,1)), color-stop(100%,rgba(241,218,54,1)));',
                'background: -webkit-linear-gradient(top,  rgba(254,252,234,1) 0%,rgba(241,218,54,1) 100%);',
                'background: -o-linear-gradient(top,  rgba(254,252,234,1) 0%,rgba(241,218,54,1) 100%);',
                'background: -ms-linear-gradient(top,  rgba(254,252,234,1) 0%,rgba(241,218,54,1) 100%);',
                'background: linear-gradient(to bottom,  rgba(254,252,234,1) 0%,rgba(241,218,54,1) 100%);',
                'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#fefcea\', endColorstr=\'#f1da36\',GradientType=0 );}',
            '.emquiz-button:hover {box-shadow: 0 0 3px white; color: black;}',
            '.emquiz-nextbutton {display: none;}',
            '.emquiz-correctanswer .emquiz-nextbutton {display: inline-block;}',
            '.emquiz-correctanswer .emquiz-answerbutton {display: none;}',
            
            '.emquiz-gradgray {background: rgb(179,190,173);',
                'background: -moz-linear-gradient(top,  rgba(179,190,173,1) 0%, rgba(223,229,215,1) 60%, rgba(252,255,244,1) 100%);',
                'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(179,190,173,1)), color-stop(60%,rgba(223,229,215,1)), color-stop(100%,rgba(252,255,244,1)));',
                'background: -webkit-linear-gradient(top,  rgba(179,190,173,1) 0%,rgba(223,229,215,1) 60%,rgba(252,255,244,1) 100%);',
                'background: -o-linear-gradient(top,  rgba(179,190,173,1) 0%,rgba(223,229,215,1) 60%,rgba(252,255,244,1) 100%);',
                'background: -ms-linear-gradient(top,  rgba(179,190,173,1) 0%,rgba(223,229,215,1) 60%,rgba(252,255,244,1) 100%);',
                'background: linear-gradient(to bottom,  rgba(179,190,173,1) 0%,rgba(223,229,215,1) 60%,rgba(252,255,244,1) 100%);',
                'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#b3bead\', endColorstr=\'#fcfff4\',GradientType=0 );}',
            '.emquiz-gradblue {background: rgb(135,224,253);,',
                'background: -moz-linear-gradient(top,  rgba(135,224,253,1) 0%, rgba(83,203,241,1) 40%, rgba(5,171,224,1) 100%);',
                'background: -webkit-gradient(linear, left top, left bottom, color-stop(0%,rgba(135,224,253,1)), color-stop(40%,rgba(83,203,241,1)), color-stop(100%,rgba(5,171,224,1)));',
                'background: -webkit-linear-gradient(top,  rgba(135,224,253,1) 0%,rgba(83,203,241,1) 40%,rgba(5,171,224,1) 100%);',
                'background: -o-linear-gradient(top,  rgba(135,224,253,1) 0%,rgba(83,203,241,1) 40%,rgba(5,171,224,1) 100%);',
                'background: -ms-linear-gradient(top,  rgba(135,224,253,1) 0%,rgba(83,203,241,1) 40%,rgba(5,171,224,1) 100%);',
                'background: linear-gradient(to bottom,  rgba(135,224,253,1) 0%,rgba(83,203,241,1) 40%,rgba(5,171,224,1) 100%);',
                'filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\'#87e0fd\', endColorstr=\'#05abe0\',GradientType=0 );}',
            // Scorebar
            '.emquiz-scorebar .emquiz-scoreblock {display: inline-block; height: 10px; box-sizing: border-box; border: 1px solid black;}',
            '.emquiz-scorebar .emquiz-scoreblock-correct {background: #B9E032;}',
            '.emquiz-scorebar .emquiz-scoreblock-wrong {background: #ef5959;}',
            // Finish
            '.emquiz-finished a.emquiz-button {display: none;}'
        ].join('\n')
    }
    
    EmathQuiz.prototype.template = [
        '<div class="emathquizwrapper ffwidget-background">',
        '<div class="emquiz-title"></div>',
        '<div class="emquiz-questionarea">',
        '    <div class="emquiz-picarea"></div>',
        '    <div class="emquiz-question">',
        '        <div class="emquiz-question-textarea"></div>',
        '        <div class="emquiz-question-answersarea"></div>',
        '    </div>',
        '</div>',
        '<div class="emquiz-feedbackarea"></div>',
        '<div class="emquiz-scorebar"></div>',
        '<div class="emquiz-controlarea"></div>',
        '</div>'
    ].join('\n');
    
    EmathQuiz.prototype.defaults = {
        autonext: false,
        overlay: false,
        score: 0,
        title: 'Quiz',
        total: 10,
        answered: 0,
        results: []
    }

})(jQuery)


