/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Dependency problem in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /******
     * Embedded element is wrapper to embed webcontent.
     * @requires 
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var EmbeddedElement = function(place, options){
        this.place = $(place);
        this.place.addClass('embeddedelement');
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init element
     * @param {Object} options - initing data of the element
     ******/
    EmbeddedElement.prototype.init = function(options){
        options = $.extend(true,{}, EmbeddedElement.defaults, options);
        this.settings = $.extend(true, {}, EmbeddedElement.defaults.settings, options.settings);
        this.setMode(options.settings.mode);
        this.data = options.data;
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        }
    }
    
    /******
     * Set style-tag if needed.
     ******/
    EmbeddedElement.prototype.setStyles = function(){
        if ($('head style#embeddedelement-style').length === 0) {
            $('head').append('<style id="embeddedelement-style" type="text/css">'+EmbeddedElement.styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    EmbeddedElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = EmbeddedElement.modes[mode] || EmbeddedElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    EmbeddedElement.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the element in different modes
     ******/
    EmbeddedElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
            this.place.attr('uilang', this.settings.uilang);
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        }
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    EmbeddedElement.prototype.view = function(){
        var html = this.__format(EmbeddedElement.templates.viewhtml,{'openinbrowser':ebooklocalizer.localize('embeddedelement:openinbrowser', this.settings.uilang)});
        this.place.html(html);
        if(this.data.elementTitle){
            this.place.find('h3.embeddedelement-title').text(this.data.elementTitle);
        }else{
            this.place.find('h3.embeddedelement-title').remove();
        }
        if(this.data.embedUrl){
            this.place.find('.embeddedelement-embedopeninbrowser a').attr('href',this.data.embedUrl);
        }else{
            this.place.find('.embeddedelement-embedopeninbrowser').remove();
        }
        this.place.find('.embeddedelement-embedviewwindow').trigger("showEmbedded");
    }
    
    /******
     * Show the element in editable mode.
     ******/
    EmbeddedElement.prototype.edit = function(){
        var html = this.__format(EmbeddedElement.templates.edithtml,{'titleplaceholder':ebooklocalizer.localize('embeddedelement:titleplaceholder', this.settings.uilang),'placeholder':ebooklocalizer.localize('embeddedelement:placeholder', this.settings.uilang),'resizeinfo':ebooklocalizer.localize('embeddedelement:resizeinfo', this.settings.uilang),'previewbox':ebooklocalizer.localize('embeddedelement:previewbox', this.settings.uilang),'previewwidth':ebooklocalizer.localize('embeddedelement:previewwidth', this.settings.uilang),'previewheight':ebooklocalizer.localize('embeddedelement:previewheight', this.settings.uilang)});
        this.place.html(html);
        if(this.data.elementTitle)this.place.find('input.embeddedelement-title').val(this.data.elementTitle);
        if(this.data.inputData)this.place.find('input.embeddedelement-editarea').val(this.data.inputData);
        if(this.data.width){
            var setWidth = this.data.width;
        }else{
            var setWidth = 200;
        }
        if(this.data.height){
            var setHeight = this.data.height;
        }else{
            var setHeight = 100;
        }
        this.place.find('input.previewresize[name="width"]').val(setWidth).parent().children('.controlvalue').text(setWidth);
        this.place.find('input.previewresize[name="height"]').val(setHeight).parent().children('.controlvalue').text(setHeight);
        this.place.find('.embeddedelement-embedpreviewwindow').trigger(jQuery.Event( "showEmbedded" ));
        
    }
    
    /******
     * Remove all event handlers
     ******/
    EmbeddedElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    EmbeddedElement.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in EmbeddedElement.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        });
            /******
         * Update the metadata of the element.
         ******/
                
        this.place.on('showEmbedded','.embeddedelement-embedpreviewwindow,.embeddedelement-embedviewwindow',function(){
            if(element.data.embedUrl && jQuery(this).find('iframe').attr('src')===element.data.embedUrl){
                jQuery(this).width(element.data.width).height(element.data.height);
            }else{
                jQuery(this).empty().append('<iframe src="'+(element.data.embedUrl?element.data.embedUrl:'')+'" '+(element.data.allowfullscreen?element.data.allowfullscreen:'')+'></iframe>').width(element.data.width).height(element.data.height);
            }
        });
        this.place.on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[embeddedelementdata]]', data);
            element.place.data('[[elementdata]]', data);
        });
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    EmbeddedElement.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    EmbeddedElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('elementfocus', function(e){
            e.stopPropagation();
            element.place.find('input.embeddedelement-editarea').trigger('focus');
        });
        this.place.on('click', '.embeddedelement-button[data-buttontype="remove"]', function(e){
            element.place.trigger('removeelement', {type: 'embeddedelement'});
        });
        this.place.on('setEmbedded',function(e){
            if(e.url){element.data.embedUrl = e.url;}
            if(e.allowfullscreen){element.data.allowfullscreen = e.allowfullscreen;}
            if(e.width){element.data.width = e.width;}
            if(e.height){element.data.height = e.height;}
            element.place.trigger('element_changed',[{type:"embeddedelement"}]);
            element.place.find('.embeddedelement-embedpreviewwindow').trigger(jQuery.Event( "showEmbedded" ));
        
        });

        this.place.find('input.previewresize').change(function(){
            var value = jQuery(this).val();
            if(jQuery(this).attr('name')==="width"){
                element.data.width = value;
                element.place.find('.embeddedelement-embedpreviewwindow').width(value);
            }else{
                element.data.height = value;
                element.place.find('.embeddedelement-embedpreviewwindow').height(value);
            }
            jQuery(this).parent().children('.controlvalue').text(value);
            element.place.trigger('element_changed',[{type:"embeddedelement"}]); 
        });
        this.place.on('mouseup','.embeddedelement-embedpreviewwindow',function(){
            var wi = parseInt(jQuery(this).width());
            var he = parseInt(jQuery(this).height());
            element.data.width = wi;
            element.data.height = he;
            element.place.find('input.previewresize[name="width"]').val(wi).parent().children('.controlvalue').text(wi);
            element.place.find('input.previewresize[name="height"]').val(he).parent().children('.controlvalue').text(he);
            element.place.trigger('element_changed',[{type:"embeddedelement"}]);            
        });
        this.place.children('input.embeddedelement-editarea').each(function() {
            var urlInput = jQuery(this);
            urlInput.data('oldVal', urlInput.val());
            // All available change events
            urlInput.on("propertychange change click keyup input paste", function(event){
                if (urlInput.data('oldVal') != urlInput.val()) {
                    urlInput.data('oldVal', urlInput.val());
                    element.data.inputData=urlInput.val();
                    var url = "";
                    if(/^(https?|s?ftp):\/\/(((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:)*@)?(((\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5]))|((([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])*([a-z]|\d|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])))\.)+(([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])*([a-z]|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])))\.?)(:\d*)?)(\/((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)?)?(\?((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|[\uE000-\uF8FF]|\/|\?)*)?(#((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|\/|\?)*)?$/i.test(urlInput.val())){
                        url=element.testUrl(urlInput.val());
                    }else if(/<iframe.*?<\/iframe>/i.test(urlInput.val())){
                        var event = jQuery.Event( "setEmbedded" );
                        if(/src="[^"]*"/.test(urlInput.val())){
                            event.url = urlInput.val().match(/src="[^"]*"/)[0].slice(5,-1);
                        }
                        if(/width="[^"]*"/.test(urlInput.val())){
                            event.width = urlInput.val().match(/width="[^"]*"/)[0].slice(7,-1);
                        }
                        if(/height="[^"]*"/.test(urlInput.val())){
                            event.height = urlInput.val().match(/height="[^"]*"/)[0].slice(8,-1);
                        }
                        if(/allowfullscreen/.test(urlInput.val())){
                            event.allowfullscreen = "allowfullscreen";
                        }
                        element.place.trigger(event);
                    }
                }
            });
        });
        this.place.children('input.embeddedelement-title').each(function() {
            var titleInput = jQuery(this);
            titleInput.data('oldVal', titleInput.val());
            // All available change events
            titleInput.on("propertychange change click keyup input paste", function(event){
                if (titleInput.data('oldVal') != titleInput.val()) {
                    titleInput.data('oldVal', titleInput.val());
                    element.data.elementTitle=titleInput.val();
                    element.place.trigger('element_changed',[{type:"embeddedelement"}]);
                    }
                
            });
        });
    }
    
    /******
     * Update the metadata of the element.
     ******/
    EmbeddedElement.prototype.testUrl = function(url){
        var returnUrl=url;
        var event = jQuery.Event( "setEmbedded" );
        if((url.match(/^(?:https?:\/\/)?(?:www\.)?(?:youtu\.be\/|youtube\.com\/(?:embed\/|v\/|watch\?v=|watch\?.+&v=))((\w|-){11})(?:\S+)?$/) ? returnUrl = "https://www.youtube.com/embed/"+RegExp.$1 : false)){
            //add features to youtubevideo
            event.allowfullscreen = "allowfullscreen";
            event.width = 600;
            event.height = 400;
        }
        event.url = returnUrl;
        this.place.trigger(event);
        
    }
    
    
    
    /******
     * Update the metadata of the element.
     ******/
    EmbeddedElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    }
    
    /******
     * Get data of the element
     * @returns {Object} data of this element of format:
     *   {
     *      "type": "embeddedelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {
                "elementTitle": this.data.elementTitle          // Element's title
                "embedUrl": this.data.embedUrl,                 // url to iframe
                "inputData": this.data.inputData,               // data that user have inputted
                "allowfullscreen": this.data.allowfullscreen    // is fullscreenmode allowed
                "width": this.data.width                        // width of embeddedelement
                "height": this.data.height                      // height of embeddedelement
            }
     *   }
     ******/
    EmbeddedElement.prototype.getData = function(){
        var result = {
            type: 'embeddedelement',
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        }
        return result;
    }
    
    
    /******
     * Template variable replacer.
     * @param {String} text - template with replaceable placeholders: '{%key%}'
     * @param {Object} data - Object with key-value-pairs
     * @returns {String} Original text with placeholders replaced with corresponding values.
     ******/
    EmbeddedElement.prototype.__format = function(text, data){
        data = $.extend(true,{}, data);
        for (var key in data) {
            var rex = RegExp('{%'+key+'%}', 'g');
            text = text.replace(rex, data[key]);
        }
        return text;
    }
    
    /******
     * Default settings
     ******/
    EmbeddedElement.defaults = {
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            elementTitle :'',
            embedUrl: '',
            inputData :'',
            allowfullscreen :'',
            width:200,
            height:100
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en'
        }
    }
    
    
    /******
     * Modes
     ******/
    EmbeddedElement.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    
    /******
     * Icons
     ******/
    EmbeddedElement.icons = {
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z" /><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    EmbeddedElement.elementinfo = {
        type: 'embeddedelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'embeddedelement',
        name: 'Embedded',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="ebook-icon ebook-icon-embed mini-icon mini-icon-embed"><path stroke="none" d="M9 8 l1 2 l-6 5 l6 5 l-1 2 l-9 -7z m12 0 l9 7 l-9 7 l-1 -2 l6 -5 l-6 -5z m-3 0 l-4 15 l-2 -1 l4 -15z" /></svg>',
        description: {
            en: 'Embedded webcontent',
            fi: 'Upotettu websisältö',
            sv: 'Inbäddat webbmaterial'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['general']
    }
    
    /******
     * Styles
     ******/
    EmbeddedElement.styles = [
        '.embeddedelement[data-elementmode="view"] {margin: 0.5em; width: auto; text-align: justify;}',
        // Input elements
        '.embeddedelement input.embeddedelement-title {min-width: 10em; min-height: 3em; width: 99%; margin: 0 0 0.5em 0;  background-color: white;}',
        '.embeddedelement input.embeddedelement-editarea {min-width: 10em; min-height: 3em; width: 99%; margin: 0;  background-color: white;}',
        // Edit / Author mode
        '.embeddedelement[data-elementmode="edit"], .embeddedelement[data-elementmode="author"] { border: 1px dashed #888; border-radius: 0.5em; margin: 0; position: relative; min-width: 15em;}',
        // Review mode
        '.embeddedelement[data-elementmode="review"] {border: 1px solid red; border-radius: 0.5em; padding: 0.5em;}',
        // Embedded content preview area
        '.embeddedelement-embedpreview{text-align: center;}',
        //Resizeinfo element
        '.embeddedelement-resizeinfo{display:block;}',
        // Embedded content preview window
        '.embeddedelement-embedpreviewwindow{margin: 0; display:inline-block; max-width:100%; min-width:100px; max-height:1000px; min-height:30px; resize: both; overflow: hidden; border:1px solid gray;border-right:none;padding-right:20px;}',
        // Embedded content preview area
        '.embeddedelement-embedview{text-align: center;}',
        // Embedded content preview window
        '.embeddedelement-embedviewwindow{margin: 0; display:inline-block; max-width:100%; min-width:100px; max-height:1000px; min-height:30px; overflow: hidden;}',
        '.embeddedelement-embedpreviewwindow iframe,.embeddedelement-embedviewwindow iframe{width:100%;height:100%;}',
        //open in browser link
        '.embeddedelement-embedview .embeddedelement-embedopeninbrowser{display:block;margin:5px 0;}',
        '.embeddedelement-embedview .embeddedelement-embedopeninbrowser a{color:#BBB;opacity:0;}',
        '.embeddedelement-embedview:hover .embeddedelement-embedopeninbrowser a{opacity:1;}'
        
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    EmbeddedElement.localization = {
        "en": {
            "embeddedelement:previewbox": "Previewbox",
            "embeddedelement:resizeinfo": "You can resize embeddedwindow as you like",
            "embeddedelement:titleplaceholder": "Element title",
            "embeddedelement:placeholder": "Embedcode or url",
            "embeddedelement:previewwidth": "Width",
            "embeddedelement:previewheight": "Height",
            "embeddedelement:remove": "Remove",
            "embeddedelement:openinbrowser": "Open in browser"
        },
        "fi": {
            "embeddedelement:previewbox": "Esikatseluruutu",
            "embeddedelement:resizeinfo": "Voit muuttaa upotusikkunan haluamaasi kokoon",
            "embeddedelement:titleplaceholder": "Otsikko",
            "embeddedelement:placeholder": "Upotuskoodi tai verkko-osoite",
            "embeddedelement:previewwidth": "Leveys",
            "embeddedelement:previewheight": "Korkeus",
            "embeddedelement:remove": "Poista",
            "embeddedelement:openinbrowser": "Avaa selaimessa"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(EmbeddedElement.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(EmbeddedElement.localization);
    }

    /******
     * Templates
     ******/
    EmbeddedElement.templates = {
        edithtml: [ '<input class="embeddedelement-title" placeholder="{%titleplaceholder%}" type="text"  />',
                    '<input class="embeddedelement-editarea" placeholder="{%placeholder%}" type="text"  />',
                    '<div class="embeddedelement-embedpreview">',
                    '<span class="embeddedelement-resizeinfo">{%resizeinfo%}</span>',
                    '<div class="embeddedelement-resizecontrol"><label>{%previewwidth%}:<span class="controlvalue"></span><input type="range" min=100 max=1200 class="previewresize" name="width" /></label><label>{%previewheight%}:<span class="controlvalue"></span><input type="range" min=30 max=1000 class="previewresize" name="height" /></label></div>',
                    '<span title="{%previewbox%}" class="embeddedelement-embedpreviewwindow"><iframe></iframe></span>',
                    '</div>'
        ].join('\n'),
        viewhtml: [ '<div class="embeddedelement-embedview">',
                    '<h3 class="embeddedelement-title"></h3>',
                    '<span class="embeddedelement-embedviewwindow"><iframe></iframe></span>',
                    '<span class="embeddedelement-embedopeninbrowser"><a>{%openinbrowser%}</a></span>',
                    '</div>'
        ].join('\n')
    }
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', EmbeddedElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', EmbeddedElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var embedelem = new EmbeddedElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.embeddedelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in embeddedelement.');
            return false;
        }
    }
    
})(jQuery);