/***
|''Name:''|emathequationarray.js|
|''Author:''|Petri Sallasmaa, Petri Salmela, Johan Ånäs|
|''Description:''|Tool for showing math in equationarray-like structure|
|''Version:''|1.0|
|''Date:''|September 23, 2013|
|''License:''|[[GNU Affero General Public License|http://www.gnu.org/licenses/agpl-3.0.html]]|
|''~CoreVersion:''|2.6.2|
|''Contact:''|pesasa@iki.fi|
|''Dependencies ''|[[DataTiddlerPlugin]]|
|''Documentation:''| |

!!!!!Revisions
<<<
20130922.1045 ''start''
<<<

!!!!!Code
***/
//{{{
/*
 * jquery.emathequationarray.js
 * jQuery plugin for equationarray
 * and TiddlyWiki macro
 *
 * E-Math -project http://emath.eu
 * Petri Salmela
 * License: AGPL 3.0 or later
 *
 */

/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - MathQuill
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.mathquill) === 'undefined' && jQuery.fn.mathquill;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };     
    
    /**
     * emathdisplaymode
     * @param options
     */
    
    $.fn.emathequationarray = function(options) {
        
        if (methods[options]){
            return methods[options].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(options) === 'object' || !options) {
                // Passing this 'this' to methods.init (so this there is also 'this')
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' +  options + ' does not exist on jQuery.emathequationarray' );
            return this;
        }
    };
    


       var methods = {
        init : function(options) {

            var useLegacyDataType = (options['type'] == null);

            var settings;
            
            options = $.extend({}, EmEqnarray.defaults, options);

            if (useLegacyDataType) {

                settings = $.extend({
                    latex : '',
                    editable : false,
                    settings : options.settings,
                    metadata : options.metadata
                }, options);
            } else {

                // options.mode may be 'view', 'edit', 'author', or 'review' and
                // 'slideshow' is one form of view mode too
                var authormode = (options.settings.mode == 'author');

                // Author always edits
                var editable = authormode || (options.settings.mode == 'edit');
                var eqnarray = options.data.eqnarray;

                settings = {
                    "latex" : '',
                    "editable" : editable,
                    "authormode" : authormode,
                    "settings" : options.settings,
                    "metadata" : options.metadata,
                    "eqnarray" : eqnarray
                };

            }

            // because this here may be an array of elements by the initial
            // jQuery-selection. This means that an array will be created for
            // them all
            return this
                    .each(function() {
                        var eqnarray = new EmEqnarray(this, settings);
                        eqnarray.init();
                    });
        },
        get : function(options) {
            this.trigger('getdata');
            return this.data('[[elementdata]]'); // return the
            // current value
        },
        getreviewdata : function(options) {
            this.trigger('get_review_data');
            return this.data('review_data'); // return the current value
        },
        setreviewdata : function(options) {
            if (options['messages']) {
                this.trigger('set_review_data', options);
            }
        },
        geteditables : function(options) {
            this.trigger('geteditable_data');
            return this.data('emathequationarray_editable_data'); // return
            // the
            // current
            // value
        },
        set : function(options) {
            if (typeof (options) === 'string') {
                this.trigger('set_data', [ options ]);
            }
        },
        reveal : function() {

            this.trigger('reveal_one_row');

        }
    };



    /***************************************************************************
     * An Equation array
     **************************************************************************/
    
    var EmEqnarray = function(place, settings){

        settings = $.extend({
            eqnarray: [{left:'', middle:'', right:'', description:''}],
            editable: false,
            authormode: false
        }, settings);
        this.inited = false;
        this.place = $(place);
        // Contains an array of rows
        this.eqnarray = [];
        this.settings = settings;
        this.metadata = settings.metadata;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.creator = this.metadata.creator || this.metadata.modifier || this.settings.username;
        this.type = settings.type;
        this.usersettings = settings.settings;
        this.editable = settings.editable;
        this.authormode = settings.authormode;
        // if metadata and type not found, data is considered legacy style
        // revieweditmode is read from review data
        this.revieweditmode = false;
    };

    EmEqnarray.prototype.init = function(){
        var html = '<div class="emathequationarraybox"><table class="emathequationarray"><tbody></tbody></table></div>';
        this.place.html(html);
        this.content = this.place.find('tbody');
        this.setAttrs();
        this.draw();
        this.initData();
        
        this.initHandlers();
        if ($('head style#ematheqnarraycss').length < 1){
            $('head').append('<style type="text/css" id="ematheqnarraycss">' + this.strings.css + '</style>');
        }
        this.inited = true;
    };
    
    EmEqnarray.prototype.initData = function(){
        // Init rows with given data.
        for (var i = 0, length = this.settings.eqnarray.length; i < length; i++){
            this.addRow(this.settings.eqnarray[i]);
        }
        if (this.usersettings.mode == 'slideshow') {
                this.content.addClass('slideshow')
                $rowList = this.content.find('tr.row');
                $.each($rowList, function() {
                        $(this).addClass('unrevealed');
                });
        }
    }
    
    /**
     * Set some attributes
     */
    EmEqnarray.prototype.setAttrs = function() {
        this.place.addClass('emathequationarray').attr('data-elementmode', this.usersettings.mode);
    };
    
    /**
     * This returns the id that should be used for identifying the row, NOT the index. 
     * @returns {Number}
     */
    EmEqnarray.prototype.getNextFreeRowId = function() {
        for (var i = 0; i < this.eqnarray.length; i++){
                
                // If we find a free slot starting from first row, just use that id
                if (!($("tr[data-loc='row_" + i + "']", this.content)[0])) {
                        return i;
                }
        }
        return this.eqnarray.length;
        
    }
    
    EmEqnarray.prototype.addRow = function(options, rownum){
        // Add a row in rownum:th position.
        options = $.extend(true, {
            left: '',
            middle: '',
            right: '',
            description: '',
            editable: this.editable,
            editmodes: {left: false, middle: false, right: false, description: false},
            authormode: this.authormode,
            reviewmode: (this.usersettings.mode == 'review')
        }, options);
        $trlist = this.content.find('tr');
        if (typeof(rownum) !== 'number') {
            rownum = $trlist.length
        }
        var rowId = 'row_' + this.getNextFreeRowId();
        var $newtr = $('<tr></tr>').attr('data-loc', rowId);
        if (rownum >= 0 && rownum < $trlist.length) {
            $trlist.eq(rownum).before($newtr);
        } else {
            this.content.append($newtr);
        }
        var row = new EmEqnRow($newtr, options, this.revieweditmode);
        this.eqnarray.splice(rownum, 0, row);
        row.draw();
    }
    
    EmEqnarray.prototype.removeRow = function(rownum){
        // Remove row with given rownumber.
        this.eqnarray.splice(rownum, 1);
        this.content.find('tr').eq(rownum).remove();
        if (this.eqnarray.length === 0) {
            this.addRow();
        }
        this.changed();
    }
    
    EmEqnarray.prototype.setFocus = function(row, col){
        // Set focus on row (number) in given col ('left', 'middle', 'right', 'description').
        if (0 <= row && row < this.eqnarray.length) {
            this.eqnarray[row].setFocus(col);
        }
    }
    
    EmEqnarray.prototype.draw = function(){
        // Draw the equationarray.
        this.place.find('table').removeClass();
        
        if (this.usersettings.mode == 'review') {
                this.place.find('table').addClass('review');
                if (this.revieweditmode) {
                        this.place.find('table').addClass('review-editable');
                } else {
                        this.place.find('table').addClass('review-readonly');                           
                }
        } else if (this.editable) {
            this.place.find('table').addClass('editable');
        }
        this.content.find('tr').empty();
        for (var i = 0; i < this.eqnarray.length; i++){
            this.eqnarray[i].draw();
        }
    }
    
    EmEqnarray.prototype.initHandlers = function(){
        var eqnarray = this;
        this.place.unbind().bind('eqnarray_next', function(event, place){
            // Move to the next field.
            var $mqelems = eqnarray.content.find('.emeqnarray-field-editable');
            var $mqelemsTextAreas = eqnarray.content.find('.emeqnarray-field-editable textarea');
            var index = $mqelems.index(place);
            if (index >= 0 && index < $mqelems.length - 1) {
                place.focusout().blur();
                $mqelemsTextAreas.eq(index + 1).focus();
            } else if (index === $mqelems.length -1) {
                place.focusout().blur();
                var trplace = place.parents('tr').eq(0);
                $(this).trigger('eqnarray_addrow', [trplace]);
            }
        }).bind('row_changed', function(event){
            eqnarray.changed();
        }).bind('eqnarray_goUp eqnarray_goDown eqnarray_goLeft eqnarray_goRight', function(event, place, direction){
            // Move in fields, Up, Down, Left, Right.
            var currentrow = place.parents('tr').eq(0);
            var $trlist = eqnarray.content.find('tr');
            var $mqlist = currentrow.find('.emeqnarray-field-editable');
            var row = $trlist.index(currentrow);
            var col = $mqlist.index(place);
            var horizmove = {
                Left: -1,
                Right: 1,
                Up: 0,
                Down: 0
            }
            var vertmove = {
                Left: 0,
                Right: 0,
                Up: -1,
                Down: 1
            }
            var newrow = Math.max(0, Math.min($trlist.length - 1, row + vertmove[direction]));
            var newcol = Math.max(0, Math.min(3, col + horizmove[direction]));
            eqnarray.content.find('.emeqnarray-field-editable textarea').eq(4*newrow + newcol).focus(); 
        }).bind('eqnarray_addrow', function(event, place){
            // Add a row after current.
            var $trlist = eqnarray.content.find('tr');
            var index = $trlist.index(place) + 1;
            eqnarray.addRow({}, index);
            eqnarray.setFocus(index, 'left');
        }).bind('eqnarray_addrowcopy', function(event, place){
            // Add a copy of current row after current row.
            var $trlist = eqnarray.content.find('tr');
            var index = $trlist.index(place) + 1;
            eqnarray.addRow(eqnarray.eqnarray[index - 1].getData(), index);
            eqnarray.setFocus(index, 'left');
        }).bind('eqnarray_removerow', function(event, place){
            // Remove this (place) row.
            var $trlist = eqnarray.content.find('tr');
            var index = $trlist.index(place);
            eqnarray.removeRow(index);
            eqnarray.setFocus(index - 1, 'right');
            // FIXME TODO WHen old book is not in use, remove get_data listener and stuff that goes within
        }).bind('get_data', function(event){
            eqnarray.place.data('emathequationarray_data', eqnarray.getData());
        }).bind('getdata', function(event){
            event.stopPropagation();
            eqnarray.place.data('[[elementdata]]', eqnarray.getData());
        }).bind('gethtml', function(event){
            event.stopPropagation();
            eqnarray.place.data('[[elementhtml]]', eqnarray.getHtmltext());
        }).bind('geteditable_data', function(event){
            eqnarray.place.data('emathequationarray_editable_data', eqnarray.getEditableFields());
        }).bind('get_review_data', function(event){
                eqnarray.place.data('review_data', eqnarray.getReviewData());
        }).bind('set_review_data', function(event, data){
                
                if (!( eqnarray.usersettings.mode == 'review' || eqnarray.usersettings.mode == 'view')) {
                        return;
                } 
                 
                if (data.editable) {
                        eqnarray.revieweditmode = true;
                } else {
                        eqnarray.revieweditmode = false;                        
                }
                eqnarray.setReviewData(data);
        }).bind('reveal_one_row', function(event){
                var $trlist = eqnarray.content.find('tr.unrevealed');
                if ($trlist.length > 0) {
                        
                        $($trlist[0]).show(400, function() {
                                 $( this ).removeClass('unrevealed'); 
                        });
                        
                        
                }
                
        });
    }
    
    // Setting the review data will automatically switch to view mode
    EmEqnarray.prototype.setReviewData = function(data){
        
        var indexToRowData = {};
        for (var ind = 0; ind < data.messages.length; ind++) {
                if (data.messages[ind] != null) {
                        var locIndex = parseInt(data.messages[ind].loc.replace("row_", ""));
                        indexToRowData['index' + locIndex] = data.messages[ind];
                }
        }
        
        $.each(this.eqnarray, function(index) {
                this.revieweditmode = data.editable;
                if (indexToRowData['index' + index] != null ) {
                        this.reviewData = indexToRowData['index' + index];                      
                } else {
                        this.reviewData = null;
                }
        });

        this.changeMode('review');
    }
    
    EmEqnarray.prototype.changed = function(){
        this.metadata.modifier = this.usersettings.username;
        this.metadata.modified = (new Date()).getTime();
        // TODO: REmoved 'changed' when old book is removed
        this.place.trigger('changed');
        this.place.trigger('element_changed', {type: 'emeqnarray'});
    }
    
    EmEqnarray.prototype.changeMode = function(mode){
        // Change the editmode ('view'/'edit').
        var modes = {view: false, edit: true, author: true, review: false, slideshow: false};
        this.usersettings.mode = mode;
        this.editable = (typeof(modes[mode]) !== undefined ? modes[mode] : this.editable);
        for (var i = 0; i < this.eqnarray.length; i++) {
            this.eqnarray[i].changeMode(mode);
        }
        this.draw();
    }
    
    
    
    /**
     * Return data the (the old format)
     */
    EmEqnarray.prototype.getData = function(){

        return this.getDataAsNewFormat();               
        
    }

    
    /**
         * Return data (new format)
         */
    EmEqnarray.prototype.getDataAsNewFormat = function(){
        // Return the data of this equation array.
        var result = {type: "emeqnarray", metadata: this.metadata, data: { eqnarray: []}};
        
                for (var i = 0; i < this.eqnarray.length; i++) {
                        result.data.eqnarray.push(this.eqnarray[i].getData());
                }
        
        return result;
    }
    
    EmEqnarray.prototype.getHtmltext = function() {
        var latex = this.getLatexArray().replace(/\\(?:(N)N|(Z)Z|(Q)Q|(R)R|(C)C|(P)P|(H)H)/g, '\\mathbb{$1$2$3$4$5$6$7}');
        var html = '<img src="/math.svg?latex='
            + encodeURIComponent(latex)
            + '" alt="'
            + escapeHTML(latex)
            + '">';
        return html;
    };
    
    EmEqnarray.prototype.getLatexArray = function() {
        var latex = [
            '\\begin{array}{rcl|l}'
        ];
        for (var i = 0; i < this.eqnarray.length; i++) {
            latex.push(this.eqnarray[i].getLatexArray() + (i < this.eqnarray.length - 1 ? ' \\\\' : ''));
        };
        latex.push('\\end{array}');
        return latex.join('\n');
    };
    
    /**
         * Return review data
         */
    EmEqnarray.prototype.getReviewData = function() {


                                
        var result = {general : "", editable: this.revieweditmode, messages: []};
                for (var i = 0; i < this.eqnarray.length; i++) {
                        
                        
                var rowHasReviewOrComment = false;
                var marks = [];
                        
                $.each(this.eqnarray[i].switchers, function(index, value) {
                                if (this.currentMark != "") {
                                        rowHasReviewOrComment = true;
                                }
                                marks.push(this.currentMark);
                        });
                        
                        var commentVal = this.eqnarray[i].place.find('.validation-field .emeqnarray-field-editable').mathquill('latex');
                if (commentVal.length > 0) {
                        rowHasReviewOrComment = true;
                }
                if (!rowHasReviewOrComment) {
                        continue;
                }
                        var rowId = this.eqnarray[i].place.attr('data-loc');
                
                var messageContent = {loc: rowId, comment: commentVal, mark: marks};
                result.messages.push(messageContent);
        } 
   
        
        return result;
    }
    
    EmEqnarray.prototype.getEditableFields = function(){
        // Return data of editable fields as an array.
        var fields = this.place.find('.emeqnarray-field-editable');
        var result = [];
        for (var i = 0; i < fields.length; i++) {
            result.push(fields.eq(i).mathquill('latex'));
        }
        return result;
    }

    EmEqnarray.prototype.strings = {
        css: [
            '.emathequationarray {text-align: center;}',
            '.emathequationarray[data-elementmode$="view"] table {margin: 1em auto;}',
            '.emathequationarraybox {display: block; margin: 0.2em 0;}',
            '.emathequationarraybox table {text-align: left; border: none;}',
            '.emathequationarraybox table td {border: none; vertical-align: top;}',
            '.emathequationarray[data-elementmode$="view"] table td {padding-top: 0.4em; padding-bottom: 0.4em;}',
            '.emathequationarraybox table.editable {padding: 0.4em 1em;}',
            '.emathequationarraybox td.emeqnarray-left {text-align: right; min-width: 3em;}',
            '.emathequationarraybox td.emeqnarray-middle {text-align: center; min-width: 1em;}',
            '.emathequationarraybox td.emeqnarray-right {text-align: left; min-width: 3em; padding-right: 20px;}',
            '.emathequationarraybox td.emeqnarray-description {text-align: left; min-width: 6em; padding-left: 5px;}',
            '.emathequationarraybox table td.emeqnarray-description .emeqnarray-field-editable {box-shadow: none; background-color: #f0f0f0; overflow-x: hidden;}',
            '.emathequationarraybox td.emeqnarray-description {padding-left: 1em;}',
            '.emathequationarraybox td.emeqnarray-rowactions {min-width: 40px; text-align: center;}',
            '.emathequationarraybox .mathquill-editable {margin: 0.1em; border: 1px solid transparent; display: block;background-color: #f8f8f8; border-radius: 5px; padding: 0.2em 0.5em; box-shadow: inset 1px 1px 2px rgba(0,0,0,0.5), inset -1px -1px 2px rgba(255,255,255,0.5);}',
            '.emathequationarraybox .mathquill-editable.hasCursor {box-shadow: inset 1px 1px 2px rgba(0,0,0,0.5), inset -1px -1px 2px rgba(255,255,255,0.5), 0 0 3px 2px #68B4DF;}',
            '.emathequationarraybox .mathquill-editable.mathquill-textbox.hasCursor {box-shadow: inset 1px 1px 2px rgba(0,0,0,0.5), inset -1px -1px 2px rgba(255,255,255,0.5), 0 0 3px 2px #EE9393;}',
            '.emathequationarraybox table:not(.editable) .emeqnarray-field-editable {background-color: white;}',
            '.emathequationarraybox span.emeqnarray-buttontext {display: none;}',
            '.emathequationarraybox .emeqnarray-rowbutton {opacity: 0.1; cursor: pointer; display: inline-block; border-radius: 2px; transition: opacity 0.02s;}',
            '.emathequationarraybox tr:hover .emeqnarray-rowbutton {opacity: 1; transition: opacity 1s;}',
            '.emathequationarraybox .emeqnarray-removerowbutton svg {fill: red;}',
            '.emathequationarraybox .emeqnarray-newrowbutton svg {fill: green;}',
            '.emathequationarraybox .emeqnarray-newrowbutton {position: relative; bottom: -15px;}',
            '.emathequationarraybox .emeqnarray-authorlock {width: 10px; height: 10px; position: absolute; border-radius: 3px; cursor: pointer;}',
            '.emathequationarraybox .emeqnarray-authorlock .emeqnarray-locked {display: none; background-color: red; width: 10px; height: 10px; border-radius: 3px;}',
            '.emathequationarraybox .emeqnarray-authorlock.emeqnarray-lock-on .emeqnarray-locked {display: block;}',
            '.emathequationarraybox .emeqnarray-authorlock .emeqnarray-unlocked {display: block; background-color: green; width: 10px; height: 10px; border-radius: 3px;}',
            '.emathequationarraybox .emeqnarray-authorlock.emeqnarray-lock-on .emeqnarray-unlocked {display: none;}',
            '.emathequationarraybox tr .emeqnarray-reviewswitch {display: inline-block; position: absolute;}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch {display: inline-block; width: 0.8em; height: 0.8em; position: absolute; border-radius: 3px; cursor: pointer; border: 1px solid black; background-color: rgba(255,255,255,0.5); }',
            '.emathequationarraybox:hover table.review-readonly tr .emeqnarray-reviewswitch {cursor: auto; border: none; background-color: rgba(255,255,255,0.0); }',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.correct {background-color: rgba(0,255,0,0.5);}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.wrong {background-color: rgba(255,0,0,0.5)}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.note {background-color: rgba(255,255,0,0.5)}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.correct::before {position: relative; left: 0.2em; top: -0.35em; content: "";}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.wrong::before {position: relative; left: 0.05em; top: -0.3em; content: "\u2714";}',
            '.emathequationarraybox:hover table.review tr .emeqnarray-reviewswitch.note::before {position: relative; left: 0.22em; top: -0.2em; content: "!";}',
            '.emathequationarraybox .validation-field {width: 10em;}',
            '.emathequationarraybox table .validation-field > .emeqnarray-field-editable  {border: 1px dotted black; border-radius: 0px;  overflow-x: hidden;}',
            '.emathequationarraybox .noselect {-webkit-touch-callout: none;-webkit-user-select: none;-khtml-user-select: none;-moz-user-select: none;-ms-user-select: none;user-select: none; }',
            '.emathequationarraybox tr.row.unrevealed {display: none; }'
        ].join('\n')
    };
    
    EmEqnarray.icons = {
        trash: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>',
        newrow: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-newrow"><path style="stroke: none;" d="M13 7 l4 0 l0 6 l6 0 l0 4 l-6 0 l0 6 l-4 0 l0 -6 l-6 0 l0 -4 l6 0z" /></svg>',
        copyrow: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 30 30" class="mini-icon mini-icon-copyrow"><path style="fill: black; stroke: none;" d="M20 2 l4 0 l0 6 l6 0 l0 4 l-6 0 l0 6 l-4 0 l0 -6 l-6 0 l0 -4 l6 0z" /><path style="stroke: none;" d="M8 11 l4 0 l0 6 l6 0 l0 4 l-6 0 l0 6 l-4 0 l0 -6 l-6 0 l0 -4 l6 0z" /></svg>'
    }
    
    
    
    /******
     * Default settings. 
     ******/
    EmEqnarray.defaults = {
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            eqnarray: [{left:'', middle:'', right:'', description:''}]
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en'
        }
    }

    EmEqnarray.elementinfo = {
        type : 'emeqnarray',
        elementtype : ['elements','studentelements'],
        jquery : 'emathequationarray',
        name : 'Equation array',
        icon : '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-equationarray"><path style="stroke: none;" d="M5 7 h3 l13 20 h-3z m13 0 h3 l-13 20 h-3z m3 -2 a3 3 0 0 1 6 0 l-3 4 h3 v2 h-6 v-1 l4 -5 a1 1 0 0 0 -2 0z"></path></svg>',
        icon2 : '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 40 40" class="mini-icon mini-icon-equationarray"><path style="stroke: none;" d="M8 6 h21 v6 h-1 a2 3 0 0 0 -2 -3 h-13 l8 10 l-8 10 h15 a2 3 0 0 0 2 -3 h1 v6 h-23 v-3 l10 -10 l-10 -10z"></path></svg>',
        description : {
            en : 'Math environment',
            fi : 'Matematiikkaympäristö',
            sv: 'Matematikomgivning'
        },
        roles: ['teacher', 'student', 'author'],
        classes : [ 'math' ],
        weight: 50
    }

    if (typeof ($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', EmEqnarray.elementinfo);
    }
    
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', EmEqnarray.elementinfo);
    }
    
    
    
    /****************************************************************
     ****************************************************************
     * Row of an equation array
     ****************************************************************
     ****************************************************************/
    
    
    var EmEqnRow = function(place, options,reviewEditMode){
        // Class for a row in equationarray
        this.place = place;
        this.left = options.left || '';
        this.middle = options.middle || '';
        this.right = options.right || '';
        this.description = options.description || '';
        this.editable = options.editable;
        this.editmodes = $.extend({}, options.editmodes);
        this.authormode = options.authormode;
        this.reviewmode = options.reviewmode;
        this.reviewData = null;
        this.revieweditmode = reviewEditMode;
        this.slideshowmode = options.slideshowmode;
        this.inited = false;
        this.switchers = [];
    }
    
    EmEqnRow.prototype.draw = function(){
        // Draw formulas. never mix reviewmode and editable. 
        // Review mode is stronger than other modes
        
        if (this.reviewmode) {
                this.showAsReview(this.revieweditmode);
        } else if (this.editable) {
            this.edit();
        } else { // also goes for slideshow mode
            this.show();
        }
        
        this.place.addClass('row');
        

    }
    
    EmEqnRow.prototype.showAsReview = function(isEditableReview){
        
        // Removing old switchers
        $.each(this.switchers, function() {
                this.shutDown();
        });
        
        this.switchers = [];
        
        var descriptionText = this.description.replace(/\$([^\$]*)\$/g, '<span class="embedded-math">$1</span>');
        
        var commentText = '';
        if (this.reviewData != null && 'comment' in this.reviewData) {
                commentText = this.reviewData.comment.replace(/\$([^\$]*)\$/g, '<span class="embedded-math">$1</span>');
        }

        // Show static math
        var html = [
            '<td class="emeqnarray-reviewswitch"></td><td class="emeqnarray-left"><span class="emeqnarray-field emeqnarray-math-type">'+escapeHTML(this.left)+'</span></td>',
            '<td class="emeqnarray-reviewswitch"></td><td class="emeqnarray-middle"><span class="emeqnarray-field emeqnarray-math-type">'+escapeHTML(this.middle)+'</span></td>',
            '<td class="emeqnarray-reviewswitch"></td><td class="emeqnarray-right"><span class="emeqnarray-field emeqnarray-math-type">'+escapeHTML(this.right)+'</span></td>',
            '<td class="emeqnarray-reviewswitch"></td><td class="emeqnarray-description"><span class="emeqnarray-field emeqnarray-mixed-type">'+escapeHTML(descriptionText)+'</span></td>',
            '<td class="emeqnarray-reviewswitch"></td><td class="validation-field"><span class="emeqnarray-field'+(isEditableReview ? '-editable' : '')+' emeqnarray-mixed-type">'+escapeHTML(commentText)+'</span></td>'
        ].join('\n');
        


        this.place.html(html);
        this.place.addClass('noselect');
        var $switchArray = this.place.find('.emeqnarray-reviewswitch');
        var outerRow = this;
        $.each($switchArray, function(index) {
                var switcher = new EmSwitcher($(this), !isEditableReview);
                outerRow.switchers.push(switcher);
                switcher.registerClickHandler();
                if (outerRow.reviewData != null) {
                        switcher.setMark(outerRow.reviewData.mark[index]);                      
                }
        });
        
        this.place.find("td.emeqnarray-right").css("border-right", "4px double");               
        
        this.place.find('.emeqnarray-field.emeqnarray-math-type').mathquill();
        this.place.find('.emeqnarray-field.emeqnarray-mixed-type .embedded-math').mathquill();
        this.place.find('.emeqnarray-field-editable.emeqnarray-math-type').mathquill('editable');
        this.place.find('.emeqnarray-field-editable.emeqnarray-mixed-type').mathquill('textbox');
        this.inited = true;
    }
    

    
    EmEqnRow.prototype.show = function(){

        var isEditableDescription = this.editmodes['description'];
        var descriptionText = null;
        if (!isEditableDescription) {
                 descriptionText = escapeHTML(this.description).replace(/\$([^\$]*)\$/g, '<span class="embedded-math">$1</span>');
        } else {
                descriptionText = escapeHTML(this.description);
        }
        
        // Show static math
        var html = [
            '<td class="emeqnarray-left"><span class="emeqnarray-field'+(this.editmodes['left'] ? '-editable' : '')+' emeqnarray-math-type">'+escapeHTML(this.left)+'</span></td>',
            '<td class="emeqnarray-middle"><span class="emeqnarray-field'+(this.editmodes['middle'] ? '-editable' : '')+' emeqnarray-math-type">'+escapeHTML(this.middle)+'</span></td>',
            '<td class="emeqnarray-right"><span class="emeqnarray-field'+(this.editmodes['right'] ? '-editable' : '')+' emeqnarray-math-type">'+escapeHTML(this.right)+'</span></td>'
        ];
        
        var htmlDescription = '<td class="emeqnarray-description"><span class="emeqnarray-field'+ (isEditableDescription ? '-editable' : '')+' emeqnarray-mixed-type">'+descriptionText+'</span></td>';
        var hasDescription = ((this.description != null && this.description.length > 0)|| isEditableDescription);
        if (hasDescription) {
                html.push(htmlDescription);
        }
        html = html.join('\n');
        this.place.html(html);
        if (hasDescription) {
                this.place.find("td.emeqnarray-right").css("border-right", "4px double");               
        }
        
        if (this.slideshowmode) {
                this.place.addClass('slideshow');
        }
        this.place.find('.emeqnarray-field.emeqnarray-math-type').mathquill();
        this.place.find('.emeqnarray-field.emeqnarray-mixed-type .embedded-math').mathquill();
        this.place.find('.emeqnarray-field-editable.emeqnarray-math-type').mathquill('editable');
        this.place.find('.emeqnarray-field-editable.emeqnarray-mixed-type').mathquill('textbox');
        this.inited = true;
    }
    
    EmEqnRow.prototype.edit = function(){
        // Show math in editmode
        var authorlock = (this.authormode ? '<div class="emeqnarray-authorlock emeqnarray-lock-on"><div class="emeqnarray-locked"></div><div class="emeqnarray-unlocked"></div></div>' : '');
        var html = [
            '<td class="emeqnarray-rowactions"><span class="emeqnarray-rowbutton emeqnarray-newrowbutton">'+EmEqnarray.icons.newrow+'<span class="emeqnarray-buttontext">New row</span></span></td>',
            '<td class="emeqnarray-left">'+authorlock+'<span class="emeqnarray-field-editable emeqnarray-math-type" data-field="left">'+escapeHTML(this.left)+'</span></td>',
            '<td class="emeqnarray-middle">'+authorlock+'<span class="emeqnarray-field-editable emeqnarray-math-type" data-field="middle">'+escapeHTML(this.middle)+'</span></td>',
            '<td class="emeqnarray-right">'+authorlock+'<span class="emeqnarray-field-editable emeqnarray-math-type" data-field="right">'+escapeHTML(this.right)+'</span></td>',
            '<td class="emeqnarray-description">'+authorlock+'<span class="emeqnarray-field-editable emeqnarray-mixed-type" data-field="description">'+escapeHTML(this.description)+'</span></td>',
            '<td class="emeqnarray-rowactions"><span class="emeqnarray-rowbutton emeqnarray-removerowbutton">'+EmEqnarray.icons.trash+'<span class="emeqnarray-buttontext">Remove</span></span></td>'
        ].join('\n');
        this.place.html(html);
        var mathElements = this.place.find('.emeqnarray-math-type').mathquill('editable');
        this.place.find('.emeqnarray-mixed-type').mathquill('textbox');
        this.place.find("td.emeqnarray-right").css("border-right", "4px double");   
        this.leftelem = mathElements.eq(0);
        this.middleelem = mathElements.eq(1);
        this.rightelem = mathElements.eq(2);
        this.initHandlers();
        this.inited = true;
    }
    
    EmEqnRow.prototype.initHandlers = function(){
        // Init handlers for fields.
        var eqnrow = this;
        this.place.undelegate('.emeqnarray-field-editable', 'focusout')
            .delegate('.emeqnarray-field-editable', 'focusout', function(event){
            // focusout-events for fields.
            var $mqfield = $(this);
            var field = $mqfield.attr('data-field');
            var newdata = $mqfield.mathquill('latex');
            if (eqnrow[field] !== newdata) {
                eqnrow[field] = newdata;
                eqnrow.changed();
            }
        }).undelegate('.emeqnarray-field-editable', 'keydown')
            .delegate('.emeqnarray-field-editable', 'keydown', function(event){
            // keypress-events for fields.
            event.stopPropagation();
            //event.preventDefault();
            var $mqfield = $(this);
            var field = $mqfield.attr('data-field');
            switch (event.keyCode){
                case 13:
                    if (event.ctrlKey) {
                        $mqfield.focusout().blur();
                        eqnrow.place.trigger('eqnarray_addrow', [eqnrow.place]);
                        eqnrow.changed();
                    } else if (event.shiftKey) {
                        $mqfield.focusout().blur();
                        eqnrow.place.trigger('eqnarray_addrowcopy', [eqnrow.place]);
                        eqnrow.changed();
                    } else {
                        eqnrow.place.trigger('eqnarray_next', [$mqfield]);
                    }
                    break;
                case 38:
                case 40:
                case 37:
                case 39:
                    if (event.ctrlKey) {
                        var keys = {37: 'Left', 38: 'Up', 39: 'Right', 40: 'Down'};
                        eqnrow.place.trigger('eqnarray_go'+keys[event.keyCode], [$mqfield, keys[event.keyCode]]);
                    }
                    break;
                case 8:
                    if (event.ctrlKey) {
                        eqnrow.place.trigger('eqnarray_removerow', [eqnrow.place]);
                        eqnrow.changed();
                    }
                default:
                    break;
            }
        }).undelegate('.emeqnarray-removerowbutton', 'click')
            .delegate('.emeqnarray-removerowbutton', 'click', function(event){
            // Click-events for removerow-buttons.
            event.preventDefault();
            eqnrow.place.trigger('eqnarray_removerow', [eqnrow.place]);
            eqnrow.changed();
        }).undelegate('.emeqnarray-newrowbutton', 'click')
            .delegate('.emeqnarray-newrowbutton', 'click', function(event){
            // Click-events for newrow-buttons.
            event.preventDefault();
            var $mqfield = $(this).parents('tr').find('.hasCursor');
            $mqfield.focusout().blur();
            eqnrow.place.trigger('eqnarray_addrow', [eqnrow.place]);
            eqnrow.changed();
        }).undelegate('.emeqnarray-authorlock', 'click')
            .delegate('.emeqnarray-authorlock', 'click', function(event){
            // Click-events for newrow-buttons.
            event.preventDefault();
            var $lock = $(this);
            var field = $lock.parent('td').find('[data-field]').attr('data-field');
            eqnrow.toggleLock(field);
            $lock.toggleClass('emeqnarray-lock-on');
            eqnrow.changed();
        });
    }
    
    EmEqnRow.prototype.setFocus = function(col){
        // Set focus in col ('left', 'middle', 'right')
        this.place.find('.emeqnarray-field-editable[data-field="'+col+'"] textarea').focus();
    }
    
    EmEqnRow.prototype.changed = function(){
        // Trigger changed-event.
        this.place.trigger('row_changed');
    }
    
    
    EmEqnRow.prototype.toggleLock = function(field){
        // Toggle editmode of field.
        this.editmodes[field] = !this.editmodes[field];
    }
    
    EmEqnRow.prototype.changeMode = function(mode){
        // Change the editmode ('view'/'edit').
        var modes = {view: false, edit: true, author: true, review: false};
        this.editable = (typeof(modes[mode]) !== undefined ? modes[mode] : this.editable);
    }
    
    EmEqnRow.prototype.getData = function(){
        // Return the data of this row.

        return {
            left: this.left,
            middle: this.middle,
            right: this.right,
            description: this.description,
            editmodes: {
                left: this.editmodes.left,
                middle: this.editmodes.middle,
                right: this.editmodes.right,
                description: this.editmodes.description
            }
        };
    }
    
    EmEqnRow.prototype.getLatexArray = function() {
        var latex = [
            this.left,
            this.middle,
            this.right,
            '\\text{' + this.description + '}'
        ];
        return latex.join(' & ');
    };


    /****************************************************************
     ****************************************************************
     * Switcher, that may reside in a reviewable row
     ****************************************************************
     ****************************************************************/
    
    
    var EmSwitcher = function($place, readOnly){
        // Class for a row in equationarray
        this.$place = $place;
        this.$place.addClass('emeqnarray-reviewswitch');


        this.readOnly = readOnly;
        this.marks =  ["", "correct", "wrong", "note"];
        this.currentMark = this.marks[0];
    }

    /**
     * Clickhandlers will only be registered if EmSwitcher is not set as readOnly
     */
    EmSwitcher.prototype.registerClickHandler = function() {
        if (this.readOnly) {
                return;
        }
        var switcher = this;
        this.$place.on('click', function() {
                switcher.nextMark();
        });
    };
    
    
    EmSwitcher.prototype.setMark = function(mark) {
        if (mark == "" || $.inArray(mark, this.marks)) {
                this.currentMark = mark;
                this.$place.removeClass("correct wrong note");
                this.$place.addClass(mark);

        }
    }
    
    EmSwitcher.prototype.nextMark = function() {
        if (this.readOnly) {
                return;
        }
        var markIndex = this.marks.indexOf(this.currentMark);
        markIndex = ((markIndex + 1) % this.marks.length);
        this.setMark(this.marks[markIndex]);
        
    }
    
    EmSwitcher.prototype.shutDown = function() {        
        this.$place.off();
        this.$place.remove();
    }

    
})(jQuery)

if (typeof(config) !== 'undefined' && typeof(config.macros) !== 'undefined'){
    // Create macro for TiddlyWiki
    config.macros.emathequationarray = {
        /******************************
         * Show emathequationarray
         ******************************/
        handler: function (place, macroName, params, wikifier, paramString, tiddler)
        {
            if (params.length < 1){
                wikify('Missing equationarray.', place, null, tiddler);
                return false;
            }
            var eqnarrayid = params[0];
            var iseditable = (params[1] === 'edit' || params[1] === 'author' || params[1] === 'authordialog');
            var isauthor = (params[1] === 'authordialog' || params[1] === 'author');
            var emtabletext = '{{emathequationarraywrapper ematheqnarray_'+eqnarrayid+'{\n}}}';
            wikify(emtabletext, place);
            if (tiddler) {
                var settings = jQuery.extend(true, {}, tiddler.data('ematheqnarray',{}));
            } else {
                var settings = {};
            }
            settings[eqnarrayid] = settings[eqnarrayid] || {};
            settings[eqnarrayid].editable = iseditable;
            settings[eqnarrayid].authormode = isauthor;
            var eqnarray = jQuery(place).find('.emathequationarraywrapper.ematheqnarray_'+eqnarrayid).last().emathequationarray(settings[eqnarrayid])
            if (iseditable &&  params[1] !== 'authordialog') {
                eqnarray.bind('changed', function(e){
                    var $emeqnarrplace = jQuery(this);
                    var data = $emeqnarrplace.emathequationarray('get');
                    var settings = tiddler.data('ematheqnarray', {});
                    settings[eqnarrayid] = data;
                    var autosavestatus = config.options.chkAutoSave;
                    config.options.chkAutoSave = false;
                    tiddler.setData('ematheqnarray', settings);
                    config.options.chkAutoSave = autosavestatus;
                });
            }
            return true;
            
        }
    }
}
//}}}