/******
@name ElementSet
@version 0.1
@author Petri Salmela <petri.salmela@fourferries.fi>
@type plugin
@requires jQuery x.x.x or newer
@class ElementSet
@description Virtual class for set of book elements. Other classes will inherit this.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Dependency problem in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - Pikaday.js
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(jQuery.fn.pikaday) === 'undefined' && jQuery.fn.pikaday.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}


(function(window, $, ebooklocalizer){
    
    /******
     * Constructor of an elementset
     * @constructor
     * @param {jQuery} place - place for the elementset
     * @param {Object} options - initing data
     ******/
    var ElementSet = function(place, options){
        this.place = jQuery(place);
        if (options) {
            this.init(options);
        }
    }
        
    /******
     * Init the place with data and handlers and show it.
     * @param {Object} options - initing data
     ******/
    ElementSet.prototype.init = function(options){
        this.initData(options);
        this.show();
    }
        
    /******
     * Init data
     * @param {Object} options - data for initing
     ******/
    ElementSet.prototype.initData = function(options){
        // As default language, use the language of the closest element with lang-attribute.
        var lang = this.place.closest('[lang]').attr('lang') || '';
        options = jQuery.extend(true, {}, ElementSet.defaults, {settings: {lang: lang}}, options);
        this.type = options.type;
        this.name = options.name || this.place.attr('data-element-name') || '';
        this.place.addClass('ebook-elementset ebook-' + this.escapeHTML(this.type));
        this.metadata = options.metadata;
        this.metadata.created = this.metadata.created || (new Date()).toString();
        this.data = options.data;
        this.settings = options.settings;
        if (this.name) {
            this.settings.gotolink.elementsetpath.push(this.name);
            this.settings.gotolink.elementid = this.name;
        };
        this.setStyles();
        this.setAttrs();
        this.setRole(this.settings.role);
        this.setMode(this.settings.mode);
        this.contents = options.data.contents;
        this.contentdata = options.data.contentdata;
        this.extras = options.data.extras;
        if (options.data.boxtype === 'textbox' ||options.data.boxtype === 'theorybox' || options.data.boxtype === 'examplebox' || options.data.boxtype === 'definitionbox' || options.data.boxtype === 'hint' || options.data.boxtype === 'extra') {
            options.data.boxclass = options.data.boxtype;
            options.data.boxtype = 'elementbox';
        }
        options.data.boxtype = options.data.boxtype || '';
        this.contentPlaces = this.contentPlaces || {};
        if (this.data.boxtype) {
            this.place.attr('data-boxtype', this.data.boxtype);
        }
        // Sanitize away html-markups
        this.place.attr('data-tags', this.escapeHTML(this.metadata.tags.join(', ')));
        this.place.attr('data-creator', this.escapeHTML(this.metadata.creator));
        this.place.attr('data-created', this.escapeHTML(this.metadata.created));
        this.place.attr('data-modifier', this.escapeHTML(this.metadata.modifier));
        this.place.attr('data-modified', this.escapeHTML(this.metadata.modified));
        //this.initSpecial(options);
    }
    
    ///******
    // * Init actions special for specific subclasses.
    // * @param {Object} options - data for initing
    // ******/
    //ElementSet.prototype.initSpecial = function(options){}
    
    /******
     * Set mode of the elementset
     * @param {String} mode - name of the mode
     ******/
    ElementSet.prototype.setMode = function(mode){
        if (typeof(ElementSet.modes[mode]) === 'undefined') {
            this.settings.mode = 'view';
        }
        this.settings.mode = mode;
        var modesettings = ElementSet.modes[mode];
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Change the mode of the elementset
     ******/
    ElementSet.prototype.changeMode = function(mode){
        if (mode !== this.settings.mode) {
            this.setMode(mode);
            this.show();
        }
    }
    
    /******
     * Set the role of the user
     ******/
    ElementSet.prototype.setRole = function(role){
        if (typeof(ElementSet.roles[role]) === 'undefined') {
            role = this.settings.role;
        };
        var roledata = ElementSet.roles[role] || ElementSet.roles['student'];
        if (this.rolechangeable || typeof(this.rolechangeable) === 'undefined') {
            this.rolechangeable = roledata.rolechangeable;
            this.settings.role = role;
            this.teacherable = roledata.teacherable;
            this.studentable = roledata.studentable;
        };
    };

    /******
     * Get content with index.
     * @param {Number} index - the index of element
     * @returns {String} identifier of the content with given index.
     ******/
    ElementSet.prototype.getContent = function(index){
        return this.contents[index] || '';
    }
        
    /******
     * Set content with index. (Replace existing)
     * @param {Number} index - the index of element
     * @param {String} content - the identifier of content
     ******/
    ElementSet.prototype.setContent = function(index, content){
        this.contents[index] = {id: content};
    }
    
    /******
     * Append content in the end.
     * @param {String} content - the identifier of content
     ******/
    ElementSet.prototype.appendContent = function(content){
        this.contents.push({id: content});
    }
    
    /******
     * Add content in the middle of the contents array. If index is not given, push to the end.
     * @param {String} content - the identifier of content
     * @param {Number} index - the index of element.
     ******/
    ElementSet.prototype.addContent = function(content, index){
        if (typeof(index) === 'number') {
            this.contents.splice(index, 0, {id: content});
        } else {
            this.contents.push({id: content});
        }
    }
    
    /******
     * Move content from one index to other.
     * @param {Number} from - the index in this.contents.
     * @param {Number} to - the index in this.contents.
     ******/
    ElementSet.prototype.moveContent = function(from, to){
        var count = this.contents.length;
        if (from !== to && from !== to - 1 && from >= 0 && to >= 0 && from < count && to < count ) {
            var temp = this.contents.splice(from, 1);
            if (to > from) {
                to = to - 1;
            }
            this.contents.splice(to, 0, temp);
        }
    }
    
    /******
     * Rename elements in this container recoursively.
     * @param {Object} namemap - mapping from old names to new names.
     * @param {Boolean} renameall - Switch to rename all children (create new names, if not given in namemap).
     ******/
    ElementSet.prototype.renameElements = function(namemap, renameall){
        namemap = namemap || {};
        renameall = !!renameall;
        var content, cid, newid, eltype;
        for (var i = 0, len = this.contents.length; i < len; i++){
            content = this.contents[i];
            cid = content.id;
            if (namemap[cid]) {
                newid = namemap[cid];
                content.id = newid;
                this.contentPlaces[newid] = this.contentPlaces[cid];
                delete this.contentPlaces[cid];
                var cplace = this.getContentPlace(newid);
                cplace.attr('data-element-name', newid);
                eltype = cplace.attr('data-ebookelement-type');
                this.getContentPlace(newid).trigger('renameelements', [namemap, true]).trigger('element_changed', {type: eltype});
            } else if (renameall) {
                var cplace = this.getContentPlace(cid);
                eltype = cplace.attr('data-ebookelement-type');
                newid = this.getNewId(eltype);
                content.id = newid;
                this.contentPlaces[newid] = this.contentPlaces[cid];
                delete this.contentPlaces[cid];
                cplace.attr('data-element-name', newid);
                this.getContentPlace(newid).trigger('renameelements', [namemap, true]).trigger('element_changed', {type: eltype});
            };
        };
        this.changed();
    };
    
    /******
     * Go through given data and rename all included elements
     * @param {Object} data - elementset-data possible with nested elementsets
     * @returns {Object} data with renamed element names
     ******/
    ElementSet.prototype.renameData = function(data){
        var contents = data.data.contents;
        var cdata = data.data.contentdata;
        var oldname, newname, olddata, elplace;
        for (var i = 0, len = contents.length; i < len; i++) {
            oldname = contents[i].id;
            olddata = cdata[oldname];
            if (typeof(olddata) === 'undefined') {
                elplace = this.contentPlaces[oldname].children('.elementset-element');
                elplace.trigger('getdata');
                olddata = elplace.data('[[elementdata]]') || {};
            };
            newname = this.getNewId(olddata.type);
            contents[i].id = newname;
            delete cdata[oldname];
            // If olddata is a nested elementset, i.e., has contents-array, rename that too.
            cdata[newname] = (olddata.contents ? this.renameData(olddata) : olddata);
        };
        return data;
    }
    
    /******
     * Get jQuery-object of the element's place.
     * @param {String} cid - the content id
     ******/
    ElementSet.prototype.getContentPlace = function(cid){
        var result = this.contentPlaces[cid] && this.contentPlaces[cid].children('.elementset-element') || $('<div></div>');
        return result;
    }

    /******
     * Get jQuery-object of the element's reference content's place.
     * @param {String} cid - the content id
     ******/
    ElementSet.prototype.getReferencePlace = function(cid){
        var result = this.contentPlaces[cid] && this.contentPlaces[cid].children('.elementset-element-refcontent') || $('<div></div>');
        return result;
    }

    /******
     * Get data
     * @returns {Object} Data of the elementset.
     ******/
    ElementSet.prototype.getData = function(){
        var datalang = this.settings.lang;
        if (this.settings.savemode === 'semilocal') {
            datalang = 'common';
        };
        var dataset = {
            type: this.type,
            name: this.name,
            metadata: {
                creator: this.metadata.creator || this.settings.username,
                created: this.metadata.created,
                modifier: this.settings.username,
                modified: (new Date()).getTime(),
                tags: JSON.parse(JSON.stringify(this.metadata.tags)),
                lang: datalang,
                contenttypes: this.getContentTypes()
            },
            data: {
                contents: [],
                contentdata: {},
                extras: []
            }
        }
        if (this.metadata.ref) {
            dataset.metadata.ref = this.metadata.ref;
        };
        if (this.data.title) {
            dataset.data.title = this.data.title;
        }
        if (this.data.boxtype) {
            dataset.data.boxtype = this.data.boxtype;
        }
        if (this.data.boxclass) {
            dataset.data.boxclass = this.data.boxclass;
        }
        if (this.data.cols) {
            dataset.data.cols = this.data.cols;
        }
        if (this.data.borders) {
            dataset.data.borders = this.data.borders;
        }
        if (this.data['size-x']) {
            dataset.data['size-x'] = this.data['size-x'];
        }
        if (this.data.alignment) {
            dataset.data.alignment = this.data.alignment;
        }
        if (this.data.caption) {
            dataset.data.caption = this.data.caption;
        }
        if (this.data.extramaterial) {
            dataset.data.extramaterial = this.data.extramaterial;
        }
        if (this.data.level) {
            dataset.data.level = this.data.level;
        }
        if (this.data.assignmentNumber) {
            dataset.data.assignmentNumber = this.data.assignmentNumber;
        }
        var contents = dataset.data.contents;
        var contentdata = dataset.data.contentdata;
        var extras = dataset.data.extras;
        for (var i = 0, len = this.contents.length; i < len; i++){
            var cont = this.contents[i];
            contents.push(JSON.parse(JSON.stringify(cont)));
            if (this.settings.savemode === 'local' && this.contentdata[cont.id]) {
                var currcontent = this.contentdata[cont.id];
                var contdata = {
                    type: currcontent.type,
                    metadata: JSON.parse(JSON.stringify(currcontent.metadata)),
                    data: JSON.parse(JSON.stringify(currcontent.data))
                }
                contentdata[cont.id] = contdata;
            };
        }
        for (var i = 0, len = this.extras.length; i < len; i++) {
            var extra = this.extras[i];
            extras.push(JSON.parse(JSON.stringify(extra)));
            if (this.settings.savemode === 'local') {
                contentdata[extra.id] = JSON.parse(JSON.stringify(this.contentdata[extra.id]));
            }
        }
        return dataset;
    }
    
    /******
     * Set data
     * @param {Object} options - data to be set
     ******/
    ElementSet.prototype.setData = function(options){
        this.initData(options);
    };
    
    /******
     * Get types of the content
     ******/
    ElementSet.prototype.getContentTypes = function(){
        var cid, celem, ctype, cbclass;
        var types = [];
        for (var i = 0, len = this.contents.length; i < len; i++) {
            cid = this.contents[i].id;
            celem = this.getContentPlace(cid);
            ctype = celem.attr('data-ebookelement-type');
            if ((ctype === 'assignment' || ctype === 'otherassignment') && types.indexOf(ctype) === -1) {
                types.push(ctype);
            };
            if (ctype === 'elementbox') {
                cbclass = celem.attr('data-boxclass');
                if (types.indexOf(cbclass) === -1) {
                    types.push(cbclass);
                };
            };
        };
        types.sort();
        return types;
    };

    /******
     * Show the elementset in current mode
     ******/
    ElementSet.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        this.place.attr('data-elementrole', this.settings.role);
        this.contentPlaces = {};
        this.removeHandlers();
        this.place.empty().html(ElementSet.template[this.data.boxtype] || ElementSet.template.html);
        this.initHandlersCommon();
        if (!this.editable) {
            this.initHandlersNoneditable();
            if (!this.deferreddraw) {
                this.view();
            };
        } else {
            this.initHandlersEditable();
            this.edit();
        }
    }
    
    /******
     * Show the elementset in view mode
     ******/
    ElementSet.prototype.view = function(){
        this.showTitle();
        this.showContent();
        this.showIcons();
    }
    
    /******
     * Show the elementset in edit mode
     ******/
    ElementSet.prototype.edit = function(){
        this.showTitle();
        this.showContent();
        this.showEditTools();
    }
    
    /******
    * Show title
    * Implemented only on elementsets with title.
    ******/
    ElementSet.prototype.showTitle = function(){};
    
    /******
     * Show draggable link icon
     ******/
    ElementSet.prototype.showIcons = function(){
        var cid, cplace, element, html;
        for (var i = 0, len = this.contents.length; i < len; i++){
            cid = this.contents[i].id;
            //cplace = this.contentPlaces[cid];
            cplace = this.getContentPlace(cid);
            html = [
                '<div class="elementset-iconbar">',
                '<div class="elementset-iconbar-icon" draggable="true" data-icon-action="gotolink">'+ElementSet.icons.link+'</div>',
                '<div class="elementset-iconbar-icon" draggable="true" data-icon-action="copyelement">'+ElementSet.icons.copy+'</div>',
                '</div>'
            ].join('');
            element = $(html);
            cplace.prepend(element);
        };
    }
    
    /******
     * Set special attributes.
     ******/
    ElementSet.prototype.setAttrs = function(){};
    
    /******
     * Set styles for this elementset type
     ******/
    ElementSet.prototype.setStyles = function(){};
    
    /******
    * Show content
    ******/
    ElementSet.prototype.showContent = function(){
        var contentlist = [];
        var reflist = [];
        var assindent = 0;
        var elemblock;
        var extraboxes;
        var eltype, pagechanged = false;
        var currcontainer = this.place.find('.ebook-elementset-body').eq(0);
        var cancontain = this.cancontain().join(' ');
        
        currcontainer.empty();
        if (this.editable) {
            currcontainer.append('<div class="elementset-droptarget" data-droptarget-cancontain="'+ cancontain +'" data-dropindex="0" tabindex="0"></div>');
        }
        // Prepare each subelement in this elementset.
        for (var i = 0, length = this.contents.length; i < length; i++){
            var content = this.contents[i];
            var cid = content.id;
            var indent = content.assignmentindent || 0;
            var elementwrapper = $('<div class="elementset-elementwrapper"><div class="elementset-elementheadbar elementset-elementcontrolbar"></div><div class="elementset-element elementset-element-empty" tabindex="-1">Here should be an element</div><div class="elementset-element-refcontent"></div></div>');
            var element = elementwrapper.children('.elementset-element');
            //elementwrapper.attr('data-element-name', cid);
            element.attr('data-element-name', cid);
            var boxtype = content.boxtype;
            if (boxtype) {
                element.attr('data-boxtype', boxtype);
            }
            this.contentPlaces[cid] = elementwrapper;
            // Decide, if we need to change indentation level.
            if (assindent < indent) {
                assindent = indent;
                currcontainer.append('<ol class="ebook-assignmentindent"><li></li></ol>');
                currcontainer = currcontainer.find('ol.ebook-assignmentindent li').last();
            } else if (assindent > indent) {
                currcontainer = currcontainer.parents('.ebook-assignmentindent').eq(assindent - indent - 1).parent();
                assindent = indent;
            }
            currcontainer.append(elementwrapper);
            //element.attr('data-ebookelement-type', ctype);
            if (this.editable) {
                currcontainer.append('<div class="elementset-droptarget" data-droptarget-cancontain="'+ cancontain +'" data-dropindex="'+(i+1)+'" tabindex="0"></div>');
            }
            if (this.contentdata[cid]) {
                // If the data of element is embedded in the elementset.
                var cdata = this.contentdata[cid];
                this.drawElement(element, cdata);
                // If this data should be saved globally, not locally!
                // Draw it and trigger an event so that it will be moved to global data.
                if (this.editable && this.settings.savemode !== 'local') {
                    eltype = cdata.type;
                    delete this.contentdata[cid];
                    element.trigger('element_changed', {type: eltype});
                    pagechanged = true;
                };
            } else {
                // If the data of element is separate from elementset.
                contentlist.push(cid);
            }
            
            // Collect the list of elements
            reflist.push(cid);
        };
        
        // Some elements were moved from local saving to global saving. So the page data was changed.
        if (pagechanged) {
            this.changed();
        };
        
        // Show extraboxes
        if (this.extras.length > 0) {
            // If this elementset has extraboxes, make a container for them.
            extraboxes = jQuery('<div class="ebookreader-extraboxset"></div>');
            this.place.find('.ebook-elementset-footer').last().append(extraboxes);
        }
        for (var i = 0; i < this.extras.length; i++) {
            // Add an iconbox for each extrabox.
            var extraid = this.extras[i].id;
            var eltype = this.escapeHTML(this.extras[i].type);
            var element = jQuery('<div class="ebookreader-extrabox" data-extratype="'+eltype+'">' + ElementSet.icons[eltype] + '</div>');
            element.attr('data-element-name', this.escapeHTML(extraid));
            extraboxes.append(element);
        }
        // Get data for those elements that don't have data given internally.
        this.getExternalData(contentlist);
        
        // Get data for linked content
        this.getLinkedData(reflist);
        
        // Refresh content, if it needs refreshing.
        this.refreshContent();
    }
    
    /******
     * Show tools for editing.
     ******/
    ElementSet.prototype.showEditTools = function(){
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        var dialoghtml = this.getDialog();
        // Settings dialog for this elementset
        if (dialoghtml) {
            var dialog = [
                '<div class="elementset-dialogwrapper ffwidget-background">',
                '    <div class="elementset-dialog">'+dialoghtml+'</div>',
                '</div>'
            ].join('');
            this.place.prepend(dialog);
        }
        
        // For each child element add to headbar the icon etc.
        var heads;
        // For each child element add remove, drag, etc.
        var controls = [
            '<div class="elementset-elementcontrols-left">',
            '</div>',
            '<div class="elementset-elementcontrols-right">',
            '    <div class="elementset-controlicon elementset-controlicon-remove" title="'+ebooklocalizer.localize('elementset:remove', uilang)+'">'+ElementSet.icons.remove+'</div>',
            '</div>'
        ].join('\n');
        var controlbars = this.getControlbars();
        var headbars = this.getHeadbars();
        var elplace, elinfo, eltype, boxtype;
        elinfo = ElementSet.availableElements.alltypes;
        for (var i = 0, len = this.contents.length; i < len; i++) {
            //controlbars.eq(i).html(controls);
            headbars.eq(i).html(controls);
            elplace = headbars.eq(i).closest('.elementset-elementwrapper').children('.elementset-element');
            eltype = elplace.attr('data-ebookelement-type');
            boxtype = elplace.attr('data-boxtype');
            eltype = boxtype || eltype;
            if (eltype) {
                headbars.eq(i).children('.elementset-elementcontrols-left').html('<div class="elementset-elementhead-icon" draggable="true">'+(elinfo[eltype] && elinfo[eltype].icon || '')+'</div>');
            }
        }
    };
    
    /******
     * Remove an element
     * @param {String} cid - id of the element
     * @param {Boolean} noconfirm - true, if the confirmation is not needed
     ******/
    ElementSet.prototype.removeElement = function(cid, noconfirm) {
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        var index = this.indexOf(cid);
        if (index > -1) {
            // If the element is saved on this level.
            var conf = (noconfirm ? true : confirm(ebooklocalizer.localize('elementset:confirm delete', uilang)));
            if (conf) {
                var elementplace = this.getContentPlace(cid);
                elementplace.trigger('getdata');
                var elementdata = elementplace.data('[[elementdata]]') || {type: 'unknown'};
                elementplace.trigger('element_removed', [{type: elementdata.type, id: cid, lang: lang}]);
                elementplace.trigger('element_destroy');
                var elementwrapper = this.contentPlaces[cid];
                elementwrapper.next('.elementset-droptarget').remove();
                elementwrapper.remove();
                delete this.contentPlaces[cid];
                this.contents.splice(index, 1);
                this.changed();
            };
            this.renumberDroptargets();
        };
    };
    
    /******
     * Append new (editable) element.
     * @param {String} ctype - type of content element.
     * @param {String} cid - id of the new content element.
     ******/
    ElementSet.prototype.appendEditElement = function(ctype, cid, index, data){
        // Add cid to the content list in correct place.
        this.addContent(cid, index);
        // Add element to DOM
        this.appendElementToDOM(ctype, cid, index, data);
        // Get the DOM-element, get the data of element and save it.
        var element = this.getContentPlace(cid);
        if (this.settings.savemode === 'local') {
            try {
                element.trigger('getdata');
                this.contentdata[cid] = element.data('[[elementdata]]');
            } catch (e) {
                console.log(e, ctype);
            }
        } else {
            element.trigger('element_changed', {type: ctype});
        };
        // Notify the change for indexing.
        element.trigger('getindexmetadata');
        var indexmetadata = element.data('[[elementindexmetadata]]') || false;
        var gotolink = JSON.parse(JSON.stringify(this.settings.gotolink));
        gotolink.elementid = cid;
        this.place.trigger('element_modified', [{type: ctype, id: cid, lang: this.settings.lang, gotolink: gotolink, metadata: indexmetadata}]);
        this.changed();
        //this.focusElement(cid);
    }
    
    /******
     * Append and show a single element to the given index.
     * @param {String} ctype - thpe of content element.
     * @param {String} cid - id of the content element.
     * @param {Number} index - the index of the element in this set.
     * @param {Object} data - the content data of the element.
     * @returns {Object} an array of elements to draw by event.
     ******/
    ElementSet.prototype.appendElementToDOM = function(ctype, cid, index, data, assindent){
        data = data || {type: ctype};
        var uilang = this.settings.uilang;
        var content = this.contents[index];
        var contentlist = [];
        assindent = assindent || 0;
        var elinfo = ElementSet.availableElements.alltypes;
        var icon = elinfo[ctype].icon;
        var controls = [
            '<div class="elementset-elementcontrols-left">',
            '    <div class="elementset-elementhead-icon" draggable="true">'+icon+'</div>',
            '</div>',
            '<div class="elementset-elementcontrols-right">',
            '    <div class="elementset-controlicon elementset-controlicon-remove" title="'+ebooklocalizer.localize('elementset:remove', uilang)+'">'+ElementSet.icons.remove+'</div>',
            '</div>'
        ].join('\n');
        var cancontain = this.cancontain().join(' ');
        var indent = content && content.assignmentindent || 0;
        var currcontainer = this.place.children('.ebook-elementset-body');
        var anchor;
        if (this.editable) {
            anchor = currcontainer.children('.elementset-droptarget[data-dropindex="'+index+'"]');
        };
        var elementwrapper = $('<div class="elementset-elementwrapper"><div class="elementset-elementheadbar  elementset-elementcontrolbar">'+controls+'</div><div class="elementset-element elementset-element-empty" tabindex="-1">Here should be an element</div><div class="elementset-element-refcontent"></div></div>');
        var element = elementwrapper.children('.elementset-element');
        element.attr('data-element-name', cid);
        var boxtype = content && content.boxtype || '';
        if (boxtype) {
            element.attr('data-boxtype', boxtype);
        }
        this.contentPlaces[cid] = elementwrapper;
        // Decide, if we need to change indentation level.
        if (assindent < indent) {
            assindent = indent;
            currcontainer.append('<ol class="ebook-assignmentindent"><li></li></ol>');
            currcontainer = currcontainer.find('ol.ebook-assignmentindent li').last();
        } else if (assindent > indent) {
            currcontainer = currcontainer.parents('.ebook-assignmentindent').eq(assindent - indent - 1).parent();
            assindent = indent;
        }
        if (anchor) {
            anchor.after(elementwrapper);
        } else {
            currcontainer.append(elementwrapper);
        };
        if (ctype) {
            element.attr('data-ebookelement-type', ctype);
        };
        if (this.editable) {
            elementwrapper.after('<div class="elementset-droptarget" data-droptarget-cancontain="'+ cancontain +'" data-dropindex="'+(index+1)+'" tabindex="0"></div>');
            this.renumberDroptargets();
        }
        var elementdata = data || this.contentdata[cid];
        if (elementdata) {
            // If the data of element is embedded in the elementset or given.
            this.drawElement(element, elementdata);
        } else {
            // If the data of element is separate from elementset, the redraw must be triggered with event.
            // Collecting list of this kind of elements.
            contentlist.push(cid);
        }
        return contentlist;
    }
    
    /******
     * Renumber droptargets of this elementset
     ******/
    ElementSet.prototype.renumberDroptargets = function() {
        var body = this.place.children('.ebook-elementset-body');
        var droptargets = body.children('.elementset-droptarget');
        droptargets.each(function(index){$(this).removeClass('elementset-draghover').attr('data-dropindex', index)});
    };
    
    /******
     * Get data for content elements from external storage.
     * @param {Array} contentlist - list of content id's.
     ******/
    ElementSet.prototype.getExternalData = function(contentlist){
        // Ask for data for all elements with non-embedded data.
        if (contentlist.length > 0) {
            // If this elementset is in 'common'-language of
            // is translated (is the same language as is selected) then ask for that language,
            // else ask the content in the same language as the elementset.
            var lang = (this.metadata.lang === 'common' || this.metadata.lang === this.settings.lang ? this.settings.lang : this.metadata.lang);
            this.place.trigger('get_bookelement', {
                contentlist: contentlist,
                lang: lang
            });
        }
    }
    
    /******
     * Get data for linked elements from external storage
     * @param {Array} reflist - list of possible reference anchors.
     ******/
    ElementSet.prototype.getLinkedData = function(anchorlist, contentTypes){
        contentTypes = contentTypes || ['booknote', 'homeassignment', 'hamessage']
        if (anchorlist.length > 0) {
            // New request for storage
            this.place.trigger('getcontent', {anchors: anchorlist, contentType: contentTypes});
        };
    };
    
    /**
     * Refresh homeassignment messagemarks
     */
    ElementSet.prototype.refreshHAmarks = function() {
        var anchors = [];
        for (var i = 0, len = this.contents.length; i < len; i++) {
            anchors.push(this.contents[i].id);
        };
        this.place.trigger('getcontent', {anchors: anchors, contentType: ['hamessage']});
    };
    
    /******
     * Trigger 'element_changed'-event.
     ******/
    ElementSet.prototype.changed = function(){
        var lang = this.settings.lang;
        if (this.settings.savemode === 'semilocal') {
            lang = 'common';
        }
        this.place.trigger('element_changed', [{type: this.type, lang: lang}]);
    };
    
    /******
     * Refresh the content
     ******/
    ElementSet.prototype.refresh = function(){
        this.place.trigger('refresh');
    }
    
    /******
     * Full refresh of the content
     ******/
    ElementSet.prototype.refreshFull = function(){
        var scrollparent = ElementSet.findScrollParent(this.place).eq(0);
        var scrolloffset = 0;
        if (scrollparent.length > 0) {
            scrolloffset = scrollparent.scrollTop();
        };
        this.place.trigger('refreshfull');
        if (scrolloffset) {
            scrollparent.scrollTop(scrolloffset);
        }
    }
    
    /******
     * Refresh the given content
     * @param {String} cid - the content id of the content to refresh.
     ******/
    ElementSet.prototype.refreshContentElements = function(cid){
        this.getContentPlace(cid).trigger('refresh');
    }
    
    /******
     * Special things for this elementset type, that must be done, when showing content.
     * Implemented in subclasses.
     ******/
    ElementSet.prototype.refreshContent = function(){}
    
    /******
     * Set focus to given element
     * @param {String} cid - the content id
     ******/
    ElementSet.prototype.focusElement = function(cid){
        this.getContentPlace(cid).trigger('elementfocus');
    }
    
    ///******
    // * Init event handlers
    // ******/
    //ElementSet.prototype.initHandlers = function(){
    //    this.initCommonHandlers();
    //    this.initSpecialHandlers();
    //}
    
    /******
     * Remove event handlers
     ******/
    ElementSet.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Draw an element
     * @param {jQuery} element - jQuery-element to draw to.
     * @param {Object} eldata - the data to draw in to the element.
     * @param {String} savemode - mode of saving (local vs. semiolocal vs. global)
     ******/
    ElementSet.prototype.drawElement = function(element, eldata, savemode){
        var eltype = eldata.type;
        savemode = savemode || this.settings.savemode;
        savemode = (savemode === 'semilocal' ? 'local' : savemode);
        eldata.settings = {
            mode: this.settings.mode,
            savemode: savemode,
            username: this.settings.username,
            lang: this.settings.lang,
            uilang: this.settings.uilang,
            defaultlang: this.settings.defaultlang,
            role: this.settings.role,
            feedback: this.settings.feedback,
            users: this.settings.users,
            userinfo: this.settings.userinfo || {},
            gotolink: JSON.parse(JSON.stringify(this.settings.gotolink)),
            usemargin: this.settings.usemargin
        };
        eldata.settings.gotolink.elementid = element.attr('data-element-name');
        element.attr('data-ebookelement-type', eltype);
        var ellang = eldata.metadata && eldata.metadata.lang || this.settings.lang;
        element.attr('lang', ellang)
        if (eldata.metadata && eldata.metadata.lang !== this.settings.lang) {
            element.addClass('nottranslated');
        };
        var elementinfo = ElementSet.availableElements.alltypes[eltype];
        if (elementinfo) {
            var jq = elementinfo.jquery;
            var boxtype = elementinfo.boxtype;
            if (boxtype) {
                eldata.data = eldata.data || {};
                eldata.data.boxclass = eldata.data.boxclass || eldata.data.boxtype || '';
                eldata.data.boxtype = boxtype;
            }
            // Use jQuery-plugin named as variable jq, if it is listed as available element type.
            try {
                element[jq](eldata);
                element.removeClass('elementset-element-empty');
            } catch (e){
                console.log(e, eltype);
            }
        } else {
            element.parents('.ebook-otherassignment').eq(0).hide();
        }
    }

    /******
     * Init event handlers common to all elementsets.
     ******/
    //ElementSet.prototype.initCommonHandlers = function(){
    ElementSet.prototype.initHandlersCommon = function(){
        var elementset = this;
        
        // Get bookelement as reply.
        this.place.off('reply_get_bookelement').on('reply_get_bookelement', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var missingTranslations = [];
            for (var cname in data){
                var cdata = data[cname];
                var cplace = elementset.getContentPlace(cname);
                elementset.drawElement(cplace, cdata);
            };
            return false;
        });
        this.place.on('getdata', function(event, options){
            if (event.target === this) {
                var data = elementset.getData();
                if (options && options.rename) {
                    data = elementset.renameData(data);
                }
                elementset.place.data('[[elementdata]]', data);
            };
        });
        this.place.on('setdata', function(event, options){
            elementset.initData(options);
            elementset.showContent();
        });
        this.place.on('setmode', function(event, mode){
            if (event.target === this) {
                event.stopPropagation();
                elementset.changeMode(mode);
                // Delegate setmode request for all subelements.
                //for (var cid in elementset.contentPlaces){
                //    elementset.contentPlaces[cid].trigger('setmode', mode);
                //}
            }
        });
        this.place.on('setrole', function(event, role){
            if (event.target === this) {
                event.stopPropagation();
                elementset.setRole(role);
                elementset.show();
            };
        });
        this.place.on('refreshfull', function(event){
            event.stopPropagation();
            if (event.target === this) {
                elementset.changeMode(elementset.settings.mode);
            }
        });
        this.place.on('refresh', function(event){
            event.stopPropagation();
            if (event.target === this) {
                elementset.setAttrs();
                elementset.showTitle();
                // Delegate refresh request for all subelements.
                //for (var cid in elementset.contentPlaces){
                for (var i = 0, len = elementset.contents.length; i < len; i++){
                    var cid = elementset.contents[i].id;
                    elementset.refreshContentElements(cid);
                }
                elementset.refreshContent();
            }
        });
        this.place.on('refresh_hamessages', function(event) {
            event.stopPropagation();
            elementset.refreshHAmarks();
        })
        this.place.on('click', '.homeassignmentelement-marking', function(event, data) {
            event.stopPropagation();
            var msgid = $(this).attr('data-hamessageid');
            elementset.place.trigger('showmessage', {data: {messageid: msgid}});
        });
        this.place.on('click', '.messagelink', function(event) {
            event.stopPropagation();
            var link = $(this);
            var linkid = elementset.getNewId('contentlink');
            var data = {
                toid: link.attr('data-toid'),
                totype: link.attr('data-totype'),
                contentdata: {
                    type: 'linklistelement',
                    metadata: {
                        creator: elementset.settings.username,
                        modifier: elementset.settings.username,
                        created: (new Date()).getTime(),
                        modified: (new Date()).getTime(),
                        tags: []
                    },
                    data: {
                        viewmode: 'list',
                        contents: [
                            {
                                id: linkid
                            }
                        ],
                        contentdata: {}
                    }
                }
            };
            var linkdata = {
                type: 'contentlink',
                name: linkid,
                metadata: {
                    creator: elementset.settings.username,
                    modifier: elementset.settings.username,
                    created: (new Date()).getTime(),
                    modified: (new Date()).getTime(),
                    tags: []
                },
                data: {
                    title: elementset.data.title || '',
                    text: '',
                    type: elementset.place.closest('.elementset-element').attr('data-ebookelement-type'),
                    link: JSON.parse(JSON.stringify(elementset.settings.gotolink))
                }
            };
            data.contentdata.data.contentdata[linkid] = linkdata;
            link.trigger('composemessagedata', [data]);
        })
        // Reply from storage
        this.place.on('reply_getcontent', function(event, data){
            // Show refdata by the type. For example: 'booknote' assiciated with each element in this elementset.
            var refplace, elemlist, refelem, elcid, eldata, eltype, elinfo, jq;
            var elementinfos = ElementSet.availableElements.alltypes;
            data = data || {};
            switch (data.contentType) {
                case 'booknote':
                    for (var anchor in data.refs) {
                        elemlist = data.refs[anchor];
                        refplace = elementset.getReferencePlace(anchor);
                        for (var i = 0, len = elemlist.length; i < len; i++) {
                            refelem = $('<div class="elementset-element-refelement"></div>');
                            refplace.append(refelem);
                            elcid = elemlist[i];
                            refelem.attr('data-element-name', elcid);
                            eldata = data.contentdata[elcid];
                            if (eldata) {
                                eldata.settings = {
                                    username: elementset.settings.username,
                                    uilang: elementset.settings.uilang,
                                    mode: 'view',
                                    role: elementset.settings.role,
                                    users: elementset.settings.users,
                                    usemargin: elementset.settings.usemargin
                                };
                                eltype = eldata.type;
                                elinfo = elementinfos[eltype];
                                if (elinfo) {
                                    jq = elinfo.jquery;
                                    try {
                                        refelem[jq](eldata);
                                    } catch (err) {
                                        console.log(err, eltype);
                                    };
                                };
                            };
                        };
                    };
                    break;
                case 'hamessage':
                    if (typeof(elementset.showHAMessages) === 'function') {
                        elementset.showHAMessages(data);
                    };
                    break;
                case 'marginnote':
                    var notelist, cplace, noteid, mndata;
                    for (var anchor in data.refs) {
                        notelist = data.refs[anchor];
                        cplace = elementset.getContentPlace(anchor);
                        for (var i = 0, len = notelist.length; i < len; i++) {
                            noteid = notelist[i];
                            mndata = data.contentdata[noteid];
                            if (mndata) {
                                cplace.trigger('showselfreview', {data: mndata});
                            };
                        };
                    };
                    break;
                case 'homeassignment':
                default:
                    break;
            };
        }).off('marginnote_changed').on('marginnote_changed', function(event, data) {
            event.stopPropagation();
            elementset.saveSendContent(data);
        }).off('showmarginnotes').on('showmarginnotes', function(event, margin) {
            event.stopPropagation();
            elementset.showMarginNotes(margin);
        }).off('hidemarginnotes').on('hidemarginnotes', function(event, data) {
            event.stopPropagation();
            elementset.hideMarginNotes();
        }).off('editmarginnotes').on('editmarginnotes', function(event, data) {
            event.stopPropagation();
            elementset.editMarginNotes();
        }).off('viewmarginnotes').on('viewmarginnotes', function(event, data) {
            event.stopPropagation();
            elementset.viewMarginNotes();
        }).off('gotolink').on('gotolink', function(event, data){
            var link = data.data && data.data.link;
            var nexttarget = link.elementsetpath[0];
            if (event.target === this) {
                event.stopPropagation();
                elementset.goToLink(data);
            }
        }).off('getelements').on('getelements', function(event, data) {
            event.stopPropagation();
            if (event.target === this) {
                var result = elementset.getElements(!!data.rename);
                if (typeof(data) === 'object') {
                    if (typeof(data.elements) !== 'object' || typeof(data.elements.length) !== 'number') {
                        data.elements = [];
                    };
                    data.elements = data.elements.concat(result);
                };
            };
        });
    };
    
    /******
     * Init event handlers for non-editable elementset.
     ******/
    ElementSet.prototype.initHandlersNoneditable = function(){
        var elementset = this;
        this.place.off('dragstart', '.elementset-iconbar .elementset-iconbar-icon[data-icon-action="gotolink"]').on('dragstart', '.elementset-iconbar .elementset-iconbar-icon[data-icon-action="gotolink"]', function(event,data){
            event.stopPropagation();
            var element = $(this).closest('.elementset-element');
            var gotolink = {
                type: 'contentlink',
                metadata: {
                    creator: elementset.settings.username,
                    modifier: elementset.settings.username,
                    created: (new Date()).getTime(),
                    modified: (new Date()).getTime(),
                    tags: []
                },
                data: {
                    title: element.children('h1.ebook-elementset-title').text() || '',
                    text: '',
                    type: element.attr('data-ebookelement-type'),
                    link: $.extend(true, {}, elementset.settings.gotolink, {elementid: element.attr('data-element-name')})
                }
            };
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text/plain', JSON.stringify(gotolink, null, 4));
            ev.dataTransfer.effectAllowed = 'move';
        });
        this.place.off('dragstart', '.elementset-iconbar .elementset-iconbar-icon[data-icon-action="copyelement"]').on('dragstart', '.elementset-iconbar .elementset-iconbar-icon[data-icon-action="copyelement"]', function(event,data){
            event.stopPropagation();
            var element = $(this).closest('.elementset-element');
            var elemname = element.attr('data-element-name');
            element.trigger('getdata', {rename: true});
            var elemdata = element.data('[[elementdata]]');
            var newname = elementset.getNewId(elemdata.type);
            elemdata.name = newname;
            var copydata = {
                type: 'dragelement',
                data: {
                    dragelement: newname,
                    dragdata: {}
                }
            };
            copydata.data.dragdata[newname] = elemdata;

            // Draggable html-text
            var htmltext = ' ';
            element.trigger('gethtml');
            htmltext = element.data('[[elementhtml]]') || htmltext;
            
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text/html', htmltext);
            ev.dataTransfer.setData('text/plain', JSON.stringify(copydata, null, 4));
            ev.dataTransfer.effectAllowed = 'copyMove';
        });
        this.place.on('click', function(event) {
            if ($(event.target).is('.ebook-elementset[data-extramaterial], .ebook-elementset[data-extramaterial]::before')) {
                event.stopPropagation();
                event.preventDefault();
                $(this).toggleClass('extramaterialvisible');
                if ($(this).is('.extramaterialvisible')) {
                    elementset.view();
                };
            };
        });
    }
    
    /******
     * Init event handlers for editable elementset.
     ******/
    ElementSet.prototype.initHandlersEditable = function(){
        var elementset = this;
        this.place.off('element_changed').on('element_changed', function(event, data){
            //if (!data.lang) {
                data.lang = elementset.settings.lang;
            //};
            if (event.target !== this) {
                var eltype = data.type;
                var lang = data.lang;
                var elname = $(event.target).attr('data-element-name');
                if (elementset.indexOf(elname) > -1) {
                    // If the element was a child of this elementset
                    var place = elementset.getContentPlace(elname);
                    if (elementset.settings.savemode === 'local') {
                        event.stopPropagation();
                        event.preventDefault();
                        place.trigger('getdata');
                        var eldata = place.data('[[elementdata]]');
                        if (!eldata.metadata) {
                            eldata.metadata = {};
                        };
                        eldata.metadata.lang = lang;
                        elementset.contentdata[elname] = eldata;
                        elementset.changed();
                    };
                    // Notify the change for indexing.
                    place.trigger('getindexmetadata');
                    var indexmetadata = place.data('[[elementindexmetadata]]') || false;
                    var gotolink = JSON.parse(JSON.stringify(elementset.settings.gotolink));
                    gotolink.elementid = elname;
                    elementset.place.trigger('element_modified', [{type: eltype, id: elname, lang: elementset.settings.lang, gotolink: gotolink, metadata: indexmetadata}]);
                };
            };
        });
        this.place.on('addelement', function(event, data){
            var index;
            if (typeof(data.index) === 'number' && isFinite(data.index)) {
                index = data.index;
            } else {
                index = elementset.contents.length || 0;
            };
            var eldata, cid;
            try {
                eldata = JSON.parse(data.data);
            } catch (err){
                eldata = {type: 'error'};
            };
            var containertypes = elementset.cancontain();
            var elemtypes = ElementSet.availableElements.alltypes[eldata.type].elementtype;
            var canadd = false;
            for (var i = 0, len = containertypes.length; i < len && !canadd; i++) {
                canadd = canadd || elemtypes.indexOf(containertypes[i]) > -1;
            };
            if (eldata.type && eldata.type !== 'error' && canadd && index > -1) {
                event.stopPropagation();
                cid = elementset.getNewId(eldata.type);
                elementset.appendEditElement(eldata.type, cid, index, eldata);
                elementset.focusElement(cid);
            };
        });
        this.place.off('removeelement').on('removeelement', function(event, data){
            if (event.target !== this) {
                if (elementset.settings.savemode === 'local') {
                    event.stopPropagation();
                    event.preventDefault();
                };
                var noconfirm = data && data.noconfirm || false;
                //var eltype = data && data.type || 'unknown';
                var elname = $(event.target).attr('data-element-name');
                elementset.removeElement(elname, noconfirm);
            }
        });
        this.place.off('renameelements').on('renameelements', function(event, data, renameall){
            event.stopPropagation();
            if (elementset.place[0] === event.target){
                elementset.renameElements(data, renameall);
                elementset.changed();
            };
        });
        this.place.on('click', '.elementset-elementcontrolbar .elementset-controlicon-remove', function(event){
            event.stopPropagation();
            var cid = $(event.target).closest('.elementset-elementwrapper').children('[data-element-name]').attr('data-element-name');
            elementset.getContentPlace(cid).trigger('destroy').trigger('removeelement');
        });
        this.place.on('focusout', '.ebook-elementset-title-editable', function(event){
            event.stopPropagation();
            var title = $(this).mathquill('latex');
            elementset.data.title = title;
            elementset.changed();
        });
        this.place.on('focusout', '.ebook-elementset-tabtitle-editable', function(event){
            event.stopPropagation();
            var $mqelem = $(this);
            var tabtitle = $mqelem.mathquill('latex');
            var $tablists = elementset.place.children('.ebook-elementset-tabbody').children('ul.ebook-elementset-tablist');
            var index = $tablists.index($mqelem.closest('ul.ebook-elementset-tablist'));
            elementset.data.contents[index].title = tabtitle;
            elementset.changed();
        });
        this.place.on('click', '.elementset-addbutton', function(event){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var etype = button.attr('data-addelement-type');
            var elid = elementset.getNewId(etype);
            elementset.appendEditElement(etype, elid);
        });
        this.place.on('click', '.elementset-menudialog', function(event){
            event.stopPropagation();
        });
        this.place.on('click', '.elementset-menuitem', function(event){
            event.stopPropagation();
            event.preventDefault();
            var $button = $(this);
            var tooltype = $button.attr('data-elemset-tool');
            var menutool = $button.closest('.elementset-menutool');
            var $thistool = menutool.find('.elementset-menudialog-'+tooltype);
            var isopen = $thistool.hasClass('menuvisible');
            menutool.find('.elementset-menudialog').removeClass('menuvisible');
            menutool.find('.elementset-menuitem').removeClass('menuopen');
            if (!isopen) {
                $button.addClass('menuopen');
                $thistool.addClass('menuvisible');
                $('body').off('click.elementsetmenu').on('click.elementsetmenu', function(ee){
                    $('.elementset-menudialog').removeClass('menuvisible');
                    $('.elementset-menuitem').removeClass('menuopen');
                    $(this).off('click.elementsetmenu');
                })
            }
        });
        this.place.on('change', '.elementset-dialog input[type="number"]', function(event){
            event.stopPropagation();
            var $input = $(this);
            var inputKey = $input.attr('data-attribute');
            var value = ($input.val() | 0);
            var min = $input.attr('min') | 0;
            var max = $input.attr('max') | 0;
            if (min <= value && value <= max) {
                elementset.data[inputKey] = value;
                elementset.changed();
                elementset.refresh();
                for (var i = 0, len = elementset.contents.length; i < len; i++){
                    var cid = elementset.contents[i].id;
                    elementset.refreshContentElements(cid);
                };
            };
        });
        this.place.on('change', '.elementset-dialog select', function(event){
            event.stopPropagation();
            var $input = $(this);
            var inputKey = $input.attr('data-attribute');
            var value = ($input.val() || '');
            elementset.data[inputKey] = value;
            elementset.changed();
            elementset.refresh();
        });
        this.place.on('change', '.elementset-dialog input[type="radio"]', function(event){
            event.stopPropagation();
            var $input = $(this);
            var inputKey = $input.attr('data-attribute');
            var value = ($input.val() || '');
            elementset.data[inputKey] = value;
            elementset.changed();
            elementset.refresh();
        });
        this.place.on('change', '.elementset-dialog input[type="checkbox"]', function(event){
            event.stopPropagation();
            var $input = $(this);
            var inputKey = $input.attr('data-attribute');
            var value = $input.is(':checked');
            elementset.data[inputKey] = value;
            elementset.changed();
            elementset.refresh();
        });
        this.place.on('change', '.elementset-menudialog-tags input', function(event){
            event.stopPropagation();
            var inputbox = $(this);
            var tags = elementset.escapeHTML(inputbox.val()).split(/,\ */);
            elementset.metadata.tags = tags;
            elementset.changed();
        });
        this.place.on('submit', 'form', function(event){
            event.stopPropagation();
            event.preventDefault();
        });
        this.place.on('elementfocus', function(event){
            event.stopPropagation();
            elementset.place.find('h1.ebook-elementset-title .mathquill-editable').trigger('focus');
        });
        this.place.on('dragend', '.elementset-elementheadbar .elementset-elementhead-icon', function(event){
            var wrapper = $(event.target).closest('.elementset-elementwrapper');
            wrapper.removeClass('elementset-thiscurrentlydragged');
            // We didn't move the element. Clear the currentlyDraggedElement.
            ElementSet.currentlyDraggedElementSet = null;
        });
        this.place.on('dragstart', '.elementset-elementheadbar .elementset-elementhead-icon', function(event){
            event.stopPropagation();
            var wrapper = $(event.target).closest('.elementset-elementwrapper');
            wrapper.addClass('elementset-thiscurrentlydragged');
            var dragelem = wrapper.children('.elementset-element');
            var cid = dragelem.attr('data-element-name');
            // Remember that this element is the one that is dragged (and should be removed, if we are moving).
            ElementSet.currentlyDraggedElementSet = elementset;
            // Create the dragged data, i.e., dragelement with data of all content of dragged element.
            var dragelement = {
                type: 'dragelement',
                data: {
                    dragelement: cid,
                    dragdata: {}
                }
            }
            var elementPlace = elementset.getContentPlace(cid);
            var eltype = elementPlace.attr('data-ebookelement-type');
            elementPlace.trigger('getdata');
            var data = dragelement.data.dragdata;
            data[cid] = elementPlace.data('[[elementdata]]');
            
            if (elementset.settings.savemode === 'global') {
                // If we have global saving, get also all childelements to the dragdata.
                // In case we are copying to another notebook/platform.
                var subel, subcid;
                var subelements = elementPlace.find('.elementset-element');
                for (var i = 0, len = subelements.length; i < len; i++){
                    subel = subelements.eq(i);
                    subcid = subel.attr('data-element-name');
                    subel.trigger('getdata');
                    data[subcid] = subel.data('[[elementdata]]');
                };
            };
            
            // Draggable html-text
            var htmltext = '';
            elementPlace.trigger('gethtml');
            htmltext = elementPlace.data('[[elementhtml]]') || htmltext;
            
            // Set the data in the drag event
            var ev = event.originalEvent;
            ev.dataTransfer.setData('text/html', htmltext);
            ev.dataTransfer.setData('text/plain', JSON.stringify(dragelement, null, 4));
            ev.dataTransfer.effectAllowed = 'copyMove';
            
            // Set classes to show drop targets and remove them after drop.
            $('body').attr('data-nowdragging-element', eltype)
                .attr('data-nowdragging-elementtypes', 'elements');
            $('body').on('dragend.elementset', function(ev){
                $('body')
                    .removeAttr('data-nowdragging-element')
                    .removeAttr('data-nowdragging-elementtypes')
                    .off('dragend.elementset');
            });
        });
        this.place.on('dragenter', '.elementset-droptarget', function(event){
            // Highlight the droptarget.
            $(this).addClass('elementset-draghover');
        });
        this.place.on('dragleave', '.elementset-droptarget', function(event){
            // Remove hightlight from the droptarget.
            $(this).removeClass('elementset-draghover');
        });
        this.place.on('dragover', '.elementset-droptarget', function(event){
            // Prevent default to allow drop!
            event.preventDefault();
        });
        this.place.on('drop', '.elementset-droptarget', function(event){
            event.stopPropagation();
            event.preventDefault();
            var ev = event.originalEvent;
            var $place = $(this);
            var setIndex = $place.attr('data-dropindex') | 0;
            if (ev.dataTransfer.files.length > 0) {
                // If a file was dropped
                var file = ev.dataTransfer.files[0];
                var filetype = file.type;
                var filematch = filetype.match(/^[^/]*/);
                var fileclass;
                if (filematch) {
                    // By file type.
                    fileclass = filematch[0];
                    switch (fileclass) {
                        case 'image':
                            var adddata = {
                                index: setIndex,
                                data: '{"type": "imageelement"}'
                            };
                            elementset.place.trigger('addelement', adddata);
                            var place = elementset.place.find('> .ebook-elementset-body > .elementset-elementwrapper').eq(setIndex).children('.elementset-element');
                            place.trigger('openfile', {file: file});
                            break;
                        case 'audio':
                            var adddata = {
                                index: setIndex,
                                data: '{"type": "audioelement"}'
                            };
                            elementset.place.trigger('addelement', adddata);
                            var place = elementset.place.find('> .ebook-elementset-body > .elementset-elementwrapper').eq(setIndex).children('.elementset-element');
                            place.trigger('openfile', {file: file});
                            break;
                        case 'video':
                            var adddata = {
                                index: setIndex,
                                data: '{"type": "videoelement"}'
                            };
                            if ((/\/ogg$/).test(filetype) && (/\.ogg$/).test(file.name)) {
                                // Firefox guesses that .ogg is video even if it was audio.
                                adddata.data = '{"type": "audioelement"}';
                            };
                            elementset.place.trigger('addelement', adddata);
                            var place = elementset.place.find('> .ebook-elementset-body > .elementset-elementwrapper').eq(setIndex).children('.elementset-element');
                            place.trigger('openfile', {file: file});
                            break;
                        default:
                            break;
                    }
                }
                
            } else {
                // Not file but something else (JSON-data) was dropped.
                var datastr = ev.dataTransfer.getData('text');
                var data = {};
                var oldcid, cid, dragdata, namemap = {};
                try {
                    data = JSON.parse(datastr);
                } catch (err) {
                    data = {type: 'error'};
                }
                if (data.type && data.type !== 'error') {
                    if (data.type === 'dragelement') {
                        // JSON-data was a dragelement, i.e., move or copy from other place.
                        if (ev.ctrlKey || ev.metaKey || ev.altKey) {
                            // Does NOT work onf Firefox.
                            // https://bugzilla.mozilla.org/show_bug.cgi?id=606885
                            // TODO: Decide whether we want to just move with same id
                            // (and subelements with same id's),
                            // or make (recoursively) copies with new id's.
                            // With control key this should be copying, but it does not work yet.
                            oldcid = data.data.dragelement;
                            dragdata = data.data.dragdata;
                            var elemdata = data.data.dragdata[oldcid];
                            for (var name in dragdata){
                                namemap[name] = elementset.getNewId(dragdata[name].type);
                                dragdata[name].name = namemap[name];
                            }
                            //cid = (!event.shiftKey && oldcid) || elementset.getNewId(elemdata.type);
                            cid = namemap[oldcid];
                            elementset.appendEditElement(elemdata.type, cid, setIndex, elemdata);
                            elementset.getContentPlace(cid).trigger('renameelements', [namemap, true]);
                        } else {
                            // TODO: Decide whether we want to just move with same id
                            // (and subelements with same id's),
                            // or make (recoursively) copies with new id's.
                            // Without control key this should be moving.
                            oldcid = data.data.dragelement;
                            dragdata = data.data.dragdata;
                            var elemdata = data.data.dragdata[oldcid];
                            //cid = (!event.shiftKey && oldcid) || elementset.getNewId(elemdata.type);
                            // Add a copy with a new cid, remove the old, rename the new back to old cid.
                            cid = elementset.getNewId(dragdata[oldcid].type);
                            namemap[oldcid] = cid;
                            //elementset.getContentPlace(cid).trigger('removeelement', {noconfirm: true});
                            if (ElementSet.currentlyDraggedElementSet) {
                                var dragelset = ElementSet.currentlyDraggedElementSet;
                                dragelset.renameElements(namemap);
                                elementset.appendEditElement(elemdata.type, oldcid, setIndex, elemdata, true);
                                dragelset.getContentPlace(cid).trigger('destroy').trigger('removeelement', {noconfirm: true});
                                ElementSet.currentlyDraggedElementSet = null;
                                //elementset.refreshFull();
                            } else {
                                for (var name in dragdata){
                                    namemap[name] = elementset.getNewId(dragdata[name].type);
                                    dragdata[name].name = namemap[name];
                                }
                                cid = namemap[oldcid];
                                elementset.appendEditElement(elemdata.type, cid, setIndex, elemdata);
                                elementset.getContentPlace(cid).trigger('renameelements', [namemap, true]);
                            }
                        }
                    } else if (data.type === 'contentlink') {
                        var adddata = {
                            index: setIndex,
                            data: '{"type": "linklistelement"}'
                        };
                        elementset.place.trigger('addelement', adddata);
                        var drtarget = elementset.place.find('> .ebook-elementset-body > .elementset-droptarget[data-dropindex="'+setIndex+'"]').next().find('.linklistelement-droptarget');
                        drtarget.trigger('linkadd', data);
                    } else {
                        // JSON-data was directly some other type of element. Maybe an empty element of some type.
                        var adddata = {
                            index: setIndex,
                            data: datastr
                        };
                        elementset.place.trigger('addelement', adddata);
                    }
                };
                // Make sure, the dragend event goes to the body.
                elementset.place.trigger('dragend');
            };
        });
    };
    
    ///******
    // * Init event handlers special for this type of elementset.
    // ******/
    //ElementSet.prototype.initSpecialHandlers = function(){}

    /******
     * Find the index of an element with given id.
     * @param {String} id - id of an element
     * @returns {Number} The index of an element with id in this.contents or -1 if not foun
     *******/
    ElementSet.prototype.indexOf = function(id){
        var index = -1;
        var i, len = this.contents.length;
        for (i = 0; i < len; i++) {
            if (this.contents[i].id === id) {
                index = i;
                break;
            }
        }
        return index;
    }
    
    /******
     * Get a jQuery-object with headbars of all (direct) subelements.
     * @returns {jQuery} headbars of subelements.
     ******/
    ElementSet.prototype.getHeadbars = function(){
        return this.place.children('.ebook-elementset-body').children('.elementset-elementwrapper').children('.elementset-elementheadbar');
    }

    /******
     * Get a jQuery-object with controlbars of all (direct) subelements.
     * @returns {jQuery} controlbars of subelements.
     ******/
    ElementSet.prototype.getControlbars = function(){
        return this.place.children('.ebook-elementset-body').children('.elementset-elementwrapper').children('.elementset-elementcontrolbar');
    }

    /******
     * Get an id for a new element.
     * @param {String} eltype - Type of an element.
     * @returns {String} An id for the new element.
     ******/
    ElementSet.prototype.getNewId = function(eltype){
        var idparts = [eltype];
        idparts.push(this.settings.username);
        var now = new Date();
        var year = now.getUTCFullYear();
        var month = ('0'+(now.getUTCMonth() +1)).slice(-2);
        var day = ('0'+now.getUTCDate()).slice(-2);
        var hour = ('0'+now.getUTCHours()).slice(-2);
        var minute = ('0'+now.getUTCMinutes()).slice(-2);
        var second = ('0'+now.getUTCSeconds()).slice(-2);
        var msecs = ('00'+now.getUTCMilliseconds()).slice(-3);
        idparts.push(year + month + day + hour + minute + second + msecs);
        idparts.push(Math.floor(1000 * Math.random()));
        return idparts.join('-');
    };
    
    ElementSet.prototype.getDialog = function(){
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        var eltype = this.data.boxtype || this.type;
        var dialogdata = ElementSet.dialogs[eltype];
        var html = [];
        if (dialogdata) {
            for (var i = 0, len = dialogdata.length; i < len; i++) {
                var data = dialogdata[i];
                var value = this.data[data.attribute] || data.defval;
                var text = data.label[uilang] || data.label['en'];
                switch (data.type){
                    case 'range':
                        html.push('<form><label title="'+text+'">'+(data.icon || text)+'</label><input type="range" min="' + data.min + '" max="' + data.max + '" value="'+(typeof(value) !== 'undefined' ? value : data.defval || 1)+'" step="1" data-attribute="'+data.attribute+'" /></form>');
                        break;
                    case 'number':
                        html.push('<form><label title="'+text+'">'+(data.icon || text)+'</label><input type="number" min="' + data.min + '" max="' + data.max + '" value="'+(typeof(value) !== 'undefined' ? value : data.defval || 1)+'" step="1" data-attribute="'+data.attribute+'" placeholder="'+(data.placeholder || '')+'" /></form>');
                        break;
                    case 'select':
                        html.push('<form><label title="'+text+'">'+(data.icon || text)+'</label><select data-attribute="' + data.attribute + '">');
                        for (var ops = 0, opslen = data.options.length; ops < opslen; ops++) {
                            var option = data.options[ops];
                            html.push('<option value="'+option.value+'" ' + (value === option.value ? 'selected="selected"' : '') + '>'+(option.label[uilang] || option.label['en'])+'</option>');
                        }
                        html.push('</select></form>');
                        break;
                    case 'radio':
                        html.push('<form><span title="'+text+'">'+(data.icon || text)+'</span>');
                        var radioname = 'dialogradio-'+(Math.random() + '').replace(/^0./,'');
                        html.push('<div class="elementset-dialog-radio ffwidget-buttonset ffwidget-horizontal">');
                        for (var ops = 0, opslen = data.options.length; ops < opslen; ops++) {
                            var option = data.options[ops];
                            html.push('<label class="elementset-radiolabel"><input type="radio" name="' + radioname + '" value="'+option.value+'" ' + (value === option.value ? 'checked="checked"' : '') + ' data-attribute="'+data.attribute+'" /><span class="elementset-radiobutton ffwidget-label ffwidget-setbutton" title="'+(option.label[uilang] || option.label['en'])+'">'+(option.icon || option.label[uilang] || option.label['en'])+'</span></label>');
                        }
                        html.push('</div>', '</form>');
                        break;
                    case 'checkbox':
                        html.push('<form><label><span title="'+text+'">'+(data.icon || text)+'</span>');
                        var checkname = 'dialogcheckbox-'+(Math.random() + '').replace(/^0./,'');
                        html.push('<input type="checkbox" name="' + checkname + '" ' + (value === true ? 'checked="checked"' : '') + ' data-attribute="' + data.attribute + '" /></label></form>');
                    default:
                        break;
                }
            }
        }
        return html.join('');
    }
    
    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    ElementSet.prototype.cancontain = function(){
        return ['containers', 'elements', 'assignments', 'assignmentsets', 'refs'];
    }
    
    /**
     * Show homeassignment messages as markings
     * @param {Object} data       - The data of hamessage from storage
     * @param {Object} data.refs  - An object with anchors as keys and an array of elementid's as values
     * @param {Object} data.contentdata - Elementids as keys and elements' contentdata as values
     */
    ElementSet.prototype.showHAMessages = function(data) {
        var uilang = this.settings.uilang;
        var elemlist, elemid, elemdata, hamessel, haplace, deadline, overtime, nowtime;
        for (var anchor in data.refs) {
            elemlist = data.refs[anchor];
            haplace = this.place.find('[data-element-name="'+anchor+'"] > .elementset-assignment-hamessages');
            haplace.empty();
            for (var i = 0, len = elemlist.length; i < len; i++) {
                elemid = elemlist[i];
                elemdata = data.contentdata[elemid];
                if (elemdata.data.deadline) {
                    deadline = new Date(elemdata.data.deadline);
                    nowtime = new Date((new Date()).toDateString());
                    if (deadline < nowtime) {
                        overtime = true;
                    } else if (deadline.getTime() === nowtime.getTime()){
                        overtime = 'today';
                    } else {
                        overtime = false;
                    };
                } else {
                    deadline = 0;
                    overtime = false;
                };
                for (var hamesselid in elemdata.data.contentdata) {
                    hamessel = elemdata.data.contentdata[hamesselid];
                    if (hamessel.data.gotolink && hamessel.data.gotolink.data.link.elementid === anchor) {
                        for (var color in hamessel.data.colors) {
                            if (hamessel.data.colors[color]) {
                                haplace.append([
                                    '<div class="homeassignmentelement-marking homeassignment-color-'+this.escapeHTML(color)+'" data-overtime="'+overtime+'" data-deadline="'+(deadline ? deadline.getTime() : 0)+'" data-hamessageid="'+this.escapeHTML(elemid)+'">',
                                    ElementSet.icons.home,
                                    '<div class="homeassignmentelement-marking-info">',
                                    '<div class="homeassignmentelement-marking-info-text">' + this.escapeHTML(hamessel.data.text) + '</div>',
                                    (deadline ? '<div class="homeassignmentelement-marking-info-deadline">' +ebooklocalizer.localize('messageelement:deadline', uilang) + ': ' + deadline.toLocaleDateString(uilang) +'</div>' : ''),
                                    '</div>',
                                    '</div>'
                                ].join(''));
                            };
                        };
                    };
                };
            };
        };
    };

    /******
     * Go to the link target
     * @param {Object} linkdata - link data to the target
     ******/
    ElementSet.prototype.goToLink = function(linkdata){
        var link = linkdata && linkdata.data && linkdata.data.link;
        var nextstop = link.elementsetpath.shift();
        if (nextstop === this.name) {
            nextstop = link.elementsetpath.shift();
        };
        if (nextstop) {
            var element = this.place.find('.elementset-element[data-element-name="'+nextstop+'"]');
            if (element.length > 0) {
                element.get(0).scrollIntoView({behavior: 'smooth', block: 'start'});
                if (link.elementsetpath.length === 0 || nextstop === link.elementid) {
                    this.highlightElement(element, 2000);
                };
                element.trigger('gotolink', linkdata);
            };
        } else if (link.elementid) {
            var element = this.place.find('.elementset-element[data-element-name="'+link.elementid+'"]');
            if (element.length > 0) {
                element.get(0).scrollIntoView({behavior: 'smooth', block: 'start'});
                this.highlightElement(element, 2000);
                element.trigger('gotolink', linkdata);
            };
        };
    };
    
    
    /******
     * Highlight element for a small time
     * @param {jQuery} element - element to highlight
     * @param {Number} time - the time in ms for highlighting
     ******/
    ElementSet.prototype.highlightElement = function(element, time){
        $(element).addClass('elementset-elementhighlight');
        if (time) {
            setTimeout(function(){$(element).removeClass('elementset-elementhighlight')}, time);
        };
    };
    
    /**
     * Show marginnotes on elements
     * @param {Object} margin  An object with functions for handling margin.
     */
    ElementSet.prototype.showMarginNotes = function(margin) {
        var anchors = [];
        for (var i = 0, len = this.contents.length; i < len; i++) {
            anchors.push(this.contents[i].id);
        };
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.selfreviewable').trigger('reply_getmarginplace', [{getMargin: margin.getMargin}]);
        this.getLinkedData(anchors, ['marginnote']);
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.ebook-elementset').trigger('showmarginnotes', [margin]);
    };
    
    /**
     * Hide marginnotes from elements
     */
    ElementSet.prototype.hideMarginNotes = function() {
        var contents = this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.selfreviewable');
        contents.trigger('hideselfreview');
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.ebook-elementset').trigger('hidemarginnotes');
    };
    
    /**
     * Edit marginnotes on elements
     */
    ElementSet.prototype.editMarginNotes = function() {
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.selfreviewable').trigger('setmode', 'noteview');
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.ebook-elementset').trigger('editmarginnotes');
    };
    
    /**
     * View marginnotes on elements
     */
    ElementSet.prototype.viewMarginNotes = function() {
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.selfreviewable').trigger('setmode', 'view');
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element.ebook-elementset').trigger('viewmarginnotes');
    };
    
    /**
     * Save / Send content (trigger event).
     * @param {Object} content       data of the content
     * @param {String} content.contentType    Type of the content
     * @param {String} content.id    id of the content
     * @param {String} content.anchorid   id where to anchor the content
     * @param {String} content.lang  The content language
     * @param {Object} content.data  Data of the content
     * @param {Array}  content.sendto    A list of recipients
     * @param {Array}  content.sendfrom  The userid of the sender
     */
    ElementSet.prototype.saveSendContent = function(content) {
        var uilang = this.settings.uilang;
        var anchors = [content.anchorid || this.name];
        var sdata = {
            "name": content.id,
            "contentType": content.contentType,
            "lang": content.lang || "common",
            "anchors": anchors,
            "contentdata": content.data,
            "contentinfo": {
                "name": content.name,
                "contentType": content.contentType,
                //"subcontext": this.name,
                "lang": content.lang || "common",
                "sendto": content.sendto || [],
                "sendfrom": content.sendfrom || this.settings.username,
                "timestamp": (new Date()).getTime(),
                "read": false,
                "recipientopened": false,
                "isoutbox": true
            }
        };
        this.place.trigger('setcontent', [sdata, true]);
        this.place.trigger('savecontent');
    }
    
    /**
     * Get a list of elements inside this elementset as an array
     * @param {Boolean} rename    Rename all elements
     * @return {Array} an array of all elements including this elementset
     */
    ElementSet.prototype.getElements = function(rename) {
        var result = [];
        var getdata;
        var i, len, cid, cplace, eldata;
        if (this.settings.savemode !== 'local') {
            for (i = 0, len = this.contents.length; i < len; i++) {
                cid = this.contents[i].id;
                getdata = {elements: [], rename: rename};
                cplace = this.contentPlaces[cid].children('.elementset-element');
                cplace.trigger('getelements', getdata);
                if (getdata.elements.length > 0) {
                    result = result.concat(getdata.elements);
                } else {
                    cplace.trigger('getdata');
                    eldata = cplace.data('[[elementdata]]');
                    eldata.name = eldata.name || cid;
                    result.push(eldata);
                };
            };
        };
        var setdata = this.getData();
        result.push(setdata);
        if (rename) {
            result = this.renameElementList(result);
        }
        return result;
    };
    
    /**
     * Rename all elements that are listed in the last elementset in the parameter list
     * @param {Array} elements    An array of element data, the last item is the data of this elementset
     * @returns {Array}           The same list with some elements renamed.
     */
    ElementSet.prototype.renameElementList = function(elements) {
        var currSet = elements[elements.length - 1];
        var elToRename = currSet.data.contents;
        var nameMap = {};
        for (let i = 0, len = elToRename.length; i < len; i++) {
            nameMap[elToRename[i].id] = true;
        };
        for (let i = 0, len = elements.length; i < len; i++) {
            let name = elements[i].name;
            if (nameMap[name]) {
                let etype = elements[i].type;
                let newname = this.getNewId(etype);
                nameMap[name] = newname;
                elements[i].name = newname;
            };
        };
        for (let i = 0, len = currSet.data.contents.length; i < len; i++) {
            let curname = currSet.data.contents[i].id;
            if (typeof(nameMap[curname]) === 'string') {
                currSet.data.contents[i].id = nameMap[curname];
            } else if (nameMap[curname] && currSet.data.contentdata[curname]) {
                let newname = this.getNewId(currSet.data.contentdata[curname].type);
                currSet.data.contents[i].id = newname;
                currSet.data.contentdata[newname] = currSet.data.contentdata[curname];
                delete currSet.data.contentdata[curname];
            };
        };
        return elements;
    };
    
    ElementSet.prototype.sanitize = function(text, options) {
        options = $.extend({
            ALLOWED_TAGS: []
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    ElementSet.prototype.escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    /******
     * Default data and settings.
     ******/
    ElementSet.defaults = {
        type: 'elementset',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            contents: [],
            contentdata: {},
            extras: []
        },
        settings: {
            mode: 'view',
            role: 'student',
            savemode: 'local',
            lang: 'en',
            username: 'Anonymous',
            uilang: 'en',
            gotolink: {
                elementsetpath: [],
                elementid: ''
            }
        }
    }
    
    /******
     * Icons for elementsets
     ******/
    ElementSet.icons = {
        extra: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="50" viewbox="0 0 50 50" class="ebook-icon ebook-icon-extra"><path stroke="none" fill="white" d="M19 11 a2 2 0 0 1 2 -2 l6 0 a2 2 0 0 1 2 2 l0 8 l8 0 a2 2 0 0 1 2 2 l0 6 a2 2 0 0 1 -2 2 l-8 0 l0 8 a2 2 0 0 1 -2 2 l-6 0 a2 2 0 0 1 -2 -2 l0 -8 l-8 0 a2 2 0 0 1 -2 -2 l0 -6 a2 2 0 0 1 2 -2 l8 0 l0 -8z" /><path class="icon-maincolor" stroke="none" fill="black" d="M20 12 a2 2 0 0 1 2 -2 l6 0 a2 2 0 0 1 2 2 l0 8 l8 0 a2 2 0 0 1 2 2 l0 6 a2 2 0 0 1 -2 2 l-8 0 l0 8 a2 2 0 0 1 -2 2 l-6 0 a2 2 0 0 1 -2 -2 l0 -8 l-8 0 a2 2 0 0 1 -2 -2 l0 -6 a2 2 0 0 1 2 -2 l8 0 l0 -8z" /></svg>',
        hint: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="50" viewbox="0 0 50 50" class="ebook-icon ebook-icon-hint"><text fill="white" x="12" y="39" style="font-size: 45px; font-weight: bold; font-family: Lucida, \'Times New Roman\', serif;">?</text><text class="icon-maincolor" fill="black" x="13" y="40" style="font-size: 45px; font-weight: bold; font-family: Lucida, \'Times New Roman\', serif;">?</text></svg>',
        settings: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="3 3 44 44" class="ebook-icon ebook-icon-gear ebook-icon-settings"><path stroke="none" fill="white" d="M21 6 a20 20 0 0 1 10 0 a3 6 22.5 0 0 5.3 2.2 a20 20 0 0 1 7 7 a3 6 67.5 0 0 2.2 5.3 a20 20 0 0 1 0 10 a3 6 112.5 0 0 -2.2 5.3 a20 20 0 0 1 -7 7 a3 6 157.5 0 0 -5.3 2.2 a20 20 0 0 1 -10 0 a3 6 -157.5 0 0 -5.3 -2.2 a20 20 0 0 1 -7 -7 a3 6 -112.5 0 0 -2.2 -5.3 a20 20 0 0 1 0 -10 a3 6 -67.5 0 0 2.2 -5.3 a20 20 0 0 1 7 -7 a3 6 -22.5 0 0 5.3 -2.2z m5 13 a7 7 0 0 0 0 14 a7 7 0 0 0 0 -14z" /><path stroke="none" fill="black" d="M20 5 a20 20 0 0 1 10 0 a3 6 22.5 0 0 5.3 2.2 a20 20 0 0 1 7 7 a3 6 67.5 0 0 2.2 5.3 a20 20 0 0 1 0 10 a3 6 112.5 0 0 -2.2 5.3 a20 20 0 0 1 -7 7 a3 6 157.5 0 0 -5.3 2.2 a20 20 0 0 1 -10 0 a3 6 -157.5 0 0 -5.3 -2.2 a20 20 0 0 1 -7 -7 a3 6 -112.5 0 0 -2.2 -5.3 a20 20 0 0 1 0 -10 a3 6 -67.5 0 0 2.2 -5.3 a20 20 0 0 1 7 -7 a3 6 -22.5 0 0 5.3 -2.2z m5 13 a7 7 0 0 0 0 14 a7 7 0 0 0 0 -14z" /></svg>',
        add: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 25 25" class="ebook-icon ebook-icon-add"><path stroke="none" fill="white" d="M11 6 l5 0 l0 5 l5 0 l0 5 l-5 0 l0 5 l-5 0 l0 -5 l-5 0 l0 -5 l5 0z" /><path stroke="none" d="M10 5 l5 0 l0 5 l5 0 l0 5 l-5 0 l0 5 l-5 0 l0 -5 l-5 0 l0 -5 l5 0z" /></svg>',
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewbox="0 0 30 30" class="mini-icon mini-icon-trashcan-open"><path style="stroke: none; fill: white;" d="M6 6.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M8 9 l16 0 l-3 20 l-10 0z M10 11 l2 15 l2 0 l-1 -15z M14.5 11 l0.5 15 l2 0 l0.5 -15z M22 11 l-3 0 l-1 15 l2 0z" /><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z" /></svg>',
        tags:'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-label mini-icon-tag"><path style="stroke: none;" d="M25 3 a2 2 0 0 1 2 2 l0 8 l-14 14 l-10 -10 l14 -14z m-1.5 2 a1.5 1.5 0 0 0 0 3 a1.5 1.5 0 0 0 0 -3z"></path><path style="stroke: none; fill: white;" d="M8 17 l8 -8 l5 5 l-8 8z m3 1 l1 1 l6 -6 l-1 -1z"></path></svg>',
        drag: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-drag"><path style="stroke: none;" d="M15 3 l5 5 l-4 0 l0 6 l6 0 l0 -4 l5 5 l-5 5 l0 -4 l-6 0 l0 6 l4 0 l-5 5 l-5 -5 l4 0 l0 -6 l-6 0 l0 4 l-5 -5 l5 -5 l0 4 l6 0 l0 -6 l-4 0 z"></path></svg>',
        link: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-link"><path style="stroke: none;" d="M17 10 a1.5 1.5 0 1 1 3 3 l-8 8 a1.5 1.5 0 0 1 -3 -3 z M20 6 l-6 6 a4 4 0 0 1 1 -4 l4 -4 a3 3 0 0 1 7 7 l-4 4 a4 4 0 0 1 -4 1 l6 -6 a2.5 2.5 0 1 0 -4 -4 z M5 21 l6 -6 a4 4 0 0 0 -4 1 l-4 4 a3 3 0 0 0 7 7 l4 -4 a4 4 0 0 0 1 -4 l-6 6 a2.5 2.5 0 1 1 -4 -4 z"></path></svg>',
        copy: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="mini-icon mini-icon-copy"><path style="stroke: none;" d="M4 2 l16 0 l0 3 l-11 0 l0 18 l-5 0z m7 5 l8 0 l0 8 l8 0 l0 11 l-16 0z m10 0 l6 6 l-6 0z" /></svg>',
        home: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path stroke="none" d="M4 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        message: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z"></path> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z"></path></svg>'
    }
    
    /******
     * Templates
     ******/
    ElementSet.template = {
        html: [
            '<div class="ebook-elementset-header"></div>',
            '<h1 class="ebook-elementset-title ebook-elementbox-title"></h1>',
            '<div class="ebook-elementset-body"></div>',
            '<div class="ebook-elementset-footer"></div>'
        ].join('\n'),
        assignment: [
            '<div class="elementset-assignment-hamessages"></div>',
            '<div class="ebook-elementset-header"></div>',
            '<div class="ebook-elementset-titlebox">',
            '    <h1 class="ebook-elementset-title ebook-elementbox-title"></h1>',
            '    <div class="elementset-assignmenticons"></div>',
            '    <div class="elementset-assignment-level"></div>',
            '</div>',
            '<div class="elementset-assignment-extras"></div>',
            '<div class="ebook-elementset-body"></div>',
            '<div class="ebook-elementset-footer"></div>'
        ].join('\n'),
        quizset: [
            '<div class="elementset-assignment-hamessages"></div>',
            '<div class="ebook-elementset-header"></div>',
            '<div class="ebook-elementset-titlebox">',
            '    <h1 class="ebook-elementset-title ebook-elementbox-title"></h1>',
            '    <div class="elementset-assignmenticons"></div>',
            '    <div class="elementset-assignment-level"></div>',
            '</div>',
            '<div class="elementset-assignment-extras"></div>',
            '<div class="ebook-elementset-body"></div>',
            '<div class="ebook-elementset-footer"></div>'
        ].join('\n'),
        tabbox: [
            '<div class="ebook-elementset-header"></div>',
            '<div class="ebook-elementset-title ebook-elementset-tabcaption"></div>',
            '<div class="ebook-elementset-body ebook-elementset-tabbody"></div>',
            '<div class="ebook-elementset-footer"></div>'
        ].join('\n'),
        imagebox: [
            '<div class="ebook-elementset-header"></div>',
            '<div class="ebook-elementset-title"></div>',
            '<div class="ebook-elementset-body"></div>',
            '<div class="ebook-elementset-footer"></div>'
        ].join('\n')
    }
    
    /******
     * Setting dialogs for elementset types.
     ******/
    ElementSet.dialogs = {};
    
    /******
     * Modes
     ******/
    ElementSet.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    };
    
    ElementSet.roles = {
        student: {
            studentable: true,
            teacherable: false,
            rolechangeable: false
        },
        teacher: {
            studentable: false,
            teacherable: true,
            rolechangeable: true
        }
    }
    
    /******
     * Localization strings
     ******/
    ElementSet.localization = {
        "en": {
            "elementset:tags": "Tags",
            "elementset:confirm delete": "Do you really want to delete this element?",
            "elementset:settings": "Settings",
            "elementset:addelement": "Add",
            "elementset:remove": "Remove",
            "elementset:creator": "Creator",
            "assignment:searchinfo": "Search for more information.",
            "assignment:externalapp": "Use an external application.",
            "assignment:nocalculator": "Don't use calculator.",
            'elementbox:theorybox': 'Theory',
            'elementbox:examplebox': 'Example',
            'elementbox:definitionbox': 'Definition'
        },
        "fi": {
            "elementset:tags": "Avainsanat",
            "elementset:confirm delete": "Haluatko varmasti poistaa tämän elementin?",
            "elementset:settings": "Asetukset",
            "elementset:addelement": "Lisää",
            "elementset:remove": "Poista",
            "elementset:creator": "Tekijä",
            "assignment:searchinfo": "Etsi lisää tietoa.",
            "assignment:externalapp": "Käytä ulkoista sovellusta.",
            "assignment:nocalculator": "Älä käytä laskinta.",
            'elementbox:theorybox': 'Teoria',
            'elementbox:examplebox': 'Esimerkki',
            'elementbox:definitionbox': 'Määritelmä'
        },
        "sv": {
            "elementset:tags": "Nyckelord",
            "elementset:confirm delete": "Är du säker på att du vill radera detta element?",
            "elementset:settings": "Inställningar",
            "elementset:addelement": "Lägg till",
            "elementset:remove": "Ta bort",
            "elementset:creator": "Upphovsman",
            "assignment:searchinfo": "Sök tilläggsinformation.",
            "assignment:externalapp": "Använd ett externt program.",
            "assignment:nocalculator": "Använd inte räknare.",
            "elementbox:theorybox": "Teori",
            "elementbox:examplebox": "Exempel",
            "elementbox:definitionbox": "Definition"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(ElementSet.localization);
    }

    /******
     * Stylesheet for elementsets.
     ******/
    ElementSet.styles = [
        '[data-boxtype] {margin: 0;}',
        '[data-boxtype="contenttable"] table.ebook-contenttable td {vertical-align: top;}',
        '.elementset-addbuttonbar {margin:0.2em 0.8em;}',
        //'.ebook-pageelement {min-width: 20em; padding-bottom: 1.5em;}',
        '.ebook-elementbox {margin: 0;}',
        '.ebook-elementbox > h1 {margin: 0;}',
        //'.ebook-pageelement:after {content: ""; display: block; clear: both;}',
        '.ebook-elementset:after {content: ""; display: block; clear: both;}',
        '.ebook-elementset[data-alignment="right"] {clear: right; max-width: 100%;}',
        '.ebook-elementset[data-alignment="left"] {clear: left; max-width: 100%;}',
        '.ebook-elementset {clear: both; position: relative;}',
        '.ebook-elementset-title-editable {display: block; margin: 0.1em 0.3em;}',
        '.ebook-elementset-title {margin: 0.1em 0.5em;}',
        '.ebook-elementset-title.ebook-elementset-tabcaption {margin: 0.2em 0.5em;}',
        '.ebook-elementset-body {padding: 0;}',
        '.elementset-addbuttonbar span.ffwidget-setbutton {padding:3px 5px;}',
        '.ebook-elementbox {position: relative;}',
        //'.ebook-elementset[data-elementmode="edit"], .ebook-elementset[data-elementmode="author"] {border: 1px solid #777; margin-top: 0; background-color: white;}', // box-shadow: 5px 5px 5px rgba(0,0,0,0.5);
        '.ebook-elementset[data-elementmode="edit"], .ebook-elementset[data-elementmode="author"] {background-color: white;}', // box-shadow: 5px 5px 5px rgba(0,0,0,0.5);
        '.ebook-elementset[data-elementmode="edit"][data-boxtype="tabbox"], .ebook-elementset[data-elementmode="author"][data-boxtype="tabbox"] {background-color: #f9f9f9;}',
        '.ebook-elementset[data-elementmode="edit"] .ebookreader-tabelement, .ebook-elementset[data-elementmode="author"] .ebookreader-tabelement {background-color: white;}',
        '[data-elementmode="edit"] > .ebook-elementset-tabbody > .ebookreader-tabelement, [data-elementmode="author"] > .ebook-elementset-tabbody > .ebookreader-tabelement {display: block; margin-bottom: 1em; border-top: none; border-radius: 0 0 5px 5px; margin-top: 0;}',
        '[data-boxtype="contenttable"][data-elementmode="edit"] table.ebook-contenttable, [data-boxtype="contenttable"][data-elementmode="author"] table.ebook-contenttable {width: 100%; box-sizing: border-box; -webkit-box-sizing: border-box; -moz-box-sizing: border-box;}',
        '[data-boxtype="contenttable"][data-elementmode="edit"] table.ebook-contenttable > tbody > tr > td, [data-boxtype="contenttable"][data-elementmode="author"] table.ebook-contenttable > tbody > tr > td {border: 1px dashed #999; min-height: 1em;}',
        '[data-borders="dotted"] table.ebook-contenttable, [data-borders="dashed"] table.ebook-contenttable, [data-borders="solid"] table.ebook-contenttable {border-collapse: collapse;}',
        '[data-borders="dotted"] > .ebook-elementset-body > table.ebook-contenttable > tbody > tr > td {border: 1px dotted #666;}',
        '[data-borders="dashed"] > .ebook-elementset-body > table.ebook-contenttable > tbody > tr > td {border: 1px dashed #666;}',
        '[data-borders="solid"] > .ebook-elementset-body > table.ebook-contenttable > tbody > tr > td {border: 1px solid #666;}',
        '.elementset-menutool {position: relative; margin-top: 40px;}',
        '.elementset-menulist {position: absolute; bottom: 0; right: 0; border-top: 1px solid #aaa; border-left: 1px solid #aaa; border-radius: 5px 0 2px 0; white-space: nowrap;}', // box-shadow: 0 5px 8px rgba(0,0,0,0.3);
        '.elementset-menuitem, .elementset-menubutton {padding: 3px; display: inline-block; text-shadow: -1px -1px 1px rgba(0,0,0,0.2), 2px 2px 1px rgba(255,255,255,0.5); cursor: pointer;}',
        '.elementset-menuitem:hover, .elementset-menubutton:hover {background-color: rgba(255,255,255,0.5);}',
        '.elementset-menuitem.menuopen {background-color: rgba(255,255,255,0.6); box-shadow: inset 1px 1px 2px rgba(0,0,0,0.3), inset -1px -1px 2px rgba(255,255,255,0.3);}',
        '.elementset-menudialog {display: none; position: absolute; top: 0; right: 0; padding: 0.2em; border-bottom: 1px solid #aaa; border-top: 1px solid #aaa; border-left: 1px solid #aaa; border-radius: 5px 0 5px 5px; box-shadow: 5px 5px 5px rgba(0,0,0,0.3); white-space: nowrap;}',
        '.elementset-menudialog.menuvisible {display: block; z-index: 2;}',
        '.elementset-menudialog input[type="number"] {width: 3em;}',
        '.elementset-menuitem svg, .elementset-menubutton svg {width: 15px; height: 15px;}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="left"], [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="left"] {float: left; clear: left; justify-content: flex-start;}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="right"], [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="right"] {float: right; float: right; justify-content: flex-end;}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="center"], [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper[data-alignment="center"] {float: none; margin: 0 auto;}',
        'div.ebook-elementset[data-elementmode="view"][data-alignment="left"], div.ebook-elementset[data-elementmode="review"][data-alignment="left"] {float: left; clear: left; margin: 0.5em 1em 1em 0.5em; z-index: 3; position: relative;}',
        'div.ebook-elementset[data-elementmode="view"][data-alignment="right"], div.ebook-elementset[data-elementmode="review"][data-alignment="right"] {float: right; clear: right; margin: 0.5em 0.5em 1em 1em; z-index: 3; position: relative;}',
        'div.ebook-elementset[data-elementmode="view"][data-alignment="center"] {float: none; margin: 0.5em auto;}',
        '.elementset-elementwrapper[data-alignment="left"], .elementset-elementwrapper[data-alignment="right"] {position: relative; z-index: 3;}',
        '[data-boxclass="imagebox"] .ebook-elementset-body {text-align: center;}',
        '[data-boxclass="imagebox"] .ebook-elementset-title {padding: 0 0.5em;}',
        '.elementset-element:focus {outline: 2px solid rgba(0,0,0,0.05); outline: 2px solid rgba(255,255,255,0.2); background-color: rgba(220,220,220,0.1);}',
        
        // Drag'n'drop
        '.elementset-droptarget span {visibility: hidden;}',
        '.ebook-elementbox[data-elementmode="edit"] .elementset-droptarget, .ebook-elementbox[data-elementmode="author"] .elementset-droptarget, .ebook-elementset[data-elementmode="edit"] .elementset-droptarget, .ebook-elementset[data-elementmode="author"] .elementset-droptarget {height: 0;}',
        '.ebook-elementbox[data-elementmode="edit"] .elementset-droptarget.elementset-draghover, .ebook-elementbox[data-elementmode="author"] .elementset-droptarget.elementset-draghover, .ebook-elementset[data-elementmode="edit"] .elementset-droptarget.elementset-draghover, .ebook-elementset[data-elementmode="author"] .elementset-droptarget.elementset-draghover {background-color: #ccc;}',
        'body[data-nowdragging-elementtypes="containers"] .elementset-droptarget[data-droptarget-cancontain~="containers"],',
        '  body[data-nowdragging-elementtypes="tabcontainers"] .elementset-droptarget[data-droptarget-cancontain~="tabcontainers"],',
        '  body[data-nowdragging-elementtypes="elements"] .elementset-droptarget[data-droptarget-cancontain~="elements"]',
        '  {background-color: #eee; margin: 0; padding: 0;}',
        'body[data-nowdragging-elementtypes="containers"] .elementset-droptarget[data-droptarget-cancontain~="containers"] span,',
        '  body[data-nowdragging-elementtypes="tabcontainers"] .elementset-droptarget[data-droptarget-cancontain~="tabcontainers"] span,',
        '  body[data-nowdragging-elementtypes="elements"] .elementset-droptarget[data-droptarget-cancontain~="elements"] span',
        '  {visibility: visible;}',
        '.elementset-thiscurrentlydragged {box-shadow: 8px 8px 12px rgba(0,0,0,0.3); opacity: 0.3;}',
        '.elementset-thiscurrentlydragged .elementset-droptarget {visibility: hidden;}',
        '.elementset-droptarget-tablecell {width: auto; text-align: center; font-weight: bold;}',
        '[data-dragtype~="element"] .elementset-droptarget[data-droptarget-cancontain~="elements"], [data-dragtype~="container"] .elementset-droptarget[data-droptarget-cancontain~="containers"], [data-dragtype~="studentelement"] .elementset-droptarget[data-droptarget-cancontain~="elements"], [data-dragtype~="studentcontainer"] .elementset-droptarget[data-droptarget-cancontain~="containers"], [data-dragtype~="assignment"] .elementset-droptarget[data-droptarget-cancontain~="assignments"], [data-dragtype~="quizelement"] .elementset-droptarget[data-droptarget-cancontain~="quizelement"], [data-dragtype~="assignment"] .elementset-droptarget[data-droptarget-cancontain~="assignmentsets"], [data-dragtype~="tabcontainer"] .elementset-droptarget[data-droptarget-cancontain~="tabcontainers"] {height: 20px; background-color: #f5f5f5; border: 1px dashed #eee; border-radius: 10px;}',
        '[data-dragtype~="element"] .elementset-droptarget[data-droptarget-cancontain~="elements"]:focus, [data-dragtype~="container"] .elementset-droptarget[data-droptarget-cancontain~="containers"]:focus, [data-dragtype~="studentelement"] .elementset-droptarget[data-droptarget-cancontain~="elements"]:focus, [data-dragtype~="studentcontainer"] .elementset-droptarget[data-droptarget-cancontain~="containers"]:focus, [data-dragtype~="assignment"] .elementset-droptarget[data-droptarget-cancontain~="assignments"]:focus, [data-dragtype~="quizelement"] .elementset-droptarget[data-droptarget-cancontain~="quizelement"]:focus, [data-dragtype~="assignment"] .elementset-droptarget[data-droptarget-cancontain~="assignmentsets"]:focus, [data-dragtype~="tabcontainer"] .elementset-droptarget[data-droptarget-cancontain~="tabcontainers"]:focus {background-color: #ccc; border: 1px solid #aaa; outline: none; box-shadow: inset 2px 2px 3px rgba(100,100,100,0.5);}',
        '[data-canmodify="false"] > .ebook-elementset-body > .elementset-droptarget {display: none;}',
        '[data-canmodify="false"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-elementcontrolbar > .elementset-elementcontrols-left, [data-canmodify="false"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-elementcontrolbar > .elementset-elementcontrols-right {display: none;}',
        '[data-elementmode="edit"] td.elementset-contenttable-lastcell .elementset-droptarget-tablecell, [data-elementmode="author"] td.elementset-contenttable-lastcell .elementset-droptarget-tablecell {height: 2em; text-align: center; font-weight: bold; color: #666; margin: 0;}',
        '.ebook-elementset .elementset-elementwrapper {margin: 0.2em 0.5em;}',
        '.ebook-elementset[data-elementmode="edit"] > .ebook-element-body > .elementset-elementwrapper, .ebook-elementset[data-elementmode="author"] > .ebook-element-body > .elementset-elementwrapper {margin: 0.5em; background-color: rgba(200,200,200,0.1);}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper, [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper, [data-elementmode="edit"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper, [data-elementmode="author"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; align-content: flex-start;}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-element, [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-element, [data-elementmode="edit"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper > .elementset-element, [data-elementmode="author"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper > .elementset-element {-webkit-flex-grow: 1; -ms-flex-grow: 1;  flex-grow: 1; border: 1px solid #aaa;}',
        '.ebook-elementset .ebook-elementset-tabbody > .elementset-elementwrapper {margin: 0;}',
        '.elementset-elementwrapper .elementset-elementcontrolbar {display: none; background-color: #eee;}',
        '[data-elementmode="edit"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-elementheadbar, [data-elementmode="author"] > .ebook-elementset-body > .elementset-elementwrapper > .elementset-elementheadbar, [data-elementmode="edit"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper > .elementset-elementheadbar, [data-elementmode="author"] > .ebook-elementset-body > .ebook-contenttable > tbody > tr > td > .elementset-elementwrapper > .elementset-elementheadbar {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; -webkit-flex-grow: 0; -ms-flex-grow: 0;  flex-grow: 0;-webkit-flex-shrink: 0; -ms-flex-shrink: 0;  flex-shrink: 0; width: 20px; padding: 2px; border-radius: 5px 0 0 5px; border: 1px solid #aaa; border-right: none; text-align: center; overflow: hidden;}',
        '[data-elementmode="edit"] .elementset-elementwrapper .elementset-elementheadbar .elementset-elementhead-icon svg, [data-elementmode="author"] .elementset-elementwrapper .elementset-elementheadbar .elementset-elementhead-icon svg {width: 15px; height: 15px;}',
        '[data-elementmode="edit"] .elementset-elementwrapper .elementset-elementheadbar .elementset-elementhead-icon, [data-elementmode="author"] .elementset-elementwrapper .elementset-elementheadbar .elementset-elementhead-icon {display: inline-block; cursor: move;}',
        '.elementset-elementheadbar:hover .elementset-elementhead-icon {margin: -2px; background-color: #ddd;}',
        '[data-elementmode="edit"] .elementset-elementheadbar:hover .elementset-elementhead-icon svg, [data-elementmode="author"] .elementset-elementheadbar:hover .elementset-elementhead-icon svg {width: 19px; height: 19px;}',
        //'[data-elementmode="edit"] .elementset-elementwrapper .elementset-elementcontrolbar, [data-elementmode="author"] .elementset-elementwrapper .elementset-elementcontrolbar {display: block; height: 20px; padding: 2px; border-radius: 0 0 0.5em 0.5em; border: 1px solid #666; border-top: none;}',
        '.elementset-elementcontrolbar .elementset-elementcontrols-left {}',
        '.elementset-elementcontrolbar .elementset-elementcontrols-right {}',
        '.elementset-elementcontrolbar .elementset-controlicon {display: inline-block; cursor: pointer; opacity: 0; transition: opacity 0.5s; -webkit-transition: opacity 0.5s; -moz-transition: opacity 0.5s; -o-transition: opacity 0.5s;}',
        '.elementset-elementwrapper:hover > .elementset-elementcontrolbar .elementset-controlicon {opacity: 1;}',
        '.elementset-elementcontrolbar .elementset-controlicon svg {height: 20px; width: 20px;}',
        '.elementset-dialogwrapper {border-bottom: black; text-align: right; padding: 2px;}',
        '.elementset-dialog {text-align: left; display: inline-block;}',
        '.elementset-dialog form {display: inline-block; margin-left: 1em;}',
        '.elementset-dialog form > span {margin-right: 0.5em;}',
        
        // elementboxes (theorybox, examplebox, definitionbox,...) (view, edit,...)
        '.ebook-elementbox[data-elementmode="view"][data-boxclass], .ebook-elementbox[data-elementmode="view"][data-boxclass][data-alignment="center"]:not([data-imgwidth]) {margin: 2em 2.5em; padding: 0;}',
        '.ebook-elementbox[data-elementmode="view"] .markdownelement[data-ebookelement-type="markdownelement"] h1 {font-size: 120%;}',
        '.ebook-elementbox[data-elementmode="view"] .markdownelement[data-ebookelement-type="markdownelement"] h2 {font-size: 115%;}',
        '.ebook-elementbox[data-elementmode="view"] .markdownelement[data-ebookelement-type="markdownelement"] h3 {font-size: 110%;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="prerequisites"] h1.ebook-elementbox-title {border-bottom: 1px solid black;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="prerequisites"] ul {border-left: 8px solid black;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="theorybox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="examplebox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="definitionbox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="historybox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="didyouknowbox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="discussionbox"] {margin: 2em 2.5em; padding: 0; border: 1px solid #777; border-radius: 4px;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="theorybox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="examplebox"], .ebook-elementbox[data-elementmode="view"][data-boxclass="definitionbox"] {box-shadow: 5px 5px 5px rgba(0,0,0,0.2);}',
        '.ebook-elementbox[data-elementmode="view"] > h1.ebook-elementbox-title {font-size: 120%; margin: 0; padding: 0.2em 0.4em; min-height: 1.1em;}',
        '.ebook-elementbox[data-boxclass="textbox"][data-elementmode="view"] > h1.ebook-elementbox-title:empty, .ebook-elementbox[data-boxclass="examplesd"][data-elementmode="view"] > h1.ebook-elementbox-title:empty {display: none;}',
        '.ebook-elementbox .markdownelement ol > li > ol > li {list-style-type: lower-alpha;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="theorybox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: #ffa; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="theorybox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="theorybox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: #ffa; border-radius: 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="examplebox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: #e7ffb7; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="examplebox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="examplebox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: #e7ffb7; border-radius: 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="definitionbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: #fcf; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="definitionbox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="definitionbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: #fcf; border-radius: 0;}',
        '.ebook-elementbox[data-elementmode][data-boxclass="historybox"] {background-color: #e3cba4;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="historybox"] {background-color: #e3cba4; border-image: url("data:image/png;base64,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") 30 33 30 30 fill repeat stretch; background-color: transparent; border-color: #e3cba4; border-style: solid; border-width: 30px 33px 30px 30px;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="historybox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: #e3cba4; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="historybox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="historybox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: transparent; border-radius: 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="didyouknowbox"] {background-color: #f7f7f7; padding: 0.5em;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="didyouknowbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: transparent; border-bottom: 2px solid black; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="didyouknowbox"] > h1.ebook-elementbox-title:hover::before {box-shadow: 0 0 10px gold;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="didyouknowbox"] > h1.ebook-elementbox-title::before {content: " "; display: inline-block; height: 40px; width: 40px; margin-right: 0.5em; text-align: bottom; background: transparent url("data:image/png;base64,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") left top no-repeat;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="didyoyknowbox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="didyoyknowbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: transparent; border-radius: 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="discussionbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.2em 0.4em; background-color: transparent; border-radius: 4px 4px 0 0;}',
        '.ebook-elementbox[data-elementmode="view"][data-boxclass="discussionbox"] > h1.ebook-elementbox-title::after {content: " "; display: block; height: 48px; width: 50px; position: absolute; top: -15px; right: -15px; background: url("data:image/png;base64,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") no-repeat center center transparent;}',
        '.ebook-elementbox[data-elementmode="edit"][data-boxclass="discussionbox"] > h1.ebook-elementbox-title, .ebook-elementbox[data-elementmode="author"][data-boxclass="discussionbox"] > h1.ebook-elementbox-title {margin: 0; padding: 0.1em 0.4em; background-color: transparent; border-radius: 0;}',
        
        // PageElement, SolutionElement
        '.ebook-pageelement > .elementset-dialogwrapper, .ebook-solutionelement > .elementset-dialogwrapper {display: none;}',
        //'.ebook-pageelement[data-elementmode="edit"], .ebook-pageelement[data-elementmode="author"] {border: none;}',
        
        // TriggerSet
        '.ebook-triggerset[data-elementmode="view"][data-triggerevent="click"] > .ebook-elementset-header > .ebook-triggerset-triggerelement, .ebook-triggerset[data-elementmode="review"][data-triggerevent="click"] > .ebook-elementset-header > .ebook-triggerset-triggerelement {cursor: pointer;}',
        '.ebook-triggerset[data-elementmode="view"] > .ebook-elementset-header > .ebook-triggerset-triggerelement, .ebook-triggerset[data-elementmode="review"] > .ebook-elementset-header > .ebook-triggerset-triggerelement {padding: 0; min-width: 0;}',
        '.ebook-triggerset[data-elementmode="edit"] > .ebook-elementset-header > .ebook-triggerset-triggerelement, .ebook-triggerset[data-elementmode="author"] > .ebook-elementset-header > .ebook-triggerset-triggerelement {background-color: transparent;}',
        '.ebook-triggerset[data-elementmode="view"] > .ebook-elementset-header > .isrunningaction, .ebook-triggerset[data-elementmode="review"] > .ebook-elementset-header > .isrunningaction {background-color: #fee;}',
        '.ebook-triggerset[data-elementmode="edit"] > .ebook-elementset-header, .ebook-triggerset[data-elementmode="author"] > .ebook-elementset-header {background-color: #eef;}',
        '.ebook-triggerset[data-elementmode="edit"] > .ebook-elementset-body, .ebook-triggerset[data-elementmode="author"] > .ebook-elementset-body {background-color: #ffe; box-shadow: inset 3px 3px 3px rgba(0,0,0,0.4), inset -3px -3px 3px rgba(255,255,255,0.5);}',
        '.ebook-triggerset[data-elementmode="view"] > .ebook-triggerset-triggerelement > .ebook-triggerset-head, .ebook-triggerset[data-elementmode="review"] > .ebook-triggerset-triggerelement > .ebook-triggerset-head {display: none;}',
        '.ebook-triggerset[data-elementmode="edit"] > .ebook-elementset-header > .ebook-triggerset-head, .ebook-triggerset[data-elementmode="author"] > .ebook-elementset-header > .ebook-triggerset-head, .ebook-triggerset[data-elementmode="edit"] > .ebook-elementset-title, .ebook-triggerset[data-elementmode="author"] > .ebook-elementset-title {font-size: 120%;}',
        '.ebook-triggerset[data-elementmode="view"] .ebook-triggerset-icon, .ebook-triggerset[data-elementmode="review"] .ebook-triggerset-icon {position: absolute; top:0; right: 0;}',
        
        // Dialog inputs
        'label.elementset-radiolabel > input[type="radio"] {display: none;}',
        //'label.elementset-radiolabel > span.elementset-radiobutton {display: inline-block; cursor: pointer; vertical-align: middle; padding: 2px; margin: 0; border: 1px solid #666; border-radius: 2px; box-shadow: inset 1px 1px 1px rgba(255,255,255,0.5), inset -1px -1px 1px rgba(0,0,0,0.3), 1px 1px 1px rgba(255,255,255,0.5), -1px -1px 1px rgba(0,0,0,0.3);}',
        'label.elementset-radiolabel > span.elementset-radiobutton {padding: 2px 6px; margin: 0;}',
        'label.elementset-radiolabel > input[type="radio"]:checked + span.elementset-radiobutton {background-color: rgba(255,255,255,0.5); border: 1px solid #666; box-shadow: inset 1px 1px 1px rgba(0,0,0,0.3), inset -1px -1px 1px rgba(255,255,255,0.5), 1px 1px 1px rgba(255,255,255,0.5), -1px -1px 1px rgba(0,0,0,0.3);}',
        'label.elementset-radiolabel > span.elementset-radiobutton svg {width: 20px; height: 20px;}',
        '.elementset-dialog input[type="number"] {width: 5em;}',
        '.elementset-dialog-radio {display: inline-block; margin: 0;}',
        
        // Missing elements
        '.elementset-element-empty {background-color: #faa;}',
        
        // Solutions
        '.elementset-solutionelement-previewtool {margin: 0.3em; width: 30px;}',
        '.elementset-solutionelement-previewtool .elementset-solutionelement-previewicon {padding: 2px;}',
        '.elementset-solutionelement-buttons .elementset-solutionelement-previewicon svg {width: 20px; height: 20px;}',
        '.elementset-solutionelement-previewtool .elementset-solutionelement-previewarea {position: absolute; top: 4px; right: 4px; left: 25px; border: 1px solid black; box-shadow: 8px 8px 12px rgba(0,0,0,0.4); background-color: white; z-index: 10;}',
        
        // Element highlight
        '.elementset-element {box-shadow: 0 0 0 0 transparent; transition: box-shadow 2s ease 0s;}',
        '.elementset-element.elementset-elementhighlight {box-shadow: 0 0 3px 5px rgba(255, 215, 0, 0.3);}',
        
        // Element icons (link, ...)
        '.elementset-element {position: relative;}',
        '.elementset-iconbar {position: absolute; top: 0; right: -18px; width: 15px; padding: 1px; opacity: 0; cursor: move; z-index: 4; border-radius: 3px; border: 1px solid #aaa; background-color: rgba(230,230,230,0.8); transition: opacity 0.3s 0.1s;}',
        '.elementset-element:focus > .elementset-iconbar {opacity: 1; transition: opacity 0.3s 0.1s; outline: none;}',
        '.ebook-imagebox {z-index: 3;}',
        '.ebook-messageelement .elementset-iconbar {display: none;}',
        
        // Extramaterial
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial] {transition: box-shadow 0.5s;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible) {cursor: pointer; height: 0.3em; padding: 0; box-shadow: none!important; border: 1px solid #aaa; padding: 0 0.5em 0.5em; border-radius: 5px;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible):hover {box-shadow: 3px 3px 5px rgba(0,0,0,0.3)!important;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible)::before {margin-top: -0.3em; box-shadow: 0 0 1px 1px #666; font-size: 200%; transition: font-size 0.5s, margin-top 0.5s;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible)::after {content: "\u25BC"; display: block; height: 0.5em; line-height: 0.5em; width: auto; text-align: center;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible) > .ebook-elementset-header, .ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible) > .ebook-elementset-title, .ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible) > .ebook-elementset-body, .ebook-elementset[data-elementmode$="view"][data-extramaterial]:not(.extramaterialvisible) > .ebook-elementset-footer {display: none;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial].extramaterialvisible > .ebook-elementset-body {padding-top: 1em;}',
        'div.ebook-elementset[data-elementmode$="view"][data-extramaterial="advanced"]:not(.extramaterialvisible) {background: #efe!important;}',
        'div.ebook-elementset[data-elementmode$="view"][data-extramaterial="advanced"] {background-color: #efe; box-shadow: 0 0 0 5px #cea!important;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial="advanced"]::before {cursor: pointer; content: "+"; display: inline-block; position: absolute; top: 0; right: 0; margin-top: 0; width: 1em; height: 1em; line-height: 100%; vertical-align: middle; text-align: center; font-weight: bold; background-color: green; color: white; border: 2px solid white; border-radius: 0 5px 0 5px; font-size: 150%; z-index: 1; transition: font-size 0.5s, margin-top 0.5s;}',
        'div.ebook-elementset[data-elementmode$="view"][data-extramaterial="helpful"]:not(.extramaterialvisible) {background: #fed!important;}',
        'div.ebook-elementset[data-elementmode$="view"][data-extramaterial="helpful"] {background-color: #fed; box-shadow: 0 0 0 5px #fed!important;}',
        '.ebook-elementset[data-elementmode$="view"][data-extramaterial="helpful"]::before {cursor: pointer; content: "?"; display: inline-block; position: absolute; top: 0; right: 0; margin-top: 0; width: 1em; height: 1em; line-height: 100%; vertical-align: middle; text-align: center; font-weight: bold; background-color: #ffa858; color: white; border: 2px solid white; border-radius: 0 5px 0 5px; font-size: 150%; z-index: 1; transition: font-size 0.5s, margin-top 0.5s;}',
        
        // Assignmentlist
        '.ebook-assignmentlist {list-style: none;}',
        '.ebook-assignmentlist > li > h1 {font-size: 200%; text-align: center; border-bottom: 3px double black; margin: 1em 0 0;}'

    ].join('\n');
    
    if ($('head style#elementset-style').length === 0) {
        $('head').append('<style id="elementset-style" type="text/css">'+ElementSet.styles+'</style>')
    }
    
    /******
     * Available elements.
     * Elements will update reference to themselves here with jQuery-plugin.
     ******/
    ElementSet.availableElements = {
        alltypes: {},
        pagecontainers: {},
        assignments: {},
        containers: {},
        elements: {},
        studentelements: {},
        studentcontainers: {},
        tabcontainers: {},
        classes: {
            general: [],
            math: []
        }
    }
    
    /******
     * Add new element type to available elements.
     * @param {Object} elemobj - Object with information about element type.
     ******/
    ElementSet.addElementType = function(elemobj){
        elemobj = $.extend(true, {}, elemobj);
        var etype;
        // If the object has desired information
        if (elemobj.type && elemobj.jquery && elemobj.name && elemobj.icon && elemobj.description && elemobj.classes) {
            for (var i = 0, len = elemobj.elementtype.length; i < len; i++){
                etype = elemobj.elementtype[i];
                // Add element to the available ones.
                if (!ElementSet.availableElements[etype]) {
                    ElementSet.availableElements[etype] = {};
                }
                ElementSet.availableElements[etype][elemobj.type] = elemobj;
            }
            ElementSet.availableElements.alltypes[elemobj.type] = elemobj;
            for (var i = 0, length = elemobj.classes.length; i < length; i++) {
                // Add the element type to the classes it can be used in.
                var classname = elemobj.classes[i];
                if (!ElementSet.availableElements.classes[classname]){
                    ElementSet.availableElements.classes[classname] = [elemobj.type];
                } else if (ElementSet.availableElements.classes[classname].indexOf(elemobj.type) === -1) {
                    ElementSet.availableElements.classes[classname].push(elemobj.type);
                }
            }
        }
    };
    
    /******
     * Find the closest scrollable parent element
     * @param {jQuery} element - jQuery-object
     * @return {jQuery} jQuery-object with all ancestors which are scrollable.
     ******/
    ElementSet.findScrollParent = function(element){
        var scrparent = element.parents().filter(function(){
            var parent = $(this);
            if (parent.css('position') === 'static') {
                return false;
            };
            return (/(auto|scroll)/).test( parent.css('overflow') + parent.css('overflow-y'));
        });
        return scrparent;
    };

    /**** jQuery-plugin **********************/
    var elementsetmethods = {
        'init': function(params){
            return this.each(function(){
                var element = new ElementSet(this, params);
            });
        },
        'addelementtype': function(data){
            ElementSet.addElementType(data);
        },
        'getelementtypes': function(data){
            data = data || 'all';
            var result;
            switch (data){
                case 'all':
                    result = $.extend(true, {}, ElementSet.availableElements.alltypes);
                    break;
                case 'classes':
                    result = $.extend(true, {}, ElementSet.availableElements.classes);
                    break;
                default:
                    result =  $.extend(true, {}, ElementSet.availableElements[data]);
            }
            return result;
        },
        'getclass': function(data){
            return ElementSet;
        }
    }
    
    $.fn.elementset = function(method){
        if (elementsetmethods[method]) {
            return elementsetmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return elementsetmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in elementset.');
            return false;
        }
    }












    /******************************************************************************************************************
     * ElementBox
     * @constructor
     * @param {jQuery} place - the place for elementbox
     * @param {Object} options . data for elementbox
     ******************************************************************************************************************/
    var ElementBox = function(place, options){
        //this.parentClass.constr.call(this, place, options);
        //this.constr(place, options);
        ElementSet.call(this, place, options);
    }
    
    // Inherit from ElementSet
    //ElementBox.prototype = new ElementSet();
    ElementBox.prototype = Object.create(ElementSet.prototype);
    ElementBox.prototype.constructor = ElementBox;
    ElementBox.prototype.parentClass = ElementSet.prototype;
    
    /******
     * Show title
     ******/
    ElementBox.prototype.showTitle = function(){
        if (this.editable) {
            this.showTitleEdit();
        } else {
            this.showTitleView();
        }
    }

    /******
     * Show title in view mode
     ******/
    ElementBox.prototype.showTitleView = function(){
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        if (this.data.boxclass === 'theorybox' || this.data.boxclass === 'examplebox'  || this.data.boxclass === 'definitionbox' || this.type === 'assignment' || this.type === 'otherassignment') {
            this.data.title = this.data.title || ' ';
        };
        var localtype = ebooklocalizer.localize('elementbox:'+this.data.boxclass, uilang);
        var titlenumber = (typeof(this.data.elementboxNumber) !== 'undefined' ? '<span class="ebook-elementbox-boxnumber">'+(localtype ? localtype+' <span class="ebook-elementbox-boxnumberplace">'+ (this.data.elementboxNumber || '') : '')+'</span></span> ' : '<span class="ebook-elementbox-boxnumber">'+(localtype ? localtype +'<span class="ebook-elementbox-boxnumberplace"></span>' : '')+'</span> ');
        if (typeof(this.data.title) === 'string' && this.data.title !== '') {
            this.place.find('.ebook-elementset-title').html(titlenumber + this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>').replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>'));
            this.place.find('.ebook-elementset-title .mathquill-embedded-latex').mathquill();
        };
        if (typeof(this.data.caption) === 'string' && this.data.caption !== '') {
            this.place.find('.ebook-elementset-footer').html('<div class="ebook-elementbox-caption">' + this.escapeHTML(this.data.caption).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>').replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>') + '</div>');
            this.place.find('.ebook-elementset-footer .mathquill-embedded-latex').mathquill();
        };
        if (!this.data.elementboxNumber) {
            var numberplace = this.place.find('.ebook-elementset-title .ebook-elementbox-boxnumberplace');
            this.place.trigger('getelementboxnumber', [{
                boxclass: this.data.boxclass,
                id: this.name,
                callback: function(boxnum) {
                    if (boxnum) {
                        numberplace.text(' '+boxnum);
                    };
                }
            }]);
        };
    };
    
    /******
     * Show title in edit mode
     ******/
    ElementBox.prototype.showTitleEdit = function(){
        var lang = this.settings.lang;
        var uilang = this.settings.uilang;
        this.data.title = this.data.title || '';
        if (this.data.boxclass === 'theorybox' || this.data.boxclass === 'examplebox' || this.data.boxclass === 'definitionbox' || this.type === 'assignment' || this.type === 'otherassignment') {
            this.data.title = this.data.title || ' ';
        }
        var titlenumber = (typeof(this.data.elementboxNumber) !== 'undefined' ? '<span class="ebook-elementbox-boxnumber">'+ebooklocalizer.localize('reader:'+this.data.boxtype, uilang)+' '+ (this.data.elementboxNumber || '') +':</span> ' : '');
        if (typeof(this.data.title) === 'string') {
            this.place.find('.ebook-elementset-title').html(titlenumber + '<span class="ebook-elementset-title-editable">' + this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '\$ $1\$') + '</span>');
            this.place.find('.ebook-elementset-title .ebook-elementset-title-editable').mathquill('textbox');
        }
    }
    
    /******
     * Set attributes
     ******/
    ElementBox.prototype.setAttrs = function(){
        if (this.data.boxclass) {
            this.place.attr('data-boxclass', this.data.boxclass);
        };
        if (this.data.borders) {
            this.place.attr('data-borders', this.data.borders);
        } else {
            this.place.removeAttr('data-borders');
        };
        if (this.data.extramaterial && this.data.extramaterial !== 'normal') {
            this.deferreddraw = true;
            this.place.attr('data-extramaterial', this.data.extramaterial);
        };
        if (this.data['size-x']) {
            this.place.attr('data-imgwidth', this.data['size-x']);
            //if (this.settings.mode === 'view' && (this.data.alignment === 'left' || this.data.alignment === 'right')) {
            if (this.data.alignment === 'left' || this.data.alignment === 'right' || this.data.alignment === 'center') {
                if (this.editable) {
                    this.place.parent().css('max-width', (this.data['size-x'] | 0) + 20);                
                } else {
                    this.place.css('max-width', (this.data['size-x'] | 0) + 20);
                };
            };
        };
        if (this.data['size-y']) {
            this.place.attr('data-imgheight', this.data['size-y']);
        };
        if (this.data.alignment) {
            this.place.attr('data-alignment', this.data.alignment);
            this.place.parent().attr('data-alignment', this.data.alignment);
        };
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
        }
    }
    
    /******
     * Show content special for elementbox.
     ******/
    ElementBox.prototype.refreshContent = function(){
        if (this.data.boxtype === 'tabbox') {
            this.makeTabs();
        }
        // TODO: check this
        if (this.data.fullpage) {
            this.place.closest('.notevisible').addClass('notefullpage');
        }
    }

    /******
     * Make tabs from subelements
     ******/
    ElementBox.prototype.makeTabs = function(){
        var tabbox = this;
        var html = [];
        if (!this.editable) {
            html.push('<ul class="ebook-elementset-tablist ffwidget-tablist">');
            for (var i = 0; i < this.data.contents.length; i++){
                var content = this.data.contents[i];
                var contentdata = this.data.contentdata[content.id];
                html.push('<li class="ebookreader-tabitem ffwidget-tabitem'+(i === 0 ? ' ebookreader-tabitem-current ffwidget-tabitem-current' : '')+'" data-tabelement-id="'+content.id+'">'+(content.title || ((i+1)+'.'))+'</li>');
                this.contentPlaces[content.id].addClass('ebookreader-tabelement ffwidget-tabelement');
            }
            html.push('</ul>');
            this.tablist = this.place.children('.ebook-elementset-tabbody');
            this.tablist.children('ul.ebook-elementset-tablist').remove();
            this.tablist.prepend(html.join('\n'));
        } else {
            for (var i = 0; i < this.data.contents.length; i++) {
                var html = [];
                var content = this.data.contents[i];
                html.push('<ul class="ebook-elementset-tablist ffwidget-tablist">');
                html.push('<li class="ebookreader-tabitem ffwidget-tabitem'+(i === 0 ? ' ebookreader-tabitem-current ffwidget-tabitem-current' : '')+'" data-tabelement-id="'+content.id+'"><span class="ebook-elementset-tabtitle-editable">'+(content.title || ((i+1)+'.'))+'</span></li>');
                this.contentPlaces[content.id].addClass('ebookreader-tabelement ffwidget-tabelement');
                html.push('</ul>');
                this.contentPlaces[content.id].before(html.join('\n'));
            }
            this.place.find('.ebook-elementset-tabtitle-editable:not(.mathquill-rendered-math)').mathquill('textbox');
            this.tablist = this.place.children('.ebook-elementset-tabbody');
        }
        if (this.data.contents[0]) {
            this.contentPlaces[this.data.contents[0].id]
                .addClass('ffwidget-tabelement-current');
        }
        this.place.find('.ebook-elementset-body').eq(0).addClass('ffwidget-tabwrapper');
        this.tablist.on('click', 'li.ebookreader-tabitem', function(e){
            e.stopPropagation();
            var $lielem = $(this);
            $lielem.parent().children('li').removeClass('ffwidget-tabitem-current');
            $lielem.addClass('ffwidget-tabitem-current');
            var elemid = $lielem.attr('data-tabelement-id');
            tabbox.place.find('.ffwidget-tabelement-current').removeClass('ffwidget-tabelement-current');
            tabbox.contentPlaces[elemid].addClass('ffwidget-tabelement-current')
            tabbox.refreshContentElements(elemid);
        });
    }
    
    /******
     * Get data
     ******/
    //ElementBox.prototype.getData = function(){
    //}
    
    /******
     * Set data
     ******/
    ElementBox.prototype.setData = function(options){
        this.initData(options);
    }
    
    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    ElementBox.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(ElementBox.elementinfo[this.data.boxtype].cancontain));
    }

    /******
     * Add setting dialogs for elementbox types.
     ******/
    ElementSet.dialogs.elementbox = [
        {
            name: 'extramaterial',
            label: {
                en: 'Extra material',
                fi: 'Lisämateriaali'
            },
            attribute: 'extramaterial',
            defval: 'normal',
            type: 'radio',
            icon: ' ',
            options: [
                {
                    label: {
                        en: 'Normal',
                        fi: 'Normaali'
                    },
                    icon: '_',
                    value: 'normal'
                },
                {
                    label: {
                        en: 'Helpfull',
                        fi: 'Auttava'
                    },
                    icon: '?',
                    value: 'helpful'
                },
                {
                    label: {
                        en: 'Advanced',
                        fi: 'Edistynyt'
                    },
                    icon: '+',
                    value: 'advanced'
                }
            ]
        },
        {
            name: 'boxclass',
            label: {
                en: 'Type',
                fi: 'Tyyppi'
            },
            attribute: 'boxclass',
            defval: 'textbox',
            type: 'radio',
            icon: ' ',
            options: [
                {
                    label: {
                        en: 'Text',
                        fi: 'Teksti'
                    },
                    icon: 'N',
                    value: 'textbox'
                },
                {
                    label: {
                        en: 'Theory',
                        fi: 'Teoria'
                    },
                    icon: 'T',
                    value: 'theorybox'
                },
                {
                    label: {
                        en: 'Example',
                        fi: 'Esimerkki'
                    },
                    icon: 'E',
                    value: 'examplebox'
                },
                {
                    label: {
                        en: 'Definition',
                        fi: 'Määritelmä'
                    },
                    icon: 'D',
                    value: 'definitionbox'
                },
                {
                    label: {
                        en: 'Prerequisites',
                        fi: 'Esitiedot'
                    },
                    icon: 'P',
                    value: 'prerequisites'
                },
                {
                    label: {
                        en: 'History',
                        fi: 'Historia'
                    },
                    icon: 'H',
                    value: 'historybox'
                },
                {
                    label: {
                        en: 'Did you know',
                        fi: 'Tiesitkö'
                    },
                    icon: '!',
                    value: 'didyouknowbox'
                },
                {
                    label: {
                        en: 'Discussion',
                        fi: 'Pohdi'
                    },
                    icon: '?',
                    value: 'discussionbox'
                }
            ]
        },
        {
            name: 'size-x',
            label: {
                en: 'Width',
                fi: 'Leveys'
            },
            attribute: 'size-x',
            type: 'number',
            min: 0,
            max: 800,
            defval: '',
            placeholder: 'auto',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-width"><path style="stroke: none;" d="M1 8 l0 14 l2 0 l0 -7 l4 4 l0 -2.5 l16 0 l0 2.5 l4 -4 l0 7 l2 0 l0 -14 l-2 0 l0 7 l-4 -4 l0 2.5 l-16 0 l0 -2.5 l-4 4 l0 -7z"></path></svg>'
        },
        {
            name: 'alignment',
            label: {
                en: 'Alignment',
                fi: 'Sijoittelu'
            },
            attribute: 'alignment',
            defval: 'center',
            type: 'radio',
            icon: ' ',
            options: [
                {
                    label: {
                        en: 'left',
                        fi: 'vasen',
                        sv: 'vänster'
                    },
                    value: 'left',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatleft"><path style="stroke: none;" d="M2 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none;" fill="#555" d="m2 3 l28 0 l0 2 l-28 0z m19 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m-19 6 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'center',
                        fi: 'keski',
                        sv: 'centrum'
                    },
                    value: 'center',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatcenter"><path style="stroke: none;" d="M9 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none;" fill="#555" d="M2 3 l28 0 l0 2 l-28 0z m0 24 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'right',
                        fi: 'oikea',
                        sv: 'höger'
                    },
                    value: 'right',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatright"><path style="stroke: none;" d="M17 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none;" fill="#555" d="M2 3 l28 0 l0 2 l-28 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l28 0 l0 2 l-28 0z" /></svg>'
                }
            ]
        }
    ];
    ElementSet.dialogs.tabbox = [];
    ElementSet.dialogs.contenttable = [
        {
            name: 'cols',
            label: {
                en: 'Columns',
                fi: 'Sarakkeita'
            },
            attribute: 'cols',
            type: 'number',
            min: 1,
            max: 6,
            defval: 2
        },
        {
            name: 'borders',
            label: {
                en: 'Borders',
                fi: 'Reunaviivat'
            },
            attribute: 'borders',
            type: 'select',
            options: [
                {
                    label: {
                        en: 'None',
                        fi: 'Ei reunaa'
                    },
                    value: ''
                },
                {
                    label: {
                        en: 'Solid',
                        fi: 'Kiinteä'
                    },
                    value: 'solid'
                },
                {
                    label: {
                        en: 'Dashed',
                        fi: 'Katkoviiva'
                    },
                    value: 'dashed'
                },
                {
                    label: {
                        en: 'Dotted',
                        fi: 'Pisteviiva'
                    },
                    value: 'dotted'
                }
            ]
        }
    ];

    /******
     * Info about element (icon, description, etc.)
     ******/
    ElementBox.elementinfo = {
        elementbox: {
            type: 'elementbox',
            elementtype: ['containers','teachercontainers'],
            boxtype: 'elementbox',
            jquery: 'elementbox',
            name: 'Elementbox',
            icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 5 l0 18 l22 0 l0 -18z m3 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z" /></svg>',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M2 2 h26 v26 h-26z m2 5 v18 h22 v-18z" /></svg>',
            description: {
                en: 'Box container',
                fi: 'Sisältölaatikko',
                sv: 'Ruta'
            },
            roles: ['student', 'teacher', 'author'],
            classes: ['container'],
            cancontain: ['containers', 'elements']
        },
        tabbox: {
            type: 'tabbox',
            elementtype: ['containers'],
            boxtype: 'tabbox',
            jquery: 'elementbox',
            name: 'Tabbox',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 5 l2 -2 l7 0 l2 2 l2 -2 l7 0 l2 2 l0 2 l2 0 l0 20 l-24 0z m1 3 l0 18 l22 0 l0 -18 l-12 0 l0 -2 l-2 -2 l-7 0 l-1 1z m3 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z" /></svg>',
            description: {
                en: 'Elementbox with tabs.',
                fi: 'Välilehdellinen elementtilaatikko.',
                sv: 'Ruta med flikar'
            },
            roles: ['author'],
            classes: ['tabcontainer'],
            cancontain: ['tabcontainers']
        },
        tabcontainer: {
            type: 'tabcontainer',
            elementtype: ['tabcontainers'],
            boxtype: 'tabcontainer',
            jquery: 'elementbox',
            name: 'Tabcontainer',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 5 l2 -2 l7 0 l2 2 l2 -2 l7 0 l2 2 l0 2 l2 0 l0 20 l-24 0z m1 3 l0 18 l22 0 l0 -18 l-12 0 l0 -2 l-2 -2 l-7 0 l-1 1z m3 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z" /></svg>',
            description: {
                en: 'One tab in a tabset',
                fi: 'Yksi välilehti',
                sv: 'En flik i en serie'
            },
            roles: ['author'],
            classes: ['tabcontainer'],
            cancontain: ['containers', 'elements']
        }
    }
    
    $.fn.elementset('addelementtype', ElementBox.elementinfo.elementbox);
    $.fn.elementset('addelementtype', ElementBox.elementinfo.tabbox);
    $.fn.elementset('addelementtype', ElementBox.elementinfo.tabcontainer);

    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', ElementBox.elementinfo.elementbox);
        $.fn.elementpanel('addelementtype', ElementBox.elementinfo.tabbox);
        $.fn.elementpanel('addelementtype', ElementBox.elementinfo.tabcontainer);
    }

    
    /********************************************************************************************
     **** jQuery-plugin
     *****/
    var boxmethods = {
        'init': function(params){
            return this.each(function(){
                var elementbox = new ElementBox(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.elementbox = function(method){
        if (boxmethods[method]) {
            return boxmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return boxmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in elementbox.');
            return false;
        }
    }
    
    //$.fn.plaintext = $.fn.elementbox;


    /******************************************************************************************************************
     * TriggerSet to trigger actions on some element from click/hover/... on some other element.
     * @constructor
     * @param {jQuery} place - the place for elementbox
     * @param {Object} options . data for elementbox
     ******************************************************************************************************************/
    var TriggerSet = function(place, options){
        //this.parentClass.constr.call(this, place, options);
        //this.constr(place, options);
        ElementSet.call(this, place, options);
    }
    
    // Inherit from ElementSet
    //ElementBox.prototype = new ElementSet();
    TriggerSet.prototype = Object.create(ElementSet.prototype);
    TriggerSet.prototype.constructor = TriggerSet;
    TriggerSet.prototype.parentClass = ElementSet.prototype;

    /******
     * Init data
     ******/
    TriggerSet.prototype.esInitData = TriggerSet.prototype.initData;
    TriggerSet.prototype.initData = function(options){
        options = jQuery.extend(true, {}, TriggerSet.defaults, options);
        this.esInitData(options);
    }

    /******
     * Init handlers edit mode
     ******/
    TriggerSet.prototype.esInitHandlersEditable = TriggerSet.prototype.initHandlersEditable;
    TriggerSet.prototype.initHandlersEditable = function(){
        var triggset = this;
        this.esInitHandlersEditable();
        this.place.off('element_changed.triggerelem', '> .ebook-elementset-header > .ebook-triggerset-triggerelement').on('element_changed.triggerelem', '> .ebook-elementset-header > .ebook-triggerset-triggerelement', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var eltype = data.type;
            var elname = $(event.target).attr('data-element-name');
            place = triggset.header.children('.ebook-triggerset-triggerelement');
            place.trigger('getdata');
            var eldata = place.data('[[elementdata]]');
            triggset.data.triggercontent = eldata;
            triggset.changed();
        });
        
    };
    
    /******
     * Init handlers view mode
     ******/
    TriggerSet.prototype.esInitHandlersNoneditable = TriggerSet.prototype.initHandlersNoneditable;
    TriggerSet.prototype.initHandlersNoneditable = function(){
        var triggset = this;
        this.esInitHandlersNoneditable();
        var trigger = this.data.triggerevent;
        var startevent = TriggerSet.eventtypes[trigger].start;
        var stopevent = TriggerSet.eventtypes[trigger].stop;
        this.place.off(startevent, '> .ebook-elementset-header > .ebook-triggerset-triggerelement:not(.isrunningaction)').on(startevent, '> .ebook-elementset-header > .ebook-triggerset-triggerelement:not(.isrunningaction)', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            triggset.place.find('> .ebook-elementset-header > .ebook-triggerset-triggerelement').addClass('isrunningaction');
            var ename, cid, cplace;
            for (var i = 0, len = triggset.data.contents.length; i < len; i++) {
                ename = triggset.data.contents[i].event || 'play';
                cid = triggset.data.contents[i].id;
                cplace = triggset.getContentPlace(cid);
                cplace.trigger(ename);
            };
        });
        this.place.off(stopevent, '> .ebook-elementset-header > .ebook-triggerset-triggerelement.isrunningaction').on(stopevent, '> .ebook-elementset-header > .ebook-triggerset-triggerelement.isrunningaction', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            triggset.place.find('> .ebook-elementset-header > .ebook-triggerset-triggerelement').removeClass('isrunningaction');
            var ename, cid, cplace;
            for (var i = 0, len = triggset.data.contents.length; i < len; i++) {
                ename = triggset.data.contents[i].event || 'stop';
                cid = triggset.data.contents[i].id;
                cplace = triggset.getContentPlace(cid);
                cplace.trigger(ename);
            };
        });
        this.place.off('mediaended').on('mediaended', function(event, data){
            event.stopPropagation();
            triggset.place.find('> .ebook-elementset-header > .ebook-triggerset-triggerelement').removeClass('isrunningaction');
        });
        
    };
    
    /******
     * Get data
     * @returns {Object} Data of the triggerset
     ******/
    TriggerSet.prototype.esGetData = TriggerSet.prototype.getData;
    TriggerSet.prototype.getData = function(){
        var dataset = this.esGetData();
        dataset.data.triggercontent = JSON.parse(JSON.stringify(this.data.triggercontent));
        dataset.data.triggerevent = this.data.triggerevent;
        return dataset;
    }
    
    /******
     * Show content special for triggerset
     ******/
    TriggerSet.prototype.refreshContent = function(){
        var uilang = this.settings.uilang;
        this.header = this.place.children('.ebook-elementset-header');
        this.footer = this.place.children('.ebook-elementset-footer');
        this.titlebox = this.place.children('.ebook-elementset-title');
        this.place.attr('data-triggerevent', this.data.triggerevent);
        var data = JSON.parse(JSON.stringify(this.data.triggercontent));
        data.settings = JSON.parse(JSON.stringify(this.settings));
        this.header.html('<h1 class="ebook-triggerset-head"></h1><div class="ebook-triggerset-triggerelement"></div>');
        this.header.children('.ebook-triggerset-triggerelement').pageelement(data);
        this.header.attr('data-triggerevent', this.data.triggerevent);
        if (this.editable) {
            this.header.children('.ebook-triggerset-head').html(ebooklocalizer.localize('triggerset:whenthishappens', uilang));
            this.titlebox.html(ebooklocalizer.localize('triggerset:elementstotrigger', uilang));
        } else {
            this.footer.html('<div class="ebook-triggerset-icon">'+TriggerSet.icons[this.data.triggerevent]+'</div>');
        };
    };

    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    TriggerSet.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(TriggerSet.elementinfo.cancontain));
    }
    
    TriggerSet.icons = {
        hover: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-hover"><path style="stroke: none;" d="M10 17 v-12 a2 2 0 0 1 4 0 v6 a2 2 0 0 1 4 0 v1 a2 2 0 0 1 4 0 v2 a2 2 0 0 1 3 2 v3 a3 8 0 0 1 -2 7 a3 6 0 0 0 -1 5 h-9 a7 14 -30 0 0 -8 -12 a2 2 0 0 1 2 -4z" /><path style="stroke: none; fill: white;" d="M11 19 v-14 a1 1 0 0 1 2 0 v7 l1 1 l1 -1 v-1 a1 1 0 0 1 2 0 v1 l1 1 l1 -1 a1 1 0 0 1 2 0 v2 l1 2 a1 1 0 0 1 2 0 v4 a3 8 0 0 1 -2 6 a3 6 0 0 0 -1 3 h-8 a7 14 -30 0 0 -8 -11 a1 1 0 0 1 2 -2z" /></svg>',
        click: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-click mini-icon-clickarrow"><path style="stroke: none;" d="M9 9 l17 6 l-5 3 l6 6 l-3 3 l-6 -6 l-3 5z m2 2 l4.5 12 l2.3 -3.8 l6 6 l1.5 -1.5 l-6 -6 l3.8 -2.3z m-2 -10 h1.5 v6 h-1.5z m-8 8 h6 v1.5 h-6z m2.5 6.5 l4 -4 l1 1 l-4 4z m0 -11.5 l1 -1 l4 4 l-1 1z m12.5 0 l-4 4 l-1 -1 l4 -4z" /></svg>',
        clickhand: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="-1 -5 25 25" class="mini-icon mini-icon-click"><path style="stroke: none;" d="M4 -4 l5 5 l-1 1 l-5 -5z m16 0 l1 1 l-5 5 l-1 -1z m4.5 8 l0 1.3 l-7.5 0 l0 -1.3z m-25 0 l7.5 0 l0 1.3 l-7.5 0z" /><path style="stroke: none;" d="M10 17 v-12 a2 2 0 0 1 4 0 v6 a2 2 0 0 1 4 0 v1 a2 2 0 0 1 4 0 v2 a2 2 0 0 1 2 2 v3 a3 8 0 0 1 -2 7 a3 6 0 0 0 -1 5 h-8 a7 14 -30 0 0 -8 -12 a2 2 0 0 1 2 -4z" /><path style="stroke: none; fill: white;" d="M11 19 v-14 a1 1 0 0 1 2 0 v7 l1 1 l1 -1 v-1 a1 1 0 0 1 2 0 v1 l1 1 l1 -1 a1 1 0 0 1 2 0 v2 l0 2 a1 1 0 0 1 2 0 v4 a3 8 0 0 1 -2 6 a3 6 0 0 0 -1 3 h-7 a7 14 -30 0 0 -8 -11 a1 1 0 0 1 2 -2z" /></svg>'
    }
    
    TriggerSet.defaults = {
        data: {
            triggercontent: {"type": "pageelement"},
            triggerevent: 'click'
        }
    }

    ElementSet.dialogs.triggerset = [
        {
            name: 'triggerevent',
            label: {
                en: 'Type',
                fi: 'Tyyppi'
            },
            attribute: 'triggerevent',
            defval: 'click',
            type: 'radio',
            icon: ' ',
            options: [
                {
                    label: {
                        en: 'Click',
                        fi: 'Klikki',
                        sv: 'Klick'
                    },
                    icon: TriggerSet.icons.click,
                    value: 'click'
                },
                {
                    label: {
                        en: 'Mouse over',
                        fi: 'Hiiri päällä',
                        sv: 'Musen över'
                    },
                    icon: TriggerSet.icons.hover,
                    value: 'hover'
                }
            ]
        }
    ]
    
    TriggerSet.eventtypes = {
        click: {
            start: 'click',
            stop: 'click'
        },
        hover: {
            start: 'mouseenter',
            stop: 'mouseleave'
        }
    }

    /******
     * Info about element (icon, description, etc.)
     ******/
    TriggerSet.elementinfo = {
        type: 'triggerset',
        elementtype: ['containers','teachercontainers'],
        boxtype: 'triggerset',
        jquery: 'triggerset',
        name: 'TriggerSet',
        icon: TriggerSet.icons.click,
        description: {
            en: 'Trigger actions to other elements',
            fi: 'Laukaise toimintoja toisille elementeille',
            sv: 'Starta ett element med ett annat'
        },
        roles: ['teacher', 'author'],
        classes: ['action'],
        cancontain: ['containers', 'elements']
    }
    
    $.fn.elementset('addelementtype', TriggerSet.elementinfo);

    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', TriggerSet.elementinfo);
    }
    
    TriggerSet.localization = {
        en: {
            'triggerset:whenthishappens': 'When this happens...',
            'triggerset:elementstotrigger': '...start these.'
        },
        fi: {
            'triggerset:whenthishappens': 'Kun tämä tapahtuu...',
            'triggerset:elementstotrigger': '...käynnistä nämä.'
        },
        sv: {
            "triggerset:whenthishappens": "När detta händer...",
            "triggerset:elementstotrigger": "...starta dessa."
        }
    };
    
    if (typeof(ebooklocalizer) !== 'undefined') {
        ebooklocalizer.addTranslations(TriggerSet.localization);
    }

    
    /********************************************************************************************
     **** jQuery-plugin
     *****/
    var triggermethods = {
        'init': function(params){
            return this.each(function(){
                var element = new TriggerSet(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.triggerset = function(method){
        if (triggermethods[method]) {
            return triggermethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return triggermethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in triggerset.');
            return false;
        }
    }



    /******************************************************************************************************************
     * ImageBox
     * @constructor
     * @param {jQuery} place - the place for elementbox
     * @param {Object} options . data for imagebox
     ******************************************************************************************************************/
    var ImageBox = function(place, options){
        options = $.extend(true, {}, ImageBox.defaults, options);
        ElementSet.call(this, place, options);
    }
    
    ImageBox.prototype = Object.create(ElementSet.prototype);
    ImageBox.prototype.constructor = ImageBox;
    ImageBox.prototype.parentClass = ElementSet.prototype;
    
    /******
     * Show title
     ******/
    ImageBox.prototype.showTitle = function(){
        if (this.editable) {
            this.showTitleEdit();
        } else {
            this.showTitleView();
        }
    }
    
    /******
     * Show title (caption) in view mode
     ******/
    ImageBox.prototype.showTitleView = function(){
        var uilang = this.settings.uilang;
        if (typeof(this.data.title) === 'string' && this.data.title !== '') {
            this.place.find('.ebook-elementset-title').html(this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>').replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>'));
            this.place.find('.ebook-elementset-title .mathquill-embedded-latex').mathquill();
        }
        if (typeof(this.data.caption) === 'string' && this.data.caption !== '') {
            this.place.find('.ebook-elementset-footer').html('<div class="imagebox-caption">' + this.escapeHTML(this.data.caption).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>').replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>') + '</div>');
            this.place.find('.ebook-elementset-footer .mathquill-embedded-latex').mathquill();
        }
    }
    
    /******
     * Show title (caption) in edit mode
     ******/
    ImageBox.prototype.showTitleEdit = function(){
        var uilang = this.settings.uilang;
        this.data.title = this.data.title || '';
        if (typeof(this.data.title) === 'string') {
            this.place.find('.ebook-elementset-title').html('<span class="ebook-elementset-title-editable">' + this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '\$ $1\$') + '</span>');
            this.place.find('.ebook-elementset-title .ebook-elementset-title-editable').mathquill('textbox');
        }
    }
    
    /******
     * Set attributes
     ******/
    ImageBox.prototype.setAttrs = function(){
        if (this.data['size-x']) {
            this.place.attr('data-imgwidth', this.data['size-x']);
            if (this.settings.mode === 'view' && (this.data.alignment === 'left' || this.data.alignment === 'right')) {
                if (this.data['size-x'] !== 'auto') {
                    this.place.css('max-width', (this.data['size-x'] | 0) + 20);
                } else {
                    this.place.css('max-width', '');
                }
            }
        }
        if (this.data['size-y']) {
            this.place.attr('data-imgheight', this.data['size-y']);
        };
        if (this.data.alignment) {
            this.place.attr('data-alignment', this.data.alignment);
        };
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
        };
    }

    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    ImageBox.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(ImageBox.elementinfo.cancontain));
    }

    /******
     * Info about ImageBox (icon, description, etc.)
     ******/
    ImageBox.elementinfo = {
        type: 'imagebox',
        elementtype: ['containers', 'studentcontainers'],
        boxtype: 'imagebox',
        jquery: 'imagebox',
        name: 'Imagebox',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-image"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 22 l22 0 l0 -22z m3 2 l15 0 l0 1 l-15 0z m1 4 l13 0 l0 13 l-13 0z m1 1 l0 11 l11 0 l0 -11z m4 2 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4 m4 3 l2 5 l-8 0 l2 -3 l2 2z" /></svg>',
        description: {
            en: 'Imagebox container for images',
            fi: 'Kuvalaatikko',
            sv: 'Bildruta'
        },
        roles: ['student', 'teacher', 'author'],
        classes: [],
        cancontain: ['elements']
    };
    
    /******
     * Default data and settings for imagebox.
     ******/
    ImageBox.defaults = {
        type: 'imagebox',
        metadata: {
            creator: 'Anonymous',
            created: 0,
            modifier: 'Anonymous',
            modified: 0,
            tags: []
        },
        data: {
            boxtype: 'imagebox',
            'size-x': 'auto',
            alignment: 'center'
        }
    }
    
    /******
     * Add settings dialog for this type
     ******/
    ElementSet.dialogs.imagebox = [
        {
            name: 'size-x',
            label: {
                en: 'Width',
                fi: 'Leveys'
            },
            attribute: 'size-x',
            type: 'number',
            min: 0,
            max: 800,
            defval: 300,
            placeholder: 'auto'
        },
        {
            name: 'alignment',
            label: {
                en: 'Alignment',
                fi: 'Sijoittelu'
            },
            attribute: 'alignment',
            type: 'radio',
            options: [
                {
                    label: {
                        en: 'left',
                        fi: 'vasen'
                    },
                    value: 'left',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatleft"><path style="stroke: none;" d="M2 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="m2 3 l28 0 l0 2 l-28 0z m19 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m-19 6 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'center',
                        fi: 'keski'
                    },
                    value: 'center',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatcenter"><path style="stroke: none;" d="M9 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="M2 3 l28 0 l0 2 l-28 0z m0 24 l28 0 l0 2 l-28 0z" /></svg>'
                },
                {
                    label: {
                        en: 'right',
                        fi: 'oikea'
                    },
                    value: 'right',
                    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-floatright"><path style="stroke: none;" d="M17 9 l14 0 l0 14 l-14 0z" /><path style="stroke: none; fill: #555;" d="M2 3 l28 0 l0 2 l-28 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l9 0 l0 2 l-9 0z m0 6 l28 0 l0 2 l-28 0z" /></svg>'
                }
            ]
        }
    ];

    // Register imagebox as an elementset and to the elementpanel.
    $.fn.elementset('addelementtype', ImageBox.elementinfo);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', ImageBox.elementinfo);
    }

    
    /********************************************************************************************
     **** jQuery-plugin
     *****/
    var imgboxmethods = {
        'init': function(params){
            return this.each(function(){
                var imagebox = new ImageBox(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.imagebox = function(method){
        if (imgboxmethods[method]) {
            return imgboxmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return imgboxmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in imagebox.');
            return false;
        }
    }
    
    
    
    /******************************************************************************************************************
     * ContentTable - An elementset with table layout
     * @constructor
     * @param {jQuery} place - the place for elementbox
     * @param {Object} options . data for contenttable
     ******************************************************************************************************************/
    var ContentTable = function(place, options){
        options = $.extend(true, {}, ContentTable.defaults, options);
        ElementSet.call(this, place, options);
    }
    
    ContentTable.prototype = Object.create(ElementSet.prototype);
    ContentTable.prototype.constructor = ContentTable;
    ContentTable.prototype.parentClass = ElementSet.prototype;
    
    /******
     * Show title
     ******/
    ContentTable.prototype.showTitle = function(){
        if (this.editable) {
            this.showTitleEdit();
        } else {
            this.showTitleView();
        }
    }

    /******
     * Show title (caption) in view mode
     ******/
    ContentTable.prototype.showTitleView = function(){
        var uilang = this.settings.uilang;
        if (typeof(this.data.title) === 'string' && this.data.title !== '') {
            this.place.find('.ebook-elementset-title').html(this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>').replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>'));
            this.place.find('.ebook-elementset-title .mathquill-embedded-latex').mathquill();
        }
    }

    /******
     * Show title (caption) in edit mode
     ******/
    ContentTable.prototype.showTitleEdit = function(){
        var uilang = this.settings.uilang;
        this.data.title = this.data.title || '';
        if (typeof(this.data.title) === 'string') {
            this.place.find('.ebook-elementset-title').html('<span class="ebook-elementset-title-editable">' + this.escapeHTML(this.data.title).replace(/\\\((.*?)\\\)/g, '\$ $1\$') + '</span>');
            this.place.find('.ebook-elementset-title .ebook-elementset-title-editable').mathquill('textbox');
        }
    }
    
    /******
     * Set attributes
     ******/
    ContentTable.prototype.setAttrs = function(){
        if (this.data.boxclass) {
            this.place.attr('data-boxclass', this.data.boxclass);
        };
        if (this.data.borders) {
            this.place.attr('data-borders', this.data.borders);
        } else {
            this.place.removeAttr('data-borders');
        };
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
        }
    }
    
    /******
     * Get list of types of elements the contenttable can contain.
     * @returns {Array} list of strings
     ******/
    ContentTable.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(ContentTable.elementinfo.cancontain));
    }

    /******
     * Get a jQuery-object with headbars of all (direct) subelements.
     * @returns {jQuery} headbars of subelements.
     ******/
    ContentTable.prototype.getHeadbars = function(){
        return this.place.find('> .ebook-elementset-body > table > tbody > tr > td > .elementset-elementwrapper > .elementset-elementheadbar');
    }

    /******
     * Get a jQuery-object with controlbars of all (direct) subelements.
     * @returns {jQuery} controlbars of subelements.
     ******/
    ContentTable.prototype.getControlbars = function(){
        return this.place.find('> .ebook-elementset-body > table > tbody > tr > td > .elementset-elementwrapper > .elementset-elementcontrolbar');
    }

    /******
     * Make a table from data
     ******/
    ContentTable.prototype.refreshContent = function(){
        var cells = this.contents.length;
        if (this.editable) {
            // We don't need (want) general droptargets. ContentTable uses droptargets of it's own.
            this.place.find('> .ebook-elementset-body .elementset-droptarget').remove();
            cells++;
        }
        var rows = Math.ceil(cells / this.data.cols);
        table='<table class="ebook-contenttable">\n<tbody>\n';
        for(var i = 0; i < rows; i++){
            table +="<tr>\n";
            for (var j = 0; j < this.data.cols; j++){
                table +="<td></td>\n";
            }
            table +="</tr>\n";
        }
        table +="</tbody>\n</table>";
        var $body = this.place.find('.ebook-elementset-body').eq(0);
        tableCells = $body.append(table).find('table').last().find('td');
        var cancontain = this.cancontain().join(' ');
        var $cell;
        for(var i = 0; i < cells; i++){
            $cell = tableCells.eq(i);
            if (this.editable) {
                $cell.html('<div class="elementset-droptarget elementset-droptarget-tablecell" data-droptarget-cancontain="'+ cancontain +'" data-dropindex="'+i+'"><span>+</span></div>');
                if (i < cells - 1) {
                    $cell.append(this.contentPlaces[this.data.contents[i].id]);
                } else {
                    $cell.addClass('elementset-contenttable-lastcell');
                }
            } else {
                $cell.append(this.contentPlaces[this.data.contents[i].id]);
            }
        }
        if ($body.children('table').length > 1) {
            $body.children('table').eq(0).remove();
        }
        
    }

    /******
     * Append and show a single element to the given index.
     * @param {String} ctype - type of content element.
     * @param {String} cid - id of the content element.
     * @param {Number} index - the index of the element in this set.
     * @param {Object} data - the content data of the element.
     * @returns {Object} an array of elements to draw by event.
     ******/
    ContentTable.prototype.appendElementToDOMParent = ContentTable.prototype.appendElementToDOM;
    ContentTable.prototype.appendElementToDOM = function(ctype, cid, index, data, assindent) {
        this.appendElementToDOMParent(ctype, cid, index, data, assindent);
        this.refreshContent();
    }
    
    /******
     * Remove an element
     * @param {String} cid - id of the element
     * @param {Boolean} noconfirm - true, if the confirmation is not needed
     ******/
    ContentTable.prototype.removeElementParent = ContentTable.prototype.removeElement;
    ContentTable.prototype.removeElement = function(cid, noconfirm) {
        this.removeElementParent(cid, noconfirm);
        this.refreshContent();
    }  
    
    /******
     * Default data and settings for contenttables.
     ******/
    ContentTable.defaults = {
        type: 'contenttable',
        metadata: {
            creator: 'Anonymous',
            created: 0,
            modifier: 'Anonymous',
            modified: 0,
            tags: []
        },
        data: {
            boxtype: 'contenttable',
            cols: 2,
            borders: 'none'
        }
    }

    /******
     * Info about ContentTable (icon, description, etc.)
     ******/
    ContentTable.elementinfo = {
        type: 'contenttable',
        elementtype: ['containers'],
        boxtype: 'contenttable',
        jquery: 'contenttable',
        name: 'ContentTable',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 10 l10 0 l0 -10z m0 12 l0 10 l10 0 l0 -10z m12 0 l0 10 l10 0 l0 -10z m0 -12 l0 10 l10 0 l0 -10z m-7 2 l4 6 l-8 0z m-4 12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z m16 -4 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m-4 -12 l8 0 l0 2 l-8 0z m0 4 l8 0 l0 2 l-8 0z" /></svg>',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M2 2 h26 v26 h-26z m2 2 l0 10 l10 0 l0 -10z m0 12 l0 10 l10 0 l0 -10z m12 0 l0 10 l10 0 l0 -10z m0 -12 l0 10 l10 0 l0 -10z" /></svg>',
        description: {
            en: 'Elementbox container for elements in a table',
            fi: 'Taulukkomuotoinen sisältölaatikko',
            sv: 'Ruta för element i tabell'
        },
        roles: ['teacher', 'author'],
        classes: ['container'],
        cancontain: ['containers', 'elements']
    };

    /******
     * Add settings dialog for this type
     ******/
    ElementSet.dialogs.contenttable = [
        {
            name: 'cols',
            label: {
                en: 'Columns',
                fi: 'Sarakkeita'
            },
            attribute: 'cols',
            type: 'number',
            min: 1,
            max: 6,
            defval: 2
        },
        {
            name: 'borders',
            label: {
                en: 'Borders',
                fi: 'Reunaviivat'
            },
            attribute: 'borders',
            type: 'select',
            options: [
                {
                    label: {
                        en: 'None',
                        fi: 'Ei reunaa'
                    },
                    value: ''
                },
                {
                    label: {
                        en: 'Solid',
                        fi: 'Kiinteä'
                    },
                    value: 'solid'
                },
                {
                    label: {
                        en: 'Dashed',
                        fi: 'Katkoviiva'
                    },
                    value: 'dashed'
                },
                {
                    label: {
                        en: 'Dotted',
                        fi: 'Pisteviiva'
                    },
                    value: 'dotted'
                }
            ]
        }
    ];

    // Register contenttable as an elementset and to the elementpanel.
    $.fn.elementset('addelementtype', ContentTable.elementinfo);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', ContentTable.elementinfo);
    }

    
    /********************************************************************************************
     **** jQuery-plugin
     *****/
    var conttablemethods = {
        'init': function(params){
            return this.each(function(){
                var ctable = new ContentTable(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.contenttable = function(method){
        if (conttablemethods[method]) {
            return conttablemethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return conttablemethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in imagebox.');
            return false;
        }
    }


    /****************************************
     ***** Assignmentlist
     ****************************************/
    
    if (typeof(ebooklocalizer) !== 'undefined') {
        ebooklocalizer.addTranslations({
            'en': {
                'assignment:basic': 'Automaticly checked',
                'assignment:Show Answer': 'Show the answer',
                'assignment:Hide answer': 'Hide the answer',
                'assignment:answer': 'Answer',
                'assignment:assignment': 'Assignment'
            },
            'fi': {
                'assignment:basic': 'Automaattisesti tarkistettavat',
                'assignment:Show Answer': 'Näytä vastaus',
                'assignment:Hide answer': 'Piilota vastaus',
                'assignment:answer': 'Vastaus',
                'assignment:assignment': 'Tehtävä'
            },
            'sv': {
                "assignment:basic": "Automatiskt rättande uppgifter",
                "assignment:Show Answer": "Visa svaret",
                "assignment:Hide answer": "Göm svaret",
                "assignment:answer": "Svar",
                "assignment:assignment": "Uppgift"
            }
        });
    }
    
    var AssignmentList = function(place, options) {
        options = $.extend(true, {}, AssignmentList.defaults, options);
        ElementSet.call(this, place, options);
    }
    
    AssignmentList.prototype = Object.create(ElementSet.prototype);
    AssignmentList.prototype.constructor = AssignmentList;
    AssignmentList.prototype.parentClass = ElementSet.prototype;
    
    /******
     * Show content
     ******/
    AssignmentList.prototype.showContent = function(options){
        this.place.empty();
        var lang = this.place.closest('[lang]').attr('lang') || 'en';
        lang = this.metadata.lang;
        var uilang = this.settings.uilang;
        this.asslist = $('<ul class="ebook-assignmentlist"></ul>');
        this.place.append(this.asslist);
        this.otherassignments = $('<li class="ebook-assignments-other"><h1>'+ebooklocalizer.localize('assignment:basic', uilang)+'</h1></li>');
        this.asslist.append(this.otherassignments);
        this.assignmentsets = [];
        var stars = '***';
        for (var i = 0; i < 3; i++){
            this.assignmentsets.push($('<li class="ebook-assignments-'+i+'"><h1>'+stars.substr(0,i+1)+'</h1></li>'));
            this.asslist.append(this.assignmentsets[i]);
        }
        this.contentPlaces = {};

        var contentlist = [];
        var reflist = [];
        
        for (var i = 0; i < this.contents.length; i++){
            var content = this.contents[i].id;
            var elementwrapper = $('<div class="elementset-elementwrapper"><div class="elementset-elementheadbar ffwidget-background"></div><div class="elementset-element" tabindex="-1"></div><div class="elementset-elementcontrolbar ffwidget-background"></div><div class="elementset-element-refcontent"></div></div>');
            var element = elementwrapper.children('.elementset-element');
            element.attr('data-element-name', content);
            this.contentPlaces[content] = elementwrapper;
            if (this.contents[i].assignmenttype === 'assignment') {
                var level = this.contents[i].level;
                if (this.assignmentsets[level]) {
                    this.assignmentsets[level].append(elementwrapper);
                }
            } else {
                this.otherassignments.append(elementwrapper);
            }
            if (this.contentdata[content]) {
                // If the data of element is embedded in the elementset.
                var cdata = this.contentdata[content];
                this.drawElement(element, cdata);
            } else {
                // If the data of element is separate from elementset.
                contentlist.push(content);
            };
            reflist.push(content);
        }

        this.getExternalData(contentlist);
        this.getLinkedData(reflist);
        this.refreshContent();
        //if (contentlist.length > 0) {
        //    this.place.trigger('get_bookelement', {contentlist: contentlist, lang: this.metadata.lang});
        //}
        if (this.otherassignments.children(':visible').length === 1) {
            this.otherassignments.remove();
        }
        for (var i = 0; i < 3; i++){
            if (this.assignmentsets[i].children().length === 1) {
                this.assignmentsets[i].remove();
            }
        }
        var assheads = this.place.find('.ebook-otherassignment:visible > .ebook-elementset-titlebox > h1.ebook-elementbox-title, .ebook-assignment:visible > .ebook-elementset-titlebox > h1.ebook-elementbox-title');
        for (var i = 0, length = assheads.length; i < length; i++){
            assheads.eq(i).prepend('<span class="ebook-assignment-number">'+ebooklocalizer.localize('assignment:assignment', uilang)+' ' + (i+1) + (assheads.eq(i).text() !== ' ' ? ':' : '') + '</span> ');
        };
        this.showSolutions();
    };
    
    /******
     * Show solutions for assignments in this assignmentlist in view mode
     ******/
    AssignmentList.prototype.showSolutions = function(){
        var assignment, aid, solutionplace;
        for (var i = 0, len = this.data.contents.length; i < len; i++) {
            assignment = this.data.contents[i];
            aid = assignment.id;
            solutionplace = this.contentPlaces[aid].find('.homeassignmentset-solution');
            solutionplace.solutiontool({
                data: {assignmentid: aid},
                settings: {
                    uilang: this.settings.uilang,
                    lang: this.settings.lang,
                    mode: 'view',
                    role: this.settings.role,
                    username: this.settings.username,
                    savemode: this.settings.savemode,
                    users: this.settings.users,
                    gotolink: JSON.parse(JSON.stringify(this.settings.gotolink))
                }
            });
        };
    };
    

    AssignmentList.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(AssignmentList.elementinfo.cancontain));
    }
    
    AssignmentList.defaults = {
        type: 'assignmentlist',
        metadata: {},
        data: {},
        settings: {
            uilang: 'en',
            savemode: 'global',
            mode: 'view'
        }
    }

    /******
     * Info about AssignmentList (icon, description, etc.)
     ******/
    AssignmentList.elementinfo = {
        type: 'assignmentlist',
        elementtype: ['containers'],
        boxtype: 'assignmentlist',
        jquery: 'assignmentlist',
        name: 'AssignmentList',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-assignmentlist"><path style="stroke: none;" d="M5 5 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z m0 15 a3 3 0 0 0 0 6 a3 3 0 0 0 0 -6z" /><text x="16" y="14" style="font-weight: bold; font-size: 15px; font-family: sans;">?</text><text x="16" y="29" style="font-weight: bold; font-size: 15px; font-family: sans;">?</text></svg>',
        description: {
            en: 'A container for assignments',
            fi: 'Säilö tehtäville',
            sv: 'Ruta för hemuppgifter'
        },
        roles: ['author'],
        classes: ['assignmentcontainer', 'viewonly'],
        cancontain: ['containers', 'elements', 'assignments']
    };

    // Register assignmentlist as an elementset and to the elementpanel.
    $.fn.elementset('addelementtype', AssignmentList.elementinfo);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', AssignmentList.elementinfo);
    }

    
    /**** jQuery-plugin *****/
    var asslistmethods = {
        'init': function(params){
            return this.each(function(){
                var asslist = new AssignmentList(this, params);
            });
        }
    }
    
    $.fn.assignmentlist = function(method){
        if (asslistmethods[method]) {
            return asslistmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return asslistmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in elementbox.');
            return false;
        };
        return false;
    };

    
    /*****************************************
    **** Assignment jQuery-plugin
    *****************************************/
    var Assignment = function(place, options){
        options = $.extend(true, {}, Assignment.defaults, options);
        options.settings.savemode = 'local';
        ElementSet.call(this, place, options);
    };
    
    Assignment.prototype = Object.create(ElementSet.prototype);
    Assignment.prototype.constructor = Assignment;
    Assignment.prototype.parentClass = ElementSet.prototype;

    /******
     * Init assignment data with ElementSet's initData and do some more.
     ******/
    Assignment.prototype.initData = function(options) {
        this.parentClass.initData.call(this, options);
        this.place.addClass('ebook-assignment');
        this.assignmentid = this.place.attr('data-element-name');
        this.answer = options['data'] && options['data']['assignmentAnswer'] || options['data']['assignmentSolution'] || '';
    };

    /******
     * Init handlers with ElementSet's initHandlersNoneditable and do some more.
     ******/
    Assignment.prototype.initHandlersNoneditableParent = Assignment.prototype.initHandlersNoneditable;
    Assignment.prototype.initHandlersNoneditable = function() {
        this.initHandlersNoneditableParent();
        var assignment = this;
        this.place.on('click','.showAnswer', function(){
            jQuery(this).parent().toggleClass('answerhidden');
        });
        this.place.off('click', '.elementset-assignment-homeassignment-button, .assignment-teachercontrol .assignment-habutton').on('click', '.elementset-assignment-homeassignment-button, .assignment-teachercontrol .assignment-habutton', function(event, data) {
            event.stopPropagation();
            assignment.markAsHomeassignment();
        });
        this.place.off('markashomeassignment').on('markashomeassignment', function(event, data) {
            event.stopPropagation();
            assignment.markAsHomeassignment();
        });
        this.place.off('click', '.assignment-teacherbuttons').on('click', '.assignment-teacherbuttons', function(event, data) {
            // Not in use anymore?
            // TODO: remove this
            event.stopPropagation();
            var button = $(this);
            var action = button.attr('data-action');
            assignment.place.trigger(action);
        });
        this.place.off('customdragstart').on('customdragstart', function(event, data) {
            event.stopPropagation();
            var evdata = {};
            switch (data.action) {
                case 'markashomeassignment':
                    evdata = {
                        "type": "homeassignmentelement",
                        "metadata": {},
                        "data": assignment.getHomeassignmentData()
                    };
                    break;
                default:
                    break;
            };
            var ev = data.event;
            ev.dataTransfer.setData('text/plain', JSON.stringify(evdata, null, 4));
            ev.dataTransfer.effectAllowed = 'move';
        });
    };

    /******
     * Init handlers with ElementSet's initHandlersEditable and do some more.
     ******/
    Assignment.prototype.initHandlersEditableParent = Assignment.prototype.initHandlersEditable;
    Assignment.prototype.initHandlersEditable = function() {
        this.initHandlersEditableParent();
        var assignment = this;
        this.place.off('change', 'h1.ebook-elementbox-title input.assignment-title-input').on('change', 'h1.ebook-elementbox-title input.assignment-title-input', function(event){
            event.stopPropagation();
            var input = $(this);
            var value = input.val();
            // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
            assignment.data.title = value;
            assignment.changed();
        });
        this.place.off('click', '.elementset-assignment-level svg.mini-icon-star').on('click', '.elementset-assignment-level svg.mini-icon-star', function(event){
            event.stopPropagation();
            event.preventDefault();
            var star = $(this);
            var level = star.index();
            assignment.data.level = level;
            star.closest('.elementset-assignment-level').attr('data-assignmentlevel', level);
            assignment.changed();
        });
        this.place.off('change', '.assignment-answer-input').on('change', '.assignment-answer-input', function(event){
            event.stopPropagation();
            event.preventDefault();
            var input = $(this);
            var value = input.val();
            // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
            assignment.answer = value;
            assignment.changed();
        });
        this.place.off('blur.assignment', '.assignment-answer-input.mathbox').on('blur.assignment', '.assignment-answer-input.mathbox', function(event){
            event.stopPropagation();
            event.preventDefault();
            var input = $(this);
            var value = input.mathquill('latex');
            // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
            assignment.answer = value;
            assignment.changed();
        });
    };

    /******
     * Show title
     ******/
    Assignment.prototype.showTitle = function(){
        if (this.editable) {
            this.showTitleEdit();
        } else {
            this.showTitleView();
        }
    }
    
    /******
     * Show title (view only)
     ******/
    Assignment.prototype.showTitleView = function(){
        var uilang = this.settings.uilang;
        var title = this.escapeHTML(this.data.title || '') || '&nbsp;';
        var assnumber = this.data.assignmentNumber || '';
        assnumber = (assnumber ? ('<span class="ebook-assignment-number">' + assnumber + '</span> ') : '');
        this.place.find('.ebook-elementset-title').eq(0).html(assnumber + title).find('.mathquill-embedded-latex').mathquill();
        var level = this.data.level | 0;
        var emptylist = [];
        emptylist[level + 1] = '';
        var levelstr = emptylist.join(Assignment.icons.star);
        this.place.find('.elementset-assignment-level').eq(0).html(levelstr);
        var assicons = this.data.assignmenticons || [];
        var assicplace = this.place.find('.elementset-assignmenticons');
        for (var i = 0, len = assicons.length; i < len; i++) {
            assicplace.append('<div class="elementset-assignment-icon" data-assignmenticon="'+assicons[i]+'" title="'+ebooklocalizer.localize('assignment:' + assicons[i], uilang)+'">'+(Assignment.icons[assicons[i]] || '')+'</div>');
        };
    }
    
    /******
     * Show title (edit)
     ******/
    Assignment.prototype.showTitleEdit = function(){
        var title = this.escapeHTML(this.data.title || '');
        var assnumber = this.data.assignmentNumber || '';
        assnumber = (assnumber ? ('<span class="ebook-assignment-number">' + assnumber + '</span> ') : '');
        this.place.find('.ebook-elementset-title').eq(0).html(assnumber + '<input class="assignment-title-input" type="text" value="' + title + '" />');
        var level = this.data.level | 0;
        var emptylist = [];
        emptylist[5] = '';
        var levelstr = emptylist.join(Assignment.icons.star);
        this.place.find('.elementset-assignment-level').eq(0).html(levelstr).attr('data-assignmentlevel', level);
    }

    /******
     * Show content
     * Use ElementBox's showContent and add some.
     ******/
    Assignment.prototype.showContentParent = Assignment.prototype.showContent;
    Assignment.prototype.showContent = function(){
        var lang = this.place.closest('[lang]').attr('lang');
        var uilang = this.settings.uilang;
        this.showContentParent();
        if (this.editable) {
            if (typeof($.fn.mathquill) === 'function') {
                this.place.append('<div class="assignmentsAnswer"><p>'+ebooklocalizer.localize('assignment:answer', uilang)+':</p><span class="assignment-answer-input mathbox">'+this.escapeHTML(this.answer)+'</span></div>');
                this.place.find('.assignmentsAnswer .assignment-answer-input.mathbox').mathquill('textbox');
            } else {
                this.place.append('<div class="assignmentsAnswer"><p>'+ebooklocalizer.localize('assignment:answer', uilang)+':</p><textarea class="assignment-answer-input">'+this.escapeHTML(this.answer)+'</textarea></div>');
            };
            //this.editSolutions();
        } else {
            if (this.answer) {
                this.place.append('<div class="assignmentsAnswer answerhidden"><button class="showAnswer ffwidget-button"><span class="showbutton">' + ebooklocalizer.localize('assignment:Show Answer', uilang) + '</span><span class="hidebutton">' + ebooklocalizer.localize('assignment:Hide answer', uilang) + '</span></button><div class="answertext"><span class="textanswer">' + ebooklocalizer.localize('assignment:answer', uilang) + ':</span> '+this.escapeHTML(this.answer).replace(/\\\(/g,'<span class="answermath">').replace(/\\\)/g,'</span>').replace(/\$(.+?)\$/g,'<span class="answermath">$1</span>')+'</span></div></div>');
                this.place.find('.answermath').mathquill();
            };
            this.viewSolutions();
        };
    };
    
    /******
     * Show solutions for assignments in this homeassignmentset in non-editable mode
     ******/
    Assignment.prototype.viewSolutions = function(){
        var assignment, aid, solutionplace;
        aid = this.assignmentid;
        if (this.data.assignmentType === 0 || this.type === 'assignment') {
            this.place.after('<div class="assignment-solutiontoolplace"></div>');
            solutionplace = this.place.next('.assignment-solutiontoolplace');
            solutionplace.solutiontool({
                data: {assignmentid: aid},
                settings: {
                    uilang: this.settings.uilang,
                    lang: this.settings.lang,
                    mode: (this.teacherable ? 'review' : 'edit'),
                    role: this.settings.role,
                    username: this.settings.username,
                    savemode: this.settings.savemode,
                    users: this.settings.users,
                    gotolink: JSON.parse(JSON.stringify(this.settings.gotolink))
                }
            });
            this.addTeacherbuttons();
        };
    };
    
    // TODO: Remove this? Is not used.
    /******
     * Show solutions for assignments in this homeassignmentset in editable mode
     ******/
    Assignment.prototype.editSolutions = function(){
        var assignment, aid, solutionplace;
        aid = this.assignmentid;
        if (this.data.assignmentType === 0 && this.type === 'assignment') {
            this.place.append('<div class="assignment-solutiontoolplace"></div>');
            solutionplace = this.place.children('.assignment-solutiontoolplace');
            solutionplace.solutiontool({
                data: {assignmentid: aid},
                settings: {
                    uilang: this.settings.uilang,
                    mode: 'edit',
                    role: this.settings.role,
                    username: this.settings.username,
                    savemode: this.settings.savemode,
                    users: this.settings.users
                }
            });
        };
    };
    
    /**
     * Add teacherbuttons
     */
    Assignment.prototype.addTeacherbuttons = function() {
        var teacherplace = this.place.find('.assignment-teachercontrol');
        if (teacherplace.length === 0) {
            teacherplace = this.place.next('.assignment-solutiontoolplace').find('.assignment-teachercontrol');
        }
        var teacherbuttons = [];
        var tbuttonset = ['<div class="ffwidget-buttonset ffwidget-horizontal assignment-teacherbuttons">'];
        if (this.settings.users && this.settings.users.canEdit({ctype: 'hamessage'})) {
            teacherbuttons.push('<button class="ffwidget-setbutton assignment-habutton" data-action="markashomeassignment" draggable="true">' + ElementSet.icons.home + '</button>');
        };
        tbuttonset.push(teacherbuttons.join(''));
        tbuttonset.push('</div>');
        if (teacherbuttons.length > 0) {
            teacherplace.append(tbuttonset.join(''));
        };
    };
    
    
    /******
     * No edit tools for Assignments.
     ******/
    //Assignment.prototype.showEditTools = function(){};
    
    /******
     * Get data of assignment
     * First call ElementSet's getData() and then do some additional stuff.
     ******/
    Assignment.prototype.getDataParent = Assignment.prototype.getData;
    Assignment.prototype.getData = function(){
        var result = this.getDataParent();
        result.data.assignmentNumber = this.data.assignmentNumber || '';
        result.data.level = this.data.level || '0';
        result.data.assignmentType = this.data.assignmentType;
        result.data.assignmentAnswer = this.answer;
        result.data.assignmenticons = (this.data.assignmenticons || []).slice();
        return result;
    }

    /**
     * Mark this assignment as a homeassignment
     */
    Assignment.prototype.markAsHomeassignment = function() {
        var hadata = this.getHomeassignmentData();
        this.place.trigger('homeassignmentdata', [hadata]);
    };
    
    /**
     * Get the homeassignmentdata
     * @returns {Object} data of this assignment as a homeassignment
     */
    Assignment.prototype.getHomeassignmentData = function() {
        var pagenumber = (this.place.closest('.notebook-viewarea').children('.notebook-pagetitle').find('.notebookview-titlenumber').text() || '');
        pagenumber = (pagenumber && `[${pagenumber}] ` || '');
        var titlenumber = this.place.find('.ebook-elementset-title .ebook-assignment-number').text() || '';
        var title = this.data.title;
        titlenumber = titlenumber.replace(/:$/, '');
        var hadata = {
            gotolink: {
                type: 'contentlink',
                metadata: {
                    creator: this.settings.username,
                    modifier: this.settings.username,
                    created: (new Date()).getTime(),
                    modified: (new Date()).getTime(),
                    tags: []
                },
                data: {
                    title: (titlenumber && title ? pagenumber + titlenumber + ': ' + title : pagenumber + titlenumber + title),
                    text: '',
                    type: this.place.closest('.elementset-element').attr('data-ebookelement-type'),
                    link: JSON.parse(JSON.stringify(this.settings.gotolink))
                }
            },
            title: (titlenumber && title ? pagenumber + titlenumber + ': ' + title : pagenumber + titlenumber + title),
            text: '',
            level: this.data.level,
            assignmentNumber: this.data.assignmentNumber
        };
        return hadata;
    };
    
    /******
     * Go to the link target
     * @param {Object} linkdata - link data to the target
     ******/
    Assignment.prototype.goToLink = function(linkdata){
        var link = linkdata && linkdata.data && linkdata.data.link;
        delete link.elementid;
        var soltool = this.place.parent().find('.assignment-solutiontoolplace.solutiontool');
        soltool.trigger('gotolink', linkdata);
    };
    
    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    Assignment.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(Assignment.elementinfo[this.type].cancontain));
    };
    
    Assignment.icons = {
        star: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-star"><path d="M15 2 l3.7 8 l9.3 1 l-7 6 l2 9 l-8 -4.8 l-8 4.8 l2 -9 l-7 -6 l9.3 -1z"></path></svg>',
        question: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="ebook-icon ebook-icon-assignment"><path stroke="none" d="M6 5 c4 -4, 8 -3.6, 11 -3 s 6 3, 6 6 s -1 5, -3 7 s-2 2, -3 3 s-1 2, -1 5 l-4 0 c0 -2, -0.1 -3.9, 1 -6 s 3 -3.2, 4 -4 s2.5 -3,2 -5 s-3.5 -3.3, -6 -3 s-5 2, -7 4z m6 20 l4 0 l0 4 l-4 0z" /></svg>',
        searchinfo: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-search"><path style="stroke: none;" d="M24 27 l-6 -6 a10 10 0 1 1 3 -3 l6 6 a1.5 1.5 0 0 1 -3 3z m-8 -8 a7 7 0 0 0 3 -3 a7.2 7.2 0 1 0 -3 3z m-3 -2 a4 4 0 0 1 -5 -5 a9 9 0 0 0 5 5z" /></svg>',
        externalapp: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-external"><path style="stroke: none;" d="M28 2 l0 10 l-4 -4 l-8 8 a3 3 0 1 1 -2 -2 l8 -8 l-4 -4z m-12 2 v3 h-10 a3 3 0 0 0 -3 3 v14 a3 3 0 0 0 3 3 h14 a3 3 0 0 0 3 -3 v-10 h3 v10 a6 6 0 0 1 -6 6 h-14 a6 6 0 0 1 -6 -6 v-14 a6 6 0 0 1 6 -6z" /></svg>',
        nocalculator: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-nocalculator"><path style="stroke: none;" d="M7 1 h16 a2 2 0 0 1 2 2 v22 a2 2 0 0 1 -2 2 h-16 a2 2 0 0 1 -2 -2 v-22 a2 2 0 0 1 2 -2z m0 2 v6 h16 v-6z m12 1 h3 v4 h-3z m1 1 v2 h1 v-2z M10 11 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m-10 5 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m-10 5 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z m5 0 a2 2 0 1 0 0 4 a2 2 0 1 0 0 -4z" /><path style="stroke: none; fill: red;" d="M29 4 l1 5 l-29 18 l-1 -5z" />'
    }

    /******
     * Stylesheets for assignments
     ******/
    Assignment.styles = [
        // Titles and levels
        '.ebook-elementset-titlebox {display: flex; flex-flow: row nowrap; justify-content: space-between; align-items: center; border-bottom: 2px groove black;}',
        // Answer and hiding with button
        '.assignmentsAnswer {clear: both; margin-bottom: 0.5em;}',
        '.assignmentsAnswer::after {content: ""; display: block; clear: both;}',
        '.assignmentsAnswer .answertext {padding: 1em; margin: 0 1em 0 12em; background-color: white; color: black; border: 1px solid #aaa; border-radius: 0 1em 1em 1em; position: absolute; z-index: 3; box-shadow: 8px 8px 15px rgba(0,0,0,0.2);}',
        '.assignmentsAnswer.answerhidden .answertext {visibility: hidden;}',
        '.assignmentsAnswer button.showAnswer {margin-left: 1em; padding: 0.5em 1em; max-width: 11em; float: left;}',
        '.assignmentsAnswer button.showAnswer .showbutton {display: none;}',
        '.assignmentsAnswer.answerhidden button.showAnswer .showbutton {display: inline;}',
        '.assignmentsAnswer.answerhidden button.showAnswer .hidebutton {display: none;}',
        '.assignmentsAnswer .mathbox.assignment-answer-input.mathquill-textbox {display: block; padding: 0.4em; margin: 0.1em;}',
        '.ebook-assignment h1.ebook-elementset-title {font-size: 110%; margin: 0; padding: 0.2em 0.5em; flex-grow: 1;}',
        '.ebook-assignment[data-elementmode$="view"] h1.ebook-elementset-title:empty {padding: 0;}',
        '.ebook-assignment[data-elementmode="view"], .ebook-otherassignment[data-elementmode="view"] {margin: 2em 0 0;}',
        '.ebook-assignment, .ebook-otherassignment {position: relative; border: 1px dashed #666; margin: 0; background-color: #f0f0f0;}',
        '.ebook-assignment[data-elementmode="edit"], .ebook-otherassignment[data-elementmode="edit"] {margin: 0;}',
        '.ebook-assignment > .ebook-elementset-header, .ebook-otherassignment > .ebook-elementset-header {position: relative;}',
        '.ebook-assignment > .ebook-elementset-titlebox > .elementset-assignment-level, .ebook-otherassignment > .ebook-elementset-titlebox > .elementset-assignment-level {margin: 2px;}',
        '.elementset-assignment-level svg.mini-icon-star, .ebook-assignment > .ebook-elementset-header svg.mini-icon-star, .ebook-otherassignment > .ebook-elementset-header svg.mini-icon-star {fill: gold; stroke: black; stroke-width: 1px;}',
        '.ebook-assignment[data-elementmode="edit"] > .ebook-elementset-header svg.mini-icon-star, .ebook-otherassignment[data-elementmode="edit"] > .ebook-elementset-header svg.mini-icon-star {cursor: pointer;}',
        '.ebook-assignment h1.ebook-elementbox-title > input.assignment-title-input {font-size: 120%; font-weight: bold; width: 100%; box-sizing: border-box;}',
        '.elementset-assignment-level[data-assignmentlevel="3"] > svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star path {fill: #ddd; stroke: #999;}',
        '.elementset-assignment-level[data-assignmentlevel="2"] > svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star path {fill: #ddd; stroke: #999;}',
        '.elementset-assignment-level[data-assignmentlevel="1"] > svg.mini-icon-star + svg.mini-icon-star + svg.mini-icon-star path {fill: #ddd; stroke: #999;}',
        '.elementset-assignment-level[data-assignmentlevel="0"] > svg.mini-icon-star + svg.mini-icon-star path {fill: #ddd; stroke: #999;}',
        'textarea.assignment-answer-input {display: block; width: 100%; box-sizing: border-box;}',
        
        // Quizset
        '.ebook-quizset .elementset-element.markdownelement {margin: 0.5em 0;}',
        '.ebook-quizset[data-elementmode="view"] > .ebook-elementset-body > .elementset-elementwrapper {margin-top: 2.5em; margin-bottom: 2.5em;}',
        '.quizset-solving-buttonarea {list-style: none; }',
        '.quizset-solving-buttonarea li {display: inline-block;}',
        '.quizset-solvingarea.quizset-issolving .quizset-solving-startnew-button {display: none;}',
        '.quizset-solvingarea .quizset-solving-send-button {display: none;}',
        '.quizset-solvingarea.quizset-waitingforsend .quizset-solving-send-button {display: inline-block; opacity: 0.3; cursor: default;}',
        '.quizset-solvingarea.quizset-readytosend .quizset-solving-send-button {display: inline-block; opacity: 1; cursor: pointer;}',
        '.quizquestion[data-elementmode$="view"] {margin-left: 2em;}',
        '.quizquestion[data-quizset-question]:before {content: attr(data-quizset-question) ")"; float: left; margin-right: 1em;}',
        '.quizset-solving-resultarea {display: none;}',
        '.quizset-userissolving .quizset-solving-resultarea {display: none;}',
        '.quizset-solving-resultarea.quizset-showresults {display: block; margin: 0.2em 1em; padding: 0.5em 1em; border: 1px solid #aaa; background-color: rgba(255,255,255,0.3);}',
        '.quizset-solvingarea .quizset-solving-buttonarea svg {vertical-align: middle; width: 25px; height: 25px;}',
        '.quizset-solvingarea .quizset-solving-buttonarea span {vertical-align: middle;}',
        
        // Quizset teacherarea
        '.quizset-teacherarea {background-color: #eee; margin: 0.3em 0; padding: 0; border-top: 1px dashed black; border-bottom: 1px dashed black;}',
        //'.quizset-teachercontrol {background-color: #eee; margin: 0.3em -0.5em; padding: 0; border-top: 1px dashed black; border-bottom: 1px dashed black;}',
        '.quizset-teacherbuttons {margin: 0.5em 1em;}',
        
        // Quizset statistics
        'table.quizstatistics {border: 1px solid #aaa; width: 95%; margin: 0.5em auto; border-collapse: collapse; background-color: #fafafa;}',
        'table.quizstatistics thead th {font-weight: bold; text-align: center; padding: 0.2em 0.5em; border-bottom: 1px solid #777;}',
        'table.quizstatistics tbody th.quizstatistics-username {font-weight: normal; text-align: left; padding: 0.2em 0.5em;}',
        'table.quizstatistics td.quizstatistics-bestscore {text-align: center;}',
        'table.quizstatistics td.quizstatistics-numoftries {text-align: center;}',
        'table.quizstatistics .quizstatistics-allcorrect {color: #080; font-weight: bold;}',
        'table.quizstatistics tr.quizstatistics-userbest {cursor: pointer;}',
        'table.quizstatistics tr.quizstatistics-userbest:hover {box-shadow: 0 0 3px rgba(0,0,0,0.5); background-color: white;}',
        'table.quizstatistics tr.quizstatistics-evenrow {background-color: #fafafa;}',
        'table.quizstatistics tr.quizstatistics-oddrow {background-color: #f0f0f0;}',
        'table.quizstatistics tr.quizstatistics-userallsols {display: none;}',
        'table.quizstatistics tr.quizstatistics-userrowopen + tr.quizstatistics-userallsols {display: table-row; box-shadow: inset 1px 1px 3px rgba(0,0,0,0.5), inset -1px -1px 3px rgba(255,255,255,0.8);}',
        'table.quizstatistics .quizstatistics-solvescore {display: inline-block; min-width: 5em; text-align: center;}',
        'table.quizstatistics .quizstatistics-solvedate {color: #555; font-size: 95%; font-style: italic;}',
        '.quizset-teacherbuttons .quizset-stats-button-show-hide .quizset-stats-button-hide-label {display: none;}',
        '.quizstats-showstats .quizset-stats-button-show-hide .quizset-stats-button-hide-label {display: inline;}',
        '.quizstats-showstats .quizset-stats-button-show-hide .quizset-stats-button-show-label {display: none;}',
        '.quizset-statsarea .quizset-stats-scoreboard {display: none;}',
        '.quizstats-showstats .quizset-statsarea .quizset-stats-scoreboard {display: block;}',


        '.quizset-stats-buttons button {padding: 0.2em 1em; margin: 0.2em 2.6em;}',
        '.quizstatistics-userallsol-list {list-style: none; padding: 0; margin: 0.5em 0;}',
        '.quizstatistics-userallsol-list li {cursor: pointer; margin: 0; padding: 0.1em 0 0.1em 2em;}',
        '.quizstatistics-userallsol-list li:hover {background-color: white; box-shadow: 0 0 3px rgba(0,0,0,0.5);}',
        '.quizstatistics-userallsol-list li.quizstatistics-selectedsol {font-weight: bold;}',
        '.quizset-solverdata {display: none; float: right; background-color: #a00; border: 1px solid #555; border-radius: 0 0.3em 0 0.3em; padding: 0.5em; font-size: 85%;}',
        '.quizset-solverdata.quizset-showsolverdata {display: block;}',
        '.quizset-solvername {color: white; font-weight: bold; text-shadow: 1px 1px 1px rgba(0,0,0,0.5);}',
        '.ebook-quizset > .ebook-elementset-body.quizset-usersolutionview {border: 1px solid #555; border-radius: 0.3em; background-color: #fefefe;}',
        
        // Teacherarea / Homeassignments
        '.assignment-teacherarea {background-color: #eee;}',
        '.assignment-teachercontrol {display: flex; justify-content: space-between; flex-flow: row nowrap;}',
        '.assignment-teacherarea button {padding: 2px 2em;}',
        '.assignment-teacherarea .ffwidget-buttonset {margin: 0; padding: 0.5em;}',
        
        // Homeassignment marks
        '.elementset-assignment-hamessages {position: absolute; top: -26px; left: 1em; border: 1px solid #888; border-color: inherit; border-style: inherit; border-bottom-width: 0; padding: 2px; padding-bottom: 0; border-radius: 4px 4px 0 0; background: inherit;}',
        '.elementset-assignment-hamessages:empty {display: none;}',
        '.homeassignmentelement-marking {display: inline-block; text-align: center; padding: 3px; width: 17px; height: 17px; line-height: 0; border-radius: 50%; margin: 0 3px; cursor: pointer; position: relative;}',
        '.homeassignmentelement-marking[data-overtime="true"] {border: 1px solid #888;}',
        '.homeassignmentelement-marking[data-overtime="today"] {border-color: #a00;}',
        '.homeassignmentelement-marking svg {width: 15px; height: 15px;}',
        '.homeassignmentelement-marking-info {display: none; position: absolute; top: 32px; left: 0; border: 1px solid #999; background-color: #ffc; padding: 0.2em; font-size: 80%; line-height: 1em;}',
        '.homeassignmentelement-marking:hover .homeassignmentelement-marking-info {display: block; z-index: 20;}',
        
        // Assignmenticons
        '.elementset-assignmenticons {padding: 2px 5px; margin: 0 0.5em; background-color: rgba(255,255,255,0.5); border-radius: 5px;}',
        '.elementset-assignmenticons:empty {display: none;}',
        '.elementset-assignment-icon {display: inline-block; margin: 0 4px;}',
        '.elementset-assignment-icon svg {height: 25px; width: auto;}',
        
        // Assignments in minimal mode
        '.ebook-assignment > .ebook-elementset-titlebox {cursor: pointer;}',
        '.ebook-assignment > .ebook-elementset-titlebox:hover {background-color: rgba(255,255,255,0.5); transition: background-color 0.2s;}',
        '.ebook-assignment.assignment-minimal {font-size: 80%;}',
        '.ebook-assignment.assignment-minimal .assignmentsAnswer {display: none;}',
        '.ebook-assignment.assignment-minimal > .ebook-elementset-body > .elementset-elementwrapper > .elementset-element {display: none;}',
        '.ebook-assignment.assignment-minimal > .ebook-elementset-body > .elementset-elementwrapper > .elementset-element[data-ebookelement-type="markdownelement"], .ebook-assignment.assignment-minimal > .ebook-elementset-body > .elementset-elementwrapper > .elementset-element[data-ebookelement-type="wikielement"] {display: block;}',
        '.ebook-assignment.assignment-minimal + .solutiontool {display: none;}'
    ].join('\n');
    
    /******
     * Assignment's default data and settings
     ******/
    Assignment.default = {
        type: 'assignment',
        metadata: {},
        data: {
            assignmentType: 0
        },
        settings: {
            savemode: 'local',
            username: 'Anonymous'
        }
    }
        
    /******
     * Info about ContentTable (icon, description, etc.)
     ******/
    Assignment.elementinfo = {
        assignment: {
            type: 'assignment',
            elementtype: ['assignments'],
            boxtype: 'assignment',
            jquery: 'assignment',
            name: 'Assignment',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-assignment"><path stroke="none" fill="black" d="M4 1 h18 a2 2 0 0 1 2 2 v4 l-2 2 v-6 h-18 v24 h18 v-10 l2 -2 v12 a2 2 0 0 1 -2 2 h-18 a2 2 0 0 1 -2 -2 v-24 a2 2 0 0 1 2 -2z M27 5 l3 3 l-14 14 l-3.5 0.5 l0.5 -3.5z m0 1.5 l-13 13 l1.5 1.5 l13 -13z M7 8 c2 -1 4 -1.5 6 0 c1.6 1.8 1.2 3 -1 5 c-1 1 -1 1 -1 2 h-2.5 c0 -1 0.2 -2 1.5 -3 c2.2 -2.2 1 -4 -3 -2z m1.5 8 h2.5 v2.5 h-2.5z" /></svg>',
            description: {
                en: 'Assignment with open answers',
                fi: 'Tehtävä avoimilla vastauksilla',
                sv: 'Öppen uppgift'
            },
            weight: 50,
            roles: ['teacher', 'author'],
            classes: ['assignments'],
            cancontain: ['containers', 'elements'],
            dragdata: '{"type": "assignment", "data": {"assignmentType": 0}}'
        },
        otherassignment: {
            type: 'otherassignment',
            elementtype: ['otherassignments'],
            boxtype: 'otherassignment',
            jquery: 'otherassignment',
            name: 'Otherassignment',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-otherassignment"><text x="10" y="24" text-anchor="middle" style="font-weight: bold; font-size: 15px; font-family: sans;">?</text><path style="stroke: none;" d="M16 17 l2 0 l3 5 l 7 -12 l2 0 l-9 14 l-2 0 "></path></svg>',
            description: {
                en: 'An assignment containing interactive elements',
                fi: 'Tehtävä, joka sisältää interaktiivisen tehtävän sisältöelementtejä',
                sv: 'Uppgift med interaktiva element'
            },
            weight: 50,
            roles: ['author'],
            classes: ['assignmentsset', 'viewonly'],
            cancontain: ['containers', 'elements']
        }
    };
    
    // Register assignmentlist as an elementset and to the elementpanel.
    $.fn.elementset('addelementtype', Assignment.elementinfo.assignment);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Assignment.elementinfo.assignment);
    }
    $.fn.elementset('addelementtype', Assignment.elementinfo.otherassignment);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Assignment.elementinfo.otherassignment);
    };
    
    /**** jQuery-plugin *****/
    var assignmentmethods = {
        'init': function(params){
            return this.each(function(){
                var Sdassignment = new Assignment(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }

    $.fn.assignment = function(method){
        if (assignmentmethods[method]) {
            return assignmentmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return assignmentmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in elementbox.');
            return false;
        }
    }

    $.fn.otherassignment = $.fn.assignment;

    var quizsetmethods = {
        'init': function(params){
            return this.each(function(){
                var quizset = new Quizset(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }


    var Quizset = function(place, options){
        options = $.extend(true, {}, Assignment.defaults, Quizset.defaults, options);
        options.settings.savemode = 'local';
        ElementSet.call(this, place, options);
    };
    
    Quizset.prototype = Object.create(ElementSet.prototype);
    Quizset.prototype.constructor = Quizset;
    Quizset.prototype.parentClass = ElementSet.prototype;

    /******
     * Init quizset data with ElementSet's initData and do some more.
     ******/
    Quizset.prototype.initData = function(options) {
        this.parentClass.initData.call(this, options);
        this.place.addClass('ebook-assignment ebook-quizset');
        this.assignmentid = this.place.attr('data-element-name');
        if (this.settings.users) {
            this.teacherable = this.settings.users.canEdit({ctype: 'review', contextid: this.assignmentid});
        };
        this.settings.feedback = options.data.feedback || this.settings.feedback;
        this.type = 'quizset';
        this.solutions = new QuizSolutionSet();
    };
    
    Quizset.prototype.getDataParent = Quizset.prototype.getData;
    Quizset.prototype.getData = function(){
        var result = Assignment.prototype.getDataParent.call(this);
        result.data.feedback = this.data.feedback;
        result.data.showfirst = this.data.showfirst;
        return result;
    }


    Quizset.prototype.initHandlersNoneditableParent = Quizset.prototype.initHandlersNoneditable;
    Quizset.prototype.initHandlersNoneditable = function(){
        var quizset = this;
        Assignment.prototype.initHandlersNoneditable.call(quizset);
        // New with storage
        this.place.off('setcontentbyref').on('setcontentbyref', function(event, data, localsave){
            // Handle only events coming from children. Not events triggered by this quizset itself.
            if (event.target !== quizset.place[0]) {
                event.stopPropagation();
                quizset.addSolutionData(data, localsave);
                var total = quizset.place.find('.quizquestion').length;
                var answered = quizset.solutions.getAnswerCount();
                if (total === answered) {
                    quizset.place.find('.quizset-solvingarea').addClass('quizset-readytosend');
                }
            }
        });
        // New with storage
        this.place.off('reply_getcontent').on('reply_getcontent', function(event, data){
            event.stopPropagation();
            if (event.target === quizset.place[0] && data.contentType === 'solution') {
                quizset.addQuizsetSolutionData(data);
            };
        });
        // Reply for requests from quiz elements (multichoice, shortanswer, shuffleassignments)
        this.place.off('getcontentbyref').on('getcontentbyref', function(event, data){
            if (event.target !== quizset.place[0] && !quizset.teacherable) {
                event.stopPropagation();
                var replydata = quizset.getSolutionData(data.refs);
                if (replydata) {
                    $(event.target).trigger('reply_getcontentbyref', replydata);
                };
            };
        })
    };

    Quizset.prototype.initHandlersEditableParent = Quizset.prototype.initHandlersEditable;
    Quizset.prototype.initHandlersEditable = Assignment.prototype.initHandlersEditable;

    Quizset.prototype.showTitle = Assignment.prototype.showTitle;

    Quizset.prototype.showTitleView = Assignment.prototype.showTitleView;

    Quizset.prototype.showTitleEdit = Assignment.prototype.showTitleEdit;

    Quizset.prototype.goToLink = Assignment.prototype.goToLink;

    Quizset.prototype.addTeacherbuttons = Assignment.prototype.addTeacherbuttons;

    Quizset.prototype.markAsHomeassignment = Assignment.prototype.markAsHomeassignment;

    // Move current showContent() before redefining it.
    Quizset.prototype.showContentParent = Quizset.prototype.showContent;
    Quizset.prototype.showContent = function(){
        var lang = this.place.closest('[lang]').attr('lang');
        var uilang = this.settings.uilang;
        // New with storage
        this.place.trigger('getcontent', {anchors: [this.assignmentid], contentType: ['solution']});
        if (this.solutions.getCount() > 0 || this.teacherable) {
            this.showContentParent();
            this.showQuizTools();
            var questions = this.place.find('.quizquestion');
            var total = questions.length;
            for (var i = 0, len = questions.length; i < len; i++) {
                questions.eq(i).attr('data-quizset-question', (i+1));
            };
            if (!this.editable) {
                if (!this.solutions.isClosed()) {
                    var solarea = this.place.find('.quizset-solvingarea');
                    solarea.addClass('quizset-issolving quizset-waitingforsend');
                    var answered = this.solutions.getAnswerCount();
                    if (total === answered) {
                        solarea.addClass('quizset-readytosend');
                    }
                } else if (this.solutions.getCount() > 0) {
                    this.showStats();
                };
                if (this.teacherable) {
                };
            };
        } else {
            if (this.data.showfirst) {
                this.showContentParent();
            };
            this.showQuizTools();
        };
    };
    
    /******
     * Show tools for Quizsets (buttons: send, retry, ...)
     ******/
    Quizset.prototype.showQuizTools = function(){
        if (!this.editable) {
            if (this.teacherable) {
                this.showQuizToolsTeacher();
            } else {
                this.showQuizToolsStudent();
            };
            this.addQuizsetHandlers();
        };
    };
    
    Quizset.prototype.showQuizToolsTeacher = function(){
        var uilang = this.settings.uilang;
        this.place.find('> .ebook-elementset-body').before([
            '<div class="quizset-teacherarea assignment-teacherarea">',
            '<div class="quizset-teachercontrol assignment-teachercontrol ffwidget-background">',
            '<div class="quizset-teacherbuttons ffwidget-buttonset ffwidget-horizontal">',
            '<button class="ffwidget-setbutton quizset-teacherbutton-clear">'+ebooklocalizer.localize('quizstats:clear solution', uilang)+'</button>',
            '<button class="ffwidget-setbutton quizset-teacherbutton-showcorrects">'+ebooklocalizer.localize('quizstats:show corrects', uilang)+'</button>',
            '<button class="ffwidget-setbutton quizset-stats-button-show-hide"><span class="quizset-stats-button-show-label" title="'+ebooklocalizer.localize('quizstats:show statistics', uilang)+'">'+ebooklocalizer.localize('quizstats:show statistics', uilang)+'</span>',
            '<span class="quizset-stats-button-hide-label" title="'+ebooklocalizer.localize('quizstats:hide statistics', uilang)+'">'+ebooklocalizer.localize('quizstats:hide statistics', uilang)+'</span></button>',
            '</div>',
            '</div>',
            '<div class="assignment-quizscoreboard quizset-statsarea"></div>',
            '</div>', // end teacherarea
            '<div class="quizset-solverdata">',
            '    <div class="quizset-solvername"></div>',
            '    <div class="quizset-solverdate"></div>',
            '    <div class="quizset-solverscore"></div>',
            '</div>'
        ].join(''));
        var statsarea = this.place.find('.quizset-statsarea');
        if (this.solutions.getCount() > 0 || this.teacherable) {
            var questions = this.place.find('.quizquestion');
            var total = 0;
            for (var i = 0, len = questions.length; i < len; i++) {
                total += questions.eq(i).attr('data-quiztotal') | 0;
            };
            var statstool = new QuizStats(this.solutions, total, this.settings.uilang, this.settings.users);
            statsarea.html(statstool.html());
        };
        this.addTeacherbuttons();
    };
    
    Quizset.prototype.showQuizToolsStudent = function(){
        var quiztool;
        this.place.append('<div class="assignment-quiztool"></div>');
        quiztool = this.place.children('.assignment-quiztool');
        quiztool.html(Quizset.templates.solving);
        this.addQuizButtons();
        var solvingarea = quiztool.find('.quizset-solvingarea');
        if ((this.data.showfirst && this.solutions.getCount() === 0) || !this.solutions.isClosed()) {
            this.startSolving();
        };
    };
    
    Quizset.prototype.addQuizButtons = function(){
        var uilang = this.settings.uilang;
        var buttonset = this.place.find('.quizset-solvingarea .quizset-solving-buttonarea');
        var html = [
            '        <li><button class="ffwidget-button quizset-solving-startnew-button" title="'+ ebooklocalizer.localize('quizset:start new a try', uilang) +'">'+Quizset.icons.edit + ' <span>'+ ebooklocalizer.localize('quizset:start new a try', uilang) +'</span></button></li>',
            '        <li><button class="ffwidget-button quizset-solving-send-button" title="'+ ebooklocalizer.localize('quizset:send quiz solution', uilang) +'">'+Quizset.icons.send+' <span>'+ ebooklocalizer.localize('quizset:send quiz solution', uilang) +'</span></button></li>'
        ].join('\n');
        buttonset.html(html);
    };
    
    Quizset.prototype.startSolving = function(){
        var place = this.place;
        // Start solving, if not already started.
        if (!place.is('.quizset-userissolving')) {
            this.place.addClass('quizset-userissolving');
            var resultarea = this.place.find('.quizset-solving-resultarea');
            resultarea.removeClass('quizset-showresults');
            this.place.find('.quizset-solvingarea').addClass('quizset-issolving quizset-waitingforsend');
        };
    };
    
    Quizset.prototype.stopSolving = function() {
        this.place.removeClass('quizset-userissolving');
    }
    
    Quizset.prototype.addQuizsetHandlers = function(){
        var quizset = this;
        this.place.off('click', '.quizset-solving-buttonarea .quizset-solving-startnew-button').on('click', '.quizset-solving-buttonarea .quizset-solving-startnew-button', function(event, data){
            event.stopPropagation();
            quizset.newSolutionset();
            quizset.showContentParent();
            var questions = quizset.place.find('.quizquestion');
            for (var i = 0, len = questions.length; i < len; i++) {
                questions.eq(i).attr('data-quizset-question', (i+1));
            };
            quizset.startSolving();
        });
        this.place.off('click', '.quizset-solvingarea.quizset-readytosend .quizset-solving-buttonarea .quizset-solving-send-button').on('click', '.quizset-solvingarea.quizset-readytosend .quizset-solving-buttonarea .quizset-solving-send-button', function(event, data){
            event.stopPropagation();
            quizset.sendSolution();
            quizset.stopSolving();
            quizset.show();
        });
        this.place.off('startsolving').on('startsolving', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            quizset.startSolving();
        });
        if (this.teacherable) {
            this.place.off('click', '.quizstatistics-userbest').on('click', '.quizstatistics-userbest', function(event, data){
                event.stopPropagation();
                event.preventDefault();
                var row = $(this);
                var allrows = row.parent('tbody').children('tr.quizstatistics-userbest');
                var isopen = row.hasClass('quizstatistics-userrowopen');
                var score = row.attr('data-score');
                var bestscorerow = row.next().find('li[data-score="'+score+'"]').last();
                allrows.removeClass('quizstatistics-userrowopen');
                if (!isopen) {
                    row.addClass('quizstatistics-userrowopen');
                    bestscorerow.click();
                } else {
                    quizset.clearSolver();
                    quizset.place.find('.quizquestion').trigger('clearanswer');
                };
            });
            this.place.off('click', '.quizset-teacherbuttons .quizset-stats-button-show-hide').on('click', '.quizset-teacherbuttons .quizset-stats-button-show-hide', function(event, data){
                event.stopPropagation();
                event.preventDefault();
                var button = $(this);
                //button.closest('[data-ebookelement-type="quizset"]').toggleClass('quizstats-showstats');
                quizset.place.toggleClass('quizstats-showstats');
                quizset.place.find('.quizset-statsarea .quizstatistics-userrowopen').removeClass('quizstatistics-userrowopen');
                quizset.clearSolver();
                quizset.place.find('.quizquestion').trigger('clearanswer');
            });
            this.place.off('click', '.quizset-teacherbuttons button.quizset-teacherbutton-clear').on('click', '.quizset-teacherbuttons button.quizset-teacherbutton-clear', function(event, data){
                event.stopPropagation();
                event.preventDefault();
                quizset.clearSolver();
                quizset.place.find('.quizquestion').trigger('clearanswer');
            });
            this.place.off('click', '.quizset-teacherbuttons button.quizset-teacherbutton-showcorrects').on('click', '.quizset-teacherbuttons button.quizset-teacherbutton-showcorrects', function(event, data){
                event.stopPropagation();
                event.preventDefault();
                quizset.clearSolver();
                quizset.place.find('.quizquestion').trigger('showcorrectanswer');
            });
            this.place.off('click', '.quizset-stats-scoreboard .quizstatistics-userallsol-list li').on('click', '.quizset-stats-scoreboard .quizstatistics-userallsol-list li', function(event, data){
                event.stopPropagation();
                event.preventDefault();
                var item = $(this);
                var solname = item.attr('data-solutionname');
                item.closest('.quizstatistics').find('.quizstatistics-userallsol-list li').removeClass('quizstatistics-selectedsol');
                item.addClass('quizstatistics-selectedsol');
                quizset.showSolution(solname);
            });
        };
    };
    
    /******
     * Create a new quizset solution
     ******/
    Quizset.prototype.newSolutionset = function(){
        if (this.solutions.isClosed()) {
            this.solutions.addSolution({name: this.getNewId('quizsetsolution'), metadata: {ref: this.assignmentid}, settings: {username: this.settings.username, lang: this.settings.lang, uilang: this.settings.uilang, role: this.settings.role}});
            this.place.find('.quizset-solvingarea').addClass('quizset-waitingforsend');
        };
    }
    
    /******
     * Add a solution data of a quizset as whole (got as 'reply_getcontent' from the system)
     * @param {Object} data - the data as type: 'quizsetsolution'.
     ******/
    Quizset.prototype.addQuizsetSolutionData = function(data){
        var refelems = data.refs[this.assignmentid] || [];
        var refname, refdata;
        for (var i = 0, len = refelems.length; i < len; i++) {
            refname = refelems[i];
            if (data.refcontentdata) {
                // TODO old format
                refdata = data.refcontentdata[refname];
            } else if (data.contentdata){
                refdata = data.contentdata[refname];
            };
            refdata.settings = {username: this.settings.username, lang: this.settings.lang, uilang: this.settings.uilang, role: this.settings.role};
            this.solutions.addSolution(refdata);
        };
    };
    
    /******
     * Add data of a single question in the quiz (got as 'setcontentbyref' from question itself)
     * @param {Object} data - the data as type of 'setcontentbyref' event.
     ******/
    Quizset.prototype.addSolutionData = function(data, localsave){
        var name = data.name;
        this.newSolutionset();
        this.solutions.addSolutionData(data);
        if (!localsave) {
            this.setRefContent({});
        }
    };
    
    Quizset.prototype.getSolutionData = function(reftos){
        var reply;
        //if (!this.solutions.isClosed()) {
        if (this.solutions.getCount() > 0) {
            reply = this.solutions.getSolutionData(reftos);
        } else {
            reply = false;
        };
        return reply;
    }
    
    /******
     * Trigger saving of content. If optional senddata is given, the data is also sent according to that.
     * @param {Object} contentinfo - optional information about sending (sendto, sendfrom, senddate, etc...)
     ******/
    Quizset.prototype.setRefContent = function(contentinfo){
        var currsoldata = this.solutions.getDataCurrent();
        if (currsoldata && currsoldata.metadata) {
            currsoldata.metadata.lang = 'common';
        };
        contentinfo = $.extend(true, {
            name: currsoldata.name,
            contentType: 'solution',
            lang: 'common', //this.settings.lang,
            pubtype: 'own',
            sendto: [],
            sendfrom: this.settings.username,
            timestamp: (new Date()).getTime(),
            read: true,
            recipientopened: false,
            isoutbox: true
        }, contentinfo);
        this.place.trigger('setcontent', [
            {
                name: currsoldata.name,
                contentType: 'solution',
                lang: 'common', // this.settings.lang,
                anchors: [this.assignmentid],
                contentdata: currsoldata,
                contentinfo: contentinfo
            },
            true  // isdirty
        ]);
        this.place.trigger('savecontent');
    };
    
    /******
     * Close and send the solution, when it is ready.
     ******/
    Quizset.prototype.sendSolution = function(){
        // Close all questions in this quizset. (Each question triggers its own saving to this quizset.)
        this.place.find('> .ebook-elementset-body > .elementset-elementwrapper > .elementset-element, > .ebook-elementset-body > .ebook-assignmentindent > li > .elementset-elementwrapper > .elementset-element').trigger('close_quizquestion', {localsave: true});
        // Close this quizset
        this.solutions.close();
        // Gather all needed information about this sending.
        var current = this.solutions.current;
        var recipients = this.settings.users.getRecipientSendto('solution');
        var contentinfo = {
            name: current,
            contentType: 'solution',
            lang: 'common', // this.settings.lang,
            //reftype: 'solutiondata',
            //reftos: [this.assignmentid],
            pubtype: 'teacher',
            sendto: recipients,
            sendfrom: this.settings.username,
            //senddate: (new Date()).toString(),
            timestamp: (new Date()).toString(),
            read: true,
            recipientopened: false,
            isoutbox: true
        }
        // Save the data of this solution together with the optinal senddata for sending.
        this.setRefContent(contentinfo);
        this.place.trigger('sendcontent');
    }
    
    /******
     * Show the solution with given name.
     * @param {String} solname - name of the solution
     ******/
    Quizset.prototype.showSolution = function(solname){
        var solution = this.solutions.soldata[solname];
        var qname, ansname, qsol, refs = solution.refs;
        var qplace, replydata;
        this.place.find('.quizquestion').trigger('clearanswer').trigger('setmode', 'view');
        for (qname in refs) {
            ansname = refs[qname];
            qsol = $.extend(true, {}, solution.solutions[ansname]);
            qsol.data.closed = true;
            qplace = this.place.find('.quizquestion[data-element-name="'+qname+'"]');
            replydata = {
                refcontentdata: {},
                refs: {}
            };
            replydata.refcontentdata[ansname] = qsol;
            replydata.refs[qname] = [ansname];
            qplace.trigger('reply_getcontentbyref', replydata);
        };
        var $solverdata = this.place.find('.quizset-solverdata');
        var $solvername = $solverdata.children('.quizset-solvername');
        $solverdata.addClass('quizset-showsolverdata')
        $solvername.html(this.settings.users && this.settings.users.getRealnameLnameFirst && this.settings.users.getRealnameLnameFirst(solution.metadata.modifier) || this.escapeHTML(solution.metadata.modifier));
        this.place.children('.ebook-elementset-body').addClass('quizset-usersolutionview questionhighlights');
    }
    
    Quizset.prototype.clearSolver = function(){
        var solverdata = this.place.find('.quizset-solverdata')
        solverdata.removeClass('quizset-showsolverdata');
        solverdata.children('.quizset-solvername, .quizset-solverdate, .quizset-solverscore').empty();
        this.place.children('.ebook-elementset-body').removeClass('quizset-usersolutionview questionhighlights');
        this.place.find('.quizstatistics-selectedsol').removeClass('quizstatistics-selectedsol');
    }

    Quizset.prototype.showStats = function(){
        var qnum = this.place.find('.quizquestion').length;
        var stats = this.solutions.getStats(qnum);
        var resultarea = this.place.find('.quizset-solving-resultarea');
        resultarea.addClass('quizset-showresults');
        resultarea.append('<p>'+ebooklocalizer.localize('quizset:you got corrects', this.settings.uilang) + ': '+stats.correct +'/'+stats.total+'.</p>' )
    }

    Quizset.prototype.getDataParent = Quizset.prototype.getData;
    Quizset.prototype.getData = Assignment.prototype.getData;

    Quizset.prototype.cancontain = function(){
        return JSON.parse(JSON.stringify(Quizset.elementinfo.cancontain));
    }
    
    Quizset.defaults = {
        type: 'quizset',
        metadata: {},
        data: {
            feedback: false,
            showfirst: true
        },
        settings: {
            savemode: 'local',
            username: 'Anonymous',
            feedback: false
        }
    }

    Quizset.icons = {
        send: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-send"><path style="stroke: none;" d="M2 18 l24 -12 l-4 18 l-7 -3 l-4 4 l-1 0 l-1 -6z m8 0 l0.7 6 l2 -4.5 l12 -12z"></path></svg>',
        edit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>'
    }
    
    Quizset.localization = {
        "en": {
            "quizset:you got corrects": "You got correct answers",
            "quizset:send quiz solution": "send solution",
            "quizset:start new a try": "Start a new try",
            "quizstats:username": "Name",
            "quizstats:best try": "Best try",
            "quizstats:number of tries": "Number of tries",
            "quizstats:show statistics": "Show statistics",
            "quizstats:hide statistics": "Hide statistics",
            "quizstats:clear solution": "Clear",
            "quizstats:show corrects": "Show corrects"
        },
        "fi": {
            "quizset:you got corrects": "Sait oikeita vastauksia",
            "quizset:send quiz solution": "Lähetä vastaus",
            "quizset:start new a try": "Aloita uusi yritys",
            "quizstats:username": "Nimi",
            "quizstats:best try": "Paras yritys",
            "quizstats:number of tries": "Yritysten määrä",
            "quizstats:show statistics": "Näytä tilastot",
            "quizstats:hide statistics": "Piilota tilastot",
            "quizstats:clear solution": "Tyhjennä",
            "quizstats:show corrects": "Näytä oikeat"
        },
        "sv": {
            "quizset:you got corrects": "Du svarade rätt",
            "quizset:send quiz solution": "Skicka lösning",
            "quizset:start new a try": "Gör ett försök",
            "quizstats:username": "Namn",
            "quizstats:best try": "Bästa försök",
            "quizstats:number of tries": "Antal försök",
            "quizstats:show statistics": "Visa statistik",
            "quizstats:hide statistics": "Göm statistik",
            "quizstats:clear solution": "Radera",
            "quizstats:show corrects": "Visa rätta svaren"
        }
    };

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Quizset.localization);
    }

    Quizset.templates = {
        solving: [
            '<div class="quizset-solvingarea">',
            '    <ul class="quizset-solving-buttonarea">',
            '    </ul>',
            '    <div class="quizset-solving-resultarea"></div>',
            '</div>'
        ].join('\n')
    }
    
    ElementSet.dialogs.quizset = [
        {
            name: 'showfirst',
            label: {
                en: 'Start directly',
                fi: 'Aloita suoraan',
                sv: 'Starta direkt'
            },
            attribute: 'showfirst',
            type: 'checkbox'
        },
        {
            name: 'feedback',
            label: {
                en: 'Instant feedback',
                fi: 'Välitön palaute',
                sv: 'Direkt respons'
            },
            attribute: 'feedback',
            type: 'checkbox'
        }
    ];
    
    Quizset.elementinfo = {
        type: 'quizset',
        elementtype: ['assignments'],
        boxtype: 'quizset',
        jquery: 'quizset',
        name: 'Quiz set',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-quizset"><path style="stroke: none;" d="M2 5 c2 -1 4 -1.5 6 0 c1.6 1.8 1.2 3 -1 5 c-1 1 -1 1 -1 2 h-2.5 c0 -1 0.2 -2 1.5 -3 c2.2 -2.2 1 -4 -3 -2z m1.5 8 h2.5 v2.5 h-2.5z"></path><path style="stroke: none;" d="M21 4 c2 -1 4 -1.5 6 0 c1.6 1.8 1.2 3 -1 5 c-1 1 -1 1 -1 2 h-2.5 c0 -1 0.2 -2 1.5 -3 c2.2 -2.2 1 -4 -3 -2z m1.5 8 h2.5 v2.5 h-2.5z"></path><g  transform="scale(1.5) translate(7 8)"><path style="stroke: none;" d="M0 0 c2 -1 4 -1.5 6 0 c1.6 1.8 1.2 3 -1 5 c-1 1 -1 1 -1 2 h-2.5 c0 -1 0.2 -2 1.5 -3 c2.2 -2.2 1 -4 -3 -2z m1.5 8 h2.5 v2.5 h-2.5z"></path></g></svg>',
        description: {
            en: 'A quiz with several small questions',
            fi: 'Quiz-tehtäviä',
            sv: 'Uppgift med flera korta frågor'
        },
        weight: 50,
        roles: ['teacher', 'author'],
        classes: ['assignments'],
        cancontain: ['containers', 'elements', 'quizelement']
    };
    

    $.fn.elementset('addelementtype', Quizset.elementinfo);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', Quizset.elementinfo);
    };
    
    if ($('head style#assignment-style').length === 0) {
        $('head').append('<style id="assignment-style" type="text/css">'+Assignment.styles+'</style>')
    }



    $.fn.quizset = function(method){
        if (quizsetmethods[method]) {
            return quizsetmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return quizsetmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in quizset.');
            return false;
        }
    };
    
    
    /*********************************************
     * A Class to handle and sort a set of multiple Quiz-solutions.
     *********************************************/
    var QuizSolutionSet = function(){
        this.solnames = [];
        this.soldata = {};
        this.current = '';
    };
    
    QuizSolutionSet.prototype.order = function() {
        var solution = this;
        var sorter = function(a, b){
            var atime = solution.soldata[a].time;
            var btime = solution.soldata[b].time;
            return (atime.getTime() < btime.getTime() ? -1 : 1);
        };
        this.solnames.sort(sorter);
    };
    
    QuizSolutionSet.prototype.addSolution = function(options){
        var solid = options.name;
        if (solid && this.solnames.indexOf(solid) === -1) {
            this.solnames.push(solid);
            this.soldata[solid] = new QuizSolution(options);
            this.order();
            this.current = this.solnames[this.solnames.length - 1];
        };
    };
    
    QuizSolutionSet.prototype.addSolutionData = function(data){
        if (!this.isClosed()) {
            this.soldata[this.current].setSolution(data.reftos, data.refcontentdata)
            //this.soldata[this.current].setSolution(data)
        }
    }
    
    QuizSolutionSet.prototype.getCurrent = function(){
        return this.soldata[this.current];
    };
    
    QuizSolutionSet.prototype.close = function(){
        this.soldata[this.current].close();
    };
    
    QuizSolutionSet.prototype.isClosed = function(){
        return (this.soldata[this.current] && this.soldata[this.current].isClosed() || this.getCount() === 0);
    };
    
    QuizSolutionSet.prototype.getDataCurrent = function(){
        return this.soldata[this.current].getData();
    };
    
    QuizSolutionSet.prototype.getCount = function(){
        return this.solnames.length;
    }
    
    QuizSolutionSet.prototype.getAnswerCount = function(){
        var curr = this.getCurrent();
        var count = curr.getCount();
        return count;
    }
    
    QuizSolutionSet.prototype.getSolutionData = function(reftos){
        var solset = this.getDataCurrent();
        var currsol = this.getCurrent();
        var soldata, solution, solname;
        var reply = {
            refs: {},
            refcontentdata: {},
            refinfo: {}
        };
        for (var i = 0, len = reftos.length; i < len; i++) {
            soldata = currsol.getSolution(reftos[i]);
            solname = soldata.name;
            solution = soldata.solution;
            reply.refs[reftos[i]] = [solname];
            reply.refcontentdata[solname] = solution;
        };
        return reply;
    }
    
    QuizSolutionSet.prototype.getStats = function(){
        var cursol = this.getCurrent();
        //var stats = {
        //    total: total,
        //    correct: 0
        //};
        //var solution, solname;
        //for (var sol in cursol.refs) {
        //    solname = cursol.refs[sol];
        //    solution = cursol.solutions[solname];
        //    if (solution.data.solved) {
        //        stats.correct++;
        //    };
        //};
        var stats = cursol.getScore();
        return stats;
    };
    
    
    /*********************************************
     * A Class to store Quiz-solutions. (key - value pairs with key as reference to quizquestion and value as solution)
     * Solution can be open or closed.
     *********************************************/
    var QuizSolution = function(options){
        options = $.extend(true, {}, QuizSolution.defaults, options);
        this.name = options.name;
        this.metadata = options.metadata;
        this.settings = options.settings;
        this.time = (this.metadata.modified ? new Date(this.metadata.modified) : new Date());
        this.closed = options.data.closed || false;
        this.refs = options.data.refs;
        this.solutions = options.data.solutions;
        this.count = 0;
        for (var ref in this.refs) {
            this.count++;
        }
    };
    
    QuizSolution.prototype.close = function(){
        this.closed = true;
    };
    
    QuizSolution.prototype.isClosed = function(){
        return this.closed;
    };
    
    QuizSolution.prototype.setSolution = function(reftos, soldata){
        if (typeof(reftos) === 'string') {
            reftos = [reftos];
        };
        var solname;
        if (!this.closed) {
            for (var i = 0, len = reftos.length; i < len; i++) {
                solname = soldata.name;
                if (!this.refs[reftos[i]]) {
                    this.count++;
                }
                this.refs[reftos[i]] = solname;
                this.solutions[solname] = soldata;
            };
            this.metadata.created = this.metadata.created || new Date();
            this.metadata.creator = this.metadata.creator || this.settings.username || '';
            this.metadata.modified = (new Date()).getTime();
            this.metadata.modifier = this.settings.username || '';
        };
    };
    
    QuizSolution.prototype.getSolution = function(refto){
        var solname = this.refs[refto];
        return {name: solname, solution: $.extend(true, {}, this.solutions[solname])};
    };
    
    QuizSolution.prototype.getData = function(){
        var reply = {
            name: this.name,
            type: 'quizsetsolution',
            metadata: $.extend(true, {}, this.metadata),
            data: {
                closed: this.closed,
                refs: $.extend(true, {}, this.refs),
                solutions: $.extend(true, {}, this.solutions)
            }
        };
        return reply;
    };
    
    QuizSolution.prototype.getCount = function(){
        return this.count;
    };
    
    QuizSolution.prototype.getScore = function(){
        var score = {
            correct: 0,
            total: 0
        };
        var solname, quizsol, resultlist;
        for (var ref in this.refs) {
            solname = this.refs[ref];
            quizsol = this.solutions[solname];
            resultlist = quizsol.data.results || [];
            for (var j = 0, jlen = resultlist.length; j < jlen; j++) {
                score.correct += resultlist[j];
            };
            score.total += (quizsol.data.total || resultlist.length || 1);
        };
        return score;
    };
    
    QuizSolution.defaults = {
        name: '',
        type: 'quizsetsolution',
        metadata: {},
        data: {
            closed: false,
            refs: {},
            solutions: {}
        }
    }
    
    
    
    /***************************************************
     * QuizStats
     ***************************************************/
    var QuizStats = function(data, total, uilang, userlist){
        this.data = $.extend(true, {}, QuizStats.defaults, data);
        this.total = total || 0;
        this.results = {};
        this.uilang = uilang || 'en';
        this.userlist = userlist;
        this.init();
    }
    
    QuizStats.prototype.init = function(){
        this.byUser = {};
        this.users = [];
        this.sort();
        this.setByUser();
    }
    
    QuizStats.prototype.sort = function(){
        var tool = this;
        var order = function(a, b){
            var datea = new Date(tool.data.soldata[a].metadata.modified);
            var dateb = new Date(tool.data.soldata[b].metadata.modified);
            return (datea.getTime() < dateb.getTime() ? -1 : 1);
        };
        this.data.solnames.sort(order);
    };
    
    QuizStats.prototype.setByUser = function(){
        var solnames = this.data.solnames;
        var soldata = this.data.soldata;
        var solname, solution;
        var username, result, scores;
        for (var i = 0, len = solnames.length; i < len; i++) {
            solname = solnames[i];
            solution = soldata[solname];
            username = solution.metadata.modifier;
            if (!this.byUser[username]) {
                this.byUser[username] = [];
            }
            this.byUser[username].push(solname);
            scores = this.findScore(solname);
            result = {
                total: scores.total,
                score: scores.score,
                date: this.findSolDate(solname)
            };
            this.results[solname] = result;
            if (this.users.indexOf(username) === -1) {
                this.users.push(username);
            };
        };
        this.users.sort();
    };
    
    QuizStats.prototype.getSolCount = function(username){
        var usersols = this.byUser[username]  || [];
        var solcount = 0;
        for (var i = 0, len = usersols.length; i < len; i++) {
            solcount = solcount + (this.data.soldata[usersols[i]].isClosed() ? 1 : 0);
        }
        return solcount;
    }
    
    QuizStats.prototype.findScore = function(solname){
        var solution = this.data.soldata[solname] || {};
        var result = {
            score: 0,
            total: 0
        };
        var questionsol, qname, sname;
        for (qname in solution.refs){
            sname = solution.refs[qname];
            questionsol = solution.solutions[sname];
            try {
                result.score += questionsol.data.results.reduce(function(a, b){return a+b;}, 0);
            } catch (err) {
                result.score += (questionsol.data.solved ? 1 : 0);
            };
            result.total += questionsol.data.total || questionsol.data.results && questionsol.data.results.length || 1;
        }
        return result;
    };
    
    QuizStats.prototype.findSolDate = function(solname){
        var solution = this.data.soldata[solname] || {};
        var date = new Date(solution.metadata.modified);
        return date;
    }
    
    QuizStats.prototype.getResult = function(username, index){
        var usersols = this.byUser[username]  || [];
        if (typeof(index) !== 'number') {
            index = usersols.length - 1;
        };
        var solname = usersols[index];
        return this.results[solname].score;
    };
    
    QuizStats.prototype.getBestResult = function(username){
        var usersols = this.byUser[username]  || [];
        var solname;
        var result = 0;
        for (var i = 0, len = usersols.length; i < len; i++) {
            solname = usersols[i];
            result = Math.max(result, this.results[solname].score);
        }
        return result;
    };
    
    QuizStats.prototype.getDate = function(username, index){
        var usersols = this.byUser[username]  || [];
        if (typeof(index) !== 'number') {
            index = usersols.length - 1;
        };
        var solname = usersols[index];
        var date = this.results[solname].date;
        //var solution = this.data.soldata[usersols[index]];
        //var date = new Date(solution.metadata.modified);
        return date;
    };
    
    QuizStats.prototype.html = function(){
        var html = ['<div class="quizset-stats-scoreboard"><table class="quizstatistics">'];
        var thead = [
            '<thead>',
            '<tr>',
            '<th>'+ebooklocalizer.localize('quizstats:username', this.uilang)+'</th>',
            '<th>'+ebooklocalizer.localize('quizstats:best try', this.uilang)+'</th>',
            '<th>'+ebooklocalizer.localize('quizstats:number of tries', this.uilang)+'</th>',
            '</tr>',
            '</thead>'
        ];
        var tbody = ['<tbody>'];
        var username;
        var usersols, solname, soldate, solscore, soltotal;
        var score, scoreclass, realname;
        for (var i = 0, len = this.users.length; i < len; i++) {
            username = this.users[i];
            realname = this.userlist && this.userlist.getRealnameLnameFirst && this.userlist.getRealnameLnameFirst(username) || username;
            score = this.getBestResult(username);
            scoreclass = (score === this.total ? ['quizstatistics-allcorrect'] : ['quizstatistics-somewrong']);
            tbody.push('<tr class="quizstatistics-userbest '+(i % 2 === 0 ? 'quizstatistics-evenrow' : 'quizstatistics-oddrow')+'" data-score="'+score+'">');
            tbody.push('<th class="quizstatistics-username">' + realname + '</th>');
            tbody.push('<td class="quizstatistics-bestscore"><span class="'+scoreclass.join(' ')+'">' + score + '/' + this.total + '</span></td>');
            //tbody.push('<td>' + (new Intl.DateTimeFormat(this.uilang, {weekday: 'short', year: 'numeric', month: 'numeric', day: 'numeric'}).format(this.getDate(username))) + '</td>');
            tbody.push('<td class="quizstatistics-numoftries">' + this.getSolCount(username) + '</td>');
            tbody.push('</tr>');
            usersols = this.byUser[username] || [];
            tbody.push('<tr class="quizstatistics-userallsols">', '<td colspan="3">','<ul class="quizstatistics-userallsol-list">');
            for (var j = 0, slen = usersols.length; j < slen; j++) {
                solname = usersols[j];
                if (!this.data.soldata[solname].isClosed()) {
                    continue;
                };
                soldate = this.results[solname].date;
                solscore = this.results[solname].score;
                soltotal = this.results[solname].total;
                tbody.push('<li data-solutionname="'+solname+'" data-score="'+solscore+'"><span class="quizstatistics-solvescore'+(solscore === soltotal ? ' quizstatistics-allcorrect' : '')+'">'+solscore+'/'+soltotal+'</span> <span class="quizstatistics-solvedate">'+(new Intl.DateTimeFormat(this.uilang, {weekday: 'long', year: 'numeric', month: 'numeric', day: 'numeric', hour: 'numeric', minute: 'numeric'}).format(soldate))+'</span></li>');
            }
            tbody.push('</ul>','</td>','</tr>');
        };
        tbody.push('</tbody>');
        html.push(thead.join('\n'), tbody.join('\n'), '</table></div>');
        return html.join('\n');
    }
    
    QuizStats.defaults = {
        soldata: {},
        solnames: []
    }


    /***************************************************
     * Homeassignmentset
     ***************************************************/
    
    /******
     * Homeassignmentset
     * @class HomeassSet
     * @constructor
     * @param {jQuery} place - Place for notebook
     * @param {Object} options - options for the notebook
     * @param {Object} solutions - an optional parameter with solutions
     ******/
    var HomeassSet = function(place, options, solutions){
        options = $.extend(true, {}, HomeassSet.defaults, options);
        ElementSet.call(this, place, options);
    };
    
    HomeassSet.prototype = Object.create(ElementSet.prototype);
    HomeassSet.prototype.constructor = HomeassSet;
    HomeassSet.prototype.parentClass = ElementSet.prototype;    

    /******
     * Set html attributes and classes
     ******/
    HomeassSet.prototype.setAttrs = function(){
        this.place.addClass('homeassignmentset');
        this.place.attr('data-elementmode', this.mode);
    };
    
    /******
     * Set the mode of element (view, edit, author, review,...)
     ******/
    HomeassSet.prototype.setMode = function(mode){
        if (!HomeassSet.modes[mode]) {
            mode = 'view';
        };
        if (!this.teacherable) {
            mode = 'view';
        }
        this.settings.mode = mode;
        this.editable = HomeassSet.modes[mode].editable || false;
    };
    
    /******
     * Set css-styles to the head, if needed.
     ******/
    HomeassSet.prototype.setStyles = function(){
        if ($('#homeassignmentset-style').length === 0) {
            $('head').append('<style type="text/css" id="homeassignmentset-style">' + HomeassSet.styles + '</style>');
        };
    };
    
    /******
     * Handlers for editable mode
     ******/
    HomeassSet.prototype.initHandlersEditableParent = HomeassSet.prototype.initHandlersEditable;
    HomeassSet.prototype.initHandlersEditable = function(){
        this.initHandlersEditableParent();
        var hassset = this;
        this.place.off('change', '.homeassignmentset-givendate input.date').on('change', '.homeassignmentset-givendate input.date', function(event, data){
            var field = $(this);
            var datetext = field.val();
            var date = (datetext ? new Date(datetext) : new Date());
            hassset.data.givenDate = date;
            hassset.changed();
        });
        this.place.off('change', '.homeassignmentset-duetodate input.date').on('change', '.homeassignmentset-duetodate input.date', function(event, data){
            var field = $(this);
            var datetext = field.val();
            var date = (datetext ? new Date(datetext) : new Date());
            hassset.data.duetoDate = date;
            hassset.changed();
        });
        this.place.off('change', 'input.homeassignmentset-title-input').on('change', 'input.homeassignmentset-title-input', function(event, data){
            var field = $(this);
            var text = field.val();
            // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
            hassset.data.title = text;
            hassset.changed();
        });
        this.place.off('change', '.homeassignmentset-text-input').on('change', '.homeassignmentset-text-input', function(event, data){
            var field = $(this);
            var text = field.val();
            // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
            hassset.data.text = text;
            hassset.changed();
        });
    }
    
    /******
     * Show the homeassignmentset in non-editable mode.
     ******/
    HomeassSet.prototype.view = function(){
        this.place.html(HomeassSet.templates.html);
        this.viewHeaders();
        this.viewAssignments();
        this.showIcons();
        //this.viewSolutions();
    };
    
    /******
     * Show homeassignmentset in editable mode.
     ******/
    HomeassSet.prototype.edit = function(){
        this.place.html(HomeassSet.templates.html);
        this.editHeaders();
        this.viewAssignments();
        //this.editSolutions();
    };
    
    /******
     * Show the headers of the homeassignmentset (non-editable).
     ******/
    HomeassSet.prototype.viewHeaders = function(){
        var uilang = this.settings.uilang;
        var headers = this.place.children('.homeassignmentset-header');
        var titlearea = headers.children('.homeassignmentset-title');
        var metaarea = headers.children('.homeassignmentset-metadata');
        var textarea = this.place.children('.homeassignmentset-text');
        titlearea.html(marked(this.data.title));
        titlearea.find('.mathquill-embedded-latex').mathquill();
        if (this.data.givenDate) {
            var gdate;
            try {
                gdate = new Date(this.data.givenDate);
            } catch (err) {
                gdate = new Date();
            };
            var givenFormatted = new Intl.DateTimeFormat(uilang, {weekday: 'short', year: 'numeric', month: 'numeric', day: 'numeric'}).format(gdate);
            metaarea.children('.homeassignmentset-givendate').html(ebooklocalizer.localize('homeassset:Given', uilang)+': ' + givenFormatted);
        };
        if (this.data.duetoDate) {
            var ddate;
            try {
                ddate = new Date(this.data.duetoDate);
            } catch (err) {
                ddate = new Date();
            };
            var duetoFormatted = new Intl.DateTimeFormat(uilang, {weekday: 'short', year: 'numeric', month: 'numeric', day: 'numeric'}).format(ddate);
            metaarea.children('.homeassignmentset-duetodate').html(ebooklocalizer.localize('homeassset:Return date', uilang)+': ' + duetoFormatted);
        }
        if (this.data.text) {
            textarea.html(marked(this.data.text));
            textarea.find('.mathquill-embedded-latex').mathquill();
        }
    };
    
    /******
     * Show the headers of the homeassignmentset (non-editable).
     ******/
    HomeassSet.prototype.editHeaders = function(){
        var uilang = this.settings.uilang;
        var headers = this.place.children('.homeassignmentset-header');
        var titlearea = headers.children('.homeassignmentset-title');
        var metaarea = headers.children('.homeassignmentset-metadata');
        var textarea = this.place.children('.homeassignmentset-text');
        titlearea.html('<input type="text" class="homeassignmentset-title-input" value="'+this.escapeHTML(this.data.title)+'" placeholder="'+ebooklocalizer.localize('homeassset:assignment_title', uilang)+'" />');
        metaarea.children('.homeassignmentset-givendate').html(ebooklocalizer.localize('homeassset:Given', uilang)+': <input type="date" class="date" value="' + this.escapeHTML(this.data.givenDate ? this.getDate(new Date(this.data.givenDate)) : '') + '" />');
        metaarea.children('.homeassignmentset-duetodate').html(ebooklocalizer.localize('homeassset:Return date', uilang)+': <input type="date" class="date" value="' + this.escapeHTML(this.data.duetoDate ? this.getDate(new Date(this.data.duetoDate)) : '') + '" />');
        textarea.html('<textarea class="homeassignmentset-text-input" placeholder="'+ebooklocalizer.localize('homeassset:hassettext', uilang)+'">' + this.escapeHTML(this.data.text || '') + '</textarea>');
        if (($('<input type="date">')[0].type === 'text') && $.fn.pikaday) {
            var uselang = (Pikaday.localization[uilang] ? uilang : 'en');
            var field = headers.find('input[type="date"].date');
            field.each(function(){$(this).pikaday({firstDay: 1, i18n: Pikaday.localization[uselang], container: $(this).parent().get(0), position: 'bottom', reposition: false})});
        }
    };
    
    /******
     * Show the assignments in this homeassignmentset (non-editable)
     ******/
    HomeassSet.prototype.viewAssignments = function(){
        var assignment, aid, adata, aelement;
        var assarea = this.place.find('.homeassignmentset-assset');
        var viewmode = (this.teacherable ? this.settings.mode : 'view');
        for (var i = 0, len = this.data.contents.length; i < len; i++){
            assignment = this.data.contents[i];
            aid = assignment.id;
            adata = $.extend(true, {}, this.data.contentdata[aid]);
            aelement = $(HomeassSet.templates.assignment);
            this.contentPlaces[aid] = aelement;
            assarea.append(aelement);
            assarea.append('<div class="elementset-droptarget" data-droptarget-cancontain="assignments" data-dropindex="'+(i+1)+'"></div>');
            aelement.attr('data-assignmentid', aid);
            adata.settings = {
                mode: viewmode,
                role: this.settings.role,
                uilang: this.settings.uilang,
                username: this.settings.username,
                users: this.settings.users,
                gotolink: this.settings.gotolink,
                usemargin: false
            };
            var asstextelement = aelement.find('.homeassignmentset-assignmenttext');
            asstextelement.attr('data-element-name', aid);
            var atype = adata.type;
            var elementinfo = ElementSet.availableElements.alltypes[atype];
            if (elementinfo) {
                var jq = elementinfo.jquery;
                try {
                    asstextelement[jq](adata);
                } catch (err){
                    console.log(err, atype);
                };
            };
        };
    };
    
    /******
     * Get the data of the homeassignmentset.
     * First get the data with ElementSet's getData method and then add homeassignmentset specific properties.
     ******/
    HomeassSet.prototype.getDataParent = HomeassSet.prototype.getData;
    HomeassSet.prototype.getData = function(){
        var dataset = this.getDataParent();
        dataset.data.text = this.data.text;
        dataset.data.duetoDate = this.data.duetoDate;
        dataset.data.givenDate = this.data.givenDate;
        return dataset;
    };
    
    /******
     * Get the array of elements that can be contained in this container
     * @returns {Array} list of containable elements
     ******/
    HomeassSet.prototype.cancontain = function(){
        return HomeassSet.elementinfo.cancontain.slice();
    }

    /******
     * Convert a Date-object ot a string in standard form (2015-05-15)
     * @param {Date} datetime - A date to convert
     * @returns {String} The date in YYYY-MM-DD -format.
     ******/    
    HomeassSet.prototype.getDate = function(datetime){
        var year = datetime.getFullYear();
        var month = ('0' + (datetime.getMonth()+1)).slice(-2);
        var date = ('0' + datetime.getDate()).slice(-2);
        return year + '-' + month + '-' + date;
    };
    
    /******
     * Properties of the modes.
     ******/
    HomeassSet.modes = {
        edit: {
            editable: true,
            reviewable: false
        },
        view: {
            editable: false,
            reviewable: false
        },
        review: {
            editable: false,
            reviewable: true
        }
    };
    
    /******
     * Default data template.
     ******/
    HomeassSet.defaults = {
        type: 'homeassignmentset',
        metadata: {},
        data: {
            title: '',
            text: '',
            givenDate: '',
            duetoDate: '',
            assignments: [],
            assignmentdata: {}
        },
        settings: {
            mode: 'view',
            uilang: 'en',
            role: 'student'
        }
    };

    /******
     * Icons
     ******/    
    HomeassSet.icons = {
        home: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path style="stroke: none;" d="M3 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        assignment: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-assignment"><text x="15" y="24" text-anchor="middle" style="font-weight: bold; font-size: 15px; font-family: sans;">???</text></svg>'
    };
    
    /******
     * Html-templates
     ******/
    HomeassSet.templates = {
        html: [
            '<div class="homeassignmentset-icon">'+HomeassSet.icons.home+'</div>',
            '<div class="homeassignmentset-header">',
            '    <div class="homeassignmentset-title"></div>',
            '    <div class="homeassignmentset-metadata">',
            '        <div class="homeassignmentset-givendate"></div>',
            '        <div class="homeassignmentset-duetodate"></div>',
            '    </div>',
            '</div>',
            '<div class="homeassignmentset-text"></div>',
            '<div class="homeassignmentset-assset">',
            '    <div class="elementset-droptarget" data-dropindex="0" data-droptarget-cancontain="assignments"></div>',
            '</div>'
        ].join('\n'),
        assignment: [
            '<div class="homeassignmentset-assignment elementset-elementwrapper">',
            '    <div class="elementset-elementheadbar elementset-elementcontrolbar ffwidget-background">',
            '        <div class="elementset-elementcontrols-left"><div class="elementset-elementhead-icon" draggable="true">'+HomeassSet.icons.assignment+'</div></div>',
            '        <div class="elementset-elementcontrols-right"><div class="elementset-controlicon elementset-controlicon-remove">'+ElementSet.icons.remove+'</div></div>',
            '    </div>',
            '    <div class="homeassignmentset-assignmenttext elementset-element" tabindex="-1"></div>',
            '    <div class="homeassignmentset-solution"></div>',
            '</div>'
        ].join('\n')
    };
    
    /******
     * CSS-stylesheets
     ******/
    HomeassSet.styles = [
        '.ebook-homeassignmentset {border: 1px solid #999; border-radius: 0.5em; margin: 0 0 1em 0;}',
        '[data-elementmode="edit"] .ebook-homeassignmentset {border-radius: 0 0 0.5em 0.5em;}',
        '.ebook-homeassignmentset .homeassignmentset-assignment {padding: 1em 0; border-top: 1px solid #666; border-bottom: 1px solid #666;}',
        '.ebook-homeassignmentset .homeassignmentset-assignment:hover {background-color: #fafafa;}',
        '.ebook-homeassignmentset .homeassignmentset-icon {float: right; margin: 5px;}',
        '.ebook-homeassignmentset .homeassignmentset-icon svg.mini-icon-home {width: 60px; height: 60px;}',
        '.ebook-homeassignmentset .homeassignmentset-title {font-size: 110%; font-weight: bold; padding: 0.2em 0.45em;}',
        '.ebook-homeassignmentset .homeassignmentset-metadata {padding: 0 0.5em;}',
        '.ebook-homeassignmentset .homeassignmentset-text {padding: 0.2em 0.5em; border: 1px solid #999; margin: 0.2em; box-shadow: inset 1px 1px 1px rgba(0,0,0,0.4), inset -1px -1px 1px rgba(255,255,255,0.5); background-color: #eee; clear: right;}',
        '.ebook-homeassignmentset .homeassignmentset-asset {padding: 0.5em;}',
        '.ebook-homeassignmentset .homeassignmentset-assignment > .elementset-elementheadbar {display: none;}',
        '.ebook-homeassignmentset[data-elementmode="edit"] .homeassignmentset-assignment > .elementset-elementheadbar, .ebook-homeassignmentset[data-elementmode="author"] .homeassignmentset-assignment > .elementset-elementheadbar {display: block; height: 20px; padding: 2px 5px; border-radius: 0.5em 0.5em 0 0; overflow: hidden;}',
        '.ebook-homeassignmentset[data-elementmode="edit"] .homeassignmentset-assignment, .ebook-homeassignmentset[data-elementmode="author"] .homeassignmentset-assignment {border: 1px solid #666; border-radius: 0.5em; padding: 0;}',
        '.ebook-homeassignmentset .homeassignmentset-header {margin-right: 100px;}',
        '.ebook-homeassignmentset .homeassignmentset-title input[type="text"] {display: block; width: 100%;}',
        '.ebook-homeassignmentset textarea.homeassignmentset-text-input {box-sizing: border-box; width: 100%; height: 5em; resize: vertical;}',
        
        // Pikaday
        '.pika-single.is-bound {position: absolute!important;}',
        ''
    ].join('\n');
    
    /******
     * Info about ContentTable (icon, description, etc.)
     ******/
    HomeassSet.elementinfo = {
        type: 'homeassignmentset',
        elementtype: ['containers','assignmentsets'],
        boxtype: 'homeassignmentset',
        jquery: 'homeassignmentset',
        name: 'Homeassignmentset',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path style="stroke: none;" d="M3 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        description: {
            en: 'An set of homeassignments',
            fi: 'Joukko kotitehtäviä',
            sv: 'En serie hemuppgifter'
        },
        roles: ['teacher', 'author'],
        classes: ['viewonly'],
        cancontain: ['assignments']
    };


    /******
     * Localization strings
     ******/
    HomeassSet.localization = {
        "en": {
            "homeassset:assignment_title": "Title of the assignment",
            "homeassset:Given": "Given",
            "homeassset:Return date": "Return date",
            "homeassset:hassettext": "Additional notes for homeassignments..."
        },
        "fi": {
            "homeassset:assignment_title": "Tehtävän otsikko",
            "homeassset:Given": "Annettu",
            "homeassset:Return date": "Palautus",
            "homeassset:hassettext": "Lisäohjeita tehtävien tekemiseen..."
        },
        "sv": {
            "homeassset:assignment_title": "Uppgiftens rubrik",
            "homeassset:Given": "Utdelad",
            "homeassset:Return date": "Deadline",
            "homeassset:hassettext": "Tilläggsinformation för hemuppgifterna..."
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(HomeassSet.localization);
    }
    
    // Register assignmentlist as an elementset and to the elementpanel.
    $.fn.elementset('addelementtype', HomeassSet.elementinfo);
    if ($.fn.elementpanel) {
        $.fn.elementpanel('addelementtype', HomeassSet.elementinfo);
    }

    /**** jQuery-plugin *****/
    var homeassmethods = {
        'init': function(params){
            return this.each(function(){
                var homeassset = new HomeassSet(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        }
    };
    
    $.fn.homeassignmentset = function(method){
        if (homeassmethods[method]) {
            return homeassmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return homeassmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in homeassignmentset.');
            return false;
        }
    };


    /**************************************************************************
     * Solutions and SolutionSets
     **************************************************************************/
    /******************************************************************************************************************
     * PageElement
     * @constructor
     * @param {jQuery} place - place for this PageElement
     * @param {Object} options - data for the page
     ******************************************************************************************************************/
    var SolutionElement = function(place, options){
        ElementSet.call(this, place, options);
        this.settings.usemargin = false;
    }
    
    // Inherit from ElementSet
    SolutionElement.prototype = Object.create(ElementSet.prototype);
    SolutionElement.prototype.constructor = SolutionElement;
    SolutionElement.prototype.parentClass = ElementSet.prototype;


    /******
     * Set mode as any as in any ElementSet, but if the solution has been sent, it's not editable.
     ******/
    SolutionElement.prototype.setModeParent = SolutionElement.prototype.setMode;
    SolutionElement.prototype.setMode = function(mode){
        if (mode === 'edit' && this.data.issent) {
            mode = 'view';
        };
        this.setModeParent(mode);
    };

    /******
     * Get list of types of elements this elementset can contain ('containers', 'elements',...)
     * @returns {Array} list of strings
     ******/
    SolutionElement.prototype.cancontain = function(){
        return ['elements'];
    };
    
    /******
     * Init handlers with added functionality
     ******/
    SolutionElement.prototype.initHandlersCommonParent = SolutionElement.prototype.initHandlersCommon;
    SolutionElement.prototype.initHandlersCommon = function(){
        this.initHandlersCommonParent();
        var solution = this;
        this.place.off('click', '.elementset-solutionelement-buttons button').on('click', '.elementset-solutionelement-buttons button', function(event){
            var button = $(this);
            var mode = button.attr('data-action');
            button.trigger('setmode', mode);
        });
        this.place.off('mouseenter', '.elementset-solutionelement-previewtool').on('mouseenter', '.elementset-solutionelement-previewtool', function(event){
            var bplace = $(this);
            if (!bplace.hasClass('solutionelement-preview-isopen')) {
                bplace.append('<div class="elementset-solutionelement-previewarea"></div>');
                var area = bplace.children('.elementset-solutionelement-previewarea');
                var data = solution.getData();
                data.settings = {mode: 'view', uilang: solution.settings.uilang, lang: solution.settings.lang, users: solution.settings.users};
                area.solutionelement(data);
                bplace.addClass('solutionelement-preview-isopen');
            };
        });
        this.place.off('mouseleave', '.elementset-solutionelement-previewtool').on('mouseleave', '.elementset-solutionelement-previewtool', function(event){
            var bplace = $(this);
            var area = bplace.children('.elementset-solutionelement-previewarea');
            area.remove();
            bplace.removeClass('solutionelement-preview-isopen');
        });
        this.place.off('showreview').on('showreview', function(event, reviewid, reviewdata){
            event.stopPropagation();
            solution.showReview(reviewid, reviewdata);
        });
        this.place.off('hidereview').on('hidereview', function(event, data){
            event.stopPropagation();
            solution.hideReview();
        });
        this.place.off('elementreview_changed').on('elementreview_changed', function(event, data){
            // Coming from children
            if (event.target !== this) {
                event.stopPropagation();
                solution.addToReview(data);
            };
        });
        
        if (this.reviewable) {
            this.place.off('focusout', '.solutionelement-review-general-text').on('focusout', '.solutionelement-review-general-text', function(event, data){
                event.stopPropagation();
                var value = $(this).val();
                // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
                solution.changeGeneralReview({
                    text: value
                });
            });
            this.place.off('change', '.solutionelement-review-general-grade').on('change', '.solutionelement-review-general-grade', function(event, data){
                event.stopPropagation();
                var value = $(this).val();
                // TODO: Should this be sanitized? How about texts like: "We have x<a and b>y"?
                solution.changeGeneralReview({
                    grade: value
                });
            });
        }
    };
    
    /******
     * Show elements and add buttons after that.
     ******/
    SolutionElement.prototype.showParent = SolutionElement.prototype.show;
    SolutionElement.prototype.show = function(){
        var uilang = this.settings.uilang;
        this.showParent();
        var mdate = (this.metadata.modified ? new Date(this.metadata.modified) : new Date());
        var creator = this.settings.users && this.settings.users.getRealname(this.metadata.creator || this.settings.username) || this.metadata.creator;
        var creatorid = this.metadata.creator;
        var classes = (this.settings.users && this.settings.users.canEdit({ctype: 'message'}) ? 'messagelink' : '');
        this.place.prepend('<div class="solutionelement-header"><span class="solutionelement-header-creator '+classes+'" data-creatorid="'+creatorid+'" data-toid="'+creatorid+'" data-totype="user">'+creator+'<span class="solutionelement-heaer-icon-sendmessage">'+ElementSet.icons.message+'</span></span>, <span class="solutionelement-header-createdate">'+(new Intl.DateTimeFormat(uilang, {weekday: 'short', year: 'numeric', month: 'numeric', day: 'numeric'}).format(mdate))+'</span></div>');
        if (!this.data.issent) {
            this.setPreview();
        };
        if (this.reviewable) {
            this.editGeneralReview();
        };
    };
    
    /******
     * Show review
     * @param {String} id - id if the review
     * @param {Object} data - data of review
     ******/
    SolutionElement.prototype.showReview = function(id, data){
        if (id) {
            this.currentReviewid = id;
        } else {
            id = this.currentReviewid;
        };
        if (data) {
            this.currentReviewdata = data;
        } else {
            data = this.currentReviewdata;
        };
        var elplace;
        if (id && data && data.data) {
            // If the id and data was given
            var rvdata;
            for (var rvname in data.data.reviewdata) {
                rvdata = data.data.reviewdata[rvname];
                elplace = this.place.find('[data-element-name="'+rvname+'"]');
                elplace.trigger('showselfreview', rvdata);
            };
            if (this.reviewable) {
                this.editGeneralReview(data.data.general);
            } else {
                this.viewGeneralReview(data.data.general);
            };
        } else {
            // New review is created
            elplace = this.place.find('[data-element-name].selfreviewable');
            elplace.trigger('showselfreview');
            if (this.reviewable) {
                this.editGeneralReview();
            } else {
                this.viewGeneralReview();
            };
            this.createReview();
            this.reviewChanged();
        };
        this.place.trigger('review_read', [id]);
    };
    
    /******
     * Hide review
     ******/
    SolutionElement.prototype.hideReview = function(){
        this.currentReviewid = '';
        this.currentReviewdata = null;
        this.place.find('.ebook-elementset-body [data-element-name]').trigger('hideselfreview');
        this.hideGeneralReview();
    };
    
    /******
     * View general review
     * @param {Object} greview - data of general review
     ******/
    SolutionElement.prototype.viewGeneralReview = function(greview) {
        var text = greview && greview.text || '';
        var grade = greview && greview.grade | 0;
        var footer = this.place.children('.ebook-elementset-footer');
        footer.find('.solutionelement-reviewarea').remove();
        var html = [
            '<div class="solutionelement-reviewarea" data-generalrevmode="view">',
            '<div class="solutionelement-review-general">',
            '<div class="solutionelement-review-general-text">',
            (marked && typeof(marked) === 'function' ? marked(text) : this.escapeHTML(text)),
            '</div>',
            '<div class="solutionelement-review-general-grade">'+grade+'/10</div>',
            '</div>',
            '</div>'
        ].join('');
        footer.append(html);
    }
    
    /******
     * Edit general review
     * @param {Object} greview - data of general review
     ******/
    SolutionElement.prototype.editGeneralReview = function(greview) {
        var text = greview && greview.text || '';
        var grade = greview && greview.grade | 0;
        var footer = this.place.children('.ebook-elementset-footer');
        footer.find('.solutionelement-reviewarea').remove();
        var html = [
            '<div class="solutionelement-reviewarea" data-generalrevmode="edit">',
            '    <div class="solutionelement-review-general">',
            '        <input class="solutionelement-review-general-grade" type="range" min="0" max="10" step="1" value="'+grade+'" oninput="this.nextElementSibling.value = this.value;">',
            '        <input class="solutionelement-review-general-grade" type="number" min="0" max="10" step="1" value="'+grade+'" oninput="this.previousElementSibling.value = this.value;">',
            '        <textarea class="solutionelement-review-general-text">',
                        this.escapeHTML(text),
                    '</textarea>',
            '    </div>',
            '</div>'
        ].join('');
        footer.append(html);
    }
    
    /******
     * Hide general review
     ******/
    SolutionElement.prototype.hideGeneralReview = function(greview) {
        this.place.find('> .ebook-elementset-footer .solutionelement-reviewarea').remove();
    };
    
    /******
     * Create new empty review
     ******/
    SolutionElement.prototype.createReview = function(){
        this.currentReviewid = 'reviewelement-' + this.settings.username + '-' + (new Date()).toJSON().replace(/[^0-9]/g, '') + '-' + Math.floor(Math.random()*1000);
        this.currentReviewdata = {
            name: this.currentReviewid,
            type: 'reviewelement',
            metadata: {
                creator: this.settings.username,
                created: (new Date()).getTime(),
                tags: [],
                lang: 'common',
                ref: this.name
            },
            data: {
                reviewdata: {},
                general: {
                    text: ''
                }
            }
        };
    };
    
    /******
     * Add new data to review
     * @param {Object} elreview - data of element's review {id: <id of sol element>, reviewdata: <data of review>}
     ******/
    SolutionElement.prototype.addToReview = function(elreview) {
        if (!this.currentReviewdata) {
            this.createReview();
        };
        this.updateReviewMeta();
        var data = this.currentReviewdata.data.reviewdata;
        data[elreview.id] = elreview.reviewdata;
        this.reviewChanged();
    };
    
    /******
     * Change general review
     * @param {Object} general - general review
     ******/
    SolutionElement.prototype.changeGeneralReview = function(general) {
        if (!this.currentReviewdata) {
            this.createReview();
        };
        this.updateReviewMeta();
        this.currentReviewdata.data.general = $.extend(true, {}, this.currentReviewdata.data.general, general);
        this.reviewChanged();
    };
    
    /******
     * Update reviewdata's meta
     ******/
    SolutionElement.prototype.updateReviewMeta = function(){
        this.currentReviewdata.metadata.modifier = this.settings.username;
        this.currentReviewdata.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Review changed
     ******/
    SolutionElement.prototype.reviewChanged = function(){
        var name = this.place.attr('data-element-name');
        var savedata = {
            "name": this.currentReviewid,
            "contentType": "review",
            "lang": "common",
            "anchors": [name],
            "contentdata": this.currentReviewdata,
            "contentinfo": {
                "name": this.currentReviewid,
                "contentType": "review",
                "lang": "common",
                // As own data until published to the "student"
                "pubtype": "own",
                // Back to the original creator
                "recipients": [{to: this.metadata.creator, pubtype: 'user'}],
                // "sendto" is empty until "send"-button is pressed
                "sendto": [],
                "sendfrom": this.settings.username,
                "timestamp": (new Date()).getTime(),
                "isoutbox": true,
                "read": false,
                "recipientopened": false
            }
        };
        this.place.trigger('elementreview_changed', savedata);
    };
    
    /******
     * Set button for changing mode (edit/preview)
     ******/
    SolutionElement.prototype.setPreview = function(){
        var header = this.place.children('.ebook-elementset-header');
        if (this.editable && !this.place.hasClass('.elementset-solutionelement-previewarea')) {
            header.html('<div class="elementset-solutionelement-previewtool"><div class="elementset-solutionelement-previewicon">'+SolutionElement.icons.preview+'</div></div>')
        };
    };
    
    /******
     * Set styles
     ******/
    SolutionElement.prototype.setStyles = function(){
        if ($('#solutionelement-style').length === 0) {
            $('head').append('<style type="text/css" id="solutionelement-style">' + SolutionElement.styles + '</style>');
        };
    };
    
    /******
     * Some icons
     ******/
    SolutionElement.icons = {
        preview: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-preview"><path style="stroke: none;" d="M3 15 a13 13 0 0 0 24 0 a13 13 0 0 0 -24 0z m1 0 a13 13 0 0 1 22 0 a13 13 0 0 1 -22 0z m6 -1 a5 5 0 0 0 10 0 a5 5 0 0 0 -10 0z m1.5 -1.6 a3 3 0 0 1 3 -2.1 a2 0.8 0 0 1 0 1 a3 3 0 0 0 -1.8 1.2 a0.5 0.5 0 0 1 -1.16 0z"></path><path class="nopreview" stroke="none" style="fill: none; stroke-width: 4;" d="M3 27 l24 -24"></path></svg>',
        edit: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>'
    };
    
    /******
     * Styles
     ******/
    SolutionElement.styles = [
        '.solutionelement-header {font: size: 90%; font-family: monospace; margin: 0.1em 0.3em; text-align: right;}',
        '.solutionelement-reviewarea {margin-top: 3em; border-top: 2px dashed #ccc;}',
        '.solutionelement-review-general {background-color: #f0f0f0; padding: 0; box-shadow: 0 0 8px rgba(0,0,0,0.5);}',
        '[data-generalrevmode="view"] .solutionelement-review-general {padding: 0; display: flex; flex-direction: row;}',
        '.solutionelement-review-general:empty {display: none;}',
        '[data-generalrevmode="edit"] .solutionelement-review-general {padding: 0.5em 0.3em;}',
        '[data-generalrevmode="edit"] .solutionelement-review-general-text {display: block; box-sizing: border-box; width: 100%; height: 8em; padding: 0.4em; overflow-y: scroll;}',
        '.solutionelement-review-general-grade[type="range"] {box-sizing: border-box; width: 90%; display: inline-block; vertical-align: middle;}',
        '.solutionelement-review-general-grade[type="number"] {width: 3em; border-radius: 0.8em; border: 1px solid #999; padding: 0.4em; vertical-align: middle; text-align: center;}',
        '[data-generalrevmode="view"] .solutionelement-review-general-grade {font-size: 200%; text-align: center; width: 3em; flex-grow: 0; flex-shrink: 0; background-color: #fafafa;}',
        '[data-generalrevmode="view"] .solutionelement-review-general-text {flex-grow: 1; padding: 0.2em 0.5em;}',
        '.messagelink {text-decoration: underline; color: blue; cursor: pointer;}',
        '.solutionelement-header-icon-sendmessage {display: none;}',
        '.messagelink .solutionelement-header-icon-sendmessage {display: inline-block; margin-left: 0.5em; vertical-align: text-bottom;}',
        '.messagelink .solutionelement-header-icon-sendmessage svg {height: 20px; width: auto;}'
    ].join('\n');
    


    /**** jQuery-plugin **********************/
    var solutionmethods = {
        'init': function(params){
            return this.each(function(){
                var solutionelem = new SolutionElement(this, params);
                //pageelem.init(params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'setdata': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        }
    }
    
    $.fn.solutionelement = function(method){
        if (solutionmethods[method]) {
            return solutionmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return solutionmethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in solutionelement.');
            return false;
        }
    }

    //SolutionElement.elementinfo = {
    //    type: 'solutionelement',
    //    elementtype: 'solutioncontainers',
    //    jquery: 'solutionelement',
    //    name: 'SolutionElement',
    //    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-elementbox"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 5 l0 18 l22 0 l0 -18z m3 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z m0 4 l16 0 l0 2 l-16 0z" /></svg>',
    //    description: {
    //        en: 'Solution of an assignment.',
    //        fi: 'Tehtävän ratkaisu.'
    //    },
    //    classes: ['solutioncontainer'],
    //    cancontain: ['elements']
    //}



    /******************************************************************************************
     * MessageElement
     ******************************************************************************************/
    /******
     * Message element
     * @constructor
     * @param {jQuery} place - place for the element
     * @param {Object} options - data for element
     ******/
    var MessageElement = function(place, options){
        this.place = $(place);
        options = $.extend(true, {}, MessageElement.defaults, options);
        this.init(options);
    };

    MessageElement.prototype = Object.create(ElementSet.prototype);
    MessageElement.prototype.constructor = MessageElement;
    MessageElement.prototype.parentClass = ElementSet.prototype;    
    
    /******
     * Init the element
     * @param {Object} options - data for the element
     ******/
    //MessageElement.prototype.init = function(options){
    //    this.metadata = options.metadata;
    //    this.data = options.data;
    //    this.settings = options.settings;
    //    this.setMode(this.settings.mode);
    //};
    
    /******
     * Set the css-styles
     ******/
    MessageElement.prototype.setStyles = function(){
        if ($('#messageelementstyles').length === 0) {
            $('head').append('<style type="text/css" id="messageelementstyles">'+MessageElement.style+'</style>');
        };
    }
    
    MessageElement.prototype.cancontain = function(){
        return MessageElement.elementinfo.cancontain.slice();
    }
    
    MessageElement.prototype.edit = function(){
        this.showHeader();
        this.showTitle();
        this.showContent();
        this.showFooter();
        this.showEditTools();
        this.initHandlersMessage();
    }
    
    MessageElement.prototype.showHeader = function(){
        var uilang = this.settings.uilang;
        var header = this.place.children('.ebook-elementset-header');
        header.addClass('virumcoursefeed-composer-header');
        header.html(MessageElement.templates.header);
        form = header.find('.virumcoursefeed-composer-headerform');
        var input, value, label, utype, utags;
        var headerform = JSON.parse(JSON.stringify(MessageElement.headerform[this.type]));
        //var people = this.settings.users.getUserList(['teachers', 'students']);
        var userlist = this.settings.users;
        var allgroups = userlist.getGroups();
        var people = userlist.getUserList(allgroups);
        //var contactlist = headerform.sendto.options;
        var contact;
        for (var i = 0, len = people.length; i < len; i++){
            // TODO: add some kind of label for teacher!
            contact = people[i];
            if (contact.getUsername() === this.settings.username) {
                // Don't add the user himself.
                continue;
            }
            //contactlist.push({
            //    name: contact.getUsername(),
            //    labelNT: contact.getRealnameLnameFirst(),
            //    value: contact.getUsername(),
            //    usertype: contact.getUsertype(),
            //    usertags: userlist.getUserTags(contact.getUsername())
            //});
        };
        //this.data.sendto = contactlist[0].value;
        var values = {
            title: this.data.title,
            //sendto: this.data.sendto,
            sendtolist: this.data.sendtolist
        }
        for (var iname in headerform) {
            input = headerform[iname];
            value = this.escapeHTML(values[input.name] || input.defaultvalue);
            switch (input.type) {
                case 'text':
                    ihtml = [
                        '<div class="virumcoursefeed-composer-header-inputblock">',
                        '    <label>'+ebooklocalizer.localize(input.label, uilang)+': <input class="virumcoursefeed-composer-input" type="'+input.type+'" value="'+value+'" data-name="'+input.name+'" placeholder="'+ebooklocalizer.localize(input.placeholder, uilang)+'"></label>',
                        '</div>'
                    ];
                    ihtml = ihtml.join('\n');
                    form.append(ihtml);
                    break;
                case 'date':
                    ihtml = [
                        '<div class="virumcoursefeed-composer-header-inputblock">',
                        '    <label>'+ebooklocalizer.localize(input.label, uilang)+': <input class="virumcoursefeed-composer-input date uninited" type="text" value="'+value+'" data-name="'+input.name+'" placeholder="'+ebooklocalizer.localize(input.placeholder, uilang)+'"><span class="virumcoursefeed-dateoutput">&nbsp;</span></label>',
                        '</div>'
                    ];
                    ihtml = ihtml.join('\n');
                    form.append(ihtml);
                    if ($.fn.pikaday) {
                        var uselang = (Pikaday.localization[uilang] ? uilang : 'en');
                        var field = form.find('input[type="text"].date.uninited');
                        field.removeClass('uninited');
                        field.each(function(){$(this).pikaday({firstDay: 1, i18n: Pikaday.localization[uselang], container: $(this).parent().get(0), position: 'bottom', reposition: false})});
                    }
                    break;


                case 'select':
                    ihtml = [
                        '<div class="virumcoursefeed-composer-header-inputblock">',
                        '    <label>'+ebooklocalizer.localize(input.label, uilang) +': ',
                        '        <select class="virumcoursefeed-composer-input" value="'+value+'" data-name="'+input.name+'">'];
                    for (var i = 0, len = input.options.length; i < len; i++) {
                        label = input.options[i].label;
                        label = (label ? ebooklocalizer.localize(input.options[i].label, uilang) : input.options[i].labelNT || ' ');
                        utype = input.options[i].usertype || '';
                        utags = input.options[i].usertags || [];
                        ihtml.push('<option value="'+input.options[i].value+'" data-usertype="'+utype+'" data-usertags="'+utags.join(' ')+'">'+label+'</option>');
                    }
                    ihtml.push('        </select>',
                        '    </label>',
                        '</div>');
                    ihtml = ihtml.join('\n');
                    form.append(ihtml);
                    break;
                case 'addressbook':
                    ihtml = [
                        '<div class="virumcoursefeed-composer-header-addressbook '+(this.settings.users.canPublish({ctype: this.getContentType(), toid: this.settings.users.contextid, totype: 'context'}) ? 'cansendtoall' : '')+'">',
                        '    <label class="virumcoursefeed-recipient-label">'+ebooklocalizer.localize(input.label, uilang) + ': ',
                        '    </label>',
                        '    <div class="virumcoursefeed-composer-header-addressbookblock"></div>',
                        '</div>'
                    ];
                    ihtml = $(ihtml.join('\n'));
                    form.append(ihtml);
                    var addressplace = ihtml.find('.virumcoursefeed-composer-header-addressbookblock');
                    var addressbook = new Addressbook(addressplace, userlist.getRecipients('message', true), uilang);
                    break;
                default:
                    ihtml = [''];
                    ihtml = ihtml.join('\n');
                    form.append(ihtml);
                    break;
            }
        };
        if (!this.data.title) {
            form.find('input[data-name="title"]').focus();
        };
    };
    
    MessageElement.prototype.showFooter = function(){
        var uilang = this.settings.uilang;
        var footer = this.place.children('.ebook-elementset-footer');
        footer.addClass('virumcoursefeed-composer-footer');
        footer.html(MessageElement.templates.footer);
        var bbars = {
            left: footer.find('.virumcoursefeed-composerbar-left'),
            right: footer.find('.virumcoursefeed-composerbar-right')
        };
        var button, bhtml;
        for (var bname in MessageElement.buttons) {
            button = MessageElement.buttons[bname];
            bhtml = [
                '<button class="virumcoursefeed-composerbar-button ffwidget-button button-'+button.name+'" data-event="'+button.event+'" title="'+ebooklocalizer.localize(button.label, uilang)+'">',
                button.icon,
                '</button>'
            ].join('');
            bbars[button.position].append(bhtml);
        };
    }
    
    /******
     * Init event handers special for messageelement (edit mode).
     ******/
    MessageElement.prototype.initHandlersMessage = function(){
        var message = this;
        this.place.on('click', 'button.virumcoursefeed-composerbar-button', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var ev = button.attr('data-event');
            button.trigger(ev);
        });
        this.place.on('coursefeedcancelmessage', function(event, data){
            message.clearAll();
        });
        this.place.on('coursefeedsendmessage', function(event, data){
            event.stopPropagation();
            message.sendMessage();
        });
        this.place.on('change', '.virumcoursefeed-composer-input', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var key = input.attr('data-name');
            // TODO: Should this be sanitized? What about: "Must be x<a and y>b"?
            var value = input.val();
            message.data[key] = value;
        });
        this.place.on('change', '.virumcoursefeed-composer-input[type="text"].date', function(event, data){
            var input = $(this);
            var value = input.val();
            var output = input.next('.virumcoursefeed-dateoutput');
            var uilang = message.settings.uilang;
            if (value !== '') {
                output.html((new Date(value)).toLocaleDateString(uilang));
            };
        });
        this.place.on('click', '.virumcoursefeed-recipient-label', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var item = $(this);
            item.next('.virumcoursefeed-composer-header-addressbookblock').toggleClass('blockopen');
        });
        this.place.on('click', '.addressbook-recipientlist', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var item = $(this);
            item.parent('.virumcoursefeed-composer-header-addressbookblock').addClass('blockopen');
        });
        this.place.on('addhomeassignmentdata', function(event, data) {
            event.stopPropagation();
            message.addHomeassignment(data);
        });
        this.place.on('addmessagedata', function(event, data) {
            event.stopPropagation();
            message.addMessageData(data);
        });
    };
    
    /******
     * Clear all from this message container.
     ******/
    MessageElement.prototype.clearAll = function(){
        this.place
            .removeClass('ebook-elementset ebook-messageelement')
            .removeAttr('data-tags')
            .removeAttr('data-creator')
            .removeAttr('data-created')
            .removeAttr('data-modifier')
            .removeAttr('data-modified')
            .removeAttr('data-elementmode')
            .removeAttr('data-elementrole')
            .empty();
    }
    
    /******
     * Get data (Extends the getData() of ElementSet.)
     * @returns {Object} the data of MessageElement.
     ******/
    MessageElement.prototype.getDataParent = MessageElement.prototype.getData;
    MessageElement.prototype.getData = function() {
        var result = this.getDataParent();
        result.data.title = this.data.title;
        try {
            this.data.sendto = JSON.parse(this.place.find('.addressbook-recipient-input').val());
        } catch (err) {
            this.data.sendto = [];
        };
        result.data.sendto = this.data.sendto;
        result.data.deadline = this.data.deadline;
        result.data.givendate = this.metadata.created;
        return result;
    }
    
    /**
     * Get the contentType
     * @returns {String} the contentType of this message (message|hamessage)
     */
    MessageElement.prototype.getContentType = function() {
        var ctype;
        switch (this.type) {
            case 'hamessage':
                ctype = 'hamessage';
                break;
            case 'messageelement':
            default:
                ctype = 'message';
                break;
        };
        return ctype;
    };
    
    /**
     * Add a homeassignment element to this message
     * @param {Object} hadata - data of the homeassignment
     */
    MessageElement.prototype.addHomeassignment = function(hadata) {
        var data = {
            type: 'homeassignmentelement',
            name: this.getNewId('homeassignmentelement'),
            data: {
                title: hadata.title,
                text: hadata.text,
                level: hadata.level | 0,
                gotolink: hadata.gotolink,
                colors: hadata.colors
            }
        };
        var index = this.contents.length;
        this.appendEditElement(data.type, data.name, index, data);
    }
    
    /**
     * Add an element to this message
     * @param {Object} eldata - data of element
     */
    MessageElement.prototype.addMessageData = function(eldata) {
        if (eldata.toid && eldata.totype) {
            // Add provided recipient to the message.
            var addressbook = this.place.find('.addressbook-selector .addressbook-list');
            var addritem = addressbook.find('[data-addritemtype="'+eldata.totype+'"][data-itemid="'+eldata.toid+'"] .addressbook-checkbox').click();
        };
        var data = eldata.contentdata;
        data.name = this.getNewId(data.type);
        var index = this.contents.length;
        this.appendEditElement(data.type, data.name, index, data);
    }
    
    /******
     * Send this message.
     ******/
    MessageElement.prototype.sendMessage = function(){
        var uilang = this.settings.uilang;
        var data = this.getData();
        var sendto;
        //var pubtype;
        //pubtype = (this.data.sendto === 'teacher' ? 'teacher' : (this.data.sendto === 'all' ? 'all' : 'user'));
        sendto = this.data.sendto;
        var contentType = this.getContentType();
        var msgid = this.getNewId(this.type);
        if (sendto.length > 0) {
            this.place.trigger('sendnewmessage', {name: msgid, sendfrom: this.settings.username, sendto: sendto, contentType: contentType, data: data});
        } else {
            if (this.settings.users.canPublish({ctype: contentType, toid: this.settings.users.contextid, totype: 'context'}) && confirm(ebooklocalizer.localize('messageelement:to_all_are_you_sure', uilang))) {
                sendto = this.data.sendto = [{to: this.settings.users.contextid, pubtype: 'context'}];
                this.place.trigger('sendnewmessage', {name: msgid, sendfrom: this.settings.username, sendto: sendto, contentType: contentType, data: data});
            } else {
                alert(ebooklocalizer.localize('messageelement:no_recipients', uilang))
            }
        }
    };
    
    /******
     * Default data for messageelement
     ******/
    MessageElement.defaults = {
        "type": "messageelement",
        "metadata": {
            "creator": "",
            "created": "",
            "modifier": "",
            "modified": "",
            "tags": []
        },
        "data": {
            "title": "",
            "contents": [],
            "contentdata": {},
            "sendto": ""
        },
        "settings": {
            "mode": "view",
            "uilang": "en",
            "role": "student"
        }
    };
    
    /******
     * Buttons
     ******/
    MessageElement.buttons = {
        cancel: {
            name: 'cancel',
            label: 'messageelement:cancel',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-cancel"><circle style="fill: white;" cx="15" cy="15" r="13" /><path style="stroke: none;" d="M15 2 a13 13 0 0 1 0 26 a13 13 0 0 1 0 -26z m0 10 l-4 -4 l-3 3 l4 4 l-4 4 l3 3 l4 -4 l4 4 l3 -3 l-4 -4 l4 -4 l-3 -3 l-4 4z" /></svg>',
            event: 'coursefeedcancelmessage',
            position: 'left'
        },
        sendmessage: {
            name: 'send',
            label: 'messageelement:sendmessage',
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-send"><path style="stroke: none;" d="M2 18 l24 -12 l-4 18 l-7 -3 l-4 4 l-1 0 l-1 -6z m8 0 l0.7 6 l2 -4.5 l12 -12z"></path></svg>',
            event: 'coursefeedsendmessage',
            position: 'right'
        }
    }
    
    /******
     * Headerform
     ******/
    MessageElement.headerform = {
        messageelement: {
            title: {
                name: 'title',
                label: 'messageelement:subject',
                type: 'text',
                defaultvalue: '',
                placeholder: 'messageelement:titleplaceholder'
            },
            sendtolist: {
                name: 'sendtolist',
                label: 'messageelement:sendto',
                type: 'addressbook',
                defaultvalue: []
            }
        },
        hamessage: {
            title: {
                name: 'title',
                label: 'messageelement:subject',
                type: 'text',
                defaultvalue: '',
                placeholder: 'messageelement:titleplaceholder'
            },
            deadline: {
                name: 'deadline',
                label: 'messageelement:deadline',
                type: 'date',
                defaultvalue: '',
                placeholder: 'messageelement:deadline'
            },
            sendtolist: {
                name: 'sendtolist',
                label: 'messageelement:sendto',
                type: 'addressbook',
                defaultvalue: []
            }
        }
    }

    /******
     * Localization strings
     ******/
    MessageElement.localization = {
        "en": {
            "messageelement:cancel": "Cancel",
            "messageelement:sendmessage": "Send",
            "messageelement:subject": "Subject",
            "messageelement:titleplaceholder": "Subject",
            "messageelement:teacher": "Teacher",
            "messageelement:all": "Everybody",
            "messageelement:sendto": "To",
            "messageelement:no_recipients": "No recipients selected!",
            "messageelement:to_all_are_you_sure": "You are sending this message to all.\nAre you sure?",
            "messageelement:deadline": "Deadline"
        },
        "fi": {
            "messageelement:cancel": "Peruuta",
            "messageelement:sendmessage": "Lähetä",
            "messageelement:subject": "Otsikko",
            "messageelement:titleplaceholder": "Otsikko",
            "messageelement:teacher": "Opettaja",
            "messageelement:all": "Kaikki",
            "messageelement:sendto": "Vastaanottaja",
            "messageelement:no_recipients": "Ei vastaanottajia valittuina!",
            "messageelement:to_all_are_you_sure": "Viesti lähetetään kaikille.\nOletko varma?",
            "messageelement:deadline": "Aikaraja"
        },
        "sv": {
            "messageelement:cancel": "Avbryt",
            "messageelement:sendmessage": "Skicka",
            "messageelement:subject": "Ärende",
            "messageelement:titleplaceholder": "Ärende",
            "messageelement:teacher": "Lärare",
            "messageelement:all": "Alla",
            "messageelement:sendto": "Mottagare",
            "messageelement:no_recipients": "Ingen mottagare är vald!",
            "messageelement:to_all_are_you_sure": "Är du säker?\nDetta meddelande kommer att skickas till alla deltagare.",
            "messageelement:deadline": "Deadline"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(MessageElement.localization);
    }
    
    /******
     * HTML-templates
     ******/
    MessageElement.templates = {
        header: [
            '<form class="virumcoursefeed-composer-headerform"></form>'
        ].join('\n'),
        footer: [
            '<div class="virumcoursefeed-composerbar"><div class="virumcoursefeed-composerbar-left"></div><div class="virumcoursefeed-composerbar-right"></div></div>'
        ].join('\n')
    }
    
    /******
     * CSS-rules for messageelements
     ******/
    MessageElement.style = [
        '.virumcoursefeed-composearea {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row wrap; -ms-flex-flow: column; flex-flow: column; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        '.virumcoursefeed-composearea > .ebook-elementset-body {background-color: white; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; margin: 0 5px; border: 1px solid #aaa; box-shadow: -1px -1px 1px rgba(0,0,0,0.3), 1px 1px 1px rgba(255,255,255,0.5); overflow-y: scroll;}',
        
        // Header (headerform)
        '.virumcoursefeed-composer-header {flex-grow: 0; flex-shrink: 0;}',
        '.virumcoursefeed-composer-headerform {margin: 0.2em 5px;}',
        '.virumcoursefeed-composer-headerform select option[data-usertype="teachers"] {color: red;}',
        '.virumcoursefeed-composer-headerform select option[data-usertags~="teachers"] {color: red;}',
        '.virumcoursefeed-recipient-label {cursor: pointer;}',
        '.virumcoursefeed-composer-header-addressbookblock .addressbook-selector {overflow: hidden; height: 0; transition: height 0.5s;}',
        '.virumcoursefeed-composer-header-addressbookblock.blockopen .addressbook-selector {overflow: auto; height: 7em;}',
        '.virumcoursefeed-composer-input[type="text"].date {display: none;}',
        '.virumcoursefeed-dateoutput {background-color: white; color: black; border: 1px solid #ccc; margin: 0.1em 0; display: inline-block; text-align: center; min-width: 8em; min-height: 1.5em; line-height: 1.5em;}',
        
        // Footer (buttonbar)
        '.virumcoursefeed-composer-footer {-webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; min-height: 25px;}',
        '.virumcoursefeed-composerbar {padding: 2px 5px; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row wrap; -ms-flex-flow: row wrap; flex-flow: row wrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        '.virumcoursefeed-composerbar button {padding: 0 5px;}',
        '.virumcoursefeed-composerbar button svg {height: 20px; width: auto; vertical-align: middle;}',
        '.virumcoursefeed-composerbar button.button-cancel svg path {fill: #a00;}',
        '.virumcoursefeed-composerbar button.button-send svg path {fill: #080;}'
        
    ].join('\n');
    
    /******
     * Elementinfo for messageelement
     ******/
    MessageElement.elementinfo = {
        type: 'messageelement',
        elementtype: ['feedelements'],
        jquery: 'messageelement',
        name: 'MessageElement',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /></svg>',
        description: {
            en: 'Message',
            fi: 'Viesti',
            sv: 'Meddelande'
        },
        roles: ['student', 'teacher', 'author'],
        cancontain: ['elements', 'containers', 'assignments', 'assignmentsets'],
        classes: ['message']
    };

    MessageElement.haelementinfo = {
        type: 'hamessage',
        elementtype: ['feedelements'],
        jquery: 'messageelement',
        name: 'HomeaAssignmentMessage',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path style="stroke: none;" d="M3 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>',
        description: {
            en: 'Home assignment message',
            fi: 'Kotitehtäväviesti',
            sv: 'Meddelanden om hemuppgifter'
        },
        roles: ['student', 'teacher', 'author'],
        cancontain: ['elements', 'containers', 'assignments', 'assignmentsets'],
        classes: ['message']
    };

    /******
     * Register the messageelement to the elementset and to the elementpanel
     ******/
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', MessageElement.elementinfo);
        $.fn.elementset('addelementtype', MessageElement.haelementinfo);
    }
    //if ($.fn.elementpanel) {
    //    $.fn.elementpanel('addelementtype', MessageElement.elementinfo);
    //}
    
    /**** jQuery-plugin *****/
    var messagemethods = {
        'init': function(params){
            return this.each(function(){
                var message = new MessageElement(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        }
    }
    
    $.fn.messageelement = function(method){
        if (messagemethods[method]) {
            return messagemethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return messagemethods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in messageelement.');
            return false;
        }
    }
    


    /**
     * Addressbook tool for selecting addresses.
     * @class Addressbook
     * @constructor
     * @param {jQuery} place   - jQuery object for the place
     * @param {Object} options - Initing options
     */
    var Addressbook = function(place, options, uilang) {
        this.place = place;
        this.uilang = uilang;
        this.recipients = options;
        this.groups = {};
        this.users = {};
        this.selected = {};
        this.setStyles();
        this.initHandlers();
        this.show();
    };
    
    Addressbook.prototype.show = function() {
        var item;
        var html = [
            '<input class="addressbook-recipient-input" style="display: none;" type="text" value="[]">',
            '<ul class="addressbook-recipientlist"></ul>',
            '<div class="addressbook-selector">'
        ];
        var htmllist = this.getHtmlList(this.recipients);
        html.push(htmllist);
        html.push('</div>');
        this.place.html(html.join(''));
        this.updateSelected();
    }
    
    Addressbook.prototype.getHtmlList = function(data, fakecheck) {
        var html = [];
        var item;
        html.push('<ul class="addressbook-list">');
        for (var i = 0, len = data.length; i < len; i++) {
            item = data[i];
            if (!fakecheck && (item.idtype === 'user' || item.togroup)) {
                html.push('<li class="addressbook-listitem addressbook-listitem-selectable" data-addritemtype="'+item.idtype+'" data-itemid="'+item.id+'">');
                html.push('<span class="addressbook-checkbox"></span>')
            } else {
                html.push('<li class="addressbook-listitem" data-addritemtype="'+item.idtype+'" data-itemid="'+item.id+'">');
                html.push('<span class="addressbook-fakecheckbox"></span>')
            }
            html.push('<span class="addressbook-name">' + item.description + '</span>');
            if (item.idtype === 'group' || item.idtype === 'context') {
                this.groups[item.id] = item;
                if (item.members) {
                    if (item.touser) {
                        html.push(' <span class="addressbook-groupcount">(<span class="addressbook-groupselected">0</span>/<span class="addressbook-grouptotal">' + item.members.length + '</span>)</span>');
                        html.push(this.getHtmlList(item.members, false));
                    } else {
                        html.push(this.getHtmlList(item.members, true));
                    };
                };
            } else {
                this.users[item.id] = item;
            };
            html.push('</li>');
        };
        html.push('</ul>')
        return html.join('');
    };
    
    Addressbook.prototype.addRecipient = function(data) {
        var item = (data.pubtype === 'group' || data.pubtype === 'context' ? this.groups[data.to] : this.users[data.to]);
        this.selected[data.to] = item;
        this.place.find('.addressbook-listitem[data-itemid="'+data.to+'"][data-addritemtype="'+data.pubtype+'"]:not([data-addritemstatus="marked"])').addClass('addressbook-disabled');
        var member, memid, memtype;
        if (item.members) {
            for (var i = 0, len = item.members.length; i < len; i++) {
                member = item.members[i];
                memid = member.id;
                memtype = member.idtype;
                delete this.selected[memid];
                this.place.find('.addressbook-listitem[data-itemid="'+memid+'"][data-addritemtype="'+memtype+'"]').addClass('addressbook-disabled');
            };
        }; //this.place.find('.addressbook-listitem[data-itemid="'+data.to+'"][data-addritemtype="'+data.pubtype+'"]').attr('data-addritemstatus','selected');
        this.updateSelected();
    };
    
    Addressbook.prototype.removeRecipient = function(data) {
        var item = this.selected[data.to];
        delete this.selected[data.to];
        this.place.find('[data-itemid="'+data.to+'"][data-addritemtype="'+data.pubtype+'"]').removeClass('addressbook-disabled');
        var member, memid, memtype;
        if (item && item.members) {
            for (var i = 0, len = item.members.length; i < len; i++) {
                member = item.members[i];
                memid = member.id;
                memtype = member.idtype;
                // Add to recipients all children that are marked somewhere else (but not children of other marked group)
                this.place.find('.addressbook-listitem[data-addritemstatus="marked"][data-itemid="'+memid+'"][data-addritemtype="'+memtype+'"]:not([data-addritemstatus="marked"] .addressbook-listitem)').trigger('addrecipient', {to: memid, pubtype: memtype});
                // Enable marking of children  in other groups, if not disabled for other reason.
                if (this.place.find('.addressbook-listitem.addressbook-disabled[data-itemid="'+memid+'"][data-addritemtype="'+memtype+'"][data-addritemstatus="marked"]').length === 0) {
                    this.place.find('.addressbook-listitem.addressbook-disabled[data-itemid="'+memid+'"][data-addritemtype="'+memtype+'"]:not([data-addritemstatus="marked"] .addressbook-listitem)').removeClass('addressbook-disabled');
                };
                this.place.find('.addressbook-listitem.addressbook-disabled[data-itemid="'+memid+'"][data-addritemtype="'+memtype+'"][data-addritemstatus="marked"]').removeClass('addressbook-disabled');
            };
        };
        this.updateSelected();
    };
    
    Addressbook.prototype.updateSelected = function() {
        var rlist = [], reclist = [], item, element, counter, marked, titlelist;
        for (var itemid in this.selected) {
            item = this.selected[itemid];
            titlelist = [];
            if (item.idtype === 'group' && item.members) {
                for (var i = 0, len = item.members.length; i < len; i++) {
                    titlelist.push(this.escapeHTML(item.members[i].description));
                };
            } else {
                titlelist.push(this.escapeHTML(item.description));
            };
            rlist.push('<li class="addressbook-recipientlist-item" title="'+titlelist.join('\n')+'" data-recipient="'+item.id+'" data-recipienttype="'+item.idtype+'"><span class="addressbook-recipientname">'+item.description+'</span> <span class="addressbook-recipient-remove">&times;</span></li>');
            reclist.push(item.send_to);
        };
        for (var gid in this.groups) {
            item = this.groups[gid];
            element = this.place.find('.addressbook-listitem[data-itemid="'+item.id+'"][data-addritemtype="'+item.idtype+'"]');
            marked = element.find('> .addressbook-list > [data-addritemstatus="marked"], > .addressbook-list > .addressbook-disabled');
            counter = element.find('> .addressbook-groupcount > .addressbook-groupselected');
            counter.text(marked.length);
            if (marked.length > 0) {
                counter.addClass('addressbook-hascount');
            } else {
                counter.removeClass('addressbook-hascount');
            };
        };
        if (rlist.length === 0) {
            rlist.push('<li class="addressbook-recipientlist-item addressbook-toeverybody"><span class="addressbook-recipientname">'+ebooklocalizer.localize('messageelement:all', this.uilang)+'</span></li>')
        }
        this.place.find('ul.addressbook-recipientlist').html(this.sanitize(rlist.join('\n')));
        this.place.find('input.addressbook-recipient-input').val(JSON.stringify(reclist));
    };
    
    Addressbook.prototype.initHandlers = function() {
        var abook = this;
        this.place.off('click', '.addressbook-listitem.addressbook-listitem-selectable').on('click', '.addressbook-listitem.addressbook-listitem-selectable', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var item = $(this);
            if (!item.is('[data-addritemstatus="marked"] .addressbook-listitem') && !item.is('.addressbook-disabled')) {
                var itemtype = item.attr('data-addritemtype');
                var itemid = item.attr('data-itemid');
                if (item.is('[data-addritemstatus="marked"]')) {
                    item.removeAttr('data-addritemstatus');
                    item.trigger('removerecipient', {to: itemid, pubtype: itemtype});
                } else {
                    item.attr('data-addritemstatus', 'marked');
                    item.trigger('addrecipient', {to: itemid, pubtype: itemtype});
                };
            };
        });
        this.place.off('addrecipient').on('addrecipient', function(event, data) {
            event.stopPropagation();
            abook.addRecipient(data);
        }).off('removerecipient').on('removerecipient', function(event, data) {
            event.stopPropagation();
            abook.removeRecipient(data);
        });
        this.place.off('click', '.addressbook-recipient-remove').on('click', '.addressbook-recipient-remove', function(event, data) {
            event.stopPropagation();
            var item = $(this).closest('.addressbook-recipientlist-item');
            var id = item.attr('data-recipient');
            var idtype = item.attr('data-recipienttype');
            abook.place.find('.addressbook-listitem[data-addritemstatus="marked"][data-itemid="'+id+'"][data-addritemtype="'+idtype+'"]').removeAttr('data-addritemstatus');
            item.trigger('removerecipient', {to: id, pubtype: idtype});
        });
    }
    
    Addressbook.prototype.setStyles = function() {
        if ($('head style#addressbookstyles').length === 0) {
            $('head').append('<style type="text/css">' + Addressbook.styles + '</style>');
        };
    };
    
    Addressbook.prototype.sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    Addressbook.prototype.escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    Addressbook.styles = [
        '.virumcoursefeed-composer-header-addressbookblock {position: relative;}',
        '.addressbook-selector {position: relative;}',
        '.addressbook-list {list-style: none; margin: 0; padding: 0;}',
        '.addressbook-selector > .addressbook-list {width: 33%; min-height: 7em;}',
        //'.addressbook-list > li:hover {background-color: rgba(255,255,255,0.5); cursor: pointer;}',
        '.addressbook-listitem:hover {background-color: rgba(255,255,255,0.5); cursor: pointer;}',
        //'.addressbook-list > li > .addressbook-list {display: none; overflow: auto;}',
        '.addressbook-listitem > .addressbook-list {display: none; overflow: auto;}',
        //'.addressbook-list > li:hover > .addressbook-list {display: block; position: absolute; top: 0; bottom: 0; left: 30%; right: 0; border: 1px solid #777;}',
        '.addressbook-listitem:hover > .addressbook-list {display: block; position: absolute; top: 0; bottom: 0; left: 33%; right: 0; border: 1px solid #aaa; padding: 0.2em; border-radius: 3px; box-shadow: inset 2px 2px 4px rgba(0,0,0,0.3), inset -2px -2px 4px rgba(255,255,255,0.5);}',
        '.addressbook-listitem {font-size: 90%;}',
        '.addressbook-listitem[data-addritemstatus="marked"] {background-color: rgba(136,221,255,0.5);}',
        '.addressbook-listitem[data-addritemstatus="marked"]:hover {background-color: rgba(136,221,255,0.8); cursor: pointer;}',
        '.addressbook-fakecheckbox {display: inline-block; width: 1em; height: 1em; vertical-align: middle; border: 1px solid transparent; margin-right: 0.3em;}',
        '.addressbook-checkbox {display: inline-block; width: 1em; height: 1em; vertical-align: middle; border: 1px solid #555; border-radius: 3px; margin-right: 0.3em; background-color: white;}',
        '.addressbook-listitem[data-addritemstatus="marked"] .addressbook-checkbox {background-color: #4bf;}',
        '.addressbook-listitem[data-addritemstatus="marked"] .addressbook-listitem, .addressbook-listitem.addressbook-disabled {background-color: transparent; color: #777;}',
        '.addressbook-listitem[data-addritemstatus="marked"] .addressbook-listitem .addressbook-checkbox, .addressbook-listitem.addressbook-disabled .addressbook-checkbox {background-color: #999;}',
        //'.addressbook-listitem[data-addritemstatus="marked"] .addressbook-groupcount {display: none;}',
        '.addressbook-groupselected.addressbook-hascount {color: #00f;}',
        // Addressbook recipientlist
        '.addressbook-recipientlist {list-style: none; margin: 0.1em 0; padding: 0; font-family: monospace; font-size: 65%; min-height: 1.4em; background-color: white; border: 1px solid #ccc; border-radius: 2px;}',
        '.addressbook-recipientlist-item {display: inline-block; margin: 0.1em; 0; padding: 0.1em; border: 1px solid #aaa; border-radius: 2px; background-color: #ccc; color: black;}',
        '.addressbook-recipientlist-item.addressbook-toeverybody {display: none;}',
        '.cansendtoall .addressbook-recipientlist-item.addressbook-toeverybody {display: inline-block;}',
        '.addressbook-recipientlist-item:hover {background-color: #ddd;}',
        '.addressbook-recipient-remove {display: inline-block; vertical-align: middle; text-align: center; width: 1em; height: 1em; border: 1px solid #bbb; border-radius: 50%; cursor: pointer; line-height: 1em; background-color: rgba(255,255,255,0.3);}',
        '.addressbook-recipient-remove:hover {background-color: rgba(255,255,255,0.5);}'
    ].join('\n');



    var normalizeLatex = function(latex) {
        var result = latex;
        result.replace(/\\NN/g, '\\mathbb{N}')
            .replace(/\\RR/g, '\\mathbb{R}')
            .replace(/\\Q/g, '\\mathbb{Q}')
            .replace(/\\ZZ/g, '\\mathbb{Z}')
            .replace(/\\P/g, '\\mathbb{P}');
        var count = 0, blck, i;
        var index = result.indexOf('\\Int{}');
        var blocks = [];
        while (index !== -1) {
            for (blck = 0; blck < 4; blck++) {
                count = 1;
                for (i = index + 1; count === 0 && bcount === 4 || i < result.length; i++) {
                }
            }
            index = result.indexOf('\\Int{}');
        }
        return result;
    }

})(window, jQuery, window.ebooklocalizer);