/******
@name ElementPalet
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class ElementPanel
@description Virtual class for set of book elements. Other classes will inherit this.
*******/

/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Dependency problem in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(ebooklocalizer) === 'undefined' && ebooklocalizer.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

(function(window, $, ebooklocalizer){
    
    /******
     * Constructor of an elementpanel
     * @constructor
     * @param {jQuery} place - place for the elementset
     * @param {Object} options - initing data
     ******/
    var ElementPanel = function(place, options){
        this.place = jQuery(place);
        this.place.addClass('elementpanel-wrapper');
        this.editcounter = 0;
        this.clipboard = {};
        this.init(options);
        if (!this.selfconfigable) {
            this.registerConfForm();
        };
    }
    
    ElementPanel.prototype.init = function(options){
        options = $.extend(true, {}, ElementPanel.defaults, options);
        this.docks = options.docks;
        this.settings = options.settings;
        this.config = options.config;
        this.confopen = false;
        this.selfconfigable = !! this.settings.selfconfigable;
        this.initPlace();
        this.setAttrs();
        this.show();
        this.setSize();
        this.addHandlers();
    }
    
    ElementPanel.prototype.setAttrs = function(){
        if (this.settings.background) {
            this.place.addClass('ffwidget-background');
        };
        this.place.attr('data-appname', 'elementpanel');
        this.setOrientation();
        this.setDocking();
        this.setAnimation();
        this.setPosition();
        this.minimize(this.config.minimized);
    };
    
    ElementPanel.prototype.setOrientation = function(orientation){
        if (typeof(orientation) === 'string') {
            this.config.orientation = orientation;
            if (this.docks[orientation]) {
                this.docks[orientation].append(this.place);
            };
        };
        this.place.attr('data-orientation', this.config.orientation);
    };
    
    ElementPanel.prototype.setDocking = function(docked){
        if (typeof(docked) === 'string' || typeof(docked) === 'boolean') {
            this.config.docked = !!docked;
        };
        this.place.attr('data-docked', this.config.docked);
        if (!this.config.docked) {
            this.setOrientation('vertical');
        }
        this.setSize();
    };
    
    ElementPanel.prototype.setAnimation = function(animation) {
        if (typeof(animation) === 'boolean') {
            this.config.animation = animation;
        };
        if (this.config.animation && (!this.config || this.config.docked === 'none')) {
            this.place.addClass('elementpanel-animation');
        } else {
            this.place.removeClass('elementpanel-animation');
        };
    };
    
    ElementPanel.prototype.setClassHiding = function(clsname, value){
        var index = this.config.hideclasses.indexOf(clsname);
        if (value) {
            if (index === -1) {
                this.config.hideclasses.push(clsname);
            };
        } else {
            if (index > -1) {
                this.config.hideclasses.splice(index, 1);
            };
        };
    };
    
    ElementPanel.prototype.setClassFolding = function(clsname, value){
        var index = this.config.foldclasses.indexOf(clsname);
        if (value) {
            if (index === -1) {
                this.config.foldclasses.push(clsname);
            };
        } else {
            if (index > -1) {
                this.config.foldclasses.splice(index, 1);
            };
        };
        this.configChanged();
    };
    
    ElementPanel.prototype.isClassFolded = function(clsname) {
        return (this.config.foldclasses.indexOf(clsname) !== -1);
    };
    
    ElementPanel.prototype.minimize = function(minimized){
        this.config.minimized = minimized;
        var panel = this;
        if (minimized) {
            this.place.addClass('elementpanel-minimizing');
            //var style = this.window.attr('style');
            //this.window.removeAttr('style');
            var currcoords = this.place.position();
            currcoords.right = 'auto';
            currcoords.bottom = 'auto';
            //this.window.attr('style', style);
            if (this.config.animation) {
                setTimeout(function(){
                    panel.window.css(currcoords);
                }, 0);
                setTimeout(function(){
                    panel.place.addClass('elementpanel-minimized');
                }, 500);
            } else {
                panel.window.css(currcoords);
                panel.place.addClass('elementpanel-minimized');
            };
        } else {
            this.place.removeClass('elementpanel-minimized');
            this.place.removeClass('elementpanel-minimizing');
            var currcoords = this.window.position();
            currcoords.right = 'auto';
            currcoords.bottom = 'auto';
            this.window.css(currcoords);
            if (this.config.animation) {
                setTimeout(function(){panel.setPosition(); panel.setSize();}, 0);
            } else {
                this.setPosition();
                this.setSize();
            };
        };
        this.configChanged();
    };
    
    ElementPanel.prototype.setPosition = function(coords){
        if (typeof(coords) === 'object') {
            this.config.position.x = coords.x;
            this.config.position.y = coords.y;
        };
        var width = this.window.width();
        var height = this.window.height();
        var wwidth = $(window).width();
        var wheight = $(window).height();
        var maxx = wwidth - width;
        var maxy = wheight - height;
        var positions = {
            top: Math.max(0, Math.min(this.config.position.y, maxy)),
            left: Math.max(0, Math.min(this.config.position.x, maxx)),
            right: 'auto',
            bottom: 'auto'
        };
        //switch (this.config.docked){
        //    case 'top':
        //        positions.top = 0;
        //        positions.bottom = 'auto';
        //        break;
        //    case 'bottom':
        //        positions.bottom = 0;
        //        positions.top = 'auto';
        //        break;
        //    case 'left':
        //        positions.left = 0;
        //        positions.right = 'auto';
        //        break;
        //    case 'right':
        //        positions.right = 0;
        //        positions.left = 'auto';
        //        break;
        //    case 'topleft':
        //        positions.top = positions.left = 0;
        //        positions.bottom = positions.right = 'auto';
        //        break;
        //    case 'topright':
        //        positions.top = positions.right = 0;
        //        positions.bottom = positions.left = 'auto';
        //        break;
        //    case 'bottomleft':
        //        positions.bottom = positions.left = 0;
        //        positions.top = positions.right = 'auto';
        //        break;
        //    case 'bottomright':
        //        positions.bottom = positions.right = 0;
        //        positions.top = positions.left = 'auto';
        //        break;
        //    default:
        //        break;
        //};
        this.window.css(positions);
    };
    
    ElementPanel.prototype.setSize = function() {
        if (!this.config.docked && this.config.orientation === 'vertical') {
            var height = this.docks.vertical.height() * 0.66;
            this.window.css('height', 'auto');
            var maxh = this.window.height();
            this.window.css('height', Math.min(Math.max(height, 500), maxh) + 'px');
        } else {
            this.window.css('height', '');
        };
    };
    
    ElementPanel.prototype.notHidden = function(clsname) {
        return this.config.hideclasses.indexOf(clsname) === -1;
    }
    
    ElementPanel.prototype.initPlace = function(){
        this.place.html(ElementPanel.templates.html);
        this.window = this.place.children('.elementpanel-window');
    }
    
    ElementPanel.prototype.configChanged = function(){
        this.place.trigger('saveconfigs', {type: 'configsave', appname: 'elementpanel', configs: this.getConfig()});
    }
    
    ElementPanel.prototype.getConfig = function(){
        var config = $.extend(true, {}, this.config);
        return config;
    }
    
    ElementPanel.prototype.setConfig = function(config) {
        this.config = $.extend({}, this.config, config);
        this.configChanged();
        this.show();
        this.setSize()
    }
    
    ElementPanel.prototype.getAllowedClasses = function() {
        var allclasses = ElementPanel.availableElements.classes;
        var allowed = allclasses;
        var result = {};
        if (this.settings.users) {
            allowed = this.settings.users.getElementClasses();
            for (var eclass in allclasses) {
                if (allowed.indexOf(eclass) !== -1) {
                    result[eclass] = allclasses[eclass];
                };
            };
        } else {
            result = allclasses;
        };
        return result;
    }
    
    ElementPanel.prototype.show = function(){
        var uilang = this.settings.uilang;
        var role = this.settings.role;
        var elbuttons = {};
        var temp = ElementPanel.templates.button;
        var text, str;
        var strlist = ['name', 'type', 'icon', 'description', 'weight'];
        var elclass, elset, eltype, elroles, elstring, weight, cdescription;
        var classes = this.getAllowedClasses();
        for (elclass in classes) {
            // Search buttons of each classes.
            if (this.notHidden(elclass)) {
                elset = classes[elclass];
                elbuttons[elclass] = [];
                for (var name in elset.elements) {
                    // Each element type in this class.
                    eltype = elset.elements[name]
                    elroles = eltype.roles || ['author'];
                    if (this.settings.users || elroles.indexOf(role) > -1) {
                        // If we are using accessrights from users-object or the element can be used in this role.
                        text = temp;
                        for (var j = 0; j < strlist.length; j++) {
                            // Replace placeholders in the template string.
                            str = strlist[j];
                            elstring = eltype[str];
                            elstring = (typeof elstring === 'object' ? elstring[uilang] || elstring['en'] : elstring);
                            text = text.replace(RegExp('{%'+str+'%}', 'g'), elstring);
                        };
                        elbuttons[elclass].push(text);
                    };
                };
            };
        };
        var elsets = this.window.find('.elementpanel-elementsets');
        var elclid;
        elsets.empty();
        for (elclass in elbuttons) {
            // Add each element class to the panel, if there are any buttons in this class.
            if (elbuttons[elclass].length > 0) {
                weight = classes[elclass].weight;
                elclid = elclass.replace(/[^a-zA-Z]/g, '_');
                cdescription = classes[elclass].description || elclass;
                elsets.append('<div class="elementpanel-elementclass-wrapper"><label class="elementpanel-elementclass-description" for="elementpanel-elementclass-'+elclid+'">' + cdescription + '</label><input id="elementpanel-elementclass-'+elclid+'" type="checkbox" class="elementpanel-elementclass-folded" data-classname="'+elclass+'" '+(this.isClassFolded(elclass) ? 'checked="checked"' : '')+'><div class="elementpanel-elementclass"   data-elementtypes="'+elclass+'" style="order: -'+weight+';" title="'+cdescription+'">'+elbuttons[elclass].join('')+'</div></div>');
            };
        };
    };
    
    //ElementPanel.prototype.showConfs = function(){
    //    var uilang = this.settings.uilang;
    //    var orient = this.config.orientation;
    //    var docked = this.config.docked;
    //    var animation = this.config.animation;
    //    var dialog = this.window.children('.elementpanel-dialog');
    //    var form = $(ElementPanel.templates.dialog);
    //    dialog.html(form);
    //    form.append('<legend>'+ebooklocalizer.localize('elementpanel:settings', uilang)+'</legend>');
    //    // Orientation
    //    form.append([
    //        '<div class="elementpanel-dialogbox">',
    //        '<p>'+ebooklocalizer.localize('elementpanel:orientation', uilang)+'</p>',
    //        '<div class="elementpanel-radiobuttons ffwidget-buttonset ffwidget-horizontal">',
    //        '<label class="elementpanel-orientation-label"><input type="radio" name="elementpanel-orientation-input" value="horizontal" '+(orient === 'horizontal' ? 'checked="checked"' : '')+'><span class="ffwidget-label ffwidget-setbutton elementpanel-dialog-label">'+ElementPanel.icons.horizontal+'</span></label>',
    //        '<label class="elementpanel-orientation-label"><input type="radio" name="elementpanel-orientation-input" value="vertical" '+(orient === 'vertical' ? 'checked="checked"' : '')+'><span class="ffwidget-label ffwidget-setbutton elementpanel-dialog-label">'+ElementPanel.icons.vertical+'</span></label>',
    //        '<label class="elementpanel-orientation-label"><input type="radio" name="elementpanel-orientation-input" value="box" '+(orient === 'box' ? 'checked="checked"' : '')+'><span class="ffwidget-label ffwidget-setbutton elementpanel-dialog-label">'+ElementPanel.icons.box+'</span></label>',
    //        '</div>',
    //        '</div>'
    //    ].join(''));
    //    // Docking
    //    form.append([
    //        '<div class="elementpanel-dialogbox">',
    //        '<p>'+ebooklocalizer.localize('elementpanel:docking', uilang)+'</p>',
    //        '<table class="ffwidget-buttongroup"><tbody>',
    //        // -------------------------------------------------
    //        '<tr>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="topleft" '+(docked === 'topleft' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.topleftdock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="top" '+(docked === 'top' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.topdock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="topright" '+(docked === 'topright' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.toprightdock+'</span></label></td>',
    //        '</tr>',
    //        // -------------------------------------------------
    //        '<tr>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="left" '+(docked === 'left' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.leftdock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="" '+(!docked ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.nonedock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="right" '+(docked === 'right' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.rightdock+'</span></label></td>',
    //        '</tr>',
    //        // -------------------------------------------------
    //        '<tr>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="bottomleft" '+(docked === 'bottomleft' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.bottomleftdock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="bottom" '+(docked === 'bottom' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.bottomdock+'</span></label></td>',
    //        '<td><label class="elementpanel-docking-label"><input type="radio" name="elementpanel-docking-input" value="bottomright" '+(docked === 'bottomright' ? 'checked="checked"' : '')+'><span class="ffwidget-label elementpanel-dialog-label">'+ElementPanel.icons.bottomrightdock+'</span></label></td>',
    //        '</tr>',
    //        // -------------------------------------------------
    //        '</tbody></table>',
    //        '</div>'
    //    ].join(''));
    //    // Animation
    //    form.append([
    //        '<div class="elementpanel-dialogbox">',
    //        '<p>' + ebooklocalizer.localize('elementpanel:animation', uilang) + '</p>',
    //        '<div class="elementpanel-radiobuttons ffwidget-buttonset ffwidget-horizontal">',
    //        '<label class="elementpanel-animation-label"><input type="radio" name="elementpanel-animation-input" value="on" '+(animation === true ? 'checked="checked"' : '')+'><span class="ffwidget-label ffwidget-setbutton elementpanel-dialog-label">on</span></label>',
    //        '<label class="elementpanel-animation-label"><input type="radio" name="elementpanel-animation-input" value="off" '+(animation === false ? 'checked="checked"' : '')+'><span class="ffwidget-label ffwidget-setbutton elementpanel-dialog-label">off</span></label>',
    //        '</div>',
    //        '</div>'
    //    ].join(''));
    //    // Hide classes
    //    var classlist = [
    //        '<div class="elementpanel-dialogbox">',
    //        '<p>' + ebooklocalizer.localize('elementpanel:elementclasses', uilang) + '</p>',
    //        '<ul class="elementpanel-classlist">'
    //    ];
    //    var classHidden;
    //    for (var cls in this.getAllowedClasses()) {
    //        classHidden = this.config.hideclasses.indexOf(cls) > -1;
    //        classlist.push('<li><label class="elementpanel-classhide-label"><input type="checkbox" name="elementpanel-classhide-input" data-classname="'+cls+'" '+(classHidden ? '' : 'checked="checked"')+'><span class="ffwidget-label">'+cls+'</span></label></li>')
    //    }
    //    classlist.push('</ul></div>');
    //    form.append(classlist.join(''));
    //    this.confopen = true;
    //    this.setPosition();
    //};
    //
    //ElementPanel.prototype.hideConfs = function(){
    //    this.window.children('.elementpanel-dialog').empty();
    //    this.confopen = false;
    //    this.setPosition();
    //}
    
    ElementPanel.prototype.startMoving = function(coords){
        var panel = this;
        this.moveOffset = coords;
        $('body').addClass('elementpanelmoving')
            .off('mousemove.elementpanel')
            .on('mousemove.elementpanel', function(event, data){
                var coords = {x: event.clientX, y: event.clientY};
                panel.move(coords);
            })
            .off('mouseup.elementpanel')
            .on('mouseup.elementpanel', function(event, data){
                var coords = {x: event.clientX, y: event.clientY};
                panel.stopMoving();
            });
        $(document).off('mouseleave.elementpanel').on('mouseleave.elementpanel', function(event, data) {
            panel.stopMoving();
        });
    }
    
    ElementPanel.prototype.stopMoving = function(){
        $('body').removeClass('elementpanelmoving')
            .off('mousemove.elementpanel')
            .off('mouseup.elementpanel');
        $(document).off('mouseleave.elementpanel');
        this.configChanged();
    }
    
    ElementPanel.prototype.move = function(coords){
        var newcoords = {
            x: coords.x - this.moveOffset.x,
            y: coords.y - this.moveOffset.y
        };
        var horizcoord = this.docks.horizontal.offset();
        var vertcoord = this.docks.vertical.offset();
        if (Math.abs(newcoords.y - horizcoord.top) < 40) {
            this.setDocking(true);
            this.setOrientation('horizontal');
        } else if (Math.abs(newcoords.x - vertcoord.left) < 40) {
            this.setDocking(true);
            this.setOrientation('vertical');
        } else {
            this.setDocking(false);
        };
        this.setPosition(newcoords);
    }
    
    /******
     * Start edit, i.e., increase editcounter with one and unminimize.
     ******/
    ElementPanel.prototype.startEdit = function(){
        this.editcounter++;
        this.minimize(false);
        this.setPosition();
    };
    
    /******
     * Stop edit, i.e., decrease editcounter and minimize, if editcouter === 0.
     ******/
    ElementPanel.prototype.stopEdit = function(){
        this.editcounter = Math.max(this.editcounter - 1, 0);
        if (this.editcounter === 0){
            this.minimize(true);
        };
    };
    
    ElementPanel.prototype.addHandlers = function(){
        var panel = this;
        this.place.on('click', '.elementpanel-elementicon', function(event, data){
            event.stopPropagation();
            var eltype = $(this).attr('data-elementtype');
            $(this).trigger('elementpanelevent', {eventname: 'paletaddelem', data: JSON.stringify({type: eltype})});
        });
        this.place.on('setconfigs', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            panel.setConfig(data);
        });
        this.place.on('dragstart', '.elementpanel-elementicon', function(ev){
            ev.stopPropagation();
            var event = ev.originalEvent;
            var eltype = $(this).attr('data-elementtype');
            var eltypes = $(this).parent().attr('data-elementtypes');
            var dragobj = {type: eltype};
            event.dataTransfer.setData('text', JSON.stringify(dragobj));
            event.dataTransfer.effectAllowed = 'copy';
            $('body').attr('data-nowdragging-element', eltype)
                .attr('data-nowdragging-elementtypes', eltypes);
            $('body').on('dragend.elementpanel', function(ev){
                $(this).off('dragend.elementpanel')
                    .removeAttr('data-nowdragging-element')
                    .removeAttr('data-nowdragging-elementtypes');
            });
        });
        this.place.on('dragstart', '.elementpanel-clipboard .elementpanel-dragicon', function(ev){
            ev.stopPropagation();
            var event = ev.originalEvent;
            event.dataTransfer.setData('text', JSON.stringify(panel.clipboard));
            event.dataTransfer.effectAllowed = 'copy';
            var eltype = panel.clipboardtype || '';
            var eltypes = panel.clipboardelset || '';
            $('body').attr('data-nowdragging-element', eltype)
                .attr('data-nowdragging-elementtypes', eltypes);
            $('body').on('dragend.elementpanel', function(ev){
                $(this).off('dragend.elementpanel')
                    .removeAttr('data-nowdragging-element')
                    .removeAttr('data-nowdragging-elementtypes');
            });
        });
        this.place.on('dragover', '.elementpanel-clipboard', function(event){
            event.preventDefault();
        });
        this.place.on('drop', '.elementpanel-clipboard', function(ev){
            ev.stopPropagation();
            ev.preventDefault();
            var event = ev.originalEvent;
            var datastr = event.dataTransfer.getData('text');
            var data = {};
            try {
                data = JSON.parse(datastr);
            } catch (err) {
                data = {type: 'error'};
            };
            if (data.type && data.type !== 'error') {
                panel.clipboard = data;
                var cid = data.data && data.data.dragelement || '';
                var element = data.data && data.data.dragdata && data.data.dragdata[cid];
                var eltype = element.type;
                panel.clipboardtype = eltype;
                var elicon = panel.place.find('.elementpanel-elementicon[data-elementtype="'+eltype+'"]');
                panel.clipboardelset = elicon.parent().attr('data-elementtypes');
                var icon = elicon.html();
                panel.place.find('.elementpanel-clipboard').html('<div class="elementpanel-dragicon" draggable="true">' + icon + '</div>');
            }
        });
        //this.place.on('click', '.elementpanel-controlbutton[data-control="configs"]', function(event, data){
        //    try {
        //    event.stopPropagation();
        //    if (panel.confopen) {
        //        panel.hideConfs();
        //    } else {
        //        panel.showConfs();
        //    };
        //    } catch (err){
        //        console.log(err);
        //    }
        //});
        this.place.on('mousedown', '.elementpanel-handle', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            panel.startMoving({x: event.offsetX, y: event.offsetY});
        });
        this.place.on('click', '.elementpanel-icon', function(event, data){
            event.stopPropagation();
            panel.minimize(!panel.config.minimized);
        });
        this.place.on('change', '.elementpanel-dialog input[name="elementpanel-orientation-input"]', function(event, data){
            var radio = $(this);
            var value = radio.val();
            panel.setOrientation(value);
            panel.setPosition();
            panel.configChanged();
        });
        this.place.on('change', '.elementpanel-dialog input[name="elementpanel-docking-input"]', function(event, data){
            var radio = $(this);
            var value = radio.val();
            panel.setDocking(value);
            panel.setAnimation();
            panel.setPosition();
            panel.configChanged();
        });
        this.place.on('change', '.elementpanel-dialog input[name="elementpanel-animation-input"]', function(event, data){
            var radio = $(this);
            var value = radio.val();
            panel.setAnimation(value === 'on');
            panel.configChanged();
        });
        this.place.on('change', '.elementpanel-dialog input[name="elementpanel-classhide-input"]', function(event, data){
            var check = $(this);
            var name = check.attr('data-classname');
            var value = !check.is(':checked');
            panel.setClassHiding(name, value);
            panel.configChanged();
            panel.show();
        });
        this.place.on('change', '.elementpanel-elementclass-folded', function(event, data){
            var check = $(this);
            var name = check.attr('data-classname');
            var value = check.is(':checked');
            panel.setClassFolding(name, value);
            panel.setSize();
        });
        $('body').off('startediting.elementpanel').on('startediting.elementpanel', function(event, data){
            panel.startEdit();
        });
        $('body').off('stopediting.elementpanel').on('stopediting.elementpanel', function(event, data){
            panel.stopEdit();
        });
        $(window).off('resize.elementpanel').on('resize.elementmanel', function(event, data){
            panel.setPosition();
            panel.setSize();
        });
    };
    
    ElementPanel.prototype.registerConfForm = function() {
        var formdata = {apptype: 'elementpanel', appid: 'elementpanel', confform: this.getConfigForm()};
        this.place.trigger('registerconfigdialog', formdata);
    }
    
    ElementPanel.prototype.getConfigForm = function() {
        var uilang = this.settings.uilang;
        var form = {
            name: 'elementpanel',
            title: ebooklocalizer.localize('elementpanel:elementpanel', uilang),
            icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-edit"><path style="stroke: none;" d="M3 27 l10 -4 l15 -15 a7 7 0 0 0 -6 -6 l-15 15z m5 -2.4 a3 3 0 0 0 -2.6 -2.6 l2 -5 a8 8 0 0 1 5.6 5.6 z"></path></svg>',
            weight: 10,
            form: [
                {
                    name: 'hideclasses',
                    label: ebooklocalizer.localize('elementpanel:elementclasses', uilang),
                    type: 'checkboxset',
                    negative: true,
                    options: [],
                    values: [],
                    attribute: 'hideclasses'
                }
            ]
        };
        var eclass;
        var options = form.form[0].options;
        var values = form.form[0].values;
        var allowed = this.getAllowedClasses()
        for (var cls in allowed) {
            elclass = {
                name: cls,
                label: cls,
                value: cls,
                weight: allowed[cls].weight
            };
            options.push(elclass);
            if (!this.notHidden(cls)) {
                values.push(cls);
            };
        };
        options.sort(function(a, b) {
            return b.weight - a.weight;
        });
        return form;
        
    }
    
    ElementPanel.availableElements = {
        classes: {
            general: {
                weight: 1000,
                elements: {}
            },
            container: {
                weight: 900,
                elements: {}
            },
            tabcontainer: {
                weight: 890,
                elements: {}
            },
            assignments: {
                weight: 800,
                elements: {}
            },
            math: {
                weight: 300,
                elements: {}
            },
            bookmarks: {
                weight: 200,
                elements: {}
            }
        }
    }
    
    ElementPanel.icons = {
        toolbox: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-toolbox"><path style="stroke: none;" d="M1 12 l0 -3 a3 3 0 0 1 3 -3 l6 0 l0 -1 a2 2 0 0 1 2 -2 l6 0 a2 2 0 0 1 2 2 l0 1 l6 0 a3 3 0 0 1 3 3 l0 3z m10.5 -6 l7 0 l0 -1.5 l-7 0z m-10.5 8 l9 0 l0 1 a2 2 0 0 0 2 2 l6 0 a2 2 0 0 0 2 -2 l0 -1 l9 0 l0 10 a3 3 0 0 1 -3 3 l-22 0 a3 3 0 0 1 -3 -3z m10.5 0 l7 0 l0 2 l-7 0z" /></svg>',
        clipboard: "<svg xmlns='http://www.w3.org/2000/svg' version='1.1' width='32' height='32' viewBox='0 0 30 30' class='mini-icon mini-icon-clipboard'><path style='stroke: none; fill: rgba(0,0,0,0.1);' d='M20 3 l5 0 a3 3 0 0 1 3 3 l0 20 a3 3 0 0 1 -3 3 l-20 0 a3 3 0 0 1 -3 -3 l0 -20 a3 3 0 0 1 3 -3 l5 0 l0 2 l-5 0 a1 1 0 0 0 -1 1 l0 20 a1 1 0 0 0 1 1 l20 0 a1 1 0 0 0 1 -1 l0 -20 a1 1 0 0 0 -1 -1 l-5 0z m-2 0 l0 1 a2 2 0 0 0 2 2 a2 2 0 0 1 2 2 l0 1 l-14 0 l0 -1 a2 2 0 0 1 2 -2 a2 2 0 0 0 2 -2 l0 -1 a3 3 0 0 1 6 0z' /></svg>",
        configs: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 32 32" class="mini-icon mini-icon-gear"><path style="stroke: none;" d="M12 0 h8 l0.9 3.5 l3.5 2 l3.55 -1.1 l4.1 7 l-2.7 2.5 v4.1 l2.7 2.5 l-4.1 7 l-3.55 -1.1 l-3.5 2 l-0.9 3.5 h-8 l-0.9 -3.5 l-3.5 -2 l-3.55 1.1 l-4.1 -7 l2.7 -2.5 v-4.1 l-2.7 -2.5 l4.1 -7 l3.55 1.1 l3.5 -2z m1.55 2 l-0.7 2.9 l-5 2.8 l-2.8 -0.8 l-2.5 4.3 l2.15 2 v5.65 l-2.15 2 l2.5 4.3 l2.8 -0.8 l5 2.8 l0.7 2.9 h4.9 l0.7 -2.9 l5 -2.8 l2.8 0.8 l2.5 -4.3 l-2.15 -2 v-5.65 l2.15 -2 l-2.5 -4.3 l-2.8 0.8 l-5 -2.8 l-0.7 -2.9z m2.45 7.55 a6.45 6.45 0 0 1 0 12.9 a6.45 6.45 0 0 1 0 -12.9z m0 2 a4.45 4.45 0 0 0 0 8.9 a4.45 4.45 0 0 0 0 -8.9z"></path></svg>',
        menu: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="15" height="15" viewBox="0 0 30 30" class="pssicon pssicon-menu"><path style="stroke: none;" d="M5 4 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z m0 9 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z m0 9 l20 0 a2.5 2.5 0 0 1 0 5 l-20 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        horizontal: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-horizontal"><path style="stroke: none;" d="M2 12 l26 0 l0 6 l-26 0z m2 1 l0 4 l2 0 l0 -4z m8 0 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z m5 0 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z m5 0 a2 2 0 0 0 0 4 a2 2 0 0 0 0 -4z" /></svg>',
        vertical: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-vertical"><path style="stroke: none;" d="M12 2 l0 26 l6 0 l0 -26z m1 2 l4 0 l0 2 l-4 0z m0 8 a2 2 0 0 1 4 0 a2 2 0 0 1 -4 0z m0 5 a2 2 0 0 1 4 0 a2 2 0 0 1 -4 0z m0 5 a2 2 0 0 1 4 0 a2 2 0 0 1 -4 0z m0 5z" /></svg>',
        box: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-box"><path style="stroke: none;" d="M5 2 l20 0 l0 28 l-20 0z m2 1 l0 3 l16 0 l0 -3z m0 10 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -10 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m6 -10 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z m0 5 a2 2 0 0 0 4 0 a2 2 0 0 0 -4 0z" /></svg>',
        topleftdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-topleftdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m14 2 l0 12 l-12 0 l0 12 l24 0 l0 -24z" /></svg>',
        toprightdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-toprightdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m14 2 l-12 0 l0 24 l24 0 l0 -12 l-12 0z" /></svg>',
        bottomleftdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-bottomleftdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 14 l12 0 l0 12 l12 0 l0 -24 l-24 0z" /></svg>',
        bottomrightdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-bottomrightdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 2 l0 24 l12 0 l0 -12 l12 0 l0 -12 l-24 0z" /></svg>',
        topdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-topdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m7 2 l-5 0 l0 24 l24 0 l0 -24 l-5 0 l0 8 l-14 0z" /></svg>',
        bottomdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-bottomdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 2 l0 24 l5 0 l0 -8 l14 0 l0 8 l5 0 l0 -24z" /></svg>',
        leftdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-leftdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 2 l0 5 l8 0 l0 14 l-8 0 l0 5 l24 0 l0 -24z" /></svg>',
        rightdock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-rightdock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 2 l0 24 l24 0 l0 -5 l-8 0 l0 -14 l8 0 l0 -5z" /></svg>',
        nonedock: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-nonedock"><path style="stroke: none;" d="M1 1 l28 0 l0 28 l-28 0z m2 2 l0 24 l24 0 l0 -24z m5 5 l14 0 l0 14 l-14 0z" /></svg>'
    }
    
    ElementPanel.templates = {
        html: [
            '<div class="elementpanel-window ffwidget-background">',
            '  <div class="elementpanel-panel">',
            '    <div class="elementpanel-handle"></div>',
            '    <div class="elementpanel-controls">',
            //'      <div class="elementpanel-controlbutton" data-control="configs">'+ElementPanel.icons.configs+'</div>',
            '    </div>',
            '    <div class="elementpanel-elementsets"></div>',
            '    <div class="elementpanel-clipboard"></div>',
            '  </div>',
            '  <div class="elementpanel-dialog"></div>',
            '</div>'
        ].join('\n'),
        button: '<div class="elementpanel-elementicon ffwidget-bordered" draggable="true" title="{%description%}" data-elementtype="{%type%}" style="order: -{%weight%};">{%icon%}<div class="elementpanel-element-description">{%description%}</div></div>',
        dialog: [
            '<fieldset>',
            '</fieldset>'
        ].join('\n')
    }
    
    ElementPanel.defaults = {
        config: {
            minimized: false,           // Show the panel as minimized (icon)
            orientation: 'horizontal', // vertical|horizontal|box
            docked: true,            // top|right|bottom|left|topleft|topright|bottomleft|bottomright|none
            position: {                // x- and y-coordinates, when floating
                x: 300,
                y: 300
            },
            animation: false,
            showconfig: true,
            foldclasses: [],
            hideclasses: []
        },
        settings: {
            //role: 'student',
            lang: 'en',
            uilang: 'en',
            background: false
        }
    }
    
    ElementPanel.addElementType = function(elemobj){
        elemobj = $.extend(true, {weight: 0}, elemobj);
        var etype, eclass;
        if (elemobj.type && elemobj.jquery && elemobj.name && elemobj.icon && elemobj.description && elemobj.classes) {
            for (var i = 0, len = elemobj.classes.length; i < len; i++) {
                eclass = elemobj.classes[i];
                if (!ElementPanel.availableElements.classes[eclass]) {
                    ElementPanel.availableElements.classes[eclass] = {weight: 0, elements: {}};
                };
                ElementPanel.availableElements.classes[eclass].elements[elemobj.type] = elemobj;
            };
        };
    };
    
    ElementPanel.styles = [
        // Wrapper
        '.elementpanel-wrapper {position: static; display: block; padding: 0; -webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none;}',
        '.elementpanel-wrapper.elementpanel-minimized {}',
        
        // Window
        '.elementpanel-wrapper .elementpanel-window {padding: 3px; border-radius: 3px; z-index: 500; border: 1px solid #888; opacity: 1;display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; min-height: 37px; max-width: 100%; max-height: 100%;}',
        '.elementpanel-wrapper[data-docked="false"] .elementpanel-window {position: fixed;}',
        '.elementpanel-wrapper.elementpanel-animation .elementpanel-window {transition: top 0.3s, left 0.3s, opacity 0.3s;}',
        '.elementpanel-wrapper.elementpanel-minimizing .elementpanel-window {opacity: 0;}',
        '.elementpanel-wrapper.elementpanel-minimized .elementpanel-window {display: none;}',
        '.elementpanelmoving .elementpanel-window {box-shadow: 5px 5px 20px rgba(0,0,0,0.5);}',
        '.elementpanelmoving .elementpanel-animation .elementpanel-window {transition: top 0s, left 0s;}',
        '.elementpanelmoving {-webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none;}',
        
        // Panel
        '.elementpanel-panel {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap;}',
        
        // Dialog for settings
        '.elementpanel-dialog {font-size: 80%;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-dialogbox {display: block;}',
        '.elementpanel-wrapper[data-orientation="box"] .elementpanel-dialogbox {display: block;}',
        '.elementpanel-dialogbox {display: inline-block; vertical-align: top;}',
        '.elementpanel-dialog .elementpanel-radiobuttons input[type="radio"] {display: none;}',
        '.elementpanel-dialog .elementpanel-radiobuttons label {padding: 0; margin: 0;}',
        '.elementpanel-dialog label span.ffwidget-label {padding: 4px; margin: 0;}',
        '.elementpanel-dialog  label svg {height: 20px; width: 20px;}',
        '.elementpanel-dialog .ffwidget-buttonset, .elementpanel-dialog .ffwidget-buttongroup {margin: 0.5em 1em;}',
        '.elementpanel-dialog ul {list-style: none; margin: 0; padding: 0;}',
        '.elementpanel-dialog .elementpanel-classhide-label .ffwidget-label {text-transform: capitalize;}',
        
        // Handle
        '.elementpanel-handle {background-color: rgba(0,0,0,0.4); order: -99999; min-height: 10px; min-width: 10px; cursor: move; -webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none; border-radius: 3px;}',
        '.elementpanel-wrapper.elementpanel-minimized .elementpanel-handle {display: none;}',
        
        // Toolbox icon
        '.elementpanel-icon {display: inline-block; margin: 0; padding: 3px; cursor: pointer; box-shadow: inset 2px 2px 2px rgba(0,0,0,0.5), inset -2px -2px 2px rgba(255,255,255,0.5);}',
        '.elementpanel-icon:hover {background-color: rgba(255,255,255,0.5);}',
        '.elementpanel-wrapper:not(.elementpanel.minimized) .elementpanel-icon svg {width: 10px; height: 10px;}',
        '.elementpanel-wrapper.elementpanel-minimized .elementpanel-icon {box-shadow: none;}',
        
        // Controlicons
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-controls {width: 15px; margin: 3px 0;}',
        '.elementpanel-controls .elementpanel-controlbutton {width: 15px; height: 15px; display: inline-block; margin: 0 3px; cursor: pointer;}',
        '.elementpanel-controls .elementpanel-controlbutton:hover {background-color: rgba(255,255,255,0.5);}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-controls .elementpanel-controlbutton {margin: 3px 0;}',
        '.elementpanel-controls .elementpanel-controlbutton svg {width: 15px; height: 15px;}',
        
        // Elementsets
        '.elementpanel-wrapper .elementpanel-elementsets {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; flex-shrink: 1;}',
        
        // Elementclass
        '.elementpanel-wrapper .elementpanel-elementclass-description {display: block; font-family: monospace; font-size: 50%; order: -9999; text-align: center; background-color: rgba(255,255,255,0.2); overflow: hidden; text-transform: capitalize; border-radius: 5px; cursor: pointer;}',
        '.elementpanel-wrapper .elementpanel-elementclass {margin: 0 2px; border: 1px solid #bbb; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row wrap; -ms-flex-flow: row wrap; flex-flow: row wrap; flex-shrink: 0;}',
        '.elementpanel-wrapper.elementpanel-minimized .elementpanel-elementclass {display: none;}',
        '.elementpanel-wrapper .elementpanel-elementclass.elementpanel-classhidden {display: none;}',
        '.elementpanel-wrapper .elementpanel-elementclass-folded {display: none;}',
        '.elementpanel-wrapper .elementpanel-elementclass-folded:checked + .elementpanel-elementclass {display: none;}',
        
        // Icons
        '.elementpanel-wrapper .elementpanel-elementicon {display: inline-block; margin: 2px; border-radius: 50%; width: 25px; height: 25px; padding: 4px; cursor: move; position: relative;}',
        '.elementpanel-wrapper .elementpanel-elementicon svg {width: 25px; height: 25px;}',
        '.elementpanel-wrapper .elementpanel-elementicon:hover {background-color: rgba(255,255,255,0.5);}',
        '.elementpanel-wrapper .elementpanel-elementicon .elementpanel-element-description {display: none;}',
        //'.elementpanel-wrapper .elementpanel-elementicon:hover .elementpanel-element-description {display: block; position: fixed; margin-top: -60px; margin-left: -20px; padding: 0.4em; font-size: 80%; color: black; background-color: rgba(255,255,170,0.9); z-index: 50; border-radius: 0.5em; border: 1px solid rgba(0,0,0,0.5); box-shadow: 8px 8px 8px rgba(0,0,0,0.3);}',

        // Horizontal
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-window {-webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-window .elementpanel-panel {-webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementsets {-webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; overflow-x: auto;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementclass-description {display: none;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementclass {-webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; border: none; border-right: 1px solid rgba(0,0,0,0.3); border-left: 2px solid rgba(255,255,255,0.5); padding-right: 5px; padding-left: 5px; margin-right: 0; margin-left: 0;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementclass:first-child {border-left: none;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementclass:last-child {border-right: none;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementsets::-webkit-scrollbar {height: 6px; background: transparent;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementsets:hover::-webkit-scrollbar {}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementsets::-webkit-scrollbar-thumb {background-color: rgba(170,170,170,0.5); border-radius: 2px;}',
        '.elementpanel-wrapper[data-orientation="horizontal"] .elementpanel-elementsets:hover::-webkit-scrollbar-thumb {background: rgba(170,170,170,0.8);}',
        
        // Vertical
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementsets {-webkit-flex-flow: row wrap; -ms-flex-flow: row wrap; flex-flow: column nowrap; overflow-y: auto; width: 60px;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementsets::-webkit-scrollbar {width: 2px; background: transparent;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementsets:hover::-webkit-scrollbar {width: 2px;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementsets::-webkit-scrollbar-thumb {background-color: rgba(170,170,170,0.5);}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementsets:hover::-webkit-scrollbar-thumb {background: rgba(170,170,170,0.8); border-radius: 2px;}',
        '.elementpanel-wrapper[data-docked="false"][data-orientation="vertical"] .elementpanel-elementsets {width: 100px;}',
        '.elementpanel-wrapper[data-docked="true"][data-orientation="vertical"] {height: 100%}',
        '.elementpanel-wrapper[data-docked="true"][data-orientation="vertical"] .elementpanel-window {height: 100%}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementclass-description {width: 100%;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementclass-wrapper {border: none; border-bottom: 1px solid #aaa; border-top: 1px solid #eee; padding-bottom: 5px; padding-top: 5px; margin-bottom: 0; margin-top: 0;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementclass {-webkit-flex-flow: row wrap; -ms-flex-flow: row wrap; flex-flow: row wrap; justify-content: space-between; border: none;}',
        '.elementpanel-wrapper[data-docked="true"][data-orientation="vertical"] .elementpanel-elementclass {justify-content: space-around;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementclass-wrapper:first-child {border-top: none;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-elementclass-wrapper:last-child {border-bottom: none;}',
        
        // Box
        '.elementpanel-wrapper[data-orientation="box"] .elementpanel-elementsets {-webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; min-width: 200px;}',
        '.elementpanel-wrapper[data-orientation="box"] .elementpanel-elementclass {flex-flow: row wrap; border: none; border-bottom: 1px solid #bbb; margin: 3px 0;}',
        
        // Clipboard
        '.elementpanel-clipboard {background: rgba(255,255,255,0.5) url("data:image/svg+xml;charcode=utf-8,'+ElementPanel.icons.clipboard+'") center center no-repeat; order: 99999; min-height: 35px; min-width: 30px; padding: 0; order: 50; margin: 0 2px; border-radius: 3px;}',
        '.elementpanel-clipboard .elementpanel-dragicon {display: inline-block;}',
        '.elementpanel-wrapper[data-orientation="vertical"] .elementpanel-clipboard .elementpanel-dragicon {display: block; text-align: center;}',
        '.elementpanel-clipboard svg {width: 30px; height: 30px;}',
        '.elementpanel-wrapper.elementpanel-minimized .elementpanel-clipboard {display: none;}',
        '.elementpanel-wrapper[data-docked="true"][data-orientation="vertical"] .elementpanel-clipboard {margin-bottom: 10px;}',
        '.elementpanel-wrapper .elementpanel-clipboard .elementpanel-element-description {display: none;}',
        
    ].join('\n');

    if ($('head style#elementpanel-style').length === 0) {
        $('head').append('<style id="elementpanel-style" type="text/css">'+ElementPanel.styles+'</style>')
    }

    
    /******
     * Localization strings
     ******/
    ElementPanel.localization = {
        "en": {
            'elementpanel:elementpanel': 'Element panel',
            'elementpanel:settings': 'Settings',
            'elementpanel:orientation': 'Orientation',
            'elementpanel:docking': 'Docking',
            'elementpanel:animation': 'Animation',
            'elementpanel:elementclasses': 'Hide classes'
        },
        "fi": {
            'elementpanel:elementpanel': 'Elementtipaneeli',
            'elementpanel:settings': 'Asetukset',
            'elementpanel:orientation': 'Asento',
            'elementpanel:docking': 'Telakointi',
            'elementpanel:animation': 'Animointi',
            'elementpanel:elementclasses': 'Piilota luokat'
        },
        "sv": {
            'elementpanel:elementpanel': 'Verktygsbalken',
            "elementpanel:settings": "Inställningar",
            "elementpanel:orientation": "Riktning",
            "elementpanel:docking": "Fastsättning",
            "elementpanel:animation": "Animation",
            'elementpanel:elementclasses': 'Göm klasser'
        }
    };
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(ElementPanel.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(ElementPanel.localization);
    };



    
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var elpanel = new ElementPanel(this, params);
            });
        },
        'addelementtype': function(data){
            ElementPanel.addElementType(data);
        }
    }

    $.fn.elementpanel = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in elementpanel.');
            return false;
        }
    }


})(window, jQuery, window.ebooklocalizer);