/******
@name VirumCourse
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Dashboard
@description A class and jQuery-plugin for a course dashboard.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /**
     * Course dashboard
     * @constructor
     * @param {jQuery} place    DOM-place for course dashboard
     * @param {Object} options  data for course dashboard
     */
    var Dashboard = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.initData();
        this.setAttrs();
        this.setStyles();
        this.show();
    }
    
    Dashboard.prototype.init = function(options) {
        this.settings = options.settings;
        this.users = this.settings.users;
        this.appid = this.settings.appid;
        this.apptype = this.settings.apptype;
    };
    
    Dashboard.prototype.initData = function() {
        this.solsByCreator = {};
        this.hasets = [];
        this.place.trigger('getcontent', {anchors: [this.settings.contextid], contentType: ['hamessage', 'solution', 'review']});
    }
    
    Dashboard.prototype.initHandlers = function() {
        var dashboard = this;
        this.place.on('reply_getcontent', function(event, data) {
            event.stopPropagation();
            dashboard.setContentdata(data);
        });
        this.place.on('refreshapp', function(event, data) {
            event.stopPropagation();
            dashboard.initData();
            dashboard.show();
        });
        this.place.on('click', '.coursedashboard-controlbuttons button', function(event, data) {
            event.stopPropagation();
            var action = $(this).attr('data-action');
            dashboard.place.trigger(action);
        });
        this.place.on('click', 'th.coursedashboard-scoreboard-haset, td.coursedashboard-scoreboard-hatotal, td.coursedashboard-scoreboard-usertotal', function(event, data) {
            event.stopPropagation();
            var hasetnum = $(this).attr('data-hasetnum') | 0;
            dashboard.toggleHasetColumn(hasetnum);
        });
        this.place.on('click', 'td.coursedashboard-scoreboard-assdatacell, td.coursedashboard-scoreboardone-assdatacell', function(event, data) {
            event.stopPropagation();
            var cell = $(this);
            var username = cell.attr('data-username') || '';
            var hasetnum = cell.attr('data-hasetnum') | 0;
            var hassnum = cell.attr('data-hassnumber') | 0;
            var solcount = cell.attr('data-solcount') | 0;
            dashboard.gotoAss({username: username, haset: hasetnum, assnumber: hassnum, solcount: solcount});
        });
        this.place.on('click', 'td.coursedashboard-scoreboard-assignment', function(event, data) {
            event.stopPropagation();
            var cell = $(this);
            var hasetnum = cell.attr('data-hasetnum') | 0;
            var hassnum = cell.attr('data-hassnumber') | 0;
            var solcount = cell.attr('data-solcount') | 0;
            var datacells = dashboard.scoreboard.find('td.coursedashboard-scoreboard-assdatacell[data-hasetnum="'+hasetnum+'"][data-hassnumber="'+hassnum+'"]:not([data-solcount="0"])');
            var users = datacells.map(function(){return $(this).attr('data-username')}).get();
            dashboard.gotoAss({username: '', haset: hasetnum, assnumber: hassnum, solcount: users.length});
        });
        this.place.on('click', 'th.coursedashboard-scoreboard-username', function(event, data) {
            event.stopPropagation();
            var username = $(this).attr('data-username') || '';
            if (dashboard.scoreboardOne && dashboard.scoreboardOne.attr('data-username') === username) {
                dashboard.scoreboardOne.empty().attr('data-username', '');
            } else {
                dashboard.scoreboardOne.attr('data-username', username);
                dashboard.showOne(username);
            };
        });
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            dashboard.close();
        });
    };
    
    Dashboard.prototype.setAttrs = function() {
        this.place.addClass('coursedashboard');
    };
    
    Dashboard.prototype.show = function() {
        if (this.settings.users.canView({ctype: 'solution'})) {
            this.place.html(this.template.all);
            this.scoreboard = this.place.find('.coursedashboard-scoreboard');
            this.scoreboardOne = this.place.find('.coursedashboard-scoreboard-one');
            this.showAll();
        } else {
            this.place.html(this.template.one);
            this.scoreboardOne = this.place.find('.coursedashboard-scoreboard-one');
            this.showOne();
        };
        this.setButtonTitles();
    };
    
    Dashboard.prototype.showAll = function() {
        var uilang = this.settings.uilang;
        var allgroups = this.users.getGroups();
        var users = this.users.getUserList(allgroups);
        // Start the table
        var table = [
            '<h1 class="coursedashboard-title">' + ebooklocalizer.localize('dashboard:homeassignments', uilang) + '</h1>',
            '<table><thead class="ffwidget-background">',
        ];
        // Table head
        // Homeassignmentsets
        table.push('<tr>');
        table.push('<th rowspan="2">' + ebooklocalizer.localize('dashboard:name', uilang) + '</th>');
        var hasetlist = [];
        var haset;
        for (var i = 0, len = this.hasets.length; i < len; i++) {
            haset = this.hasets[i];
            hasetlist.push('<th class="coursedashboard-scoreboard-haset" colspan="'+(haset.size()+1)+'" data-hasetnum="'+i+'"><div class="coursedashboard-scoreboard-hasetnumber">'+(i+1)+'</div><div class="coursedashboard-scoreboard-hasetdeadline" title="'+(haset.getDeadlineLocaleLong(uilang))+'">'+haset.getDeadlineLocale(uilang)+'</div><div class="coursedashboard-scoreboard-hasettitle">'+escapeHTML(haset.getTitle())+'</div></th>');
        };
        table.push(hasetlist.join('\n'));
        table.push('<th class="ffwidget-background" rowspan="2">' + ebooklocalizer.localize('dashboard:name', uilang) + '</th>');
        table.push('</tr>');
        // Homeassignments
        var halist = [];
        var assignment, hatitle, hatext;
        table.push('<tr>');
        for (var i = 0, len = this.hasets.length; i < len; i++) {
            haset = this.hasets[i];
            for (var j = 0, jlen = haset.size(); j < jlen; j++) {
                assignment = haset.assignments[j];
                hatitle = assignment.hatitle || assignment.asstitle || '';
                hatext = (assignment.hatext || assignment.asstext || '').substr(0, 50);
                halist.push('<td class="coursedashboard-scoreboard-assignment" data-hacolors="'+escapeHTML(haset.getColors(j).join(' '))+'" data-hasetnum="'+i+'" data-hassnumber="'+j+'" title="'+escapeHTML(hatitle + '\n' + hatext)+'"><div class="coursedashboard-scoreboard-assnumber">'+(j+1)+'</div></td>');
            };
            halist.push('<td class="coursedashboard-scoreboard-hatotal ffwidget-background-inverse" data-hasetnum="'+i+'">'+haset.size()+'</td>');
        }
        table.push(halist.join('\n'));
        table.push('</tr></thead><tbody>');
        // Table body
        var username, assid, sols, counter, percentage, allrevs, currevlist, currevid, currev, currevscore, bestrev, bestrevtext;
        var userhaset, userhaindex, uhaset;
        for (var i = 0, len = users.length; i < len; i++) {
            table.push('<tr>');
            username = users[i].getUsername();
            userhaset = this.getFilteredHasets(username);
            userhaindex = 0;
            table.push('<th class="coursedashboard-scoreboard-username" data-username="'+escapeHTML(username)+'">' + escapeHTML(users[i].getRealnameLnameFirst()) + '</th>');
            for (var j = 0, jlen = this.hasets.length; j < jlen; j++) {
                haset = this.hasets[j];
                uhaset = userhaset[userhaindex];
                if (uhaset && haset.getId() === uhaset.getId()) {
                    // Show only, if this assignmentset is sent to the username.
                    counter = 0;
                    for (var k = 0, klen = haset.size(); k < klen; k++) {
                        allrevs = true;
                        assid = haset.getAssignment(k);
                        sols = this.solsByCreator[username] && this.solsByCreator[username][assid];
                        counter += (sols && sols.length > 0 ? 1 : 0);
                        bestrev = -1;
                        bestrevtext = '';
                        for (var l = 0, rlen = (sols && sols.length || 0); l < rlen; l++) {
                            currevlist = this.reviews[sols[l].getId()] || [];
                            for (var m = 0, crlen = currevlist.length; m < crlen; m++) {
                                currevid = currevlist[m];
                                currev = currevid && this.reviewdata[currevid];
                                currevscore = (currev ? currev.data.general.grade : -1) | 0; // Making sure, the value is Number.
                                if (currevscore > bestrev) {
                                    bestrev = currevscore;
                                    bestrevtext = currev && currev.data.general.text || '';
                                };
                            };
                            allrevs = allrevs && currevlist.length > 0;
                        };
                        table.push('<td class="coursedashboard-scoreboard-assdatacell" data-assignmentid="'+escapeHTML(assid)+'" data-username="'+escapeHTML(username)+'" data-hasetnum="'+j+'" data-hassnumber="'+k+'" data-solcount="'+(sols ? sols.length : '0')+'" data-allreviewed="'+escapeHTML(allrevs)+'"><div class="coursedashboard-scoreboard-assdatacontent"><div class="coursedashboard-scoreboard-solcount">'+(sols ? sols.length : '')+'</div><span class="coursedashboard-scoreboard-bestreview" title="'+escapeHTML(bestrevtext)+'">'+(bestrev >= 0 ? bestrev : '-')+'</span></div></td>');
                    };
                    percentage = 100 * counter / (Math.max(klen, 1));
                    table.push('<td class="coursedashboard-scoreboard-usertotal" data-hasetnum="'+j+'" data-username="'+username+'"><div class="coursedashboard-hasetdata" data-doneasses="'+(counter || '0')+'"><div class="coursedashboard-hasetpercentage" style="width: '+percentage+'%"></div><div class="coursedashboard-hasetdata-contenttext">'+(counter ? counter + '/' + klen : '-')+'</div></div></td>');
                    userhaindex++;
                } else {
                    // Show as empty, if this assignmentset wasn't sent to this username.
                    table.push('<td colspan="'+(haset.size() + 1)+'" class="coursedashboard-scoreboard-empty"></td>')
                };
            }
            table.push('<th>' + escapeHTML(users[i].getRealnameLnameFirst()) + '</th>');
            table.push('</tr>');
        }
        table.push('</tbody>');
        table.push('<thead class="ffwidget-background"><tr>');
        // Lower table head
        table.push('<th rowspan="2">' + ebooklocalizer.localize('dashboard:name', uilang) + '</th>');
        table.push(halist.join('\n'));
        table.push('<th rowspan="2">' + ebooklocalizer.localize('dashboard:name', uilang) + '</th>');
        table.push('</tr>','<tr>');
        table.push(hasetlist.join('\n'));
        table.push('</tr>');
        table.push('</thead></table>')
        this.scoreboard.html(table.join('\n'));
    };
    
    Dashboard.prototype.showOne = function(username) {
        username = username || this.settings.username;
        var uilang = this.settings.uilang;
        var maxcount = 0;
        this.scoreboardOne.html('Sorry, '+ escapeHTML(this.users.getRealname(username)) + ', no can do!');
        var table = [
            '<table><thead class="ffwidget-background">',
            '<tr><th class="coursedashboard-scoreboardone-userhead" colspan="2">' + escapeHTML(this.users.getRealnameLnameFirst(username)) + '</th></tr>',
            '<tr class="coursedashboard-scoreboardone-headrow"><th>' + ebooklocalizer.localize('dashboard:assignmentgroup', uilang) + '</th>',
            //'<th class="coursedashboard-scoreboardone-assignmenthead" colspan="1"></th>',
            '</thead>',
            '<tbody>'
        ];
        var userhasets = this.getFilteredHasets(username);
        var haset, sols, assid, assignment, hatitle, hatext, counter, percentage, size, allrevs, lastreved, currevlist, currevid, currev, currevscore, bestrev, bestrevtext;
        for (var i = 0, len = userhasets.length; i < len; i++) {
            maxcount = Math.max(maxcount, userhasets[i].size());
        };
        var userhaindex = 0;
        var uhaset;
        for (var i = 0, len = this.hasets.length; i < len; i++) {
            uhaset = userhasets[userhaindex];
            haset = this.hasets[i];
            if (uhaset && haset.getId() === uhaset.getId()) {
                table.push(
                    '<tr>',
                    '<td class="coursedashboard-scoreboardone-assgroup">',
                    '<div class="coursedashboard-scoreboardone-assgroupitem">',
                    '<span class="coursedashboard-scoreboardone-assgroupno">' + (i+1) + '</span>',
                    '<span class="coursedashboard-scoreboardone-assgrouptitle">' + escapeHTML(haset.getTitle()) + '</span>',
                    '</div>',
                    '<div class="coursedashboard-scoreboardone-assgroupdl">' + haset.getDeadlineLocale(uilang) + '</div>');
                counter = 0;
                size = haset.size();
                for (var j = 0; j < size; j++) {
                    allrevs = true;
                    lastreved = false;
                    assid = haset.getAssignment(j);
                    assignment = haset.assignments[j];
                    sols = this.solsByCreator[username] && this.solsByCreator[username][assid];
                    counter += (sols && sols.length > 0 ? 1 : 0);
                    bestrev = -1;
                    bestrevtext = '';
                    for (var k = 0, slen = (sols && sols.length); k < slen; k++) {
                        currevlist = this.reviews[sols[k].getId()] || [];
                        for (var m = 0, crlen = currevlist.length; m < crlen; m++) {
                            currevid = currevlist[m];
                            currev = currevid && this.reviewdata[currevid];
                            currevscore = (currev ? currev.data.general.grade : -1) | 0; // Making sure, the value is Number.
                            if (currevscore > bestrev) {
                                bestrev = currevscore;
                                bestrevtext = currev && currev.data.general.text || '';
                            }
                        };
                        allrevs = allrevs && currevlist.length > 0;
                        lastreved = currevlist.length > 0;
                    };
                    hatitle = assignment.hatitle || assignment.asstitle || '';
                    hatext = (assignment.hatext || assignment.asstext || '').substr(0, 50);
                    table.push(
                        '<td class="coursedashboard-scoreboardone-assdatacell" title="'+escapeHTML(hatitle + '\n' + hatext)+'" data-assignmentid="' + escapeHTML(assid) + '" data-username="' + escapeHTML(username) + '" data-hasetnum="' + i + '" data-hassnumber="' + j + '" data-solcount="' + (sols ? sols.length : '0') + '" data-allreviewed="'+escapeHTML(allrevs)+'" data-lastreviewed="'+escapeHTML(lastreved)+'">',
                        '<div class="coursedashboard-scoreboardone-assnum" data-hacolors="' + escapeHTML(haset.getColors(j).join(' ')) + '">' + (j+1) + '</div>',
                        '<div class="coursedashboard-scoreboardone-solcount"><div class="coursedashboard-scoreboard-solcount">' + (sols ? sols.length : '') + '</div><span class="coursedashboard-scoreboard-bestreview" title="' + escapeHTML(bestrevtext) + '">'+(bestrev >= 0 ? bestrev : '-')+'</span></div>',
                        '</td>'
                    );
                };
                for (var j = 0; j < maxcount - size; j++) {
                    table.push('<td class="coursedashboard-scoreboardone-empty"></td>');
                };
                percentage = 100 * counter / (Math.max(size, 1));
                table.push('<td class="coursedashboard-scoreboardone-usertotal" data-hasetnum="'+j+'" data-username="'+escapeHTML(username)+'"><div class="coursedashboard-hasetdata" data-doneasses="'+(counter || '0')+'"><div class="coursedashboard-hasetpercentage" style="width: '+percentage+'%"></div><div class="coursedashboard-hasetdata-contenttext">'+(counter ? counter + '/' + size : '-')+'</div></div></td>');
                table.push('</tr>');
                userhaindex++;
            };
        };
        table.push(
            '</tbody>',
            '<thead class="ffwidget-background">',
            '<tr class="coursedashboard-scoreboardone-headrow"><th>' + ebooklocalizer.localize('dashboard:assignmentgroup', uilang) + '</th>',
            //'<th class="coursedashboard-scoreboardone-assignmenthead" colspan="1"></th>',
            '</tr>',
            '</thead>',
            '</table>'
        );
        this.scoreboardOne.html(table.join('\n'));
        var headnums = [];
        for (var i = 0; i < maxcount; i++) {
            headnums.push('<th>' + (i+1) + '</th>');
        };
        headnums.push('<th></th>');
        this.scoreboardOne.find('.coursedashboard-scoreboardone-headrow').append(headnums.join(''));
        //this.scoreboardOne.find('.coursedashboard-scoreboardone-assignmenthead').attr('colspan', maxcount);
        this.scoreboardOne.find('.coursedashboard-scoreboardone-userhead').attr('colspan', maxcount + 2);
    };
    
    Dashboard.prototype.toggleHasetColumn = function(hasetnum) {
        var selectedcol = this.scoreboard.find('th[data-hasetnum="'+escapeHTML(hasetnum)+'"], td[data-hasetnum="'+escapeHTML(hasetnum)+'"]');
        var ison = selectedcol.is('.coursedashboard-currentha');
        if (!ison) {
            selectedcol.addClass('coursedashboard-currentha');
        } else {
            selectedcol.removeClass('coursedashboard-currentha');
        };
    };
    
    Dashboard.prototype.gotoAss = function(data) {
        var gotolink = this.hasets[data.haset] && this.hasets[data.haset].getGotolink(data.assnumber);
        if (data.solcount > 0) {
            var assid = this.hasets[data.haset] && this.hasets[data.haset].getAssignment(data.assnumber);
            var solution = '';
            if (data.username) {
                solution = this.solsByCreator[data.username][assid][data.solcount-1].getId();
            };
            gotolink.data.link.solution = true;
            gotolink.data.link.solname = solution;
            gotolink.data.link.solver = data.username;
        };
        this.place.trigger('gotolink', gotolink);
    }
    
    Dashboard.prototype.getFilteredHasets = function(username) {
        var usergroups = this.settings.users.getGroupsByUser(username);
        var result = [];
        for (var i = 0, len = this.hasets.length; i < len; i++) {
            if (this.hasets[i].hasUser(username) || this.hasets[i].hasGroup(usergroups)) {
                result.push(this.hasets[i]);
            };
        };
        return result;
    };
    
    Dashboard.prototype.setContentdata = function(data) {
        switch (data.contentType) {
            case 'solution':
                this.solsByCreator = {};
                var solution, creator, assid;
                for (var solid in data.contentdata) {
                    solution = new Solution(data.contentdata[solid]);
                    creator = solution.getCreator();
                    assid = solution.getAssignment();
                    if (!this.solsByCreator[creator]) {
                        this.solsByCreator[creator] = {};
                    }
                    if (!this.solsByCreator[creator][assid]) {
                        this.solsByCreator[creator][assid] = [];
                    };
                    this.solsByCreator[creator][assid].push(solution);
                    this.solsByCreator[creator][assid].sort(function(a, b) {
                        return a.getModified().getTime() - b.getModified().getTime();
                    });
                }
                break;
            case 'hamessage':
                this.hasets = [];
                var haset;
                var hamess;
                for (var hamessid in data.contentdata) {
                    data.contentdata[hamessid].data.name = hamessid;
                    hamess = data.contentdata[hamessid];
                    if (!hamess.data.givendate) {
                        hamess.data.givendate = hamess.metadata.created;
                    };
                    haset = new HAset(hamess.data);
                    this.hasets.push(haset);
                    this.hasets.sort(function(a, b) {
                        return (a.getGivendate().getTime() - b.getGivendate().getTime);
                    });
                };
                break;
            case 'review':
                this.reviews = {};
                this.reviewdata = {};
                var review, solname;
                for (var rev in data.contentdata) {
                    review = data.contentdata[rev];
                    solname = review.metadata.ref || '';
                    if (solname) {
                        if (!this.reviews[solname]) {
                            this.reviews[solname] = [];
                        };
                        this.reviews[solname].push(rev);
                        this.reviewdata[rev] = review;
                    };
                };
                break;
            default:
                break;
        };
    };
    
    Dashboard.prototype.setButtonTitles = function() {
        var uilang = this.settings.uilang;
        this.place.find('button[data-action="refreshapp"]').attr('title', ebooklocalizer.localize('dashboard:refresh', uilang));
    }
    
    Dashboard.prototype.close = function() {
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    }
    
    Dashboard.prototype.setStyles = function() {
        if ($('head style#coursedashboardstyles').length === 0) {
            $('head').append('<style type="text/css" id="coursedashboardstyles">'+this.styles+'</style>')
        }
    }
    
    Dashboard.prototype.defaults = {
        settings: {
            uilang: 'en'
        }
    };
    
    Dashboard.prototype.styles = [
        '.coursedashboard {background-color: #eee; font-family: helvetica, Arial, sans-serif; max-height: 100%; overflow: auto;}',
        '.coursedashboard h1 {text-align: center;}',
        '.coursedashboard-scoreboard {overflow: auto;}',
        '.coursedashboard-scoreboard tbody th {text-align: left; font-weight: normal; white-space: nowrap;}',
        '.coursedashboard-scoreboard table {border-collapse: collapse; border: 1px solid #aaa; margin: 1em auto;}',
        '.coursedashboard-scoreboard thead th {border: 1px solid #aaa; text-align: center;}',
        '.coursedashboard-scoreboard thead td {border: 1px solid #aaa; background-color: white; color: black; text-align: center;}',
        'th.coursedashboard-scoreboard-haset, td.coursedashboard-scoreboard-assignment, td.coursedashboard-scoreboard-hatotal, td.coursedashboard-scoreboard-usertotal {cursor: pointer;}',
        'th.coursedashboard-scoreboard-haset {vertical-align: top;}',
        '.coursedashboard-scoreboard-assnumber {font-size: 90%;}',
        '.coursedashboard-scoreboard tbody tr:nth-child(even) {background-color: rgba(200,200,200,0.2);}',
        '.coursedashboard-scoreboard tbody tr:nth-child(odd) {background-color: rgba(255,255,255,0.8);}',
        '.coursedashboard-scoreboard tbody tr:hover {background-color: rgba(255,255,170, 0.5);}',
        '.coursedashboard-scoreboard tbody th {padding-left: 0.3em; padding-right: 0.3em;}',
        '.coursedashboard-scoreboard-hasetnumber {text-align: center; font-weight: bold;}',
        '.coursedashboard-scoreboard-hasetdeadline {text-align: center; font-size: 60%; font-weight: normal; font-style: italic;}',
        '.coursedashboard-scoreboard-hasettitle {font-size: 60%; font-weight: normal;}',
        '.coursedashboard-scoreboard-username {cursor: pointer;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="black"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid black;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="red"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #ff4141;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="orange"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #ff9900;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="yellow"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid gold;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="green"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #009100;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="blue"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #0084c8;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="violet"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #ba00ff;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="brown"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #804d00;}',
        '.coursedashboard-scoreboard-assignment[data-hacolors="brow"] > .coursedashboard-scoreboard-assnumber {border-top: 2px solid #804d00;}', // Ooops
        // Score cells
        '.coursedashboard-scoreboard tbody td {border-left: 1px solid #aaa; border-right: 1px solid #aaa; text-align: center; font-family: monospace; font-size: 90%; line-height: 1.3em;}',
        '.coursedashboard-scoreboard-assdatacontent, .coursedashboard-scoreboard-assnumber {overflow: hidden; max-width: 0; transition: max-width 0.5s, padding 0.5s;}',
        '.coursedashboard-scoreboard-assdatacontent {padding: 0; cursor: pointer; position: relative;}',
        '.coursedashboard-scoreboard-solcount {font-size: 60%; position: absolute; top: 0; left: 0; line-height: 1em;}',
        '.coursedashboard-scoreboard-bestreview {font-weight: bold;}',
        '.coursedashboard-currentha .coursedashboard-scoreboard-assdatacontent, .coursedashboard-currentha .coursedashboard-scoreboard-assnumber  {max-width: 3em;}',
        '.coursedashboard-currentha .coursedashboard-scoreboard-assdatacontent {padding: 0 0.5em;}',
        '.coursedashboard-hasetdata {background-color: #ffa; position: relative; min-width: 2em;}',
        '.coursedashboard-hasetdata[data-doneasses="0"] {background-color: #faa;}',
        '.coursedashboard-hasetpercentage {width: 0; position: absolute; top: 0; bottom: 0; left: 0; background-color: rgba(0, 255, 0, 0.5);}',
        '.coursedashboard-scoreboard-assdatacell[data-allreviewed="false"] .coursedashboard-scoreboard-solcount {color: red;}',
        '.coursedashboard-reviewer .coursedashboard-scoreboardone-assdatacell[data-allreviewed="false"] .coursedashboard-scoreboard-solcount {color: red;}',
        '.coursedashboard-myown .coursedashboard-scoreboardone-assdatacell[data-lastreviewed="true"] .coursedashboard-scoreboard-solcount {color: blue;}',
        '.coursedashboard-hasetdata-contenttext {position: relative; font-size: 80%; color: #444;}',
        '.coursedashboard-scoreboard-empty {background-color: #bbb; border: none;}',
        // Scoreboard one
        '.coursedashboard-scoreboard-one table {border-collapse: collapse; border: 1px solid #aaa; margin: 1em auto; background-color: #bbb;}',
        '.coursedashboard-scoreboard-one thead th {border: 1px solid #aaa;}',
        '.coursedashboard-scoreboard-one thead tr.coursedashboard-scoreboardone-headrow th {font-weight: normal;}',
        '.coursedashboard-scoreboard-one tbody tr:nth-child(even) {background-color: #eee; color: black;}',
        '.coursedashboard-scoreboard-one tbody tr:nth-child(odd) {background-color: #fff; color: black;}',
        '.coursedashboard-scoreboard-one tbody td {border-left: 1px solid #aaa; min-width: 3em; padding: 0 0.3em;}',
        '.coursedashboard-scoreboard-one tbody td.coursedashboard-scoreboardone-assgroup {padding: 0 0.3em;}',
        '.coursedashboard-scoreboard-one tbody td.coursedashboard-scoreboardone-assdatacell {text-align: center; cursor: pointer;}',
        '.coursedashboard-scoreboard-one tbody .coursedashboard-scoreboardone-assnum {visibility: hidden; white-space: nowrap; display: none;}',
        '.coursedashboard-scoreboard-one tbody tr:hover .coursedashboard-scoreboardone-assnum {visibility: visible;}',
        '.coursedashboard-scoreboard-one tbody .coursedashboard-scoreboardone-assgroupitem {font-weight: bold; font-size: 80%;}',
        '.coursedashboard-scoreboard-one tbody .coursedashboard-scoreboardone-assgroupdl {font-style: italic; font-size: 80%;}',
        '.coursedashboard-scoreboard-one tbody .coursedashboard-scoreboardone-usertotal {padding: 0; text-align: center; font-size: }',
        '.coursedashboard-scoreboard-one tbody .coursedashboard-scoreboardone-usertotal .coursedashboard-hasetdata {height: 1.5em;}',
        '.coursedashboard-scoreboardone-solcount {position: relative;}',
        '.coursedashboard-scoreboardone-empty {background-color: #bbb; border: none;}',
        // buttons
        '.coursedashboard-controlbuttons button {line-height: 100%; padding: 0;}',
        '.coursedashboard-controlbuttons button svg {height: 30px; width: 30px;}'
    ].join('\n');
    
    Dashboard.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-stats"><path style="stroke: none;" d="M5 28 a2 2 0 0 1 0 -4 h20 a2 2 0 0 1 0 4z m-1 -6 v -7 h4 v7 z m6 0 v-12 h4 v12 z m6 0 v-18 h4 v18z m6 0 v-12 h4 v12z" /></svg>',
        refresh: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-refresh"><path style="stroke: none;" d="M16 14 l8 0 l0 -8 l-3 3 a8.5 8.5 0 1 0 0 12 l-2 -2 a5.6 5.6 0 1 1 0 -8 z"></path></svg>'
    }
    
    Dashboard.prototype.template = {
        all: [
            '<div class="coursedashboard-controlbuttons ffwidget-buttonset ffwidget-horizontal"><button class="ffwidget-setbutton" data-action="refreshapp">'+Dashboard.prototype.icons.refresh+'</button></div>',
            '<div class="coursedashboard-scoreboard"></div>',
            '<div class="coursedashboard-scoreboard-one coursedashboard-reviewer"></div>'
        ].join('\n'),
        one: [
            '<div class="coursedashboard-controlbuttons ffwidget-buttonset ffwidget-horizontal"><button class="ffwidget-setbutton" data-action="refreshapp">'+Dashboard.prototype.icons.refresh+'</button></div>',
            '<div class="coursedashboard-scoreboard-one coursedashboard-myown"></div>'
        ].join('\n')
    };
    
    Dashboard.localization = {
        "en": {
            "dashboard:homeassignments": "Homeassignments",
            "dashboard:name": "Name",
            "dashboard:assignmentgroup": "Assignment group",
            "dashboard:refresh": "Refresh"
        },
        "fi": {
            "dashboard:homeassignments": "Kotitehtävät",
            "dashboard:name": "Nimi",
            "dashboard:assignmentgroup": "Tehtäväryhmä",
            "dashboard:refresh": "Päivitä"
        },
        "sv": {
            "dashboard:homeassignments": "Hemuppgifter",
            "dashboard:name": "Namn",
            "dashboard:assignmentgroup": "Uppgiftgrupp",
            "dashboard:refresh": "Uppdatera"
        }
    }

    try {
        ebooklocalizer.addTranslations(Dashboard.localization);
    } catch (err) {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Dashboard.localization);
    }
    
    
    /**
     * Homeassignmentset
     */
    var HAset = function(data) {
        data = $.extend(true, {}, this.defaults, data);
        this.deadline = (data.deadline ? new Date(data.deadline) : '');
        this.givendate = (data.givendate ? new Date(data.givendate) : new Date(0));
        this.id = data.name;
        this.title = data.title;
        this.assignments = [];
        this.tousers = [];
        this.togroups = [];
        var cid, ass, i, len;
        for (i = 0, len = data.contents.length; i < len; i++) {
            cid = data.contents[i].id;
            if (data.contentdata[cid].type === 'homeassignmentelement') {
                this.assignments.push(new Assignment(data.contentdata[cid].data));
            };
        };
        var toid, recipient;
        for (i = 0, len = data.sendto.length; i < len; i++) {
            recipient = data.sendto[i];
            toid = recipient.to;
            if (recipient.pubtype === 'user' && this.tousers.indexOf(toid) === -1) {
                this.tousers.push(toid);
            } else if (recipient.pubtype === 'group' && this.togroups.indexOf(toid) === -1) {
                this.togroups.push(toid);
            }
        }
    };
    
    HAset.prototype.getId = function() {
        return this.id;
    };
    
    HAset.prototype.getTitle = function() {
        return this.title;
    };
    
    HAset.prototype.getDeadline = function() {
        return this.deadline;
    };
    
    HAset.prototype.getGivendate = function() {
        return this.givendate;
    };
    
    HAset.prototype.getDeadlineLocale = function(lang) {
        var result = '';
        if (this.deadline) {
            result = this.deadline.toLocaleDateString(lang)
        };
        return result;
    };
    
    HAset.prototype.getDeadlineLocaleLong = function(lang) {
        var result = '';
        if (this.deadline) {
            result = this.deadline.toLocaleDateString(lang, {weekday: 'long', year: 'numeric', month: 'long', day: 'numeric'});
        };
        return result;
    };
    
    HAset.prototype.size = function() {
        return this.assignments.length;
    };
    
    HAset.prototype.getColors = function(index) {
        var result = [];
        if (index < this.assignments.length && -1 < index) {
            result = this.assignments[index].getColors();
        };
        return result;
    };
    
    HAset.prototype.getAssignment = function(index) {
        var result = '';
        if (-1 < index && index < this.assignments.length) {
            result = this.assignments[index].getId();
        };
        return result;
    };
    
    HAset.prototype.getGotolink = function(index) {
        var result = {};
        if (-1 < index && index < this.assignments.length) {
            result = this.assignments[index].getGotolink();
        };
        return result;
    };
    
    HAset.prototype.hasUser = function(username) {
        return (this.tousers.length === 0 || (this.tousers.indexOf(username) > -1));
    };
    
    HAset.prototype.hasGroup = function(groupname) {
        if (typeof(groupname) === 'string') {
            groupname = [groupname];
        };
        var result = false;
        for (var i = 0, len = groupname.length; i < len && !result; i++) {
            result = result || (this.togroups.indexOf(groupname[i]) > -1);
        };
        return result;
    };
    
    HAset.prototype.defaults = {
        contentdata: {},
        contents: [],
        deadline: 0,
        title: ''
    };
    
    /**
     * Assignment
     */
    var Assignment = function(data) {
        data = $.extend(true, {}, this.defaults, data);
        this.colors = data.colors;
        this.level = data.level;
        this.hatitle = data.title;
        this.hatext = data.text;
        this.gotolink = data.gotolink;
        this.id = this.gotolink.data.link.elementid || '';
        this.asstitle = data.gotolink.data.title;
        this.asstext = data.gotolink.data.text;
        this.asstype = data.gotolink.data.type;
    };
    
    Assignment.prototype.getId = function() {
        return this.id;
    };
    
    Assignment.prototype.getTitle = function() {
        return this.hatitle;
    };
    
    Assignment.prototype.getText = function() {
        return this.hatext;
    };
    
    Assignment.prototype.getColors = function() {
        var result = [];
        for (var key in this.colors) {
            if (this.colors[key]) {
                result.push(key);
            };
        };
        return result;
    };
    
    Assignment.prototype.getLevel = function() {
        return this.level;
    }
    
    Assignment.prototype.getGotolink = function() {
        return JSON.parse(JSON.stringify(this.gotolink));
    };
    
    Assignment.prototype.defaults = {
        colors: {},
        gotolink: {
            data: {
                link: {},
                title: '',
                text: '',
                type: 'assignment'
            }
        },
        level: 0,
        title: '',
        text: ''
    };
    
    /**
     * Solutionset
     */
    var Solutionset = function(data) {
        data = $.extend(true, {}, this.defaults, data);
        this.solutions = {};
    }
    
    Solutionset.prototype.addSolution = function(solution) {
        solution = $.extend(true, {}, this.solutio)
    }
    
    Solutionset.prototype.defaults = {
        
    }
    
    /**
     * Solution
     */
    var Solution = function(data) {
        data = $.extend(true, {}, this.defaults, data);
        this.created = new Date(data.metadata.created);
        this.modified = new Date(data.metadata.modified);
        this.creator = data.metadata.creator;
        this.modifier = data.metadata.modifier;
        this.assignment = data.metadata.ref;
        this.id = data.name;
    };
    
    Solution.prototype.getId = function() {
        return this.id;
    };
    
    Solution.prototype.getCreator = function() {
        return this.creator;
    };
    
    Solution.prototype.getModifier = function() {
        return this.modifier;
    };
    
    Solution.prototype.getCreated = function() {
        return this.created;
    };
    
    Solution.prototype.getModified = function() {
        return this.modified;
    };
    
    Solution.prototype.getAssignment = function() {
        return this.assignment;
    };
    
    
    Solution.prototype.defaults = {
        metadata: {
            creator: '',
            modifier: '',
            created: 0,
            modified: 0,
            ref: ''
        },
        name: ''
    };
    
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var db = new Dashboard(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[dashboarddata]]');
            return data;
        },
        'geticon': function() {
            return Dashboard.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.coursedashboard = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in coursedashboard.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
