/******
@name VirumCourse
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class VirumCourse
@description A class and jQuery-plugin for a virum course.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - ElementSet
 * - EbookLocalizer
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.apply;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(ebooklocalizer) === 'undefined' && ebooklocalizer.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };

    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    /******
     * Some data structures from Elpp libs
     ******/
    var ContentStorage = window.Elpp.ContentStorage;
    var Userinfo = window.Elpp.Userinfo;
    var Userlist = window.Elpp.Userlist;

    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var course = new VirumCourse(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[virumcoursedata]]');
            return data;
        }
    }
    
    $.fn.virumcourse = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in virumcourse.');
            return false;
        }
    }
    
    /******
     * VirumCourse class
     * @class VirumCourse
     * @constructor
     * @param {jQuery} place - Place for virumcourse
     * @param {Object} options - options for the virumcourse
     ******/
    var VirumCourse = function(place, options){
        this.place = $(place);
        this.init(options);
        this.addHandlers();
        this.show();
        this.setServerSync({serversync: this.context.serversync});
        this.showPendingNotifs();
        this.place.trigger('courseready');
        this.place.trigger('contextready', {contextid: this.context.id, contexttype: this.context.type});
        window.Elpp.virumcourse = this;
    };
    
    /******
     * Init the course
     * @param {Object} options - the settings for the course
     ******/
    VirumCourse.prototype.init = function(options){
        options = $.extend(true, {}, VirumCourse.defaults, options);
        var storage = $.extend(true, {}, options.storage);
        var storageparts = $.extend(true, [], options.storageparts);
        var updateparts = $.extend(true, {}, options.updateparts);
        updateparts = this.validateUpdates(updateparts);
        storageparts = storageparts.concat(updateparts);
        this.setStyles();
        this.context = options.context;
        this.course = this.context;  // Alias
        this.apps = this.context.apps;
        this.appdata = options.appdata;
        this.configforms = {};
        this.material = this.context.material;
        this.materialdata = options.materialdata;
        this.setAttrs();
        this.refreshFeed = false;
        if (options.settings.role === 'admin') {
            this.adminable = true;
            options.settings.role = 'teacher';
        };
        this.settings = options.settings;
        this.margin = {};
        this.shareables = {};
        this.users = new Userlist({
            username: this.settings.username,
            contextid: this.context.id,
            contexttype: this.context.type,
            contexttitle: this.context.titles.common || '',
            contextshortdesc: this.context.short_desc,
            contextdescription: this.context.description,
            users: this.context.users,
            groups: this.context.groups,
            rights: this.context.rights
        });
        this.configs = $.extend(true, {}, options.configs, this.loadConfigs());
        this.pendingnotifications = options.notifications.slice();
        // TODO old format
        //this.initRefData(options.refdata);
        if (this.settings.hasparent) {
            this.configable = false;
        } else {
            this.configable = true;
        };
        // Init storage
        this.storage = new ContentStorage(storage, this.settings.singlelangmode);
        this.updatestorage = new ContentStorage({}, this.settings.singlelangmode);
        // Append storageparts and updateparts to the storage.
        for (var i = 0, len = storageparts.length; i < len; i++) {
            if (storageparts[i].contentinfo) {
                this.setStorageData(storageparts[i]);
            };
        };
        // Give the data whole storage to the outer system for saving.
        this.saveStorage();
    };
    
    /******
     * Set some attributes for the DOM-place
     ******/
    VirumCourse.prototype.setAttrs = function(){
        this.place.addClass('virumcourse-wrapper')
            .attr('data-courseid', escapeHTML(this.course.id))
            .attr('data-contextid', escapeHTML(this.context.id));
    };
    
    /******
     * Get context data
     * @returns {Object} - the same kind of data that this NotebookHandler is inited with.
     ******/
    VirumCourse.prototype.getContextData = function() {
        var context = {
            context: JSON.parse(JSON.stringify(this.context)),
            storage: this.storage.getAllData(),
            storageparts: [],
            updateparts: {}
        };
        return context;
    };
    
    /******
     * Get context savedata (applications' data and all storages/storageparts)
     * @returns {Object} - appdata, storage, storageparts and updateparts
     ******/
    VirumCourse.prototype.getContextSave = function() {
        var context = {
            contextinfo: {
                context: JSON.parse(JSON.stringify(this.context)),
                apps: JSON.parse(JSON.stringify(this.apps))
            },
            contextdata: {
                storage: this.storage.getAllData(),
                storageparts: [],
                updateparts: {}
            },
            saveneeded: true
        };
        return context;
    };
    
    /******
     * Set the title of the course
     * @param {String} [title]   The title to set (optional)
     ******/
    VirumCourse.prototype.setTitle = function(title){
        var place = this.place.find('.virumcourse-control .virumcourse-title');
        title = this.sanitize(title || this.context.titles[this.settings.lang.split(/[-_]/)[0]] || this.context.titles.common || this.course.title || '');
        place.text(title);
        $('head title').text(title);
    }
    
    /**
     * Set the title of an app
     * @param {Object} options   Options for setting the title
     * @param {String} options.appid    The id if the app
     * @param {String} options.title    The title to change
     */
    VirumCourse.prototype.setAppTitle = function(options) {
        this.place.find('.virumcourse-applist .virumcourse-applistitem[data-appid="'+options.appid+'"] .virumcourse-apptitle').text(this.sanitize(options.title, {ALLOWED_TAGS: []}));
    };
    
    /**
     * Set serversync mode
     * @param {Object} options
     * @param {Boolean} options.serversync - true: context can be synced to the server, false: context can not be synced
     * @param {String} options.message    - The message for notification.
     */
    VirumCourse.prototype.setServerSync = function(options) {
        var serversync = options.serversync;
        var message = options.message;
        this.serversync = serversync;
        this.place.attr('data-serversync', escapeHTML(serversync));
        if (serversync) {
            this.denotify({
                id: 'serversync',
                nclass: 'warning'
            });
            if (message) {
                this.notify({
                    id: 'serversync',
                    nclass: 'quick',
                    message: message
                });
            }
        } else {
            this.notify({
                id: 'serversync',
                nclass: 'warning',
                message: message || 'Is not synced to the server.',
                data: {contextid: this.context.id, events: [], data: []}
            });
            if (message) {
                this.notify({
                    id: 'serversync',
                    nclass: 'quick',
                    message: message
                });
            };
        };
    };
    
    /******
     * Close all apps
     ******/
    VirumCourse.prototype.close = function(){
        this.appsOpen = this.appsOpen || 0;
        var $apps = this.place.find('.virumcourse-app[data-appid] > .virumcourse-appcontent');
        $apps.trigger('closechildrenapp');
    };
    
    /**
     * One children app is closed
     */
    VirumCourse.prototype.appClosed = function(data) {
        this.appsOpen--;
        this.place.find('.virumcourse-app[data-apptype="'+data.apptype+'"][data-appid="'+data.appid+'"]').remove();
        this.place.find('.virumcourse-applist .virumcourse-applistitem[data-apptype="'+data.apptype+'"][data-appid="'+data.appid+'"]').remove();
        if (this.shareables[data.appid]) {
            delete this.shareables[data.appid];
        };
        if (this.appsOpen <= 0) {
            // If all apps are closed, data can be saved and this context can be closed.
            this.saveContent();
            //this.sendFromOutbox();
            this.place.trigger('getcontextsave');
            this.place.trigger('closeok', {contexttype: this.context.type, contextid: this.context.id});
            // TODO:
            // 1. sendFromOutbox()
            // 2. Wait for reply or reasonable timeout
            // 3. Save storage
            // 4. Wait for reply of reasonable timeout
            // 5. If reply: close : else : show error message.
        };
    };
    
    /******
     * Show the course
     ******/
    VirumCourse.prototype.show = function(){
        this.place.html(VirumCourse.templates.html);
        this.notificationarea = this.place.find('.virumcourse-control-notifications');
        this.notificationarea.vnotifier($.extend({}, VirumCourse.notification, {level: (this.adminable ? 10 : 2)}));
        this.place.find('.virumcourse-mathpanel').mqmathpanel({settings: {uilang: this.settings.uilang.split(/[-_]/)[0]},configs: this.getConfig('mqmathpanel')});
        this.showControls();
        this.setTitle();
        //var apparea = this.place.find('.virumcourse-applicationarea');
        var apparea = this.place.find('.virumcourse-apparea').eq(0);
        //var materialarea = this.place.find('.virumcourse-materialarea');
        var materialarea = this.place.find('.virumcourse-apparea').eq(0);
        var applist = this.place.find('.virumcourse-applist-right');
        var materiallist = this.place.find('.virumcourse-applist-left');
        var appplace, app, appname, appconf, appdata;
        this.appsOpen = 0;
        var i;
        for (i = 0, len = this.material.length; i < len; i++){
            app = this.material[i];
            this.showApp(materialarea, materiallist, app, 'material');
        };
        for (i = 0, len = this.apps.length; i < len; i++){
            app = this.apps[i];
            this.showApp(apparea, applist, app, 'app');
        };
        this.startStopUpdatePoll();
    }
    
    /**
     * Show an app or a material
     * @param {jQuery} place    - Place for adding apps or material.
     * @param {jQuery} listplace  Place for adding listitems for apps or material.
     * @param {Object} app      - Object describing app or material
     *                            {type: "...", jq: "<jQuery-method>", id: "..."}
     * @param {String} apptype  - Type of app ("app"|"material")
     */
    VirumCourse.prototype.showApp = function(place, listplace, app, apptype) {
        var jq = app.jq;
        var appplace = $(VirumCourse.templates[apptype]);
        appplace.attr('data-apptype', escapeHTML(app.type)).attr('data-appid', escapeHTML(app.id));
        if (apptype === 'app') {
            appplace.addClass('sidepanel');
            place.append(appplace);
        } else {
            place.prepend(appplace);
        };
        var appcontent = appplace.children('.virumcourse-appcontent');
        var appliplace = $(VirumCourse.templates.listitem);
        appliplace.attr('data-apptype', escapeHTML(app.type)).attr('data-appid', escapeHTML(app.id));
        listplace.append(appliplace);
        var appicon = '', apptitle = '';
        var datastorage = {};
        switch (apptype) {
            case 'material':
                datastorage = this.materialdata || {};
                break;
            case 'app':
                datastorage = this.appdata || {};
                break;
        }
        var appdata = datastorage[app.id] || {};
        appdata.type = appdata.type || app.type;
        appdata.name = appdata.name || app.id;
        appdata.settings = $.extend(true, {}, {
            appid: app.id,
            apptype: app.type,
            username: this.settings.username,
            mode: 'view',
            role: this.settings.role,
            lang: this.settings.lang.split(/[-_]/)[0],
            uilang: this.settings.uilang.split(/[-_]/)[0],
            readonly: this.settings.readonly,
            courseid: this.context.id,
            contextid: this.context.id,
            context: {
                id: this.context.id,
                type: this.context.type,
                titles: this.context.titles,
                short_desc: this.context.short_desc,
                description: this.context.description,
                teacher: this.context.teacher,
                schools: this.context.schools,
                subjects: this.context.subjects
            },
            users: this.users,
            userinfo: this.settings.user,
            libraryinfo: this.settings.library,
            elementpanel: !this.settings.elementpanel,
            hasparent: true,
            devmode: this.settings.devmode,
            gotolink: {
                courseid: this.context.id,
                contextid: this.context.id,
                appname: appdata.name
            },
            margin: this.margin
        }, appdata.settings);
        try {
            var appname = appdata.data && appdata.data.bookid || appdata.name;
            if (appname) {
                // If the name was in data, then use the configs here.
                appdata.configs = $.extend(true, {}, appdata.configs, this.configs[appname]);
            }
            appcontent[jq](appdata);
            if (!appname) {
                // If the name was not in the data, get it now and use configs.
                appname = appcontent.attr('data-appname');
                appconf = this.configs[appname];
                if (appconf) {
                    appcontent.trigger('setconfigs', JSON.parse(JSON.stringify(appconf)));
                };
            };
            this.appsOpen++;
            appicon = appcontent[jq]('geticon');
            apptitle = this.sanitize(appcontent[jq]('gettitle'), {ALLOWED_TAGS: []});
        } catch (err){
            console.log('virumcourse-apperror', err);
            appcontent.addClass('virumcourse-apperror');
        };
        appliplace.html((appicon ? '<div class="virumcourse-apptitle ffwidget-background">'+escapeHTML(apptitle)+'</div>' : '') + '<div class="virumcourse-appicon">' + sanitize(appicon) + '</div><div class="virumcourse-appclose"><span class="virumcourse-appclose-button">&#x00d7;</span></div>');
    };
    
    /**
     * Toggle the visibility of selected application
     * @param {String} appid   Id of the application
     * @param {Boolean} [forceon]  Set this mode and don't toggle
     */
    VirumCourse.prototype.toggleApp = function(appid, forceon) {
        var ison;
        var app = this.place.find('.virumcourse-app.sidepanel[data-appid="'+appid+'"]');
        var all = this.place.find('.virumcourse-app.sidepanel');
        var alltab = this.place.find('.virumcourse-applist-right li');
        var tab = alltab.filter('[data-appid="'+appid+'"]');
        if (typeof(forceon) === 'boolean') {
            ison = !forceon;
        } else {
            ison = app.is('.sidepanel-visible');
        };
        all.removeClass('sidepanel-visible');
        alltab.removeClass('tabactive');
        if (!ison) {
            app.addClass('sidepanel-visible');
            tab.addClass('tabactive');
            app.children('.virumcourse-appcontent').trigger('refreshapp');
        };
        var width = app.width();
        this.place.trigger('appresized', {appname: '', change: -width});
    };
    
    /**
     * Toggle app fullscreen
     * @param {String} appid   The id of the app
     * @param {Boolean} [forcemode]  Force fullscreen on or off
     */
    VirumCourse.prototype.toggleAppFullscreen = function(appid, forcemode) {
        //this.toggleApp(appid, true);
        var app = this.place.find('.virumcourse-app.sidepanel[data-appid="'+appid+'"]');
        var all = this.place.find('.virumcourse-app.sidepanel');
        var button = app.find('.virumcourse-apptopbar button[data-action="togglemaximize"]');
        var winbutton = app.find('.virumcourse-apptopbar button[data-action="togglewindowize"]');
        var allbutton = all.find('.virumcourse-apptopbar button[data-action="togglemaximize"]');
        var ison;
        if (typeof(forcemode) === 'boolean') {
            ison = !forcemode;
        } else {
            ison= app.is('.virumcourse-appfullscreen');
        };
        all.removeClass('virumcourse-appfullscreen');
        allbutton.removeClass('buttonselected');
        if (!ison) {
            app.addClass('virumcourse-appfullscreen');
            app.removeClass('virumcourse-appwindowized');
            button.addClass('buttonselected');
            winbutton.removeClass('buttonselected');
        };
        this.place.trigger('appresized', {appname: '', change: 0});
    };
    
    /**
     * Toggle app windowized
     * @param {String} appid   The id of the app
     * @param {Boolean} [forcemode]  Force fullscreen on or off
     */
    VirumCourse.prototype.toggleAppWindowize = function(appid, forcemode) {
        //this.toggleApp(appid, true);
        var app = this.place.find('.virumcourse-app.sidepanel[data-appid="'+appid+'"]');
        var maxbutton = app.find('.virumcourse-apptopbar button[data-action="togglemaximize"]');
        var button = app.find('.virumcourse-apptopbar button[data-action="togglewindowize"]');
        var ison;
        if (typeof(forcemode) === 'boolean') {
            ison = !forcemode;
        } else {
            ison= app.is('.virumcourse-appwindowized');
        };
        if (!ison) {
            app.addClass('virumcourse-appwindowized');
            app.removeClass('virumcourse-appfullscreen');
            button.addClass('buttonselected');
            maxbutton.removeClass('buttonselected');
        } else {
            app.removeClass('virumcourse-appwindowized');
            button.removeClass('buttonselected');
        };
        var width = app.width();
        this.place.trigger('appresized', {appname: '', change: -width});
    };
    
    /******
     * Show control buttons
     ******/
    VirumCourse.prototype.showControls = function(){
        var controlarea = this.place.find('.virumcourse-control-buttons');
        var config = $.extend(true, {minimized: true}, this.configs.elementpanel);
        var dockvertical = this.place.find('.virumcourse-dock[data-dockorient="vertical"]');
        var dockhorizontal = this.place.find('.virumcourse-dock[data-dockorient="horizontal"]');
        var dock = (config.orientation === 'vertical' ? dockvertical : dockhorizontal);
        var elementpanel = $('<div class="virumcourse-elementpanel"></div>');
        dock.append(elementpanel);
        //var settings = $.extend(true, {}, this.settings);
        var settings = {
            username: this.settings.username,
            users: this.users,
            role: this.settings.role,
            uilang: (this.settings.uilang || 'en').split(/[-_]/)[0],
            lang: (this.settings.lang || 'en').split(/[-_]/)[0]
        };
        elementpanel.elementpanel({
            docks: {
                horizontal: dockhorizontal,
                vertical: dockvertical
            },
            settings: settings,
            config: config
        });
        var configbutton = $('<div class="virumcourse-configbutton virumcourse-control-button">'+VirumCourse.icons.config+'</div>');
        controlarea.append(configbutton);
    };
    
    /******
     * Show config dialog
     ******/
    VirumCourse.prototype.showConfigs = function(){
        var uilang = this.settings.uilang.split(/[-_]/)[0];
        var confforms = JSON.parse(JSON.stringify(this.configforms));
        var confvalues = JSON.parse(JSON.stringify(this.configs));
        var formsets = [];
        var apps = this.place.find('[data-appname]');
        var appname;
        for (var i = 0, len = apps.length; i < len; i++) {
            appname = apps.eq(i).attr('data-appname');
            if (confforms[appname]) {
                formsets.push(appname);
            };
        };
        if (!this.dialog) {
            this.dialog = $('<div class="configdialog"></div>');
            this.place.append(this.dialog);
        };
        var confconf = confvalues['Courseconfig'];
        this.dialog.formdialog({
            type: 'dialog',
            name: 'Courseconfig',
            data: {
                title: ebooklocalizer.localize('virumcourse:configs', uilang),
                icon: VirumCourse.icons.config,
                formsets: formsets,
                forms: confforms,
                values: confvalues
            },
            configs: confconf,
            settings: {
                uilang: uilang,
                lang: this.settings.lang,
                modal: false
            }
        });
    };
    
    /******
     * Do something when course gets new data (send notifications etc.)
     * @param {Object} data - one element object.
     ******/
    VirumCourse.prototype.newcoursedata = function(data) {
        var notifdata = {};
        var sendnotif = false;
        this.refreshFeed = false;
        switch (data.contentType){
            case 'message':
                notifdata.nclass = 'messages';
                var sendfrom = this.users.getRealname(data.contentinfo.sendfrom) || '';
                notifdata.message = (data.contentdata.data.title || '') + ' (' + sendfrom + ')';
                notifdata.data = {messageid: data.name};
                sendnotif = !data.contentinfo.read && data.contentinfo.sendfrom !== this.settings.username;
                this.refreshFeed = true;
                break;
            case 'homeassignment':
            case 'hamessage':
                notifdata.nclass = 'messages';
                notifdata.message = (data.contentdata.data.title || '') + ' (' + (data.contentinfo.sendfrom || '') + ')';
                notifdata.icon = '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path style="stroke: none;" d="M3 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path></svg>';
                notifdata.data = {messageid: data.name};
                sendnotif = true;
                this.refreshFeed = true;
                break;
            default:
                break;
        }
        if (sendnotif) {
            this.notify(notifdata);
        };
    }
    
    /******
     * Trigger notifications
     ******/
    VirumCourse.prototype.notify = function(ntification){
        this.notificationarea.trigger('newnotification', ntification);
    }
    
    /******
     * Trigger removing of  notifications
     ******/
    VirumCourse.prototype.denotify = function(ntification){
        this.notificationarea.trigger('removenotification', ntification);
    }
    
    /******
     * Remove all event handlers attached to DOM-place
     ******/
    VirumCourse.prototype.removeHanderls = function(){
        this.place.off();
    };
    
    /******
     * Add event handlers
     ******/
    VirumCourse.prototype.addHandlers = function(){
        var course = this;
        this.removeHanderls();
        /******
         * Handle notifications to the notification toolbar.
         ******/
        this.place.off('notification').on('notification', function(event, data){
            event.stopPropagation();
            course.notify(data);
        }).off('denotification').on('denotification', function(event, data){
            event.stopPropagation();
            course.denotify(data);
        }).off('edit_started').on('edit_started', function(event, data){
            data.contexttype = course.context.type;
            data.contextid = course.context.id;
        }).off('edit_stopped').on('edit_stopped', function(event, data){
            data.contexttype = course.context.type;
            data.contextid = course.context.id;
        }).off('getcontextdata').on('getcontextdata', function(event, data){
            event.stopPropagation();
            var cdata = course.getContextData();
            course.place.trigger('contextdata', [cdata]);
        }).off('getcontextsave').on('getcontextsave', function(event, data){
            event.stopPropagation();
            var cdata = course.getContextSave();
            var appinfo = {
                contexttype: course.context.type,
                contextid: course.context.id,
            };
            course.place.trigger('contextsave', [cdata, appinfo]);
        }).off('flushalldirty').on('flushalldirty', function(event, data){
            event.stopPropagation();
            course.place.find('[data-appname]').trigger('flushdirty');
        }).off('init_refresh_hamessages').on('init_refresh_hamessages', function(event, data){
            event.stopPropagation();
            course.place.find('.virumcourse-app .ebook-elementset').trigger('refresh_hamessages');
        }).off('makeupdaterequest').on('makeupdaterequest', function(event, data) {
            event.stopPropagation();
            course.requestUpdates();
        }).off('quickmsgack').on('quickmsgack', function(event, data) {
            event.stopPropagation();
            if (data && data.data) {
                var events = data.data.events;
                var evdata = data.data.data;
                for (var i = 0, len = events.length; i < len; i++) {
                    course.place.trigger(events[i], [evdata[i]]);
                };
            };
        });
        /******
         * Add context information to 'appready' event
         ******/
        this.place.off('appready').on('appready', function(event, data) {
            data.contexttype = course.course.type;
            data.contextid = course.course.id;
        });
        /******
         * Merge new content data to apps
         ******/
        this.place.on('mergecontent', function(event, data){
            event.stopPropagation();
            var appdata;
            for (var appname in data){
                appdata = data[appname];
                course.place.find('[data-appname="'+appname+'"]').trigger('mergecontent', appdata);
            };
        });
        /******
         * Save contentinfo of messages from feed
         ******/
        this.place.on('setcontentinfo', function(event, data){
            event.stopPropagation();
            course.storage.updateContentInfo(data, true); // true === mark as dirty
        });
        /******
         * Get contentinfo from storage. (With queries)
         ******/
        this.place.off('getcontentinfo').on('getcontentinfo', function(event, data) {
            // get contentinfo from storage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var target = $(event.target);
            course.getContentinfo(data, target);
        });
        /******
         * Get content from storage. (With queries)
         ******/
        this.place.off('getcontent').on('getcontent', function(event, data, isinit) {
            // get content from storage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var target = $(event.target);
            course.getContent(data, target, isinit);
        });
        /**
         * Get updated content from updatestorage. (With queries)
         */
        this.place.off('getupdatecontent').on('getupdatecontent', function(event, data) {
            // get content from updatestorage (by anchor or by name)
            event.stopPropagation();
            event.preventDefault();
            var query, ctype;
            var target = $(event.target);
            var result;
            for (var i = 0, len = data.contentType.length; i < len; i++){
                ctype = data.contentType[i];
                if (data.anchors) {
                    result = course.updatestorage.getContentByAnchor({
                        contentType: ctype,
                        lang: data.lang,
                        anchors: data.anchors
                    });
                } else if (data.names) {
                    result = course.updatestorage.getContentByName({
                        contentType: ctype,
                        lang: data.lang,
                        names: data.names
                    });
                } else {
                    result = {contentType: ctype, contentdata: {}};
                };
                var contentlist = [];
                for (var cname in result.contentdata) {
                    contentlist.push(cname);
                };
                course.updatestorage.removeContentByName(ctype, contentlist, data.lang);
                target.trigger('reply_getcontent', result);
            };
        });
        /******
         * Save content to the storage.
         ******/
        this.place.off('setcontent').on('setcontent', function(event, data, isdirty){
            event.stopPropagation();
            event.preventDefault();
            if (typeof(data) === 'object' && typeof(data.length) === 'undefined'){
                data = [data];
            }
            var cid = course.context.id;
            for (var i = 0, len = data.length; i < len; i++) {
                // Add this context as anchor.
                if (data[i].anchors.indexOf(cid) === -1) {
                    data[i].anchors.push(cid);
                };
                // Set the data as dirty data.
                course.storage.setData(data[i], isdirty);
                if (data[i].contentinfo && data[i].contentinfo.isoutbox) {
                    course.storage.sendOutbox(data[i].contentinfo);
                };
                course.newcoursedata(data[i]);
            };
            course.feedRefresh();
        });
        /**
         * Save content to the updatestorage.
         * Use data that is saved in the course.storage
         */
        this.place.off('setupdatecontent').on('setupdatecontent', function(event, data, isdirty){
            event.stopPropagation();
            event.preventDefault();
            if (typeof(data) === 'object' && typeof(data.length) === 'undefined'){
                data = [data];
            }
            var stdata;
            for (var i = 0, len = data.length; i < len; i++) {
                stdata = course.storage.getData({name: data[i].name, lang: data[i].lang, contentType: data[i].contentType});
                course.updatestorage.setData(stdata, isdirty);
                if (stdata.contentinfo && stdata.contentinfo.isoutbox) {
                    course.updatestorage.sendOutbox(stdata.contentinfo);
                };
            };
            course.place.find('[data-appname]').trigger('updates_available');
        });
        this.place.off('setsavecontent').on('setsavecontent', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            course.place.trigger('setcontent', [data, false]); // false = no isdirty
            course.place.trigger('setupdatecontent', [data, false]);
            course.saveStorage();
        });
        /******
         * Mark element for sending, if given, and send all content from outbox.
         ******/
        this.place.on('sendcontent', function(event, contentinfo){
            event.stopPropagation();
            if (contentinfo) {
                course.storage.sendOutbox(contentinfo);
            };
            course.sendFromOutbox();
        });
        /******
         * Save all dirty content from storage
         ******/
        this.place.on('savecontent', function(event, isreadonly){
            event.stopPropagation();
            course.saveContent(isreadonly);
        });
        /******
         * Handle 'closeappok'events from children applications
         ******/
        this.place.on('closeappok', function(event, data){
            event.stopPropagation();
            // TODO count closings and trigger 'closeok', when all children apps are closed.
            course.appClosed(data);
        });
        /******
         * Handle 'closeok' events from children applications
         ******/
        this.place.on('closeok', function(event, data){
            if (event.target !== this) {
                event.stopPropagation();
                // TODO count closings and trigger 'closeok', when all children apps are closed.
            };
        });
        /**
         * Show the feed app
         */
        this.place.on('showfeed', function(event, data){
            event.stopPropagation();
            var feedapp = course.place.find('[data-apptype="coursefeed"]');
            var appid = feedapp.attr('data-appid');
            course.toggleApp(appid, true);
        });
        /**
         * Show marginnotes, if annotation is clicked.
         */
        this.place.on('click', '.commentorhighlightelement', function(event, data) {
            var marginapp = course.place.find('[data-apptype="marginnotes"]');
            var appid = marginapp.attr('data-appid');
            course.toggleApp(appid, true);
        });
        /******
         * Ask feed to show the message with given name.
         ******/
        this.place.on('showmessage', function(event, data){
            event.stopPropagation();
            var feedapp = course.place.find('[data-apptype="coursefeed"]');
            var appid = feedapp.attr('data-appid');
            course.toggleApp(appid, true);
            feedapp.children('.virumcourse-appcontent').trigger('showmessage', data.data);
        });
        this.place.on('warningnotifhandler', function(event, data){
            event.stopPropagation();
            var trigdata = data.data;
            var targetid = trigdata.appid || trigdata.contextid;
            var targettype = (trigdata.contextid ? 'contextid' : (trigdata.appid ? 'appname' : ''));
            for (var i = 0, len = trigdata.events.length; i < len; i++) {
                course.place.find('[data-'+targettype+'="'+trigdata.appid+'"]').trigger(trigdata.events[i], [trigdata.data[i]])
            };
        });
        /**
         * Reply from contentdatasave (and storagesave)
         */
        this.place.on('reply_contentdatasave reply_storagesave', function(event, message) {
            event.stopPropagation();
            message = message || [];
            if (typeof(message) === 'object' && typeof(message.length) === 'undefined') {
                message = [message];
            };
            for (var i = 0, len = message.length; i < len; i++) {
                if (message[i].success) {
                    course.dataSaved(message[i]);
                } else {
                    course.dataSaveFailed(message[i]);
                };
            };
        });
        /**
         * Reply from contentdatasend
         */
        this.place.on('reply_contentdatasend', function(event, message) {
            event.stopPropagation();
            message = message || [];
            if (typeof(message) === 'object' && typeof(message.length) === 'undefined') {
                message = [message];
            };
            for (var i = 0, len = message.length; i < len; i++) {
                if (message[i].success) {
                    course.dataSent(message[i]);
                } else {
                    course.dataSendFailed(message[i]);
                };
            };
        });
        this.place.on('closeapp', function(event){
            event.stopPropagation();
            course.close();
        });
        /**
         * Set the serversync property of the context.
         */
        this.place.on('serversync', function(event, data) {
            event.stopPropagation();
            course.setServerSync(data);
        });
        /******
         * Go to link
         ******/
        this.place.on('gotolink', function(event, data){
            if (data.data && data.data.link && (
                    (data.data.link.contextid && data.data.link.contextid === course.context.id) ||
                    (data.data.link.courseid && data.data.link.courseid === course.course.id)
                )) {
                event.stopPropagation();
                course.toggleAppFullscreen('', false);
                course.goToLink(data);
            };
        });
        /**
         * Update the title
         */
        this.place.on('titledata', function(event, titledata) {
            event.stopPropagation();
            event.preventDefault();
            var title = [];
            if (titledata.shorttitle) {
                title.push(titledata.shorttitle);
            };
            if (titledata.title) {
                title.push(titledata.title);
            };
            course.setTitle(title.join(' – ') || '');
        });
        /**
         * Update the context about changes in application
         */
        this.place.on('appdatachanged', function(event, appdata) {
            event.stopPropagation();
            event.preventDefault();
            course.appChangedHandler(appdata);
        });
        /**
         * Update the context about changes in it (coming from other instances/windows)
         */
        this.place.on('apply_contextchange', function(event, appdata) {
            event.stopPropagation();
            event.preventDefault();
            course.appChangedHandler(appdata, true /* stopbubble*/);
        });
        /**
         * Homeassignment requests
         */
        this.place.on('homeassignmentdata', function(event, data) {
            event.stopPropagation();
            course.place.trigger('showfeed');
            course.place.find('[data-apptype="coursefeed"] .virumcourse-appcontent').trigger('homeassignmentdata', [data]);
        });
        /**
         * Request for message composing
         */
        this.place.on('composemessagedata', function(event, data) {
            event.stopPropagation();
            course.place.trigger('showfeed');
            course.place.find('[data-apptype="coursefeed"] .virumcourse-appcontent').trigger('composenewmessage', [data]);
        })
        /******
         * Request for list of users.
         ******/
        this.place.on('getuserlist', function(event, data){
            event.stopPropagation();
            var users = course.getUserlist();
            var target = $(event.target);
            target.trigger('userlist', users);
        });
        /******
         * Get the currently focused content element when moving mouse over menu
         ******/
        this.place.off('focusout.coursecurrentfocus').on('focusout.coursecurrentfocus', function(event){
            course.currentFocus = $(event.target);
        });
        /******
         * Elementpanelevents for elementpanel
         ******/
        this.place.off('elementpanelevent').on('elementpanelevent', function(event, data){
            event.stopPropagation();
            var uilang = course.settings.uilang;
            switch (data.eventname){
                case 'paletaddelem':
                    var lastonpage = course.place.find('.notebook-viewarea').first().find('.elementset-droptarget').last();
                    var focused = course.currentFocus || lastonpage;
                    focused = (focused.is(':visible') ? focused : lastonpage);
                    var elset;
                    if (focused && typeof(focused.length) === 'number' && focused.length > 0 && focused.is(':visible')) {
                        var droptarget, wrapper, elset;
                        wrapper = focused.closest('.elementset-elementwrapper');
                        elset = focused.closest('.ebook-elementset');
                        if (focused.is('.elementset-droptarget')) {
                            droptarget = focused;
                        } else {
                            if (wrapper.length > 0) {
                                droptarget = wrapper.next('.elementset-droptarget');
                            } else if (elset.length > 0) {
                                droptarget = elset.find('> .ebook-elementset-body > .elementset-droptarget').last();
                            } else {
                                droptarget = lastonpage;
                                elset = droptarget.closest('.ebook-elementset');
                            };
                        }
                        var index = droptarget.attr('data-dropindex') | 0;
                        elset.trigger('addelement', {index: index, data: data.data});
                    } else {
                        course.place.trigger('notification', {
                            "id": "noplaceforadding-" + (new Date()).getTime(),
                            "nclass": "quick",
                            "message": ebooklocalizer.localize('virumcourse:nofocusforadding', uilang)
                        });
                    };
                    break;
                default:
                    break;
            }
        });
        
        this.place.on('click', '.virumcourse-applist-right .virumcourse-applistitem', function(event, data) {
            event.stopPropagation();
            var item = $(this);
            var appid = item.attr('data-appid');
            course.toggleApp(appid);
        });
        
        this.place.on('click', '.virumcourse-apptopbar button[data-action="togglemaximize"]', function(event, data) {
            event.stopPropagation();
            var item = $(this).closest('.virumcourse-app');
            var appid = item.attr('data-appid');
            item.trigger('toggleappfullscreen', [{appid: appid}]);
        });
        
        this.place.on('click', '.virumcourse-apptopbar button[data-action="togglewindowize"]', function(event, data) {
            event.stopPropagation();
            var item = $(this).closest('.virumcourse-app');
            var appid = item.attr('data-appid');
            item.trigger('toggleappwindowize', [{appid: appid}]);
        });
        
        this.place.on('toggleappfullscreen', function(event, data) {
            event.stopPropagation();
            var appid = data.appid || '';
            course.toggleAppFullscreen(appid);
        });
        
        this.place.on('toggleappwindowize', function(event, data) {
            event.stopPropagation();
            var appid = data.appid || '';
            course.toggleAppWindowize(appid);
        });
        
        /**
         * Some app has resized. Trigger resizing for others
         */
        this.place.on('appresized', function(event, data) {
            event.stopPropagation();
            var appname = data.appname;
            course.place.find('[data-apptype]:not([app-name="'+appname+'"]) .virumcourse-appcontent').trigger('resizerequest', data);
        });
        
        /**
         * Start videoconferencing
         */
        this.place.on('videoconf_start', function(event){
            event.stopPropagation();
            event.preventDefault();
            var username = course.settings.username;
            var data = {
                contextid: course.context.id,
                contexttype: course.context.type,
                action: 'videoconf_start',
                username: username,
                fullname: ''
            };
            if (course.users) {
                data.fullname = course.users.getRealname(username);
            };
            course.place.trigger('startvideocall', [data]);
        });
        
        /**
         * Get margin information
         */
        this.place.on('marginannouncement', function(event, data) {
            event.stopPropagation();
            course.margin.getMargin = data.data.getMargin;
            course.margin.clearMargin = data.data.clearMargin;
        }).on('getmarginplace', function(event, data){
            event.stopPropagation();
            var place = $(event.target);
            if (typeof(course.margin.getMargin) === 'function') {
                place.trigger('reply_getmarginplace', {getMargin: course.margin.getMargin});
            };
        }).on('pageredraw', function(event, data){
            event.stopPropagation();
            if (typeof(course.margin.clearMargin) === 'function') {
                var appid = data.appid || '';
                course.margin.clearMargin(appid);
            };
        }).on('setmarginnotes_on', function(event, data){
            event.stopPropagation();
            course.place.find('[data-apptype="notebook"] > .virumcourse-appcontent, [data-apptype="reader"] > .virumcourse-appcontent').trigger('marginnotes_on', [course.margin]);
        }).on('setmarginnotes_off', function(event, data){
            event.stopPropagation();
            course.place.find('[data-apptype="notebook"] > .virumcourse-appcontent, [data-apptype="reader"] > .virumcourse-appcontent').trigger('marginnotes_off');
        }).on('setmarginnotes_edit_on', function(event, data){
            event.stopPropagation();
            course.place.find('[data-apptype="notebook"] > .virumcourse-appcontent, [data-apptype="reader"] > .virumcourse-appcontent').trigger('marginnotes_edit_on');
        }).on('setmarginnotes_edit_off', function(event, data){
            event.stopPropagation();
            course.place.find('[data-apptype="notebook"] > .virumcourse-appcontent, [data-apptype="reader"] > .virumcourse-appcontent').trigger('marginnotes_edit_off');
        });
        
        /******
         * Request for file export (show dialog and save the given file)
         ******/
        this.place.on('exportfileas', function(event, data){
            $('.virumcourse-popupdialog').remove();
            var dialog = $('<div class="virumcourse-popupdialog" style="display: none;"><a class="ffwidget-button" download="' + data.filename + '" href="' + data.objecturl + '">Download</a></div>');
            jQuery('body').append(dialog);
            var mevent = new MouseEvent('click', {view: window, publes: true, cancelable: false});
            dialog.find('a').get(0).dispatchEvent(mevent);
            dialog.on('click', function(event){$(this).remove();});
        });
        /******
         * Request for file import (show dialog and open file)
         ******/
        this.place.on('importfile', function(event, data){
            var nbelement = $(event.target);
            $('.virumcourse-popupdialog').remove();
            var dialog = $('<div class="virumcourse-popupdialog" style="display: none;"><label class="ffwidget-button"><input type="file" name="importfile" style="display: none;"><span><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-folder"><path stroke="none" fill="black" d="M2 6 l5 0 l2 2 l14 0 l0 2 l-16 0 l-4 14 l5 -12 l22 0 l-5 13 l-23 0z"></path></svg></span></input></label><br/><button class="virumcourse-popupclose ffwidget-button">Close</button></div>');
            $('body').append(dialog);
            var mevent = new MouseEvent('click', {view: window, publes: true, cancelable: false});
            dialog.find('label').click();
            dialog.find('input').on('change', function(event){
                var thefile = event.target.files[0];
                if (thefile.type === 'application/json' || thefile.type === '') {
                    var reader = new FileReader();
                    reader.onload = function(ev){
                        var jsondata;
                        try {
                            jsondata = JSON.parse(ev.target.result);
                        } catch (err) {
                            jsondata = {};
                            console.log('parse error: ', err);
                        };
                        if (jsondata.type === 'book') {
                            jsondata.settings = {
                                username: course.settings.username,
                                lang: course.settings.lang.split(/[-_]/)[0],
                                uilang: course.settings.uilang.split(/[-_]/)[0],
                                role: course.settings.role,
                                readonly: course.settings.readonly,
                                users: course.users,
                                elementpanel: false,
                                hasparent: true
                            };
                            var bookname = jsondata.name;
                            jsondata.configs = JSON.parse(JSON.stringify(course.getConfig(bookname)));
                            // If we can import, we are already in 'devmode'. So, stay in devmode.
                            jsondata.settings.devmode = true;
                            nbelement.notebookview(jsondata);
                            dialog.remove();
                        } else {
                            alert('The file was invalid format');
                        }
                    };
                    reader.readAsText(thefile);
                }
            });
            dialog.on('click', 'button.virumcourse-popupclose', function(event){dialog.remove();});
        });
        /******
         * Configs of some sub application have changed
         ******/
        this.place.on('config_changed', function(event, data){
            var ctype = data.type;
            var key = data.key;
            var config = data.config;
            course.configs[key] = config;
            course.saveConfigs();
        }).on('registerconfigdialog', function(event, data){
            event.stopPropagation();
            course.configforms[data.appid] = data.confform;
        }).on('saveconfigs', function(event, data){
            if (event.target !== this) {
                event.stopPropagation();
                course.saveConfigs(data);
            };
        }).on('formdialog-valuechange', function(event, data){
            event.stopPropagation();
            var dialogdata = data.dialogdata;
            var apps = dialogdata.formsets;
            var values = dialogdata.values;
            var appname, confs;
            for (var i = 0, len = apps.length; i < len; i++){
                appname = apps[i];
                confs = values[appname];
                course.place.find('[data-appname="'+appname+'"]').trigger('setconfigs', confs);
            };
        });
        /**
         * Get the information about shareables
         */
        this.place.on('getshareables', function(event, data) {
            event.stopPropagation();
            var target = $(event.target);
            var shareables = [];
            var appshares;
            for (var appid in course.shareables) {
                appshares = course.shareables[appid];
                for (var i = 0, len = appshares.length; i < len; i++) {
                    shareables.push(JSON.parse(JSON.stringify(appshares[i])));
                };
            };
            target.trigger('reply_getshareables', [shareables]);
        }).on('register_shareables', function(event, data){
            event.stopPropagation();
            course.registerShareables(data);
        }).on('shareevent', function(event, data){
            event.stopPropagation();
            course.place.find('.virumcourse-apparea > [data-apptype][data-appid="'+data.appid+'"] > .virumcourse-appcontent').trigger(data.event);
        }).on('pagesharedonline', function(event, data){
            event.stopPropagation();
            course.place.find('.virumcourse-apparea > [data-apptype="sharer"] > .virumcourse-appcontent').trigger('sharedone', data);
        }).on('click', '.virumcourse-configbutton', function(event, data){
            event.stopPropagation();
            course.showConfigs();
        });
    };
    
    /******
     * Get config data of given app
     * @param {String} appname - id of the app
     ******/
    VirumCourse.prototype.getConfig = function(appname){
        return $.extend(true, {}, this.configs[appname]);
    };
    
    /**
     * Get contentinfo from storage by anchors or by names and return to asking application
     * @param {Object} query           Data for the query in the storage
     * @param {String[]} query.contentType   A list of content types to search for
     * @param {String[]} [query.anchors]     A list of anchors to search for or
     * @param {String[]} [query.names]       A list of element id's to search for
     * @param {String[]} query.lang          The language of searched elements
     * @param {jQuery} returntarget          The jQuery-element where to trigger the reply
     */
    VirumCourse.prototype.getContentinfo = function(query, returntarget) {
        var ctype, result;
        for (var i = 0, len = query.contentType.length; i < len; i++) {
            ctype = query.contentType[i];
            if (query.anchors) {
                // Query is by anchors
                result = this.storage.getContentInfoByAnchor({
                    contentType: ctype,
                    lang: query.lang,
                    anchors: query.anchors
                });
            } else if (query.names) {
                result = this.storage.getContentInfoByName({
                    contentType: ctype,
                    lang: query.lang,
                    names: query.names
                });
            } else {
                result = {contentType: ctype, contentinfo: {}};
            };
            returntarget.trigger('reply_getcontentinfo', result);
        };
    };
    
    /**
     * Get content from storage by anchors or by names and return to asking application
     * @param {Object} query           Data for the query in the storage
     * @param {String[]} query.contentType   A list of content types to search for
     * @param {String[]} [query.anchors]     A list of anchors to search for or
     * @param {String[]} [query.names]       A list of element id's to search for
     * @param {String[]} query.lang          The language of searched elements
     * @param {jQuery} returntarget          The jQuery-element where to trigger the reply
     * @param {Boolean} isinit               This is initing request. Relay this back to the target.
     */
    VirumCourse.prototype.getContent = function(query, returntarget, isinit) {
        var ctype, result;
        for (var i = 0, len = query.contentType.length; i < len; i++) {
            ctype = query.contentType[i];
            if (query.anchors) {
                // Query is by anchors
                result = this.storage.getContentByAnchor({
                    contentType: ctype,
                    lang: query.lang,
                    anchors: query.anchors
                });
            } else if (query.names) {
                result = this.storage.getContentByName({
                    contentType: ctype,
                    lang: query.lang,
                    names: query.names
                });
            } else {
                result = {contentType: ctype, contentdata: {}};
            };
            returntarget.trigger('reply_getcontent', [result, isinit]);
        };
    };

    /**
     * Parse and validate data in updateparts. Check that contentType matches with the
     * type given inside the data.
     * @param {Object} updates  - updated data ordered by contentType.
     *                            Each contentType has an array of JSON.stringified
     *                            elementobjects.
     * @return {Array} An array of JSON.parsed and typechecked element data.
     */
    VirumCourse.prototype.validateUpdates = function(updates) {
        var contentType, carr, i, len, jsonelem, element;
        var result = [];
        for (contentType in updates) {
            carr = updates[contentType];
            for (i = 0, len = carr.length; i < len; i++) {
                try {
                    element = JSON.parse(carr[i]);
                } catch(err) {
                    console.log('Error: Not valid updates', err);
                    element = null;
                };
                if (element && element.contentType === contentType && element.contentinfo && element.contentinfo.contentType === contentType) {
                    result.push(element);
                };
            };
        };
        return result;
    }
    
    /**
     * Get the savedata (all dirty elements)
     * @param {Boolean} readonly - if the notebook is in readonly mode
     * @returns {Array} an array of all drity elements
     */
    VirumCourse.prototype.getSavedata = function(readonly){
        var ctypes = this.storage.getContentTypes();
        if (readonly) {
            var nbindex = ctypes.indexOf('notebookcontent');
            ctypes.splice(nbindex, 1);
        };
        var ctype;
        var savearray = [];
        var dirtydata;
        for (var i = 0, len = ctypes.length; i < len; i++) {
            // Give the outer system a list of objects to save.
            ctype = ctypes[i];
            dirtydata = this.storage.getDirtyContent({contentType: ctype});
            for (var j = 0, dirlen = dirtydata.length; j < dirlen; j++) {
                savearray.push(dirtydata[j]);
            };
        };
        return savearray;
    };
    
    /**
     * Get the senddata from outbox
     * @returns {Array} an array of element datas to be sent
     */
    VirumCourse.prototype.getSenddata = function() {
        var ctypes = this.storage.getContentTypes();
        var ctype;
        var sendarray = [], senddata;
        var obdata;
        for (var i = 0, len = ctypes.length; i < len; i++) {
            // Send as an array with objects with needed attributes and the content data.
            // Outer systen can then just process the list.
            ctype = ctypes[i];
            obdata = this.storage.getOutboxContent({contentType: ctype});
            for (var j = 0, oblen = obdata.length; j < oblen; j++) {
                // Mark all elements as sent and dirty.
                // (If sending is successfull, that is true and the data in server is correct.)
                // (If element comes from server as update, it is dirty, i.e., must be saved locally)
                obdata[j].contentinfo.isoutbox = false;
                senddata = {
                    contexttype: this.context.type,
                    contextid: this.context.id,
                    subcontext: obdata[j].contentinfo.subcontext || this.context.id,
                    dataid: obdata[j].name,
                    data: JSON.stringify(obdata[j]),
                    send_to: obdata[j].contentinfo.sendto || [],
                    datatype: obdata[j].contentType,
                    datamodified: obdata[j].contentinfo.timestamp,
                    lang: obdata[j].contentinfo.lang
                };
                sendarray.push(senddata);
            };
        };
        return sendarray;
    };

    /******
     * Save all dirty content from storages
     * @param {Boolean} readonly    True, if the notebook is in readonly mode
     ******/
    VirumCourse.prototype.saveContent = function(readonly){
        var savearray = this.getSavedata(readonly);
        var sendarray = this.getSenddata();
        if (savearray.length + sendarray.length > 0) {
            this.place.trigger('contentdatasave', [savearray, sendarray, {contexttype: this.context.type, contextid: this.context.id, apptype: this.context.type, appid: this.course.id}]);
        };
    };
    
    /******
     * Save the whole storage as it is.
     ******/
    VirumCourse.prototype.saveStorage = function(){
        // Get an array of all dirty elements
        var dirties = this.storage.getDirtyContent();
        var dirtylist = [];
        var dirty;
        for (var i = 0, len = dirties.length; i < len; i++) {
            dirtylist.push({
                success: true,
                data: dirties[i]
            });
        };
        // Get the data of the storage
        var storage = this.getStorageAllData();
        // Give the data whole storage to the outer system for saving.
        this.place.trigger('storagesave', {apptype: this.context.type, appid: this.context.id, storage: storage, dirtylist: dirtylist});
    }
    
    /******
     * Send all content from outbox
     ******/
    VirumCourse.prototype.sendFromOutbox = function(){
        var sendarray = this.getSenddata();
        if (sendarray.length > 0) {
            this.place.trigger('contentdatasend', [sendarray]);
        };
    };
    
    /******
     * Set data to storage
     * @param {Object} contentdata - contentdata to be stored.
     *     contentdata = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     ******/
    VirumCourse.prototype.setStorageData = function(contentdata) {
        this.storage.setData(contentdata);
    };
    
    /**
     * Check if the storage is dirty
     * @returns {boolean} dirtyness of storage
     */
    VirumCourse.prototype.isDirty = function() {
        return this.storage.isDirty();
    };
    
    /******
     * Get data from storage
     * @param {Object} query - object that tells what to get from storage.
     *     query = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *     }
     * @returns {Object} - object with all data of element in same format as given to .setStorageData(data).
     *      {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *      }
     ******/
    VirumCourse.prototype.getStorageData = function(query){
        result = this.storage.getData(query);
        return result;
    };
    
    /******
     * Get all the data from the storage
     ******/
    VirumCourse.prototype.getStorageAllData = function() {
        return this.storage.getAllData();
    };
    
    /**
     * Data is saved. Mark dirty elements clean
     * @param {Object} item                The reply message of successfull saving
     * @param {Boolean} item.success       Saving was successfull. Always true
     * @param {Object|Object[]} item.data     The data for identifying the element to be cleared
     * @param {String} item.data[].name         The id of the content
     * @param {String} item.data[].contentType  The type of the content
     * @param {Object} item.data[].contentdata  The data of the content
     * @param {Object} item.data[].contentinfo  The info of the content
     * @param {String} [item.message]      The optional message for notifications.
     */
    VirumCourse.prototype.dataSaved = function(item) {
        var data = item.data;
        var message = item.message || '';
        if (typeof(data.length) === 'undefined') {
            data = [data];
        };
        for (var i = 0, len = data.length; i < len; i++) {
            this.storage.cleanDirty(data[i].contentinfo);
            switch (data.contentType) {
                case 'notebookcontent':
                    var trigdata = {
                        name: data[i].name,
                        lang: data[i].contentinfo.lang,
                        timestamp: data[i].contentinfo.timestamp,
                        type: data[i].contentdata && data[i].contentdata.type || ''
                    };
                    this.place.find('[data-apptype="notebook"]').trigger('datasaved', [trigdata]);
                    break;
                default:
                    break;
            };
        };
        this.notify({
            "nclass": "localsave",
            "message": message || "Save done.",
            "state": "black",
            "ttl": 300
        });
    };
    
    /**
     * Data save failed
     * @param {Object} item                The reply message of failed saving
     * @param {Boolean} item.success       Saving failed. Always false
     * @param {Object|Object[]} item.data     The data for identifying the element to be cleared
     * @param {String} item.data[].name         The id of the content
     * @param {String} item.data[].contentType  The type of the content
     * @param {Object} item.data[].contentdata  The data of the content
     * @param {Object} item.data[].contentinfo  The info of the content
     * @param {String} [item.message]      The optional message for notifications.
     */
    VirumCourse.prototype.dataSaveFailed = function(item) {
        this.notify({
            "nclass": "localsave",
            "message": item.message || "Could not save data. Trying later.",
            "state": "red",
            "ttl": 1000
        });
    };
    
    /**
     * Data is sent
     * @param {Object} item                The reply message of successfull sending
     * @param {Boolean} item.success       Sending was successfull. Always true
     * @param {Object|Object[]} item.data     The data for identifying the element to be cleared
     * @param {String} item.data[].name         The id of the content
     * @param {String} item.data[].contentType  The type of the content
     * @param {Object} item.data[].contentdata  The data of the content
     * @param {Object} item.data[].contentinfo  The info of the content
     * @param {String} [item.message]      The optional message for notifications.
     */
    VirumCourse.prototype.dataSent = function(item) {
        var uilang = this.settings.uilang;
        var data = item.data;
        var message = item.message;
        if (typeof(data) === 'string') {
            try {
                data = JSON.parse(data);
            } catch (err) {
                data = [];
            };
        };
        if (typeof(data.length) === 'undefined') {
            data = [data];
        };
        var ctypes = {};
        var citem;
        for (var i = 0, len = data.length; i < len; i++) {
            citem = data[i];
            ctypes[citem.contentType] = true;
            this.storage.cleanOutbox(citem.contentinfo);
            switch (citem.contentType) {
                case 'message':
                case 'hamessage':
                    this.place.find('[data-apptype="coursefeed"] .virumcourse-appcontent').trigger('sendsuccess', citem);
                    this.refreshFeed = true;
                    break;
                default:
                    break;
            };
        };
        this.feedRefresh();
        this.notify({
            "nclass": "network",
            "message": message || "Sending done.",
            "state": "black",
            "ttl": 300
        });
    };
    
    /**
     * Data send failed
     * @param {Object} item                The reply message of failed sending
     * @param {Boolean} item.success       Sending failed. Always false
     * @param {Object|Object[]} item.data     The data for identifying the element to be cleared
     * @param {String} item.data[].name         The id of the content
     * @param {String} item.data[].contentType  The type of the content
     * @param {Object} item.data[].contentdata  The data of the content
     * @param {Object} item.data[].contentinfo  The info of the content
     * @param {String} [item.message]      The optional message for notifications.
     */
    VirumCourse.prototype.dataSendFailed = function(item) {
        var uilang = this.settings.uilang;
        var data = item.data;
        var message = item.message;
        if (typeof(data) === 'string') {
            try {
                data = JSON.parse(data);
            } catch (err) {
                data = [];
            };
        };
        if (typeof(data.length) === 'undefined') {
            data = [data];
        };
        var ctypes = {};
        var citem;
        for (var i = 0, len = data.length; i < len; i++) {
            citem = data[i];
            switch (citem.contentType) {
                case 'message':
                case 'hamessage':
                    this.place.find('[data-apptype="coursefeed"] .virumcourse-appcontent').trigger('sendfail', citem);
                    break;
                default:
                    break;
            };
        };
        this.feedRefresh();
        this.notify({
            "nclass": "network",
            "message": message || "Could not send data. Trying later.",
            "state": "red",
            "ttl": 1000
        });
    };
    
    /**
     * Trigger feed refresh
     */
    VirumCourse.prototype.feedRefresh = function() {
        if (this.refreshFeed) {
            this.place.find('[data-apptype="coursefeed"] .virumcourse-appcontent').trigger('coursefeedrefresh');
            this.refreshFeed = false;
        };
    };
    
    /******
     * Start/Stop updatepoll
     ******/
    VirumCourse.prototype.startStopUpdatePoll = function(){
        var udpoll = this.settings.updatepoll;
        var timeout = (typeof(udpoll) === 'number' && udpoll !== 0 ? udpoll : 120000); // millisecons === 2 minutes
        if (this.updatepollid) {
            cancelInterval(this.updatepollid);
        }
        var course = this;
        this.requestUpdates();
        if (this.settings.updatepoll) {
            this.updatepollid = setInterval(function(){
                course.requestUpdates();
            }, timeout);
        };
    };
    
    /******
     * Go to the link inside the course
     * @param {Object} linkdata - data of the link destination
     ******/
    VirumCourse.prototype.goToLink = function(linkdata){
        var link = linkdata.data && linkdata.data.link;
        delete link.courseid;
        delete link.contextid;
        if (link.appname) {
            var appplace = this.place.find('.virumcourse-apparea [data-appname="'+link.appname+'"]');
            appplace.trigger('gotolink', linkdata);
        };
    };
    
    /******
     * Request updates from outer system.
     ******/
    VirumCourse.prototype.requestUpdates = function(){
        this.place.trigger('updaterequest', {contextid: this.context.id, contexttype: this.context.type, courseid: this.course.id, appid: this.context.id, apptype: this.context.type});
    };
    
    /******
     * Get lists of users in this course (teachers, students,...?)
     * @returns {Object} object of format: {teachers: [{...},{...}], students: [{...},{...}]}
     ******/
    VirumCourse.prototype.getUserlist = function(){
        return this.users.getUserList();
    }
    
    /******
     * Get realname for given username
     * @param {String} username - username
     * @returns {String} the realname for given username
     ******/
    VirumCourse.prototype.getRealname = function(username){
        return this.users.getRealname(username);
    }
    
    /******
     * Show all pending notifications
     ******/
    VirumCourse.prototype.showPendingNotifs = function(){
        while (this.pendingnotifications.length > 0){
            this.notify(this.pendingnotifications.shift());
        };
    };
    
    /**
     * Register shareables
     * @param {Object} shares   Data about shareables
     */
    VirumCourse.prototype.registerShareables = function(shares) {
        var appid = shares.appid;
        var apptype = shares.apptype;
        var shareables = shares.shareables;
        this.shareables[appid] = shareables;
    };
    
    /**
     * Handle application change activities
     * @param {Object} data   Data for changed things in the application
     * @param {Boolean} stopbubble    Don't "bubble" the iformation outside this context.
     */
    VirumCourse.prototype.appChangedHandler = function(data, stopbubble) {
        var appid = data.info.appid;
        var apptype = data.info.apptype;
        var changes = data.data;
        var item, key, chtype;
        var lang = this.settings.lang.split(/[-_]/)[0];
        var uilang = this.settings.uilang.split(/[-_]/)[0];
        switch (apptype) {
            case 'notebook':
                // Change comes from a notebook
                for (var i = 0, len = changes.length; i < len; i++) {
                    item = changes[i];
                    key = item.key;
                    chtype = item.chtype;
                    switch (key) {
                        case 'titles':
                            // Titles are changed
                            if (chtype === 'info' || chtype === 'edit') {
                                var title = item.data['titles'][lang] || item.data['titles'].common || '';
                                if (this.context.type === 'own_notes' && item.data.titles) {
                                    // Context is own_notes, i.e., use same titles for the context as for notebook.
                                    if (title) {
                                        // Show the title at the top of the context.
                                        this.setTitle(title);
                                    };
                                    this.context.titles = $.extend(true, this.context.titles, item.data.titles);
                                    if (chtype === 'edit' && !stopbubble) {
                                        // If the title of the context was edited, then send it aout as information.
                                        this.place.trigger('contextchanged', {
                                            contextid: this.context.id,
                                            contexttype: this.context.type,
                                            info: {
                                                contextid: this.context.id,
                                                contexttype: this.context.type,
                                                appid: appid,
                                                apptype: apptype,
                                            },
                                            data: [{
                                                key: key,
                                                chtype: chtype,
                                                data: {
                                                    titles: this.context.titles
                                                }
                                            }]
                                        });
                                    };
                                };
                                // Update the application's title, wherever it is shown.
                                this.setAppTitle({appid: appid, title: title});
                            };
                            break;
                        default:
                            break;
                    };
                };
                break;
            default:
                break;
        };
    };
    
    /******
     * Save configs either locally (localStorage) or in parent module.
     * @param {Object} data - Optional config data to be saved.
     *                        If not given, just save all configs.
     *                        format: {appname: '<application id>', configs: {<configs to be saved>}}
     ******/
    VirumCourse.prototype.saveConfigs = function(data){
        if (this.configable) {
            // Save given data or current data in this module
            if (typeof(data) !== 'undefined') {
                // Add given config data
                var appname = data.appname;
                var configs = data.configs;
                this.configs[appname] = configs;
            }
            // Save all config data
            var confstring = JSON.stringify(this.configs);
            try {
                localStorage.setItem('virumcourse_configs', confstring);
            } catch (err) {
                console.log('error saving virumcourse configs', err);
            };
        } else {
            // Save config data in parent module
            if (typeof(data) === 'undefined') {
                // Not given data, but save all. (Else save only given data)
                data = {
                    type: 'configsave',
                    appname: this.course.id,
                    configs: JSON.parse(JSON.stringify(this.configs))
                };
            };
            this.place.trigger('saveconfigs', data);
        }
    };
    
    /******
     * Load configs from localStorage
     * @returns {Object} config-object
     ******/
    VirumCourse.prototype.loadConfigs = function(){
        var confs = {};
        try {
            confs = JSON.parse(localStorage.getItem('virumcourse_configs')) || {};
        } catch (err){
            confs = {};
        };
        return confs;
    };
    
    /******
     * Add <style>-tag with CSS-rules for course.
     ******/
    VirumCourse.prototype.setStyles = function(){
        if ($('#virumcoursestyles').length === 0) {
            $('head').append('<style type="text/css" id="virumcoursestyle">'+VirumCourse.style+'</style>');
        };
    };

    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    VirumCourse.prototype.sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /******
     * Some icons
     ******/
    VirumCourse.icons = {
        config: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 32 32" class="mini-icon mini-icon-gear"><path style="stroke: none;" d="M12 0 h8 l0.9 3.5 l3.5 2 l3.55 -1.1 l4.1 7 l-2.7 2.5 v4.1 l2.7 2.5 l-4.1 7 l-3.55 -1.1 l-3.5 2 l-0.9 3.5 h-8 l-0.9 -3.5 l-3.5 -2 l-3.55 1.1 l-4.1 -7 l2.7 -2.5 v-4.1 l-2.7 -2.5 l4.1 -7 l3.55 1.1 l3.5 -2z m1.55 2 l-0.7 2.9 l-5 2.8 l-2.8 -0.8 l-2.5 4.3 l2.15 2 v5.65 l-2.15 2 l2.5 4.3 l2.8 -0.8 l5 2.8 l0.7 2.9 h4.9 l0.7 -2.9 l5 -2.8 l2.8 0.8 l2.5 -4.3 l-2.15 -2 v-5.65 l2.15 -2 l-2.5 -4.3 l-2.8 0.8 l-5 -2.8 l-0.7 -2.9z m2.45 7.55 a6.45 6.45 0 0 1 0 12.9 a6.45 6.45 0 0 1 0 -12.9z m0 2 a4.45 4.45 0 0 0 0 8.9 a4.45 4.45 0 0 0 0 -8.9z"></path></svg>',
        config2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="pssicon pssicon-gear"><path style="stroke: none;" d="M17.26 2.2 a13 13 0 0 1 7.7 4.44 a3 8 60 0 0 2.256 3.91 a13 13 0 0 1 0 8.89 a3 8 120 0 0 -2.256 3.91 a13 13 0 0 1 -7.7 4.44 a3 8 0 0 0 -4.52 0 a13 13 0 0 1 -7.7 -4.44 a3 8 60 0 0 -2.256 -3.91 a13 13 0 0 1 0 -8.89 a3 8 120 0 0 2.256 -3.91 a13 13 0 0 1 7.7 -4.44 a3 8 0 0 0 4.52 0 M15 11 a4 4 0 0 0 0 8 a4 4 0 0 0 0 -8 z"></path></svg>',
        warning: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-warning"><path style="stroke: none;" fill="red" d="M15 2 l13 26 h-26z m-2 10 l1 10 h2 l1 -10z m1 12 v2 h2 v-2z"></path></svg>',
        maximize: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-maximize"><path style="stroke: none;" d="M18 1 h10 a1 1 0 0 1 1 1 v10 a1 1 0 0 1 -3 0 v-6 l-8 8 a1 1 0 0 1 -2 -2 l8 -8 h-6 a1 1 0 0 1 0 -3z m-17 17 a1 1 0 0 1 3 0 v6 l8 -8 a1 1 0 0 1 2 2 l-8 8 h6 a1 1 0 0 1 0 3 h-10 a1 1 0 0 1 -1 -1z" /></svg>',
        unmaximize: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-unmaximize"><path style="stroke: none;" d="M3 16 h10 a1 1 0 0 1 1 1 v10 a1 1 0 0 1 -3 0 v-6 l-8 8 a1 1 0 0 1 -2 -2 l8 -8 h-6 a1 1 0 0 1 0 -3z m13 -13 a1 1 0 0 1 3 0 v6 l8 -8 a1 1 0 0 1 2 2 l-8 8 h6 a1 1 0 0 1 0 3 h-10 a1 1 0 0 1 -1 -1z" /></svg>',
        windowize: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-windowize"><path style="stroke: none;" d="M4 4 h22 v25 h-22z m1 4 v20 h20 v-20z" /></svg>'
    };
    
    /******
     * html-templates
     ******/
    VirumCourse.templates = {
        html: [
            '<div class="virumcourse-control ffwidget-background-colored"><div class="virumcourse-control-buttons"></div><div class="virumcourse-title ffwidget-title"></div><div class="virumcourse-control-notifications"></div><div class="virumcourse-mathpanel"></div></div>',
            '<div class="virumcourse-appset">',
            '    <div class="virumcourse-dock" data-dockorient="vertical"></div>',
            '    <ul class="virumcourse-applist virumcourse-applist-left"></ul>',
            '    <div class="virumcourse-apparea virumcourse-materialarea virumcourse-apparea-left"></div>',
            '    <div class="virumcourse-apparea virumcourse-applicationarea virumcourse-apparea-right"></div>',
            '    <ul class="virumcourse-applist virumcourse-applist-right ffwidget-background"></ul>',
            '</div>',
            '<div class="virumcourse-dock" data-dockorient="horizontal"></div>'
        ].join('\n'),
        material: [
            '<div class="virumcourse-app">',
            //'<div class="virumcourse-apptopbar">',
            //'<div class="virumcourse-apptopbar-left"></div>',
            //'<div class="virumcourse-apptopbar-right"><button></button><button></button></div>',
            //'</div>',
            '<div class="virumcourse-appcontent"></div>',
            '</div>'
        ].join('\n'),
        app: [
            '<div class="virumcourse-app">',
            '<div class="virumcourse-apptopbar ffwidget-background">',
            '<div class="virumcourse-apptopbar-left"></div>',
            '<div class="virumcourse-apptopbar-right ffwidget-buttonset ffwidget-horizontal"><button class="ffwidget-setbutton" data-action="togglewindowize"><span class="virumcourse-appbuttonicon virumcourse-windowize">'+VirumCourse.icons.windowize+'</span></button><button class="ffwidget-setbutton" data-action="togglemaximize"><span class="virumcourse-appbuttonicon virumcourse-maximize">'+VirumCourse.icons.maximize+'</span></button></div>',
            '</div>',
            '<div class="virumcourse-appcontent"></div>',
            '</div>'
        ].join('\n'),
        listitem: [
            '<li class="virumcourse-applistitem"></li>'
        ].join('\n')
    }
    
    /******
     * CSS-styles
     ******/
    VirumCourse.style = [
        '.virumcourse-wrapper {position: relative; overflow: hidden; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; font-size: 12pt;}',
        // control
        '.virumcourse-control {margin: 0; min-height: 40px; height: 40px; -webkit-flex-grow: 0; -ms-flex-grow: 0; flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0; flex-shrink: 0; background-color: #89a02c; background-color: #677821; background-color: #a02c2c; background-color: #d3bc5f; background-color: #ffd700; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; border-bottom: 1px solid transparent;}',
        '.virumcourse-control .virumcourse-title {padding: 5px 10px; flex-grow: 1; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;}',
        //'.virumcourse-control .virumcourse-title {color: black; text-shadow: 1px 1px 10px rgba(255,255,255,0.5), -1px -1px 10px rgba(255,255,255,0.5), 1px -1px 10px rgba(255,255,255,0.5), -1px 1px 10px rgba(255,255,255,0.5); font-weight: normal; font-family: "GlacialIndifference", "Helvetica Neue", Helvetica, Arial, "Lucida Grande", sans-serif; font-size: 25px; padding: 5px 10px;}',
        '.virumcourse-control .virumcourse-control-buttons {margin-left: 2em;}',
        '.virumcourse-control .virumcourse-control-button {display: inline-block; cursor: pointer;}',
        '.virumcourse-control .virumcourse-control-button:hover {background-color: rgba(255,255,255,0.5);}',
        '.virumcourse-control .virumcourse-control-notifications {min-width: 200px; text-align: right; z-index: 30;}',
        // apparea
        '.virumcourse-appset {display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; align-items: stretch; flex-grow: 1; overflow: hidden;}',
        '.virumcourse-applist {list-style: none; margin: 0; padding: 0; width: 40px; background-color: #eee; background-color: #fafafa; flex-shrink: 0; padding-top: 100px; position: relative;}',
        '.virumcourse-applist li {cursor: pointer;}',
        '.virumcourse-applist .virumcourse-appicon {text-align: center;}',
        '.virumcourse-applist .virumcourse-appicon svg {width: 30px; height: auto;}',
        '.virumcourse-applist .virumcourse-appclose {text-align: center; margin: 3px 0;}',
        '.virumcourse-applist .virumcourse-appclose .virumcourse-appclose-button {display: inline-block; width: 16px; height: 16px; line-height: 16px; vertical-align: middle; border: 1px solid #a00; background-color: rgba(170,0,0,0.7); color: #fff; box-shadow: -1px -1px 1px rgba(0,0,0,0.3), inset 1px 1px 1px rgba(255,255,255,0.5), inset -1px -1px 1px rgba(0,0,0,0.3), 1px 1px 1px rgba(255,255,255,0.5); font-weight: bold; border-radius: 50%; font-size: 12px;}',
        '.virumcourse-applist-left {width: 6px; margin-right: -6px; opacity: 0; position: relative; z-index: 2; overflow: hidden; transition: width 0.2s, opacity 0s 0.2s; box-shadow: inset -6px 0 15px rgba(0,0,0,0.2); border-right: 1px solid #aaa;}',
        '.virumcourse-applist-left:hover {width: 45px; opacity: 1; transition: width 0.2s, opacity 0s 0s, z-index 0s 0.1s; overflow: visible;}',
        '.virumcourse-applist-left li {margin: 0; padding: 8px 2px; overflow: hidden;}',
        '.virumcourse-applist-left li:hover {background-color: rgba(255,255,255,0.9); box-shadow: inset 2px 2px 3px rgba(0,0,0,0.3), inset -1px -1px 3px rgba(255,255,255,0.5);}',
        '.virumcourse-applist-left li .virumcourse-appclose {display: none;}',
        '.virumcourse-applist-left li .virumcourse-apptitle {display: none;}',
        '.virumcourse-applist-left li:hover .virumcourse-apptitle {display: block; position: absolute; left: 65px; z-index: 20; padding: 0.3em; font-family: helvetica, Arial, sans-serif; font-size: 80%; border: 1px solid black; box-shadow: 3px 3px 8px rgba(0,0,0,0.4); border-radius: 5px;}',
        '.virumcourse-applist-left li:hover .virumcourse-apptitle::before {display: block; background-color: transparent; content: ""; width: 0; height: 0; position: absolute; top: 50%; left: -20px; margin-top: -10px; border: 10px solid transparent; border-right: 20px solid #666; border-left: none;}',
        '.virumcourse-applist-right {width: 35px;}',
        '.virumcourse-applist-right:empty {width: 0;}',
        '.virumcourse-applist-right li {width: 30px; margin-left: 5px; margin-bottom: 10px; background-color: white; border-radius: 6px 0 0 6px; border: 1px solid #777; border-right: none; transition: width 0.3s, margin 0.3s; box-shadow: 0 0 5px rgba(0,0,0,0.4);}',
        '.virumcourse-applist-right li .virumcourse-appclose {display: none;}',
        '.virumcourse-applist-right li.tabactive .virumcourse-appclose {display: block;}',
        '.virumcourse-applist-right li.tabactive, .virumcourse-applist-right li:hover {width: 40px; margin-left: -5px; background-color: rgba(255,255,255,0.9);}',
        '.virumcourse-applicationarea {-webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; -webkit-flex-shrink: 1; -ms-flex-shrink: 1; flex-shrink: 0; position: relative; margin: 0; padding: 0; background-color: white; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        '.virumcourse-apparea-left {box-shadow: -3px 0 6px rgba(0,0,0,0.3); border-left: 1px solid #777;}',
        '@media screen and (max-width: 1100px) {.virumcourse-applicationarea {flex-shrink: 1;}}',
        '.virumcourse-materialarea {overflow: hidden; width: 100%; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; position: relative; margin: 0; padding: 0; background-color: white; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        // app
        '.virumcourse-app {position: relative; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; -webkit-flex-shrink: 1; -ms-flex-shrink: 1; flex-shrink: 1; width: 50%; overflow: hidden; display: flex; flex-direction: column; background-color: #ccc;}',
        '.virumcourse-app.virumcourse-appfullscreen.sidepanel {position: absolute; right: 0; top: 0; bottom: 0; width: 100%!important; z-index: 20;}',
        '.virumcourse-app.virumcourse-appwindowized.sidepanel {position: absolute; bottom: 0; right:0; z-index: 20; max-height: 70%; overflow-y: auto; border: 1px solid #999; box-shadow: -2px -2px 10px rgba(0,0,0,0.5);}',
        
        '.virumcourse-app.virumcourse-apperror {background-color: #eee; box-shadow: 0 0 5px #faa;}',
        '.virumcourse-appcontent {flex-grow: 1;}',
        
        // App topbar
        '.virumcourse-apptopbar {flex-shrink: 0; display: flex; flex-flow: row nowrap; justify-content: space-between; align-items: stretch; border-radius: 2px; box-shadow: inset 0 0 2px 1px rgba(0,0,0,0.5); padding: 2px;}',
        '.virumcourse-apptopbar button {width: 20px; height: 20px; padding: 0!important;}',
        '.virumcourse-apptopbar button svg {width: 12px; height: 12px;}',
        '.virumcourse-appbuttonicon {display: inline-block; width: 15px; height: 15px; line-height: 15px; vertical-align: middle; text-align: center;}',
        
        // sidepanel
        //'.virumcourse-app.sidepanel {position: relative; width: 28em; transition: width 0.5s;}',
        //'.virumcourse-app.sidepanel.sidepanel-hidden {width: 0; flex-grow: 0;}',
        '.virumcourse-app.sidepanel {position: relative; width: 0; transition: width 0.5s; flex-grow: 0;}',
        '.virumcourse-app.sidepanel.sidepanel-visible {width: 28em; flex-grow: 1;}',
        '.virumcourse-app.sidepanel-hidden .sidepanel-tabblock {position: absolute; top: 100px; left: -30px; width: 30px; height: 70px; background-color: rgba(255,255,255,0.7); border: 1px solid #aaa; border-right: none; border-radius: 5px 0 0 5px; transition: width 0.2s, left 0.2s, background-color 0.2s; box-shadow: -4px 4px 10px rgba(0,0,0,0.3);}',
        '.virumcourse-app.sidepanel-hidden .sidepanel-tabblock:hover {width: 40px; left: -40px; background-color: rgba(255,255,255,0.95);}',
        '.virumcourse-app.sidepanel-hidden .sidepanel-tabblock svg {width: 15px; height: 15px;}',
        
        // Popupdialog
        '.virumcourse-popupdialog {z-index: 30; position: fixed; top: 3em; left: 3em; width: 15em; height: 5em; padding: 2em; background-color: #eee; border: 1px solid black; border-radius: 1em; box-shadow: 8px 8px 12px rgba(0,0,0,0.3);}',
        '.virumcourse-popupdialog label {cursor: pointer;}',
        
        // Print css
        '@media print {',
            '.virumcourse-wrapper {position: static!important; overflow: visible; display: block;}',
            '.virumcourse-control {display: none;}',
            '.virumcourse-applist {display: none;}',
            '.virumcourse-appset {display: block; overflow: visible;}',
            '.virumcourse-apparea {display: block; overflow: visible; border: none; box-shadow: none;}',
            '.virumcourse-app {display: block; width: auto;}',
        '}'
    ].join('\n');
    
    /******
     * Default input for the course
     ******/
    VirumCourse.defaults = {
        apps: [],
        appdata: {},
        context: {
            type: "course",
            id: "",
            titles: {
                common: "Notebook"
            },
            title: "Notebook",
            material: [],
            teachers: [],
            users: {
                teachers: [],
                students: []
            },
            groups: []
        },
        storage: {},
        storageparts: [],
        updateparts: {},
        notifications: [],
        settings: {
            user: {
                username: 'Anonymous',
                firstname: '',
                lastname: ''
            },
            uilang: 'en-US',
            lang: 'en-US',
            role: 'student',
            readonly: false,
            updatepoll: 600000,
            hasparent: false,
            elementpanel: true
        },
        configs: {}
    }
    
    VirumCourse.notification = {
        classes: [
            {
                name: 'network',
                label: 'Network',
                type: 'trafficlight',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="-5 -5 40 40" class="mini-icon mini-icon-cloudup"><path style="stroke: none;" d="M5 25 a4 4 0 1 1 2 -10 a6 6 0 0 1 12 0 a4 4 0 0 1 7 3 a3 3 0 0 1 0 7z m7 -2 l4 0 l0 -3 l2 0 l-4 -4 l-4 4 l2 0z" /></svg>',
                //icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-led"><path style="stroke: none;" d="M15 5 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z"></path><path style="stroke: none; fill: rgba(255,255,255,0.2);" d="M5 15 a10 10 0 0 1 20 0 a12 12 0 0 0 -20 0z"></path><path style="stroke: none; fill: rgba(255,255,255,0.5);" d="M5 15 a10 10 0 0 1 20 0 a10.2 10.2 0 0 0 -20 0z"></path><path style="stroke: none; fill: rgba(0,0,0,0.1);" d="M5 15 a10 10 0 0 0 20 0 a12 12 0 0 1 -20 0z"></path><path style="stroke: none; fill: rgba(0,0,0,0.5);" d="M5 15 a10 10 0 0 0 20 0 a10.2 10.2 0 0 1 -20 0z"></path></svg>',
                event: 'networknotifclick',
                level: 0
            },
            {
                name: 'localsave',
                label: 'Saving',
                type: 'trafficlight',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="-7 -7 44 44" class="mini-icon mini-icon-save mini-icon-disk"><path style="stroke: none;" d="M1 1 l23 0 l5 5 l0 23 l-28 0z m5 2 l0 8 l17 0 l0 -8z m12 1 l3 0 l0 6 l-3 0z m-13 10 l0 14 l20 0 l0 -14z m3 3 l14 0 l0 2 l-14 0z m0 3 l14 0 l0 2 l-14 0z m0 3 l14 0 l0 2 l-14 0z"></path></svg>',
                event: 'localsavenotifclick',
                level: 0
            },
            {
                name: 'messages',
                label: 'Messages',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /></svg>',
                event: 'showmessage',
                level: 0
            },
            {
                name: 'warning',
                label: 'Warning',
                icon: VirumCourse.icons.warning,
                event: 'warningnotifhandler',
                level: 0
            },
            {
                name: 'quick',
                label: 'Quick messages',
                type: 'quickmsg',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /></svg>',
                event: 'quickmsgack',
                level: 0
            },
            {
                name: 'chat',
                label: 'Chat',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-chat"><path style="stroke: none;" d="M11 6 a4 4 0 0 1 4 -4 l10 0 a4 4 0 0 1 4 4 l0 2 a4 4 0 0 1 -4 4 a8 8 0 0 0 2 4 a12 12 0 0 1 -8 -4 l-4 0 a4 4 0 0 1 -4 -4z m-1 2 l0 2 a4 4 0 0 0 4 4 l6 0 a4 4 0 0 1 -4 4 l-3 0 a12 12 0 0 1 -8 4 a8 8 0 0 0 2 -4 l-2 0 a4 4 0 0 1 -4 -4 l0 -2 a4 4 0 0 1 4 -4z" /></svg>',
                event: 'showchat',
                level: 0
            },
            {
                name: 'debug',
                label: 'Debug',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 50 50" class="mini-icon mini-icon-wrench"><path style="stroke: none;" fill="black" d="M35 5 a12 12 0 1 0 9 9 l-9 9 a10 10 0 0 1 -9 -9 l9 -9z m-15 17.5 l-17 17 a2 2 0 0 0 7 7 l17 -17z m-16 18 a2 2 0 0 1 5 5 a2 2 0 0 1 -5 -5z"></path></svg>',
                event: 'showdebug',
                level: 9
            }
        ],
        level: 3,
        align: 'right'
        
    }
    
    /******
     * Localization strings
     ******/
    VirumCourse.localization = {
        "en": {
            "virumcourse:nofocusforadding": "No focused place to add element.",
            "virumcourse:configs": "Settings"
        },
        "fi": {
            "virumcourse:nofocusforadding": "Ei paikkaa, johon lisätä elementti.",
            "virumcourse:configs": "Asetukset"
        },
        "sv": {
            "virumcourse:nofocusforadding": "Ingen plats har markerats.",
            "virumcourse:configs": "Inställningar"
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(VirumCourse.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(VirumCourse.localization);
    }

    
    /***********************************************************************************************
     * Course feed
     * @constructor
     * @param {jQuery} place - DOM-place for course feed
     * @param {Object} options - data for course feed
     ***********************************************************************************************/
    var CourseFeed = function(place, options){
        options = $.extend(true, {}, CourseFeed.defaults, options);
        this.place = $(place);
        this.setStyles();
        this.addHandlers();
        this.init(options);
    };
    
    /******
     * Init the course feed
     * @param {Object} options - data for course feed
     ******/
    CourseFeed.prototype.init = function(options){
        this.name = options.name;
        this.appid = this.name;
        this.apptype = 'coursefeed';
        this.setAttrs();
        this.feed = [];
        this.filterlist = [];
        this.contentdata = options.contentdata;
        this.refinfo = options.refinfo;
        this.newMessageAnchors = {};
        this.singleopen = options.singleopen;
        this.settings = options.settings;
        this.users = options.settings.users;
        this.elinfos = $.fn.elementset('getelementtypes', 'feedelements');
        this.show();
        this.currentMessage = '';
        this.fetchMessages();
    };
    
    /******
     * Set some attributes for DOM-place
     ******/
    CourseFeed.prototype.setAttrs = function(){
        this.hidden = true;
        this.place.addClass('virumcoursefeed-wrapper sidepanel sidepanel-hidden').attr('data-appname', escapeHTML(this.name));
    };
    
    /******
     * Add filter to the filter list (show only elements matching the filter)
     * @param {String} filter - type of elements to show
     ******/
    CourseFeed.prototype.addFilter = function(filter){
        if (this.filterlist.indexOf(filter) === -1) {
            this.filterlist.push(filter);
        };
        this.filterMessageList(true);
    };
    
    /******
     * Remove a filter from the list
     * @param {String} filter - type of elements to not show
     ******/
    CourseFeed.prototype.removeFilter = function(filter){
        var index = this.filterlist.indexOf(filter);
        if (index !== -1) {
            this.filterlist.splice(index, 1);
        };
        this.filterMessageList(true);
    };
    
    /******
     * Select the feed messages matching to the filter list
     * @param {Array} msgtype - An array of strings to use as a filter
     ******/
    CourseFeed.prototype.filter = function(msgtype){
        if (typeof(msgtype) === 'undefined') {
            msgtype = this.filterlist.slice();
        }
        if (msgtype.length === 0) {
            // If list of message types to show is empty, use all available types in this.elinfos
            for (var feedtype in this.elinfos) {
                msgtype.push(feedtype);
            };
            msgtype.push('unread');
        };
        this.feed = [];
        var filterRead = (msgtype.indexOf('unread') !== -1);
        var dataitem, datainfo, unread, modified;
        for (var item in this.contentdata) {
            dataitem = this.contentdata[item];
            datainfo = this.refinfo[item];
            unread = !datainfo.read && filterRead;
            modified = (dataitem.metadata.modified && new Date(dataitem.metadata.modified) || new Date(0));
            this.feed.push({
                id: item,
                date: modified,
                filtered: (msgtype.indexOf(dataitem.type) === -1 && !unread ? true : false)
            });
        };
        this.feed.sort(function(a, b){return (a.date < b.date ? -1 : 1);})
    };
    
    /******
     * Get the number of messages of given type
     * @param {String} msgtype - name of message type
     ******/
    CourseFeed.prototype.getNumOfTypes = function(msgtype){
        var count = 0;
        var dataitem;
        if (msgtype === 'unread') {
            for (var item in this.contentdata) {
                dataitem = this.refinfo[item];
                if (!dataitem || !dataitem.read ) {
                    count += 1;
                };
            };
        } else {
            for (var item in this.contentdata) {
                dataitem = this.contentdata[item];
                if (msgtype === dataitem.type) {
                    count += 1;
                };
            };
        };
        return count;
    }
    
    /******
     * Show the course feed
     ******/
    CourseFeed.prototype.show = function(){
        var uilang = this.settings.uilang;
        this.place.html(CourseFeed.templates.html);
        this.showControlButtons();
    };
    
    /******
     * Show the messagelist
     ******/
    CourseFeed.prototype.showMessageList = function(clearall){
        var uilang = this.settings.uilang;
        this.messagelist = this.place.find('.virumcoursefeed-messagelist');
        this.messagearea = this.place.find('.virumcoursefeed-messagearea');
        if (clearall) {
            this.messagelist.empty();
        }
        this.filter();
        var message, msginfo, msg, msgmdata, elinfo, icon, ctype, sendicon, resendtitle, subject, deadline, creator, msgdate, msgdatestr, html, oldmessage, oldisopen, read, dlasses, recipients, recipientlist;
        sendicon = CourseFeed.icons.resend;
        resendtitle = ebooklocalizer.localize('coursefeed:resend_failed', uilang);
        for (var i = 0, len = this.feed.length; i < len; i++) {
            msg = this.feed[i];
            filtered = msg.filtered;
            message = this.contentdata[msg.id];
            msginfo = this.refinfo[msg.id];
            read = !!msginfo.read;
            classes = ['virumcoursefeed-messagelistitem'];
            elinfo = this.elinfos[message.type];
            msgmdata = message.metadata;
            msgdata = message.data;
            icon = elinfo && elinfo.icon || '';
            subject = this.sanitize(msgdata.subject || msgdata.title || '', {ALLOWED_TAGS: []});
            deadline = (msgdata.deadline ? (new Date(msgdata.deadline)).toLocaleDateString(uilang) : '');
            sender = msginfo.sendfrom || msgmdata.creator || '';
            sender = this.getRealname(sender) || sender;
            recipientlist = msginfo.sendto || msgdata.sendto || '';
            if (typeof(recipientlist) === 'string') {
                recipientlist = [{pubtype: 'user', to: recipientlist}];
            };
            recipients = [];
            for (var j = 0, rlen = recipientlist.length; j < rlen; j++) {
                if (recipientlist[j].pubtype === 'user') {
                    recipients.push(this.getRealname(recipientlist[j].to));
                } else if (recipientlist[j].pubtype === 'group') {
                    recipients.push(this.getGroupDescription(recipientlist[j].to));
                } else if (recipientlist[j].pubtype === 'context') {
                    recipients.push(this.getContextTitle());
                };
            };
            //recipient = this.getRealname(recipient) || (recipient === 'teacher' || recipient === 'all' ? ebooklocalizer.localize('messageelement:' + recipient, uilang) : recipient);
            msgdate = (msgmdata.modified && new Date(msgmdata.modified) || '');
            msgdatestr = (new Intl.DateTimeFormat(uilang, {weekday: 'long', year: 'numeric', month: 'numeric', day: 'numeric'}).format(msgdate));
            if (filtered) {
                classes.push('filteredmessage');
            };
            if (!read) {
                classes.push('unreadmessage');
            }
            ctype = (message.type === 'messageelement' ? 'message' : escapeHTML(message.type));
            html = [
                '<li class="'+classes.join(' ')+'" data-feedindex="'+i+'" data-feedid="'+msg.id+'" data-messagetype="'+message.type+'" tabindex="0">',
                '  <div class="virumcoursefeed-messageheader" data-outboxstatus="'+msginfo.isoutbox+'">',
                '    <div class="virumcoursefeed-messageheader-date">',
                '        <div class="virumcoursefeed-messagelist-date">('+msgdatestr+')</div>',
                '    </div>',
                '    <div class="virumcoursefeed-messageheader-title">',
                '        <div class="virumcoursefeed-messagelist-icon">'+icon+'</div>',
                '        <div class="virumcoursefeed-messagelist-subject">'+subject+'</div>',
                '    </div>',
                '    <div class="virumcoursefeed-messageheader-sendfailed virumcoursefeed-controlbutton" title="'+resendtitle+'" data-event="sendcontent" data-ctype="message">'+sendicon+'</div>',
                '    <div class="virumcoursefeed-messageheader-namedate">',
                '        <div class="virumcoursefeed-fromto">',
                '            <div class="virumcoursefeed-messagelist-sender">'+sender+'</div>',
                '            <span>&rarr;</span>',
                '            <div class="virumcoursefeed-messagelist-recipient">'+recipients.join('</div>, <div class="virumcoursefeed-messagelist-recipient">')+'</div>',
                '        </div>',
                '    </div>',
                '    <div class="virumcoursefeed-messageheader-deadline">',
                '        <div class="virumcoursefeed-messagelist-deadline">'+ebooklocalizer.localize('coursefeed:deadline', uilang)+': <span class="virumcoursefeed-messagelist-deadlinedate">'+deadline+'</span></div>',
                '    </div>',
                '  </div>',
                '  <div class="virumcoursefeed-messageitemarea"></div>',
                '</li>'
            ].join('\n');
            html = this.sanitize(html, {
                FORBID_ATTR: ['onclick', 'onerror', 'onload']
            });
            oldmessage = this.messagelist.find('li[data-feedid="'+msg.id+'"]');
            if (oldmessage.length === 0) {
                this.messagelist.prepend(html)
            } else {
                oldisopen = oldmessage.hasClass('selectedmessage');
                oldmessage.find('.virumcoursefeed-messagelist-date').html('('+ msgdatestr + ')');
                oldmessage.find('.virumcoursefeed-messageheader').attr('data-outboxstatus', escapeHTML(msginfo.isoutbox));
                if (oldisopen) {
                    this.showMessage(msg.id);
                };
            };
        };
        this.refreshFilterCounters();
    }
    
    /******
     * Filter existing messagelist
     ******/
    CourseFeed.prototype.filterMessageList = function(){
        this.filter();
        var msg, mid, filtered;
        var message;
        for (var i = 0, len = this.feed.length; i < len; i++) {
            msg = this.feed[i];
            mid = msg.id;
            filtered = msg.filtered;
            message = this.messagelist.children('li.virumcoursefeed-messagelistitem[data-feedid="'+mid+'"]');
            if (filtered) {
                message.addClass('filteredmessage');
            } else {
                message.removeClass('filteredmessage');
            };
        };
    };
    
    
    /******
     * Show the control buttons of the feed
     ******/
    CourseFeed.prototype.showControlButtons = function(){
        var uilang = this.settings.uilang;
        this.controlbuttons = this.place.find('ul.virumcoursefeed-control-buttons');
        this.controlfilters = this.place.find('ul.virumcoursefeed-control-filters');
        var html = [], item, feedtype, button, filter;
        var buttonset = CourseFeed.buttons.control || {};
        for (item in buttonset) {
            button = buttonset[item];
            if (!button.ctype || this.settings.users.canEdit({ctype: button.ctype})) {
                html.push(`<li class="virumcoursefeed-control-buttonitem virumcoursefeed-controlbutton ffwidget-button" title="${ebooklocalizer.localize(button.label, this.settings.uilang)}" data-event="${button.event}" ${(button.ctype ? 'data-ctype="'+button.ctype+'"' : '')}><span>${button.icon}</span></li>`);
            };
        };
        this.controlbuttons.html(html.join('\n'));
        var selected, count;
        count = this.getNumOfTypes('unread');
        var filterhtml = ['<li class="virumcoursefeed-control-filteritem virumcoursefeed-filterbutton ffwidget-setbutton" title="'+(ebooklocalizer.localize('coursefeed:filterunread', uilang))+'" data-filtername="unread"><span>'+CourseFeed.icons.unreadmessages+'</span><span class="virumcoursefeed-filtercount">'+count+'</span></li>'];
        for (feedtype in this.elinfos) {
            selected = (this.filterlist.indexOf(feedtype) === -1 ? '' : ' buttonselected');
            filter = this.elinfos[feedtype];
            count = this.getNumOfTypes(feedtype);
            filterhtml.push('<li class="virumcoursefeed-control-filteritem virumcoursefeed-filterbutton ffwidget-setbutton'+selected+'" title="'+(filter.description[uilang] || filter.description['en'])+'" data-filtername="'+filter.type+'"><span>'+filter.icon+'</span><span class="virumcoursefeed-filtercount">'+count+'</span></li>');
        };
        this.controlfilters.html(filterhtml.join(''));
    };
    
    /******
     * Refresh filter counts
     ******/
    CourseFeed.prototype.refreshFilterCounters = function(){
        var place;
        place = this.controlfilters.find('[data-filtername="unread"] .virumcoursefeed-filtercount');
        place.html(this.getNumOfTypes('unread'));
        for (var feedtype in this.elinfos) {
            place = this.controlfilters.find('[data-filtername="'+feedtype+'"] .virumcoursefeed-filtercount');
            place.html(this.getNumOfTypes(feedtype));
        };
    };
    
    /******
     * Show the selected message
     * @param {String} messageid - the id of the message to be shown
     ******/
    CourseFeed.prototype.showMessage = function(messageid){
        this.currentMessage = (messageid || this.feed[0].id);
        var index = 0;
        for (var i = 0, len = this.feed.length; i < len; i++) {
            if (this.feed[i].id === messageid) {
                index = i;
                break;
            };
        };
        this.markMessageRead(messageid);
        var msgitems = this.messagelist.children('li');
        if (this.signleopen) {
            msgitems.removeClass('selectedmessage').children('.virumcoursefeed-messageitemarea').empty();
        };
        var msglitem = msgitems.filter('[data-feedid="'+messageid+'"]');
        if (msglitem.length > 0) {
            msglitem.get(0).scrollIntoView({behavior: 'smooth'});
            msglitem.addClass('selectedmessage');
            if (msglitem.is('.unreadmessage')) {
                msglitem.removeClass('unreadmessage');
                this.filterMessageList();
                this.refreshFilterCounters();
            };
        };
        var uilang = this.settings.uilang;
        var msg = this.feed[index];
        var message = this.contentdata[msg.id];
        var elementinfo = this.elinfos[message.type];
        var element = $('<div></div>');
        var messagearea = msglitem.children('.virumcoursefeed-messageitemarea');
        messagearea.html(element);
        message.settings = {
            mode: 'view',
            uilang: uilang,
            username: this.settings.username,
            role: this.settings.role,
            savemode: 'local',
            users: this.settings.users
        };
        element.messageelement(message);
        var height = messagearea.height();
        messagearea.height(0);
        messagearea.height(height);
        setTimeout(function(){messagearea.attr('style','');}, 500);
    };
    
    /******
     * Hide the selected message
     * @param {String} messageid - the id of the message to be hidden
     ******/
    CourseFeed.prototype.hideMessage = function(messageid){
        var msgitem = this.messagelist.children('li[data-feedid="'+messageid+'"]');
        msgitem.removeClass('selectedmessage');
        var msgarea = msgitem.children('.virumcoursefeed-messageitemarea');
        var height = msgarea.height();
        msgarea.height(height);
        msgarea.empty().height(0);
        setTimeout(function(){msgarea.attr('style','');}, 500);
    }
    
    /******
     * Fetch all messages referenced to belong to this course.
     * Do this by triggering 'getcontentbyref'-event.
     ******/
    CourseFeed.prototype.fetchMessages = function(){
        this.place.trigger('makeupdaterequest');
        this.place.trigger('getcontent', {anchors: [this.settings.contextid, this.name], contentType: ['message', 'hamessage']});
        this.showMessageList();
    }
    
    /******
     * Store new messages
     * @param {Object} msgdata - data of the message
     ******/
    CourseFeed.prototype.storeMessage = function(msgdata){
        
    }
    
    /******
     * Mark message with given id as read
     * @param {String} msgid - message id
     ******/
    CourseFeed.prototype.markMessageRead = function(msgid){
        var msginfo = this.refinfo[msgid];
        if (!msginfo.read) {
            msginfo.read = true;
            this.setMessageInfo(msgid);
        };
    };
    
    /******
     * Send message info to outer system
     * @param {String} msgid - message id
     ******/
    CourseFeed.prototype.setMessageInfo = function(msgid){
        var refinfo = this.refinfo[msgid];
        var contentinfo = {
            name: msgid,
            contentType: refinfo.contentType || refinfo.reftype,
            subcontext: this.name,
            lang: refinfo.lang,
            isoutbox: refinfo.isoutbox,
            read: refinfo.read,
            sendto: refinfo.sendto,
            sendfrom: refinfo.sendfrom,
            recipientopened: refinfo.recipientopened,
            timestamp: refinfo.timestamp || refinfo.senddate
        };
        this.place.trigger('setcontentinfo', contentinfo);
        this.place.trigger('savecontent');
    };
    
    /******
     * Start composing
     ******/
    CourseFeed.prototype.startComposing = function(mtype, data){
        mtype = mtype || 'messageelement';
        this.newMessageAnchors = {};
        this.place.addClass('iscomposing');
        this.place.find('.virumcoursefeed-composearea').messageelement({
            type: mtype,
            settings: {
                uilang: this.settings.uilang,
                username: this.settings.username,
                role: this.settings.role,
                mode: 'edit',
                users: this.settings.users
            }
        });
        this.place.trigger('startediting');
        if (data) {
            this.addMessageData(data);
        };
    };
    
    /******
     * Stop composing
     ******/
    CourseFeed.prototype.stopComposing = function(){
        this.place.removeClass('iscomposing');
        this.place.find('.virumcoursefeed-composearea').removeClass('ebook-hamessage ebook-messageelement');
        this.place.trigger('stopediting');
    };
    
    /**
     * Add a new homeassignment element to a hamessage
     * @parameter {Object} hadata - data of homeassignment
     */
    CourseFeed.prototype.addHomeassingment = function(hadata) {
        // If the composer is not already open, start a new hamessage.
        if (!this.place.is('.iscomposing')) {
            this.startComposing('hamessage');
        };
        var composer = this.place.find('.virumcoursefeed-composearea');
        if (composer.is('.ebook-hamessage')) {
            composer.trigger('addhomeassignmentdata', hadata);
        };
    };
    
    /**
     * Add a new  element to a message. Create a new message, if note yet composing
     * @parameter {Object} hadata - data of new element
     */
    CourseFeed.prototype.addMessageData = function(data) {
        // If the composer is not already open, start a new hamessage.
        if (!this.place.is('.iscomposing')) {
            this.startComposing('messageelement');
        };
        var composer = this.place.find('.virumcoursefeed-composearea');
        if (composer.is('.ebook-messageelement')) {
            composer.trigger('addmessagedata', data);
        };
    };
    
    /******
     * Send new message out
     * @param {Object} sdata - send data {name: <msgid>, publishType: <user|toteacher|all>, sendto: <username|group>, data: <messageelement>}
     ******/
    CourseFeed.prototype.sendMessage = function(sdata){
        var uilang = this.settings.uilang;
        var anchors = [this.settings.contextid];
        for (var anc in this.newMessageAnchors) {
            if (this.newMessageAnchors[anc] > 0) {
                anchors.push(anc);
            };
        };
        var content = {
            "name": sdata.name,
            "contentType": sdata.contentType || "message",
            "lang": 'common',
            "anchors": anchors,
            "contentdata": sdata.data,
            "contentinfo": {
                "name": sdata.name,
                "contentType": sdata.contentType || "message",
                "subcontext": this.name,
                "lang": 'common',
                //"pubtype": (sdata.sendto === 'teacher' ? 'teacher' : sdata.sendto === 'all' ? 'all' : 'user'),
                "sendto": sdata.sendto,
                "sendfrom": sdata.sendfrom,
                "timestamp": (new Date()).getTime(),
                "read": false,
                "recipientopened": false,
                "isoutbox": true
            }
        };
        this.place.trigger('setcontent', [content, true]);
        this.place.trigger('savecontent');
        // End composing and show this new message
        this.stopComposing();
        this.showMessage(sdata.name);
        if (sdata.contentType === 'hamessage') {
            this.place.trigger('init_refresh_hamessages');
        };
    };
    
    /******
     * Get the realname for given username (from this.settings.users)
     * @param {String} username
     * @returns {String} the realname of the user
     ******/
    CourseFeed.prototype.getRealname = function(username){
        return this.users.getRealname(username);
    };
    
    /**
     * Get the group description for given groupid (from this.settings.users)
     * @param {String} gid   - Group id
     * @returns {String} the description of the group
     */
    CourseFeed.prototype.getGroupDescription = function(gid) {
        return this.users.getGroupDescription(gid);
    };
    
    /**
     * Get the context title
     * @returns {String} the title of the context
     */
    CourseFeed.prototype.getContextTitle = function() {
        return this.users.getContextTitle();
    };
    
    /**
     * Get the context description
     * @returns {String} the description of the context
     */
    CourseFeed.prototype.getContextDescription = function() {
        return this.users.getContextDescription();
    };
    
    /******
     * Set feed hiding
     * @param {Boolean} hidden - true, if hidden
     ******/
    CourseFeed.prototype.setFeedHide = function(hidden){
        this.hidden = !!hidden;
        var width = this.place.width();
        if (this.hidden) {
            this.place.addClass('sidepanel-hidden');
        } else {
            this.place.removeClass('sidepanel-hidden');
        }
        this.place.trigger('appresized', [{appname: this.name, change: -width}]);
    }
    
    /******
     * Hide feed
     ******/
    CourseFeed.prototype.hideFeed = function(){
        this.setFeedHide(true);
    };
    
    /******
     * Unhide feed
     ******/
    CourseFeed.prototype.unhideFeed = function(){
        this.setFeedHide(false);
    };
    
    /******
     * Toggle hide/unhide feed
     ******/
    CourseFeed.prototype.toggleHideFeed = function(){
        this.setFeedHide(!this.hidden);
    };
    
    /**
     * The sending of a message was success
     * @param {Object} data - the data of the successfull message
     */
    CourseFeed.prototype.sendSuccess = function(item) {
        var uilang = this.settings.uilang;
        var msgid = item.name;
        var msgdata = item.contentdata.data;
        this.refinfo[msgid] = item.contentinfo;
        // Trigger notification about message sending.
        this.place.trigger('notification', {
            "id": "sentmessage-" + msgid,
            "nclass": "quick",
            "message": ebooklocalizer.localize('coursefeed:messagesent', uilang) + ':\n' + (msgdata.title || msgdata.subject || '')
        });
        this.place.trigger('denotification', {
            "id": "sentmessage-" + msgid,
            "nclass": "warning"
        });
    }
    
    /**
     * The sending of a message was fail
     * @param {Object} data - the data of the failed message
     */
    CourseFeed.prototype.sendFail = function(item) {
        var uilang = this.settings.uilang;
        var msgid = item.name;
        var msgdata = item.contentdata.data;
        this.refinfo[msgid] = item.contentinfo;
        // Trigger notification about message sending.
        this.place.trigger('notification', {
            "id": "sentmessage-" + msgid,
            "nclass": "quick",
            "message": ebooklocalizer.localize('coursefeed:messagesend_failed', uilang) + ':\n' + (msgdata.title || msgdata.subject || ''),
            "icon": VirumCourse.icons.warning
        });
        this.place.trigger('notification', {
            "id": "sentmessage-" + msgid,
            "nclass": "warning",
            "message": ebooklocalizer.localize('coursefeed:messagesend_failed', uilang) + ':\n' + (msgdata.title || msgdata.subject || ''),
            "data": {appid: this.name, events: ['showfeed', 'showmessage'], data: [null, {messageid: msgid}]},
            "icon": VirumCourse.icons.warning
        });
    }
    
    /******
     * Remove all event handlers from this DOM-place
     ******/
    CourseFeed.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Add event handlers
     ******/
    CourseFeed.prototype.addHandlers = function(){
        var feed = this;
        this.removeHandlers();
        this.place.on('click', '.virumcoursefeed-messagelistitem .virumcoursefeed-messageheader', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var item = $(this).closest('li');
            var msgid = item.attr('data-feedid') || '';
            var isopen = item.hasClass('selectedmessage');
            if (isopen) {
                feed.hideMessage(msgid);
            } else {
                feed.showMessage(msgid);
            };
        });
        this.place.on('click', '.virumcoursefeed-tabblock', function(event, data){
            feed.toggleHideFeed();
        });
        this.place.on('toggleshow', function(event, data){
            feed.toggleHideFeed();
        });
        this.place.on('showfeed', function(event, data){
            feed.unhideFeed();
        });
        this.place.on('keydown', '.virumcoursefeed-messagelistitem', function(event, data){
            var item = $(this);
            var newitem;
            var click = false;
            switch (event.which){
                case 38:
                    newitem = item.prev();
                    event.stopPropagation();
                    event.preventDefault();
                    break;
                case 40:
                    newitem = item.next();
                    event.stopPropagation();
                    event.preventDefault();
                    break;
                case 33:
                case 36:
                    newitem = item.parent().children().first();
                    event.stopPropagation();
                    event.preventDefault();
                    break;
                case 34:
                case 35:
                    newitem = item.parent().children().last();
                    event.stopPropagation();
                    event.preventDefault();
                    break;
                case 13:
                    newitem = item;
                    click = true;
                    event.stopPropagation();
                    event.preventDefault();
                    break;
                default:
                    newitem = $();
            };
            if (newitem.length > 0) {
                newitem.focus();
                if (click) {
                    newitem.children('.virumcoursefeed-messageheader').click();
                }
            };
        });
        this.place.on('reply_getcontent', function(event, data){
            if (event.target === this && (data.contentType === 'message' || data.contentType === 'hamessage')) {
                event.stopPropagation();
                var refs = data.refs[feed.settings.contextid] || [];
                var content = data.contentdata;
                var info = data.contentinfo;
                var contentType = data.contentType;
                var ref, notif, dataitem;
                for (var i = 0, len = refs.length; i < len; i++) {
                    ref = refs[i];
                    info[ref] = info[ref] || {};
                    info[ref].contentType = data.contentType || info[ref].reftype;
                    dataitem = content[ref];
                    feed.contentdata[ref] = dataitem;
                    feed.refinfo[ref] = info[ref];
                };
            };
        });
        this.place.on('click', '.virumcoursefeed-filterbutton', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var filter = button.attr('data-filtername');
            if (button.hasClass('buttonselected')) {
                button.removeClass('buttonselected');
                feed.removeFilter(filter);
            } else {
                button.addClass('buttonselected');
                feed.addFilter(filter);
            };
        });
        this.place.on('click', '.virumcoursefeed-controlbutton', function(event){
            event.stopPropagation();
            event.preventDefault();
            var button = $(this);
            var action = button.attr('data-event');
            var ctype = button.attr('data-ctype');
            var data = ctype && CourseFeed.buttons.control[ctype].data || {};
            button.trigger(action, [data]);
        });
        this.place.on('coursefeedrefresh', function(event, data){
            event.stopPropagation();
            feed.fetchMessages();
        });
        this.place.on('showmessage', function(event, data){
            event.stopPropagation();
            feed.showMessage(data.messageid);
        });
        this.place.on('coursefeednewhamessage', function(event, data){
            event.stopPropagation();
            if (!feed.place.is('.iscomposing')) {
                feed.startComposing('hamessage');
            };
        });
        this.place.on('coursefeednewmessage', function(event, data){
            event.stopPropagation();
            if (!feed.place.is('.iscomposing')) {
                feed.startComposing('messageelement', data);
            };
        });
        this.place.on('coursefeedcancelmessage', function(event, data){
            event.stopPropagation();
            feed.stopComposing();
        });
        this.place.on('sendnewmessage', function(event, data){
            event.stopPropagation();
            feed.sendMessage(data);
        });
        this.place.on('sendsuccess', function(event, data) {
            event.stopPropagation();
            feed.sendSuccess(data);
        });
        this.place.on('sendfail', function(event, data) {
            event.stopPropagation();
            feed.sendFail(data);
        });
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            feed.close();
        });
        this.place.on('homeassignmentdata', function(event, data) {
            event.stopPropagation();
            if (feed.hidden) {
                feed.unhideFeed();
            };
            feed.addHomeassingment(data);
        });
        this.place.on('composenewmessage', function(event, data) {
            event.stopPropagation();
            if (feed.hidden) {
                feed.unhideFeed();
            };
            feed.addMessageData(data);
        });
        this.place.off('setcontent').on('setcontent', function(event, data) {
            // Pass the event through and add the id of this notebook as anchor.
            // Don't stop propagation!!!
            var appid = feed.name;
            var dataarr;
            if (typeof(data) === 'object' && typeof(data.length) === 'number') {
                dataarr = data;
            } else {
                dataarr = [data];
            };
            for (var i = 0, len = dataarr.length; i < len; i++) {
                if (dataarr[i].anchors.indexOf(appid) === -1) {
                    dataarr[i].anchors.push(appid);
                };
            };
        });
        this.place.on('addmessageanchor', function(event, data) {
            event.stopPropagation();
            var anchor = data.anchor;
            if (!feed.newMessageAnchors[anchor]) {
                feed.newMessageAnchors[anchor] = 1;
            } else {
                feed.newMessageAnchors[anchor]++;
            };
        });
        this.place.on('removemessageanchor', function(event, data) {
            event.stopPropagation();
            var anchor = data.anchor;
            if (feed.newMessageAnchors[anchor]) {
                feed.newMessageAnchors[anchor] = Math.max(0, feed.newMessageAnchors[anchor] - 1);
            };
        });
    };
    
    /**
     * Close the feed
     */
    CourseFeed.prototype.close = function() {
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    }
    
    /******
     * Add <style>-tag with CSS-rules for the course feed.
     ******/
    CourseFeed.prototype.setStyles = function(){
        if ($('#virumcoursefeedstyles').length === 0) {
            $('head').append('<style type="text/css" id="virumcoursefeedstyle">'+CourseFeed.style+'</style>');
        };
    };
    
    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    CourseFeed.prototype.sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };
    
    /******
     * Default data for the course feed
     ******/
    CourseFeed.defaults = {
        feed: [],
        contentdata: {},
        refinfo: {},
        singleopen: false,
        hidden: true,
        settings: {
            uilang: 'en',
            username: 'Anonymous',
            courseid: '0',
            contextid: '0',
            users: {}
        }
    };
    
    /******
     * Buttons that should be shown in the panels.
     ******/
    CourseFeed.buttons = {
        control: {
            message: {
                label: 'coursefeed:newmessage',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-newmessage"><path style="stroke: none; fill: white;" d="M3 5 l12 -4 l12 4 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l12 -4 l12 4 l0 18 l-24 0z m1 1 l0 2 l2.2 2 l0 -4 l17.6 0 l0 4 l2.2 -2 l0 -2 l-11 -3.5z m0 16 l22 0 l-8 -7.5 l-6 0z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z m-3 -14.3 l-16 0 l0 3.8 l4.3 3.2 l7.6 0 l4.3 -3.2z m-1 2 l0 1 l-14 0 l0 -1z m-2 3 l0 1 l-10 0 l0 -1z" /><path style="stroke: none; fill: #0a0;" d="M23 2 a6 6 0 0 0 0 12 a6 6 0 0 0 0 -12z" /><path style="stroke: none; fill: white;" d="M22 4 l2 0 l0 3 l3 0 l0 2 l-3 0 l0 3 l-2 0 l0 -3 l-3 0 l0 -2 l3 0z" /></svg>',
                event: 'coursefeednewmessage',
                ctype: 'message',
                data: {contentdata: {type: 'markdownelement'}}
            },
            hamessage: {
                label: 'coursefeed:newhamessage',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-home"><path style="stroke: none;" d="M3 30 l0 -11 l12 -12 l12 12 l0 11 l-9 0 l0 -6 l-6 0 l0 6 z m-1.5 -12 l-1.5 -1.5 l15 -15 l15 15 l-1.5 1.5 l-13.5 -13.5 z m2 -6 l0 -7 l4 0 l0 3z"></path><path style="stroke: none; fill: #0a0;" d="M23 2 a6 6 0 0 0 0 12 a6 6 0 0 0 0 -12z" /><path style="stroke: none; fill: white;" d="M22 4 l2 0 l0 3 l3 0 l0 2 l-3 0 l0 3 l-2 0 l0 -3 l-3 0 l0 -2 l3 0z" /></svg>',
                aicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-newmessage"><path style="stroke: none; fill: white;" d="M3 5 l12 -4 l12 4 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 5 l12 -4 l12 4 l0 18 l-24 0z m1 1 l0 2 l2.2 2 l0 -4 l17.6 0 l0 4 l2.2 -2 l0 -2 l-11 -3.5z m0 16 l22 0 l-8 -7.5 l-6 0z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z m-3 -14.3 l-16 0 l0 3.8 l4.3 3.2 l7.6 0 l4.3 -3.2z m-1 2 l0 1 l-14 0 l0 -1z m-2 3 l0 1 l-10 0 l0 -1z" /><path style="stroke: none; fill: #0a0;" d="M23 2 a6 6 0 0 0 0 12 a6 6 0 0 0 0 -12z" /><path style="stroke: none; fill: white;" d="M22 4 l2 0 l0 3 l3 0 l0 2 l-3 0 l0 3 l-2 0 l0 -3 l-3 0 l0 -2 l3 0z" /></svg>',
                event: 'coursefeednewhamessage',
                ctype: 'hamessage'
            },
            refresh: {
                label: 'coursefeed:refresh',
                icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-refresh"><path style="stroke: none;" d="M17 13 l11 0 l0 -11 l-4 4 a12.7 12.7 0 1 0 0 18 a1.5 1.5 0 0 0 -3 -3 a8.46 8.46 0 1 1 0 -12 z"></path></svg>',
                event: 'coursefeedrefresh'
            }
            //sendmessage: {
            //    label: 'coursefeed:sendmessage',
            //    icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-send"><path style="stroke: none;" d="M2 18 l24 -12 l-4 18 l-7 -3 l-4 4 l-1 0 l-1 -6z m8 0 l0.7 6 l2 -4.5 l12 -12z"></path></svg>',
            //    event: 'coursefeedsendmessage'
            //},
        }
    };
    
    /******
     * Html-templates
     ******/
    CourseFeed.templates = {
        html: [
            '<div class="virumcoursefeed-controls ffwidget-background">',
            //'    <div class="virumcoursefeed-tabblock sidepanel-tabblock"><div class="virumcoursefeed-appicon"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="200" height="200" viewBox="0 0 30 30" class="mini-icon mini-icon-feedapp"><path style="stroke: none; fill: white;" d="M3 3 l24 0 l0 18 l-24 0z" /> <path style="stroke: none;" d="M3 3 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" /><path style="stroke: none;" fill="#585" class="openfeed" d="M6 23 l7 -7 l0 3 l5 0 l0 8 l-5 0 l0 3z" /><path style="stroke: none;" fill="#855" class="closefeed" d="M12 19 l5 0 l0 -3 l7 7 l-7 7 l0 -3 l-5 0z" /></svg></div></div>',
            '    <ul class="virumcoursefeed-control-buttons"></ul>',
            '    <ul class="virumcoursefeed-control-filters ffwidget-buttonset ffwidget-horizontal"></ul>',
            '</div>',
            '<div class="virumcoursefeed-contentarea" data-dragtype="element">',
            '    <div class="virumcoursefeed-composearea ffwidget-background"></div>',
            //'    <div class="virumcoursefeed-composearea ffwidget-background"><div class="virumcoursefeed-composer"></div><div class="virumcoursefeed-composerbar"></div></div>',
            '    <ul class="virumcoursefeed-messagelist"></ul>',
            '    <div class="virumcoursefeed-messagearea"></div>',
            '</div>'
        ].join('\n')
    }

    /******
     * CSS-rules for the course feed
     ******/
    CourseFeed.style = [
        '.virumcoursefeed-wrapper {border-left: 1px solid #333; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        '.virumcoursefeed-wrapper.closed {display: none;}',
        '.virumcoursefeed-wrapper.maximized {width: 100%;}',
        '.virumcoursefeed-contentarea {overflow: hidden; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; position: relative;  display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: column nowrap; -ms-flex-flow: column nowrap; flex-flow: column nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between; height: 500px; /** Height just to fool Chrome browser **/}',
        
        // Controls
        '.virumcoursefeed-controls {min-height: 36px; height: 36px; padding: 2px; display: -webkit-box; display: -ms-flex; display: -webkit-flex; display: flex; -webkit-flex-flow: row nowrap; -ms-flex-flow: row nowrap; flex-flow: row nowrap; -webkit-align-items: stretch; -ms-align-items: stretch; align-items: stretch; -webkit-justify-content: space-between; -ms-justify-content: space-between; justify-content: space-between;}',
        '.virumcoursefeed-controls > ul {list-style: none; margin: 0 0.5em; padding: 0; flex-grow: 1;}',
        '.virumcoursefeed-controls > ul > li {display: inline-block; margin: 0; padding: 1px; width: 30px; height: 30px;}',
        '.virumcoursefeed-controls > ul > li > button {padding: 1px;}',
        '.virumcoursefeed-controls > ul > li svg {width: 25px; height: 25px;}',
        '.virumcoursefeed-controls > ul.virumcoursefeed-control-filters {text-align: right;}',

        // sidepanel tabblock
        '.virumcoursefeed-controls .virumcoursefeed-tabblock {width: 39px; height: 40px; margin: -2px 0; flex-grow: 0; flex-shrink: 0; background-color: rgba(255,255,255,0.5); cursor: pointer; border-right: 1px solid #aaa;}',
        '.virumcoursefeed-controls .virumcoursefeed-tabblock .virumcoursefeed-appicon {display: inline-block; vertical-align: middle; padding: 5px;}',
        '.virumcoursefeed-controls .virumcoursefeed-tabblock .virumcoursefeed-appicon svg {width: 30px; height: 30px;}',
        '.sidepanel:not(.sidepanel-hidden) .virumcoursefeed-controls .virumcoursefeed-tabblock .virumcoursefeed-appicon svg .openfeed {fill: transparent;}',
        '.virumcoursefeed-controls .virumcoursefeed-tabblock:hover .virumcoursefeed-appicon svg .closefeed {fill: #a00;}',
        '.sidepanel-hidden .virumcoursefeed-controls .virumcoursefeed-tabblock:hover .virumcoursefeed-appicon svg .openfeed {fill: #0a0;}',
        '.sidepanel-hidden .virumcoursefeed-controls .virumcoursefeed-tabblock .virumcoursefeed-appicon svg .closefeed {fill: transparent;}',
        
        // Filters
        'ul.virumcoursefeed-control-filters li.virumcoursefeed-control-filteritem {height: 22px; width: auto;}',
        'ul.virumcoursefeed-control-filters li.virumcoursefeed-control-filteritem .virumcoursefeed-filtercount {display: inline-block; font-weight: normal; margin: 0 0.1em;}',
        'ul.virumcoursefeed-control-filters li.virumcoursefeed-control-filteritem svg {height: 20px; width: 20px;}',
        
        // Messagelist
        '.virumcoursefeed-messagelist {list-style: none; border-right: 1px solid black; margin: 0; padding: 0; position: relative; left: 0; right: 0; top: 0; bottom: 0; background-color: #ddd; overflow-y: auto; transition: top 0.5s; -webkit-flex-grow: 1; -ms-flex-grow: 1; flex-grow: 1; -webkit-flex-shrink: 1; -ms-flex-shrink: 1; flex-shrink: 1;}',
        //'.virumcoursefeed-messagelist {list-style: none; border-right: 1px solid black; margin: 0; padding: 0; width: 120px; position: absolute; left: 0; top: 0; bottom: 0; background-color: #ccc; overflow-y: auto;}',
        '.virumcoursefeed-messagelistitem {border: 1px solid #ddd; border-bottom: 1px solid #aaa; border-right: 1px solid #aaa; overflow: hidden; background-color: #eee; margin: 0.2em; border-radius: 0.7em 0.7em 0 0.7em; box-shadow: none;}',
        '.virumcoursefeed-messagelistitem .virumcoursefeed-messageheader {padding: 5px; cursor: pointer; transition: padding 0.3s, background-color 1s; border-left: 5px solid transparent; font-size: 75%;}',
        '.virumcoursefeed-messagelistitem.selectedmessage {background-color: white;}',
        // Message date
        '.virumcoursefeed-messagelist-date {float: right;}',
        '.virumcoursefeed-messagelist-icon {text-align: center; display: block; float: left; margin: 0 0.5em 0.1em 0;}',
        '.virumcoursefeed-messagelist-icon svg {width: 20px; height: 20px;}',
        '.virumcoursefeed-messagelist-subject {font-weight: bold; display: block; font-family: Helvetica, Calibri, sans-serif;}',
        'virumcoursefeed-messageheader-title::after {display: block; clear: both; content: " ";}',
        '.virumcoursefeed-messageheader-namedate {}',
        '.virumcoursefeed-messagelist-fromto {color: #777; font-style: italic; display: inline-block;}',
        '.virumcoursefeed-messagelist-sender, .virumcoursefeed-messagelist-recipient {display: inline-block; font-family: monospace; background-color: rgba(255,255,255,0.5); border-radius: 3px; border: 1px solid #ddd; margin: 0.1em; padding: 0 0.1em;}',
        '.virumcoursefeed-messageheader-sendfailed {display: none; clear: both; float: left;}',
        '.virumcoursefeed-messageheader-sendfailed svg {fill: #a00;}',
        '.virumcoursefeed-messageheader[data-outboxstatus="true"] .virumcoursefeed-messageheader-sendfailed {display: block;}',
        '.virumcoursefeed-messagelist-date {font-size: 80%; color: #666; display: inline-block;}',
        '.virumcoursefeed-messageheader::after {display: block; clear: both; content: " ";}',
        '.virumcoursefeed-messagelistitem.selectedmessage {margin: 0.5em; border: 1px solid #777; box-shadow: 3px 3px 6px rgba(0,0,0,0.2);}',
        '.virumcoursefeed-messagelistitem {transition: margin 0.3s, background-color 1s;}',
        '.virumcoursefeed-messagelistitem.selectedmessage .virumcoursefeed-messageheader {border-radius: 0.5em; border: 1px solid #333; background-color: #f1e4ad; padding: 0.5em; margin: 5px; afont-size: 100%;}',
        //'.virumcoursefeed-messagelistitem.selectedmessage .virumcoursefeed-messageheader .virumcoursefeed-messagelist-recipient, .virumcoursefeed-messagelistitem.selectedmessage .virumcoursefeed-messageheader .virumcoursefeed-messagelist-sender {font-size: 85%;}',
        '.virumcoursefeed-messageitemarea {transition: height 0.3s;}',
        '.virumcoursefeed-messageheader-deadline {display: none;}',
        'li.virumcoursefeed-messagelistitem[data-messagetype="hamessage"] .virumcoursefeed-messageheader-deadline {display: block;}',
        
        // Unread message
        '.virumcoursefeed-messagelistitem.unreadmessage .virumcoursefeed-messageheader {border-left: 5px solid red;}',
        
        // Focused message
        'li.virumcoursefeed-messagelistitem:not(.selectedmessage):focus {background-color: #ffa; border-top: 1px solid gold; border-bottom: 1px solid gold;}',
        'li.virumcoursefeed-messagelistitem:not(.selectedmessage):focus {background-color: #8df; border-top: 1px solid #0be; border-bottom: 1px solid #0be; outline: none;}',
        'li.virumcoursefeed-messagelistitem.selectedmessage:focus {border-top: 1px solid #0be; border-bottom: 1px solid #0be; box-shadow: 0 0 5px #0be; outline: none;}',
        
        // Filtering
        '.virumcoursefeed-messagelistitem.filteredmessage:not(.selectedmessage) {height: 2px; padding: 0; background-color: #ccc;}',
        '.virumcoursefeed-messagelistitem.filteredmessage:not(.selectedmessage) > div {display: none;}',
        
        // Messagearea
        '.virumcoursefeed-messagearea {position: absolute; left: 120px; top: 0; bottom: 0; right: 0; margin: 0.5em; font-size: 90%; overflow-y: scroll; display: none;}',
        '.virumcoursefeed-message-header {border: 1px solid #333; background-color: #f1e4ad; border-radius: 0.5em; padding: 0.5em;}',
        '.virumcoursefeed-message-subject {font-weight: bold;}',
        '.virumcoursefeed-message-sender {font-style: italic;}',
        
        // Composearea
        '.virumcoursefeed-composearea {height: 0; overflow: hidden; transition: height 0.5s; position: relative; -webkit-flex-grow: 0; -ms-flex-grow: 0;  flex-grow: 0; -webkit-flex-shrink: 0; -ms-flex-shrink: 0;  flex-shrink: 0;}',
        '.virumcoursefeed-composearea > .ebook-elementset-body {color: black;}',
        //'.virumcoursefeed-composearea .virumcoursefeed-composer {background-color: white; overflow-y: scroll; overflow-x: auto; position: absolute; top: 0; bottom: 30px; left: 5px; right: 5px;}',
        //'.virumcoursefeed-composearea .virumcoursefeed-composerbar {min-height: 30px; position: absolute; bottom: 0; left: 0; right: 0;}',
        '.virumcoursefeed-wrapper.iscomposing .virumcoursefeed-composearea {height: 50%; min-height: 400px; padding: 0; -webkit-flex-grow: 1; -ms-flex-grow: 1;  flex-grow: 1; -webkit-flex-shrink: 0; -ms-flex-shrink: 0;  flex-shrink: 0;}',
        '.virumcoursefeed-wrapper.iscomposing .virumcoursefeed-messagelist {height: 50%;}',
        '.virumcoursefeed-composearea .elementset-droptarget[data-droptarget-cancontain="elements"] {background-color: #fafafa; box-shadow: 0 0 1px 1px rgba(0,0,0,0.3);}'
    ].join('\n');

    
    /******
     * Some icons
     ******/
    CourseFeed.icons = {
        unreadmessages: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-unreadmessage"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z"></path> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z"></path><path style="stroke: none; fill: #a00;" d="M13 8 l4 0 l0 10 l-4 0z m0 12 l4 0 l0 4 l-4 0z" /></svg>',
        resend: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="25" height="25" viewBox="0 0 30 30" class="mini-icon mini-icon-send"><path style="stroke: none;" d="M2 18 l24 -12 l-4 18 l-7 -3 l-4 4 l-1 0 l-1 -6z m8 0 l0.7 6 l2 -4.5 l12 -12z"></path></svg>',
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-message"><path style="stroke: none; fill: white;" d="M3 5 l24 0 l0 18 l-24 0z"></path> <path style="stroke: none;" d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z"></path></svg>'
    }

    /******
     * Localization strings
     ******/
    CourseFeed.localization = {
        "en": {
            "coursefeed:sendmessage": "Send",
            "coursefeed:newmessage": "New message",
            "coursefeed:refresh": "Refresh",
            "coursefeed:filterunread": "Unread messages",
            "coursefeed:messagesent": "Message sent",
            "coursefeed:messagesend_failed": "Message send failed!",
            "coursefeed:deadline": "Deadline",
            "coursefeed:resend_failed": "Sending failed.\nResend.",
            "coursefeed:newhamessage": "New homeassignment message"
        },
        "fi": {
            "coursefeed:sendmessage": "Lähetä",
            "coursefeed:newmessage": "Uusi viesti",
            "coursefeed:refresh": "Päivitä",
            "coursefeed:filterunread": "Lukemattomat viestit",
            "coursefeed:messagesent": "Viesti lähetetty",
            "coursefeed:messagesend_failed": "Viestin lähetys epäonnistui!",
            "coursefeed:deadline": "Aikaraja",
            "coursefeed:resend_failed": "Lähetys epäonnistui.\nLähetä uudelleen.",
            "coursefeed:newhamessage": "Uusi kotitehtäväviesti"
        },
        "sv": {
            "coursefeed:sendmessage": "Skicka",
            "coursefeed:newmessage": "Ny meddelanden",
            "coursefeed:refresh": "Uppdatera",
            "coursefeed:filterunread": "Olästa meddelanden",
            "coursefeed:messagesent": "Meddelandet har skickats",
            "coursefeed:messagesend_failed": "Meddelandet kunde INTE skickas!",
            "coursefeed:deadline": "Deadline",
            "coursefeed:resend_failed": "Sändningen misslyckades. Försök på nytt. ",
            "coursefeed:newhamessage": "Ny hemuppgift meddelanden"
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(CourseFeed.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(CourseFeed.localization);
    }
    
    /**** jQuery-plugin *****/
    var feedmethods = {
        'init': function(params){
            return this.each(function(){
                var feed = new CourseFeed(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[virumcoursefeeddata]]');
            return data;
        },
        'geticon': function() {
            return CourseFeed.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.virumcoursefeed = function(method){
        if (feedmethods[method]) {
            return feedmethods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return feedmethods.init.apply(this, arguments);
        } else {
            console.log('Method ' + method + ' does not exist in virumcoursefeed.');
            return false;
        }
    };

    
    ///******
    // * Userinfo - a class for holding information about user (name etc.)
    // * @constructor
    // ******/
    //
    //var Userinfo = function(userdata, utype){
    //    userdata = $.extend(true, {}, Userinfo.defaults, userdata);
    //    this.username = userdata.username;
    //    this.realname = userdata.realname;
    //    this.usertype = utype || '';
    //}
    //
    //Userinfo.prototype.getUsername = function(){
    //    return this.username;
    //}
    //
    //Userinfo.prototype.getRealname = function(){
    //    var name = [];
    //    var nameparts = ['firstname', 'middlename', 'lastname'];
    //    for (var i = 0, len = nameparts.length; i < len; i++) {
    //        if (nameparts[i]) {
    //            name.push(this.realname[nameparts[i]]);
    //        };
    //    };
    //    return name.join(' ');
    //};
    //
    //Userinfo.prototype.getRealnameLnameFirst = function(){
    //    var name = [this.realname.lastname + ','];
    //    var nameparts = ['firstname', 'middlename'];
    //    for (var i = 0, len = nameparts.length; i < len; i++) {
    //        if (nameparts[i]) {
    //            name.push(this.realname[nameparts[i]]);
    //        };
    //    };
    //    return name.join(' ');
    //};
    //
    //Userinfo.prototype.getUsertype = function(){
    //    return this.usertype;
    //}
    //
    //Userinfo.defaults = {
    //    username: 'Anonymous',
    //    realname: {
    //        firstname: '',
    //        middlename: '',
    //        lastname: ''
    //    }
    //}
    //
    //
    ///********************************************
    // * Userlist - class for holding user objects
    // * @constructor
    // ********************************************/
    //var Userlist = function(userdata){
    //    userdata = $.extend(true, {}, Userlist.defaults, userdata);
    //    this.initData(userdata);
    //}
    //
    //Userlist.prototype.initData = function(userdata){
    //    this.users = {};
    //    this.groups = {};
    //    this.userByUsername = {};
    //    var utype, ulist, user, username, i, len;
    //    for (utype in userdata.users){
    //        ulist = userdata.users[utype];
    //        this.users[utype] = [];
    //        for (i = 0, len = ulist.length; i < len; i++) {
    //            username = ulist[i].username;
    //            user = new Userinfo(ulist[i], utype);
    //            this.users[utype].push(user);
    //            this.userByUsername[username] = user;
    //        };
    //    };
    //    // TODO: groups.
    //};
    //
    //Userlist.prototype.getRealname = function(username){
    //    var user = this.userByUsername[username];
    //    return user && user.getRealname() || '';
    //};
    //
    //Userlist.prototype.getRealnameLnameFirst = function(username){
    //    var user = this.userByUsername[username];
    //    return user && user.getRealnameLnameFirst() || '';
    //};
    //
    //Userlist.prototype.getUserList = function(typelist, sortbyfname){
    //    var ulist = [], tmplist;
    //    var i, len, utype;
    //    for (i = 0, len = typelist.length; i < len; i++){
    //        utype = typelist[i];
    //        if (this.users[utype]) {
    //            tmplist = this.users[utype].slice();
    //            if (sortbyfname) {
    //                // Sort by first name
    //                tmplist.sort(function(a, b){
    //                    return (a.getRealname() < b.getRealname() ? -1 : 1);
    //                });
    //            } else {
    //                // Sort by last name
    //                tmplist.sort(function(a, b){
    //                    return (a.getRealnameLnameFirst() < b.getRealnameLnameFirst() ? -1 : 1);
    //                });
    //            }
    //            ulist = ulist.concat(tmplist);
    //        };
    //    };
    //    return ulist;
    //};
    //
    //Userlist.defaults = {
    //    users: {
    //        teachers: [],
    //        students: [],
    //    },
    //    groups: []
    //}
    
    
    
})(window, jQuery, window.ebooklocalizer);
