/******
@name Contextboard
@version 0.1
@author Petri Salmela <petri.salmela@fourferries.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Contextboard
@description A class and jQuery-plugin for a user interface for modifying context details such as name, description, members, etc.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - Elpp-libs
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
        typeof(Elpp) === 'undefined' && Elpp.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Helper functions
     */
    
    /**
     * Sanitize text
     * @param {String} text    Text to sanitize
     * @param {Object} options Options for sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    };

    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
   
    /**
     * Context board
     * @constructor
     * @param {jQuery} place    DOM-place for course dashboard
     * @param {Object} options  data for course dashboard
     */
    var Contextboard = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.initData();
        this.setAttrs();
        this.setStyles();
        this.show();
    }
    
    Contextboard.prototype.init = function(options) {
        this.settings = options.settings;
        this.users = this.settings.users;
        this.appid = this.settings.appid;
        this.apptype = this.settings.apptype;
    };
    
    Contextboard.prototype.initData = function() {
        this.initSchools();
        var subjects = this.settings.libraryinfo && this.settings.libraryinfo.subjects || [];
        this.subjects = new Elpp.SubjectSet(subjects);
        //this.place.trigger('getcontent', {anchors: [this.settings.contextid], contentType: ['hamessage', 'solution', 'review']});
    };
    
    Contextboard.prototype.initSchools = function() {
        this.schoolsById = {};
        var schools = this.settings.libraryinfo && this.settings.libraryinfo.schools || [];
        var sid;
        for (var i = 0, len = schools.length; i < len; i++) {
            sid = schools[i].id;
            this.schoolsById[sid] = schools[i];
        };
    };
    
    Contextboard.prototype.initHandlers = function() {
        var board = this;
        //this.place.on('reply_getcontent', function(event, data) {
        //    event.stopPropagation();
        //    board.setContentdata(data);
        //});
        this.place.off('click', '.contextboard-messagelink').on('click', '.contextboard-messagelink', function(event) {
            event.stopPropagation();
            var toid = $(this).attr('data-toid');
            board.composeMessage(toid);
        });
        this.place.on('refreshapp', function(event, data) {
            event.stopPropagation();
            board.initData();
            board.show();
        });
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            board.close();
        });
    };
    
    Contextboard.prototype.setAttrs = function() {
        this.place.addClass('contextboard');
    };
    
    Contextboard.prototype.show = function() {
        this.view();
    };
    
    Contextboard.prototype.view = function() {
        this.place.html(this.template.html);
        this.viewInfo();
        if (this.users.canEdit({ctype: 'notebookcontent', contextid: this.settings.contextid})) {
            this.viewUserList();
        };
    };
    
    Contextboard.prototype.viewInfo = function() {
        var uilang = this.settings.uilang;
        var context = this.settings.context;
        var place = this.place.children('.contextboard-contextinfo');
        place.empty();
        var html = ['<table class="contextboard-infotable">', '<tbody>'];
        html.push(`<tr><td>${this.getSubjectIcons()}</td><th>${sanitize(context.titles[uilang] || context.titles.common || '')}</th></tr>`)
        if (this.settings.context.teacher) {
            html.push(`<tr><td class="contextboard-infolabel">${ebooklocalizer.localize('contextboard:teacher', uilang)}:</td><td>${sanitize(this.users.getRealname(this.settings.context.teacher.username || ''))}</td></tr>`);
        };
        html.push(`<tr><td class="contextboard-infolabel">${ebooklocalizer.localize('contextboard:shortdescription', uilang)}:</td><td>${sanitize(context.short_desc)}</td></tr>`);
        html.push(`<tr><td class="contextboard-infolabel">${ebooklocalizer.localize('contextboard:description', uilang)}:</td><td>${sanitize(context.description)}</td></tr>`);
        html.push('</tbody>', '</table>');
        place.html(html.join('\n'));
    };
    
    Contextboard.prototype.viewUserList = function() {
        var cboard = this;
        var uilang = this.settings.uilang;
        var place = this.place.children('.contextboard-userlist');
        place.empty();
        var html = {};
        var groups = this.users.getGroups(), gid, i, j, len, glen;
        for (i = 0, len = groups.length; i < len; i++) {
            gid = groups[i];
            html[gid] =  [
                '<table class="contextboard-usertable">',
                `<caption>${this.users.groups[gid].getDescription()}</caption>`,
                '<thead class="ffwidget-background-colored">',
                `<tr>
                <th>${ebooklocalizer.localize('contextboard:name', uilang)}</th>
                <th>${ebooklocalizer.localize('contextboard:username', uilang)}</th>
                <th>${ebooklocalizer.localize('contextboard:email', uilang)}</th>
                <th>${ebooklocalizer.localize('contextboard:school', uilang)}</th>
                </tr>`,
                '</thead>',
                '<tbody>'
            ];
        }
        console.log('this', this);
        var userlist = this.users.getUserList(groups);
        var user, row, ugroups, schools, schoolnames;
        for (i = 0, len = userlist.length; i < len; i++) {
            user = userlist[i];
            schools = user.getSchools();
            schoolnames = schools.map(function(id) {return cboard.schoolsById[id].name});
            row = `
                <tr>
                    <td data-userdata="realname">${user.getRealnameLnameFirst()}</td>
                    <td data-userdata="username">${user.getUsername()} ${this.getMessageLink(user.getUsername())}</td>
                    <td data-userdata="email"><a href="mailto:${encodeURIComponent(user.data.email)}">${user.getEmail()}</a></td>
                    <td data-userdata="schools">${sanitize(schoolnames.join(',<br>'))}</td>
                </tr>
            `;
            ugroups = user.getGroups();
            for (j = 0, glen = ugroups.length; j < glen; j++) {
                html[ugroups[j]].push(row);
            };
        };
        for (gid in this.users.groups) {
            html[gid].push('</tbody>', '</table>');
            place.append(html[gid].join('\n'));
        };
    };
    
    Contextboard.prototype.getSubjectIcons = function() {
        var lang = this.settings.uilang;
        var result = ['<ul class="contextboard-subjectlist">'];
        var subjects = this.settings.context.subjects;
        var subj;
        for (var i = 0, len = subjects.length; i < len; i++) {
            subj = subjects[i];
            result.push(`
                <li>
                    <div class="contextboard-subjecticon">${this.subjects.getIcon(subj)}</div>
                    <div class="contextboard-subjecttext">${this.subjects.getTitle(subj, lang)}</div>
                </li>
            `);
        };
        result.push('</ul>');
        return result.join('\n');
    };
    
    Contextboard.prototype.getMessageLink = function(username) {
        var html = `<span class="contextboard-messagelink" data-toid="${sanitize(username)}">${this.icons.message}</span>`;
        return html;
    };
    
    Contextboard.prototype.composeMessage = function(toid) {
        var data = {
            toid: toid,
            totype: 'user',
            contentdata: {
                type: 'markdownelement',
                metadata: {
                    creator: this.settings.username,
                    modifier: this.settings.username,
                    created: (new Date()).getTime(),
                    modified: (new Date()).getTime(),
                    tags: []
                },
                data: {
                    text: ''
                }
            }
        }
        this.place.trigger('composemessagedata', [data]);
    };
    
    Contextboard.prototype.close = function() {
        this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    };
    
    Contextboard.prototype.setStyles = function() {
        if ($('head style#contextboardstyles').length === 0) {
            $('head').append('<style type="text/css" id="contextboardstyles">'+this.styles+'</style>')
        };
    };
    
    Contextboard.prototype.defaults = {
        settings: {
            uilang: 'en'
        }
    };
    
    Contextboard.prototype.styles = [
        '.contextboard {background-color: #eee; font-family: helvetica, Arial, sans-serif; max-height: 100%; overflow: auto; padding: 1em;}',
        '.contextboard h1 {text-align: center;}',
        '.contextboard-usertable {margin: 1em auto; border-collapse: collapse; border: 1px solid #888; font-size: 80%; width: 100%; background-color: white;}',
        '.contextboard-usertable th, .contextboard-usertable td {padding: 0.2em 0.5em; text-align: left; vertical-align: top;}',
        '.contextboard-usertable tbody tr:nth-child(even) {background-color: rgba(0,0,0,0.05);}',
        '.contextboard-usertable tbody tr:nth-child(odd) {background-color: rgba(255,255,255,0.2);}',
        '.contextboard-infotable {width: 100%;}',
        '.contextboard-infotable td, .contextboard-infotable th {padding: 0.2em 0.5em; text-align: left; vertical-align: top;}',
        '.contextboard-subjectlist {list-style: none; margin: 0; padding: 0;}',
        '.contextboard-subjectlist li {display: inline-block; margin: 0 0.5em; padding: 4px; text-align: center; border-radius: 4px; background-color: rgba(255,255,255,0.3); max-width: 50px; overflow: hidden;}',
        '.contextboard-subjecticon {line-height: 0;}',
        '.contextboard-subjecttext {font-family: sans-serif; font-size: 70%;}',
        '.contextboard-infolabel {font-family: sans-serif;}',
        '.contextboard-messagelink svg {width: 1.3em; height: 1.3em; display: inline-block; vertical-align: bottom; cursor: pointer;}'
    ].join('\n');
    
    Contextboard.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-contextboard"><path style="stroke: none;" d="M15 9 a5 5 0 0 1 0 10 a5 5 0 0 1 0 -10z m-10 16 a6 6 0 0 1 6 -6 l4 4 l4 -4 a6 6 0 0 1 6 6 v3 h-20z m0 -14 a3 3 0 1 1 2 0 a5 5 0 0 1 2 1 a6 6 0 0 0 1 5 h-9 v-2 a5 5 0 0 1 5 -4z m19 0 a5 5 0 0 1 5 4 v2 h-9 a6 6 0 0 0 1 -5 a5 5 0 0 1 2 -1 a3 3 0 1 1 2 0z" /></svg>',
        refresh: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-refresh"><path style="stroke: none;" d="M16 14 l8 0 l0 -8 l-3 3 a8.5 8.5 0 1 0 0 12 l-2 -2 a5.6 5.6 0 1 1 0 -8 z"></path></svg>',
        message: '<svg class="mini-icon mini-icon-message" viewBox="0 0 30 30" height="20" width="20" xmlns="http://www.w3.org/2000/svg"><path d="M3 5 l24 0 l0 18 l-24 0z" style="stroke: none; fill: white;"></path> <path d="M3 5 l24 0 l0 18 l-24 0z m1 1 l0 2 l11 8 l11 -8 l0 -2z m0 16 l22 0 l-8 -7 l-3 2 l-3 -2z m0 -1 l7 -6.5 l-7 -5.5z m22 0 l0 -12 -7 5.5z" style="stroke: none;"></path></svg>'
    }
    
    Contextboard.prototype.template = {
        html: `
            <div class="contextboard-contextinfo">
            </div>
            <div class="contextboard-userlist">
            </div>
        `
    };
    
    Contextboard.localization = {
        "en": {
            "contextboard:name": "Name",
            "contextboard:username": "Username",
            "contextboard:email": "Email",
            "contextboard:school": "School",
            "contextboard:teacher": "Teacher",
            "contextboard:shortdescription": "Short description",
            "contextboard:description": "Description"
        },
        "fi": {
            "contextboard:name": "Nimi",
            "contextboard:username": "Tunnus",
            "contextboard:email": "Sähköposti",
            "contextboard:school": "Koulu",
            "contextboard:teacher": "Opettaja",
            "contextboard:shortdescription": "Lyhyt kuvaus",
            "contextboard:description": "Kuvaus"
        },
        "sv": {
            "contextboard:name": "Namn",
            "contextboard:username": "Användarnamn",
            "contextboard:email": "E-post",
            "contextboard:school": "Skolan",
            "contextboard:teacher": "Läraren",
            "contextboard:shortdescription": "Kort beskrivning",
            "contextboard:description": "Beskrivning"
        }
    }

    try {
        ebooklocalizer.addTranslations(Contextboard.localization);
    } catch (err) {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Contextboard.localization);
    }
    
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var board = new Contextboard(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[contextboarddata]]');
            return data;
        },
        'geticon': function() {
            return Contextboard.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.contextboard = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in coursedashboard.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
