/******
@name ContentList element
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class ContentList
@description A class and jQuery-plugin for "contentlist element" on notebook pages.
    Element generates a list (or a grid) from elements of given content type on this notebook.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Dependency problem in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 * - DOMPurify
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.localize;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
        typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){
    
    /**
     * Helper functions
     */
    
    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var element = new ContentList(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        }
    }
    
    $.fn.contentlistelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in %ccontentlist element%c.', 'color: green; background-color: #efe;');
            return false;
        };
    };

    /******
     * ContentList class
     * @class ContentList
     * @constructor
     * @param {jQuery} place - Place for the element
     * @param {Object} options - options for the element
     ******/

    var ContentList = function(place, options) {
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.show();
        this.setStyles();
        this.setAttrs();
    };
    
    /******
     * Init the element
     * @param {Object} options - initing data
     ******/
    ContentList.prototype.init = function(options) {
        options = $.extend(true, {}, this.defaults, options);
        this.settings = options.settings;
        this.data = options.data;
        this.metadata = $.extend(true, {
            creator: this.settings.username,
            created: (new Date()).getTime(),
            modifier: this.settings.username,
            modified: (new Date()).getTime(),
            tags: []
        }, options.metadata);
        this.itemdata = {};
        this.name = this.place.attr('data-element-name');
        this.setMode(this.settings.mode);
    };
    
    /******
     * Show the element
     ******/
    ContentList.prototype.show = function() {
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        };
    };
    
    /******
     * Show the element in view mode
     ******/
    ContentList.prototype.view = function() {
        this.place.html(this.templates[this.data.viewmode] || '');
        this.itemlist = this.place.children('.contentlist-itemlist');
        this.place.trigger('getindexdata', {type: this.data.type});
    };
    
    /******
     * Show the element in edit mode
     ******/
    ContentList.prototype.edit = function() {
        var uilang = this.settings.uilang;
        var html = [
            '<ul class="contentlist-configitem">',
            '<li><label title="'+ebooklocalizer.localize('contentlist:list', uilang)+'"><input class="contentlist-input" data-inputkey="viewmode" type="radio" value="list" name="contentlist-viewmode-'+escapeHTML(this.name)+'" '+(this.data.viewmode === 'list' ? 'checked="checked"' : '')+'> '+ContentList.icons.listview+'</label></li>',
            '<li><label title="'+ebooklocalizer.localize('contentlist:grid', uilang)+'"><input class="contentlist-input" data-inputkey="viewmode" type="radio" value="grid" name="contentlist-viewmode-'+escapeHTML(this.name)+'" '+(this.data.viewmode === 'grid' ? 'checked="checked"' : '')+'> '+ContentList.icons.gridview+'</label></li>',
            '</ul>',
            '<div class="contentlist-configitem">',
            '<select class="contentlist-input" data-inputkey="type">',
            '<option value="imageelement" '+(this.data.type === 'imageelement' ? 'selected' : '')+'>Kuvat</option>',
            '<option value="mediaelement" '+(this.data.type === 'mediaelement' ? 'selected' : '')+'>Media</option>',
            '</select>',
            '</div>'
        ]
        this.place.html(html.join('\n'));
    };
    
    /******
     * Show the list of elements after fetching
     * @param {Object} data - data of indexes {type: "<type of element>", indexdata: [{indexdatas}, ...]}
     ******/
    ContentList.prototype.showData = function(data) {
        var itemhtml = [], itemdata, pageid = '', newpageid;
        for (var i = 0, len = data.indexdata.length; i < len; i++) {
            itemdata = data.indexdata[i];
            this.itemdata[itemdata.id] = itemdata;
            newpageid = itemdata.gotolink.chapid;
            if (this.data.viewmode === 'list' && pageid !== newpageid) {
                pageid = newpageid;
                itemhtml.push('</ul>');
                itemhtml.push('<h3><span class="contentlist-pagenum">' + escapeHTML(itemdata.pagenumber) + '</span> <span class="contentlist-pagetitle">' + escapeHTML(itemdata.pagetitle) + '</span></h3>');
                itemhtml.push('<ul class="contentlist-itemlist">')
            }
            itemhtml.push('<li class="contentlist-listitem" data-elementtype="' + escapeHTML(data.type) + '" data-elementid="'+escapeHTML(itemdata.id)+'">');
            itemhtml.push(this.itemHtml[data.type](itemdata));
            itemhtml.push('</li>');
        };
        this.itemlist.html(itemhtml.join('\n'));
    };
    
    /******
     * Set some attributes in html
     ******/
    ContentList.prototype.setAttrs = function() {
        this.place.attr('data-viewmode', this.data.viewmode);
    };
    
    /******
     * Do when data changed
     ******/
    ContentList.prototype.changed = function() {
        this.updateMetadata();
        this.place.trigger('element_changed', {type: 'contentlistelement'});
    };
    
    /******
     * Update metadata
     ******/
    ContentList.prototype.updateMetadata = function() {
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Init event handlers
     ******/
    ContentList.prototype.initHandlers = function() {
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEditable();
        } else {
            this.initHandlersNoneditable();
        };
    };
    
    /******
     * Init common event handlers
     ******/
    ContentList.prototype.initHandlersCommon = function() {
        var element = this;
        this.place.on('getdata', function(event){
            event.stopPropagation();
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    }
    
    /******
     * Init non-editable event handlers
     ******/
    ContentList.prototype.initHandlersNoneditable = function() {
        var element = this;
        this.place.off('reply_getindexdata').on('reply_getindexdata', function(event, data) {
            event.stopPropagation();
            if (element.data.type === data.type) {
                element.showData(data);
            };
        }).off('click', '.contentlist-listitem').on('click', '.contentlist-listitem', function(event, data) {
            event.stopPropagation();
            var item = $(this);
            var elid = item.attr('data-elementid');
            var gotolink = JSON.parse(JSON.stringify(element.itemdata[elid].gotolink));
            var linkdata = {
                type: 'contentlink',
                metadata: {},
                data: {
                    link: gotolink
                }
            };
            item.trigger('gotolink', linkdata);
        }).off('click', '.contentlist-item-source a').on('click', '.contentlist-item-source a', function(event, data) {
            event.stopPropagation();
            // Only follow link, don't do "gotolink".
        });
    };
    
    /******
     * Init editable event handlers
     ******/
    ContentList.prototype.initHandlersEditable = function() {
        var element = this;
        this.place.off('change', '.contentlist-input').on('change', '.contentlist-input', function(event, data){
            event.stopPropagation();
            var input = $(this);
            var inputkey = input.attr('data-inputkey');
            var value = input.val();
            element.data[inputkey] = value;
            element.changed();
        });
    }
    
    /******
     * Set mode
     ******/
    ContentList.prototype.setMode = function(mode) {
        var rights = this.modes[mode];
        if (typeof(rights) === 'undefined') {
            mode = 'view';
            rights = this.modes[mode];
        };
        this.settings.mode = mode;
        this.editable = rights.editable;
        this.authorable = rights.authorable;
        this.reviewable = rights.reviewable;
    };
    
    /******
     * Get data
     * @returns {Object} data to be saved
     ******/
    ContentList.prototype.getData = function() {
        var result = {
            type: 'contentlistelement',
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: JSON.parse(JSON.stringify(this.data))
        };
        return result;
    }
    
    /******
     * Set styles, if needed
     ******/
    ContentList.prototype.setStyles = function() {
        if ($('style#contentliststyles').length === 0) {
            $('head').append('<style id="contentliststyles" type="text/css">' + this.styles + '</style>');
        };
    };
    
    /******
     * Functions to map index data to html list item by the type.
     * @param {Object} data - the index data of an element
     ******/
    ContentList.prototype.itemHtml = {
        imageelement: function(data) {
            var meta = data.metadata || {};
            var html = [
                '<div class="contentlist-itemthumbnail">',
                (meta.thumbnail ? '<img src="' + escapeHTML(meta.thumbnail) + '">' : ContentList.icons.image),
                '</div>',
                '<div class="contentlist-itemtext">',
                '<div class="contentlist-item-notes">' + (escapeHTML(meta.notes) || '') + ' [' + escapeHTML(data.lang) + ']' + '</div>',
                '<div class="contentlist-item-filename">' + (escapeHTML(meta.file) || '') + '</div>',
                '<div class="contentlist-item-author">' + (escapeHTML(meta.author) || '') + '</div>',
                '<div class="contentlist-item-license">' + (escapeHTML(meta.license) || '') + '</div>',
                (meta.source ? '<div class="contentlist-item-source"><a href="'+escapeHTML(meta.source)+'" target="_blank">' + escapeHTML(meta.source) + '</a></div>' : ''),
                '</div>'
            ];
            return html.join('\n');
        },
        mediaelement: function(data) {
            var meta = data.metadata || {};
            var html = [
                '<div class="contentlist-itemthumbnail">',
                (meta.thumbnail ? (meta.thumbnail.substr(0,5) === 'data:' ? '<img src="' + escapeHTML(meta.thumbnail) + '">' : sanitize(meta.thumbnail)) : ContentList.icons.audio),
                '</div>',
                '<div class="contentlist-itemtext">',
                '<div class="contentlist-item-filename">' + (escapeHTML(meta.file) || '') + '</div>',
                '<div class="contentlist-item-author">' + (escapeHTML(meta.author) || '') + '</div>',
                '<div class="contentlist-item-license">' + (escapeHTML(meta.license) || '') + '</div>',
                (meta.source ? '<div class="contentlist-item-source"><a href="'+escapeHTML(meta.source)+'" target="_blank">' + escapeHTML(meta.source) + '</a></div>' : ''),
                '</div>'
            ];
            return html.join('\n');
        }
    };
    
    ContentList.prototype.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    };
    
    ContentList.prototype.defaults = {
        type: 'contentlistelement',
        metadata: {},
        data: {
            type: 'imageelement',
            viewmode: 'list'
        },
        settings: {
            lang: 'common',
            uilang: 'en',
            mode: 'view',
            role: 'student',
            username: 'Anonymous'
        }
    };
    
    ContentList.prototype.templates = {
        list: [
            '<ul class="contentlist-itemlist">',
            '</ul>'
        ].join('\n'),
        grid: [
            '<ul class="contentlist-itemlist">',
            '</ul>'
        ].join('\n')
    }
    
    ContentList.prototype.styles = [
        '.contentlist-itemlist {list-style: none; margin: 0 1em; padding: 0;}',
        '[data-viewmode="list"] .contentlist-listitem {margin: 0; padding: 0.3em; display: flex;}',
        '[data-viewmode="grid"] .contentlist-itemlist {display: -ms-flex; display: -webkit-flex; display: flex; flex-direction: row; flex-wrap: wrap; justify-content: flex-start; align-items: stretch; align-content: flex-start;}',
        '[data-viewmode="grid"] .contentlist-listitem {margin: 0.2em; padding: 0.3em; vertical-align: top; width: 180px; overflow: hidden;}',
        '.contentlist-listitem:nth-child(odd) {background-color: rgba(0,0,0,0.05);}',
        '.contentlist-listitem:nth-child(even) {background-color: rgba(0,0,0,0.02);}',
        '.contentlist-itemthumbnail {text-align: center; min-width: 56px; min-height: 56px; cursor: pointer;}',
        '.contentlist-itemthumbnail svg {display: inline-block;}',
        'ul.contentlist-configitem {list-style: none; padding: 0;}',
        '.contentlist-configitem {display: inline-block; vertical-align: top; margin: 1em;}',
        '.contentlist-configitem label {cursor: pointer;}',
        '.contentlist-configitem label svg, .contentlist-configitem label input {display: inline-block; vertical-align: middle;}',
        '.contentlist-item-filename {overflow-wrap: break-word; word-wrap: break-word; -ms-word-break: break-all; font-family: monospace; font-size: 80%;}'
    ].join('\n');
    
    ContentList.icons = {
        image: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="50" height="50" viewBox="0 0 30 30" class="mini-icon mini-icon-image"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 22 l22 0 l0 -22z m7 2 a5 5 0 0 0 0 10 a5 5 0 0 0 0 -10 m8 7 l5 10 l-16 0 l4 -6 l3 3z" /></svg>',
        audio: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-audio"><path class="mini-icon-maincontent" style="stroke: none;" d="M5 12 h3 l8 -8 v22 l-8 -8 h-3z m13 -1 a8 8 0 0 1 0 8 l1 1 a9 9 0 0 0 0 -10z m3 -3 a11 11 0 0 1 0 14 l1 1 a12 12 0 0 0 0 -16z"></path></svg>',
        listview: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-list"><path style="stroke: none;" d="M5 4 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m0 9 a2.5 2.5 0 0 1 0 5 a2.5 2.5 0 0 1 0 -5z m7.5 -18 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z m0 9 l12.5 0 a2.5 2.5 0 0 1 0 5 l-12.5 0 a2.5 2.5 0 0 1 0 -5z"></path></svg>',
        gridview: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-grid"><path style="stroke: none;" d="M3 3 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m-18 9 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z m9 0 l6 0 l0 6 l-6 0z"></path></svg>'
    };
    
    ContentList.localization = {
        "en": {
            "contentlist:view": "View",
            "contentlist:list": "List",
            "contentlist:grid": "Grid",
            "contentlist:images": "Images",
            "contentlist:boxes": "Boxes"
        },
        "fi": {
            "contentlist:view": "Näkymä",
            "contentlist:list": "Lista",
            "contentlist:grid": "Ruudukko",
            "contentlist:images": "Kuvat",
            "contentlist:boxes": "Laatikot"
        },
        "sv": {
            "contentlist:view": "Vy",
            "contentlist:list": "Lista",
            "contentlist:grid": "Rutnät",
            "contentlist:images": "Bilder",
            "contentlist:boxes": "Lådor"
        }
    };
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(ContentList.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(ContentList.localization);
    };

    /******
     * Info about element (icon, description, etc.)
     ******/
    ContentList.elementinfo = {
        type: 'contentlistelement',
        elementtype: ['elements','teacherelements'],
        jquery: 'contentlistelement',
        name: 'ContentList',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-image"><path style="stroke: none;" d="M3 3 l24 0 l0 24 l-24 0z m1 1 l0 22 l22 0 l0 -22z m7 2 a5 5 0 0 0 0 10 a5 5 0 0 0 0 -10 m8 7 l5 10 l-16 0 l4 -6 l3 3z" /></svg>',
        description: {
            en: 'Content list',
            fi: 'Sisältöluettelo',
            sv: 'Innehållsförteckning'
        },
        roles: ['teacher', 'author'],
        classes: ['lists'],
        weight: 950
    }
    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', ContentList.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', ContentList.elementinfo);
    }

    
})(window, jQuery, window.ebooklocalizer);