/**
 * Requirements:
 * - jQuery
 * - DOMPurify
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
    typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

;(function(window, $){
    
    /******
     * TODO:
     * - .changed() - triggers "reviewchanged"-event (or whatever) with the data
     ******/

    /**
     * Helper functions
     */
    
    /**
     * Sanitizer
     */
    var sanitize = function(text, options) {
        options = $.extend({
            SAFE_FOR_JQUERY: true
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };

    /******
     * Commentor for reviewing
     ******/
    var Commentor = function(place, options) {
        options = $.extend(true, {}, this.defaults, options);
        this.place = $(place);
        this.place.addClass('commentortoolinuse');
        this.id = options.data.id || this.createId();
        var annotations = options.data.annotations;
        this.annotations = [];
        this.annotationsById = {};
        var annotation;
        for (var i = 0, len = annotations.length; i < len; i++) {
            annotation = annotations[i];
            this.annotationsById[annotation.id] = annotation;
            this.annotations.push(annotation.id);
        };
        if (options.settings.margin) {
            this.margin = options.settings.margin;
        } else {
            this.margin = $('<div></div>');
            $('body').append(this.margin);
        };
        this.removeHandlers();
        this.margin.addClass('commentor-margin');
        this.margin.html(this.templates.marginarea);
        this.marginarea = this.margin.children('.commentor-marginarea');
        this.annoui = this.margin.children('.commentor-annoui');
        this.dialog = this.margin.find('.commentor-annoui-dialogwindow');
        this.initButtons();
        this.usemargin = options.settings.usemargin;
        this.dialogpos = {xcoord: 10, ycoord: 140};
        this.settings = options.settings;
        this.setMode();
        this.setAttrs();
        this.initHandlers();
        this.show();
        this.setStyles();
    };
    
    /******
     * Create new id for commentor
     ******/
    Commentor.prototype.createId = function(){
        var id = ['commentortool'];
        id.push((new Date()).getTime());
        var rnd = '000000' + (Math.floor(Math.random() * 1000000));
        id.push(rnd.substr(rnd.length - 6));
        return id.join('');
    };
    
    /******
     * Init editing buttons
     ******/
    Commentor.prototype.initButtons = function() {
        var ctype;
        var uibuttons = this.annoui.find('.commentor-annoui-buttonset');
        var html = [];
        for (var i = 0, len = this.ctypes.length; i < len; i++) {
            ctype = this.ctypes[i];
            html.push('<li class="ffwidget-setbutton" data-annotationtype="'+ctype.type+'"><span class="commentor-annoui-buttonbg"><span class="commentor-annoui-example">' + ctype.icon + '</span></span></li>');
        };
        uibuttons.html(html.join(''));
    };
    
    /******
     * Convert given node and offset to a location (xpath, offset). Exclude commentor's own span's.
     * @param {DOM-node} node - DOM-node (textnode or elementnode)
     * @param {Number} offset - offset inside given node.
     * @returns {Object} Location object as {xpath: '<string>', offset: <number>}
     ******/
    Commentor.prototype.nodeToLocation = function(node, offset) {
        var xp = '';
        var totalOffset = offset;
        var root = this.place.get(0);
        var currnode = node;
        var tagname, parent, siblings, sibling, index, offsetFound = false, offsetadd;
        while (currnode !== root && currnode !== document.body) {
            tagname = currnode.tagName;
            parent = currnode.parentNode;
            //if (currnode.nodeType !== 3) {
                // Do this, if not textnode
                siblings = parent.childNodes;
                index = 1;
                offsetadd = 0;
                for (var i = 0, len = siblings.length; i < len; i++) {
                    // Count siblings before and with same tag.
                    // Count offset to add (if needed)
                    sibling = siblings[i];
                    if (sibling === currnode) {
                        // Reached the current node.
                        if (currnode.nodeType !== 3 && !currnode.classList.contains('commentorhighlightelement')) {
                            // Ignore, if commentorelement, otherwise add to xpath
                            xp = '/' + tagname + '[' + index + ']' + xp;
                            // Offset base is reached, no need for counting offset after this.
                            offsetFound = true;
                        };
                        break;
                    };
                    // This is done for siblings before current node.
                    // Add the length of this sibling to the offset (if needed)
                    offsetadd += (sibling.nodeType === 3 ? sibling.length : sibling.textContent.length);
                    // Add the count of siblings with same tag as current node.
                    if (sibling.nodeType === 1 && sibling.tagName === tagname) {
                        index++;
                    };
                };
                // If the offset base was not yet found, increase the total offset.
                if (!offsetFound){
                    totalOffset += offsetadd;
                }
            //};
            // Continue iterating with parent as current node.
            currnode = parent;
        };
        return {xpath: xp.toLowerCase(), offset: totalOffset};
    };
    
    /******
     * Convert given location to a node and offset.
     * @param {String} xpath
     * @param {Number} offset
     * @returns {Object} an object in format {node: <DOM-node>, offset: <number>}
     ******/
    Commentor.prototype.locationToNode = function(xpath, offset) {
        var xplist = xpath.split('/');
        var result = {};
        var root = this.place;
        root.addClass('commentorfoundpath');
        var node = root.get(0);
        var parts, tagname, index, candidates, error = false;
        for (var i = 1, len = xplist.length; i < len; i++) {
            parts = xplist[i].split('[');
            tagname = parts[0];
            index = parseInt(parts[1]);
            candidates = this.findNodesOfType(node, tagname);
            if (candidates[index - 1]) {
                node = candidates[index - 1];
                node.classList.add('commentorfoundpath');
            } else {
                error = true;
            };
        };
        if (error) {
            result = false;
        } else {
            result = this.findNodeWithOffset(node, offset);
        };
        this.place.find('.commentorfoundpath').removeClass('commentorfoundpath');
        root.removeClass('commentorfoundpath');
        return result;
    };
    
    /******
     * Find a list of nodes with given tagname. Only node itself, if matching, or
     * search from children recoursively, if node has class commentorhighlightelement.
     * @param {DOM-node} node
     * @param {String} tagname - tag to search for.
     * @returns {Array} an array of nodes of tagname
     ******/
    Commentor.prototype.findNodesOfType = function(node, tagname) {
        var result = [];
        if (node.classList.contains('commentorhighlightelement') || node.classList.contains('commentorfoundpath')) {
            var children = node.childNodes;
            for (var i = 0, len = children.length; i < len; i++) {
                if (children[i].nodeType !== 3) {
                    result = result.concat(this.findNodesOfType(children[i], tagname));
                };
            };
        } else if (node.tagName.toLowerCase() === tagname.toLowerCase()) {
            result.push(node);
        };
        return result;
    };
    
    /******
     * Find the (text)node and offset relative to it with given node and it's offset
     * @param {DOM-node} node
     * @param {Number} offset - offset from the beginning of node
     * @returns {Object} object of format {node: <DOM-textnode>, offset: <number>}
     ******/
    Commentor.prototype.findNodeWithOffset = function(node, offset) {
        var result = {
            node: node,
            offset: offset
        };
        if (node.nodeType !== 3) {
            // If the node is not textnode, search for its children
            var children = node.childNodes;
            var sibling, osleft = offset;
            for (var i = 0, len = children.length; i < len; i++) {
                sibling = children[i];
                if (sibling.nodeType === 3 && osleft <= sibling.length) {
                    // This is the textnode we are looking for.
                    result.node = sibling;
                    result.offset = osleft;
                    break;
                } else if (sibling.nodeType !== 3 && osleft <= sibling.textContent.length) {
                    // This is children we are lookin for, but we must go inside recoursively.
                    result = this.findNodeWithOffset(sibling, osleft);
                    break;
                } else {
                    // These are not the droids you are lookin for.
                    // Decrease osleft by length of this sibling and go for the next one.
                    osleft = osleft - (sibling.nodeType === 3 ? sibling.length : sibling.textContent.length);
                };
            };
        };
        return result;
    };
    
    /******
     * Get annotation quote
     * @param {String} aid - id of annotation
     * @returns {String} text of the annotation's range
     ******/
    Commentor.prototype.getAnnotationQuote = function(aid) {
        var quote = [];
        var annotation = this.annotationsById[aid];
        var range, start, end, selrange;
        for (var i = 0, len = annotation.ranges.length; i < len; i++) {
            range = annotation.ranges[i];
            start = this.locationToNode(range.start, range.startOffset);
            end = this.locationToNode(range.end, range.endOffset);
            try {
                if (start && end) {
                    selrange = document.createRange();
                    selrange.setStart(start.node, start.offset);
                    selrange.setEnd(end.node, end.offset);
                    selection = document.getSelection();
                    selection.removeAllRanges();
                    selection.addRange(selrange);
                    quote.push(selection.toString());
                };
            } catch (err) {
                console.log('Error selecting comment range:', err);
            }
        };
        return quote.join('');
    };
    
    /******
     * Add annotation data to the list of annotations
     * @param {Object} annotation - annotation data
     * @returns {String} id of the new annotation
     ******/
    Commentor.prototype.addAnnotation = function(annotation) {
        this.annotations.push(annotation.id);
        this.annotationsById[annotation.id] = annotation;
        this.changed();
        return annotation.id;
    };
    
    /******
     * Remove annotation data from the list of annotations
     * @param {Object} aid - annotation id
     ******/
    Commentor.prototype.removeAnnotation = function(aid) {
        this.hideAnnotation(aid);
        delete this.annotationsById[aid];
        var index = this.annotations.indexOf(aid);
        this.annotations.splice(index, 1);
        this.marginarea.children('[data-annotationid="'+aid+'"][data-commentorid="'+this.id+'"]').remove();
        this.changed();
    };
    
    /******
     * Create new annotation from selection
     * @param {Object} selection - from document.getSelection()
     * @returns {Object} - new annotation object
     ******/
    Commentor.prototype.createNewAnnotation = function(selection) {
        var rnd = '0000' + Math.floor(1000*Math.random());
        rnd = rnd.substr(rnd.length - 4);
        var anno = {
            id: 'commentorannotation' + (new Date()).getTime() + rnd,
            quote: selection.toString(),
            text: '',
            ranges: [],
            type: '1',
            timestamp: (new Date()).getTime(),
            userid: this.settings.username
        };
        var range, start, end;
        for (var i = 0, len = selection.rangeCount; i < len; i++){
            range = selection.getRangeAt(i);
            start = this.nodeToLocation(range.startContainer, range.startOffset);
            end = this.nodeToLocation(range.endContainer, range.endOffset);
            anno.ranges.push({
                start: start.xpath,
                startOffset: start.offset,
                end: end.xpath,
                endOffset: end.offset
            });
        };
        this.changed();
        return anno;
    };
    
    /******
     * Show comments
     ******/
    Commentor.prototype.show = function(){
        if (this.editable) {
            this.edit();
        } else {
            this.view();
        };
    };
    
    /******
     * Show comments in view mode
     ******/
    Commentor.prototype.view = function(){
        this.marginarea.empty();
        for (var i = 0, len = this.annotations.length; i < len; i++) {
            this.hideAnnotation(this.annotations[i]);
            this.showAnnotation(this.annotations[i]);
        };
    };
    
    /******
     * Hide all comments
     ******/
    Commentor.prototype.hide = function() {
        this.marginarea.empty();
        for (var i = 0, len = this.annotations.length; i < len; i++) {
            this.hideAnnotation(this.annotations[i]);
        };
    };
    
    /******
     * Show comments in edit mode
     ******/
    Commentor.prototype.edit = Commentor.prototype.view;

    /******
     * Show highlighting
     * @param {String} aid - id of annotation object with format:
     *      {
     *          id: '<id of the annotation>',
     *          quote: '<the highlighted string>',
     *          text: '<comment text>',
     *          ranges: [{
     *              start: '<xpath string>',
     *              startOffset: <number>,
     *              end: '<spath string>',
     *              endOffset: <number>
     *          },
     *              ...
     *          ],
     *          type: <string>
     *      }
     ******/
    Commentor.prototype.showAnnotation = function(aid){
        var annotation = this.annotationsById[aid];
        var range, start, end, quote, selrange, selection, fixelems, fxelem;
        quote = this.getAnnotationQuote(aid);
        var q = this.annotationsById[aid].quote;
        this.place.attr('spellcheck', 'false').attr('contenteditable', true);
        for (var i = 0, len = annotation.ranges.length; i < len; i++) {
            range = annotation.ranges[i];
            start = this.locationToNode(range.start, range.startOffset);
            end = this.locationToNode(range.end, range.endOffset);
            if (start && end) {
                // Start and end locations were found successfully.
                try {
                    selrange = document.createRange();
                    selrange.setStart(start.node, start.offset);
                    selrange.setEnd(end.node, end.offset);
                    selection = document.getSelection();
                    selection.removeAllRanges();
                    selection.addRange(selrange);
                    document.execCommand('hiliteColor', null, 'LightGoldenRodYellow');
                    fixelems = this.place.find('span[style="background-color: LightGoldenRodYellow;"], span[style="background-color: lightgoldenrodyellow;"], span[style="background-color: rgb(250, 250, 210);"]');
                    for (var j = 0, flen = fixelems.length; j < flen; j++) {
                        // Change highlighting from inline styling to classess and attributes.
                        fxelem = fixelems.eq(j);
                        annotation.type = (quote === annotation.quote ? annotation.type : '0');
                        fxelem
                            .removeAttr('style')
                            .addClass('commentorhighlightelement')
                            .attr('data-annotationtype', escapeHTML(annotation.type))
                            .attr('data-annotationid', escapeHTML(annotation.id))
                            .attr('data-commentorid', escapeHTML(this.id))
                            .attr('data-annotationtext', escapeHTML(annotation.text));
                        // Remove empty spans.
                        if (/^\s*$/.test(fxelem.html())) {
                            fxelem.replaceWith(fxelem.html());
                        };
                    };
                    selection.removeAllRanges();
                } catch (err) {
                    console.log('Error on selecting comment range: ', err);
                }
            };
        };
        this.place.removeAttr('contenteditable').removeAttr('spellcheck');
        this.updateMarginbox(aid);
        //var margindata = {
        //    annotationid: annotation.id,
        //    text: annotation.text
        //};
        //this.marginarea.append(this.fillTemplate(this.templates.marginbox, margindata));
    };
    
    /******
     * Hide annotation from the page
     ******/
    Commentor.prototype.hideAnnotation = function(aid) {
        var elements = this.place.find('.commentorhighlightelement[data-annotationid="'+aid+'"]');
        var elem;
        for (var i = 0, len = elements.length; i < len; i++) {
            elem = elements.eq(i);
            elem.replaceWith(elem.html());
        };
    };
    
    /******
     * Update or create the marginbox
     * @param {String} aid - annotation id
     ******/
    Commentor.prototype.updateMarginbox = function(aid) {
        var annotation = this.annotationsById[aid];
        var marginbox = this.marginarea.children('.commentor-marginbox[data-annotationid="'+aid+'"][data-commentorid="'+this.id+'"]');
        var margindata = {
            annotationid: escapeHTML(aid),
            text: this.render(annotation.text),
            quote: '<p>'+escapeHTML(annotation.quote)+'</p>',
            commentorid: escapeHTML(this.id),
            atype: escapeHTML(annotation.type),
            date: escapeHTML(annotation.timestamp ? (new Intl.DateTimeFormat(this.settings.uilang, {weekday: 'short', year: 'numeric', month: 'numeric', day: 'numeric', hour: 'numeric', minute: 'numeric'}).format(new Date(annotation.timestamp))) : '')
        };
        if (marginbox.length === 0) {
            this.marginarea.append(this.fillTemplate(this.templates.marginbox, margindata));
            marginbox = this.marginarea.children('.commentor-marginbox[data-annotationid="'+aid+'"][data-commentorid="'+this.id+'"]');
        } else {
            marginbox.html(this.fillTemplate(this.templates.marginboxcontent, margindata));
            marginbox.attr('data-annotationtype', annotation.type);
        };
        if (annotation.text === '') {
            marginbox.addClass('commentor-marginbox-empty');
        } else {
            marginbox.removeClass('commentor-marginbox-empty');
        }
    };
    
    /******
     * Render annotation text as markdown, if possible. Otherwise just sanitize.
     * @param {String} text - annotation text
     * @returns {String} rendered text.
     ******/
    Commentor.prototype.render = function(text) {
        var html = text;
        try {
            if (typeof(marked) === 'function') {
                html = marked(text);
            } else {
                html = '<div class="commentor-plaintext">' + escapeHTML(text) + '</div>';
            };
        } catch (err) {
            html = '<div class="commentor-plaintext">' + escapeHTML(text) + '</div>';
        };
        return html;
    };
    
    /******
     * Init handlers of the commentor.
     ******/
    Commentor.prototype.initHandlers = function(){
        this.initHandlersCommon();
        if (this.editable) {
            this.initHandlersEdit();
        } else {
            this.initHandlersView();
        }
    };
    
    /******
     * Remove all handlers (of previous instances with same id)
     ******/
    Commentor.prototype.removeHandlers = function() {
        this.place
            .off('getdata')
            .off('showcomments.commentor'+this.id)
            .off('hidecomments.commentor'+this.id)
            .off('mousemove.commentor'+this.id, '.commentorhighlightelement')
            .off('mouseleave.commentor'+this.id, '.commentorhighlightelement')
            .off('mouseup.commentor'+this.id)
            .off('click.commentor'+this.id, '.commentorhighlightelement[data-annotationid][data-commentorid="'+this.id+'"]')
            .off('commentordestroy');
        this.margin.off('mousemove.commentor'+this.id, '.commentor-marginbox')
            .off('mouseleave.commentor'+this.id, '.commentor-marginbox')
            .off('click.commentor'+this.id, '.commentor-annoui-buttonset li')
            .off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="remove"]')
            .off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="cancel"]')
            .off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="save"]')
            .off('focusout.commentor'+this.id, '.commentor-annoui-textbox')
            .off('click.commentor'+this.id, '.commentorhighlightelement[data-annotationid][data-commentorid="'+this.id+'"]')
            .off('click.commentor'+this.id, '.commentor-marginbox[data-annotationid][data-commentorid="'+this.id+'"]');
    };
    
    /******
     * Init handlers of the commentor common for edit and view mode
     ******/
    Commentor.prototype.initHandlersCommon = function(){
        var anno = this;
        this.place.off('getdata').on('getdata', function(event, data) {
            event.stopPropagation();
            anno.place.data('commentordata', anno.getData());
        });
        this.place.off('showcomments.commentor'+this.id).on('showcomments.'+this.id, function(event, data) {
            event.stopPropagation();
            anno.show();
        });
        this.place.off('hidecomments.commentor'+this.id).on('hidecomments.'+this.id, function(event, data) {
            event.stopPropagation();
            anno.hide();
        });
        this.place.off('commentordestroy').on('commentordestroy', function(event, data) {
            event.stopPropagation();
            anno.removeHandlers();
        })
        this.place.off('mousemove.commentor'+this.id, '.commentorhighlightelement').on('mousemove.commentor'+this.id, '.commentorhighlightelement', function(event, data){
            //event.stopPropagation();
            var highlight = $(this);
            var aid = highlight.attr('data-annotationid');
            var xcoord = event.originalEvent.clientX;
            var ycoord = event.originalEvent.clientY;
            anno.place.find('[data-annotationid="'+aid+'"]').addClass('commentor-mousehover');
            anno.showMarginbox(aid, xcoord, ycoord);
            //anno.marginarea.find('[data-annotationid="'+aid+'"]').addClass('commentor-mousehover');
        });
        this.place.off('mouseleave.commentor'+this.id, '.commentorhighlightelement').on('mouseleave.commentor'+this.id, '.commentorhighlightelement', function(event, data){
            //event.stopPropagation();
            var highlight = $(this);
            var aid = highlight.attr('data-annotationid');
            anno.place.find('[data-annotationid="'+aid+'"]').removeClass('commentor-mousehover');
            //anno.marginarea.find('[data-annotationid="'+aid+'"]').removeClass('commentor-mousehover');
            anno.hideMarginbox(aid);
        });
        this.margin.off('mousemove.commentor'+this.id, '.commentor-marginbox').on('mousemove.commentor'+this.id, '.commentor-marginbox', function(event, data){
            //event.stopPropagation();
            var highlight = $(this);
            var aid = highlight.attr('data-annotationid');
            anno.place.find('[data-annotationid="'+aid+'"]').addClass('commentor-mousehover');
            anno.marginarea.find('[data-annotationid="'+aid+'"]').addClass('commentor-mousehover');
        });
        this.margin.off('mouseleave.commentor'+this.id, '.commentor-marginbox').on('mouseleave.commentor'+this.id, '.commentor-marginbox', function(event, data){
            //event.stopPropagation();
            var highlight = $(this);
            var aid = highlight.attr('data-annotationid');
            anno.place.find('[data-annotationid="'+aid+'"]').removeClass('commentor-mousehover');
            anno.marginarea.find('[data-annotationid="'+aid+'"]').removeClass('commentor-mousehover');
        });
    };
    
    /******
     * Init handlers of the commentor in edit mode
     ******/
    Commentor.prototype.initHandlersEdit = function(){
        var anno = this;
        this.place.off('mouseup.commentor'+this.id).on('mouseup.commentor'+this.id, function(event, data){
            if (!document.getSelection().isCollapsed) {
                // We have selection: Make a new annotation
                event.stopPropagation();
                anno.newAnnotation = anno.createNewAnnotation(document.getSelection());
                var xcoord = event.originalEvent.clientX;
                var ycoord = event.originalEvent.clientY;
                anno.showAnnoUi(xcoord, ycoord);
            } else if (anno.margin.is('.commentor-editannotation')) {
                // We don't have selection, but the editing window was open:
                // Update comment from editing window and close it.
                var textarea = anno.dialog.find('textarea');
                var value = textarea.val();
                var aid = anno.dialog.attr('data-annotationid');
                anno.updateComment(aid, value);
                anno.hideAnnoUi();
            } else {
                // No selection, no window.
                anno.hideAnnoUi();
            };
        });
        this.margin.off('click.commentor'+this.id, '.commentor-annoui-buttonset li').on('click.commentor'+this.id, '.commentor-annoui-buttonset li', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var isdialog = $(this).closest('.commentor-annoui-dialogwindow').length > 0;            
            var ctype = $(this).attr('data-annotationtype');
            var aid;
            if (isdialog) {
                aid = anno.dialog.attr('data-annotationid');
                var annotation = anno.annotationsById[aid];
                anno.updateType(aid, ctype);
                anno.updateQuote(aid, anno.getAnnotationQuote(aid));
                anno.dialog.find('.commentor-annoui-buttonset li').removeClass('buttonselected');
                $(this).addClass('buttonselected');
                anno.hideAnnotation(aid);
                anno.showAnnotation(aid);
            } else {
                anno.newAnnotation.type = ctype;
                aid = anno.addAnnotation(anno.newAnnotation);
                anno.showAnnotation(aid);
                anno.order();
                anno.showCommentDialog(aid);
            }
        });
        this.margin.off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="remove"]').on('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="remove"]', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var aid = anno.dialog.attr('data-annotationid');
            anno.removeAnnotation(aid);
            anno.hideCommentDialog();
        });
        this.margin.off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="cancel"]').on('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="cancel"]', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var aid = anno.dialog.attr('data-annotationid');
            anno.annotationsById[aid] = anno.canceldata;
            anno.changed();
            anno.hideAnnotation(aid);
            anno.showAnnotation(aid);
            anno.hideCommentDialog();
        });
        this.margin.off('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="save"]').on('click.commentor'+this.id, '.commentor-annoui-dialog-buttonbar button[data-action="save"]', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var textarea = anno.dialog.find('textarea');
            var value = textarea.val();
            var aid = anno.dialog.attr('data-annotationid');
            anno.updateComment(aid, value);
            anno.hideCommentDialog();
        });
        this.margin.off('focusout.commentor'+this.id, '.commentor-annoui-textbox').on('focusout.commentor'+this.id, '.commentor-annoui-textbox', function(event, data) {
            event.stopPropagation();
            event.preventDefault();
            var textarea = anno.dialog.find('textarea');
            var value = textarea.val();
            var aid = anno.dialog.attr('data-annotationid');
            anno.updateComment(aid, value);
        });
        this.place.off('click.commentor'+this.id, '.commentorhighlightelement[data-annotationid][data-commentorid="'+this.id+'"]').on('click.commentor'+this.id, '.commentorhighlightelement[data-annotationid][data-commentorid="'+this.id+'"]', function(event, data){
            var commid = $(this).attr('data-commentorid');
            if (document.getSelection().isCollapsed) {
                event.stopPropagation();
                event.preventDefault();
                var aid = $(this).attr('data-annotationid');
                anno.showCommentDialog(aid);
            };
        });
        this.margin.off('click.commentor'+this.id, '.commentor-marginbox[data-annotationid][data-commentorid="'+this.id+'"]').on('click.commentor'+this.id, '.commentor-marginbox[data-annotationid][data-commentorid="'+this.id+'"]', function(event, data){
            event.stopPropagation();
            event.preventDefault();
            var aid = $(this).attr('data-annotationid');
            anno.showCommentDialog(aid);
        });
        // Moving dialog
        this.dialog.off('mousedown touchstart', '.commentor-annoui-headbar').on('mousedown touchstart', '.commentor-annoui-headbar', function(event, data){
            event.stopPropagation();
            var parent = anno.dialog;
            var offset = {top: parseInt(parent.css('top')), left: parseInt(parent.css('left'))};
            anno.startMovingDialog({x: event.clientX - offset.left, y: event.clientY - offset.top});
        });
        this.dialog.off('mousemove touchmove', '.commentor-annoui-headbar').on('mousemove touchmove', '.commentor-annoui-headbar', function(event, data){
            event.preventDefault();
        });
        this.dialog.off('click', '.commentor-annoui-headbar').on('click', '.commentor-annoui-headbar', function(event, data){
            event.preventDefault();
        });
    };
    
    /******
     * Init handlers of the commentor in view mode
     ******/
    Commentor.prototype.initHandlersView = function(){
        var anno = this;
    };
    
    /******
     * Show marginbox. If this.usemargin === true, highlight the marginbox, if false, then popup the marginbox.
     * @param {String} aid - annotation id
     * @param {Number} xcoord - x-coordinate of the mouse
     * @param {Number} ycoord - y-coordinate of the mouse
     ******/
    Commentor.prototype.showMarginbox = function(aid, xcoord, ycoord) {
        var marginbox = this.marginarea.find('[data-annotationid="'+escapeHTML(aid)+'"]');
        marginbox.addClass('commentor-mousehover');
        if (!this.usemargin) {
            this.marginarea.css({top: ycoord + 'px', left: xcoord + 'px'});
        };
    };
    
    /******
     * Hide marginbox. If this.usemargin === true, stop highlighting the marginbox, if false, then hide the popup.
     * @param {String} aid - annotation id
     ******/
    Commentor.prototype.hideMarginbox = function(aid) {
        var marginbox = this.marginarea.find('[data-annotationid="'+escapeHTML(aid)+'"]');
        marginbox.removeClass('commentor-mousehover');
        this.marginarea.removeAttr('style');
    };
    
    /******
     * Order annotations by order in the page
     ******/
    Commentor.prototype.order = function(){
        var hilits = this.place.find('.commentorhighlightelement[data-annotationid][data-commentorid="'+this.id+'"]');
        var annoids = [], hilit, aid;
        for (var i = 0, len = hilits.length; i < len; i++) {
            hilit = hilits.eq(i);
            aid = hilit.attr('data-annotationid');
            if (annoids.indexOf(aid) === -1) {
                annoids.push(aid);
            };
        };
        var mboxes;
        if (annoids.length === this.annotations.length) {
            this.annotations = annoids;
            mboxes = this.marginarea.children('.commentor-marginbox');
            mboxes.sort(function(a, b){
                var aid = $(a).attr('data-annotationid');
                var bid = $(b).attr('data-annotationid');
                var aindex = annoids.indexOf(aid);
                var bindex = annoids.indexOf(bid);
                return aindex - bindex;
            });
            this.marginarea.append(mboxes);
        };
    };
    
    /******
     * Set some attributes
     ******/
    Commentor.prototype.setAttrs = function(){
        if (this.usemargin) {
            this.marginarea.addClass('commentor-usemargin');
        } else {
            this.marginarea.addClass('commentor-popupmargin');
        };
    };
    
    /******
     * Set the mode of commentor ('edit'|'view'). Includes the checking of the validity.
     * @param {String} mode - the mode (optional)
     ******/
    Commentor.prototype.setMode = function(mode) {
        mode = this.availableModes[mode || this.settings.mode] || 'view';
        this.settings.mode = mode;
        this.editable = (mode === 'edit');
    };

    /******
     * Show user interface for adding new annotation
     * @param {Number} xcoord - mouse x-coordinate
     * @param {Number} ycoord - mouse y-coordinate
     ******/
    Commentor.prototype.showAnnoUi = function(xcoord, ycoord) {
        this.margin.addClass('commentor-showbuttons');
        this.annoui.find('.commentor-annoui-buttons').css({top: ycoord + 'px', left: xcoord + 'px'});
    };
    
    /******
     * Hide user interface for adding new annotation
     ******/
    Commentor.prototype.hideAnnoUi = function() {
        this.margin.removeClass('commentor-editannotation commentor-showbuttons');
    };

    /******
     * Show comment dialog
     ******/
    Commentor.prototype.showCommentDialog = function(aid) {
        this.margin.removeClass('commentor-showbuttons');
        this.margin.addClass('commentor-editannotation');
        this.updateDialogPos();
        this.dialog.attr('data-annotationid', aid);
        var annotation = this.annotationsById[aid];
        var ctype = annotation.type;
        var buttons = this.dialog.find('.commentor-annoui-buttonset li');
        buttons.removeClass('buttonselected');
        var selected = this.dialog.find('.commentor-annoui-buttonset li[data-annotationtype="'+ctype+'"]');
        selected.addClass('buttonselected');
        this.canceldata = JSON.parse(JSON.stringify(annotation));
        var textarea = this.dialog.find('textarea');
        textarea.val(annotation.text || '');
        textarea.focus();
        var selection = document.getSelection();
    };

    /******
     * Hide comment dialog
     ******/
    Commentor.prototype.hideCommentDialog = function() {
        this.margin.removeClass('commentor-editannotation');
        this.dialog.find('textarea').val('');
    };
    
    /******
     * Update the position of dialog
     * @param {Number} xcoord - x-coordinate
     * @param {Number} ycoord - y-coordinate
     ******/
    Commentor.prototype.updateDialogPos = function(xcoord, ycoord) {
        if (typeof(xcoord) !== 'undefined' && typeof(ycoord) !== 'undefined') {
            this.dialogpos.xcoord = xcoord;
            this.dialogpos.ycoord = ycoord;
        };
        this.margin.find('.commentor-annoui-dialogwindow').css({left: this.dialogpos.xcoord, top: this.dialogpos.ycoord});
    };
    
    /******
     * Update the textvalue of the currently editable annotation
     * @param {String} aid - annotation id
     * @param {String} text - new comment text
     ******/
    Commentor.prototype.updateComment = function(aid, text) {
        var annotation = this.annotationsById[aid];
        if (annotation && annotation.text !== text) {
            annotation.text = text;
            this.updateMarginbox(aid);
            this.changed();
        };
    };
    
    /******
     * Update the comment type of given annotation
     * @param {String} aid - annotation id
     * @param {String} atype - annotation type
     ******/
    Commentor.prototype.updateType = function(aid, atype) {
        var annotation = this.annotationsById[aid];
        annotation.type = atype;
        this.changed();
    };
    
    /******
     * Update the quote of given annotation
     * @param {String} aid - annotation id
     * @param {String} quote - annotation quote
     ******/
    Commentor.prototype.updateQuote = function(aid, quote) {
        var annotation = this.annotationsById[aid];
        annotation.quote = quote;
        if (annotation.text === '') {
            this.updateMarginbox(aid);
        };
        this.changed();
    };
    
    /******
     * Start moving dialog window
     ******/
    Commentor.prototype.startMovingDialog = function(coords) {
        var anno = this;
        this.moveOffset = coords;
        $('body').addClass('annotationdialogmoving')
            .off('mousemove.commentordialog touchmove.commentordialog')
            .on('mousemove.commentordialog touchmove.commentordialog', function(event, data){
                var coords = {x: event.clientX, y: event.clientY};
                anno.moveDialog(coords);
            })
            .off('mouseup.commentordialog touchend.commentordialog')
            .on('mouseup.commentordialog touchend.commentordialog', function(event, data){
                anno.stopMovingDialog();
            });
    };
    
    /******
     * Stop moving the dialog
     ******/
    Commentor.prototype.stopMovingDialog = function(){
        $('body').removeClass('annotationdialogmoving')
            .off('mousemove.commentordialog touchmove.commentordialog')
            .off('mouseup.commentordialog touchend.commentordialog');
    };
    
    /******
     * Move the form dialog
     ******/
    Commentor.prototype.moveDialog = function(coords){
        var position = {
            xcoord: coords.x - this.moveOffset.x,
            ycoord: coords.y - this.moveOffset.y
        };
        this.updateDialogPos(position.xcoord, position.ycoord);
    };
    
    /******
     * Trigger event ect. when comments have changed
     ******/
    Commentor.prototype.changed = function() {
        this.place.trigger('commentorchanged', [this.getData()]);
    };

    /******
     * Get the data of this commentor
     * @returns {Object} data of commentor
     ******/
    Commentor.prototype.getData = function() {
        var result ={
            data: {
                id: this.id,
                annotations: []
            }
        };
        var aid, annotation;
        for (var i = 0, len = this.annotations.length; i < len; i++) {
            aid = this.annotations[i];
            annotation = this.annotationsById[aid];
            result.data.annotations.push({
                id: annotation.id,
                quote: annotation.quote,
                text: annotation.text,
                ranges: JSON.parse(JSON.stringify(annotation.ranges)),
                type: annotation.type,
                timestamp: annotation.timestamp,
                userid: annotation.userid || ''
            });
        };
        return result;
    }
    
    /******
     * Set CSS-styles, if needed.
     ******/
    Commentor.prototype.setStyles = function(){
        if ($('#commentorstylesheets').length === 0) {
            $('head').append('<style id="commentorstylesheets" type="text/css">' + this.styles + '</style>');
        };
    };
    
    /******
     * Fill data in template text
     * @param {String} template - text with placeholders as '{{% variablename %}}'
     * @param {Object} mapping - object that maps variablenames to values
     * @returns {String} template replaced with given values for variables
     ******/
    Commentor.prototype.fillTemplate = function(template, mapping) {
        var rex;
        for (var key in mapping) {
            rex = RegExp('{{%\\s*'+key+'\\s*%}}', 'g')
            template = template.replace(rex, mapping[key] + '');
        };
        return template;
    };
    
    Commentor.prototype.availableModes = {
        edit: 'edit',
        view: 'view'
    };
    
    Commentor.prototype.defaults = {
        data: {
            id: '',
            annotations: [],
            margin: null,
            usemargin: false
        },
        settings: {
            mode: 'view',
            uilang: 'en'
        }
    };
    
    Commentor.prototype.ctypes = [
        {
            name: 'Highlight 1',
            type: '1',
            icon: 'A'
        },
        {
            name: 'Highlight 2',
            type: '2',
            icon: 'B'
        },
        {
            name: 'Highlight 3',
            type: '3',
            icon: 'C'
        },
        {
            name: 'Highlight 4',
            type: '4',
            icon: 'D'
        },
        {
            name: 'Underline 1',
            type: '5',
            icon: 'E'
        },
        {
            name: 'Underline 2',
            type: '6',
            icon: 'F'
        }
    ];
    
    Commentor.prototype.icons = {
        remove: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-remove"><path style="stroke: none;" d="M5 5.5 l7 -2 l-0.2 -1 l2 -0.4 l0.2 1 l7 -2 l0.6 2 l-16 4.4 z M7 8 l16 0 l-3 20 l-10 0z M9 10 l2 15 l2 0 l-1 -15z M13.5 10 l0.5 15 l2 0 l0.5 -15z M21 10 l-3 0 l-1 15 l2 0z"></path></svg>',
        cancel: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-cancel"><circle style="fill: white;" cx="15" cy="15" r="13"></circle><path style="stroke: none;" d="M15 2 a13 13 0 0 1 0 26 a13 13 0 0 1 0 -26z m0 10 l-4 -4 l-3 3 l4 4 l-4 4 l3 3 l4 -4 l4 4 l3 -3 l-4 -4 l4 -4 l-3 -3 l-4 4z"></path></svg>',
        save: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-ok"><circle style="fill: white;" cx="15" cy="15" r="13"></circle><path style="stroke: none;" d="M15 2 a13 13 0 0 1 0 26 a13 13 0 0 1 0 -26z m-2 15 l-4 -4 l-3 3 l7 7 l11 -11 l-3 -3z"></path></svg>'
    }
    
    Commentor.prototype.templates = {
        marginarea: [
            '<div class="commentor-marginarea"></div>',
            '<div class="commentor-annoui">',
            '    <div class="commentor-annoui-buttons"><ul class="commentor-annoui-buttonset ffwidget-buttonset ffwidget-horizontal"></ul></div>',
            '    <div class="commentor-annoui-dialogwindow ffwidget-background">',
            '        <div class="commentor-annoui-headbar ffwidget-background-colored"></div>',
            '        <div class="commentor-annoui-toolbar"><ul class="commentor-annoui-buttonset ffwidget-buttonset ffwidget-horizontal"></ul></div>',
            '        <div class="commentor-annoui-textarea"><textarea class="commentor-annoui-textbox" rows="8"></textarea></div>',
            '        <div class="commentor-annoui-dialog-buttonbar ffwidget-buttonset ffwidget-horizontal"><button class="ffwidget-setbutton" data-action="remove">'+Commentor.prototype.icons.remove+'</button><button class="ffwidget-setbutton" data-action="cancel">'+Commentor.prototype.icons.cancel+'</button><button class="ffwidget-setbutton" data-action="save">'+Commentor.prototype.icons.save+'</button></div>',
            '    </div>',
            '</div>'
        ].join('\n'),
        marginbox: [
            '<div class="commentor-marginbox" data-annotationtype="{{% atype %}}" data-annotationid="{{% annotationid %}}" data-commentorid="{{% commentorid %}}">',
            '  <div class="commentor-marginbox-head">',
            '    <div class="commentor-marginbox-date">{{% date %}}</div>',
            '  </div>',
            '  <div class="commentor-marginbox-quote">{{% quote %}}</div>',
            '  <div class="commentor-marginbox-body">',
            '    {{% text %}}',
            '  </div>',
            '</div>'
        ].join('\n'),
        marginboxcontent: [
            '  <div class="commentor-marginbox-head">',
            '    <div class="commentor-marginbox-date">{{% date %}}</div>',
            '  </div>',
            '  <div class="commentor-marginbox-quote">{{% quote %}}</div>',
            '  <div class="commentor-marginbox-body">',
            '    {{% text %}}',
            '  </div>',
        ].join('\n')
    }
    
    Commentor.prototype.styles = [
        // Margin and marginboxes
        '.commentor-margin .commentor-marginbox {box-sizing: border-box; margin: 0.7em 0.4em; font-size: 75%; border: 1px solid #aaa; border-radius: 0.2em; background-color: #fafafa; box-shadow: 0 0 3px rgba(100, 100, 100, 0.5); cursor: pointer;}',
        '.commentor-margin .commentor-marginbox-body {padding: 0.2em 0.5em;}',
        '.commentor-marginarea.commentor-popupmargin {position: fixed; margin: 5px; padding: 0; z-index: 1000;}',
        '.commentor-marginarea.commentor-popupmargin .commentor-marginbox {display: none;}',
        '.commentor-marginarea.commentor-popupmargin .commentor-marginbox.commentor-mousehover {display: block; max-width: 30em; padding: 0; font-family: sans-serif; background-color: #ffd; box-shadow: 5px 5px 5px rgba(0,0,0,0.5);}',
        '.commentor-margin .commentor-marginbox[data-annotationtype="0"] {border: 1px solid red;}',
        '.commentor-marginarea.commentor-popupmargin .commentor-marginbox.commentor-mousehover:empty {display: none;}',
        '.commentor-plaintext {white-space: pre-wrap;}',
        '.commentor-marginbox ul, .commentor-marginbox ol {padding-left: 1.5em;}',
        '.commentor-marginbox table {border-collapse: collapse; margin: 0 auto;}',
        '.commentor-marginbox table td, .commentor-marginbox table th {padding: 0.2em 0.4em; border: 1px solid #888;}',
        '.commentor-marginbox-head {background-color: rgba(200,200,200,0.5);}',
        '.commentor-marginbox-date {font-size: 80%; text-align: right; margin: 0 0.2em;}',
        '.commentor-marginbox-quote {display: none; padding: 0.2em 0.5em;}',
        '.commentor-marginbox-empty .commentor-marginbox-quote {display: block;}',
        // Annoui
        '.commentor-margin .commentor-annoui-buttons {display: none; transform: translateX(-50%); margin-top: 0.5em; box-shadow: 5px 5px 10px rgba(0,0,0,0.4);}',
        '.commentor-margin .commentor-annoui-buttonset {list-style: none; margin: 0; padding: 0; min-height: 20px; min-width: 20px; display: flex; flex-direction: row; background-color: #eee;}',
        '.commentor-margin.commentor-showbuttons .commentor-annoui-buttons {display: block; position: fixed;}',
        '.commentor-annoui {min-width: 30em;}',
        '.commentor-annoui-headbar {min-height: 8px; border-radius: 4px;}',
        // Annoui buttons
        '.commentor-margin .commentor-annoui-buttonset li {padding: 4px; cursor: pointer;}',
        '.commentor-margin .commentor-annoui-buttonset li.buttonselected {box-shadow: inset 2px 2px 4px rgba(0,0,0,0.3), inset -2px -2px 4px rgba(255,255,255,0.5); background-color: rgba(255,255,255,0.5);}',
        '.commentor-margin .commentor-annoui-buttonset li:hover {background-color: rgba(255,255,255,0.5);}',
        '.commentor-annoui-buttonset li span {padding: 0 0.3em;}',
        // Annoui dialog
        '.commentor-margin .commentor-annoui-dialogwindow {display: none;-webkit-touch-callout: none; -webkit-user-select: none; -khtml-user-select: none; -moz-user-select: none; -ms-user-select: none; user-select: none; z-index: 1000;}',
        '.commentor-margin.commentor-editannotation .commentor-annoui-dialogwindow {display: block; position: fixed; padding: 2px; border: 1px solid #999; border-radius: 2px; background-color: #eee;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-headbar {min-height: 16px; border-radius: 8px; cursor: move; background-color: #aaa;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-textarea textarea {box-sizing: border-box; min-width: 30em; padding: 0.3em; overflow-y: scroll; background: white; color: black;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-dialog-buttonbar {margin: 0; display: flex;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-dialog-buttonbar button {flex-grow: 1;}',
        '.commentor-margin .mini-icon-remove path {fill: black;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-dialog-buttonbar button .mini-icon-cancel path {fill: #a00;}',
        '.commentor-margin .commentor-annoui-dialogwindow .commentor-annoui-dialog-buttonbar button .mini-icon-ok path {fill: #0a0;}',
        // Highlight
        '.commentortoolinuse .commentorhighlightelement {cursor: pointer;}',
        // buttons
        '.commentor-annoui-buttonset span.commentor-annoui-buttonbg {background-color: white; display: inline-block; padding: 0; margin: 0.2em; border-radius: 2px;color: black;}',
        // Default styles for annotation types
        // Type 0 (error - wrong quote text)
        '.commentorhighlightelement[data-annotationtype="0"] {background: transparent url("data:image/svg+xml;utf8,<svg xmlns=\'http://www.w3.org/2000/svg\' version=\'1.1\' width=\'10\' height=\'5\' viewBox=\'0 0 20 10\'><path style=\'fill: none; stroke: red; stroke-width: 4px;\' d=\'M0 0 l10 10 l10 -10\'></path></svg>") left bottom repeat-x;}',
        // Type 1
        '.commentorhighlightelement[data-annotationtype="1"], .commentor-annoui-buttonset [data-annotationtype="1"] span.commentor-annoui-example {background-color: rgba(255,255,170,0.6);}',
        // Type 2
        '.commentorhighlightelement[data-annotationtype="2"], .commentor-annoui-buttonset [data-annotationtype="2"] span.commentor-annoui-example{background-color: rgba(255,170,170,0.6);}',
        // Type 3
        '.commentorhighlightelement[data-annotationtype="3"], .commentor-annoui-buttonset [data-annotationtype="3"] span.commentor-annoui-example {background-color: rgba(100,200,255,0.6);}',
        // Type 4
        '.commentorhighlightelement[data-annotationtype="4"], .commentor-annoui-buttonset [data-annotationtype="4"] span.commentor-annoui-example {background-color: rgba(170,255,170,0.6);}',
        // Type 5
        '.commentorhighlightelement[data-annotationtype="5"], .commentor-annoui-buttonset [data-annotationtype="5"] span.commentor-annoui-example {border-bottom: 2px solid red;}',
        '.commentor-annoui-buttonset [data-annotationtype="5"] span.commentor-annoui-example {box-shadow: inset 0 -2px 0 red;}',
        // Type 6
        '.commentorhighlightelement[data-annotationtype="6"], .commentor-annoui-buttonset [data-annotationtype="6"] span.commentor-annoui-example {border-bottom: 2px solid blue;}',
        '.commentor-annoui-buttonset [data-annotationtype="6"] span.commentor-annoui-example {box-shadow: inset 0 -2px 0 blue;}',
        // Hover highlight
        //'.commentorhighlightelement.commentor-mousehover {box-shadow: 0 0 0px 2px rgba(0,0,0,0.4); border-radius: 2px;}',
        '.commentorhighlightelement.commentor-mousehover {box-shadow: 0 0 8px 2px rgba(255,255,0,0.3), 0 0 0px 1px rgba(0,0,0,0.4); border-radius: 2px;}',
        //'.commentorhighlightelement.commentor-mousehover {background-color: rgba(255,100,100,0.5);}',
        '.commentor-marginbox.commentor-mousehover {box-shadow: 0 0 3px rgba(100, 100, 100, 0.5), inset 0 0 15px rgba(255,255,170,0.9); background-color: #ffd;}'
    ].join('\n');

    
   

    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var tool = new Commentor(this, params);
            });
        },
        'getdata': function(){
            this.trigger('getdata');
            return this.data('commentordata');
        }
    };
    
    $.fn.commentor = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in commentor.');
            return false;
        }
    }
    
})(window, jQuery);