/**
 * Requirements:
 * - jQuery
 */
try {
    typeof(jQuery) === 'undefined' && jQuery;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 * - ElementSet
 * - ElementPanel
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) === 'undefined' && EbookLocalizer.localize;
        typeof(jQuery.fn.elementset) === 'undefined' && jQuery.fn.elementset.apply;
        typeof(jQuery.fn.elementpanel) === 'undefined' && jQuery.fn.elementpanel.apply;
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

/**
 * Runtime requirements
 * - CodeMirror
 */

if (typeof(checkRuntimeRequirements) !== 'undefined' && checkRuntimeRequirements) {
    try {
        typeof(CodeMirror) === 'undefined' && CodeMirror.apply;
    } catch (err) {
        throw new Error('Missing runtime dependency in ' + err.fileName + '\n' + err);
    }
}

(function($){
    
    
    /**
     * Helper functions
     */
    
    /**
     * Escape html for security
     */
    var escapeHTML = function(html) {
        return document.createElement('div')
            .appendChild(document.createTextNode(html))
            .parentNode
            .innerHTML
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
    };
    
    /******
     * Code element to show and edit programming code.
     * @requires codemirror.js code-editor library <https://codemirror.net/>
     * @constructor
     * @param {jQuery} place - place for element
     * @param {Object} options - data for the element
     ******/
    var CodeElement = function(place, options){
        this.place = $(place);
        this.place.addClass('codeelement');
        this.setStyles();
        this.init(options);
        this.show();
    }
    
    /******
     * Init codeelement element
     * @param {Object} options - initing data of the element
     ******/
    CodeElement.prototype.init = function(options){
        options = $.extend(true, {}, CodeElement.defaults, options);
        this.settings = $.extend(true, {}, CodeElement.defaults.settings, options.settings);
        this.type = 'codeelement';
        this.setMode(options.settings.mode);
        this.text = options.data.text;
        this.codelang = options.data.codelang;
        this.height = options.data.height;
        this.metadata = {
            creator: options.metadata.creator,
            created: options.metadata.created,
            modifier: options.metadata.modifier,
            modified: options.metadata.modified,
            lang: options.metadata.lang,
            tags: options.metadata.tags
        };
        this.codemodes = {};
        this.codemodelist = [];
        var cmode;
        var cmodelist;
        if (this.settings.codemirror) {
            for (var lang in CodeMirror.modes) {
                if (CodeMirror.modes.hasOwnProperty(lang) && CodeElement.codemodes[lang]) {
                    cmodelist = CodeElement.codemodes[lang];
                    for (var i = 0, len = cmodelist.length; i < len; i++) {
                        this.codemodes[cmodelist[i].name] = cmodelist[i];
                        this.codemodelist.push(cmodelist[i]);
                    };
                } else {
                    cmode = {
                        name: lang,
                        label: lang,
                        mode: lang
                    };
                    this.codemodes[lang] = cmode;
                    this.codemodelist.push(cmode);
                };
            };
            this.codemodelist.sort(function(a, b){
                return (a.label < b.label ? -1 : 1);
            });
        };
    };
    
    /******
     * Set style-tag if needed.
     ******/
    CodeElement.prototype.setStyles = function(){
        if ($('head style#codeelement-style').length === 0) {
            $('head').append('<style id="codeelement-style" type="text/css">'+CodeElement.styles+'</style>')
        }
    }
    
    /******
     * Set the mode of the element.
     * @param {String} mode - the mode of element: 'view', 'edit', 'review', 'author',...
     ******/
    CodeElement.prototype.setMode = function(mode){
        this.settings.mode = mode || 'view';
        var modesettings = CodeElement.modes[mode] || CodeElement.modes.view;
        this.editable = modesettings.editable;
        this.authorable = modesettings.authorable;
        this.reviewable = modesettings.reviewable;
    }
    
    /******
     * Set and use mode
     * @param {String} mode - the mode of element
     ******/
    CodeElement.prototype.changeMode = function(mode){
        this.setMode(mode);
        this.show();
    }
    
    /******
     * Show the CodeElement element in different modes
     ******/
    CodeElement.prototype.show = function(){
        this.place.attr('data-elementmode', this.settings.mode);
        if (this.metadata.lang) {
            this.place.attr('lang', this.metadata.lang);
            this.place.attr('uilang', this.settings.uilang);
        }
        this.removeHandlers();
        this.initHandlersCommon();
        if (!this.editable) {
            this.view();
            this.initHandlersNoneditable();
        } else {
            this.edit();
            this.initHandlersEditable()
        }
    }
    
    /******
     * Show the element in non-editable view mode.
     ******/
    CodeElement.prototype.view = function(){
        this.place.empty().html(CodeElement.templates.view);
        this.cblock = this.place.children('.codeelement-codeblock');
        if (this.settings.codemirror) {
            var options = $.extend(true, {}, CodeElement.cmsettings, {
                readOnly: true,
                value: this.text,
                mode: this.codemodes[this.codelang].mode || 'null'
            });
            if (this.height > 0) {
                //options.viewportMargin = 40;
                this.place.removeClass('codeblockAutoresize');
                this.codemirror = CodeMirror(this.cblock.get(0), options);
                this.place.find('.CodeMirror').css('height', 1.4*this.height + 'em');
            } else {
                options.viewportMargin = Infinity;
                this.place.addClass('codeblockAutoresize');
                this.codemirror = CodeMirror(this.cblock.get(0), options);
                this.place.find('.CodeMirror').css('height', '');
            };
        } else {
            this.cblock.html([
                '<pre class="codeelement-codetext">',
                escapeHTML(this.text),
                '</pre>'
            ].join('\n'));
        };
    }
    
    /******
     * Show the element in editable mode.
     ******/
    CodeElement.prototype.edit = function(){
        this.place.empty();
        this.place.html(CodeElement.templates.edit);
        this.cblock = this.place.children('.codeelement-codeblock');
        if (this.settings.codemirror) {
            this.showControls();
            var options = $.extend(true, {}, CodeElement.cmsettings, {
                value: this.text,
                mode: this.codemodes[this.codelang].mode || 'null'
            });
            if (this.height > 0) {
                //options.viewportMargin = 40;
                this.place.removeClass('codeblockAutoresize');
                this.codemirror = CodeMirror(this.cblock.get(0), options);
                this.place.find('.CodeMirror').css('height', 1.4*this.height + 'em');
            } else {
                options.viewportMargin = Infinity;
                this.place.addClass('codeblockAutoresize');
                this.codemirror = CodeMirror(this.cblock.get(0), options);
                this.place.find('.CodeMirror').css('height', '');
            };
        } else {
            this.cblock.html([
                '<textarea class="codeelement-codetext">',
                escapeHTML(this.text),
                '</textarea>'
            ].join('\n'));
        };
    };
    
    /******
     * Show controlbar for editing
     ******/
    CodeElement.prototype.showControls = function(){
        var uilang = this.settings.uilang;
        var controlbar = this.place.find('.codeelement-controlbar');
        controlbar.empty();
        var html = ['<form>'];
        // Height range
        html.push('<label title="' + ebooklocalizer.localize('codeelement:height', uilang) + '">'+CodeElement.icons.height+'</label><span class="codeelement-controlheight"></span><input class="codeelement-inputheight" type="range" value="'+escapeHTML(this.height)+'" max="50" min="0" />');
        // Language selection
        html.push('<select data-key="codelang" class="codeelement-input">');
        var mode, selected;
        for (var i = 0, len = this.codemodelist.length; i < len; i++) {
            mode = this.codemodelist[i];
            selected = (this.codelang === mode.name ? 'selected="selected"' : '');
            html.push('<option value="'+mode.name+'" '+selected+'>'+mode.label+'</option>');
        };
        html.push('</select>')
        html.push('</form>');
        controlbar.append(html.join('\n'));
        this.updateHeight();
    };
    
    /******
     * Update the height in controlbar
     ******/
    CodeElement.prototype.updateHeight = function(){
        var height = this.height;
        if (height === 0) {
            height = 'auto';
        }
        this.place.find('.codeelement-controlheight').html(height);
    };
    
    /******
     * Remove all event handlers
     ******/
    CodeElement.prototype.removeHandlers = function(){
        this.place.off();
    }
    
    /******
     * Init handlers for all modes.
     ******/
    CodeElement.prototype.initHandlersCommon = function(){
        var element = this;
        this.place.on('setdata', function(e, data){
            e.stopPropagation();
            element.init(data);
        }).on('setmode', function(e, data){
            e.stopPropagation();
            if (data in CodeElement.modes) {
                element.changeMode(data);
            }
        }).on('view', function(e){
            e.stopPropagation();
            element.changeMode('view');
        }).on('edit', function(e){
            e.stopPropagation();
            element.changeMode('edit');
        }).on('review', function(e){
            e.stopPropagation();
            element.changeMode('review');
        }).on('author', function(e){
            e.stopPropagation();
            element.changeMode('author');
        }).on('getdata', function(e){
            var data = element.getData();
            element.place.data('[[elementdata]]', data);
        });
    }
    
    /******
     * Init handlers for non-editable mode
     ******/
    CodeElement.prototype.initHandlersNoneditable = function(){
        var element = this;
        this.place.on('refresh', function(e){
            e.stopPropagation();
            element.view();
        });
    }
    
    /******
     * Init handlers for editable mode
     ******/
    CodeElement.prototype.initHandlersEditable = function(){
        var element = this;
        this.place.on('blur', '.codeelement-codeblock', function(event, data){
            var changed = element.getText();
            if (changed) {
                element.updateMetadata();
                element.changed();
            };
        });
        this.place.on('change', '.codeelement-controlbar .codeelement-input', function(event, data){
            var input = $(this);
            var value = input.val();
            var key = input.attr('data-key');
            element[key] = value;
            element.changed();
            element.show();
        });
        this.place.on('input', '.codeelement-controlbar .codeelement-inputheight', function(event, data){
            var input = $(this);
            var value = input.val() | 0;
            element.height = value;
            element.updateHeight();
            element.changed();
        });
        this.place.on('change', '.codeelement-controlbar .codeelement-inputheight', function(event, data){
            var input = $(this);
            var value = input.val() | 0;
            element.height = value;
            element.updateHeight();
            element.changed();
            element.show();
        });
    }
    
    /******
     * Update the metadata of the element.
     ******/
    CodeElement.prototype.updateMetadata = function(){
        this.metadata.creator = this.metadata.creator || this.settings.username;
        this.metadata.created = this.metadata.created || (new Date()).getTime();
        this.metadata.modifier = this.settings.username;
        this.metadata.modified = (new Date()).getTime();
    };
    
    /******
     * Trigger element_changed event.
     ******/
    CodeElement.prototype.changed = function(){
        this.place.trigger('element_changed', {type: 'codeelement'});
    }

    /******
     * Update the content text from editor
     ******/
    CodeElement.prototype.getText = function(){
        var oldtext = this.text;
        if (this.settings.codemirror) {
            this.text = this.codemirror.getDoc().getValue();
        } else {
            this.text = this.cblock.children('textarea').val();
        };
        // Return true, if the text has changed.
        return (oldtext !== this.text);
    };
    
    /******
     * Get data of the element
     * @returns {Object} data of this markdown element of format:
     *   {
     *      "type": "markdownelement",
     *      "metadata": {
     *          "creator": "...",
     *          "created": "...",
     *          "modifier": "...",
     *          "modified": "...",
     *          "tags": []
     *      },
     *      "data": {"text": this.text}
     *   }
     ******/
    CodeElement.prototype.getData = function(){
        var result = {
            type: this.type,
            metadata: JSON.parse(JSON.stringify(this.metadata)),
            data: {
                text: this.text,
                codelang: this.codelang,
                height: this.height
            }
        }
        return result;
    }
    
    /******
     * Default settings
     ******/
    CodeElement.defaults = {
        type: 'codeelement',
        metadata: {
            creator: '',
            created: '',
            modifier: '',
            modified: '',
            tags: []
        },
        data: {
            text: "",
            codelang: "text",
            height: 20
        },
        settings: {
            mode: 'view',
            preview: false,
            uilang: 'en',
            lang: 'en',
            codemirror: (typeof(CodeMirror) === 'function')
        }
    }
    
    /******
     * CodeMirror settings
     ******/
    CodeElement.cmsettings = {
        lineNumbers: true,
        showCursorWhenSelecting: true,
        indentUnit: 4,
        viewportMargin: 40
    };
    
    /******
     * CodeMirror language modes
     ******/
    CodeElement.codemodes = {
        null: [
            {
                name: 'text',
                label: 'Text',
                mode: 'null',
            }
        ],
        python: [
            {
                name: 'python',
                label: 'Python',
                mode: 'python'
            }
        ],
        javascript: [
            {
                name: 'javascript',
                label: 'Javascript',
                mode: 'javascript',
            }
        ],
        htmlmixed: [
            {
                name: 'htmlmixed',
                label: 'HTML',
                mode: 'htmlmixed'
            }
        ],
        markdown: [
            {
                name: 'markdown',
                label: 'Markdown',
                mode: 'markdown'
            }
        ],
        css: [
            {
                name: 'css',
                label: 'CSS',
                mode: 'css'
            }
        ],
        php: [
            {
                name: 'php',
                label: 'PHP',
                mode: 'application/x-httpd-php'
            }
        ],
        clike: [
            {
                name: 'c',
                label: 'C',
                mode: 'text/x-csrc'
            },
            {
                name: 'cpp',
                label: 'C++',
                mode: 'text/x-c++src'
            },
            {
                name: 'java',
                label: 'Java',
                mode: 'text/x-java'
            },
            {
                name: 'objectivec',
                label: 'Objective-C',
                mode: 'text/x-objectivec'
            },
            {
                name: 'scala',
                label: 'Scala',
                mode: 'text/x-scala'
            },
            {
                name: 'csharp',
                label: 'C#',
                mode: 'text/x-csharp'
            }
        ]
    }
    
    /******
     * Modes
     ******/
    CodeElement.modes = {
        view: {
            editable: false,
            authorable: false,
            reviewable: false
        },
        edit: {
            editable: true,
            authorable: false,
            reviewable: false
        },
        review: {
            editable: false,
            authorable: false,
            reviewable: true
        },
        author: {
            editable: true,
            authorable: true,
            reviewable: false
        }
    }
    
    /******
     * Templates
     ******/
    CodeElement.templates = {
        view: [
            '<div class="codeelement-codeblock"></div>'
        ].join('\n'),
        edit: [
            '<div class="codeelement-controlbar ffwidget-background"></div>',
            '<div class="codeelement-codeblock"></div>'
        ].join('\n')
    }
    
    /******
     * Icons
     ******/
    CodeElement.icons = {
        height: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewBox="0 0 30 30" class="mini-icon mini-icon-height"><path style="stroke: none;" d="M8 1 l14 0 l0 2 l-7 0 l4 4 l-2.5 0 l0 16 l2.5 0 l-4 4 l7 0 l0 2 l-14 0 l0 -2 l7 0 l-4 -4 l2.5 0 l0 -16 l-2.5 0 l4 -4 l-7 0z" /></svg>'
    }
    
    /******
     * Info about element (icon, description, etc.)
     ******/
    CodeElement.elementinfo = {
        type: 'codeelement',
        elementtype: ['elements', 'studentelements'],
        jquery: 'codeelement',
        name: 'CodeMirror',
        icon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-codeelement mini-icon-code"><path style="stroke: none;" d="M8 2 h2 v12 h-2 v-10 h-1z m7 4 a4 4 0 0 1 8 0 v4 a4 4 0 0 1 -8 0z m2 0 v4 a2 2 0 0 0 4 0 v-4 a2 2 0 0 0 -4 0z m-11 14 a4 4 0 0 1 8 0 v4 a4 4 0 0 1 -8 0z m2 0 v4 a2 2 0 0 0 4 0 v-4 a2 2 0 0 0 -4 0z m12 -4 h2 v12 h-2 v-10 h-1z m7 4 " /></svg>',
        icon2: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-codeelement"><path style="stroke: none;" d="M1 10 a5 5 0 0 1 10 0 l0 10 a5 5 0 0 1 -10 0z m2 0 l0 8 l6 -8 a3 3 0 0 0 -6 0z m6 2 l-6 8 a3 3 0 0 0 6 0z m5 -7 l2 0 l0 18 l2 0 l0 2 l-6 0 l0 -2 l2 0 l0 -14 l-2 0z m5 5 a5 5 0 0 1 10 0 l0 10 a5 5 0 0 1 -10 0z m2 0 l0 8 l6 -8 a3 3 0 0 0 -6 0z m6 2 l-6 8 a3 3 0 0 0 6 0z" /></svg>',
        description: {
            en: 'Code block',
            fi: 'Koodielementti',
            sv: 'Kodelement'
        },
        roles: ['teacher', 'student', 'author'],
        classes: ['programming']
    }
    
    /******
     * Styles
     ******/
    CodeElement.styles = [
        '.codeelement {max-width: 60em; min-width: 20em;}',
        '.codeelement-codeblock .CodeMirror {border: 1px solid #eee;}',
        '.codeelement.codeblockAutoresize .codeelement-codeblock .CodeMirror {height: auto;}',
        '.codeelement-controlbar {text-align: right; padding: 0 3px;}',
        '.codeelement-controlbar svg {height: 20px; width: auto; display: inline-block;}',
        '.codeelement-controlbar form {text-align: left; display: inline-block;}',
        '.codeelement-controlbar form > * {display: inline-block; vertical-align: middle;}',
        '.codeelement-controlbar .codeelement-controlheight {width: 2em; text-align: center; display: inline-block;}',
        '.codeelement[data-elementmode="view"] .CodeMirror-cursor {display: none;}',
        '.codeelement[data-elementmode="edit"] .codeelement-codeblock textarea.codeelement-codetext {box-sizing: border-box; width: 100%; min-height: 20em;}'
    ].join('\n');
    
    /******
     * Localization strings
     ******/
    CodeElement.localization = {
        "en": {
            "codeelement:height": "Height"
        },
        "fi": {
            "codeelement:height": "Korkeus"
        },
        "sv": {
            "codeelement:height": "Höjd"
        }
    }
    
    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(CodeElement.localization);
    } else {
        var ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(CodeElement.localization);
    }

    
    if (typeof($.fn.elementset) === 'function') {
        $.fn.elementset('addelementtype', CodeElement.elementinfo);
    }
    if (typeof($.fn.elementpanel) === 'function') {
        $.fn.elementpanel('addelementtype', CodeElement.elementinfo);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var codeelem = new CodeElement(this, params);
            });
        },
        'get': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[elementdata]]');
            return data;
        },
        'set': function(params){
            var $place = $(this);
            $place.trigger('setdata', [params]);
        },
        'setmode': function(params){
            var $place = $(this);
            $place.trigger('setmode', [params]);
        }
    }
    
    $.fn.codeelement = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in codeelement.');
            return false;
        }
    }

})(jQuery);