/**
 * Requirements:
 * - jQuery
 * - DOMPurify
 */

try {
    typeof(jQuery) === 'undefined' && jQuery;
    typeof(DOMPurify) === 'undefined' && DOMPurify.apply;
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

;(function(window, $, ebooklocalizer){
    
    window.Elpp = {};

    /*******************************************************************************************
     * Content data
     ******************************************************************************************/
    /**
     * TypeStorage - A class for storing referece data (Named contentdata which is connected to some reference-elements).
     * @constructor
     * @name TypeStorage
     * @param {String} contentType - the type of the content
     * @param {Object} data - the content data in ... format.
     * @param {String} defaultlag  The default language
     * @param {Boolean} singlalangmode   Use only one language ('common').
     */
    var TypeStorage = function(contentType, data, defaultlang, singlelangmode){
        if (typeof(data) === 'undefined') {
            data = {};
        };
        var lang, name, element;
        this.defaultlang = defaultlang;
        this.singlelangmode = singlelangmode;
        // Default language of content. This is used, if asked one is not found.
        // The last fallback is 'common' language.
        this.refs = JSON.parse(JSON.stringify(data.refs || {}));
        // A mapping of all references. Key: anchor element's id, value: an array of element id's.
        // this.refs = {
        //     "<anchorid>": ["<elementid>", ... <array of elementids associated to refid>],
        //     "<anchorid>": ["<elementid>", ... <array of elementids associated to refid>],
        //     ...
        // }
        this.contentdata = JSON.parse(JSON.stringify(data.contentdata || {}));
        // A storage for elements by languages. Key: elementid, value: element's data.
        // this.contentdata = {
        //     "common": {
        //         "<elementid>": {<element's data>},
        //         "<elementid>": {<element's data>},
        //         ...,
        //     },
        //     "en": {
        //         ...
        //     },...
        // }
        this.contentinfo = JSON.parse(JSON.stringify(data.contentinfo || {}));
        // Info about elements in this.contentdata grouped by language
        // this.contentinfo = {
        //     "common": {
        //         "<elementid>": {
        //             "name": "<elementid>",
        //             "contenttype": "<type of content>",
        //             "sendto": "<recipientid>",
        //             "sendfrom": "<senderid>",
        //             "timestamp": "<timestamp>",
        //             "read": false,
        //             "recipientopened": false,
        //             "isoutbox": false
        //         },
        //         ....
        //     },
        //     "en": {
        //         ...
        //     }
        // }
        this.contentType = contentType;
        if (this.singlelangmode) {
            for (lang in this.contentdata) {
                if (lang !== this.defaultlang) {
                    for (name in this.contentdata[lang]) {
                        element = this.contentdata[lang][name];
                        element.metadata = $.extend(true, {}, element.metadata, {lang: this.defaultlang});
                        this.contentdata[this.defaultlang][name] = element;
                    };
                    delete this.contentdata[lang];
                };
            };
            for (lang in this.contentinfo) {
                if (lang !== this.defaultlang) {
                    for (name in this.contentinfo[lang]) {
                        element = this.contentinfo[lang][name];
                        element.lang = this.defaultlang;
                        this.contentinfo[this.defaultlang][name] = element;
                    };
                    delete this.contentinfo[lang];
                };
            };
        };
        this.dirty = {};
        // For each language an array of elementids that are waiting to be saved locally.
        this.outbox = {};
        // For each language an array of elementids that are waiting to be sent to the server.
        var cinfo;
        for (lang in this.contentinfo) {
            for (name in this.contentinfo[lang]) {
                cinfo = this.contentinfo[lang][name];
                if (cinfo.isoutbox) {
                    this.sendOutbox({name: name, lang: lang, timestamp: cinfo.timestamp});
                };
            };
        };
    };
    
    window.Elpp.TypeStorage = TypeStorage;
    
    /**
     * Check if there are elements with anchor as anchor
     * @param {String} anchor - Id of anchor element
     * @returns {Boolean} true/false
     */
    TypeStorage.prototype.hasAnchor = function(anchor){
        return !!this.refs[anchor] && this.refs[anchor].length > 0;
    }
    
    /**
     * Check if there is content with this name and language
     * @param {String} name - name of a content element
     * @param {String} lang - the language of content
     * @returns {Boolean} true/false
     */
    TypeStorage.prototype.hasContent = function(name, lang){
        return !!this.contentdata[lang] && !!this.contentdata[lang][name];
    };
    
    /**
     * Check if there is content with this name in any language
     * @param {String} name - name of a content element
     * @returns {Boolean} true/false
     */
    TypeStorage.prototype.hasContentAnyLang = function(name){
        var result = false;
        var lang;
        for (lang in this.contentdata) {
            result = result || !!this.contentdata[lang][name];
        };
        return result;
    };
    
    /**
     * Check if there is contentinfo with this name and language
     * @param {String} name - the name of a content element
     * @param {String} lang - the language of content
     * @returns {Boolean} true/false whether contentinfo exists.
     */
    TypeStorage.prototype.hasContentinfo = function(name, lang){
        return !!this.contentinfo[lang] && !!this.contentinfo[lang][name];
    };
    
    /**
     * Set new contentdata with name and references to all anchor elements in refs
     * @param {Object} data - the data of the new content element
     *     data: {
     *         name: "<elementid>",
     *         lang: "<language>",
     *         contentType: "<type of content>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     * @param {Boolean} isdirty - indicates, if the data should be marked as dirty
     */
    TypeStorage.prototype.setData = function(data, isdirty){
        var name = data.name || '';
        var lang = (this.singlelangmode ? this.defaultlang : data.lang || this.defaultlang);
        var currTimestamp = this.getTimestamp(name, lang);
        var anchorlist, anchor, reflist;
        if (currTimestamp < (new Date(data.contentinfo.timestamp || 0).getTime())) {
            // Update only, if the data has newer timestamp.
            anchorlist = data.anchors || [];
            for (var i = 0, len = anchorlist.length; i < len; i++){
                anchor = anchorlist[i];
                if (!this.hasAnchor(anchor)) {
                    this.refs[anchor] = [];
                };
                reflist = this.refs[anchor];
                if (reflist.indexOf(name) === -1) {
                    reflist.push(name);
                };
            };
            if (!this.contentdata[lang]) {
                this.contentdata[lang] = {};
            };
            if (this.singlelangmode) {
                data.contentdata.metadata = $.extend(true, {}, data.contentdata.metadata, {lang: lang});
                data.contentinfo = $.extend(true, {}, data.contentinfo, {lang: lang});
            };
            this.contentdata[lang][name] = data.contentdata;
            if (!this.contentinfo[lang]) {
                this.contentinfo[lang] = {};
            };
            this.contentinfo[lang][name] = $.extend({
                name: name,
                contentType: this.contentType,
                lang: lang
            }, TypeStorage.defaultInfo, data.contentinfo);
            if (isdirty) {
                this.setDirty(data.contentinfo);
            };
            if (data.contentinfo.isoutbox) {
                // Mark outbox, if contentinfo says it is outbox.
                // Use timestamp from contentinfo or "now".
                this.sendOutbox({name: name, timestamp: data.contentinfo.timestamp || (new Date).getTime(), lang: lang});
            } else {
                // Not in outbox. Clean to be sure.
                this.cleanOutbox({name: name, timestamp: data.contentinfo.timestamp || (new Date).getTime(), lang: lang});
            };
        };
    };
    
    /**
     * Get content data in the same format as it was in setData.
     * @param {String} name - the name of a content element
     * @param {String} lang - the language
     * @returns {Object} - the data of content element
     *     result = {
     *         name: "<elementid>",
     *         lang: "<language>",
     *         contentType: "<type of content>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     */
    TypeStorage.prototype.getData = function(name, lang){
        var result = {};
        var anchor, anchorlist, index;
        if (this.hasContent(name, lang)) {
            result = {
                name: name,
                contentType: this.contentType,
                lang: lang,
                anchors: [],
                contentdata: this.getContentByName(name, lang),
                contentinfo: this.getContentInfo(name, lang)
            };
            for (anchor in this.refs) {
                anchorlist = this.refs[anchor];
                index = anchorlist.indexOf(name);
                if (index > -1) {
                    result.anchors.push(anchor);
                };
            };
        };
        return result;
    };
    
    /**
     * Get the current timestamp of the content
     * @param {String} name   the name of a content element
     * @param {String} lang   the language
     * @returns {Number}      The current timestamp (millis)
     */
    TypeStorage.prototype.getTimestamp = function(name, lang) {
        var result = 0;
        var cinfo;
        if (this.hasContent(name, lang)) {
            cinfo = this.contentinfo[lang] && this.contentinfo[lang][name] || {timestamp: 0}
            result = cinfo.timestamp;
        };
        return (new Date(result || 0)).getTime();
    };
    
    /**
     * Remove contentdata by element's name and all references of it.
     * @param {String} name - the name of content element
     * @param {String} lang - the language
     */
    TypeStorage.prototype.removeData = function(name, lang){
        if (this.hasContent(name, lang)) {
            if (this.isDirty(name, lang)) {
                this.cleanDirty({name: name, lang: lang, timestamp: 0, force: true});
            };
            if (this.isOutbox(name, lang)) {
                this.cleanOutbox({name: name, lang: lang, timestamp: 0, force: true});
            };
            delete this.contentinfo[lang][name];
            delete this.contentdata[lang][name];
            var reflist, index;
            if (!this.hasContentAnyLang(name)) {
                // Remove anchors, if the content doesn't exist in any language anymore.
                for (var anchor in this.refs) {
                    reflist = this.refs[anchor];
                    index = reflist.indexOf(name);
                    if (index > -1) {
                        reflist.splice(index, 1);
                    };
                    if (reflist.length === 0) {
                        delete this.refs[anchor];
                    };
                };
            };
        };
    };
    
    /**
     * Get a list of names of content data, which are connected to given anchor element.
     * @param {String} anchor - Id of the anchor element
     * @returns {String[]}   A list of names of content data
     */
    TypeStorage.prototype.getNamesByAnchor = function(anchor){
        return (this.refs[anchor] || []).slice();
    };
    
    /**
     * Get the content data with given name and language.
     * @param {String} name - name of a content element
     * @param {String} lang - the language of the content
     * @returns {Object} the content data of named element or false
     */
    TypeStorage.prototype.getContentByName = function(name, lang){
        var result = false;
        if (this.hasContent(name, lang)) {
            // Use asked language, if found.
            result = this.contentdata[lang][name];
        } else if (this.hasContent(name, this.defaultlang)) {
            // Else use default language, if found
            result = this.contentdata[this.defaultlang][name];
        } else if (this.hasContent(name, 'common')) {
            // Else fallback to 'common'-language, if found.
            result = this.contentdata['common'][name];
        };
        result = JSON.parse(JSON.stringify(result));
        return result;
    };
    
    /**
     * Get the elements for all asked anchors
     * @param {Array} anchorlist - a list of elements referred to (anchor elements)
     * @param {String} lang - the language the content is asked in. (optional)
     * @returns {Object} result with structure
     *    result: {
     *        contentType: "<contentType>",
     *        lang: "<content language>",
     *        refs: {
     *            "<anchorid>": ["<elementid>", ...],
     *            ...
     *        },
     *        contentdata: {
     *            "<elementid>": {<element data>},
     *            ...
     *        },
     *        contentinfo: {
     *            "<elementid">: {<element's contentinfo>},
     *            ....
     *        }
     *    }
     */
    TypeStorage.prototype.getContentByAnchor = function(anchorlist, lang){
        if (typeof(anchorlist) === 'string') {
            anchorlist = [anchorlist];
        };
        var name, anchor, reflist, uselang;
        var result = {
            contentType: this.contentType,
            refs: {},
            contentdata: {},
            contentinfo: {}
        };
        for (var i = 0, anclen = anchorlist.length; i < anclen; i++) {
            anchor = anchorlist[i];
            reflist = this.getNamesByAnchor(anchor);
            result.refs[anchor] = [];
            for (var j = 0, len = reflist.length; j < len; j++){
                name = reflist[j];
                uselang = false;
                if (this.hasContent(name, lang)) {
                    uselang = lang;
                } else if (this.hasContent(name, this.defaultlang)) {
                    uselang = this.defaultlang;
                } else if (this.hasContent(name, 'common')) {
                    uselang = 'common';
                } else {
                    //for (var testlang in this.contentdata) {
                    //    if (this.hasContent(name, testlang)) {
                    //        uselang = testlang;
                    //        break;
                    //    };
                    //};
                };
                if (uselang) {
                    result.contentdata[name] = this.getContentByName(name, uselang);
                    result.refs[anchor].push(name);
                    result.contentinfo[name] = this.getContentInfo(name, uselang);
                };
            };
        };
        result.lang = lang;
        return result;
    };
    
    /**
     * Get the elements' contentinfo for all asked anchors
     * Same as getContentByAnchor(), but witout contentdata.
     * @param {Array} anchorlist - a list of elements referred to (anchor elements)
     * @param {String} lang - the language the content is asked in. (optional)
     * @returns {Object} result with structure
     *    result: {
     *        contentType: "<contentType>",
     *        lang: "<content language>",
     *        refs: {
     *            "<anchorid>": ["<elementid>", ...],
     *            ...
     *        },
     *        contentinfo: {
     *            "<elementid">: {<element's contentinfo>},
     *            ....
     *        }
     *    }
     */
    TypeStorage.prototype.getContentInfoByAnchor = function(anchorlist, lang){
        if (typeof(anchorlist) === 'string') {
            anchorlist = [anchorlist];
        };
        var name, anchor, reflist, uselang;
        var result = {
            contentType: this.contentType,
            refs: {},
            contentinfo: {}
        };
        for (var i = 0, anclen = anchorlist.length; i < anclen; i++) {
            anchor = anchorlist[i];
            reflist = this.getNamesByAnchor(anchor);
            result.refs[anchor] = [];
            for (var j = 0, len = reflist.length; j < len; j++){
                name = reflist[j];
                uselang = false;
                if (this.hasContentinfo(name, lang)) {
                    uselang = lang;
                } else if (this.hasContentinfo(name, this.defaultlang)) {
                    uselang = this.defaultlang;
                } else if (this.hasContentinfo(name, 'common')) {
                    uselang = 'common';
                } else {
                    for (var testlang in this.contentinfo) {
                        if (this.hasContentinfo(name, testlang)) {
                            uselang = testlang;
                            break;
                        };
                    };
                };
                if (uselang) {
                    result.refs[anchor].push(name);
                    result.contentinfo[name] = this.getContentInfo(name, uselang);
                };
            };
        };
        result.lang = lang;
        return result;
    };
    
    /**
     * Set contentinfo related to given element with asked language.
     * @param {Object} contentinfo - contentinfo (includes name-attribute)
     * @param {String} lang - the language
     */
    TypeStorage.prototype.setContentInfo = function(contentinfo, lang){
        var name = contentinfo.name;
        contentinfo.lang = lang;
        if (name && lang && this.contentdata[lang] && this.contentdata[lang][name]) {
            if (!this.contentinfo[lang]) {
                this.contentinfo[lang] = {};
            };
            this.contentinfo[lang][name] = $.extend({}, TypeStorage.defaultInfo, contentinfo);
        };
    };
    
    /**
     * Update contentinfo related to given element. (changes only attributes given in contentinfo)
     * @param {Object} contentinfo - key-value pairs (includes name-attribute)
     * @param {String} lang - the language
     */
    TypeStorage.prototype.updateContentInfo = function(contentinfo, lang, isdirty){
        contentinfo = JSON.parse(JSON.stringify(contentinfo));  // Make a copy
        var name = contentinfo.name;
        if (name && lang && this.contentdata[lang] && this.contentdata[lang][name]) {
            if (!this.contentinfo[lang]) {
                this.contentinfo[lang] = {};
            };
            // Don't update isoutbox this way
            if (typeof(contentinfo.isoutbox) !== 'undefined') {
                delete contentinfo.isoutbox;
            };
            var cinfo = $.extend({}, this.contentinfo[lang][name], contentinfo);
            this.contentinfo[lang][name] = cinfo;
            if (isdirty) {
                this.setDirty(cinfo);
            }
        };
    };
    
    /**
     * Get content elements info for all asked elements
     * @param {String} elemid - the elementid
     * @param {String} lang - the language
     * @return {Object} result with elementids as key and contentinfos as values
     */
    TypeStorage.prototype.getContentInfo = function(elemid, lang){
        var result = {};
        if (this.contentinfo[lang]) {
            // Don't fallback to other languages, if version in asked language does not exist.
            result = JSON.parse(JSON.stringify(this.contentinfo[lang][elemid] || {name: elemid}));
            result.contentType = this.contentType;
        };
        return result;
    }
    
    /**
     * Mark the element as dirty (= waiting for local saving)
     * @param {Object} elementinfo - info about element to be marked as dirty
     *    {
     *        name: "<elementid>",
     *        timestamp: "<timestamp">,
     *        lang: "<language>"
     *    }
     */
    TypeStorage.prototype.setDirty = function(elementinfo){
        var name = elementinfo.name || '';
        var lang = elementinfo.lang || '';
        var timestamp = elementinfo.timestamp || '';
        if (name && timestamp) {
            if (!this.dirty[lang]) {
                this.dirty[lang] = [];
            };
            if (this.dirty[lang].indexOf(name) === -1) {
                this.dirty[lang].push(name);
            };
            this.updateContentInfo({name: name, timestamp: timestamp}, lang);
        };
    };
    
    /**
     * Clean the element from dirty list
     * @param {Object} element - the name and modified timestamp of the element to clean (has been saved)
     *      element = {
     *          name: "<element id>",
     *          lang: "<language>",
     *          timestamp: "<timestamp>",
     *          force: true|false
     *      }
     */
    TypeStorage.prototype.cleanDirty = function(element){
        var name = element.name || '';
        var lang = element.lang;
        var timestamp = new Date(element.timestamp || 0).getTime();
        if (this.contentinfo[lang] && this.dirty[lang]) {
            var cinfo = this.contentinfo[lang][name];
            var index = this.dirty[lang].indexOf(name);
            if (index > -1 && cinfo) {
                var lasttime = new Date(cinfo.timestamp || 0).getTime();
                if (timestamp >= lasttime || element.force) {
                    this.dirty[lang].splice(index, 1);
                };
            };
        };
    };
    
    /**
     * Check if the given element is dirty
     * @param {String} name - the name of the element
     * @param {String} lang - the language
     * @returns {Boolean} dirty status of named element
     */
    TypeStorage.prototype.isDirty = function(name, lang){
        var result = false;
        var dirtylist = [];
        if (lang) {
            // If language is given, use that language.
            dirtylist = this.dirty[lang] || [];
        } else {
            // Else, use all languages.
            for (var dlang in this.dirty) {
                dirtylist = dirtylist.concat(this.dirty[dlang]);
            };
        };
        if (name) {
            // If name is given, search for that name in dirtylist.
            result = (dirtylist.indexOf(name) > -1);
        } else {
            // Else, check if there are any dirty elements.
            result = dirtylist.length > 0;
        };
        return result;
    };
    
    /**
     * Get a list of all dirty elements
     * @param {String} lang - the language (optional)
     * @returns {Array} an array of all dirty elements
     */
    TypeStorage.prototype.getDirty = function(lang){
        var result, ddata = [];
        if (lang) {
            result = {};
            if (this.dirty[lang]) {
                ddata = this.dirty[lang].slice()
            };
            result[lang] = ddata;
        } else {
            result = JSON.parse(JSON.stringify(this.dirty));
        };
        return result;
    }
    
    /**
     * Get list of all elements in dirty
     * @param {String} lang - the language (optional)
     * @returns {Array} an array of element in dirty.
     */
    TypeStorage.prototype.getDirtyContent = function(lang){
        var result = [];
        if (lang && this.dirty[lang]) {
            var list = this.dirty[lang];
            for (var i = 0, len = list.length; i < len; i++) {
                result.push(this.getData(list[i], lang));
            };
        } else if (!lang) {
            var list;
            for (lang in this.dirty){
                list = this.dirty[lang];
                for (var i = 0, len = list.length; i < len; i++) {
                    result.push(this.getData(list[i], lang));
                };
            };
        };
        return result;
    };

    
    /**
     * Mark the element for sending (put in the outbox)
     * @param {Object} elementinfo - info about element to send to outbox
     *    {
     *        name: "<elementid>",
     *        timestamp: "<timestamp">,
     *        lang: "<language>"
     *    }
     * @param {String} lang - the language
     */
    TypeStorage.prototype.sendOutbox = function(elementinfo){
        var name = elementinfo.name || '';
        var lang = elementinfo.lang || '';
        var timestamp = elementinfo.timestamp || '';
        if (name && timestamp && lang) {
            if (!this.outbox[lang]) {
                this.outbox[lang] = [];
            };
            if (this.outbox[lang].indexOf(name) === -1) {
                this.outbox[lang].push(name);
            };
            elementinfo.lang = lang;
            elementinfo.timestamp = timestamp;
            elementinfo.isoutbox = true;
            this.updateContentInfo(elementinfo, lang);
        };
    };
    
    /**
     * Clean the element from outbox, if the current version has been sent
     * @param {Object} element - the name and modified timestamp of the element that has been sent
     *      element = {
     *          name: "<element id>",
     *          lang: "<language>",
     *          timestamp: "<timestamp>",
     *          force: true|false
     *      }
     */
    TypeStorage.prototype.cleanOutbox = function(element){
        var name = element.name || '';
        var lang = element.lang;
        var timestamp = new Date(element.timestamp || 0).getTime();
        if (this.contentinfo[lang] && this.outbox[lang]) {
            var cinfo = this.contentinfo[lang][name];
            var index = this.outbox[lang].indexOf(name);
            if (index > -1 && cinfo) {
                var lasttime = new Date(cinfo.timestamp || 0).getTime();
                if (timestamp >= lasttime || element.force) {
                    this.outbox[lang].splice(index, 1);
                    cinfo.isoutbox = false;
                    this.setDirty(element);
                };
            };
        };
    };
    
    /**
     * Check if the given element is in outbox
     * @param {String} name - the name of the element
     * @param {String} lang - the language
     * @returns {Boolean} dirty status of named element
     */
    TypeStorage.prototype.isOutbox = function(name, lang){
        var result = false;
        var oblist = [];
        if (lang && this.outbox[lang]) {
            // If language is given, use that language.
            oblist = this.outbox[lang];
        } else {
            // Else, use all languages.
            for (var oblang in this.outbox) {
                oblist = oblist.concat(this.outbox[oblang]);
            };
        };
        if (name) {
            // If name is given, search for that name in oblist.
            result = (oblist.indexOf(name) > -1);
        } else {
            // Else, check if there are any outboxed elements.
            result = oblist.length > 0;
        };
        return result;
    };
    
    /**
     * Get an object of names of all elements in outbox ordered by language
     * @param {String} lang - the language (optional)
     * @returns {Object} arrays of element names in outbox ordered by languages.
     */
    TypeStorage.prototype.getOutbox = function(lang){
        var result;
        if (lang && this.outbox[lang]) {
            result = {};
            result[lang] = this.outbox[lang].slice();
        } else {
            result = JSON.parse(JSON.stringify(this.outbox));
        }
        return result;
    };
    
    /**
     * Get list of all elements in outbox
     * @param {String} lang - the language (optional)
     * @returns {Array} an array of element in outbox.
     */
    TypeStorage.prototype.getOutboxContent = function(lang){
        var result = [];
        if (lang && this.outbox[lang]) {
            var list = this.outbox[lang];
            for (var i = 0, len = list.length; i < len; i++) {
                result.push(this.getData(list[i], lang));
            };
        } else if (!lang) {
            var list;
            for (lang in this.outbox){
                list = this.outbox[lang];
                for (var i = 0, len = list.length; i < len; i++) {
                    result.push(this.getData(list[i], lang));
                };
            };
        };
        return result;
    };
    
    /**
     * get all reference data
     * @returns {Object} enough data to re-create this TypeStorage
     */
    TypeStorage.prototype.getAllData = function(){
        var refdata = {
            refs: $.extend(true, {}, this.refs),
            contentdata: $.extend(true, {}, this.contentdata),
            contentinfo: $.extend(true, {}, this.contentinfo)
        };
        return refdata;
    };
    
    /**
     * Default data for TypeStorage
     */
    TypeStorage.defaults = {
        "anchors": [],
        "contentdata": {},
        "contentinfo": {}
    };
    
    /**
     * Default data for TypeStorage's refinfo
     */
    TypeStorage.defaultInfo = {
        "sendto": "",
        "sendfrom": "",
        "read": false,
        "recipientopened": false,
        "isoutbox": false
    };
    
    
    
    /**
     * ContentStorage - A class for containing different types of refdata
     * @constructor
     * @name ContentStorage
     * @param {Object} data - Object with contentTypes as keys and refdata as value
     *      data = {
     *          "defaultlang": "<defaultlanguage>",
     *          "storages": {
     *              "<contenttype>": {
     *                  "common": {
     *                      "<elementid>": {name: "<elementid>", anchors: [...], contentdata: {...}, contentinfo: {...}},
     *                      "<elementid>": {name: "<elementid>", anchors: [...], contentdata: {...}, contentinfo: {...}},
     *                      ...
     *                  },
     *                  "en": {....},
     *                  ....
     *              "<contenttype>": {...},
     *              ...
     *          }
     *      }
     * @param {Boolean} singlelangmode   If true, use 'common' as lang for all data.
     */
    var ContentStorage = function(data, singlelangmode){
        data = $.extend({defaultlang: 'common', storages: {}}, data);
        this.singlelangmode = !!singlelangmode;
        this.defaultlang = (this.singlelangmode ? 'common' : data.defaultlang);
        this.storages = {};
        for (var contentType in data.storages){
            this.storages[contentType] = new TypeStorage(contentType, data.storages[contentType], this.defaultlang, this.singlelangmode);
        };
    };
    
    window.Elpp.ContentStorage = ContentStorage;
    
    /**
     * Set content data into the ContentStorage
     * @param {Object} data - reference data
     *     data = {
     *         name: "<elementid>",
     *         contentType: "<type of content>",
     *         lang: "<element language (optional)>",
     *         anchors: ["<anchorid>",...],
     *         contentdata: {<content of element>},
     *         contentinfo: {<contentinfo of element>}
     *     }
     * @param {Boolean} isdirty - indicates that data should be marked dirty
     */
    ContentStorage.prototype.setData = function(data, isdirty){
        var contentType = data.contentType;
        if (typeof(data.lang) === 'undefined') {
            data.lang = this.defaultlang;
        };
        if (typeof(this.storages[contentType]) === 'undefined') {
            // If new contentType, create storage for it
            this.storages[contentType] = new TypeStorage(contentType, {}, this.defaultlang, this.singlelangmode);
        };
        this.storages[contentType].setData(data, isdirty);
    };
    
    /**
     * Get data of an element (name, contentType, lang, anchors, contentdata, contentinfo)
     * @param {Object} query - the data to search for
     *    query = {
     *        contentType: '<type of content>',
     *        name: '<elementid>',
     *        lang: '<language>'
     *    }
     * @returns {Object} object with all data of element in same format as given to .setData(data).
     */
    ContentStorage.prototype.getData = function(query){
        var result = {};
        if (query.contentType && query.name && query.lang) {
            result = this.storages[query.contentType] && this.storages[query.contentType].getData(query.name, query.lang || this.defaultlang) || result;
        };
        return result;
    };
    
    /**
     * Get content data associated to anchors with given contentTypes from the ContentStorage
     * @param {String} contentType - An array of types of content data
     * @param {Array|String} anchors - An array of elementnames referenced to or just one anchor as a string.
     * @param {String} lang - the language the content is asked in. (optional)
     *  or
     * @param {Object} contentType - with all parameters above as attributes. {contentType: '...', anchors: ..., lang: '...'}
     * @returns {Object} with content data ordered by contentTypes
     *    result = {
     *        contentType: "<contentType>",
     *        lang: "<content language>",
     *        refs: {
     *            "<anchorid>": ["<elementid>", ...],
     *            ...
     *        },
     *        contentdata: {
     *            "<elementid>": {<element data>},
     *            ...
     *        },
     *        contentinfo: {
     *            "<elementid>": {<element's contentinfo>},
     *            ...
     *        }
     *    }
     */
    ContentStorage.prototype.getContentByAnchor = function(contentType, anchors, lang){
        if (typeof(contentType) === 'object' && !anchors && !lang) {
            lang = contentType.lang;
            anchors = contentType.anchors;
            contentType = contentType.contentType;
        }
        var result = {contentType: contentType, refs: [], contentdata: {}, contentinfo: {}, lang: lang};
        if (typeof(anchors) === 'string') {
            anchors = [anchors];
        };
        result = this.storages[contentType] && this.storages[contentType].getContentByAnchor(anchors, lang) || result;
        return result;
    };
    
    /**
     * Get content info associated to anchors with given contentTypes from the ContentStorage
     * Same as getContentByAnchor(), but without contentdata.
     * @param {String} contentType - An array of types of content data
     * @param {Array|String} anchors - An array of elementnames referenced to or just one anchor as a string.
     * @param {String} lang - the language the content is asked in. (optional)
     *  or
     * @param {Object} contentType - with all parameters above as attributes. {contentType: '...', anchors: ..., lang: '...'}
     * @returns {Object} with content data ordered by contentTypes
     *    result = {
     *        contentType: "<contentType>",
     *        lang: "<content language>",
     *        refs: {
     *            "<anchorid>": ["<elementid>", ...],
     *            ...
     *        },
     *        contentinfo: {
     *            "<elementid>": {<element's contentinfo>},
     *            ...
     *        }
     *    }
     */
    ContentStorage.prototype.getContentInfoByAnchor = function(contentType, anchors, lang){
        if (typeof(contentType) === 'object' && !anchors && !lang) {
            lang = contentType.lang;
            anchors = contentType.anchors;
            contentType = contentType.contentType;
        }
        var result = {contentType: contentType, refs: [], contentinfo: {}, lang: lang};
        if (typeof(anchors) === 'string') {
            anchors = [anchors];
        };
        result = this.storages[contentType] && this.storages[contentType].getContentInfoByAnchor(anchors, lang) || result;
        return result;
    };
    
    /**
     * Get content data by name of the element, the type of the content and the language
     * @param {String} contentType - the type of the content
     * @param {Array|String} names - the name/id of the content element as an array or just one string
     * @param {String} lang - the language the content is asked in. (optional)
     * @returns {Object} an object with elementid's as keys and element data as value.
     */
    ContentStorage.prototype.getContentByName = function(contentType, names, lang){
        if (typeof(contentType) === 'object' && !names && !lang) {
            lang = contentType.lang;
            names = contentType.name;
            contentType = contentType.contentType;
        };
        if (typeof names === 'string') {
            names = [names];
        };
        var result = {};
        for (var i = 0, len = names.length; i < len; i++) {
            result[names[i]] =  this.storages[contentType] && this.storages[contentType].getContentByName(names[i], lang) || {};
        };
        return result;
    };
    
    /**
     * Remove content by element's type, name and language. And all references of it.
     * @param {String} contentType - the type of the content
     * @param {Array|String} names - the name/id of the content element as an array or just one string
     * @param {String} lang - the language the content is asked in.
     */
    ContentStorage.prototype.removeContentByName = function(contentType, names, lang) {
        if (typeof(contentType) === 'object' && !names && !lang) {
            lang = contentType.lang;
            names = contentType.name;
            contentType = contentType.contentType;
        };
        if (typeof(names) === 'string') {
            names = [names];
        };
        for (var i = 0, len = names.length; i < len; i++) {
            this.storages[contentType] && this.storages[contentType].removeData(names[i], lang);
        };
    };
    
    /**
     * Get all data from ContentStorage
     * @param {Array} exclude - a list of types to exclude
     * @returns {Object} object that can be given to ContentStorage()-constructor to recreate the storage.
     */
    ContentStorage.prototype.getAllData = function(exclude) {
        exclude = exclude || [];
        var result = {
            defaultlang: this.defaultlang,
            storages: {}
        };
        for (var contentType in this.storages) {
            if (exclude.indexOf(contentType) === -1) {
                // If the contentType is not excluded
                result.storages[contentType] = this.storages[contentType].getAllData();
            };
        };
        return result;
    }
    
    /**
     * Get list of all contentTypes
     * @returns {Array} all content types
     */
    ContentStorage.prototype.getContentTypes = function() {
        var result = [];
        for (var ctype in this.storages) {
            result.push(ctype);
        };
        return result;
    };
    
    /**
     * Update Contentinfo of given element with new data
     * @param {Object} contentinfo - new contentinfo (includes name-, contentType- and lang-attributes)
     */
    ContentStorage.prototype.updateContentInfo = function(contentinfo, isdirty){
        var lang = contentinfo.lang;
        var contentType = contentinfo.contentType;
        if (this.storages[contentType]) {
            this.storages[contentType].updateContentInfo(contentinfo, lang, isdirty);
        };
    };
    
    /**
     * Check, if storage/element is dirty
     * @param {String|Object} contentType     The type of the content or an object with
     *                        contentType, name and lang.
     * @param {String} [lang]    The language
     * @param {String} [name]    The elementid
     * @returns {Boolean} true, if there are any dirty elements matching conditions.
     */
    ContentStorage.prototype.isDirty = function(contentType, lang, name) {
        if (typeof(contentType) === 'object' && !lang && !name) {
            name = contentType.name;
            lang = contentType.lang;
            contentType = contentType.contentType;
        };
        var result = false;
        if (typeof(contentType) === 'undefined') {
            for (var ctype in this.storages) {
                result = result || this.storages[ctype].isDirty(name, lang);
            };
        } else {
            result = this.storages[contentType].isDirty(name, lang);
        };
        return result;
    };
    
    /**
     * Get all dirty elements names
     * @param {String|Object} contentType   The type of content or an object with
     *                        contentType and lang.
     * @param {String} [lang]     The language
     * @returns {Object} names of all dirty elements ordered by contentType and language
     */
    ContentStorage.prototype.getDirty = function(contentType, lang) {
        if (typeof(contentType) === 'object' && !lang) {
            lang = contentType.lang;
            contentType = contentType.contentType;
        };
        var result = {};
        if (typeof(contentType) === 'undefined') {
            for (var ctype in this.storages) {
                result[ctype] = this.storages[ctype].getDirty(lang);
            };
        } else {
            var dempty = {};
            dempty[lang] = [];
            result[contentType] = this.storages[contentType] && this.storages[contentType].getDirty(lang) || dempty;
        };
        return result;
    };
    
    /**
     * Get all dirty elements as an array from dirty
     * @param {Object} query - query parameters as attributes
     *     query = {
     *         contentType: "<contentType>",
     *         lang: "<language>"
     *     }
     * @returns {Array} an array of all element's data
     */
    ContentStorage.prototype.getDirtyContent = function(query) {
        query = $.extend({}, query);
        var contentType = query.contentType;
        var lang = query.lang;
        var result = [];
        if (contentType && this.storages[contentType]) {
            result = this.storages[contentType].getDirtyContent(lang) || [];
        } else if (!contentType) {
            for (var ctype in this.storages) {
                result = result.concat(this.storages[ctype].getDirtyContent(lang) || []);
            };
        };
        return result;
    };
    
    /**
     * Clean an element from the dirty list (after it has been successfully saved locally).
     * @param {Array|Object} data - an object or array of objects with data what to clean
     *      data = [{
     *          name: "<elementid>",
     *          contentType: "<contentType>",
     *          timestamp: "<timestamp>",
     *          lang: "<language>"
     *      },...]
     */
    ContentStorage.prototype.cleanDirty = function(data){
        if (typeof(data.length) === 'undefined') {
            data = [data];
        };
        var cleandata;
        for (var i = 0, len = data.length; i < len; i++) {
            cleandata = data[i];
            if (this.storages[cleandata.contentType]) {
                this.storages[cleandata.contentType].cleanDirty(cleandata);
            };
        };
    };

    /**
     * Get all unsent elements names from outbox
     * @param {String} contentType - type of content (optional)
     * @param {String} lang - the language (optional)
     *   or
     * @param {Object} - query as an object with all parameters as attributes.
     * @returns {Object} names of all dirty elements ordered by contentType and language
     */
    ContentStorage.prototype.getOutbox = function(contentType, lang) {
        if (typeof(contentType) === 'object' && !lang) {
            lang = contentType.lang;
            contentType = contentType.contentType;
        };
        var result = {};
        if (typeof(contentType) === 'undefined') {
            for (var ctype in this.storages) {
                result[ctype] = this.storages[ctype].getOutbox(lang);
            };
        } else {
            var obempty = {};
            obempty[lang] = [];
            result[contentType] = this.storages[contentType] && this.storages[contentType].getOutbox(lang) || obempty;
        };
        return result;
    };
    
    /**
     * Get all unset elements as an array from outbox
     * @param {Object} query - query parameters as attributes
     *     query = {
     *         contentType: "<contentType>",
     *         lang: "<language>"
     *     }
     * @returns {Array} an array of all element's data
     */
    ContentStorage.prototype.getOutboxContent = function(query) {
        query = $.extend({}, query);
        var contentType = query.contentType;
        var lang = query.lang;
        var result = [];
        if (contentType && this.storages[contentType]) {
            result = this.storages[contentType].getOutboxContent(lang) || [];
        } else if (!contentType) {
            for (var ctype in this.storages) {
                result = result.concat(this.storages[ctype].getOutboxContent(lang) || []);
            };
        };
        return result;
    }
    
    /**
     * Send to outbox, i.e., mark refelement to be waiting for sending
     * @param {Object} elementinfo - info about element to send
     *     {
     *         name: "<elementid>",
     *         timestamp: "<timestamp>",
     *         contentType: "<type of content>",
     *         lang: "<language>"
     *     }
     */
    ContentStorage.prototype.sendOutbox = function(elementinfo){
        var contentType = elementinfo.contentType;
        if (this.storages[contentType]) {
            this.storages[contentType].sendOutbox(elementinfo);
        };
    };
    
    /**
     * Clean an element from the outbox (after it has been successfully sent to the server).
     * @param {Array|Object} data - an object or array of objects with data what to clean
     *      data = [{
     *          name: "<elementid>",
     *          contentType: "<contentType>",
     *          timestamp: "<timestamp>",
     *          lang: "<language>"
     *      }, ...]
     */
    ContentStorage.prototype.cleanOutbox = function(data){
        if (typeof(data.length) === 'undefined') {
            data = [data];
        };
        var cleandata;
        for (var i = 0, len = data.length; i < len; i++) {
            cleandata = data[i];
            if (this.storages[cleandata.contentType]) {
                this.storages[cleandata.contentType].cleanOutbox(cleandata);
            };
        };
    };
    
    
    
    
    
    
    
    
    
    
    
    
    /**
     * Userinfo - a class for holding information about a user (name etc.)
     * @constructor
     * @name Userinfo
     * @param {Object} userdata - data about the user
     * @param {Array} groups - an optinal list of groups the user is member of
     */
    var Userinfo = function(userdata, groups){
        userdata = $.extend(true, {}, Userinfo.defaults, userdata);
        if (!(typeof(groups) === 'object' && typeof(groups.length) === 'number')) {
            groups = [];
        };
        this.data = userdata;
        this.usertype = '';
        this.groups = groups.slice();
    }
    
    // Copy to globally available object
    window.Elpp.Userinfo = Userinfo;
    
    /**
     * Get the username
     * @returns {String} the username
     */
    Userinfo.prototype.getUsername = function(){
        return this.data.username.toLowerCase();
    };
    
    /**
     * Get the realname of the user in firstname first order ("Peter von Bagh")
     * @returns {String} the realname
     */
    Userinfo.prototype.getRealname = function(){
        var name = [];
        var nameparts = ['firstname', 'middlename', 'lastname'];
        for (var i = 0, len = nameparts.length; i < len; i++) {
            if (this.data.realname[nameparts[i]]) {
                name.push(this.data.realname[nameparts[i]]);
            };
        };
        return sanitize(name.join(' '));
    };
    
    /**
     * Get the realname of the user in lastname first order ("Bagh, Peter von")
     * @returns {String} the realname
     */
    Userinfo.prototype.getRealnameLnameFirst = function(){
        var name = [this.data.realname.lastname + ','];
        var nameparts = ['firstname', 'middlename'];
        for (var i = 0, len = nameparts.length; i < len; i++) {
            if (this.data.realname[nameparts[i]]) {
                name.push(this.data.realname[nameparts[i]]);
            };
        };
        return sanitize(name.join(' '));
    };
    
    /**
     * Get the email address of the user.
     * @returns {String}  the email address
     */
    Userinfo.prototype.getEmail = function() {
        return this.data.email || '';
    };
    
    /**
     * Get the list of schools (if there is one).
     */
    Userinfo.prototype.getSchools = function() {
        return (this.data.schools || []).slice();
    };
    
    /**
     * Get the list of groups the user is member of
     * @returns {Array} list of groups
     */
    Userinfo.prototype.getGroups = function() {
        return this.groups.slice();
    };
    
    /**
     * Add a group the user is member of
     * @param {String} gid     The group id
     */
    Userinfo.prototype.addGroup = function(gid) {
        if (this.groups.indexOf(gid) === -1) {
            this.groups.push(gid);
        };
    };
    
    /**
     * Remove the user from a group
     * @param {String} gid     The groupid
     */
    Userinfo.prototype.removeGroup = function(gid) {
        var index = this.groups.indexOf(gid);
        if (index !== -1) {
            this.groups.splice(index, 1);
        };
    };
    
    /**
     * Get the type of the user
     * @returns {String} usertype
     */
    Userinfo.prototype.getUsertype = function(){
        return this.usertype;
    };
    
    /**
     * Default settings
     */
    Userinfo.defaults = {
        username: 'Anonymous',
        realname: {
            firstname: '',
            middlename: '',
            lastname: ''
        }
    };
    
    
    /**
     * Groupinfo - a class for holding information about groups
     * @constructor
     * @name Groupinfo
     * @param {Object} groupdata      The data about group
     */
    var Groupinfo = function(groupdata) {
        groupdata = $.extend(true, {}, Groupinfo.defaults, groupdata);
        this.data = groupdata;
    };
    
    // Copy to globally available object
    window.Elpp.Groupinfo = Groupinfo;

    /**
     * Get groupid
     * @returns {String} groupid
     */
    Groupinfo.prototype.getGid = function() {
        return this.data.gid;
    };

    /**
     * Get group's description
     * @returns {String} the description of this group
     */
    Groupinfo.prototype.getDescription = function() {
        return sanitize(this.data.description);
    };
    
    /**
     * Set group's description
     * @param {String} description     The description of this group
     */
    Groupinfo.prototype.setDescription = function(description) {
        this.data.description = sanitize(description || '');
    };
    
    /**
     * Get a list of members (usernames) in this group
     * @return {Array} a list of usernames in this group
     */
    Groupinfo.prototype.getMembers = function() {
        return this.data.members.slice();
    };
    
    /**
     * Add a members in this group
     * @param {Array|String} usernames     A list of members (or a single member) to add on this group
     * @returns {Array} a list of added members (not counting users that were already members)
     */
    Groupinfo.prototype.addMembers = function(usernames) {
        if (typeof(usernames) === 'string') {
            usernames = [usernames];
        };
        var user, added = [];
        for (var i = 0, len = usernames.length; i < len; i++) {
            user = usernames[i].toLowerCase();
            if (this.data.members.indexOf(user) === -1) {
                this.data.members.push(user);
                added.push(user);
            };
        };
        return added;
    };
    
    /**
     * Remove members from this group
     * @param {Array|String} usernames     A list of members (or a single member) to remove from this group.
     * @returns {Array} list of removed members (not counting the ones that were not in this group)
     */
    Groupinfo.prototype.removeMembers = function(usernames) {
        if (typeof(usernames) === 'string') {
            usernames = [usernames];
        };
        var user, index, removed = [];
        for (var i = 0, len = usernames.length; i < len; i++) {
            user = usernames[i].toLowerCase();
            index = this.data.members.indexOf(user);
            if (index !== -1) {
                removed.push(this.data.members[index]);
                this.data.members.splice(index, 1)
            };
        };
        return removed;
    };
    
    /**
     * Check, if the given user is member of this group
     * @param {String} username     The username of the user
     * @returns {Boolean} true, if the username is a member in this group
     */
    Groupinfo.prototype.isMember = function(username) {
        username = username.toLowerCase();
        return (this.data.members.indexOf(username) !== -1);
    };
    
    /**
     * Get a list of tags
     * @returns {Array} list of all tags on this group
     */
    Groupinfo.prototype.getTags = function() {
        return this.data.tags.slice();
    };
    
    /**
     * Add a tag on this group
     * @param {String} tag     A new tag
     */
    Groupinfo.prototype.addTag = function(tag) {
        if (this.data.tags.indexOf(tag) === -1) {
            this.data.tags.push(tag);
        };
    };
    
    /**
     * Remove a tag from this group
     * @param {String} tag     A tag to be removed
     * @returns {Boolean} true, if tag was found, false else.
     */
    Groupinfo.prototype.removeTag = function(tag) {
        var index = this.data.tags.indexOf(tag);
        if (index !== -1) {
            this.data.tags.splice(index, 1);
        };
        return (index !== -1);
    };
    
    /**
     * Check, if the group has given tag
     * @param {String} tag      A tag to be checked
     * @returns {Boolean} true, if this group has this tag, else false
     */
    Groupinfo.prototype.hasTag = function(tag) {
        var index = this.data.tags.indexOf(tag);
        return (index !== -1);
    };
    
    /**
     * Get all data of this group
     * @returns {Object} the data of this group
     */
    Groupinfo.prototype.getData = function() {
        return JSON.parse(JSON.stringify(this.data));
    };
    
    
    Groupinfo.defaults = {
        gid: 'Anongroup',
        description: 'A group without name',
        tags: [],
        members: []
    }
    
    /****************************************
     * Userlist - class for holding user objects
     * @constructor
     * @name Userlist
     * @param {Object} userdata      The data about all users
     ***************************************/
    var Userlist = function(userdata){
        userdata = $.extend(true, {}, Userlist.defaults, userdata);
        this.initData(userdata);
    };
    
    // Copy to globally available object
    window.Elpp.Userlist = Userlist;
    
    /**
     * Init the userlist
     * @param {Object} userdata     The data about all users
     */
    Userlist.prototype.initData = function(userdata){
        this.username = userdata.username.toLowerCase();
        this.contextid = userdata.contextid;
        this.contexttitle = userdata.contexttitle;
        this.contextshortdesc = userdata.contextshortdesc;
        this.contextdescription = userdata.contextdescription;
        this.users = {};
        this.groups = {};
        var user, username, group, gid, i, len, j, mlen, member;
        var ulist = userdata.users;
        for (i = 0, len = ulist.length; i < len; i++) {
            username = ulist[i].username.toLowerCase();
            user = new Userinfo(ulist[i]);
            this.users[username] = user;
        };
        var glist = userdata.groups;
        for (i = 0, len = glist.length; i < len; i++) {
            gid = glist[i].gid;
            group = new Groupinfo(glist[i]);
            this.groups[gid] = group;
            for (j = 0, mlen = glist[i].members.length; j < mlen; j++) {
                member = glist[i].members[j];
                if (this.users[member]) {
                    this.users[member].addGroup(gid);
                };
            };
        };
        var filter = {user: [this.username], group: this.getGroupsByUser(this.username)};
        this.rights = new UserRight(userdata.rights, filter);
    };
    
    /**
     * Add user into the userlist with given list of groups
     * @param {String} userdata     The data about the user
     * @param {Array} groups        A list of groups the user should be member of (optional)
     */
    Userlist.prototype.addUser = function(userdata, groups) {
        var username = userdata.username.toLowerCase();
        var user = new Userinfo(userdata);
        var gid;
        if (!(typeof(groups) === 'object' && typeof(groups.length) === 'number')) {
            groups = [];
        };
        for (var i = 0, len = groups.length; i < len; i++) {
            gid = groups[i];
            if (this.groups[gid]) {
                this.groups[gid].addMembers(username);
                this.users[username].addGroup(gid);
            };
        };
    };
    
    /**
     * Remove user from userlist
     * @param {String} username    The username of the user
     */
    Userlist.prototype.removeUser = function(username) {
        username = username.toLowerCase();
        if (this.users[username]) {
            var groups = this.users[username].getGroups();
            for (var i = 0, len = groups.length; i < len; i++) {
                this.groups[groups[i]].removeMembers(username);
            };
            delete this.users[username];
        };
    };
    
    /**
     * Add group
     * @param {Object} groupdata - data of group with: gid, description and members
     */
    Userlist.prototype.addGroup = function(groupdata) {
        var gid = groupdata.gid;
        if (gid && !this.groups[gid]) {
            this.groups[gid] = new Groupinfo(groupdata);
            var members = this.groups[gid].getMembers();
            this.addMembers(members, gid);
        };
    };
    
    /**
     * Remove group
     * @param {String} gid    The groupid of the group
     */
    Userlist.prototype.removeGroup = function(gid) {
        if (this.groups[gid]) {
            var members = this.groups[gid].getMembers();
            this.removeMembers(members, gid);
            delete this.groups[gid];
        };
    };
    
    /**
     * Add user(s) into given group(s)
     * @param {Array|String} usernames   A list of usernames to be added to groups
     * @param {Array|String} groups      A list of groups the users should be added to
     */
    Userlist.prototype.addMembers = function(usernames, groups) {
        if (typeof(usernames) === 'string') {
            usernames = [usernames];
        };
        if (typeof(groups) === 'string') {
            groups = [groups];
        }
        var i, j, ulen, glen, user, gid;
        for (i = 0, ulen = usernames.length; i < ulen; i++) {
            user = usernames[i].toLowerCase();
            for (j = 0, glen = groups.length; j < glen; j++) {
                gid = groups[j];
                this.groups[gid] && this.groups[gid].addMembers(user);
                this.users[user] && this.users[user].addGroup(gid);
            };
        };
    };
    
    /**
     * Remove user(s) from given group(s)
     * @param {Array|String} usernames    A list of usernames to be removed from groups
     * @param {Array|String} groups       A list of groups the users should be removed from
     */
    Userlist.prototype.removeMembers = function(usernames, groups) {
        if (typeof(usernames) === 'string') {
            usernames = [usernames];
        };
        if (typeof(groups) === 'string') {
            groups = [groups];
        }
        var i, j, ulen, glen, user, gid;
        for (i = 0, ulen = usernames.length; i < ulen; i++) {
            user = usernames[i].toLowerCase();
            for (j = 0, glen = groups.length; j < glen; j++) {
                gid = groups[j];
                this.groups[gid] && this.groups[gid].removeMembers(user);
                this.users[user] && this.users[user].removeGroup(gid);
            };
        };
    };
    
    /**
     * Get the title of the context
     * @returns {String} The title of the context
     */
    Userlist.prototype.getContextTitle = function() {
        return this.contexttitle || '';
    }
    
    /**
     * Get the short description of the context
     * @returns {String} The short description of the context
     */
    Userlist.prototype.getContextShortDescription = function() {
        return this.contextshortdesc || '';
    }
    
    /**
     * Get the description of the context
     * @returns {String} The description of the context
     */
    Userlist.prototype.getContextDescription = function() {
        return this.contextdescription || '';
    }
    
    /**
     * Get the description of the group by group id
     * @param {String} gid - The group id
     * @returns {String} The description of the group.
     */
    Userlist.prototype.getGroupDescription = function(gid) {
        return this.groups[gid] && this.groups[gid].getDescription() || '';
    };
    
    /**
     * Get the realname of asked user in firstname first order ("Peter von Bagh")
     * @param {String} username     The username of the user
     * @returns {String} the realname of username user
     */
    Userlist.prototype.getRealname = function(username){
        username = username.toLowerCase();
        var user = this.users[username];
        return user && user.getRealname() || '';
    };
    
    /**
     * Get the realname of asked user in lastname first order ("Bagh, Peter von")
     * @param {String} username - the username of the user
     * @returns {String} the realname of username user
     */
    Userlist.prototype.getRealnameLnameFirst = function(username){
        username = username.toLowerCase()
        var user = this.users[username];
        return user && user.getRealnameLnameFirst() || '';
    };
    
    /**
     * Check if the user is member in a group
     * @param {String} username    The name of the user
     * @param {String} gid    The group id
     * @returns {Boolean} true, if the username is member of gid, else false
     */
    Userlist.prototype.isMember = function(username, gid) {
        username = username.toLowerCase();
        var result = false;
        if (this.groups[gid]) {
            result = this.groups[gid].isMember(username);
        };
        return result;
    };
    
    /**
     * Get tags in groups of given user
     * @param {String} username   The username of the user
     * @returns {Array} a list of all tags in all groups of this user
     */
    Userlist.prototype.getUserTags = function(username) {
        username = username.toLowerCase();
        var groups = this.getGroupsByUser(username);
        var tags = [], gid, i, len;
        for (i = 0, len = groups.length; i < len; i++) {
            gid = groups[i];
            tags = tags.concat(this.groups[gid].getTags());
        };
        tags.sort();
        // Remove duplicates.
        var plusone = '';
        for (i = tags.length -1; i >= 0; i--) {
            // Go the ordered taglist from the end to the beginning.
            if (tags[i] === plusone) {
                // If the current tag is same as the next one, remove current.
                tags.splice(i, 1);
            } else {
                // If the current tag is different from the next one, the current is the new next one.
                plusone = tags[i];
            };
        };
        return tags;
    };
    
    /**
     * Get all groups
     * @returns {Array} a list of all groups
     */
    Userlist.prototype.getGroups = function() {
        var result = [];
        for (var gid in this.groups) {
            result.push(gid);
        };
        return result;
    };
    
    /**
     * Get all groups of given user
     * @param {String} username     Username of the user
     * @returns {Array} a list of groups (gid's) the user is member of.
     */
    Userlist.prototype.getGroupsByUser = function(username) {
        username = username.toLowerCase();
        var result = [];
        if (this.users[username]) {
            result = this.users[username].getGroups();
        };
        return result;
    };
    
    /**
     * Get a list of usernames in given groups
     * @param {Array|String} groups     A list of gid's (or a single gid)
     * @returns {Array} a list of all users in asked groups
     */
    Userlist.prototype.getUsernamesInGroups = function(groups) {
        if (typeof(groups) === 'undefined') {
            groups = [];
        }
        if (typeof(groups) === 'string') {
            groups = [groups];
        };
        var users = [], gusers, gid, i, j, len, glen, user;
        for (i = 0, len = groups.length; i < len; i++) {
            gid = groups[i];
            gusers = this.groups[gid] && this.groups[gid].getMembers() || [];
            for (j = 0, glen = gusers.length; j < glen; j++) {
                user = gusers[j].toLowerCase();
                if (users.indexOf(user) === -1) {
                    users.push(user);
                };
            };
        };
        return users;
    };
    
    /**
     * Get a list of users of given type. Sort by lastname by default, or optionally by firstname.
     * @param {Array} group    A list of groups (String)
     * @param {Boolean} sortbyfname     true: sort by firstname, false|undefined: sort by lastname
     * @returns {Array} a list of Userinfo-objects.
     */
    Userlist.prototype.getUserList = function(grouplist, sortbyfname){
        var usernames = this.getUsernamesInGroups(grouplist);
        var ulist = [];
        for (var i = 0, len = usernames.length; i < len; i++) {
            ulist.push(this.users[usernames[i]] || (new Userinfo({username: usernames[i].toLowerCase(), realname: {}}, [])));
        };
        if (sortbyfname) {
            // Sort by first name
            ulist.sort(function(a, b) {
                return (a.getRealname() < b.getRealname() ? -1 : 1);
            });
        } else {
            // Sort by last name
            ulist.sort(function(a, b) {
                return (a.getRealnameLnameFirst() < b.getRealnameLnameFirst() ? -1 : 1);
            });
        };
        return ulist;
    };
    
    /**
     * Check if the user can publish given type of content to the given user or group.
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype      - The contentType of the element
     * @param {String} query.toid      - Recipient id for the element to be published
     * @param {String} query.totype    - Type of the recipient id ('user'|'group')
     * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
     */
    Userlist.prototype.canPublish = function(query) {
        var result = false;
        if (query.totype === 'user') {
            var groups = this.getGroupsByUser(query.toid);
            for (var i = 0, len = groups.length; i < len; i++) {
                result = result || this.rights.canPublish({
                    ctype: query.ctype,
                    toid: groups[i],
                    totype: 'group',
                    tomember: true
                });
                if (result) {
                    break;
                };
            };
        } else {
            result = this.rights.canPublish(query);
        };
        return result;
    };

    /**
     * Check if given type of element can be edited in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype      - The contentType of the element
     * @param {String} query.contextid - id of the context where the editing would be done.
     * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
     */
    Userlist.prototype.canEdit = function(query) {
        return this.rights.canEdit(query);
    };
    
    /**
     * Check if given type of element can be viewed in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype      - The contentType of the element
     * @param {String} query.contextid - id of the context where the editing would be done.
     * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
     */
    Userlist.prototype.canView = function(query) {
        return this.rights.canView(query);
    };
    
    /**
     * Check if given class of elements can be used
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.eclass    - The name of the element class
     * @param {String} query.contextid - id of the context where the editing would be done.
     * @returns {Boolean}    True, if the element class can be used in given context, else false.
     */
    Userlist.prototype.canUseElementClass = function(query) {
        return this.rights.canUseElementClass(query);
    };
    
    /**
     * Get the list of all element types that can be used
     * @returns {String[]}  An array of names of element classes
     */
    Userlist.prototype.getElementClasses = function() {
        return this.rights.getElementClasses();
    }
    
    /**
     * Get the list of recipients this kind of content can be published to.
     * @param {String} ctype           - Type of the content.
     * @returns {Array}    An array of recipients
     */
    Userlist.prototype.getRecipients = function(ctype) {
        var rec = this.rights.getRecipients(ctype);
        var result = [], item, id, idtype, listitem, group;
        for (var i = 0, len = rec.length; i < len; i++) {
            item = rec[i];
            id = item.toid;
            idtype = item.totype;
            listitem = {
                id: id,
                idtype: idtype,
                description: (idtype === 'user' ? this.getRealnameLnameFirst(id) : (idtype === 'group' ? this.getGroupDescription(id) : this.getContextTitle())),
                touser: item.touser,
                togroup: item.togroup,
                send_to: {to: id, pubtype: idtype},
                members: []
            };
            if (idtype === 'group') {
                // For groups, find the list of all members.
                listitem.members = [];
                group = this.getUsernamesInGroups(id);
                for (var j = 0, jlen = group.length; j < jlen; j++) {
                    listitem.members.push({
                        id: group[j],
                        idtype: 'user',
                        description: this.getRealnameLnameFirst(group[j]),
                        send_to: {to: group[j], pubtype: 'user'}
                    });
                };
            } else if (idtype === 'context') {
                // For context, get the list of all users.
                listitem.members = this.getUsersAsRecipients();
            };
            listitem.members.sort(function(a, b) {
                // Sort the list of members alphabetically.
                return (a.description < b.description ? -1 : 1);
            });
            result.push(listitem);
        };
        return result;
    };
    
    /**
     * Get all users as a recipeint list
     * @returns {Array} an array of users as recipients
     */
    Userlist.prototype.getUsersAsRecipients = function() {
        var result = [];
        for (var user in this.users) {
            user = user.toLowerCase();
            result.push({
                id: user,
                idtype: 'user',
                description: this.getRealnameLnameFirst(user),
                send_to: {to: user, pubtype: 'user'}
            });
        };
        return result;
    };
    
    /**
     * Get a list of send_to -object for all the recipients the queried type of content can be sent.
     * @param {String} ctype    The contentType
     * @returns {Array}  An array of objects of format {to: 'uid|gid', pubtype: 'user|group|context'}
     */
    Userlist.prototype.getRecipientSendto = function(ctype) {
        var rec = this.rights.getRecipients(ctype);
        var result = [], item, id, idtype, listitem;
        for (var i = 0, len = rec.length; i < len; i++) {
            item = rec[i];
            id = item.toid;
            idtype = item.totype;
            listitem = {
                to: id,
                pubtype: idtype
            };
            if (idtype === 'user' || idtype === 'context') {
                result.push(listitem);
            };
            if (idtype === 'group') {
                if (item.togroup) {
                    // Add the group itself to the list.
                    result.push(listitem);
                };
                if (item.touser) {
                    // Add all members of the group to the list.
                    group = this.getUsernamesInGroups(id);
                    for (var j = 0, jlen = group.length; j < jlen; j++) {
                        result.push({to: group[j], pubtype: 'user'});
                    };
                };
            };
        };
        return result;
    }
    
    /**
     * Default settings
     */
    Userlist.defaults = {
        username: '',
        contexttitle: 'All',
        contextshortdesc: '',
        contextdescription: 'All',
        users: [],
        groups: [],
        rights: {}
    }


    /* --------------------------------------------------------------------------- */
    /**
     * UserRight
     * @class UserRight
     * @name UserRight
     * @constructor
     * @param {Object} options   Set of available user rights.
     * @param {Object} filter    Filter only rights concerning the user(s) identified
     *                           with this filter.
     *                           Format: {user: [<usernames>], group: [<groupids>]}
     * @description 
     */
    var UserRight = function(options, filter) {
        options = $.extend(true, {}, UserRight.defaults, options);
        this.edit = new AccessRight(options.edit, filter);
        this.publish = new AccessRight(options.publish, filter);
        this.view = new AccessRight(options.view, filter);
        this.elementclasses = new AccessRight(options.elementclasses, filter);
        //this.addEdits(options.edit);
        //this.addPublish(options.publish);
    };
    
    // Copy to globally available object
    window.Elpp.UserRight = UserRight;

    /**
     * Check if given type of element can be edited in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype      - The contentType of the element
     * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
     */
    UserRight.prototype.canEdit = function(query) {
        var result = false;
        try {
            result = this.edit.hasEditRight(query);
        } catch (err) {
            console.error(err);
        };
        return result;
    };

    /**
     * Check if given class of elements can be used
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.eclass    - The element class of the element
     * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
     */
    UserRight.prototype.canUseElementClass = function(query) {
        var result = false;
        try {
            result = this.elementclasses.hasElementclassRight(query);
        } catch (err) {
            console.error(err);
        };
        return result;
    };

    /**
     * Get the list of all element classes that can be used
     * @returns {String[]} - An array of names of element classes
     */
    UserRight.prototype.getElementClasses = function() {
        return this.elementclasses.getElementClasses();
    };
    
    /**
     * Check if given type of element can be published in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype     - The contentType of the element
     * @param {String} query.toid      - Recipient id for the element to be published
     * @param {String} query.totype    - Type of the recipient id ('user'|'group')
     * @param {Boolean} (query.tomember) - Optional parameter, for totype === 'group', if pubright is asked
     *                                   as a member of a group (not for whole group).
     * @returns {Boolean}         true: if the element of given contentType can be published
     *                                   to given recipient.
     *                                   false: if the element of given contentType can not be published
     *                                   to given recipient.
     */
    UserRight.prototype.canPublish = function(query) {
        var result = false;
        try {
            result = this.publish.hasPubRight(query);
        } catch (err) {
            console.error(err);
        };
        return result;
    };
    
    /**
     * Get the list of all recipients that given type of elements can be published to.
     * @param {String} ctype - The contentType of the element
     * @returns {Object[]} - An array of objects with format {to: "<user or group id>", ttype: "<user|group>"}
     */
    UserRight.prototype.getRecipients = function(ctype) {
        return this.publish.getRecipients(ctype);
    };
    
    /**
     * Check if given type of element can be viewed in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype     - The contentType of the element
     * @returns {Boolean}         true: if the element of given contentType can be published
     *                                   to given recipient.
     *                                   false: if the element of given contentType can not be published
     *                                   to given recipient.
     */
    UserRight.prototype.canView = function(query) {
        var result = false;
        try {
            result = this.view.hasViewRight(query);
        } catch (err) {
            console.error(err);
        };
        return result;
    };
    
    UserRight.defaults = {
        "edit": [],
        "publish": [],
        "view": [],
        "elementclasses": []
    };
    
    ///**
    // * Class for editing rights
    // * @class EditRight
    // * @name EditRight
    // * @constructor
    // * @param {Object[]} rights         An array of editright objects
    // * @param {String} rights[].ctype   Content type
    // * @param {Object} filter           Filtering by username or groups {user: [...], group: [...]}
    // */
    //var EditRight = function(rights, filter) {
    //    this.types = {};
    //    this.rights = {};
    //    if (typeof(rights) === 'object') {
    //        if (typeof(rights.length) === 'number') {
    //            for (var i = 0, len = rights.length; i < len; i++) {
    //                this.addRight(rights[i], filter);
    //            };
    //        } else {
    //            for (var ctype in rights) {
    //                this.addTypeRight(ctype, rights[ctype], filter);
    //            };
    //        };
    //    };
    //};
    //
    //EditRight.prototype.addRight = function(right, filter) {
    //    var ctype = right.ctype;
    //    var id = right.id;
    //    if (!this.types[ctype]) {
    //        this.types[ctype] = {};
    //    };
    //    if (!this.rights[ctype]) {
    //        this.rights[ctype] = {};
    //    };
    //    if (!this.types[ctype][id]) {
    //        this.types[ctype][id] = {};
    //    };
    //    var who;
    //    for (var i = 0, len = right.who.length; i < len; i++) {
    //        who = right.who[i];
    //        if (!filter || (filter[who.idtype] || []).indexOf(who.id) !== -1){
    //            // If there is no filter or if who.id is listed in filter in who.idtype list (user or group).
    //            this.types[ctype][id][who.id] = who.idtype;
    //            this.rights[ctype][id] = true;
    //        };
    //    };
    //};
    //
    //EditRight.prototype.addTypeRight = function(ctype, tright, filter) {
    //    if (!this.types[ctype]) {
    //        this.types[ctype] = {};
    //    };
    //    if (!this.rights[ctype]) {
    //        this.rights[ctype] = {};
    //    }
    //    var rights, who;
    //    for (var id in tright) {
    //        if (!this.types[ctype][id]) {
    //            this.types[ctype][id] = {};
    //        };
    //        rights = tright[id];
    //        for (var i = 0, len = rights.length; i < len; i++) {
    //            who = rights[i];
    //            if (!filter || (filter[who.idtype] || []).indexOf(who.id) !== -1){
    //                // If there is no filter or if who.id is listed in filter in who.idtype list (user or group).
    //                this.types[ctype][id][who.id] = who.idtype;
    //                this.rights[ctype][id] = true;
    //            };
    //        };
    //    };
    //};
    //
    ///**
    // * Check if given type of element can be edited in given type of context
    // * @param {Object} query           - The description of element to be edited
    // * @param {String} query.type      - The contentType of the element
    // * @param {String} query.contextid - id of the context where the editing would be done.
    // * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
    // */
    //EditRight.prototype.hasRight = function(query) {
    //    return !!(this.rights[query.type] && this.rights[query.type][query.contextid]);
    //};
    //
    //
    //
    //
    ///**
    // * Class for publishing rights
    // * @class PublishRight
    // * @name PublishRight
    // * @constructor
    // * @param {Object[]} rights         An array of publishright objects
    // * @param {String} rights[].ctype   Content type 
    // * @param {Object} filter           Filtering by username or groups {user: [...], group: [...]}
    // */
    //PublishRight = function(rights, filter){
    //    this.types = {};
    //    this.rights = {};
    //    if (typeof(rights) === 'object') {
    //        if (typeof(rights.length) === 'number') {
    //            for (var i = 0, len = rights.length; i < len; i++) {
    //                this.addRight(right[i], filter);
    //            };
    //        } else {
    //            for (var ctype in rights) {
    //                this.addTypeRight(ctype, rights[ctype], filter);
    //            };
    //        };
    //    };
    //};
    //
    //PublishRight.prototype.addRight = function(right, filter) {
    //    var ctype = right.ctype;
    //    var from = right.from;
    //    var ftype = right.ftype;
    //    var whoto = right.to;
    //    var ttype = right.ttype;
    //    if (!this.types[ctype]) {
    //        this.types[ctype] = {};
    //    };
    //    if (!this.rights[ctype]) {
    //        this.rights[ctype] = {};
    //    };
    //    if (!filter || (filter[ftype] || []).indexOf(from) !== -1) {
    //        // If there is no filter or if the from user/group is found in the filter.
    //        if (!this.types[ctype][from]) {
    //            this.types[ctypes][from] = {};
    //        };
    //        this.types[ctype][from][whoto] = ttype;
    //        this.rights[ctype][whoto] = rights[i].totype;
    //    };
    //};
    //
    //PublishRight.prototype.addTypeRight = function(ctype, tright, filter) {
    //    if (!this.types[ctype]) {
    //        this.types[ctype] = {};
    //    };
    //    if (!this.rights[ctype]) {
    //        this.rights[ctype] = {};
    //    };
    //    var rights, idtype;
    //    for (var id in tright) {
    //        idtype = tright[id].idtype;
    //        rights = tright[id].rights;
    //        if (!filter || (filter[idtype] || []).indexOf(id) !== -1) {
    //            // If there is no filter of the id is found in the filter's list of usernames or groups.
    //            if (!this.types[ctype][id]) {
    //                this.types[ctype][id] = {};
    //            };
    //            for (var i = 0, len = rights.length; i < len; i++) {
    //                this.types[ctype][id][rights[i].to] = rights[i].totype;
    //                this.rights[ctype][rights[i].to] = rights[i].totype;
    //            };
    //        };
    //    };
    //};
    //
    ///**
    // * Check if given type of element can be published in given type of context
    // * @param {Object} query           - The description of element to be edited
    // * @param {String} query.type      - The contentType of the element
    // * @param {Object} query.to        - Recipient of the element to be published
    // * @param {String} query.to.id     - Recipient id for the element to be published
    // * @param {String} query.to.ttype  - Type of the recipient id ('user'|'group')
    // * @returns {Boolean}    True, if the element of given contentType can be edited in given context, else false.
    // */
    //PublishRight.prototype.hasRight = function(query) {
    //    return !!(this.rights[query.type] && this.rights[query.type][query.to.id] === query.to.ttype);
    //};
    //
    ///**
    // * Get the list of all recipients that given type of elements can be published to.
    // * @param {String} ctype - The contentType of the element
    // * @returns {Object[]} - An array of objects with format {to: "<user or group id>", pubtype: "<user|group>"}
    // */
    //PublishRight.prototype.getRecipients = function(ctype) {
    //    var rights = this.rights[ctype];
    //    var result = [];
    //    if (rights) {
    //        for (var whoto in rights) {
    //            result.push({
    //                to: whoto,
    //                pubtype: rights[whoto]
    //            });
    //        };
    //    };
    //    return result;
    //}
    
    
    /**
     * Class for access rights
     * @class AccessRight
     * @name AccessRight
     * @constructor
     * @param {Object[]} rights            An array of editright objects
     * @param {String} rights[].ctype      Content type
     * @param {String} rights[].whoid      Id of the right owner
     * @param {String} rights[].whotype    Type of the right owner (user|group|context)
     * @param {String} rights[].toid       Id of the "right target"
     * @param {String} rights[].totype     Type of the "right target"
     * @param {Object} filter           Filtering by username or groups {user: [...], group: [...]}
     */
    var AccessRight = function(rights, filter) {
        this.filter = filter;
        this.rights = {};
        var ctype;
        if (typeof(rights) === 'object') {
            if (typeof(rights.length) === 'number') {
                // If the rights are a list of rights
                for (var i = 0, len = rights.length; i < len; i++) {
                    ctype = rights[i].ctype;
                    this.addRight(ctype, rights[i]);
                };
            } else {
                // If the rights are grouped by the content type
                for (var ctype in rights) {
                    for (var i = 0, len = rights[ctype].length; i < len; i++) {
                        this.addRight(ctype, rights[ctype][i]);
                    }
                };
            };
        };
    };

    /**
     * Add new right to this object
     * @param {String} ctype     The type of content ("message", "notebookcontent", "solution", "review",..)
     * @param {Object} right     The object with one right
     * @param {String} right.whoid       Id of the user|group|context tho whome the right is given
     * @param {String} right.whotype     The type of whoid (user|group|context)
     * @param {String} right.toid        Id of the recipient (if publish right)
     * @param {String} right.totype      The type of toid (user|group|context)
     * @param {Boolean} right.tomembers   True: can publish to the members of group|context,
     *                                   False: can publish to the group|context (not single members)
     */
    AccessRight.prototype.addRight = function(ctype, right) {
        if (right.whotype === 'context' || !this.filter || (this.filter[right.whotype] || []).indexOf(right.whoid) !== -1){
            // If the right is for all in this context, there is no filter
            // or if right.whoid is listed in filter in right.whotype list (user or group).

            // Add the right, if it does not exist yet.
            this.rights[ctype] = this.rights[ctype] || {};
            if (right.toid) {
                // Add the right to publish to somebody
                this.rights[ctype][right.toid] = this.rights[ctype][right.toid] || {totype: right.totype};
                if (right.tomembers) {
                    this.rights[ctype][right.toid].touser = true;
                } else {
                    this.rights[ctype][right.toid].togroup = true;
                };
            };
        };
    };

    /**
     * Check if given type of element can be edited
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype     - The contentType of the element
     * @returns {Boolean}            True, if the element of given contentType can be edited,
     *                               else false.
     */
    AccessRight.prototype.hasEditRight = function(query) {
        return !!this.rights[query.ctype];
    };
    
    /**
     * Check if given type of element can be viewed. (Similar to the edit one.)
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype     - The contentType of the element
     * @returns {Boolean}            True, if the element of given contentType can be edited,
     *                               else false.
     */
    AccessRight.prototype.hasViewRight = AccessRight.prototype.hasEditRight;

    /**
     * Check if given class of elements can be used
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.eclass    - The element class of the element
     * @returns {Boolean}            True, if the element of given contentType can be edited,
     *                               else false.
     */
    AccessRight.prototype.hasElementclassRight = function(query) {
        return !!this.rights[query.eclass];
    };

    /**
     * Get the list of all element classes that can be used
     * @returns {String[]}    - An array of names of element classes
     */
    AccessRight.prototype.getElementClasses = function() {
        var result = [];
        for (var eclass in this.rights) {
            result.push(eclass);
        };
        return result;
    }

    /**
     * Check if given type of element can be publish in given type of context
     * @param {Object} query           - The description of element to be edited
     * @param {String} query.ctype     - The contentType of the element
     * @param {String} query.toid      - Recipient id for the element to be published
     * @param {String} query.totype    - Type of the recipient id ('user'|'group'|'context')
     * @param {Boolean} (query.tomember)  Optional parameter, for totype === 'group', if pubright is asked
     *                                   as a member of a group (not for whole group).
     * @returns {Boolean}                true: if the element of given contentType can be published
     *                                   to given recipient.
     *                                   false: if the element of given contentType can not be published
     *                                   to given recipient.
     */
    AccessRight.prototype.hasPubRight = function(query) {
        var rightsobj = this.rights[query.ctype] && this.rights[query.ctype][query.toid];
        var result = !!rightsobj;
        if (rightsobj) {
            switch (query.totype) {
                case 'group':
                    var membership = (query.tomember ? rightsobj.touser : rightsobj.togroup);
                    result = (result && rightsobj.totype === 'group' && membership);
                    break;
                case 'user':
                    result = result && rightsobj.totype === 'user';
                    break;
                default:
                    break;
            };
        };
        return result;
    };
    
    /**
     * Get the list of all recipients that given type of elements can be published to.
     * @param {String} ctype - The contentType of the element
     * @returns {Object[]}   - An array of objects with format
     *                        {to: "<user or group id>", pubtype: "<user|group|context>"}
     */
    AccessRight.prototype.getRecipients = function(ctype) {
        var rights = this.rights[ctype];
        var result = [];
        if (rights) {
            for (var whoto in rights) {
                result.push({
                    toid: whoto,
                    totype: rights[whoto].totype,
                    touser: !!rights[whoto].touser,
                    togroup: !!rights[whoto].togroup
                });
            };
        };
        return result;
    }
    
    
    
    /**
     * Class for school subjects
     * @class SubjectSet
     * @name SubjectSet
     * @constructor
     * @param {Array} subjectlist  optional list of subjectobjects
     */
    var SubjectSet = function(subjectlist) {
        if (!subjectlist || !subjectlist.length) {
            subjectlist = [];
        };
        this.subjects = {};
        this.subjectlist = []
        var sid;
        for (var i = 0, len = subjectlist.length; i < len; i++) {
            sid = subjectlist[i].id;
            this.subjects[sid] = subjectlist[i];
            this.subjectlist.push(subjectlist[i]);
        };
    };
    
    /**
     * Get list of subject id's in alphabetical weighted order (lights first, same weight in alphabetical order)
     * @param {String} lang    The language of titles
     */
    SubjectSet.prototype.getSortedIds = function(lang) {
        lang = lang || 'en';
        this.subjectlist.sort(function(a, b) {
            var atitle = a.title[lang] || a.title['en'];
            var btitle = b.title[lang] || b.title['en'];
            return (a.weight < b.weight || (a.weight === b.weight && atitle < btitle) ? -1 : 1);
        });
        var result = [];
        for (var i = 0, len = this.subjectlist.length; i < len; i++) {
            result.push(this.subjectlist[i].id);
        };
        return result;
    };
    
    /**
     * Get the title of the asked subject in given language
     * @param {String} id    The id of the subject
     * @param {String} lang  The language of the title
     */
    SubjectSet.prototype.getTitle = function(id, lang) {
        lang = lang || 'en';
        var subject = this.subjects[id];
        var title = subject && subject.title[lang] || '';
        return title;
    };
    
    /**
     * Get the icon of the asked subject
     * @param {String} id    The id of the subject
     */
    SubjectSet.prototype.getIcon = function(id) {
        var subject =  this.subjects[id];
        var icon = subject && subject.icon || '';
        return icon;
    };
    
    window.Elpp.SubjectSet = SubjectSet;
    
    /**
     * Class for one school subject
     * @class SchoolSubject
     * @name SchoolSubject
     * @constructor
     * @param {Object} subject   An object describing the subject.
     * @param {String} subject.id        Id of the subject
     * @param {String} subject.name      The name of the subject
     * @param {Object} subject.title     The localized subject names
     * @param {String} subject.icon      The svg-icon of the subject
     * @param {Number} subject.weight    The weight of the subject (for sorting)
     */
    var SchoolSubject = function(subject) {
        subject = $.extend({}, this.defaults, subject);
        this.data = subject;
    };
    
    SchoolSubject.prototype.getTitle = function(lang) {
        return this.data.title[lang] || '';
    };
    
    SchoolSubject.prototype.getWeight = function() {
        return this.data.weight;
    };
    
    SchoolSubject.prototype.getIcon = function() {
        return this.data.icon;
    };
    
    SchoolSubject.prototype.getId = function() {
        return this.data.id;
    };
    
    SchoolSubject.prototype.getName = function() {
        return this.data.name;
    };
    
    SchoolSubject.prototype.defaults = {
        id: 'unknown',
        name: 'unknown',
        title: {
            en: 'unknown',
            fi: 'tuntematon',
            sv: 'okänd'
        },
        icon: '',
        weight: 50
    };
    
    
    var sanitize = function(text, options) {
        options = $.extend({
            ALLOWED_TAGS: []
        }, options);
        return DOMPurify.sanitize(text, options);
    }
    
    /**
     * Tool to convert notebook's materialdata and storagedata to merged materialdata
     * @param {Object} data                All data given
     * @param {String} data.id             Id of the notebook
     * @param {Object} data.materialdata   materialdata
     * @param {Object} data.storage        storage's data
     */
    window.Elpp.getNotebookData = function(data) {
        var id = data.id || data.materialdata.id || data.materialdata.name;
        var mdata = data.materialdata[id]
        if (mdata.data && mdata.data.content && mdata.data.content.frontpage) {
            var common = mdata.data.content;
            mdata.data.content = {
                common: common
            };
        };
        var defaultdata = {id: id, materialdata: {}, storage: {}}
        defaultdata.materialdata[id] = {
            type: "book",
            metadata: {
                creator: "Anonymous",
                created: (new Date()).toString()
            },
            data: {
                bookid: id,
                title: {
                    common: "Notebook"
                },
                toc: {
                    type: "toc",
                    metadata: {},
                    data: {
                        firstpage: {}
                    }
                },
                pagetitles: {
                    common: {}
                },
                index: {},
                content: {},
                defaultlang: "fi",
                langs: [
                    "fi"
                ]
            },
            name: id
        };
        defaultdata.storage = {
            storages: {
                notebookcontent: {
                    refs: {},
                    contentdata: {},
                    contentinfo: {}
                }
            }
        }
        data = $.extend(true, {}, defaultdata, data);
        
        var result = data.materialdata[id];
        var storage = data.storage.storages.notebookcontent;
        var reflist = storage.refs[id] || [];
        var cdata, cid, content;
        for (var lang in storage.contentdata) {
            cdata = storage.contentdata[lang];
            for (var i = 0, len = reflist.length; i < len; i++) {
                cid = reflist[i];
                content = cdata[cid];
                if (content) {
                    switch (content.type) {
                        case 'toc':
                            if (lang === 'common') {
                                result.data.toc = content;
                            };
                            break;
                        case 'pagetitles':
                            if (!result.data.pagetitles[lang]) {
                                result.data.pagetitles[lang] = {};
                            };
                            result.data.pagetitles[lang] = content.data;
                            break;
                        case 'index':
                            if (!result.data.index[lang]) {
                                result.data.index[lang] = {};
                            };
                            result.data.index[lang] = content.data;
                            break;
                        default:
                            if (!result.data.content[lang]) {
                                result.data.content[lang] = {};
                            };
                            result.data.content[lang][cid] = content;
                            break;
                    };
                };
            };
        };
        return result;
    };
    
})(window, jQuery, window.ebooklocalizer);
