/***************************************************
 * BookChapter -class
 ***************************************************/
BookChapter = function(ebook, options){
    options = jQuery.extend({
        chapid: '',
        content: '',
        types: [],
        headimage: '',
        subheadimage: '',
        contentlevel: '',
        chapters: [],
        extra: false
    }, options);
    this.ebook = ebook;
    this.chapid = options.chapid;
    this.content = options.content;
    this.types = options.types;
    this.headimage = options.headimage;
    this.subheadimage = options.subheadimage;
    this.contentlevel = options.contentlevel;
    this.chapters = options.chapters;
    this.extra = options.extra;
    this.parent = false;
    this.next = false;
    this.prev = false;
}

/******
 * Initialize the chapter from toc data.
 ******/
BookChapter.prototype.initialize = function(chapdata, level, parent){
    if (typeof(parent) === 'undefined'){
        parent = false;
    }
    chapdata = jQuery.extend({
        chapid: '',
        content: '',
        types: [],
        headimage: '',
        subheadimage: '',
        contentlevel: '',
        chapters: [],
        extra: false
    }, chapdata);
    this.chapid = chapdata.chapid;
    this.number = chapdata.number;
    this.content = chapdata.content;
    this.types = chapdata.types;
    this.extra = chapdata.extra;
    this.contentlevel = chapdata.contentlevel;
    this.headimage = chapdata.headimage;
    this.subheadimage = chapdata.subheadimage;
    this.level = level;
    this.parent = parent;
    if (chapdata.frontImage) {
        this.frontImage = chapdata.frontImage;
    }
    if (chapdata.before && chapdata.before.content) {
        var newchap = new BookChapter(this.ebook);
        this.ebook.appendChapter(chapdata.before.chapid);
        newchap.initialize(chapdata.before, this.level + 1, this);
        this.ebook.addToDict(newchap);
        this.before = newchap;
    }
    for (var i = 0; i < chapdata.chapters.length; i++){
        var chapter = chapdata.chapters[i];
        var newchap = new BookChapter(this.ebook);
        if (chapdata.chapters[i].content !== ''){
            this.ebook.appendChapter(chapter.chapid);
        }
        newchap.initialize(chapter, this.level + 1, this);
        this.ebook.addToDict(newchap);
        this.chapters.push(newchap);
    }
    if (chapdata.after && chapdata.after.content) {
        var newchap = new BookChapter(this.ebook);
        this.ebook.appendChapter(chapdata.after.chapid);
        newchap.initialize(chapdata.after, this.level + 1, this);
        this.ebook.addToDict(newchap);
        this.after = newchap;
    }
}

/******
 * Get the parent chapter node.
 ******/
BookChapter.prototype.getParent = function(){
    return this.parent;
}

/******
 * Add a chapter to the end of the chapters-array.
 ******/
BookChapter.prototype.addSubchapter = function(chapter){
    this.chapters.push(chapter);
}

/******
 * Remove a chapter from the chapters-array.
 ******/
BookChapter.prototype.removeSubchapter = function(chapid){
    var i = this.indexOfSub(chapid);
    this.chapters.splice(i,1);
}

/******
 * Find the index of chapter with chapid in the chapters-array.
 ******/
BookChapter.prototype.indexOfSub = function(chapid){
    var i = 0;
    while (i < this.chapters.length && this.chapters[i].chapid !== chapid){
        i++;
    }
    if (this.chapters[i] && this.chapters[i].chapid === chapid){
        return i;
    } else {
        return -1;
    }
}

/******
 * Get headimage name
 ******/
BookChapter.prototype.getHeadimage = function(chapid){
    var headimage = {
        type: 'headimage',
        text: ''
    };
    if (this.subheadimage && (chapid === this.chapid)) {
        headimage.type = 'fullheadimage';
        headimage.text = this.subheadimage;
    } else if (this.headimage) {
        headimage.text = this.headimage;
    } else {
        headimage = (this.parent && this.parent.getHeadimage() || headimage);
    };
    return headimage;
    //return this.headimage || (this.parent && this.parent.getHeadimage() || '');
}

/**
 * Get the content level of the chapter. If the chapter does not have contentlevel
 * given, inherit from parent. If there is no root, return ''.
 * @returns {String}  contentlevel ('', 'normal', 'helpful', 'advanced', ...?)
 */
BookChapter.prototype.getContentlevel = function() {
    return this.contentlevel || (this.parent && this.parent.getContentlevel()) || '';
}

/******
 * Get the title of this chapter in given language, or in the default language,
 * if language is not available.
 ******/
BookChapter.prototype.getTitle = function(lang){
    return this.ebook.getTitle(this.chapid, lang);
}

/******
 * Get the level of chapter.
 ******/
BookChapter.prototype.getLevel = function(){
    return this.level || 0;
}

/******
 * Get the (name/id of the) content.
 ******/
BookChapter.prototype.getContent = function(){
    return this.content;
}

/******
 * Get frontImage
 ******/
BookChapter.prototype.getFrontImage = function(){
    return this.frontImage || '';
}

/******
 * Get the subtoc beginning from this chapter as a html-list.
 ******/
BookChapter.prototype.tocHtml = function(lang, emptylist){
    var title = this.getTitle(lang);
    var simpletitle = title;
    title = title.replace(/\\\((.*?)\\\)/g, '<span class="mathquill-embedded-latex">$1</span>')
        .replace(/\$(.*?)\$/g, '<span class="mathquill-embedded-latex">$1</span>');
    var chapnumber = this.getNumberLong();
    var html = '<li data-ebtocchapid="'+this.chapid+'" data-ebtoclevel="'+this.level+'" data-ebchaptypes="'+this.getTypes().join(' ')+'" data-ebisextrachap="'+this.getExtra().toString()+'" title="'+this.htmlEscape(simpletitle)+'" class="ebooktocitem">\n';
    html += '<span data-chapid="'+this.chapid+'"><span class="ebtocnumber">'+chapnumber+'</span> <span class="ebtoctitle">'+ title + '</span></span>\n';
    var endhtml = '</li>\n';
    if (this.before) {
        html += '<ul class="ebooktocbefore">\n';
        html += this.before.tocHtml(lang, emptylist);
        html += '</ul>\n';
    }
    if (this.chapters.length > 0){
        html += '<ul class="ebooktocsubchap">\n';
        for (var i=0; i < this.chapters.length; i++){
            html += this.chapters[i].tocHtml(lang, emptylist);
        }
        html += '</ul>\n';
    } else if (emptylist){
        html += '<ul class="ebooktocsubchap"></ol>\n';
    }
    if (this.after) {
        html += '<ul class="ebooktocafter">\n';
        html += this.after.tocHtml(lang, emptylist);
        html += '</ul>\n';
    }
    html += endhtml;
    return html;
}

/******
 * Get the chapter as a toc-object.
 ******/
BookChapter.prototype.getObject = function(){
    var json = {
        title: this.title,
        types: this.types,
        chapid: this.chapid,
        content: this.content,
        chapters: []
    }
    if (this.headimage){
        json.headimage = this.headimage;
    }
    if (this.subheadimage){
        json.subheadimage = this.subheadimage;
    }
    if (this.before) {
        json.before = this.before.getObject();
    }
    if (this.after) {
        json.after = this.after.getObject();
    }
    for (var i = 0; i < this.chapters.length; i++){
        json.chapters.push(this.chapters[i].getObject());
    }
    // Return a copy, not the original.
    return JSON.parse(JSON.stringify(json));
}

/******
 * Get the array of types.
 ******/
BookChapter.prototype.getTypes = function(){
    return this.types.slice();
}

/******
 * Set the array of types.
 ******/
BookChapter.prototype.setTypes = function(types){
    this.types = types.slice();
}

/******
 * Get the extra status (boolean).
 ******/
BookChapter.prototype.getExtra = function(){
    return this.extra;
}

/******
 * Set the extra status (boolean).
 ******/
BookChapter.prototype.setExtra = function(extra){
    this.extra = extra;
}

/******
 * Get list of children.
 ******/
BookChapter.prototype.getChildren = function(lang){
    var children = [];
    if (this.before) {
        children.push({title: this.before.getTitle(lang), chapid: this.before.chapid, chapnumber: this.before.getNumber()});
    }
    for (var i = 0; i < this.chapters.length; i++) {
        var chap = this.chapters[i];
        children.push({title: chap.getTitle(lang), chapid: chap.chapid, chapnumber: chap.getNumber()});
    }
    if (this.after) {
        children.push({title: this.after.getTitle(lang), chapid: this.after.chapid, chapnumber: this.after.getNumber()});
    }
    return children;
}


/******
 * Get the number of the chapter as a list of numbers in the path from root to chapter.
 ******/
BookChapter.prototype.getNumberList = function(chapid){
    if (typeof(this.numberlist) === 'undefined') {
        if (!this.parent) {
            this.numberlist = [this.number || ''];
        } else {
            this.numberlist = this.parent.getNumberList(this.chapid);
            var thisnum = this.getNumber();
            this.numberlist.push(thisnum);
        }
    }
    return this.numberlist.slice();
}

/******
 * Get the number of the chapter as a string.
 ******/
BookChapter.prototype.getNumber = function(){
    if (typeof(this.number) === 'undefined') {
        if (!this.parent) {
            this.number = '';
        } else {
            var index = this.parent.indexOfSub(this.chapid);
            this.number =  (index > -1 ? (index + 1)+'' : '');
        }
    }
    //if (typeof(this.number) === 'undefined') {
    //    this.number = this.getNumberList().join('.');
    //}
    return this.number;
}

/******
 * Get the number of the chapter as a string in the long format (chapter.section.subsection).
 ******/
BookChapter.prototype.getNumberLong = function(chapid){
    var nlist = this.getNumberList();
    while (nlist.length > 0 && nlist[0] === ''){
        nlist.shift(); // Remove empty strings
    }
    return nlist.join('.');
    //return  this.number;
}

/******
 * Get the path i.e. an array of titles from root to this chapter.
 ******/
BookChapter.prototype.getPath = function(lang, chapid){
    if (!this.parent) {
        var pathlist = [];
    } else {
        var pathlist = this.parent.getPath(lang, this.chapid);
    }
    pathlist.push([this.getTitle(lang), this.chapid]);
    return pathlist;
}

/******
 * Escape given text with html entities.
 ******/
BookChapter.prototype.htmlEscape = function(text){
    return jQuery('<div></div>').text(text).html();
}