/************************************************************************
 * Book -class
 * Class for representing the ebook.
 ************************************************************************/
function Book(options){
    //toc = jQuery.extend({}, toc);
    this.preRequisites = [];
    this.type = 'Book';
    this.version = 1.0;
    this.store = (options.getContent ?  options : new BookStorage(options));
    var selfbook = this;
    this.init(options);
    //this.initialize(toc, store);
}

/**
 * Init the book (syncronously)
 * @param {Object} options - data for initing
 */
Book.prototype.init = function(options) {
    this.toc = this.getContentDataSync('toc.json')['toc.json'].data;
    if (typeof(this.toc) === 'string') {
        try {
            this.toc = JSON.parse(this.toc);
        } catch (err) {
            this.toc = {};
        }
    }
    this.elementboxNumbers = this.getContentDataSync('elementboxNumbers')['elementboxNumbers'] || {};
    this.toc = jQuery.extend(true, {
        bookid: '',
        booktype: [],
        curriculum: 'FI',
        defaultlang: 'en',
        langs: [],
        titles: {},
        firstpage: {},
        chapters: [],
        author: [],
        secondaryauthor: [],
        vendorURL: ''
    }, this.toc);
    this.bookid = this.toc.bookid;
    this.booktype = this.toc.booktype;
    this.curriculum = this.toc.curriculum;
    this.defaultlang = this.toc.defaultlang;
    this.langs = this.toc.langs;
    this.enabledLangs = [];
    this.author = this.toc.author;
    this.secondaryauthor = this.toc.secondaryauthor;
    this.vendorURL = this.toc.vendorURL;
    this.vendorLogo = this.toc.vendorLogo;
    this.bookserielogo = this.toc.bookserielogo;
    this.titles = {};
    this.initTitlesSync();
    this.frontpage = new BookChapter(this);
    this.chapters = [this.toc.firstpage.chapid];
    this.tocdict = {};
    this.frontpage.initialize(this.toc.firstpage, 0);
    this.tocdict[this.frontpage.chapid] = this.frontpage;
    this.initNextPrev();
}

/******
 * Init the ebook from data in json or object format.
 ******/
Book.prototype.initialize = function(params){
    params = jQuery.extend({
        lang: null,
        initCallback: function(){}
    }, params);
    var bookitem = this;
    var initLang = params.lang;
    var initCallback = params.initCallback;
    this.elementboxNumbers = {};
    
    var init2 = function(){
        bookitem.frontpage = new BookChapter(bookitem);
        // Linear array of pages' ids.
        bookitem.chapters = [bookitem.toc.firstpage.chapid];
        bookitem.tocdict = {};
        bookitem.frontpage.initialize(bookitem.toc.firstpage, 0);
        bookitem.tocdict[bookitem.frontpage.chapid] = bookitem.frontpage;
        bookitem.initNextPrev();

        initCallback();
        //this.titles = toc.titles;
        //this.store = store;
    }
    
    var initTitles = function(){
        bookitem.initTitles(init2);
    }
    
    var init1 = function(){
        // TODO Typecheck toc and store via duck typing to avoid errors on this level.
        if (typeof(bookitem.toc) === 'string'){
            try {
                bookitem.toc = JSON.parse(bookitem.toc);
            } catch(err) {
                bookitem.toc = {};
            }
        }
        bookitem.toc = jQuery.extend(true, {
            bookid: '',
            booktype: [],
            curriculum: 'FI',
            defaultlang: 'en',
            langs: [],
            titles: {},
            firstpage: {},
            chapters: [],
            author: [],
            secondaryauthor: [],
            vendorURL: ''
        }, bookitem.toc);
        bookitem.bookid = bookitem.toc.bookid;
        bookitem.booktype = bookitem.toc.booktype;
        bookitem.curriculum = bookitem.toc.curriculum;
        bookitem.defaultlang = bookitem.toc.defaultlang;
        bookitem.langs = bookitem.toc.langs;
        bookitem.enabledLangs = [];
        bookitem.author = bookitem.toc.author;
        bookitem.secondaryauthor = bookitem.toc.secondaryauthor;
        bookitem.vendorURL = bookitem.toc.vendorURL;
        bookitem.vendorLogo = bookitem.toc.vendorLogo;
        bookitem.bookserielogo = bookitem.toc.bookserielogo;
        bookitem.titles = {};
        initTitles();
    }
    
    var loadNumbers = function () {
        bookitem.getContentData(['elementboxNumbers'], false, function(data){
            bookitem.elementboxNumbers = (data['elementboxNumbers'].type !== 'error') && data['elementboxNumbers'] || {};
            init1();
        });
    }
    
    var loadToc = function(){
        bookitem.getContentData(['toc.json'], false, function(data){
            bookitem.toc = data['toc.json'].data;
            loadNumbers();
        });
    }
    
    loadToc();
}

/******
 * Clear the data in ebook
 ******/
Book.prototype.clear = function(){
    this.bookid = '';
    this.titles = {};
    this.frontpage = null;
    this.chapters = [];
    this.tocdict = {};
}

/******
 * Init titles in asked language
 ******/
Book.prototype.initTitles = function(callback){
    if (typeof(callback) !== 'function') {
        callback = function(){};
    }
    var bookitem = this;
    var titles = [];
    for (var i = 0; i < this.langs.length; i++) {
        var lang = this.langs[i].id;
        if (this.isAvailableLang(lang)) {
            titles.push(lang + '/titles.json');
        }
    }
    this.getContentData(titles, false, function(data){
        for (var titlekey in data){
            var lang = titlekey.split('/')[0];
            if (bookitem.isAvailableLang(lang)) {
                bookitem.enabledLangs.push(lang);
                bookitem.titles[lang] = data[titlekey].data;
            }
        }
        callback();
    });
}

/******
 * Init titles in asked language
 ******/
Book.prototype.initTitlesSync = function(){
    var titles = [];
    for (var i = 0; i < this.langs.length; i++) {
        var lang = this.langs[i].id;
        if (this.isAvailableLang(lang)) {
            titles.push(lang + '/titles.json');
        }
    }
    var titleset = this.getContentDataSync(titles, false);
    var langname;
    for (var titlekey in titleset) {
        langname = titlekey.split('/')[0];
        if (this.isAvailableLang(langname)) {
            this.enabledLangs.push(langname);
            this.titles[langname] = titleset[titlekey].data;
        }
    }
}

/******
 * Init the linear next and previous links in chapter nodes.
 ******/
Book.prototype.initNextPrev = function(){
    for (var i = 0; i < this.chapters.length; i++){
        if (i > 0){
            this.tocdict[this.chapters[i]].prev = this.chapters[i-1];
        }
        if (i < this.chapters.length - 1){
            this.tocdict[this.chapters[i]].next = this.chapters[i+1];
        }
    }
}

/******
 * Get bookid
 ******/
Book.prototype.getBookid = function(){
    return this.bookid;
}

/******
 * Get the chapter object with given id.
 ******/
Book.prototype.getChapter = function(chapid){
    return this.tocdict[chapid];
}

/******
 * Get the id of the next page of the page / chapter with given id.
 ******/
Book.prototype.nextPage = function(currentpage){
    var newpage = this.getChapter(currentpage).next;
    if (!newpage){
        newpage = currentpage;
    }
    return newpage;
}

/******
 * Get the id of the previous page of the page / chapter with given id.
 ******/
Book.prototype.prevPage = function(currentpage){
    var newpage = this.getChapter(currentpage).prev;
    if (!newpage){
        newpage = currentpage;
    }
    return newpage;
}

/******
 * Get the id of the parent page of the page / chapter with given id.
 ******/
Book.prototype.parentPage = function(currentpage){
    var newpage = this.getChapter(currentpage).parent;
    if (typeof(newpage) === 'undefined'){
        newpage = currentpage;
    }
    return newpage.chapid;
}

/******
 * Get the title of the book in given language.
 ******/
Book.prototype.getBookTitle = function(lang){
    lang = lang || this.defaultlang;
    return (this.titles[lang] && this.titles[lang].booktitle) ||
        (this.titles[this.defaultlang] && this.titles[this.defaultlang].booktitle)
        || '';
}

/******
 * Get the title, shorttitle and shorttitlefill of the book in given language.
 ******/
Book.prototype.getTitleData = function(lang){
    result = {};
    lang = lang || this.defaultlang;
    //result.title = (this.titles[lang] && this.titles[lang].booktitle) ||
    //    (this.titles[this.defaultlang] && this.titles[this.defaultlang].booktitle)
    //    || '';
    result.title = this.getTitle(this.frontId(), lang);
    result.shorttitle = (this.titles[lang] && this.titles[lang].bookshorttitle) ||
        (this.titles[this.defaultlang] && this.titles[this.defaultlang].bookshorttitle)
        || '';
    result.shorttitlefill = (this.titles[lang] && this.titles[lang].shorttitlefill) ||
        (this.titles[this.defaultlang] && this.titles[this.defaultlang].shorttitlefill)
        || '#6c3';
    return result;
}

/******
 * Get the array of booktypes.
 ******/
Book.prototype.getBookTypes = function(){
    return this.booktype.slice();
}

/******
 * Get authordata of the book
 ******/
Book.prototype.getAuthors = function(){
    var authors = {
        author: this.author.slice(),
        secondaryauthor: $.extend(true, [], this.secondaryauthor)
    };
    return authors;
}

/******
 * Check if the book has booktype.
 ******/
Book.prototype.hasBookType = function(booktype){
    return this.booktype.indexOf(booktype) > -1;
}

/******
 * Get the title of a chapter with given language.
 ******/
Book.prototype.getTitle = function(chapid, lang){
    var title = '';
    if (this.getChapter(chapid)) {
        lang = this.titles[lang] && lang || this.defaultlang;
        title = this.titles[lang].chaptertitles[chapid];
    }
    return title;
}

/******
 * Get the title of chapter with numbering.
 ******/
Book.prototype.getNumberedTitle = function(chapid, lang){
    var number = this.getChapNumberLong(chapid).join('.');
    var title = this.getTitle(chapid, lang);
    return number + ' ' + title;
}

/******
 * Return an array of types of given chapter.
 ******/
Book.prototype.getTypes = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getTypes() || [];
}

/******
 * Get the content id of the given chapter.
 ******/


/*****
 * Get the path of given chapter.
 ******/
Book.prototype.getPath = function(chapid, lang){
    if (typeof(lang) === 'undefined') {
        lang = this.defaultlang;
    }
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getPath(lang) || '';
}

/*****
 * Get the children of given chapter.
 ******/
Book.prototype.getChildren = function(chapid, lang){
    if (typeof(lang) === 'undefined') {
        lang = this.defaultlang;
    }
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getChildren(lang) || [];
}

/******
 * Get the default language.
 ******/
Book.prototype.getDefaultLang = function(){
    return this.defaultlang;
}

/******
 * Check if given lang is available.
 ******/
Book.prototype.isAvailableLang = function(lang){
    if (!this.langs || !this.langs.length) {
        return false;
    }
    for (var i = this.langs.length -1; i > -1; i--) {
        if (this.langs[i].id === lang) {
            break;
        }
    }
    return i > -1;
}

/******
 * Get chapter number. (chapter, section or subsection)
 ******/
Book.prototype.getChapNumber = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getNumber() || '';
}

/******
 * Get chapter number in long format. (chapter.section.subsection)
 ******/
Book.prototype.getChapNumberLong = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getNumberLong() || '';
}

/******
 * Get an array with chapter number. [chapter, section, subsection]
 ******/
Book.prototype.getChapNumberList = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getNumberList() || [];
}

/******
 * Get the breadcrumb as html.
 ******/
Book.prototype.getBreadcrumb = function(chapid, lang){
    chapid = chapid || 'front';
    var chapter = this.getChapter(chapid);
    if (!chapter){
        return '';
    }
    var breadcrumb = [];
    var brnumber = this.getChapNumberList(chapid);
    var breadcrumbpath = this.getPath(chapid, lang);
    //breadcrumbpath.shift();
    var brnumberafter = (chapid === this.frontId()) ? '' : '.';
    breadcrumb.push('<span class="breadcrumbchapter" chapid="'+breadcrumbpath[0][1]+'"><span class="breadcrumbchaptertext" title="'+breadcrumbpath[0][0]+'"><svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="20" height="20" viewbox="0 0 50 50" class="ebook-icon ebook-icon-book mini-icon mini-icon-book"><path stroke="none" d="M19 10 q2 -4 6 -4 l20 0 q5 0 3 5 l-14 20 q-3.5 5 -8 5 l-21 0 a4 6 0 0 0 0 10 l21 0 q4.5 0 8 -5 l11.2 -16 q3 -1 2 3 l-11.2 16 q-3.5 5 -8 5 l-24 0 a8 10 0 0 1 0 -16z" /></svg></span></span>');
    for (var i = 1; i<breadcrumbpath.length; i++){
        breadcrumb.push('<span class="breadcrumbchapter" chapid="'+breadcrumbpath[i][1]+'">'+(typeof(brnumber[i]) !== 'undefined' && brnumber[i] !== '' ? '<span class="breadcrumbnumber">'+brnumber[i] + brnumberafter + '</span> ' : '') + '<span class="breadcrumbchaptertext">'+breadcrumbpath[i][0]+'</span></span>');
    }
    return breadcrumb.join('<span class="breadcrumbdelimiter ffwidget-bordercolored"> &gt; </span>') + '<span class="breadcrumbdelimiter ffwidget-bordercolored">' +(chapter.chapters.length > 0 ? ' &gt;  ' : '') + '</span>';
    // \ufe40 down angle
}

/******
 * Get the name of the content element of given chapter.
 ******/
Book.prototype.getContentById = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.content || '';
}

/******
 * Get the frontImage of given chapter
 ******/
Book.prototype.getFrontImage = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getFrontImage();
}

/******
 * Get vendor logo of the current book
 ******/
Book.prototype.getVendorLogo = function(){
    return this.vendorLogo || '';
}

/******
 * Get bookserie's logo for the coverpage
 ******/
Book.prototype.getBookSerieLogo = function(){
    return this.bookserielogo || '';
}

/******
 * Get the toc as an object
 ******/
Book.prototype.getToc = function(){
    var tocobj = {
        "bookid": this.bookid,
        "title": this.title,
        "curriculum": this.curriculum || '',
        "defaultlang": this.defaultlang,
        "langs": this.langs || [],
        "author": this.author || [],
        "secondaryauthor": this.secondaryauthor || [],
        "vendorURL": this.vendorURL || ''
    }
    if (this.frontpage){
        tocobj["firstpage"] = this.frontpage.getObject();
    } else {
        tocobj["firstpage"] = {};
    }
    return tocobj;
}

/******
 * Get the toc as JSON string.
 ******/
Book.prototype.getTocJSON = function(){
    return JSON.stringify(this.getToc());
}

/******
 * Get the toc as html.
 ******/
Book.prototype.getTocHtml = function(beginId, lang){
    if (!beginId) {
        beginId = this.frontId();
    }
    lang = lang || this.defaultlang;
    var html = '<div class="ebooktoc">\n';
    var title = this.titles[lang].booktitle;
    if (beginId === -1 || beginId === this.frontpage.chapid){
        html += '<h1>'+title+'</h1>\n';
    }
    html += '<ul>\n';
    if (beginId === -1){
        html += this.frontpage.tocHtml(lang);
    }else{
        html += (this.getChapter(beginId) && this.getChapter(beginId).tocHtml(lang) || '');
    }
    html += '</ul>\n</div>\n';
    return html;
}

/******
 * Add a given chapter id to the dict.
 ******/
Book.prototype.addToDict = function(chapter){
    this.tocdict[chapter.chapid] = chapter;
}

/******
 * Append chapter id in the end of the linear array.
 ******/
Book.prototype.appendChapter = function(chapid){
    this.chapters.push(chapid);
}

/******
 * Add a new chapter /section / subsection under one with id chapid.
 * by default add to the end, if the index is not given.
 ******/
Book.prototype.addChapterUnder = function(chapid, data, index){
    // TODO!
    var chapter = this.getChapter(chapid);
    if (typeof(index) === 'undefined') {
        index = chapter.chapters.length;
    }
    var prev = chapter.chapters[index - 1];
    var next = chapter.chapters[index];
    var newchap = new BookChapter(data);
    newchap.prev = prev.chapid;
    newchap.next = next && next.chapid;
    prev.next = next.prev = newchap.chapid;
    chapter.chapters.splice(index, 0, newchap);
    var cindex = this.chapters.indexOf(prev.chapid);
    this.chapters.splice(cindex, 0, newchap.chapid);
}

/******
 * Convert number to string of characters.
 ******/
Book.prototype.numtochar = function(index, rest){
    if (typeof(rest) === "undefined") { rest = ""; }
    var characters = 'abcdefghijklmnopqrstuvwxyz';
    if (index/26 > 1){
        return this.numtochar((index/26 | 0) - 1, characters[index%26] + rest);
    } else if (index/26 === 1){
        return 'aa' + rest;
    } else {
        return characters[index%26] + rest;
    }
}

/******
 * Find first free chapter id.
 ******/
Book.prototype.getFreeChapid = function(){
    var last = this.lastFreeChapid || 0;
    while (this.getChapter(this.numtochar(last))){
        last++;
    }
    this.lastFreeChapid = last;
    return this.numtochar(last);
}

Book.prototype.writeTocText = function(){
    // Export toc to plain text list.
    var text = '';
    text += '!'+this.bookid+';'+this.title+'\n';
    text += this.frontpage.title +';'+ this.frontpage.chapid +'\n';
    for (var i = 1; i < this.chapters.length; i++){
        var chapter = this.tocdict[this.chapters[i]];
        for (var j = 0; j<chapter.chapid.length; j++){
            text += '#';
        }
        text += ' ' + chapter.title + ';' + (chapter.types ? chapter.types.join(' ') : '') + ';' + chapter.chapid + ';' + chapter.content + '\n';
    }
    return text;
}

Book.prototype.readTocText = function(tiddler){
    // Read / import toc from plain text list in given tiddler or from <bookid>_toc_text.
    if (!tiddler){
        tiddler = this.bookid + '_toc_text';
    }
    if (!store.tiddlerExists(tiddler)){
        return false;
    }
    var characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    var text = store.getTiddlerText(tiddler);
    var splitted = text.split('\n');
    var booktoc = {};
    var bookhead = splitted[0].replace(/^\!*/,'').split(';');
    booktoc.bookid = bookhead[0];
    booktoc.title = bookhead[1] || '';
    booktoc.alttitles = {};
    booktoc.curriculum = bookhead[2] || '';
    booktoc.lang = bookhead[3] || '';
    booktoc.altlangs = bookhead[4].split(' ') || [];
    if (booktoc.altlangs.length == 1 && booktoc.altlangs[0] == ''){
        booktoc.altlangs = [];
    }
    booktoc.style = bookhead[5] || '';
    booktoc.frontpage = {};
    var fronttext = splitted[1].split(';');
    booktoc.frontpage.title = fronttext[0] || 'Frontpage';
    booktoc.frontpage.chapid = fronttext[1] || 'front';
    booktoc.frontpage.types = ["front"];
//    booktoc.frontpage.contentLevel = 0;
    booktoc.frontpage.content = booktoc.bookid + '_Frontpage';
    booktoc.frontpage.chapters = [];
    var chapterNum = 0;
    var sectionNum = 0;
    var subsectionNum = 0;
    var chaptext = splitted.join('\n');
    var chapters = chaptext.split(/\n#(?!#)/);
    for (var i = 1; i<chapters.length && i<characters.length; i++){
        var chapter = {};
        var ctexts = chapters[i].split('\n')[0].split(';');
        chapter.title = (ctexts[0] || '').trim();
        chapter.types = ["chapter"].concat((ctexts[1] || '').trim().split(' '));
        if (chapter.types[chapter.types.length - 1] === ''){
            chapter.types.pop();
        }
//        chapter.contentLevel = 1;
        chapter.chapid = (ctexts[2] || characters[i]).trim();
        chapter.content = (ctexts[3] || booktoc.bookid + '_chapter_' + chapterNum).trim();
        chapterNum ++;
        chapter.chapters = [];
        var sections = chapters[i].split(/\n##(?!#)/);
        for (var j = 1; j<sections.length && j<characters.length; j++){
            var section = {};
            var stexts = sections[j].split('\n')[0].split(';');
            section.title = (stexts[0] || '').trim();
            section.types = ["section"].concat((stexts[1] || '').trim().split());
            if (section.types[section.types.length - 1] === ''){
                section.types.pop();
            }
//            section.contentLevel = 2;
            section.chapid = (stexts[2] || characters[i] + characters[j]).trim();
            section.content = (stexts[3] || booktoc.bookid + '_section_' + sectionNum).trim();
            sectionNum++;
            section.chapters = [];
            var subsections = sections[j].split(/\n###(?!#)/);
            for (var k = 1; k<subsections.length && k<characters.length; k++){
                var subsection = {};
                var sstexts = subsections[k].split('\n')[0].split(';');
                subsection.title = (sstexts[0] || '').trim();
                subsection.types = (sstexts[1] || '').trim().split();
                if (subsection.types[subsection.types.length - 1] === ''){
                    subsection.types.pop();
                }
//                subsection.contentLevel = 3;
                subsection.chapid = (sstexts[2] || characters[i] + characters[j] + characters[k]).trim();
                subsection.content = (sstexts[3] || booktoc.bookid + '_subsection_' + subsectionNum).trim();
                subsectionNum++;
                subsection.chapters = [];
                section.chapters.push(subsection);
            }
            chapter.chapters.push(section);
        }
        booktoc.frontpage.chapters.push(chapter);
    }
    return JSON.stringify(booktoc);
}

/******
 * Get the id of frontpage.
 ******/
Book.prototype.frontId = function(){
    return this.frontpage.chapid;
}

/******
 * Reverse search the chapid by content name.
 ******/
Book.prototype.getIdByContent = function(contentName){
    if (typeof(this.contentindex) == 'undefined'){
        var chapter = this.frontpage;
        this.contentindex = {};
        this.contentindex[chapter.content] = chapter.chapid;
        while (chapter.next !== false){
            chapter = this.tocdict[chapter.next];
            this.contentindex[chapter.content] = chapter.chapid;
        }
    }
    return this.contentindex[contentName];
}

/******
 * Reverse search the title by content name.
 ******/
Book.prototype.getTitleByContent = function(contentName, lang){
    return this.getTitle(this.getIdByContent(contentName), lang);
}

/******
 * Reverse search chapid by chapter number.
 ******/
Book.prototype.getIdByNumber = function(numberstr){
    if (typeof(numberstr) != 'string'){
        return false;
    }
    var chapid, chapter;
    var found = false;
    for (var i = 0; !found && i < this.chapters.length; i++) {
        chapid = this.chapters[i];
        chapter = this.getChapter(chapid);
        if (chapter.getNumberLong() === numberstr) {
            found = true;
        }
    }
    return chapid;
}

/******
 * 
 ******/
Book.prototype.getSubToc = function(chapid, currentid, sublevel, lang) {
    if (!this.tocdict[chapid]) {
        return '';
    }
    var chapter = this.tocdict[chapid];
    var beforehtml = '';
    var afterhtml = '';
    var itemnumber = this.getNumberedTitle(chapid, lang).match(/^(?:[0-9]+\.)*[0-9]+/) || '';
    var itemtext = this.getTitle(chapid, lang);
    if (itemtext == ''){
        itemtext = this.getTitle(chapid);
    }

    var conthtml = '';
    if (!sublevel){
        beforehtml = '<div class="subtocwrapper">\n<div class="subtocinner">\n'+beforehtml;
        afterhtml = afterhtml + '</div>\n</div>\n';
        conthtml += '<div class="subtocitem" chapid="'+chapid+'"><span class="tocitemnumber">'+ itemnumber + '</span> <a href="javascript:;">' + itemtext + '</a></div>\n';
    } else {
        var classes = ['subtocitem'];
        if (chapid == currentid){
            classes.push('iscurrentpage');
        }
        conthtml = '<li class="'+classes.join(' ')+'" chapid="'+chapid+'"><a href="javascript:;"><span class="tocitemnumber">'+ itemnumber + '</span> ' + itemtext + '</a>\n';
    }
    if (chapter.chapters.length > 0){
        conthtml += '<ul class="subtoc" chapid="'+chapid+'">\n';
        for (var i = 0; i < chapter.chapters.length; i++){
            conthtml += this.getSubToc(chapter.chapters[i].chapid, currentid, true, lang);
        }
        conthtml += '</ul>\n';
    }
    if (!!sublevel){
        conthtml += '</li>\n';
    }
    return beforehtml + conthtml + afterhtml;
}

/******
 * Get an array of all available languages in this book.
 ******/
Book.prototype.getLangs = function(){
    //var langs = [];
    //for (lang in this.langs) {
    //    langs.push(lang);
    //}
    langs = JSON.parse(JSON.stringify(this.langs));
    return langs;
}

/**
 * Check if given language is available.
 * @param {String} lang   Asked language
 * @return {String}  Usable language. This is lang, if it is available, otherwise this.defaultlang or the first in this.langs.
 */
Book.prototype.getUsableLang = function(lang) {
    var result;
    var found = false;
    for (var i = 0, len = this.langs.length; i < len && !found; i++) {
        if (this.langs[i].id === lang) {
            found = true;
        };
    };
    if (found) {
        result = lang;
    } else {
        result = this.defaultlang || this.langs[0] || '';
    };
    return result;
}

/**
 * Get the contentlevel of a given chapter
 * @param {String} chapid    The id of the chapter
 * @returns {String} the contentlevel
 */
Book.prototype.getContentlevel = function(chapid) {
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getContentlevel() || '';
}

/******
 * Get the headimage.
 ******/
Book.prototype.getHeadimage = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getHeadimage(chapid) || {type: 'headimage', text: ''};
    
}

/******
 *Get link for headimage source
 ******/
Book.prototype.getHeadlink = function(chapid){
    // TODO!
}

/******
 * Get the pagetype of given chapter
 ******/
Book.prototype.getPagetype = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getPagetype() || '';
}

/******
 * Set the pagetype of given chapter
 ******/
Book.prototype.setPagetype = function(chapid, ptype){
    var chapter = this.getChapter(chapid);
    if (chapter){
        chapter.setPagetype(ptype);
    }
}

/******
 * Get the extra-information (boolean) of given chapter
 ******/
Book.prototype.getExtra = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getExtra() || false;
}

/******
 * Set the extra-information of given chapter
 ******/
Book.prototype.setExtra = function(chapid, extra){
    var chapter = this.getChapter(chapid);
    if (chapter){
        chapter.setExtra(extra);
    }
}

/******
 * Returns true if "page" have no content or subcontent
 ******/
// TODO: Does not work like this!
Book.prototype.isEmptyById = function(chapid){
    var chapter = this.getChapter(chapid);
    if (!chapter){
        return true;
    }
    return chapter && chapter.chapters.length === 0 && !this.hasText(chapid);
}

/******
 * Returns true if "page" has content
 ******/
// TODO: Does not work like this!
Book.prototype.hasContent = function(chapid){
    var contentId;
    var contentData = false;
    var chapter = this.getChapter(chapid);
    if (chapter) {
        contentId = chapter.getContent();
        contentData = this.getPageData(contentId);
    }
    return !!contentData;
}

/******
 * Get the level of given chapter
 ******/
Book.prototype.getLevel = function(chapid){
    var chapter = this.getChapter(chapid);
    return chapter && chapter.getLevel();
}

/******
 * Get the toc as linear list.
 ******/
Book.prototype.getLinearToc = function(lang){
    lang = lang || this.defaultlang;
    var linear = [];
    for (var i = 0; i < this.chapters.length; i++) {
        var chapid = this.chapters[i];
        var chapter = this.getChapter(chapid);
        var item = {
            "id": chapid,
            "level": chapter.getLevel(),
            "title": chapter.getTitle(lang, this.defaultlang),
            "label": this.getChapNumberLong(chapid)
        }
        linear.push(item);
    }
    return linear;
}

/******
 * Get the page content from store.
 ******/
Book.prototype.getContentDataSync = function(contentIds, lang, callback){
    if (typeof(contentIds) === 'string') {
        contentIds = [contentIds];
    }
    var contentItems = {};
    for (var i = 0; i < contentIds.length; i++) {
        contentItems[contentIds[i]] = this.store.getContent(contentIds[i]);
    }
    return contentItems;
}

/******
 * Get the page content from store.
 ******/
Book.prototype.getContentData = function(contentIds, lang, callback){
    var book = this;
    if (typeof(contentIds) === 'string') {
        contentIds = [contentIds];
    }
    var itemsToFind = contentIds.length;
    var contentItems = {};
    var __handleContent = function(element){
        --itemsToFind;
        if (element.metadata && element.metadata.id) {
            var elemid = element.metadata.id
            var parts = elemid.split('/');
            var elementid = parts[parts.length - 1];
            if (book.elementboxNumbers[elementid]) {
                element.data.elementboxNumber = book.elementboxNumbers[elementid];
            }
            contentItems[elemid] = element;
        } else {
            // Should not come here.
            console.log('error');
            console.log(element);
        }
        if (itemsToFind === 0) {
            callback.call(window, contentItems);
        }
    }
    for (var i = 0; i < contentIds.length; i++) {
        //var idstring = (lang ? lang + '/' + contentIds[i] : contentIds[i]);
        var idstring = contentIds[i];
        //this.store.getContent(idstring, __handleContent);
        this.store.getContentData(idstring, lang, this.defaultlang, __handleContent);
    }
}



/**
 * BookStorage class for storaging the element content of the book
 * @class BookStorage
 * @param {Object} store - the data of the book.
 */
var BookStorage = function(options) {
    this.content = options.content;
    this.contentPrefix = options.contentPrefix;
}

BookStorage.prototype.getContent = function(contentID, callback) {
    var cid = this.contentPrefix + contentID;
    
    if (typeof(this.content[cid]) !== 'undefined') {
        var result = JSON.parse(JSON.stringify(this.content[cid]));
        
        // Add id and content prefix to the loaded content.
        if (typeof(result.metadata) === 'undefined') {
            result.metadata = {};
        };
        result.metadata.id = contentID;
        result.metadata.contentPrefix = this.contentPrefix;
        
        if (callback) {
            callback.call(window, result);
        }
    } else {
        result = {type: 'response', metadata: {}, data: {}};
        result.metadata.id = contentID;
        result.metadata.contentPrefix = this.contentPrefix;
        result.data[cid] = '';
        if (callback) {
            callback.call(window, result);
        }
    }
    return result;
}

BookStorage.prototype.getContentData = function(contentID, lang, deflang, callback) {
    var langs = [lang, deflang, ''];
    var cid, result;
    for (var i = 0, len = langs.length; i < len; i++) {
        if (langs[i]) {
            cid = this.contentPrefix + langs[i] + '/' + contentID;
        } else {
            cid = this.contentPrefix + contentID;
        }
        if (typeof(this.content[cid]) !== 'undefined') {
            result = JSON.parse(JSON.stringify(this.content[cid]));
            
            // Add id and content prefix to the loaded content.
            if (typeof(result.metadata === 'undefined')) {
                result.metadata = {};
            }
            result.metadata.id = contentID;
            result.metadata.lang = result.metadata.lang || langs[i];
            if (callback) {
                callback.call(window, result);
            }
            break;
        }
    }
    if (!result) {
        result = {type: 'response', metadata: {}, data: {}};
        result.metadata.id = contentID;
        result.metadata.contentPrefix = this.contentPrefix;
        result.data[this.contentPrefix + contentID] = '';
        if (callback) {
            callback.call(window, result);
        }
    }
    return result;
}




//}}}