/*****************************************************************
 * Application filesystem for node-webkit
 * Constructor
 * 20.5.2014
 * Petri Sallasmaa
 * Petri Salmela
 *****************************************************************/
    function appFileSystem(params) {
        var extended = 
            {
                config: {
                    general: {
                        version: 2015.0302
                    }
                },
                pathTo:{
                    main:{
                        js  : "data/js",
                        css : "data/css",
                        root: ".",
                        credits: "data/credits",
                        settings: "data/settings"
                    },
                    system:{
                        js  : "data/js/system",
                        css : "data/css/system",
                        themes: "data/css/systemThemes",
                        themetemplates: "data/css/templateObjects"
                    },
                    applications: {
                        main  : "data/applications"
                    },
                    plugins: {
                        main: "data/js/plugins",
                        js  : "data/js/plugins/js",
                        css : "data/js/plugins/css"
                    },
                    content: {
                        main: "data/content"
                    }
                },
                reloadNeeded : false
            };
            // },
            // params
        // );
        
        for (var item in extended) {
            this[item] = extended[item];
        }

        try{
            this.place = params.place
            
        }catch(e){
            this.place = window.document.body;
        }
        this.fileSystem = new require("fs");
        
    }
    
    /******
     * gives path to type
     ******/
    appFileSystem.prototype.getPath = function(category, type) {
        try {
            return this.pathTo[category][type];
        } catch(err) {
            console.log("FilepathError:", err, category, type);
        };
    };
    
    appFileSystem.prototype.getContentPath = function(info){
        try{
            if(info.category === "content"){
                var mainpath = this.getPath(info.category, info.type);
                if (mainpath) {
                    if (info.context && info.context.contextid) {
                        return mainpath + "/" + info.context.contextid;
                    } else {
                        return mainpath;
                    };
                };
                console.log("ERROR with content path mainpath", info);
                return "";
            }else{
                console.log("ERROR with content path category", info);
                return "";
            };
        } catch(err) {
            console.log("getContentPath:", err, info);
        };
    };
    /******
     * Add content path
     ******/
    appFileSystem.prototype.addContentPath = function(pathinfo) {
        var type = pathinfo.type;
        var subpaths = pathinfo.subpath;
        this.pathTo.content[type] = this.pathTo['content']['main'] + "/" + type;
        if (typeof(subpath) !== "undefined") {
            this.pathTo.content[type].sub = subpaths;
        };
        if (this.checkTypeofFile(this.pathTo['content']['main']+"/"+type) === "nonexists") {
            this.fileSystem.mkdirSync(this.pathTo['content']['main'] + "/" + type);
        };
    };
    
    /******
     * Removes file-extensions for given filelist
     ******/
    appFileSystem.prototype.removeExtensions = function(filelist) {
        for (var i = 0; i < filelist.length; i++){
            filelist[i] = filelist[i].replace(/\.[^/.]+$/, "");
        };
        return filelist;
    };
    
    /******
     *Removes all but wanted extesion files in filelist
     *keepExtension (boolean) option keeps extension in filenames
     ******/
    appFileSystem.prototype.removeButExtensions = function(filelist, extension, keepExtension) {
        for (var i = 0; i < filelist.length; true) {
            if (!filelist[i].match(new RegExp('\.'+extension+'+$'))) {
                filelist.splice(i,1);
            } else {
                if (!keepExtension) {
                    filelist[i] = filelist[i].replace(/\.[^/.]+$/, "");
                };
                i++;
            };
        };
        return filelist;
    };
    
    /******
     * Removes directories from filelist
     ******/
    appFileSystem.prototype.removeDirectories = function(path,filelist) {
         for(var i=0;i<filelist.length;true){
            if(!this.fileSystem.lstatSync(path+"/"+filelist[i]).isFile()){
                filelist.splice(i,1);
            }else{
                i++;
            }
        }
        return filelist;
    }
    
    /**
     * Mac generates extra files with filename "._<filename>" for each file on FAT filesystem.
     * We want to remove those and other possible dotfiles.
     */
    appFileSystem.prototype.removeDotfiles = function(filelist) {
        for (var i = filelist.length - 1; i > -1; i--) {
            if (filelist[i][0] === '.') {
                filelist.splice(i, 1);
            };
        };
        return filelist;
    };
    
    /******
     * Returns filelist for given path
     * if extension is defined list includes only that type of files
     * with returnExtension in list can be file-extensions or not
     ******/
    appFileSystem.prototype.getfileList = function(pathTofiles, extension, returnExtension) {
        var filelist = this.fileSystem.readdirSync(pathTofiles, 'utf8');
        // Mac might have some stupid ._files
        filelist = this.removeDotfiles(filelist);
        filelist = this.removeDirectories(pathTofiles, filelist);
        if (typeof(extension) === "undefined") {
            return {path: pathTofiles, list: filelist};
        } else {
            return this.removeButExtensions(filelist, extension, !!returnExtension);
        };
    };
    
    /******
     * returns all system javascript files
     ******/
    appFileSystem.prototype.getSystemJS = function() {
        return this.getfileList(this.getPath('system','js'));
    }
    /******
     * returns all system theme css files
     ******/
    appFileSystem.prototype.getSystemThemeCSS = function() {
        return this.getfileList(this.getPath('system','themes'));
    }
    /******
     * returns all system css files
     ******/
    appFileSystem.prototype.getSystemCSS = function() {
        return this.getfileList(this.getPath('system','css'));
    }
    /******
     * returns list of system-applications
     ******/
    appFileSystem.prototype.getApplicationList = function() {
        return this.getfileList(this.getPath('applications','main'),'js',false);
    }
    /******
     * returns list of application plugins
     ******/
    appFileSystem.prototype.getPluginList = function() {
        return this.getfileList(this.getPath('plugins','main'),'json',false);
    }
    /******
     * returns list of system files
     * -TODO:
     *   get main systemfiles 
     *   for now return hardcoded list
     ******/
    appFileSystem.prototype.getSystemList = function() {
        return ["mainSystem", "systemJS"]
    }
    
    /******
     * create contentinfo json file of given contentlist(list of filenames)
     * infolistfile is writen to disk and list is also returned
     ******/
    appFileSystem.prototype.createContentInfo = function(searchPath,contentList,type) {
        var infoList =[];
        if(typeof(contentList) ==="undefined") contentList = this.fileSystem.readdirSync(searchPath);
        if(typeof(type) ==="undefined") type = "file";
        if(type ==="file"){
            contentList = this.removeDirectories(searchPath, contentList);
        }
        for(var i=0;i<contentList.length;i++){
            if(contentList[i]==="info.json"){continue;}
            if(type ==="folder"){
                var contentPath = searchPath+"/"+contentList[i]+"/metadata.json";
            }else{
                var contentPath = searchPath+"/"+contentList[i];
            }
            try{
                var fileContent = JSON.parse(this.fileSystem.readFileSync(contentPath,"utf8"));
                //Check if content have information about titles
                //TODO: dataformat should be same so titles should be in same structures
                var titles={};
                if(fileContent["titles"]){//for reader-pluging data
                    titles = fileContent["titles"];
                }else if(fileContent['data'] && fileContent['data']["title"]){// for notebook data
                    titles = fileContent['data']["title"];
                }else if(fileContent['title'] ){// for course data
                    titles = fileContent["title"];
                }
                var oneInfo = {"titles":titles,"filename":contentList[i]};
                if(fileContent['type'] && fileContent[fileContent['type']]){
                    oneInfo[fileContent['type']] = fileContent[fileContent['type']];
                }
                infoList.push(oneInfo); 
            }catch(e){
                infoList.push({"error":contentList[i]}); 
            }
        }
        this.fileSystem.writeFileSync(searchPath+"/info.json",JSON.stringify(infoList));
        return infoList;
        
    }
    /******
     * returns list of available content of given path ( list of filenames)
     ******/
    appFileSystem.prototype.getAvailableContent = function(path,type) {
        
        var searchPath= this.getPath('content','main')+path;
        if(this.fileSystem.existsSync(searchPath+"/info.json")){
            try{
                var infoObject = JSON.parse(this.fileSystem.readFileSync(searchPath+"/info.json","utf8"));
                if(this.fileSystem.readdirSync(searchPath).length-1 === infoObject.length){
                    return JSON.parse(this.fileSystem.readFileSync(searchPath+"/info.json","utf8"));
                }
            }catch(e){
                //infofile not readable or corrupted
                this.fileSystem.unlinkSync(searchPath+"/info.json");
            }
        }      
        return this.createContentInfo(searchPath,this.fileSystem.readdirSync(searchPath),type);
    }
    
    /******
     * returns file content in format of utf-8 string
     ******/
    appFileSystem.prototype.readFileToString = function(file) {
        return this.fileSystem.readFileSync(file,"utf8");
    }
    /******
     * writes content (string/object) to file async when ready triggeres file-saved event
     * TODO: option for binary buffer to write options
     *       trigger also jQuery event of file-saved
     *       event could be also filename+"ready" or something
     ******/
    appFileSystem.prototype.writeToFile = function(contextinfo,filename,file,buffer) {
        var appfs = this;
        if(buffer){
            if(typeof(file)==="object" ){
                if(file.type && file.type ==="Buffer"){
                    file = new Buffer(file);
                }
            }else if(typeof(file)==="string"){
                file = new Buffer(JSON.parse(file));
            }
        }else{
            if(typeof(file)==="object"){
                file = JSON.stringify(file);
            }
        }
        var readyevent = new CustomEvent("file-saved",{detail:{"filename":filename}});
        try{
            this.fileSystem.writeFileSync(this.getPath(contextinfo.category,contextinfo.type)+"/"+filename,file);
        }catch(err){
            console.log('writeTofile Error',err);
        }
        // this.fileSystem.writeFile(this.getPath(contextinfo.category,contextinfo.type)+"/"+filename,file,function(){
        	// //TODO: trigger also jQuery event?? if jQuery loaded
            // //console.log("file-saved",{detail:{"filename":filename}});
            // try{
                // document.body.
                        // dispatchEvent(readyevent);
            // }catch(e){
                // console.log("Error trigger writeok",e,filename);
            // }
        // });
    }
    /******
     * syncronised version of writeToFile function
     ******/
    appFileSystem.prototype.writeToFileSync = function(contextinfo, filename, file, buffer) {
        try{
            if (buffer) {
                if (typeof(file) === "object") {
                    if (file.type && file.type === "Buffer") {
                        file = new Buffer(file);
                    };
                } else if (typeof(file) === "string"){
                    file = new Buffer(JSON.parse(file));
                };
            } else {
                if (typeof(file) === "object"){
                    file = JSON.stringify(file);
                };
            };
            if (typeof(contextinfo) !== "undefined") {
                filename = this.getPath(contextinfo.category, contextinfo.type) + "/" + filename;
            };
            if (!(/\.\.\//).test(filename)) { // Security check
                this.fileSystem.writeFileSync(filename, file);
                var event = new CustomEvent("file-saved", {"filename": filename});
                document.getElementsByTagName("body")[0].dispatchEvent(event);
                return true;
            } else {
                console.log('writeToFileSync: Invalid filename / path (includes "../")')
                return false;
            }
        } catch(err) {
            console.log("writeToFileSync", err);
            return false;
        };
    };
    
    /******
     *
     ******/
    appFileSystem.prototype.writeContext = function(type,contextid) {
        
    }
    /******
     * returns object of application-plugin info json file
     ******/
    appFileSystem.prototype.readPluginListFileToObject = function(pluginname) {
        try{
            return JSON.parse(this.readFileToString(this.getPath('plugins','main')+"/"+pluginname+".json"));
        }catch(e){
            console.log("Parse error");
            console.log(e);
            //TODO:Parse error: throw somekind of error
            return {};
        }
    }
    /******
     * returns content type file (file "sub"-path have to included like books/1.json) as javascript list object
     * TODO: subpath have to be given now maybe can also given as type (notebook,course, ect..) 
     ******/
    appFileSystem.prototype.readFileToList = function(file) {
        var filepath = this.getPath(file.category,file.type)+(typeof(file.subpath) !=="undefined"?"/"+file.subpath:"")+"/"+file.filename;
        return this.readFileTo(filepath,"list");
    }
    /******
     * returns content type file (file "sub"-path have to included like books/1.json) as javascript object
     * TODO: subpath have to be given now maybe can also given as type (notebook,course, ect..) 
     ******/
    appFileSystem.prototype.readFileToObject = function(file) {
        var filepath = this.getPath(file.category,file.type)+(typeof(file.subpath) !=="undefined"?"/"+file.subpath:"")+"/"+file.filename;
        return this.readFileTo(filepath,"object");
    }
    /******
     * returns content type file (file "sub"-path have to included like books/1.json) as javascript object
     * TODO: subpath have to be given now maybe can also given as type (notebook,course, ect..) 
     ******/
    appFileSystem.prototype.readFileTo = function(fullFilePath,returnType) {
        try{
            var returnObj = JSON.parse(this.readFileToString(fullFilePath));
            if(returnType ==="list" && typeof(returnObj.length) === "undefined"){
                return [returnObj];
            }
            return returnObj;
        }catch(e){
            console.log("Parse error");
            console.log(e);
            //TODO:Parse error: throw somekind of error
            if(returnType ==="list"){
                return [];
            }
            return {};
        }
        
    }
    /******
     * 
     ******/
    appFileSystem.prototype.readContextToObject = function(context) {
        try {
            var filepath, filelist, tempObj = [], returnObj = {};
            filepath = this.getContentPath(context) + "/";
            try {
                filelist = this.fileSystem.readdirSync(filepath);
            } catch(e) {
                console.log("can't read filedir", e);
                return {};
            };
            for (var i = 0; i < filelist.length; i++) {
                try {
                    tempObj = {};
                    tempObj[filelist[i].replace(/\.[a-zA-Z0-9]*$/,"")] 
                        = JSON.parse(
                                this.readFileToString(
                                    filepath + filelist[i]
                                )
                            );
                    returnObj = jQuery.extend(
                        true,
                        {},
                        returnObj,
                        tempObj
                    );
                }catch(e){
                    console.log("Parse error");
                    console.log(e);
                    //TODO:Parse error: throw somekind of error
                    // return {};
                }
                
            }
            return returnObj;
        }catch(e){
            console.log('Error:readFileListToObject',e);
        }
    };
    
    /**
     * Read the tutorial file as object
     */
    appFileSystem.prototype.readTutorialToObject = function(lang) {
        var filepath = this.getContentPath({category: 'content', type: 'main'});
        var filename = filepath + '/notebook-tutorial-' + lang + '.json';
        if (!this.fileSystem.existsSync(filepath)) {
            filename = filepath + '/notebook-tutorial-en.json';
        };
        var fileobj = {};
        try {
            fileobj = {
                storage: JSON.parse(this.readFileToString(filename))
            };
        } catch (err) {
            fileobj = {};
            console.log('could not find valid tutorial: ' + filename, err);
        };
        return fileobj;
    };
    /******
     * returns  content type filelist  as javascript object
     * TODO: subpath have to be given now maybe can also given as type (notebook,course, ect..)
     *       add option to set "object" as parameters witch has keys like main_path, filelist, ect...
     ******/
    appFileSystem.prototype.readFileListToObject = function(file) {
        try{
        var filepath,filelist,tempObj = [],returnObj = {};
        filepath = this.getPath(file.category,file.type) +(typeof(file.subpath) !=="undefined"?"/"+file.subpath:"")+ "/";
        
        try{
            filelist = this.fileSystem.readdirSync(filepath);
        }catch(e){
            console.log("can't read filedir",e);
            return {};
        }
        for(var i=0;i<filelist.length;i++){
            try{
                tempObj = {};
                tempObj[filelist[i].replace(/\.[a-zA-Z0-9]*$/,"")] 
                    = JSON.parse(
                            this.readFileToString(
                                filepath+filelist[i]
                            )
                        );
                returnObj = jQuery.extend(
                    true,
                    {},
                    returnObj,
                    tempObj
                );
            }catch(e){
                console.log("Parse error");
                console.log(e);
                //TODO:Parse error: throw somekind of error
                // return {};
            }
            
        }
        return returnObj;
        }catch(e){
            console.log('Error:readFileListToObject',e);
        }
    }
    /******
     * returns list of mainsystem applications
     ******/
    appFileSystem.prototype.getApplicationJS = function() {
        return this.getfileList(this.getPath('applications','main'));
    }
    /******
     *checkPlugins existing
     * -plugins can be found as pathTo['plugins']+plugingname+".json" filesystem
     *  these files are list of needed js and css files for every plugin
     ******/
    appFileSystem.prototype.checkPlugins = function(pluginlist) {
        var missingList=[];
        // Removed json file no more in use
        // for(var plugin in pluginlist){
            // if(!this.fileSystem.existsSync(this.getPath('plugins')+"/"+pluginlist[plugin]+".json")){
                // missingList.push(pluginlist[plugin]);
            // }
        // }
        return missingList;
    }
    /******
     * plugin loadlist
     ******/
    appFileSystem.prototype.getPluginloadList = function(plugin) {
        var path = this.getPath('plugins','main');
        var files_list = {
            "path"          : path.replace("data/", ""),
            "jslist"        : this.removeDirectories(path + "/js", this.fileSystem.readdirSync(path + "/js")),
            "csslist"       : this.removeDirectories(path + "/css", this.fileSystem.readdirSync(path + "/css"))
        }
        for (var i = 0; i < files_list.jslist.length; i++) {
            if (files_list.jslist[i].substr(-9) === "_start.js" && files_list.jslist[i] !== plugin + "_start.js") {
                files_list.jslist.splice(i,1);
            }
        }
        files_list.jslist.sort();
        return files_list;
    }
    /******
     * New implementation of getting plugin loadlist assist with plugin-json-file
     * returns list of javasript- and css files that have to load that plugin works also filepath to files
     * loadlist is got by getPluginloadList function now
     ******/
    appFileSystem.prototype.getPluginloadListJSON = function(plugin) {
        var path = this.getPath('plugins','main');
        // var listObject = this.readPluginListFileToObject(plugin);
        return jQuery.extend(true,{},{"path":path.replace("data/","")},this.readPluginListFileToObject(plugin));
    }
    /******
     * Seach if plugin have themeCSS files
     * returns list of  css files that can be loaded for themes
     ******/
    appFileSystem.prototype.getPluginthemeFiles = function(plugin) {
        var path = this.getPath('plugins','css');
        try{
            return this.fileSystem.readdirSync(path+"/"+plugin+"-themes");
        }catch(e){
            console.log("getPluginthemeFiles ERROR",e);
            return [];
        }
    }
    /******
     * returns an javascriptobject of settings by given filename
     ******/
    appFileSystem.prototype.getOptions = function(filename) {
        var settingsFile = this.getPath('main','settings')+"/"+filename;
        if(this.fileSystem.existsSync(settingsFile)){
            try{
                return JSON.parse(this.readFileToString(settingsFile));
            }catch(e){
                console.log(e+"systemoption parse error",filename);
                return {};
            }
        }
        return {};
                
    }
    /******
     * returns saved general settings
     ******/
    appFileSystem.prototype.getSystemOptions = function() {
        return jQuery.extend(
            true,
            {},
            this.getOptions('systemoptions.json'),
            {includeApplications : this.getApplicationList()},
            {includePlugings : this.getPluginList()},
            {includeSystem : this.getSystemList()}
            );                
    }
    /******
     * returns saved usersettings
     ******/
    appFileSystem.prototype.getUserOptions = function() {
        return this.getOptions('useroptions.json');                
    }
    /******
     * returns saved librarysettings
     ******/
    appFileSystem.prototype.getLibraryOptions = function() {
        var result = $.extend({}, this.getOptions('libraryoptions.json'), this.defaults.libraryoptions);
        return result;
    }
    /******
     * returns saved generalsettings
     ******/
    appFileSystem.prototype.getGeneralOptions = function() {
        return this.getOptions('generaloptions.json');                
    }
    /******
     * returns saved contentsettings
     ******/
    appFileSystem.prototype.getContentOptions = function() {
        return this.getOptions('contentoptions.json');                
    }
    /******
     * returns saved updateinfo
     ******/
    appFileSystem.prototype.getUpdateInfo = function() {
        return this.getOptions('updateinfo.json');                
    }
    /******
     * returns saved updateinfo
     ******/
    appFileSystem.prototype.getUiThemeTemplates = function() {
        var returnObj = {};
        var templatefiles = this.getfileList(this.getPath('system','themetemplates'));
        for (var i=0; i<templatefiles.list.length;i++){
            returnObj[templatefiles.list[i].replace(/\.json$/,'')] = this.readFileTo(templatefiles.path+"/"+templatefiles.list[i],"object");
        }
        return returnObj;              
    }
    /******
     * saves user options to file
     * triggeres event file-saved
     * TODO: maybe user_settings-ready event?
     ******/
    appFileSystem.prototype.saveUserOptions = function(options) {
        this.writeToFile({category:"main",type:"settings"},"useroptions.json",options);                
    }
    /******
     * saves library options to file
     * triggeres event file-saved
     * TODO: maybe library_settings-ready event?
     ******/
    appFileSystem.prototype.saveLibraryOptions = function(options) {
        this.writeToFile({category: "main", type: "settings"}, "libraryoptions.json", options);                
    };
    /******
     *saves systemsettings to file
     * triggeres event file-saved
     * TODO: maybe general_settings-ready event?
     ******/
    appFileSystem.prototype.saveSystemOptions = function(options) {
        this.writeToFile({category:"main",type:"settings"},"systemoptions.json",options);
    }
    /******
     *saves systemsettings to file
     * triggeres event file-saved
     * TODO: maybe general_settings-ready event?
     ******/
    appFileSystem.prototype.saveGeneralOptions = function(options) {
        this.writeToFile({category:"main",type:"settings"},"generaloptions.json",options);
    }
    /******
     *saves contentsettings to file
     * triggeres event file-saved
     * TODO: maybe content_settings-ready event?
     ******/
    appFileSystem.prototype.saveContentOptions = function(options) {
        this.writeToFile({category:"main",type:"settings"},"contentoptions.json",options);
    }
    /******
     *saves updateinfo to file
     * triggeres event file-saved
     * TODO: maybe updateinfo_settings-ready event?
     ******/
    appFileSystem.prototype.saveUpdateInfo = function(options) {
        this.writeToFile({category: "main", type: "settings"}, "updateinfo.json", options);
    }
    /******
     * checks that filename will be unique
     ******/
    appFileSystem.prototype.checkFileUniqueExists = function(context,filename,username) {
        var path;
        // switch(type){
            // case "notebook":
                
                // path = this.getPath('content')+"/notebooks/"+filename+"json";
            // break;
        // }
        path = this.getPath(context.category,context.type)+(typeof(context.contextid) !=="undefined"?"/"+context.contextid+"/":"/")+filename;
        if(path && this.fileSystem.existsSync(path)){
            return filename;
        }else{
            return this.checkFileUniqueExists(context,this.generateFilename(username),username);
        }

    }
    /******
     * checks if file exist
     ******/
    appFileSystem.prototype.checkTypeofFile = function(filepath) {
        try {
            if (filepath === false) return "error";
            var exists = this.fileSystem.existsSync(filepath);
            if (!exists){
                return "nonexists";
            } else {
                var lstat = this.fileSystem.lstatSync(filepath);
                if (lstat.isFile()) {
                    return "file";
                } else if(lstat.isDirectory()) {
                    return "directory";
                } else {
                    return "nofile";
                };
            };
        } catch(err) {
            return "error";
        };
    };
    /******
     * checks if file exist
     ******/
    appFileSystem.prototype.checkFileExists = function(filepath) {
        try{
            return this.fileSystem.existsSync(filepath);
        }catch(err){
            console.log(err);
            return false;
        }
    }
    /******
     * checks if context exist
     ******/
    appFileSystem.prototype.checkContextExists = function(context) {
        var path = this.getPath(context.category,context.type)+(typeof(context.contextid) !=="undefined"?"/"+context.contextid:"")
        if(path && this.fileSystem.existsSync(path)){
            return true;
        }else{
            return false;
        }
    }
    appFileSystem.prototype.makeUniqueContext= function(context,username){
        if(!this.checkContextExists(context)){
            //allready unique
            return context;
        }else{
            //make unique
            context.contextid = this.generateContextid(username);
            return this.makeUniqueContext(context,username);
        }
    }
    /******
     * generates unique id using username and timestamp
     ******/
    appFileSystem.prototype.generateContextid = function(username) {
            if(typeof(username)==="undefined")username = "anonymous";
            return username+(new Date).getTime();
    }
    /******
     * generates unique filename using username and timestamp
     ******/
    appFileSystem.prototype.generateFilename = function(username,type) {
            if(typeof(username)==="undefined")username = "";
            return username+(new Date).getTime()+(typeof(type)!=="undefined"?"."+type:"");
    }
    /******
     * get new context
     ******/
    appFileSystem.prototype.getNewContext = function(username,contenttype,isloop) {
        var newContextid = this.generateFilename(username);
        if(typeof(isloop) !=="undefined"){
            if(isloop > 5){
                console.log("loopError new content");
                return newContextid+Math.random().toString(36).replace(/[^a-z]+/g, '').substr(0, 10);
            }
        }else{
            isloop =0;
        }
        if (!this.checkContextExists({category:'content',type:contenttype,contextid:newContextid})){
            return newContextid;
        }else{
            return this.getNewContext(username,contenttype,isloop+1);
        }
    }
    /******
     * get new filename
     ******/
    appFileSystem.prototype.getNewFilename = function(username,contenttype,filetype) {
            return this.checkFileExists(contenttype,this.generateFilename(username,(typeof(filetype) !== "undefined"?filetype:"json")),username);
    }
    /******
     * creates new content based on some default information of content
     ******/
    appFileSystem.prototype.createNewContent = function(contentdata) {
        if(contentdata.newcontent.copymaterialfromcontext){
            if(contentdata.startdata){
                try{
                    var materialdata = contentdata.startdata.value;
                }catch(err){
                    console.log("StartdataError",err);
                    var materialdata = this.readFileToObject({category:"content",type:contentdata.newcontent.copymaterialfromcontext.context_type,subpath:contentdata.newcontent.copymaterialfromcontext.contextid,filename:"materialdata.json"});
                    if(contentdata.includeStorage){
                        var storage = this.readFileToObject({category:"content",type:contentdata.newcontent.copymaterialfromcontext.context_type,subpath:contentdata.newcontent.copymaterialfromcontext.contextid,filename:"storage.json"});
                        var meltedData = this.getNotebookData({"materialdata":materialdata,"storage":storage,"id":contentdata.newcontent.copymaterialfromcontext.contextid});
                        materialdata = {};
                        materialdata[contentdata.newcontent.copymaterialfromcontext.contextid] = meltedData;
                    }
                }
            }else{
                var materialdata = this.readFileToObject({category:"content",type:contentdata.newcontent.copymaterialfromcontext.context_type,subpath:contentdata.newcontent.copymaterialfromcontext.contextid,filename:"materialdata.json"});
                if(contentdata.includeStorage){
                    var storage = this.readFileToObject({category:"content",type:contentdata.newcontent.copymaterialfromcontext.context_type,subpath:contentdata.newcontent.copymaterialfromcontext.contextid,filename:"storage.json"});
                    var meltedData = this.getNotebookData({"materialdata":materialdata,"storage":storage,"id":contentdata.newcontent.copymaterialfromcontext.contextid});
                    materialdata = {};
                    materialdata[contentdata.newcontent.copymaterialfromcontext.contextid] = meltedData;
                }
            }
            contentdata.newcontent.contentdata['materialdata'] = materialdata;

        }
        try{
            this.saveContentData({
                savetype:contentdata.newcontent.savetype,
                contentType:{
                    category:"content",
                    type:(contentdata.newcontent.contenttype ==="notebooks" || contentdata.newcontent.contenttype ==="notebook"?"own_notes":contentdata.newcontent.contenttype),
                    context:{
                        contextid:contentdata.newcontent.contextid
                    }
                },
                saveobj:contentdata.newcontent.contentdata
            });
            return true;
        }catch(e){
            console.log("createNewContent Error");
            console.log(e);
        }
    }
    /******
     * creates new content based on some default information of content
     ******/
    appFileSystem.prototype.makeStartdata = function(contentdata) {
        if(contentdata.copymaterialfromcontext){
            var materialdata = this.readFileToObject({category:"content",type:contentdata.copymaterialfromcontext.context_type,subpath:contentdata.copymaterialfromcontext.contextid,filename:"materialdata.json"});
            if(contentdata.includeStorage){
                var storage = this.readFileToObject({category:"content",type:contentdata.copymaterialfromcontext.context_type,subpath:contentdata.copymaterialfromcontext.contextid,filename:"storage.json"});
                var meltedData = this.getNotebookData({"materialdata":materialdata,"storage":storage,"id":contentdata.copymaterialfromcontext.contextid});
                materialdata = {};
                materialdata[contentdata.copymaterialfromcontext.contextid] = meltedData;
            }
            return {"key":"materialdata.json","value":materialdata} ;

        }else{
            //TODO:
            //- what else it could be?
            return {"key":"startdata.json","value":{}};
        }
    }
     
    /**
     * Tool to convert notebook's materialdata and storagedata to merged materialdata
     * @param {Object} data                All data given
     * @param {String} data.id             Id of the notebook
     * @param {Object} data.materialdata   materialdata
     * @param {Object} data.storage        storage's data
     */
    appFileSystem.prototype.getNotebookData = function(data) {
        var id = data.id || data.materialdata.id || data.materialdata.name;
        var mdata = data.materialdata[id]
        if (mdata.data && mdata.data.content && mdata.data.content.frontpage) {
            var common = mdata.data.content;
            mdata.data.content = {
                common: common
            };
        };
        var defaultdata = {id: id, materialdata: {}, storage: {}}
        defaultdata.materialdata[id] = {
            type: "book",
            metadata: {
                creator: "Anonymous",
                created: (new Date()).toString()
            },
            data: {
                bookid: id,
                title: {
                    common: "Notebook"
                },
                toc: {
                    type: "toc",
                    metadata: {},
                    data: {
                        firstpage: {}
                    }
                },
                pagetitles: {
                    common: {}
                },
                index: {},
                content: {},
                defaultlang: "fi",
                langs: [
                    "fi"
                ]
            },
            name: id
        };
        defaultdata.storage = {
            storages: {
                notebookcontent: {
                    refs: {},
                    contentdata: {},
                    contentinfo: {}
                }
            }
        }
        data = $.extend(true, {}, defaultdata, data);
        
        var result = data.materialdata[id];
        var storage = data.storage.storages.notebookcontent;
        var reflist = storage.refs[id] || [];
        var cdata, cid, content;
        for (var lang in storage.contentdata) {
            cdata = storage.contentdata[lang];
            for (var i = 0, len = reflist.length; i < len; i++) {
                cid = reflist[i];
                content = cdata[cid];
                if (content) {
                    switch (content.type) {
                        case 'toc':
                            if (lang === 'common') {
                                result.data.toc = content;
                            };
                            break;
                        case 'pagetitles':
                            if (!result.data.pagetitles[lang]) {
                                result.data.pagetitles[lang] = {};
                            };
                            result.data.pagetitles[lang] = content.data;
                            break;
                        case 'index':
                            if (!result.data.index[lang]) {
                                result.data.index[lang] = {};
                            };
                            result.data.index[lang] = content.data;
                            break;
                        default:
                            if (!result.data.content[lang]) {
                                result.data.content[lang] = {};
                            };
                            result.data.content[lang][cid] = content;
                            break;
                    };
                };
            };
        };
        return result;
    };
    /******
     *
     ******/
    appFileSystem.prototype.processUpdateFile = function(data) {
        
        var isBuffer = (data.subtype ==="buffer"?true:false);
        /*switch(data.subtype){
            case"string":
                this.writeToFileSync([data.updatefor,data.path],data.filename,data.data);
                // console.log([data.updatefor,data.path],data.filename,data.data);
            break;
            case"buffer":
            break;
        }*/
        this.writeToFileSync({"category": data.updatefor, "type": data.path}, data.filename, data.data, isBuffer);
        if (isBuffer) {
            console.log([data.updatefor, data.path], data.filename, data.data, isBuffer);
        };
    };

    /******
     *
     ******/
    appFileSystem.prototype.processContentUpdates = function(data) {
        var returnobj = {"updateerror": {}};
        for (var contentype in data) {
            for (var contextid in data[contentype]) {
                var updateparts = this.readFileToObject({category: "content", type: contentype, subpath: contextid, filename: "updateparts.json"});
                for (var datatype in data[contentype][contextid]) {
                    if (typeof(updateparts[datatype]) === "undefined") {
                        updateparts[datatype] = data[contentype][contextid][datatype];
                    } else {
                        updateparts[datatype] = updateparts[datatype].concat(data[contentype][contextid][datatype]);
                    };
                };
                if (!this.saveContentData({
                    savetype: "objtofiles",
                    saveobj: {"updateparts": updateparts},
                    contentType: {
                        category: "content",
                        type: contentype,
                        context: {contextid: contextid}
                    }
                })){
                    if (typeof(returnobj.updateerror[contentype]) === "undefined") {
                        returnobj.updateerror[contentype] = [];
                    };
                    returnobj.updateerror[contentype].push(contextid);
                };
            }
        }
        return returnobj;
    }
    /******
     *
     ******/
    appFileSystem.prototype.processSystemUpdate = function(data) {
        
        if (typeof(data) === "string") {
            try {
                data = JSON.parse(data);
            } catch(e) {
                console.log("UpdateData parse error");
                console.log(e);
                console.log(data);
            };
        };
        if (data.reload) {this.reloadNeeded = true;}
        switch (data.type) {
            case "file":
                this.processUpdateFile(data);
                break;
            case "json":
                break;
        };
    };

    /******
     *
     ******/
    appFileSystem.prototype.processUpdates = function(updateData, eventplace, handlingdata) {
        console.log('Processing Updates:', updateData, eventplace, handlingdata);
        var contentUpdateData = {};
        for (var i = 0; i < updateData.systemupdate.length; i++){
            this.processSystemUpdate(updateData.systemupdate[i]);
        };
        var sentid;
        for (var i = 0; i < updateData.contentupdate.length; i++){
            var currentData = updateData.contentupdate[i];
            sentid = currentData.contexttype + currentData.contextid + currentData.dataid + currentData.datamodified;
            try {
                var indexofsent = handlingdata.contentupdates.sent.indexOf(sentid);
            } catch(e) {
                console.log("Sentindex error",e);
                //if some errors for sentinfo process update anyway
                var indexofsent = -1;
            };
            if (indexofsent === -1) {
                if (typeof(contentUpdateData[currentData.contexttype]) === "undefined") {
                    contentUpdateData[currentData.contexttype] = {};
                };
                if (typeof(contentUpdateData[currentData.contexttype][currentData.contextid]) === "undefined") {
                    contentUpdateData[currentData.contexttype][currentData.contextid] = {};
                };
                if (typeof(contentUpdateData[currentData.contexttype][currentData.contextid][currentData.datatype]) === "undefined") {
                    contentUpdateData[currentData.contexttype][currentData.contextid][currentData.datatype] = [];
                };
                contentUpdateData[currentData.contexttype][currentData.contextid][currentData.datatype].push(currentData.data);
            } else {
                handlingdata.place.trigger('removesentinfo', {type: "contentupdates", item: sentid, index: indexofsent});
            }
            // this.processUpdate(updateData.contentupdate[i],handlingdata);
        }
        var contentupdates = this.processContentUpdates(contentUpdateData);
        if (this.reloadNeeded) {
            require('nw.gui').Window.get().reload(); 
        } else {
            eventplace.trigger(updateData.requesttype + "_done");
            try {
                handlingdata.place.trigger('saveupdatesettings.system');
            } catch(e) {
                console.log('saveupdatesettings error');
            };
            return {content: contentupdates};
        }
        
    }
    /******
     *
     ******/
    appFileSystem.prototype.saveObjectToFile = function(fileInfo) {
        return this.writeToFileSync(fileInfo.contentType,fileInfo.contentFile,fileInfo.contentData); 
    }
    /******
     * refresh infofiles
     ******/
    appFileSystem.prototype.refreshFileInfo = function(refreshtype) {
        return this.createContentInfo(this.getPath(['content',refreshtype]));
        
    }
    /******
     * Node file remove
     ******/
    appFileSystem.prototype.fileremove = function(filepath) {
            this.fileSystem.unlinkSync(filepath);        
    }
    /*****
     * 
     *****/
     appFileSystem.prototype.dirremove = function(path) {
        var files = [];
        var appFS = this;
        if( this.fileSystem.existsSync(path) ) {
            files = this.fileSystem.readdirSync(path);
            files.forEach(function(file,index){
                var curPath = path + "/" + file;
                if(appFS.fileSystem.lstatSync(curPath).isDirectory()) { // recurse
                    appFS.dirremove(curPath);
                } else { // delete file
                    appFS.fileSystem.unlinkSync(curPath);
                }
            });
            this.fileSystem.rmdirSync(path);
        }
    };
    /******
     *
     ******/
    appFileSystem.prototype.removeContent = function(content) {
        try{
            this.dirremove(this.getContentPath(content));
        }catch(err){
            console.log(err);
            try{
                this.removeFile(content);
            }catch(err){
                console.log(err);
            }
        }
    };
    /******
     *
     ******/
    appFileSystem.prototype.removeFile = function(file) {
        if(typeof(file)==="string"){
            var filePath= this.getPath('content')+file;
        }else if(typeof(file)==="object"){
            var filePath = this.getPath(file.path)+file.filename;
        }
        try{
            this.fileremove(filePath);
            return {success:true}
        }catch(e){
            console.log("file remove Error");
            console.log(e);
            //TODO:Parse error: throw somekind of error
            return {};
        }
    }
    /******
     *
     ******/
    appFileSystem.prototype.writeManifest = function(appkey,debugmode) {
        var packagejson = this.fileSystem.readFileSync('package.json', 'utf8');
        var pkgjson;
        try {
            pkgjson = JSON.parse(packagejson);
        } catch (err) {
            pkgjson = {
                "name"          : appkey,
                "main"          : "./main.html",
                "chromium-args" : "--enable-experimental-web-platform-features",
                "window": {
                    "toolbar"   : !!debugmode,
                    "width"     : 1000,
                    "height"    : 1000
                },
                "webkit": {
                    "plugin"    : true
                }
            }
        };
        pkgjson.name = appkey;
        pkgjson.window.toolbar = !!debugmode;
        this.fileSystem.writeFileSync('package.json', JSON.stringify(pkgjson));
    }
    /******
     *
     ******/
    appFileSystem.prototype.saveContentData = function(savedata) {
        var returnarray = []
        switch(savedata.savetype){
            case"objtofiles":
                var newsaves = function(){var valuelist =[];for(var key in savedata.saveobj){valuelist.push({"key":key,"value":savedata.saveobj[key]})};return valuelist}();
                var writepath = this.getContentPath(savedata.contentType) || false;
                switch(this.checkTypeofFile(writepath)){
                    case"directory":
                        //all ok
                        break;
                    case"nonexists":
                        //make one
                        this.fileSystem.mkdirSync(writepath);
                        break;
                    default:
                        //some kind of error
                        console.log("Error content directory");
                        return ;
                        break;
                }
                for(var i=0;i<newsaves.length;i++){
                    var fullfilename = writepath+"/"+newsaves[i].key+".json";
                    var exists = this.checkFileExists(fullfilename);
                    if(savedata.savemethod ==="write" || !exists){
                        this.fileSystem.writeFileSync(fullfilename,(typeof(newsaves[i].value) === "object" ? JSON.stringify(newsaves[i].value):newsaves[i].value));
                    }else{
                        var oldcontent = JSON.parse(this.readFileToString(fullfilename));
                        var oldcontenttype = (oldcontent instanceof Array ? "array" :typeof(oldcontent));
                        var newcontenttype = (typeof(newsaves[i].value) ==="object" && typeof(newsaves[i].value.length) !=="undefined" ? "array" : typeof(newsaves[i].value));
                        var savecontent = "";
                        var type_error = false;
                        switch(oldcontenttype){
                            case"string":
                                if(newcontenttype ==="string"){
                                    savecontent = oldcontent + newsaves[i].value;
                                }else{
                                    console.log("typeError:string");
                                    type_error =true;
                                }
                                break;
                            case "object":
                                if(newcontenttype ==="object"){
                                    savecontent = jQuery.extend(true,{},oldcontent , newsaves[i].value);
                                }else{
                                    console.log("typeError:object");
                                    type_error = true;
                                }
                                break;
                            case "array":
                                savecontent = oldcontent.slice();
                                if(newcontenttype ==="string" || newcontenttype ==="object"){
                                    savecontent.push(newsaves[i].value)
                                }else if(newcontenttype ==="array"){
                                    savecontent = savecontent.concat(newsaves[i].value);
                                }else{
                                    console.log("typeError:array");
                                    type_error = true;
                                }
                                break;
                            default:
                                type_error = true;
                            break;
                        }
                        if(!type_error){
                            try{
                                if(oldcontenttype ==="string"){
                                    this.fileSystem.writeFileSync(fullfilename,savecontent);
                                }else{
                                    this.fileSystem.writeFileSync(fullfilename,JSON.stringify(savecontent));
                                }
                                returnarray.push({success:true,data:newsaves[i].value});
                            }catch(err){
                                console.log("saveContentData file write error",err);
                                returnarray.push({success:false,data:newsaves[i].value});
                            }
                        }else{
                            console.log("Contentdata save Errror",newsaves[i],writepath);
                            returnarray.push({success:false,data:newsaves[i].value});
                        }
                    }
                }
                
            break;
            default:
            break;
        }
        return returnarray;
    }
    appFileSystem.prototype.deleteFolderRecursive = function(path) {
        var fs = this.fileSystem;
        var appFileSelf = this;
        if( fs.existsSync(path) ) {
            fs.readdirSync(path).forEach(function(file,index){
                var curPath = path + "/" + file;
                if(fs.lstatSync(curPath).isDirectory()) { // recurse
                    appFileSelf.deleteFolderRecursive(curPath);
                } else { // delete file
                    fs.unlinkSync(curPath);
                }
            });
            fs.rmdirSync(path);
        }
    };
    appFileSystem.prototype.clearContentData = function(list){
        var fileSystemSelf = this;
        list.forEach(function(type,index){
            fileSystemSelf.deleteFolderRecursive(fileSystemSelf.getContentPath({category:"content",type:type}));
        });
        
    }
    
    appFileSystem.prototype.defaults = {
        libraryoptions: {
            "subjects": [
                {
                    "id": "biology",
                    "name": "biology",
                    "weight": 1,
                    "title": {
                        "en": "biology",
                        "fi": "biologia",
                        "sv": "biologi"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-biology\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M7 22 a12 12 0 0 1 6 -16 v2 a10 10 0 0 1 11 -2 a10 10 0 0 1 -2 11 h2 a12 12 0 0 1 -16 6 l-2 2 l-1 -1z m2 -2 l1 1 l 12 -12.5z\"></path></svg>"
                },
                {
                    "id": "chemistry",
                    "name": "chemistry",
                    "weight": 1,
                    "title": {
                        "en": "chemistry",
                        "fi": "kemia",
                        "sv": "kemi"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-chemistry\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M20 5 l5 5 l-14 14 a2.5 2.5 0 0 1 -5 -5z m0 1 l-8 8 h8 l4 -4z m-6 10 a1 1 0 0 0 0 2 a1 1 0 0 0 0 -2z m-4 1 a1 1 0 0 0 0 2 a1 1 0 0 0 0 -2z m-1 4 a1 1 0 0 0 0 2 a1 1 0 0 0 0 -2z\"></path></svg>"
                },
                {
                    "id": "physics",
                    "name": "physics",
                    "weight": 1,
                    "title": {
                        "en": "physics",
                        "fi": "fysiikka",
                        "sv": "fysik"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-physics\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M15 13 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m0 -12 a5 14 0 0 1 0 28 a5 14 0 0 1 0 -28z m0 1 a4 13 0 0 0 0 26 a4 13 0 0 0 0 -26z m-12.2 5.8 a5 14 -60 0 1 24.4 14.4 a5 14 -60 0 1 -24.4 -14.4z m0.7 0.7 a4 13 -60 0 0 23 13 a4 13 -60 0 0 -23 -13z m24.4 0 a5 14 60 0 1 -24.4 14.4 a5 14 60 0 1 24.4 -14.4z m-0.7 0.7 a4 13 60 0 0 -23 13 a4 13 60 0 0 23 -13z\"></path></svg>"
                },
                {
                    "id": "english",
                    "name": "english",
                    "weight": 1,
                    "title": {
                        "en": "English",
                        "fi": "englanti",
                        "sv": "engelska"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-english\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><text x=\"15\" y=\"19\" style=\"text-anchor: middle; font-size: 12px; font-family: sans-serif; font-weight: bold; stroke: none;\" class=\"mini-icon-foreground\">[en]</text></svg>"
                },
                {
                    "id": "finnish",
                    "name": "finnish",
                    "weight": 1,
                    "title": {
                        "en": "Finnish",
                        "fi": "suomi",
                        "sv": "finska"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-finnish\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><text x=\"15\" y=\"19\" style=\"text-anchor: middle; font-size: 12px; font-family: sans-serif; font-weight: bold; stroke: none;\" class=\"mini-icon-foreground\">[fi]</text></svg>"
                },
                {
                    "id": "geography",
                    "name": "geography",
                    "weight": 1,
                    "title": {
                        "en": "geography",
                        "fi": "maantiede",
                        "sv": "geografi"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-geography\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M8 19 a7 7 0 0 1 13 -13 l2 -2 a0.5 0.5 0 0 1 1 1 a12 12 0 0 1 -9 20 v2 h3 v1 h-7 v-1 h3 v-2 a12 12 0 0 1 -7 -3 a0.5 0.5 0 0 1 -1 -1z m1 1 l-1 1 a10.61 10.61 0 0 0 15 -15 l-1 1 a7 7 0 0 1 -13 13z m1 -1 a8 8 0 0 0 13.4 -4 l-3 2 l-1 -2 l-1 -1 l-3 2 l-3 3 l-2 -1 l0 -5 l-2 -2 l2 -4 l4 1 l-3 -2 l2 -2 a8.5 8.5 0 0 0 -3.4 15z\"></path></svg>"
                },
                {
                    "id": "history",
                    "name": "history",
                    "weight": 1,
                    "title": {
                        "en": "history",
                        "fi": "historia",
                        "sv": "historia"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-history\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M15 1 l13 8 h-26z m-10 9 h2 v12 h-2z m6 0  h2 v12 h-2z m6 0  h2 v12 h-2z m6 0  h2 v12 h-2z m-19 13 h22 v2 h-22z\"></path></svg>"
                },
                {
                    "id": "mathematics",
                    "name": "mathematics",
                    "weight": 1,
                    "title": {
                        "en": "mathematics",
                        "fi": "matematiikka",
                        "sv": "matematik"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-math\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M3 17 h3 l2 4 l3 -15 h14 a1 1 0 0 1 0 2 h-13 l-3 17 a1 1 0 0 1 -1 0 l-3 -6 h-2 a1 1 0 0 1 0 -2z m11 -3 a1 1 0 0 1 2 -2 l4 4 l4 -4 a1 1 0 0 1 2 2 l-4 4 l4 4 a1 1 0 0 1 -2 2 l-4 -4 l-4 4 a1 1 0 0 1 -2 -2 l4 -4z\"></path></svg>"
                },
                {
                    "id": "swedish",
                    "name": "swedish",
                    "weight": 1,
                    "title": {
                        "en": "Swedish",
                        "fi": "ruotsi",
                        "sv": "svenska"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-swedish\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><text x=\"15\" y=\"19\" style=\"text-anchor: middle; font-size: 12px; font-family: sans-serif; font-weight: bold; stroke: none;\" class=\"mini-icon-foreground\">[sv]</text></svg>"
                },
                {
                    "id": "languages",
                    "name": "languages",
                    "weight": 2,
                    "title": {
                        "en": "other languages",
                        "fi": "muut kielet",
                        "sv": "andra språk"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-languages\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M4 21 a2 3 0 0 0 2 -2 a6 5 0 0 1 7 -8 a6 5 0 0 0 2.5 5.5 a6 5 0 0 1 -7.5 3.5 a3 2.5 0 0 1 -4 1z m17.5 -4.5 a6 5 0 1 1 2 -1 a2 3 0 0 0 2 3 a4 3 0 0 1 -4 -2z m-8.5 1.5 l-2 -6 h-2 l-2 6 h1.5 l0.66 -2 h1.66 l0.66 2 h1.5z m-2.4 -2.8 h-1.2 l0.6 -1.9z m10 -4.5 a8 8 0 0 1 -1.5 1.5 a8 8 0 0 1 -1 -1 l-0.7 0.7 a8 8 0 0 0 1 1 a8 8 0 0 1 -2 1 v1 a8 8 0 0 0 2.8 -1.4 a8 8 0 0 0 2.8 1.4 v-1 a8 8 0 0 1 -2 -1 a8 8 0 0 0 1.8 -2.2 h1 v-1 h-2.5 v-1.3 h-1.3 v1.3 h-3 v1z\" /></svg>"
                },
                {
                    "id": "psychology",
                    "name": "psychology",
                    "weight": 1,
                    "title": {
                        "en": "psychology",
                        "fi": "psykologia",
                        "sv": "psykologi"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-psychology\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M12 25 v-4 h-2 a2 2 0 0 1 -2 -2 v-2 h-2 l2 -4 a7 7 0 1 1 13 3 a2 4 0 0 0 -2 4 v5z m3 -6 a1 1 0 0 0 0 -2 a1 1 0 0 0 0 2z m-1 -4 a1 1 0 0 0 2 0 v-2 a3 3 0 1 0 -4 -3 h1 a1.7 1.7 0 1 1 2 2 a1 1 0 0 0 -1 1z\"></path></svg>"
                },
                {
                    "id": "philosophy",
                    "name": "philosophy",
                    "weight": 1,
                    "title": {
                        "en": "philosophy",
                        "fi": "filosofia",
                        "sv": "filosofi"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-philosophy\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M24 25 h-14 v-2 h1 c-0.5 -0.5 -0.5 -1 1 -2 c0.5 -0.5 -0.5 -1 -4 -4 c-1 -0.3 -2 -2.5 1 -3 c-1.5 -2.5 -3 -3 -1 -5 c-4 -2 -0.5 -6 2 -4 c2 1 2 1 4 1 c4 1 6 4 7 9 c0 1 0.2 1.5 -1 2.4 c2 0 2 0 2 2 v1 c0.5 0.5 0.5 2 1 3 h1z m-8 -10.5 c0 -0.9 -2.3 -3.5 -3 -3.5 l-1 3z m-6.6 -4.5 c-1.5 1 -1.5 1 0.6 2.6 c0.5 -0.3 0.6 -0.8 0.8 -2.6 c-0.2 -0.2 0.2 -1.6 -1.4 0z\"></path></svg>"
                },
                {
                    "id": "religion",
                    "name": "religion",
                    "weight": 1,
                    "title": {
                        "en": "religion and life stance education",
                        "fi": "uskonto ja elämänkatsomustieto",
                        "sv": "religion och Livsåskådning"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-religion\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\"  style=\"stroke: none;\" d=\"M15 2 c0 3 3 4 3 6 a3 3 0 0 1 -2.5 3 l-0.5 -3 l-0.5 3 a3 3 0 0 1 -2.5 -3 c0 -2 2 -5 3 -6z m-3 10 a5 5 0 0 0 6 0 l1 15 h-8z\" /></svg>"
                },
                {
                    "id": "social-studies",
                    "name": "social studies",
                    "weight": 1,
                    "title": {
                        "en": "social studies",
                        "fi": "yhteiskuntaoppi",
                        "sv": "samhällslära"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-social-studies\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M21 8 h-4 c0 -2 -4 -2 -4 0 c0 2 7.2 3 8 6 c0.5 2 0 3 -1 4 c1 1 1 2 1 3 c0 6 -12 6 -12 0 h4 c0 2 4 2 4 0 c0 -2 -7.2 -3 -8 -6 c-0.5 -2 0 -3 1 -4 c-1 -1 -1 -2 -1 -3 c0 -6 12 -6 12 0z m-8.5 4 c-2 3.5 4 4.5 5 5 c2 -3.5 -4 -4.5 -5 -5z\" /></svg>"
                },
                {
                    "id": "music",
                    "name": "music",
                    "weight": 1,
                    "title": {
                        "en": "music",
                        "fi": "musiikki",
                        "sv": "musik"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-music\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\"  style=\"stroke: none;\" d=\"M10 6 l12 -3 v15 a2.6 2 -30 0 1 -6 2 a3 2 -15 0 1 5 -2.3 v-10 l-10 2.5 v12 a2.6 2 -30 0 1 -6 2 a3 2 -15 0 1 5 -2.3z\" /></svg>"
                },
                {
                    "id": "visual-art",
                    "name": "visual art",
                    "weight": 1,
                    "title": {
                        "en": "visual art",
                        "fi": "kuvaamataito",
                        "sv": "bildkonst"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-visual-art\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\"  style=\"stroke: none;\" d=\"M20 7 c0 2 -4 11 -6 14 l-2 -1 c1 -3 5.5 -10.5 8 -13z m-6.5 14.5 c1 1 1 3 -2 3 s-1 1 -2 2 c-2 -2 -1 -6 2 -6z m-5 -13 l2 -1.5 l4.5 6 l-1.5 2.5z m9 7.5 l4.5 6 l-2.5 2 l-3.75 -5z m5.1 6.8 l0.5 3 l-3 -1z\" /></svg>"
                },
                {
                    "id": "physical-education",
                    "name": "physical education",
                    "weight": 1,
                    "title": {
                        "en": "physical education",
                        "fi": "liikunta ja terveystieto",
                        "sv": "hälsokunskap och gymnastik"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-physical-education\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M20 5 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m-2 4 l3 3 l3 -3 a1 1 0 0 1 1.4 1.4 l-3 3 a2 2 0 0 1 -2.8 0 l-1.2 -1.2 l-3 3 l3 3 a2 2 0 0 1 1 2 l-2 5 a1 1 0 0 1 -2.5 -1 l1.5 -3.75 l-3 -3 l-2 2 a2 2 0 0 1 -1 0.5 h-5 a1.4 1.4 0 0 1 0 -2.8 h 3.8 l6 -6 h-2 l-3 3 a1 1 0 0 1 -1.4 -1.4 l3.7 -3.7z\" /></svg>"
                },
                {
                    "id": "misc",
                    "name": "misc",
                    "weight": 100,
                    "title": {
                        "en": "other",
                        "fi": "muu",
                        "sv": "andra"
                    },
                    "icon": "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" width=\"30\" height=\"30\" viewBox=\"0 0 30 30\" class=\"mini-icon mini-icon-miscsubject\"><circle class=\"mini-icon-background\" stroke=\"none\" fill=\"none\" cx=\"15\" cy=\"15\" r=\"15\" /><path class=\"mini-icon-foreground\" style=\"stroke: none;\" d=\"M15 3 a12 12 0 0 1 0 24 a12 12 0 0 1 0 -24z m 0 2 a10 10 0 0 0 0 20 a10 10 0 0 0 0 -20z m-4 5 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m8 0 a2 2 0 0 1 0 4 a2 2 0 0 1 0 -4z m0 9 a1 1 0 0 1 1 1 a7 7 0 0 1 -10 0 a1 1 0 0 1 1 -1 a5 5 0 0 0 8 0z\"></path></svg>"
                }
            ]
        }
    }