/******
@name Credits
@version 0.1
@author Petri Salmela <petri.salmela@abo.fi>
@author Petri Sallasmaa <petri.sallasmaa@abo.fi>
@type plugin
@requires jQuery x.x.x or newer
@class Credits
@description A class and jQuery-plugin for showing credits given as JSON-file.

TODO:
*******/

/**
 * Requirements:
 * - jQuery
 */

try {
    typeof(jQuery) !== 'undefined';
} catch (err) {
    throw new Error('Missing dependency in ' + err.fileName + '\n' + err);
}

/**
 * Optional requirements
 * - EbookLocalizer
 */

if (typeof(checkOptionalRequirements) !== 'undefined' && checkOptionalRequirements) {
    try {
        typeof(EbookLocalizer) !== 'undefined';
        typeof(ebooklocalizer) !== 'undefined';
    } catch (err) {
        throw new Error('Missing optional dependency in ' + err.fileName + '\n' + err);
    }
}

;(function(window, $, ebooklocalizer){

    /**
     * Credits and copyright showing tool
     * @constructor
     * @param {jQuery} place    DOM-place for credits
     * @param {Object} options  data of credits
     */
    var Credits = function(place, options) {
        options = $.extend({}, this.defaults, options);
        this.place = $(place);
        this.init(options);
        this.initHandlers();
        this.setAttrs();
        this.setStyles();
        this.show();
    }
    
    Credits.prototype.init = function(options) {
        this.settings = options.settings;
        this.data = options.data;
        this.data.credits.sort(function(a, b) {
            return (a.type < b.type || (a.type === b.type && a.name.toLowerCase() < b.name.toLowerCase()) ? -1 : 1);
        });
    };
    
    Credits.prototype.initHandlers = function() {
        var credits = this;
        this.place.on('closechildrenapp', function(event, data){
            event.stopPropagation();
            credits.close()
        }).on('click', '.credits-itemlist-item a', function(event){
            event.stopPropagation();
            event.preventDefault();
            var link = $(this);
            var selector = link.attr('href');
            var target = $(selector);
            target[0].scrollIntoView({behavior: 'smooth', block: 'start'});
        });
    };
    
    Credits.prototype.setAttrs = function() {
        this.place.addClass('creditview');
    };
    
    Credits.prototype.show = function() {
        this.view();
    };
    
    Credits.prototype.view = function() {
        var uilang = this.settings.uilang;
        var html = this.template.html;
        this.place.html(html);
        var title = this.place.find('.credits-title');
        var list = this.place.find('.credits-itemlist');
        var area = this.place.find('.credits-creditarea');
        title.html(ebooklocalizer.localize('credits:creditsncopyrights', uilang));
        var credits = this.data.credits;
        var creditem, currtype, ctype, item, itemname;
        var contributors, licenses, files;
        for (var i = 0, len = credits.length; i < len; i++) {
            creditem = credits[i];
            console.log(creditem.name);
            list.append(this.fillTemplate(this.template.listitem, {
                typeicon: this.icons[creditem.type],
                name: creditem.name,
                nameid: creditem.name.replace(/[^a-zA-z0-9\-_]/g, '-')
            }));
            if (currtype !== creditem.type) {
                ctype = ebooklocalizer.localize('credits:' + creditem.type, uilang);
                area.append('<h1 class="credits-credtype">' + ctype + '</h1>');
                currtype = creditem.type;
            };
            contributors = [];
            for (var j = 0, jlen = creditem.contributors && creditem.contributors.length || 0; j < jlen; j++) {
                item = creditem.contributors[j];
                itemname = (this.settings.hasnet && item.url ? this.fillTemplate(this.template.contributorlink, {
                    conturl: item.url,
                    contname: item.name
                }) : item.name);
                contributors.push(this.fillTemplate(this.template.contributor, {
                    contname: itemname,
                    contemail: (item.email ? '&lt;' + item.email + '&gt;' : '')
                }));
            };
            licenses = [];
            for (var j = 0, jlen = creditem.licenses && creditem.licenses.length || 0; j < jlen; j++) {
                item = creditem.licenses[j];
                itemname = (item.url ? this.fillTemplate(this.template.licenselink, {
                    licurl: (this.settings.hasnet ? item.url : '../' + item.file),
                    licname: item.name
                }) : item.name);
                licenses.push(this.fillTemplate(this.template.license, {
                    licname: itemname,
                    licfile: item.file
                }));
            };
            files = [];
            for (var j = 0, jlen = creditem.files && creditem.files.length || 0; j < jlen; j++) {
                item = creditem.files[j];
                files.push(this.fillTemplate(this.template.file, {
                    filename: item
                }));
            };
            area.append(this.fillTemplate((this.settings.hasnet ? this.template.creditblock : this.template.creditblocknonet), {
                type: creditem.type,
                name: creditem.name,
                nameid: creditem.name.replace(/[^a-zA-z0-9\-_]/g, '-'),
                description: creditem.description,
                url: creditem.url,
                contributorlist: contributors.join('\n'),
                contributortitle: ebooklocalizer.localize('credits:contributortitle', uilang),
                licenselist: licenses.join('\n'),
                licensetitle: ebooklocalizer.localize('credits:licensetitle', uilang),
                filelist: files.join('\n'),
                filetitle: ebooklocalizer.localize('credits:filetitle', uilang)
            }));
        };
    };
    
    Credits.prototype.close = function() {
        //this.place.trigger('closeappok', {appid: this.appid, apptype: this.apptype});
    }
    
    Credits.prototype.setStyles = function() {
        if ($('head style#creditviewstyles').length === 0) {
            $('head').append('<style type="text/css" id="creditviewstyles">'+this.styles+'</style>')
        }
    }

    /**
     * Fill data in template text
     * @param {String} template   Text with placeholders as '{{% variablename }}'
     * @param {Object} mapping    Object that maps variablenames to values
     * @returns {String}          Template replaced with given values for variables.
     */
    Credits.prototype.fillTemplate = function(template, mapping) {
        var rex;
        for (var key in mapping) {
            rex = RegExp('{{%\\s*'+key+'\\s*%}}', 'g')
            template = template.replace(rex, (mapping[key] || '') + '');
        };
        return template;
    };
    
    Credits.prototype.defaults = {
        type: 'creditdata',
        data: {
            credits: []
        },
        settings: {
            uilang: 'en',
            hasnet: true
        }
    };
    
    Credits.prototype.styles = [
        '.creditview {position: absolute; top: 0; bottom: 0; left: 0; right: 0; display: flex; flex-flow: column nowrap; justify-content: space-between; overflow: hidden;}',
        'h1.credits-title {flex-grow: 0; flex-shrink: 0;}',
        '.credits-creditview {flex-grow: 1; display: flex; flex-flow: row nowrap; justify-content: space-between; overflow: hidden;}',
        '.credits-itemlist {flex-grow: 0; flex-shrink: 0; overflow: auto; list-style: none; margin: 0; padding: 0.5em;}',
        '.credits-itemlist-item {margin: 0; padding: 0; border: 1px solid #aaa;}',
        '.credits-itemlist-item a {display: block; color: inherit; margin: 0; padding: 0.3em 0.5em; text-decoration: none;}',
        '.credits-creditarea {flex-grow: 1; flex-shrink: 1; overflow: auto;padding: 0 1em;}',
        '.credits-credtype {border-bottom: 3px solid black;}',
        '.credits-creditblock {border-radius: 0.5em; border: 1px solid #aaa; margin: 2em 0; padding: 0.5em; background-color: white;}',
        // Contributors
        '.credits-contributoremail {font-style: italic;}',
        // Files
        '.credits-files ul {list-style: none;}',
        '.credits-fileitem {font-family: monospace;}'
    ].join('\n');
    
    Credits.prototype.icons = {
        appicon: '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="30" height="30" viewBox="0 0 30 30" class="mini-icon mini-icon-videoconf"><path style="stroke: none;" d="M15 5 a5 5 0 0 1 0 10 a5 5 0 0 1 0 -10z m-10 16 a6 6 0 0 1 6 -6 l4 4 l4 -4 a6 6 0 0 1 6 6z m-4 -17 a2 2 0 0 1 2 -2 h24 a2 2 0 0 1 2 2 v18 a2 2 0 0 1 -2 2 h-11 v2 h6 v2 h-14 v-2 h6 v-2 h-11 a2 2 0 0 1 -2 -2z m2 0 v18 h24 v-18z" /></svg>',
        javascript: 'JS',
        css: 'CSS',
        font: 'F',
        image: 'I',
        code: 'C'
    }
    
    Credits.prototype.template = {
        html: [
            '<h1 class="credits-title"></h1>',
            '<div class="credits-creditview">',
            '    <ul class="credits-itemlist"></ul>',
            '    <div class="credits-creditarea"></div>',
            '</div>'
        ].join('\n'),
        listitem: [
            '<li class="credits-itemlist-item ffwidget-background">',
            '  <a href="#{{% nameid %}}">',
            '    <span class="credits-item-icon">{{% typeicon %}}</span>',
            '    <span class="credits-item-name">{{% name %}}</span>',
            '  </a>',
            '</li>'
        ].join('\n'),
        creditblock: [
            '<div class="credits-creditblock" data-itemtype="{{% type %}}" id="{{% nameid %}}">',
            '    <h2 class="credits-block-title">{{% name %}}</h2>',
            '    <div class="credits-block-description">{{% description %}}</div>',
            '    <div class="credits-block-url"><a target="_blank" href="{{% url %}}">{{% url %}}</a></div>',
            '    <div class="credits-contributors">',
            '        <h3>{{% contributortitle %}}</h3>',
            '        <ul class="credits-contributorlist">',
            '            {{% contributorlist %}}',
            '        </ul>',
            '    </div>',
            '    <div class="credits-licenses">',
            '        <h3>{{% licensetitle %}}</h3>',
            '        <ul class="credits-licenselist">',
            '            {{% licenselist %}}',
            '        </ul>',
            '    </div>',
            '    <div class="credits-files">',
            '        <h3>{{% filetitle %}}</h3>',
            '        <ul class="credits-filelist">',
            '            {{% filelist %}}',
            '        </ul>',
            '    </div>',
            '</div>'
        ].join('\n'),
        creditblocknonet: [
            '<div class="credits-creditblock" data-itemtype="{{% type %}}" id="{{% nameid %}}">',
            '    <h2 class="credits-block-title">{{% name %}}</h2>',
            '    <div class="credits-block-description">{{% description %}}</div>',
            '    <div class="credits-block-url"><span>{{% url %}}</span></div>',
            '    <div class="credits-contributors">',
            '        <h3>{{% contributortitle %}}</h3>',
            '        <ul class="credits-contributorlist">',
            '            {{% contributorlist %}}',
            '        </ul>',
            '    </div>',
            '    <div class="credits-licenses">',
            '        <h3>{{% licensetitle %}}</h3>',
            '        <ul class="credits-licenselist">',
            '            {{% licenselist %}}',
            '        </ul>',
            '    </div>',
            '    <div class="credits-files">',
            '        <h3>{{% filetitle %}}</h3>',
            '        <ul class="credits-filelist">',
            '            {{% filelist %}}',
            '        </ul>',
            '    </div>',
            '</div>'
        ].join('\n'),
        contributor: [
            '<li class="credits-contributoritem">',
            '  <span class="credits-contributorname">{{% contname %}}</span>',
            '  <span class="credits-contributoremail">{{% contemail %}}</span>',
            '</li>'
        ].join('\n'),
        contributorlink: '<a href="{{% conturl %}}" target="_blank">{{% contname %}}</a>',
        license: [
            '<li class="credits-licenseitem" data-licensefile="{{% licfile %}}">',
            '  <div class="credits-licensename">{{% licname %}}</div>',
            '</li>'
        ].join('\n'),
        licenselink: '<a href="{{% licurl %}}" target="_blank">{{% licname %}}</a>',
        file: [
            '<li class="credits-fileitem">',
            '  <div class="credits-filename">{{% filename %}}</div>',
            '</li>'
        ].join('\n')
    };
    
    Credits.localization = {
        "en": {
            "credits:creditsncopyrights": "Credits and Copyrights",
            "credits:code": "Programs",
            "credits:css": "CSS-styles",
            "credits:font": "Fonts",
            "credits:javascript": "Javascript",
            "credits:contributortitle": "Contributors",
            "credits:licensetitle": "Licenses",
            "credits:filetitle": "Files"
        },
        "fi": {
            "credits:creditsncopyrights": "Tekijät ja tekijänoikeudet",
            "credits:code": "Ohjelmat",
            "credits:css": "CSS-tyylit",
            "credits:font": "Fontit",
            "credits:javascript": "Javascriptit",
            "credits:contributortitle": "Tekijät",
            "credits:licensetitle": "Lisenssit",
            "credits:filetitle": "Tiedostot"
        },
        "sv": {
        }
    }

    if (ebooklocalizer) {
        ebooklocalizer.addTranslations(Credits.localization);
    } else {
        ebooklocalizer = {
            translations: {},
            addTranslations: function(trans){
                this.translations = $.extend(true, this.translations, trans);
            },
            localize: function(key, lang){
                lang = (this.translations[lang] ? lang : 'en');
                return this.translations[lang] && this.translations[lang][key] || key;
            }
        }
        ebooklocalizer.addTranslations(Credits.localization);
    }
    
    /**** jQuery-plugin *****/
    var methods = {
        'init': function(params){
            return this.each(function(){
                var creds = new Credits(this, params);
            });
        },
        'getdata': function(){
            var $place = $(this).eq(0);
            $place.trigger('getdata');
            var data = $place.data('[[creditsdata]]');
            return data;
        },
        'geticon': function() {
            return Credits.prototype.icons.appicon;
        },
        'gettitle': function() {
            return '';
        }
    }
    
    $.fn.creditview = function(method){
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof(method) === 'object' || !method) {
            return methods.init.apply(this, arguments);
        } else {
            $.error('Method ' + method + ' does not exist in creditview.');
            return false;
        }
    };

    
})(window, jQuery, window.ebooklocalizer);
